/**
 * $Id$
 *
 * 特殊フォルダ「SQL」関連の関数をまとめたヘッダファイル
 */
#ifndef INCLUDE_TF_RDBO_SQL_H
#define INCLUDE_TF_RDBO_SQL_H

#include "apr.h"
#include "apr_pools.h"
#include "tf_rdbo_util.h"
#include "tf_rdbo.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare public functions 
  ----------------------------------------------------------------------------*/
/**
 * SQL の表示情報の検索・取得および、SQLの解析を実施する。
 * (note)
 *	sqlinformationsearch が実施するDB検索およびSQL解析処理は
 *	ここで実装されています。
 *	このSEARCH専用になっていますので、他の用途には使用してはなりません。
 * 
 * @param request_rec * apache request_rec構造体
 * @param divy_rdbo_resource ** 検索結果格納
 * @param const char * SQL文
 * @param apr_hash_t * divy_sqldepend 取得用 topid のハッシュ
 * @param apr_hash_t * sqlpreference 時の クライアントでキャッシュされた
 * 			requiredsql
 * @param divy_rdbo_rsvalue * クライアントから送られてきたrequiredsql のリスト
 * @param int 検索オプション see search.h
 * @return int 0:成功、1:失敗
 */
DIVY_DECLARE(int) divy_rdbo_sqlinformationsearch(request_rec *r,
						divy_rdbo_resource **rdb_r,
						const char *sqlstr,
						apr_hash_t *sqldtopid_h,
						apr_hash_t *cachedreqsql,
						divy_rdbo_rsvalue *rsvalue,
						int searchopt);

/**
 * 指定されたsqluri が示すSQLの情報を取得して返却する。
 *
 * select対象のテーブル: divy_sql
 * (note)
 *	sqluri = /$sqlname
 *	※sqluri は $root/.management/SQL/$sqlname とは異なります。
 *
 * @param r request_rec *
 * @param sqluri const char * SQLのuri(相対URI)を表す文字列
 * @param sql_pr divy_rdbo_sql ** 取得したSQL情報
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_sql_property(request_rec *r, const char *sqluri,
							divy_rdbo_sql **sql_pr);

/**
 * 指定されたsql_pr が表すSQLを新規登録する。
 *
 * insert対象のテーブル: divy_sql, divy_sqlcontent
 * (note) sql_pr の必須項目:
 *	relativeuri, labelname, sql, active, usedbmsname, sqltype
 *	registdt は自動的に補完します。
 *
 * (note) sql_pr->sqlcnt_pr の必須項目:
 *	id, name
 *	但し、contype DIVY_SQL_CONTYPE_REQUIRED であれば sqlmodeも必須
 *	上記以外のデータは、指定されていた値をそのまま反映します。
 * (note)
 *    同名のSQLがあってはなりません。
 *
 * @param r request_rec *
 * @param sql_pr const divy_rdbo_sql * 登録するSQL
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_insert_sql_property(request_rec *r,
						const divy_rdbo_sql *sql_pr);

/**
 * 指定されたsql_pr が表す内容でSQLを更新する。
 *
 * (note) 機能
 *   ・sql_pr が示すSQLのプロパティを変更
 *   ・sql_pr が持っていたRequiredSQL間の依存関係を変更
 *
 * update 対象のテーブル: divy_sql, divy_sqlcontent, divy_sqldepend
 *
 * (note) この関数を呼び出すには以下を呼び出し元にて保証して下さい。
 *
 *   ・active フラグのを非アクティブにする場合、sql_pr が示すSQLと
 *     別のSQLとの間に依存関係がないこと。
 *     (非アクティブ化によって、他のSQLが実行できなくなる状況を回避する)
 *
 *   ・SQLの種類(sqltype)をRequiredSQLから通常SQLに変更する時には、
 *     このRequiredSQLが他のSQLから利用されていないこと。
 *
 *   ・SQLの依存関係やSQL文を変えることで、依存関係の循環参照が起きないこと
 *
 * (note) sql_pr の必須項目
 *	sqlid, type, usedbmsname, relativeuri,
 *	subname(RequiredSQL,名前付きバインド変数), next※
 *
 *	※next には、更新前のsqlが持っていたsql_prへのポインタを繋げて下さい。
 * (note) sql_pr->sqlcnt_pr の必須項目
 *	id, name
 * (note) 以下のデータは更新できません。
 * 	(sql_pr) sqlid, relativeuri, registdt
 *	上記以外のデータは、指定されていた値をそのまま反映します。
 *
 * (note) ts_ctxがない場合は失敗します。(他の関数との違い！！)
 *
 * @param r request_rec *
 * @param sql_pr const divy_rdbo_sql * 更新するSQL
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_sql_property(request_rec *r,  
					const divy_rdbo_sql *sql_pr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたsql_pr が示す１つのSQLを削除する。
 * (note) 機能
 *   ・sql_pr が示すSQLの削除
 *   ・sql_pr->sqlcnt_pr が示すWHERE句プロパティの削除
 *   ・sql_pr を利用するグループから使用権限を没収
 *   ・sql_pr が持っていたRequiredSQL間の依存関係を消去
 *
 * delete 対象のテーブル： divy_sql, divy_sqlcontent, divy_sqldepend
 * update 対象のテーブル:  divy_sqlmem
 *
 * (note) この関数を呼び出すには以下を呼び出し元にて保証して下さい。
 *
 *   ・sql_pr が示すSQLと別のSQLとの間に依存関係がないこと
 *     (このSQL削除によって影響を受けないこと)
 *
 * @param r request_rec *
 * @param sql_pr const divy_rdbo_sql * 削除するSQL
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_sql_property(request_rec *r,
						const divy_rdbo_sql *sql_pr);

/**
 * 指定されたsrc_sql_pr が示すSQLをdst_sql_pr が示すSQLとして移動する。
 *
 * 更新対象テーブル:	divy_sql
 *
 * src_sql_pr の必須項目: relativeuri, sqlid
 * dst_sql_pr の必須項目: relativeuri, labelname
 *
 * 変更される情報:
 * 	relativeuri, labelname,	updatedt
 *	updatedt は、自動的に補完されます。
 *
 * 変更されない情報:
 * 	sqlid, registdt, sqlcnt_pr,
 * 	src_sql_pr を利用可能なグループの利用可能状態
 *
 *
 * @param r request_rec *
 * @param src_sql_pr const divy_rdbo_sql * 移動元SQL
 * @param dst_sql_pr const divy_rdbo_sql * 移動先SQL
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_move_sql_property(request_rec *r,
					const divy_rdbo_sql *src_sql_pr,
					const divy_rdbo_sql *dst_sql_pr);

/**
 * 指定されたサブ名称のリストsubname_listを使ってSQL文を取得する.
 *
 * select 対象テーブル: divy_sql
 *
 * @param r request_rec *
 * @param subname_list divy_rdbo_clist * サブ名称からなるリスト
 * @param sql_pr_hash apr_hash_t ** サブ名称をキーとしてdivy_rdbo_sqlを持つハッシュ
 * @param ts_ctx divy_db_transaction_ctx トランザクションコンテキスト
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_sql_by_subname(request_rec *r,
						divy_rdbo_clist *subname_list,
						apr_hash_t **sql_pr_hash,
						divy_db_transaction_ctx *ts_ctx);

/**
 * dbmsname が示すDBMS に対しSQL文sql を実行し、result_value に値をいれて返却する.
 * なお、isTopOnlyが1だった時には、実行結果セットの先頭の１件だけを取得します。
 * この関数はRequiredSQLの実行に使用します。通常SQLの実行には使用できません。
 *
 * (note) ts_ctx が持っているDbConnと同じDbConnを使用するときには、
 * 	トランザクションを引き継ぎます。
 *
 * @param r request_rec *
 * @param dbmsname char * DBMS 名称(DBプロバイダ名)
 * @param sql char * SQL文
 * @param bindvals divy_array_t * バインド変数集合
 * @param isTopOnly int 先頭の１件だけを取得するかどうか(1: １件だけ /0: 取れただけ)
 * @param result_value divy_rdbo_rsvalue ** 結果リスト
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 * 	0: 成功 / 1: 失敗
 * 	DIVY_STCODE_TOOMANY_COLUMN : SELECT句のカラム数が多かった
 * 	DIVY_STCODE_TOOFEW_COLUMN  : SELECT句のカラム数が少なかった
 * 	結果リストが取得出来なかったときには、処理ステータスは0で、
 * 	*result_value をNULLとして返却します。
 */
DIVY_DECLARE(int) divy_rdbo_exec_sql(request_rec *r,
						char *dbmsname,
						char *sql,
						divy_array_t *bindvals,
						int isTopOnly,
						divy_rdbo_rsvalue **result_value,
						divy_db_transaction_ctx *ts_ctx);

/**
 * あるsubname を持つrequiredsql、名前付きバインド変数が divy_sql に存在するか
 * 調べる
 *
 * (note)
 * アクティブなrequiredsql、名前付きバインド変数 のみdivy_sql から検索します。
 *
 * @param  r request_rec *
 * @param  apr_hash_t * チェック対象となるsubname のハッシュ
 * @param  apr_hash_t ** 存在しなかったsubname のハッシュ
 * @param  int 存在しなかったsubname を返却しない(0)/する(1)のフラグ
 * 		返却しない場合は存在しないsubname が見つかった時点で処理終了 
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 * 		0:成功、1:失敗、
 * 		DIVY_STCODE_REQUIRED_NF : 存在しないsubname があった
 */
DIVY_DECLARE(int) divy_rdbo_check_required(request_rec *r,
					apr_hash_t *subname_h,
					apr_hash_t **subname_nf_h,
					int retflg,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたサブ名称の集合subname_setを使ってSQLプロパティを取得する.
 *
 * select 対象テーブル: divy_sql
 * (note) 取得プロパティ
 * 	sqlid, subname, sql, active, type, usedbmsname	
 * (note)
 * 	継続中のトランザクションを引き継ぎます。
 *
 * @param r request_rec *
 * @param subname_set divy_cset_t * サブ名称文字列の文字列集合
 * @param sql_pr_h apr_hash_t ** 取得したSQLプロパティ(１件もなければNULL)
 * @param ts_ctx divy_db_transaction_ctx トランザクションコンテキスト
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_sql_properties_by_subnames(request_rec *r,
						divy_cset_t *subname_set,
						apr_hash_t **sql_pr_h,
						divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたid または subname の集合idsubname_hを使ってSQLラベル名称を取得する
 *
 * select 対象テーブル: divy_sql
 * (note)
 * 	継続中のトランザクションを引き継ぎます。
 *
 * @param r request_rec *
 * @param idsubname_h apr_hash_t * id または subname をキーとして持つハッシュ
 * @param labelname_h apr_hash_t ** SQLラベル名称(１件もなければNULL)
 * @param ts_ctx divy_db_transaction_ctx トランザクションコンテキスト
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_sql_labelname_by_idsubname(request_rec *r,
						apr_hash_t *idsubname_h,
						apr_hash_t **labelname_h,
						divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたid または subname の集合idsubname_h
 * かつ 指定されたsubname を sqlcontent にもつ
 * かつ 指定された値取得モード を sqlcontent にもつ
 * SQL 文の SQLラベル名称を取得する
 *
 * select 対象テーブル: divy_sql,divy_sqlcontent
 * (note)
 * 	継続中のトランザクションを引き継ぎます。
 * 	subname, 値取得モードは必ず指定してください。
 *
 * @param r request_rec *
 * @param idsubname_h apr_hash_t * id または subname をキーとして持つハッシュ
 * @param char * sqlcontent の subname 
 * @param divy_rdbo_sqlmode sqlcontent の 値取得モード
 * @param labelname_h apr_hash_t ** SQLラベル名称(１件もなければNULL)
 * @param ts_ctx divy_db_transaction_ctx トランザクションコンテキスト
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_sql_labelname_by_idsubname_sqlcsubnamemode
						(request_rec *r,
						apr_hash_t *idsubname_h,
						char *sqlcsubname,
						divy_rdbo_sqlmode sqlcdepmode,
						apr_hash_t **labelname_h,
						divy_db_transaction_ctx *ts_ctx);

/**
 * dbmsname のDBMS情報を使っているSQLの絶対URI一覧を取得して返却する。
 * 使用されていなければ*usingdbms_set がNULLになります。
 *
 * なお、DBMSを直接利用していたSQLに関してのみ、情報を返却することにします。
 * 間接的にこのDBMSを利用していたものについては特に報告しません。
 *
 * @param r request_rec *
 * @param dbmsname const char * DBMS 識別名称
 * @param usingdbms_set apr_hash_t ** このDBMSを直接利用していたSQL一覧
 */
DIVY_DECLARE(int) divy_rdbo_find_sql_usingdbms(request_rec *r,
						const char *dbmsname,
						apr_hash_t **usingdbms_set);

/**
 * 指定されたtop_sqlid が持っているSQLの依存関係リストを取得して返却する。
 * select 対象テーブル: divy_sqldepend
 *
 * (note)
 * 	topid_h に含まれる文字列がRequiredSQLまたは名前付きバインド変数を
 *	 表している時には、divy_sqlをJOINして取得します。
 * (note) top_sqlid_h がNULLの場合には、divy_sqldependテーブルに存在する全ての
 * 	  依存関係を取得します。
 * 	  top_sqlid_h が複数の値を持つ場合は全てのid の依存関係を取得します。
 * (note) divy_rdbo_sqldepend のtopid は設定されません。
 * (note) トランザクションを継続します。
 *
 * @param r request_rec *
 * @param topid_h apr_hash_t * トップSQL の id のハッシュ
 * @param sqldepend divy_rdbo_sqldepend ** SQL依存関係を持つリスト
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト 
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_sqldepend_list(request_rec *r,
					apr_hash_t *topid_h,
					divy_rdbo_sqldepend **sqldepend,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたnames(サブ名称またはSQLシーケンス番号)の配列を使って
 * divy_sqlテーブルを検索し、uriを取得する。
 *
 * (note) names について
 * 	names には、divy_sqldepend テーブルのsqld_pt_subname_vcまたは
 * 	sqld_cl_subname_vcカラムの値が指定されます。
 * 	このカラムには、サブ名称($$Bxxxx, $$SSxxxx, $$SMxxxx)とSQL
 * 	シーケンス番号(nnnnnnnn)が含まれているため、両者の混合を許して
 * 	います。
 * 	どちらに属するのかは、マクロIS_REQSQL_NODE によって決定します。
 *
 * @param r request_rec *
 * @param names const char ** サブ名称、SQLシーケンス番号からなる配列 (NULL終端)
 * @param sql_pr divy_rdbo_sql ** 取得した結果からなるリスト
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_sqluri_by_subname(request_rec *r,
							const char **names,
							divy_rdbo_sql **sql_pr);

#ifdef __cplusplus
}
#endif

#endif /* INCLUDE_TF_RDBO_SQL_H */

