/**
 * $Id$
 *
 * 特殊フォルダ「ユーザ」関連の関数をまとめたヘッダファイル
 * (note)
 *   ユーザはやグループやリソースとのとの癒着が多く、全てをtf_rdbo.h から
 *   移行出来た訳ではありません. 更新系の関数は殆どNGでした.ご注意下さい.
 */
#ifndef INCLUDE_TF_RDBO_USER_H
#define INCLUDE_TF_RDBO_USER_H

#include "apr.h"
#include "apr_pools.h"
#include "tf_rdbo_util.h"
#include "tf_rdbo.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare public functions 
  ----------------------------------------------------------------------------*/
/**
 * 指定されたparams, screen によりユーザ情報をリポジトリDBから検索して返却する
 * (userinformationsearch のcontent とdetaillist)
 *
 * (note)
 *   * divy_rdbo_userinformationsearch が複雑化しすぎたため分離.
 *   * Oracle 対応は止めました
 *
 * @param params divy_rdbo_search_params * 入力パラメータ
 * @param useris_iscreen divy_search_useris_iscreen * 入力値
 * @param output divy_rdbo_search_output * 出力結果へのポインタ(アロケート済み)
 * @return int 処理ステータス (0:成功、1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_userinformation_detaillist(divy_rdbo_search_params *params,
						divy_search_useris_iscreen *screen,
						divy_rdbo_search_output *output);

/**
 * 指定されたparams, screen によりユーザ情報をリポジトリDBから検索して返却する.
 * (userinformationsearch のcontent)
 *
 * (note)
 *   * divy_rdbo_userinformationsearch が複雑化しすぎたため分離.
 *   * Oracle 対応は止めました
 *
 * @param params divy_rdbo_search_params * 入力パラメータ
 * @param useris_iscreen divy_search_useris_iscreen * 入力値
 * @param output divy_rdbo_search_output * 出力結果へのポインタ(アロケート済み)
 * @return int 処理ステータス (0:成功、1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_userinformation_content(divy_rdbo_search_params *params,
						divy_search_useris_iscreen *screen,
						divy_rdbo_search_output *output);


/**
 * 指定されたparams, screen によりユーザのグループ所属情報
 * をリポジトリDBから検索して返却する(userinformationsearch の availablegroup)
 *
 * (note)
 *   * divy_rdbo_userinformationsearch が複雑化しすぎたため分離.
 *   * Oracle 対応は止めました
 *
 * @param params divy_rdbo_search_params * 入力パラメータ
 * @param useris_iscreen divy_search_useris_iscreen * 入力値
 * @param output divy_rdbo_search_output * 出力結果へのポインタ(アロケート済み)
 * @return int 処理ステータス (0:成功、1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_userinformation_availablegroup(divy_rdbo_search_params *params,
						divy_search_useris_iscreen *screen,
						divy_rdbo_search_output *output);

/**
 * 指定されたparams, screen によりユーザ情報をリポジトリDBから検索して返却する
 * (userinformationsearch のtreelist)
 *
 * (note)
 *   * divy_rdbo_userinformationsearch が複雑化しすぎたため分離.
 *
 * @param params divy_rdbo_search_params * 入力パラメータ
 * @param useris_iscreen divy_search_useris_iscreen * 入力値
 * @param output divy_rdbo_search_output * 出力結果へのポインタ(アロケート済み)
 * @return int 処理ステータス (0:成功、1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_userinformation_treelist(divy_rdbo_search_params *params,
						divy_search_useris_iscreen *screen,
						divy_rdbo_search_output *output);

/**
 * 統計情報 に表示する情報の取得
 * (note)
 *	statusinformationsearch で使用します。
 *	このSEARCH専用になっていますので、他の用途には使用してはなりません。
 * 
 * @param params divy_rdbo_search_params * 入力パラメータ
 * @param output divy_rdbo_search_output * 出力結果へのポインタ(アロケート済み)
 * @return int 処理ステータス (0:成功 / 1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_statusinformationsearch(divy_rdbo_search_params *params,
							divy_rdbo_search_output *output);

/**
 * 指定されたuserid が示すユーザ情報を取得して返却する。
 *
 * select 対象のテーブル: divy_usr, divy_usrdiskquota, dav_resource, divy_passpolicystatus
 * (note)
 *   * Oracle 対応は止めました
 *
 * @param r request_rec *
 * @param userid const char * ユーザIDを表す文字列
 * @param usr_pr divy_rdbo_usr ** 取得したユーザ情報
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_user_property(request_rec *r,
						const char *userid,
						divy_rdbo_usr **usr_pr);

/**
 * r->user (アクセスユーザ) のユーザ情報を取得してr->pool にキャッシュする. (auth.c 用)
 * なおこの値の取り出しは、util.c の対応する関数を使用してください。
 *
 * LDAPによりユーザ作成を行なえるようになっている設定の場合は
 * ユーザがTeamFileに存在しなかった値としてDIVY_STCODE_USER_NOT_EXIST
 * を返却する
 *
 * select対象テーブル: divy_usr, divy_usraccessdeny, divy_grp, divy_grpmem
 * 
 * @param r request_rec *
 * @param force int	キャッシュ値を使うかどうか (0: キャッシュ値優先 / 1: 強制取り直し)
 * @return int ( 0: 成功 / 1:失敗/ DIVY_STCODE_USER_NOT_EXIST: ユーザがいない)
 */
DIVY_DECLARE(int) divy_rdbo_cache_userinfo(request_rec *r, int force);

/**
 * 指定されたユーザのユーザQuotaを取得する。
 * usr_pr で指定されたユーザが存在しなければ、DIVY_STCODE_USER_NOT_EXISTを
 * 返却します。以下の例外を除き、DIVY_STCODE_USER_NOT_EXISTが返却される
 * ケースでは、トランザクションが分断されている場合、トランザクションは
 * commit されます。(rollbackではないことに注意)
 * (note)
 * 	do_entrylock フラグが立っていたらエントリをロックして返却します。
 * 	その際、ts_ctx がNULLであれば、エラーとなります。
 * 	また、DIVY_STCODE_USER_NOT_EXIST が返却されるような状況でも
 * 	エラーとなります。
 * (note)
 * 	デッドロックが起きないことは呼び出し側で保証して下さい。
 * 	この関数は一切感知しません。
 *
 * @param r request_rec *
 * @param usr_pr divy_rdbo_usr * 対象ユーザIDと取得したQuota 情報
 * @param do_entrylock int 1: ロックをかけてエントリを取得 / 0: ロックをかけない
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 * 	(0: 成功 / 1: 失敗 / DIVY_STCODE_USER_NOT_EXIST: ユーザが存在しない)
 */
DIVY_DECLARE(int) divy_rdbo_get_userquota(request_rec *r,
					divy_rdbo_usr *usr_pr,
					int do_entrylock,
			   		divy_db_transaction_ctx *ts_ctx);

/**
 * 現時点で全ユーザに割り当てられたQuota数の合計を取得する。
 *
 * select対象テーブル: divy_usrdiskquota
 *
 * @param r request_rec *
 * @param sum_uquota divy_rdbo_usr ** ユーザQuotaの合計
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_allocated_userquota(request_rec *r,
						divy_rdbo_usr **sum_uquota);

/**
 * 指定されたusr_pr が表すユーザを新規登録する。
 * ユーザが持っているプライベートコレクションも作成する。
 *
 * insert対象のテーブル:  divy_usr, divy_usrdiskquota, dav_resource, divy_usraccessdeny
 * (note) usr_pr の必須項目
 *	usrid, password, fullname
 *	registdt は自動的に補完します。
 *	上記以外のデータは、指定されていた値をそのまま反映します。
 *	プライベートコレクションは新規作成と同様に作ります。
 * (note)
 *	同じusrid のユーザは登録出来ません。
 *
 * (note) トランザクションが継続中であれば引き継ぎます
 *
 * @param r request_rec *
 * @param usr_pr const divy_rdbo_usr * 登録するユーザ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_insert_user_property(request_rec *r,
						const divy_rdbo_usr *usr_pr,
						divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたusr_pr が表す内容でユーザを更新する。
 *
 * update 対象テーブル:updateflgによる。
 *	divy_usr, divy_usrdiskquota, divy_usraccessdeny, dav_resource
 * (note) usr_pr 必須項目
 *	usrid, password, fullname, rsid
 *     updatedt は自動的に補完します。
 * [divy_usrdiskquota]
 *	maxst, maxres
 *	updatedt は自動的に補完します。
 * (note) 以下の値は更新出来ません。
 *	usrid, rsid, usrseq, usedst, registdt, usedres, lastaccess
 * (note)
 * 	ts_ctxがない場合は失敗します。
 *
 * @param r request_rec *
 * @param usr_pr const divy_rdbo_usr * 更新するユーザ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_user_property(request_rec *r, 
					const divy_rdbo_usr *usr_pr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたusr_pr が示すユーザを削除する。
 * ユーザが保持していたプライベートコレクション以下のリソースを全て削除します。
 * (note)
 * 	* 中身が残っていたら削除させない等の条件があるのなら
 * 	  それは呼び出し側で保証して下さい。
 * 	* トランザクションが継続中であればそれを引き継ぎます。
 *
 * usr_pr の必須項目: userid, rsid, prvcol_uri
 *
 * delete 対象テーブル:
 * 	divy_usr           : このユーザのエントリ
 * 	divy_usrdiskquota  : このユーザのエントリ
 * 	divy_usraccessdeny : このユーザのアクセス制限エントリ
 * 	divy_grpmem        : このユーザのエントリ
 * 	divy_mailwatch     : このユーザのmailwatch プロパティ
 * 	dav_resource       : このユーザのプライベートコレクション以下のリソース
 * 	dav_lock           : プライベートコレクション以下に残っていた不正ロック
 * 	divy_trash_info    : ごみ箱フォルダのコレクション
 *
 * @param r request_rec *
 * @param usr_pr const divy_rdbo_usr * 削除対象のユーザ
 * @param del_filelist divy_linkedlist_t ** プライベートコレクション以下に存在したリソース
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_user_property(request_rec *r,
					const divy_rdbo_usr *usr_pr,
					divy_linkedlist_t **del_filelist,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたsrc_usr_pr が示すユーザをdst_usr_pr が示すユーザとしてコピーする。
 * 基本的に、新しいユーザをsrc_usr_pr の情報を使って作るだけです。
 * src_usr_pr が持っていたリソース情報はコピーされません。
 *
 * 更新対象テーブル:
 *    divy_usr, divy_usrdiskquota, dav_resource, divy_grpmem, divy_usraccessdeny
 *
 * src_usr_pr の必須項目: usrid
 * dst_usr_pr の必須項目: usrid, extstatus
 *    registdt は自動的に補完します。
 *
 * コピーされる情報:
 *	password, fullname, mailaddr, adminmode, maxst, maxres
 *	src_usr_pr が所属していたグループへの所属状態
 *
 * コピーされない情報:
 *	usrid, rsid, usrseq, usedst, usedres,
 *	registdt, updatedt,
 *	src_usr_pr が持っていたプライベート・コレクション
 *
 * @param r request_rec *
 * @param src_usr_pr const divy_rdbo_usr * コピー元ユーザ
 * @param dst_usr_pr const divy_rdbo_usr * コピー先ユーザ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_copy_user_property(request_rec *r,
					const divy_rdbo_usr *src_usr_pr,
					const divy_rdbo_usr *dst_usr_pr);

/**
 * 指定されたsrc_usr_pr が示すユーザをdst_usr_pr が示すユーザとして移動する。
 * (note)
 * 	##### FIXME
 * 	この関数のコールは本質的にデッドロックの可能性を回避できません。
 * 	デッドロックしないようにするためには、移動対象ユーザの操作を
 * 	一旦全て禁止させなければなりませんが、このことが不可能であるため
 * 	です。
 * 	デッドロックが起きても検出できないDBMSを使用した場合、この
 * 	コードは破綻します。
 *
 * 更新対象テーブル:
 * 	divy_usr           : ユーザ情報
 * 	divy_usrdiskquota  : Quota 情報
 * 	divy_usraccessdeny : アクセス拒否リスト
 * 	dav_resource       : プライベートコレクションURI、更新者/作成者ID
 * 	dav_lock           : 認証ユーザ
 * 	divy_mailwatch     : メール監視ユーザのユーザID
 * 	divy_grpmem        : 所属ユーザのユーザID
 *
 * src_usr_pr の必須項目: usrid, rsid, prvcol_uri
 * dst_usr_pr の必須項目: usrid
 *
 * 変更される情報:
 * 	usrid, updatedt,
 * 	src_usr_pr が持っていたプライベート・コレクションの全uri と displayname, depth
 *	updatedt は、自動的に補完されます。
 *
 * 変更されない情報:(変更されない情報)
 *	rsid, usrseq, usedst, adminmode, usedst, maxst, usedres, maxres, registdt
 *
 * @param r request_rec *
 * @param src_usr_pr const divy_rdbo_usr * 移動元ユーザ
 * @param dst_usr_pr const divy_rdbo_usr * 移動先ユーザ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_move_user_property(request_rec *r,
					const divy_rdbo_usr *src_usr_pr,
					const divy_rdbo_usr *dst_usr_pr);

/**
 * ownerid が示すグループリーダが、自分自身の管理下に置かれた
 * ユーザを保持しているかどうか.
 *
 * @param r request_rec *
 * @param ownerid const char * オーナID
 * @param has_user int * ユーザを保持しているかどうか (1: 保持している / 0: 保持していない)
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_has_owner_users(request_rec *r, const char *ownerid, int *has_user);

/**
 * ownerid が示すグループリーダが管理しているユーザの人数を返却する
 * (note)
 *   Other ユーザは勘定に入れません.
 *
 * @param r request_rec *
 * @param ownerid const char * オーナID
 * @param nuser int * 抱えているユーザの数
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_count_owner_users(request_rec *r, const char *ownerid, int *nuser);

/**
 * userid が示すユーザのオーナIDを指定されたownerid に変更する.
 *
 * (note)
 *    * 外部仕様的な意味では、
 *      Other ユーザuserid をownerid が示すユーザの管理下に置くことに相当します.
 *
 *    * ownerid がNULLの場合も許容します(即ちオーナなし)
 *
 * @param r request_rec *
 * @param userid const char * Other ユーザのユーザID
 * @param ownerid const char * オーナのユーザID
 * @param ts_ctx divy_db_transaction_ctx *
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_update_user_ownerinfo(request_rec *r, const char *userid,
							const char *ownerid, divy_db_transaction_ctx *ts_ctx);

/**
 * mail addressが該当するユーザ一覧をすべて取得する。主にSAML向け
 *
 * @param r request_rec *
 * @param mailaddr const char* メールアドレス
 * @param usr_pr divy_rdbo_usr **	ユーザ構造体
 * @param count	int * 該当ユーザ数
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_get_userlist_from_mail(request_rec *r, const char* mailaddr, divy_rdbo_usr **usr_pr, int *count);

#ifdef __cplusplus
}
#endif

#endif /* INCLUDE_TF_RDBO_USER_H */

