/**
 * $Id$
 *
 * tf_rdbo には分類できなかった関数を集めたヘッダファイル.
 * このヘッダファイルは、tf_rdbo.h に依存してはなりません. 互いに必要と
 * し合ってDead lock します.
 * (note)
 *   tf_rdbo.c/tf_rdbo.h は今や当初の目的から大きく逸れてただひたすら大きな
 *   "SQLを書いておくソース" になってしまいました。
 *   もはや手の施しようの無いほど。
 *   そうであるのなら、せめて、ORマッピングの役割の果たすものだけをtf_rdbo に
 *   入れておき、それ以外のものは何処かに追いやってしまおうと考えて
 *   このヘッダを定義しました。
 *   これを見た誰かに改善を託します。
 */
#ifndef INCLUDE_TF_RDBO_UTIL_H
#define INCLUDE_TF_RDBO_UTIL_H

#include "apr.h"
#include "apr_pools.h"
#include "apr_time.h"
#include "apr_hash.h"
#include "httpd.h"

#include "tfr.h"
#include "tf_db.h"
#include "util_db.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Fixed values
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structures
  ----------------------------------------------------------------------------*/
typedef enum   __divy_rdbo_jointype		divy_rdbo_jointype;
typedef struct __divy_rdbo_tableinfo	divy_rdbo_tableinfo;
typedef struct __divy_rdbo_extstatus	divy_rdbo_extstatus;	/* 不完全型 */

/*
 * 拡張ステータスの種類を表す列挙型
 */
typedef enum __divy_rdbo_extstatus_type	divy_rdbo_extstatus_type;
enum __divy_rdbo_extstatus_type {
	EXTSTATUS_TYPE_USR = 0,	/* ユーザ拡張ステータスを表す */
	EXTSTATUS_TYPE_GRP,		/* グループ拡張ステータス(グループ制約属性)を表す */
};

/**
 * basicsearch で各種テーブルと結合する際に使用する結合の種類を表す値。
 * (note)
 * 	以下で定義されたタイプ以外の結合をサポートするには、ロジックの
 * 	修正が必要となります。
 */
enum __divy_rdbo_jointype {
	DIVY_JOIN_NONE = 0,	/* 結合しない */
	DIVY_JOIN_INNER,	/* 内部結合   */
	DIVY_JOIN_LEFT,		/* 左外部結合 */
};

/**
 * あるカラムをJOINする際に必要となるデータベースオブジェクトの情報と
 * JOINで結合するキーに関する情報を持つ構造体 (basicsearch専用)
 */
struct __divy_rdbo_tableinfo {
	int propid;		/* このカラムが表すLiveプロパティのプロパティID */
	char *colname;		/* テーブルのカラム名                     */
	char *tablename;	/* coloumnameのカラムを持つテーブルの名前 */
	char *refname;		/* tablenameに付けられた別名(SQL文に使用する) */
	divy_rdbo_jointype jointype;	/* 結合の種類 */
	char *l_joinkey;	/* jointype 結合のキー(DIVY_JOIN_NONE ならばNULL) */
	char *r_joinkey;	/* jointype 結合のキー(DIVY_JOIN_NONE ならばNULL) */
};

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 *  リポジトリDB からリソースIDを採番して取得する。
 *  (note)
 *    ts_ctx のトランザクションを継続します.
 * 
 * @param r request_rec *
 * @param char ** 取得したリソースID文字列. 処理に失敗したらNULL
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト(NULLで自動生成)
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_create_rsid(request_rec *r, char **rsid, divy_db_transaction_ctx *ts_ctx);

/**
 * リソースID・グループIDを採番号して返却する。
 *
 * (note)
 *	* tf_rdbo_group.h のpublic 化に伴いtf_rdbo.c から移動しました.
 *	* トランザクションが継続中であれば引き継ぎます。
 *
 * @param r request_rec *
 * @param rsid char ** 採番したリソースID
 * @param grpid char ** 採番したグループID
 * @param ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理結果 (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_create_rsgrpid(request_rec *r, char **rsid, char **grpid,
											divy_db_transaction_ctx *ts_ctx);

/**
 * 指定された文字列 str からエスケープされるワイルドカードを探し、
 * dbに対応した形に変換して返却する。
 * 
 * (note) 現在は以下のように変換します。(postgres用)
 *	%（パーセント) -> \%
 *	_ (アンダバー) -> \_
 *
 * @param p apr_pool_t *
 * @param str const char * 変換元文字列
 * @return 変換後の文字列
 */
DIVY_DECLARE(char *) divy_rdbo_escape_wildcard(apr_pool_t *p, const char *str);

/**
 * 指定されたdbconn がアクティブなコネクションであることを検証する。
 * (note)
 * 	この関数は、DBコネクション管理マネージャが使用します。
 * 	特殊な考慮が入っていますので、他からは使用しないで下さい。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param dbconn DbConn * テストするコネクション
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_validate_activedbconn(apr_pool_t *p, DbConn *dbconn);

/**
 * 指定されたプロパティ名の取得元となっているテーブル名とカラム名を
 * 構造体divy_rdbo_tableinfoとして返却する
 *
 * @param r request_rec * request_rec構造体へのポインタ
 * @param propname const char * プロパティ名
 * @return divy_rdbo_tableinfo*
 */
DIVY_DECLARE(divy_rdbo_tableinfo *)divy_rdbo_get_tableinfo(request_rec *r, 
                                  	           const char *propname);

/**
 * ログインしているユーザは有効期限切れか?
 *
 * @param r request_rec *
 * @return int 有効期限切れかどうか(1: 期限切れ / 0: 有効期限内
 */
DIVY_DECLARE(int) divy_rdbo_is_expired_user(request_rec *r);

/**
 * type が示す拡張ステータスを表す文字列 extstatus_str 値が正しい状態に
 * なっているかどうか検証する。
 *
 * @param p apr_pool_t * リクエストプール (ログ出力のため)
 * @param extstatus_str const char * ユーザ拡張ステータスからなる文字列
 * @param type divy_rdbo_extstatus_type 拡張ステータスの種類
 * @return int 正しい状態かどうか (1: 不正 / 0: 正しい)
 */
DIVY_DECLARE(int) divy_rdbo_validate_extstatus(apr_pool_t *p,
						const char *extstatus_str, divy_rdbo_extstatus_type type);

/**
 * type が示す拡張ステータスを表す文字列 extstatus_str をパースする。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param extstatus_str const char * ユーザ拡張ステータスを表す文字列
 * @param type divy_rdbo_extstatus_type 拡張ステータスの種類
 * @return divy_rdbo_extstatus * パース結果を保持する構造体へのポインタ
 */
DIVY_DECLARE(divy_rdbo_extstatus *) divy_rdbo_parse_extstatus(
					apr_pool_t *p, const char *extstatus_str, divy_rdbo_extstatus_type type);

/**
 * デフォルトのtype に応じた拡張ステータスを取得する。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param type divy_rdbo_extstatus_type 拡張ステータスの種類
 * @return divy_rdbo_extstatus * デフォルトの拡張ステータスへのポインタ
 */
DIVY_DECLARE(divy_rdbo_extstatus *) divy_rdbo_create_default_extstatus(apr_pool_t *p, divy_rdbo_extstatus_type type);

/**
 * type が示す拡張ステータスの文字列表現を取得する。
 *
 * @param extstatus const divy_rdbo_extstatus * type が示す拡張ステータスへのポインタ
 * @param type divy_rdbo_extstatus_type 拡張ステータスの種類
 * @return char * ユーザ拡張ステータスの文字列表現。この値は、テーブルに直接
 * 	INSERT/UPDATE してもよい値になっています。
 */
DIVY_DECLARE(char *) divy_rdbo_extstatus2str(const divy_rdbo_extstatus *extstatus, divy_rdbo_extstatus_type type);

/**
 * 指定されたtype の拡張ステータスextstatus に含まれるステータスのフラグ値を
 * 全てリセット(非アクティブ)にする.
 * 権限フラグは全て権限なしの状態にセットします.
 *
 * @param extstatus divy_rdbo_extstatus * type の拡張ステータスへのポインタ
 * @param type divy_rdbo_extstatus_type 拡張ステータスの種類
 */
DIVY_DECLARE(void) divy_rdbo_reset_extstatus(divy_rdbo_extstatus *extstatus, divy_rdbo_extstatus_type type);

/**
 * type の示す拡張ステータスのワイルドカード文字列表現を取得する。
 * ワイルドカード文字列になるのはアクティブでないステータスだけです。
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへのポインタ
 * @param type divy_rdbo_extstatus_type 拡張ステータスの種類
 * @return char * ユーザ拡張ステータスのワイルドカード文字列表現。
 * 	この値は、テーブルにINSERT/UPDATE してならない。LIKE 句の右辺値に利用できるだけです。
 */
DIVY_DECLARE(char *) divy_rdbo_extstatus2wildcardstr(const divy_rdbo_extstatus *extstatus, divy_rdbo_extstatus_type type);

/**
 * 指定されたユーザ拡張ステータスextstatus に含まれるステータスの内、ユーザ権限フラグだけを
 * リセットする. 全てをリセットするには、divy_rdbo_reset_extstatus を使用して下さい.
 *
 * @param extstatus divy_rdbo_extstatus * ユーザ拡張ステータスへのポインタ
 */
DIVY_DECLARE(void) divy_rdbo_reset_userprivilege(divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスにアクティブフラグが立っているかどうか
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @return int	アクティブかどうか(1: アクティブ / 0: アクティブではない)
 */
DIVY_DECLARE(int) divy_rdbo_is_active_user(const divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスにアクティブフラグを立てる
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_active_user(divy_rdbo_extstatus *extstatus, int on);

/**
 * ユーザ拡張ステータスに信頼ユーザフラグが立っているかどうか
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @return int	信頼されているかどうか(1: 信頼されている / 0: 信頼されていない)
 */
DIVY_DECLARE(int) divy_rdbo_is_trusty_user(const divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスに信頼ユーザフラグを立てる
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_trusty_user(divy_rdbo_extstatus *extstatus, int on);

/**
 * ユーザ拡張ステータスにグループリーダフラグが立っているかどうか.
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @return int	グループリーダかどうか(1: グループリーダである / 0: それ以外である)
 */
DIVY_DECLARE(int) divy_rdbo_is_groupleader(const divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスにグループリーダフラグを立てる
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_groupleader(divy_rdbo_extstatus *extstatus, int on);

/**
 * ユーザ拡張ステータスにread 権限が付いているかどうか
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @return int	read 権限を持つかどうか(1: 持つ  / 0: 持たない)
 */
DIVY_DECLARE(int) divy_rdbo_has_read_privilege(const divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスにread 権限をつける
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_read_privilege(divy_rdbo_extstatus *extstatus, int on);

/**
 * ユーザ拡張ステータスにupload 権限が付いているかどうか
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @return int	upload 権限を持つかどうか(1: 持つ  / 0: 持たない)
 */
DIVY_DECLARE(int) divy_rdbo_has_upload_privilege(const divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスにupload 権限をつける
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_upload_privilege(divy_rdbo_extstatus *extstatus, int on);

/**
 * ユーザ拡張ステータスにreadwrite 権限が付いているかどうか
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @return int	readwrite 権限を持つかどうか(1: 持つ  / 0: 持たない)
 */
DIVY_DECLARE(int) divy_rdbo_has_readwrite_privilege(const divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスにreadwrite 権限をつける
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_readwrite_privilege(divy_rdbo_extstatus *extstatus, int on);

/**
 * ユーザ拡張ステータスにview属性設定 権限が付いているかどうか
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @return int	view属性設定 権限を持つかどうか(1: 持つ  / 0: 持たない)
 */
DIVY_DECLARE(int) divy_rdbo_has_setview_privilege(const divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスにview属性設定 権限をつける
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_setview_privilege(divy_rdbo_extstatus *extstatus, int on);

/**
 * ユーザ拡張ステータスにシステム実行権限が付いているかどうか
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @return int	システム実行権限を持つかどうか(1: 持つ  / 0: 持たない)
 */
DIVY_DECLARE(int) divy_rdbo_has_sysexec_privilege(const divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスにシステム実行権限を付ける
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_sysexec_privilege(divy_rdbo_extstatus *extstatus, int on);

/**
 * ユーザ拡張ステータスにグループ制約属性「無視」するが付いているかどうか.
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @return int	グループ制約の適用を無視する(1) / 無視しない(0)
 */
DIVY_DECLARE(int) divy_rdbo_has_user_groupconstraints_ignore(const divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスにグループ制約属性の「無視」状態を設定する.
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_user_groupconstraints_ignore(divy_rdbo_extstatus *extstatus, int on);

/**
 * ユーザ拡張ステータスに「Otherユーザを管理下に置くことができる」が付いているかどうか.
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @return int	Otherユーザを管理下に置くことができる(1) / できない(0)
 */
DIVY_DECLARE(int) divy_rdbo_has_control_otheruser(const divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスに「Otherユーザを管理下に置くことができる」状態を設定する.
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_control_otheruser(divy_rdbo_extstatus *extstatus, int on);

/**
 * ユーザ拡張ステータスに「アカウントがロックされた」がついているかどうか
 *
 * @param extstatus  divy_rdbo_extstatus
 * @return int	アカウントがロックアウトされている(1) / されていない(0)
 */
DIVY_DECLARE(int) divy_rdbo_has_lockout(const divy_rdbo_extstatus *extstatus);

/**
 * ユーザ拡張ステータスに「アカウントがロックされた」を設定する
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_lockout(divy_rdbo_extstatus *extstatus, int on);

/**
 * グループ制約に書き込み制約がついているかどうか.
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @return int 付いている(1) / 付いていない(0)
 */
DIVY_DECLARE(int) divy_rdbo_has_write_groupconstraints(const divy_rdbo_extstatus *extstatus);

/**
 * r を持つアクセスユーザが、指定されたuri, infotype のリソース・コレクションに対し
 * ユーザのグループ制約適用の有無やグループの書き込み制約を総合的に評価した上で
 * 書き込み制限が掛かっているかどうか判定する. (ユーティリティ関数)
 *
 * (note)
 *   * ユーザ拡張ステータスとグループ制約を総合的に判断します.
 *   * ただし、ユーザのread-only や upload-only は評価しません.
 *   * グループ単体の書き込み制約の有無は関数divy_rdbo_has_write_groupconstraints を使います.
 *   * アクセスユーザという観点で評価する点に注意. 他の関数群とは趣が違います.
 *
 * @param r request_rec *
 * @param uri const char * 調べるURI
 * @param infotype divy_infotype uriのタイプ
 * @return int 1: 書き込み制約あり / 0: 書き込み制約なし
 */
DIVY_DECLARE(int) divy_rdbo_has_write_constraints_on_uri(request_rec *r, const char *uri, divy_infotype infotype);

/**
 * グループ制約に書き込み制約を設定する
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_write_groupconstraints(divy_rdbo_extstatus *extstatus, int on);

/**
 * グループ制約に操作ログ制約がついているかどうか.
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @return int 付いている(1) / 付いていない(0)
 */
DIVY_DECLARE(int) divy_rdbo_has_oplog_groupconstraints(const divy_rdbo_extstatus *extstatus);

/**
 * グループ制約に操作ログ制約を設定する
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_oplog_groupconstraints(divy_rdbo_extstatus *extstatus, int on);

/**
 * グループ制約にプロパティ表示制約がついているかどうか.
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @return int 付いている(1) / 付いていない(0)
 */
DIVY_DECLARE(int) divy_rdbo_has_showprop_groupconstraints(const divy_rdbo_extstatus *extstatus);

/**
 * グループ制約にプロパティ表示制約を設定する
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_showprop_groupconstraints(divy_rdbo_extstatus *extstatus, int on);

/**
 * グループ拡張ステータスにアクティブフラグが立っているかどうか.
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @return int	アクティブかどうか(1: アクティブ / 0: アクティブではない)
 */
DIVY_DECLARE(int) divy_rdbo_is_active_group(const divy_rdbo_extstatus *extstatus);

/**
 * グループ拡張ステータスにアクティブフラグをセットする.
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_active_group(divy_rdbo_extstatus *extstatus, int on);

/**
 * グループ拡張ステータスにBOX機能フラグが立っているかどうか.
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @return int	BOX機能が有効かどうか(1: 有効 / 0: 無効)
 */
DIVY_DECLARE(int) divy_rdbo_is_box_group(const divy_rdbo_extstatus *extstatus);

/**
 * グループ拡張ステータスにBOX有効フラグをセットする.
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_box_group(divy_rdbo_extstatus *extstatus, int on);

/**
 * グループ拡張ステータスにアップロードポリシーフラグが立っているかどうか
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @return int	アップロードポリシー機能が有効かどうか(1: 有効 / 0: 無効)
 */
DIVY_DECLARE(int) divy_rdbo_is_uploadpolicy_group(const divy_rdbo_extstatus *extstatus);

/**
 * グループ拡張ステータスにアップロードポリシーフラグをセットする
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_uploadpolicy_group(divy_rdbo_extstatus *extstatus, int on);

/**
 * r が示すアクセスユーザが管理リソースにアクセスできる権限を持っているかどうか.
 * [ アクセスできる条件 ]
 *   * 管理者であること or
 *   * グループリーダであること
 *
 * @param r request_rec *
 * @return int 処理ステータス (1: 権限を持っている / 0: 権限を持っていない)
 */
DIVY_DECLARE(int) divy_rdbo_has_administrative_privilege(request_rec *r);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_RDBO_UTIL_H */

