/**
 * $Id$
 *
 * tf_sqlparser.h
 *
 * SQLのパースを行う関数群のヘッダファイル
 *
 */
#ifndef INCLUDE_TF_SQLPARSER_H
#define INCLUDE_TF_SQLPARSER_H

#include "httpd.h"
#include "apr.h"
#include "apr_strings.h"
#include "apr_pools.h"
#include "apr_tables.h"
#include "apr_hash.h"

#include "mod_dav_tf.h"
#include "util.h"
#include "tf_rdbo.h"
#include "util_db.h"
#include "tf_array.h"
#include "search.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*--------------------------------------------------------------
  Define fixed values and macro
  --------------------------------------------------------------*/
/*
 * グローバル変数の定義と宣言の調整を行うためのdefine 値を定義
 * (see mod_dav_tf.h)
 *
 * SQLP_GLOBAL_VALUE_DEFINE の定義は、mod_dav_tfモジュールでは
 * tf_sqlparser.c で、それ以外のスタンドアローン型のプログラムでは
 * main関数があるファイルで行ってください。
 * SQLP_GLOBAL_VALUE_DEFINE の定義は"#include "tf_sqlparser.h" を
 * 記述する前に行う必要があります。
 * 複数のファイルで定義することは禁止です。(コンパイルエラーになります。)
 */
#ifdef SQLP_GLOBAL_VALUE_DEFINE
#define SQLP_GLOBAL
#else
#define SQLP_GLOBAL extern
#endif  /* SQLP_GLOBAL_VALUE_DEFINE */

/**
 * 処理ステータス
 * (note)
 * 	内部でしか使用されないステータスもあります
 */
#define DIVY_SQLP_ST_OK                0	/* 成功 */
#define DIVY_SQLP_ST_ERR               1	/* 予期しないエラー */ 
#define DIVY_SQLP_ST_UNSUPPORT         2	/* 解析がサポートされていないSQLが指定された */
#define DIVY_SQLP_ST_INVALID           3	/* SQL構文に誤りがある可能性が高い */
#define DIVY_SQLP_ST_WRONG_LVAL        4	/* 不正な左辺値(Lvalue)だった */
#define DIVY_SQLP_ST_WRONG_OPER        5	/* 不正なSQL演算子が指定されていた */
#define DIVY_SQLP_ST_TOOMANY_REQSQL    6	/* 右辺値に指定されたRequiredSQLの個数が多い */
#define DIVY_SQLP_ST_WRONG_M_PREF      7	/* $$SMxxx が誤った場所で使用されていた */
#define DIVY_SQLP_ST_NOTHING_TODO      8	/* 何もしなかった */
#define DIVY_SQLP_ST_LOOP_DETECTED     9	/* 依存グラフのループを検出した */
#define DIVY_SQLP_ST_USINGSQL_FOUND   10	/* あるSQLを利用するSQLが存在した */
#define DIVY_SQLP_ST_WRONG_RELATION   11	/* 依存グラフの親子関係がおかしい */
#define DIVY_SQLP_ST_EXEC_FAILED      12	/* 依存SQLのどれかの実行に失敗した */
#define DIVY_SQLP_ST_MULTIPLE_TOP     13	/* トップノードが複数個存在した */
#define DIVY_SQLP_ST_NOTFOUND         14	/* 見つからなかった */
#define DIVY_SQLP_ST_FOUND_NBIND      15	/* (廃止) 名前付きバインド変数の構文だった */
#define DIVY_SQLP_ST_WRONG_REQSQLNUM  16	/* SQL文中のRequiredSQLと置換値集合の数が不一致 */
#define DIVY_SQLP_ST_MISMATCH_ID      17	/* RequiredSQL出現位置が不正 */
#define DIVY_SQLP_ST_MISSING_ID       18	/* 置換値集合の中のid が歯抜けだった */
#define DIVY_SQLP_ST_WRONG_BINDVAL    19	/* (廃止) 不正なBINDVAL構文だった */
#define DIVY_SQLP_ST_NONSELECT        20	/* "SELECT" がない */
#define DIVY_SQLP_ST_NONFROM	      21	/* "FROM" がない */
#define DIVY_SQLP_ST_WRONG_NBIND_VAL  22	/* (廃止) BINDVAL指定値がクオートで囲まれていない */
#define DIVY_SQLP_ST_WRONG_NBIND_KEY  23	/* (廃止) BINDVAL構文に名前付きバインド変数がない */
#define DIVY_SQLP_ST_WRONG_NBIND_OP   24	/* (廃止) BINDVAL構文のキーと値を結ぶ演算子が不正 */
#define DIVY_SQLP_ST_MISSING_SQLCNT   25	/* (廃止) content が必要なSQLなのに content がない */
#define DIVY_SQLP_ST_MISSING_SQLDIS   26	/* (廃止) sqldiscovery がない */
#define DIVY_SQLP_ST_MISSING_SQLNAME  27	/* (廃止) labelname がない */
#define DIVY_SQLP_ST_MISMATCH_NAMEURI 28	/* (廃止) labelname と uri の名称部が一致しない */
#define DIVY_SQLP_ST_MISSING_SQLSTMT  29	/* sqlstatement がない */
#define DIVY_SQLP_ST_MISSING_REQSUB   30	/* RequiredSQL,名前付きバインド変数 で subname がない */
#define DIVY_SQLP_ST_WRONG_SUBNAME    31	/* subname の形式が正しくない */
#define DIVY_SQLP_ST_MISMATCH_REQURI  32	/* 同一subname のRequiredSQL,名前付きバインド変数がDBに存在するがURIが一致しない */
#define DIVY_SQLP_ST_REQSQL_NF	      33	/* SQL文中のRequiredSQL,名前付きバインド変数がDBに存在しない */
#define DIVY_SQLP_ST_CONFLICT_SQLCNT  34	/* SQL文のWHEREプロパティとkeyvalue エレメントの内容に矛盾 */
#define DIVY_SQLP_ST_WRONG_BIND_POS   35	/* 通常バインド変数が指定不可能な箇所に指定された */
#define DIVY_SQLP_ST_UNQUOTED_STR     36	/* SQL文中に閉じられていないシングルクオートがある */
#define DIVY_SQLP_ST_MISMATCH_SQLTYPE 37	/* sqltype とSQL文の種類が一致しない */
#define DIVY_SQLP_ST_NOEFFECTIVE_BODY 38	/* BINDVAL構文に意味の無いWHERE句プロパティが指定された */
#define DIVY_SQLP_ST_OVER_MAX_CHAIN   39	/* 依存SQLの階層数が最大数を超えた */
#define DIVY_SQLP_ST_MUSTCOL_NF       40	/* MUST のカラムがselect句に含まれていない */
#define DIVY_SQLP_ST_SQLTYPE_UNKNOWN  41	/* sqltype が指定されていないor未知のものだった */
#define DIVY_SQLP_ST_IERR	      42	/* INTERNAL SERVER ERROR (リクエスト修正では対応できないエラー) */
#define DIVY_SQLP_ST_TOOMANY_COLUMN   43	/* SELECT句に指定されたカラム数が多い */
#define DIVY_SQLP_ST_TOOFEW_COLUMN    44	/* SELECT句に指定されたカラム数が少ない */
#define DIVY_SQLP_ST_FOUND_NBIND_VAL  45	/* 名前付きバインド変数に値が設定されていた */
#define DIVY_SQLP_ST_NF_NBIND_VAL     46	/* 名前付きバインド変数に値が設定されていない */
#define DIVY_SQLP_ST_FOUND_UPDATESTMT 47	/* 更新系DML、DDL、DCLが指定された */
#define DIVY_SQLP_ST_UNKNOWN_STMT     48	/* 理解出来ないSQL構文・識別子が指定された */
#define DIVY_SQLP_ST_UNQUOTED_DQSTR   49	/* ダブルクオート文字が正しく閉じられていない */
#define DIVY_SQLP_ST_FOUND_ASTERISK   50	/* SELECT句に'*'が指定された */
#define DIVY_SQLP_ST_UNQUOTED_CTSTR   51	/* SQL文中に閉じられていないSQLコメントがある */
#define DIVY_SQLP_ST_MISSING_BINDVAL  52	/* (廃止) BINDVAL識別子が指定されていない */
#define DIVY_SQLP_ST_MISSING_STIDENT  53	/* (廃止) SQL構文の開始に必要な識別子がない */
#define DIVY_SQLP_ST_USELESS_SUBNAME  54	/* 通常SQLやリポジトリSQLにサブ名称が指定された */
#define DIVY_SQLP_ST_CACHE_NONBIND    55	/* 名前付きバインド変数以外に「キャッシュする」が指定された */
#define DIVY_SQLP_ST_CACHE_NBIND_VAL  56	/* 値のある名前付きバインド変数に「キャッシュする」が指定された */
#define DIVY_SQLP_ST_WRONG_COMB_TYPE  57	/* sqltype とサブ名称の組み合わせがおかしい */
#define DIVY_SQLP_ST_EMPTY_HIDDEN_NBIND 58	/* 他のSQLからHIDDENで利用されている名前付きバインド変数のデフォルト値を空にした */
#define DIVY_SQLP_ST_NEWSQL_LABELNAME 59	/* 新規登録で同じlabelnameが既に存在する */
#define DIVY_SQLP_ST_OLDSQL_LABELNAME 60	/* 更新でで同じlabelnameが存在しない */
#define DIVY_SQLP_ST_RESERVED_SQL     61	/* 予約されたSQLである */
#define DIVY_SQLP_ST_UNRESERVED_SQL   62	/* 予約されていないSQLであった */
#define DIVY_SQLP_ST_LACKOF_LICENSE   63	/* DB機能ライセンス不足のため"通常SQL"は利用不可 */

/*
 * 関数divy_sql_parser_get_sqlmodelist で使用するフラグ値
 */
#define DIVY_SQL_NO_DEFAULT_VAL  0	/* デフォルト値を持たない */
#define DIVY_SQL_HAS_DEFAULT_VAL 1	/* デフォルト値を持つ     */

/**
 * RequiredSQL, 名前付きバインド変数かどうかを判定するマクロ
 * @return int 1: 真 / 0: 偽
 */
#define IS_REQSQL_NODE(s)	(s != NULL && strlen(s) > 3 && \
				 s[0] == '$' && s[1] == '$' && \
			 	 ((s[2] == 'S' && s[3] == 'S') || \
				  (s[2] == 'S' && s[3] == 'M') || \
				  (s[2] == 'B')))
/**
 * $$SSxxx のRequiredSQLかどうかを判定するマクロ
 * @return int 1: 真 / 0: 偽
 */
#define IS_S_REQSQL_NODE(s)	(s != NULL && strlen(s) > 4 && \
				 s[0] == '$' && s[1] == '$' && \
			 	 s[2] == 'S' && s[3] == 'S')

/**
 * $$SMxxx のRequiredSQLかどうかを判定するマクロ
 * @return int 1: 真 / 0: 偽
 */
#define IS_M_REQSQL_NODE(s)	(s != NULL && strlen(s) > 4 && \
				 s[0] == '$' && s[1] == '$' && \
			 	 s[2] == 'S' && s[3] == 'M')

/**
 * 名前付きバインド変数($$Bxxx) かどうかを判定するマクロ
 * @return int 1: 真 / 0: 偽
 */
#define IS_NBIND_NODE(s)	(s != NULL && strlen(s) > 3 && \
				 s[0] == '$' && s[1] == '$' && s[2] == 'B')


/*--------------------------------------------------------------
  Declare structures and enum
  --------------------------------------------------------------*/
typedef struct __divy_sql_parser	divy_sql_parser;
typedef struct __divy_sql_parser_ctx	divy_sql_parser_ctx;
typedef struct __divy_sql_parser_err	divy_sql_parser_err;

typedef enum   __divy_sql_nodetype	divy_sql_nodetype;
typedef struct __divy_sql_elemnode	divy_sql_elemnode;

typedef struct __divy_sql_dependnode	divy_sql_dependnode;
typedef struct __divy_sql_dependnodeList divy_sql_dependnodeList;

/*--------------------------------------------------------------
  Define structures and enum
  --------------------------------------------------------------*/
/**
 * SQL parser を表す構造体。(内部用)
 *
 * (note)
 *   この構造体は、parser が作成された時に生成され、
 *   SQLのパース処理において、一時的に状態を記録したりする
 *   ために使用されます。
 * (note)
 * 	内部処理にのみ使います。外からも見えますが使わないように。
 */
struct __divy_sql_parser {
	/* 外部パーサ用コンテキスト */
	divy_sql_parser_ctx *p_ctx;

	apr_pool_t *p;	/* 作業用のプール */
	request_rec *r;	/* 作業用のRequestレコード */
	divy_db_transaction_ctx *ts_ctx;	/* 作業用 */
};

/**
 * 構文解析・字句解析パーサと値のやりとりをする際に使用する
 * コンテキスト構造体 (内部用)
 */
struct __divy_sql_parser_ctx {

	apr_pool_t *p;	/* 作業用のプール */

	/* 解析対象のSQL文 */
	const char *str;

	/*
	 * 以下は構文解析パーサ(yacc)で使用する変数
	 */
	int errcd;			/* SQL解析エラーコード */
	int select_colcnt;		/* SELECTカラムの個数 */
	int max_select_colcnt;		/* SELECT句が複数ある場合に使用するひとつ前のカウント */
	int select_id;			/* SELECT句識別ID(副問い合わせSQL毎にインクリメント) */
	divy_sql_elemnode *sqlenode;	/* SQL解析結果へのポインタ */

	int sqlposition;		/* SQL文中の位置を記録する一時領域 */

	/* 以下 字句解析パーサ(lex)で使用する変数 */
	/* 
	 * lex scanner に渡す解析対象の文字列へのポインタ
	 * (sqlとは字句解析パーサ内での扱いが異なります)
	 */
	char *scanbuf;

	/*
	 * 字句解析パーサのパース開始条件(%x と対応)
	 */
#define SCANNER_START_MODE_NORMAL  0	/* INITIAL で始まる開始条件でスタート */
#define SCANNER_START_MODE_REPLACE 1	/* xrp で始まる開始条件でスタート */
	int xstart;

	/*
	 * 以下は値置き換え処理において使用する変数
	 */

	/* RequiredSQL、名前付きバインド変数置き換え処理で置き換えられた
	 * 結果を格納する変数 */
	divy_sbuf *new_sql_buf;

	/* RequiredSQL、名前付きバインド変数を先頭から順番に格納する配列 */
	divy_array_t *bindvals;

	/* バインド変数を格納するリスト */
	divy_search_ldbs_bind *bvalue;

	/* RequiredSQL、名前付きバインド変数のSQL文中における出現位置(カウント) */
	int ipos;

	/* バインド変数のSQL文中における出現位置(カウント) */
	int ibpos;

	/* 置換対象のRequiredSQLの名称と置換値を保持するリスト */
	divy_rdbo_rsvalue *reqsql_value;

	/* identifier が示すキーワードの種類を保持する */
	void *keyword;
};

/**
 * SQLパーサで発生した１つのエラー情報記述する構造体
 */
struct __divy_sql_parser_err {
	int retcd;		/* パーサから返却されるエラーリターン値 */
	const char *code;	/* ステータスコード */
	const char *msg;	/* エラーメッセージ */
};

/*-----------------------------------------------------------------------------
   SQL文の構造を表す構造体群の定義
 ----------------------------------------------------------------------------*/
/**
 * divy_sql_elemnode の共用体メンバとなる構造体の種類を表す定数
 * (note) 命名規則
 * 	"DIVY_NT_" + (構造体の名前)
 */
enum __divy_sql_nodetype {
	DIVY_NT_UNKNOWN  = 0,	/* 未分類(別の分類法に従っていることを示す) */

	/* トップノードを表すタイプ */
	DIVY_NT_SELECT,

	/* SQL文中の所属句を表すタイプ */
	DIVY_NT_SELECT_CLAUSE,

	/*
	 * TeamFile で導入された概念を表すタイプ
	 */
	DIVY_NT_BIND,		/* バインド変数      */
	DIVY_NT_NAMEDBIND,	/* 名前付きバインド  */
	DIVY_NT_S_REQUIREDSQL,	/* RequiredSQL($$SS) */
	DIVY_NT_M_REQUIREDSQL,	/* RequiredSQL($$SM  */

	/*
	 * SQL構成要素を表すタイプ
	 */
	DIVY_NT_FUNC,		/* SQL関数         */
	DIVY_NT_OPERATOR,	/* 演算子          */
	DIVY_NT_COLUMN,		/* カラム名        */
	DIVY_NT_SLITERAL,

	DIVY_NT_END	/* sentinel */
};

/**
 * SQL の構成要素を表す構造体 (SQLエレメントノード)
 *
 * (note) 構造
 *   同階層のSQL構成要素をリストとして保持するためのポインタ(nextメンバ)と
 *   １つ下のSQL構成要素に対するポインタ(first_childメンバ)を持っている。
 *   apr_xml_elem と似た構造になっています。(似せて作りました)
 *
 * (例)
 * 	SELECT a, b FROM table WHERE id = $$SSxxx
 * 	-->
 * 	このSELECT文そのものも、その構成要素a,b も全てこの構造体
 * 	によって表現出来ます。
 *
 * (note) 式をdivy_sql_elemnode に格納する方法
 *
 * 	基本的には、SELECTノードの下(first_child)に式のノードを連結して保持します。
 * 	ある1つの式のノードは、オペレータをトップに左辺値(LVALUE)と右辺値(RVALUE)
 * 	から構成されています。取り出しは先に括弧内に書かれたマクロを使用します。
 * 	UNIONで結ばれたSELECTノードは同階層に、サブクエリーで連結されたSELECT
 * 	ノードはfirst_child に連結されます。
 * 	WHERE句やHAVING句に指定された構文は式の連結リストとして保持します。
 *
 * 	(例)
 * 	-----------------------------------------------------------------------
 * 	SELECT $$Bxxx, id, val FROM abc
 * 	WHERE id2 = $$SSxxx and	id3 = (SELECT id4 FROM zzz WHERE id5 IN $$SMxxx)
 * 	UNION SELECT ...
 * 	-----------------------------------------------------------------------
 * --->
 * 	SELECT
 * 	  |
 * 	  $$Bxxx -> '='     ->      '='        ->        SELECT
 * 	             |               |                     | 
 * 	             id2 -> $$SSxxx  id3 -> SELECT         ...
 * 	                                       |
 * 	                                       IN
 * 	                                       |
 * 	                                       id5 -> $$SMxxx
 *  "->"  : next
 *  "|"   : first_child (マクロCHILD_NODE(node) で取得)
 *
 *  (note) 式の詳細
 * 
 *   ○ 単体の式(xxxx) の場合
 *   ---------------------------------------------
 *   	nodetype    : 適切なノードを表す列挙型
 *   	name        : RequiredSQL、名前付きバインド変数、通常バインド変数の名前
 *   	(残り)      : NULL
 *
 *   ○ ２項式 (xxxx op expr) の場合
 *   ---------------------------------------------
 *   	nodeype           : 左辺値と右辺値を結ぶオペレータタイプ(DIVY_NT_OPERATOR)
 *   	name              : nodetype が示すオペレータの名前("="や">="など)
 *   	first_child       : 左辺値(xxxx) のdivy_sql_elemnode *
 *   	first_child->next : 右辺値(expr) のdivy_sql_elemnode *
 *   	(残り)            : NULL
 *
 *   ○ N 多項式(xxxx op expr1, ..., exprn) の場合
 *   ---------------------------------------------
 *   	nodetype          : 左辺値と右辺値を結ぶオペレータタイプ(DIVY_NT_OPERATOR)
 *   	name              : nodetype が示すオペレータの名前("="や">="など)
 *   	first_child       : 左辺値(xxxx) のdivy_sql_elemnode *
 *   	first_child->next : 右辺値(expr1) のdivy_sql_elemnode *
 *   	first_child->next
 *   	      ->...->next : 右辺値(exprn) のdivy_sql_elemnode *
 *
 * (note) このような構造を採用した理由
 *
 *   多くのSQLパーサでは、SQL構成要素をＮ組(Ｎ=3,4)によって表現しています。
 *   一般的になSQLパーサは、SQLの意味解析や最適化を行う目的で、字句解析と
 *   構文解析を実施するので、データ構造もそれに見合ったものを採用するのは
 *   当然です。ただ、これらのデータ構造は大掛かりなものになり易く、今回の
 *   ようなRequiredSQLの周辺部だけ理解できればよい、という種類の簡易解析
 *   においては、寧ろ、分かり難いだろうと思われます。
 *   そのため、可読性を高める意味もこめて、XMLの解析ノードの形に似せて
 *   設計しています。
 *   当然、このようなデータ構造を使用することで、意味解析は非常に困難に
 *   なります。(WHERE句も分からないので。)
 *   将来、意味解析が必要になったならば、データ構造の見直しが必要になる
 *   でしょう。(RequiredSQLにそこまで必要とすれば、ですが。)
 */
struct __divy_sql_elemnode {

	/* 要素の種類を表す列挙型 */
	divy_sql_nodetype nodetype;

	/*
	 * SQLエレメントノードを識別する名前
	 * このノードがDBカラムやRequiredSQLなどの名前付きの
	 * SQL構成要素を表す時には、その名称が入ります
	 * 演算子の場合には、演算子を表す文字列が入ります。
	 */
	char *name;

	/* 所属ノードを表す */
	divy_sql_nodetype belong_nodetype;

	/* このノードのSQL解析位置 */
	int pos;

	/* 同Tree階層に並ぶSQLノードへのポインタ(NULL終端) */
	divy_sql_elemnode *next;

	/* １レベル下(Child) SQLエレメントノードへのポインタ(NULL終端) */
	divy_sql_elemnode *first_child;
};

/* divy_sql_elemnode を操作するためのマクロ */
#define LVALUE(n)	((n) ? (n)->first_child : NULL)
#define RVALUE(n)	((n) && (n)->first_child ? (n)->first_child->next : NULL)
#define CHILD_NODE(n)	((n) ? (n)->first_child : NULL)

/*-----------------------------------------------------------------------------
   SQL 間の関係を表す構造体群の定義
 ----------------------------------------------------------------------------*/
/**
 * RequiredSQL間の依存関係を表す構造体 (SQL依存グラフノード)
 *
 * (note) 構造
 * -------------
 * ある１つRequiredSQLと、それを利用するRequiredSQLの一覧(Parent)、
 * 利用されるRequiredSQLの一覧(Child) を持っている。
 *
 * ParentノードからChildノードへの向きは、child_listで表現します。
 * (あるSQL依存グラフノードがchild_listを持つ時、自身はchild_list
 *  から見て親ということになります。)
 * 自身の親ノードからなるリストparent_list は、処理の都合上
 * どうしても戻って探索したい場合や自分に親があるかどうかの
 * 判定に使用する目的で導入しています。
 * SQL依存グラフの構造を決定する役割は持っていません。
 *
 * (例) p +--> c1 (p からc1, c2 への向きを持つ有向グラフ)
 *        |--> c2 
 *
 *      ==> c1 = p->child_list->d_node
 *          c2 = p->child_list->next->d_node
 *
 * (例) ある１つのSQL依存グラフノードのイメージ
 *
 *        divy_sql_dependnode *x
 *        +------------------+  
 * p1---> | name = "$$SSxx"  | ---> c1
 * p2---> | nid  = nnnnn     | ---> c2
 *        |                  | ---> c3
 *        +------------------+
 */
struct __divy_sql_dependnode {
	/* 
	 * このSQL依存グラフノードを識別する数.
	 * １から始まる。同一判定などはこの値を用いる
	 */
	unsigned int nid;

	/* このSQL依存グラフノードの名前(nnnnnnnn, $$SSxxxx, $$SMxxxx, $$Bxxx) */
	char *name;

	/* このSQL依存グラフノードがRequiredSQL、名前付きバインド変数かどうか */
	int isRequired;

	/* SQLエレメントノードが持つ値(キー値,表示値)リスト */
	divy_rdbo_rsvalueset *valueset;

	/* このノードの"親"を要素とするSQL依存グラフノードのリスト */
	divy_sql_dependnodeList *parent_list;

	/* このノードの"子"を要素とするSQL依存グラフノードのリスト */
	divy_sql_dependnodeList *child_list;

	/* ノードを巡回する際に使用するワークフラグ */
#define INIT    0	/* このノードは未操作               */
#define VISIT   1 	/* このノードは１度通ったことがある */
#define RESOLVE 2	/* このノードの値は解決済みである   */
	int state;

	/* SQL依存グラフの閉回路検出に使用するワーク変数(初期値:0) */
	unsigned int path_nid;
};

/**
 * divy_sql_dependnode * のリストを表現する構造体
 */
struct __divy_sql_dependnodeList {

	/* 要素へのポインタ */
	divy_sql_dependnode *d_node;
	
	/* 次のリストへのポインタ */
	divy_sql_dependnodeList *next;
};


/*-----------------------------------------------------------------------------
  Declare array 
 ----------------------------------------------------------------------------*/
/**
 * SQLパーサから戻されたエラーリターン値とエラーコード・メッセージの関係を
 * 持つ構造体の配列。
 */
SQLP_GLOBAL const divy_sql_parser_err sql_parser_errs[]
#ifdef SQLP_GLOBAL_VALUE_DEFINE
= {
	{ DIVY_SQLP_ST_ERR,
	  "S00001",
	  "Unexpected fatal error."
	},
	{ DIVY_SQLP_ST_UNSUPPORT,
	  "S00002",
	  "Unsupported sql statement."
	},
	{ DIVY_SQLP_ST_INVALID, 
	  "S00003",
	  "There is a syntax error in the sql statement."
	},
	{ DIVY_SQLP_ST_WRONG_LVAL, 
	  "S00004",
	  "Wrong left value."
	},
	{ DIVY_SQLP_ST_WRONG_OPER,
	  "S00005",
	  "Wrong operator."
	},
	{ DIVY_SQLP_ST_TOOMANY_REQSQL,
	  "S00006",
	  "Too many requiredsql specified."
	},
	{ DIVY_SQLP_ST_WRONG_M_PREF, 
	  "S00007",
	  "The position of multiple requiredsql is bad."
	},
	{ DIVY_SQLP_ST_NOTHING_TODO,
	  "S00008",
	  "Noting todo"
	},
	{ DIVY_SQLP_ST_LOOP_DETECTED, 
	  "D00001",
	  "Loop detected."
	},
	{ DIVY_SQLP_ST_USINGSQL_FOUND,
	  "D00002",
	  "This Requiredsql has dependency with other sql."
	},
	{ DIVY_SQLP_ST_WRONG_RELATION,
	  "D00003",
	  ""
	},
	{ DIVY_SQLP_ST_EXEC_FAILED,
	  "D00004",
	  ""
	},
	{ DIVY_SQLP_ST_MULTIPLE_TOP,
	  "D00005",
	  "The multiple top sql is found."
	},
	{ DIVY_SQLP_ST_NOTFOUND,
	  "D00006",
	  ""
	},
	{ DIVY_SQLP_ST_FOUND_NBIND,	/* 廃止 */
	  "D00007",
	  NULL
	},
	{ DIVY_SQLP_ST_EMPTY_HIDDEN_NBIND,
	  "D00008",
	  "The default value of Namedbind used by other sql in HIDDEN mode."
	},

	{ DIVY_SQLP_ST_MISMATCH_ID,
	  "S00009",
	  "The specified subname in resolve-value data is different from that in the sql."
	},
	{ DIVY_SQLP_ST_MISSING_ID,
	  "S00010",
	  "The \"id\" attribute in resolve-value data is missing."
	},
	{ DIVY_SQLP_ST_WRONG_BINDVAL,	/* 廃止 */
	  "S00011",
	  NULL
	},
	{ DIVY_SQLP_ST_NONSELECT,
	  "S00012",
	  "SQL statement must have SELECT phrase."
	},
	{ DIVY_SQLP_ST_NONFROM,
	  "S00013",
	  "SQL statement must have FROM phrase."
	},
	{ DIVY_SQLP_ST_WRONG_NBIND_VAL, /* 廃止 */
	  "S00014",
	  NULL
	},
	{ DIVY_SQLP_ST_WRONG_NBIND_KEY, /* 廃止 */
	  "S00015",
	  NULL
	},
	{ DIVY_SQLP_ST_WRONG_NBIND_OP,	/* 廃止 */
	  "S00016",
	  NULL
	},
	{ DIVY_SQLP_ST_MISSING_SQLCNT,	/* 廃止 */
	  "S00017",
	  NULL 
	},
	{ DIVY_SQLP_ST_MISSING_SQLDIS,	/* 廃止 */
	  "S00018",
	  NULL 
	},
	{ DIVY_SQLP_ST_MISSING_SQLNAME,	/* 廃止 */
	  "S00019",
	  NULL 
	},
	{ DIVY_SQLP_ST_MISMATCH_NAMEURI, /* 廃止 */
	  "S00020",
	  NULL
	},
	{ DIVY_SQLP_ST_MISSING_SQLSTMT,
	  "S00021",
	  "The sqlstatement property is missing."
	},
	{ DIVY_SQLP_ST_MISSING_REQSUB,
	  "S00022",
	  "The subname property is missing."
	},
	{ DIVY_SQLP_ST_WRONG_SUBNAME,
	  "S00023",
	  "The format of subname is wrong."
	},
	{ DIVY_SQLP_ST_MISMATCH_REQURI,
	  "S00024",
	  "The name of subname is already used."
	},
	{ DIVY_SQLP_ST_REQSQL_NF,
	  "S00025",
	  "The RequiredSQL or namedbind in SQL statement is not found."
	},
	{ DIVY_SQLP_ST_CONFLICT_SQLCNT,
	  "S00026",
	  "The content of where property conflicts with the condition of sql statement."
	},
	{ DIVY_SQLP_ST_WRONG_BIND_POS,
	  "S00027",
	  "Could not specify the BIND variable in thie sql."
	},
	{ DIVY_SQLP_ST_UNQUOTED_STR,
	  "S00028",
	  "The single-quote is not closed."
	},
	{ DIVY_SQLP_ST_MISMATCH_SQLTYPE,
	  "S00029",
	  "The conbinations of type of sql statement and sqltype are not correct."
	},
	{ DIVY_SQLP_ST_NOEFFECTIVE_BODY,
	  "S00030",
	  "The namedbind sentence could not have WHERE property."
	},
	{ DIVY_SQLP_ST_OVER_MAX_CHAIN,
	  "S00031",
	  "The count of sql-dependence exceeded the max value."
	},
	{ DIVY_SQLP_ST_MUSTCOL_NF,
	  "S00032",
	  "The SELECT phrase must include more culumn(s)."
	},
	{ DIVY_SQLP_ST_SQLTYPE_UNKNOWN,
	  "S00033",
	  "Type of sql is not selected or unknowwn."
	},
	{ DIVY_SQLP_ST_TOOMANY_COLUMN,
	  "S00034",
	  "The number of SELECT column is large."
	},
	{ DIVY_SQLP_ST_TOOFEW_COLUMN,
	  "S00035",
	  "The number of SELECT column is small."
	},
	{ DIVY_SQLP_ST_FOUND_NBIND_VAL,
	  "S00036",
	  ""
	},
	{ DIVY_SQLP_ST_NF_NBIND_VAL,
	  "S00037",
	  ""
	},
	{ DIVY_SQLP_ST_FOUND_UPDATESTMT,
	  "S00038",
	  "The update-DML or DDL, DCL statement is specified."
	},
	{ DIVY_SQLP_ST_UNKNOWN_STMT,
	  "S00039",
	  "Could not understand the specified statement."
	},
	{ DIVY_SQLP_ST_UNQUOTED_DQSTR,
	  "S00040",
	  "The double-quote is not closed."
	},
	{ DIVY_SQLP_ST_FOUND_ASTERISK,
	  "S00041",
	  "Could not use '*' column."
	},
	{ DIVY_SQLP_ST_UNQUOTED_CTSTR,
	  "S00042",
	  "The c-style comment is not closed."
	},
	{ DIVY_SQLP_ST_MISSING_BINDVAL,	/* 廃止 */
	  "S00043",
	  NULL
	},
	{ DIVY_SQLP_ST_MISSING_STIDENT,	/* 廃止 */
	  "S00044",
	  NULL
	},
	{ DIVY_SQLP_ST_USELESS_SUBNAME,
	  "S00045",
	  "Could not set \"subname\" for normalsql or repository sql."
	},
	{ DIVY_SQLP_ST_CACHE_NONBIND,
	  "S00046",
	  "Could not set \"cache\" mode other then namedbind."
	},
	{ DIVY_SQLP_ST_CACHE_NBIND_VAL,
	  "S00047",
	  "Could not set \"cache\" mode namedbind with default value."
	},
	{ DIVY_SQLP_ST_WRONG_COMB_TYPE,
	  "S00048",
	  "The combinations of subname and sqltype are not correct."
	},
	{ DIVY_SQLP_ST_NEWSQL_LABELNAME,
	  "S00049",
	  "The labelname of newsql exists."
	},
	{ DIVY_SQLP_ST_OLDSQL_LABELNAME,
	  "S00050",
	  "The labelname of oldsql does not exist."
	},
	{ DIVY_SQLP_ST_RESERVED_SQL,
	  "S00051",
	  "The reserved sql could not be set with \"cache\" mode."
	},
	{ DIVY_SQLP_ST_LACKOF_LICENSE,
	  "S00052",
	  "The license of DB collaboration had been missing."
	},
	{ -1, NULL, NULL }	/* sentinel */
  }
#endif	/* SQLP_GLOBAL_VALUE_DEFINE */
;

#ifdef SQLP_GLOBAL_VALUE_DEFINE
#define SQL_PARSER_ERR_LEN	sizeof(sql_parser_errs)/sizeof(divy_sql_parser_err)
#else
#define SQL_PARSER_ERR_LEN
#endif	/* SQLP_GLOBAL_VALUE_DEFINE */
 
/*--------------------------------------------------------------
  Declare public function
  --------------------------------------------------------------*/
/**
 * SQL parser を表す構造体のインスタンスを生成する。
 * (note)
 * 	作成したparserオブジェクトは、parserが内部的な情報を保持するために
 * 	使用します。呼び出し側がこの値を意識する必要はありません。
 *
 * @param p apr_pool_t *
 * @param parser divy_sql_parser ** 生成したparser へのポインタ
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK  : 成功
 * 	DIVY_SQLP_ST_ERR : 失敗 (500)
 */
DIVY_DECLARE(int) divy_sql_parser_create(apr_pool_t *p, divy_sql_parser **parser);

/**
 * パーサのコンテキスト構造体を初期化する. (内部専用)
 * (note)
 * 	この関数は、字句解析・構文解析パーサとのやり取りでのみ使用するように
 * 	して下さい。それ以外の用途では、divy_sql_parser_create でパーサ
 * 	コンテキストも一緒に初期化しますので、そちらを使用して下さい。
 *
 * @para p apr_pool_t *
 * @param p_ctx divy_sql_parser_ctx ** パーサコンテキスト
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK  : 成功
 * 	DIVY_SQLP_ST_ERR : 失敗 (500)
 */
DIVY_DECLARE(int) divy_sql_parser_init_pctx(apr_pool_t *p,
					divy_sql_parser_ctx **p_ctx);

/**
 * SQLパーサの戻り値retcd から、詳細なエラー情報(divy_sql_parser_err)を
 * 取得して返却する。
 * エラーコードから詳細なエラーメッセージを取得する用途において使用して
 * 下さい。
 *
 * @param p apr_pool_t *
 * @param retcd int SQLパーサから返却されたエラーコード
 * @return divy_sql_parser_err * 対応するエラー構造体へのポインタ
 */
DIVY_DECLARE(divy_sql_parser_err *) divy_sql_parser_get_parser_err(
						apr_pool_t *p, int retcd);

/**
 * SELECT に指定されたカラムが決められたカラム集合に含まれている
 * かどうかを検証する。
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param sql const char * SELECT句を含むSQL文
 * @param must_selcol_hash apr_hash_t ** 決められたカラム集合に含まれているが、
 * 					SELECT句に含まれていないカラム
 * (note) SELECT 句に全て含まれていた場合はNULL
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK               : 問題なし。成功
 * 	DIVY_SQLP_ST_ERR              : 失敗 (400/500)
 * 	DIVY_SQLP_ST_INVALID          : 誤りのある可能性が高いSQL構文エラーを発見した(400)
 * 	DIVY_SQLP_ST_WRONG_LVAL       : RequiredSQLが左辺値として指定された (400)
 * 	DIVY_SQLP_ST_WRONG_OPER       : RequiredSQLに比較演算子以外が適用された (400)
 * 	DIVY_SQLP_ST_WRONG_M_PREF     : $$SMxxx がIN句の右辺値以外に指定された (400)
 * 	DIVY_SQLP_ST_NONSELECT        : SELECT がない (400/500/おためしエラー)
 * 	DIVY_SQLP_ST_NONFROM          : FROM がない (400/500/おためしエラー)
 * 	DIVY_SQLP_ST_MUSTCOL_NF       : MUST のカラムが含まれていない(400/500/おためしエラー)
 * 	DIVY_SQLP_ST_WRONG_BIND_POS   : 通常バインド変数がRequiredSQLの中に指定された
 * 	DIVY_SQLP_ST_UNQUOTED_STR     : SQL文中に正しく閉じられていないシングルクオートがある (400)
 * 	DIVY_SQLP_ST_MISMATCH_SQLTYPE : sqltype とSQL文の種類が一致しない (400)
 * 	DIVY_SQLP_ST_TOOMANY_COLUMN   : SELECT句に指定されたカラム数が多い (400)
 * 	DIVY_SQLP_ST_TOOFEW_COLUMN    : SELECT句に指定されたカラム数が少ない (400)
 * 	DIVY_SQLP_ST_FOUND_UPDATESTMT : 更新系DML、DDL、DCLが指定された(400)
 * 	DIVY_SQLP_ST_UNKNOWN_STMT     : 理解出来ないSQL構文・識別子が指定された(400)
 * 	DIVY_SQLP_ST_UNQUOTED_DQSTR   : ダブルクオート文字が正しく閉じられていない(400)
 * 	DIVY_SQLP_ST_FOUND_ASTERISK   : SELECT句に'*'が指定された (400)
 * 	DIVY_SQLP_ST_UNQUOTED_CTSTR   : SQL文中に閉じられていないSQLコメントがある (400)
 */
DIVY_DECLARE(int) divy_sql_parser_validate_selectcol(divy_sql_parser *parser,
						const char *sql,
						apr_hash_t **must_selcol_hash);

/**
 * [ RequiredSQLの依存関係の最下層を調べる（MUSTなRequiredを調べる）]
 *
 * dependlist のような依存関係がある状態で、値が解決されているRequiredSQL集合
 * reqsql_set が与えられた時、SQLの実行に最低限必要なRequiredSQLの一覧を
 * 返却する。
 *
 * (note) dependlist について
 * ---------------------------
 * 	dependlist には、トップSQL(頂点ノード)がただ１つだけ含まれるように
 * 	データを入れて渡すようにして下さい。
 * 	トップSQLとは、divy_sqldepend テーブルのsqld_top_sql_id_cがある値の
 * 	SQLのことを示しています。SQL依存グラフ上では、親グラフノードを持たない
 *	依存グラフノードであると定義されます。
 *
 * (note) 判定ロジックについて
 * ---------------------------
 *   以下のような依存関係があったとします。(矢印の先が必要とされている側を示す)
 *
 *   $$SMxx1 +-> $$SSxx2 --> $$SSxx3 --> $$Bxx4
 *           |-> $$SSxx5 --> $$SSxx6
 *
 *   依存グラフの最下層ノードは、$$Bxx4 と $$SSxx6 になります。
 *
 *   (基本)
 *   ・reqsql_setが空であれば、依存グラフの最下層にある名前付きバインド変数
 *     $$Bxx4 がmust_reqsql_set として報告されます。
 *
 *   (仮定)
 *   ・最下層に存在する$$SSや$$SMのRequiredSQLは可変のWHERE句を持っていてはならない
 *     (もし持っていたとすれば、このSQL自身が最下層であるという事実と矛盾する。
 *      このようなこれは関係があるとすれば、データ不整合以外に有り得ない。)
 *
 *   (例外規則)
 *   ・$$Bxx4 にデフォルト値が設定されていれば報告しません。
 *   ・$$SSxx3 がreqsql_setに格納されていれば、$$Bxx4 が未解決であっても解決済み。
 *     (依存グラフの中間に一つでも解決されているノードがあれば、階層は無視されます)
 *     (パフォーマンス改善のためと、キャッシュ値を有効に活用するため。)
 *
 * (note) DB のトランザクションが継続中であれば引き継ぎます
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param r request_rec *
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @param dependlist divy_rdbo_sqldepend * 依存関係を表すリスト
 * @param reqsql_set apr_hash_t * 値が解決済みのRequiredSQLからなる集合
 * @param must_reqsql_set apr_hash_t ** 値を解決しなければならないRequiredの集合
 * 					解決すべきものがなければNULLが戻ります。
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK             : 成功 or 何もする必要は無かった
 * 	DIVY_SQLP_ST_ERR            : 失敗 (500)
 * 	DIVY_SQLP_ST_LOOP_DETECTED  : 依存関係の不正. (500)
 * 	DIVY_SQLP_ST_EXEC_FAILED    : 名前付きバインド変数の取得に失敗した(500)
 */
DIVY_DECLARE(int) divy_sql_parser_find_sqldependency(divy_sql_parser *parser,
						request_rec *r,
						divy_db_transaction_ctx *ts_ctx,
						divy_rdbo_sqldepend *dependlist,
						apr_hash_t *reqsql_set,
						apr_hash_t **must_reqsql_set);

/**
 * 指定されたSQL文sqlに含まれるRequiredSQLまたは名前付きバインド変数を
 * SQLバインド変数("?") で置換し、それぞれのSQLバインド位置に対応するよう
 * RequiredSQL及び名前付きバインド変数を並べ替えて配列として返す。
 * IN句指定された場合にはこれらの句の性質に合致するよう値を変形する。
 *
 * この関数で得られた変換されたSQL文 *new_sql を実行するには、
 *   (1) *new_sql をプリペアする
 *   (2) bindvals の0番目から順番にsetXXXX 関数を使ってバインド
 *   (3) executeQueryを使って実行
 *
 * して下さい。rsvalue がNULLでなければ(2) を実施しないとSQLエラーとなります。
 *
 * [ IN句に指定された場合 ]
 * ---------------------------
 *   RequiredSQLから得られる値が複数であった時には、値をカンマで区切り、
 *   値集合の前後を括弧で閉じる。
 *   (例) $$SMxxxx1 = 1,2,3 の場合
 *        --> *new_sql => IN (?,?,?)
 *            bindvals => (1,2,3)
 *
 * (note) 同名のRequiredSQL、名前付きバインド変数が指定された場合の置換方式
 *   一つのSQLの中に同じRequiredSQLを複数指定することが可能である外部仕様に
 *   なっています。そのため、RequiredSQLまたは名前付きバインド変数の置き換え
 *   は、その名前と出現位置によって行っています。
 *   (例)
 * 	sql = "SELECT id, value FROM xxxx WHERE id2 = $$Bxxxx1
 * 		 AND id3 = $$SSxxx2 OR id4 = $$Bxxxx1"
 *
 * 	rsvalue: id = 1, name = "$$Bxxxx1", value = "abcd"
 * 	rsvalue: id = 2, name = "$$SSxxx2", value = "tanaka"
 * 	rsvalue: id = 3, name = "$$Bxxxx1", value = "cccc"
 *
 * 	-->
 * 	*new_sql = "SELECT id, value FROM xxxx WHERE id2 = ? 
 * 		AND id3 = ? OR id4 = ? 
 * 	bindvals = ("abcd", "tanaka", "cccc")
 *
 * 	$$Bxxxx1 は２回出現しますが、異なる値で置換されていることに注意して下さい。
 *
 * (note) LIKE句に指定されたRequiredSQLについて
 *   ワイルドカード文字"%" を補ったりはしません。ワイルドカードを得られた
 *   値のどこに(前、後)、どれ(%, _) を指定したらいいのか分からないからです。
 *
 * (note) !!注意!!
 * 	この関数はmutexを使ったロックを使用する関数を使っています。
 * 	mutex の中からこの関数をCallすることは厳禁です。
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param sql const char * 置き換え対象のSQL文
 * @param bvalue divy_search_ldbs_bind * バインド変数リスト
 * @param rsvalue  divy_rdbo_rsvalue *
 * 				置き換え対象のRequiredSQLまたは名前付き
 * 				バインド変数の名前と値を持つ構造体へのポインタ
 * @param sql char ** 置き換えられたSQL文
 * @param bindvals	divy_array_t ** バインド位置をインデックスとするバインド変数の配列
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK              : 成功
 * 	DIVY_SQLP_ST_ERR             : 失敗 (500)
 * 	DIVY_SQLP_ST_WRONG_REQSQLNUM : sql中のRequiredSQL数とrsvalueのリスト数が異なる (400)
 * 	DIVY_SQLP_ST_MISMATCH_ID     : sql中のRequiredSQL出現位置とrsvalue->idの値が異なる(400)
 * 	DIVY_SQLP_ST_MISSING_ID      : rsvalue->id が歯抜けだったため、置換できなかった(400)
 * 	DIVY_SQLP_ST_UNQUOTED_STR    : SQL文中に正しく閉じられていないシングルクオートがある (400)
 * 	DIVY_SQLP_ST_UNQUOTED_DQSTR  : ダブルクオート文字が正しく閉じられていない(400)
 * 	DIVY_SQLP_ST_UNQUOTED_CTSTR  : SQL文中に閉じられていないSQLコメントがある (400)
 */
DIVY_DECLARE(int) divy_sql_parser_replace_reqsql(divy_sql_parser *parser,
						const char *sql,
						divy_search_ldbs_bind *bvalue,
						divy_rdbo_rsvalue *rsvalue,
						char **new_sql,
						divy_array_t **bindvals);

/**
 * [ WHERE句プロパティの作成 ]
 *
 * 指定されたSQL文sqlを解析し、WHERE句プロパティwhere_pr を生成して返却
 * する。
 * (note)
 * 	名前付きバインド変数のデフォルト値は指定できません。
 * 	(構文エラーになります)
 *
 * (note) WHERE句プロパティについて
 *   WHERE句プロパティとは、エンドユーザがlinkdbsearch を実施する際に入力
 *   しなければならない値をエンドユーザに示すための情報のことを示します。
 *   "エンドユーザに示すための情報"とは、具体的には、
 *
 *   	・入力値を端的に示す名称	(SQLラベル名)
 *   	・入力値に関する書式		(データフォーマット)
 *   	・入力値を自動補完するための値	(RequiredSQL、名前付きバインド変数)
 *   のことを示します。
 *   これらは全て管理者がSQLを作成する際に設定してもらう必要がありますが、
 *   SQL文に直接記述することは出来ません。(そういう外部Specです。)
 *   これらの指定には独自のプロパティシートを利用しますので、このプロパティ
 *   シートのデータを生成する処理をここで行っています。
 *   ただ、SQL文には、上記のうちの一部しか含まれていません。
 *
 * (note) 返却情報の形式
 * 	(例)
 * 	sql = "SELECT id, $$SSxxxx2, value FROM xxxx WHERE id2 = $$Bxxxx1"
 *
 * 	DBカラム「id2」は名前付きバインド変数「$$Bxxxx1」を値生成SQLとして
 * 	指定しているので、この関係をdivy_rdbo_sqlcontent に格納して返却する。
 * 	SELECT 句に指定されたRequiredSQL「$$SSxxxx2」には対応するDBカラムが
 * 	存在しないため、DBカラムをNULLにして返却します。
 *
 * (note) 返却対象にならないWHERE句プロパティ
 *
 * 	・サポートされていない演算子と共に使用されたRequiredSQL、名前付きバインド変数
 *	・エスケープされたRequiredSQL、名前付きバインド変数
 *	・通常バインド変数、RequiredSQL、名前付きバインド変数以外の式
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param sql const char * SQL文
 * @param where_pr divy_rdbo_sqlcontent ** 生成したWHERE句プロパティ
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK               : 成功
 * 	DIVY_SQLP_ST_ERR              : 失敗 (500)
 * 	DIVY_SQLP_ST_INVALID          : 誤りのある可能性が高いSQL構文エラーを発見した(400)
 *	DIVY_SQLP_ST_WRONG_LVAL       : RequiredSQLが左辺値として指定された (400)
 * 	DIVY_SQLP_ST_WRONG_M_PREF     : $$SMxxx がIN句の右辺値以外に指定された (400)
 * 	DIVY_SQLP_ST_UNQUOTED_STR     : SQL文中に閉じられていないシングルクオートがある(400)
 * 	DIVY_SQLP_ST_FOUND_UPDATESTMT : 更新系DML、DDL、DCLが指定された(400)
 * 	DIVY_SQLP_ST_UNKNOWN_STMT     : 理解出来ないSQL構文・識別子が指定された(400)
 * 	DIVY_SQLP_ST_UNQUOTED_DQSTR   : ダブルクオート文字が正しく閉じられていない(400)
 * 	DIVY_SQLP_ST_FOUND_ASTERISK   : SELECT句に'*'が指定された (400)
 * 	DIVY_SQLP_ST_UNQUOTED_CTSTR   : SQL文中に閉じられていないSQLコメントがある (400)
 */
DIVY_DECLARE(int) divy_sql_parser_make_whereprop(divy_sql_parser *parser,
						const char *sql,
						divy_rdbo_sqlcontent **where_pr);

/**
 * 指定されたSQL文sqlに含まれるRequiredSQL及び通常・名前付きバインド変数が指定規則に
 * 基づいて指定されているかどうか検証する。
 * (note)
 * 	名前付きバインド変数構文(sqltype = DIVY_SQL_TYPE_BIND) のsql の内容は
 * 	単なる値とみなすようになりました。常に正常であると返却されます。
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param sqltype divy_rdbo_sqltype 解析対象SQLのsqltype
 * @param sql const char * SQL文
 * @return int 処理ステータスと検証結果
 * 	DIVY_SQLP_ST_OK               : 問題なし。成功。 
 * 	DIVY_SQLP_ST_ERR              : 予期しないエラーのため中止 (500)
 * 	DIVY_SQLP_ST_INVALID          : 誤りのある可能性が高いSQL構文エラーを発見した(400)
 * 	DIVY_SQLP_ST_WRONG_LVAL       : RequiredSQLが左辺値として指定された (400)
 * 	DIVY_SQLP_ST_WRONG_OPER       : RequiredSQLに比較演算子以外が適用された (400)
 * 	DIVY_SQLP_ST_WRONG_M_PREF     : $$SMxxx がIN句の右辺値以外に指定された (400)
 * 	DIVY_SQLP_ST_NONSELECT        : "SELECT" がない(400)
 * 	DIVY_SQLP_ST_WRONG_BIND_POS   : 通常バインド変数がRequiredSQLの中に指定された
 * 	DIVY_SQLP_ST_UNQUOTED_STR     : SQL文中に正しく閉じられていないシングルクオートがある (400)
 * 	DIVY_SQLP_ST_MISMATCH_SQLTYPE : sqltype とSQL文の種類が一致しない (400)
 * 	DIVY_SQLP_ST_TOOMANY_COLUMN   : SELECT句に指定されたカラム数が多い (400)
 * 	DIVY_SQLP_ST_TOOFEW_COLUMN    : SELECT句に指定されたカラム数が少ない (400)
 * 	DIVY_SQLP_ST_FOUND_UPDATESTMT : 更新系DML、DDL、DCLが指定された(400)
 * 	DIVY_SQLP_ST_UNKNOWN_STMT     : 理解出来ないSQL構文・識別子が指定された(400)
 * 	DIVY_SQLP_ST_UNQUOTED_DQSTR   : ダブルクオート文字が正しく閉じられていない(400)
 * 	DIVY_SQLP_ST_FOUND_ASTERISK   : SELECT句に'*'が指定された (400)
 * 	DIVY_SQLP_ST_UNQUOTED_CTSTR   : SQL文中に閉じられていないSQLコメントがある (400)
 */
DIVY_DECLARE(int) divy_sql_parser_validate_reqsql(divy_sql_parser *parser,
						divy_rdbo_sqltype sqltype,
						const char *sql);

/**
 * 指定されたSQL文sqlに含まれるRequiredSQL及び通常・名前付きバインド変数が指定規則に
 * 基づいて指定されているかどうか検証し、さらに指定されたWHERE句プロパティとの
 * 整合性を確認する.
 * (note)
 * 	名前付きバインド変数構文(sqltype = DIVY_SQL_TYPE_BIND) のsql の内容は
 * 	単なる値とみなすようになりました。常に正常であると返却されます。
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param sqltype divy_rdbo_sqltype 解析対象SQLのsqltype
 * @param sql const char * 解析対象のSQL
 * @param where_pr divy_rdbo_sqlcontent * WHERE句プロパティ.存在しない場合にはNULL
 * @return int 処理ステータスと検証結果
 * 	DIVY_SQLP_ST_OK               : 問題なし。成功。 
 * 	DIVY_SQLP_ST_ERR              : 予期しないエラーのため中止 (500)
 * 	DIVY_SQLP_ST_INVALID          : 誤りのある可能性が高いSQL構文エラーを発見した(400)
 * 	DIVY_SQLP_ST_WRONG_LVAL       : RequiredSQLが左辺値として指定された (400)
 * 	DIVY_SQLP_ST_WRONG_OPER       : RequiredSQLに比較演算子以外が適用された (400)
 * 	DIVY_SQLP_ST_WRONG_M_PREF     : $$SMxxx がIN句の右辺値以外に指定された (400)
 * 	DIVY_SQLP_ST_NONSELECT        : "SELECT" がない(400)
 * 	DIVY_SQLP_ST_CONFLICT_SQLCNT  : SQL文のWHEREプロパティとkeyvalue エレメントの内容に矛盾(400)
 * 	DIVY_SQLP_ST_WRONG_BIND_POS   : 通常バインド変数がRequiredSQLの中に指定された
 * 	DIVY_SQLP_ST_UNQUOTED_STR     : SQL文中に正しく閉じられていないシングルクオートがある (400)
 * 	DIVY_SQLP_ST_MISMATCH_SQLTYPE : sqltype とSQL文の種類が一致しない (400)
 * 	DIVY_SQLP_ST_TOOMANY_COLUMN   : SELECT句に指定されたカラム数が多い (400)
 * 	DIVY_SQLP_ST_TOOFEW_COLUMN    : SELECT句に指定されたカラム数が少ない (400)
 * 	DIVY_SQLP_ST_FOUND_UPDATESTMT : 更新系DML、DDL、DCLが指定された(400)
 * 	DIVY_SQLP_ST_UNKNOWN_STMT     : 理解出来ないSQL構文・識別子が指定された(400)
 * 	DIVY_SQLP_ST_UNQUOTED_DQSTR   : ダブルクオート文字が正しく閉じられていない(400)
 * 	DIVY_SQLP_ST_FOUND_ASTERISK   : SELECT句に'*'が指定された (400)
 * 	DIVY_SQLP_ST_UNQUOTED_CTSTR   : SQL文中に閉じられていないSQLコメントがある (400)
 */
DIVY_DECLARE(int) divy_sql_parser_validate_reqsql_withbody(
						divy_sql_parser *parser,
						divy_rdbo_sqltype sqltype,
						const char *sql,
						divy_rdbo_sqlcontent *where_pr);

/**
 * [ RequiredSQL の値を解決する ]  
 *
 * 値解決が既に行われているRequiredSQLの値集合rsvalueを使って、RequiredSQL間の
 * 依存関係dependlistの中で値が解決されていないRequiredSQLの値を求め、
 * トップ階層(RequiredSQLを使用する最上位)の値のみを返却する。
 *
 * (例) a --> b --> c --> d という依存関係があった時、
 * 	・d がrsvalueとして与えられた場合: c, b の順番にSQLを実行し、aのWHERE句に埋める
 * 	・c がrsvalueとして与えられた場合: b のSQLだけを実行し、aのWHERE句に埋める
 *
 * 	a のことをトップ階層と呼んでいる。a は通常SQL又はRequiredSQLの何れかである。
 *
 * (note)
 * 	dependlist は、入次数が０である唯一SQLを頂点とするグラフに
 * 	なっていなければなりません。
 *
 * (note) rsvalue 必須項目: name, valueset
 * (note) 依存関係にあるSQLの実行に失敗した場合、DIVY_SQLP_ST_EXEC_FAILED を
 *        返却し、処理を中断します。(外部仕様)
 *        このとき、top_rsvalue に失敗したSQLの名前を格納します。
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param r request_rec *
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @param dependlist divy_rdbo_sqldepend * 依存関係を表すリスト
 * @param rsvalue divy_rdbo_rsvalue * 解決済みRequiredSQLの名前と値のリスト
 * @param top_rsvalue_hash apr_hash_t ** サブ名称をキーとして、トップ階層の
 * 				RequiredSQL名称と値(divy_rdbo_rsvalue*) を
 * 				値としてもつハッシュ
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK          : 成功
 * 	DIVY_SQLP_ST_ERR         : 失敗(500)
 * 	DIVY_SQLP_ST_EXEC_FAILED : 依存SQLのどれかの実行に失敗したため処理を中断した(207 & 404)
 *	DIVY_SQLP_ST_TOOMANY_COLUMN : SELECT句のカラムが多すぎた(500)
 *	DIVY_SQLP_ST_TOOFEW_COLUMN  : SELECT句のカラムが少なすぎた(500)
 *	ただし、DIVY_SQLP_ST_TOOMANY_COLUMNとDIVY_SQLP_ST_TOOFEW_COLUMNは
 *	SQLの実行時エラーチェックであるため、試し実行ではエラーコードを返すこと
 */
DIVY_DECLARE(int) divy_sql_parser_resolve_reqsql(divy_sql_parser *parser,
						request_rec *r,
						divy_db_transaction_ctx *ts_ctx,
						divy_rdbo_sqldepend *dependlist,
						divy_rdbo_rsvalue *rsvalue,
						apr_hash_t **top_rsvalue_hash);

/**
 * dependlist のような依存関係がある時、この依存関係から作られるSQL依存グラフが
 * 閉回路(closed path)を作らないかどうか判定する。
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param dependlist divy_rdbo_sqldepend * 依存関係を表すリスト
 * @return int 依存グラフを作っていたかどうか
 * 	DIVY_SQLP_ST_OK            : 閉回路は存在しない
 * 	DIVY_SQLP_ST_LOOP_DETECTED : 閉回路が存在した(400)
 * 	DIVY_SQLP_ST_ERR           : 予期しないエラーが発生した(500)
 */
DIVY_DECLARE(int) divy_sql_parser_validate_closedpath(divy_sql_parser *parser,
						divy_rdbo_sqldepend *dependlist);

/**
 * dependlist のような依存関係がある時、この依存関係から作られるSQL依存グラフの
 * 依存連鎖数が最大値(DIVY_REQSQL_MAX_CHAING) を超えないかどうかを判定する。
 *
 * (note)
 * 	dependlist は、入次数が０である唯一SQLを頂点とするグラフに
 * 	なっていなければなりません。
 * 	さもなければ、全く依存関係の無いSQLグループで発生していた
 * 	DIVY_SQLP_ST_OVER_MAX_CHAINを見てしまいます。
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param dependlist divy_rdbo_sqldepend * 依存関係を表すリスト
 * @return int 依存グラフを作っていたかどうか
 * 	DIVY_SQLP_ST_OK             : SQL依存グラフは正常である
 * 	DIVY_SQLP_ST_MULTIPLE_TOP   : SQLの頂点ノードが複数個存在した (400)
 * 	DIVY_SQLP_ST_OVER_MAX_CHAIN : 依存SQLの階層数が最大数を超えた (400)
 * 	DIVY_SQLP_ST_ERR            : 予期しないエラーが発生した(500)
 */
DIVY_DECLARE(int) divy_sql_parser_validate_maxchain(divy_sql_parser *parser,
						divy_rdbo_sqldepend *dependlist);

/**
 * dependlist のような依存関係がある時、subname のSQLが依存グラフに
 * 含まれていないかどうか検証し、含まれていれば、subnameのSQLを直接利用するSQLの
 * 名称をリストして返却する。
 * (例)
 * 	$$SSxxxx1 -> $$SSxxxx2 -> $$SSxxxx3 -> $$Bxxxx4
 * 	subname = $$SSxxxx3 のとき、
 * 	$$SSxxxx2 を返却する。$$SSxxx1も間接的に利用しているといえるが、
 * 	ここでは返却しない。
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param subname const char * SQLのサブ名称
 * @param dependlist divy_rdbo_sqldepend * SQLの全依存関係を表すリスト
 * @param usingsql_set divy_cset_t ** subname のSQLを利用するSQLのサブ名称からなる集合 
 * @return int 処理ステータス
 *
 * 	DIVY_SQLP_ST_OK             : このSQLを使用しているSQLは存在しない
 * 	DIVY_SQLP_ST_USINGSQL_FOUND : このSQLを使用しているSQLが見つかった
 * 	DIVY_SQLP_ST_ERR            : 処理自体が異常終了した(500)
 */
DIVY_DECLARE(int) divy_sql_parser_find_usingsql(divy_sql_parser *parser,
						const char *subname,
						divy_rdbo_sqldepend *dependlist,
						divy_cset_t **usingsql_set);

/**
 * dependlist のような依存関係がある時、subname_setの集合に含まれるRequiredSQL
 * が利用するRequiredSQLと名前付きバインド変数との親子関係、つまり依存グラフの
 * 辺(edge)を全て取得してdivy_rdbo_sqldepend のリストとして返却する。
 *
 * (例)
 * 	$$SSxxxx1 -> $$SSxxxx2 -> $$SSxxxx3 -> $$Bxxxx4
 *	$$SSxxxx2 が指定された時、
 *	$$SSxxxx2 - $$SSxxxx3, $$SSxxxx3 - $$Bxxxx4 が返却される。
 *
 * (note)
 * 	同じ親子関係を持つedgelist は返却しません。
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param subname_set divy_cset_t * サブ名称の集合
 * @param dependlist divy_rdbo_sqldepend * SQLの全依存関係を表すリスト
 * @param edgelist divy_rdbo_sqldepend ** 依存グラフの辺リスト
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK             : このSQLを使用しているSQLは存在しない
 * 	DIVY_SQLP_ST_USINGSQL_FOUND : このSQLを使用しているSQLが見つかった
 * 	DIVY_SQLP_ST_ERR            : 処理自体が異常終了した(500)
 */
DIVY_DECLARE(int) divy_sql_parser_find_edgelist(divy_sql_parser *parser,
						divy_cset_t *subname_set,
						divy_rdbo_sqldepend *dependlist,
						divy_rdbo_sqldepend **edgelist);

/**
 * 指定されたSQLが持っているRequiredSQL及び名前付きバインド変数を取得して、
 * rsql_set につめて返却する。(通常バインド変数は返却されません)
 * なお、同名のRequiredSQLや名前付きバインド変数が複数使用されていたとしても
 * それは１つであるとカウントします。
 *
 * (note)
 * 	名前付きバインド変数のデフォルト値をsql に入れてコールしてはなりません。
 *
 * (note) !!注意!!
 * 	この関数は、scanner_mutex を使用しています。
 * 	scanner_mutex を使用する他の関数を絶対に内部から呼び出さないで下さい。
 *
 * (note) 取得ロジックについて
 *   基本的には、プレフィックスが「$$SS」「$$SM」「$$B」として宣言された変数を
 *   取得します。
 *   但し、「\」(バックスラッシュ)によってエスケープされたもの、「$$$SS」のような
 *   プレフィックス文字列が部分一致しただけのものは、RequiredSQLまたは名前付き
 *   バインド変数であるとは認識されません。
 *   多くの場合、指定ミスであると考えられますが、「$」を特別な変数と見立てている
 *   RDBMS が実在するため、この関数では厳密に扱います。
 *
 * @param parser divy_sql_parser * パーサを表す構造体
 * @param sql const char * 解析対象のSQL
 * @param rqsql_hash apr_hash_t ** １件も無い時にはNULL.そうでなければ以下の値を持つ
 * 		key  : SQLに含まれていたRequiredSQL or 名前付きバインド変数名(char *)
 *		value: 出現位置(int *)
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK               : 成功
 * 	DIVY_SQLP_ST_ERR              : 失敗(500)
 * 	DIVY_SQLP_ST_UNQUOTED_STR     : SQL文中に正しく閉じられていないシングルクオートがある (400)
 * 	DIVY_SQLP_ST_UNQUOTED_DQSTR   : ダブルクオート文字が正しく閉じられていない(400)
 * 	DIVY_SQLP_ST_UNQUOTED_CTSTR   : SQL文中に閉じられていないSQLコメントがある (400)
 */
DIVY_DECLARE(int) divy_sql_parser_extract_reqsql(divy_sql_parser *parser,
						const char *sql,
						apr_hash_t **reqsql_hash);

/**
 * 名前付きバインド変数にデフォルト値が設定されているかどうかを判別する。
 *
 * @param parser divy_sql_parser * パーサを表す構造体
 * @param sql const char * SQL文(BINDVAL構文を表す文字列)
 * @return int 値を持っていたかどうか
 * 	DIVY_SQLP_ST_FOUND_NBIND_VAL : 持っていた
 * 	DIVY_SQLP_ST_NF_NBIND_VAL    : 持っていなかった
 *	DIVY_SQLP_ST_ERR             : 予期しないエラー
 */
DIVY_DECLARE(int) divy_sql_parser_is_nbind_setting(divy_sql_parser *parser,
						const char *sql);

/**
 * SQLタイプtypeのWHERE句プロパティにサブ名称subnameのSQLが指定された時、
 * 持つことが出来るSQLモード一覧を返却する。
 * なお、subname が名前付きバインド変数の場合には、デフォルト値があるか
 * どうかをhas_default_valに指定する。
 *
 * @param parser divy_sql_parser * パーサを表す構造体
 * @param type divy_rdbo_sqltype SQL本体のSQLタイプ
 * 				DIVY_SQL_TYPE_BIND ならばsqlmodelistはNULLです
 * @param subname const char * WHERE句プロパティに指定されたSQLのサブ名称
 * @param has_default_val int デフォルト値を持つかどうか
 * 			DIVY_SQL_NO_DEFAULT_VAL  : 持たない
 * 			DIVY_SQL_HAS_DEFAULT_VAL : 持つ
 * @param sqlmodelist divy_rdbo_sqlmode ** 取得したモードからなる配列
 * 			終端の要素はDIVY_SQL_MODE_UNKNOWNになっている
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK	 : 成功
 * 	DIVY_SQLP_ST_ERR : 予期しないエラー
 */
DIVY_DECLARE(int) divy_sql_parser_get_sqlmodelist(divy_sql_parser *parser,
						 divy_rdbo_sqltype type,
						 const char *subname,
						 int has_default_val,
						 divy_rdbo_sqlmode **sqlmodelist);

/**
 * 指定されたサブ名称のSQLがシステムで特別に予約されたものであるかどうかを判定する。
 * (note)
 * 	システムで特別に予約されたSQLとは、クライアント・サーバ間で意識している
 * 	SQLのことを指します。
 *
 * @param parser divy_sql_parser * パーサを表す構造体
 * @param subname const char * 調べるSQLのサブ名称
 * @return int 予約されたSQLかどうか
 *	DIVY_SQLP_ST_RESERVED_SQL   : 予約されたSQLであった
 *	DIVY_SQLP_ST_UNRESERVED_SQL : 予約されていないSQLであった
 *	DIVY_SQLP_ST_ERR            : 予期しないエラー
 */
DIVY_DECLARE(int) divy_sql_parser_is_special_reservedsql(divy_sql_parser *parser,
							const char * subname);

/**
 * サブ名称subname とsqluri, sqltype との組み合わせ検証
 * (note)
 * 	この関数は、クライアントから入力されたサブ名称の検証に使用します。
 * 	それ以外の用途には不向きなので使用しないで下さい。(サブルーチン的な
 * 	性質を持っているからです。)
 *
 * @param parser divy_sql_parser * パーサを表す構造体
 * @param r request_rec *
 * @param sqltype divy_rdbo_sqltype このサブ名称をもつSQLの種類
 * @param sqluri const char * このサブ名称をもつSQLの相対URI
 * @param isnew int このサブ名称をもつSQLは新規登録であるとマークされているか
 * @param subname const char * 検証するサブ名称
 * @return int 検証結果
 * 	DIVY_SQLP_ST_OK              : 成功。問題なし
 * 	DIVY_SQLP_ST_ERR             : 予期しないエラーが発生した
 * 	DIVY_SQLP_ST_IERR            : 内部エラー(DB検索に失敗した)
 * 	DIVY_SQLP_ST_MISSING_REQSUB  : RequiredSQL,名前付きバインド変数にサブ名称が未指定
 * 	DIVY_SQLP_ST_WRONG_SUBNAME   : サブ名称の形式が正しくない
 * 	DIVY_SQLP_ST_WRONG_COMB_TYPE : sqltype とサブ名称の組み合わせがおかしい
 * 	DIVY_SQLP_ST_MISMATCH_REQURI : 同名のサブ名称が既にDBに登録されていた
 */
DIVY_DECLARE(int) divy_sql_parser_validate_subname(divy_sql_parser *parser,
						request_rec *r,
						divy_rdbo_sqltype sqltype,
						const char *sqluri,
						int isnew,
						const char *subname);

/**
 * キャッシュモードcachemode とsqltype, sql との組み合わせ検証
 * (note)
 * 	この関数は、クライアントから入力されたキャッシュモードの検証に使用します。
 * 	それ以外の用途には不向きなので使用しないで下さい。(サブルーチン的な
 * 	性質を持っているからです。)
 *
 * @param parser divy_sql_parser * パーサを表す構造体
 * @param sqltype divy_rdbo_sqltype このキャッシュモードをもつSQLの種類
 * @param subname const char * サブ名称
 * @param sql const char * このキャッシュモードを持つSQL
 * @param cachemode divy_rdbo_sqlcachemode 検証対象のSQLキャッシュモード
 * @return int 検証結果
 * 	DIVY_SQLP_ST_OK              : 成功。問題なし
 * 	DIVY_SQLP_ST_ERR             : 予期しないエラーが発生した
 * 	DIVY_SQLP_ST_CACHE_NONBIND   : 名前付きバインド変数以外に「キャッシュする」が指定
 * 	DIVY_SQLP_ST_CACHE_NBIND_VAL : 値のある名前付きバインド変数に「キャッシュする」が指定
 * 	DIVY_SQLP_ST_RESERVED_SQL    : 予約されたSQLに対して「キャッシュする」が指定
 */
DIVY_DECLARE(int) divy_sql_parser_validate_cachemode(divy_sql_parser *parser,
						divy_rdbo_sqltype sqltype,
						const char *subname,
						const char *sql,
						divy_rdbo_sqlcachemode cachemode);


#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_SQLPARSER_H */

