/**
 * $Id$
 *
 * ファイルストレージ管理用の構造体、関数の定義＆宣言
 *
 * (note) このファイル導入の背景
 * 	ここに定義されているものは、元々リポジトリプロバイダ(repos.c) に
 * 	実装されていたもので、これを多少汎用化して外に出しました。
 * 	ストレージを触る処理が多くなったにも関わらず、リポジトリプロバイダが
 * 	それを独占してしまっていたため、それを打開する目的でこのようにしました。
 * 	時間が無かったので単に外だしにしたというイメージが強く、あまり汎用的では
 * 	ありません。
 * 	本当はApacheに依存しないストレージ操作APIを目指していましたが、全ての
 * 	ファイル、ストレージ操作をラップするには尋常ではない修正が必要である
 * 	ため断念しました。
 * 	元々、TeamFileというDAVプロバイダそのものが1つの"DAVストレージ"の実装
 * 	であるという特性もあって、操作を全てラップするのはイコール内部に小さな
 * 	TeamFileを作ると同じ意味を持ってしまうので道程は厳しいでしょう。
 * 	十分は時間が出てきたら誰かチャレンジして下さい。
 *
 */
#ifndef INCLUDE_TF_STORAGE_H
#define	INCLUDE_TF_STORAGE_H

#include "apr.h"
#include "apr_pools.h"
#include "httpd.h"
#include "util_filter.h"

#include "tfr.h"
#include "util_common.h"
#include "tf_linkedlist.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * リソース格納ディレクトリの中に格納される最大ファイル数のデフォルト値
 * (note)
 *   httpd.conf に"TfMaxResourcePerDir" が指定されなかったときに使用されます
 */
#define DIVY_MAX_RESOURCE_PER_DIR 15000

/**
 * 1ディレクトリに格納されるユーザID SEQ名のフォルダ数
 * !!! 1度でもD-IVYサーバが動作してしまったら、変更しないで下さい。!!!
 */
#define DIVY_MAX_USER_PER_DIR  5000

/**
 * divy_pfile_mkfixedtemp が使用する固定ディレクトリの名称
 */
#define DIVY_FIXED_RESDIR_NAME	".tmp"

/**
 * 一時ファイル名の生成に使用するテンプレート
 */
#define DIVY_TMP_FILE_TEMPLATE_NAME "tmp_XXXXXX"

/**
 * パス区切り文字
 */
#ifdef WIN32
#define PATH_DELIMITER_CH	'\\'
#else	/* !WIN32 */
#define PATH_DELIMITER_CH	'/'
#endif	/* WIN32 */

/*------------------------------------------------------------------------------
  Declare structure/enum
  ----------------------------------------------------------------------------*/
/**
 * 1つのストレージ(仮想的なDisk)を表す構造体 [ 不完全型 ]
 * [ 役割 ]
 * 	* 物理ストレージ情報を保持する
 *
 * ストレージ関連のコンフィグ情報を保持している構造体です。
 * ユーザSEQなどといった個々のファイル毎に異なる情報は保持しません。
 */
typedef struct __divy_fstorage_t	divy_fstorage_t;

/**
 * 物理ファイルの抽象パスを表す構造体 [ 不完全型 ]
 *
 * あくまでも"抽象パス表現"ですので、この構造体を作成したからといって、
 * 物理ファイルが作成される訳ではありません。これは、apr_file_t とは
 * 異なります。
 */
typedef struct __divy_pfile_t		divy_pfile_t;

/**
 * 物理ディレクトリの抽象パスを表す構造体 [ 不完全型 ]
 *
 * divy_pfile_t の親ディレクトリを表現する目的で使用しています。
 * 通常イメージするようなファイルコンテナとしての役割は持っていません。
 * 単なるpfile の親情報という程度です。
 */
typedef struct __divy_pdir_t		divy_pdir_t;

/*
 * ある1つのリソースをコピーするに必要な情報を保持する構造体
 */
typedef struct __divy_copyfile_info	divy_copyfile_info;

/*------------------------------------------------------------------------------
  Define structure/enum
  ----------------------------------------------------------------------------*/
/**
 * ファイルストレージ関数が返却するステータスを表す定数
 */
enum {
	DIVY_FS_ST_OK = 0,		/* 成功 */
	DIVY_FS_ST_LESSTHAN,		/* a < b (divy_pfile_compare で使用) */
	DIVY_FS_ST_EQUAL,		/* a = b (divy_pfile_compare で使用) */
	DIVY_FS_ST_MORETHAN,		/* a > b (divy_pfile_compare で使用) */
	DIVY_FS_ST_CLOSED,		/* 既にストレージは閉じられていた    */
	DIVY_FS_ST_INVALID_PARAM,	/* 引数不正 */
	DIVY_FS_ST_INVALID_CONFIG,	/* コンフィグファイル不正 */
	DIVY_FS_ST_FORMAT_MISSMATH,	/* 形式が一致しなかった */
	DIVY_FS_ST_FORMAT_SPECIAL,	/* 特殊形式だった */
	DIVY_FS_ST_MPTYPE_MISSMATCH,	/* mptype と呼び出し関数の関係が正しくない */
	DIVY_FS_ST_DIFFERENT_ROOT,	/* ルートパスが異なっていた */
	DIVY_FS_ST_ERR,			/* 失敗(詳細な理由は不明) */
};

/**
 * ある1つのリソースをコピーするに必要な情報を保持する構造体
 * [ 歴史 ]
 * 	かつてこの構造体は、tf_rdbo.h に定義されていました。ですが、
 * 	それではここから参照し難い状況だったので、定義をこちらに
 * 	移動してしまいました。不恰好ですが、変更を最小限に抑えるため
 * 	仕方ありません。
 */
struct __divy_copyfile_info {
	char *src_relativepath;		/* src の相対物理パス */
	char *dst_relativepath;		/* dst の相対物理パス */

	divy_pfile_t *src_pfile;
	divy_pfile_t *dst_pfile;

	/* 以下作業用の情報 */
	apr_int64_t getcontentlength;
	int isfile;			/* ファイルかどうか(1: ファイル) */
	char *rsid;			/* dst のリソースID (作業用) */

#ifdef DIVY_SUPPORT_PREFLIGHT_PLUGIN
	const char *dispname;
#endif	/* DIVY_SUPPORT_PREFLIGHT_PLUGIN */
	divy_copyfile_info *next;	/* 次へ */
};


/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * ファイルストレージを表す構造体のインスタンスを生成し、ファイルストレージが
 * "開いた"とみなせる状態にする。(ファイルストレージが開いていれば、いつでも
 * ファイル操作が可能である)
 *
 * @param r request_rec * コンフィグを保持するレコード構造体
 * @param p apr_pool_t * 作業用のプール (*fstorage とライフサイクルを共にする)
 * @param fstorage divy_fstorage_t ** ファイルストレージを表す構造体へのポインタ
 * @return int 処理ステータス
 * 	DIVY_FS_ST_OK             : 成功
 * 	DIVY_FS_ST_INVALID_PARAM  : 引数がNULLだった
 * 	DIVY_FS_ST_INVALID_CONFIG : コンフィグファイルのエントリ(fsrootpath)が不正
 * 	DIVY_FS_ST_ERR            : 予期しない失敗
 */
DIVY_DECLARE(int) divy_fstorage_open(request_rec *r, apr_pool_t *p,
						divy_fstorage_t **fstorage);

/**
 * ファイルストレージを"閉じた"とみなせる状態にする。
 *
 * @param fstorage divy_fstorage_t * ファイルストレージを表す構造体へのポインタ
 * @return int 処理ステータス。常にDIVY_FS_ST_OKを返します。
 */
DIVY_DECLARE(int) divy_fstorage_close(divy_fstorage_t *fstorage);


/**
 * 一時ファイルの"抽象パス表現"を生成する。
 * (note)
 * 	* この関数は「URI階層と物理階層が一致"しない"構造」の場合のみに意味を持ちます。
 * 	  それ以外のケースでは使用しても意味がありませんし有害です。
 *
 * [ 一時ファイルパスの一意性 ]
 *   相異なるパス生成要求に対して返却されるファイルパスの一意性は
 *   以下の条件を1つ以上満たす場合に保証される
 *
 *   * アクセスユーザが異なる
 *   * アクセスプロセスが異なる
 *   * アクセスサーバのホスト名が異なる
 *   * アクセス時間が異なる
 *   * 一時ファイルを作成するパスの親が異なる
 *
 * ただし、同時アクセスに対して異なる名称を出力しない保証はありません。
 * 同一ファイルが存在するかどうかチェックしていないためです。
 * これを保証するには、この関数を使って得られた文字列を関数 apr_file_mktemp に
 * 渡すなどとすればよいでしょう。この場合、suffix には、apr_file_mktemp に渡す
 * 文字列パターンを記述して下さい。
 *
 * (note)
 * 	ファイルの実体が作られることはありませんが、そのファイルが格納される
 * 	であろう物理ディレクトリは仮想ストレージの種類によっては
 * 	実際に作成されることがあります。意識する必要はありませんが、実体が
 * 	作成されることで問題があるオペレーションでは気をつけてください。
 *
 * @param fstorage divy_fstorage_t * ファイルストレージを表す構造体へのポインタ
 * @param p apr_pool_t * 作業用のプール (*pfile とライフサイクルを共にする)
 * @param suffix const char * 一時ファイルの末尾につく名前
 * @param pfile divy_pfile_t ** 生成した抽象パス表現
 * @return int 処理ステータス (DIVY_FS_ST_OK: 成功 / 上記以外: 失敗)
 *	DIVY_FS_ST_OK               : 正常
 *	DIVY_FS_ST_INVALID_PARAM    : fstorage, suffix がNULLだった(引数不正)
 *	DIVY_FS_ST_CLOSED           : fstorage は既にclose されていた
 *	DIVY_FS_ST_MPTYPE_MISSMATCH : use_mptype と fstorage のマッピングタイプが不一致
 * 	DIVY_FS_ST_ERR              : 予期しない失敗
 */
DIVY_DECLARE(int) divy_pfile_mktemp(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *suffix, divy_pfile_t **pfile);

/**
 * 一時ファイルの"抽象パス表現"を生成する。(URI版)
 * (note)
 * 	* この関数は「URI階層と物理階層が一致"する"構造」の場合のみに意味を持ちます。
 * 	  それ以外のケースでは使用しても意味がありませんし有害です。
 *
 * @param fstorage divy_fstorage_t * ファイルストレージを表す構造体へのポインタ
 * @param p apr_pool_t * 作業用のプール (*pfile とライフサイクルを共にする)
 * @param uri const char * 対象物理ファイルのURI(自身を含む)
 * @param suffix const char * 一時ファイルの末尾につく名前
 * @param pfile divy_pfile_t ** 生成した抽象パス表現
 * @return int 処理ステータス (DIVY_FS_ST_OK: 成功 / 上記以外: 失敗)
 */
DIVY_DECLARE(int) divy_pfile_mktempByUri(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *uri, const char *suffix,
					divy_pfile_t **pfile);

/**
 * 一時ファイルの"抽象パス表現"を生成する。(autoindex, サーバ主体メール専用)
 *
 * 関数divy_pfile_mktemp との違いは、常に同じ親ディレクトリを一時ファイル
 * 格納場所として返却する点です。
 * この関数は、サーバが内部で一時的に出力するファイルの格納場所を示す目的で
 * のみ使用して下さい。
 * 正式にPUTされてきたファイルの一時格納場所に使用しますと、問題を引き起こします。
 *
 * (note)
 * 	* この関数は「URI階層と物理階層が一致"しない"構造」の場合のみに意味を持ちます。
 * 	  それ以外のケースでは使用しても意味がありませんし有害です。
 *
 * @param fstorage divy_fstorage_t * ファイルストレージを表す構造体へのポインタ
 * @param p apr_pool_t * 作業用のプール (*pfile とライフサイクルを共にする)
 * @param suffix const char * 一時ファイルの末尾につく名前
 * @param pfile divy_pfile_t ** 生成した抽象パス表現
 * @return int 処理ステータス (DIVY_FS_ST_OK: 成功 / 上記以外: 失敗)
 *	DIVY_FS_ST_OK               : 正常
 *	DIVY_FS_ST_INVALID_PARAM    : fstorage, suffix がNULLだった(引数不正)
 *	DIVY_FS_ST_CLOSED           : fstorage は既にclose されていた
 *	DIVY_FS_ST_MPTYPE_MISSMATCH : use_mptype と fstorage のマッピングタイプが不一致
 * 	DIVY_FS_ST_ERR              : 予期しない失敗
 */
DIVY_DECLARE(int) divy_pfile_mkfixedtemp(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *suffix, divy_pfile_t **pfile);


/**
 * 指定されたpath の"抽象パス表現"を生成する。
 * (note)
 * 	ファイルの実体が作られることはありませんが、そのファイルが格納される
 * 	であろう物理ディレクトリは仮想ストレージの種類によっては
 * 	実際に作成されることがあります。意識する必要はありませんが、実体が
 * 	作成されることで問題があるオペレーションでは気をつけてください。
 *
 * @param fstorage divy_fstorage_t * ファイルストレージを表す構造体へのポインタ
 * @param p apr_pool_t * 作業用のプール (*pfile とライフサイクルを共にする)
 * @param relativepath const char * 物理パス名の文字列表現(ルートパスからの相対パス)
 * @param pfile divy_pfile_t ** 生成した抽象パス表現
 * @return int 処理ステータス
 *	DIVY_FS_ST_OK               : 正常
 *	DIVY_FS_ST_INVALID_PARAM    : fstorage, relativepath がNULLだった(引数不正)
 *	DIVY_FS_ST_CLOSED           : fstorage は既にclose されていた
 *	DIVY_FS_ST_MPTYPE_MISSMATCH : use_mptype と fstorage のマッピングタイプが不一致
 * 	DIVY_FS_ST_ERR              : 予期しない失敗
 */
DIVY_DECLARE(int) divy_pfile_create(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *relativepath,
					divy_pfile_t **pfile);

/**
 * 指定されたURI文字列から"抽象パス表現"を生成する。
 * (note)
 * 	* この関数は「URI階層と物理階層が一致"する"構造」の場合のみに意味を持ちます。
 * 	  それ以外のケースでは使用しても意味がありませんし有害です。
 * 	* 実際にファイルが作られることはありません。"抽象パス"を算出するだけです
 *
 * @param fstorage divy_fstorage_t * ファイルストレージを表す構造体へのポインタ
 * @param p apr_pool_t * 作業用のプール (*pfile とライフサイクルを共にする)
 * @param uri const char * URIの文字列表現
 * @param pfile divy_pfile_t ** 生成した抽象パス表現
 * @return int 処理ステータス (DIVY_FS_ST_OK: 成功 / 上記以外: 失敗)
 */
DIVY_DECLARE(int) divy_pfile_createByUri(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *uri, divy_pfile_t **pfile);

/**
 * ソースとディスティネーションの情報を持つcopyinfo から"抽象パス表現"を生成する。
 * 生成したdivy_pfile_t は、copyinfo に詰めて返却します。
 *
 * (note)
 * 	* この関数は、TeamFileの物理ファイルコピー仕様を非常に強く意識しています。
 * 	  それ以外の用途では使い道すらわからないはず。用途外の使用は禁止。
 * 	* ファイルの実体が作られることはありませんが、そのファイルが格納される
 * 	  であろう物理ディレクトリは仮想ストレージの種類によっては
 * 	  実際に作成されることがあります。意識する必要はありませんが、実体が
 * 	  作成されることで問題があるオペレーションでは気をつけてください。
 *
 *
 * @param fstorage divy_fstorage_t * ファイルストレージを表す構造体へのポインタ
 * @param p apr_pool_t * 作業用のプール (*pfile とライフサイクルを共にする)
 * @param copyinfo divy_copyfile_info * コピー情報
 * @return int 処理ステータス
 *	DIVY_FS_ST_OK               : 正常
 *	DIVY_FS_ST_INVALID_PARAM    : fstorage, copyinfo がNULLだった(引数不正)
 *	DIVY_FS_ST_CLOSED           : fstorage は既にclose されていた
 *	DIVY_FS_ST_MPTYPE_MISSMATCH : use_mptype と fstorage のマッピングタイプが不一致
 * 	DIVY_FS_ST_ERR              : 予期しない失敗
 */
DIVY_DECLARE(int) divy_pfile_createByCopyinfo(divy_fstorage_t *fstorage, apr_pool_t *p,
					divy_copyfile_info *copyinfo);

/**
 * 指定されたpfile1 とpfile2 の抽象パス表現が等しいかどうか判定する。
 * (note)
 * 	取得した文字列の内容を書き換えてもpfile の内部表現を変更することは
 * 	出来ません。所定の手続き(divy_pfile_renameなど)を利用して下さい。
 *
 * @param pfile1 divy_pfile_t * 抽象パス表現へのポインタ
 * @param pfile2 divy_pfile_t * 抽象パス表現へのポインタ
 * @return int 処理ステータス
 * 	DIVY_FS_ST_LESSTHAN : pfile1 < pfile2 (抽象パスの長さが小さい)
 * 	DIVY_FS_ST_EQUAL    : pfile1 = pfile2
 * 	DIVY_FS_ST_MORETHAN : pfile1 > pfile2
 * 	DIVY_FS_ST_ERR      : 予期しない失敗
 */
DIVY_DECLARE(int) divy_pfile_compare(divy_pfile_t *pfile1, divy_pfile_t *pfile2);

/**
 * 指定されたpfile の名前(Final Path Segment) をname に変更する。
 *
 * @param pfile divy_pfile_t * 抽象パス表現へのポインタ
 * @param name const char * 変更後の名称
 * @return int 処理ステータス
 *	DIVY_FS_ST_OK               : 正常
 *	DIVY_FS_ST_INVALID_PARAM    : name がNULLだった(引数不正)
 * 	DIVY_FS_ST_ERR              : 予期しない失敗
 */
DIVY_DECLARE(int) divy_pfile_rename(divy_pfile_t *pfile, const char *name);

/**
 * 指定されたpfile の相対パスをrelativepath に変更する。
 * また、pfile を開いたfstorageがNULLであればこれも失敗します。
 * (note)
 * 	相対パスとは、ストレージルートパスを含まないパス文字列のことです。
 *
 * @param pfile divy_pfile_t * 抽象パス表現へのポインタ
 * @param relativepath const char * 変更後の相対パス
 * @return int 処理ステータス
 *	DIVY_FS_ST_OK               : 正常
 *	DIVY_FS_ST_INVALID_PARAM    : relativepath がNULLだった(引数不正)
 *	DIVY_FS_ST_DIFFERENT_ROOT   : relativepath とpfile のルートパスが異なる
 * 	DIVY_FS_ST_ERR              : 予期しない失敗
 */
DIVY_DECLARE(int) divy_pfile_relativemove(divy_pfile_t *pfile, const char *relativepath);

/**
 * 指定されたpfile のフルパスをfullpath に変更する。
 * 但し、fullpath のルートパスとpfile のルートパスが異なる場合には
 * 変更は失敗します。
 * また、pfile を開いたfstorageがNULLであればこれも失敗します。
 *
 * @param pfile divy_pfile_t * 抽象パス表現へのポインタ
 * @param fullpath const char * 変更後のフルパス
 * @return int 処理ステータス
 *	DIVY_FS_ST_OK               : 正常
 *	DIVY_FS_ST_INVALID_PARAM    : fullpath がNULLだった(引数不正)
 *	DIVY_FS_ST_DIFFERENT_ROOT   : fullpath とpfile のルートパスが異なる
 * 	DIVY_FS_ST_ERR              : 予期しない失敗
 */
DIVY_DECLARE(int) divy_pfile_fullmove(divy_pfile_t *pfile, const char *fullpath);


/*
 * 以下はアクセサの宣言
 */

/**
 * 指定されたpfile のフルパスを取得する。
 * (note)
 * 	取得した文字列の内容を書き換えてもpfile の内部表現を変更することは
 * 	出来ません。所定の手続き(divy_pfile_renameなど)を利用して下さい。
 *
 * @param pfile divy_pfile_t * 抽象パス表現へのポインタ
 * @return const char * フルパスを表す文字列。
 * 			pfile がNULLならばNULLを返却します。
 */
DIVY_DECLARE(const char *) divy_pfile_get_fullpath(divy_pfile_t *pfile);

/**
 * 指定されたpfile の相対パス(ストレージルートを除いたパス)を取得する。
 * (note)
 * 	取得した文字列の内容を書き換えてもpfile の内部表現を変更することは
 * 	出来ません。所定の手続き(divy_pfile_renameなど)を利用して下さい。
 *
 * @param pfile divy_pfile_t * 抽象パス表現へのポインタ
 * @return const char * 相対パスを表す文字列。
 * 			pfile がNULLならばNULLを返却します。
 */
DIVY_DECLARE(const char *) divy_pfile_get_relativepath(divy_pfile_t *pfile);

/**
 * 指定されたpfile の名前(final path segment)を取得する。
 * (note)
 * 	取得した文字列の内容を書き換えてもpfile の内部表現を変更することは
 * 	出来ません。所定の手続き(divy_pfile_renameなど)を利用して下さい。
 *
 * @param pfile divy_pfile_t * 抽象パス表現へのポインタ
 * @return const char * 名前(final path segment)を表す文字列。
 * 			pfile がNULLならばNULLを返却します。
 */
DIVY_DECLARE(const char *) divy_pfile_get_name(divy_pfile_t *pfile);

/**
 * 指定されたpfile の仮想ストレージを取得する。
 *
 * @param pfile divy_pfile_t * 抽象パス表現へのポインタ
 * @return divy_fstorage_t * ストレージへのポインタ
 * 			pfile がNULLならばNULLを返却します。
 */
DIVY_DECLARE(divy_fstorage_t *) divy_pfile_get_fstorage(divy_pfile_t *pfile);

/**
 * 指定されたpfile の親ディレクトリのフルパスを取得する。
 * (note)
 * 	取得した文字列の内容を書き換えてもpfile の内部表現を変更することは
 * 	出来ません。
 *
 * @param pfile divy_pfile_t * 抽象パス表現へのポインタ
 * @return const char * 親ディレクトリのフルパスを表す文字列。
 * 			pfile がNULLならばNULLを返却します。
 */
DIVY_DECLARE(const char *) divy_pfile_get_pfullpath(divy_pfile_t *pfile);

/**
 * 指定されたpfile の親ディレクトリの相対パス(ストレージルートを除いたパス)を
 * 取得する。
 * (note)
 * 	取得した文字列の内容を書き換えてもpfile の内部表現を変更することは
 * 	出来ません。
 *
 * @param pfile divy_pfile_t * 抽象パス表現へのポインタ
 * @return const char * 親ディレクトリの相対パスを表す文字列。
 * 			pfile がNULLならばNULLを返却します。
 */
DIVY_DECLARE(const char *) divy_pfile_get_prelativepath(divy_pfile_t *pfile);


/*
 * 以下はユーティリティ
 */

/**
 * fullpath が示す物理ファイルパスから名称パート(final path segment) を取り出して返却する。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param fullpath const char * 物理ファイルのフルパス
 * @return const char * 名称パート。fullpath がNULLだったり、取得に失敗したらNULL。
 * 			「/」だけの場合には、NULLになります。
 */
DIVY_DECLARE(const char *) divy_extract_finalpath(apr_pool_t *p, const char *fullpath);

/**
 * fullpath が示す物理ファイルパスから親ディレクトリのパスを取り出して返却する。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param fullpath const char * 物理ファイルのフルパス
 * @return const char * 親ディレクトリパス。fullpath がNULLだったり、取得に失敗したらNULL。
 * 			「/」だけの場合には、NULLになります。
 */
DIVY_DECLARE(const char *) divy_extract_parentath(apr_pool_t *p, const char *fullpath);

/**
 * relativepath の絶対パスをfstorage のストレージ上に作成する。
 * (note)
 * 	絶対パス文字列を合成するだけです。実体が作成されることはありません。
 *
 * @param fstorage divy_fstorage_t ** ファイルストレージを表す構造体へのポインタ
 * @param relativepath const char * 相対パス
 * @return const char * 作成された絶対パス。fstorage がNULLならばNULL。
 * 			relativepath がNULL/空ならばfstorageのルートパス。
 */
DIVY_DECLARE(const char *) divy_make_fullpath(divy_fstorage_t *fstorage,
						const char *relativepath);

/**
 * 一時ファイル名称を組み立てて返却する
 *
 * @param fstorage divy_fstorage_t ** ファイルストレージを表す構造体へのポインタ
 * @return const char * 組み立てられた一時ファイル名
 * 			fstorage がNULLならばNULLを返す.
 */
DIVY_DECLARE(const char *) divy_make_temporaryname(divy_fstorage_t *fstorage,
							const char *suffix);

/**
 * サイズがgetcontentlength バイトのfilepath が示すファイルを
 * output ストリームに流す。
 * (note)
 * 	この関数は、かつてrepos.c で_download_file とよばれていた関数を
 * 	Publicにしたものです。
 *
 * @param r request_rec * リクエスト構造体
 * @param output ap_filter_t* outputフィルタchainを表す構造体へのポインタ
 * @param getcontentlength apr_int64_t Liveプロパティのファイルサイズ
 * 					-1 ならば無視します
 * @param filepath const char * ファイルのフルパス
 * @return int 処理ステータス※Apacheのステータスコードです。
 * 	OK : 成功
 */
DIVY_DECLARE(int) divy_sendfile2stream(request_rec *r,
					ap_filter_t *output,
					apr_int64_t getcontentlength,
					const char *filepath);

/**
 * pathlist が示す物理ファイルをストレージから削除する。
 *
 * @param r request_rec * リクエスト構造体
 * @param pathlist divy_linkedlist_t * ファイルパスからなるリスト
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_remove_physical_files(request_rec *r,
					divy_linkedlist_t *pathlist);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_STORAGE_H */

