/**
 * $Id$
 *
 * サムネイルファイル周りの操作を行う関数、構造体の定義&宣言
 *
 * (note)
 * 	サムネイル関連の関数が散らばってしまうので、一箇所に纏めました。
 * 	機能単位に纏め上げるのは、あまり良いとは言えませんが、サムネイル
 * 	特有の処理や定義が分散してしまうよりはマシという考えからこのように
 * 	しました。
 */
#ifndef INCLUDE_TF_THUMBNAIL_H
#define INCLUDE_TF_THUMBNAIL_H

#include "apr.h"
#include "apr_pools.h"

#include "tfr.h"
#include "util_common.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * エラーコード
 */
#define DIVY_THMNL_ST_OK	0	/* 正常 */
#define DIVY_THMNL_ST_INVALID_PARAM 1	/* パラメータ不正 */
#define DIVY_THMNL_ST_NOTSUPPORTED  2	/* サムネイル機能はサポートされていない */
#define DIVY_THMNL_ST_NOTFOUND	10	/* サムネイルファイルが見つからなかった */
#define DIVY_THMNL_ST_READERR	11	/* サムネイルファイルの読み込みに失敗   */
#define DIVY_THMNL_ST_WRITEERR	12	/* サムネイルファイルへの書き込みに失敗 */
#define DIVY_THMNL_ST_REMOVEERR	13	/* サムネイルファイルの削除に失敗       */
#define DIVY_THMNL_ST_COPYERR	14	/* サムネイルファイルのコピーに失敗     */
#define DIVY_THMNL_ST_NOREMOVEDIR	15	/* 削除する必要なし */

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * 抽出元イメージの相対パスrelativepath からサムネイルファイルを算出し、
 * 存在すれば、サムネイルデータを読み込んで文字列として返却する.
 *
 * @param p apr_pool_t * 出力結果を割り当てるプール
 * @param r request_rec *
 * @param relativepath const char * 抽出元イメージの相対パス
 * @param data char ** 読み込んだデータ(NULL終端ではありません)
 * @param nreadbyte apr_size_t * 読み込んだデータのサイズ(byte単位)
 * @return int 処理ステータス
 * 		DIVY_THMNL_ST_OK       : 正常にデータが取得できた
 * 		DIVY_THMNL_ST_NOTFOUND : サムネイルファイルがなかった
 * 		DIVY_THMNL_ST_READERR  : サムネイルファイルの読み込みに失敗
 * 		DIVY_THMNL_ST_INVALID_PARAM : パラメータ不正
 * 		DIVY_THMNL_ST_NOTSUPPORTED  : サムネイル機能をサポートしていない
 */
DIVY_DECLARE(int) divy_thumbnail_read(apr_pool_t *p, request_rec *r,
                                       const char *relativepath,
                                       char **data,
                                       apr_size_t *nreadbyte);

/**
 * 抽出元イメージの相対パスrelativepath からサムネイルファイルパスを算出し、
 * 指定されたサムネイルデータdata を書き込む.
 *
 * @param p apr_pool_t * 出力結果を割り当てるプール
 * @param r request_rec *
 * @param relativepath const char * 抽出元イメージの相対パス
 * @param data char * 書き込むデータ(NULL終端ではありません)
 * @param nwritebyte apr_size_t 書き込むデータサイズ(byte単位)
 * @return int 処理ステータス
 * 		DIVY_THMNL_ST_OK       : 書き込み成功
 * 		DIVY_THMNL_ST_WRITEERR : 書き込み失敗
 * 		DIVY_THMNL_ST_INVALID_PARAM : パラメータ不正
 * 		DIVY_THMNL_ST_NOTSUPPORTED  : サムネイル機能をサポートしていない
 */
DIVY_DECLARE(int) divy_thumbnail_write(apr_pool_t *p, request_rec *r,
                                        const char *relativepath,
                                        char *data,
                                        apr_size_t nwritebyte);

/**
 * 抽出元イメージの相対パスrelativepath からサムネイルファイルパスを算出し、
 * このファイルの持つサムネイルファイルを物理的に削除する.
 * (サムネイル機能をサポートしていなくてもこの関数はコールできます)
 *
 * @param p apr_pool_t * 出力結果を割り当てるプール
 * @param r request_rec *
 * @param relativepath const char * 抽出元イメージの相対パス
 * @return int 処理ステータス
 * 		DIVY_THMNL_ST_OK        : 削除成功
 * 		DIVY_THMNL_ST_REMOVEERR : 削除失敗
 * 		DIVY_THMNL_ST_INVALID_PARAM : パラメータ不正
 */
DIVY_DECLARE(int) divy_thumbnail_remove(apr_pool_t *p, request_rec *r,
                                        const char *relativepath);

/**
 * 抽出元イメージが格納された親フォルダのパスpfullpath から
 * サムネイルファイルパスを算出し、このサムネイルパスの親ディレクトリが
 * 削除できれば削除する.
 * (サムネイル機能をサポートしていなくてもこの関数はコールできます)
 *
 * @param p apr_pool_t * 出力結果を割り当てるプール
 * @param r request_rec *
 * @param pfullpath const char * 抽出元イメージの親パスの絶対パス
 * @return int 処理ステータス
 * 		DIVY_THMNL_ST_OK        : 削除成功
 * 		DIVY_THMNL_ST_REMOVEERR : 削除失敗
 * 		DIVY_THMNL_ST_INVALID_PARAM : パラメータ不正
 * 		DIVY_THMNL_ST_NOREMOVEDIR : 中身があったので削除しなかった(成功扱い)
 * 		DIVY_THMNL_ST_NOTSUPPORTED  : サムネイル機能をサポートしていない
 */
DIVY_DECLARE(int) divy_thumbnail_removepdir(apr_pool_t *p, request_rec *r,
                                            const char *pfullpath);

/**
 * ソースの抽出元イメージを相対パスsrc_relativepath から dst_relativepath に
 * コピーするとき、サムネイルファイルもコピーする.
 *
 * @param p apr_pool_t * 出力結果を割り当てるプール
 * @param r request_rec *
 * @return int 処理ステータス
 * 		DIVY_THMNL_ST_OK        : コピー成功
 * 		DIVY_THMNL_ST_COPYERR   : コピー失敗
 * 		DIVY_THMNL_ST_INVALID_PARAM : パラメータ不正
 * 		DIVY_THMNL_ST_NOTSUPPORTED  : サムネイル機能をサポートしていない
 */
DIVY_DECLARE(int) divy_thumbnail_copy(apr_pool_t *p, request_rec *r,
                                       const char *src_relativepath,
                                       const char *dst_relativepath);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_THUMBNAIL_H */

