/**
 * $Id$
 *
 * tf_validator.h
 *
 * 入力値、アクセス制御の検証を行うのに使用する関数、構造体を宣言、定義するヘッダ
 * [ 導入理由 ]
 *   * 入力値の検証ロジックが実装上、色々なファイル上に散在してしまい、結果、
 *     新しい特殊フォルダを導入する際などに、修正箇所が不明確であった。
 *
 *   * うまくすれば使い回しできる検証ロジックがファイルが別であるため、再利用しない
 *     若しくは無理矢理共通化していたため、ロジックがわかりにくくなっていた。
 *
 *   * 検証ロジックが多すぎるためこのロジックだけで各種DAVプロバイダの実装が膨れて
 *     しまい、メインロジックの見通しが悪くなっていた
 *
 * 結局、入力値の検証ロジックを1ファイルに収めて、多少共通化しただけなので
 * それ程便利になったとは言えません。もっとよい方法があれば、そうして下さい。
 */
#ifndef INCLUDE_TF_VALIDATOR_H
#define INCLUDE_TF_VALIDATOR_H

#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"

#include "mod_dav.h"

#include "tf_env.h"
#include "mod_dav_tf.h"
#include "util.h"
#include "tf_rdbo.h"
#include "tf_folder.h"
#include "tf_parser.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define Fixed values and macro
  ----------------------------------------------------------------------------*/
/*
 * divy_validate_xxx の chk_mode の値
 */
#define DIVY_VALIDATE_INSERT	0x01	/* 登録時の検証 */
#define DIVY_VALIDATE_UPDATE	0x02	/* 更新時の検証 */
#define DIVY_VALIDATE_DELETE	0x04	/* 削除時の検証 */
#define DIVY_VALIDATE_COPY		0x08	/* コピー時の検証 */
#define DIVY_VALIDATE_LDAP_UPDATE		0x10	/* LDAP経由の更新の検証 */

/*
 * 関数 divy_validate_pathsegment専用
 * not_allowed_set に指定されている値を無視する時に使用するフラグ値
 */
#define DIVY_IGNORE_BS       0x01		/* '\' を無視 */
#define DIVY_IGNORE_SLASH    0x02		/* '/' を無視 */
#define DIVY_IGNORE_COLON    0x04		/* ':' を無視 */
#define DIVY_IGNORE_ASTERISK 0x08		/* '*' を無視 */
#define DIVY_IGNORE_QUESTION 0x10		/* '?' を無視 */
#define DIVY_IGNORE_DQUOTE   0x20		/* '"' を無視 */
#define DIVY_IGNORE_LT       0x40		/* '<' を無視 */
#define DIVY_IGNORE_GT       0x80		/* '>' を無視 */
#define DIVY_IGNORE_BAR      0x100		/* '|' を無視 */

/**
 * SQLプロパティの検証処理において、DB 未登録のSQLがもつRequiredの
 * 循環参照チェック時に、そのSQL が直接もつRequiredのparentとして
 * 仮に使用する値
 */
#define DIVY_SQL_TOPPTVALUE "tmpparentvalue"

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * 猶予期間(日) を秒数に直すマクロ
 */
#define DIVY_PASSPOLICY_PROBATION_SEC(d)	(d*24*3600)

/**
 * パスワード有効期限(日) を秒数に直すマクロ
 */
#define DIVY_PASSPOLICY_PASSPERIOD_SEC(d)	(d*24*3600)
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/*------------------------------------------------------------------------------
  Define enum and structure
  ----------------------------------------------------------------------------*/
#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * 関数 divy_set_resultstatus で利用するヘッダの種別を表す値.
 */
typedef enum divy_resultstatus_type divy_resultstatus_type;
enum divy_resultstatus_type {
	DIVY_RESULTSTATUS_TYPE_UNKNOWN = 0,	/* (未定義) */
	DIVY_RESULTSTATUS_TYPE_CHANGENOW,	/* パスワード変更が直ぐに必要 */	
	DIVY_RESULTSTATUS_TYPE_FIRST_PROBATION,	/* 初回ログイン時の猶予期間 */
	DIVY_RESULTSTATUS_TYPE_FIRST_EXPIRED,	/* 初回ログイン時の期限切れ */
	DIVY_RESULTSTATUS_TYPE_PROBATION,		/* 猶予期間 */
	DIVY_RESULTSTATUS_TYPE_EXPIRED,			/* 期限切れ */
	DIVY_RESULTSTATUS_TYPE_INACTIVE_GROUP,	/* 対象グループは非アクティブ */
	DIVY_RESULTSTATUS_TYPE_NODELETE_GROUPLEADER,	/* 子ユーザが居るのでリーダを削除できない */
	DIVY_RESULTSTATUS_TYPE_NODEMOTE_GROUPLEADER,	/* 子ユーザが居るのでリーダを降格できない */
	DIVY_RESULTSTATUS_TYPE_NOCHANGE_ADMINTYPE,		/* グループリーダは子ユーザの種類を上位に変更できない */
	DIVY_RESULTSTATUS_TYPE_NOCHANGE_OTHERUSER,		/* グループリーダはOtherユーザのプロパティを変更不可 */
	DIVY_RESULTSTATUS_TYPE_NODELETE_OTHERUSER,		/* グループリーダはOtherユーザを削除できない */
	DIVY_RESULTSTATUS_TYPE_NOCREATE_TOPGROUP,		/* グループリーダはトップグループを作成できない */
	DIVY_RESULTSTATUS_TYPE_NOCHANGE_TOPGROUP,		/* グループリーダはトップグループを編集できない */
	DIVY_RESULTSTATUS_TYPE_NODELETE_TOPGROUP,		/* グループリーダはトップグループを削除できない */
	DIVY_RESULTSTATUS_TYPE_NOCREATE_SUBGROUP,		/* グループリーダは他人のサブグループを作成できない */
	DIVY_RESULTSTATUS_TYPE_NOCHANGE_SUBGROUP,		/* グループリーダは他人のサブグループを編集できない */
	DIVY_RESULTSTATUS_TYPE_EXCEED_MAXCREATION,		/* グループリーダが最大作成数制限を越えて登録しようとした */
	DIVY_RESULTSTATUS_TYPE_ALREADY_EXISTS_LEADER,	/* 対象グループには既にグループリーダが任命済みだった */
	DIVY_RESULTSTATUS_TYPE_NOCONTROL_OTHERUSER,		/* グループリーダは権限不足により、Otherユーザを引き込めない */
	DIVY_RESULTSTATUS_TYPE_NOCHANGE_APPOINTED_GROUPLEADER, /* 任命済みグループリーダの所属替え / 削除はNG */
	DIVY_RESULTSTATUS_TYPE_CONFIRMREADING_NO_REMOVE_OTHER,	/* 他人の開封通知は解除できません */
	DIVY_RESULTSTATUS_TYPE_CONFIRMREADING_NO_EXISTENCE,	/* 開封通知が存在しないので解除できない */
	DIVY_RESULTSTATUS_TYPE_CONFIRMREADING_ALREADY_EXISTENCE,	/* 開封通知が存在しないので解除できない */
	DIVY_RESULTSTATUS_TYPE_UPLOADPOLICY_RULE_VIOLATION,	/* アップロードポリシー違反 */
	DIVY_RESULTSTATUS_TYPE_2FA_AUTHENTICAITON_REQUIRED,	/* 二段階認証のコードを要求 */
	DIVY_RESULTSTATUS_TYPE_BOX_AUTHENTICATION_FAILED,	/* BOXログイン失敗 */
	DIVY_RESULTSTATUS_TYPE_BOX_INVALID_USER,			/* BOXの許可されたユーザではない */
	DIVY_RESULTSTATUS_TYPE_BOX_DISABLED_WILDCARD_TOMAILADDRESS,	/* BOXの相手先メールアドレスはサーバポリシーにより禁止されています */
	DIVY_RESULTSTATUS_TYPE_SENDMAIL,				/* メールを送信しました */
};
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * 関数 divy_validate_password_term で使用するBodyデータの種類
 */
typedef enum divy_reqbody_type	divy_reqbody_type;
enum divy_reqbody_type {
	DIVY_REQBODY_TYPE_UNKNOWN = 0,	/* 未定義 or BODYなし or BODYは無視してチェック */
	DIVY_REQBODY_TYPE_OTHER,		/* 下記以外 */
	DIVY_REQBODY_TYPE_SEARCH_ALLLIST,	/* SEARCH rootreesearch, alllist */
	DIVY_REQBODY_TYPE_SEARCH_GROUPLIST,	/* SEARCH グループコレクション一覧 */
	DIVY_REQBODY_TYPE_SEARCH_SQLSTMT,	/* SEARCH SQL一覧 (dbilnk) */
	DIVY_REQBODY_TYPE_SEARCH_REPOSSQLSTMT,	/* SEARCH リポジトリSQL一覧(reposdblink) */
	DIVY_REQBODY_TYPE_SEARCH_EXECSQLSTMT,	/* SEARCH 試し実行SQL */
	DIVY_REQBODY_TYPE_SEARCH_USERDETAILLIST, /* SEARCH userinformationsearch detaillist */
	DIVY_REQBODY_TYPE_SEARCH_USERCONTENT, /* SEARCH userinformationsearch detaillist */
};
#endif	/* DIVY_SUPPORT_PASSPOLICY */
/*------------------------------------------------------------------------------
  Declar prototype public functions
  ----------------------------------------------------------------------------*/
/**
 * 指定されたu_spec のURIに対し発行されたメソッドが利用可能かどうかを
 * "必要最低限だけ"検証して返却する。
 * 必要最低限のチェックしか行っていませんので"許可されすぎてしまう"ケースがあります。
 * 必要な検証は各プロバイダや関数 divy_validate_request_resourceで実施して下さい。
 *
 * (歴史的経緯)
 *   このメソッドが導入された当初はここにあるものだけで全てのメソッド、URIに
 *   対する検証が済んでいました。ただ、拡張を繰り返していくうちにそうでは
 *   なくなり、結果、ここでの検証内容だけでは不十分となりました。
 *   それが、"必要最小限のチェック" しか行っていない理由です。
 *
 * @param r request_rec *
 * @param u_spec divy_uri_spec * 解析されたuri の情報。
 * @return dav_error * アクセス可能かどうか(NULL の場合アクセス可能)
 */
DIVY_DECLARE(dav_error *) divy_validate_minimum_access_guarantee(request_rec *r,
							divy_uri_spec *u_spec);

/**
 * 指定された"リクエスト"リソース resource を受け入れることが出来るかどうか検証する。
 * この関数を通す前に、関数divy_validate_minimum_access_guarantee() を使って
 * 必要最小限のチェックを済ませておくこと.
 *
 * (note) resource について
 *   * resource はリクエストの基点となったリソースでなければなりません。
 *     (resource 以下のメンバではNG)
 *
 *   * 指定するresource にはDB検索結果を反映すること。
 *     検索の結果、DBに格納されていなくて問題ありません。以下のメンバを正しく設定すること。
 *      exists,
 *      collection,
 *      info,
 *      info->rdb_r,
 *      info->rdb_r->u_spec,
 *      info->rdb_r->rstate_pr
 *
 * @param r request_rec *
 * @param resource const dav_resource * DB検索結果が反映された判定対象のリソース
 * @return dav_error * アクセス可能かどうか(NULL の場合アクセス可能)
 */
DIVY_DECLARE(dav_error *) divy_validate_request_resource(request_rec *r,
							const dav_resource *resource);

/**
 * URIのpath segment に含めてはならない文字が無いかどうか検証する。
 *
 * (note) path segment に含めてはならない文字
 * 	\/:,;*?"<>|	(いずれもWindowsのファイル名の制限)
 * (note) path segment の定義
 * 	path segment とは、URIを構成する文字列の"/"と"/"の間の文字列のこと.
 * 	スラッシュ自身は含まれないことに注意。
 *
 * @param p apr_pool_t *
 * @param ignore int 無視する禁則文字のビット値.無視しなければ0.
 * @param str const char * 検査対象の文字列
 * @return int 検証結果 (0: 問題ない / 1: 問題あり)
 */
DIVY_DECLARE(int) divy_validate_pathsegment(apr_pool_t *p, int ignore,
							const char *str);

/**
 * ユーザプロパティが正しいかどうか検証する。
 *
 * (note) チェック成功条件
 * ------------------------------
 * ・usr_pr がNULLではないこと (userdiscovery)
 * ・userdiscovery を保持できるURIであること
 * ・usr_pr の usrid が登録処理ではNULLでないこと、更新処理ではNULLであること
 * ・usr_pr の usrid には、値があり、エンティティであり、禁則文字を含まず、
 * 		resource->uri のusrid パート(final path segment) が同じ (登録時)
 * ・usr_pr の usrid は特殊フォルダの名称とバッティングしないこと
 * ・usr_pr の usrid には、値がないこと (更新時)
 * ・usr_pr の fullname, password には値があること
 * ・usr_pr の mailaddr のフォーマットが正しいこと
 * ・usr_pr の accessdeny は全て0ではないこと
 * ・usr_pr の 操作者自身の"ユーザ"フォルダをアクセス拒否リストに加えないこと
 * ・usr_pr の usedst, usedres に値が無いこと
 * ・usr_pr の maxst, maxres は、正の自然数であること(0を含まない)
 * ・usr_pr の registdt, updatedt に値が無いこと
 * ・ユーザ追加ライセンスの範囲に収まっているかどうか(新規作成時)
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param iscreen const divy_user_iscreen * 検証するユーザプロパティscreen
 * @return dav_error * 400, 403, 500. NULL の時には検証に成功したということ
 */
DIVY_DECLARE(dav_error *) divy_validate_user_property(request_rec *r,
					int chk_mode,
					const divy_user_iscreen *iscreen);

/**
 * アップデートユーザがアクティブになる場合、サーバのライセンスに接触していない
 * か調査を行う。
 *
 * @param r request_rec *
 * @return dav_error * NULLの場合はライセンス的には問題がないということ
 */
DIVY_DECLARE(dav_error *) divy_validate_update_user_limit(request_rec* r);

/**
 * グループプロパティが正しいかどうか検証する。
 *
 * (note) チェック成功条件
 * ------------------------------
 * ・grp_pr がNULLではないこと
 * ・groupdiscovery を保持できるURIであること
 * ・grpid に値がないこと
 * ・grp_pr の name には、値があり、エンティティであり、禁則文字を含まず、
 * 		resource->uri のname パート(final path segment) が同じ (登録時)
 * ・grp_pr の name に値がないこと (更新時)
 * ・グループ階層の作成が制限されているとき、単階層のグループであること
 * ・メールオプションが無いのに、grp_pr->mailwatch_pr がある
 * ・grp_pr->mailwatch_pr があるにも関わらず、トリガが未設定
 * ・grp_pr->mailwatch_pr->notification のChildが不正
 * ・grp_pr の registdt, updatedt に値がないこと
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param iscreen const divy_group_iscreen * 検証するグループプロパティscreen
 * @return dav_error * 400, 500. NULL の時には検証に成功したということ
 */
DIVY_DECLARE(dav_error *) divy_validate_group_property(request_rec *r,
					int chk_mode,
					const divy_group_iscreen *iscreen);

/**
 * SQL プロパティが正しいかどうか検証する。
 *
 * (note) チェック成功条件
 * ------------------------------
 * ・sql_pr がNULLではないこと (sqldiscovery)
 * ・sqldiscovery を保持できるURIであること
 * ・sql_pr->sqlid に値がないこと (sqlno)
 * ・sql_pr->labelname には、値があり、エンティティであり、禁則文字を含まず、
 * 		resource->uri のname パート(final path segment) が同じ (登録時)
 * ・sql_pr->labelname には、値がないこと (更新時)
 *
 * [ 名前付きバインド変数構文 以外 ]
 * ・DB機能が利用出来ること (type)
 * ・sql_pr->sql に値があること (statement)
 * ・sql_pr->dbms_pr に値が無いこと (dbmslist)
 * ・sql_pr->usedbmsname に値があること (usedbmsname)
 * ・sql_pr->registdt に値が無いこと
 * ・sql_pr->updatedt に値が無いこと
 *
 * [ RequiredSQL構文, 名前付きバインド変数構文 ]
 * 	・sql_pr->subname に値があること (subname)
 * 	・sql_pr->subname の形式($$B, $$SS, $$SM)が正しいこと
 * 	・sql_pr->subname とsql_pr->type の組み合わせが正しいこと
 * 	・sql_pr->subname のSQLがDBに登録されていないこと(登録時),
 * 	・sql_pr->subname が既に登録されていたとしたら、それは自身が持っていること(更新時)
 *
 * [ リポジトリ検索SQL構文 ]
 * 	・sql_pr->sql のSELECT句が存在すること
 * 	・sql_pr->sql のSELECTカラムが必須集合を含んでいること
 *
 * [ WHERE句プロパティあり ]
 * 	・sql_pr->sqlcnt_pr のname が存在すること (dispcolumnname)
 * 	・名前付きバインド変数構文ではないこと
 *
 * [ WHERE句プロパティあり & $$B / $$SS / $$SM / ? が指定された]
 * 	・name が指定されていること (reqsqlname)
 * 	・name の形式($$B, $$SS, $$SM)が正しいこと (reqsqlname)
 * 	・sqlmode が指定されていないこと(通常バインド変数)
 *	・sqlmode が指定されていること($$B / $$SS / $$SM)
 * 	・sqlmode とtype (sqltype) との組合せが正しいこと
 *
 * [ WHERE句プロパティあり ]
 * 	・sql_pr->sqlcnt_pr->id の先頭が1であること
 * 	・sql_pr->sqlcnt_pr->id が重複していないこと
 * 	・sql_pr->sqlcnt_pr->id に歯抜けがないこと
 *
 * ・sql_pr->sql の文法とWHERE句プロパティとの整合性が取れていること
 *
 * [ WHERE句プロパティあり & $$B / $$SS / $$SM が指定された]
 * 	・$$B / $$SS / $$SM がDBに登録され、アクティブであること
 *
 * [ WHERE句プロパティあり & RequiredSQL or 名前付きバインド変数構文 & 
 *   非アクティブ化 & 更新処理] 
 *      このRequiredSQLを利用する他のSQLが存在しないこと。
 *
 * [ WHERE句プロパティあり & RequiredSQL ]
 * 	SQL依存関係がループしないこと (自己参照を含む)
 *
 * [ WHERE句プロパティあり ]
 * 	SQLの連鎖数が最大値を超えないこと
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param iscreen const divy_sql_iscreen * 検証するSQLプロパティscreen
 * @return dav_error 400, 500. NULL の時には検証に成功したということ
 */
DIVY_DECLARE(dav_error *) divy_validate_sql_property(request_rec *r,
					int chk_mode,
					const divy_sql_iscreen *iscreen);

/**
 * システムメッセージプロパティが正しいかどうか検証する。
 *
 * (note) チェック成功条件
 * ------------------------------
 * ・sysmsg_pr がNULLではないこと
 * ・systemdiscovery を保持できるURIであること
 * ・sysmsg_pr の msgid には、値があり、数値であり、
 * 		resource->uri のmsgid パート(final path segment) が同じ (登録時)
 * ・sysmsg_pr の msgid に値がないこと (更新時)
 * ・sysmsg_pr の registdt, updatedt に値が無いこと
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param iscreeen const divy_sysms_iscreen * 検証するシステムメッセージを持つscreen
 * @return dav_error * 400, 500. NULL の時には検証に成功したということ
 */
DIVY_DECLARE(dav_error *) divy_validate_sysmsg_property(request_rec *r,
					int chk_mode,
					const divy_sysmsg_iscreen *iscreen);

/**
 * DBMS プロパティが正しいかどうか検証する。
 *
 * (note) チェック成功条件
 * ------------------------------
 * ・dbms_pr がNULLではないこと
 * ・dbmsdiscovery を保持できるURIであること
 * ・dbms_pr の dbmsid に値が無いこと
 * ・dbms_pr の name には、値があり、エンティティであり、禁則文字を含まず、
 * 		resource->uri のname パート(final path segment) が同じ (登録時)
 * ・dbms_pr の name には、値がないこと (更新時)
 * ・dbms_pr の type に値があること
 * ・dbms_pr の hostname に値があること
 * ・dbms_pr の port が正の数であること
 * ・dbms_pr の dbname に値があること
 * ・dbms_pr の username に値があること
 * ・dbms_pr の password に値があること
 * ・dbms_pr の active が非アクティブの時、利用しているSQLが存在しないこと (更新時)
 * ・dbms_pr の registdt, updatedt に値がないこと
 *
 * 	active のチェックはすでに済んでいるものとします。
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param iscreen const divy_dbms_iscreen * 検証するDBMSプロパティを持つscreen
 * @return dav_error * 400, 424, 500. NULL の時には検証に成功したということ
 */
DIVY_DECLARE(dav_error *) divy_validate_dbms_property(request_rec *r,
					int chk_mode,
					const divy_dbms_iscreen *iscreen);

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * systemquota プロパティの解析結果をチェックする。
 *
 * @param r request_rec *
 * @param sysquota_iscreen divy_sysquota_iscreen * systemquota の内容をもつ
 * @return dav_error * DAVエラー(400, 403, 507), エラーがなければNULL.
 */
DIVY_DECLARE(dav_error *) divy_validate_systemquota_property(request_rec *r,
					divy_sysquota_iscreen *sysquota_iscreen);
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

/**
 * mailwatch プロパティの解析結果をチェックする。
 *
 * (note) チェック成功条件
 * ------------------------------
 * ・メール機能ライセンスがあること
 * ・制限ユーザではないこと (拡張ステータスあり)
 * ・メール監視フラグを設定可能なリソースであること
 * ・mwatch_pr がNULLではないこと
 * ・mwatch_pr の notification が指定されていること
 * ・mwatch_pr の trigger_methods が指定されていること
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param mwatch_iscreen const divy_mailwatch_iscreen * 検証するメール監視プロパティ
 * @return dav_error * DAVエラー(400,409,500). NULL の時には成功
 */
DIVY_DECLARE(dav_error *) divy_validate_mailwatch_property(request_rec *r,
					int chk_mode,
					const divy_mailwatch_iscreen *mwatch_iscreen);

/**
 * sharedcollection プロパティの解析結果をチェックする。
 *
 * (note) チェック成功条件
 * ------------------------------
 * ・shlink_pr がNULLではないこと
 * ・Read-Only、Upload-Onlyユーザではないこと (ユーザ拡張ステータスが有効時)
 * ・リクエストURIはlinkdbsearch結果フォルダであること
 * ・shlink_pr のdisplayname が存在し、禁則文字が指定されていないこと
 * ・shlink_pr のuri が存在し、禁則文字が指定されていないこと、また
 * 	このuri は共有コレクションのURIであること
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param shcol_iscreen divy_sharedcollection_iscreen * 検証する共有コレクションプロパティ
 * @return dav_error * DAVエラー(400,409,500). NULL の時には成功
 */
DIVY_DECLARE(dav_error *) divy_validate_sharedcollection_property(request_rec *r,
					int chk_mode,
					const divy_sharedcollection_iscreen *shcol_iscreen);

/**
 * updatediscovery プロパティの解析結果をチェックする。
 *
 * (note) チェック成功条件
 * ------------------------------
 * ・clmodule_pr がNULLではないこと
 * ・updatediscovery を保持できるURIであること
 * ・clmodule_pr の name は指定されていないこと
 * ・clmodule_pr の version, lineup, digest が指定されていること
 * ・clmodule_pr の が指定されていないこと
 *
 * @param r request_rec *
 * @param clmodule_pr divy_rdbo_clmodule * 検証する更新クライアントプロパティ
 * @return dav_error * 400, 500. NULL の時には検証に成功したということ
 */
DIVY_DECLARE(dav_error *) divy_validate_updatediscovery_property(request_rec *r, 
					const divy_clmodule_iscreen *clmodule_iscreen);

/**
 * resourcestate プロパティの解析結果をチェックする。
 *
 * (note) チェック成功条件
 * ------------------------------
 * ・rstate_pr がNULLではないこと(resourcestate プロパティが指定されていること)
 * ・ユーザ拡張ステータスが有効であること
 * ・View属性設定権限を持っていること
 * ・制限ユーザではないこと
 * ・resourcestate が付加/除去可能なコレクション/リソースであること
 * ・[ set (published) ]
 *   ・ 既にView属性を保持していないこと --> 寛容に扱い見逃すことになりました
 *   ・ View属性の連続性が保たれていること
 *     (上位階層のコレクションがView属性を持っている / グループコレクション直下)
 *
 * ・[ remove, set(published が指定されていない) ]
 *   ・* View属性が既に存在すること --> 寛容に扱い見逃すことになりました
 *
 * (note) このプロパティは、グループゴミ箱ではないグループコレクションより下の
 * 	リソース及びコレクションが持っています。
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param rstate_iscreen divy_resourcestate_iscreen * 検証するプロパティ
 * @return dav_error * DAVエラー(400,409,500). NULL の時には成功
 */
DIVY_DECLARE(dav_error *) divy_validate_resourcestate_property(request_rec *r,
					int chk_mode,
					divy_resourcestate_iscreen *rstate_iscreen);

/**
 * thumbnail プロパティの解析結果をチェックする。
 *
 * (note) チェック成功条件
 * ------------------------------
 * ・サムネイル機能が有効であること
 * ・thumbnail プロパティが値を持つこと
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param thumbnail_iscreen divy_thumbnail_iscreen * 検証するプロパティを持つリソース
 * @return dav_error * DAVエラー(400,409,500). NULL の時には成功
 */
DIVY_DECLARE(dav_error *) divy_validate_thumbnail_property(request_rec *r,
					int chk_mode,
					divy_thumbnail_iscreen *thumbnail_iscreen);

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passwordpolicy プロパティの解析結果をチェックする。
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param passpolicy_iscreen divy_passwordpolicy_iscreen * 検証するプロパティを持つリソース
 * @return dav_error * DAVエラー(400,409,500). NULL の時には成功
 */
DIVY_DECLARE(dav_error *) divy_validate_passwordpolicy_property(request_rec *r,
					int chk_mode,
					divy_passwordpolicy_iscreen *passpolicy_iscreen);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * changepassword プロパティの解析結果をチェックする。
 * [ 有効性 ]
 *   パスワードポリシー機能が「サポート/未サポート」「有効/無効」に関わらず
 *   定義済みとなります.
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param changepassword_iscreen divy_changepassword_iscreen * 検証するプロパティを持つリソース
 * @return dav_error * DAVエラー(400,409,500). NULL の時には成功
 */
DIVY_DECLARE(dav_error *) divy_validate_changepassword_property(request_rec *r,
					int chk_mode,
					divy_changepassword_iscreen *changepassword_iscreen);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * changeowner プロパティの解析結果をチェックする。
 * [ 有効性 ]
 *   グループリーダ機能能が「有効/無効」に関わらず定義済みとなります.
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param changeowner_iscreen divy_changeowner_iscreen * 検証するプロパティを持つリソース
 * @return dav_error * DAVエラー(400,409,500). NULL の時には成功
 */
DIVY_DECLARE(dav_error *) divy_validate_changeowner_property(request_rec *r,
					int chk_mode,
					divy_changeowner_iscreen *changeowner_iscreen);

/**
 * changeleader プロパティの解析結果をチェックする。
 * [ 有効性 ]
 *   グループリーダ機能能が「有効/無効」に関わらず定義済みとなります.
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param changeleader_iscreen divy_changeleader_iscreen * 検証するプロパティを持つリソース
 * @return dav_error * DAVエラー(400,409,500). NULL の時には成功
 */
DIVY_DECLARE(dav_error *) divy_validate_changeleader_property(request_rec *r,
					int chk_mode,
					divy_changeleader_iscreen *changeleader_iscreen);

/**
 * confirmreading プロパティの解析結果をチェックする。
 * [ 有効性 ]
 *   開封通知機能が「有効/無効」に関わらず定義済みとなります.
 *
 * @param r request_rec *
 * @param chk_mode int チェックモード
 * @param iscreen divy_confirmreading_iscreen * 検証するプロパティを持つリソース
 * @return dav_error * DAVエラー(400,409,500). NULL の時には成功
 */
DIVY_DECLARE(dav_error *) divy_validate_confirmreading_property(request_rec *r,
					int chk_mode,
					divy_confirmreading_iscreen *iscreen);


/**
 * 指定されたsrc が表すエンティティソースをdst のエンティティ
 * ディスティネーションとしてコピー可能かどうかを検証する。
 *
 * (note) 検証にパスする条件
 *   * src とdst が共に同じタイプのリソースであること。
 *   * src とdst は、エンティティであること。(リレーションではない)
 *   * dst のfinal path segment に禁則文字がないこと。
 *   * dst のfinal path segment にリレーションプレフィックスが付いていないこと
 *
 * @param r request_rec *
 * @param src_u_spec const divy_uri_spec * ソースURI
 * @param dst_u_spec const divy_uri_spec * ディスティネーションURI
 * @return dav_error * DAVエラー(400, 409). 問題なければNULL
 */
DIVY_DECLARE(dav_error *) divy_validate_copy_entity(request_rec *r,
					const divy_uri_spec *src_u_spec,
					const divy_uri_spec *dst_u_spec);

/**
 * 指定されたsrc が表すエンティティソースをdst のエンティティ
 * ディスティネーションとして移動可能かどうかを検証する。
 *
 * [ 検証にパスする条件 ]
 *   * ソースとディスティネーションが共に同じタイプのリソースであること。
 *     (例外) 
 *	* プライベート, グループフォルダ <-> 通常フォルダは許可する
 *	* 更新クライアント -> プライベート,グループ,通常フォルダは許可する
 *	* 共有コレクション -> 通常コレクションの下は許可する
 *   * ソースとディスティネーションは、エンティティである。(リレーションではない)
 *   * dst のfinal path segment に禁則文字がないこと。
 *   * dst のfinal path segment にリレーションプレフィックスが付いていないこと
 *
 * @param r request_rec *
 * @param src_u_spec const divy_uri_spec * ソースURI
 * @param dst_u_spec const divy_uri_spec * ディスティネーションURI
 * @return dav_error 問題なければNULL, 問題があればエラーを返す(409)
 */
DIVY_DECLARE(dav_error *) divy_validate_move_entity(request_rec *r,
					const divy_uri_spec *src_u_spec,
					const divy_uri_spec *dst_u_spec);

/**
 * 指定されたsrc が表すリレーションソースをdst のリレーション
 * ディスティネーションとしてコピー可能かどうかを検証する。
 *
 * [ 検証にパスする条件 ]
 *   * src と dst が共に同じタイプのリソースであること。
 *   * src と dst はリレーションである。(エンティティではない)
 *   * dst のfinal path segment に禁則文字が含まれていないこと
 *   * リレーションのリネームは禁止
 * (note)
 * 	リレーションの作成はCOPYメソッドですが、このValidateロジックを
 * 	用いてはなりません。_validate_create_relationを利用して下さい。
 * (note)
 * 	dst のfinal path segment はsrc のそれと常に同じでなければならないので、
 * 	禁則チェックはやっても意味がありません。
 * 	(src のfinal path segmentは正しくDBに格納されているという仮定)
 *
 * @param r request_rec *
 * @param src_u_spec const divy_uri_spec * ソースURI
 * @param dst_u_spec const divy_uri_spec * ディスティネーションURI
 * @return dav_error * DAVエラー(400, 403, 409)、成功の場合NULL
 */
DIVY_DECLARE(dav_error *) divy_validate_copy_relation(request_rec *r,
					const divy_uri_spec *src_u_spec,
					const divy_uri_spec *dst_u_spec);

/**
 * 指定されたsrc が表すリレーションソースをdst のリレーション
 * ディスティネーションとして移動可能かどうかを検証する。
 *
 * [ 検証にパスする条件 ]
 *   * src と dst が共に同じタイプのリソースであること。
 *   * src と dst はリレーションであること。(エンティティではない)
 *   * リレーションのリネームは禁止
 * (note)
 * 	move におけるdst のfinal path segment はsrc のそれと常に同じ
 * 	でなければならないので、禁則チェックはやっても意味がありません。
 * 	(src のfinal path segmentは正しくDBに格納されているという仮定)
 *
 * @param r request_rec *
 * @param src_u_spec const divy_uri_spec * ソースURI
 * @param dst_u_spec const divy_uri_spec * ディスティネーションURI
 * @return dav_error * DAVエラーを返す(400, 403, 409)
 */
DIVY_DECLARE(dav_error *) divy_validate_move_relation(request_rec *r,
					const divy_uri_spec *src_u_spec,
					const divy_uri_spec *dst_u_spec);

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * userid のユーザのパスワード有効期限、猶予期間のチェック
 *
 * @param r request_rec *
 * @param userid const char * チェック対象のユーザのユーザID
 * @param u_spec divy_uri_spec * URIの種類
 * @param type divy_reqbody_type Bodyデータの種類
 */
DIVY_DECLARE(dav_error *) divy_validate_password_term(request_rec *r,
					const char *userid,
					divy_uri_spec *u_spec, divy_reqbody_type type);
#endif	/* DIVY_SUPPORT_PLUGIN */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * 定義値DIVY_HEADER_RESULTSTATUS が示すヘッダ文字列を生成して
 * ヘッダにセットする.
 *
 * @param r request_rec *
 * @param type divy_resultstatus_type セットするヘッダの種類
 */
DIVY_DECLARE(void) divy_set_resultstatus(request_rec *r, divy_resultstatus_type type);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * 定義値DIVY_HEADER_RESULTSTATUS が示すヘッダ文字列をリセットする.
 *
 * @param r request_rec *
 * @param type divy_resultstatus_type セットするヘッダの種類
 */
DIVY_DECLARE(void) divy_reset_resultstatus(request_rec *r);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * 強制削除がクライアントから指示されているかどうか.
 * (note)
 *     強制削除の指示は、HTTPリクエストヘッダ経由で行われます.
 *
 * @param r request_rec *
 * @return int 1: 指示されていた / 0: 指示されていない
 */
DIVY_DECLARE(int) divy_validate_is_forcedelete(request_rec *r);

/**
 * リソースを調査します。アップロードポリシーで利用されます
 *
 * @param r			request_rec *
 * @param resource	dav_resource *
 *
 * @return int 1: 問題が見つかった / 0: 問題は見つからなかった
 *
 */
DIVY_DECLARE(int) divy_resource_inspection(request_rec *r, const dav_resource *resource);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_VALIDATOR_H */

