/**
 * $Id$
 *
 * プールにキャッシュされる値を管理するのに必要な手順やデータを定義するヘッダ
 *
 * (note) 導入の背景
 * 	関数 apr_pool_userdata_set / apr_pool_userdata_get を使ってapr_pool_t に
 * 	ユーザデータを記録するという手法を主としてパフォーマンス改善を目的に
 * 	各所で行っていた。だが、これらの関数は、プールにキャッシュする際に使う
 * 	key値のユニーク性を実装者が保証しなければならず、結果、キャッシュ値の
 * 	増大と共に問題を抱えるようになった。
 * 	この問題を本ヘッダファイル及び実装ファイルによって以下のように解決
 * 	します。
 *
 * 	[ 前提 ]
 *
 * 	  * プールを使ってキャッシュしたい値の種類は以下である
 * 	  ** keyに関連付けられた値
 * 	  ** 1度通った事を記録するためのフラグ値
 *
 * 	[ 基本手法 ]
 *
 *        * key値及びプレフィックスをこのヘッダファイルに登録させます
 * 	  * 関数 apr_pool_userdata_set / apr_pool_userdata_get に替わる新しい
 * 	    キャッシュ用関数を導入します。
 * 	    このキャッシュ関数に値をset / get するには登録されたkey値や
 * 	    プレフィックス文字列に対して採番されたIDを用いなくては
 * 	    ならないものとします。--> 必ず登録しないとならないことを強制できる
 * 	  * フラグ目的で利用できる関数を別立てします。
 * 	    コードの可読性を高めるためです。
 *
 * これらの手法の欠点は、(擬似的ではあるが一種の)カプセル機構を破壊します。
 * つまり、ロックプロバイダだけ知っていればよいキャッシュ値を全く別の何かが
 * 認識できてしまうことです。残念ながら、プールは基本的にグローバルであるため、
 * 元来カプセル化を主張できる種類のものではなかったということです。諦めて下さい。
 */
#ifndef INCLUDE_TF_VALUECACHE_H
#define INCLUDE_TF_VALUECACHE_H

#include "apr.h"
#include "apr_pools.h"

#include "tfr.h"

#if APR_HAVE_STDARG_H
#include <stdarg.h>
#endif	/* APR_HAVE_STDARG_H */

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structure/enum
  ----------------------------------------------------------------------------*/
/**
 * key値に対応するIDを表す列挙型
 *
 * * 新しいIDの追加方法
 * ----------------------
 *   (1) DIVY_PCACHE_END の"直前"位置に新しいkeyを追加
 *   (2) 配列 _key_relation (tf_valuecache.c に定義) の"最後尾"にkeyと
 *       keyの文字列表現(NULL,空以外)を追加。
 *       文字列表現は同階層に並ぶ他の文字列と一致してはなりません。
 * (note)
 * 	_key_relationの要素の並び位置と divy_pcache_keyの数値は一致させて下さい。
 * 	一致させなくても問題はおきないようにしますが可読性が下がります。
 * 	(またパフォーマンスも下がります)
 */
enum __divy_pcache_key {
	/*
	 * テーブルカラム名-Liveプロパティのリレーション(see divy_rdbo_get_tableinfo, グローバル)
	 */
	DIVY_PCACHE_DAT_GL_TABLEINFO = 0,
	/*
	 * ロックレコード (see dav_divy_open_lockdb, Request単位)
	 */
	DIVY_PCACHE_DAT_REQ_LOCKREC,
	/*
	 * uri解析結果 (see divy_parse_uri, uri単位)
	 */
	DIVY_PCACHE_DAT_URI_USPEC,
	/*
	 * 所属グループ情報 (see _validate_group_access, ユーザ単位)
	 */
	DIVY_PCACHE_DAT_USER_GRPINFO,
	/*
	 * 最終アクセス記録フラグ (see dav_divy_update_lastaccess, KeepAlive単位)
	 */
	DIVY_PCACHE_FLG_KA_ACCESS,
	/*
	 * DBプロバイダID (see _register_temp_DbConnPool, Request単位)
	 */
	DIVY_PCACHE_DAT_REQ_DBPROVIER,
	/*
	 * セマフォ取得 (see divy_util_ldap_init_lock, グローバル)
	 */
	DIVY_PCACHE_FLG_GL_SEM,
	/*
	 * LDAP util (see divy_util_ldap_initialize, グローバル)
	 */
	DIVY_PCACHE_DAT_GL_LDAPUTIL,
	/*
	 * LDAPコンフィグ (see _divy_util_ldap_create_config, Request単位)
	 */
	DIVY_PCACHE_DAT_REQ_LDAPCONFIG,
	/*
	 * ロギングに必要な情報 (see init_log_env, Request単位)
	 */
	DIVY_PCACHE_DAT_REQ_LOGENV,
	/*
	 * サーバ主体メール送信状態 (see _send_servermail, KeepAlive単位)
	 */
	DIVY_PCACHE_FLG_KA_SVRMAILSEND,
	/*
	 * Requestの間だけ保持される一時的なオブジェクト専用のアロケータ(Request単位)
	 */
	DIVY_PCACHE_DAT_REQ_ALLOCATOR,
	/*
	 * ライセンスキーの取得状態 (see _query_license, KeepAlive単位)
	 */
	DIVY_PCACHE_FLG_KA_LICENSEKEY,
#ifdef DIVY_SUPPORT_PLUGIN
	/*
	 * プラグインローダーへのポインタ (see divy_pi_init, グローバル) 
	 */
	DIVY_PCACHE_DAT_GL_PLLOADER,
	/*
	 * プラグインローダー初期化状態 (see divy_pi_init, グローバル)
	 */
	DIVY_PCACHE_FLG_GL_PLLOADER,
	/*
	 * プラグイン用Childプロセスプール (see divy_pi_cinit, Request単位)
	 */
	DIVY_PCACHE_DAT_GL_PLPOOL,
	/*
	 * プラグイン用リクエストレベルのプール (see divy_pi_rinit, Request単位)
	 */
	DIVY_PCACHE_DAT_REQ_PLPOOL,
#endif	/* DIVY_SUPPORT_PLUGIN */

	/*
	 * Content-Type、拡張子マッピング情報 (see _read_mimetype_map, グローバル)
	 */
	DIVY_PCACHE_DAT_GL_CMAP,

	/*
	 * アクセスユーザのユーザ情報 (see divy_rdbo_cache_userinfo, Request & ユーザ単位)
	 */
	DIVY_PCACHE_DAT_REQ_USERINFO,
#ifdef DIVY_SUPPORT_PASSPOLICY
	/*
	 * キャッシュされたパスワードポリシー情報 (see _useris_do_process, Request単位)
	 */
	DIVY_PCACHE_DAT_REQ_PASSPOLICY,
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/*
	 * キャッシュされた認証元情報
	 */
	DIVY_PCACHE_FLG_USE_SAML,

	/*
	 * キャッシュされたアクセスユーザの所属グループの情報
	 * (see divy_rdbo_cache_userinfo, Request単位 & ユーザ単位)
	 */
	DIVY_PCACHE_DAT_REQ_AVAILABLE_GRPINFO,

	/*
	 * キャッシュされたセキュリティ保護されたグループ一覧
	 * (see _insert_prop_resourcestate, Request単位)
	 */
	DIVY_PCACHE_DAT_REQ_SECUREDGRP,

	/*
	 * LDAP認証の結果をキャッシュ (see divy_util_ldap_check_userid, KeepAlive + uid)
	 */
	DIVY_PCACHE_DAT_KA_LDAPAUTH,

	/*
	 * LDAP用の共有メモリ初期化状態 (see divy_util_ldap_initialize)
	 */
	DIVY_PCACHE_FLG_GL_SHM,

	/*
	 * POST時に利用される。セッションID
	 */
	DIVY_PCACHE_DAT_SES_SID,

	/*
	 * jobの初期化状態(see divy_start_watchdog)
	 */
	DIVY_PCACHE_FLG_GL_JOBS,

	/* sentinel */
	DIVY_PCACHE_END
};
typedef enum __divy_pcache_key divy_pcache_key;

/**
 * divy_pcache_key とkey値との対応関係を表す構造体
 */
struct __divy_pcache_key_relation {
	divy_pcache_key key;
	const char *skey;	/* 文字列表現のキー値 */
};
typedef struct __divy_pcache_key_relation divy_pcache_key_relation;

/*------------------------------------------------------------------------------
  Declare "value cache" functions
  ----------------------------------------------------------------------------*/
/**
 * プールp にkey でキャッシュされた値を取得する。
 *
 * @param p apr_pool_t * キャッシュ値を保持するプールへのポインタ
 * @param key divy_pcache_key key値
 * @return void * keyでキャッシュされていた"何か"へのポインタ
 */
DIVY_DECLARE(void *) divy_pcache_get_data(apr_pool_t *p, divy_pcache_key key);

/**
 * プールp に組合せキー(key1 + ...) でキャッシュされた値を取得する。
 * 組合せキーは引数の順序と一致します。
 * なお、vkey以降のキー値が重複しないことは呼び出し元で保証して下さい。
 * そこまではカバーできません。
 * (例) 
 * 	void *a = divy_pcache_get_data(p, 1, "Japan", "Tokyo", NULL);
 * 	void *b = divy_pcache_get_data(p, 1, "Tokyo", "Japan", NULL);
 *
 * 	-> a, b は異なるキーに関連付けられています(キー順序に依存)
 * (note)
 * 	指定するキーの数は、多くても3つ程度の留めてください。
 * 	パフォーマンス劣化及び記憶領域の無駄になります。
 *
 * @param p apr_pool_t * キャッシュ値を保持するプールへのポインタ
 * @param key1 divy_pcache_key 最初のkey値
 * @param ... const char * 2番目以降のkey値。必ずNULL終端にすること
 * @return void * key1, ... でキャッシュされていた"何か"へのポインタ
 */
DIVY_DECLARE(void *) divy_pcache_vget_data(apr_pool_t *p,
						divy_pcache_key key1, ...);

/**
 * プールp にkey でdata をキャッシュする。(ポインタキャッシュ)
 * (note)
 * 	厳密にはキャッシュするのはdataへのポインタ値です。
 * 	ですので、キャッシュした後dataの中身を書き換えてしまうと
 * 	キャッシュ値も変わってしまいます。注意して下さい。
 *
 * @param p apr_pool_t * キャッシュ値を保持するプールへのポインタ
 * @param data const void * keyでキャッシュする"何か"へのポインタ
 * @param key divy_pcache_key キャッシュのkey
 */
DIVY_DECLARE(void) divy_pcache_set_data(apr_pool_t *p,
					const void *data,
					divy_pcache_key key);

/**
 * プールp に組合せキー(key1 + ...) でdata をキャッシュする。
 * (ポインタキャッシュ)
 *
 * @param p apr_pool_t * キャッシュ値を保持するプールへのポインタ
 * @param data const void * keyでキャッシュする"何か"へのポインタ
 * @param key1 divy_pcache_key 最初のkey
 * @param ... const char * 2番目以降のkey値。必ずNULL終端にすること
 */
DIVY_DECLARE(void) divy_pcache_vset_data(apr_pool_t *p,
					const void *data,
					divy_pcache_key key1, ...);

/**
 * プールp にkey でdata をキャッシュする。(ポインタキャッシュ)
 * プールが破棄された時にdata を引数に渡してcleanup をコールする
 *
 * @param p apr_pool_t * キャッシュ値を保持するプールへのポインタ
 * @param data const void * keyでキャッシュする"何か"へのポインタ
 * @param cleanup クリーンアップ関数へのポインタ(NOT NULL)
 * @param key divy_pcache_key キャッシュのkey
 */
DIVY_DECLARE(void) divy_pcache_fset_data(apr_pool_t *p,
					const void *data,
					apr_status_t (*cleanup)(void *),
					divy_pcache_key key);

/**
 * プールp に組合せキー(key1 + ...) でdata をキャッシュする。
 * プールが破棄された時にdata を引数に渡してcleanup をコールする
 * (ポインタキャッシュ)
 *
 * @param p apr_pool_t * キャッシュ値を保持するプールへのポインタ
 * @param data const void * keyでキャッシュする"何か"へのポインタ
 * @param cleanup クリーンアップ関数へのポインタ(NOT NULL)
 * @param key1 divy_pcache_key 最初のkey
 * @param ... const char * 2番目以降のkey値。必ずNULL終端にすること
 */
DIVY_DECLARE(void) divy_pcache_vfset_data(apr_pool_t *p,
					const void *data,
					apr_status_t (*cleanup)(void *),
					divy_pcache_key key1, ...);

/*------------------------------------------------------------------------------
  Declare "flag cache" functions
  ----------------------------------------------------------------------------*/
/**
 * key でフラグキャッシュp からフラグ値を取得する。
 *
 * @param p apr_pool_t * フラグを保持するプールへのポインタ
 * @param key divy_pcache_key key値
 * @return int フラグ値 (key, p がNULLならば0です)
 */
DIVY_DECLARE(int) divy_pcache_get_flag(apr_pool_t *p, divy_pcache_key key);

/**
 * 組合せキー(key1 + ...) でフラグキャッシュp からフラグ値を取得する。
 *
 * @param p apr_pool_t * フラグを保持するプールへのポインタ
 * @param key1 divy_pcache_key 最初のkey値
 * @param ... const char * 2番目以降のkey値。必ずNULL終端にすること
 * @return int フラグ値
 */
DIVY_DECLARE(int) divy_pcache_vget_flag(apr_pool_t *p,
					divy_pcache_key key1, ...);

/**
 * key でフラグキャッシュpにflag値を記録する。
 *
 * @param p apr_pool_t * フラグを保持するプールへのポインタ
 * @param flag int フラグ値
 * @param key divy_pcache_key key値
 */
DIVY_DECLARE(void) divy_pcache_set_flag(apr_pool_t *p, int flag,
					divy_pcache_key key);

/**
 * 組合せキー(key1 + ...) でフラグキャッシュp にflag値を記録する。
 *
 * @param p apr_pool_t * フラグを保持するプールへのポインタ
 * @param flag int フラグ値
 * @param key1 divy_pcache_key 最初のkey値
 * @param ... const char * 2番目以降のkey値。必ずNULL終端にすること
 */
DIVY_DECLARE(void) divy_pcache_vset_flag(apr_pool_t *p, int flag,
					divy_pcache_key key1, ...);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_VALUECACHE_H */
