/**
 * $Id$
 *
 * util.h
 *
 * ユーティリティ関数用ヘッダファイル
 *
 * (お願い)
 *  このファイルには、共通関数として使用できそうな関数を
 *  入れて下さい。
 *  各プロバイダ固有の処理などは、そのプロバイダ専用の
 *  ファイルに記述するようにして下さい。
 *
 * 変更履歴
 * 
 * 2003/03/04 Tue takehara NEW
 *
 */
#ifndef INCLUDE_UTIL_H
#define INCLUDE_UTIL_H

#include "apr.h"
#include "apr_time.h"
#include "apr_pools.h"
#include "apr_hash.h"
#include "httpd.h"
#include "mod_dav.h"

#if APR_HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif

#include "tfr.h"
#include "tf_misc.h"
#include "util_common.h"
#include "mod_dav_tf.h"
#include "tf_rdbo.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define fixed value
  ----------------------------------------------------------------------------*/

/*--------------------------------------------------------------
  Define enum
  --------------------------------------------------------------*/
/*
 * ApacheのMPM情報
 */
enum __divy_mpmq {
	DIVY_MPMQ_UNKNOWN = -1,
	DIVY_MPMQ_HARD_LIMIT_THREADS = 0,
	DIVY_MPMQ_HARD_LIMIT_DAEMONS,
	DIVY_MPMQ_IS_ASYNC,
	DIVY_MPMQ_MAX_THREADS,
	DIVY_MPMQ_MAX_DAEMONS,
};
typedef enum __divy_mpmq divy_mpmq;
/*------------------------------------------------------------------------------
  Declare structure
  ----------------------------------------------------------------------------*/
/**
 * 関数divy_get_accept_language_param で戻される言語タグパラメータを表す構造体
 */
typedef struct __divy_language_tag	divy_language_tag;
struct __divy_language_tag {
	const char *lang;
	double qval;	/* q パラメータ */

	divy_language_tag *next;	/* 次へ */
};

/**
 * 解析されたViaヘッダの情報を表す
 */
typedef struct __divy_httpheader_via	divy_httpheader_via;
struct __divy_httpheader_via {

	char *sHttpVersion;	/* HTTPバージョン情報 */
	char *sHost;		/* ホスト名情報 */
	char *sPort;		/* ポート番号情報 */
	char *sComment;		/* コメント情報 */

	char *sVia;	/* 解析前のVia ヘッダのオリジナル文字列 */
	divy_httpheader_via *next;	/* 次のViaヘッダへのポインタ */
};

/**
 * 関数 divy_construct_url2 のtype に指定可能な数値
 */
enum {
	DIVY_URLTYPE_PUBLIC = 1,	/* グローバルに公開するためのURL. メールに添付されるURLなどがこれ */
	DIVY_URLTYPE_PROTECTED,		/* 利用者から見てPublicなURL */
	DIVY_URLTYPE_INNER,			/* 内部プログラムが利用するURL. システムCGIのURLなどがこれ */
};

/**
 * 関数 divy_construct_url2 によって組み立てられるURLのパスセグメントを表す構造体
 */
typedef struct __divy_buildurl_segment	divy_buildurl_segment;
struct __divy_buildurl_segment {
	char *schema;	/* スキーマ名パート("http" / "https") */
	char *host;		/* ホスト名パート */
	int port;		/* ポート名パート(80, 443の場合には、組み立てられたURLにport を含めません */
	char *uri;		/* URIパート (divy_construct_url2 に指定された引数) */
};

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * ロックの種類のうち、"排他ロック"を表すXML文字列を取得する。
 * 
 * 引数が存在しないことからも分かるように、この関数は、リソースや
 * コレクションが排他ロックを持っているかどうかとは全く"無関係"に
 * 単に組み立てられた文字列を返却するだけです。
 * このようなWebDAV固有の処理は呼び出し元において行ってください。
 *
 * @return const char* "排他ロック"を表す文字列
 */
DIVY_DECLARE(const char *) dav_divy_get_exclusivelock_string(void);

/**
 * ロックのデータベースをOPENして引数で与えられたURI以下のデータを取得する
 * HOOKに存在するdav_divy_open_lockdbの変形版
 * (サーチメソッドで利用されることを目的として作成しました。)
 * 
 * @param r request_rec *       リクエスト
 * @param lockdb dav_lockdb *   lockdb構造体
 * @param key const char *      検索対象となるキー（URI)
 * @param depth int             keyからの階層の深さ(0 or 1 or INFINITY)
 * @param expire int            タイムアウトチェック(0: no / 1: yse)
 * @return 結果ステータス( 0: 正常 / 1: 失敗)
 * 
 */
DIVY_DECLARE(int) dav_divy_open_lockdb_private(request_rec *r, dav_lockdb **lockdb,
				 const char *key, int depth,
				 int expire);

/**
 * dav_divy_openlockdb_privateで作られたlockdb構造体を利用して
 * 引数で与えられたロック情報を返却する。dav_lock構造体で返却する。
 * 存在しない場合NULLを返却する。
 * dav_divy_open_lockdb_privateで作成されたlockdb構造体でなければNULLを
 * 返します。
 *
 * @param lockdb dav_lockdb *    lockdb構造体
 * @param uri    const char *    確認するURI
 * @param locks  dav_lock   **   dav_lock構造体
 * @return dav_lock *            ロック構造体
 *
 */
DIVY_DECLARE(int) dav_divy_fetch_lock_record(dav_lockdb *lockdb, const char *uri,
	       						dav_lock **locks);

/**
 * dav_divy_open_lockdb_privateで作られたlockdb構造体を利用して
 * 引数で与えられたuriのロック情報を返却する。
 * dav_divy_open_lockdb_privateで作成されたlockdb構造体でなければNULLを
 * 返します。
 * それ以外ではEMPTYタグを返却します。
 * (サーチメソッドで利用されることを目的として作成しました)
 *
 * @param lockdb dav_lockdb *   lockdb構造体
 * @param uri const char *      確認するURI
 * @return const char *         lockdiscovery内のタグすべて
 *
 */
DIVY_DECLARE(const char *) dav_divy_lock_get_activelock(dav_lockdb *lockdb, const char *uri);

/**
 * dav_divy_open_lockdb_privateで作成されたlockdb構造体を閉じる（ふりをする）
 * 特に処理は行いません。
 * ダミー的な関数です。
 * (サーチメソッドで利用されることを目的として作成しました)
 * 
 * @param lockdb dav_lockdb **
 * @return 結果ステータス( 0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) dav_divy_close_lockdb_private(dav_lockdb *lockdb);

/**
 * ログインしているユーザの名前を取得する。
 * (note)
 * 	取得した値の中身を変更してはなりません。
 * 	変更するとキャッシュ値が変わってしまいます。
 *
 * @param r request_rec *
 * @return const char *	 名前(ユーザID ではない) 
 */
DIVY_DECLARE(const char *) divy_get_fullname(request_rec *r);

/**
 * ログインしているユーザのパスワードを取得する
 * (note)
 * 	取得した値の中身を変更してはなりません。
 * 	変更するとキャッシュ値が変わってしまいます。
 *
 * @param r request_rec *
 * @return const char *
 */
DIVY_DECLARE(const char *) divy_get_password(request_rec *r);

/**
 * ログインしているユーザのメールアドレスを取得する
 * (note)
 * 	取得した値の中身を変更してはなりません。
 * 	変更するとキャッシュ値が変わってしまいます。
 *
 * @param r request_rec *
 * @return const char *
 */
DIVY_DECLARE(const char *) divy_get_mailaddr(request_rec *r);

/**
 * ログインしているユーザの管理者権限を取得する。
 *
 * @param r request_rec *
 * @return divy_rdbo_adminmode (DIVY_ADMINMODE_NORMAL / DIVY_ADMINMODE_ADMIN)
 */
DIVY_DECLARE(divy_rdbo_adminmode) divy_get_adminmode(request_rec *r);

/**
 * ログインしているユーザのIDを取得する。
 * (note)
 * 	取得した値の中身を変更してはなりません。
 * 	変更するとキャッシュ値が変わってしまいます。
 *
 * @param r request_rec *
 * @return const char *
 */
DIVY_DECLARE(const char *) divy_get_userid(request_rec *r);

/**
 * ログインしているユーザのResourceIDを取得する。
 * (note)
 * 	取得した値の中身を変更してはなりません。
 * 	変更するとキャッシュ値が変わってしまいます。
 *
 * @param r request_rec *
 * @return const char *
 */
DIVY_DECLARE(const char *) divy_get_resourceid(request_rec *r);

/**
 * ログインしているユーザの物理ファイルのシーケンス番号を取得する。
 * (note)
 * 	取得した値の中身を変更してはなりません。
 * 	変更するとキャッシュ値が変わってしまいます。
 *
 * @param r request_rec *
 * @return const char *
 */
DIVY_DECLARE(const char *) divy_get_userseq(request_rec *r);

/**
 * ログインしているユーザの最終アクセス時間を取得する。
 * (note)
 * 	取得した値の中身を変更してはなりません。
 * 	変更するとキャッシュ値が変わってしまいます。
 *
 * @param r request_rec *
 * @return time_t 最終アクセス時間
 */
DIVY_DECLARE(time_t) divy_get_lastaccess(request_rec *r);

/**
 * ログインしているユーザが利用しているクライアントを表す文字列(User-Agent)を
 * 取得する。
 * (note)
 * 	取得した値の中身を変更してはなりません。
 * 	変更するとキャッシュ値が変わってしまいます。
 *
 * @param r request_rec *
 * @return const char * (最終アクセス時間)
 */
DIVY_DECLARE(const char *) divy_get_lastaccessclient(request_rec *r);

/**
 * ログインしているユーザのアクセス拒否フォルダの状態を取得する。
 * (note)
 * 	取得した値の中身を変更してはなりません。
 * 	変更するとキャッシュ値が変わってしまいます。
 *
 * @param r request_rec *
 * @return const int * 要素数DIVY_FOLDER_ID_END個のアクセス状態
 * 		値が０の時には、アクセスを許可、１の時には拒否することを示す
 */
DIVY_DECLARE(const int *) divy_get_accessdeny(request_rec *r);

/**
 * ログインしているユーザの有効期限を取得する。
 *
 * @param r request_rec *
 * @return time_t 有効期限のエポックタイム
 */
DIVY_DECLARE(time_t) divy_get_expiration(request_rec *r);

/**
 * ログインしているユーザの権限を取得する。
 * (note)
 * 	取得した値の中身を変更してはなりません。
 * 	変更するとキャッシュ値が変わってしまいます。
 *
 * @param r request_rec *
 * @return const divy_rdbo_extstatus * divy_rdbo_extstatus へのポインタ
 */
DIVY_DECLARE(const divy_rdbo_extstatus *) divy_get_extstatus(request_rec *r);

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ログインしているユーザのパスワード状態を取得する.
 *
 * @param r request_rec *
 * @return const divy_rdbo_passpolicystatus *
 */
DIVY_DECLARE(const divy_rdbo_passpolicystatus *) divy_get_passpolicystatus(request_rec *r);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * ログインしているユーザのコメントを取得する。
 * (note)
 * 	取得した値の中身を変更してはなりません。
 * 	変更するとキャッシュ値が変わってしまいます。
 *
 * @param r request_rec *
 * @return const char *
 */
DIVY_DECLARE(const char *) divy_get_usercomment(request_rec *r);

/**
 * ログインしているユーザの最大サイズQuotaを取得する。
 *
 * @param r request_rec *
 * @return apr_int64_t
 */
DIVY_DECLARE(apr_int64_t) divy_get_usermsquota(request_rec *r);

/**
 * ログインしているユーザの最大ファイル数Quotaを取得する。
 *
 * @param r request_rec *
 * @return apr_int64_t
 */
DIVY_DECLARE(apr_int64_t) divy_get_usermfquota(request_rec *r);

/**
 * ログインしているユーザの使用サイズQuotaを取得する。
 *
 * @param r request_rec *
 * @return apr_int64_t
 */
DIVY_DECLARE(apr_int64_t) divy_get_userusquota(request_rec *r);

/**
 * ログインしているユーザの使用ファイル数Quotaを取得する。
 *
 * @param r request_rec *
 * @return apr_int64_t
 */
DIVY_DECLARE(apr_int64_t) divy_get_userufquota(request_rec *r);

/**
 * ログインしているユーザの全ユーザプロパティを取得する.
 *
 * @param r request_rec *
 * @return const divy_rdbo_usr *
 */
DIVY_DECLARE(const divy_rdbo_usr *) divy_get_usrprop(request_rec *r);

/**
 * ログインしているユーザのオーナIDを取得する.
 *
 * @param r request_rec *
 * @return const char * オーナID
 */
DIVY_DECLARE(const char *) divy_get_userownerid(request_rec *r);

/**
 * ログインしているユーザが作成可能な(=管理可能な)最大ユーザ数を取得する.
 *
 * @param r request_rec *
 * @return apr_int32_t 数値
 */
DIVY_DECLARE(apr_int32_t) divy_get_usermaxusercreation(request_rec *r);

/**
 * ログインしているユーザの許可されているアドレスを取得する
 *
 * @param r request_rec *
 * @return divy_array_t* IPアドレス配列
 */
DIVY_DECLARE(const char *) divy_get_allowhosts(request_rec *r);

/**
 * ログインしているユーザのログイン失敗回数を取得する
 *
 * @param r request_rec *
 * @return apr_int32_t ログイン失敗回数
 */
DIVY_DECLARE(apr_int32_t) divy_get_loginfailedcount(request_rec *r);

/**
 * Location ディレクティブで指定されたURI (root uri) を取得する。
 * 
 * @param r request_rec *
 * @return const char* URIを表す文字列
 */
DIVY_DECLARE(const char *) dav_divy_get_root_uri(request_rec *r);

/**
 * 指定されたuriが示す文字列を、XMLのPCDATAとして指定可能な正しい形式に
 * なるよう文字列をエスケープして返却する。
 *
 * @param apr_pool_t *
 * @param uri const char * 変換対象のuri
 * @return const char * 変換されたURI
 */
DIVY_DECLARE(const char *) dav_divy_escape_uri(apr_pool_t *p, const char *uri);

/**
 * 指定されたuriが示す文字列を URL デコードする。
 * (note)
 * 	この関数はEBCDIC 系でエンコードされたuri 文字列を処理できません。
 * (note)
 * 	"%2f"が"/"に変換されるため、uriにURL文字列が指定された場合には
 * 	path segment が増えてしまうことがあります。この関数を使用する場合
 * 	には、不正アクセス目的でこのようなURLを意図的に送信するクライアントが
 * 	いることを考慮して下さい。(そして適切な対策をとって下さい。)
 *
 * @param apr_pool_t *
 * @param uri const char * 変換対象のuri
 * @param char ** 変換されたURI
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) dav_divy_unescape_uri(apr_pool_t *p, const char *uri, 
					char **decode);

/**
 * サーバライセンスキーを復号する。
 *
 * @param p apr_pool_t *
 * @param ciphertext const char * サーバライセンスキー
 * @return const char * 復号したサーバライセンス
 */
DIVY_DECLARE(const char *) dav_divy_decipher_svrlicense(apr_pool_t *p,
							const char *ciphertext);

/**
 * サーバライセンスキーを暗号化する。
 *
 * @param p apr_pool_t *
 * @param plaintext const char * 平文
 * @return const char * 復号したサーバライセンス
 */
DIVY_DECLARE(const char *) dav_divy_encipher_svrlicense(apr_pool_t *p,
							const char *plaintext);

/**
 * D-IVY サーバに登録できるユーザ数の上限値(契約上の上限値)を取得する。
 *
 * @param s server_rec *
 * @return int 契約上の上限値
 */
DIVY_DECLARE(int) dav_divy_get_contract_user_count(server_rec *s);

/**
 * dav_divy_get_max_user_count で取得されるユーザの最大値を超えて登録できる
 * ユーザ数の上限値。
 * (note)
 *	dav_divy_get_allow_user_count + dav_divy_get_max_user_count が
 *	登録できる最大ユーザ数になります。
 *
 * @param s server_rec *
 * @return int 上限値 
 */
DIVY_DECLARE(int) dav_divy_get_allow_user_count(server_rec *s);

/**
 * mod_dav#util.c の関数dav_xml_get_cdata の改良バージョン。
 * dav_xml_get_cdata において存在していた以下の動作を変更しています。
 *
 * (1) strip_white = 1 が動作しない件 (これはバグです！！)
 * (2) LFがCRを伴わずに現れた場合、CRLFに変換する
 *
 * (note) (1) について
 * 	strip_white = 1の時には、取得したテキストの"前後"についた以下の文字を
 * 	取り除きます。
 * 	スペー ス, フォー ムフィード ('\f'), 改行(newline) ('\n'), 
 * 	復帰(carriage return) ('\r'), 水平タブ ('\t'), 垂直タブ ('\v')
 *
 * (note) (2) について
 * 	Apache2.0.47現在のApache XMLパーサは、CDATAに指定されたCRLFの改行を
 * 	LFだけに変換してしまいます。XMLの仕様によれば、非検証XMLパーサには、
 * 	改行、復帰、スペースなどの空白記号(white space)と呼ばれる記号をどのように
 * 	扱うのか判らないので、アプリケーションに通知することになっていますが、
 * 	Apacheは変更してしまいます。
 * 	正確には、xml:space という属性がなければデフォルトの動作になりますので、
 * 	Apacheのデフォルト動作であると主張されれば、問題があるとは言えないことに
 * 	なります。
 * 	ただ、TeamFileの動作としては困ることが多いので、ApacheによってCRLFから
 * 	LFに変換されてしまったコードを再度CRLFに戻すことにしました。
 *
 * @param elem const apr_xml_elem * CDATAを持つXMLエレメントノード
 * @param pool apr_pool_t * 作業用のプール
 * @param strip_white int テキストの前後から空白記号を除去するか(0:しない/1:する)
 * @return const char * 取得したテキスト
 */
DIVY_DECLARE(const char *) divy_xml_get_cdata(const apr_xml_elem *elem,
						apr_pool_t *pool, int strip_white);

/**
 * 最終アクセス情報を更新の必要性を判断し、必要であれば更新する。
 * (note)
 * 	更新が不要であると判断する条件は以下の通り。
 * 	・SEARCH, PROPFIND, OPTIONS の場合
 * 	・"クライアント更新"領域に対するGETリクエストである場合
 * 	・同一KeepAliveの中で、既に更新されている場合
 *
 * @param r request_rec *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) dav_divy_update_lastaccess(request_rec *r);

/**
 * data が示すURLパラメータデータ(POST, GETと共に提出されるデータ)を
 * 解析して、キー値と値からなるハッシュに焼きなおして返却する。
 * [ parse_h の内容 ]
 *     * key : URLパラメータのキー値
 *     * val : divy_linkedlist_t* からなるリスト
 *
 * @param p apr_pool_t * parsed_h が割り当てられるプール
 * @param data const char * URLパラメータ
 * @param parsed_h apr_hash_t ** 解析されたURLパラメータを持つハッシュ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */ 
DIVY_DECLARE(int) dav_divy_parse_urlparam(apr_pool_t *p,
						const char *data,
						apr_hash_t **parsed_h);

/**
 * 指定されたuri の親リソースURIを算出して返却する。
 *
 * なお、算出されたURIは単に計算によって求められたものであり、
 * それが現実に存在するかどうかまでは検知しません。
 *
 * @param p apr_pool_t * 算出したURI文字列を割り当てるプール
 * @param uri const char * 対象URI
 * @return char * 親リソースURI. uri がNULLまたは空文字であればNULLを返却.
 */
DIVY_DECLARE(char *) divy_get_parenturi(apr_pool_t *p, const char *uri);

/**
 * 指定されたuri の全ての親リソースURIを算出して返却する。
 *
 * なお、算出されたURIは単に計算によって求められたものであり、
 * それが現実に存在するかどうかまでは検知しません。
 *
 * (例) uri = "/a/b/c/d"
 *   --> "/a", "/a/b", "/a/b/c"
 *
 * @param p apr_pool_t * 算出したURI文字列を割り当てるプール
 * @param uri const char * 対象URI
 * @return divy_cset_t * 親リソースURIからなる集合。
 * 		uri がNULLまたは空文字であればNULLを返却.
 */
DIVY_DECLARE(divy_cset_t *) divy_get_parenturi_set(apr_pool_t *p, const char *uri);

/**
 * 指定されたuri 自身および全親リソースURIを算出して返却する。
 *
 * なお、算出されたURIは単に計算によって求められたものであり、
 * それが現実に存在するかどうかまでは検知しません。
 *
 * @param p apr_pool_t * 算出したURI文字列を割り当てるプール
 * @param uri const char * 対象URI
 * @return divy_cset_t * 自身および親リソースURIからなる集合。
 * 		uri がNULLまたは空文字であればNULLを返却.
 */
DIVY_DECLARE(divy_cset_t *) divy_get_uri_set(apr_pool_t *p, const char *uri);

/**
 * User-Agent: ヘッダの内容を取得する。
 *
 * @param r request_rec *
 * @return const char * User-Agentヘッダの内容
 */
DIVY_DECLARE(const char *) dav_divy_get_user_agent(request_rec *r);

/**
 * Accept-Language から言語パラメータを取得して返却する.
 * [ 書式 ]
 * 	xxx;q=0.n, yyy;q=0.m, zzz
 * (note)
 *
 * @param r request_rec *
 * @return divy_language_tag * 言語パラメータのリスト(Accept-Language の指定順)
 */
DIVY_DECLARE(divy_language_tag *) divy_get_accept_language_param(request_rec *r);

/**
 * Cookie: クッキーヘッダを取得する
 *
 * @param r request_rec *
 * @return const char * Cookie ヘッダの内容
 */
DIVY_DECLARE(const char*) dav_divy_get_cookie(request_rec *r);

/**
 * Cookieに設定されている言語種別を取得する
 *
 * @param r request_rec *
 * @return const char * (NULLあり)
 */
DIVY_DECLARE(const char*) dav_divy_get_cookie_lang(request_rec *r);

/**
 * サーバで使用する言語パラメータ名称を取得する.
 *
 * (note) divy_get_accept_language_param との違い
 * 	関数 divy_get_accept_language_param はクライアントから提出された
 * 	Accept-Languageをそのまま取得しますが、この関数ではApache環境
 * 	変数の値や特殊なマッピング規則に基づいてAccept-Languageを
 * 	"ja" かそれ以外にマッピングし直します。
 *
 * @param r request_rec *
 * @return const char * 言語パラメータ(en, ja)
 */
DIVY_DECLARE(const char *) divy_get_language_param(request_rec *r);

/**
 * デフォルトの言語パラメータを取得する.
 *
 * @param r request_rec *
 */
const char * divy_get_default_languagetag(request_rec *r);

/**
 * request_rec から Content-Type の値を取得して返却する。
 *
 * @param r request_rec *
 * @return const char * 取得したContent-Typeの値。
 *                      取得できなければ NULL を返す。
 */
DIVY_DECLARE(const char *) divy_get_content_type(request_rec *r);

/**
 * request_rec から Content-Language の値を取得して返却する。
 *
 * @param r request_rec *
 * @return const char * 取得したContent-Language の値。
 *                      取得できなければ NULL を返す。
 */
DIVY_DECLARE(const char *) divy_get_content_language(request_rec *r);

/**
 * request_rec から Content-Length の値を取得して返却する。
 *
 * @param r request_rec *
 * @return apr_int64_t  取得したContent-Length の値。
 *                      取得できなければ 0 を返す。
 */
DIVY_DECLARE(apr_int64_t) divy_get_content_length(request_rec *r);

/**
 * request_rec から X-Expected-Entity-Length の値を取得して返却する。
 *
 * @param r request_rec *
 * @return apr_int64_t	取得したX-Expected-Entity-Length の値。
 *						取得できなければ 0 を返す。
 */
DIVY_DECLARE(apr_int64_t) divy_get_x_expected_entity_length(request_rec *r);

/**
 * 指定されたuri を "http://server:port/uri" 形式のURLにして返却する。
 * (note) 2007/04/25 Wed takehara
 *   この関数は、リバースプロキシ環境下において正しく動作しません.
 *   バイナリ互換性のためだけに残しておきますが、今後は利用しないで下さい.
 *   変わりにdivy_construct_url2() を使って下さい.
 *
 * この関数は、ap_construct_url と殆ど同じですが、UseCanonicalName
 * ディレクティブがOffであっても正しいポート番号を返却します。
 * なお、ホスト名パートは、UseCanonicalNameの設定に従います。
 *
 * (note) この関数導入の背景
 * 	この関数の殆どは、ap_construct_url のコピーコードです。
 * 	ap_construct_url はUseCanonicalNameの値によってクライアント提出値を
 * 	使用するか、DNSルックアップするかを判定します。しかし、パフォーマンス
 * 	の観点からUseCanonicalName Onとすることは有り得ない為、結果、多くの場合
 * 	ap_construct_url では正しいポート番号を取得出来ないことになります。
 * 	何故なら、リクエストURIにポート番号がついていないからです。
 * 	これでは困ってしまうので、UseCanonicalName Offであっても、正しいポート
 * 	番号を取得するためにこの関数を作成しました。ap_construct_url の実装が
 * 	変更されたら、こちらのコードも同期する必要があるでしょう。
 *
 * @param p apr_pool_t * 返却値が割り当てられるプール
 * @param host const char * URLのホスト名パート(NULL だと自動的につけます)
 * @param uri const char * 組み立てるURI文字列
 * @param r request_rec * リクエスト構造体
 * @return char * 組み立てられたURL文字列
 */
DIVY_DECLARE(char *) divy_construct_url(apr_pool_t *p, const char *host,
					const char *uri, request_rec *r);

/**
 * 指定されたuri を指定された種類type に応じて適切なフルURL (schema://host:port/uri)を
 * 組み立てて返却する.
 * (note)
 *   * この関数はリバースプロキシに対応する目的で導入されたdivy_construct_url の改良版です.
 *     今後は、こちらを利用して下さい.
 *
 *   * (2007/04/27 Fri)
 *     現時点では、urltype == DIVY_URLTYPE_PROTECTED が正しく動作しません. Reverse-Proxy からの
 *     リクエストスキーマを正しくautodetect できないためです. 少なくともSquid は. 使用しないこと.
 *
 * @param p apr_pool_t * 返却値が割り当てられるプール
 * @param type int 組み立てるURLの種類 (DIVY_URLTYPE_PUBLIC / DIVY_URLTYPE_PROTECTED / DIVY_URLTYPE_INNER)
 * @param uri const char * ホストパートの後ろに付くURI文字列
 * @param r request_rec * リクエスト構造体
 * @param url_segment divy_buildurl_segment * 組み立てれたURL文字列を分解した情報. 不要なればNULLにすること
 * @return char * 組み立てられたURL文字列
 */
DIVY_DECLARE(char *) divy_construct_url2(apr_pool_t *p, int urltype,
					const char *uri, request_rec *r, divy_buildurl_segment *url_segment);

/**
 * 専用のメモリアロケータを持つプール(r->pool のサブプール) を返却する。
 * 既に作成済みであればそれを返します。そうでなければ生成します。
 *
 * 返却されるプールの特徴:
 * ------------------------
 *   * r->pool のサブプールですので、r->poolとライフサイクルとを共にします。
 *     つまり、r が確定しないステージではこの関数を利用できませんし、r が破棄される
 *     ステージでも駄目です。
 *   * リクエストが終わるとこのプールが利用していたメモリ領域はOSに返却されます
 *     (pchild のようにメモリ領域のキャッシュは行われません)
 *
 * (note) 使い道
 *   * リクエストが終われば用の無いオブジェクトの確保
 *   * アロケートサイズが大きすぎてメモリキャッシュに入れるには気の引けるオブジェクトの確保
 *
 * (note)
 *   * r->pool が破棄された後、この関数を使って得たプール及び、このプールから割り当てられた
 *     オブジェクトは利用できません。
 *   * 取得したプールを破棄する必要はありません。こちらで面倒を見ます。
 *     逆に破棄してはなりません。
 *
 * @param r request_rec *
 * @return apr_pool_t * 生成されたサブプール
 * 	何らかの理由でアロケートに失敗した場合にはNULLを返す。
 */
DIVY_DECLARE(apr_pool_t *) divy_get_request_temporary_pool(request_rec *r);

/**
 * リクエストのメソッド番号を取得する。(プロトコルゲートウェイ対応)
 * (note)
 * 	r->method_number を直接参照するのではなく、この関数で必ず取得して下さい。
 *
 * @param r request_rec *
 * @return int メソッド番号 (see httpd.h)
 */
DIVY_DECLARE(int) divy_get_method_number(request_rec *r);

/**
 * リクエストのメソッドの文字列を取得する. (プロトコルゲートウェイ対応)
 * (note)
 * 	r->method を直接参照するのではなく、この関数で必ず取得して下さい。
 *
 * @param r request_rec *
 * @return int メソッド番号 (see httpd.h)
 */
DIVY_DECLARE(char *) divy_get_method_name(request_rec *r);

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * r->pool にキャッシュされたパスワードポリシー情報を取得する.
 * キャッシュされていなければDB から取得します.
 *
 * @param r request_rec *
 * @param policyid int ポリシーID
 * @param use_cache int 必ずキャッシュから取得する. DB は見ない
 * @param divy_rdbo_passwordpolicy ** 取得したパスワードポリシー
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_get_cached_passwordpolicy(request_rec *r, int policyid,
		int use_cache, divy_rdbo_passwordpolicy **passpolicy_pr);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * Via リクエストヘッダをパースして返却する.
 *
 * @param r request_rec *
 * @return divy_httpheader_via *
 */
DIVY_DECLARE(divy_httpheader_via *) divy_parse_via_header(request_rec *r);

/**
 * クライアントがリクエストしてきたポート番号とスキーマを調べて返却する.
 *
 * @param r request_rec * リクエストパラメータ
 * @param schema char ** スキーマ名("http", "https"へのポインタ)
 * @param port int * ポート番号
 * @return int 処理ステータス (1: 失敗 / 0: 成功)
 */
DIVY_DECLARE(int) divy_get_request_conninfo(request_rec *r, char **schema, int *port);

/**
 * r->pool にキャッシュされたアクセスユーザの所属グループ情報を取得する.
 * キャッシュは関数divy_rdbo_cache_userinfo() の中で行われます.
 *
 * @param r request_rec *
 * @param apr_hath_t ** グループコレクションURIをキーとしてグループ情報を持つハッシュへのポインタ
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_get_cached_availablegroup(request_rec *r, apr_hash_t **grp_h);

/**
 * r->pool にキャッシュされたアクセスユーザの所属グループ情報の中から
 * uri のリソースが所属するグループのグループ拡張ステータスを取得する.
 * キャッシュは関数divy_rdbo_cache_userinfo() の中で行われます.
 *
 * @param r request_rec *
 * @param uri const char * 対象リソースのURI (グループコレクションに調整しなくてもいい)
 * @param grp_extstatus divy_rdbo_extstatus ** 取得したグループ拡張ステータス
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_get_cached_availablegroupextstatus(request_rec *r,
		const char *uri, divy_rdbo_extstatus **grp_extstatus);

/**
 * r->pool にキャッシュされたアクセスユーザの所属グループ情報の中から
 * uri のリソースが所属するグループ情報を取得する.
 * キャッシュは関数divy_rdbo_cache_userinfo() の中で行われます.
 *
 * @param r request_rec *
 * @param uri const char * 対象リソースのURI (グループコレクションに調整しなくてもいい)
 * @param grp_pr divy_rdbo_grp ** 取得したグループ情報
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_get_cached_availablegroup_by_uri(request_rec *r,
									const char *uri, divy_rdbo_grp **grp_pr);

/**
 * 公開チケット用絶対アドレスURL文字列を生成する.
 *
 * 短縮URLをサポートしている場合
 * $schema://$root/.st/JnDmxDnes
 * 短縮URLをサポートしていない場合
 * $schema://$root/.ticket?u=xxx&WEBDAV_METHOD=PROPFIND
 *
 * @param r request_rec *
 * @param p apr_pool_t *
 * @param rdb_r divy_rdbo_resource * リソース情報
 * @return char * 生成された監視フォルダアクセスURL文字列
 * 	uri がNULLであれば空文字を返す
 */
DIVY_DECLARE(char *) divy_construct_oneclick_url(request_rec *r, apr_pool_t *p, divy_rdbo_resource *rdb_r);

/**
 * オーナownerid が作ったユーザは、アクセスユーザ(r->user) から見ると「Otherユーザ」かどうか.
 *
 * @param r request_rec *
 * @param ownerid const char * ユーザのオーナ
 * @return int Otherユーザである(1) / Otherユーザではない(0)
 */
DIVY_DECLARE(int) divy_is_otheruser(request_rec *r, const char *ownerid);

/**
 * リクエストされているURLのFinalPathがリソースフォークファイルかを調べる
 * 厳密なリソースフォークの評価は行わない単純にURIとエージェントを確認する
 * だけです。
 * リソースフォークの条件
 * FinalPathセグメントの先頭二文字が == "._"
 * user agentがWebDAVFS/で始まる
 * 
 * @param r request_rec *
 * @return int リソースフォーク(1) / それ以外(0)
 *
 */
DIVY_DECLARE(int) divy_is_resourcefork(request_rec *r);

/**
 * ユーザエージェントがOffice系なのかを調べます
 *
 * @param r request_rec *
 * @return int Office(1) / それ以外(0) 
 */
DIVY_DECLARE(int) divy_is_Office(request_rec *r);

/**
 * ブラウザのパスワード変更表示をなくすか？
 *
 * @param request_rec *
 * @param int 表示しないかどうか (1: 表示しない / 0: 表示する)
 */
DIVY_DECLARE(int) divy_disable_change_password(request_rec *r);

/**
 * 同じキーが経過時間にあったかどうかを調べます。存在しない場合は、記録します。
 *
 * @param r	request_rec *
 * @param key const char*
 * @param value const char*
 * @param timeout apr_uint32_t
 *
 * @return int (1: 存在した / 0: しなかった)
 */
DIVY_DECLARE(int) divy_has_action_in_time(request_rec *r, const char* key, const char* value, apr_uint32_t timeout);

/**
 * カスタムのイメージパスを取得する
 *
 * @param request_rec *
 * @param char * fname	ファイル名
 *
 * @return char *	イメージのフルパス	
 */
DIVY_DECLARE(char *) divy_get_custom_image_path(request_rec *r, const char* fname);

/**
 * MPMのさまざまな情報を所得して設定する
 *
 * @param none
 * @return none
 */
DIVY_DECLARE(void) dav_divy_query_mpm_info(void);

/**
 * MPMの取得済みの情報を入手する
 *
 * @param mpmq divy_mpmq (see util/include/util_thread.h)
 * @return int 結果
 */
DIVY_DECLARE(int) dav_divy_get_mpm_info(divy_mpmq mpmq);
	
#ifdef __cplusplus
}
#endif

#endif /* INCLUDE_UTIL_H */

