/**
 * $Id$
 *
 * util_auth.h
 *
 * Authユーティリティ関数用ヘッダファイル
 *
 *
 */

#ifndef INCLUDE_UTIL_AUTH_H
#define INCLUDE_UTIL_AUTH_H

#include "mod_dav_tf.h"
#include "tfr.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define fixed value
  ----------------------------------------------------------------------------*/
/* セッション名 */
#define TF_COOKIE_SES_NAME "tf-session"	

/*------------------------------------------------------------------------------
  Declare structure
  ----------------------------------------------------------------------------*/
typedef struct __divy_auth_session	divy_auth_session;

struct __divy_auth_session {
	char *sid;			/* セッションID */
	char *uid;			/* ユーザーID	*/
	char *password;		/* パスワード	*/
	char *lang;			/* 言語			*/
	char *validmail;	/* (2FA) メールアドレス */
	char *authcode;		/* (2FA) 文字列 */
	char *uniqueid;		/* ユニークID	*/
};

/*
 *
 */
enum {
	DIVY_AUTH_SESSION_OK = 0,
	DIVY_AUTH_SESSION_ERROR,
	DIVY_AUTH_SESSION_TIMEOUT,
};
/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/

/**
 * Cookieを取得してログインを正常に行っているか調べる
 *
 * @param r request_rec*
 * @param password char**
 * @param sessionid char**
 * @return 1: 正常 / 0:失敗
 */
DIVY_DECLARE(int)
divy_util_auth_parse_cookie(request_rec* r, char** password, char** sessionid);

/**
 * リクエストのクライアントがブラウザかどうかを調べる
 * 調べ方がAcceptヘッダの有無だけという単純なチェックになっている
 *  @param r request_rec*
 *  @return 1: ブラウザ / 0: それ以外
 */
DIVY_DECLARE(int) divy_util_auth_request_is_browser(request_rec *r);

/**
 * 認証を行わない(スルーする)URIかを調査する
 * @param r request_rec*
 * @return int 1: スルー（認証しないでいい) / 0: 認証が必要
 */
DIVY_DECLARE(int) dav_divy_auth_through_uri(request_rec *r);

/**
 *  許可された場所からのパスワードなしログインなのかを調べる
 *
 *  @param r request_rec*
 *	@param sent_pw const char*	送信されたパスワード
 *  @return 1: 許可されている / 0: 許可されない
 */
DIVY_DECLARE(int) divy_util_auth_allow_ignore_login(request_rec *r, const char* sent_pw);

/**
 * login画面へ移動する
 *
 * @param r request_rec*
 * @param sid const char*	セッションID
 * @param reason int 遷移理由(エラーコード)
 * @return int (HTTP_STATUS)
 */
DIVY_DECLARE(int) divy_redirect_login(request_rec *r, const char *sid, int reason);

/**
 * Basic認証のヘッダーを作成する
 * @param r request_rec *
 * @return void
 *
 */
DIVY_DECLARE(void) divy_note_auth_failure(request_rec *r);

/**
 * セッションクッキーを削除するヘッダを作る
 * @param r request_rec*
 */
DIVY_DECLARE(void) divy_set_delete_session_cookie(request_rec *r);

/**
 * 	セッションから該当のデータを削除する
 * 	@param r	request_rec *
 * 	@param sid	const char *
 *
 * 	@return void
 */
DIVY_DECLARE(void) divy_util_delete_session(request_rec *r, const char *sid);

/**
 * セッションにユーザー情報を追加する
 * 	@param r		request_rec *
 * 	@param uid 		const char* 	ユーザーID
 * 	@param password const char*		パスワード
 * 	@param sid		char**			セッションID
 * 	@param int 		tmp				一時利用なら 1
 * 	@return 0: 成功 / 1:失敗
 *
 * 	(note)
 * 	tmp の-1
 * 	通常memcacheのタイムアウトはディレクティブの TfSessionTimeOutを利用します
 * 	1 にする場合は、タイムアウトは固定値で1となります。（つまり1秒）
 * 	実際には、POST認証を行いリダイレクトされて認証が行われる為の一時的な
 * 	扱いを行うセッション利用をするために利用します。
 *
 */
#if 0
DIVY_DECLARE(int) 
divy_util_auth_set_memcache_session(request_rec *r, const char* uid,
										 const char* password, char** sid,
										 int tmp);
#endif
DIVY_DECLARE(int) 
divy_util_auth_set_memcache_session(request_rec *r, divy_auth_session *session, int tmp);

/**
 * 指定されたセッションをリフレッシュする。内容の書き換えはない。
 * @param r		request_rec *
 * @param sid	const char* セッション
 *
 * @return 0:成功 / 1:失敗
 */
DIVY_DECLARE(int)
divy_util_auth_refresh_session(request_rec *r, const char* sid);

/**
 * 指定されたセッションを削除します。無い場合はリクエストから取得します
 * @param r		request_rec *
 * @param sid	const char* セッション
 *
 */
DIVY_DECLARE(int)
divy_util_auth_remove_session(request_rec *r, const char* sid);

/**
 * BOXの認証をクリアしているかを調べます
 * @param r request_rec*
 * @param const char* boxへのアクセスキー
 * @param update int アップデートを行う
 * @return int 0 : 成功 / 1: 失敗（認証画面を出しなさい)
 */
DIVY_DECLARE(int) divy_util_box_authenticate_user(request_rec *r, const char* key, int update);

DIVY_DECLARE(int) divy_util_box_set_cookie(request_rec *r, divy_auth_session *session);

DIVY_DECLARE(int) divy_util_auth_session_manager(request_rec* r, divy_auth_session *session);

/**
 * BOXを展開できるユーザかを調べます
 * @param r request_rec*
 * @param id ユーザID若しくはSID
 * @param boxvalidmailaddr boxを設定したユーザが入力したメール
 * @return int (-1: 未設定 / 0:　成功 / 1: 失敗)
 */
DIVY_DECLARE(int)
divy_util_box_validate_user(request_rec *r,
								const char* id, const char* boxvalidmailaddr);

/**
 * セッション上のパスワードを変更する
 *
 * @param r 			request_rec*
 * @param uid			const char*	 ユーザーID
 * @param newpassword	const char*	 パスワード
 * @return int 0 : 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_util_auth_update_session_password(request_rec* r, const char* uid, const char* newpassword);

/**
 * セッションからユーザ情報を取得する
 *
 * @param r				request_rec *		リクエスト構造体
 * @param session		divy_auth_session* セッション
 * @return 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_util_auth_get_memcache_userinfo(request_rec* r, divy_auth_session *session);

/**
 * 指定されたcookieが送られてきたか?
 * @param r request_rec * リクエスト構造体
 * @param name	const char*	Cookie名
 * @return int (1: ある / 0: ない)
 */
DIVY_DECLARE(int) divy_util_auth_has_cookie(request_rec *r,
												const char* cookiename);

/**
 * 2FA認証を行ったユーザか？
 * @param r 		request_rec *	リクエスト構造体
 * @param id		const char*		ID文字列 (ユーザIDやセッションID）
 * @return int 		認証をパスしているかどうか（-1:期限切れ/1:パス/0:していない）
 */
DIVY_DECLARE(int) divy_util_confirm_2FA_passed(request_rec *r, const char* id);

#ifdef __cplusplus
}
#endif

#endif /* INCLUDE_UTIL_AUTH_H */
 
