/*
 *	$Id: libtfcrypt.h,v 1.7 2005/06/29 15:31:00 yone Exp $
 *
 *	opensslの暗号化を行なう
 *	(note)
 *	引数がcharであることに注意してください。
 */
#ifndef INCLUDE_UTIL_CRYPT_H
#define INCLUDE_UTIL_CRYPT_H

#include "config.h"
#include "httpd.h"
#include "apr.h"
#include "apr_pools.h"
#include "mod_dav_tf.h"

#ifdef __cplusplus
extern "C" {
#endif 

/*
 *	内部で利用するコンテキスト
 */
typedef struct __tf_crypt_info_t	tf_crypt_info_t;


/*
 *	暗号化の種類
 *	tf_crypt_stream_init()のcipherで利用します
 */
typedef enum __TFCRTYPT_CIPHERTYPE {
	TFCRYPT_CIPHER_NULL,		    /* none cipher  */
#if 0
	TFCRYPT_CIPHER_DES_CBC,         /* DES CBC      */
	TFCRYPT_CIPHER_DES3_CBC,        /* DES3 CBC     */
	TFCRYPT_CIPHER_BF_CBC,		    /* blowfish CBC */
	TFCRYPT_CIPHER_CAST5_CBC,		/* CAST5 CBC	*/
	TFCRYPT_CIPHER_AES128_CBC,	    /* AES 128 CBC  */
	TFCRYPT_CIPHER_AES192_CBC,	    /* AES 192 CBC  */
	TFCRYPT_CIPHER_AES256_CBC,      /* AES 256 CBC  */
#endif
	TFCRYPT_CIPHER_DES_CFB,         /* DES CFB      */
	TFCRYPT_CIPHER_DES_OFB,         /* DES OFB      */
	TFCRYPT_CIPHER_DES3_CFB,		/* DES3 CFB     */
	TFCRYPT_CIPHER_DES3_OFB,        /* DES3 OFB     */
	TFCRYPT_CIPHER_BF_CFB,		    /* blowfish CFB */
	TFCRYPT_CIPHER_BF_OFB,		    /* blowfish OFB */
	TFCRYPT_CIPHER_CAST5_OFB,	    /* CAST5 OFB    */
	TFCRYPT_CIPHER_CAST5_CFB,       /* CAST5 CFB    */
	TFCRYPT_CIPHER_AES128_CFB,	    /* AES 128 CFB  */
	TFCRYPT_CIPHER_AES128_OFB,      /* AES 128 OFB  */
	TFCRYPT_CIPHER_AES192_CFB,	    /* AES 192 CFB  */
	TFCRYPT_CIPHER_AES192_OFB,      /* AES 192 OFB  */
	TFCRYPT_CIPHER_AES256_CFB,      /* AES 256 CFB  */
	TFCRTPY_CIPHER_AES256_OFB,      /* AES 256 OFB  */
	TFCRTPY_CIPHER_NULL,            /* NULL         */
} TFCRYPT_CIPHERTYPE;

typedef struct __TFCRYPT_CIPHERTYPE_TABLE {
	TFCRYPT_CIPHERTYPE type;
	const char*        string;
} _TFCRYPT_CIPHERTYPE_TABLE;

static const _TFCRYPT_CIPHERTYPE_TABLE TFCRYPT_CIPHERTYPE_TABLE[] = 
{
	{   TFCRYPT_CIPHER_DES_CFB,    "des-cfb"	},
	{   TFCRYPT_CIPHER_DES_OFB,    "des-ofb"    },
	{   TFCRYPT_CIPHER_DES3_CFB,   "des3-cfb"   },
	{   TFCRYPT_CIPHER_DES3_OFB,   "des3-ofb"   },
	{   TFCRYPT_CIPHER_BF_CFB,     "bf-cfb"     },
	{   TFCRYPT_CIPHER_BF_OFB,     "bf-ofb"     },
	{   TFCRYPT_CIPHER_CAST5_OFB,  "cast5-ofb"  },
	{   TFCRYPT_CIPHER_CAST5_CFB,  "case5-cfb"  },
	{   TFCRYPT_CIPHER_AES128_CFB, "aes128-cfb" },
	{   TFCRYPT_CIPHER_AES128_OFB, "aes128-ofb" },
	{   TFCRYPT_CIPHER_AES192_CFB, "aes192-cfb" },
	{   TFCRYPT_CIPHER_AES192_OFB, "aes192-ofb" },
	{   TFCRYPT_CIPHER_AES256_CFB, "aes256-cfb" },
	{   TFCRTPY_CIPHER_AES256_OFB, "aes256-ofb" },
	{	TFCRYPT_CIPHER_NULL,		NULL		}  /* centinel */
};

/* 暗号モード */
enum TFCRYPT_CIPHER_MODE {
	TFCRYPT_CIPHERMODE_ENCRYPT = 1,	/* encrypt      */
	TFCRYPT_CIPHERMODE_DECRYPT = 2	/* decrypt      */
};

/* public function */

/**
 * ファイルの暗号化を行う
 * 
 * @param r           request_rec * 
 * @param filepath    const char *      暗号化を行うファイル
 * @param type        int               暗号化の種類
 * @param key         const char *		キー
 * @param iv          unsigned char *	情報データ
 * @return const char* 暗号化されたファイルパス
 */
DIVY_DECLARE(const char*)
tf_encrypt_file(request_rec *r, const char *filepath, int type, const char *key, unsigned char* iv);

/**
 * ファイルの復号化を行う
 * 
 * @param r           request_rec * 
 * @param filepath    const char *      復号化を行うファイル
 * @param type        int               暗号化の種類
 * @param key         const char *		キー
 * @param iv          unsigned char *	情報データ
 * @return const char* 暗号化されたファイルパス
 */
DIVY_DECLARE(const char*)
tf_decrypt_file(request_rec *r, const char *filepath, int type, const char *key, unsigned char* iv);

/**
 * ファイルをSHA1ダイジェストする 
 * 
 * @param	p			apr_pool_t*	
 * @param 	filepath	const char*		ファイルパス
 * @param	digest		unsigned char*	ダイジェスト値(160bits = 20bytes)
 *
 * @return  1:　成功 / 0: 失敗
 */
DIVY_DECLARE(int)
tf_cipher_file2sha1(apr_pool_t* p, const char *filepath, unsigned char* digest);

/**
 * 文字列をSHA1ダイジェストする 
 * @param p             apr_pool_t*
 * @param string        const char* SHA1する文字列
 * @param	digest		unsigned char*	ダイジェスト値(160bits = 20bytes)
 *
 * @return  1:　成功 / 0: 失敗
 */ 
DIVY_DECLARE(int)
tf_cipher_str2sha1(apr_pool_t* p, const char* string, unsigned char* digest);

/**
 *	暗号化を行なう構造体の初期化を行なう。
 *
 *	@param	info	tf_crypt_info_t**		コンテキスト
 *	@param	mode	int						( 1: 暗号化 / 2: 複合化 )
 *	@param	cipher	int                     利用する暗号タイプ
 *
 */
int
tf_crypt_stream_init(tf_crypt_info_t** info, int mode, int cipher);

/**
 *	処理に利用するキーを設定する
 *
 *	@param	info	tf_crypt_info_t**		コンテキスト
 *	@param	key		const char*	            キー
 *	@param	keylen	int						キーの長さ
 *
 */
int
tf_crypt_stream_set_key(tf_crypt_info_t* info, const char* key, int keylen);

/**
 * tf_crypt_info_tの内容にもとづいて処理を開始する
 *
 *	@param	info	tf_crypt_info_t**		コンテキスト
 *	@param	out		char*					処理された結果が戻ってくるバッファ
 *	@param	in		const char*             処理するバッファ
 *	@param	buflen	int                     バッファの長さ
 *	@return int                             1: 成功 / 2: 失敗
 *
 *	(note)
 *	outとinのバッファのサイズはbuflenが示す長さをあらかじめ用意
 *	する事。
 *
 */
int
tf_crypt_stream(tf_crypt_info_t* info, char* out, const char* in, int buflen);

/**
 *	終了処理
 *
 *	@param	info	tf_crypt_info_t*        コンテキスト
 *	@return int                             1: 成功 / 0: 失敗
 *
 *	(note)
 *	tf_crypt_infoを開放します。tf_crypt_stream()の成功・失敗関わらず
 *	必ず実行してください。
 */
int
tf_crypt_stream_cleanup(tf_crypt_info_t* info);


/**
 *	文字列を暗号化する
 *
 *	@param	out		char*			結果	[out]
 *	@param	outlen	outlen*			結果の長さ[out]
 *	@param	in		const char*		入力[in]
 *	@param	inlen	int				入力の長さ[in]
 *	@param	key		const char*		NULLで終わる文字列[in]
 *	@param	mode	int				1: 暗号化 / 2:複合化
 *	@param	cipher	int             利用する暗号タイプ
 *
 *	@return	1: 成功 / 0: 失敗
 *
 *	(note)
 *	内部でストリーム暗号をかけます。outバッファはinと同じサイズのバッファを
 *	あらかじめ与えておくこと。あとでFreeすることを忘れないようにしてください。
 *
 */
int
tf_crypt_string(char* out, int* outlen, const char* in, int inlen, const char* key, int mode, int cipher);

/**
 * 引数で渡された文字列から暗号化種別を返す
 *
 * @param string const char*
 *
 * @return int TFCRYPT_CIPHERTYPE / 0:みつからなかった
 */
int
tf_crypt_get_ciphertype(const char* string);

#ifdef __cplusplus
}
#endif

#endif


