/**
 * $Id$
 *
 * util_db.h
 *
 * TeamFileモジュールがDBプロバイダにアクセスするのに必要な手続き
 * (関数、構造体)を宣言するヘッダファイル。
 * このヘッダには、DBプロバイダとTeamFileとの間にある一定のアクセスレイヤを
 * 作る役割があります。これはDBプロバイダをApache本体から切り離すために
 * 必要となります。
 *
 * (note)
 * 	このヘッダファイルのincludeおよび宣言された構造体・関数を
 * 	DBプロバイダが利用してはなりません。
 *
 * 2004/02/08 Sun takehara NEW
 */
#ifndef INCLUDE_UTIL_DB_H
#define	INCLUDE_UTIL_DB_H

#include "apr.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "httpd.h"

#include "tfr.h"
#include "tf_linkedlist.h"
#include "tf_cset.h"
#include "tf_db.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define fixed values and define types.
  ----------------------------------------------------------------------------*/
/**
 * DBコネクションプールのデフォルトリフレッシュ間隔[sec]
 * (note) 指定可能sec
 * 	DB_CONNMGR_PROCESS_INTERVAL < n < apr_time_t型の最大値
 * (note)
 * 	httpd.conf に"TfDbRefreshInterval" が指定されなかった時に使用されます
 */
#define DIVY_DBCPOOL_REFRESH_INTERVAL 28800

/**
 * DB カラムの長さ(1)
 * (note) DBMS 毎に異なるものはここでは定義しません。
 */
#define DIVY_DB_RSID_LEN	"12"	/* リソースID */
#define DIVY_DB_DISPNAME_LEN	"255"	/* 表示名称 (dav_resource) */
#define DIVY_DB_ETAG_LEN	"50"	/* Etag (dav_resource) */
#define DIVY_DB_USRID_LEN	"60"	/* ユーザID */
#define DIVY_DB_FULLNAME_LEN	"60"	/* ユーザのフル名称 */
#define DIVY_DB_PHYSICPATH_LEN	"50"	/* 物理パス(dav_resource) */
#define DIVY_DB_UPDATEDT_LEN	"20"	/* 登録日付、更新日付 */
#define DIVY_DB_GRPID_LEN	"8"	/* グループID */
#define DIVY_DB_GRPNAME_LEN	"80"	/* グループ名称 */
#define DIVY_DB_DBIDNAME_LEN	"30"	/* DBMS 識別名称 */
#define DIVY_DB_NSNAME_LEN	"255"	/* ネームスペースURI(dav_namespace) */
#define DIVY_DB_DSTYPE_LEN	"10"	/* DBMS種類 (divy_dbms) */
#define DIVY_DB_DSHOSTNAME_LEN	"30"	/* DBMSホスト名称 (divy_dbms) */
#define DIVY_DB_DSUSERNAME_LEN	"50"	/* DBMSユーザID (divy_dbms) */
#define DIVY_DB_DSPASSWORD_LEN	"50"	/* DBMSパスワード (divy_dbms) */
#define DIVY_DB_USREXTSTATUS_LEN	"32"	/* ユーザの拡張ステータスの長さ */
#define DIVY_DB_ALLOWHOSTS_LEN  "255"  /* 許可ホスト */

/*
 * DB シーケンスID の桁数を表す定義値
 */
#define DIVY_DB_RES_SEQ_LEN	"12"
#define DIVY_DB_GRP_SEQ_LEN	"8"
#define DIVY_DB_SQL_SEQ_LEN	"8"
#define DIVY_DB_DS_SEQ_LEN	"4"

/*------------------------------------------------------------------------------
  PostgreSQL
 -----------------------------------------------------------------------------*/
#if defined(DIVY_DBMS_POSTGRES)

/* SQL演算子 */
/* 文字列連結演算子 */
#define DIVY_DBOP_CONCAT	"||"

/* SQL関数 */
/* SQLシーケンス作成 */
#if 0
#define DIVY_DBFUNC_CREATE_RES_SEQ "lpad(nextval(\'res_seq\'),"DIVY_DB_RES_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_GRP_SEQ "lpad(nextval(\'grp_seq\'),"DIVY_DB_GRP_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_SQL_SEQ "lpad(nextval(\'sql_seq\'),"DIVY_DB_SQL_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_DS_SEQ  "lpad(nextval(\'ds_seq\'),"DIVY_DB_DS_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_USR_SEQ "nextval(\'usr_seq\')"
#define DIVY_DBFUNC_CREATE_DP_SEQ  "nextval(\'dp_seq\')"

/* 文字列の長さを調べる */
#define DIVY_DBFUNC_CHARLEN(col)     "char_length("col")"

/* エスケープ句 */
#define DIVY_DBFUNC_ESCAPE_CLAUSE	""
#endif

#define DIVY_DBFUNC_CREATE_RES_SEQ "lpad(nextval(\'res_seq\')::text,"DIVY_DB_RES_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_GRP_SEQ "lpad(nextval(\'grp_seq\')::text,"DIVY_DB_GRP_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_SQL_SEQ "lpad(nextval(\'sql_seq\')::text,"DIVY_DB_SQL_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_DS_SEQ  "lpad(nextval(\'ds_seq\')::text,"DIVY_DB_DS_SEQ_LEN",\'0\')"

#define DIVY_DBFUNC_CREATE_USR_SEQ "nextval(\'usr_seq\')"
#define DIVY_DBFUNC_CREATE_DP_SEQ  "nextval(\'dp_seq\')"

/* 文字列の長さを調べる */
#define DIVY_DBFUNC_CHARLEN(col)     "char_length("col")"

/* エスケープ句 */
#define DIVY_DBFUNC_ESCAPE_CLAUSE	""

/*
 * 文字列の切り出し関数を埋め込むマクロ
 *
 * @param str char * ソースとなる文字列
 * @param start int 切り出し開始位置
 * @return 文字列リテラル
 */
#define DIVY_DBFUNC_SUBSTRING(str,start)  "substring("str", "start")"
#define DIVY_DBFUNC_SUBSTRING2(str,start,end)  "substring("str", "start","end")"

/*
 * マクロ DIVY_DBEXPR_CAST で使用するデータ型
 * (note) PostgreSQLでは空にする
 */
#define DIVY_COLTYPE_CHAR(len)    ""	/* 固定長文字列型 (SQL_CHAR) */
#define DIVY_COLTYPE_VARCHAR(len) ""	/* 可変長文字列型 */
#define DIVY_COLTYPE_INTEGER      ""	/* 数値型(SQL_INTEGER)       */
#define DIVY_COLTYPE_BIGINT       ""	/* 数値型(SQL_BIGINT)        */

/*
 * CAST 式用マクロ
 * 	
 * @param col キャストするカラムまたはバインド変数("?"). Double Quotaで囲むこと.
 * @param type データ型をあらわす定義値(DIVY_COLTYPE_xxxx)
 */
#define DIVY_DBEXPR_CAST(col,type)	col

/*
 * DBカラムの長さ(2)
 */
#define DIVY_DB_URI_LEN	"1000"

/*------------------------------------------------------------------------------
  Oracle
 -----------------------------------------------------------------------------*/
#elif defined(DIVY_DBMS_ORACLE)

/* SQL演算子 */
/* 文字列連結演算子 */
#define DIVY_DBOP_CONCAT	"||"

/* SQL関数 */
/* SQLシーケンス作成 */
#define DIVY_DBFUNC_CREATE_RES_SEQ "lpad(res_seq.nextval,"DIVY_DB_RES_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_GRP_SEQ "lpad(grp_seq.nextval,"DIVY_DB_GRP_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_SQL_SEQ "lpad(sql_seq.nextval,"DIVY_DB_SQL_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_DS_SEQ  "lpad(ds_seq.nextval,"DIVY_DB_DS_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_USR_SEQ "usr_seq.nextval"
#define DIVY_DBFUNC_CREATE_DP_SEQ  "dp_seq.nextval"

/* 文字列の長さを調べる */
#define DIVY_DBFUNC_CHARLEN(col)     "length("col")"

/* エスケープ句 */
#define DIVY_DBFUNC_ESCAPE_CLAUSE	"ESCAPE \'\\\'"

/*
 * 文字列の切り出し関数を埋め込むマクロ
 *
 * @param str char * ソースとなる文字列
 * @param start int 切り出し開始位置
 * @return 文字列リテラル
 */
#define DIVY_DBFUNC_SUBSTRING(str,start)  "SUBSTR("str", "start")"
#define DIVY_DBFUNC_SUBSTRING2(str,start,end)  "SUBSTR("str", "start","end")"

/**
 * マクロ DIVY_DBEXPR_CAST で使用するデータ型
 * (note) PostgreSQLでは空にする
 */
#define DIVY_COLTYPE_CHAR(len)    ""	/* 固定長文字列型 (SQL_CHAR) */
#define DIVY_COLTYPE_VARCHAR(len) ""	/* 可変長文字列型 */
#define DIVY_COLTYPE_INTEGER      ""	/* 数値型(SQL_INTEGER)       */
#define DIVY_COLTYPE_BIGINT       ""	/* 数値型(SQL_BIGINT)        */

/*
 * CAST 式用マクロ
 * 	
 * @param col キャストするカラムまたはバインド変数("?"). Double Quotaで囲むこと.
 * @param type データ型をあらわす定義値(DIVY_COLTYPE_xxxx)
 */
#define DIVY_DBEXPR_CAST(col,type)	col

/*
 * DBカラムの長さ(2)
 */
#define DIVY_DB_URI_LEN	"1000"


/*------------------------------------------------------------------------------
  Sybase
 -----------------------------------------------------------------------------*/
#elif defined(DIVY_DBMS_SYBASE)

/*------------------------------------------------------------------------------
  DB2
 -----------------------------------------------------------------------------*/
#elif defined(DIVY_DBMS_DB2)

/* SQL演算子 */
/* 文字列連結演算子 */
#define DIVY_DBOP_CONCAT	"||"

/* SQL関数 */
/* SQLシーケンス作成 */
#define DIVY_DBFUNC_CREATE_RES_SEQ "tf_lpad(rtrim(char(nextval for res_seq)),"DIVY_DB_RES_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_GRP_SEQ "tf_lpad(rtrim(char(nextval for grp_seq)),"DIVY_DB_GRP_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_SQL_SEQ "tf_lpad(rtrim(char(nextval for sql_seq)),"DIVY_DB_SQL_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_DS_SEQ  "tf_lpad(rtrim(char(nextval for ds_seq)),"DIVY_DB_DS_SEQ_LEN",\'0\')"
#define DIVY_DBFUNC_CREATE_USR_SEQ "nextval for usr_seq"
#define DIVY_DBFUNC_CREATE_DP_SEQ  "nextval for dp_seq"

/* 文字列の長さを調べる */
#define DIVY_DBFUNC_CHARLEN(col)     "length("col")"

/* エスケープ句 */
#define DIVY_DBFUNC_ESCAPE_CLAUSE	"escape \'\\\'"

/*
 * 文字列の切り出し関数を埋め込むマクロ
 *
 * @param str char * ソースとなる文字列
 * @param start int 切り出し開始位置
 * @return 文字列リテラル
 */
#define DIVY_DBFUNC_SUBSTRING(str,start)  "substr("str", "start")"
#define DIVY_DBFUNC_SUBSTRING2(str,start,end)  "substr("str", "start","end")"

/**
 * マクロ DIVY_DBEXPR_CAST で使用するデータ型
 */
#define DIVY_COLTYPE_CHAR(len)    "char("len")"		/* 固定長文字列型 (SQL_CHAR) */
#define DIVY_COLTYPE_VARCHAR(len) "varchar("len")"	/* 可変長文字列型 */
#define DIVY_COLTYPE_INTEGER      "integer"		/* 数値型(SQL_INTEGER)       */
#define DIVY_COLTYPE_BIGINT       "bigint"		/* 数値型(SQL_BIGINT)        */

/*
 * CAST 式用マクロ
 * 	
 * @param col キャストするカラムまたはバインド変数("?"). Double Quotaで囲むこと.
 * @param type データ型をあらわす定義値(DIVY_COLTYPE_xxxx)
 */
#define DIVY_DBEXPR_CAST(col,type)	"CAST("col" as "type")"

/*
 * DBカラムの長さ(2)
 */
#define DIVY_DB_URI_LEN	"800"


/*------------------------------------------------------------------------------
  MySQL
 -----------------------------------------------------------------------------*/
#elif defined(DIVY_DBMS_MYSQL)

#endif


/*--------------------------------------------------------------
  Define structure
  --------------------------------------------------------------*/
typedef struct __divy_db_transaction_ctx	divy_db_transaction_ctx;
typedef struct __divy_db_bind_ctx		divy_db_bind_ctx;	/* 不完全型 */
typedef struct __divy_db_bind_ctx_idx		divy_db_bind_ctx_idx;

/**
 * DB トランザクションを継続するために必要な情報をもつコンテキスト。
 * 複数の異なるトランザクションデータソース間に渡って、トランザクションを
 * 継続する際に使用して下さい。
 * この構造体は、トランザクションを最初に開始する側で作成し、あとは
 * 引き回して使います。
 * 
 * (note)
 * 	request 間に渡ってこのコンテキストを使ってはなりません。
 * 	DbConn がrequest間で維持されないためと、変更が確定されるかどうかは、
 * 	クライアント依存になってしまうためです。
 * 	例え変更を加えていなくても使用は厳禁です。
 */ 
struct __divy_db_transaction_ctx {

	/* 接続済みのコネクションオブジェクト */
	DbConn *dbconn;	

	/* 現在の進行中のトランザクション状態 (bit値) */
#define DIVY_TRANS_NOTREADY 0x01	/* 開始も終了もされていない             */
#define DIVY_TRANS_START    0x02	/* 開始されて、まだ終わっていない       */
#define	DIVY_TRANS_ABORT    0x04	/* 継続しているが、処理に失敗している   */
#define	DIVY_TRANS_END      0x08	/* トランザクションは終了している       */
	int status;

	/* 
	 * トランザクションを継続するために必要な"何か"を指し示すポインタ
	 *
	 * このコンテキストオブジェクトの利用者と設定者との間で
	 * トランザクションを保つために必要な情報があれば、ここに設定して
	 * 下さい。
	 */
	void *data;

	/* このコンテキストの生成に利用したプール(作業用) */
	apr_pool_t *p;
};


/*--------------------------------------------------------------
  Declare Public Functions
  --------------------------------------------------------------*/
/**
 * DBプロバイダ管理コンテキストを初期化する。
 * この関数は複数のスレッドからアクセスされてはなりません。
 * (note)
 * 	この関数は、DBコネクションを使用するどの全ての関数よりも先に呼び出される
 * 	必要があります。child_initステージが適切です。
 * 	この関数は、１度コールされると、destroy_dbprvmng_env がコールされるまで
 * 	何も実行しなくなります。(マルチスレッドは駄目)
 * (note)
 * 	この関数を複数回呼び出さないようコール元で保証して下さい。
 *
 * @param pchild apr_pool_t * Childプロセスとライフサイクルを共にするプール
 * @param s server_rec * main serverのserver_rec
 */
DIVY_DECLARE(void) init_dbprvmng_env(apr_pool_t *pchild, server_rec *s);

/**
 * DBプロバイダ管理コンテキストの終了処理
 * (note)
 * 	この関数がコールされた後、DB に対しあらゆるオペレーションは
 * 	実行してはなりません。
 * 	この関数は、child がexitする時(pchild のクリーンアップハンドラ)
 * 	に呼び出して下さい。それ以外で適切なステージがあればその限りでは
 * 	ありません。
 * (note)
 * 	この関数を複数回呼び出さないようコール元で保証して下さい。
 */
DIVY_DECLARE(void) destroy_dbprvmng_env(void);

/**
 * r->server のvirtual host に存在するDB プロバイダの中から、providerNmae が示す
 * DB プロバイダ構造体(DbDataSource)取得して返却する。
 * (note)
 * 	この関数は、mod_dav_tf モジュールでしか使用できないフック関数を使用
 * 	しています。mod_dav_tf 以外のモジュールはコールしないこと。
 *
 * @param r request_rec *
 * @param providerName const char* プロバイダ名
 * @return DbDataSource* DB プロバイダ構造体へのポインタ
 */
DIVY_DECLARE(DbDataSource *) lookup_db_provider(request_rec *r,
						const char *providerName);

/**
 * r->server のvirtual host に存在するDB プロバイダの中から、リポジトリDBの
 * DB プロバイダ構造体(DbDataSource)取得して返却する。
 * (note)
 * 	この関数は、mod_dav_tf モジュールでしか使用できないフック関数を使用
 * 	しています。mod_dav_tf 以外のモジュールはコールしないこと。
 *
 * @param r request_rec*
 * @return DbDataSource* リポジトリDBプロバイダ構造体へのポインタ
 *                       登録されていないときにはNULLを返却.
 */
DIVY_DECLARE(DbDataSource *) lookup_reposdb_provider(request_rec *r);

/**
 * 指定したproviderType が示すDBプロバイダ"種類"がリポジトリDBプロバイダとして
 * 登録されているかどうか。
 *
 * (注意) httpd.confに、TfDbDbmsType が設定されていなければなりません。
 *        設定されていない場合には、常にDB_FALSEを返却します。
 *
 * @param r request_rec *
 * @param providerType const char* 判定するプロバイダの種類名
 * @return int DB_TRUE: リポジトリDBである / DB_FALSE: リポジトリDBではない
 */
DIVY_DECLARE(int) is_reposdb_provider(request_rec *r, const char *providerType);

/**
 * リポジトリDBとの接続がアクティブなDbConn構造体を取得する。
 *
 * この関数はDB コネクションプールに対するアクセスも自動的に行います。
 * これを使用することで、DBプロバイダ使用者がプールの存在を意識する必要は
 * なくなります。
 * (note)
 * 	この関数は、リクエストプールが破棄されるまでのステージの中で使用
 * 	することが出来ます。
 * 	リポジトリDBからDbConnを取得する際には、DbDataSource のgetDbConn
 * 	ハンドラではなく、この関数を使って取得して下さい。
 * 	パフォーマンスが向上します。
 * (note)
 * 	ここで取得したDbConn をclose で閉じてはなりません。
 * 	DbConn の破棄はmod_dav_tf が面倒をみます。
 *
 * (note) この関数は、lookup_tagged_DbConn のラッパです。
 *
 * @param request_rec *r request_rec構造体へのポインタ
 * @param dbconn DbConn ** 取得したDbConnへのポインタ
 * 			何らかの原因でDbConnが取得できない時にはNULLを返却
 */
DIVY_DECLARE(void) lookup_reposdb_DbConn(request_rec *r, DbConn **dbconn);

/**
 * 指定されたproviderNameが示すDBプロバイダからアクティブなDbConn を取得する。
 *
 * この関数はDB コネクションプールに対するアクセスも自動的に行います。
 * これを使用することで、DBプロバイダ使用者がプールの存在を意識する必要は
 * なくなります。
 *
 * (note)
 * 	この関数は、リクエストプールが破棄されるまでのステージの中で使用
 * 	することが出来ます。
 * 	DbConnを取得する際には、DbDataSource のgetDbConnハンドラではなく
 * 	この関数を使って取得して下さい。
 * 	パフォーマンスが向上します。
 *
 * (note)
 * 	ここで取得したDbConn をclose で閉じてはなりません。
 * 	DbConn の破棄はmod_dav_tf が面倒をみます。
 *
 * 	常に新しいDbConnが必要であるようなケース(例えば、DBトランザクションを
 * 	プログラムフロー上のトランザクションとして使用する場合など)では、この
 * 	関数を使用することによって副作用が生じる場合もあります。これは、
 * 	この関数で取得されたDbConnが別の関数コールにおいても使用されてしまう
 * 	ためで、トランザクションが確定されてしまうかもしれないからです。
 * 	この用途には、新しいDbConnを取得する関数lookup_tagged_DbConn を使用
 * 	して下さい。

 * (note) 導入動機
 *	この関数は、１リクエストの中で、複数の接続先に何回も接続する
 *	用途が生じたため、パフォーマンスを改善する目的で導入しました。
 *	多くのRDBMSでは、初期接続に対し、比較的大きなメモリリソースと
 *	時間を必要とするため、この関数を使用することにより不要な
 *	パフォーマンス劣化を防ぐことが出来ます。
 *
 * @param request_rec *r request_rec構造体へのポインタ
 * @param providerName const char * DBプロバイダの名称 (not null !!)
 * @param dbconn DbConn ** 取得したDbConnへのポインタ
 * 			何らかの原因でDbConnが取得できない時にはNULLを返却
 */
DIVY_DECLARE(void) lookup_cached_DbConn(request_rec *r,
					const char *providerName, DbConn **dbconn);

/**
 * 指定されたproviderName のDBプロバイダから、DbConn識別タグctag を持つ
 * アクティブなコネクション(DbConn *)を取得する。
 * ctag が付いたDbConnが無ければ新たにDbConnを取得して返却します。
 * ctag がNULLの場合、lookup_cached_DbConn と同じ動作になります。
 * (note) 用途
 * 	この関数は、DBとの接続を1リクエストに渡って排他的に使用したい場合に
 * 	使用することが出来ます。同じctag を指定する限り、1リクエストの間では
 * 	常に同じDbConnを取得することが可能です。
 * 	それ以外の用途では、lookup_cached_DbConnやlookup_reposdb_DbConn を
 * 	使用して下さい。
 * (note)
 * 	この関数は、リクエストプールが破棄されるまでのステージの中で使用
 * 	することが出来ます。
 * (note)
 * 	ここで取得したDbConn をclose で閉じてはなりません。
 * 	DbConn の破棄はmod_dav_tf が面倒をみます。
 *
 * @param request_rec *r request_rec構造体へのポインタ
 * @param ctag const char * 取得したいDbConnを識別するタグ名称
 * @param providerName const char * DBプロバイダの名称
 * @param dbconn DbConn ** 取得したDbConnへのポインタ
 * 			何らかの原因でDbConnが取得できない時にはNULLを返却
 */
DIVY_DECLARE(void) lookup_tagged_DbConn(request_rec *r, const char *ctag,
					const char *providerName, DbConn **dbconn);

/**
 * リポジトリDBのDBMS接続情報をサーバコンフィグから取り出し
 * DB プロバイダの環境を初期化する。
 * (note)
 *	repos にセットされる文字列の領域はr->pool から確保されます。
 *	テスト用途にしか使用されていません。
 *
 * @param r request_rec *
 * @param repos DbDataSource * リポジトリDB プロバイダ構造体へのポインタ
 * @return RET_STAT 処理終了ステータスコード. DB_SUCCESS: 成功 / DB_ERROR: 失敗
 */
DIVY_DECLARE(RET_STAT) init_reposdb_provider(request_rec *r, DbDataSource *repos);

/**
 * dbconn が使用不可であるとマークする。
 * (note)
 * 	使用不可としてマークされたコネクションはリクエスト終了後、自動的に
 * 	破棄されます。
 *
 * @param dbconn DbConn * マーク対象のDbConn
 */
DIVY_DECLARE(void) mark_inactive_dbconn(DbConn *dbconn);

/**
 * 指定されたproviderType のDBプロバイダにライセンスがあるかどうかを判定する。
 *
 * (note) ライセンスがあると判断する条件
 * 	(1) DBプロバイダライセンスキーが登録されていること
 * 	(2) ライセンス期限内もしくは、無制限ライセンスであること
 *
 * @param p apr_pool_t *
 * @param s server_rec *
 * @param providerType const char * プロバイダの種類
 * @return int ライセンスがあるかどうか
 * 	DB_LICENSE_VALID   : 持っている
 * 	DB_LICENSE_NOTHING : 持っていない
 * 	DB_LICENSE_EXPIRE  : 持っていたが切れてしまった
 */
DIVY_DECLARE(RET_STAT) divy_db_has_license(apr_pool_t *p, server_rec *s,
		 				const char *providerType);

/**
 * ライセンスがあって使用可能な(ロードされている)DBプロバイダの種類一覧を返却する。
 * (note)
 * 	この関数はmutex を使用します。他のmutexロックスコープの中から
 * 	コールしてはなりません。
 *
 * @param r request_rec *
 * @param type_set apr_hash_t ** 有効なDBMS種類一覧
 * 			有効な種類一覧がなければ*type_set がNULLになります。
 * @return RET_STAT 処理ステータス
 * 	DB_SUCCESS : 成功
 * 	DB_ERROR   : 失敗
 */
DIVY_DECLARE(RET_STAT) divy_db_get_licensed_dbmstype(request_rec *r,
						apr_hash_t **type_set);

/**
 * リポジトリDBのDBMS識別名称を生成して返却する。
 * この関数によって得られる名称が同じになるのは次の条件を全て満たす
 * 場合のみとなります。
 *  ・Location パス名称が等しいこと
 *  ・DB接続情報が等しいこと
 *  ・DBコネクションプール設定が等しいこと
 *
 * @param p apr_pool_t * リポジトリDBのDBMS識別名称を割り当てるプール
 * @param dconf dav_divy_dir_conf * ディレクトリコンフィグ
 * @return char * 生成したリポジトリDBのDBMS識別名称
 * 		作成に失敗したらNULLを返却
 */
DIVY_DECLARE(char *) divy_db_build_reposdbmsname(apr_pool_t *p,
						dav_divy_dir_conf *dconf);

/*------------------------------------------------------------------------------
  Utility public functions
  ----------------------------------------------------------------------------*/
/**
 * トランザクションコンテキストを生成して、トランザクション開始前の状態に
 * 設定して渡す。
 *
 * @param r request_rec *
 * @param ts_ctx divy_db_transaction_ctx ** トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_db_create_transaction_ctx(request_rec *r,
					divy_db_transaction_ctx **ts_ctx);

/**
 * トランザクションコンテキストを生成して、新しいDbConn を作成し、
 * トランザクション開始前の状態に設定して渡す。
 * (note)
 * 	ここで作成されたDBコネクションはこの関数を呼び出したスレッドが
 * 	１リクエストの間だけ"独占的"に使用することが保証されています。
 * 	但し、この関数から得られたts_ctx を破棄または忘れてしまうと
 * 	同じDBコネクションは２度と取得出来ません。
 * 	独占的に使用したいのであれば、ts_ctx を１リクエストの間、ずっと
 * 	保持するよう呼び出し側で対応して下さい。
 *
 * @param r request_rec *
 * @param ts_ctx divy_db_transaction_ctx ** トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_db_create_transaction_ctx_new(request_rec *r,
					divy_db_transaction_ctx **ts_ctx);

/**
 * 指定されたdbconn を持つトランザクションコンテキストを生成して
 * トランザクション開始前の状態に設定して渡す。
 *
 * @param r request_rec *
 * @param dbconn DbConn * DBコネクション
 * @param ts_ctx divy_db_transaction_ctx ** トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_db_create_transaction_ctx_by_dbconn(request_rec *r,
					DbConn *dbconn,
					divy_db_transaction_ctx **ts_ctx);

/**
 * 関数divy_db_create_transaction_ctxまたは 
 * divy_db_create_transaction_ctx_newによって生成されたトランザクション
 * コンテキストのトランザクションを開始する。
 * (note)
 * 	既にトランザクションが開始していれば何もしません。
 *
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_db_start_transaction(divy_db_transaction_ctx *ts_ctx);

/**
 * 関数divy_db_create_transaction_ctxまたは 
 * divy_db_create_transaction_ctx_newによって開始されたトランザクションを
 * 確定する。
 *
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_db_commit_transaction(divy_db_transaction_ctx *ts_ctx);

/**
 * 関数divy_db_create_transaction_ctxまたは 
 * divy_db_create_transaction_ctx_newによって開始されたトランザクションを
 * 破棄する。
 *
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_db_rollback_transaction(divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたts_ctx が示すトランザクションは、処理を継続できる正しい状態
 * であるかどうかを調べる。
 *
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int (0: 継続できない / 1: 継続できるまたは新規作成できる)
 *             ts_ctx が NULL であれば、2 を返却します。
 */
DIVY_DECLARE(int) divy_db_is_transaction_valid_state(divy_db_transaction_ctx *ts_ctx);

/**
 * 渡されたハッシュのキーに指定された文字列をcondcnt の数だけ','区切りで
 * 文字列合成し、condcnt の単位で divy_linkedlist_t * につめて返却する。
 *
 * (note)
 *   * cond_h のキーに文字列を入れてください。value は参照しません。
 *   * SQL 文の IN 句を数を指定して作成したい時などに使ってください。
 *
 * @param  p apr_pool_t *
 * @param  apr_hash_t * ',' で接続していく文字列集合へのポインタ
 * @param  int 接続するデータ数
 * @return divy_linkedlist_t * 接続後の文字列の構造体
 */
DIVY_DECLARE(divy_linkedlist_t *) divy_db_make_incond(apr_pool_t *p,
					apr_hash_t *cond_h, int condcnt);

/**
 * 指定された文字列集合str_set からバインド変数コンテキストを生成し返却する。
 *
 * (note) バインド変数コンテキストとは?
 * 	任意の数のバインド変数をバインドしなければならないとき、指定した数分だけ
 * 	バインド変数識別文字"?"をSQLに埋め込まなければなりません。またバインド変数を
 * 	IN句の右辺値としてい指定する場合などは、DBMSによって指定できる数に制限を
 * 	受けることがあります。これらは何れも定型的であるので、呼び出し元が一々
 * 	意識するのは面倒です。これらを機械的に行うことを目標にバインド変数コンテキスト
 * 	という概念を導入しました。
 * 	何のことはなく、バインド変数の総数、バインドする値、バインド位置を覚えている
 * 	だけのコンテキストです。strtok で渡すコンテキストのような働きを期待しています。
 *
 * (例) divy_db_create_bind_context(p, uri_set, 5);
 * 	--> uri_set に格納された文字列が5個を1組とするグループに分類されて
 * 	    バインド変数コンテキストが生成される。
 *
 * @param p apr_pool_t *
 * @param str_set divy_cset_t * 文字列集合へのポインタ
 * @param unitnum int 連結されるバインド変数の数。-1 ならばstr_cset 内の全要素数と同数とみなす
 * @return divy_db_bind_ctx * 生成されたバインドコンテキストへのポインタ
 * 	str_cset がNULLならばNULL。
 */
DIVY_DECLARE(divy_db_bind_ctx *) divy_db_bindcontext_make(apr_pool_t *p,
						divy_cset_t *str_set, int unitnum);

/**
 * バインドコンテキストbindctx から最初のイテレータへのポインタを取り出す。
 *
 * @param bindctx divy_db_bind_ctx * バインドコンテキスト
 * @return divy_db_bind_ctx_idx * 最初のイテレータへのポインタ
 */
DIVY_DECLARE(divy_db_bind_ctx_idx *) divy_db_bindcontext_first(divy_db_bind_ctx *bindctx);

/**
 * イテレータbindctx_idx から次のイテレータへのポインタを取り出す。
 *
 * @param bindctx_idx divy_db_bind_ctx_idx *
 * @return divy_db_bind_ctx_idx * 次のイテレータへのポインタ
 */
DIVY_DECLARE(divy_db_bind_ctx_idx *) divy_db_bindcontext_next(divy_db_bind_ctx_idx *bindctx_idx);

/**
 * 現在のイテレータbindctx_idx に格納されたバインド変数の集合をリストとして取り出す。
 *
 * @param bindctx_idx divy_db_bind_ctx_idx *
 * @param list const divy_linkedlist_t ** リンクリストへのポインタ
 */
DIVY_DECLARE(void) divy_db_bindcontext_get_list(divy_db_bind_ctx_idx *bindctx_idx, divy_linkedlist_t **list);

/**
 * 指定されたイテレータbindctx_idx に格納されたバインド変数と同数のバインド変数指示文字"?" を","で連結し
 * 文字列として返却する。
 *
 * @param  p apr_pool_t *
 * @param total int バインド変数の総数
 * @param part int divy_linkedlist_t に詰めるバインド変数の個数
 * @return divy_linkedlist_t * 接続後の文字列の構造体
 * 	total/part が0 であればNULLを返します。
 * 	part が -1 であれば、1つのdivy_linkedlist_t* に全ての値を入れます。
 */
DIVY_DECLARE(void) divy_db_bindcontext_get_bindstr(divy_db_bind_ctx_idx *bindctx_idx,
							const char **bindstr);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_UTIL_DB_H */

