/**
 * $Id$
 *
 * util_ml.h
 *
 * メールユーティリティ関数用ヘッダファイル
 *
 *
 */
#ifndef INCLUDE_UTIL_MAIL_H
#define INCLUDE_UTIL_MAIL_H

#include "httpd.h"
#include "apr.h"
#include "apr_pools.h"
#include "apr_tables.h"

#include "tfr.h"
#include "mod_dav_tf.h"
#include "tf_ml.h"
#include "tf_rdbo.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*--------------------------------------------------------------
  Define fixed value
  --------------------------------------------------------------*/
/**
 * 何もしないメールプロバイダを表す値(caseignore)
 * (より正確には、メール送信をしてはならないかどうかを判断するための
 *  定義値です。)
 */
#define DIVY_ML_OFF	"off"

/**
 * メールのURIはそのユーザがどちらのURIをWatchしているかを
 * 確認する必要があるため追加しました。
 */
#define _BIT(n)	(1 << (n))
enum __MlWatchPlace {
	ML_NOWATCH  = _BIT(0),		/* MailWatchはしていない */
	ML_SRCWATCH = _BIT(1),		/* 送信元 */
	ML_DSTWATCH = _BIT(2)		/* 送信先 */
};

typedef enum __MlWatchPlace MlWatchPlace;

/**
 * 関数divy_ml_notify_passwordexpiredで使用するメール本文の種類
 */
enum {
	DIVY_ML_NOTIFY_UNKNOWN = 0,			/* (未定義)                       */
	DIVY_ML_NOTIFY_PASS_PROBATION,		/* パスワード猶予期限メール       */
	DIVY_ML_NOTIFY_PASS_EXPIRED,		/* パスワードロックアウトメール   */
	DIVY_ML_NOTIFY_FLOGIN_PROBATION,	/* 初回ログイン猶予期限メール     */
	DIVY_ML_NOTIFY_FLOGIN_EXPIRED,		/* 初回ログインロックアウトメール */
	DIVY_ML_NOTIFY_FLOGIN_CHANGENOW,	/* 初回ログイン即時パスワード変更 */
};

/**
 * 関数divy_ml_notify_passwordexpired に渡すデータ
 */
typedef struct divy_ml_passwordexpired_data	divy_ml_passwordexpired_data;
struct divy_ml_passwordexpired_data {
	time_t period;			/* 完全期限切れの日時 */
	time_t probation_last;	/* 最後の猶予日時 */
	time_t flprobation_last;	/* 初回ログインの猶予期間が終わる最後の日時 */
};

/**
 * パースされた監視メールのデータを保持する構造体
 * (see divy_ml_parse_mailrequest_xml)
 */
typedef struct divy_ml_mailwatch_data	divy_ml_mailwatch_data;
struct divy_ml_mailwatch_data {
	char *srcuri;	/* src uri (監視対象リソースのsrc 親パス) */
	char *dsturi;	/* dst uri (監視対象リソースのdst 親パス) */
	char *from;		/* 送信元ユーザのユーザID */
	char *method;	/* 操作メソッド */
	char *subject;	/* メールの件名 */
	char *body;		/* メールの本文(ユーザ入力文を含む) */
	char *fbody;	/* メール本文のフッター */
	char *filelist;	/* ファイル一覧 */
	char *brandname;/* ブランド名 */
};

/*
 * divy_ml_boxで利用するタイプ
 */
enum {
	DIVY_ML_BOX_TYPE_NONE = 0,		/* 未指定 */
	DIVY_ML_BOX_TYPE_OPEN_NOTIFY,	/* BOXが開かれた */
	DIVY_ML_BOX_TYPE_OPEN_REPORT,	/* BOXを公開した（新規作成) */
};

/*--------------------------------------------------------------
  Declare public function
  --------------------------------------------------------------*/
/**
 * メールプロバイダの環境を初期化する。
 * この関数は複数のスレッドからアクセスされてはなりません。
 * (note)
 * 	この関数は、メールプロバイダを使用するどの全ての関数よりも先に呼び出される
 * 	必要があります。child_initステージが適切です。
 * (note)
 * 	この関数を複数回呼び出さないようコール元で保証して下さい。
 *
 * @param pchild apr_pool_t * Childプロセスとライフサイクルを共にするプール
 * @param s server_rec * main serverのserver_rec
 */
DIVY_DECLARE(void) init_mlprv_env(apr_pool_t *pchild, server_rec *s);

/**
 * 指定されたproviderTypeのメールプロバイダ(MlDataSource)を取得する。
 * (note)
 * 	プロバイダが初期化されていなければ、初期化してから返却します。
 * (note)
 * 	この関数は、mod_dav_tf モジュールでしか使用できないフック関数を使用
 * 	しているため、このモジュールの中でしか使用できません。
 * 	他のモジュールからCallしないこと。
 *
 * @param r request_rec *
 * @param providerType const char* プロバイダタイプ
 * @return MlDataSource* プロバイダ提供の構造体へのポインタ
 */
DIVY_DECLARE(MlDataSource *) lookup_ml_provider(request_rec *r,
						const char *providerType);

/**
 * 指定されたファイルネームを元にメールを送信する。
 * メールの機能は全てこの関数を利用すれば勝手にメールが送られます。
 *
 * @param r request_rec *       リクエスト構造体
 * @param filename const char * 絶対パスのファイル名
 * @return 処理ステータス (ML_ERROR | ML_SUCCESS)
 *
 */
DIVY_DECLARE(int) divy_ml_send_clientmail(request_rec *r, const char *filename);

/**
 * MlMessage構造体の内容が正しく送信できるかチェックする
 *
 * @param	pool	apr_pool_t*	プール
 * @param	msg	MlMessage**	メールメッセージ構造体
 *
 */
DIVY_DECLARE(int) divy_ml_validate_mlmessage(apr_pool_t *pool, MlMessage *mlmsg);

/**
 * filename が示す監視メールXMLをパースして*mldata に解析結果を格納する.
 *
 * (note) ignore_body について
 * 	ignore_body = 1 にするとbodyfodder エレメントを読み捨てます.
 * 	また、body エレメントデータのうち、エンドユーザが入力していない本文も
 * 	読み捨てます.
 *
 * @param r request_rec *
 * @param wp apr_pool_t * 作業用のプール
 * @param filename const char *
 * @param ignore_body int もはや古い形式で使われなくなったbodyとbodyfodder の一部を切り捨てる(1/0)
 * @param mldata divy_ml_mailwatch_data ** 解析されたXMLデータを保持する構造体へのポインタ
 * @return int 処理ステータス (1: 失敗 / 0: 成功)
 */
DIVY_DECLARE(int) divy_ml_parse_mailrequest_xml(request_rec *r, apr_pool_t *wp, const char *filename,
												int ignore_body, divy_ml_mailwatch_data **mldata);

/**
 * fd が示す監視メールのディスクリプタからXMLをパースして*mldata に解析結果を格納する.
 * (note)
 *    divy_ml_parse_mailrequest_xml のディスクリプタ版です.
 *
 * @param r request_rec *
 * @param wp apr_pool_t * 作業用のプール
 * @param fd apr_file_t *
 * @param ignore_body int もはや古い形式で使われなくなったbodyとbodyfodder の一部を切り捨てる(1/0)
 * @param mldata divy_ml_mailwatch_data ** 解析されたXMLデータを保持する構造体へのポインタ
 * @return int 処理ステータス (1: 失敗 / 0: 成功)
 */
DIVY_DECLARE(int) divy_ml_parse_mailrequest_xml_by_fd(request_rec *r, apr_pool_t *wp, apr_file_t *fd,
												int ignore_body, divy_ml_mailwatch_data **mldata);

/**
 *
 */
DIVY_DECLARE(MlProperty *)divy_ml_createMlProperty(request_rec *r,
	       						      const char *uid);
DIVY_DECLARE(MlDataSource *) lookup_activeml_provider(request_rec *r);
DIVY_DECLARE(char *) divy_ml_folding_str(apr_pool_t *p, char *str, int len, char **offset);
DIVY_DECLARE(const char *)divy_ml_getHeader(apr_pool_t *p, MlMessage *mlmsg,
	       				   const char *key, char *delimiter);
DIVY_DECLARE(int) divy_ml_encode_msg(apr_pool_t *pool, MlMessage *msg);

/**
 * MlMessageに基づいてSMTPのヘッダを作成する。
 * MlMessage内のheaderを一度クリアしてから設定します。
 * 何度流しても同じ結果になります。
 *
 * @param p apr_pool_t *
 * @param msg MlMessage *
 * @return ML_SUCCESS:成功 / ML_ERROR: 失敗
 */
DIVY_DECLARE(int) divy_ml_setSMTPHeader(apr_pool_t *p, MlMessage *mlmsg);

/**
 * SMTPヘッダにヘッダを追加する。
 * リプレース引数により追加するか置き換えるかを設定できる
 *
 * @param headers apr_table_t *
 * @param key   const char *
 * @param value const char *
 * @param replace int 
 * @return 処理ステータス
 *
 */
DIVY_DECLARE(int) divy_ml_setHeader(apr_table_t *headers, const char *key,
	       			   const char *value, int replace);

/**
 * MlMessage構造体のスケルトンを作成する。
 * (note)
 * 	MlMessage構造体はリスト構造になっているが、この関数では一件しか
 * 	作りません。
 *
 * @param pool 	apr_pool_t *
 * @param mlmsg	MlMessage **
 * @return int 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_ml_create_mlmsg(apr_pool_t *pool, MlMessage **mlmsg);

/**
 * MlAddress構造体を作成する。
 * useridへはNULLをはめるとディレクティブのサーバ管理者のユーザを作成します。
 *
 * @param  r request_rec *
 * @param pool 	apr_pool_t *
 * @param  userid	const char *
 * @param  addr	MlAddress **
 * @return 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_ml_create_mladdress(request_rec *r, const char* userid, MlAddress **addr);

/**
 * サーバ管理者へユーザからメールを送るMlMessage構造体を作成する。
 *
 * @param r request_rec *
 * @param mlmsg	MlMessage **
 * @return 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_ml_create_admin2user_msg(request_rec *r, MlMessage **mlmsg);

/**
 * mlmsgの内容をそのままメール送信する
 *
 * @param r	request_rec*
 * @param mlmsg	MlMessage*
 * @return int	1: 成功/ 0:失敗
 *
 */
DIVY_DECLARE(int) divy_ml_sendmaildirect(request_rec *r, MlMessage *mlmsg);

/**
 * メールアドレスの情報を取得した後（データベースからとか）
 * 複数はいっている可能性のあるメールアドレスのうち一番初めの
 * 一件だけを返す。
 *
 * @param	pool		apr_pool_t* プール
 * @param	address		char *	調べたいメールアドレス
 * @return	mailaddress	char *	一番先頭のメールアドレス
 */
DIVY_DECLARE(char *) divy_ml_get_first_mail_address(apr_pool_t *pool, char *address);

/**
 * カンマで区切られた複数このメールアドレス文字列address のフォーマットを検証する。
 * [ 書式 ]
 *	xxxxx@xxxxx.xxx,yyyy@yyyy.yyy,....
 * 	x,y = 英数字, "-"(ハイフン), "_"(アンダーバ), "."(ピリオド)
 * 	ピリオドは少なくとも１つ入らなければならない
 *
 * @param p apr_pool_t * 作業用のプール
 * @param address const char * カンマで区切られたメールアドレスリスト
 * @return int 処理ステータス
 * 	ML_SUCCESS	: 書式に問題なし
 * 	ML_ERROR	: 書式またはその他で何らかの問題が起きた
 */
DIVY_DECLARE(int) divy_ml_validate_mailaddr(apr_pool_t *p, const char *address);

/**
 * Addrのユーザのメールアドレスが複数だった場合(カンマ区切り）にaddrの内容から
 * もうひとつのアドレス構造体へ対象者を分割する。
 *
 * @param p		apr_pool_t *
 * @param addr		MlAddress **
 * @param addr2		MlAddress **
 * @return void
 */
DIVY_DECLARE(void) divy_ml_setMultiAddress(apr_pool_t *p,
	       			MlAddress **addr, MlAddress **addr2);

/**
 * サーバ主体のメール送信機能が有効かどうかを判定する。
 *
 * [ 有効であると判定する条件 ]
 * 	* メール機能が有効であること
 * 	* サーバ主体のメール送信機能が有効であること
 * 	* メールテンプレートルートパスが設定されていること
 * 	* メール送信が禁止されたクライアントではないこと
 * 	* rdb_r がメール監視フラグを持つコレクション
 * 	  またはコレクション以下のリソースである(グループコレクション以下だけ)
 * (note)
 * 	この関数はrepos.c に実装されていましたが、メール関連のコードであったため
 * 	ここに移動しました。
 *
 * @param r request_rec * リクエスト構造体
 * @param rdb_r divy_rdbo_resource *
 * @return int 有効かどうか (1: 有効である / 0: 有効ではない)
 */
DIVY_DECLARE(int) divy_ml_enable_svrsendmail(request_rec *r,
						divy_rdbo_resource *rdb_r);

/**
 * 指定されたuri において、クライアント主体のメール送信機能が有効かどうかを判定する。
 *
 * [ 有効であると判定する条件 ]
 * 	* メール機能が有効であること
 * 	* rdb_r がメール監視フラグを持つコレクション
 * 	  またはコレクション以下のリソースである
 *
 * @param r request_rec * リクエスト構造体
 * @param uri const char * 調べたいリソースのURI
 * @return int 有効かどうか (1: 有効である / 0: 有効ではない)
 */
DIVY_DECLARE(int) divy_ml_enable_clientsendmail(request_rec *r, const char *uri);

/**
 * ユーザ作成メール通知が有効かどうか判定する
 *
 * [ 有効であると判定する条件 ]
 * 	* メール機能が有効であること
 * 	* 作成ユーザにメールアドレスが存在すること
 * 	* メールテンプレートルートパスが設定されていること
 * 	* アクション通知メールの送信が許可されていること
 * (note)
 * 	この関数はrepos.c に実装されていましたが、メール関連のコードであったため
 * 	ここに移動しました。
 *
 * @param r request_rec * リクエスト構造体
 * @param usr_pr divy_rdbo_usr * ユーザプロパティ
 * @return int 有効かどうか (1: 有効である / 0: 有効ではない)
 */
DIVY_DECLARE(int) divy_ml_enable_notify_usercreation(request_rec *r,
						divy_rdbo_usr *usr_pr);

/**
 * Virus detection alert メール通知が有効かどうか判定する
 *
 * [ 有効であると判定する条件 ]
 * 	* メール機能が有効であること
 * 	* メールテンプレートルートパスが設定されていること
 *
 * @param r request_rec * リクエスト構造体
 * @return int 有効かどうか (1: 有効である / 0: 有効ではない)
 */
DIVY_DECLARE(int) divy_ml_enable_virul_detection_alert(request_rec *r);

/**
 * subject にブランディング用プレフィックスを付けて返す
 *
 * @param wp apr_pool_t * 作業用のプール
 * @param r request_rec *
 * @param subject const char * サブジェクト
 * @return char * 合成されたサブジェクト
 */
DIVY_DECLARE(char *) divy_ml_append_subject_prefix(apr_pool_t *wp, request_rec *r,
                                                    const char *subject);

/**
 * サーバ主体のメールを送信する.
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource * リクエスト情報
 * @param dst_rdb_r divy_rdbo_resource * Destination 情報
 */
DIVY_DECLARE(void) divy_ml_send_servermail(request_rec *r,
					divy_rdbo_resource *rdb_r, divy_rdbo_resource *dst_rdb_r);

/**
 * ユーザ作成通知を送信する.
 *
 * @param r request_rec *
 * @param usr_pr divy_rdbo_usr * ユーザ情報
 */
DIVY_DECLARE(void) divy_ml_notify_usercreation(request_rec *r, divy_rdbo_usr *usr_pr);

/**
 * ユーザ変更通知を送信する。
 *
 * @param r request_rec *
 * @param usr_pr divy_rdbo_usr * ユーザ情報
 */
DIVY_DECLARE(void) divy_ml_notify_update_user(request_rec *r, divy_rdbo_usr *usr_pr);

/**
 * パスワード変更通知を送信する
 * 
 * @param r request_rec *
 * @param usr_pr const divy_rdbo_usr * ユーザ情報
 */
DIVY_DECLARE(int) divy_ml_notify_changed_password(request_rec *r, const divy_rdbo_usr *usr_pr);

/**
 * ウィルス感染したことをメールで通知する.
 * [ 送信先 ]
 *   * TeamFileサーバ管理者
 *   * アップロード実行者
 *
 * [ 送信元 ]
 *   * TeamFile 管理者のメールアドレス(表示名なし)
 *
 * @param r request_rec *
 * @param infect_status int 感染状況
 * @param rdb_r divy_rdbo_resource * 感染したファイルの情報
 */
DIVY_DECLARE(void) divy_ml_notify_virusdetection(request_rec *r,
					int infect_status, divy_rdbo_resource *rdb_r);

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * パスワード期限切れ状態をメールで通知する.
 * (note)
 *   この関数は、中でDB検索を実行します。
 *   そのため、トランザクションを維持した状態ではこの関数をコールすることはできません。
 * [ 送信先 ]
 *   * TeamFileサーバ管理者
 *   * 操作したユーザ
 *
 * [ 送信元 ]
 *   * TeamFile 管理者のメールアドレス(表示名なし)
 *
 * @param r request_rec *
 * @param mltype int 送信メールの種類 (DIVY_ML_NOTIFY_xxx)
 * @param usr_pr divy_rdbo_usr *
 * @param data const divy_ml_passwordexpired_data * メールに入れるデータ
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_ml_notify_passwordexpired(request_rec *r,
								int mltype, const divy_rdbo_usr *usr_pr,
								divy_ml_passwordexpired_data *data);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * オーナーの変更をメールで通知する
 * [ 送信先 ]
 *   * オーナーになったユーザ
 *
 * [ 送信元 ]
 *   * TeamFile 管理者のメールアドレス(表示名なし)
 *
 * @param r request_rec *
 * @param appointed_groupleader 任命された(1) / 解任された(0)
 * @param grp_pr const divy_rdbo_grp *
 * @param owner_pr divy_rdbo_usr * オーナー情報
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_ml_notify_ownerchange(request_rec *r, int appointed_groupleader,
											const divy_rdbo_grp *grp_pr, divy_rdbo_usr *owner_pr);

/**
 * 開封通知をメールで送信する.
 *
 * [ 送信先 ]
 *   * rdb_r->uri に開封通知を仕掛けている人 (N人)
 *
 * [ 送信元 ]
 *   * TeamFile 管理者のメールアドレス(表示名なし)
 *
 * (note) rdb_r の必須プロパティ
 * 	uri, displayname
 *
 * @param r request_rec *
 * @param 
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_notify_confirmreading(request_rec *r, divy_rdbo_resource *rdb_r);

/**
 * 存在する限りのロックされた本人、オーナー、管理者にメールを送ります
 *
 * @param r request_rec *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_notify_failed_login(request_rec *r);

/**
 * BOX画面を開いたときに送信するメール
 * @param r request_rec *
 * @param type int (メールの種類 DIVY_ML_BOX_TYPE_XXXX)
 * @param box divy_rdbo_box *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_box(request_rec *r, int type, divy_rdbo_box *box);

/**
 * BOXを渡す利用者に対しての通知を送る
 *
 * @param r request_rec *
 * @param box divy_rdbo_box *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_box_invitation(request_rec *r, divy_rdbo_box *box);

/**
 * アップロードポリシーを適用した時に所属グループメンバに送られるメール
 * @param r request_rec *
 * @param grp_pr const divy_rdbo_grp *
 * @param usr_pr divy_rdbo_usr *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_group_policy_apply_notify(request_rec *r,
												const divy_rdbo_grp *grp_pr,
												divy_rdbo_usr *usr_pr);

/**
 * グループポリシーに違反したときに飛ばされるメール
 * @param r request_rec *
 * @param filename const char *		対象となったファイル
 * @param grp_pr	divy_rdbo_grp *
 * @param prefix	int				ファイル名のパス(1: pass / 0: NG)
 * @param char		int				文字種のパス(1: pass / 0: NG)
 * @param len		int				長さのパス(1: pass / 0: NG)
 * @param suffix	int				拡張子のパス(1: pass / 0: NG)
 *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_group_policy_inspection_result(request_rec *r,
												const char* filename,
												divy_rdbo_grp *grp_pr,
												int prefix_pass, int char_pass,
												int len_pass, int suffix_pass);

/**
 * 二段階認証のトークンを送る
 * @param r			request_rec *		リクエスト構造体
 * @param mailaddr	const char * 		メールアドレス
 * @param token		char *				トークン文字列
 * @param expire	apr_time_t 			有効期限
 *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_2FA_token(request_rec *r, const char *mailaddr,
											char* token, apr_time_t expire);

#ifdef __cplusplus
}
#endif

#endif /* INCLUDE_UTIL_H */
