/**
 * $Id$
 *
 * util_saml.h
 *
 * SAML対応ユーティリティ関数
 *
 */
#ifndef INCLUDE_UTIL_SAML_H
#define INCLUDE_UTIL_SAML_H

#include "httpd.h"
#include "apr.h"
#include "apr_pools.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define fixed values and define types.
  ----------------------------------------------------------------------------*/
/* IdP選択画面をリフレッシュする時間(秒) */
#define IDP_REFRESH_TIME 1200

/* ログイン画面を表示させ続けておく時間 */
#define SAML_LOCAL_LOGIN_INTERVAL 600

#define SAML_STATUS_SUCCESS "urn:oasis:names:tc:SAML:2.0:status:Success"
#define SAML_URN_ASSERTION "urn:oasis:names:tc:SAML:2.0:assertion"

/*--------------------------------------------------------------
  Define structures and enums
  --------------------------------------------------------------*/
typedef struct __divy_saml_idp_metadata	divy_saml_idp_metadata;
typedef struct __divy_saml_idp_binding	divy_saml_idp_binding;
typedef struct __divy_saml_assertion    divy_saml_assertion;
typedef struct __divy_saml_attribute	divy_saml_attribute;
typedef struct __divy_samlinfo_iscreen  divy_samlinfo_iscreen;
typedef struct __divy_saml_tfuser		divy_saml_tfuser;

/**
 * binding urnの種類
 */
enum __divy_saml_binding_type {
	SAML_URN_BINDING_NONE,		/* なし */
	SAML_URN_BINDING_REDIRECT, 	/* HTTP リダイレクト */
	SAML_URN_BINDING_POST,		/* HTTP ポスト*/
	SAML_URN_BINDING_ARTIFACT,	/* Artifact */
	SAML_URN_BINDING_SOAP		/* SOAP */
};
typedef enum __divy_saml_binding_type divy_saml_binding_type;

struct divy_saml_subject {
	/* NameID */
	char *nameid;

	/* ・皈ス・テ・ノ */
	char *method;

	/* SubjectConfirmationData/Recipient */
	char *recipient;

	/* SubjectConfirmationData/NotOnOrAfter */
	char *notonorafter;

	/* SubjectConfirmationData/InResponseTo */
	char *inresponseto;
};

struct divy_saml_conditions {
	char *notbefore;
	char *notonorafter;
};

struct __divy_saml_attribute {
	char *name;
	char *value;
	divy_saml_attribute *next;
};

struct __divy_saml_assertion {

	/* ・「・オ。シ・キ・逾ャエ゛、゛、譴討い陣MLそのもの */
	char *xml;

	char *issuer;
	char *status;

	struct divy_saml_subject *subject;

	struct divy_saml_conditions *cond;

	/* 公開鍵 */
	const char *x509certificate;

	divy_saml_attribute *attr;
};

struct __divy_saml_idp_binding {

	const char *http_redirect;

	const char *http_post;

	const char *http_artifact;

	const char *soap;
};

struct __divy_saml_tfuser {
	const char *usrid;
	const char *fullname;
	divy_saml_tfuser *next;
};

/*
 * SAMLの情報を格納する
 */
struct __divy_samlinfo_iscreen {

	/* リクエスト構造体 */
	request_rec *r;

	/* アクションを行うURL */
	char *actionurl;

	/* next url */
	char *next;

	/* SAMLのRelayState */
	char *relaystate;

	/* SAMLRequest文字列(base64されています) */
	char *samlxml;

	/* 署名用証明書のパス */
	char *x509path;

	/*
	 * TeamFileのユーザーID
	 * IdPから戻ってきたNameIDを利用したときユーザーがTeamFileに複数いた場合
	 * 選択画面を表示する必要があります。
	 * usrはそのユーザ一覧を保持しています。
	 * 更にtfuserは選ばれたユーザIDが入ります。
	 */
	divy_saml_tfuser* usr_pr;

	char *selectedtfuser;

	/* トークン */
	char *token;

	/* ステータスコード */
	char *statuscode;

	/* メッセージ */
	char *msg;

	/* ブラウザのスタイル */
	int browserstyle;

	/* ユニークID */
	const char *uniqueid;
};

/**
 * IdPのMETAデータを表す構造体
 */
struct __divy_saml_idp_metadata {

	/* IdPメタデータのファイルパス */
	const char* filepath;

	/* IdP entityID */
	char *entityID;

	/* 認証リクエストの要求 0: 要求しない / 1: 要求する */
	int wantauthnreqsigned;

	/* 署名用x509証明書 */
	char *signedx509data;

	/* 暗号化用x509証明書 */
	char *encryptionx509data;

	/* 選択されたbinding type */
	divy_saml_binding_type choose_binding_type;

#define SAML_URN_BINDING_REDIRECT_STR "urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Redirect"
#define SAML_URN_BINDING_POST_STR "urn:oasis:names:tc:SAML:2.0:bindings:HTTP-POST"
#define SAML_URN_BINDING_ARTIFACT_STR "urn:oasis:names:tc:SAML:2.0:bindings:HTTP-Artifact"
#define SAML_URN_BINDING_SOAP_STR "urn:oasis:names:tc:SAML:2.0:bindings:SOAP"
	/* Single Sign on Service Binding Location */
	divy_saml_idp_binding *signon;

	/* Single Logout Service Binding Location */
	divy_saml_idp_binding *logout;

	divy_saml_idp_metadata *next;
};

/*--------------------------------------------------------------
  Declare Public Functions
  --------------------------------------------------------------*/

/**
 * SAMLResponseデータをパースする
 *
 * @param apr_pool_t *pool
 * @param data const char *
 * @param assertion divy_saml_assertion **
 *
 * 	@return 0: 成功 / 1:失敗
 */
DIVY_DECLARE(int)
divy_saml_parse_samlresponse(apr_pool_t *pool, const char* data, divy_saml_assertion **assertion);

/**
 * SAML認証画面で利用する独自XMLデータを作成する
 *
 * @param request_rec *r
 * @param xml char ** 作成されたXMLデータ
 *
 * 	@return 0: 成功 / 1:失敗
 */
DIVY_DECLARE(int)
divy_saml_build_autoindex_xml(request_rec *r, divy_samlinfo_iscreen* screen, char **xml);

/**
 * IdPメタデータを開いて構造体に詰める
 * 
 * @param	pool		apr_pool_t* プール
 * @param 	filepath	const char* IdPメタファイルフルパス
 * @param	idpmeta		divy_saml_idp_metadata	メタデータ構造体
 *
 * @return 0: 成功 / 1: エラー
 */
DIVY_DECLARE(int)
divy_saml_open_metadata(apr_pool_t* pool, const char* filepath, divy_saml_idp_metadata **idpmeta);

/**
 * SAMLのAuth認証で利用するRelayStateをmemcahedに記録する
 *
 * (note)
 * このRelayStateはTeamFileのブラウザはnext=(ここ）を記録します。
 * 利用目的は同じですが、RelayStateは一旦IdPへリダイレクトされた後に
 * 戻ってくる値です。80byteまでの制限があります。
 *
 * @param request_rec *r
 * @param data const char* 保存したいデータ next=(ここの[URL])
 * @param timeout int 記録有効期間(秒指定）
 * @param token char **　RelayStateのトークン文字列
 *
 * @return 0: 成功 / 1: エラー
 */
DIVY_DECLARE(int)
divy_saml_save_relaystate(request_rec *r, const char* data, 
												int timeout, char **token);

/**
 * 記録されているRelayStateの情報を取得する
 *
 * @param request_rec *r
 * @param token const char* キーとなるトークン（relaysteteの値)
 * @param updatetime 取得した後のキーの有効期限(0: 即削除 : 秒指定）
 * @@aram data char **	保存されていた内容（nextの値)
 *
 * @return 0: 成功 / 1: エラー
 */
DIVY_DECLARE(int)
divy_saml_load_relaystate(request_rec *r, const char* token,
										unsigned int updatetime, char **data);

/**
 * XML署名を検証する
 * 
 */
DIVY_DECLARE(int) divy_saml_verifying_xml_signed(apr_pool_t *pool, divy_saml_assertion *assertion);

/**
 * SAMLResponseの内容をチェックします
 * @param r	request_rec *
 * @param assertion divy_saml_assertion *
 * @param token const char*
 *
 * @return dav_error *
 *
 */
DIVY_DECLARE(dav_error *)
divy_saml_validate_samlresponse(request_rec *r, divy_saml_assertion *assertion, const char *token);

/**
 * SAMLのユーザを新規作成する
 */
DIVY_DECLARE(int)
divy_saml_provisioning_account(request_rec *r, divy_saml_assertion *assertion);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_UTIL_SAML_H */

