/***************************************************************************
                          tfldaputil.c  -  description
                             -------------------
    begin                : Tue.  Apr., 15 2004
    copyright            : (C) 2004 by Lei Jiang
    email                : jiang-lei@8107.co.jp
 ***************************************************************************/
#include "util.h"
#include "tfldaputil.h"
#include "searchnode.h"

/*------------------------------------------------------------------------------
  Define static values and macro
  ----------------------------------------------------------------------------*/
APLOG_USE_MODULE(dav_tf);

void TFLDAPOptions_initDefault(TFLDAPOptions* pOptions);
apr_status_t TFLDAPUtil_cleanUp(void *pvData);
void TFLDAPUtil_initLDAP(TFLDAPUtil * pUtil, apr_pool_t * pPool);
TFLRESULT TFLDAPUtil_initSHM(TFLDAPUtil * pUtil, apr_pool_t * pPool,
                             TFLDAPOptions * pOptions);
void TFLDAPUtil_buildFilter(char chFilterBuf[], apr_pool_t * pRequestPool,
                            TFLDAPConf * pConf, const char *pszUsername);
TFLRESULT TFLDAPUtil_setupConnections(TFLDAPUtil * pUtil, TFLDAPConf * pConf,
                                      apr_pool_t * pRequestPool,
                                      const char *pszUsername,
                                      const char *pszPassword);

LDAP_DECLARE(unsigned long)
TFHashString (int nStr, ...)
{
  int     i;
  va_list args;
  unsigned long h = 0, g;
  char   *str, *p;

  va_start (args, nStr);
  for (i = 0; i < nStr; ++i) {
    str = va_arg (args, char *);
    for (p = str; *p; ++p) {
      h = (h << 4) + *p;
      if ((g = h & 0xf0000000)) {
	h = h ^ (g >> 24);
	h = h ^ g;
      }
    }
  }
  va_end (args);

  return h;
}

LDAP_DECLARE(void)
TFLDAPUtil_lock (TFLDAPUtil * pUtil, TFLULOCKTYPE enuLockType)
{
  if (!pUtil)
    return;
  if (pUtil->pfnLockCB)
    pUtil->pfnLockCB (enuLockType, pUtil->pvUserData);
}

void
TFLDAPOptions_initDefault (TFLDAPOptions* pOptions)
{
  pOptions->pszShmPath = TF_LDAP_DEFAULTSHMPATH;
  pOptions->unCacheBytes = TF_LDAP_MINCACHEBYTES;
  pOptions->unTTL = TF_LDAP_DEFAULTTTL;
  pOptions->unMaxMemBlocks = TF_LDAP_MINMEMBLOCKS;
  pOptions->unURLHashTblSize = TF_LDAP_DEFURLHASHSIZE;
  pOptions->unSearchHashTblSize = TF_LDAP_DEFSCHHASHSIZE;
  pOptions->fMarkPercentage = TF_LDAP_DEFMARKPERCENT;
  pOptions->fPurgePercentage = TF_LDAP_DEFPURGEPERCENT;
      
  pOptions->pszCAFile = NULL;
  pOptions->enuCAType = LDAP_CA_TYPE_UNKNOWN;

}

apr_status_t
TFLDAPUtil_cleanUp (void *pvData)
{
  TFLDAPUtil *pUtil = (TFLDAPUtil *) pvData;
  if (pUtil != NULL) {
    apr_shm_t *pShm = pUtil->pShm;
    apr_status_t result = apr_shm_destroy (pShm);
    pUtil->pShm = NULL;
    return result;
    /* disconnect from LDAP server */
    TFLDAPUtil_closeConnection (pUtil, &pUtil->connAuth);
    TFLDAPUtil_closeConnection (pUtil, &pUtil->connBind);
  }

/**  util_ald_destroy_cache (util_ldap_cache);*/
  return APR_SUCCESS;
}



void
TFLDAPUtil_initLDAP (TFLDAPUtil * pUtil, apr_pool_t * pPool)
{
  int     nRet = LDAP_SUCCESS;
/*  apr_pool_cleanup_register (p, st, tf_util_ldap_cleanup,
			     tf_util_ldap_cleanup);*/

  /* initialize SSL support if requested */
  if (pUtil->pszCAFile) {
#if APR_HAS_LDAP_SSL		/* compiled with ssl support */
#if APR_HAS_NETSCAPE_LDAPSDK
    /* Netscape sdk only supports a cert7.db file */
    if (pUtil->enuCAType == LDAP_CA_TYPE_CERT7_DB)
      nRet = ldapssl_client_init (pUtil->pszCAFile, NULL);
    else
      nRet = -1;
#elif APR_HAS_NOVELL_LDAPSDK
    /* Novell SDK supports DER or BASE64 files */
    if (pUtil->enuCAType == LDAP_CA_TYPE_DER ||
	pUtil->enuCAType == LDAP_CA_TYPE_BASE64) {
      nRet = ldapssl_client_init (NULL, NULL);
      if (LDAP_SUCCESS == nRet) {
	if (pUtil->enuCAType == LDAP_CA_TYPE_BASE64)
	  nRet = ldapssl_add_trusted_cert (pUtil->pszCAFile,
					   LDAPSSL_CERT_FILETYPE_B64);
	else
	  nRet = ldapssl_add_trusted_cert (pUtil->pszCAFile,
					   LDAPSSL_CERT_FILETYPE_DER);

	if (LDAP_SUCCESS != nRet)
	  ldapssl_client_deinit ();
      }
    } else {
      nRet = -1;
    }
#elif APR_HAS_OPENLDAP_LDAPSDK
    /* OpenLDAP SDK supports BASE64 files */
    if (pUtil->enuCAType == LDAP_CA_TYPE_BASE64)
      nRet =
	ldap_set_option (NULL, LDAP_OPT_X_TLS_CACERTFILE, pUtil->pszCAFile);
    else
      nRet = -1;
#elif APR_HAS_MICROSOFT_LDAPSDK
    /* Microsoft SDK use the registry certificate store - always
     * assume support is always available */
    nRet = LDAP_SUCCESS;
#else
    nRet = -1;
#endif /* APR_HAS_NETSCAPE_LDAPSDK */
#else /* not compiled with SSL Support */
    nRet = -1;
#endif /* APR_HAS_LDAP_SSL */
    if (LDAP_SUCCESS == nRet)
      pUtil->bSSLSupport = 1;
    else
      pUtil->bSSLSupport = 0;
  }
  /* The Microsoft SDK uses the registry certificate store -
   * always assume support is available
   */
#if APR_HAS_MICROSOFT_LDAPSDK
  pUtil->bSSLSupport = 1;
#endif

}

TFLRESULT
TFLDAPUtil_initSHM (TFLDAPUtil * pUtil, apr_pool_t * pPool, TFLDAPOptions * pOptions)
{
  apr_status_t nResult;
  RMRESULT enuResult;
  RM_HANDLE hRootCache;
  int     bAttach;
  int	  nStgLen;
  void   *pMem;
  double  fMarkPercent;
  double  fPurgePercent;
  unsigned long unMarkBytes;
  unsigned long unPurgeBytes;
  unsigned long unMarkBlocks;
  unsigned long unPurgeBlocks;
    
  RM_ASSERT(pOptions);
  
  if (pUtil->pShm)
    return TFLROK;

  bAttach = 0;
  /* This function call is post config stage only!!!! */
  apr_shm_remove(pUtil->pszShmPath, pPool);
  nResult = apr_shm_create (&(pUtil->pShm),
			    pUtil->unCacheBytes, pUtil->pszShmPath, pPool);
  if (EEXIST == nResult) {
    bAttach = 1;
    nResult = apr_shm_attach (&(pUtil->pShm), pUtil->pszShmPath, pPool);
  }

  if (APR_SUCCESS != nResult) {
    return TFLRSHMINITFAILED;
  }

  pMem = (void *) apr_shm_baseaddr_get (pUtil->pShm);

  TFLU_RDLOCK (pUtil);
  if (bAttach) {
    /* attach memory manager */
    enuResult = RMAttach (&(pUtil->RMMgr), pMem, 0, 0);	/*TODO: lock */
    if (RMRSOK != enuResult) {
      TFLU_UNLOCK (pUtil);
      return TFLROUTOFMEM;	/* out of mem */
    }
    RM_VERIFY( RMRSOK == RMGetUserData (&(pUtil->RMMgr), (RM_USER_DATA_T *) & hRootCache));
    if (hRootCache) {
      pUtil->pRootCache = (LDAPCache *) RMHandleToPtr (&(pUtil->RMMgr), hRootCache);
      RM_ASSERT (pUtil->pRootCache);

      /*
       *  TODO: read more mem settings from cache........????
       */
      pUtil->unURLHashTblSize = pUtil->pRootCache->unURLHashTblSize;
      pUtil->unSearchHashTblSize = pUtil->pRootCache->unSearchHashTblSize;
      pUtil->unMaxMemBlocks = rmGetMaxBlockCount(&pUtil->RMMgr);
      nStgLen = rmGetStgAreaLength(&pUtil->RMMgr);      
      TFLU_UNLOCK (pUtil);
      return TFLROK;
    }
  }

  TFLU_UNLOCK (pUtil);
  /* if bAttach=1 falls through, it could
   * mean that prev creation operation failed*/
  /* create memory manager */
  TFLU_WRLOCK (pUtil);
  /*
   *  TODO: apply more mem settings ...............????
   */
  
  enuResult = RMCreate (&(pUtil->RMMgr), pMem, pUtil->unCacheBytes, pUtil->unMaxMemBlocks, 0, 0);
  if (RMRSOK != enuResult) {
    TFLU_UNLOCK (pUtil);
    return TFLROUTOFMEM;	/* out of mem */
  }
  nStgLen = rmGetStgAreaLength(&pUtil->RMMgr);
  hRootCache = LDAPCache_Create (&(pUtil->RMMgr), pUtil->unURLHashTblSize, pUtil->unSearchHashTblSize);
  pUtil->pRootCache = (LDAPCache *) RMHandleToPtr (&(pUtil->RMMgr), hRootCache);
  if (!pUtil->pRootCache) {
    TFLU_UNLOCK (pUtil);
    return TFLROUTOFMEM;
  }
  RMSetUserData (&(pUtil->RMMgr), (RM_USER_DATA_T) hRootCache);

  /* registering cleanup callback */
  apr_pool_cleanup_register (pPool, pUtil, TFLDAPUtil_cleanUp,
			     apr_pool_cleanup_null);

  /*
   * normalize free bytes Mark, purge
   * and block mark, purge values here
   */
  fMarkPercent = pOptions->fMarkPercentage;
  if(fMarkPercent < 0.0 || fMarkPercent > TF_LDAP_MAXMARKPERCENT)
    fMarkPercent = TF_LDAP_MAXMARKPERCENT;
  fPurgePercent = pOptions->fPurgePercentage;
  if(fPurgePercent < 0.0 || fPurgePercent > TF_LDAP_MAXPURGEPERCENT)
    fPurgePercent = TF_LDAP_MAXPURGEPERCENT;
    
  unMarkBytes = (unsigned long)(((double)(nStgLen))*(fMarkPercent)/100.0);
  unPurgeBytes = (unsigned long)(((double)(nStgLen))*(fPurgePercent)/100.0);
  unMarkBlocks = (unsigned long)(((double)(pUtil->unMaxMemBlocks))*(fMarkPercent)/100.0);
  unPurgeBlocks = (unsigned long)(((double)(pUtil->unMaxMemBlocks))*(fPurgePercent)/100.0);

  pUtil->unMarkBytes = unMarkBytes;
  pUtil->unPurgeBytes = unPurgeBytes;
  pUtil->unMarkBlocks = unMarkBlocks;
  pUtil->unPurgeBlocks = unPurgeBlocks;
  TFLU_UNLOCK (pUtil);
  return TFLROK;
}


LDAP_DECLARE(TFLDAPUtil *)
TFLDAPUtil_Create (apr_pool_t * pChildPool, TFLDAPOptions * pOptions, TFLDAPLOCKCB pfnLockCB, void *pvUserData)
{
  TFLRESULT enuRet;
  TFLDAPOptions defOptions;
  TFLDAPUtil *pRet =
    (TFLDAPUtil *) apr_pcalloc (pChildPool, sizeof (TFLDAPUtil));
  if (!pRet)
    return 0;
  pRet->pPool = pChildPool;
  pRet->pShm = NULL;
  pRet->pszCAFile = NULL;
  pRet->pfnLockCB = pfnLockCB;
  pRet->pvUserData = pvUserData;
  
  if(!pOptions) {
    pOptions = &defOptions;  
    TFLDAPOptions_initDefault (pOptions);
  }

  pRet->pszShmPath = apr_pstrdup (pChildPool,
    pOptions->pszShmPath ? pOptions->pszShmPath : TF_LDAP_DEFAULTSHMPATH);
  pRet->unCacheBytes = pOptions->unCacheBytes > TF_LDAP_MINCACHEBYTES ?
    pOptions->unCacheBytes : TF_LDAP_MINCACHEBYTES;
  pRet->unTTL = pOptions->unTTL > 0 ?
    pOptions->unTTL : 0;
  pRet->pszCAFile = pOptions->pszCAFile ?
    apr_pstrdup (pChildPool, pOptions->pszCAFile) : NULL;
  pRet->unMaxMemBlocks = pOptions->unMaxMemBlocks > TF_LDAP_MINMEMBLOCKS ?
    pOptions->unMaxMemBlocks : TF_LDAP_MINMEMBLOCKS;

  pRet->unURLHashTblSize = pOptions->unURLHashTblSize > TF_LDAP_MINURLHASHSIZE ?
    pOptions->unURLHashTblSize : TF_LDAP_MINURLHASHSIZE;
  pRet->unSearchHashTblSize = pOptions->unSearchHashTblSize > TF_LDAP_MINSEARCHHASHSIZE ?
    pOptions->unSearchHashTblSize : TF_LDAP_MINSEARCHHASHSIZE;
  
  pRet->enuCAType = pOptions->enuCAType;
  pRet->bSSLSupport = 0;	/* will be set in initLDAP()*/

  TFLDAPUtil_initLDAP (pRet, pChildPool);
  TMP_RDLOCK(pRet);
  enuRet = TFLDAPUtil_initSHM (pRet, pChildPool, pOptions);
  TMP_UNLOCK(pRet);
  if (TFLROK != enuRet)
    return NULL;
  return pRet;
}



void
TFLDAPUtil_buildFilter (char chFilterBuf[], apr_pool_t * pRequestPool,
			TFLDAPConf * pConf, const char *pszUsername)
{
  char   *p, *q, *pszFilterEnd;
  char   *pszUser;

  if (pszUsername != NULL) {
    pszUser = apr_pstrdup (pRequestPool, pszUsername);
  } else {
    return;
  }

  /*
   * Create the first part of the filter, which consists of the
   * config-supplied portions.
   */
  apr_snprintf (chFilterBuf, TF_LDAP_FILTERBUFLEN, "(&(%s)(%s=",
		pConf->pszFilter, pConf->pszAttribute);

  /*
   * Now add the client-supplied username to the filter, ensuring that any
   * LDAP filter metachars are escaped.
   */
  pszFilterEnd = chFilterBuf + TF_LDAP_FILTERBUFLEN - 1;
  for (p = pszUser, q = chFilterBuf + strlen (chFilterBuf);
       *p && q < pszFilterEnd; *q++ = *p++) {
    if (strchr ("*()\\", *p) != NULL) {
      *q++ = '\\';
      if (q >= pszFilterEnd) {
	break;
      }
    }
  }
  *q = '\0';

  /*
   * Append the closing parens of the filter, unless doing so would
   * overrun the buffer.
   */
  if (q + 2 <= pszFilterEnd)
    strcat (chFilterBuf, "))");

  ERRLOG1(pRequestPool, APLOG_DEBUG, DIVY_FST_NORMAL + DIVY_SST_DEBUG,
				  "Build LDAP Filter String. %s", chFilterBuf);
}


TFLRESULT
TFLDAPUtil_setupConnections (TFLDAPUtil * pUtil, TFLDAPConf * pConf,
			     apr_pool_t * pRequestPool,
			     const char *pszUsername, const char *pszPassword)
{
  TFLDAPConn *pLink;
  pLink = &pUtil->connBind;
  if (!pLink->pszHost || strcmp (pLink->pszHost, pConf->pszHost)) {
    pLink->pszHost = apr_pstrdup (pRequestPool, pConf->pszHost);
  }
  if (!pLink->pszBindDN || strcmp (pLink->pszBindDN, pConf->pszBindDN)) {
    pLink->pszBindDN = apr_pstrdup (pRequestPool, pConf->pszBindDN);
  }
  if (!pLink->pszBindPW || strcmp (pLink->pszBindPW, pConf->pszBindPW)) {
    pLink->pszBindPW = apr_pstrdup (pRequestPool, pConf->pszBindPW);
  }
  pLink->bSSL = pConf->bSSL;
  pLink->nPort = pConf->nPort;
  pLink->enuDeref = pConf->enuDeref;

  pLink = &pUtil->connAuth;
  if (!pLink->pszHost || strcmp (pLink->pszHost, pConf->pszHost)) {
    pLink->pszHost = apr_pstrdup (pUtil->pPool, pConf->pszHost);
  }
  /* auth connection DN shall be retrived with bind connection
   * in checkUserID()
   */
  /*pLink->pszBindDN = pszUsername; */
  pLink->pszBindPW = pszPassword;
  pLink->bSSL = pConf->bSSL;
  pLink->nPort = pConf->nPort;
  pLink->enuDeref = pConf->enuDeref;

  return TFLROK;
}

/*
 *  internal function
 */
TFLRESULT
TFLDAPUtil_checkUserID (TFLDAPUtil * pUtil, TFLDAPConf * pConf,
			apr_pool_t * pRequestPool, const char *pszFilter,
			const char *pszPasswd, char **ppszDN,
			char ***pppszVals, int bAuth)
{
  RM_HANDLE hURLNode, hSearchNode;
  TFLRESULT enuRet;
  URLNode *pURLNode;
  SearchNode *pSearchNode;
  RMMANAGER *pRMMgr;
  apr_time_t tmNow;
  LDAPMessage *pLDAPRes, *pLDAPEntry;
  int bPasswdMismatch;
  int nRetry, nCount;
  int nInsRetryCount;
  char *pszDN;
  int nUsedBytes;
  int nUsedBlocks;
  int bMarked;
  int bOutOfMem;
  int nSvrDownCount;

  RM_ASSERT (pUtil->pRootCache);
  pRMMgr = &(pUtil->RMMgr);

  /*
   * First check if we need to mark or purge the cache
   */
  TFLU_RDLOCK (pUtil);
  bMarked = pUtil->pRootCache->bMarked;
  nUsedBytes = rmGetStgUsage(pRMMgr);
  nUsedBlocks = rmGetAllocatedBlockCount(pRMMgr);
  RM_ASSERT(nUsedBytes > 0);
  RM_ASSERT(nUsedBlocks > 0);
  TFLU_UNLOCK (pUtil);
  ERRLOG2(pRequestPool, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "UsedBytes = %d UsedBlocks = %d", nUsedBytes, nUsedBlocks);
  if(nUsedBytes > pUtil->unPurgeBytes || nUsedBlocks > pUtil->unPurgeBlocks) {
    TFLU_WRLOCK (pUtil);
    ERRLOG2(pRequestPool, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_OS, "Purge LDAP shared memory. (UsedBytes = %d, UsedBlocks = %d)", nUsedBytes, nUsedBlocks);
    LDAPCache_Purge(pUtil->pRootCache, pRMMgr);
    pUtil->pRootCache->bMarked = 0;	/* reset mark flag */
    TFLU_UNLOCK (pUtil);
  } 
  if(nUsedBytes > pUtil->unMarkBytes || nUsedBlocks > pUtil->unMarkBlocks){
    if(!bMarked) {
      TFLU_WRLOCK (pUtil);
      pUtil->pRootCache->tmMark = apr_time_now();
      pUtil->pRootCache->bMarked = 1;
      TFLU_UNLOCK (pUtil);
    }
  }
  
  
  TFLU_RDLOCK (pUtil);
  hURLNode = LDAPCache_GetURLNode (pUtil->pRootCache, pRMMgr, pConf->pszURL);
  TFLU_UNLOCK (pUtil);

  if (!hURLNode) {
    /* create a new node and insert it into cache */
    TFLU_WRLOCK (pUtil);
    hURLNode =
      LDAPCache_CreateURLNode (pUtil->pRootCache, pRMMgr, pConf->pszURL);
    TFLU_UNLOCK (pUtil);
  }
  pURLNode = (URLNode *) RMHandleToPtr (pRMMgr, hURLNode);
  if (!pURLNode)
  {
	ERRLOG0(pRequestPool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"out of mem (pURLNode).");
    return TFLROUTOFMEM;
  }

  /* Get the search node */
  TFLU_RDLOCK (pUtil);
  hSearchNode = URLNode_GetSearchNode (pURLNode, pRMMgr, pszFilter);
  TFLU_UNLOCK (pUtil);

  if (hSearchNode) {
    TFLU_RDLOCK (pUtil);
    pSearchNode = (SearchNode *) RMHandleToPtr (pRMMgr, hSearchNode);
    RM_ASSERT (pSearchNode);

    /* check if passwds mismatch */
    bPasswdMismatch = 0;
    if (bAuth) {
      RM_HANDLE hszCachedPasswd;
      const char *pszCachedPasswd;
      RM_ASSERT (pszPasswd);

      hszCachedPasswd = pSearchNode->hszBindPW;
      pszCachedPasswd =
	(const char *) RMHandleToPtr (pRMMgr, hszCachedPasswd);
      if (!pszCachedPasswd) {
	bPasswdMismatch = 1;
      } else if (strcmp (pszCachedPasswd, pszPasswd) != 0) {
	bPasswdMismatch = 1;
      }
    }
    TFLU_UNLOCK (pUtil);


    /* check search node time stamp ... */
    tmNow = apr_time_now ();
    if (tmNow - pSearchNode->tmLastBind > pUtil->unTTL) {
      TFLU_WRLOCK (pUtil);
      /* printf("Expired Item [%s] removed\n", pszFilter);*/
      URLNode_RemoveSearchNode (pURLNode, pRMMgr, pszFilter);
      TFLU_UNLOCK (pUtil);
    } else if (bPasswdMismatch) {
      /* check if the passwd matches */
      TFLU_WRLOCK (pUtil);
      /*printf("Item [%s] password mismatch, removed\n", pszFilter);*/
      URLNode_RemoveSearchNode (pURLNode, pRMMgr, pszFilter);
      TFLU_UNLOCK (pUtil);
      /*return TFLRAUTHFAILED; */
    } else {
      /* the record is ok, return */
      const char *pszCachedDN;
      char  **ppszVals;
      const char *pszVal;
      int     nIndex, nValCount;
      RM_HANDLE *phVals;

      TFLU_RDLOCK (pUtil);
      /* copy DN */
      pszCachedDN = (const char *) RMHandleToPtr (pRMMgr, pSearchNode->hszDN);
      RM_ASSERT (pszCachedDN);
      RM_ASSERT (ppszDN);
      *ppszDN = apr_pstrdup (pRequestPool, pszCachedDN);

      /* copy values */
      nValCount = pSearchNode->nValCount;
      phVals = (RM_HANDLE *) RMHandleToPtr (pRMMgr, pSearchNode->hhszVals);
      RM_ASSERT (phVals);
      ppszVals =
	(char **) apr_pcalloc (pRequestPool,
			       nValCount * sizeof (const char *));
      RM_ASSERT (ppszVals);
      *pppszVals = ppszVals;

      if (nValCount) {
	for (nIndex = 0; nIndex < nValCount; nIndex++) {
	  pszVal = (const char *) RMHandleToPtr (pRMMgr, phVals[nIndex]);
	  ppszVals[nIndex] = pszVal ? apr_pstrdup (pRequestPool, pszVal) : 0;
	}
      }
      TFLU_UNLOCK (pUtil);

      return TFLROK;
    }
  }

  /* The Retry Loop
   *   We come here either because there is no
   *   matching record in the cache yet, or because
   *   the record needs to be refreshed
   */
  nSvrDownCount = 0;
  for (nRetry = 0; nRetry < TF_LDAP_SEARCHRETRY; nRetry++) {
    int     nResult;
    if (TFLROK != (enuRet = TFLDAPUtil_openConnection (pUtil, &pUtil->connBind)))
      return enuRet;
    nResult = ldap_search_ext_s (pUtil->connBind.pLdap, pConf->pszBaseDN,
				 pConf->scope, pszFilter,
				 pConf->ppszAttributes, 0, NULL, NULL, NULL,
				 -1, &pLDAPRes);
    if (LDAP_SERVER_DOWN == nResult) {
      enuRet = TFLRSERVERDOWN;
	  if (++nSvrDownCount > 3 && nRetry + 1 < TF_LDAP_SEARCHRETRY) {
		  /* close connection and retry */
		  (void) TFLDAPUtil_closeConnection (pUtil, &pUtil->connBind);
		  nSvrDownCount = 0;
	  }
      continue;
    }

    if (LDAP_SUCCESS != nResult) {
      ERRLOG1(pRequestPool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
              "ldap_search_ext_s failed. [ %s ]", ldap_err2string(nResult));
      (void) TFLDAPUtil_closeConnection (pUtil, &pUtil->connBind);
      return TFLRSEARCHFAILED;
    }

    /* search successful , count entries */
    nCount = ldap_count_entries (pUtil->connBind.pLdap, pLDAPRes);
    if (nCount != 1) {
      if (nCount == 0) {
	enuRet = TFLROBJNOTFOUND;
      } else {
	enuRet = TFLROBJNOTUNIQUE;
      }
      ldap_msgfree (pLDAPRes);
      (void) TFLDAPUtil_closeConnection (pUtil, &pUtil->connBind);
      return enuRet;
    }

    pLDAPEntry = ldap_first_entry (pUtil->connBind.pLdap, pLDAPRes);
    pszDN = ldap_get_dn (pUtil->connBind.pLdap, pLDAPEntry);
    *ppszDN = apr_pstrdup (pRequestPool, pszDN);
    ldap_memfree (pszDN);


    /* TODO: use for-loop to handle out-of-mem */ 
    TFLU_WRLOCK (pUtil);
    bOutOfMem = 0;
    for(nInsRetryCount = 0; nInsRetryCount < 5; nInsRetryCount++) {
      RM_HANDLE hszDN, hszBindPW;

      /* first check if we need to purge the cache
       */
	  if(bOutOfMem) {
		if(!pUtil->pRootCache->bMarked) {
	  		pUtil->pRootCache->tmMark = apr_time_now();
	  		pUtil->pRootCache->bMarked = 1;
		}

		LDAPCache_Purge(pUtil->pRootCache, pRMMgr);
		pUtil->pRootCache->bMarked = 0;	/* reset mark flag */
		bOutOfMem = 0; /* reset out-of-mem flag */
		continue; /* refresh mem retry */

	  }

      /* 
       * insert the node into cache 
       */
      hSearchNode = URLNode_CreateSearchNode (pURLNode, pRMMgr, pszFilter);
      pSearchNode = (SearchNode *) RMHandleToPtr (pRMMgr, hSearchNode);

    if (!pSearchNode) {
		bOutOfMem = 1;
		continue;
		/*
		TFLU_UNLOCK (pUtil);
		return TFLROUTOFMEM;
		*/
	}

      /*user name already copied in CreateSearchNode */
      /*pSearchNode->hszUserName = RMUstrdup(pRMMgr, pszFilter, 0); */
      hszDN = RMUstrdup (pRMMgr, *ppszDN, 0);
      hszBindPW = RMUstrdup (pRMMgr, pszPasswd, 0);

      pSearchNode->hszDN = hszDN;
      pSearchNode->hszBindPW = pszPasswd ? hszBindPW : 0;

	  if(!hszDN) {
		RMFree(pRMMgr, hszBindPW);
		URLNode_RemoveSearchNode(pURLNode, pRMMgr, pszFilter);
		bOutOfMem = 1;
		continue;
      }
      
      /* copy the attribute values */
    if (pConf->ppszAttributes) {
		int     nValIndex;
		int     nValCount;
		char  **ppszVals;
		RM_HANDLE *phszVals;
	
		/* first count attributes */
		for (nValCount = 0; pConf->ppszAttributes[nValCount]; nValCount++);
		pSearchNode->nValCount = nValCount;
	
		if (nValCount) {
			ppszVals = (char **) apr_pcalloc (pRequestPool, nValCount * sizeof(char *));
			pSearchNode->hhszVals = RMCalloc (pRMMgr, nValCount * sizeof (const char *));

			if(!pSearchNode->hhszVals) {
				RMFree(pRMMgr, hszBindPW);
				RMFree(pRMMgr, hszDN);
				URLNode_RemoveSearchNode(pURLNode, pRMMgr, pszFilter);
				bOutOfMem = 1;
				continue;
			}

			phszVals = (RM_HANDLE *) RMHandleToPtr (pRMMgr, pSearchNode->hhszVals);
			RM_ASSERT (ppszVals);
			RM_ASSERT (phszVals);
			*pppszVals = ppszVals;
			for (nValIndex = 0; nValIndex < nValCount; nValIndex++) {
				RM_HANDLE hszAttrVal;
				char   *pszVal = NULL;
				char  **ppszLdapValues
						= ldap_get_values (pUtil->connBind.pLdap,
										   pLDAPEntry,
										   pConf->ppszAttributes[nValIndex]);
				if (ppszLdapValues && ppszLdapValues[0]) {
						pszVal = apr_pstrdup (pRequestPool, ppszLdapValues[0]);
				}
				ldap_value_free (ppszLdapValues);
				ppszVals[nValIndex] = pszVal;
				hszAttrVal = 0;
				if(pszVal) {
					hszAttrVal = RMUstrdup (pRMMgr, pszVal, 0);
					if(!hszAttrVal) {
						bOutOfMem = 1;
						break;
					}
				}
				phszVals[nValIndex] = hszAttrVal; 
			}
			if(bOutOfMem) {
				/* rolling back */
				for(nValIndex-- ;nValIndex >= 0; nValIndex--) {
					RMFree(pRMMgr, phszVals[nValIndex]);
				}

				URLNode_RemoveSearchNode(pURLNode, pRMMgr, pszFilter);
				continue;
			}
			*pppszVals = ppszVals;
		} /* ValCount if*/
	} /* attr if */
	bOutOfMem = 0;
	break;
	}	/* insert loop */

	(void) TFLDAPUtil_closeConnection (pUtil, &pUtil->connBind);

    TFLU_UNLOCK (pUtil);
    ldap_msgfree (pLDAPRes);
      
    if(bOutOfMem) {
	  ERRLOG0(pRequestPool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					  "Give up. Insert attributes.(out of mem)");
      return TFLROUTOFMEM;
    }   

    /* authenticate user if necessary */
    if (bAuth) {
      pUtil->connAuth.pszBindDN = *ppszDN;
      enuRet = TFLDAPUtil_openConnection (pUtil, &pUtil->connAuth);
      TFLDAPUtil_closeConnection (pUtil, &pUtil->connAuth);
      if (TFLRSERVERDOWN != enuRet) {
	/* something went wrong */
	if (TFLROK != enuRet) {
	  enuRet = TFLRAUTHFAILED;
	  TFLU_WRLOCK (pUtil);
	  URLNode_RemoveSearchNode (pURLNode, pRMMgr, pszFilter);
	  TFLU_UNLOCK (pUtil);
	}
	break;
      }
    } else {
      /* not doing auth */
      return TFLROK;
    }
  }				/* the retry loop */
  return enuRet;
}

LDAP_DECLARE(void)
TFLDAPUtil_SetLockCallback (TFLDAPUtil * pUtil,
			    TFLDAPLOCKCB pfnLockCB, void *pvUserData)
{
  if (!pUtil)
    return;
  pUtil->pfnLockCB = pfnLockCB;
  pUtil->pvUserData = pvUserData;
}


LDAP_DECLARE(TFLRESULT)
TFLDAPUtil_CheckUserID (TFLDAPUtil * pUtil,
			TFLDAPConf * pConf,
			apr_pool_t * pRequestPool,
			const char *pszUsername, const char *pszPasswd)
{
  int     nRetry;
  char    chFilterBuf[TF_LDAP_FILTERBUFLEN];
  char   *pszDN, **ppszVals;
  TFLRESULT enuRet = TFLROK;

  if (!(pUtil && pConf && pRequestPool && pszUsername && pszPasswd))
    return TFLRINVALIDARG;
  TFLDAPUtil_buildFilter (chFilterBuf, pRequestPool, pConf, pszUsername);

  TMP_WRLOCK(pUtil);
  for (nRetry = 0; nRetry < 5; nRetry++) {
    /* first make sure the bind conn is valid */
    TFLDAPUtil_setupConnections (pUtil, pConf, pRequestPool, chFilterBuf,
				 pszPasswd);
    if (TFLRSERVERDOWN !=
	(enuRet =
	 TFLDAPUtil_checkUserID (pUtil, pConf, pRequestPool, chFilterBuf,
				 pszPasswd, &pszDN, &ppszVals, 1))) {
      break;
    }
  }
  TMP_UNLOCK(pUtil);
  return enuRet;
}

LDAP_DECLARE(TFLRESULT)
TFLDAPUtil_GetAttributes (TFLDAPUtil * pUtil,
			  TFLDAPConf * pConf,
			  apr_pool_t * pRequestPool,
			  const char *pszUsername,
			  char **ppszDN, char ***pppszAttributes)
{
  int     nRetry;
  char    chFilterBuf[TF_LDAP_FILTERBUFLEN];
  char   *pszDN, **ppszVals;
  TFLRESULT enuRet = TFLROK;

  if (!(pUtil && pConf && pRequestPool && pszUsername))
    return TFLRINVALIDARG;
  if (!ppszDN)
    ppszDN = &pszDN;
  if (!pppszAttributes)
    pppszAttributes = &ppszVals;
  TFLDAPUtil_buildFilter (chFilterBuf, pRequestPool, pConf, pszUsername);
  TMP_WRLOCK(pUtil);
  for (nRetry = 0; nRetry < 5; nRetry++) {
    /* first make sure the bind conn is valid */
    TFLDAPUtil_setupConnections (pUtil, pConf, pRequestPool, chFilterBuf,
				 NULL);
    if (TFLRSERVERDOWN !=
	(enuRet =
	 TFLDAPUtil_checkUserID (pUtil, pConf, pRequestPool, chFilterBuf,
				 NULL, ppszDN, pppszAttributes, 0))) {
      break;
    }
  }
  TMP_UNLOCK(pUtil);
  return enuRet;
}

LDAP_DECLARE(TFLDAPConf *)
TFLDAPConf_Create (apr_pool_t * pChildPool)
{
  TFLDAPConf *pRet =
    (TFLDAPConf *) apr_pcalloc (pChildPool, sizeof (TFLDAPConf));
  if (!pRet)
    return NULL;
  pRet->pPool = pChildPool;
  /*
     pRet->pszURL;
     pRet->pszBaseDN;
     pRet->pszAttribute;
     pRet->ppszAttributes;
     pRet->cope;
     pRet->pszFilter;
     pRet->enuDeref;
     pRet->bUserIsDN;
     pRet->pszHost;
     pRet->nPort;
     pRet->pszBindDN;
     pRet->pszBindPasswd;
   */
  pRet->bSSL = 0;
  return pRet;
}


LDAP_DECLARE(const char *)
TFLDAPUtil_FormatMessage (TFLRESULT enuResult)
{
  switch (enuResult) {
  case TFLROK:
    return " operation successful ";
  case TFLRINVALIDARG:
    return " argument(s) invalid ";
  case TFLROUTOFMEM:
    return " out of memory ";
  case TFLRSHMINITFAILED:
    return " failed to initialize shared memory ";
  case TFLRLDAPINITFAILED:
    return " failed to initialize ldap client ";
  case TFLRBINDFAILED:
    return " failed to bind to ldap server ";
  case TFLRCONNNOTFOUND:
    return " LDAP connection not found(not likely to happen) ";
  case TFLRSERVERDOWN:
    return " LDAP server down ";
  case TFLRSSLNOTSUPPORTED:
    return " SSL connection is not supported ";
  case TFLRSEARCHFAILED:
    return " LDAP search failed ";
  case TFLROBJNOTFOUND:
    return " object(user) not found ";
  case TFLROBJNOTUNIQUE:
    return " object(user) not unique ";
  case TFLRINVALIDPASSWD:
    return " password invalid(empty) ";
  case TFLRAUTHFAILED:
    return " ldap auth failed ";
  default:
    return "[message not found]";
  }
}

TFLRESULT
TFLDAPUtil_closeConnection (TFLDAPUtil * pUtil, TFLDAPConn * pConn)
{
  if (!pConn->bBound) {
    return TFLROK;
  } else {
    ldap_unbind_s (pConn->pLdap);
    pConn->pLdap = NULL;
    pConn->bBound = 0;
    return TFLROK;
  }
}

TFLRESULT
TFLDAPUtil_openConnection (TFLDAPUtil * pUtil, TFLDAPConn * pConn)
{
  TFLRESULT enuRet;
  int     nRetry;
  int     nVersion = LDAP_VERSION3;
  int     nResult = LDAP_SUCCESS;
  
  if (pConn->bBound)
    return TFLROK;


  if (!pConn->pLdap) {
    /*try to initialize */
    if (!pConn->bSSL) {
      /* raw connection */
      pConn->pLdap = ldap_init ((const char *) pConn->pszHost, pConn->nPort);
    } else {
      /* SSL connection */
      if (pUtil->bSSLSupport) {
#if APR_HAS_LDAP_SSL
#if APR_HAS_NOVELL_LDAPSDK
	pConn->pLdap = ldapssl_init (pConn->pszHost, pConn->nPort, 1);
#elif APR_HAS_NETSCAPE_LDAPSDK
	pConn->pLdap = ldapssl_init (pConn->pszHost, pConn->nPort, 1);
#elif APR_HAS_OPENLDAP_LDAPSDK
	pConn->pLdap = ldap_init (pConn->pszHost, pConn->nPort);
	if (pConn->pLdap) {
	  int     nSSLmode = LDAP_OPT_X_TLS_HARD;
	  nResult = ldap_set_option (pConn->pLdap, LDAP_OPT_X_TLS, &nSSLmode);
	  if (LDAP_SUCCESS != nResult) {
	    ldap_unbind_s (pConn->pLdap);
	    pConn->pLdap = NULL;
	  }
	}
#elif APR_HAS_MICROSOFT_LDAPSDK
	pConn->pLdap =
	  ldap_sslinit ((const char *) pConn->pszHost, pConn->nPort, 1);

#else
	enuRet = TFLRSSLNOTSUPPORTED;
	/*ldc->reason = "LDAP: ssl connections not supported"; */
#endif /* APR_HAS_NOVELL_LDAPSDK */

#endif /* APR_HAS_LDAP_SSL */
      } else {
	enuRet = TFLRSSLNOTSUPPORTED;
	/*ldc->reason = "LDAP: ssl connections not supported"; */
      }
    }

    /* if ldap object is still not initialized... */
    if (!pConn->pLdap) {
      pConn->bBound = 0;
      return TFLRLDAPINITFAILED;
    }

    /* Set the alias dereferencing option */
    ldap_set_option (pConn->pLdap, LDAP_OPT_DEREF, &(pConn->enuDeref));

    /* always default to LDAP V3 */
    ldap_set_option (pConn->pLdap, LDAP_OPT_PROTOCOL_VERSION, &nVersion);


    /* add the cleanup to the pool */
    /*
       apr_pool_cleanup_register(ldc->pool, ldc,
       util_ldap_connection_destroy,
       apr_pool_cleanup_null);
     */
  }


  for (nRetry = 0; nRetry < TF_LDAP_CONNMAXRETRY; nRetry++) {
    nResult = ldap_simple_bind_s (pConn->pLdap,
				  (const char *) pConn->pszBindDN,
				  (const char *) pConn->pszBindPW);
    if (LDAP_SERVER_DOWN != nResult)
      break;
  }

  pConn->bBound = 1;

  /* free the handle if there was an error
   */
  if (LDAP_SUCCESS != nResult) {
    ldap_unbind_s (pConn->pLdap);
    pConn->pLdap = NULL;
    pConn->bBound = 0;
    if (LDAP_SERVER_DOWN == nResult) {
      return TFLRSERVERDOWN;
	}
	else {
      ERRLOG1(pUtil->pPool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			  "ldap open connection failed. [ %s ]", ldap_err2string(nResult));
      ERRLOG2(pUtil->pPool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			  "BindDN = %s, BindPW = %s", pConn->pszBindDN, pConn->pszBindPW);
      return TFLRLDAPINITFAILED;
	}
  }

  return TFLROK;
}
