/**
 * $Id$
 *
 * liveprop.c
 *
 * Live property・プロバイダ
 *
 * 変更履歴
 *
 * 2003/02/13 Thu takehara NEW
 */
/* Apache header files */
#include "httpd.h"
#include "http_protocol.h"
#include "util_md5.h"
#include "apr.h"
#include "apr_strings.h"
#include "apr_pools.h"
#include "apr_hash.h"
#include "mod_dav.h"

/* document management headers */
#include "mod_dav_tf.h"
#include "liveprop.h"
#include "tf_rdbo.h"
#include "tf_rdbo_sql.h"
#include "tf_rdbo_sysmsg.h"
#include "tf_rdbo_dbms.h"
#include "tf_rdbo_user.h"
#include "tf_rdbo_group.h"
#include "tf_rdbo_clupdate.h"
#include "util.h"
#include "util_auth.h"
#include "tf_sqlparser.h"
#include "util_ml.h"
#include "tf_folder.h"
#include "tf_validator.h"
#include "tf_valuecache.h"
#include "tf_thumbnail.h"
#include "tf_parser.h"
#include "tf_plugin.h"
#include "tf_confirmreading.h"
#include "tf_box.h"

#include "tfs_hashset.h"
#include "tfs_pools.h"

/*------------------------------------------------------------------------------
  Define Fixed values and macro
  ----------------------------------------------------------------------------*/
APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * [ 不完全型の定義 ]
 * ある一つのリソースまたはコレクションが持つ１つのプロパティに対して
 * 加えられた変更を後で、必要に応じて取り消しor反映するのに必要な
 * 情報をもつコンテキスト。(see mod_dav.h)
 * (note) 単位
 * １個のプロパティに対して1個だけ存在します。
 * このコンテキストは、PROPPATCH による変更を行う前に生成され、
 * 何らかの問題が発生して、変更を取消す際に利用されます。
 */
struct dav_liveprop_rollback {

	/* プロパティのproperty id */
	int propid;

	/* 変更を加えられる前に持っていたプロパティの値 */
	divy_rdbo_keylist   *keylist_pr;	/* keylist             */
	divy_rdbo_mailwatch *mwatch_pr;		/* mailwatch           */
	divy_rdbo_shlink    *shlink_pr;		/* sharedcollection    */
	divy_rdbo_clmodule  *clmodule_pr;	/* updatediscovery     */
	divy_rdbo_usr       *usr_pr;		/* userdiscovery       */
	divy_rdbo_grp       *grp_pr;		/* groupdiscovery      */
	divy_rdbo_sql       *sql_pr;		/* sqldiscovery        */
	divy_rdbo_sysmsg    *sysmsg_pr;		/* sysmsgdiscovery     */
	divy_rdbo_shsrccol  *shsrccol_pr;
	divy_rdbo_dbms      *dbms_pr;		/* dbmsdiscovery       */
#ifdef DIVY_SUPPORT_EXTENDQUOTA
	divy_rdbo_diskquota *sysquota_pr;	/* systemquota         */
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

	/* トランザクションコンテキストへのポインタ */
	divy_db_transaction_ctx *ts_ctx;
};

/**
 * あるLive プロパティを表すコンテナとしての役割を持つ
 * 複数のLive プロパティをまとめて保持しています。
 */
struct __divy_liveprop_ctx {
	
	/* 以下のプロパティのproperty id */
	int propid;
	
	/* PROPPATCH オペレーションの種類を表す値 */
	int operation;

	/* Live プロパティの一覧 */
	divy_rdbo_mailwatch *mwatch_pr;		/* mailwatch           */
	divy_rdbo_shlink    *shlink_pr;		/* sharedcollection    */
	divy_rdbo_clmodule  *clmodule_pr;	/* updatediscovery     */
	divy_rdbo_keylist   *keylist_pr;	/* keylist             */
	divy_rdbo_sql       *sql_pr;		/* sqldiscovery        */

	divy_user_iscreen   *user_iscreen;	/* userdiscovery       */
	divy_group_iscreen  *group_iscreen;	/* groupdiscovery      */
	divy_sysmsg_iscreen *sysmsg_iscreen;	/* sysmsgdiscovery     */
	divy_dbms_iscreen   *dbms_iscreen;	/* dbmsdiscovery       */

#ifdef DIVY_SUPPORT_EXTENDQUOTA
	divy_sysquota_iscreen *sysquota_iscreen;/* systemquota         */
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

	divy_resourcestate_iscreen *rstate_iscreen;	/* resourcestate */
	divy_thumbnail_iscreen *thumbnail_iscreen;	/* thumbnail */
#ifdef DIVY_SUPPORT_PASSPOLICY
	divy_passwordpolicy_iscreen *passpolicy_iscreen;	/* passwordpolicy */
	divy_changepassword_iscreen *changepassword_iscreen;	/* changepassword */
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	divy_changeowner_iscreen *changeowner_iscreen;	/* changeowner */
	divy_changeleader_iscreen *changeleader_iscreen;	/* changeleader */
	divy_confirmreading_iscreen *confirmreading_iscreen;	/* confirmreading */
};
typedef struct __divy_liveprop_ctx	divy_liveprop_ctx;


/*------------------------------------------------------------------------------
  Declar structures
  ----------------------------------------------------------------------------*/

/* forward-declare */
const dav_hooks_liveprop dav_divy_hooks_liveprop;

/*------------------------------------------------------------------------------
  Define Hook array etc..
  ----------------------------------------------------------------------------*/
/**
 * サポートするネームスペースURIの定義
 *
 * このLive プロパティプロバイダがサポートしているプロパティの
 * ネームスペースURIとプレフィックス番号(配列の要素番号)を定義しています。
 * (note)
 * 	この配列の要素を並べ替えるときには、liveprop.h で定義した配列の添え字
 * 	番号を示す以下も正しく書き換えてください。
 * 	・DAV_NS_PREFIX_IDX
 * 	・DIVY_NS_PREFIX_IDX
 * (note)
 *	dav_namespace テーブルのエントリはDeadプロパティプロバイダが使用する
 *	ネームスペースですので、Liveプロパティプロバイダが定義するネームスペース
 *	と同じかどうかは感知しません。同じであっても一向に構わないフレームワーク
 *	になっています。
 */
static const char * const dav_divy_namespace_uris[] = {
	"DAV:",
	DIVY_NS_DEFAULT_URI,
	NULL		/* sentinel */
};

/**
 * 構造体dav_liveprop_spec の nsメンバの値
 */
enum {
	DAV_DIVY_URI_DAV,       /* the DAV: namespace URI */
	DAV_DIVY_URI_MYPROPS    /* the namespace URI for our custom props */
};

/**
 * 構造体dav_liveprop_spec の is_writableメンバの値
 */
enum {
	DAV_DIVY_PROP_READONLY,   /* プロパティはRead-Onlyである */
	DAV_DIVY_PROP_READWRITE   /* プロパティは読み書きできる  */
};

/**
 * dav_liveprop_spec構造体の初期化
 *
 * ・ネームスペースインデックス(Liveプロパティプロバイダローカル)、
 * ・プロパティ名、
 * ・プロパティID（Liveプロパティプロバイダローカル）
 * ・書き込み可能かどうか
 *
 * を持つ構造体からなる配列.
 * dav_liveprop_spec構造体は、ある特定のプロパティがLiveプロパティかどうかを
 * 判断するために使用されます。
 *
 * (note)
 * 	この配列の要素は、Liveプロパティ名をASCIIの昇順で並べられています。
 * 	これは、dav_divy_find_livepropでバイナリサーチを実施するためです。
 * 	並びを変える時には、先の関数とmod_dav_tf.h の列挙型
 * 	dav_divy_propid_info も一緒に変えて下さい。
 *
 * [ 新しいLiveプロパティの追加方法 ]
 *   (1) include/mod_dav_tf の列挙型dav_divy_propid_info に新しいLiveプロパティの
 *       プロパティIDを追加する。定義済みLiveプロパティの間には入れてはならない。
 *       Liveプロパティ名がASCIIの昇順に並ぶ位置にいれること。
 *
 *   (2) liveprop.c の構造体配列dav_divy_props に新しいLiveプロパティの定義を追加する。
 *       追加位置はLiveプロパティ名がASCIIの昇順に並ぶように入れること。
 *
 *   (3) 必要であればrepos.c のdav_divy_walk などで新しいLiveプロパティの値を
 *       DBから取得する
 *   (4) 関数 _insert_prop に新しいLiveプロパティの値を書き込むようコードを入れる
 *
 *   (5) PROPFIND (all) の調整は関数dav_divy_insert_all_liveprops で実施。
 *       値を返却したくないのならそのようにコードを入れる。
 */
static const dav_liveprop_spec dav_divy_props[] = {

	{	DAV_DIVY_URI_MYPROPS,
		"analysiserrdiscovery",
		DAV_DIVY_PROPID_analysiserrdiscovery,
		DAV_DIVY_PROP_READWRITE
	},
	{	DAV_DIVY_URI_MYPROPS,
		"changeleader",
		DAV_DIVY_PROPID_changeleader,
		DAV_DIVY_PROP_READWRITE
	},
	{	DAV_DIVY_URI_MYPROPS,
		"changeowner",
		DAV_DIVY_PROPID_changeowner,
		DAV_DIVY_PROP_READWRITE
	},
#ifdef DIVY_SUPPORT_PASSPOLICY
	{	DAV_DIVY_URI_MYPROPS,
		"changepassword",
		DAV_DIVY_PROPID_changepassword,
		DAV_DIVY_PROP_READWRITE
	},
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	{	DAV_DIVY_URI_MYPROPS,
		"confirmreading",
		DAV_DIVY_PROPID_confirmreading,
		DAV_DIVY_PROP_READWRITE
	},
	{	DAV_DIVY_URI_DAV,
		"creationdate",
		DAV_DIVY_PROPID_creationdate,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"creator",		/* (note) 名称変更時にはdav_divy_walk も変更する */
		DAV_DIVY_PROPID_creator,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"dbmsdiscovery",
		DAV_DIVY_PROPID_dbmsdiscovery,
		DAV_DIVY_PROP_READWRITE
	},
	{	DAV_DIVY_URI_MYPROPS,
		"dbmserrdiscovery",
		DAV_DIVY_PROPID_dbmserrdiscovery,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"dependsqldiscovery",
		DAV_DIVY_PROPID_dependsqldiscovery,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_DAV,
		"displayname",
		DAV_DIVY_PROPID_displayname,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"displayorder",
		DAV_DIVY_PROPID_displayorder,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"foldertype",
		DAV_DIVY_PROPID_foldertype,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_DAV,
		"getcontentlanguage",
		DAV_DIVY_PROPID_getcontentlanguage,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_DAV,
		"getcontentlength",
		DAV_DIVY_PROPID_getcontentlength,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_DAV,
		"getcontenttype",
		DAV_DIVY_PROPID_getcontenttype,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_DAV,
		"getetag",
		DAV_DIVY_PROPID_getetag,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_DAV,
		"getlastmodified",
		DAV_DIVY_PROPID_getlastmodified,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"groupdiscovery",
		DAV_DIVY_PROPID_groupdiscovery,
		DAV_DIVY_PROP_READWRITE
	},
	{	DAV_DIVY_URI_MYPROPS,
		"groupstate",
		DAV_DIVY_PROPID_groupstate,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"keylist",
		DAV_DIVY_PROPID_keylist,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"lastmodifier",		/* (note) 名称変更時にはdav_divy_walk も変更する */
		DAV_DIVY_PROPID_lastmodifier,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"linkdberrdiscovery",
		DAV_DIVY_PROPID_linkdberrdiscovery,
		DAV_DIVY_PROP_READONLY
	},
	/* Core live プロパティプロバイダに任せるので定義できない */
#if 0
	{	DAV_DIVY_URI_DAV,
		"lockdiscovery",
		DAV_DIVY_PROPID_lockdiscovery,
		DAV_DIVY_PROP_READONLY
	},
#endif
	{	DAV_DIVY_URI_MYPROPS,
		"mailwatch",		/* (note) 名称変更時にはdav_divy_walk も変更する */
		DAV_DIVY_PROPID_mailwatch,
		DAV_DIVY_PROP_READWRITE
	},
#ifdef DIVY_SUPPORT_PASSPOLICY
	{	DAV_DIVY_URI_MYPROPS,
		"passwordpolicy",
		DAV_DIVY_PROPID_passwordpolicy,
		DAV_DIVY_PROP_READWRITE
	},
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	{	DAV_DIVY_URI_MYPROPS,
		"resourcestate",
		DAV_DIVY_PROPID_resourcestate,
		DAV_DIVY_PROP_READWRITE
	},
	{	DAV_DIVY_URI_DAV,
		"resourcetype",
		DAV_DIVY_PROPID_resourcetype,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"sharedcollection",	/* (note) 名称変更時にはdav_divy_walk も変更する */
		DAV_DIVY_PROPID_sharedcollection,
		DAV_DIVY_PROP_READWRITE
	},
	{	DAV_DIVY_URI_MYPROPS,
		"shorten",
		DAV_DIVY_PROPID_shorten,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_DAV,
		"source",
		DAV_DIVY_PROPID_source,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"sqldiscovery",
		DAV_DIVY_PROPID_sqldiscovery,
		DAV_DIVY_PROP_READWRITE
	},
	{	DAV_DIVY_URI_MYPROPS,
		"sqlpreferencediscovery",
		DAV_DIVY_PROPID_sqlpreferencediscovery,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"statusdiscovery",
		DAV_DIVY_PROPID_statusdiscovery,
		DAV_DIVY_PROP_READONLY
	},
	/* Core live プロパティプロバイダに任せるので定義出来ない */
#if 0
	{	DAV_DIVY_URI_DAV,
		"supportedlock",
		DAV_DIVY_PROPID_supportedlock,
		DAV_DIVY_PROP_READONLY
	},
#endif
	{	DAV_DIVY_URI_MYPROPS,
		"sysmsgdiscovery",
		DAV_DIVY_PROPID_sysmsgdiscovery,
		DAV_DIVY_PROP_READWRITE
	},
#ifdef DIVY_SUPPORT_EXTENDQUOTA
	{	DAV_DIVY_URI_MYPROPS,
		"systemquota",
		DAV_DIVY_PROPID_systemquota,
		DAV_DIVY_PROP_READWRITE
	},
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
	{	DAV_DIVY_URI_MYPROPS,
		"thumbnail",
		DAV_DIVY_PROPID_thumbnail,
		DAV_DIVY_PROP_READWRITE
	},
	{	DAV_DIVY_URI_MYPROPS,
		"trashdiscovery",
		DAV_DIVY_PROPID_trashdiscovery,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"trashinformation",
		DAV_DIVY_PROPID_trashinformation,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"updatediscovery",
		DAV_DIVY_PROPID_updatediscovery,
		DAV_DIVY_PROP_READWRITE
	},
	{	DAV_DIVY_URI_MYPROPS,
		"user-privilege-grant-set",
		DAV_DIVY_PROPID_user_privilege_grant_set,
		DAV_DIVY_PROP_READONLY
	},
	{	DAV_DIVY_URI_MYPROPS,
		"userdiscovery",
		DAV_DIVY_PROPID_userdiscovery,
		DAV_DIVY_PROP_READWRITE
	}
};

/* 配列dav_divy_props の長さを表す定義値 */
#define DAV_DIVY_PROPS_LEN	sizeof(dav_divy_props)/sizeof(dav_liveprop_spec)

/**
 * プロパティID をインデックス番号として、dav_liveprop_spec 
 * 構造体へのポインタを持つ配列
 *
 * (note) 導入の背景
 * 	この配列は、性能向上対策の一環として導入されました。
 * 	ただ１度だけ初期化され、使いまわされます。
 * 	もっとよい方法があれば、変更して下さい。
 */
static dav_liveprop_spec **dav_divy_props_array = NULL;

/*------------------------------------------------------------------------------
  Declare private function 
  ----------------------------------------------------------------------------*/
static apr_status_t _cleanup_props_array(void *ctx);
static dav_prop_insert _insert_prop(const dav_resource *resource,
					int propid,
					dav_prop_insert what,
					apr_text_header *phdr,
					int propfind_type);
static char * _insert_prop_resourcestate(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_resource *rdb_r);
static char * _insert_prop_user_privilege_grant_set(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					int force_append_prop,
					divy_rdbo_resource *rdb_r);
static char * _insert_prop_mailwatch(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_resource *rdb_r);
#ifdef DIVY_SUPPORT_EXTENDQUOTA
static char * _insert_prop_systemquota(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_resource *rdb_r);
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
static char * _insert_prop_thumbnail(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_resource *rdb_r);
#ifdef DIVY_SUPPORT_PASSPOLICY
static char * _insert_prop_passwordpolicy(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_passwordpolicy *passpolicy_pr);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
static char * _insert_prop_groupstate(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_grp *grp_pr);
static char * _insert_prop_confirmreading(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_resource *rdb_r);
static char * _insert_prop_shorten(apr_pool_t *p,
					request_rec *r,
					const char* nsname,
					divy_rdbo_resource *rdb_r);
static char * _build_liveprop_nsname(apr_pool_t *p, int global_ns);

/* proppatch 処理 */
static dav_error * _set_mailwatch(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_mailwatch *mwatch_pr);
static dav_error * _remove_mailwatch(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_mailwatch *mwatch_pr);
static dav_error * _set_sharedcollection(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_shlink *shlink_pr);
static dav_error * _remove_sharedcollection(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_shlink *shlink_pr);
static dav_error * _set_user(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_user_iscreen *user_iscreen);
static dav_error * _set_group(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_group_iscreen *group_iscreen);
static dav_error * _set_sql(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_sql *sql_pr);
static dav_error * _set_clmodule(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_clmodule *clmodule_pr);
static dav_error * _set_sysmsg(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_sysmsg_iscreen *sysmsg_iscreen);
static dav_error * _set_dbms(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_dbms_iscreen *dbms_iscreen);
#ifdef DIVY_SUPPORT_EXTENDQUOTA
static dav_error * _set_systemquota(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_sysquota_iscreen *sysquota_iscreen);
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
static dav_error * _set_resourcestate(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_resourcestate_iscreen *rstate_iscreen);
static dav_error * _set_thumbnail(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_thumbnail_iscreen *thumbnail_iscreen);
static dav_error * _remove_thumbnail(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_thumbnail_iscreen *thumbnail_iscreen);

#ifdef DIVY_SUPPORT_PASSPOLICY
static dav_error * _set_passwordpolicy(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_passwordpolicy_iscreen *passpolicy_iscreen);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
#ifdef DIVY_SUPPORT_PASSPOLICY
static dav_error * _remove_passwordpolicy(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_passwordpolicy_iscreen *passpolicy_iscreen);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
#ifdef DIVY_SUPPORT_PASSPOLICY
static dav_error * _set_changepassword(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_changepassword_iscreen *changepassword_iscreen);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
static dav_error * _set_changeowner(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_changeowner_iscreen *changeowner_iscreen);
static dav_error * _set_changeleader(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_changeleader_iscreen *changeleader_iscreen);

static dav_error * _set_confirmreading(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_confirmreading_iscreen *iscreen);
static dav_error * _remove_confirmreading(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_confirmreading_iscreen *iscreen);


/*------------------------------------------------------------------------------
  Create & Initialize Live property group Hook structure
  ----------------------------------------------------------------------------*/
/**
 * mod_davで宣言されたdav_liveprop_group の別名.
 * ここでは、Liveプロパティの集合(ある特定のプロパティがLiveであるかどうか
 * を判断するために使用される構造体の配列）と、ネームスペースURIからなる
 * 配列と、Liveプロパティを処理するハンドラ構造体を設定し、初期化を行って
 * いる。
 */
const dav_liveprop_group dav_divy_liveprop_group = {
	dav_divy_props,
	dav_divy_namespace_uris,
	&dav_divy_hooks_liveprop
};

/*------------------------------------------------------------------------------
  Define Hook functions
  ----------------------------------------------------------------------------*/
/**
 * 指定されたプロパティID のプロパティ名と値を phdr に文字列として追加
 * する。
 *
 * @param resource const dav_resource *
 * @param propid int プロパティID
 * @param what dav_prop_insert 動作の種類
 * @param phdr apr_text_header *
 * @return dav_prop_insert どの動作を行ったか。
 *              DAV_PROP_INSERT_NAME      : プロパティ名の追加
 *              DAV_PROP_INSERT_VALUE     : プロパティ名・値の追加
 *              DAV_PROP_INSERT_SUPPORTED : (未サポート. DeltaV 用)
 *              DAV_PROP_INSERT_NOTSUPP   : このリソースでは未定義
 */
static dav_prop_insert dav_divy_insert_prop(const dav_resource *resource,
                                            int propid, dav_prop_insert what,
                                            apr_text_header *phdr)
{
	/* prop の処理を実施する */
	return _insert_prop(resource, propid, what, phdr, DAV_PROPFIND_IS_PROP);
}

/**
 * 指定されたプロパティIDのLiveプロパティが書き込み可能かどうか。
 *
 * Liveプロパティ一覧に存在しないプロパティID(propid)を指定されたら
 * "Read-Only"であると通知する。(基本的にLiveプロパティはRead-Only
 * としているからです。)
 *
 * @param resource const dav_resource * 
 * @param propid int プロパティID
 * @return 0: 書き込み不可能(Read-Only) / 1: 書き込み可能)
 */
static int dav_divy_is_writable(const dav_resource *resource, int propid)
{
	const dav_liveprop_spec *lspec = NULL;
	apr_pool_t *p = resource->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(resource->info->r);
	const divy_rdbo_extstatus *extstatus = NULL;

	TRACE(p);
	
	/* propid のdav_divy_liveprop_group * を取得する */
	(void) divy_get_liveprop_info(propid, resource->info->r,
				      &dav_divy_liveprop_group, &lspec);
	if (lspec != NULL) {
		/* サーバ主体かハイブリッドのメール送信の場合、
		 * メール監視フラグをPROPPATCHすることは禁止 */
		/* ハイブリッドはサーバ主体と同じ扱いです */
		if (lspec->propid == DAV_DIVY_PROPID_mailwatch &&
		    (dconf->mlserversend == DIVY_MLSERVERSEND_ON || 
			 dconf->mlserversend == DIVY_MLSERVERSEND_MIDDLE)) {

			extstatus = divy_get_extstatus(resource->info->r);
			if (divy_get_adminmode(resource->info->r)
										!= DIVY_ADMINMODE_ADMIN &&
				!divy_rdbo_is_groupleader(extstatus)) {
				return 0;
			}
		}
		return lspec->is_writable;
	}

	/* Liveプロパティ一覧にないプロパティが指定されたら？ */
	ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
		"You SHOULD NOT specify NONE-live property. "
	        "(propid = %d). We return Read-Only status, "
	        "BUT please check server program.", propid);
	
	return DAV_DIVY_PROP_READONLY;
}

/**
 * Liveプロパティに値を割り当てることができるのか、そして、与えられた値が
 * 正しいのかを確認する。
 *
 * @param resource const dav_resource *
 * @param elem apr_xml_elem *
 * @param operation int  PROPPATCHオペレーションの種類
 * @param context void ** Liveプロパティの解析結果を持つコンテキスト
 * @param defer_to_dead int * このプロパティをLiveではなくDeadプロパティとして
 *                            扱うかどうか。1: Dead 扱い / 0: Live 扱い
 * @return DAV エラー
 */
static dav_error * dav_divy_patch_validate(const dav_resource *resource,
                                           const apr_xml_elem *elem,
                                           int operation,
                                           void **context,
                                           int *defer_to_dead)
{
	dav_elem_private *priv      = elem->priv;
	int propid                  = priv->propid;
	apr_pool_t *p               = resource->pool;
	request_rec *r              = resource->info->r;
	divy_uri_spec  *u_spec      = resource->info->rdb_r->u_spec;
	dav_error *err              = NULL;
	divy_liveprop_ctx *ctx      = NULL;

	TRACE(p);

	/* 初期化 */
	*context = NULL;

	/*
	 * PROPPATCHが認められているかどうかの確認
	 * (note)
	 * 	Read-OnlyなLiveプロパティはこの関数に入ってこないので、
	 * 	敢えて拒否する必要はありません。
	 */

	/* [ PROPPATCH remove が許可されるpropid ]
	 *
	 *   * DAV_DIVY_PROPID_mailwatch
	 *   * DAV_DIVY_PROPID_sharedcollection
	 *   * DAV_DIVY_PROPID_resourcestate (ユーザ拡張ステータス有効時)
	 *   * DAV_DIVY_PROPID_thumbnail
	 *   * DAV_DIVY_PROPID_passwordpolicy (パスワードポリシーサポート時)
	 *   * DAV_DIVY_PROPID_confirmreading (開封通知)
	 */
	if (operation == DAV_PROP_OP_DELETE &&
		!(propid == DAV_DIVY_PROPID_mailwatch ||
		  propid == DAV_DIVY_PROPID_sharedcollection ||
		  propid == DAV_DIVY_PROPID_thumbnail ||
#ifdef DIVY_SUPPORT_PASSPOLICY
		  (divy_support_passpolicy(r) && propid == DAV_DIVY_PROPID_passwordpolicy) ||
#endif	/* DIVY_SUPPORT_PASSPOLICY */
		  (divy_support_extenduserstatus(r) && propid == DAV_DIVY_PROPID_resourcestate) ||
		  (divy_support_confirmreading(r) && propid == DAV_DIVY_PROPID_confirmreading))) {

		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The remove operation of this live property is not supported. "
				"(uri = %s, name = %s)", resource->uri, elem->name);

		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* Live プロパティコンテキストの生成 *
	 */
	ctx = apr_pcalloc(p, sizeof(divy_liveprop_ctx));
	ctx->propid    = propid;	/* propid をSaveする */
	ctx->operation = operation;

	/*
	 * 指定されたプロパティの値が正しいかどうか判定する
	 */
	switch (propid) {

		/* mailwatch プロパティ(メール監視フラグ) */
		case DAV_DIVY_PROPID_mailwatch:
		{
			divy_mailwatch_iscreen iscreen;

			if (operation == DAV_PROP_OP_SET) {
				/* パース */
				if (divy_parse_mailwatch_property(p, elem, &ctx->mwatch_pr)) {
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
			}
			else {
				/* DAV_PROP_OP_DELETE の時には何もしない */
				ctx->mwatch_pr = apr_pcalloc(p, sizeof(divy_rdbo_mailwatch));
			}
			iscreen.u_spec    = resource->info->rdb_r->u_spec;
			iscreen.mwatch_pr = ctx->mwatch_pr;

			/* 値の検証 */
			if (operation == DAV_PROP_OP_SET) {
				err = divy_validate_mailwatch_property(r, DIVY_VALIDATE_UPDATE, &iscreen);
			}
			else {
				err = divy_validate_mailwatch_property(r, DIVY_VALIDATE_DELETE, &iscreen);
			}
			if (err) return err;

			break;
		}

		/* sharedcollection プロパティ(共有コレクションのソース) */
		case DAV_DIVY_PROPID_sharedcollection:
		{
			divy_sharedcollection_iscreen iscreen;

			if (operation == DAV_PROP_OP_SET) {
				/* パース */
				if (divy_parse_sharedcollection_property(p, elem, &ctx->shlink_pr)) {
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
			}
			else {
				/* DAV_PROP_OP_DELETE の時には何もしない */
				ctx->shlink_pr = apr_pcalloc(p, sizeof(divy_rdbo_shlink));
			}
			iscreen.u_spec    = resource->info->rdb_r->u_spec;
			iscreen.shlink_pr = ctx->shlink_pr;

			/* 値の検証 */
			if (operation == DAV_PROP_OP_SET) {
				err = divy_validate_sharedcollection_property(r, DIVY_VALIDATE_UPDATE, &iscreen);
			}
			else {
				err = divy_validate_sharedcollection_property(r, DIVY_VALIDATE_DELETE, &iscreen);
			}
			if (err) return err;

			break;
		}

		/* resourcestate プロパティ */
		case DAV_DIVY_PROPID_resourcestate:
		{
			divy_resourcestate_iscreen *iscreen;

			if (operation == DAV_PROP_OP_SET) {
				/* パース */
				if (divy_parse_resourcestate_property(p, elem, &ctx->rstate_iscreen)) {
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
			}
			else {
#if 0
				/* View属性(公開/非公開状態) の削除(非公開)を設定 */
				ctx->rstate_iscreen = apr_pcalloc(p, sizeof(divy_resourcestate_iscreen));
				ctx->rstate_iscreen->type = DIVY_RSTATE_TYPE_VIEW;
				ctx->rstate_iscreen->state.published = 0;	/* 非公開にする */
				ctx->rstate_iscreen->next = NULL;	/* 今は1種類しかない */
#endif
				if (divy_parse_resourcestate_property(p, elem, &ctx->rstate_iscreen)) {
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
			}

			/* 足りなかった幾つかの項目を補う */
			for (iscreen = ctx->rstate_iscreen; iscreen; iscreen = iscreen->next) {
				iscreen->uri    = resource->info->rdb_r->uri;
				iscreen->u_spec = resource->info->rdb_r->u_spec;
				iscreen->resourcetype = resource->info->rdb_r->resourcetype;

				if (iscreen->type == DIVY_RSTATE_TYPE_BOX) {
					/* リソースIDを設定する（短縮URLに利用する）*/
					iscreen->optional.boxinfo->rsid = resource->info->rdb_r->rsid;
					/* リクエストのユーザをBOX作成者とする */
					iscreen->optional.boxinfo->creator = divy_get_userid(r);
				}
			}

			/* 値の検証 */
			if (operation == DAV_PROP_OP_SET) {
				err = divy_validate_resourcestate_property(r,
							DIVY_VALIDATE_UPDATE, ctx->rstate_iscreen);
			}
			else {
				err = divy_validate_resourcestate_property(r,
							DIVY_VALIDATE_DELETE, ctx->rstate_iscreen);
			}
			if (err) return err;

			break;
		}

		/* userdiscovery プロパティ(ユーザ編集) */
		case DAV_DIVY_PROPID_userdiscovery:

			/* パース */
			if (divy_parse_user_property(p, elem, &ctx->user_iscreen)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			ctx->user_iscreen->u_spec     = u_spec;
			ctx->user_iscreen->src_usr_pr = resource->info->rdb_r->usr_pr;

			/* 値の検証 */
			err = divy_validate_user_property(r, DIVY_VALIDATE_UPDATE, ctx->user_iscreen);
			if (err) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"userdiscovery\" property is wrong. (uri = %s)",
					resource->uri);
				return err;
			}

			break;

		/* groupdiscovery プロパティ(グループ編集) */
		case DAV_DIVY_PROPID_groupdiscovery:

			/* パース */
			if (divy_parse_group_property(p, elem, &ctx->group_iscreen)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			ctx->group_iscreen->u_spec     = u_spec;
			ctx->group_iscreen->src_grp_pr = resource->info->rdb_r->grp_pr;

			/* 値の検証 */
			err = divy_validate_group_property(r, DIVY_VALIDATE_UPDATE, ctx->group_iscreen);
			if (err) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"groupdiscovery\" property is wrong. (uri = %s)",
					resource->uri);
				return err;
			}

			break;

		/* sqldiscovery プロパティ (SQLの編集) */
		case DAV_DIVY_PROPID_sqldiscovery:
		{
			divy_sql_iscreen iscreen;

			/* パース */
			if (divy_parse_sql_property(p, elem, &ctx->sql_pr)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			iscreen.u_spec    = resource->info->rdb_r->u_spec;
			iscreen.sql_pr    = ctx->sql_pr;
			iscreen.db_sql_pr = resource->info->rdb_r->sql_pr;

			/* 値の検証 */
			err = divy_validate_sql_property(r, DIVY_VALIDATE_UPDATE, &iscreen);
			if (err) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"sqldiscovery\" property is wrong. (uri = %s)",
					resource->uri);
				return err;
			}

			break;
		}

		/* updatediscovery プロパティ 
		 * (クライアントモジュールの登録・編集・削除) */
		case DAV_DIVY_PROPID_updatediscovery:
		{
			divy_clmodule_iscreen iscreen;

			/* コレクションには付かない */
			if (resource->collection) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"This type of resource cannot have the "
					"\"updatediscovery\" element.(uri = %s)", resource->uri);
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* パース */
			if (divy_parse_updatediscovery_property(p, elem, &ctx->clmodule_pr)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			iscreen.u_spec      = resource->info->rdb_r->u_spec;
			iscreen.clmodule_pr = ctx->clmodule_pr;

			/* 値の検証 */
			err = divy_validate_updatediscovery_property(r, &iscreen);
			if (err) return err;

			break;
		}
		/* sysmsgdiscovery プロパティ(メッセージ編集) */
		case DAV_DIVY_PROPID_sysmsgdiscovery:

			/* パース */
			if (divy_parse_sysmsg_property(p, elem, &ctx->sysmsg_iscreen)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			ctx->sysmsg_iscreen->u_spec = u_spec;

			/* 値の検証 */
			err = divy_validate_sysmsg_property(r, DIVY_VALIDATE_UPDATE, ctx->sysmsg_iscreen);
			if (err) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"sysmsgdiscovery\" property is wrong. (uri = %s)",
					resource->uri);
				return err;
			}

			break;
		/* dbmsdiscovery プロパティ(編集) */
		case DAV_DIVY_PROPID_dbmsdiscovery:

			/* パース */
			if (divy_parse_dbms_property(p, elem, &ctx->dbms_iscreen)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			ctx->dbms_iscreen->u_spec = u_spec;

			/* 値の検証 */
			err = divy_validate_dbms_property(r, DIVY_VALIDATE_UPDATE, ctx->dbms_iscreen);
			if (err) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"dbmsdiscovery\" property is wrong. (uri = %s)",
					resource->uri);
				return err;
			}

			break;
#ifdef DIVY_SUPPORT_EXTENDQUOTA
		/* systemquota プロパティ */
		case DAV_DIVY_PROPID_systemquota:

			/* パース */
			if (divy_parse_systemquota_property(p, elem, &ctx->sysquota_iscreen)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			ctx->sysquota_iscreen->u_spec = u_spec;

			/* 値の検証 */
			err = divy_validate_systemquota_property(r, ctx->sysquota_iscreen);
			if (err) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"systemquota\" property is wrong. (uri = %s)",
					resource->uri);
				return err;
			}

			break;
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

		/* thumbnail プロパティ */
		case DAV_DIVY_PROPID_thumbnail:

			/* パース */
			if (divy_parse_thumbnail_property(p, elem, &ctx->thumbnail_iscreen)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			ctx->thumbnail_iscreen->u_spec = u_spec;

			/* 値の検証 */
			if (operation == DAV_PROP_OP_SET) {
				err = divy_validate_thumbnail_property(r, DIVY_VALIDATE_UPDATE,
								ctx->thumbnail_iscreen);
			}
			else {
				err = divy_validate_thumbnail_property(r, DIVY_VALIDATE_DELETE,
								ctx->thumbnail_iscreen);
			}
			if (err) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"thumbnail\" property is wrong. (uri = %s)", resource->uri);
				return err;
			}

			break;
#ifdef DIVY_SUPPORT_PASSPOLICY
		/* パスワードポリシープロパティ */
		case DAV_DIVY_PROPID_passwordpolicy:
			/* パスワードポリシーはサポートされているか? */
			if (!divy_support_passpolicy(r)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The \"passwordpolicy\" property was not supported.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* パース */
			if (divy_parse_passwordpolicy_property(p, elem, &ctx->passpolicy_iscreen)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			ctx->passpolicy_iscreen->u_spec = u_spec;

			/* 値の検証 */
			if (operation == DAV_PROP_OP_SET) {
				err = divy_validate_passwordpolicy_property(r, DIVY_VALIDATE_UPDATE,
								ctx->passpolicy_iscreen);
			}
			else {
				err = divy_validate_passwordpolicy_property(r, DIVY_VALIDATE_DELETE,
								ctx->passpolicy_iscreen);
			}
			if (err) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"passwordpolicy\" property is wrong. (uri = %s)",
					resource->uri);
				return err;
			}
			break;

#endif	/* DIVY_SUPPORT_PASSPOLICY */
#ifdef DIVY_SUPPORT_PASSPOLICY
		/* パスワード変更専用プロパティ
		 * [ 有効性 ]
		 *   パスワードポリシー機能が「サポート/未サポート」「有効/無効」に関わらず
		 *   定義済みとなります.
		 */
		case DAV_DIVY_PROPID_changepassword:
			/* パース */
			if (divy_parse_changepassword_property(p, elem, &ctx->changepassword_iscreen)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			ctx->changepassword_iscreen->u_spec     = u_spec;
			ctx->changepassword_iscreen->src_usr_pr = resource->info->rdb_r->usr_pr;

			/* 値の検証 */
			err = divy_validate_changepassword_property(r, DIVY_VALIDATE_UPDATE,
									ctx->changepassword_iscreen);
			if (err) return err;
			break;
#endif	/* DIVY_SUPPORT_PASSPOLICY */
		/*
		 * Otherユーザを管理下に置く
		 */
		case DAV_DIVY_PROPID_changeowner:
			if (!divy_support_groupleader(r)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The \"changeowner\" property was not supported.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* パース */
			if (divy_parse_changeowner_property(p, elem, &ctx->changeowner_iscreen)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			ctx->changeowner_iscreen->u_spec = u_spec;
			ctx->changeowner_iscreen->src_usr_pr = resource->info->rdb_r->usr_pr;

			/* 値の検証 */
			err = divy_validate_changeowner_property(r, DIVY_VALIDATE_UPDATE,
									ctx->changeowner_iscreen);
			if (err) return err;
			break;
		/*
		 * グループリーダに任命/解任
		 */
		case DAV_DIVY_PROPID_changeleader:
			if (!divy_support_groupleader(r)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The \"changeleader\" property was not supported.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* パース */
			if (divy_parse_changeleader_property(p, elem, &ctx->changeleader_iscreen)) {
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			ctx->changeleader_iscreen->u_spec = u_spec;
			ctx->changeleader_iscreen->src_grp_pr = resource->info->rdb_r->grp_pr;

			/* 値の検証 */
			err = divy_validate_changeleader_property(r, DIVY_VALIDATE_UPDATE,
									ctx->changeleader_iscreen);
			if (err) return err;
			break;

		/* 開封通知 プロパティ */
		case DAV_DIVY_PROPID_confirmreading:
		{
			if (!divy_support_confirmreading(r)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The \"confirmreading\" property was not supported.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			if (operation == DAV_PROP_OP_SET) {
				/* パース */
				if (divy_parse_confirmreading_property(p, elem, &ctx->confirmreading_iscreen)) {
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
			}
			else {
				ctx->confirmreading_iscreen = apr_pcalloc(p, sizeof(divy_confirmreading_iscreen));
			}

			ctx->confirmreading_iscreen->uri          = resource->info->rdb_r->uri;
			ctx->confirmreading_iscreen->userid       = apr_pstrdup(p, divy_get_userid(r));
			ctx->confirmreading_iscreen->creationdate = 0;	/* デフォルト値 */
			ctx->confirmreading_iscreen->u_spec       = u_spec;
			ctx->confirmreading_iscreen->rdb_r        = resource->info->rdb_r;

			/* 値の検証 */
			if (operation == DAV_PROP_OP_SET) {
				err = divy_validate_confirmreading_property(r,
							DIVY_VALIDATE_UPDATE, ctx->confirmreading_iscreen);
			}
			else {
				err = divy_validate_confirmreading_property(r,
							DIVY_VALIDATE_DELETE, ctx->confirmreading_iscreen);
			}
			if (err) return err;

			break;
		}

		default:
			/* ここにくるのはバグ(read-only にしていなかったとか) */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"The PROPPATCH operation of this live property "
				"is not supported.(uri = %s, name = %s)",
				resource->uri, elem->name);

			return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}
	/* context を渡す */
	*context = ctx;

	return NULL;
}

/**
 * Live プロパティに対し、プロパティ値の変更または削除を行う。
 * (note)
 * この関数では、Liveプロパティの変更をDBには反映しません。
 * patch_commit またはpatch_rollback されたときに最終的な動作を
 * 決定します。ですのでここでは値を記録しておくだけとします。
 *
 * @param resource const dav_resource *
 * @param elem const apr_xml_elem *
 * @param operation int
 * @param context void *
 * @param rollback_ctx dav_liveprop_rollback ** ロールバックコンテキスト
 */
static dav_error * dav_divy_patch_exec(const dav_resource *resource,
                                       const apr_xml_elem *elem,
                                       int operation,
                                       void *context,
                                       dav_liveprop_rollback **rollback_ctx)
{
	dav_error *err            = NULL;
	apr_pool_t *p             = resource->pool;
	request_rec *r            = resource->info->r;
	divy_liveprop_ctx *ctx    = (divy_liveprop_ctx *) context;
	const char *uri           = resource->uri;

	TRACE(p);

	/* 
	 * ロールバックコンテキストを作成する
	 */
	*rollback_ctx = apr_pcalloc(p, sizeof(dav_liveprop_rollback));
	(*rollback_ctx)->propid = ctx->propid;

	/* トランザクションコンテキストが無ければ作る */
	if (resource->info->liveprop_ts_ctx == NULL) {
		if (divy_db_create_transaction_ctx_new(r, &resource->info->liveprop_ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get transaction ctx for proppatch operation."
				"(uri = %s)", uri);
			resource->info->liveprop_ts_ctx = NULL;
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}
	(*rollback_ctx)->ts_ctx = resource->info->liveprop_ts_ctx;

	/*
	 * 指定されたLive プロパティを処理する
	 * commit/rollbackはmod_dav側で行うためここでは行いません。
	 */
	switch (ctx->propid) {
		/* mailwatch プロパティ(メール監視フラグ) */
		case DAV_DIVY_PROPID_mailwatch:
			
			/* 削除の場合 */
			if (operation == DAV_PROP_OP_DELETE) {
				err = _remove_mailwatch(r, resource,
						*rollback_ctx, ctx->mwatch_pr);
			}
			/* 設定の場合 */
			else {
				err = _set_mailwatch(r, resource,
						*rollback_ctx, ctx->mwatch_pr); 
			}
			if (err) return err;

			break;

		/* sharedcollection プロパティ(共有コレクションのソース) */
		case DAV_DIVY_PROPID_sharedcollection:

			/* 削除の場合 */
			if (operation == DAV_PROP_OP_DELETE) {
				err = _remove_sharedcollection(r, resource, *rollback_ctx, ctx->shlink_pr);
			}
			/* 設定の場合 */
			else {
				err = _set_sharedcollection(r, resource, *rollback_ctx, ctx->shlink_pr);
			}
			if (err) return err;

			break;

		/* resourcestate プロパティ */
		case DAV_DIVY_PROPID_resourcestate:

			/* 変更オペレーションをしなくてもいい場合 */
			if (!ctx->rstate_iscreen->need_to_update) {
				return NULL;	/* 成功したことにしてしまう */
			}

			/* プロパティの追加, 変更, 削除 */
			err = _set_resourcestate(r, resource, *rollback_ctx, ctx->rstate_iscreen);
			if (err) return err;

			break;

		/* userdiscovery プロパティ(ユーザの編集・削除) */
		case DAV_DIVY_PROPID_userdiscovery:
			
			/* プロパティのset */
			err = _set_user(r, resource, *rollback_ctx, ctx->user_iscreen);
			if (err) return err;
			
			break;

		/* groupdiscovery プロパティ(グループの編集・削除) */
		case DAV_DIVY_PROPID_groupdiscovery:
			
			/* プロパティのset */
			err = _set_group(r, resource, *rollback_ctx, ctx->group_iscreen);
			if (err) return err;
			
			break;

		/* sqldiscovery プロパティ (SQLの編集・削除) */
		case DAV_DIVY_PROPID_sqldiscovery:
			
			/* プロパティのset */
			err = _set_sql(r, resource, *rollback_ctx, ctx->sql_pr);
			if (err) return err;
			
			break;

		/* updatediscovery プロパティ 
		 * (クライアントモジュールの登録・編集・削除) */
		case DAV_DIVY_PROPID_updatediscovery:
			
			/* プロパティのset */
			err = _set_clmodule(r, resource,
						*rollback_ctx, ctx->clmodule_pr);
			if (err) return err;
			
			break;

		/* sysmsgdiscovery プロパティ(メッセージ編集・削除) */
		case DAV_DIVY_PROPID_sysmsgdiscovery:
			
			/* プロパティのset */
			err = _set_sysmsg(r, resource, *rollback_ctx, ctx->sysmsg_iscreen);
			if (err) return err;
			
			break;
		/* dbmsdiscovery プロパティ(DBMS編集) */
		case DAV_DIVY_PROPID_dbmsdiscovery:

			/* プロパティのset */
			err = _set_dbms(r, resource, *rollback_ctx, ctx->dbms_iscreen);
			if (err) return err;
			
			break;
#ifdef DIVY_SUPPORT_EXTENDQUOTA
		/* systemquota プロパティ */
		case DAV_DIVY_PROPID_systemquota:
			err = _set_systemquota(r, resource, *rollback_ctx, ctx->sysquota_iscreen);
			if (err) return err;
			
			break;
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
		/* thumbnail プロパティ */
		case DAV_DIVY_PROPID_thumbnail:
			/* 削除の場合 */
			if (operation == DAV_PROP_OP_DELETE) {
				err = _remove_thumbnail(r, resource, *rollback_ctx, ctx->thumbnail_iscreen);
			}
			/* 設定の場合 */
			else {
				err = _set_thumbnail(r, resource, *rollback_ctx, ctx->thumbnail_iscreen);
			}

			if (err) return err;
			break;
#ifdef DIVY_SUPPORT_PASSPOLICY
		/* パスワードポリシープロパティ */
		case DAV_DIVY_PROPID_passwordpolicy:
			/* 削除の場合 */
			if (operation == DAV_PROP_OP_DELETE) {
				err = _remove_passwordpolicy(r, resource, *rollback_ctx, ctx->passpolicy_iscreen);
			}
			/* 設定の場合 */
			else {
				err = _set_passwordpolicy(r, resource, *rollback_ctx, ctx->passpolicy_iscreen);
			}

			if (err) return err;
			break;
#endif	/* DIVY_SUPPORT_PASSPOLICY */
#ifdef DIVY_SUPPORT_PASSPOLICY
		/* パスワード変更専用のプロパティ */
		case DAV_DIVY_PROPID_changepassword:
			/* (note) remove は存在しません */
			err = _set_changepassword(r, resource, *rollback_ctx, ctx->changepassword_iscreen);
			if (err) return err;
			break;
#endif	/* DIVY_SUPPORT_PASSPOLICY */

		/* Otherユーザを管理下に置く */
		case DAV_DIVY_PROPID_changeowner:
			/* (note) remove は存在しません */
			err = _set_changeowner(r, resource, *rollback_ctx, ctx->changeowner_iscreen);
			if (err) return err;
			break;

		/* グループリーダに任命/解任 */
		case DAV_DIVY_PROPID_changeleader:
			/* (note) remove は存在しません */
			err = _set_changeleader(r, resource, *rollback_ctx, ctx->changeleader_iscreen);
			if (err) return err;
			break;

		/* 開封通知 */
		case DAV_DIVY_PROPID_confirmreading:
			
			/* 削除の場合 */
			if (operation == DAV_PROP_OP_DELETE) {
				err = _remove_confirmreading(r, resource,
						*rollback_ctx, ctx->confirmreading_iscreen);
			}
			/* 設定の場合 */
			else {
				err = _set_confirmreading(r, resource,
						*rollback_ctx, ctx->confirmreading_iscreen); 
			}
			if (err) return err;

			break;

	}

	return NULL;
}

/**
 * 指定されたoperation のPROPPATCH 動作による変更を確定する。
 * (note) 呼び出し単位
 *   １個のLiveプロパティ単位。
 *
 * (note) patch_commit について
 *   mod_dav のフレームワークに従うと、この関数で発生したエラーは全て無視され
 *   成功したものとみなされます。即ち、この関数では、エラーを発生させることが
 *   事実上出来なくなります。ですが、文字通りの変更の確定作業を行うに際し、
 *   レコード値の重複など、マルチユーザによる使用に起因してエラーが起きる
 *   かもしれません。これは、決して、極めてレアーであるとは断定できない
 *   ケースです。
 *
 *   TeamFile では同時にPROPPATCH出来るLiveプロパティの数を１つに限定し、
 *   かつDBのトランザクションをLiveプロパティのトランザクションと一致
 *   させることで不整合が生じないよう考慮していますが、上記の制限は
 *   複数のLiveプロパティをPROPPATCHする時にまた問題になるでしょう。
 *
 * @param resource const dav_resource *
 * @param operation int 操作の種類
 * @param context void *
 * @param rollback_ctx dav_liveprop_rollback *
 */
static void dav_divy_patch_commit(const dav_resource *resource,
                                  int operation,
                                  void *context,
                                  dav_liveprop_rollback *rollback_ctx)
{
	divy_db_transaction_ctx *ts_ctx;

	/* トランザクションがなければ何もしません。 */
	if (rollback_ctx == NULL || rollback_ctx->ts_ctx == NULL) {
		return;
	}
	ts_ctx = rollback_ctx->ts_ctx;
	
	/* トランザクションを確定する 
	 * 失敗時はログ出力ぐらいしか出来ません。*/
	if (divy_db_commit_transaction(ts_ctx)) {
		ERRLOG2(resource->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to commit transaction. "
			"(uri = %s, operation = %d)", resource->uri, operation);
	}

	/* dbconn は解放してはならないので何もしない */
	
	return;
}

/**
 * 指定されたoperation のPROPPATCH 動作による変更を破棄する。
 * (note) 呼び出し単位
 *   １個のLiveプロパティ単位。
 *
 * (note)
 *   patch_exec で変更は既にDBに書き込まれているため、それを打ち消す
 *   オペレーションを実施します。
 *
 * @param resource const dav_resource * PROPPATCH 対象のリソース
 * @param operation int 操作の種類
 * @param context void * (divy_rdbo_resource のこと)
 * @param rollback_ctx dav_liveprop_rollback * 変更を加えられる前の値を
 * 						保持するコンテキスト
 */
static dav_error * dav_divy_patch_rollback(const dav_resource *resource,
                                           int operation,
                                           void *context,
                                           dav_liveprop_rollback *rollback_ctx)
{
	divy_db_transaction_ctx *ts_ctx;
	
	/* トランザクションがなければ何もしません。 */
	if (rollback_ctx == NULL || rollback_ctx->ts_ctx == NULL) {
		return NULL;
	}
	ts_ctx = rollback_ctx->ts_ctx;
	
	/* トランザクションを破棄する */
	if (divy_db_rollback_transaction(ts_ctx)) {
		ERRLOG2(resource->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to rollback transaction. "
			"(uri = %s, operation = %d)", resource->uri, operation);
		return dav_new_error(resource->pool,
					HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	/* dbconn は解放してはならないので何もしない */

	return NULL;
}

/*--------------------------------------------------------------
  Define APR_HOOK functions
  --------------------------------------------------------------*/

/**
 * このLiveプロパティプロバイダが定義している全てのLive プロパティを
 * "<D:プロパティ名>値</D:プロパティ名>" の形に整形し、引数phdr に
 * 文字列として追加する。
 * 
 * 実際の処理では、既に取得済みの情報を使ってXMLを整形しているだけ。
 * この関数の中で、DBを検索している訳ではありません。
 *
 * この関数は、フック登録関数 dav_hook_insert_all_livepropsによって
 * コールバック登録されている。
 *
 * @param r request_rec *
 * @param resource const dav_resource * 処理対象のリソースまたはコレクション
 * @param what dav_prop_insert DAV_PROP_INSERT_NAME : <propname>を処理する
 *                             DAV_PROP_INSERT_VALUE: <allprop>を処理する
 * @param phdr apr_text_header * レスポンスXMLを追加する構造体
 */
DIVY_DECLARE(void) dav_divy_insert_all_liveprops(request_rec * r,
                                   const dav_resource * resource,
                                   dav_prop_insert what,
                                   apr_text_header * phdr)
{
	int i, propid, tbllen;

	/* 
	 * 指定された resource を処理するリポジトリプロバイダが
	 * D-IVY のリポジトリプロバイダではないとき、処理をしては
	 * ならない。
	 * (note)
	 * この関数は、mod_dav が独自に定義したApacheのフック関数の
	 * ハンドラとして登録されています。そのため、divyと一緒に
	 * mod_dav_fsやmod_dav_repos などのDAVプロバイダがLoadModule
	 * されているだけで、コールバックされてしまいます。
	 * 以下の処理は、divy以外のDAVプロバイダが持っているリソースや
	 * コレクションを処理対象から除外する目的で入れています。
	 */
	if (resource->hooks != &dav_divy_hooks_repository) {
		return;
	}

	/* 対象リソースまたはコレクションが無かった場合 */
	if (resource->exists == 0) {
		return;
	}

	/*
	 * XML 文字列を生成する。
	 */
	tbllen = DAV_DIVY_PROPS_LEN;
	for (i = 0; i < tbllen; i++) {
		propid = dav_divy_props[i].propid;

		switch (propid) {
			default:
				(void) _insert_prop(resource, propid, what, phdr,
						    DAV_PROPFIND_IS_ALLPROP);
				break;

			/* 以下は例外的な処理です。*/
			/* 
			 * mod_dav のcore live プロバイダは、<allprop> の時、
			 * <resourcetype> を全てのLiveプロバイダより前に
			 * 処理してしまいます。<prop> の時には、このようには
			 * しません。よって、以下では、resourcetypeを処理
			 * しないようにしています。
			 * この事情はとても込み入っていて、dav_fsやdav_repos
			 * では発生しません。TeamFileでは、独自のプロパティID
			 * を使っているため、このような結果になっているのですが、
			 * 独自プロパティIDを使うこと自体は実装仕様違反では
			 * ないため mod_dav 側の実装考慮もれと思われます。
			 * mod_dav 側もそう簡単には修正できない事柄です。
			 */
			case DAV_DIVY_PROPID_resourcetype:
				break;

			/*
			 * DAV_PROP_INSERT_NAME の場合、以下のプロパティは
			 * 無視します。
			 * (note) 無視してもよい理由
			 *   mod_dav が定義した関数 dav_get_allprops が 以下の
			 *   ２つのプロパティに対して盲目的に発行するサブ
			 *   リクエストを排除する目的で、Dead プロパティ
			 *   プロバイダが以下のプロパティの名前をmod_dav に報告
			 *   しています。ですので、Live プロパティプロバイダ側
			 *   ではこれらに関する処理を行いません。
			 *   なお、dav_get_props の場合には、Live プロパティ
			 *   プロバイダが先に呼ばれるので、処理しなければ
			 *   なりません。
			 */
			case DAV_DIVY_PROPID_getcontenttype:
			case DAV_DIVY_PROPID_getcontentlanguage:
				if (what == DAV_PROP_INSERT_NAME) {
					break;	/* 無視する(理由は上) */
				}
				else {
					(void) _insert_prop(resource, propid, what, phdr,
							    DAV_PROPFIND_IS_ALLPROP);
				}
				break;

			/* 以下は、allprop において返却する必要なし */
			case DAV_DIVY_PROPID_source: 
			case DAV_DIVY_PROPID_foldertype:
			case DAV_DIVY_PROPID_displayorder:
			case DAV_DIVY_PROPID_keylist:
			case DAV_DIVY_PROPID_userdiscovery:
			case DAV_DIVY_PROPID_groupdiscovery:
			case DAV_DIVY_PROPID_sqldiscovery:
			case DAV_DIVY_PROPID_updatediscovery:
			case DAV_DIVY_PROPID_statusdiscovery:
			case DAV_DIVY_PROPID_sysmsgdiscovery:
			case DAV_DIVY_PROPID_dbmsdiscovery:
			case DAV_DIVY_PROPID_dependsqldiscovery:
			case DAV_DIVY_PROPID_sqlpreferencediscovery:
			case DAV_DIVY_PROPID_dbmserrdiscovery:
			case DAV_DIVY_PROPID_analysiserrdiscovery:
			case DAV_DIVY_PROPID_linkdberrdiscovery:
			case DAV_DIVY_PROPID_trashdiscovery:
			case DAV_DIVY_PROPID_resourcestate:
			case DAV_DIVY_PROPID_user_privilege_grant_set:
			case DAV_DIVY_PROPID_thumbnail:
#ifdef DIVY_SUPPORT_PASSPOLICY
			case DAV_DIVY_PROPID_passwordpolicy:
#endif	/* DIVY_SUPPORT_PASSPOLICY */
			case DAV_DIVY_PROPID_confirmreading:
			case DAV_DIVY_PROPID_shorten:
				break;
		}
	}

	return;
}

/**
 * 指定されたLiveプロパティ名と対応する プロパティID (プロバイダローカル, 
 * mod_dav_divy.hで定義)を取得して返却し、このLiveプロパティを処理する
 * Liveプロパティプロバイダ(の構造体)へのポインタを hooksに設定する。
 * 
 * D-IVY のLiveプロパティプロバイダもLiveプロパティを独自に定義する
 * ことが出来るため、拡張されたプロパティを探す目的でmod_davの
 * dav_find_liveprop_provider関数から呼び出されている.
 * 
 * [取得条件]
 *  ・指定されたリソースを取り扱うリポジトリプロバイダがdivyのものであること
 *  ・指定されたネームスペース名(ns_uri)が"DAV:"であること
 *  ・指定されたLiveプロパティ名(name)がLiveプロパティのキーと値を持つ
 *    ハッシュテーブル(resource->info->db_r->lpr_hash)に登録されていること.
 *
 * mod_dav独自のハンドラdav_hook_find_livepropによってフックハンドラとして
 * 登録されている関数.
 *
 * @param resource const dav_resource * 処理対象のリソースまたはコレクション
 * @param ns_uri const char * ネームスペースURI
 * @param name const char * プロパティ名
 * @param hooks const dav_hooks_liveprop ** このプロパティを処理する
 *                                          Liveプロパティプロバイダへのポインタ
 * @return int プロパティID 
 * 	処理すべきリソースではないまたは、教えたくない(?) 、知らない
 * 	Liveプロパティである場合には0 を返却する。(mod_davとのインターフェース)
 */
DIVY_DECLARE(int) dav_divy_find_liveprop(const dav_resource * resource, 
                           		const char *ns_uri, const char *name,
                           		const dav_hooks_liveprop ** hooks)
{
	int low, high, mid, ret, ns;
	const char * const *uris = dav_divy_liveprop_group.namespace_uris;
	const dav_liveprop_spec *spec;

	/* 
	 * 指定された resource を処理するリポジトリプロバイダが
	 * D-IVY のリポジトリプロバイダではないとき、処理をしては
	 * ならない。(関数 dav_divy_insert_all_livepropsと同じ理由です。)
	 */
	if (resource->hooks != &dav_divy_hooks_repository) {
		return 0;
	}

	/*
	 * プロパティ名からプロパティIDを取得して返却する
	 */
	/* ns_uri を探す */
	for (ns = 0; uris[ns] != NULL; ns++) {
		if (strcmp(ns_uri, uris[ns]) ==0) {
			break;
		}
	}

	if (uris[ns] == NULL) {
		return 0;
	}

	/* プロパティIDを探す(バイナリサーチ) */
	for (low = 0, high = DAV_DIVY_PROPS_LEN; low < high; ) {
		mid = (low + high) / 2;
		spec = &dav_divy_props[mid];
		ret = strcmp(spec->name, name);
		if (ret == 0 && ns == spec->ns) {
			*hooks = dav_divy_liveprop_group.hooks;
			return spec->propid;
		}
		else if (ret < 0) {
			low = mid + 1;
		}
		else {
			high = mid;
		}
	}

	return 0;
}

/*--------------------------------------------------------------
  Define public function
  --------------------------------------------------------------*/
DIVY_DECLARE(int) divy_get_liveprop_info(int propid, 
                                        request_rec *r,
                                        const dav_liveprop_group *group,
                                        const dav_liveprop_spec **info)
{
	/* 
	 * 性能向上対策は、このLiveプロパティプロバイダが管理している
	 * プロパティにしか適用されません。
	 * 他のプロパティの場合には、core のロジックに任せます。
	 */
	if (group != &dav_divy_liveprop_group) {
		return dav_get_liveprop_info(propid, group, info);
	}

	/* 情報がキャッシュされていなければ作成する */
	if (dav_divy_props_array == NULL) {
		int i, idx;
		int len = DAV_DIVY_PROPS_LEN;
		apr_status_t st;
		/* ロックをかける */
		st = apr_thread_mutex_lock(init_mutex);
		if (st != APR_SUCCESS) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to lock for creating dav_divy_props_array.(code = %d)", st);
			/* 継続してみる */
		}

		/*
		 * (note) r->server->process->poolに格納しても良い根拠
		 * 	・この配列はサーバが再起動するまで変更されない
		 * 	・全てのユーザ、ディレクティブが同じ値をReadできればよい
		 * 以上の条件を１つでも満たさない状況になったら、考え直して下さい。
		 */
		if (dav_divy_props_array == NULL) {	/* ダブルチェックパターン */

			dav_divy_props_array = apr_pcalloc(r->server->process->pool, 
						sizeof(dav_liveprop_spec *) * len);
			apr_pool_cleanup_register(r->server->process->pool, NULL, 
						  _cleanup_props_array,
						  apr_pool_cleanup_null);
			/* 
			 * プロバイダIDが採番されているにも関わらず、dav_divy_props
			 * 配列上にエントリがなければ、この配列のその部分の要素は
			 * NULL になります。
			 * でもそれは非常に少ないので気にしないことにします。
			 */
			for (i = 0; i < len; i++) {
				idx = dav_divy_props[i].propid - 
					(DAV_DIVY_PROPID_BEGIN + 1);
				dav_divy_props_array[idx] 
					= (dav_liveprop_spec *) &dav_divy_props[i];
			}
		}
		/* ロックを解除 */
		if (st == APR_SUCCESS) {	/* ロックを掛けるのに成功していた場合 */
			st = apr_thread_mutex_unlock(init_mutex);
		}
		if (st != APR_SUCCESS) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to unlock for creating dav_divy_props_array. "
			        "Code = %d", st);
		}
	}

	/* 配列からdav_liveprop_spec 構造体を取り出す */
	*info = dav_divy_props_array[propid - (DAV_DIVY_PROPID_BEGIN + 1)];
	if (*info != NULL) {
		return dav_get_liveprop_ns_index(group->namespace_uris[(*info)->ns]);
	}

	/* 値が取得出来なかった */
	*info = NULL;
	return 0;
}

/**
 * レスポンスで使用するLiveプロパティのネームスペース文字列を組み立てて返却する。
 */
DIVY_DECLARE(const char *) divy_make_liveprop_ns(apr_pool_t *p, int want)
{
	char *ns_str = NULL;
	const char * const *uris = dav_divy_liveprop_group.namespace_uris;

	/* (note) 何度も文字列合成したくないので少々汚いコードになります。*/
	if ((want & DIVY_GET_DAV_NS) && (want & DIVY_GET_DIVY_NS)) {
		ns_str = apr_psprintf(p,
				" xmlns:"DAV_NS_PREFIX"=\"%s\""
				" xmlns:"DIVY_NS_PREFIX"=\"%s\"",
				uris[DAV_NS_PREFIX_IDX], uris[DIVY_NS_PREFIX_IDX]);
	}
	else if (want & DIVY_GET_DAV_NS) {
		ns_str = apr_psprintf(p,
				" xmlns:"DAV_NS_PREFIX"=\"%s\"",
				uris[DAV_NS_PREFIX_IDX]);
	}
	else if (want & DIVY_GET_DIVY_NS) {
		ns_str = apr_psprintf(p,
				" xmlns:"DIVY_NS_PREFIX"=\"%s\"",
				uris[DIVY_NS_PREFIX_IDX]);
	}
	else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Invalid \"want\" arg.(want = %d)", want);
		ns_str = NULL;
	}

	return ns_str;
}

/**
 * プロパティID DAV_DIVY_PROPID_user_privilege_grant_set のエレメントを作成して
 * phdr につめて返却する。(SEARCH専用のバックドア)
 *
 */
DIVY_DECLARE(void) divy_insert_prop_user_privilege_grant_set(request_rec *r,
						const char *nsname,
						int mode,
						divy_rdbo_resource *rdb_r,
						apr_text_header *phdr)
{
	apr_pool_t *p = r->pool;
	char *inner_element, *s;
	const dav_liveprop_spec *lspec = NULL;

	if (phdr == NULL) return;	/* 何もできない */

	/* ユーザ拡張ステータスをサポートしていなければ何もしない */
	if (!divy_support_extenduserstatus(r)) return;

	/* Liveプロパティの名前を取得する */
	(void) divy_get_liveprop_info(DAV_DIVY_PROPID_user_privilege_grant_set, r,
				      		&dav_divy_liveprop_group, &lspec);
	if (lspec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get dav_liveprop_spec."
			"(propid = DAV_DIVY_PROPID_user_privilege_grant_set)");
		return;
	}

	/* プレフィックスが指定されていない場合 */
	if (IS_EMPTY(nsname)) {
		nsname = DIVY_NS_PREFIX;
	}

	/* Liveプロパティの内部エレメントを取得する */
	inner_element = _insert_prop_user_privilege_grant_set(p, r, nsname, mode, rdb_r);

	/* Liveプロパティの組み立て */
	if (IS_FILLED(inner_element)) {
		s = apr_psprintf(p, "<%s:%s>%s</%s:%s>" CRLF,
				nsname, lspec->name, inner_element,
				nsname, lspec->name);
	}
	else {
		/* EMPTY にする */
		s = apr_psprintf(p, "<%s:%s/>" CRLF, nsname, lspec->name);
	}
	apr_text_append(p, phdr, s);
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * デフォルトのユーザパスワードポリシー状態を生成して返却する.
 * デフォルト値を入れるだけです. 必要であれば後で置き換えてください.
 */
DIVY_DECLARE(divy_rdbo_passpolicystatus *)
	divy_create_user_passwordpolicy(apr_pool_t *p, const char *userid)
{
	divy_rdbo_passpolicystatus *ps = NULL;
	
	ps = apr_pcalloc(p, sizeof(divy_rdbo_passpolicystatus));
	ps->policyid        = DIVY_DEFAULT_POLICY_ID;	/* デフォルトID */
	ps->usrid           = apr_pstrdup(p, userid);
	ps->lastchange      = 0L;
	ps->sendexpiredmail = 0L;
	ps->firstlogin      = 0L;

	return ps;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */


/*--------------------------------------------------------------
  Define private function 
  --------------------------------------------------------------*/
static apr_status_t _cleanup_props_array(void *ctx)
{
	dav_divy_props_array = NULL;
	return APR_SUCCESS;
}

/**
 * 指定されたプロパティID のプロパティ名と値を phdr に文字列として追加
 * する。
 * (note) 役割
 * 	この関数は、propfind_type を区別する目的で導入されました。
 * 	propの時には、ハンドラdav_divy_insert_propが呼ばれ、
 * 	allpropsの時には、ハンドラdav_divy_insert_all_livepropsが
 * 	呼ばれる性質を利用して、この時に適切なpropfind_type を設定し
 * 	区別します。
 *
 * @param resource const dav_resource *
 * @param propid int プロパティID
 * @param what dav_prop_insert 動作の種類
 * @param phdr apr_text_header *
 * @param propfind_type int propかallprops を区別するフラグ。
 * 			propnameもallpropsに含まれます。
 * 			(DAV_PROPFIND_IS_ALLPROP / DAV_PROPFIND_IS_PROP)
 * @return dav_prop_insert どの動作を行ったか。
 *              DAV_PROP_INSERT_NAME      : プロパティ名の追加
 *              DAV_PROP_INSERT_VALUE     : プロパティ名・値の追加
 *              DAV_PROP_INSERT_SUPPORTED : (未サポート. DeltaV 用)
 *              DAV_PROP_INSERT_NOTSUPP   : このリソースでは未定義
 */
static dav_prop_insert _insert_prop(const dav_resource *resource,
					int propid,
					dav_prop_insert what,
					apr_text_header *phdr,
					int propfind_type)
{
	char *s = NULL, *value = NULL, *nsname = NULL;
	const dav_liveprop_spec *lspec = NULL;
	request_rec *r            = resource->info->r;
	divy_rdbo_resource *rdb_r = resource->info->rdb_r;
	apr_pool_t *p             = resource->pool;
	int global_ns;
	char *tmp, *datebuf = NULL;

	/*
	 * プロパティID からdav_liveprop_spec * を取得する
	 */
	global_ns = divy_get_liveprop_info(propid, r, &dav_divy_liveprop_group, &lspec);
	if (lspec == NULL) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get dav_liveprop_spec."
			"(propid = %d)", propid);
		return DAV_PROP_INSERT_NOTDEF;
	}

	/*
	 * プロパティID 毎に処理する
	 */
	switch (propid) {
		case DAV_DIVY_PROPID_creationdate:	/* 常に存在する */
			if (what == DAV_PROP_INSERT_NAME) break;

			/* 値が0 だったらEMPTYにする */
			if (rdb_r->creationdate == 0) {
				value = NULL;
				break;
			}
			/* フォーマット変換 (Epoch time -> ISO8601) */
			if (divy_format_time_t(p, rdb_r->creationdate, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
				value = NULL;	/* EMPTYにする */
			}
			else {
				value = datebuf;
			}
			break;

		case DAV_DIVY_PROPID_displayname:	/* 常に存在する */
			if (what == DAV_PROP_INSERT_NAME) break;

			value = (char *) dav_divy_escape_xmlstr(p,
						rdb_r->displayname,
						DIVY_XML_T2T_QUOTE);

			/* NFC -> NFD 変換を行う(環境変数が設定されていた場合)
			 * (note)
			 * href の変換は訳あってmod_dav が実施。また、rdb_r->displayname の値は
			 * 置き換えません。出力値だけ変換します。他への影響を避けるためです。*/
			if (apr_table_get(r->subprocess_env, DIVY_APENV_PROPFINDNFC2NFD)) {
				value = (char *) convert_utf8_to_utf8_formac(r, value, strlen(value), p);
			}
			break;

		case DAV_DIVY_PROPID_getcontentlanguage:

			if (resource->collection) {
				if (propfind_type == DAV_PROPFIND_IS_PROP) {
					/* prop: 404 Not Found */
					return DAV_PROP_INSERT_NOTDEF;
				}
				else {	/* allprops: 未指定 */
					return what;
				}
			}
			else {
				value = rdb_r->getcontentlanguage;
				if (IS_EMPTY(value)) {
					if (propfind_type == DAV_PROPFIND_IS_PROP) {
						/* prop: 404 Not Found */
						return DAV_PROP_INSERT_NOTDEF;
					}
					else {	/* allprop: 未指定 */
						return what;
					}
				}
			}
			break;

		case DAV_DIVY_PROPID_getcontentlength:

			if (resource->collection) {
				if (propfind_type == DAV_PROPFIND_IS_PROP) {
					/* prop: 404 Not Found */
					return DAV_PROP_INSERT_NOTDEF;
				}
				else {	/* allprops: 未指定 */
					return what;
				}
			}
			else {
				if (what == DAV_PROP_INSERT_NAME) break;
				value = apr_psprintf(p, "%"APR_INT64_T_FMT, rdb_r->getcontentlength);
			}
			break;

		case DAV_DIVY_PROPID_getcontenttype:

			value = rdb_r->getcontenttype;
			if (IS_EMPTY(value)) {
				if (propfind_type == DAV_PROPFIND_IS_PROP) {
					/* prop: 404 Not Found */
					return DAV_PROP_INSERT_NOTDEF;
				}
				else {	/* allprop: 未指定 */
					return what;
				}
			}
			break;

		case DAV_DIVY_PROPID_getetag:	/* 常に存在する */
			if (what == DAV_PROP_INSERT_NAME) break;

			value = rdb_r->getetag;
			break;

		case DAV_DIVY_PROPID_getlastmodified:	/* 常に存在する */
			if (what == DAV_PROP_INSERT_NAME) break;

			/* 値が0 だったらEMPTYにする */
			if (rdb_r->getlastmodified == 0) {
				value = NULL;
				break;
			}

			/* フォーマット変換 (Epoch time -> RFC822) */
			if (divy_format_time_t(p, rdb_r->getlastmodified, DIVY_TIME_STYLE_RFC822, &datebuf)) {
				value = NULL;	/* EMPTYにする */
			}
			else {
				value = datebuf;
			}
			break;

		case DAV_DIVY_PROPID_resourcetype:	/* 常に存在する */
			if (what == DAV_PROP_INSERT_NAME) break;

			if (resource->collection) {
				value = "<"DAV_NS_PREFIX":collection/>";
			}
			else {
				value = "";
			}
			break;

		/* Core live プロパティプロバイダに任せるので定義してはならない */
#if 0
		case DAV_DIVY_PROPID_lockdiscovery:
		case DAV_DIVY_PROPID_supportedlock:
#endif
		case DAV_DIVY_PROPID_creator:	/* 常に存在する */
			if (what == DAV_PROP_INSERT_NAME) break;

			value = (char *) dav_divy_escape_xmlstr(p, rdb_r->creator, DIVY_XML_T2T_CDATA);
			break;

		case DAV_DIVY_PROPID_lastmodifier:	/* 常に存在する */
			if (what == DAV_PROP_INSERT_NAME) break;

			value = (char *) dav_divy_escape_xmlstr(p, rdb_r->lastmodifier, DIVY_XML_T2T_CDATA);
			break;

		case DAV_DIVY_PROPID_sharedcollection:
		{
			dav_buffer sbuf = { 0 };
			divy_rdbo_shlink *shlink;

			/* リソースだった or 値が無かった場合 */
			if (!resource->collection || rdb_r->shlink_pr == NULL) {
				if (propfind_type == DAV_PROPFIND_IS_PROP) {
					/* prop: 404 Not Found */
					return DAV_PROP_INSERT_NOTDEF;
				}
				else {	/* allprop: 未指定 */
					return what;
				}
			}

			if (what == DAV_PROP_INSERT_NAME) break;

			nsname = _build_liveprop_nsname(p, global_ns);

			dav_set_bufsize(p, &sbuf, 1024);
			sbuf.cur_len = 0;
			for (shlink = rdb_r->shlink_pr; 
			     shlink; shlink = shlink->next) {

				tmp = apr_psprintf(p, 
					"<%s:sharedcollectioninfo>" CRLF
					"<%s:shname>%s</%s:shname>" CRLF
					"<%s:linkuri>%s/</%s:linkuri>" CRLF
					"</%s:sharedcollectioninfo>" CRLF,
					nsname, nsname, 
					dav_divy_escape_xmlstr(p, shlink->displayname, DIVY_XML_T2T_CDATA),
					nsname, nsname,
					dav_divy_escape_uri(p, shlink->uri),
					nsname, nsname);
				dav_buffer_append(p, &sbuf, tmp);
			}

			/* バッファの文字列を渡す */
			value = sbuf.buf;
			break;
		}

		case DAV_DIVY_PROPID_resourcestate:

			/* サポートしていなければ404 Not Found */
			if (!divy_support_extenduserstatus(r)) {
				return DAV_PROP_INSERT_NOTDEF;
			}

			/* 値が無かった場合 */
			if (rdb_r->rstate_pr == NULL && rdb_r->grp_pr == NULL) {
				if (propfind_type == DAV_PROPFIND_IS_PROP) {
					/* prop: 404 Not Found */
					return DAV_PROP_INSERT_NOTDEF;
				}
				else {	/* allprop: 未指定 */
					return what;
				}
			}

			if (what == DAV_PROP_INSERT_NAME) break;

			nsname = _build_liveprop_nsname(p, global_ns);
			
			/* Liveプロパティの組み立て */
			value = _insert_prop_resourcestate(p, r, nsname, rdb_r);
			break;

		case DAV_DIVY_PROPID_user_privilege_grant_set:

			/* サポートしていなければ404 Not Found */
			if (!divy_support_extenduserstatus(r)) {
				return DAV_PROP_INSERT_NOTDEF;
			}

			if (what == DAV_PROP_INSERT_NAME) break;

			nsname = _build_liveprop_nsname(p, global_ns);

			/* Liveプロパティの組み立て */
			value = _insert_prop_user_privilege_grant_set(p, r, nsname,
							DIVY_GET_ALL_PRIVILEGE, rdb_r);
			break;
				
		case DAV_DIVY_PROPID_mailwatch:

			/* 値が無かった場合 */
			if (rdb_r->mwatch_pr == NULL) {
				if (propfind_type == DAV_PROPFIND_IS_PROP) {
					/* prop: 404 Not Found */
					return DAV_PROP_INSERT_NOTDEF;
				}
				else {	/* allprop: 未指定 */
					return what;
				}
			}
			if (what == DAV_PROP_INSERT_NAME) break;

			nsname = _build_liveprop_nsname(p, global_ns);

			/* Liveプロパティの組み立て */
			value = _insert_prop_mailwatch(p, r, nsname, rdb_r);
			break;

#ifdef DIVY_SUPPORT_EXTENDQUOTA
		case DAV_DIVY_PROPID_systemquota:

			if (rdb_r->u_spec == NULL ||
				rdb_r->u_spec->infotype != DIVY_INFOTYPE_root) {
				/* prop: 404 Not Found */
				return DAV_PROP_INSERT_NOTDEF;
			}

			if (rdb_r->sysquota_pr == NULL) {
				return DAV_PROP_INSERT_NOTDEF;
			}
			if (what == DAV_PROP_INSERT_NAME) break;

			nsname = _build_liveprop_nsname(p, global_ns);

			/* Liveプロパティの組み立て */
			value = _insert_prop_systemquota(p, r, nsname, rdb_r);
			break;

#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
		case DAV_DIVY_PROPID_thumbnail:
			if (resource->collection) {
				if (propfind_type == DAV_PROPFIND_IS_PROP) {
					/* prop: 404 Not Found */
					return DAV_PROP_INSERT_NOTDEF;
				}
				else {	/* allprop: 未指定 */
					return what;
				}
			}
			if (what == DAV_PROP_INSERT_NAME) break;

			nsname = _build_liveprop_nsname(p, global_ns);

			/* Liveプロパティの組み立て */
			value = _insert_prop_thumbnail(p, r, nsname, rdb_r);
			if (value == NULL) {
				return DAV_PROP_INSERT_NOTDEF;
			}
			break;
		case DAV_DIVY_PROPID_trashinformation:
			if (rdb_r->trash_pr == NULL) {
				return DAV_PROP_INSERT_NOTDEF;
			}

			if (what == DAV_PROP_INSERT_NAME) break;

			nsname = _build_liveprop_nsname(p, global_ns);
			value = apr_psprintf(p, CRLF
				"<%s:deleter>%s</%s:deleter>" CRLF
				"<%s:deletion>%s</%s:deletion>" CRLF,
				nsname,
				dav_divy_escape_xmlstr(p, rdb_r->trash_pr->deleter_name, DIVY_XML_T2T_CDATA),
				nsname, nsname, rdb_r->trash_pr->daletion_date, nsname);
			break;
#ifdef DIVY_SUPPORT_PASSPOLICY
		case DAV_DIVY_PROPID_passwordpolicy:
		{
			divy_rdbo_passwordpolicy *passpolicy_pr = NULL;

			if (rdb_r->u_spec == NULL ||
				rdb_r->u_spec->infotype != DIVY_INFOTYPE_m_user) {
				/* prop: 404 Not Found */
				return DAV_PROP_INSERT_NOTDEF;
			}

			/* passwordpolicy プロパティを取得する */
			if (divy_get_cached_passwordpolicy(r, DIVY_DEFAULT_POLICY_ID,
						0, &passpolicy_pr)) {
				ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get passwordpolicy property.");
				return DAV_PROP_INSERT_NOTDEF;
			}

			if (passpolicy_pr == NULL) {
				return DAV_PROP_INSERT_NOTDEF;
			}
			if (what == DAV_PROP_INSERT_NAME) break;

			nsname = _build_liveprop_nsname(p, global_ns);

			/* Liveプロパティの組み立て */
			value = _insert_prop_passwordpolicy(p, r, nsname, passpolicy_pr);
			break;
		}
#endif	/* DIVY_SUPPORT_PASSPOLICY */
		case DAV_DIVY_PROPID_groupstate:

			/* サポートしていなければ404 Not Found */
			if (!divy_support_groupleader(r)) {
				return DAV_PROP_INSERT_NOTDEF;
			}

			/* グループフォルダで定義されるプロパティ */
			if (rdb_r->u_spec == NULL ||
				rdb_r->u_spec->infotype != DIVY_INFOTYPE_group_e) {
				/* prop: 404 Not Found */
				return DAV_PROP_INSERT_NOTDEF;
			}

			if (rdb_r->grp_pr == NULL) {
				return DAV_PROP_INSERT_NOTDEF;
			}
			if (what == DAV_PROP_INSERT_NAME) break;

			nsname = _build_liveprop_nsname(p, global_ns);

			/* Liveプロパティの組み立て */
			value = _insert_prop_groupstate(p, r, nsname, rdb_r->grp_pr);
			break;

		case DAV_DIVY_PROPID_confirmreading:

			/* サポートしていなければ404 Not Found */
			if (!divy_support_confirmreading(r)) {
				return DAV_PROP_INSERT_NOTDEF;
			}

			/* 値が無かった場合 */
			if (rdb_r->confirmreading_pr == NULL) {
				if (propfind_type == DAV_PROPFIND_IS_PROP) {
					/* prop: 404 Not Found */
					return DAV_PROP_INSERT_NOTDEF;
				}
				else {	/* allprop: 未指定 */
					return what;
				}
			}
			if (what == DAV_PROP_INSERT_NAME) break;

			nsname = _build_liveprop_nsname(p, global_ns);

			/* Liveプロパティの組み立て */
			value = _insert_prop_confirmreading(p, r, nsname, rdb_r);
			break;

		case DAV_DIVY_PROPID_shorten:

			/* サポートしていなければ404 Not Found */
			if (!divy_support_shorten(r)) {
				return DAV_PROP_INSERT_NOTDEF;
			}

			/* 値が無かった場合 通常ありえない */
			if (IS_EMPTY(rdb_r->rsid)) {
				return DAV_PROP_INSERT_NOTDEF;
			}

			if (what == DAV_PROP_INSERT_NAME) break;

			nsname = _build_liveprop_nsname(p, global_ns);

			/* Liveプロパティの組み立て */
			value = _insert_prop_shorten(p, r, nsname, rdb_r);
			break;


		/* このプロバイダでは定義されていません */
		case DAV_DIVY_PROPID_source:
			return DAV_PROP_INSERT_NOTDEF;

		/*
		 * 以下はPROPFIND 出来るリソースでは決して定義
		 * されないタイプのLiveプロパティです。SEARCHで
		 * 利用されるだけです。
		 */
		case DAV_DIVY_PROPID_foldertype:
		case DAV_DIVY_PROPID_displayorder:
		case DAV_DIVY_PROPID_keylist:
		case DAV_DIVY_PROPID_userdiscovery:
		case DAV_DIVY_PROPID_groupdiscovery:
		case DAV_DIVY_PROPID_sqldiscovery:
		case DAV_DIVY_PROPID_updatediscovery:
		case DAV_DIVY_PROPID_statusdiscovery:
		case DAV_DIVY_PROPID_sysmsgdiscovery:
		case DAV_DIVY_PROPID_dbmsdiscovery:
		case DAV_DIVY_PROPID_dependsqldiscovery:
		case DAV_DIVY_PROPID_sqlpreferencediscovery:
		case DAV_DIVY_PROPID_dbmserrdiscovery:
		case DAV_DIVY_PROPID_analysiserrdiscovery:
		case DAV_DIVY_PROPID_linkdberrdiscovery:
		case DAV_DIVY_PROPID_trashdiscovery:
#ifdef DIVY_SUPPORT_PASSPOLICY
		case DAV_DIVY_PROPID_changepassword:
#endif	/* DIVY_SUPPORT_PASSPOLICY */
		case DAV_DIVY_PROPID_changeowner:
		case DAV_DIVY_PROPID_changeleader:

			/* 未定義とします (404 Not Found) */
			return DAV_PROP_INSERT_NOTDEF;

		/* 上記以外のプロパティを受け取った場合 */
		default:
			/* 未定義とします (404 Not Found) */
			return DAV_PROP_INSERT_NOTDEF;
	}

	/* まだ採っていなければネームスペースサフィックスを取得 */
	if (IS_EMPTY(nsname)) {
		nsname = _build_liveprop_nsname(p, global_ns);
	}

	/*
	 * 動作の種類による処理分岐
	 */
	switch (what) {	
		case DAV_PROP_INSERT_NAME:	/* プロパティ名の追加 */
			s = apr_psprintf(p, "<%s:%s/>" CRLF, nsname, lspec->name);
			break;

		case DAV_PROP_INSERT_VALUE:	/* プロパティ名・値の追加 */
			if (IS_FILLED(value)) { 
				s = apr_psprintf(p, "<%s:%s>%s</%s:%s>" CRLF,
						nsname, lspec->name, value,
						nsname, lspec->name);
			}
			else {
				s = apr_psprintf(p, "<%s:%s/>" CRLF,
						nsname, lspec->name);
			}
			break;

		case DAV_PROP_INSERT_SUPPORTED:	/* 将来拡張用(DeltaV) */
			s = apr_psprintf(p, 
				"<"DAV_NS_PREFIX":supported-live-property "
				DAV_NS_PREFIX":name=\"%s\" "
				DAV_NS_PREFIX":namespace=\"%s\"/>" CRLF,
				lspec->name, nsname);
			break;

		default:
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Unsupported dav_prop_insert (type = %d)", what);
			return DAV_PROP_INSERT_NOTDEF;
	}

	/* 文字列を追加 */
	apr_text_append(p, phdr, s);
	return what;
}

/**
 * プロパティID DAV_DIVY_PROPID_resourcestate の内部エレメントを
 * 組み立てて文字列形式で返却する。
 * (note)
 * 	* 制限ユーザには、"published"エレメントを返しません(外部仕様)
 * 	* ユーザ拡張ステータスをサポートしていない状況ではこの関数はコールできません
 *
 * @param p apr_pool_t *
 * @param resource const dav_resource * このプロパティのリソース
 * @param nsname const char * ネームスペースプレフィックス
 */
static char * _insert_prop_resourcestate(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_resource *rdb_r)
{
	dav_buffer sbuf = { 0 };
	divy_rdbo_resourcestate *rstate;
	int is_trusty = divy_rdbo_is_trusty_user(divy_get_extstatus(r));
	tfs_hset_t *groupid_set = NULL;
	divy_rdbo_box *box_pr = NULL;
	char *datebuf = NULL;

	/* 可変長バッファの初期化 */
	dav_set_bufsize(p, &sbuf, 64);
	sbuf.cur_len = 0;

	/**
	 * 公開属性の組み立て
	 */
	if (rdb_r->rstate_pr != NULL) {
		for (rstate = rdb_r->rstate_pr; rstate; rstate = rstate->next) {
			/* 公開状態になっていた && 制限ユーザではない */
			if (IS_EFFECTIVE_STATE(rstate) &&
					rstate->type == DIVY_RSTATE_TYPE_VIEW && is_trusty) {
				dav_buffer_append(p, &sbuf, "<");
				dav_buffer_append(p, &sbuf, nsname);
				dav_buffer_append(p, &sbuf, ":published/>");
			}
			// BOX公開状態になっていた && 制限ユーザではない
			else if (IS_EFFECTIVE_STATE(rstate) && 
					rstate->type == DIVY_RSTATE_TYPE_BOX && is_trusty) {

				box_pr = rstate->optional.box_pr;

				if (box_pr == NULL) {
					dav_buffer_append(p, &sbuf, CRLF "<");
					dav_buffer_append(p, &sbuf, nsname);
					dav_buffer_append(p, &sbuf, ":box/>");
				}
				else {

					dav_buffer_append(p, &sbuf, CRLF "<");
					dav_buffer_append(p, &sbuf, nsname);
					dav_buffer_append(p, &sbuf, ":box>");

					if (box_pr->flag & BOX_FLAG_OPEN) {
						dav_buffer_append(p, &sbuf, "<");
						dav_buffer_append(p, &sbuf, nsname);
						dav_buffer_append(p, &sbuf, ":sealingwax/>" CRLF);
					}

					if (box_pr->flag & BOX_FLAG_PRIV) {
						dav_buffer_append(p, &sbuf, "<");
						dav_buffer_append(p, &sbuf, nsname);
						dav_buffer_append(p, &sbuf, ":privacy/>" CRLF);
					}

					if (box_pr->flag & BOX_FLAG_LOCK) {
						dav_buffer_append(p, &sbuf, "<");
						dav_buffer_append(p, &sbuf, nsname);
						dav_buffer_append(p, &sbuf, ":lockurl/>" CRLF);
					}

					if (box_pr->flag & BOX_FLAG_DELE) {
						dav_buffer_append(p, &sbuf, "<");
						dav_buffer_append(p, &sbuf, nsname);
						dav_buffer_append(p, &sbuf, ":autodelete/>" CRLF);
					}

					if (box_pr->flag & BOX_FLAG_OPEN && IS_FILLED(box_pr->shorten)) {
						/* BOX専用の短縮URLを作成する 
						 * 例：
						 * https://www.teamfile.com/teamfile/.st/txwsng?cmd=box
						 */
						dav_buffer_append(p, &sbuf,
							apr_psprintf(p, "<%s:linkurl>%s/."
											DIVY_URIPARAM_SHORTEN
											"/%s?"
											DIVY_URIPARAM_CMD
											"="
											DIVY_CMD_BOX
											"</%s:linkurl>"
											CRLF,
											nsname,
											divy_construct_url2(p,
												DIVY_URLTYPE_PUBLIC,
												dav_divy_get_root_uri(r),
												r, NULL),
											box_pr->shorten, nsname));
					}
					else {
						dav_buffer_append(p, &sbuf, "<");
						dav_buffer_append(p, &sbuf, nsname);
						dav_buffer_append(p, &sbuf, ":linkurl/>" CRLF);
					}

					if (IS_FILLED(box_pr->allowed_origin)) {
						dav_buffer_append(p, &sbuf,
							apr_psprintf(p, "<%s:allowedorigin><![CDATA[%s"
											"]]></%s:allowedorigin>" CRLF,
											nsname, box_pr->allowed_origin,
											nsname));
					}
					else {
						dav_buffer_append(p, &sbuf, "<");
						dav_buffer_append(p, &sbuf, nsname);
						dav_buffer_append(p, &sbuf, ":allowedorigin/>" CRLF);
					}

					if (IS_FILLED(box_pr->creator_usr_id)) {
						dav_buffer_append(p, &sbuf, 
							apr_psprintf(p, "<%s:creator>%s</%s:creator>" CRLF,
											nsname, box_pr->creator_usr_id, nsname));
					}
					else {
						/*
						 * 作成者は当然入っているが名前（IDではない）になる
						 * ために、空白名のユーザもいるかもしれない
						 */
						dav_buffer_append(p, &sbuf, "<");
						dav_buffer_append(p, &sbuf, nsname);
						dav_buffer_append(p, &sbuf, ":creator/>" CRLF);
					}

					if (box_pr->creationdate != 0) {
						divy_format_time_t(p, box_pr->creationdate,
											DIVY_TIME_STYLE_ISO8601, &datebuf);
						dav_buffer_append(p, &sbuf, 
							apr_psprintf(p, "<%s:publicationdate>%s"
											"</%s:publicationdate>" CRLF,
											nsname, datebuf,
											nsname));
					}
					else {
						dav_buffer_append(p, &sbuf, "<");
						dav_buffer_append(p, &sbuf, nsname);
						dav_buffer_append(p, &sbuf, ":publicationdate/>" CRLF);
					}

					if (IS_FILLED(box_pr->password)) {
						dav_buffer_append(p, &sbuf, 
							apr_psprintf(p, "<%s:password><![CDATA[%s"
											"]]></%s:password>" CRLF,
											nsname, box_pr->password,
											nsname));
					}
					else {
						dav_buffer_append(p, &sbuf, "<");
						dav_buffer_append(p, &sbuf, nsname);
						dav_buffer_append(p, &sbuf, ":password/>" CRLF);
					}

					if (box_pr->expirationdate != 0) {
						divy_format_time_t(p, box_pr->expirationdate,
											DIVY_TIME_STYLE_ISO8601, &datebuf);
						dav_buffer_append(p, &sbuf, 
							apr_psprintf(p, "<%s:expiration>%s"
											"</%s:expiration>" CRLF,
											nsname, datebuf,
											nsname));
					}
					else {
						dav_buffer_append(p, &sbuf, "<");
						dav_buffer_append(p, &sbuf, nsname);
						dav_buffer_append(p, &sbuf, ":expiration/>" CRLF);
					}

					if (IS_FILLED(box_pr->greeting)) {
						dav_buffer_append(p, &sbuf, 
							apr_psprintf(p, "<%s:greeting><![CDATA[%s"
											"]]></%s:greeting>" CRLF,
											nsname, box_pr->greeting,
											nsname));
					}

					if (IS_FILLED(box_pr->message)) {
						dav_buffer_append(p, &sbuf, 
							apr_psprintf(p, "<%s:message><![CDATA[%s"
											"]]></%s:message>" CRLF,
											nsname, box_pr->message,
											nsname));
					}

					/* 表示回数 */
					dav_buffer_append(p, &sbuf,
						apr_psprintf(p, "<%s:viewcount>%"APR_INT64_T_FMT
										"</%s:viewcount>" CRLF,
										nsname, box_pr->viewcount,
										nsname));

					/* 二要素認証（相手先メールアドレス) */
					if (divy_support_2FA(r)) {
						/*
						 * テーブルをALTERしたため、既存データは
						 * NULLの場合がある
						 */
						if (box_pr->tomailaddr == NULL) {
							box_pr->tomailaddr = "";
						}
						dav_buffer_append(p, &sbuf,
							apr_psprintf(p, "<%s:tomailaddr><![CDATA[%s"
											"]]></%s:tomailaddr>" CRLF,
											nsname, box_pr->tomailaddr,
											nsname));
					}


					dav_buffer_append(p, &sbuf, "</");
					dav_buffer_append(p, &sbuf, nsname);
					dav_buffer_append(p, &sbuf, ":box>" CRLF);


				}
			}
			else {
				/* サポートされていません */
			}
		}
	}

	/*
	 * セキュリティ保護されたフォルダかどうか
	 */
	if (rdb_r->grp_pr != NULL && IS_FILLED(rdb_r->grp_pr->grpid)) {

		/* セキュリティ保護されたグループ一覧を取り出す */
		groupid_set = divy_pcache_get_data(p, DIVY_PCACHE_DAT_REQ_SECUREDGRP);
		if (groupid_set == NULL) {
			divy_pi_cciphered_grpset(r, &groupid_set);
			if (groupid_set == NULL) {
				tfs_pool_t *pp = divy_get_pi_req_pool(r);

				/* 何度も繰り返さないようにダミーを挿入しておく */
				tfs_hset_create(pp, &groupid_set);
			}
			divy_pcache_set_data(p, groupid_set, DIVY_PCACHE_DAT_REQ_SECUREDGRP);
		}
		if (tfs_hset_contains(groupid_set, rdb_r->grp_pr->grpid)) {
			dav_buffer_append(p, &sbuf, "<");
			dav_buffer_append(p, &sbuf, nsname);
			dav_buffer_append(p, &sbuf, ":secured/>");
		}
	}

	/* 1つも追加されていなかった場合 */
	if (sbuf.cur_len == 0) {
		return "";	/* EMPTYにする */
	}

	/* バッファの文字列を返す */
	return sbuf.buf;
}

/**
 * プロパティID DAV_DIVY_PROPID_user_privilege_grant_set の内部エレメントを
 * 組み立てて文字列形式で返却する。
 * (note)
 * 	ユーザ拡張ステータスをサポートしていない状況では何もしません
 *
 * rdb_r 必須項目: rstate_pr(あれば), uri
 *
 * @param p apr_pool_t *
 * @param r request_rec *
 * @param nsname const char * ネームスペースプレフィックス
 * @param mode int 取得方法を現すビット地
 * 		DIVY_DEL_VIEW_PRIVILEGE : append-view, remove-viewを取らない
 * @param rdb_r divy_rdbo_resource *
 * @return char * 合成した文字列
 */
static char * _insert_prop_user_privilege_grant_set(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					int mode,
					divy_rdbo_resource *rdb_r)
{
	const divy_rdbo_extstatus *extstatus;
	int is_trusty;
	dav_buffer sbuf = { 0 };
	const char *root_uri  = dav_divy_get_root_uri(r);
	divy_infotype infotype, p_infotype;
	int is_write_constraints = 0;
	divy_rdbo_extstatus *grp_extstatus = NULL;
	int enable_box = 0;

	/* サポートしていなければ何もしない */
	if (!divy_support_extenduserstatus(r)) return "";

	if (rdb_r == NULL || IS_EMPTY(rdb_r->uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rdb_r or uri is EMPTY.");
		return "";
	}

	/* 可変長バッファの初期化 */
	dav_set_bufsize(p, &sbuf, 128);
	sbuf.cur_len = 0;

	/* アクセスユーザの権限を取得する */
	extstatus = divy_get_extstatus(r);

	/* 信頼されたユーザかどうか */
	is_trusty = divy_rdbo_is_trusty_user(extstatus);

	/* グループの属性を取得 */
	(void)divy_get_cached_availablegroupextstatus(r, rdb_r->uri, &grp_extstatus);

	/* BOX設定されているグループかどうか */
	enable_box = divy_rdbo_is_box_group(grp_extstatus);

	/* uriのパース */
	if (rdb_r->u_spec == NULL) {
		divy_parse_uri(p, root_uri, rdb_r->uri, &rdb_r->u_spec);
	}
	infotype   = rdb_r->u_spec->infotype;
	p_infotype = rdb_r->u_spec->p_infotype;

	/* グループ制約による書き込み制約の影響を受けるかどうか. */
	is_write_constraints = divy_rdbo_has_write_constraints_on_uri(r, rdb_r->uri, infotype);

	/* read / upload / readwrite は排他です.
	 * 複数の状態を持っていた場合、評価順位は以下。
	 * readwrite -> upload -> read */

	/* readwrite 権限がある / プライベートコレクション以下のリソース＆コレクション */
	if (divy_rdbo_has_readwrite_privilege(extstatus) ||
		infotype == DIVY_INFOTYPE_user_e         ||
		infotype == DIVY_INFOTYPE_user_trash     ||
		infotype == DIVY_INFOTYPE_user_trash_e0  ||
		infotype == DIVY_INFOTYPE_user_trash_ex  ||
		infotype == DIVY_INFOTYPE_user_e_regular) {

		dav_buffer_append(p, &sbuf, "<");
		dav_buffer_append(p, &sbuf, nsname);
		/* グループに書き込み制約があれば<read>にする */
		if (is_write_constraints) {
			dav_buffer_append(p, &sbuf, ":read/>");
		}
		else {
			dav_buffer_append(p, &sbuf, ":readwrite/>");
		}
	}
	/* readwrite権限を持たない &&
	 * 	* グループコレクション以下 (グループゴミ箱含む)
	 * 	* linkdbsearch結果フォルダ以下
	 * 	* 共有コレクション以下
	 */
	else if (infotype == DIVY_INFOTYPE_group_e            ||
		 infotype == DIVY_INFOTYPE_group_trash        ||
		 infotype == DIVY_INFOTYPE_group_trash_e0     ||
		 infotype == DIVY_INFOTYPE_group_trash_ex     ||
		 infotype == DIVY_INFOTYPE_group_e_regular    ||
		 infotype == DIVY_INFOTYPE_dbfolder_e         ||
		 infotype == DIVY_INFOTYPE_dbfolder_e_regular ||
		 infotype == DIVY_INFOTYPE_dbshfolder_e       ||
		 infotype == DIVY_INFOTYPE_dbshfolder_e_regular) {

		/* upload 権限はある? */
		if (divy_rdbo_has_upload_privilege(extstatus)) {
			dav_buffer_append(p, &sbuf, "<");
			dav_buffer_append(p, &sbuf, nsname);
			dav_buffer_append(p, &sbuf, ":upload/>");

			/* グループに書き込み制約があれば<read>をつける.
			 * (note)
			 *    但し、古いクライアントにこれを付けると動作不良を起こすので意図的に付けない */
			if (is_write_constraints && divy_get_clientinfo(r, DIVY_CLFUNC_GROUPCONSTRAINTS)) {
				dav_buffer_append(p, &sbuf, "<");
				dav_buffer_append(p, &sbuf, nsname);
				dav_buffer_append(p, &sbuf, ":read/>");
			}
		}
		/* read 権限はある? */
		else if (divy_rdbo_has_read_privilege(extstatus)) {
			dav_buffer_append(p, &sbuf, "<");
			dav_buffer_append(p, &sbuf, nsname);
			dav_buffer_append(p, &sbuf, ":read/>");
			/* (note) グループの書き込み制約があろうがなかろうがread はそのまま. */
		}
	}

	/* View属性の権限を判定する */
	if (!(mode & DIVY_DEL_VIEW_PRIVILEGE) && is_trusty && !enable_box &&
		infotype == DIVY_INFOTYPE_group_e_regular && divy_rdbo_has_setview_privilege(extstatus)) {

		int add_element = 0;

		/* View属性を付加できる条件
		 * -------------------------
		 *   * ユーザ拡張ステータス機能が有効 (View属性の有効性1)
		 *   * グループフォルダより下の通常コレクション/リソース (View属性の有効性2)
		 *   * box機能が有効になっているグループではない
		 *   ** グループゴミ箱は含まれません
		 *   * 制限ユーザではないこと
		 *   * View属性設定権限を持っている
		 *   * 既にView属性が付いて"いない" / "有効な"View属性が付いていないこと
		 * (note)
		 *   親の状態・属性プロパティが継承するかどうかなどの考慮は呼び出し側でやって下さい。
		 *   ここでは rstate_pr に継承された結果得られた情報も含まれているものとします。
		 */
		/* グループフォルダ直下のリソース＆コレクションの場合 */
		if (p_infotype == DIVY_INFOTYPE_group_e &&
		    infotype == DIVY_INFOTYPE_group_e_regular) {
			/* 公開属性を持っていない / 非公開だった */
			if (IS_INEFFECTIVE_STATE(rdb_r->rstate_pr)) {
				add_element = 1;
			}
		}
		/* グループフォルダ直下の下のリソース＆コレクション場合 */
		else if (p_infotype == DIVY_INFOTYPE_group_e_regular &&
			 infotype   == DIVY_INFOTYPE_group_e_regular) {
			/* 自分自身が非公開だった場合 */
			if (IS_INEFFECTIVE_STATE(rdb_r->rstate_pr) &&
					rdb_r->rstate_pr && !rdb_r->rstate_pr->inherit) {
				add_element = 1;
			}
		}

		if (add_element) {
			dav_buffer_append(p, &sbuf, "<");
			dav_buffer_append(p, &sbuf, nsname);
			dav_buffer_append(p, &sbuf, ":append-view/>");
		}

		/* View属性を除去できる条件
		 * -------------------------
		 *   * ユーザ拡張ステータス機能が有効 (View属性の有効性1)
		 *   * 制限ユーザではないこと
		 *   * グループコレクション以下の通常リソースであること (View属性の有効性2)
		 *   * View属性設定権限を持っている
		 *   * "有効な"View属性が付いて"いる"こと
		 */
		if (IS_EFFECTIVE_STATE(rdb_r->rstate_pr) &&
					rdb_r->rstate_pr->type == DIVY_RSTATE_TYPE_VIEW) {

			dav_buffer_append(p, &sbuf, "<");
			dav_buffer_append(p, &sbuf, nsname);
			dav_buffer_append(p, &sbuf, ":remove-view/>");
		}
	}

	/*
	 * メール監視状態を設定できるかどうか
	 *
	 * [ 付加される条件 ]
	 *   * ユーザ拡張ステータス機能が有効
	 *   * 制限ユーザではないこと
	 *   * BOX機能が有効なグループでないこと
	 *   * グループコレクション自身またはそれ以下の通常リソース(ゴミ箱含む) /
	 *     linkdbsearch 結果フォルダ以下 / 共有コレクション以下
	 */
	if (is_trusty && !enable_box &&
		(infotype == DIVY_INFOTYPE_group_e            ||
		 infotype == DIVY_INFOTYPE_group_trash        ||
		 infotype == DIVY_INFOTYPE_group_trash_e0     ||
		 infotype == DIVY_INFOTYPE_group_trash_ex     ||
		 infotype == DIVY_INFOTYPE_group_e_regular    ||
		 infotype == DIVY_INFOTYPE_dbfolder_e         ||
		 infotype == DIVY_INFOTYPE_dbfolder_e_regular ||
		 infotype == DIVY_INFOTYPE_dbshfolder_e       ||
		 infotype == DIVY_INFOTYPE_dbshfolder_e_regular)) {

		dav_buffer_append(p, &sbuf, "<");
		dav_buffer_append(p, &sbuf, nsname);
		dav_buffer_append(p, &sbuf, ":becomewatchowner/>");
	}

	/*
	 * 制約書き込み属性を入れるかどうか
	 *
	 * [ 付加される条件 ]
	 *   * ユーザ拡張ステータス機能が有効
	 *   * 制限ユーザであること
	 *   * グループフォルダ&linkdbsearch結果フォルダ&共有コレクション自身 or
	 *     上記フォルダの直下にあるリソース/コレクションであること
	 */
	if (!is_trusty) {
		int use_elem = 0;

		/* グループフォルダ自身、linkdbsearch結果フォルダ自身
		 * 共有コレクション自身、グループゴミ箱の場合 */
		if (infotype == DIVY_INFOTYPE_group_e            ||
		    infotype == DIVY_INFOTYPE_group_trash        ||
		    infotype == DIVY_INFOTYPE_group_trash_e0     ||
		    infotype == DIVY_INFOTYPE_group_trash_ex     ||
		    infotype == DIVY_INFOTYPE_dbfolder_e         ||
		    infotype == DIVY_INFOTYPE_dbshfolder_e) {
			use_elem = 1;
		}
		/* 上記フォルダ直下の場合 */
		else if ((infotype   == DIVY_INFOTYPE_group_e_regular &&
			  p_infotype == DIVY_INFOTYPE_group_e)    ||
			 (infotype   == DIVY_INFOTYPE_dbfolder_e_regular &&
			  p_infotype == DIVY_INFOTYPE_dbfolder_e) ||
			 (infotype   == DIVY_INFOTYPE_dbshfolder_e_regular &&
			  p_infotype == DIVY_INFOTYPE_dbshfolder_e)) {
			use_elem = 1;
		}

		if (use_elem) {
			dav_buffer_append(p, &sbuf, "<");
			dav_buffer_append(p, &sbuf, nsname);
			dav_buffer_append(p, &sbuf, ":limitedwrite/>");
		}
	}

	/* BOX属性を判定する */
	if (!(mode & DIVY_DEL_BOX_PRIVILEGE) && is_trusty &&
		infotype == DIVY_INFOTYPE_group_e_regular && 
		rdb_r->resourcetype == DIVY_TYPE_COLLECTION && enable_box) {

		int add_element = 0;

		/* BOX属性を不可できる条件
		 * -----------------------
		 *   * ユーザ拡張ステータスが有効
		 *   * 拡張ステータスにBOX機能が有効なグループであること
		 *   * グループフォルダ直下の下のコレクション
		 *   * ゴミ箱は含まれない
		 *   * 制限ユーザではないこと
		 */

		/* グループフォルダ直下のコレクションの場合 */
		if (p_infotype == DIVY_INFOTYPE_group_e &&
			infotype == DIVY_INFOTYPE_group_e_regular && 
			rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
			if (IS_INEFFECTIVE_STATE(rdb_r->rstate_pr)) {
				add_element = 1;
			}
		}

		if (add_element) {
			dav_buffer_append(p, &sbuf, "<");
			dav_buffer_append(p, &sbuf, nsname);
			dav_buffer_append(p, &sbuf, ":append-box/>");
		}

		if (IS_EFFECTIVE_STATE(rdb_r->rstate_pr) &&
					rdb_r->rstate_pr->type == DIVY_RSTATE_TYPE_BOX) {

			dav_buffer_append(p, &sbuf, "<");
			dav_buffer_append(p, &sbuf, nsname);
			dav_buffer_append(p, &sbuf, ":remove-box/>");
		}
	}

	/* 制約属性や権限が無かった場合 */
	if (sbuf.cur_len == 0) {
		return "";	/* EMPTY にする(not found ではない) */
	}
	else {
		/* バッファの文字列を渡す */
		return sbuf.buf;
	}
}

/**
 * プロパティID DAV_DIVY_PROPID_mailwatch の内部エレメントを
 * 組み立てて文字列形式で返却する。
 *
 * @param p apr_pool_t *
 * @param resource const dav_resource * このプロパティのリソース
 * @param nsname const char * ネームスペースプレフィックス
 */
static char * _insert_prop_mailwatch(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_resource *rdb_r)
{
	dav_buffer sbuf = { 0 };
	int i, first = 1;
	divy_rdbo_mailwatch *mwatch_pr = rdb_r->mwatch_pr;

	/* 可変長バッファの初期化 */
	dav_set_bufsize(p, &sbuf, 1024);
	sbuf.cur_len = 0;

	/*
	 * (note) DTD
	 *  <!ELEMENT mailwatch (watchowner?, triggermethod) >
	 *  <!ELEMENT watchowner (watchmethod) >
	 *  <!ELEMENT triggermethod (#PCDATA) >
	 */

	/* 操作者か? */
	if (mwatch_pr->owner_trigger_methos != NULL) {
		dav_buffer_append(p, &sbuf, "<");
		dav_buffer_append(p, &sbuf, nsname);
		dav_buffer_append(p, &sbuf, ":watchowner><");
		dav_buffer_append(p, &sbuf, nsname);
		dav_buffer_append(p, &sbuf, ":watchmethod>");

		first = 1;
		for (i = 0; i < METHODS; i++) {
			if (mwatch_pr->owner_trigger_methos[i]) {
				if (!first) {
					dav_buffer_append(p, &sbuf, " ");
				}
				first = 0;
				dav_buffer_append(p, &sbuf, ap_method_name_of(p, i));
			}
		}
		dav_buffer_append(p, &sbuf, "</");
		dav_buffer_append(p, &sbuf, nsname);
		dav_buffer_append(p, &sbuf, ":watchmethod></");
		dav_buffer_append(p, &sbuf, nsname);
		dav_buffer_append(p, &sbuf, ":watchowner>");
	}

	dav_buffer_append(p, &sbuf, "<");
	dav_buffer_append(p, &sbuf, nsname);
	dav_buffer_append(p, &sbuf, ":triggermethod>");

	/* トリガメソッドの取得 */
	first = 1;
	for (i = 0; i < METHODS; i++) {
		if (mwatch_pr->trigger_methods[i]) {
			if (!first) {
				dav_buffer_append(p, &sbuf, " ");
			}
			first = 0;

			dav_buffer_append(p, &sbuf, ap_method_name_of(p, i));
		}
	}
	dav_buffer_append(p, &sbuf, "</");
	dav_buffer_append(p, &sbuf, nsname);
	dav_buffer_append(p, &sbuf, ":triggermethod>");
#ifdef DIVY_SUPPORT_MESSENGER
	dav_buffer_append(p, &sbuf, "<");
	dav_buffer_append(p, &sbuf, nsname);
	dav_buffer_append(p, &sbuf, ":notification>");

	if (mwatch_pr->notification == DIVY_NOTICE_MSG) {
		dav_buffer_append(p, &sbuf, "<");
		dav_buffer_append(p, &sbuf, nsname);
		dav_buffer_append(p, &sbuf, ":msg/>");
	}
	else if (mwatch_pr->notification == DIVY_NOTICE_BOTH) {
		dav_buffer_append(p, &sbuf, "<");
		dav_buffer_append(p, &sbuf, nsname);
		dav_buffer_append(p, &sbuf, ":msg/><");
		dav_buffer_append(p, &sbuf, nsname);
		dav_buffer_append(p, &sbuf, ":mail/>");
	}
	else {
		dav_buffer_append(p, &sbuf, "<");
		dav_buffer_append(p, &sbuf, nsname);
		dav_buffer_append(p, &sbuf, ":mail/>");
	}
	dav_buffer_append(p, &sbuf, "</");
	dav_buffer_append(p, &sbuf, nsname);
	dav_buffer_append(p, &sbuf, ":notification>");
#endif	/* DIVY_SUPPORT_MESSENGER */

	return sbuf.buf;
}

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * プロパティID DAV_DIVY_PROPID_systemquota の内部エレメントを
 * 組み立てて文字列形式で返却する。
 *
 * @param p apr_pool_t *
 * @param resource const dav_resource * このプロパティのリソース
 * @param nsname const char * ネームスペースプレフィックス
 */
static char * _insert_prop_systemquota(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_resource *rdb_r)
{
	char *ignore_quota_str = "", *value;
	apr_uint64_t avail_bytes, total_bytes, avail_files, total_files;
	divy_rdbo_diskquota *sysquota_pr = rdb_r->sysquota_pr;

	if (sysquota_pr->sfs != NULL) {
		total_bytes = sysquota_pr->sfs->total_bytes;
		avail_bytes = sysquota_pr->sfs->avail_bytes;
		total_files = sysquota_pr->sfs->total_files;
		avail_files = sysquota_pr->sfs->avail_files;
	}
	else {
		total_bytes = APR_INT64_C(0);
		avail_bytes = APR_INT64_C(0);
		total_files = APR_INT64_C(0);
		avail_files = APR_INT64_C(0);
	}

	/* 物理DiskのCapacityよりも契約容量の方が大きいか? */
	if ((sysquota_pr->maxst != DIVY_UNLIMITED_SYSQUOTA_BYTES &&
	     sysquota_pr->maxst > total_bytes) || 
	    (sysquota_pr->maxres != DIVY_UNLIMITED_SYSQUOTA_FILES &&
	     sysquota_pr->maxres > total_files)) {
		ignore_quota_str = apr_psprintf(p,
			"<%s:ignore-assigned-quota/>"CRLF, nsname);
	}

	value = apr_psprintf(p, CRLF
			"<%s:used-bytes>%"APR_INT64_T_FMT"</%s:used-bytes>" CRLF
			"<%s:available-bytes>%"APR_INT64_T_FMT"</%s:available-bytes>" CRLF
			"<%s:assigned-bytes>%"APR_UINT64_T_FMT"</%s:assigned-bytes>" CRLF
			"<%s:free-bytes>%"APR_UINT64_T_FMT"</%s:free-bytes>" CRLF
			"<%s:used-files>%"APR_INT64_T_FMT"</%s:used-files>" CRLF
			"<%s:available-files>%"APR_INT64_T_FMT"</%s:available-files>" CRLF
			"<%s:assigned-files>%"APR_UINT64_T_FMT"</%s:assigned-files>" CRLF
			"<%s:free-files>%"APR_UINT64_T_FMT"</%s:free-files>" CRLF
			"%s",
			nsname, rdb_r->sysquota_pr->usedst, nsname,
			nsname, rdb_r->sysquota_pr->maxst, nsname,
			nsname, total_bytes, nsname,
			nsname, avail_bytes, nsname,
			nsname, rdb_r->sysquota_pr->usedres, nsname,
			nsname, rdb_r->sysquota_pr->maxres, nsname,
			nsname, total_files, nsname,
			nsname, avail_files, nsname, ignore_quota_str);

	return value;
}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

/**
 * プロパティDAV_DIVY_PROPID_thumbnail の内部エレメントを
 * 組み立てて文字列形式で返却する。
 *
 * @param p apr_pool_t *
 * @param resource const dav_resource * このプロパティのリソース
 * @param nsname const char * ネームスペースプレフィックス
 */
static char * _insert_prop_thumbnail(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_resource *rdb_r)
{
	divy_sbuf *sbuf = NULL;
	char *buf = NULL;
	apr_size_t nreadbyte = 0;
	int ret;
	const divy_rdbo_extstatus *extstatus;

	/* アクセスユーザの権限を取得する */
	extstatus = divy_get_extstatus(r);

	/* アップロード専用ユーザにはサムネイルを見せてはならない */
	if (divy_support_extenduserstatus(r) && divy_rdbo_has_upload_privilege(extstatus)) {
		return NULL;
	}
	/* (note) 実体暗号化グループであっても
	 * ファイルの混在が許可されているので、サムネイルを見せてもよいことになりました */

	/* サムネイルファイルからデータを取得 */
	ret = divy_thumbnail_read(p, r, rdb_r->physicalpath, &buf, &nreadbyte);
	if (ret == DIVY_THMNL_ST_NOTFOUND || ret == DIVY_THMNL_ST_NOTSUPPORTED) {
		/* サムネイルがないか、サポートされていない場合 */
		return NULL;
	}
	else if (ret != DIVY_THMNL_ST_OK) {
		return NULL;
	}

	/* 可変長バッファの初期化 */
	divy_sbuf_create(p, &sbuf, nreadbyte + 20);
	divy_sbuf_append(sbuf, "<![CDATA[");
	divy_sbuf_appendbyte(sbuf, nreadbyte, buf);
	divy_sbuf_append(sbuf, "]]>");

	return divy_sbuf_tostring(sbuf);
}


#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * プロパティDAV_DIVY_PROPID_passwordpolicy の内部エレメントを
 * 組み立てて文字列形式で返却する。
 *
 * @param p apr_pool_t *
 * @param resource const dav_resource * このプロパティのリソース
 * @param nsname const char * ネームスペースプレフィックス
 */
static char * _insert_prop_passwordpolicy(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_passwordpolicy *passpolicy_pr)
{
	divy_sbuf *sbuf = NULL;

	if (passpolicy_pr == NULL) {
		return NULL;
	}

	/* 可変長バッファの初期化 */
	divy_sbuf_create(p, &sbuf, 256);
	if (passpolicy_pr->status) {
		divy_sbuf_append(sbuf, CRLF"<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":policy-on/>"CRLF);
	}
	else {
		divy_sbuf_append(sbuf, CRLF"<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":policy-off/>"CRLF);
	}
	if (passpolicy_pr->minlen > 0) {
		divy_sbuf_append(sbuf,
			apr_psprintf(p, "<%s:minimum-password-length>%d"
						 "</%s:minimum-password-length>"CRLF,
						 nsname, passpolicy_pr->minlen, nsname));
	}
	if (passpolicy_pr->change_pw_cycle > 0) {
		divy_sbuf_append(sbuf,
			apr_psprintf(p, "<%s:change-password-cycle>%d"
						 "</%s:change-password-cycle>"CRLF,
						 nsname, passpolicy_pr->change_pw_cycle, nsname));

		/* 猶予期間は、パスワードの有効期限が設定されて初めて有効になる */
		divy_sbuf_append(sbuf,
			apr_psprintf(p, "<%s:password-probation>%d"
						 "</%s:password-probation>"CRLF,
						 nsname, passpolicy_pr->probation, nsname));
	}
	if (passpolicy_pr->history_pw_num >= 0) {
		divy_sbuf_append(sbuf,
			apr_psprintf(p, "<%s:change-history-number>%d"
						 "</%s:change-history-number>"CRLF,
						 nsname, passpolicy_pr->history_pw_num, nsname));
	}
	if (passpolicy_pr->ngword != NULL) {
		divy_cset_index_t *idx;
		const char *val;

		divy_sbuf_append(sbuf, "<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":ngwords><![CDATA[");
		for (idx = divy_cset_first(passpolicy_pr->ngword->ngword_set);
								idx != NULL; idx = divy_cset_next(idx)) {
			divy_cset_this(idx, &val);
			divy_sbuf_append(sbuf, val);
		}
		divy_sbuf_append(sbuf, "]]></");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":ngwords>"CRLF);
	}
	if (passpolicy_pr->denyUseridString) {
		divy_sbuf_append(sbuf, "<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":deny-userid-string/>"CRLF);
	}
	if (passpolicy_pr->isSymbolChrNeeded) {
		divy_sbuf_append(sbuf, "<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":must-include-symbol/>"CRLF);
	}
	if (passpolicy_pr->isUpAlphabet) {
		divy_sbuf_append(sbuf, "<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":must-include-upper-alphabet/>"CRLF);
	}
	if (passpolicy_pr->isLoAlphabet) {
		divy_sbuf_append(sbuf, "<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":must-include-lower-alphabet/>"CRLF);
	}
	if (passpolicy_pr->isNumericNeeded) {
		divy_sbuf_append(sbuf, "<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":must-include-numeric/>"CRLF);
	}
	if (passpolicy_pr->denyCycleChr) {
		divy_sbuf_append(sbuf, "<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":deny-cycle-character/>"CRLF);
	}
	/* (note) -1 であれば返却しない */
	if (passpolicy_pr->firstlogin_limitday >= 0) {
		divy_sbuf_append(sbuf,
			apr_psprintf(p, "<%s:firstlogin-limitday>%d"
						 "</%s:firstlogin-limitday>"CRLF,
						 nsname, passpolicy_pr->firstlogin_limitday, nsname));
	}

	return divy_sbuf_tostring(sbuf);
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * プロパティDAV_DIVY_PROPID_groupstate の内部エレメントを
 * 組み立てて文字列形式で返却する。
 *
 * @param p apr_pool_t *
 * @param resource const dav_resource * このプロパティのリソース
 * @param nsname const char * ネームスペースプレフィックス
 */
static char * _insert_prop_groupstate(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_grp *grp_pr)
{
	divy_sbuf *sbuf = NULL;

	if (grp_pr == NULL) {
		return NULL;
	}

	/* 可変長バッファの初期化 */
	divy_sbuf_create(p, &sbuf, 256);
	if (IS_FILLED(grp_pr->ownername)) {
		divy_sbuf_append(sbuf, apr_psprintf(p,
						"<%s:ownername>%s"
						"</%s:ownername>"CRLF,
						nsname, grp_pr->ownername, nsname));
	}
	else {
		divy_sbuf_append(sbuf, "<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":ownername/>"CRLF);
	}

	/* BOXをサポートしているなら属性を調べて追加 */
	if (divy_support_tfbox(r)) {
		/* BOX設定されているグループかどうか */
		if (divy_rdbo_is_box_group(grp_pr->grp_extstatus)) {
			divy_sbuf_append(sbuf, "<");
			divy_sbuf_append(sbuf, nsname);
			divy_sbuf_append(sbuf, ":box/>"CRLF);
		}
	}

	/* アップロードポリシーをサポートしているなら属性を調べて追加 */
	if (divy_support_upload_policy(r)) {
		if (divy_rdbo_is_uploadpolicy_group(grp_pr->grp_extstatus)) {
			divy_sbuf_append(sbuf, "<");
			divy_sbuf_append(sbuf, nsname);
			divy_sbuf_append(sbuf, ":uploadpolicy/>"CRLF);
		}
	}

	return divy_sbuf_tostring(sbuf);
}

/**
 * プロパティDAV_DIVY_PROPID_confirmreading の内部エレメントを
 * 組み立てて文字列形式で返却する。
 *
 * @param p apr_pool_t *
 * @param resource const dav_resource * このプロパティのリソース
 * @param nsname const char * ネームスペースプレフィックス
 * @param rdb_r divy_rdbo_resource *
 */
static char * _insert_prop_confirmreading(apr_pool_t *p,
					request_rec *r,
					const char *nsname,
					divy_rdbo_resource *rdb_r)
{
	divy_sbuf *sbuf = NULL;
	divy_rdbo_confirmreading *cr_pr;
	int isOwn = 0, isOther = 0;
	const char *own_userid = divy_get_userid(r);

	/* (note) 開封通知機能をサポートしていることが前提 */

	if (rdb_r == NULL || rdb_r->confirmreading_pr == NULL) {
		return NULL;
	}

	for (cr_pr = rdb_r->confirmreading_pr; cr_pr; cr_pr = cr_pr->next) {
		if (IS_FILLED(cr_pr->userid)) {
			if (strcmp(own_userid, cr_pr->userid) == 0) {
				isOwn = 1;
			}
			else {
				isOther = 1;
			}
		}
	}

	divy_sbuf_create(p, &sbuf, 32);
	if (isOwn) {
		divy_sbuf_append(sbuf, "<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":own/>");
	}

	if (isOther) {
		divy_sbuf_append(sbuf, "<");
		divy_sbuf_append(sbuf, nsname);
		divy_sbuf_append(sbuf, ":other/>");
	}

	return divy_sbuf_tostring(sbuf);
}

/**
 * プロパティDAV_DIVY_PROPID_shorten の内部エレメントを
 * 組み立てて文字列形式で返却する。
 *
 * @param p apr_pool_t *
 * @param resource const dav_resource * このプロパティのリソース
 * @param nsname const char * ネームスペースプレフィックス
 * @param rdb_r divy_rdbo_resource *
 */
static char * _insert_prop_shorten(apr_pool_t *p,
				request_rec *r, 
				const char* nsname, 
				divy_rdbo_resource *rdb_r) 
{

	if (IS_EMPTY(rdb_r->rsid)) {
		return NULL;
	}

	return divy_get_rid2shorten(p, rdb_r->rsid, NULL);

}

/**
 * global_ns からネームスペースプレフィックス文字列を生成して返却する。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param global_ns int グローバルネームスペースID
 * @return char * 生成したネームスペースプレフィックス文字列
 */
static char * _build_liveprop_nsname(apr_pool_t *p, int global_ns)
{
	return apr_psprintf(p, "lp%d", global_ns);
}

/**
 * resource が示すリソースにmwatch_pr のメール監視プロパティをset する.
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _set_mailwatch(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_mailwatch *mwatch_pr)
{
	apr_pool_t *p           = r->pool;
	divy_rdbo_mailwatch *mw = NULL;

	divy_infotype infotype = resource->info->rdb_r->u_spec->infotype;
	divy_rdbo_rusr *rusr    = NULL;
	divy_rdbo_grp  *grp_pr  = NULL;

	if (infotype == DIVY_INFOTYPE_m_group_ruser) {
		if (divy_rdbo_parse_rusr_uri(r, resource->uri, &rusr)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to parse resource uri of "
				"user-relation resource. (uri = %s)", resource->uri);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		mwatch_pr->usrid   = (char *) rusr->usrid;
		if (divy_rdbo_get_hierarchy_group_property(r, rusr->grp_uri,
					divy_count_dirs(rusr->grp_uri), &grp_pr, rbk_ctx->ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get hierarchy group information."
			"(relativeuri = %s)", rusr->grp_uri);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		mwatch_pr->uri     = (char *)grp_pr->grpcol_uri;
	}
	else {
		/* uri、usridを設定する */
		mwatch_pr->uri   = (char *) resource->uri;
		mwatch_pr->usrid = (char *) divy_get_userid(r);
	}

	/* 
	 * mailwatchを取得する
	 */
	if (divy_rdbo_get_mailwatch_property(r, mwatch_pr->uri,
						mwatch_pr->usrid,
						&mw, rbk_ctx->ts_ctx)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get mailwatch property for set. "
			"(uri = %s, userid = %s)",
			mwatch_pr->uri, mwatch_pr->usrid);

		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* 初登録の場合 */
	if (mw == NULL) {
		if (divy_rdbo_insert_mailwatch_property(r, mwatch_pr,
							rbk_ctx->ts_ctx)) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to insert mailwatch property."
				"(uri = %s, userid = %s)",
				mwatch_pr->uri, mwatch_pr->usrid);

			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}
	/* 登録済みの場合 */
	else {
		if (divy_rdbo_update_mailwatch_property(r, mwatch_pr,
							rbk_ctx->ts_ctx)) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to update mailwatch property."
				"(uri = %s, userid = %s)",
				mwatch_pr->uri, mwatch_pr->usrid);

			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * resource が示すリソースからmwatch_pr のメール監視プロパティをremove する.
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _remove_mailwatch(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_mailwatch *mwatch_pr)
{
	apr_pool_t *p           = r->pool;
	divy_rdbo_mailwatch *mw = NULL;

	divy_infotype infotype = resource->info->rdb_r->u_spec->infotype;
	divy_rdbo_rusr *rusr    = NULL;
	divy_rdbo_grp  *grp_pr  = NULL;

	if (infotype == DIVY_INFOTYPE_m_group_ruser) {
		if (divy_rdbo_parse_rusr_uri(r, resource->uri, &rusr)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to parse resource uri of "
				"user-relation resource. (uri = %s)", resource->uri);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		mwatch_pr->usrid   = (char *) rusr->usrid;
		if (divy_rdbo_get_hierarchy_group_property(r, rusr->grp_uri,
					divy_count_dirs(rusr->grp_uri), &grp_pr, rbk_ctx->ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get hierarchy group information."
			"(relativeuri = %s)", rusr->grp_uri);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		mwatch_pr->uri     = (char *)grp_pr->grpcol_uri;
	}
	else {
		/* uri、usridを設定する */
		mwatch_pr->uri   = (char *) resource->uri;
		mwatch_pr->usrid = (char *) divy_get_userid(r);
	}
	
	/* 
	 * mailwatchを取得する
	 */
	if (divy_rdbo_get_mailwatch_property(r, mwatch_pr->uri,
						mwatch_pr->usrid,
						&mw, rbk_ctx->ts_ctx)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get mailwatch property for remove. "
			"(uri = %s, userid = %s)",
			mwatch_pr->uri, mwatch_pr->usrid);

		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* 誰かが消してしまった */
	if (mw == NULL) return NULL;

	/* 論理削除フラグの設定 */
	if (mw->trigger_methods[M_PROPPATCH]) {
		mw->delflag = 1;	/* 論理削除する */
	}
	else {
		mw->delflag = 0;	/* 物理削除する */
	}

	/*
	 * 削除の実施
	 */
	if (divy_rdbo_remove_mailwatch_property(r, mw, rbk_ctx->ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to remove mailwatch property."
			"(uri = %s)", resource->uri);

		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * resourceが示すリソースに共有コレクションプロパティをsetする
 */
static dav_error * _set_sharedcollection(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_shlink *shlink_pr)
{
	apr_pool_t *p  = r->pool;
	int ret        = 0;

	/* sharedcollectionはmkcolで作成済みのため常にupdate */
	ret = divy_rdbo_update_sharedcollection_property(r,
				resource->uri, shlink_pr, rbk_ctx->ts_ctx);

	/* プロパティとして指定された共有コレクションが存在しなかった場合 */
	if (ret == DIVY_STCODE_SHCOL_NOTFOUND) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"The specified sharedcollection did not exist."
			"(uri = %s)", resource->uri);
		return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}
	else if (ret) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to update sharedcollection property."
			"(uri = %s)", resource->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * resourceが示すリソースからshlink_pr の共有コレクションプロパティをremoveする
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _remove_sharedcollection(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_shlink *shlink_pr)
{
	/*
	 * 共有コレクションプロパティの削除
	 */
	if (divy_rdbo_update_sharedcollection_property(r,
					resource->uri, NULL, rbk_ctx->ts_ctx)) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to remove sharedcollection property."
			"(uri = %s)", resource->uri);

		return dav_new_error(r->pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * resource が示すリソースにusr_pr のプロパティをset する.
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _set_user(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_user_iscreen *user_iscreen)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_usr *usr_pr = NULL;
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
	int support_accesscontrol  = divy_support_access_control(r);
	int support_session        = divy_support_session(r);

	/* iscreen -> rdbo */
	divy_map_user_property(p, user_iscreen, &usr_pr);

	/* useridをuriから切り出す */
	usr_pr->usrid = (char *) user_iscreen->u_spec->final_path_segment;
	usr_pr->rsid  = user_iscreen->src_usr_pr->rsid;

	/* 現DBに格納されたパスワードと入力パスワードが異なる場合 */
	if (strcmp(usr_pr->password, user_iscreen->src_usr_pr->password) != 0) {
		dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

		/* パスワードがDBの値と違っていて、暗号化パスワードが有効な場合
		 * (note)
		 *   暗号化パスワードが有効であっても、入力されたパスワードを
		 *   無条件に暗号化してはなりません。ユーザがパスワードを変更した
		 *   場合のみ実施すべきです。本来であれば”変更した”という事実を
		 *   教えてもらう必要があるのですが、現機構では不可能であるため、
		 *   以下の仮定を行って現行パスワードとの比較だけで済ましています。
		 *
		 *   * エンドユーザが入力できるパスワードの文字数は、暗号化
		 *     パスワードで生成されるパスワードよりも短い
		 *     
		 *     -> エンドユーザが暗号化パスワードと同じ文字列を送信してきたと
		 *        したら、それはパスワードを変更しないままユーザプロパティの
		 *        変更を要求したとみなせるはず。
		 *
		 *   この仮定が成り立つためには、パスワード入力欄が十分短い必要があります。
		 *   注意して下さい。
		 */
		if (dconf->encryptpassword == DIVY_ENCRYPTPASSWORD_ON) {
			if (support_session) {
				const char* sid = divy_pcache_get_data(r->pool, DIVY_PCACHE_DAT_SES_SID);
				divy_rdbo_update_session_password(r, NULL, sid, r->user, usr_pr->password);
			}
			usr_pr->password = ap_md5(p, (const unsigned char*)usr_pr->password);
		}

#ifdef DIVY_SUPPORT_PASSPOLICY
		/* パスワードポリシーをサポートしていた場合
		 * (note)
		 *   パスワードポリシーの有効/無効に関わらず、最終更新日付や
		 *   初回ログイン日付は更新します (外部仕様)
		 */
		if (divy_support_passpolicy(r)) {
			time_t now, updatedt;
	
			now = dav_divy_get_now_epoch();
			updatedt = dav_divy_iso8601totime_t(p, usr_pr->updatedt);
			usr_pr->passpolicy_status = user_iscreen->src_usr_pr->passpolicy_status;

			/* ユーザのパスワードポリシー状態が存在しなかった場合
			 * (途中でポリシーをON/OFFした場合) */
			if (usr_pr->passpolicy_status == NULL) {

				/* デフォルト値でユーザのパスワードポリシー状態を作成する */
				usr_pr->passpolicy_status = divy_create_user_passwordpolicy(p, usr_pr->usrid);

				/* 少なくとも1 回login していたか? */
				if (usr_pr->lastaccess > 0L && usr_pr->lastaccess >= updatedt) {
					/* 最終アクセス日時を初回ログイン日付とする */
			 		usr_pr->passpolicy_status->firstlogin = usr_pr->lastaccess;
				}
				else {
					/* まだ1度もログインしていない */
			 		usr_pr->passpolicy_status->firstlogin = 0L;
				}
			}
			else {
				/* (互換性問題)
				 * 自分自身のパスワードを変更する場合には、
				 * 初回ログインステータスを変更しなければならない.  */
				if (strcmp(usr_pr->usrid, divy_get_userid(r)) == 0) {
					/* firstlogin が済んでいない場合 */
					if (usr_pr->passpolicy_status->firstlogin == 0L) {
						usr_pr->passpolicy_status->firstlogin = now; /* 時刻をつける */
					}
				}
				/*
				 * ユーザIDが異なる場合は管理者による更新となるため、
				 * firstloginをリセットしてパスワード変更を促す */
				else {
					usr_pr->passpolicy_status->firstlogin = 0L; /* リセット */
				}
			}

			/* 最終期限切れメール送信日付 をクリアーする */
			usr_pr->passpolicy_status->sendexpiredmail = 0L;

			/* 完全期限切れ基点日付をクリアする */
			usr_pr->passpolicy_status->specialstart = 0L;

			/* パスワード更新日付を更新する */
			usr_pr->passpolicy_status->lastchange = now;
		}
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	}

	/* 古いクライアントであれば新しい値を使わない */
	if (user_iscreen->is_old_client) {
		usr_pr->expiration = user_iscreen->src_usr_pr->expiration;
		usr_pr->extstatus  = user_iscreen->src_usr_pr->extstatus;
	}

	/*
	 * 昔のユーザ制約無視属性をそのまま利用するかどうか. (プロパティ消失防止. WebDAV違反かも..)
	 * [ 条件 ]
	 * 	* グループ制約機能 on  && 旧クライアント (理解できないから正しい制約を送ってこないため)
	 * 	* グループ制約機能 off && 新旧クライアント
	 * 		(旧クライアントは理解できないし、新クライアントはoffを検出して判らないフリをするので)
	 */ 
	if ((support_grpconstraints && !divy_get_clientinfo(r, DIVY_CLFUNC_GROUPCONSTRAINTS)) ||
		!support_grpconstraints) {

		int onoff = divy_rdbo_has_user_groupconstraints_ignore(user_iscreen->src_usr_pr->extstatus);
		divy_rdbo_set_user_groupconstraints_ignore(usr_pr->extstatus, onoff);
	}

	/* (2007/06/21 Thu) 以下のコードは廃止.
	 * 理由は、ディレクティブをon -> off -> ユーザ情報を変更 -> on にすると
	 * ユーザが持っていたグループ制約を無視するかどうかがクリアされてしまいます.
	 * 恐らくこれは意図した動作ではありません. off になっただけで、サーバ内部では
	 * うまく機能が動作しないようになりますので、敢えてユーザプロパティを更新する必要もありません. */
#if 0
	/* グループ制約属性をサポートしていなければ、制約属性の無視フラグは「無視しない」にする(デフォルト) */
	if (!support_grpconstraints) {
		divy_rdbo_set_user_groupconstraints_ignore(usr_pr->extstatus, 0);
	}
#endif
	/*
	 * 古いクライアントからのリクエストの場合 (グループ管理者機能)
	 */
	if (support_groupleader && !divy_get_clientinfo(r, DIVY_CLFUNC_GROUPLEADER)) {
		/* maxusercreation, Other ユーザを引き込めるかどうかを昔の値にする */
		usr_pr->maxusercreation = user_iscreen->src_usr_pr->maxusercreation;
		divy_rdbo_set_control_otheruser(usr_pr->extstatus,
					divy_rdbo_has_control_otheruser(user_iscreen->src_usr_pr->extstatus));
	}

	/*
	 * 管理者が対象ユーザを"管理者"または"グループリーダ"に変更した時
	 * 対象ユーザはオーナーから独立した存在になる (外部仕様)
	 * つまり、オーナーが存在しない状態になります
	 */
	if (support_groupleader) {
		/* 管理者ユーザ && 対象を"管理者" or "グループリーダ" に変更 */
		if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN &&
			(divy_rdbo_is_groupleader(usr_pr->extstatus) || usr_pr->adminmode == DIVY_ADMINMODE_ADMIN)) {
			/* ownerid をクリアする */
			usr_pr->ownerid = NULL;
		}
		else {
			/* ownerid には昔の値を使う */
			usr_pr->ownerid = user_iscreen->src_usr_pr->ownerid;
		}
	}

	/*
	 * 対象ユーザがグループリーダ以外の場合、"管理ユーザ数"や"ユーザを管理下に置くことができる"
	 * と設定されていても無意味なので、デフォルト状態に戻します.
	 */
	if (support_groupleader && !divy_rdbo_is_groupleader(usr_pr->extstatus)) {
		usr_pr->maxusercreation = 0;	/* 無制限 */
		divy_rdbo_set_control_otheruser(usr_pr->extstatus, 0);	/* 制御できない */
	}

	/*
	 * 今が制限ユーザ以外であり、新しく制限ユーザになる場合に限り
	 * そのユーザが持っていた開封通知を全て削除する
	 */
	if (divy_support_extenduserstatus(r) && divy_support_confirmreading(r) &&
		(divy_rdbo_is_trusty_user(user_iscreen->src_usr_pr->extstatus) &&
		 !divy_rdbo_is_trusty_user(usr_pr->extstatus))) {

		if (divy_rdbo_remove_confirmreading(r, NULL/*URI無視*/, usr_pr->usrid, 0, rbk_ctx->ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove confirmreading property of limited user."
					"(userid = %s)", usr_pr->usrid);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	/*
	 * 古いクライアントからのリクエストの場合 (アクセス制御機能)
	 */
	if (support_accesscontrol && !divy_get_clientinfo(r, DIVY_CLFUNC_ACCESSCONTROL)) {
		/* 昔の値をそのまま利用する */
		usr_pr->allowhosts = user_iscreen->src_usr_pr->allowhosts;
	}

	/* userdiscoveryはmkcolで作成済みのため常にupdate */
	if (divy_rdbo_update_user_property(r, usr_pr, rbk_ctx->ts_ctx)){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to update userdiscovery property. (uri = %s)", resource->uri);

		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* 更新されたメールを送信する */
	if (divy_ml_enable_notify_usercreation(r, usr_pr)) {
		/* グループリーダが更新者であれば、リーダを入れておく */
		if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus)) {
			usr_pr->ownerid = apr_pstrdup(p, divy_get_userid(r));
			usr_pr->ownername = apr_pstrdup(p, divy_get_fullname(r));
		}
		divy_ml_notify_update_user(r, usr_pr);
	}

	return NULL;
}

/**
 * resource が示すリソースにgrp_pr のプロパティをset する.
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _set_group(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_group_iscreen *group_iscreen)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_grp *src_grp_pr = resource->info->rdb_r->grp_pr;
	divy_rdbo_grp *grp_pr     = NULL;

	/* iscreen -> rdbo */
	divy_map_group_property(p, group_iscreen, &grp_pr);

	/* src_grp_pr から昔の値を取り出す */
	grp_pr->grpid       = src_grp_pr->grpid;
	grp_pr->name        = src_grp_pr->name;
	grp_pr->rsid        = src_grp_pr->rsid;
	grp_pr->relativeuri = src_grp_pr->relativeuri;
	grp_pr->grpcol_uri  = src_grp_pr->grpcol_uri;

	/* mailwatchがある場合はuri、usridを設定する */
	if (grp_pr->mailwatch_pr != NULL) {
		grp_pr->mailwatch_pr->uri   = (char *) resource->uri;
		grp_pr->mailwatch_pr->usrid = (char *)divy_get_userid(r);
	}

	/* サーバがグループ制約機能をサポートしていて、
	 * グループ制約を理解しないクライアントからのリクエストであれば
	 * 制約属性は古い値をそのまま使用します (プロパティの消失防止. 本来WebDAV違反...) */
	if (divy_support_grpconstraints(r) && !divy_get_clientinfo(r, DIVY_CLFUNC_GROUPCONSTRAINTS)) {
		grp_pr->grp_extstatus = src_grp_pr->grp_extstatus;
	}

	/* サーバがグループ管理者機能をサポートしていて、
	 * グループ管理者機能を理解しないクライアントからのリクエストであれば
	 * アクティブは古い値をそのまま適用します */
	if (divy_support_groupleader(r) && !divy_get_clientinfo(r, DIVY_CLFUNC_GROUPLEADER)) {
		if (grp_pr->grp_extstatus != NULL) {
			divy_rdbo_set_active_group(grp_pr->grp_extstatus,
					divy_rdbo_is_active_group(src_grp_pr->grp_extstatus));
		}
		else {
			grp_pr->grp_extstatus = src_grp_pr->grp_extstatus;
		}
	}

	/* サーバがBOX機能をサポートしていて
	 * BOX機能を理解しないクライアントからのリクエストであれば
	 * BOX機能は古い値をそのまま利用します
	 */
	if (divy_support_tfbox(r) && !divy_get_clientinfo(r, DIVY_CLFUNC_BOX)) {
		divy_rdbo_set_box_group(grp_pr->grp_extstatus,
							divy_rdbo_is_box_group(src_grp_pr->grp_extstatus));
	}

	/* サーバがアップロードポリシーをサポートしていて
	 * アップロードポリシー機能を理解しないクライアントからのリクエストは
	 * アップロードポリシー機能は古い値をそのまま利用します
	 */
	if (divy_support_upload_policy(r)
						&& !divy_get_clientinfo(r, DIVY_CLFUNC_UPLOADPOLICY)) {
		divy_rdbo_set_uploadpolicy_group(grp_pr->grp_extstatus,
					divy_rdbo_is_uploadpolicy_group(src_grp_pr->grp_extstatus));
		grp_pr->policy = src_grp_pr->policy;
	}

	/* groupdiscoveryはmkcolで作成済みのため常にupdate */
	if (divy_rdbo_update_group_property(r, grp_pr, rbk_ctx->ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert groupdiscovery property."
			"(uri = %s)", resource->uri);

		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * resource が示すリソースにsql_pr のプロパティをset する.
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _set_sql(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_sql *sql_pr)
{
	apr_pool_t *p = r->pool;

	/* 相対URI, SQLシーケンス番号を取り出す */
	sql_pr->relativeuri = (char *) resource->info->rdb_r->u_spec->other_part;
	sql_pr->sqlid       = resource->info->rdb_r->sql_pr->sqlid;
	sql_pr->next        = resource->info->rdb_r->sql_pr;

	/* sqldiscoveryはmkcolで作成済みのため常にupdate */
	if (divy_rdbo_update_sql_property(r, sql_pr, rbk_ctx->ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to update sqldiscovery property."
			"(uri = %s)", resource->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * resource が示すリソースにclmodule_pr のプロパティをset する.
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _set_clmodule(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_rdbo_clmodule *clmodule_pr)
{
	apr_pool_t *p = r->pool;

	/* uriを設定する */
	clmodule_pr->uri = (char *) resource->uri;

	/* リソースがクライアントモジュールプロパティを持っていなければ新規登録 */	
	if (resource->info->list.clmodule_pr == NULL){
		if (divy_rdbo_insert_clmodule_property(r, clmodule_pr,
							rbk_ctx->ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to insert updatediscovery property."
				"(uri = %s)", resource->uri);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}
	/* 更新 */
	else {
		if (divy_rdbo_update_clmodule_property(r, clmodule_pr,
							rbk_ctx->ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to update updatediscovery property."
				"(uri = %s)", resource->uri);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * resource が示すリソースにsysmsg_pr のプロパティをset する.
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _set_sysmsg(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_sysmsg_iscreen *sysmsg_iscreen)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_sysmsg *sysmsg_pr = NULL;

	/* iscreen -> rdbo */
	divy_map_sysmsg_property(p, sysmsg_iscreen, &sysmsg_pr);

	/* msgidをuriから切り出す */
	sysmsg_pr->msgid = (char *) resource->info->rdb_r->u_spec->final_path_segment;
			
	/* sysmsgdiscoveryはmkcolで作成済みのため常にupdate */
	if (divy_rdbo_update_sysmsg_property(r, sysmsg_pr, rbk_ctx->ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to update sysmsgdiscovery property."
			"(uri = %s)", resource->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * resource が示すリソースにdbms_pr のプロパティをset する.
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _set_dbms(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_dbms_iscreen *dbms_iscreen)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_dbms *dbms_pr = NULL;

	/* iscreen -> rdbo */
	divy_map_dbms_property(p, dbms_iscreen, &dbms_pr);

	/* dbmsidを取り出す */
	dbms_pr->dbmsid = (char *) resource->info->rdb_r->dbms_pr->dbmsid;

	/* dbmsdiscoveryはmkcolで作成済みのため常にupdate */
	if (divy_rdbo_update_dbms_property(r, dbms_pr, rbk_ctx->ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to update dbmsdiscovery property."
			"(uri = %s)", resource->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * resource が示すリソースにsysquota_pr のプロパティをset する.
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _set_systemquota(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_sysquota_iscreen *sysquota_iscreen)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_diskquota *sysquota_pr = NULL;

	/* iscreen -> rdbo */
	divy_map_systemquota_property(p, sysquota_iscreen, &sysquota_pr);

	/* systemquota を登録 / 更新する */
	if (divy_rdbo_update_systemquota_property(r, sysquota_pr, rbk_ctx->ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to update systemquota property.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */


/**
 * resource が示すリソースにrstate_iscreen のプロパティをset する。
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _set_resourcestate(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_resourcestate_iscreen *rstate_iscreen)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_resourcestate *rstate_pr = NULL;

	/* iscreen -> rdbo */
	divy_map_resourcestate_property(p, rstate_iscreen, &rstate_pr);

	/* resourcestate を登録, 更新, 削除する */
	if (divy_rdbo_update_resourcestate_property(r, rstate_pr, rbk_ctx->ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to update resourcestate property.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * resource が示すリソースにthumbnail_iscreen のプロパティをset する。
 *
 * @return dav_error * DAVエラー(500), 成功したらNULL
 */
static dav_error * _set_thumbnail(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_thumbnail_iscreen *thumbnail_iscreen)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_resource *rdb_r = resource->info->rdb_r;
	int ret;

	/* サムネイルデータを書き込む */
	ret = divy_thumbnail_write(p, r, rdb_r->physicalpath,
								thumbnail_iscreen->data,
								strlen(thumbnail_iscreen->data));
	/* サムネイル機能はサポートされていて、エラーが出た場合 */
	if (ret != DIVY_THMNL_ST_OK && ret != DIVY_THMNL_ST_NOTSUPPORTED) {
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * thumbnail プロパティを削除する
 */
static dav_error * _remove_thumbnail(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_thumbnail_iscreen *thumbnail_iscreen)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_resource *rdb_r = resource->info->rdb_r;
	int ret;

	/* サムネイルファイルを削除(サポートの有無は関係なし) */
	ret = divy_thumbnail_remove(p, r, rdb_r->physicalpath);
	if (ret != DIVY_THMNL_ST_OK && ret != DIVY_THMNL_ST_NOTSUPPORTED) {
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passwordpolicy プロパティを追加/更新する.
 * (note)
 *   この関数は、パスワードポリシー機能が「サポート」されているケースでのみ
 *   コールできます. 呼び出し元で考慮してください.
 */
static dav_error * _set_passwordpolicy(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_passwordpolicy_iscreen *passpolicy_iscreen)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_passwordpolicy *passpolicy_pr = NULL, *old = NULL;

	/* iscreen -> rdbo */
	divy_map_passwordpolicy_property(p, passpolicy_iscreen, &passpolicy_pr);

	/* 古いpasswordpolicy プロパティを取得する */
	if (divy_get_cached_passwordpolicy(r, DIVY_DEFAULT_POLICY_ID, 0, &old)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get passwordpolicy property.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* パスワードポリシー開始日付を設定する.
	 *
	 * [ 条件 ] (どれか1つ)
	 *   * 新規でパスワードポリシーを登録する
	 *   * 既存パスワードポリシーのステータスを「無効」から「有効」にする
	 */
	if ((old == NULL && passpolicy_pr->status) ||
		(old != NULL && !old->status && passpolicy_pr->status)) {
		/* 今の日付をスタート日とする */
		passpolicy_pr->startdt = dav_divy_get_now_epoch();
	}
	else {
		/* 昔の設定値あり */
		if (old != NULL) {
			/* 昔の値をそのまま使う */
			passpolicy_pr->startdt = old->startdt;
		}
		else {
			passpolicy_pr->startdt = 0L;	/* 初期値 */
		}
	}

	/* passwordpolicy を登録/更新する */
	if (divy_rdbo_update_passwordpolicy_property(r, passpolicy_pr, rbk_ctx->ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to update passwordpolicy property.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */
#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passwordpolicy プロパティを削除する.
 * (note)
 *   この関数は、パスワードポリシー機能が「サポート」されているケースでのみ
 *   コールできます. 呼び出し元で考慮してください.
 */
static dav_error * _remove_passwordpolicy(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_passwordpolicy_iscreen *passpolicy_iscreen)
{
	apr_pool_t *p = r->pool;

	/* エントリーを削除する */
	if (divy_rdbo_remove_passwordpolicy_property(r, DIVY_DEFAULT_POLICY_ID,
													rbk_ctx->ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to remove passwordpolicy property."
				"(uri = %s)", resource->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * パスワードの変更を行なう.
 * [ 有効性 ]
 *   パスワードポリシー機能がサポートされていなければ
 *   ユーザベースのパスワード状態は反映しない.
 */
static dav_error * _set_changepassword(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_changepassword_iscreen *changepassword_iscreen)
{
	int support_session = divy_support_session(r);
	apr_pool_t *p = r->pool;
	divy_rdbo_usr *usr_pr;
	const divy_rdbo_usr *src_usr_pr;
	const char *password;
	dav_divy_dir_conf *dconf;
	time_t now;

	password   = changepassword_iscreen->password;	/* 入力パスワード */;
	src_usr_pr = changepassword_iscreen->src_usr_pr;

	/* パスワードは変更されたのか?
	 *
	 * (note) 何故単純に比較して変更してもいいのか?
	 *   DBに入っていたパスワードがクライアントサイドにそのまま表示されることは
	 *   外部仕様上、存在しません. これを信じて、送信されてくるパスワードは
	 *   全てエンドユーザが再入力したものとみなします. 例えMD5パスワードを
	 *   入力されたとしても.
	 */
	if (src_usr_pr != NULL && IS_FILLED(src_usr_pr->password) &&
		strcmp(password, src_usr_pr->password) == 0) {
		/* DBの値と同じパスワードであれば何もしてはなりません */
		return NULL;
	}
	now = dav_divy_get_now_epoch();	/* 今の日付を取得する */

	usr_pr = apr_pcalloc(p, sizeof(divy_rdbo_usr));
	usr_pr->next  = NULL;
	usr_pr->usrid = src_usr_pr->usrid;

	/* パスワードポリシー機能がサポートされていた場合 */
	if (divy_support_passpolicy(r)) {

		/* ユーザのパスワードポリシー状態が存在しなかった場合
		 * (途中でポリシーをOFF -> ON した場合など) */
		if (src_usr_pr->passpolicy_status == NULL) {
			/* パスワードポリシー機能が有効/無効に関わらず
			 * パスワードポリシー状態を生成する */
			usr_pr->passpolicy_status =
				divy_create_user_passwordpolicy(p, usr_pr->usrid);
			usr_pr->passpolicy_status->firstlogin = now;
		}
		/* 既に存在していた場合 */
		else {
			usr_pr->passpolicy_status = src_usr_pr->passpolicy_status;
			/* firstlogin が済んでいない場合 */
			if (usr_pr->passpolicy_status->firstlogin == 0L) {
				usr_pr->passpolicy_status->firstlogin = now; /* 時刻をつける */
			}
		}

		/* 最終期限切れメール送信日付 をクリアする */
		 usr_pr->passpolicy_status->sendexpiredmail = 0L;

		/* パスワード更新日付を更新する */
		usr_pr->passpolicy_status->lastchange = now;

		/* 完全期限切れ基点日付をクリアする */
		usr_pr->passpolicy_status->specialstart = 0L;
	}
	/* パスワードポリシー機能がサポートされていない場合 */
	else {
		usr_pr->passpolicy_status = NULL;	/* ステータスなし */
	}

	/* パスワードの入れ替え */
	dconf = dav_divy_get_dir_config(r);
	if (dconf->encryptpassword == DIVY_ENCRYPTPASSWORD_ON) {
		/* セッション情報には平文のパスワードを記録しておく */
		if (support_session) {
				divy_rdbo_update_session_password(r, NULL, divy_pcache_get_data(r->pool, DIVY_PCACHE_DAT_SES_SID), r->user, password);
		}
		/* 暗号化して入れ替える */
		usr_pr->password = ap_md5(p, (const unsigned char*)password);
	}
	else {
		usr_pr->password = (char *)password;
	}

	/* パスワード を登録/更新する */
	if (divy_rdbo_update_userpassword(r, usr_pr, rbk_ctx->ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to update changepassword property.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* memcachedが有効ならmemcachedのセッションのパスワードを更新する */
	if (divy_support_session(r) && divy_enable_memcache(r)) {
		divy_util_auth_update_session_password(r, r->user, password);
	}

	/* メールで通知を行っていいなら投げ込む */
	if (divy_ml_enable_notify_usercreation(r,
										(divy_rdbo_usr *)src_usr_pr)) {
		/* 変更通知を本人へ送信 */
		divy_ml_notify_changed_password(r, src_usr_pr);
	}

	return NULL;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */


/**
 * Otherユーザを管理下に置く操作を実施する.
 */
static dav_error * _set_changeowner(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_changeowner_iscreen *changeowner_iscreen)
{
	apr_pool_t *p          = r->pool;
	char *userid           = changeowner_iscreen->src_usr_pr->usrid;
	const char *own_userid = divy_get_userid(r);

	/* 操作者が管理者の場合、ownerid はNULL にしなければならない */
	if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN) {
		own_userid = NULL;
	}

	/* オーナ情報の更新 */
	if (divy_rdbo_update_user_ownerinfo(r, userid, own_userid, rbk_ctx->ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to update changeowner property.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}


/**
 * グループリーダに任命/解任する.
 */
static dav_error * _set_changeleader(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_changeleader_iscreen *changeleader_iscreen)
{
	apr_pool_t *p = r->pool;
	const char *ownerid;
	const char *top_grpuri;

	if (changeleader_iscreen->action == DIVY_CHLEADER_ACTION_APPOINT) {
		ownerid = changeleader_iscreen->ownerid;
	}
	else {
		ownerid = NULL;	/* オーナをNULLにする */
	}

	top_grpuri = changeleader_iscreen->src_grp_pr->relativeuri;

	/* DB を更新 */
	if (divy_rdbo_update_group_ownerinfo(r, top_grpuri, ownerid, rbk_ctx->ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to update changeleader property.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* メール通知を行う */
	(void) divy_ml_notify_ownerchange(r,
				(changeleader_iscreen->action == DIVY_CHLEADER_ACTION_APPOINT) ? 1 : 0,
				changeleader_iscreen->src_grp_pr, changeleader_iscreen->usr_pr);

	return NULL;
}

/**
 * 開封通知を設定する.
 *
 */
static dav_error * _set_confirmreading(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_confirmreading_iscreen *iscreen)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_confirmreading cr_pr = { 0 };

	cr_pr.uri          = iscreen->uri;
	cr_pr.userid       = iscreen->userid;
	cr_pr.creationdate = dav_divy_get_now_epoch();
	cr_pr.next         = NULL;

	/* DB への書き込み */
	if (divy_rdbo_insert_confirmreading(r, &cr_pr, rbk_ctx->ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to insert confirmreading property.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * 開封通知を解除する.
 *
 */
static dav_error * _remove_confirmreading(request_rec *r,
					const dav_resource *resource,
					dav_liveprop_rollback *rbk_ctx,
					divy_confirmreading_iscreen *iscreen)
{
	apr_pool_t *p = r->pool;

	if (IS_FILLED(iscreen->uri) && IS_FILLED(iscreen->userid)) {
		if (divy_rdbo_remove_confirmreading(r, iscreen->uri, iscreen->userid,
					0, rbk_ctx->ts_ctx)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove confirmreading property.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	return NULL;
}


/*--------------------------------------------------------------
  Create & Initialize LIVE PROPERTY provider Hook structure
  --------------------------------------------------------------*/
const dav_hooks_liveprop dav_divy_hooks_liveprop = {
	dav_divy_insert_prop,
	dav_divy_is_writable,
	dav_divy_namespace_uris,
	dav_divy_patch_validate,
	dav_divy_patch_exec,
	dav_divy_patch_commit,
	dav_divy_patch_rollback,
	NULL
};

