/**
 * $Id$
 *
 * lock.c
 *
 * Locking・プロバイダ
 *
 * 変更履歴
 *
 * 2003/02/13 Thu takehara NEW
 *
 */

/* Apache header files */
#include "httpd.h"
#include "apr.h"
#include "apr_strings.h"
#include "apr_uuid.h"
#include "apr_hash.h"
#include "mod_dav.h"

/* document management headers */
#include "mod_dav_tf.h"
#include "lock.h"
#include "tf_rdbo.h"
#include "util.h"
#include "liveprop.h"
#include "tf_valuecache.h"
#include "tf_rdbo_user.h"

APLOG_USE_MODULE(dav_tf);

/*--------------------------------------------------------------
  Define incomplete type structure
  --------------------------------------------------------------*/

/* lock内で利用するプライベート構造体 */
struct dav_lockdb_private {

	/* リクエスト構造体 */
	request_rec *r;

	/* ロック対象となるレコードを保持する構造体 */
	divy_rdbo_lock_resource *lp;
};

typedef struct 
{
	dav_lockdb pub;
	dav_lockdb_private priv;
}
dav_divy_lockdb_combined;

/*
 * (see mod_dav.h)
 */
struct dav_lock_private {
	const char *key;
};

typedef struct
{
	dav_lock pub;
	dav_lock_private priv;
	dav_locktoken token;
}
dav_divy_lock_combined;

/*--------------------------------------------------------------
 *   Declare prototype function (private scope functions)
 *--------------------------------------------------------------*/
static dav_lock *_dav_divy_alloc_lock(dav_lockdb * , const char * key, 
					    const dav_locktoken *);

static divy_rdbo_lock *_dav_divy_load_lock_record(dav_lockdb * lockdb,
						  const char *key,
						  int add_method);

static int _dav_divy_save_lock_record(dav_lockdb *lockdb,
			 		     const char *key);

static int _dav_divy_do_refresh(divy_rdbo_lock * lockrec, 
				const dav_locktoken_list * ltl,
				time_t new_time);

static char * _divy_convert_lockowner(request_rec *p, const char * lockowner);

/**
 * ロックトークンの存在確認をして、存在してなければ新規作成を行う。
 *
 * @param  *lockdb	dav_lockdb
 * @param  *key		const char
 * @param  *tocktoken	const dav_locktoken ロックトークン
 * @return &comb->pub	dav_lock_combined
 */
static dav_lock *_dav_divy_alloc_lock(dav_lockdb *lockdb, const char *key,
					   const dav_locktoken *locktoken)
{

	dav_divy_lock_combined *comb;
	apr_uuid_t uuid;
	char char_uuid[APR_UUID_FORMATTED_LENGTH + 1];
	TRACE(lockdb->info->r->pool);

	comb = apr_pcalloc(lockdb->info->r->pool, sizeof(*comb));
	comb->pub.rectype = DAV_LOCKREC_DIRECT;
	comb->pub.scope   = DAV_LOCKSCOPE_UNKNOWN;
	comb->pub.type    = DAV_LOCKTYPE_UNKNOWN;
	comb->pub.depth   = 0;
	comb->pub.timeout = 0;
	comb->pub.info    = &comb->priv;
	comb->priv.key    = key;

	/* 
	 * ロックトークンの存在を確認する。
	 */
	if (locktoken == NULL)  {
		comb->pub.locktoken = &comb->token;
		apr_uuid_get(&uuid);
		apr_uuid_format(char_uuid, &uuid);
		comb->token.uuid = apr_pstrdup(lockdb->info->r->pool,
				               char_uuid);
	}
	else {
		comb->pub.locktoken = locktoken;
	}

	return &comb->pub;

}

/**
 * 指定されたキーのロック情報を取得する。
 *
 * @param lockdb dav_lockdb *
 * @param key    const char *
 * @param add_method int
 *
 */
static divy_rdbo_lock *_dav_divy_load_lock_record(dav_lockdb *lockdb,
						  const char *key,
						  int add_method)
{

	divy_rdbo_lock		*lockrec = NULL;
	divy_rdbo_lock_resource *rlp;
	request_rec		*r = lockdb->info->r;
	apr_pool_t		*p = lockdb->info->r->pool;
	const char		*resolvkey;

	/*
	 * open_lockdbで作成されたハッシュテーブルよりkeyの
	 * divy_rdbo_lock構造体を検索する
	 */
	lockrec = apr_hash_get(lockdb->info->lp->token_list_hash,
						key, APR_HASH_KEY_STRING);
	/*
	 * 存在しない場合は直接ロックを探してる可能性があります。
	 * OPENLOCKで取得をすればいいのだがリクエストの階層が異なるとか
	 * DEPTH0でロック解除を行うばあい、直接ロックのレコードを取得
	 * していないケースがある為です。
	 * 再度１件だけ(depth 0)検索してみる。
	 */
	if (lockrec == NULL && add_method == DIVY_LOCK_READ_RESOLVED) {
		/* 1件だけ調査するため、depthは0とする*/
		if (divy_rdbo_get_lock_record(r, &rlp, key, 0, 1) != 0) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			       "A parent lock does not exist.(URI = %s)",
			       REPLACE_NULL(key));
			return NULL;
		}

		/* lockのひき方はget_locksと同じく最後の/を削除してから探す*/
		resolvkey = dav_divy_remove_endslash(r->pool, key);
		lockrec = apr_hash_get(rlp->token_list_hash,
						resolvkey, APR_HASH_KEY_STRING);

		/*
		 * 次の検索に同じ条件で利用する可能性がある為、
		 * 新たにハッシュテーブルに追加する。
		 */
		apr_hash_set(lockdb->info->lp->token_list_hash,
						key, APR_HASH_KEY_STRING,
					       	lockrec);
	}

	return lockrec;
}
/**
 * すべてのロック情報をデータベースへ反映させる
 *
 * @param lockdb dav_lockdb *
 * @param key const char *
 * @return int 0:成功 / 1:失敗
 */
static int _dav_divy_save_lock_record(dav_lockdb *lockdb,
					      const char *key)
{
	int        		ret;
	divy_rdbo_lock 		*lockrec = NULL;
	apr_pool_t		*p       = lockdb->info->r->pool;

	TRACE(p);

	/*
	 * 新規登録案件(LOCKメソッド)を検索する
	 */
	lockrec = _dav_divy_load_lock_record(lockdb, APPEND_TOKEN, 0);
	if (lockrec != NULL) {
		ret = divy_rdbo_lock_insert_record(lockdb->info->r, lockrec);
		if (ret != 0) {
			lockdb->info->lp->status |= TF_LOCK_TRANS_ABORT;
			return ret;
		}

		/**
		 * 登録が完了したら、対象のlockrecはクリア（NULL)する
		 */
		(void)apr_hash_set(lockdb->info->lp->token_list_hash,
						   	APPEND_TOKEN, APR_HASH_KEY_STRING, NULL);
	}

	/*
	 * 直接ロックしたものの間接ロック案件を検索する。
	 */
	lockrec = _dav_divy_load_lock_record(lockdb, DIRECT_INDIRECT, 0);
	if (lockrec != NULL) {
		ret = divy_rdbo_lock_insert_record(lockdb->info->r, lockrec);
		if (ret != 0) {
			lockdb->info->lp->status |= TF_LOCK_TRANS_ABORT;
			return ret;
		}
	}

	lockrec = NULL;
	/*
	 * 削除するURI(timeout or UNLOCKメソッド)を検索する
	 */
	lockrec = _dav_divy_load_lock_record(lockdb, EXPIRED_TOKEN, 0);
	if (lockrec != NULL) {

		ret = divy_rdbo_remove_locktoken(lockdb->info->r,
			       			 lockrec->locktoken);
		if (ret != 0) {
			lockdb->info->lp->status |= TF_LOCK_TRANS_ABORT;
			return ret;
		}
	}

	lockrec = NULL;
	/*
	 * LOCKNULLされているリソース(REMOVE LOCKNULL)を検索する。
	 */
	lockrec = _dav_divy_load_lock_record(lockdb, REMOVE_LOCKNULL, 0);
	if (lockrec != NULL) {
		ret = divy_rdbo_remove_locknull_member(lockdb->info->r,
						       lockrec);
		if (ret != 0) {
			lockdb->info->lp->status |= TF_LOCK_TRANS_ABORT;
			return ret;
		}
	}
						
	lockrec = NULL;
	/*
	 * 更新される案件(lockリフレッシュ)を検索する。
	 */
	lockrec = _dav_divy_load_lock_record(lockdb, REFRESH_TOKEN, 0);
	if (lockrec != NULL) {

		ret = divy_rdbo_update_locktoken(lockdb->info->r,
						 lockrec->locktoken,
						 lockrec->timeout);
		if (ret != 0) {
			lockdb->info->lp->status |= TF_LOCK_TRANS_ABORT;
			return ret;
		}
	}

	lockrec = NULL;
	/*
	 * NULL ロックを解除する案件を検索する。
	 */
	lockrec = _dav_divy_load_lock_record(lockdb, LOCK_STATE, 0);
	if (lockrec != NULL) {

		ret = divy_rdbo_update_locknull_state(lockdb->info->r, lockrec);
		if (ret != 0) {
			lockdb->info->lp->status |= TF_LOCK_TRANS_ABORT;
			return ret;
		}			
	}

	return 0;

}
/**
 * ロックの状態をチェックする。
 * timeoutはtime_t型をmod_dav.cで定義されている。変更は不可
 * @param expires time_t
 * @return 1(ture) ロックはされていない。（過去に解除された）0(ロック）
 *
 */
DIVY_DECLARE(int) dav_divy_lock_expired(time_t expires)
{

	return expires != DAV_TIMEOUT_INFINITE && time(NULL) >= expires;

}

/**
 * ロックプロバイダが正常にCLOSEをしたか確認する
 *
 * @param r request_rec *
 * @return 0: 正常 1:失敗
 *
 */
DIVY_DECLARE(int) dav_divy_lock_close_status(request_rec *r)
{
	void		*data;
	divy_rdbo_lock_resource	*lp;

	/* リクエスト構造体にlockのリソースがあるか確認する */
	data = divy_pcache_get_data(r->pool, DIVY_PCACHE_DAT_REQ_LOCKREC);

	/* lockのデータが無い場合はそのまま正常終了 */
	if (data == NULL) return 0;

	/*
	 * ロックの処理が失敗しているかを確認する
	 */
	lp = (divy_rdbo_lock_resource *)data;
	if (lp->status & TF_LOCK_TRANS_ABORT) return 1;

	return 0;
}
/**
 * ロックのデータベースをOPENして引数で与えられたURI以下のデータを取得する
 * HOOKに存在するdav_divy_open_lockdbの変形版
 * (サーチメソッドで利用されることを目的として作成しました。)
 *
 * @param r request_rec *	リクエスト
 * @param lockdb dav_lockdb *	lockdb構造体
 * @param key const char *	検索対象となるキー（URI)
 * @param depth int		keyからの階層の深さ(0 or 1 or INFINITY)
 * @param expire int		タイムアウトチェック(0: no / 1: yse)
 * @return 結果ステータス( 0: 正常 / 1: 失敗)
 *
 */
DIVY_DECLARE(int) dav_divy_open_lockdb_private(request_rec *r,
						dav_lockdb **lockdb,
						const char *key, int depth,
						int expire)
{
	apr_pool_t			*p    = r->pool;
	dav_divy_lockdb_combined	*comb = NULL;
	divy_rdbo_lock_resource		*lp   = NULL;
	int 				ret   = 0;

	TRACE(p);

	comb = apr_pcalloc(p, sizeof(*comb));
	lp   = apr_pcalloc(p, sizeof(*lp));

	/*
	 * LOCKプロバイダのフックは利用しない。
	 * これが通常のフックのopen_lockdbと異なる部分
	 */
	comb->pub.hooks = NULL;	
	comb->pub.ro 	= 0;

	ret = divy_rdbo_get_lock_record(r, &lp, key, depth, expire);
	if (ret) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DEBUG,
			"Failed to get Lock (URI = %s)", REPLACE_NULL(key));
		return 1;
	}

	/* トランザクションの開始 */
	lp->status = TF_LOCK_TRANS_NOREADY;

	/* プライベート構造体へリクエストとロック情報をセットする       */
	comb->priv.r   = r;
	comb->priv.lp  = lp;
	comb->pub.info = &comb->priv;

	/* lockdbへ設定した構造体を返却する */
	*lockdb = &comb->pub;

	return 0;
}

/**
 * dav_divy_open_lockdb_privateで作られたlockdb構造体を利用して
 * 引数で与えられたuriのロック情報をdav_lock構造体に設定する。
 * dav_divy_open_lockdb_privateで作成されたlockdb構造体でなければ1を
 * 返します。
 *
 * @param lockdb dav_lockdb *     dav_lockdb構造体
 * @param uri    const char *     取得したいURI
 * @param locks  dav_lock   **    返却されるdav_lock構造体
 * @return int (処理ステータス 0: 成功 / 1: 失敗)
 *
 */
DIVY_DECLARE(int) dav_divy_fetch_lock_record(dav_lockdb *lockdb,
						const char *uri,
						dav_lock **locks)
{

	apr_pool_t			*p	 = lockdb->info->r->pool;
	divy_rdbo_lock			*lockrec = NULL;

	TRACE(p);

	*locks = NULL;

	/*
	 * uriが指定されていない場合と
	 * HOOKSが設定されているのはprivateで作成されたlockdb構造体ではない
	 */
	if (uri == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DEBUG,
			"uri is not specified.");
		return 1;
	}

	if (lockdb->hooks) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DEBUG,
			"Inaccurate data is contained in lockdb.");
		return 1;
	}

	lockrec = _dav_divy_load_lock_record(lockdb, uri,
		       					DIVY_LOCK_READ_NOMAL);

	if (lockrec != NULL) {

		*locks              = _dav_divy_alloc_lock(lockdb, uri, NULL);
		(*locks)->locktoken = lockrec->locktoken;
		(*locks)->scope     = lockrec->scope;
		(*locks)->depth     = lockrec->lockdepth;
		(*locks)->timeout   = lockrec->timeout;
		(*locks)->owner     = lockrec->owner;
		(*locks)->auth_user = lockrec->userid;

	}

	return 0;
}

/**
 * dav_divy_fetch_lock_recordを利用してlockdiscoveryタグを作成する。
 * mod_davで定義されているdav_lock_get_activelockを最終的に利用しています。
 * (サーチメソッドで利用されることを目的として作成しました)
 *
 * @param lockdb dav_lockdb *	lockdb構造体
 * @param uri const char *	確認するURI
 * @return const char * 	lockdiscovery内のタグすべて
 *
 */
DIVY_DECLARE(const char *)dav_divy_lock_get_activelock(dav_lockdb *lockdb, 
							const char * uri)
{

	const char	*rettag  = "<"DAV_NS_PREFIX":lockdiscovery/>";
	dav_lock	*lock    = NULL;

	if (!dav_divy_fetch_lock_record(lockdb, uri, &lock)) {
		if (lock != NULL) {
			rettag = dav_lock_get_activelock(lockdb->info->r,
				       				lock, NULL);
		}
	}

	return rettag;

}

/**
 * dav_divy_open_lockdb_privateで作成されたlockdb構造体を閉じる（ふりをする）
 * 特に処理は行いません。
 * ダミー的な関数です。
 * (サーチメソッドで利用されることを目的として作成しました)
 * 
 * @param lockdb dav_lockdb **
 * @return 結果ステータス( 0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) dav_divy_close_lockdb_private(dav_lockdb *lockdb)
{
	return 0;
}


/*--------------------------------------------------------------
  Define Hook functions
  --------------------------------------------------------------*/
/**
 * 指定されたリソースがサポートしているロックの種類(supportedlock)を
 * XML形式(タグで囲まれた)文字列として返却する。
 *
 * この関数で返却された内容がsupportedlockの結果として出力される
 */
static const char * dav_divy_get_supportedlock(const dav_resource *resource)
{
	/* 
	 * リソース・コレクションに関わらず、
	 * サポートしているのは排他ロックだけ --> util.h を使う 
	 */
	return dav_divy_get_exclusivelock_string();
}

/**
 * ロックトークンをパースする。
 * @param p apr_pool_t *
 * @param char_token const char *
 * @param locktoken_p dav_locktoken **
 */
static dav_error * dav_divy_parse_locktoken(apr_pool_t *p, 
                                            const char *char_token,
                                            dav_locktoken **locktoken_p)
{
	dav_locktoken *locktoken;
	apr_uuid_t uuid;

	TRACE(p);

	/* opaquelocktokenのポインタが正しいかチェックする。*/
	if (divy_strstr_c((char*)char_token,"opaquelocktoken:") != char_token) {
		return dav_new_error(p,
				     HTTP_BAD_REQUEST,
				     DAV_ERR_LOCK_UNK_STATE_TOKEN, 0,
				     "The lock token uses an unknown "
				     "State-token "
				     "format and could not be parsed.");
	}
	char_token += 16;
	locktoken = apr_pcalloc(p, sizeof(*locktoken));

	if (apr_uuid_parse(&uuid, char_token)) {
		return dav_new_error(p, HTTP_BAD_REQUEST,
			             DAV_ERR_LOCK_PARSE_TOKEN, 0,
				     "The opaquelocktoken has an "
				     "incorrect format and could "
				     "not be parsed.");
	}

	locktoken->uuid = apr_pstrdup(p ,char_token);
	*locktoken_p = locktoken;
	return NULL;
}

/**
 * ロックトークンを作成する。
 *
 * @param *p 		apr_pool_t
 * @param *locktoken	const dav_locktoken
 */
static const char * dav_divy_format_locktoken(apr_pool_t *p,
                                              const dav_locktoken *locktoken)
{
	return apr_pstrcat(p, "opaquelocktoken:", locktoken->uuid, NULL);
}

/**
 * ロックトークンlt1とlt2の差をとる。
 *
 * @param const lt1 dav_locktoken *
 * @param const lt2 dav_locktoken *
 * @return int
 */
static int dav_divy_compare_locktoken(const dav_locktoken *lt1,
                                      const dav_locktoken *lt2)
{
	return memcmp(lt1->uuid, lt2->uuid, strlen(lt1->uuid));
}

/**
 * lockデータベースをオープンする。
 * lockプロバイダで必要とされる構造体を作成する。
 * 
 * @param *r request_rec
 * @param ro int
 * @param force int
 * @param **lockdb dab_lockdb
 *
 */
static dav_error * dav_divy_open_lockdb(request_rec *r, 
                                        int ro,
                                        int force, 
                                        dav_lockdb **lockdb)
{

	dav_error 		 *err;
	dav_divy_lockdb_combined *comb;
	divy_rdbo_lock_resource  *lp;
	const char 		 *key;
	int     		 depth;
	apr_pool_t		 *p = r->pool;
	void			 *data;

	TRACE(r->pool);

	comb = apr_pcalloc(r->pool, sizeof(*comb));
	lp   = apr_pcalloc(r->pool, sizeof(divy_rdbo_lock_resource));

	comb->pub.hooks = &dav_divy_hooks_locks;
	comb->pub.ro = ro;

	/*
	 * r->uriの以下のロック情報をすべて取得する
	 */
	key = apr_pstrdup(r->pool, r->uri);
	ap_no2slash((char *)key);
	depth = dav_get_depth(r, DAV_INFINITY);

	/*
	 * すでにOPENされて保持されているロック情報があるかチェックする
	 * 存在しなかった場合だけデータベースを検索する
	 */
	data = divy_pcache_get_data(p, DIVY_PCACHE_DAT_REQ_LOCKREC);
	if (data != NULL) {
		lp = (divy_rdbo_lock_resource *)data;
	}
	else {
		/*
		 * リクエスト内でのロック処理を開始した。
		 */
		if (divy_rdbo_get_lock_record(r, &lp, key, depth, 1) != 0) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Could not open LockDB.");
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			return err;
		}
		lp->status = TF_LOCK_TRANS_NOREADY;

		/* エラーがなければキャッシュする */
		divy_pcache_set_data(p, lp, DIVY_PCACHE_DAT_REQ_LOCKREC);
	}

	/* プライベート構造体へリクエストとロック情報をセットする	*/
	comb->priv.r = r;
	comb->priv.lp = lp;
	comb->pub.info = &comb->priv;

	/* what force mean? */
	/*##### force=強制的にを意味すると思うが今は無視 */
	if (force) {

	};

	/* lockdbへ設定した構造体を返却する */
	*lockdb = &comb->pub;
	
	return NULL;
}

/**
 * lockデータベースの後処理を行う。
 * lockはデータベースへの更新は全てこの関数より実行する。
 *
 * @param lockdb dav_lockdb **
 * @return viod
 */
static void dav_divy_close_lockdb(dav_lockdb *lockdb)
{

	/*
	 * すべてのデータをDBへ格納する
	 */

	if (lockdb != NULL) {
		(void)_dav_divy_save_lock_record(lockdb, NULL);
	}

	return;
}

/**
 * ある特定のメソッドは(PUT, MKCOL, COPY, MOVE VERSION)その動作の
 * 前にロックをかけトランザクションを確立するようになっている。
 * ロックNULLは上記のメソッドが正常終了した場合にはロックの解除を
 * 行うのではなく、NULLロックを通常のロックに行う必要がある。
 * これはリソース（もしくはコレクション）が存在しているもののロック
 * はNULLロックであるはずがないということで実装されているからです。
 * メソッドのアクションを起こす前にNULLロックが存在しない場合はこの
 * ハンドラは呼ばれません。
 *
 * @param lockdb dav_lockdb *
 * @param resource dav_resource *
 * @param エラーメッセージ dav_error *
 */
static dav_error * dav_divy_remove_locknull_state(dav_lockdb *lockdb,
                                                  const dav_resource *resource)
{

	const char 		*uri     = NULL;
	divy_rdbo_lock		*lockrec = NULL;
	apr_pool_t		*p       = lockdb->info->r->pool;
	divy_rdbo_lock_resource *lp      = lockdb->info->lp;
	divy_rdbo_lock		*newlockrec = NULL;

	TRACE(p);

	/*
	 * この関数へはロックされている場合にだけ呼ばれる為
	 * 必ずロックレコードが取得できるはずです。
	 */

	/*
	 * null lockは必ずリソースである(最後に/が存在しない)為
	 * 最後の/を削除する
	 * メソッドがMKCOLで呼ばれてきた場合はNULLLOCKを通常ロック
	 * に変更するだけではINDIRECT用が足りない為、追加する。
	 */
	uri = dav_divy_remove_endslash(p, resource->uri);
	lockrec = _dav_divy_load_lock_record(lockdb, uri, 0);
	for (; lockrec != NULL; lockrec = lockrec->next) {
		if (lockrec->is_null)  {
			lockrec->is_null = 0;
			apr_hash_set(lp->token_list_hash, LOCK_STATE,
			             APR_HASH_KEY_STRING, lockrec);

			/* MKCOLによって発生したNULLLOCK解除の場合の追加処理 */
			if (divy_get_method_number(lockdb->info->r) == M_MKCOL) {
				newlockrec = apr_pcalloc(p,
					       		sizeof(divy_rdbo_lock));

				newlockrec->uri = apr_pstrcat(p, lockrec->uri,
					       			     "/", NULL);
				newlockrec->locktoken = lockrec->locktoken;
				newlockrec->kbn       = DAV_LOCK_INDIRECT;
				newlockrec->key = apr_pstrdup(p, lockrec->uri);
				newlockrec->depth     = lockrec->depth;
				newlockrec->timeout   = lockrec->timeout;
				newlockrec->is_null   = lockrec->is_null;

				apr_hash_set(lp->token_list_hash, APPEND_TOKEN,
					       APR_HASH_KEY_STRING, newlockrec);

			}

		}
	}

	return NULL;
}

/*
 * ロックトークンを新規作成する。
 *
 * @param lockdb dav_lockdb *
 * @param resource const dav_resource *
 * @param lock dav_lock **
 * @return dav_error
 */
static dav_error * dav_divy_create_lock(dav_lockdb *lockdb, 
                                        const dav_resource *resource, 
                                        dav_lock **lock)
{

	*lock = _dav_divy_alloc_lock(lockdb, resource->uri, NULL);

	/*
	 * ロックがNULLロックかどうかセットする。
	 */

	(*lock)->is_locknull = !resource->exists;

	return NULL;
}

/**
 * ロック情報を取得する。
 *
 * @param lockdb dav_lockdb *
 * @param calltype int
 * @param locks dav_lock **
 * @return err dav_error
 */
static dav_error * dav_divy_get_locks(dav_lockdb *lockdb, 
                                      const dav_resource *resource,
                                      int calltype, 
                                      dav_lock **locks)
{

	const char *key;
	dav_error *err;

	divy_rdbo_lock		*lockrec = NULL;
	divy_rdbo_lock		*resolvrec = NULL;
	dav_lock		*lock    = NULL;
	dav_lock		*newlock = NULL;
	apr_pool_t		*p	 = lockdb->info->r->pool;
	int			loadtype = 0;

	TRACE(p);

	*locks = NULL;
	key = dav_divy_remove_endslash(p, resource->uri);

	if (calltype == DAV_GETLOCKS_PARTIAL) {
		loadtype = DIVY_LOCK_READ_RESOLVED;
	}

	/*
	 * 対象のuriを検索して内部のリストを取得する。
	 */
	lockrec = _dav_divy_load_lock_record(lockdb, key, loadtype);

	for (; lockrec != NULL; lockrec = lockrec->next) {
		newlock 	  = _dav_divy_alloc_lock(lockdb, key,
			       				 lockrec->locktoken);
		newlock->is_locknull = lockrec->is_null;
		if ((calltype == DAV_GETLOCKS_RESOLVED ||
		     calltype == DAV_GETLOCKS_PARTIAL) && lockrec->key != NULL) {

			/*
			 * 直接ロックより必要な情報を取得する。
			 * 間接ロック(INDIRECT)は以下の情報を保持していない
			 * lockrec->keyが存在しない場合は致命的なエラーとして
			 * 処理を行う。
			 */
			resolvrec = _dav_divy_load_lock_record(lockdb,
				   			       lockrec->key,
							       DIVY_LOCK_READ_RESOLVED);

			/*
			 * 親のリソースが存在しない場合はINTERNALERRORとする
			 */
			if (resolvrec == NULL) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"A parent resource(direct "
					"resource)does not exist.");
				err = dav_new_error(p, 
						HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				return err;

			}

			newlock->scope 	   = resolvrec->scope;
			newlock->depth     = resolvrec->lockdepth;
			newlock->timeout   = resolvrec->timeout;
			newlock->owner     = resolvrec->owner;
			newlock->auth_user = resolvrec->userid;

			/*
			 * PARTIALLOCKはフォルダ内の独立的なロックを行なう処理である
			 * そのため、ここではnewlockのキーとなるuri(newlock->info->key)
			 * にロックのおおもとを保存するようにしなければならない。
			 *
			 * mod_davからはリソース(resource)は一つ親のフォルダが渡って
			 * きます。これでは親がロックのおおもとになっていない場合
			 * 問題が起きてしまいます。そのための処理です。
			 */
			if (calltype == DAV_GETLOCKS_PARTIAL) {
				newlock->rectype     = DAV_LOCKREC_INDIRECT_PARTIAL;
				newlock->info->key   = resolvrec->uri;
			}

		}
		else {
			newlock->scope	     = lockrec->scope;
			newlock->depth       = lockrec->lockdepth;
			newlock->timeout     = lockrec->timeout;
			newlock->owner       = lockrec->owner;
			newlock->auth_user   = lockrec->userid;

		}

		newlock->next = lock;
		lock          = newlock;
	}

	/*
	 * Microsoft Office製品では保存(アップロードを行う時）にLOCKメソッドを
	 * 行った後に、PUT処理が行われます。通常の保存の場合はLOCKとPUTの時間差は
	 * なくエラーは起きません。
	 *
	 * また、ファイルを開きっぱなしの場合でもLOCKが行われています。
	 * このロックは定期的にリフレッシュが行われることによりクライアントの
	 * マシンが応答しなくても自動的にロックが消される仕組みになっています。
	 *
	 * しかし、この消される仕組みが問題となり、たとえばクライアントマシン
	 * のスクリーンセーバ動作によりリフレッシュが行われず、ロックがなくなると
	 * いう現象が発生してしまいます。
	 *
	 * この場合Ifヘッダではロックトークンを保持している為にトークンのエラー
	 * （サーバ上では既にない）となり保存が行えないことがあります。
	 * 以下の処理はこのアップロード（PUT)の時であった場合、そのロックを作成し
	 * あたかも始めからあるロックのように偽装を行う為に導入されました。
	 *
	 * 2019/02/19
	 * この副作用により、すでにロック済みのフォルダに新規ファイルをアップロード
	 * に対してもこの処理を行うようになってしまった。
	 * しかし、このきっかけとなったロック処理は実体があるファイルに対しての
	 * 対策であるため、Null Lockすることが誤りだったわけです。
	 */
	if (lock == NULL &&
			divy_get_method_number(resource->info->r) == M_PUT) {
		
		dav_locktoken_list *ltl;
		divy_rdbo_lock *lockrec, *newlockrec = NULL;
		/* ロックトークンが存在しないと処理ができない */
		dav_get_locktoken_list(resource->info->r, &ltl);
		if (ltl != NULL)  {
			newlock = _dav_divy_alloc_lock(lockdb, resource->uri, ltl->locktoken);
			newlock->is_locknull = !resource->exists;
			newlock->scope       = DAV_LOCKSCOPE_EXCLUSIVE;
			newlock->depth       = DAV_INFINITY;
			newlock->timeout     = dav_get_timeout(resource->info->r);
			newlock->owner       = apr_psprintf(p, "<ns0:owner xmlns:ns0=\"DAV:\">%s</ns0:owner>", divy_get_fullname(resource->info->r));
			newlock->auth_user   = resource->info->r->user; 
			lock = newlock;

			newlockrec = apr_pcalloc(p, sizeof(divy_rdbo_lock));
			newlockrec->uri       = apr_pstrdup(p, resource->uri);
			newlockrec->locktoken = (dav_locktoken *)newlock->locktoken;
			newlockrec->kbn       = DAV_LOCK_DIRECT;
			newlockrec->scope     = newlock->scope;
			newlockrec->depth     = divy_count_dirs(resource->uri); 
			newlockrec->timeout   = newlock->timeout;
			newlockrec->owner     = apr_pstrdup(p, newlock->owner);
			newlockrec->userid    = apr_pstrdup(p, newlock->auth_user);
			/* 実体としてロックすべき null lockではない */
			newlockrec->is_null   = resource->exists;
			newlockrec->lockdepth = newlock->depth;
			newlockrec->key       = NULL;
			lockrec = _dav_divy_load_lock_record(lockdb, APPEND_TOKEN, 0);
			if (lockrec == NULL) {
				lockrec = newlockrec;
			}
			else {
				newlockrec->next = lockrec;
				lockrec          = newlockrec;
			}

			apr_hash_set(lockdb->info->lp->token_list_hash,
							APPEND_TOKEN, APR_HASH_KEY_STRING, lockrec);

		}
	}

	*locks = lock;
	return NULL;

}

/**
 *
 * @param lockdb dav_lockdb *
 * @param resource const dav_resource *
 * @param locktoken const dav_locktoken *
 * @param partial_ok int
 * @param lock dav_lock **
 * @return err dav_error
 *
 */
static dav_error * dav_divy_find_lock(dav_lockdb *lockdb, 
                                      const dav_resource *resource,
                                      const dav_locktoken *locktoken,
                                      int partial_ok,
                                      dav_lock **lock)
{
	divy_rdbo_lock          *lockrec;
	dav_lock		*newlock = NULL;
	apr_pool_t		*p  = lockdb->info->r->pool;
	const char *key;

	TRACE(p);

	*lock = NULL;

	key = dav_divy_remove_endslash(p, resource->uri);

	lockrec = _dav_divy_load_lock_record(lockdb, key, 0);
	if (lockrec != NULL ) {
		if (!dav_divy_compare_locktoken(locktoken,
				        	lockrec->locktoken)) {
			newlock = _dav_divy_alloc_lock(lockdb, key,
						       lockrec->locktoken);
			newlock->is_locknull = !resource->exists;
			newlock->scope       = lockrec->scope;
			newlock->depth 	     = lockrec->lockdepth;
			newlock->timeout     = lockrec->timeout;
			newlock->owner       = lockrec->owner;
			newlock->auth_user   = lockrec->userid;
		}
	}

	*lock = newlock;

	return NULL;
}

/**
 * 対象のリソースがロックされているかをチェックする
 * ただしここでチェックするものはNULLリソースのみです。
 * ここで[通常リソース|コレクション]のロックをチェックすることはありません。
 * util_lock.cのdav_get_resource_stateのみで利用されるフックです。
 *
 * @param *lockdb			ロック構造体
 * @param *resource dav_resource	リソース
 * @param locks_present int		ロックの存在フラグ( 0:なし / 1:ロック)
 * @return dav_error *			エラーメッセージ
 *
 */
static dav_error * dav_divy_has_locks(dav_lockdb *lockdb,
                                      const dav_resource *resource, 
                                      int *locks_present)
{

	const char	*key;
	apr_pool_t	*p = lockdb->info->r->pool;
	divy_rdbo_lock	*lockrec;

	TRACE(p);

	*locks_present = 0;

	/* コレクションの場合は末尾に/をつける */
	if (resource->collection) {
		key = dav_divy_append_endslash(p, resource->uri);
	}
	else {
		key = resource->uri;
	}
	
	/* ロックレコードの構造体を取得する */
	if ((lockrec = _dav_divy_load_lock_record(lockdb, key, 0)) != NULL) {
		*locks_present = 1;
	}

	return NULL;
}

/**
 * mod_davより取得した内容を元にロック情報の確認して追加処理
 * を行う。
 * 
 * @param lockdb dav_lockdb *
 * @param resource dav_resource *
 * @param make_indirect int
 * @param lock const dav_lock *
 * @returnn dav_error
 */
static dav_error * dav_divy_append_locks(dav_lockdb *lockdb,
                                         const dav_resource *resource, 
                                         int make_indirect,
                                         const dav_lock *lock)
{

	divy_rdbo_lock_resource *lp_p = lockdb->info->lp;
	apr_pool_t 	*p            = lockdb->info->r->pool;

	divy_rdbo_lock 	*newlockrec   = NULL;	/* lockレコード構造体 */
	divy_rdbo_lock 	*lockrec      = NULL;	/* lockレコード構造体 */
	divy_rdbo_lock 	*dlockrec     = NULL;   /* lockレコード構造体 */
	const char 	*key          = NULL;
	const char 	*lockkey      = NULL;
	const char 	*directkey    = NULL;

	/*
	 * [ 2005.03.29 Tue ]
	 *     親リソースが存在しない時にはresource に対するロックを成功させては
	 *     ならないとRFC2518に記述されていた。それを実現しようとすると
	 *     以下のようなコードになるのだが、このコードでは通常リソースしか
	 *     さばくことができず、またinfo のparent_resource メンバはフローに利用
	 *     できるようには設計されていないので誤作動する可能性がある。
	 *     (単にパフォーマンスを稼ぐために用意しただけ)
	 *     現時点では検証しきれないのでコメントアウトしておきます。
	 */
#if 0
	if (resource->info->parent_resource != NULL &&
		!resource->info->parent_resource->exists) {
		return dav_new_error(p, HTTP_CONFLICT, 0, "");
	}
#endif

	TRACE(p);

	/*
	 * TeamfileではLOCKスコープはEXCULUSIVEのみのサポートです。
	 * その他のLOCKSCOPEは利用できません。
	 * しかし、コピーの動作でさらにその配下にファイルが一つでも
	 * ある場合mod_davがDAV_LOCKSCOPE_UNKNOWNを渡してきます。
	 * これは、LOCKが親によってかけられたことを意味する為
	 * これは許可する必要があります。
	 * RFC2518ではその他のスコープにはSHARED(共有ロック）が
	 * 存在します。
	 */
	if (lock->scope == DAV_LOCKSCOPE_SHARED) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					     "The lockscope is not supported."
					     "Telnet or the user who performed "
					     "unjest access may exist."
					     " (user: %s)",
					     lockdb->info->r->user);

		return dav_new_error(p,
				     HTTP_BAD_REQUEST,
				     0, 0,
				     "");

	}

	/*
	 * resource->uriを元にlockレコードの構造体を作成する
	 *
	 */
	key   = dav_divy_remove_endslash(p, resource->uri);

	/* 領域確保 */
	newlockrec = apr_pcalloc(p, sizeof(divy_rdbo_lock));
	lockrec    = apr_pcalloc(p, sizeof(divy_rdbo_lock));

	newlockrec->uri     	    = apr_pstrdup(p, key);
	newlockrec->locktoken       = (dav_locktoken *)lock->locktoken;

	/* MEMO:
	 * make_indirectはdepthがinfinityの場合に呼ばれる
	 * LOCK METHODでのdepth1はRFC的にありえません。
	 */
	if (make_indirect) {
		lockkey = dav_divy_remove_endslash(p, lock->info->key);
		newlockrec->kbn     = DAV_LOCK_INDIRECT;
		newlockrec->key     = apr_pstrdup(p, lockkey);
		newlockrec->depth   = divy_count_dirs(newlockrec->uri);
	}
	else if (lock->rectype == DAV_LOCKREC_DIRECT) {

		if (resource->collection) {
			/*
		 	* 直接ロックの間接ロックのレコードを生成する
		 	*  DIRECT_LOCKで作られるレコードの間接ロック版
		 	*
		 	*  LOCKのSTATUSの表示の為に利用するだけのレコード
		 	*  DIRECTのレコードと異なる部分は以下のとおり
		 	*  (1) uri	:  /a/a/ <- 最後の/がある
		 	*  (2) kbn	: DIRECTのレコードだがINDIRECTとする
		 	*  (3) scope   : NULL
		 	*  (4) depth   : NULL
		 	*  (5) owner   : NULL
		 	*  (6) key     : DIRECTのURIが入る
		 	*
		 	*/
			dlockrec = apr_pcalloc(p, sizeof(divy_rdbo_lock));
			directkey = dav_divy_append_endslash(p,lock->info->key);
			dlockrec->uri       = apr_pstrdup(p, directkey);
			dlockrec->locktoken = (dav_locktoken *)lock->locktoken;
			dlockrec->kbn       = DAV_LOCK_INDIRECT;
			dlockrec->scope     = 0;
			dlockrec->depth     = divy_count_dirs(directkey) - 1;
			dlockrec->timeout   = lock->timeout;
			dlockrec->owner     = NULL;
			dlockrec->userid    = apr_pstrdup(p ,lock->auth_user);
			dlockrec->is_null   = !resource->exists;
			dlockrec->key       = apr_pstrdup(p, lock->info->key);
			apr_hash_set(lp_p->token_list_hash, DIRECT_INDIRECT,
						APR_HASH_KEY_STRING, dlockrec);
		}

		newlockrec->kbn       = DAV_LOCK_DIRECT;
		newlockrec->scope     = lock->scope;
		newlockrec->depth     = divy_count_dirs(newlockrec->uri);
		newlockrec->owner     =
		       		_divy_convert_lockowner(lockdb->info->r,
								   lock->owner);

		newlockrec->key       = NULL;
		newlockrec->lockdepth = lock->depth;
	}
	else {
		newlockrec->kbn     = DAV_LOCK_INDIRECT;
		newlockrec->key     = apr_pstrdup(p, lock->info->key);
		newlockrec->depth   = divy_count_dirs(newlockrec->uri);
	}

	/* DIRECT INDIRECT 共通 */
	newlockrec->timeout = lock->timeout;
	newlockrec->userid  = apr_pstrdup(p, lock->auth_user);
	newlockrec->is_null = !resource->exists;

	/*
	 * ハッシュテーブルよりAPPEND_TOKENを探す
	 */
	lockrec = _dav_divy_load_lock_record(lockdb, APPEND_TOKEN, 0);
	if (lockrec == NULL) {
		lockrec = newlockrec;
	}
	else {
		newlockrec->next = lockrec;
		lockrec          = newlockrec;
	}

	/*
	 * ハッシュテーブルに追加する
	 */
	apr_hash_set(lp_p->token_list_hash, APPEND_TOKEN,
						APR_HASH_KEY_STRING, lockrec);
	return NULL;
}

/**
 * 対象のリソースを対象のロックトークンで削除する。
 * @param lockdb dav_lockdb *
 * @param resource const dav_resource *
 * @param locktoken const dav_locktoken *
 * @return dav_error *
 *
 */
static dav_error * dav_divy_remove_lock(dav_lockdb *lockdb, 
                                        const dav_resource *resource,
                                        const dav_locktoken *locktoken)
{
	divy_rdbo_lock		*lockrec    = NULL;
	divy_rdbo_lock		*dellockrec = NULL;
	divy_rdbo_lock_resource *lp         = lockdb->info->lp;
	const char		*key        = NULL;
	const char		*hashname   = NULL;

	/*
	 * ロックトークンがNULLであるとき、リクエストURI(r->uri) 以下の
	 * リソースが持つ全てのロックを解除するためにリクエストURIのlockrecを
	 * 記録する。
	 * (note)
	 * 	mod_davのフレームワークに従えば、本来はresourceのロックだけを
	 * 	解除すべき。だが、この場合、多量のSQLが発行されることとなり
	 * 	著しい性能劣化を引き起こす。そのため、以下の仮定のもと
	 * 	リクエストURI以下のロックを無条件に解除することにした。
	 * 	現バージョンのmod_davは以下の性質を満たしている。mod_davが
	 * 	変更された場合、引き続き以下の性質を満たしているかどうか必ず
	 * 	検証して下さい。
	 * 	(仮定)
	 * 	・ロックトークンがNULLであるときは、リクエストURI以下全ての
	 * 	  リソースに対してもNULLが指定されている
	 * 	・リクエストURI以下の一部のリソースがロック解除対象から除外
	 * 	  されることはない。
	 * 	・全てのロックトークンがNULL指定されたので、排他・共有ロックの
	 * 	  種類に関わらずリソースに付加された全てのロックを取り去る
	 * 	  ことが出来る。
	 */
	if (locktoken == NULL) {	/* '\0' は開始条件に含めない !! */
		/* 既に記録済みであるか ? */
		dellockrec = _dav_divy_load_lock_record(lockdb, REMOVE_LOCKNULL, 0);
		if (dellockrec == NULL) {
			/* r->uri からkey を作る */
			key = dav_divy_remove_endslash(lockdb->info->r->pool,
							lockdb->info->r->uri);
			ap_no2slash((char *)key);
			lockrec = _dav_divy_load_lock_record(lockdb, key, 0);
			if (lockrec != NULL) {
				apr_hash_set(lp->token_list_hash, REMOVE_LOCKNULL,
						APR_HASH_KEY_STRING, lockrec);
			}
		}
		return NULL;	/* 以下を実施する必要なし */
	}

	key = dav_divy_remove_endslash(lockdb->info->r->pool, resource->uri);

	/*
 	 *  ロックテーブルより対象のURIをハッシュテーブルより探す
 	 */
	if ((lockrec = _dav_divy_load_lock_record(lockdb, key, 0)) == NULL) {
		return NULL;
	}

	/*
	 * ロックトークンが渡されてきたかを確認する。
	 * ここでロックトークンが渡されないケースは必ず全てのロックを
	 * 解除する必要があります
	 */

	/*
	 * 削除対象のロックトークンであるかを確認する
	 */ 
	if (dav_divy_compare_locktoken(locktoken, lockrec->locktoken)) {
		return NULL;
	}

	hashname = apr_pstrdup(lockdb->info->r->pool, EXPIRED_TOKEN);

	/*
	 * 削除用のハッシュコードをロックテーブルより探す
	 */
	dellockrec = _dav_divy_load_lock_record(lockdb, hashname, 0);
	if (dellockrec == NULL) {
		apr_hash_set(lp->token_list_hash, hashname,
				       APR_HASH_KEY_STRING, lockrec);
	}
	else {
		lockrec->next = dellockrec;
		dellockrec    = lockrec;
		apr_hash_set(lp->token_list_hash, hashname,
					       APR_HASH_KEY_STRING, dellockrec);
	}

	return NULL;
}

/**
 * ロックのタイムアウトの時間をリフレッシュ（new_timeへ変更)する。
 * ただしロックトークンが異なった場合は行わない。
 * dav_divy_refresh_locks()の内部で行うプライベート関数です。
 *
 * @param lockrec divy_rdbo_lock *
 * @param ltl dav_locktoken *
 * @param new_time time_t
 * @return dirty ( 0: 変更なし / 1: 変更有り)
 *
 */
static int _dav_divy_do_refresh(divy_rdbo_lock * lockrec,
			       const dav_locktoken_list * ltl,
			       time_t new_time)
{

	int dirty = 0;

	/*
	 * ロックトークンはリストとなっている為(ltl)
	 * 内容すべてをチェックする
	 */
	for (; ltl != NULL; ltl = ltl->next) {
		if (dav_divy_compare_locktoken(lockrec->locktoken,
					       ltl->locktoken) == 0) {
			lockrec->timeout = new_time;
			dirty = 1;
		}
	}

	return dirty;
}

/**
 * 指定されたロックトークンに該当するリソースorコレクションを
 * 新しいtimeoutに変更する。
 * LOCKメソッドのリフレッシュロックに利用する。
 *
 * @param lockdb dav_lockdb *
 * @param resource dav_resource *
 * @param ltl const dav_locktoken_list *	ロックトークンのリスト
 * @param new_time time_t			新しいロックタイムアウト
 * @param locks dav_lock **
 * @return err dav_error
 *
 */
static dav_error * dav_divy_refresh_locks(dav_lockdb *lockdb,
                                          const dav_resource *resource,
                                          const dav_locktoken_list *ltl,
                                          time_t new_time,
                                          dav_lock **locks)
{

	const char *key;
	apr_pool_t		*p          = lockdb->info->r->pool;
	divy_rdbo_lock_resource *lp         = lockdb->info->lp;
	divy_rdbo_lock		*lockrec    = NULL;
	divy_rdbo_lock		*reflockrec = NULL;
	dav_lock 		*newlock    = NULL;

	*locks = NULL;

	TRACE(p);

	key = resource->uri;
	/* 対象となるロックレコードを取得する */
	lockrec = _dav_divy_load_lock_record(lockdb, key, 0);

	/*
	 * lockrecのリストでロックトークンに該当するものを探す
	 */
	for (; lockrec != NULL; lockrec = lockrec->next) {

		if (_dav_divy_do_refresh(lockrec, ltl, new_time)) {
			/*
			 * リフレッシュ可能なトークンが存在した
			 * 場合REFRESH_TOKENのキーでハッシュテーブル
			 * より探し出す。
			 */
			reflockrec = apr_hash_get(lp->token_list_hash,
				       			REFRESH_TOKEN,
						     	APR_HASH_KEY_STRING);
			if (reflockrec == NULL) {
				apr_hash_set(lp->token_list_hash,
							REFRESH_TOKEN,
						       	APR_HASH_KEY_STRING,
						       		    lockrec);
			}
			else {
				/* 
				 * リフレッシュされようとしているのが
				 * 既にあった
				 */
				lockrec->next = reflockrec;
				reflockrec    = lockrec;
				apr_hash_set(lp->token_list_hash,
							REFRESH_TOKEN,
						       	APR_HASH_KEY_STRING,
								reflockrec);
			}

			newlock = _dav_divy_alloc_lock(lockdb,
				       		       lockrec->key,
						       lockrec->locktoken);
			newlock->is_locknull = lockrec->is_null;
			newlock->scope       = lockrec->scope;
			newlock->depth       = lockrec->depth;
			newlock->timeout     = lockrec->timeout;
			newlock->auth_user   = lockrec->userid;

			newlock->next = *locks;
			*locks        = newlock;
		}
	}

	return NULL;
}

#if 0
/**
 * resourceを返却する為、LOCKでは行わない。
 * その代わり、dav_divy_find_lock()を用いる。
 * この関数の機能は以下の通りです。
 * 間接ロックが自分をロックさせた直接ロックのリソースを探し出し
 * そのリソースを返却する。
 *
 */
static dav_error * dav_divy_lookup_resource(dav_lockdb *lockdb,
                                            const dav_locktoken *locktoken,
                                            const dav_resource *start_resource,
                                            const dav_resource **resource)
{
	return NULL;
}
#endif

/**
 * ロックを操作するときのユーザIDのチェックを行うか否かを調べる
 * 通常は、ロックオーナーとリクエストのユーザは同じでなければ
 * ロックの解除は行えない。
 * ここではTeamFile用としてそれをスキップできるようになっています。
 *
 * @param lockowner	const char *	ロックオーナーID
 * @param resource const dav_resource * DAVリソース
 *
 * @return 0: スキップしない / 1: スキップできる
 *
 */
static int dav_divy_skip_auth_operation(const char *lockowner,
										const dav_resource *resource)
{

	request_rec *r;
	const divy_rdbo_extstatus *extstatus;
	divy_rdbo_usr *usr_pr;

    if (resource == NULL || resource->info == NULL) {
		return 0; /* skipさせない */
	}

	r = resource->info->r;

	if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN) {
		return 1; /* 管理者はスキップできる */
	}

	if (!divy_support_groupleader(r)) {
		return 0; /* グループリーダー機能がないならスキップできない */
	}

	extstatus = divy_get_extstatus(r);
	if (!divy_rdbo_is_groupleader(extstatus)) {
		/* 管理者は既に上でチェック済み */
		return 0; /* グループリーダーでなければスキップできない */
	}

	if (divy_rdbo_get_user_property(r, lockowner, &usr_pr) == 0) {
		if (usr_pr->ownerid &&
				strcmp(usr_pr->ownerid, divy_get_userid(r)) == 0) {
			return 1; /* ロックオーナーのユーザを管理しているユーザならスキップ */
		}
	}	

	return 0; /* デフォルトはスキップしない */
}

/**
 * lockownerのタグをdivy用に拡張する。
 * lockownerはmod_davよりownerエレメント内すべてが渡される為
 * divyで利用する為にvalueをフルネームに変更して格納する。
 * タグの不整合等問題があった場合そのまま返却する。
 *
 * @param r request_rec
 * @param owner const char *
 * @return タグを含んだ文字列 char *
 */
static char * _divy_convert_lockowner(request_rec *r, const char *owner)
{
	char *op;
	char *starttag, *endtag;
	char *retowner = NULL;
	const char *fullname;

	TRACE(r->pool);

	/* フルネームを取得する */
	fullname = divy_get_fullname(r);
	if (fullname == NULL) {
		return NULL;
	}

	starttag = apr_pcalloc(r->pool, sizeof(char) * strlen(owner));
	endtag   = apr_pcalloc(r->pool, sizeof(char) * strlen(owner));

	/* タグ始まりを調べて全てコピーする */
	op = strchr((char *) owner, '>');
	if (op == NULL) {
		return (char *)owner;
	}
	strncpy(starttag, owner, op - owner + 1);

	/* タグの終わりを調べて全てコピーする */
	op = divy_strrchr((char *) owner, '<');
	if (op == NULL) {
		return (char *)owner;
	}
	strcat(endtag, op);

	/* すべてを連結する */
	retowner = apr_pstrcat(r->pool, starttag, fullname, endtag, CRLF ,NULL);

	return retowner;
}

/**
 * 指定されたユーザID userid のユーザが持っていた全てのロック及び
 * ロックNULLリソースを削除する。
 *
 */
DIVY_DECLARE(int) divy_lock_remove_user_lock(request_rec *r, const char *userid,
						divy_db_transaction_ctx *ts_ctx)
{
	return divy_rdbo_remove_user_lock(r, userid, ts_ctx);
}

/*--------------------------------------------------------------
  Create & Initialize LOOK provider Hook structure
  --------------------------------------------------------------*/
const dav_hooks_locks dav_divy_hooks_locks = {
	dav_divy_get_supportedlock,	/* サポートされているロックタイプ表示*/
	dav_divy_parse_locktoken,	/* ロックトークンをパースする	*/
	dav_divy_format_locktoken,	/* ロックトークンを作成する	*/
	dav_divy_compare_locktoken,	/* ロックトークンをチェック	*/
	dav_divy_open_lockdb,		/* ロックDBを開く		*/
	dav_divy_close_lockdb,		/* ロックDBを閉じる（保存/削除）*/
	dav_divy_remove_locknull_state, /* ロックNULLを通常ロックにする */
	dav_divy_create_lock,		/* ロックを新規に作成する	*/
	dav_divy_get_locks,		/* ロック情報を取得する		*/
	dav_divy_find_lock,		/* 		*/
	dav_divy_has_locks,		/* ロックの確認			*/
	dav_divy_append_locks,		/* 新規にロックを追加する	*/
	dav_divy_remove_lock,
	dav_divy_refresh_locks,
	NULL,				/* dav_divy_lookup_resource	*/
	NULL,
	dav_divy_skip_auth_operation,
};
