/**
 * $Id$
 *
 * tf_misc_util.c
 *
 * 動的に切り離し可能な関数を定義するファイル
 * ここに定義された関数は、tf_misc.[ch] 経由でないと使用できない
 * ようになっています。ですので、ヘッダファイルを作っていません。
 */
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"
#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif	/* APR_HAVE_STDLIB_H */
#include "iconv_hook/iconv.h"

#if defined(DIVY_DEBUG_PROVIDER_LOADER) && APR_HAVE_UNISTD_H
#include <unistd.h>
#endif	/* defined(DIVY_DEBUG_PROVIDER_LOADER) && APR_HAVE_UNISTD_H */

#include "tf_provider.h"
#include "util_common.h"
#include "tf_misc.h"

/*------------------------------------------------------------------------------
  Define struct
  ----------------------------------------------------------------------------*/
typedef struct __divy_strmap	divy_strmap;
struct __divy_strmap {
	const char *from;
	const char *to;
};

/*------------------------------------------------------------------------------
  Define Global value
  ----------------------------------------------------------------------------*/
/* 一部非互換文字といわゆる半角文字を全角文字に変換するためのマップ */
static const divy_strmap _smap[] = {
	{"\xef\xbc\x8d", "\xe2\x88\x92"},	/* FULLWIDTH_HYPHEN_MINUS */
	{"\xef\xbd\x9e", "\xe3\x80\x9c"},	/* FULLWIDTH_FILDE        */
	{"\xef\xbd\xa1", "\xe3\x80\x82"},	/* HALFWIDTH IDEOGRAPHIC FULL STOP    */
	{"\xef\xbd\xa2", "\xe3\x80\x8c"},	/* HALFWIDTH LEFT CORNER BRACKET      */
	{"\xef\xbd\xa3", "\xe3\x80\x8d"},	/* HALFWIDTH RIGHT CORNER BRACKET     */
	{"\xef\xbd\xa4", "\xe3\x80\x81"},	/* HALFWIDTH IDEOGRAPHIC COMMA        */
	{"\xef\xbd\xa5", "\xe3\x83\xbb"},	/* HALFWIDTH KATAKANA MIDDLE DOT      */
	{"\xef\xbd\xa6", "\xe3\x83\xb2"},	/* HALFWIDTH KATAKANA LETTER WO       */
	{"\xef\xbd\xa7", "\xe3\x82\xa1"},	/* HALFWIDTH KATAKANA LETTER SMALL A  */
	{"\xef\xbd\xa8", "\xe3\x82\xa3"},	/* HALFWIDTH KATAKANA LETTER SMALL I  */
	{"\xef\xbd\xa9", "\xe3\x82\xa5"},	/* HALFWIDTH KATAKANA LETTER SMALL U  */
	{"\xef\xbd\xaa", "\xe3\x82\xa7"},	/* HALFWIDTH KATAKANA LETTER SMALL E  */
	{"\xef\xbd\xab", "\xe3\x82\xa9"},	/* HALFWIDTH KATAKANA LETTER SMALL O  */
	{"\xef\xbd\xac", "\xe3\x83\xa3"},	/* HALFWIDTH KATAKANA LETTER SMALL YA */
	{"\xef\xbd\xad", "\xe3\x83\xa5"},	/* HALFWIDTH KATAKANA LETTER SMALL YU */
	{"\xef\xbd\xae", "\xe3\x83\xa7"},	/* HALFWIDTH KATAKANA LETTER SMALL YO */
	{"\xef\xbd\xaf", "\xe3\x83\x83"},	/* HALFWIDTH KATAKANA LETTER SMALL TU */
	{"\xef\xbd\xb0", "\xe3\x83\xbc"},	/* HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK */
	{"\xef\xbd\xb1", "\xe3\x82\xa2"},	/* HALFWIDTH KATAKANA LETTER A  */
	{"\xef\xbd\xb2", "\xe3\x82\xa4"},	/* HALFWIDTH KATAKANA LETTER I  */
	{"\xef\xbd\xb3", "\xe3\x82\xa6"},	/* HALFWIDTH KATAKANA LETTER U  */
	{"\xef\xbd\xb4", "\xe3\x82\xa8"},	/* HALFWIDTH KATAKANA LETTER E  */
	{"\xef\xbd\xb5", "\xe3\x82\xaa"},	/* HALFWIDTH KATAKANA LETTER O  */
	{"\xef\xbd\xb6", "\xe3\x82\xab"},	/* HALFWIDTH KATAKANA LETTER KA */
	{"\xef\xbd\xb7", "\xe3\x82\xad"},	/* HALFWIDTH KATAKANA LETTER KI */
	{"\xef\xbd\xb8", "\xe3\x82\xaf"},	/* HALFWIDTH KATAKANA LETTER KU */
	{"\xef\xbd\xb9", "\xe3\x82\xb1"},	/* HALFWIDTH KATAKANA LETTER KE */
	{"\xef\xbd\xba", "\xe3\x82\xb3"},	/* HALFWIDTH KATAKANA LETTER KO */
	{"\xef\xbd\xbb", "\xe3\x82\xb5"},	/* HALFWIDTH KATAKANA LETTER SA */
	{"\xef\xbd\xbc", "\xe3\x82\xb7"},	/* HALFWIDTH KATAKANA LETTER SI */
	{"\xef\xbd\xbd", "\xe3\x82\xb9"},	/* HALFWIDTH KATAKANA LETTER SU */
	{"\xef\xbd\xbe", "\xe3\x82\xbb"},	/* HALFWIDTH KATAKANA LETTER SE */
	{"\xef\xbd\xbf", "\xe3\x82\xbd"},	/* HALFWIDTH KATAKANA LETTER SO */
	{"\xef\xbe\x80", "\xe3\x82\xbf"},	/* HALFWIDTH KATAKANA LETTER TA */
	{"\xef\xbe\x81", "\xe3\x83\x81"},	/* HALFWIDTH KATAKANA LETTER TI */
	{"\xef\xbe\x82", "\xe3\x83\x84"},	/* HALFWIDTH KATAKANA LETTER TU */
	{"\xef\xbe\x83", "\xe3\x83\x86"},	/* HALFWIDTH KATAKANA LETTER TE */
	{"\xef\xbe\x84", "\xe3\x83\x88"},	/* HALFWIDTH KATAKANA LETTER TO */
	{"\xef\xbe\x85", "\xe3\x83\x8a"},	/* HALFWIDTH KATAKANA LETTER NA */
	{"\xef\xbe\x86", "\xe3\x83\x8b"},	/* HALFWIDTH KATAKANA LETTER NI */
	{"\xef\xbe\x87", "\xe3\x83\x8c"},	/* HALFWIDTH KATAKANA LETTER NU */
	{"\xef\xbe\x88", "\xe3\x83\x8d"},	/* HALFWIDTH KATAKANA LETTER NE */
	{"\xef\xbe\x89", "\xe3\x83\x8e"},	/* HALFWIDTH KATAKANA LETTER NO */
	{"\xef\xbe\x8a", "\xe3\x83\x8f"},	/* HALFWIDTH KATAKANA LETTER HA */
	{"\xef\xbe\x8b", "\xe3\x83\x92"},	/* HALFWIDTH KATAKANA LETTER HI */
	{"\xef\xbe\x8c", "\xe3\x83\x95"},	/* HALFWIDTH KATAKANA LETTER HU */
	{"\xef\xbe\x8d", "\xe3\x83\x98"},	/* HALFWIDTH KATAKANA LETTER HE */
	{"\xef\xbe\x8e", "\xe3\x83\x9b"},	/* HALFWIDTH KATAKANA LETTER HO */
	{"\xef\xbe\x8f", "\xe3\x83\x9e"},	/* HALFWIDTH KATAKANA LETTER MA */
	{"\xef\xbe\x90", "\xe3\x83\x9f"},	/* HALFWIDTH KATAKANA LETTER MI */
	{"\xef\xbe\x91", "\xe3\x83\xa0"},	/* HALFWIDTH KATAKANA LETTER MU */
	{"\xef\xbe\x92", "\xe3\x83\xa1"},	/* HALFWIDTH KATAKANA LETTER ME */
	{"\xef\xbe\x93", "\xe3\x83\xa2"},	/* HALFWIDTH KATAKANA LETTER MO */
	{"\xef\xbe\x94", "\xe3\x83\xa4"},	/* HALFWIDTH KATAKANA LETTER YA */
	{"\xef\xbe\x95", "\xe3\x83\xa6"},	/* HALFWIDTH KATAKANA LETTER YU */
	{"\xef\xbe\x96", "\xe3\x83\xa8"},	/* HALFWIDTH KATAKANA LETTER YO */
	{"\xef\xbe\x97", "\xe3\x83\xa9"},	/* HALFWIDTH KATAKANA LETTER RA */
	{"\xef\xbe\x98", "\xe3\x83\xaa"},	/* HALFWIDTH KATAKANA LETTER RI */
	{"\xef\xbe\x99", "\xe3\x83\xab"},	/* HALFWIDTH KATAKANA LETTER RU */
	{"\xef\xbe\x9a", "\xe3\x83\xac"},	/* HALFWIDTH KATAKANA LETTER RE */
	{"\xef\xbe\x9b", "\xe3\x83\xad"},	/* HALFWIDTH KATAKANA LETTER RO */
	{"\xef\xbe\x9c", "\xe3\x83\xaf"},	/* HALFWIDTH KATAKANA LETTER WA */
	{"\xef\xbe\x9d", "\xe3\x83\xb3"},	/* HALFWIDTH KATAKANA LETTER N  */
	{"\xef\xbe\x9e", "\xe3\x82\x9b"},	/* HALFWIDTH KATAKANA VOICED SOUND MARK */
	{"\xef\xbe\x9f", "\xe3\x82\x9c"},	/* HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK */
	{"\xe2\x88\xa5", "\xe2\x80\x96"},	/* DOUBLE VERTICAL LINE */
	{"\xef\xbf\xa0", "\xc2\xa2"},		/* CENT SIGN */
	{"\xef\xbf\xa1", "\xc2\xa3"},		/* POUND SIGN */
	{"\xef\xbf\xa2", "\xc2\xac"},		/* NOT SIGN */
	{"\xe2\x85\xa0", "\x49"},				/* Roman Numeral One */
	{"\xe2\x85\xa1", "\x49\x49"},			/* Roman Numeral Two */
	{"\xe2\x85\xa2", "\x49\x49\x49"},			/* Roman Numeral Three */
	{"\xe2\x85\xa3", "\x49\x56"},			/* Roman Numeral Four */
	{"\xe2\x85\xa4", "\x56"},				/* Roman Numeral Five */
	{"\xe2\x85\xa5", "\x56\x49"},			/* Roman Numeral Six */
	{"\xe2\x85\xa6", "\x56\x49\x49"},		/* Roman Numeral Seven */
	{"\xe2\x85\xa7", "\x56\x49\x49\x49"},	/* Roman Numeral Eight */
	{"\xe2\x85\xa8", "\x49\x58"},			/* Roman Numeral Nine */
	{"\xe2\x85\xa9", "\x58"},				/* Roman Numeral Ten */
	{"\xe2\x91\xa0", "\x28\x31\x29"},		/* Circled Digit One */
	{"\xe2\x91\xa1", "\x28\x32\x29"},		/* Circled Digit Two */
	{"\xe2\x91\xa2", "\x28\x33\x29"},		/* Circled Digit Three */
	{"\xe2\x91\xa3", "\x28\x34\x29"},		/* Circled Digit Four */
	{"\xe2\x91\xa4", "\x28\x35\x29"},		/* Circled Digit Five */
	{"\xe2\x91\xa5", "\x28\x36\x29"},		/* Circled Digit Six */
	{"\xe2\x91\xa6", "\x28\x37\x29"},		/* Circled Digit Seven */
	{"\xe2\x91\xa7", "\x28\x38\x29"},		/* Circled Digit Eight */
	{"\xe2\x91\xa8", "\x28\x39\x29"},		/* Circled Digit Nine */
	{"\xe2\x91\xa9", "\x28\x31\x30\x29"},	/* Circled Digit Ten */
	{"\xe2\x91\xaa", "\x28\x31\x31\x29"},	/* Circled Digit Eleven */
	{"\xe2\x91\xab", "\x28\x31\x32\x29"},	/* Circled Digit Twelve */
	{"\xe2\x91\xac", "\x28\x31\x33\x29"},	/* Circled Digit Thirteen */
	{"\xe2\x91\xad", "\x28\x31\x34\x29"},	/* Circled Digit Fourteen */
	{"\xe2\x91\xae", "\x28\x31\x35\x29"},	/* Circled Digit Fifteen */
	{"\xe2\x91\xaf", "\x28\x31\x36\x29"},	/* Circled Digit Sixteen */
	{"\xe2\x91\xb0", "\x28\x31\x37\x29"},	/* Circled Digit Seventeen */
	{"\xe2\x91\xb1", "\x28\x31\x38\x29"},	/* Circled Digit Eighteen */
	{"\xe2\x91\xb2", "\x28\x31\x39\x29"},	/* Circled Digit Nineteen */
	{"\xe2\x91\xb3", "\x28\x32\x30\x29"},	/* Circled Digit Twenty */
	{"\xe3\x8d\x89", "\xe3\x83\x9f\xe3\x83\xaa"},				/* Square Miri */
	{"\xe3\x8c\x94", "\xe3\x82\xad\xe3\x83\xad"},				/* Square Kiro */
	{"\xe3\x8c\xa2", "\xe3\x82\xbb\xe3\x83\xb3\xe3\x83\x81"},	/* Square Senti */
	{"\xe3\x8d\x8d", "\xe3\x83\xa1\xe3\x83\xbc\xe3\x83\x88\xe3\x83\xab"},	/* Square Meetoru */
	{"\xe3\x8c\x98", "\xe3\x82\xb0\xe3\x83\xa9\xe3\x83\xa0"},	/* Square Guramu */
	{"\xe3\x8c\xa7", "\xe3\x83\x88\xe3\x83\xb3"},				/* Square Ton */
	{"\xe3\x8c\x83", "\xe3\x82\xa2\xe3\x83\xbc\xe3\x83\xab"},	/* Square Aaru */
	{"\xe3\x8c\xb6", "\xe3\x83\x98\xe3\x82\xaf\xe3\x82\xbf\xe3\x83\xbc\xe3\x83\xab"},	/* Square Hekutaaru */
	{"\xe3\x8d\x91", "\xe3\x83\xaa\xe3\x83\x83\xe3\x83\x88\xe3\x83\xab"},	/* Square Rittoru */
	{"\xe3\x8d\x97", "\xe3\x83\xaf\xe3\x83\x83\xe3\x83\x88"},				/* Square Watto */
	{"\xe3\x8c\x8d", "\xe3\x82\xab\xe3\x83\xad\xe3\x83\xaa\xe3\x83\xbc"},	/* Square Karori */
	{"\xe3\x8c\xa6", "\xe3\x83\x89\xe3\x83\xab"},	/* Square Doru */
	{"\xe3\x8c\xa3", "\xe3\x82\xbb\xe3\x83\xb3\xe3\x83\x88"},	/* Square Sento */
	{"\xe3\x8c\xab", "\xe3\x83\x91\xe3\x83\xbc\xe3\x82\xbb\xe3\x83\xb3\xe3\x83\x88"},	/* Square Paasento */
	{"\xe3\x8d\x8a", "\xe3\x83\x9f\xe3\x83\xaa\xe3\x83\x90\xe3\x83\xbc\xe3\x83\xab"},	/* Square Miribaaru */	
	{"\xe3\x8c\xbb", "\xe3\x83\x9a\xe3\x83\xbc\xe3\x82\xb8"},	/* Square Peegi */
	{"\xe3\x8e\x9c", "\x6d\x6d"},	/* Square Mm */
	{"\xe3\x8e\x9d", "\x63\x6d"},	/* Square Cm */
	{"\xe3\x8e\x9e", "\x6b\x6d"},	/* Square Km */
	{"\xe3\x8e\x8e", "\x6d\x67"},	/* Square Mg */
	{"\xe3\x8e\x8f", "\x6b\x67"},	/* Square Kg */
	{"\xe3\x8f\x84", "\x63\x63"},	/* Square Cc */
	{"\xe3\x8e\xa1", "\x6d\x32"},	/* Square M Squared */
	{"\xe3\x8d\xbb", "\xe5\xb9\xb3\xe6\x88\x90"},	/* Square Era Name Heisei */
	{"\xe3\x80\x9d", "\xe2\x80\x9d"},	/* Reversed Double Prime Quotation Mark */
	{"\xe3\x80\x9f", "\xe2\x80\x9d"},	/* Low Double Prime Quotation Mark */
	{"\xe2\x84\x96", "\x4e\x6f\x2e"},	/* Numero Sign */
	{"\xe3\x8f\x8d", "\x4b\x2e\x4b\x2e"},		/* Square Kk */
	{"\xe2\x84\xa1", "\x54\x65\x6c"},	/* Telephone Sign */
	{"\xe3\x8a\xa4", "\x28\xe4\xb8\x8a\x29"},	/* Circled Ideograph High   */
	{"\xe3\x8a\xa5", "\x28\xe4\xb8\xad\x29"},	/* Circled Ideograph Center */
	{"\xe3\x8a\xa6", "\x28\xe4\xb8\x8b\x29"},	/* Circled Ideograph Low    */
	{"\xe3\x8a\xa7", "\x28\xe5\xb7\xa6\x29"},	/* Circled Ideograph Left   */
	{"\xe3\x8a\xa8", "\x28\xe5\x8f\xb3\x29"},	/* Circled Ideograph Right  */
	{"\xe3\x88\xb1", "\x28\xe6\xa0\xaa\x29"},	/* Parenthesized Ideograph Stock */
	{"\xe3\x88\xb2", "\x28\xe6\x9c\x89\x29"},	/* Parenthesized Ideograph Have  */
	{"\xe3\x88\xb9", "\x28\xe4\xbb\xa3\x29"},	/* Parenthesized Ideograph Represent */
	{"\xe3\x8d\xbe", "\xe6\x98\x8e\xe6\xb2\xbb"},	/* Square Era Name Meizi   */
	{"\xe3\x8d\xbd", "\xe5\xa4\xa7\xe6\xad\xa3"},	/* Square Era Name Taisyou */
	{"\xe3\x8d\xbc", "\xe6\x98\xad\xe5\x92\x8c"},	/* Square Era Name Syouwa  */
	{NULL, NULL}
};

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
#if defined(DIVY_LIB_DYNAMIC_EXPORT) && defined(DIVY_USE_GNU_LD)
static void _misc_util_init(void) __attribute__ ((constructor));
static void _misc_util_fini(void) __attribute__ ((destructor));
#endif	/* DIVY_LIB_DYNAMIC_EXPORT && DIVY_USE_GNU_LD */
static int _misc_util_init_func(apr_pool_t *pconf);
static int _get_miscfunctions(apr_pool_t *p,
				const char *func_t_name, void **func_p);

/*------------------------------------------------------------------------------
  Define provider library hook structure
  ----------------------------------------------------------------------------*/
DIVY_PROVIDER_HOOK_STRUCT(misc,util,
		_misc_util_init_func,		/* 初期化ハンドラ */
		divy_lib_nop_fini_func);	/* 終了ハンドラ(nop) */

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * fromfmt でエンコードされている文字列fromstr を、tofmtでエンコードされている
 * 文字列tostrに変換して返却する。
 * (note)
 * 	tostr には(strlen(fromstr)*4+1) 文字分の領域を確保します。
 * 	fromstrの文字列はfromfmtが示すエンコードと一致していなければ
 * 	なりません。(失敗します)
 *
 * @param p apr_pool_t * プールへのポインタ
 * @param fromfmt const char * 変換前のエンコード
 * @param tofmt const char *   変換後のエンコード
 * @param fromstr char * 変換前の文字列
 * @param tostr char **  変換後の文字列
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _encode_str(apr_pool_t *p, const char *fromfmt,
				const char *tofmt, char *fromstr, char **tostr)
{
	char *outbuf, *marker, *srcstr = NULL;
	size_t srclen, outlen;
	iconv_t cd;
	int ierr;

	*tostr = NULL;		/* 初期化 */

	if (IS_EMPTY(fromstr)) {
		return 0;
	}

	if (IS_EMPTY(fromfmt)) {
		LERRLOG0(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"\"fromfmt\" is EMPTY.");
		return 1;
	}

	if (IS_EMPTY(tofmt)) {
		LERRLOG0(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"\"tofmt\" is EMPTY.");
		return 1;
	}

	/* 開く */
	if ((cd = iconv_open(tofmt, fromfmt)) == (iconv_t)(-1)) {
		 LERRLOG2(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to execute iconv_open. "
			"Maybe iconv_open could not understand the name of encoding."
			"(to = %s, from = %s)", tofmt, fromfmt);
		 return 1;
	}

	srcstr = fromstr;	/* デフォルト */

	/* UTF-8 からISO-2022-JP に変換する時だけの特別考慮 */
	if (strcasecmp(fromfmt, "UTF-8") == 0 &&
	    strcasecmp(tofmt, "ISO-2022-JP") == 0) {
		int i;
		const divy_strmap *e;

		srcstr = fromstr;
		for (i = 0, e = &_smap[0]; e->from != NULL;
						i++, e = &_smap[i]) {
			/* e->from の文字列をe->to で置き換える */
			srcstr = (char *) dav_divy_replace_str(p, srcstr,
							e->from, e->to, NULL);
		}
	}
	srclen = strlen(srcstr);
	outlen = srclen * 4;

	/* 領域確保　先頭を保存 */
	marker = outbuf = apr_pcalloc(p, outlen + 1);

	/*
	 * エンコーディングスキームの変換をする
	 */
	if (iconv(cd, (char **) &srcstr, &srclen,
					&outbuf, &outlen) == (size_t)(-1)) {
		char *errmsg;
		ierr = errno; /* エラー番号を取得する */

		errmsg = apr_psprintf(p, "fromfrm : \"%s\", tofmt : \"%s\", "
					 "fromstr : %s, errno = %d",
					 fromfmt, tofmt, srcstr, ierr);

		LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to convert string.(%s)", errmsg);
		iconv_close(cd);
		return 1;
	}

	/*
	 * シフト状態を元に戻す(シフト系エンコーディングスキーム対応)
	 */
	if (iconv(cd, NULL, NULL, &outbuf, &outlen) == (size_t)(-1)) {
		ierr = errno;	/* エラー番号を取得する */
		LERRLOG3(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to insert shift-state into encorded string."
			"(fromfrm : \"%s\", tofmt : \"%s\", errno = %d",
			fromfmt, tofmt, ierr);
		iconv_close(cd);
		return 1;
	}

	*outbuf = '\0';
	*tostr = marker;

	/* 閉じる */
	iconv_close(cd);

	return 0;
}

/*-----------------------------------------------------------------------------
  Define provider library handler
 *---------------------------------------------------------------------------*/
#if defined(DIVY_LIB_DYNAMIC_EXPORT)
/**
 * スタートアップハンドラ (コンストラクタ関数)
 * (note)
 * 	このハンドラはdlopen 終了直後に呼び出されます。また、このライブラリが
 * 	ダイナミックリンクライブラリとして使用された時には、リンク先のmain 関数
 * 	コールの直前に呼び出されます。
 * (note)
 * 	この関数はプロバイダライブラリがstatic コンパイルされると自動的に
 * 	デバックアウトされます。従って、この関数には、
 * 	    * dlopen された時だけ実施する特別な初期化処理
 * 	    * 関数divy_register_lib_provider のコール
 * 	以外の処理を"書いてはなりません"。
 */
#if defined(DIVY_USE_GNU_LD)
static void _misc_util_init(void)
#else
void _init(void)
#endif	/* DIVY_USE_GNU_LD */
{
	/* プロバイダライブラリ構造体の登録 */
	divy_register_lib_provider(&DIVY_PROVIDER_HOOK_STRUCT_NAME(misc,util));
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"misc(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
}

/**
 * 終了ハンドラ (デストラクタ関数)
 * (note)
 * 	このハンドラはdlclose 終了直後に呼び出されます。また、このライブラリが
 * 	ダイナミックリンクライブラリとして使用された時には、リンク先のmain 関数
 * 	が終了した後、コールされます。
 * (note)
 * 	この関数はプロバイダライブラリがstatic コンパイルされると自動的に
 * 	デバックアウトされます。従って、この関数には、
 * 	    * dlclose された時だけ実施する特別な終了処理
 * 	以外の処理を"書いてはなりません"。
 */
#if defined(DIVY_USE_GNU_LD)
static void _misc_util_fini(void)
#else
void _fini(void)
#endif	/* DIVY_USE_GNU_LD */
{
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"misc(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
	return;
}
#endif	/* DIVY_LIB_DYNAMIC_EXPORT */

/**
 * 初期化ハンドラ
 * (note)
 * 	このハンドラは、Apacheモジュールの一部として動作する場合には
 * 	post_config ステージ(まだChildも起動していないステージ) で
 * 	呼び出されます。
 *
 * 	Child 起動前に実施しなければならない処理や、pconf を使った
 * 	処理を記述できます。
 * 	Child_init でコールバックを受け取るには、このハンドラの
 * 	中で関数divy_hook_child_init を使ってコールバックハンドラを
 * 	レジスタして下さい。
 *
 * @pconf apr_pool_t * コンフィグプール
 * @return int 処理ステータス
 * 	DIVY_LIB_ST_OK  : 正常
 * 	DIVY_LIB_ST_ERR : 失敗
 */
static int _misc_util_init_func(apr_pool_t *pconf)
{
	/* get_miscfunction ハンドラへの登録 */
	divy_hook_get_miscfunction(_get_miscfunctions,
					NULL, NULL, APR_HOOK_MIDDLE);
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"misc(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
	return DIVY_LIB_ST_OK;
}

/**
 * 指定されたproviderType が自身のDBプロバイダを示していた時、
 * 自身のDbDataSource を生成してdbds に返却する。
 *
 * @param p apr_pool_t *
 * @param func_t_name const char * 関数タイプの名前
 * @param func_p void ** func_t_name が示す関数のポインタ
 * @return
 * 	MISC_OK       : func_t_name が示すタイプを実装していた
 * 	MISC_DECLINED : func_t_name が示すタイプを実装していない
 */
static int _get_miscfunctions(apr_pool_t *p,
				const char *func_t_name, void **func_p)
{
	*func_p = NULL;	/* 初期化 */

	/*
	 * func_t_name を定義しているか?
	 */
	if (IS_EMPTY(func_t_name) ||
		strcmp(func_t_name, "encode_str_t") != 0) {
		return MISC_DECLINED;
	}

	/*
	 * 関数へのポインタを渡す
	 */
	*func_p = (encode_str_t) _encode_str;

	return MISC_OK;
}

