/**
 * $Id$
 *
 * tf_ml_default.c
 *
 */

/* Apache header files */
#include "apr.h"
#include "apr_strings.h"
#include "apr_lib.h"
#include "apr_tables.h"
#include "apr_pools.h"
#include "apr_errno.h"
#include "apr_network_io.h"
#include "apr_time.h"
#include "apr_base64.h"
#include "apr_portable.h"

#include "openssl/ssl.h"
#include "openssl/err.h"

#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif	/* APR_HAVE_STDLIB_H */

#if defined(DIVY_DEBUG_PROVIDER_LOADER) && APR_HAVE_UNISTD_H
#include <unistd.h>
#endif	/* defined(DIVY_DEBUG_PROVIDER_LOADER) && APR_HAVE_UNISTD_H */

/* document management header files */
#include "tf_ml_default.h"
#include "util_common.h"
#include "tf_provider.h"

/*--------------------------------------------------------------
  Define fixed value
  --------------------------------------------------------------*/
#define DIVY_MAIL_DATA_TERMINATE_CHAR "."

typedef enum __ml_connect_status {
	TF_ML_DEFAULT_NOREADY	= 0x01,
	TF_ML_DEFAULT_NOEHLO    = 0x02
}
ml_connect_status;

/**
 * このモジュールが扱うメールプロバイダの種類。
 * 他のメールプロバイダと重複してはならない.
 */
#define DEFAULT_ML_PROVIDER_TYPE "default"

enum __ml_rcpt_stat {
	ML_RCPT_STAT_NONE	= 0x00,
	ML_RCPT_STAT_OK 	= 0x01,
	ML_RCPT_STAT_FAIL 	= 0x02,
	ML_RCPT_STAT_NEXT	= 0x04 
};
typedef enum __ml_rcpt_stat	ml_rcpt_stat;

/*--------------------------------------------------------------
  Define incomplete type structure
  --------------------------------------------------------------*/

struct __ml_dsContext {
};

struct __ml_sesionCntxt {
	/* Request_rec から取得したプール */
	apr_pool_t	*pool;
	/* ソケット */
	apr_socket_t	*socket;
        SSL_CTX         *ssl_ctx;
        SSL             *ssl;
	/* 状態 */
	int		status;
};

struct __ml_messageCntxt {

#define ML_RCPT_TO 	0x01
#define ML_RCPT_CC 	0x02
#define ML_RCPT_BCC 	0x04
	int		rcpt_status;

	MlAddress	*currentaddr;
};

/*--------------------------------------------------------------
  Prototype Declare provider function
  --------------------------------------------------------------*/
/*
 * These functions are used for MlDataSource structure.
 */
static MlSession * default_mlds_getMlSession(MlDataSource *mlds, 
				const MlProperty *mlprop, apr_pool_t *p);
static int  default_mlds_getCode(const MlDataSource *mlds);
static char * default_mlds_getNativeCode(const MlDataSource *mlds);
static char * default_mlds_getMsg(const MlDataSource *mlds);

/*
 * These functions are used for MlSession structure.
 */
static int default_mlsession_send(MlSession *mlsession, MlMessage *mlmsg);
static int default_mlsession_getCode(const MlSession *mlsession);
static char * default_mlsession_getNativeCode(const MlSession *mlsession);
static char * default_mlsession_getMsg(const MlSession *mlsession);




/*--------------------------------------------------------------
  Declare private functions
  --------------------------------------------------------------*/
static int _divy_ml_connect(MlSession *session);
static int _divy_ml_set_socket_timeout(MlSession *session, int which, int timeout);
static int _divy_ml_cmd_ehlo(MlSession *session);
static int _divy_ml_cmd_starttls(MlSession *session);
static int _divy_ml_cmd_auth(MlSession *session);
static int _divy_ml_cmd_mail(MlSession *session, MlMessage *mlmsg);
static ml_rcpt_stat _divy_ml_cmd_rcpt(MlSession *session, MlMessage *mlmsg);
static ml_rcpt_stat _divy_ml_cmd_rcpt_address(MlSession *session, MlMessage *msg, int rcpt_type);
static int _divy_ml_cmd_data(MlSession *session, MlMessage *mlmsg);
static int _divy_ml_cmd_rset(MlSession *session);
static int _divy_ml_cmd_quit(MlSession *session);
static int _divy_ml_send_cmd(MlSession *session, const char *cmdline);
static char * _divy_ml_socket_readline(MlSession *session, apr_pool_t *pool);
static int  _divy_ml_read_response(MlSession *session, MlResponse **response);
static void _divy_ml_socket_close(MlSession *session);
static int _divy_ml_start_ssl(MlSession *session);
static int  _divy_ml_delivery(MlSession *mlsession, MlMessage *mlmsg);
static void _divy_ml_set_smtp_extention(MlSession *session, MlResponse *response);

static void init_mlds_method(MlDataSource *mlds);
static void init_mlsession_method(MlSession *mlsession);

#if defined(DIVY_LIB_DYNAMIC_EXPORT) && defined(DIVY_USE_GNU_LD)
static void _ml_default_init(void) __attribute__ ((constructor));
static void _ml_default_fini(void) __attribute__ ((destructor));
#endif	/* DIVY_LIB_DYNAMIC_EXPORT && DIVY_USE_GNU_LD */
static int _ml_default_init_func(apr_pool_t *pconf);
static int _ml_default_create_mldatasource(apr_pool_t *p,
					const char *providerType,
					MlDataSource **mlds);

/*--------------------------------------------------------------
  Define provider library hook structure
  --------------------------------------------------------------*/
DIVY_PROVIDER_HOOK_STRUCT(ml,default,
		_ml_default_init_func,		/* 初期化ハンドラ */
		divy_lib_nop_fini_func);	/* 終了ハンドラ(nop) */

/*--------------------------------------------------------------
  Define provider function
  --------------------------------------------------------------*/

/**
 * セッション構造体を作成する
 *
 * @param mlds	MlDataSource *
 * @param mlds	MlProperty const *
 * @param p	apr_pool_t *
 * @return session MlSession *
 */
static MlSession * default_mlds_getMlSession(MlDataSource *mlds, 
				const MlProperty *mlprop, apr_pool_t *p)
{

	MlSession 	*session;
       
	/* セッションの構造体を割り当てる */
	session	              = apr_pcalloc(p, sizeof(*session));
	session->mlprop       = (MlProperty *)mlprop;
	session->scntxt       = apr_pcalloc(p, sizeof(MlSessionCntxt));
	session->scntxt->pool = p;
	session->mlds         = mlds;
	session->__msg        = apr_pcalloc(p, sizeof(char) * ML_MSGLEN);
	session->__code	      = ML_SUCCESS;
	session->__nativecode = apr_pcalloc(p, sizeof(char) * ML_MSGLEN);

	/* 関数の実体を提供する */
	init_mlsession_method(session);

	return session;
}

/**
 * データソースのエラーを取得する
 * @param mlds	MlDataSource const *
 * @return __code char *
 */
static int  default_mlds_getCode(const MlDataSource *mlds)
{
	return mlds->__code;
}

/**
 * データソースのNativeエラーを取得する
 * @param mlds	MlDataSource const *
 * @return __nativecode char *
 */
static char * default_mlds_getNativeCode(const MlDataSource *mlds)
{
	return mlds->__nativecode;
}

/**
 * データソースのエラーメッセージを取得する
 * 
 * @param mlds	MlDataSource const *
 * @return msg char *
 */
static char * default_mlds_getMsg(const MlDataSource *mlds)
{
	return mlds->__msg;
}

/**
 * メールを送信するメインルーチン
 * 一人ずつ処理を行う
 *
 * @param mlsession *MlSession
 * @param mlmsg     *MlMessage
 * @return int      ML_SUCCESS | ML_ERROR
 */
static int default_mlsession_send(MlSession *mlsession, MlMessage *mlmsg)
{
    int rt = ML_SUCCESS;
    MlMessage *m = NULL;
    apr_interval_time_t msec = apr_time_from_msec(100);

    memset(mlsession->__msg , '\0', sizeof(char) * ML_MSGLEN);

    if (mlsession == NULL) {
        sprintf(mlsession->__msg, "mlsession is NULL");
        return ML_ERROR;
    }

    // サーバに接続
    if ((rt = _divy_ml_connect(mlsession)) != ML_SUCCESS) {
        return ML_ERROR;
    }

    // SMTPS: 接続直後に即TLS
    if (mlsession->__extention & DIVY_ML_EX_IMMEDIATE_SSL) {
        if (_divy_ml_start_ssl(mlsession) != ML_SUCCESS) {
            _divy_ml_socket_close(mlsession);
            return ML_ERROR;
        }

        // TLS後、EHLO
        if ((rt = _divy_ml_cmd_ehlo(mlsession)) != ML_SUCCESS) {
            _divy_ml_socket_close(mlsession);
            return ML_ERROR;
        }

    } else {
        // 通常SMTP: EHLO → STARTTLS → TLS → 再EHLO

        if ((rt = _divy_ml_cmd_ehlo(mlsession)) != ML_SUCCESS) {
            _divy_ml_socket_close(mlsession);
            return ML_ERROR;
        }

        if ((mlsession->__extention & DIVY_ML_EX_STARTTLS) && 
				atoi(mlsession->mlprop->hostport) != 25) {
            if ((rt = _divy_ml_cmd_starttls(mlsession)) != ML_SUCCESS) {
                _divy_ml_socket_close(mlsession);
                return ML_ERROR;
            }

            if ((rt = _divy_ml_cmd_ehlo(mlsession)) != ML_SUCCESS) {
                _divy_ml_socket_close(mlsession);
                return ML_ERROR;
            }
        }
    }

    // AUTH
    if ((mlsession->__extention & DIVY_ML_EX_AUTH) &&
        mlsession->mlprop->username && mlsession->mlprop->password) {

        if ((rt = _divy_ml_cmd_auth(mlsession)) != ML_SUCCESS) {
            _divy_ml_socket_close(mlsession);
            return ML_ERROR;
        }
    }

    // メッセージ送信
    for (m = mlmsg; m != NULL; m = m->next) {
        if (m->from_addr == NULL) continue;

        if ((_divy_ml_delivery(mlsession, m)) != ML_SUCCESS) {
            _divy_ml_socket_close(mlsession);
            return ML_ERROR;
        }

        apr_sleep(msec);
    }

    // QUIT
    if ((rt = _divy_ml_cmd_quit(mlsession)) != ML_SUCCESS) {
        _divy_ml_socket_close(mlsession);
        return ML_ERROR;
    }

    _divy_ml_socket_close(mlsession);
    return ML_SUCCESS;
}

static int  default_mlsession_getCode(const MlSession *mlsession)
{
	return mlsession->__code;
}

static char * default_mlsession_getNativeCode(const MlSession *mlsession)
{
	return mlsession->__nativecode;
}

static char * default_mlsession_getMsg(const MlSession *mlsession)
{
	return mlsession->__msg;
}

/*--------------------------------------------------------------
  Define public functions
  --------------------------------------------------------------*/

/**
 * メールのデータソースを作成する。
 *
 * @param spool apr_pool_t *	サーバプール
 *
 */
DIVY_DECLARE(MlDataSource *) default_mlds_createMlDs(apr_pool_t *spool)
{

	apr_pool_t	*sp  = spool;
	MlDataSource    *ds  = NULL;

	ds               = apr_pcalloc(sp, sizeof(MlDataSource));
	ds->dscontxt     = apr_pcalloc(sp, sizeof(MldsCntxt));
	ds->__msg        = apr_pcalloc(sp, sizeof(char) * ML_MSGLEN);
	ds->__nativecode = apr_pcalloc(sp, sizeof(char) * ML_MSGLEN);

	/* 関数の実体を登録する */
	init_mlds_method(ds);

	return ds;
}

/*--------------------------------------------------------------
  Define private functions
  --------------------------------------------------------------*/

/**
 * サーバとの接続を確立してsocketに状態を保持する。
 *
 * @param session 	MlSession **
 * @return 処理ステータス
 *
 */
static int _divy_ml_connect(MlSession *session)
{

	apr_status_t 	st;
	apr_sockaddr_t	*sockaddr_dest  = NULL;
	apr_pool_t	*p	  = session->scntxt->pool;
	apr_socket_t	*socket	  = NULL;
	MlProperty	*mlprop   = session->mlprop;
	int		int_port  = 0;
	apr_port_t	port;

	if (mlprop == NULL) {
		sprintf(session->__msg, "Failed to get mail mlprop.");
		return ML_ERROR;
	}

	/*
	 * ##### mlpropの中身も調べる必要あります。
	 */
	int_port = atoi(mlprop->hostport);
	port     = (apr_port_t)int_port;

	/* 送り先のサーバ（SMTPサーバ）の情報を設定する	*/
	if ((st = apr_sockaddr_info_get(&sockaddr_dest, mlprop->hostname,
					APR_UNSPEC,
				       	port,
				       	0, p)) != APR_SUCCESS) {
		sprintf(session->__msg, "Failed to get mailhost addressinfo.( hostname = %s )",
								REPLACE_NULL(mlprop->hostname));
		return ML_ERROR;
	}

	/* socketを作成する */
	if ((st = apr_socket_create(&socket, sockaddr_dest->family,
					SOCK_STREAM, APR_PROTO_TCP, p)) != APR_SUCCESS) {
		sprintf(session->__msg, "Failed to creating socket (hostname = %s , port = %s)",
								mlprop->hostname, mlprop->hostport);
			
		return ML_ERROR;
	}

	/* CONNECT */
	if ((st = apr_socket_connect(socket, sockaddr_dest)) != APR_SUCCESS) {
		sprintf(session->__msg, "Failed to connect mail server (hostname = %s)",
			       							mlprop->hostname);
		sprintf(session->__nativecode, "%d", st);
		apr_socket_close(socket);
		return ML_ERROR;
	}

	/* 作成されたソケットをMlSession構造体に設定する */
	session->scntxt->socket = socket;

    if (port == 465) {
        // ここではまだSSL_connectしない → 呼び出し元で行う方が良い
        session->__extention |= DIVY_ML_EX_IMMEDIATE_SSL;
    }

	return ML_SUCCESS;

}

/**
 * プロトコルのタイムアウトを設定する
 * タイムアウトする場所を指定してそのタイムアウトをapr_time_tで設定する。
 * timeoutが0の場合は、デフォルトのタイムアウトを設定する。
 * 
 * @param mlsock	apr_socket_t *
 * @param which		int
 * @param timeout	apr_time_t 
 */

static int _divy_ml_set_socket_timeout(MlSession *session,
	       			int which,
				int timeout)
{
	apr_socket_t	*socket = session->scntxt->socket;
	apr_time_t	mintime = 0L;
	apr_time_t	settime	= 0L;

	/*===========================================================
	 * ##### FIXME
	 * ##### RFC 2822のタイムアウトの実装を地味に守っていないかも
	 * ##### しれません
	 * ==========================================================
	 */

	/*
	 * タイムアウトをかける場所をデフォルト値を取得する
	 */
	switch (which) {
		case CMD_HELO:
		case CMD_EHLO:
			mintime = TIMEOUT_GREETING_DEFAULT;
			break;

                case CMD_MAIL:
                        mintime = TIMEOUT_ENVELOPE_DEFAULT;
                        break;
                case CMD_AUTH:
                        mintime = TIMEOUT_ENVELOPE_DEFAULT;
                        break;
                case CMD_STARTTLS:
                        mintime = TIMEOUT_ENVELOPE_DEFAULT;
                        break;

		case CMD_DATA:
			mintime = TIMEOUT_DATA_DEFAULT;
			break;

		case CMD_RCPT:
			mintime = TIMEOUT_TRANSFER_DEFAULT;
			break;

		case CMD_QUIT:
			mintime = TIMEOUT_DATAQUIT_DEFAULT;
			break;

		default:
			sprintf(session->__msg, "Failed to SMTP Timeout setting.");
			return ML_ERROR;
	}

	/*
	 * デフォルトより大きいタイムアウトは設定させない
	 */
	settime = mintime < timeout ? mintime : timeout;

	/*
	 * タイムアウトの設定を行う。
	 * 設定ができなかった場合ログを出力して終了とします。
	 */
	if ((apr_socket_timeout_set(socket, apr_time_from_sec(settime)))
		       					      != APR_SUCCESS) {
		sprintf(session->__msg, "Failed to socketimeout.socket close ( Timeout = %"APR_TIME_T_FMT" )", apr_time_from_sec(timeout));

		apr_socket_close(socket);
		return ML_ERROR;
	}

	return ML_SUCCESS;
}

/**
 * ESMTPコマンド( EHLO )
 *
 * @param session MlSession *
 * @return 結果ステータス int
 *
 */
static int _divy_ml_cmd_ehlo(MlSession *session)
{

	apr_pool_t	*p	   = session->scntxt->pool;
	const char	*localhost = session->mlprop->localhost;
	MlResponse	*res_conn  = NULL;
	MlResponse	*res_ehlo  = NULL;

	int		rt	   = ML_ERROR;
	const char	*query	   = NULL;

	/* タイムアウトの設定	*/
	if ((_divy_ml_set_socket_timeout(session, CMD_EHLO,
				   TIMEOUT_GREETING_DEFAULT)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	/* EHLOコマンドの作成	*/
	query = apr_psprintf(p, "EHLO %s%s", localhost, DIVY_CRLF);

	if ((rt = _divy_ml_send_cmd(session, query)) != ML_SUCCESS) {
		sprintf(session->__msg, "Failed to EHLO command to server.");
		return ML_ERROR;
	}

	/*
	 * SMTPの処理メッセージの取得
	 * 通常このメッセージはサーバが生きていることだけを確認するだけです
	 */
	if ((rt = _divy_ml_read_response(session, &res_conn)) != ML_SUCCESS) {
		/* ERROR処理が必要です */
		/* サーバが確認できなかったエラーを出力する */
		return ML_ERROR;
	}

	/* サーバレスポンスの確認 2XX以外はエラー */
	if (session->__nativecode != NULL && session->__nativecode[0] != '2') {
		return ML_ERROR;
	}

	/*
	 * EHLOメッセージの取得
	 */
	if ((rt = _divy_ml_read_response(session, &res_ehlo)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	/* サーバレスポンスの確認 2XX以外はエラー */
	if (session->__nativecode != NULL && session->__nativecode[0] != '2') {
		return ML_ERROR;
	}

	/*
	 * EHLOコマンドは戻りにサーバがSMTP拡張されている
	 * 情報を戻してきます。
	 * 後のコマンド操作で利用できるようにセッション構造体
	 * に値をセットしておきます。
	 */

	_divy_ml_set_smtp_extention(session, res_ehlo);

	return ML_SUCCESS;
}

static int _divy_ml_start_ssl(MlSession *session)
{
	SSL_library_init();
	SSL_load_error_strings();

	session->scntxt->ssl_ctx = SSL_CTX_new(SSLv23_client_method());
	if (session->scntxt->ssl_ctx == NULL) {
			return ML_ERROR;
	}
	SSL_CTX_set_options(session->scntxt->ssl_ctx,
					SSL_OP_NO_SSLv2 | SSL_OP_NO_SSLv3);

	session->scntxt->ssl = SSL_new(session->scntxt->ssl_ctx);
	if (session->scntxt->ssl == NULL) {
			SSL_CTX_free(session->scntxt->ssl_ctx);
			session->scntxt->ssl_ctx = NULL;
			return ML_ERROR;
	}

	SSL_set_tlsext_host_name(session->scntxt->ssl,
					session->mlprop->hostname);

	apr_os_sock_t sd;
	apr_os_sock_get(&sd, session->scntxt->socket);
	SSL_set_fd(session->scntxt->ssl, (int)sd);

	int ret;
	do {
		ret = SSL_connect(session->scntxt->ssl);
		if (ret <= 0) {
			int err = SSL_get_error(session->scntxt->ssl, ret);
			if (err == SSL_ERROR_WANT_READ || err == SSL_ERROR_WANT_WRITE) {
					continue;
			}
			else {
				SSL_free(session->scntxt->ssl);
				SSL_CTX_free(session->scntxt->ssl_ctx);
				session->scntxt->ssl = NULL;
				session->scntxt->ssl_ctx = NULL;
				return ML_ERROR;
			}
		}
	} while (ret <= 0);

	return ML_SUCCESS;
}

static int _divy_ml_cmd_starttls(MlSession *session)
{
	apr_pool_t      *p      = session->scntxt->pool;
	const char      *query  = NULL;
	int             rt     = ML_ERROR;
	MlResponse      *res_tls = NULL;

	if ((_divy_ml_set_socket_timeout(session, CMD_STARTTLS,
									TIMEOUT_GREETING_DEFAULT)) != ML_SUCCESS) {
			return ML_ERROR;
	}

	query = apr_psprintf(p, "STARTTLS%s", DIVY_CRLF);
	if ((rt = _divy_ml_send_cmd(session, query)) != ML_SUCCESS) {
			return ML_ERROR;
	}

	if ((rt = _divy_ml_read_response(session, &res_tls)) != ML_SUCCESS) {
			return ML_ERROR;
	}
	if (session->__nativecode == NULL || session->__nativecode[0] != '2') {
			return ML_ERROR;
	}

	if (_divy_ml_start_ssl(session) != ML_SUCCESS) {
			return ML_ERROR;
	}

	return ML_SUCCESS;
}

static int _divy_ml_cmd_auth(MlSession *session)
{
    apr_pool_t      *p    = session->scntxt->pool;
    const char      *user = session->mlprop->username;
    const char      *pass = session->mlprop->password;
    char            *auth_str, *b64;
    const char      *query;
    int             rt;
    MlResponse      *res_auth = NULL;

    if (!user || !pass) {
        sprintf(session->__msg, "Auth information is missing.");
        return ML_ERROR;
    }

    if ((_divy_ml_set_socket_timeout(session, CMD_AUTH, TIMEOUT_ENVELOPE_DEFAULT)) != ML_SUCCESS) {
        return ML_ERROR;
    }

    // \0username\0password 形式
    int len = 2 + strlen(user) + strlen(pass); // includes two nulls
    auth_str = apr_palloc(p, len + 1);
    auth_str[0] = '\0';
    strcpy(auth_str + 1, user);
    auth_str[1 + strlen(user)] = '\0';
    strcpy(auth_str + 2 + strlen(user), pass);

    // base64 encode
    b64 = apr_palloc(p, apr_base64_encode_len(len) + 1);
    apr_base64_encode(b64, auth_str, len);

    query = apr_psprintf(p, "AUTH PLAIN %s%s", b64, DIVY_CRLF);

    if ((rt = _divy_ml_send_cmd(session, query)) != ML_SUCCESS) {
        return ML_ERROR;
    }

    if ((rt = _divy_ml_read_response(session, &res_auth)) != ML_SUCCESS) {
        sprintf(session->__msg, "Failed to send AUTH PLAIN.");
        return ML_ERROR;
    }

    if (!session->__nativecode || session->__nativecode[0] != '2') {
		LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_NET, 
					"AUTH PLAIN failed %s", session->__nativecode);
        return ML_ERROR;
    }

    return ML_SUCCESS;
}

/**
 * SMTPコマンド( MAIL )
 *
 * @param session MlSession *
 * @return 結果ステータス int
 *
 */
static int _divy_ml_cmd_mail(MlSession *session, MlMessage *mlmsg)
{
	apr_pool_t      *p         = session->scntxt->pool;
	const char	*query     = NULL;
	int		rt         = 0;
	MlAddress	*address   = NULL;
	MlResponse	*response  = NULL;

	/* タイムアウトの設定 */
	if ((_divy_ml_set_socket_timeout(session, CMD_MAIL,
				   TIMEOUT_ENVELOPE_DEFAULT)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	/*
	 * セッションとメッセージが設定されていればfromユーザを
	 * address構造体へ設定する
	 */
	if (session == NULL || mlmsg == NULL) {
		sprintf(session->__msg, "Failed to function. socket or msg is NULL");
		return ML_ERROR;
	}
	address = mlmsg->from_addr;

	if (address == NULL || address->addr == NULL || *address->addr == '\0') {
		sprintf(session->__msg, "Failed to FROM address "
					 "is NULL or SPACE."
					 "Transmission of mail was "
					 "interrupted.");
		return ML_ERROR;
	}

	/*
	 * From はMlAddressでリスト構造なってはいるがSMTP規則ではMAILコマンド
	 * 発行のたびリセット発行される為、何度まわしても意味がない。
	 * ここでは一番始めのアドレスだけを設定する。
	 *
	 * NOTE :
	 * 発行のたびにリセットされるはずであるが、どうもうまくいきません。
	 * その為に、なにかあったら必ずリセットする事をお勧めします。
	 */
	query   = apr_psprintf(p, "MAIL FROM: <%s>", address->addr);

	/*
	 * CharsetがUTF8の場合で、尚且つ
	 * 8BITMIME BODYが利用できるならばFROMの後ろに文字列を追加する
	 */
	if ((strcasecmp(mlmsg->charset, ML_CHARSET_UTF8)  == 0 ||
	     strcasecmp(mlmsg->charset, ML_CHARSET_UTF_8) == 0) &&
		       		session->__extention & DIVY_ML_EX_8BITMIME) {
		query = apr_pstrcat(p, query, " BODY=8BITMIME", NULL);
	}

	/* 最後はCRLFで終わらせる */
	query = apr_pstrcat(p, query, DIVY_CRLF, NULL);

	if ((rt = _divy_ml_send_cmd(session, query)) != ML_SUCCESS) {
		sprintf(session->__msg, "Failed to MAIL Command to Server");
		return ML_ERROR;
	}

	/*
	 * MAILコマンドサーバレスポンスの取得
	 */
	if ((rt = _divy_ml_read_response(session, &response)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	/* サーバレスポンスの確認 2XX以外はエラー */
	if (session->__nativecode != NULL && session->__nativecode[0] != '2') {
		return ML_ERROR;
	}

	return ML_SUCCESS;
}

/**
 * SMTPコマンド( RCPT )
 * (note)
 * TO,CC,BCCのすべてを処理する関数。RCPTコマンドの基本はTOであるため、
 * 特別扱いとしてTO以外はエラーが起きても正常として返します。
 *
 * @param session MlSession *
 * @param mlmsg	  MlMessage *
 * @return st ml_rcpt_stat
 *
 */
static ml_rcpt_stat _divy_ml_cmd_rcpt(MlSession *session, MlMessage *mlmsg)
{
	apr_pool_t      *p         = session->scntxt->pool;
	ml_rcpt_stat	st	   = ML_RCPT_STAT_NONE;

	/* タイムアウトの設定 */
	if ((_divy_ml_set_socket_timeout(session, CMD_RCPT,
				   TIMEOUT_ENVELOPE_DEFAULT)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	if (session == NULL || mlmsg == NULL) {
		session->__msg = apr_pstrdup(p, "Failed to function. "
						"socket or msg is NULL");
		return ML_RCPT_STAT_FAIL;
	}

	/* いずれかアドレスが入っていない場合は送ることはできないのでエラー */
	if (mlmsg->to_addr == NULL &&  
			mlmsg->cc_addr == NULL && mlmsg->bcc_addr) {
		session->__msg = apr_pstrdup(p,"Failed to TO Address is NULL");
		return ML_RCPT_STAT_FAIL;
	}

	/*
	 * 送信先分ループする
	 * アドレスが存在しなかった場合でも他のユーザへは送信する
	 * to,cc,bccすべてにおいて行なうがすべてのユーザへ送れるか
	 * 保証はできない。その為、メールサーバがエラーを返した場合(552)
	 * 処理を次回に回すようにしてあります。
	 *
	 * -------------------------------------------------------------
	 * TOのアドレスさえうまくいけばCC,BCCはエラーは見逃すようにする。
	 * -------------------------------------------------------------
	 */
	if (mlmsg->msgcntxt->rcpt_status & ML_RCPT_TO) {
		st = _divy_ml_cmd_rcpt_address(session, mlmsg, ML_RCPT_TO);
		if (st == ML_RCPT_STAT_OK)
			mlmsg->msgcntxt->rcpt_status &= ~ML_RCPT_TO;
	}
	/* TOが失敗したら必ずエラー */
	if (st != ML_RCPT_STAT_OK) return st;

	if (mlmsg->msgcntxt->rcpt_status & ML_RCPT_CC) {
		st = _divy_ml_cmd_rcpt_address(session, mlmsg, ML_RCPT_CC);
		if (st == ML_RCPT_STAT_OK)
			mlmsg->msgcntxt->rcpt_status &= ~ML_RCPT_CC;
	}

	if (mlmsg->msgcntxt->rcpt_status & ML_RCPT_BCC) {
		st = _divy_ml_cmd_rcpt_address(session, mlmsg, ML_RCPT_BCC);
		if (st == ML_RCPT_STAT_OK)
			mlmsg->msgcntxt->rcpt_status &= ~ML_RCPT_BCC;
	}

	return ML_RCPT_STAT_OK;
}

/**
 * ユーザ毎にRCPTコマンドを発行する。
 *
 * @param session	MlSession *
 * @param msg		MlMessage *
 * @param rcpt_type	int
 *
 */
static ml_rcpt_stat _divy_ml_cmd_rcpt_address(MlSession *session,
	       					MlMessage *msg, int rcpt_type)
{

	MlAddress	*a 	   = NULL;
	apr_pool_t      *p         = session->scntxt->pool;
	/* sendintervalはメールの送信をずらす為に導入された
	   値はミリセカンドとして記載する事
	 */
	int sendinterval		   = session->mlprop->interval;
	const char	*query     = NULL;
	int		success    = 0;
	int		rt	   = 0;
	MlResponse	*response  = NULL;

	if (p == NULL || msg == NULL) return ML_ERROR;

	if (msg->msgcntxt->currentaddr == NULL) {
		switch(rcpt_type) {
			case ML_RCPT_TO:
				a = msg->to_addr;
				break;
			case ML_RCPT_CC:
				a = msg->cc_addr;
				break;
			case ML_RCPT_BCC:
				a = msg->bcc_addr;
				break;
			default:
				return ML_RCPT_STAT_FAIL;
		}
	}
	else {
		a =  msg->msgcntxt->currentaddr;
	}

	for (; a != NULL; a = a->next) {
		if (a->addr == NULL || *a->addr == '\0') {
			/*
			 * ここにsendintervalがある理由は下のsendintervalが
			 * ある部分でアドレスを消している行為があるためです。
			 * もちろん本当にアドレスが無い場合もあるため、sendinterval
			 * に数字が入っている場合のみSleepします。
			 */
			if (sendinterval)
				apr_sleep(sendinterval);
			/* アドレスが存在しないだけなのでスキップ */
			continue;
		}

		query = apr_psprintf(p, "RCPT TO: <%s>%s",
				a->addr, DIVY_CRLF);
		if ((rt = _divy_ml_send_cmd(session, query)) != ML_SUCCESS) {
			sprintf(session->__msg, "Failed to RCPT Command to Server");
			return ML_RCPT_STAT_FAIL;
		}

		/*
		 * RCPTコマンドサーバレスポンスの取得
		 */
		if ((rt = _divy_ml_read_response(session,
						&response)) != ML_SUCCESS) {
			return ML_RCPT_STAT_FAIL;
		}

		/* サーバレスポンスの確認 */
		/* サーバメッセージはメールサーバによって不明確なところが
		 * 多い。
		 *
		 * 2xx系は基本的に正常として処理する。
		 * 450の場合はpostfixはuser unknownということが分かった為
		 * とりあえずSKIPする(但し正常とは認めない）
		 *
		 */
		if (session->__nativecode != NULL) {
			if (session->__nativecode[0] == '2') {
				/* 成功した */
				success = 1;
				if (sendinterval) {
					a->addr = NULL;
					return ML_RCPT_STAT_NEXT; /* 次回に持ち越す */
				}
			}
			else if (session->__nativecode[0] == '5' &&
				 session->__nativecode[1] == '5' &&
				 session->__nativecode[2] == '2') {
				/* ストレージオーバーならここで諦める */
				msg->msgcntxt->currentaddr = a;
				return ML_RCPT_STAT_NEXT; /* 次回に持ち越す */
			}
			else if (session->__nativecode[0] == '4' && 
				 session->__nativecode[0] == '5' &&
				 session->__nativecode[0] == '0') {
				 /* user unknownの場合はここにくる (postfix)
				 * 処理の継続を許可するがsuccess成功には
				 * 含めません
				 */
				continue;
			}
			else {
				return ML_RCPT_STAT_FAIL;
			}

			LERRLOG3(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_NET,
					"SMTP RCPT [%s] code=[%s] msg=[%s]", a->addr, session->__nativecode, IS_EMPTY(session->__msg)?"OK":session->__msg);
		}
		else {
			return ML_RCPT_STAT_FAIL;
		}
	}

	/*
	 * ここまできたということはすべてのアドレスが処理できた
	 * という意味であるため、未処理のアドレスもすべてクリアする
	 */
	if (msg->msgcntxt->currentaddr != NULL)
		msg->msgcntxt->currentaddr = NULL;

	return (success) ? ML_RCPT_STAT_OK : ML_RCPT_STAT_FAIL;
}

/**
 * SMTPコマンド( DATA )
 *
 * @param session MlSession *
 * @return 結果ステータス int
 *
 */
static int _divy_ml_cmd_data(MlSession *session, MlMessage *mlmsg)
{
	apr_pool_t      *p         = session->scntxt->pool;
	char		*query	   = NULL;
	int		rt	   = ML_SUCCESS;
	MlResponse	*response  = NULL;
	const apr_array_header_t *arr;
	const apr_table_entry_t	 *elts;
	int		i	   = 0;
	char		*data	   = NULL;
	char		*body	   = NULL;
	int 		first_set, to_cnt, cc_cnt, bcc_cnt;

	/* タイムアウトの設定 */
	if ((_divy_ml_set_socket_timeout(session, CMD_DATA,
				   TIMEOUT_DATA_DEFAULT)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	query = apr_psprintf(p, "DATA%s", DIVY_CRLF);

	if ((rt = _divy_ml_send_cmd(session, query)) != ML_SUCCESS) {
		sprintf(session->__msg, "Failed to DATA Command to Server");
		return ML_ERROR;
	}
	/*
	 * DATAコマンドサーバレスポンスの取得
	 */
	if ((rt = _divy_ml_read_response(session, &response)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	/* サーバレスポンスの確認 3XX以外はエラー */
	if (session->__nativecode != NULL && session->__nativecode[0] != '3') {
		return ML_ERROR;
	}

	/* タイムアウトの設定 */
	if ((_divy_ml_set_socket_timeout(session, CMD_DATA,
				   TIMEOUT_DATAQUIT_DEFAULT)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	//TRACE(p);

	/* ヘッダの設定 */
	arr  = apr_table_elts(mlmsg->headers);
	elts = (const apr_table_entry_t *)arr->elts;
	for (i = 0; i < arr->nelts; ++i) {
		if (elts[i].key == NULL)
			break;

		/*
		 * To,Cc,Bccの処理とサブジェクトの処理はほとんど同じなのですが
		 * ヘッダの送信方法が微妙に違いました。
		 * Toは複数人を送る為人の区切りにカンマを必要とします。
		 * しかし、subjectはそういうものはいりません。
		 * その差だけです。
		 *
		 */
		if (strcmp(elts[i].key, "To") == 0) 
			continue;
		if (strcmp(elts[i].key, "Cc") == 0)
			continue;
		if (strcmp(elts[i].key, "Bcc") == 0)
			continue;

		if (strcmp(elts[i].key, "Subject") == 0) {
			/* Subjectヘッダ処理 */
			short 	first_set  = 0;
			char	*token	   = NULL;
			char	*tok_cntx  = NULL;
			char    *elts_val;

			elts_val = apr_pstrdup(p, elts[i].val);
			/* カンマが存在しない場合は一件だけの処理 */
			if (divy_strstr(elts_val, ",") == NULL) {
				data = apr_psprintf(p, "%s: %s%s",
					       	elts[i].key,
						elts[i].val,
						DIVY_CRLF);
				if ((rt = _divy_ml_send_cmd(session, data)) !=
								ML_SUCCESS) {
					sprintf(session->__msg,
							"Failed to DATA Command"
							" to Server");
					return ML_ERROR;
				}
				continue;
			}

			/* トークンが存在するのは複数処理 */
			while ((token = apr_strtok(elts_val, ",", &tok_cntx)) != NULL)
		       	{
				elts_val = NULL;
				if (first_set == 0) {
					data = apr_psprintf(p, "%s: %s%s",
						       	elts[i].key, token,
						       	DIVY_CRLF);
				}
				else {
					/* (ここが違う) */
					data = apr_psprintf(p, " %s%s",
							token,
						       	DIVY_CRLF);
				}
				/* Send DATA Command */
				if ((rt = _divy_ml_send_cmd(session, data)) !=
								ML_SUCCESS) {
					sprintf(session->__msg,
							"Failed to DATA Command"
							" to Server");
					return ML_ERROR;
				}

				first_set = 1;

			} /* while (token = ... */ 

		}
		else {
			/* その他ヘッダ */
			data = apr_psprintf(p, "%s: %s%s", elts[i].key,
				       			   elts[i].val,
				       			   DIVY_CRLF);

			if ((rt = _divy_ml_send_cmd(session, data)) !=
				       				ML_SUCCESS) {
				sprintf(session->__msg,
					       	"Failed to DATA Command"
						" to Server");
				return ML_ERROR;
			}
		}
			
	} /* for (i=0; i<arr->nelts */

	/* To, Cc, Bccヘッダの処理 */
	first_set = to_cnt = cc_cnt = bcc_cnt = 0;
	arr  = apr_table_elts(mlmsg->headers);
	elts = (const apr_table_entry_t *)arr->elts;
	for (i = 0; i < arr->nelts; ++i) {
		char *data = NULL;

		if (elts[i].key == NULL)
			break;

		if (strcmp(elts[i].key, "To") == 0) {
			first_set = to_cnt;
			to_cnt = 1;
		}
		else 
		if (strcmp(elts[i].key, "Cc") == 0) {
			first_set = cc_cnt;
			cc_cnt = 1;
		}
		else 
		if (strcmp(elts[i].key, "Bcc") == 0) {
			first_set = bcc_cnt;
			bcc_cnt = 1;
		}
		else {
			/* 該当しないヘッダは既にやっているので除外 */
			continue;
		}

		if (first_set == 0) {
			data = apr_psprintf(p, "%s: %s%s",
					elts[i].key,
					elts[i].val, DIVY_CRLF);
		}
		else {
			data = apr_psprintf(p, "\t,%s%s",
					elts[i].val, DIVY_CRLF);
		}

		/* Send DATA Command */
		if ((rt = _divy_ml_send_cmd(session, data)) !=
				ML_SUCCESS) {
			sprintf(session->__msg,
					"Failed to DATA Command"
					" to Server");
			return ML_ERROR;
		}
	}

	/* ボディの設定 */
	/* 終端を本文につけて送信する */
	body = apr_psprintf(p, "%s%s%s%s%s",
				DIVY_CRLF,
				mlmsg->body,
				DIVY_CRLF,
				DIVY_MAIL_DATA_TERMINATE_CHAR,
				DIVY_CRLF);

	if ((rt = _divy_ml_send_cmd(session, body)) != ML_SUCCESS) {
		sprintf(session->__msg, "Failed to DATA Command to Server");
			return ML_ERROR;
	}

	/*
 	 * DATAコマンドサーバレスポンスの取得
 	 */
	if ((rt = _divy_ml_read_response(session,
					&response)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	/* サーバレスポンスの確認 2XX以外はエラー */
	if (session->__nativecode != NULL &&
			session->__nativecode[0] != '2') {
		return ML_ERROR;
	}

	return ML_SUCCESS;
}

/**
 * SMTPコマンド( RSET )
 *
 * @param session MlSession *
 * @return 結果ステータス int
 *
 */
static int _divy_ml_cmd_rset(MlSession *session)
{
	apr_pool_t      *p         = session->scntxt->pool;
	int		rt         = ML_SUCCESS;
	const char	*query	   = NULL;
	MlResponse	*response  = NULL;

	/* RSETコマンドの作成してサーバーに送信する	*/
	query = apr_psprintf(p, "RSET%s", DIVY_CRLF);
	if ((rt = _divy_ml_send_cmd(session, query)) != ML_SUCCESS) {
		sprintf(session->__msg, "Failed To RSET Commnd to Searver");
		return ML_ERROR;
	}

	/*
 	 * RSETコマンドサーバレスポンスの取得
 	 */
	if ((rt = _divy_ml_read_response(session,
					&response)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	/* サーバレスポンスの確認 2XX以外はエラー */
	if (session->__nativecode != NULL &&
			session->__nativecode[0] != '2') {
		return ML_ERROR;
	}

	return ML_SUCCESS;
}

/**
 * SMTPコマンド( QUIT )
 *
 * @param session MlSession *
 * @return 結果ステータス int
 *
 */
static int _divy_ml_cmd_quit(MlSession *session)
{
	apr_pool_t	*p	   = session->scntxt->pool;
	int		rt         = 0;
	const char	*query	   = NULL;
	MlResponse	*response  = NULL;

	/* タイムアウトの設定 */
	if ((_divy_ml_set_socket_timeout(session, CMD_QUIT,
				   TIMEOUT_DATAQUIT_DEFAULT)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	/* QUITコマンドの作成してサーバへ送信する */
	query = apr_psprintf(p, "QUIT%s", DIVY_CRLF);
	if ((rt = _divy_ml_send_cmd(session, query)) != ML_SUCCESS) {
		sprintf(session->__msg, "Failed to command QUIT.");
		return ML_ERROR;
	}

	/*
 	 * QUITコマンドサーバレスポンスの取得
 	 */
	if ((rt = _divy_ml_read_response(session,
					&response)) != ML_SUCCESS) {
		return ML_ERROR;
	}

	/* サーバレスポンスの確認 2XX以外はエラー */
	if (session->__nativecode != NULL &&
			session->__nativecode[0] != '2') {
		return ML_ERROR;
	}

	return ML_SUCCESS;
}

/**
 * サーバへ指定されたコマンド文字列を送信する
 * @param socket apr_socket_t *
 * @param cmdline const char *
 * @return 処理結果 (ML_SUCCESS | ML_ERROR)
 *
 */
static int _divy_ml_send_cmd(MlSession *session, const char *cmdline)
{
    apr_size_t total_sent = 0;
    apr_size_t sendlen = strlen(cmdline);
    apr_status_t st;
    apr_interval_time_t delay = apr_time_from_msec(100); // 100ms
    int retry = 10;

    if (!session || !session->scntxt || !session->scntxt->socket || !cmdline || *cmdline == '\0') {
        return ML_ERROR;
    }

    while (total_sent < sendlen && retry-- > 0) {
        int n;

        if (session->scntxt->ssl) {
            n = SSL_write(session->scntxt->ssl, cmdline + total_sent, sendlen - total_sent);
            if (n > 0) {
                total_sent += n;
                continue;
            } else {
                int err = SSL_get_error(session->scntxt->ssl, n);
                if (err == SSL_ERROR_WANT_WRITE || err == SSL_ERROR_WANT_READ) {
                    apr_sleep(delay);
                    continue;
                } else {
                    return ML_ERROR;
                }
            }
        } else {
            apr_size_t thislen = sendlen - total_sent;
            st = apr_socket_send(session->scntxt->socket, cmdline + total_sent, &thislen);
            if (st == APR_SUCCESS) {
                total_sent += thislen;
                continue;
            } else if (APR_STATUS_IS_EAGAIN(st)) {
                apr_sleep(delay);
                continue;
            } else {
                return ML_ERROR;
            }
        }
    }

    return (total_sent == sendlen) ? ML_SUCCESS : ML_ERROR;
}

/**
 * ソケットより一行読み取る(CRLFまでが一行)
 * LFを確認して一行読み取ったことを理解する。
 *
 * @param socket	apr_socket_t *
 * @param pool		apr_pool_t *
 * @return char 	char * (文字列: 正常 / NULL: 失敗);
 *
 */
#define RETRY_INTERVAL_MS 100
#define RETRY_MAX_COUNT 10
static char * _divy_ml_socket_readline(MlSession *session, apr_pool_t *pool)
{
    apr_socket_t *socket = session->scntxt->socket;
    SSL *ssl = session->scntxt->ssl;
    apr_status_t st;
    apr_size_t recvlen = 1;
    char *buffer = apr_pcalloc(pool, sizeof(char) * (ML_MSGLEN + 1));
    char *tmpbuf = buffer;
    int retry_count = 0;

    if (socket == NULL) {
        return NULL;
    }

    do {
        recvlen = 1;
        retry_count = 0;

        while (retry_count < RETRY_MAX_COUNT) {
            if (ssl) {
                int rc = SSL_read(ssl, tmpbuf, recvlen);
                if (rc > 0) {
                    recvlen = rc;
                    st = APR_SUCCESS;
                    break;
                } else {
                    int err = SSL_get_error(ssl, rc);
                    if (err == SSL_ERROR_WANT_READ || err == SSL_ERROR_WANT_WRITE) {
                        apr_sleep(apr_time_from_msec(RETRY_INTERVAL_MS));
                        retry_count++;
                        continue;
                    } else {
                        return NULL;
                    }
                }
            } else {
                st = apr_socket_recv(socket, tmpbuf, &recvlen);
                if (st == APR_SUCCESS) {
                    break;
                } else if (APR_STATUS_IS_EAGAIN(st) || APR_STATUS_IS_EINTR(st)) {
                    apr_sleep(apr_time_from_msec(RETRY_INTERVAL_MS));
                    retry_count++;
                    continue;
                } else {
                    return NULL;
                }
            }
        }

		if (retry_count >= RETRY_MAX_COUNT) {
			snprintf(session->__msg, ML_MSGLEN,
				"Failed to send full message after retries.");
            return NULL;
        }

        if (*tmpbuf == DIVY_LF) {
            tmpbuf++;
            *tmpbuf = '\0';
            break;
        }

        tmpbuf++;

    } while ((tmpbuf - buffer) < ML_MSGLEN);

    return buffer;
}

/**
 * ソケットよりサーバのレスポンスをリストで取得する
 *
 * @param socket apr_socket_t *
 * @param pool   apr_pool_t *
 * @param response MlResponse *
 * @return int ステータス(ML_SUCCESS | ML_ERROR)
 */
static int _divy_ml_read_response(MlSession *session, MlResponse **response)
{

	apr_socket_t	*socket   = session->scntxt->socket;
	apr_pool_t	*p        = session->scntxt->pool;
	MlResponse	*tmp_res  = NULL;
	char 		*line     = NULL;
	int		firstflg  = 0;

	if (socket == NULL) {
		sprintf(session->__msg, "Failed to socket is NULL");
		session->__code = ML_ERROR;
		return ML_ERROR;
	}

	/* レスポンス構造体を作成する */
	*response = apr_pcalloc(p, sizeof(MlResponse));
	tmp_res   = *response;

	/*
	 * 全てのコマンドはこのnativecodeを確認する為に
	 * 初期化しておく
	 */
	memset(session->__nativecode, '\0', ML_MSGLEN);

	/* 
	 * レスポンスを一行ずつ取得してリストを作成する
	 * 結果が\d\d\d-であった場合は繰り返しで取得する
	 */
	do {
		line = _divy_ml_socket_readline(session, p);
		if (line == NULL) {
			return ML_ERROR;
		}

		if (firstflg == 0) {
			tmp_res->line = line;
			firstflg = 1;
		}
		else {
			/* レスポンス構造体を作成する */
			tmp_res->next = apr_pcalloc(p, sizeof(MlResponse));
			tmp_res       = tmp_res->next;
			tmp_res->line = line;
			tmp_res->next = NULL;
		}

	} while (apr_isdigit(line[0]) &&
		 apr_isdigit(line[1]) &&
		 apr_isdigit(line[2]) &&
		 line[3] == '-');

	/*
	 * ステータスをセッション構造体へ設定する
	 * メッセージへはそのままサーバの応答メッセージを設定します
	 * サーバレスポンスはマルチで戻ってきている可能性がある為、
	 * 最後のレスポンスコードを設定する
	 * 最後のメッセージとは
	 *
	 *      \d\d\d (連続の3桁の数字にスペース)のことを指しています。
	 */
	session->__code       = ML_SUCCESS;

	for (tmp_res = *response; tmp_res != NULL; tmp_res = tmp_res->next) {

		if (tmp_res->next == NULL &&
		    apr_isdigit(tmp_res->line[0]) &&
		    apr_isdigit(tmp_res->line[1]) &&
		    apr_isdigit(tmp_res->line[2]) &&
		    tmp_res->line[3] == ' ') {

			/* SMTPエラーの場合のみレスポンスメッセージを入れる*/
			if (tmp_res->line[0] == '4' || tmp_res->line[0] == '5') 
				sprintf(session->__msg, "%s", tmp_res->line);

			memcpy(session->__nativecode, tmp_res->line, 3);
		}
	}

	return ML_SUCCESS;
}

/*
 * ソケットを閉じる
 * 
 * @param socket apr_socket_t *
 *
 */
static void _divy_ml_socket_close(MlSession *session)
{
        if (session->scntxt->ssl) {
                SSL_shutdown(session->scntxt->ssl);
                SSL_free(session->scntxt->ssl);
                session->scntxt->ssl = NULL;
        }
        if (session->scntxt->ssl_ctx) {
                SSL_CTX_free(session->scntxt->ssl_ctx);
                session->scntxt->ssl_ctx = NULL;
        }
        if (session->scntxt->socket != NULL) {
                apr_socket_close(session->scntxt->socket);
                session->scntxt->socket = NULL;
        }

        return;
}

/**
 * メールを送信する単位（人）ごとにメールを送信する。
 *
 * @param session MlSession *
 * @param msmsg   MlMessage *
 * @return 結果 (ML_SUCCESS | ML_ERROR)
 *
 */
static int _divy_ml_delivery(MlSession *mlsession, MlMessage *mlmsg)
{

	int 		rt = ML_SUCCESS;
	ml_rcpt_stat	st = ML_RCPT_STAT_NONE;

	mlmsg->msgcntxt = apr_pcalloc(mlsession->scntxt->pool,
							sizeof(MlMessageCntxt));

	if (mlmsg == NULL) return ML_ERROR;

	/* RCPTを行なうユーザがどれだけいるかチェックする */
	if (mlmsg->to_addr != NULL)
	       	mlmsg->msgcntxt->rcpt_status |= ML_RCPT_TO;
	if (mlmsg->cc_addr != NULL)
	       	mlmsg->msgcntxt->rcpt_status |= ML_RCPT_CC;
	if (mlmsg->bcc_addr != NULL)
	       	mlmsg->msgcntxt->rcpt_status |= ML_RCPT_BCC;

	/*
	 * RCPTは複数件数処理を行なうが、件数が多い場合FROMコマンドから
	 * やり直す必要があります。
	 * ここではRCPTの処理結果に応じて処理を継続するかを調べます
	 *
	 * MAILは以下のフローを守らないとエラーになります
	 * FROM -> RCPT -> DATA
	 */
	do {	
		/*-----------------------------------------------------
		 * MAIL
		 */
		if ((rt = _divy_ml_cmd_mail(mlsession, mlmsg)) != ML_SUCCESS) {
			return ML_ERROR;
		}

		/*-----------------------------------------------------
		 * REPT
		 */
		if ((st = _divy_ml_cmd_rcpt(mlsession, mlmsg))
							== ML_RCPT_STAT_FAIL) {
			/* 一人にRCPTが失敗しても次のメールアドレスは成功の
			 * 可能性があるため、正常として処理を続けるがこの処理
			 * 自体は失敗のためリセット処理だけは行う。
			 */
			LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_NET,
					"Failed to RCPT Cmd. (Reason: %s)", mlsession->__msg);
			if ((rt = _divy_ml_cmd_rset(mlsession)) != ML_SUCCESS) {
				return ML_ERROR;
			}

			return ML_SUCCESS;
		}

		/*-----------------------------------------------------
		 * DATA
		 */
		if ((rt = _divy_ml_cmd_data(mlsession, mlmsg)) != ML_SUCCESS) {
			return ML_ERROR;
		}

	} while(st == ML_RCPT_STAT_NEXT);

	return ML_SUCCESS;
}



/**
 * レスポンスの内容を元にSTMP拡張をセットする
 *
 * @param session MlSession *
 * @param response MlResponse *
 *
 */
static void _divy_ml_set_smtp_extention(MlSession *session,
	       				     MlResponse *response)
{

	MlResponse	*tmp_res = response;

	if (session == NULL || tmp_res == NULL) {
		return;
	}

	/*
	 * SMTP 拡張の応答文字列セットする
	 */
	for (; tmp_res != NULL; tmp_res = tmp_res->next) {
		if (divy_strstr(tmp_res->line, "ENHANCEDSTATUSCODES") != NULL) {
			session->__extention |= DIVY_ML_EX_ENHANCEDSTATUSCODES;
		}
	       	else if (divy_strstr(tmp_res->line, "PIPELINING") != NULL) {
			session->__extention |= DIVY_ML_EX_PIPELINING;
		}
		else if (divy_strstr(tmp_res->line, "DSN") != NULL) {
			session->__extention |= DIVY_ML_EX_DSN;
		}
		else if (divy_strstr(tmp_res->line, "AUTH") != NULL) {
			session->__extention |= DIVY_ML_EX_AUTH;
		}
		else if (divy_strstr(tmp_res->line, "STARTTLS") != NULL) {
			session->__extention |= DIVY_ML_EX_STARTTLS;
		}
		else if (divy_strstr(tmp_res->line, "SIZE") != NULL) {
			session->__extention |= DIVY_ML_EX_SIZE;
		}
		else if (divy_strstr(tmp_res->line, "CHUNKING") != NULL) {
			session->__extention |= DIVY_ML_EX_CHUNKING;
		}
		else if (divy_strstr(tmp_res->line, "BINARYMIME") != NULL) {
			session->__extention |= DIVY_ML_EX_BINARYMIME;
		}
		else if (divy_strstr(tmp_res->line, "8BITMIME") != NULL) {
			session->__extention |= DIVY_ML_EX_8BITMIME;
		}
		else if (divy_strstr(tmp_res->line, "DELIVERBY") != NULL) {
			session->__extention |= DIVY_ML_EX_DELIVERBY;
		}
		else if (divy_strstr(tmp_res->line, "ETRN") != NULL) {
			session->__extention |= DIVY_ML_EX_ETRN;
		}
		else if (divy_strstr(tmp_res->line, "XUSR") != NULL) {
			session->__extention |= DIVY_ML_EX_XUSR;
		}
	}

	return;

}

/**
 * MlDataSourceの中で宣言された関数へのポインタに実体の関数を提供する.
 *
 * @param dbds DbDataSource*
 *
 */
static void init_mlds_method(MlDataSource *mlds)
{
	mlds->getMlSession  = default_mlds_getMlSession;
	mlds->getCode       = default_mlds_getCode;
	mlds->getNativeCode = default_mlds_getNativeCode;
	mlds->getMsg        = default_mlds_getMsg;

	return;
}

/**
 * MlSessionの中で宣言された関数へのポインタに実体の関数を提供する.
 *
 * @param mlsession * 
 */
static void init_mlsession_method(MlSession *mlsession)
{
	mlsession->send          = default_mlsession_send;
	mlsession->getCode       = default_mlsession_getCode;
	mlsession->getNativeCode = default_mlsession_getNativeCode;
	mlsession->getMsg        = default_mlsession_getMsg;
	return;
}

/*-----------------------------------------------------------------------------
  Define provider library handler
 *---------------------------------------------------------------------------*/
#if defined(DIVY_LIB_DYNAMIC_EXPORT)
/**
 * スタートアップハンドラ (コンストラクタ関数)
 * (note)
 * 	このハンドラはdlopen 終了直後に呼び出されます。また、このライブラリが
 * 	ダイナミックリンクライブラリとして使用された時には、リンク先のmain 関数
 * 	コールの直前に呼び出されます。
 * (note)
 * 	この関数はプロバイダライブラリがstatic コンパイルされると自動的に
 * 	デバックアウトされます。従って、この関数には、
 * 	    * dlopen された時だけ実施する特別な初期化処理
 * 	    * 関数divy_register_lib_provider のコール
 * 	以外の処理を"書いてはなりません"。
 */
#if defined(DIVY_USE_GNU_LD)
static void _ml_default_init(void)
#else
void _init(void)
#endif	/* DIVY_USE_GNU_LD */
{
	/* プロバイダライブラリ構造体の登録 */
	divy_register_lib_provider(&DIVY_PROVIDER_HOOK_STRUCT_NAME(ml,default));
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"ml(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
}

/**
 * 終了ハンドラ (デストラクタ関数)
 * (note)
 * 	このハンドラはdlclose 終了直後に呼び出されます。また、このライブラリが
 * 	ダイナミックリンクライブラリとして使用された時には、リンク先のmain 関数
 * 	が終了した後、コールされます。
 * (note)
 * 	この関数はプロバイダライブラリがstatic コンパイルされると自動的に
 * 	デバックアウトされます。従って、この関数には、
 * 	    * dlopen された時だけ実施する特別な終了処理
 * 	以外の処理を"書いてはなりません"。
 */
#if defined(DIVY_USE_GNU_LD)
static void _ml_default_fini(void)
#else
void _fini(void)
#endif	/* DIVY_USE_GNU_LD */
{
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"ml(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
	return;
}
#endif	/* DIVY_LIB_DYNAMIC_EXPORT */

/**
 * 初期化ハンドラ
 * (note)
 * 	このハンドラは、Apacheモジュールの一部として動作する場合には
 * 	post_config ステージ(まだChildも起動していないステージ) で
 * 	呼び出されます。
 *
 * 	Child 起動前に実施しなければならない処理や、pconf を使った
 * 	処理を記述できます。
 * 	Child_init でコールバックを受け取るには、このハンドラの
 * 	中で関数divy_hook_child_init を使ってコールバックハンドラを
 * 	レジスタして下さい。
 *
 * @pconf apr_pool_t * コンフィグプール
 * @return int 処理ステータス
 * 	DIVY_LIB_ST_OK  : 正常
 * 	DIVY_LIB_ST_ERR : 失敗
 */
static int _ml_default_init_func(apr_pool_t *pconf)
{
	/* create_mldatasource ハンドラへの登録 */
	divy_hook_create_mldatasource(_ml_default_create_mldatasource,
					NULL, NULL, APR_HOOK_MIDDLE);
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"ml(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
	return DIVY_LIB_ST_OK;
}

/**
 * 指定されたproviderType が自身のDBプロバイダを示していた時、
 * 自身のDbDataSource を生成してdbds に返却する。
 *
 * @param p apr_pool_t *
 * @param providerType const char * プロバイダタイプ
 * @param mlds MlDataSource ** 作成したMlDataSource構造体
 * @return
 * 	ML_OK       : providerType が示すプロバイダは自分自身、かつ処理が成功した
 * 	ML_DECLINED : providerType が示すプロバイダは自分ではなかった
 */
static int _ml_default_create_mldatasource(apr_pool_t *p,
					const char *providerType,
					MlDataSource **mlds)
{
	*mlds = NULL;

	/*
	 * providerType は自分を示しているか？
	 */
	if (providerType == NULL ||
	    strcmp(providerType, DEFAULT_ML_PROVIDER_TYPE) != 0) {
		return ML_DECLINED;
	}

	*mlds = default_mlds_createMlDs(p);

	return ML_OK;
}


