/**
 * $Id$
 *
 * props.c
 *
 * Property database・プロバイダ
 *
 * 変更履歴
 *
 * 2003/02/13 Thu takehara NEW
 *
 */
/* Apache header files */
#include "httpd.h"
#include "apr.h"
#include "apr_strings.h"
#include "apr_pools.h"
#include "apr_hash.h"
#include "mod_dav.h"

/* document management headers */
#include "mod_dav_tf.h"
#include "tf_rdbo.h"
#include "util.h"

APLOG_USE_MODULE(dav_tf);

/*--------------------------------------------------------------
  Define Fixed values and macro
  --------------------------------------------------------------*/
/**
 * Dead プロパティのネームスペースURIの接頭語
 */
#define DIVY_DEAD_NS_PREFIX "ns"

/*
 * mod_dav 騙しモードかどうかを判定するマクロ
 * (note) "騙しモード"については、dav_divy_open のコメントを参照
 */
#define IS_CHEAT_DAV(p_db)  ((p_db)->is_cheat && (p_db)->dummy_lp_cnt > 0)

/*--------------------------------------------------------------
  Define incomplete type structure
  --------------------------------------------------------------*/
/**
 * [ 不完全型の実装 ]
 *
 * Open されたDead プロパティDataBaseを表す構造体。
 * Dead プロパティの処理に必要なデータを保持しています。
 */
struct dav_db {
	int ro;                       /* Read-Only フラグ (=1 : Read-Only)   */
	const char *rsid;             /* このリソースのrsid                  */
	const char *uri;              /* このリソースのuri                   */
	divy_rdbo_dproperty *d_pr;    /* このリソースのDead プロパティ       */
	apr_hash_t *ns_id_hash;       /* ns -> ns_uri のハッシュ             */
	apr_hash_t *ns_uri_hash;      /* ns_uri -> ns のハッシュ             */
	request_rec *r;               /* リクエスト構造体へのポインタ        */
	divy_uri_spec *u_spec;        /* uriの種類をあらわす構造体へのポインタ */
	int is_cheat;                 /* mod_dav 騙しモードかどか。(see dav_divy_open)   */
	int dummy_lp_cnt;             /* live プロパティカウンタ(mod_dav を騙す為に使用) */

	/* エラー情報を記録する構造体へのポインタ.
	 * (note) この変数の意味
	 * 	mod_dav はエラーステータスを調べてくれないため、何処かのハンドラでエラーが
	 * 	発生しても構わず処理を続行します。そうすると、必要なデータが無かったり
	 * 	エラー状態であったりするので後々問題を引き起こします。
	 * 	mod_dav がその気なら、こっちは1度起きたエラーステータスを記憶しておき
	 * 	続けてリクエストされてもエラー終了するくらいしか方法はありません。
	 * 	このポインタはその目的のために使用されます。 */
	dav_error *err;
};

/*
 * [ 不完全型の実装 ]
 *
 * response として出力されるネームスペースPrefixを作成するために使用する
 * ネームスペースIndex(int 型の数値)を保持する構造体。
 *
 * グローバルなネームスペースマッピングからこのDeadプロパテlプロバイダ
 * 内部で定義されたネームスペースマッピングに変換するための変換テーブルの
 * 役割を持っています。
 *
 * (構造)
 *   メンバ ns_map はint型の配列。
 *   配列のindex : グローバルネームスペースマッピングのインデックス(ns)
 *   配列のvalue : DAVプロバイダ内部のネームスペースマッピングのインデックス
 *
 * (note) ns とは？
 * Apache のXML関数は、request のXMLを解析すると、XMLドキュメントに
 * 存在していたネームスペースURIをapr_array_header_t 型として
 * 保持します。このapr_array_header_t型は、ネームスペースURIの値と
 * ns という一意なIdを持っています。
 */
struct dav_namespace_map {
    int *ns_map;
};

/**
 * Dead プロパティに対して加えられた変更をロールバックする際に必要となる
 * 情報を保持するコンテキスト。1つのプロパティ毎に存在します。 (see mod_dav.h)
 *
 * このコンテキストは、PROPPATCH による変更を行う前に生成され
 * (dav_divy_get_rollback)、何らかの問題が発生して、変更を取消す際に利用されます。
 * (dav_divy_apply_rollback)
 */
struct dav_deadprop_rollback {

	/* 変更を加える前に持っていた値 */
	const divy_rdbo_dproperty *b_d_pr;

	/* 対象プロパティの情報 */
	const dav_prop_name *name;
};

/*--------------------------------------------------------------
  Define array
  --------------------------------------------------------------*/
/*
 * mod_dav 騙しモードで騙すプロパティ(dav_prop_name)を持つ配列
 */
const dav_prop_name cheat_props[] = {
	{ "DAV:", "getcontenttype"	},
	{ "DAV:", "getcontentlanguage"	}
};

/* 配列cheat_props の長さ */
#define CHEAT_PROPS_LEN	sizeof(cheat_props)/sizeof(dav_prop_name)

/*--------------------------------------------------------------
  Declare  private functions
  --------------------------------------------------------------*/
static divy_rdbo_dproperty * _find_dproperty(dav_db *db, const dav_prop_name *name);
static void _define_namespaces(dav_db *db, dav_xmlns_info *xi);

/*--------------------------------------------------------------
  Define Hook functions
  --------------------------------------------------------------*/
/*
 * Property DataBaseを"Open"する。(Open したとみなせる状態にする)
 * (note)
 * 性能向上を図るため、walk と共にDead プロパティの取得が行われるとき即ち
 * Read-Onlyフラグが立っている時には、walk 側で値を取ってもらっています。
 * １回のSQLで必要な全てのDeadプロパティが得られるため性能が著しく向上
 * します。反面、このプロバイダとリポジトリプロバイダが癒着してしまいました。
 *
 * ro = 0 の時には、walk で取得できる値が無いので、自分で検索しています。
 * 
 * (note) リソース単位でCallbackされます。
 *
 * @param ro int Read-Only フラグ (0: writable / 1: read-only)
 * @param resource const dav_resource * Deadプロパティを持つリソース
 * @param pdb dav_db ** Openしたプロバイダ情報を保持する構造体
 * @return dav_error * DAV エラー (常にNULLが返ります)
 */
static dav_error * dav_divy_open(apr_pool_t *p, 
                                 const dav_resource *resource, 
                                 int ro, dav_db **pdb)
{
	/* dav_db 構造体のインスタンスを生成 */
	*pdb = apr_pcalloc(p, sizeof(dav_db));
	(*pdb)->err = NULL;	/* エラーは存在しない */

	/* Read-Onlyで、かつd_pr が無かった場合 */
	if (ro && resource->info->rdb_r->d_pr == NULL) {
		/*
		 * Dead プロパティは存在しなかったが、mod_dav のdav_get_allprops
		 * 関数をだますために値を設定する。
		 * (note)
		 *   関数dav_get_allprops は、Dead プロパティの中に
		 *   "getcontenttype"と"getcontentlanguage" が無ければ、subreqを
		 *   発行してしまいます。例えLive プロパティプロバイダ側で
		 *   ちゃんと見つかっていてもやめてくれません。
		 *   これは負荷が高いので、mod_dav を欺いて、結果的にsubreq を
		 *   発行させないようにしてしまいました。具体的には、Dead 
		 *   プロパティプロバイダは、プロパティが見つかったと嘘の報告を
		 *   行い、その値を文字列としてappend しなさいとmod_dav から
		 *   CallBackされたときに、値を設定しないようにします。
		 *
		 *   mod_dav がこのような不思議なフレームワークを採用しているのは
		 *   恐らく、dav_fs と分離できなかった頃の名残であろうと思われ
		 *   ます。
		 */
		(*pdb)->d_pr         = NULL;
		(*pdb)->r            = resource->info->r;
		(*pdb)->dummy_lp_cnt = CHEAT_PROPS_LEN;   /* 騙すLiveプロパティの個数 */
		(*pdb)->is_cheat     = 1;   /* mod_dav 騙しモード！！*/

		return NULL;
	}

	/* 以下は、d_pr が存在していた or Dead プロパティのPROPPATCH の場合 */
	(*pdb)->ro          = ro;
	(*pdb)->rsid        = resource->info->rdb_r->rsid;
	(*pdb)->uri         = resource->info->rdb_r->uri;
	(*pdb)->d_pr        = resource->info->rdb_r->d_pr;
	(*pdb)->ns_id_hash  = resource->info->ns_id_hash;
	(*pdb)->ns_uri_hash = resource->info->ns_uri_hash;
	(*pdb)->u_spec      = resource->info->rdb_r->u_spec;
	(*pdb)->r           = resource->info->r;

	if (ro) {
		(*pdb)->dummy_lp_cnt= CHEAT_PROPS_LEN;  /* 騙すLiveプロパティの個数 */
		(*pdb)->is_cheat    = 1;  /* mod_dav 騙しモード！！   */

		/* Read-Only ならば、もうすることはありません。*/
	}
	else {
		divy_uri_spec *u_spec = (*pdb)->u_spec;
		request_rec *r        = (*pdb)->r;

		/*
		 * PROPPATCH が実施できるかどうかチェック(アクセス権限チェック)
		 */

		/* Read-Only, Upload-Onlyであればプライベートコレクション以外の
		 * PROPPATCHを拒絶する(ユーザ拡張ステータス) */
		if (divy_support_extenduserstatus(r)) {
			const divy_rdbo_extstatus *extstatus = divy_get_extstatus(r);

			if ((divy_rdbo_has_upload_privilege(extstatus) ||
				divy_rdbo_has_read_privilege(extstatus)) &&
				(u_spec->infotype != DIVY_INFOTYPE_user_e_regular)) {

				ERRLOG1(r->pool, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The server dose not accept PROPPATCH on this resource "
					"for read-only or upload-only user.(uri = %s)", (*pdb)->uri);
				(*pdb)->err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");	/* エラーを記録 */
				return (*pdb)->err;
			}

			/* 書き込み制約があればPROPPATCHを許可しない */
			if (divy_rdbo_has_write_constraints_on_uri(r, (*pdb)->uri, u_spec->infotype)) {
				ERRLOG1(r->pool, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The server dose not accept PROPPATCH on this resource "
						"for read-only group.(uri = %s)", (*pdb)->uri);
					(*pdb)->err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");	/* エラーを記録 */
				return (*pdb)->err;
			}
		}

		/* 以下のリソース以外にはDeadプロパティのPROPPATCHを許可しない
		 *  ・通常リソース / 通常コレクション
		 *  ・共有コレクション以下
		 *  ・linkdbsearch結果フォルダ以下 */
		if (!((u_spec->uritype & DIVY_URITYPE_REGULAR) ||
			(u_spec->infotype == DIVY_INFOTYPE_dbfolder_e) ||
			(u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e))) {

			ERRLOG1(r->pool, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The server does not accept PROPPATCH on this resource."
				"(uri = %s)", (*pdb)->uri);
			(*pdb)->err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			return (*pdb)->err;
		}

		/* 騙す必要はありません。まじめにやります。*/
		(*pdb)->dummy_lp_cnt= 0;
		(*pdb)->is_cheat    = 0;

		/* Read/Write ならば、自前でハッシュを取得する */
		if (divy_rdbo_build_nsmap_hash((*pdb)->r,
						&(*pdb)->ns_id_hash,
						&(*pdb)->ns_uri_hash)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get namespace hash for dead property. "
				"(uri = %s)", (*pdb)->uri);
			(*pdb)->err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			return (*pdb)->err;
		}

		/* Read/Write ならば、自前でdivy_rdbo_dproperty を取得する */
		if (divy_rdbo_get_dead_property_by_uri((*pdb)->r, (*pdb)->uri, &(*pdb)->d_pr)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get divy_rdbo_dproperty "
				"for dead property.(uri = %s)", (*pdb)->uri);
			(*pdb)->err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			return (*pdb)->err;
		}
	}

	return NULL;
}

/**
 * Property DataBase を"Close"する。(Close したとみなせる状態にする)
 *
 * (note) リソース単位でCallbackされます。
 *
 * @param db dav_db * Property DataBase を表す構造体へのポインタ
 */
static void dav_divy_close(dav_db *db)
{
	return;	/* 特にすることはありません。*/
}

/**
 * dav_db と関連する(Dead プロパティが定義されているリソースの) Dead プロパティ
 * 名と値のネームスペースを取得して xi のメンバに値を設定する。
 * mod_dav からの要請により、次のように値を組み立ててxi メンバに設定します。
 * (設定するメンバはapr_hash_t です)
 *
 * ・xi->prefix_uri : key = DIVY_DEAD_NS_PREFIX + (ネームスペースID), value = uri
 * ・xi->uri_prefix : key = uri, value = DIVY_DEAD_NS_PREFIX + (ネームスペースID) 
 *
 * (note)
 * xi のメンバには、ネームスペースID が小さいものから並べなければなりません。
 * 但し、rdb_dp が示すリストは必ずネームスペースが小さい順に整列されている
 * ことが保証されています。(divy_rdbo.c 側で考慮)
 * リソース単位でCallbackされます。
 *
 * @param xi dav_xmlns_info * ネームスペースを扱うための構造体へのポインタ
 * @return dav_error * DAV エラー
 */
static dav_error * dav_divy_define_namespaces(dav_db *db, dav_xmlns_info *xi)
{
	/* 取得済みか？ */
	if (apr_hash_count(xi->uri_prefix)) return NULL;

	_define_namespaces(db, xi);
	return NULL;
}

/**
 * 指定された name のDead プロパティの値を取得して、XMLタグを組み立て、
 * phdr に追加する。
 * (note) リソースが持っているDead プロパティの単位でCallbackされます。
 *
 * @param db dav_db * Property DataBase を表す構造体へのポインタ
 * @param name const dav_prop_name * Dead プロパティ
 * @param xi dav_xmlns_info *
 * @param phdr apr_text_header *
 * @param found int * (1: 見つかった, 0: 見つからなかった)
 * @return dav_error DAV エラー
 */
static dav_error * dav_divy_output_value(dav_db *db, 
                                         const dav_prop_name *name,
                                         dav_xmlns_info *xi,
                                         apr_text_header *phdr,
                                         int *found)
{
	divy_rdbo_dproperty *d_pr = NULL;
	const char *xml_str       = NULL;
	char *ns_prefix;

	/* name の Dead プロパティを探す */
	d_pr = _find_dproperty(db, name);
	if (d_pr == NULL) {
		*found = 0;	/* 見つからなかった */
		return NULL;
	}

	*found = 1;	/* 見つかった */

	/*
	 * namespace 名を取得
	 * (note)
	 * name->ns   : ネームスペースURI
	 * name->name : プロパティ名
	 */
	if (apr_hash_count(xi->uri_prefix) == 0) {
		/*
		 * ネームスペースURIからプレフィックスを求めるxiを生成する
		 * (note)
		 * 	mod_dav.h のコメントや実装によると、output_valueの前に
		 * 	define_namespacesハンドラをコールしない場合ばあるそうで
		 * 	必要ならオンデマンドで作れということ。
		 * 	何故そのようなコードになるのか分からないが、xiが空なので
		 * 	ここで作ることにしました。
		 */
		_define_namespaces(db, xi);
	}

	ns_prefix = apr_hash_get(xi->uri_prefix, name->ns, APR_HASH_KEY_STRING);
	if (IS_EMPTY(ns_prefix)) {
		ns_prefix = "";
	}
	else {
		ns_prefix = apr_pstrcat(db->r->pool, ns_prefix, ":", NULL);
	}

	/*
	 * XML 文字列を組み立てる
	 */
	/* EMPTY タグを組み立てる */
	if (IS_EMPTY(d_pr->value)) {
		xml_str = apr_psprintf(db->r->pool,
				"<%s%s/>" CRLF, ns_prefix, d_pr->name);
	}
	/* xml:lang なしでタグを組み立てる */
	else if (IS_EMPTY(d_pr->lang_tag)) {
		/*
		 * 2005/03/14 Mon
		 * 値を実体参照やCDATAつきに変換してはならないことになりました。
		 * PROPPATCHでは常にきちんと変換された状態で格納されるので
		 * 問題は起きません。裏からDBを変更する場合には注意が必要です。*/
		xml_str = apr_psprintf(db->r->pool,
				"<%s%s>%s</%s%s>" CRLF,
				ns_prefix, d_pr->name,
				d_pr->value,
				//dav_divy_escape_xmlstr(db->r->pool,
				//	(char *)d_pr->value, DIVY_XML_T2T_QUOTE),
				ns_prefix, d_pr->name);
	}
	/* xml:lang ありでタグを組み立てる */
	else {
		xml_str = apr_psprintf(db->r->pool,
				"<%s%s xml:lang=\"%s\">%s</%s%s>" CRLF,
				ns_prefix, d_pr->name, d_pr->lang_tag,
				d_pr->value,
				//dav_divy_escape_xmlstr(db->r->pool,
				//	(char *)d_pr->value, DIVY_XML_T2T_QUOTE),
				ns_prefix, d_pr->name);
	}

	/* phdr に作成した文字列を足しこむ */
	apr_text_append(db->r->pool, phdr, xml_str);

	return NULL;
}

/**
 * 与えられた"global" namespaces が持っているネームスペースURIに
 * "Deadプロパティプロバイダ"で定義されたネームスペースIDを対応付けて
 * その対応表をmapping として返却する。
 * 登録されていないネームスペースURIが指定されたら、そのURIと
 * 対応するネームスペースIDを採番し、dav_namespace テーブルに格納します。
 *
 * (note) 対応付けの方法 --> 構造体dav_namespace_map を参照。
 *
 * (note) この関数の意味
 *  dav_divy_store() に先立ち、与えられた"global" namespaces のURIと対応する
 *  このDeadプロパティプロバイダのネームスペースIDを算出する役割を与えられて
 *  います。
 *  従って、Read/WriteモードでOpenされた場合にのみ、この関数をCallする
 *  ことが許されています。
 *  (mod_dav はちゃんとそのように呼んでくれていますが。)
 *
 * (note) set/reomve かつ prop で指定されたプロパティの個数分だけCallbackされます。
 *
 * @param db dav_db * Property DataBase を表す構造体へのポインタ
 * @param namespaces const apr_array_header_t *
 * @param mapping dav_namespace_map ** ネームスペースIndex を保持するMap
 * @return dav_error DAV エラー (409, 500)
 */
static dav_error * dav_divy_map_namespaces(dav_db *db, 
                                           const apr_array_header_t *namespaces,
                                           dav_namespace_map **mapping)
{
	int *ns_map = NULL, *first_ns_map = NULL;
	const char **p_ns_uri = NULL;
	char *ns_uri          = NULL;
	char *ns_id_str       = NULL;
	apr_pool_t *p         = db->r->pool;
	int i;

	/* 既にエラー状態にあったか? */
	if (db->err != NULL) {
		/* エラー構造体のインスタンスは必ず再作成する!! */
		db->err = dav_new_error(p, db->err->status,
								db->err->error_id, 0, db->err->desc);
		return db->err;
	}

	TRACE(p);

	/* Read/Write モードでOpenされていなければなりません */
	if (db->ro) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"This function is only in preparation "
			"for a series of store() calls. so, "
			"MUST be open for ro = 0.");
		db->err = dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
		return db->err;
	}

	/* namespaces->nelts 個の要素を持つint 型の配列を作成する */
	ns_map = apr_pcalloc(p, namespaces->nelts * sizeof(*ns_map));
	first_ns_map = ns_map;	/* 先頭を覚えておく */

	/*
	 * namespace を回して namespaces のns とネームスペースID を関連付ける
	 * (note)
	 * 逆回しにするのは、こうすると、PROPPATCHで指定されたネームスペースの
	 * 順番と一致するためです。(逆にApacheがはめてしまう？)
	 */
	for (i = namespaces->nelts, p_ns_uri = (const char **)namespaces->elts;
						i-- > 0; p_ns_uri++, ns_map++) {
		ns_id_str = apr_hash_get(db->ns_uri_hash,
					*p_ns_uri, APR_HASH_KEY_STRING);
		/* ネームスペースURI が未登録だった場合 */
		if (IS_EMPTY(ns_id_str)) {

			ns_uri = apr_pstrdup(p, *p_ns_uri);
			if (divy_rdbo_insert_dav_namespace(db->r, ns_uri, &ns_id_str)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to register a new namespace uri. "
					"(ns_uri = %s)", ns_uri);
				db->err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				return db->err;
			}

			/* ns_uri_hash に追加 */
			apr_hash_set(db->ns_uri_hash, ns_uri,
					APR_HASH_KEY_STRING, ns_id_str);  
			/* ns_id_hash にも追加しておく */
			apr_hash_set(db->ns_id_hash, ns_id_str,
					APR_HASH_KEY_STRING, ns_uri); 
		}

		/* (note) ns_id_str がネームスペースプレフィックスの番号パートになります */
		*ns_map = atoi(ns_id_str);
	}

	/* mapping に領域を割り当てる
	 * (note)
	 * 	mapping には別のDeadプロパティと共に定義されたネームスペース
	 * 	マッピングが格納されているかもしれないのだが、db->ns_uri_hash
	 * 	に同じ情報を持つようにしているため、mappingは潰しても問題
	 * 	ないのである。
	 * */
	*mapping = apr_pcalloc(p, sizeof(dav_namespace_map));
	(*mapping)->ns_map = first_ns_map;

	return NULL;
}

/**
 * 指定された name が示すプロパティ名＆ネームスペースのDead プロパティを
 * Property Database に登録する。
 *
 * (note) set かつ prop で指定されたプロパティの個数分だけCallbackされます。
 *
 * @param db dav_db * Property DataBase を表す構造体へのポインタ
 * @param name const dav_prop_name * プロパティ
 * @param elem const apr_xml_elem *
 * @param mapping dav_namespace_map *
 * @return dav_error * (500, 409)
 */
static dav_error * dav_divy_store(dav_db *db,
                                  const dav_prop_name *name,
                                  const apr_xml_elem *elem,
                                  dav_namespace_map *mapping)
{
	divy_rdbo_dproperty *d_pr = NULL, *find_d_pr;
	char *val = NULL;
	size_t l_val;
	apr_pool_t *p = db->r->pool;

	/* 既にエラー状態にあったか? */
	if (db->err != NULL) {
		/* エラー構造体のインスタンスは必ず再作成する!! */
		db->err = dav_new_error(p, db->err->status,
								db->err->error_id, 0, db->err->desc);
		return db->err;
	}

	TRACE(p);

	/* 名前がなかった場合 (ありえないが一応) */
	if (IS_EMPTY(name->name)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"name->name is NULL.");
		db->err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto append_store_list;
	}

	/*
	 * (note) mod_dav は map_namespacesハンドラでエラーを返してもそれを
	 *        全く見てくれないので、mapping がNULLのままになることがある。
	 *        継続は不可能なのでエラー終了する。
	 */
	if (mapping == NULL || mapping->ns_map == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The namespace mapping is NULL.");
		db->err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto append_store_list;
	}

	/*
	 * XML をパースする
	 */
	/* value 値の取得 */
	apr_xml_quote_elem(p, (apr_xml_elem *) elem);
	apr_xml_to_text(p, elem, APR_XML_X2T_INNER, NULL,
			mapping->ns_map, (const char **) &val, &l_val);

	d_pr = apr_pcalloc(p, sizeof(divy_rdbo_dproperty));
	d_pr->rsid         = (char *) db->rsid;
	d_pr->ns_id        = mapping->ns_map[elem->ns];
	d_pr->name         = apr_pstrdup(p, name->name);
	d_pr->lang_tag     = apr_pstrdup(p, elem->lang);
	d_pr->value        = val;
	d_pr->patch_action = DIVY_DP_NOTHING;	/* 何もしていない */
	d_pr->next         = NULL;

	/* Deadプロパティを登録する
	 * (note)
	 * 	同時更新により既に登録されているケースもあるので
	 * 	これを考慮してくれるupdate関数を使う
	 */
	/* 登録 or 更新する */
	if (divy_rdbo_update_dead_property(db->r, d_pr)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to register dead property.");
		db->err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto append_store_list;
	}
	d_pr->patch_action = (DIVY_DP_APPLAYPATCH | DIVY_DP_SET);

append_store_list:
	if (db->err != NULL) {
		if (d_pr == NULL) {
			d_pr = apr_pcalloc(p, sizeof(divy_rdbo_dproperty));
		}
		d_pr->patch_action = DIVY_DP_NOTHING;	/* 何もしなくていい */
	}

	/* Dead プロパティは登録されていないか? */
	find_d_pr = _find_dproperty(db, name);
	if (find_d_pr == NULL) {
		if (db->d_pr == NULL) {
			db->d_pr = d_pr;
		}
		else {
			divy_rdbo_dproperty *pr;
			for (pr = db->d_pr; pr->next; pr = pr->next);
			pr->next = d_pr;	/* List に繋げる */
		}
	}
	else {
		find_d_pr->rsid         = d_pr->rsid;
		find_d_pr->ns_id        = d_pr->ns_id;
		find_d_pr->name         = d_pr->name;
		find_d_pr->value        = d_pr->value;
		find_d_pr->lang_tag     = d_pr->lang_tag;
		find_d_pr->patch_action = d_pr->patch_action;
	}

	return db->err;
}

/**
 * 指定された name が示すプロパティ名＆ネームスペースのDead プロパティを
 * Property Database から削除する。
 *
 * (note) remove かつ prop で指定されたプロパティの個数分だけCallbackされます。
 *
 * @param db dav_db * Property DataBase を表す構造体へのポインタ
 * @param name const dav_prop_name * プロパティ
 * @return DAVエラー (500)
 */
static dav_error * dav_divy_remove(dav_db *db, const dav_prop_name *name)
{
	divy_rdbo_dproperty *d_pr = NULL, *find_d_pr;
	apr_pool_t *p             = db->r->pool;
	char *ns_id_str           = NULL;
	int ns_id;

	/* 既にエラー状態にあったか? */
	if (db->err != NULL) {
		/* エラー構造体のインスタンスは必ず再作成する!! */
		db->err = dav_new_error(p, db->err->status,
								db->err->error_id, 0, db->err->desc);
	}

	TRACE(p);

	if (IS_EMPTY(name->name)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"name->name is NULL.");
		db->err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto append_remove_list;
	}

	/*
	 * Dead プロパティの削除に必要な情報を集める
	 */
	ns_id_str = apr_hash_get(db->ns_uri_hash, name->ns, APR_HASH_KEY_STRING); 
	if (IS_EMPTY(ns_id_str)) {
		/* あり得ないが一応。*/
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get ns_id string.(rsid = %s)", db->rsid);
		db->err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto append_remove_list;
	}
	ns_id = atoi(ns_id_str);

	d_pr = apr_pcalloc(p, sizeof(divy_rdbo_dproperty));
	d_pr->rsid  = (char *) db->rsid;
	d_pr->ns_id = ns_id;
	d_pr->name  = apr_pstrdup(p, name->name);
	d_pr->next  = NULL;

	/* Dead プロパティの削除 */
	if (divy_rdbo_remove_dead_property(db->r, d_pr)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to delete dead property.");
		db->err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR,0,0,"");
		goto append_remove_list;
	}
	d_pr->patch_action = (DIVY_DP_APPLAYPATCH | DIVY_DP_REMOVE);

append_remove_list:
	if (db->err != NULL) {
		d_pr->patch_action = DIVY_DP_NOTHING;	/* 何もしなくていい */
	}

	/* name のプロパティが登録されていないか? */
	find_d_pr = _find_dproperty(db, name);
	if (find_d_pr == NULL) {
		if (db->d_pr == NULL) {
			db->d_pr = d_pr;
		}
		else {
			divy_rdbo_dproperty *pr;
			for (pr = db->d_pr; pr->next; pr = pr->next);
			pr->next = d_pr;	/* List に繋げる */
		}
	}
	else {
		find_d_pr->rsid         = d_pr->rsid;
		find_d_pr->ns_id        = d_pr->ns_id;
		find_d_pr->name         = d_pr->name;
		find_d_pr->value        = NULL;
		find_d_pr->lang_tag     = NULL;
		find_d_pr->patch_action = d_pr->patch_action;
	}

	return db->err;
}

/**
 * 指定された name が示すプロパティ名＆ネームスペースのDead プロパティが
 * 存在するかどうか。
 *
 * @param db dav_db * Property DataBase を表す構造体へのポインタ
 * @param name const dav_prop_name * プロパティ
 * @return int (1: 存在する / 0: 存在しない)
 */
static int dav_divy_exists(dav_db *db, const dav_prop_name *name)
{
	if (_find_dproperty(db, name) != NULL) {
		return 1;
	}

	return 0;
}

/**
 * リソースのDead プロパティとして保持されているリストのうち、先頭の
 * 要素を返却する。
 * (note) リソース単位でCallbackされます。
 *
 * @param db dav_db * Property DataBase を表す構造体へのポインタ
 * @param name const dav_prop_name * 取り出したプロパティ
 * @return DAV エラー (常にNULLです)
 */ 
static dav_error * dav_divy_first_name(dav_db *db, dav_prop_name *pname)
{
	/* Dead プロパティが１つも無かった */
	if (db->d_pr == NULL) {
		/* mod_dav 騙しモードの場合 */
		if (IS_CHEAT_DAV(db)) {
			const dav_prop_name *p = &cheat_props[--db->dummy_lp_cnt];
			pname->ns   = p->ns;
			pname->name = p->name;
		}
		else {
			pname->ns   = NULL;	/* sentinel */
			pname->name = NULL;
		}
		return NULL;
	}

	/* divy_rdbo_dproperty から値を取り出す */
	pname->ns   = apr_hash_get(db->ns_id_hash,
				apr_itoa(db->r->pool, db->d_pr->ns_id),
				APR_HASH_KEY_STRING);
	pname->name = db->d_pr->name;
	return NULL;
}

/**
 * リソースのDead プロパティとして保持されているリストから、未処理の
 * プロパティを取り出して要素を返却する。
 * (note) リソースが持っているDead プロパティの単位でCallbackされます。
 *
 * @param db dav_db * Property DataBase を表す構造体へのポインタ
 * @param name const dav_prop_name * 取り出したプロパティ
 * @return DAV エラー (常にNULLです)
 */ 
static dav_error * dav_divy_next_name(dav_db *db, dav_prop_name *pname)
{
	/* 終端に達しているか? */
	if (db->d_pr == NULL || db->d_pr->next == NULL) {
		/* 終端に達していて、かつmod_dav 騙しモードの場合 */
		if (IS_CHEAT_DAV(db)) {
			const dav_prop_name *p = &cheat_props[--db->dummy_lp_cnt];
			pname->ns   = p->ns;
			pname->name = p->name;
			return NULL;
		}
		else {
			pname->ns   = NULL;	/* sentinel */
			pname->name = NULL;
			return NULL;
		}
	}

	/* divy_rdbo_dproperty のListを１つ進める */
	db->d_pr = db->d_pr->next;

	/* 全てのDead プロパティを検索し終えていないかどうか調べる */
	pname->ns   = apr_hash_get(db->ns_id_hash,
				apr_itoa(db->r->pool, db->d_pr->ns_id),
				APR_HASH_KEY_STRING);
	pname->name = db->d_pr->name;
	return NULL;
}

/**
 * 指定された name が示すプロパティ名＆ネームスペースのDead プロパティに
 * 加えられた変更を後で取消す(apply_rollback)するために、情報を記録しておく。
 * (note)
 * dav_deadprop_rollback は、プロパティ毎に存在します。
 * set/remove かつ prop で指定されたプロパティの個数分だけCallbackされます。
 *
 * @param db dav_db * Property DataBase を表す構造体へのポインタ
 * @param name const dav_prop_name * プロパティ
 * @param prollback dav_deadprop_rollback ** ロールバックに必要なコンテキスト
 * @return DAV エラー (500)
 */
static dav_error * dav_divy_get_rollback(dav_db *db, 
                                         const dav_prop_name *name,
                                         dav_deadprop_rollback **prollback)
{
	dav_error *err = NULL;
	divy_rdbo_dproperty *d_pr = NULL, *b_d_pr = NULL;
	dav_prop_name *b_name     = NULL;
	apr_pool_t *p             = db->r->pool;

	/* 既にエラー状態にあったか? */
	if (db->err != NULL) {
		/* エラー構造体のインスタンスは必ず再作成する!! */
		db->err = dav_new_error(p, db->err->status,
								db->err->error_id, 0, db->err->desc);
		return db->err;
	}

	TRACE(p);

	/* rollback コンテキストの生成 */
	*prollback = apr_pcalloc(p, sizeof(dav_deadprop_rollback));

	/*
	 * 与えられた name のプロパティを退避する
	 */
	b_name = apr_pcalloc(p, sizeof(dav_prop_name));
	b_name->ns   = apr_pstrdup(p, name->ns);
	b_name->name = apr_pstrdup(p, name->name);
	(*prollback)->name   = (const dav_prop_name *) b_name;

	/* name の Dead プロパティを探す */
	d_pr = _find_dproperty(db, name);
	if (d_pr != NULL) {

		/* 対象のDeadプロパティの最新状態をDBから検索してrollback
		 * コンテキストに保存する */
		if (divy_rdbo_get_dead_property(db->r, d_pr->rsid, d_pr->ns_id,
										d_pr->name, &b_d_pr)) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get rollback context (rsid = %s,"
					"nsid = %d, name = %s)", d_pr->rsid, d_pr->ns_id, d_pr->name);

			/* rollback コンテキストが取得できなかったのでrollback出来ない */
			b_d_pr = apr_pcalloc(p, sizeof(divy_rdbo_dproperty));
			b_d_pr->patch_action = DIVY_DP_NOTHING;
			b_d_pr->next = NULL;
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		else {
			/* DB検索が成功していて、Deadプロパティが存在すれば
			 * 更新対象Deadプロパティのアクション(d_pr) を退避しておく */
			if (b_d_pr != NULL) {
				b_d_pr->patch_action = d_pr->patch_action;
			}
		}
	}
	else {
		/* 対象リソースのDeadプロパティを探す */
		divy_rdbo_dproperty *first = NULL;
		char *ns_id_str;

		if (divy_rdbo_get_dead_property_by_uri(db->r, db->uri, &first)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get rollback context (uri = %s)", db->uri);

			/* rollback コンテキストが取得できなかったのでrollback出来ない */
			b_d_pr = apr_pcalloc(p, sizeof(divy_rdbo_dproperty));
			b_d_pr->patch_action = DIVY_DP_NOTHING;
			b_d_pr->next = NULL;
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		else {

			ns_id_str = apr_hash_get(db->ns_uri_hash, b_name->ns, APR_HASH_KEY_STRING);
			if (IS_FILLED(ns_id_str)) {
				apr_int32_t ns_id = atoi(ns_id_str);

				for (b_d_pr = first; b_d_pr; b_d_pr = b_d_pr->next) {
					if (b_d_pr->ns_id == ns_id &&
						strcmp(b_d_pr->name, b_name->name) == 0) {
						b_d_pr->patch_action = DIVY_DP_NOTHING;
						break;	/* 見つかった */
					}
				}
			}
			/* ns_id_str が無かった場合 */
			else {
				/* Deadプロパティもないはず -> rollback時には消していい */
					b_d_pr = NULL;
			}
		}
	}
	(*prollback)->b_d_pr = (const divy_rdbo_dproperty *) b_d_pr;

	return err;
}

/**
 * 指定された name が示すプロパティ名＆ネームスペースのDead プロパティに
 * 加えられた変更を取消して、元の状態に戻す。
 * (note)
 * dav_deadprop_rollback は、プロパティ毎に存在します。
 * set/remove かつ prop で指定されたプロパティの個数分だけCallbackされます。
 *
 * @param db dav_db * Property DataBase を表す構造体へのポインタ
 * @param prollback dav_deadprop_rollback * ロールバックに必要なコンテキスト
 * @return DAV エラー (500)
 */
static dav_error * dav_divy_apply_rollback(dav_db *db,
                                           dav_deadprop_rollback *rollback)
{
	divy_rdbo_dproperty *d_pr = NULL;
	const divy_rdbo_dproperty *b_d_pr;
	apr_pool_t *p = db->r->pool;

	/* (note)
	 * db->err をチェックしてはならない。patch_action の値で
	 * 決めないとならない */

	/* rollback コンテキストが無ければ何もしなくていい */
	if (rollback == NULL) return NULL;
	b_d_pr = rollback->b_d_pr;

	if (db->d_pr != NULL) {
		/* 書き戻すDeadプロパティ値を取得 */
		d_pr = _find_dproperty(db, rollback->name);
	}

	/* rollback の適用は必要か？ */
	if (d_pr != NULL && !(d_pr->patch_action & DIVY_DP_APPLAYPATCH)) {
		/* store を行う前に失敗していたのでrollback処理は不要.
		 * インスタンスの値だけ書き戻す */
		if (b_d_pr != NULL) {
			d_pr->ns_id        = b_d_pr->ns_id;
			d_pr->name         = b_d_pr->name;
			d_pr->value        = b_d_pr->value;
			d_pr->lang_tag     = b_d_pr->lang_tag;
			d_pr->patch_action = b_d_pr->patch_action;
		}
		return NULL;
	}

	TRACE(p);

	/*
	 * Deadプロパティのrollback 処理
	 *
	 *  d_pr           b_d_pr          Action        ケース番号
	 * -------------  -------------   -------------  --------------
	 *  SET            (not null)      update         [1]
	 *  SET            (null)          delete         [2]
	 *  REMOVE         (not null)      insert         [3]
	 *  REMOVE         (null)          (nothing)      [4]
	 *
	 */
	if (d_pr != NULL && (d_pr->patch_action & DIVY_DP_SET)) {
		/* [1] update */
		if (b_d_pr != NULL) {
			if (divy_rdbo_update_dead_property(db->r, b_d_pr)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update dead property for rollback."); /* 続ける */
			}
			d_pr->patch_action = (DIVY_DP_APPLAYPATCH | DIVY_DP_SET);
		}
		/* [2] remove */
		else {
			if (divy_rdbo_remove_dead_property(db->r, d_pr)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to delete dead property for rollback."); /* 続ける */
			}
			d_pr->patch_action = (DIVY_DP_APPLAYPATCH | DIVY_DP_REMOVE);
		}
	}
	else {
		/* [3] insert */
		if (b_d_pr != NULL && IS_FILLED(b_d_pr->rsid)) {
			if (divy_rdbo_insert_dead_property(db->r, b_d_pr)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to insert dead property for rollback."); /* 続ける */
			}
			if (d_pr != NULL) {
				d_pr->patch_action = (DIVY_DP_APPLAYPATCH | DIVY_DP_SET);
			}
		}
		/* [4] (nothing) */
		else {
			if (d_pr != NULL) {
				d_pr->patch_action = (DIVY_DP_APPLAYPATCH | DIVY_DP_REMOVE);
			}
			return NULL;
		}
	}

	/* b_d_pr の値を書き込む */
	if (b_d_pr != NULL) {
		d_pr->ns_id    = b_d_pr->ns_id;
		d_pr->name     = b_d_pr->name;
		d_pr->value    = b_d_pr->value;
		d_pr->lang_tag = b_d_pr->lang_tag;
	}

	return NULL;
}

/*--------------------------------------------------------------
  Define private functions
  --------------------------------------------------------------*/
/**
 * 指定されたdb->d_pr のリストから、name のプロパティのdivy_rdbo_dproperty を
 * 探して、返却する。
 *
 * @param d_pr_list divy_rdbo_dproperty * Dead プロパティのリスト
 * @param name const dav_prop_name * Dead プロパティ (検索対象)
 * @return divy_rdbo_dproperty * 見つかったDead プロパティ構造体
 * 				見つからなければNULLを返す。
 */
static divy_rdbo_dproperty * _find_dproperty(dav_db *db, const dav_prop_name *name)
{
	divy_rdbo_dproperty *d_pr;
	int ns_id;
	char *ns_id_str;


	/* 引数が不正 or プロパティが1個も無かった場合 */
	if (name == NULL || name->ns == NULL ||
	    name->name == NULL || db->d_pr == NULL || db->ns_uri_hash == NULL) {
		return NULL;
	}

	/* ネームスペースID を算出する */
	ns_id_str = apr_hash_get(db->ns_uri_hash, name->ns, APR_HASH_KEY_STRING); 
	if (IS_EMPTY(ns_id_str)) return NULL;
	ns_id = atoi(ns_id_str);

	/* name のプロパティを探す */
	for (d_pr = db->d_pr; d_pr; d_pr = d_pr->next) {
		if (d_pr->ns_id == ns_id &&
		    strcmp(d_pr->name, name->name) == 0) {
			return d_pr;	/* 見つかった */
		}
	}
	return NULL;
}

/**
 * Dead プロパティ名と値のネームスペースを取得して xi のメンバに値を設定する。
 * mod_dav からの要請により、次のように値を組み立ててxi メンバに設定します。
 *
 * ・xi->prefix_uri : key = DIVY_DEAD_NS_PREFIX + (ネームスペースID), value = uri
 * ・xi->uri_prefix : key = uri, value = DIVY_DEAD_NS_PREFIX + (ネームスペースID)
 *
 * @param db dav_db *
 * @param xi dav_xmlns_info * ネームスペースを扱うための構造体へのポインタ
 */
static void _define_namespaces(dav_db *db, dav_xmlns_info *xi)
{
	divy_rdbo_dproperty *d_pr = NULL;
	apr_pool_t *p = db->r->pool;
	const char *ns_uri, *xmlns_str;

	for (d_pr = db->d_pr; d_pr; d_pr = d_pr->next) {
		/* ネームスペースURIを取り出す */
		ns_uri = apr_hash_get(db->ns_id_hash,
					apr_itoa(p, d_pr->ns_id),
					APR_HASH_KEY_STRING);
		if (IS_FILLED(ns_uri)) {
			xmlns_str = apr_psprintf(p, DIVY_DEAD_NS_PREFIX"%d",
							d_pr->ns_id);
			dav_xmlns_add(xi, xmlns_str, ns_uri); 
		}
	}
}

/*--------------------------------------------------------------
  Create & Initialize PROPERTY DATABASE provider Hook structure
  --------------------------------------------------------------*/
const dav_hooks_propdb dav_divy_hooks_propdb = {
	dav_divy_open, 
	dav_divy_close,
	dav_divy_define_namespaces,
	dav_divy_output_value,
	dav_divy_map_namespaces,
	dav_divy_store,
	dav_divy_remove,
	dav_divy_exists,
	dav_divy_first_name,
	dav_divy_next_name,
	dav_divy_get_rollback,
	dav_divy_apply_rollback,
	NULL
};


