/**
 * $Id$
 * repos.c
 *
 * Repository・プロバイダ
 *
 * 変更履歴
 *
 * 2003/02/13 Thu takehara NEW (empty implementation)
 *
 */

/* Apache header files */
#include "httpd.h"
#include "http_protocol.h"
#include "http_core.h"

#include "apr.h"
#include "apr_strings.h"
#include "apr_tables.h"
#include "apr_time.h"
#include "apr_date.h"
#include "apr_file_info.h"
#include "apr_file_io.h"
#include "apr_general.h"
#include "mod_dav.h"

#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif	/* APR_HAVE_STDLIB_H */
#if APR_HAVE_UNISTD_H
#include <unistd.h>     /* for getpid */
#endif	/* APR_HAVE_UNISTD_H */

/* document management headers */
#include "mod_dav_tf.h"
#include "repos.h"
#include "util.h"
#include "tf_parser.h"
#include "util_ml.h"
#include "util_auth.h"
#include "tf_rdbo.h"
#include "tf_rdbo_sql.h"
#include "tf_rdbo_sysmsg.h"
#include "tf_rdbo_dbms.h"
#include "tf_rdbo_group.h"
#include "tf_rdbo_user.h"
#include "tf_rdbo_clupdate.h"
#include "tf_sqlparser.h"
#include "lock.h"
#include "tf_storage.h"
#include "tf_autoindex.h"
#include "util_vscan.h"
#include "tf_folder.h"
#include "tf_validator.h"
#include "tf_valuecache.h"
#include "util_db.h"
#include "tf_thumbnail.h"
#include "tf_autodelete.h"
#include "tf_extcmd.h"
#include "tf_parser.h"
#include "tf_confirmreading.h"
#include "tf_extmap.h"
#ifdef TF_SUPPORT_CIPHER_STORAGE
#include "util_crypt.h"
#endif

#ifdef DAV_SUPPORT_POST_HANDLING
//#include "apreq_module_apache2.h"
#include "apreq2/apreq.h"
#include "apreq2/apreq_util.h"
#endif	/* DAV_SUPPORT_POST_HANDLING */

#ifdef DIVY_SUPPORT_PLUGIN
#include "tf_plugin.h"
#endif

APLOG_USE_MODULE(dav_tf);

/*--------------------------------------------------------------
  Define Fixed values
  --------------------------------------------------------------*/
/*
 * GET method処理をRepositoryプロバイダが処理する.
 * (note)
 * DAVプロバイダ(Repositoryプロバイダ)が、GETで取得されるresourceを
 * filesystemに直接マッピングするような方式を採用していた場合
 * (dav_fsのReleaseバージョン)、GET処理はRepositoryプロバイダが
 * 面倒を見る必要は無い。その場合には、special GET Handlingを行わない
 * と定義(dav_hooks_repository構造体のhandle_getを0に)することで、
 * Repositoryプロバイダは、set_headers、deliverの実装を行う必要が
 * なくなる。
 *
 * このサーバでは、special GET Handlingが必要であり、かつ、SEARCH method
 * でもset_headersを使用するため、実装は絶対に必要。
 * 
 */
#define ENABLE_GET_HANDLING  1

/**
 * propertyupdate エレメントと共に指定されるprop エレメントの中に
 * 複数のプロパティを指定できるかどうか
 * (関数_validate_mkcol_propertyupdate で使用)
 */
enum __divy_allow_prop_num {
	DIVY_ALLOW_SINGLE_PROP = 0,	/* 指定できない */
	DIVY_ALLOW_MULTI_PROP		/* 指定できる   */
};
typedef enum __divy_allow_prop_num divy_allow_prop_num;

/* 上書きPUTの前に退避される物理ファイルに付けられるサフィックス */
#define DIVY_BAKFILE_SUFFIX	"bak"

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare prototype function (private scope functions)
  ----------------------------------------------------------------------------*/
/* 分類されないその他特殊処理 */
static dav_stream * _create_dav_stream(apr_pool_t *p,
					const dav_resource *resource);
static void _fill_default_collection_info(request_rec *r,
						divy_rdbo_resource *rdb_r);
static dav_resource * _create_empty_resource(apr_pool_t *p,
					const char *uri, divy_uri_spec *u_spec, request_rec *r,
					int is_collection, int use_header);
static dav_error * _parse_sharedcollectioncreate_elem(apr_pool_t *p,
		const apr_xml_doc *doc, divy_rdbo_shsrccol **shsrccol_pr);
static apr_uint32_t _build_propflag(request_rec *r, dav_walker_ctx *ctx,
					int depth, divy_rdbo_resource *rdb_r);
static dav_error * _open_empty_physical_resource(request_rec *r,
					int fflags, const char *oldpath,
					divy_pfile_t **pfile, apr_file_t **fd);
static dav_error * _commit_physical_resource(request_rec *r,
					const char *path, const char *newpath,
					divy_rdbo_resource *rdb_r, int is_update);

/* validator */
static dav_error * _validate_parent_collection(apr_pool_t *p,
					const dav_resource *resource);
static dav_error * _validate_mkcol_propertyupdate(apr_pool_t *p,
					divy_allow_prop_num allow_multiprop,
					const apr_xml_doc *doc);
static dav_error * _validate_mkcol_dbfolder(apr_pool_t *p,
					const divy_rdbo_resource *rdb_r,
					const divy_rdbo_keylist *keylist_pr);
static dav_error * _validate_mkcol_dbshfolder(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					const divy_rdbo_shsrccol *shsrccol_pr);
static dav_error * _validate_remove_user(request_rec *r,
					const dav_resource *resource,
					const divy_rdbo_usr *usr_pr,
					dav_response **response);
static dav_error * _validate_remove_group(request_rec *r,
					const dav_resource *resource,
					const divy_rdbo_grp *grp_pr,
					dav_response **response);
static dav_error * _validate_remove_sql(request_rec *r,
					const dav_resource *resource,
					const divy_rdbo_sql *sql_pr,
					dav_response **response);
static dav_error * _validate_remove_dbms(request_rec *r,
					const dav_resource *resource,
					const divy_rdbo_dbms *dbms_pr,
					dav_response **response);
static dav_error * _validate_create_relation(request_rec *r,
					const dav_resource *src,
					const dav_resource *dst);
static dav_error * _validate_inheritsql(request_rec *r,
					const dav_resource *resource,
					dav_response **response);
static dav_error * _validate_copy_group(request_rec *r,
					const dav_resource *src,
					const dav_resource *dst);
static dav_error * _validate_move_user(request_rec *r,
					const dav_resource *src,
					const dav_resource *dst,
					dav_response **response);
static dav_error * _validate_move_group(request_rec *r,
					const dav_resource *src,
					const dav_resource *dst,
					dav_response **response);

/* 作成処理 */
static dav_error * _create_regular_collection(request_rec *r,
					divy_rdbo_resource *rdb_r);
static dav_error * _create_dbfolder(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc);
static dav_error * _create_sharedcollection(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc);
static dav_error * _create_user(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc);
static dav_error * _create_group(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc);
static dav_error * _create_sql(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc);
static dav_error * _create_sysmsg(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc);
static dav_error * _create_dbms(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc);
static dav_error * _create_ruser(request_rec *r,
					const dav_resource *src,
					dav_resource *dst);
static dav_error * _create_rgrp(request_rec *r,
					const dav_resource *src,
					dav_resource *dst);
static dav_error * _create_rsql(request_rec *r,
					const dav_resource *src,
					dav_resource *dst);

/* コピー処理 */
static dav_error * _copy_regular_resource(request_rec *r,
					int depth,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r);
static dav_error * _copy_user(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r);
static dav_error * _copy_group(request_rec *r,
					int depth,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r);
static dav_error * _copy_sysmsg(request_rec *r,
					const dav_resource *src,
					dav_resource *dst);
static dav_error * _copy_dbms(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r);
static dav_error * _copy_ruser(request_rec *r,
					const dav_resource *src,
					dav_resource *dst);
static dav_error * _copy_rsql(request_rec *r,
					const dav_resource *src,
					dav_resource *dst,
					dav_response **response);
static dav_error * _copy_rgrp(request_rec *r,
					const dav_resource *src,
					dav_resource *dst,
					dav_response **response);

/* 移動処理 */
static dav_error * _move_regular_resource(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r);
static dav_error * _move_user(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r);
static dav_error * _move_group(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r);
static dav_error * _move_sql(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r);
static dav_error * _move_dbms(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r);
static dav_error * _move_ruser(request_rec *r,
					const dav_resource *src,
					dav_resource *dst);
static dav_error * _move_rsql(request_rec *r,
					const dav_resource *src,
					dav_resource *dst,
					dav_response **response);
static dav_error * _move_rgrp(request_rec *r,
					const dav_resource *src,
					dav_resource *dst,
					dav_response **response);
/* 削除処理 */
static dav_error * _remove_regular_resource(request_rec *r,
					divy_rdbo_resource *rdb_r);
static void _sort_mkcol_keylist_elem(apr_pool_t *p,
					const divy_rdbo_keylist *src,
					divy_rdbo_keylist **dst);
static dav_error * _remove_user(request_rec *r, divy_rdbo_usr *usr_pr);
static dav_error * _remove_group(request_rec *r, divy_rdbo_grp *grp_pr);
/* 特殊PROPFIND */
static dav_error * _get_rootfolder(request_rec *r,
					int depth,
					divy_rdbo_resource *rdb_r);
static dav_error * _get_groupfolder(request_rec *r,
					divy_rdbo_resource *rdb_r);
static dav_error * _get_special_entity(request_rec *r,
					const char *registdt,
					const char *updatedt,
					divy_rdbo_resource *rdb_r);
static dav_error * _get_special_relation(request_rec *r,
					const char *prefix,
					divy_rdbo_resource *rdb_r);
static dav_error * _get_virtual_entiry(request_rec *r,
					divy_rdbo_resource *rdb_r);
static dav_error * _walk_resource(request_rec *r,
					apr_pool_t *wp,
					const dav_walk_params *params,
					divy_rdbo_resource *rdb_r,
					int islocknull,
					dav_response **response);
/* ウィルス検索関連 */
static dav_error * _setup_virusscan(request_rec *r, dav_stream *d_stream);
static dav_error * _scanvirus(request_rec *r, dav_stream *stream,
					const char *filepath);
/* セキュリティ関係 */
static void _set_enhance_security_headers(request_rec *r);
#ifdef DIVY_SUPPORT_PRINTER
/* プリンタ関連 */
static void _print_file(request_rec *r, divy_rdbo_resource *rdb_r, const char *filepath);
typedef struct cups_option_t cups_option_t;
extern int		cupsPrintFile(const char *printer, const char *filename,
			              const char *title, int num_options,
				      cups_option_t *options);
extern int	cupsLastError(void);
#endif	/* DIVY_SUPPORT_PRINTER */
#ifdef DIVY_SUPPORT_PREFLIGHT_PLUGIN
static void _send_preflightfolder(request_rec *r, divy_uri_spec *bu_spec,
								const char *filepath, const char *name);
#endif	/* DIVY_SUPPORT_PREFLIGHT_PLUGIN */

/*--------------------------------------------------------------
  Define Hook functions
  --------------------------------------------------------------*/
/**
 * 指定されたリクエストに含まれるuri が示すリソースまたはコレクションを
 * 取得して、resource に渡す。
 * uri は必ず一意であるため、resource も一つだけである。
 *
 * (note) この関数が取得するプロパティについて
 * 	この関数は、以下のLiveプロパティ及び全てのDeadプロパティを取得
 * 	しません。
 *
 * 	・mailwatch
 * 	・sharedcollection
 *
 * 	これらのLiveプロパティ及びDeadプロパティは、全てのメソッドから
 * 	呼ばれるこの関数で取得するにはコストが高いという判断からです。
 * 	必要であれば、別の関数で独自に取得して下さい。
 * 	なお、dav_divy_walk ではDIVYで定義されている全てのプロパティを
 * 	取得するようになっています。
 *
 * @param r request_rec *
 * @param root_dir const char * root path を表すURI
 * @param label const char *
 * @param resource dav_resource ** 取得されたリソースへのポインタ
 * @return dav_error エラー (500, 400)
 */
static dav_error * dav_divy_get_resource(request_rec *r,
                                         const char *root_dir,
                                         const char *label,
                                         int use_checked_in, 
                                         dav_resource **resource)
{
	dav_error *err             = NULL;
	dav_resource *res          = NULL;
	dav_resource_private *info = NULL;
	divy_rdbo_resource *rdb_r  = NULL;
	apr_pool_t *p              = r->pool;
	char *uri                  = NULL;
	divy_uri_spec *u_spec      = NULL;
	apr_uint32_t propflag      = 0;

#ifdef DIVY_SUPPORT_LOGACCESS
	/* ロガー用にロケーション名をセットする */
	divy_logger_set_location(r, root_dir);
#endif	/* DIVY_SUPPORT_LOGACCESS */

	TRACE(p);

	*resource = NULL;	/* 初期化 */

	/* uri の末尾からスラッシュを全て除去し、uri の中に含まれる
	 * 2つ以上のスラッシュを1つのスラッシュに変更する */
	uri = dav_divy_remove_endslash(p, r->uri);
	ap_no2slash(uri);

	/*
	 * URIをパースして対象リソースの種類を判定する
	 */
	if (divy_parse_uri(p, root_dir, uri, &u_spec)) {
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * このリクエストが必要最低限のアクセス許可条件を満たしているかどうか
	 * を検証する
	 */
	err = divy_validate_minimum_access_guarantee(r, u_spec);
	if (err) {
		return err;
	}

	/* 空リソースの生成 */
	res = _create_empty_resource(p, uri, u_spec, r, 0, 1);
	if (res == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to create empty resource instance.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	info  = res->info;
	rdb_r = res->info->rdb_r;

	/*
	 * 通常のリソース・コレクション,
	 * $root/$userid,
	 * ユーザごみ箱,
	 * $root/Group Folder/$groupid,
	 * グループごみ箱,
	 * $root/.dbfolder/$keys,
	 * $root/.dbshfolder/$keys,
	 * $root/.management/UPDATE/$clmodule
	 * $root
	 * $root/Utility_Folder/Updated_Client/$clmodule だった時
	 */
	if (u_spec->uritype & DIVY_URITYPE_REGULAR ||
	    u_spec->infotype == DIVY_INFOTYPE_user_e         ||
	    u_spec->infotype == DIVY_INFOTYPE_user_trash     ||
	    u_spec->infotype == DIVY_INFOTYPE_user_trash_e0  ||
	    u_spec->infotype == DIVY_INFOTYPE_user_trash_ex  ||
	    u_spec->infotype == DIVY_INFOTYPE_group_e        ||
	    u_spec->infotype == DIVY_INFOTYPE_group_trash    ||
	    u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
	    u_spec->infotype == DIVY_INFOTYPE_group_trash_ex ||
	    u_spec->infotype == DIVY_INFOTYPE_dbfolder_e     ||
	    u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e   ||
	    u_spec->infotype == DIVY_INFOTYPE_m_update_e     ||
	    u_spec->infotype == DIVY_INFOTYPE_root           ||
	    u_spec->infotype == DIVY_INFOTYPE_u_update_e) {
		char *swap_uri = NULL;

		/* fullname を取得する */
		if (u_spec->infotype == DIVY_INFOTYPE_root) {
			propflag |= DIVY_GET_PROP_fullname;
		}
		/* URIの変換を実施(偽バインド) */
		if (u_spec->infotype == DIVY_INFOTYPE_u_update_e) {
			swap_uri = rdb_r->uri;
			/* $root/.management/UPDATE/$modulename にしてしまう */
			rdb_r->uri = divy_build_m_update_uri(p, dav_divy_get_root_uri(r),
								u_spec->final_path_segment);
		}

		/* ユーザ拡張ステータスをサポートしていれば
		 * View属性の取得が必要 (グループフォルダ自身はdepth =0 では不要) */
		if (divy_support_extenduserstatus(r) &&
				u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {
			propflag |= DIVY_GET_PROP_resourcestate;
		}

		/*
		 * リポジトリDB から対象リソースまたはコレクションの
		 * プロパティを取得する
		 */
		if (divy_rdbo_get_hierarchy_property(r, rdb_r, 0, propflag, NULL)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get a resource property from repository.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* URIを元に戻す */
		if (u_spec->infotype == DIVY_INFOTYPE_u_update_e) {
			rdb_r->uri = swap_uri;
		}

		/* 存在していた */
		if (IS_FILLED(rdb_r->rsid)) {
			res->exists = 1;
			if (rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
				res->collection = 1;
			}
			else {
				res->collection = 0;
			}

			/* Last-Modified 時刻をリポジトリから取り出した値にする */
			apr_time_ansi_put(&r->mtime, rdb_r->getlastmodified);

			/* 更新クライアントだった場合 */
			if (u_spec->infotype == DIVY_INFOTYPE_m_update_e) {
				if (divy_rdbo_get_clmodule_property(r, rdb_r->uri, &info->list.clmodule_pr)) {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get client module property.");
					return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				}
			}
		}
		/* 存在していなかった */
		else {
			res->exists     = 0;
			/*
			 * 存在していなかった場合、collection は 0, 
			 * rdb_r->resourcetype をDIVY_TYPE_RESOURCE にする
			 * (note)
			 *  lock-null esource をLock プロバイダで正しく
			 *  扱うための対応
			 */
			res->collection = 0;
			rdb_r->resourcetype = DIVY_TYPE_RESOURCE;

			/*
			 * path_info を空にする。
			 * mod_dav 側でnull resource のnull lock 有無のチェック
			 * を行わせるためには、path_info をこのようにしなければ
			 * ならない。不自然だがmod_dav の要請なので仕方がない。
			 * (意味)
			 * exists が 0であるにも関わらず path_info つまり、
			 * PUT /divy/xxxx のxxxx があるということは、すなわち
			 * null resource であることを意味するという解釈。
			 */
			r->path_info = '\0';
		}
	}
	/*
	 * "処理する場所"かつ"仮想フォルダ" を表していた場合 
	 */
	else if (u_spec->uritype & DIVY_URITYPE_ARBITER &&
		 u_spec->uritype & DIVY_URITYPE_VIRTUAL) {
		/* 存在するものとして返却すべき */
		rdb_r->resourcetype = DIVY_TYPE_COLLECTION;
		res->exists     = 1;
		res->collection = 1;
	}
	/*
	 * "処理する場所" を表していた場合
	 * (note)
	 * 	"仮想フォルダ" である可能性は前段でチェックされているため
	 * 	ここに該当するのは純粋に"処理する場所"を表しているURIだけ。
	 * 	SEARCH, OPTIONS 以外のメソッドは本来許可できません。
	 * 	この判定はこれらのif 文に入る前にで行われているものとします。
	 */
	else if (u_spec->uritype & DIVY_URITYPE_ARBITER) {
		/* 存在するものとして返却すべき */
		rdb_r->resourcetype = DIVY_TYPE_COLLECTION;
		res->exists     = 1;
		res->collection = 1;
	}
	/*
	 * 特殊フォルダを表していた場合
	 */
	else if (u_spec->uritype & DIVY_URITYPE_SPECIAL) {
		char *err_str       = NULL;
		res->exists         = 0;
		res->collection     = 0;
		switch (u_spec->infotype) {

			/* グループフォルダ */
			case DIVY_INFOTYPE_group:
				/* 存在するものとして返却すべき */
				rdb_r->resourcetype = DIVY_TYPE_COLLECTION;
				res->exists     = 1;
				res->collection = 1;
				break;

			/* ユーザ, 統計情報(in 管理者フォルダ) */
			case DIVY_INFOTYPE_m_user_e:
			case DIVY_INFOTYPE_m_status_e:
				if (divy_rdbo_get_user_property(r,
							u_spec->final_path_segment, &rdb_r->usr_pr))
					err_str = "user";

				if (rdb_r->usr_pr) {
					res->exists     = 1;
					res->collection = 1;
				}
				break;

			/* グループ(in 管理者フォルダ) */
			case DIVY_INFOTYPE_m_group_e:
				if (divy_rdbo_get_hierarchy_group_property(r,
							u_spec->other_part, 0, &rdb_r->grp_pr, NULL))
					err_str = "group";

				if (rdb_r->grp_pr) {
					res->exists     = 1;
					res->collection = 1;
				}
				break;

			/* SQL(in 管理者フォルダ) */
			case DIVY_INFOTYPE_m_sql_e:
				if (divy_rdbo_get_sql_property(r,
							u_spec->other_part, &rdb_r->sql_pr))
					err_str = "sql";

				if (rdb_r->sql_pr) {
					res->exists     = 1;
					res->collection = 1;
				}
				break;

			/* システムメッセージ(in 管理者フォルダ) */
			case DIVY_INFOTYPE_m_msg_e:
				if (divy_rdbo_get_sysmsg_property(r,
							u_spec->final_path_segment, &info->list.sysmsg_pr))
					err_str = "sysmsg";

				if (info->list.sysmsg_pr) {
					res->exists     = 1;
					res->collection = 1;
				}
				break;

			/* DBMS 接続情報(in 管理者フォルダ) */
			case DIVY_INFOTYPE_m_dbms_e:
				if (divy_rdbo_get_dbms_property(r,
							u_spec->final_path_segment, &rdb_r->dbms_pr))
					err_str = "dbms";

				if (rdb_r->dbms_pr) {
					int mnum = divy_get_method_number(r);

					/* リポジトリDBに対するアクセスは禁止 */
					if (strcmp(rdb_r->dbms_pr->dbmsid, DIVY_REPOS_DBMSID) == 0 &&
						(mnum == M_DELETE || mnum == M_PROPPATCH ||
						 mnum == M_MKCOL  ||
						 mnum == M_MOVE   || mnum == M_COPY ||
						 mnum == M_LOCK   || mnum == M_UNLOCK)) {

						ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
								"Could not access respository database entry.");
						return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
					}
					res->exists     = 1;
					res->collection = 1;
				}
				break;
			/* ユーザリレーション */
			case DIVY_INFOTYPE_m_group_ruser:
				if (divy_rdbo_get_rusr(r, uri, &info->list.rusr_pr))
					err_str = "user relation";

				if (info->list.rusr_pr) {
					res->exists     = 1;
					res->collection = 1;
				}
				break;

			/* SQL リレーション */
			case DIVY_INFOTYPE_m_group_rsql:
				if (divy_rdbo_get_rsql(r, uri, &info->list.rsql_pr))
					err_str = "sql relation";

				if (info->list.rsql_pr) {
					res->exists     = 1;
					res->collection = 1;
				}
				break;

			/* グループリレーション */
			case DIVY_INFOTYPE_m_user_rgrp:
			case DIVY_INFOTYPE_m_sql_rgrp:
				if (divy_rdbo_get_rgrp(r, uri, &info->list.rgrp_pr))
					err_str = "group relation";

				if (info->list.rgrp_pr) {
					res->exists     = 1;
					res->collection = 1;
				}
				break;

			/* 送信メール */
			case DIVY_INFOTYPE_mail_e:
				/* DB に格納されることはないので必ずnot found */
				rdb_r->resourcetype = DIVY_TYPE_RESOURCE;
				res->exists     = 0;
				res->collection = 0;

				break;

#ifdef DAV_SUPPORT_POST_HANDLING
		/* システムCGI */
			case DIVY_INFOTYPE_cgibin_e:
				if (divy_cgi_get_property(r, u_spec, &info->list.cgi_pr)) {
					err_str = "cgi property";
				}
				if (info->list.cgi_pr) {
					res->exists     = 1;
					res->collection = 0;	/* リソースである */
				}
				break;

#endif	/* DAV_SUPPORT_POST_HANDLING */
#ifdef DIVY_SUPPORT_PLUGIN
			/* 個々のプラグイン */
			case DIVY_INFOTYPE_plugin_e:
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, ""); 

			/* プラグイン専用のアップロードファイル */
			case DIVY_INFOTYPE_plugin_uploads_e:
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, ""); 
				break;

			/* プラグインCGI */
			case DIVY_INFOTYPE_plugin_cgibin_e:
				if (divy_cgi_get_property(r, u_spec, &info->list.cgi_pr)) {
					err_str = "plugin's cgi property";
				}
				if (info->list.cgi_pr) {
					res->exists     = 1;
					res->collection = 0;	/* リソースである */
				}
				break;
			/* プラグインコンフィグファイル */
			case DIVY_INFOTYPE_plugin_conf_e:
				res->exists     = 0;
				res->collection = 0;	/* リソースである */
				break;
#endif	/* DIVY_SUPPORT_PLUGIN */
			/* 短縮URL */
			case DIVY_INFOTYPE_shorten:
				res->exists = 1;
				/* TODO: 
				   短縮URLがコレクションなのかリソースなのかを判断しなさい
				 */
				res->collection = 0;

				break;
			/* 上記以外の場合 */
			default:
				/* 事前に防いでいるはず */
				ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Unexpected infotype for get resource."
					"(uri = %s, uritype = %d, infotype = %d)",
					uri, u_spec->uritype, u_spec->infotype);
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, ""); 
				break;
		}

		if (err_str) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to read property of \"%s\".", err_str);
			/*
			 * BugTrack-TeamFile_Server/1035 対策
			 *
			 * err_strに値が入ってくるのは致命的であるが、それがはっきりと
			 * 分かってしまうINTERNAL_SERVER_ERRORを返さないようにします。
			 */
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		if (res->collection) {
			rdb_r->resourcetype = DIVY_TYPE_COLLECTION;
		}

		if (res->exists == 0) {
			r->path_info = '\0';
		}
	}
	/*
	 * URIが正しくない or 処理場所を示さない仮想フォルダの場合
	 */
	else {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Could not access this resource.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* 値をセットして終了 */
	*resource = res;

	/*
	 * リソースへのアクセスが正当かどうかの検証
	 */
	err = divy_validate_request_resource(r, res);
	if (err) return err;

	return NULL;
}

/**
 * 指定されたresource の親リソース(コレクション)を取得して、
 * parent_resource に渡す。
 * なお、指定されたresource がルートコレクションだったら、その親は存在しない
 * ので、NULL が渡されることに注意する。
 *
 * @param resource const dav_resource * 検査対象のリソースまたはコレクション
 * @param parent_resource dav_resource ** 取得された親リソース
 * @return dav_error エラー (500, 400)
 */
static dav_error * dav_divy_get_parent_resource(const dav_resource *resource,
                                                dav_resource **parent_resource)
{
	dav_resource *res          = NULL;
	divy_rdbo_resource *rdb_r  = NULL;
	apr_pool_t *p              = resource->pool;
	request_rec *r             = resource->info->r;
	char *parent_uri           = NULL;
	divy_uri_spec *u_spec      = NULL;
	apr_uint32_t propflag      = 0;
	
	TRACE(p);

	/* resource のuri が $root だったら NULL を返す (see mod_dav.h) */
	if (resource->uri == NULL ||
	    !strcmp(resource->uri, dav_divy_get_root_uri(r))) {
		*parent_resource = NULL;
		return NULL;
	}

	/*
	 * resource->info->parent_resource があれば、何も行わない。
	 * (note)
	 *   resource->info->parent_resourceは、１回目の本関数呼び出しで
	 *   作成されます。
	 *   このコードは、１回のリクエストの中で本関数がが何度もCall
	 *   されてしまう状況を鑑みて、性能向上を目的として導入されました。
	 */
	if (resource->info->parent_resource) {
		*parent_resource = resource->info->parent_resource;
		return NULL;
	}

	/* 親コレクションのURI を取得する */
	parent_uri = divy_get_parenturi(p, resource->uri);

	/* 空リソースの生成 */
	res = _create_empty_resource(p, parent_uri, NULL, r, 1, 1);
	if (res == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to create empty parent resource instance.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	rdb_r  = res->info->rdb_r;
	u_spec = rdb_r->u_spec;

	/*
	 * 通常のリソース・コレクション,
	 * $root/$userid,
	 * ユーザごみ箱,
	 * $root/Group Folder/$groupid,
	 * グループごみ箱,
	 * $root/.dbfolder/$keys,
	 * $root/.dbshfolder/$keys, 
	 * $root	だった時
	 */
	if (u_spec->uritype & DIVY_URITYPE_REGULAR ||
	    u_spec->infotype == DIVY_INFOTYPE_user_e         ||
	    u_spec->infotype == DIVY_INFOTYPE_user_trash     ||
	    u_spec->infotype == DIVY_INFOTYPE_user_trash_e0  ||
	    u_spec->infotype == DIVY_INFOTYPE_user_trash_ex  ||
	    u_spec->infotype == DIVY_INFOTYPE_group_e        ||
	    u_spec->infotype == DIVY_INFOTYPE_group_trash    ||
	    u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
	    u_spec->infotype == DIVY_INFOTYPE_group_trash_ex ||
	    u_spec->infotype == DIVY_INFOTYPE_dbfolder_e     ||
	    u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e   ||
	    u_spec->infotype == DIVY_INFOTYPE_root) {
		/*
		 * リポジトリDB から親コレクションのプロパティを取得する
		 */
		if (divy_rdbo_get_hierarchy_property(r, rdb_r, 0, propflag, NULL)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get a resource property of parent"
				"from repository.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	
		/* 存在していた */
		if (IS_FILLED(rdb_r->rsid)) {
			res->exists = 1;
			res->collection = (rdb_r->resourcetype == DIVY_TYPE_COLLECTION) ? 1 : 0;
		}
		/* 存在していなかった */
		else {
			res->exists     = 0;
			res->collection = 0;
		}
	}
	/*
	 * 仮想フォルダを表していた場合
	 */
	else if (u_spec->uritype & DIVY_URITYPE_VIRTUAL) {
		/* 親が仮想フォルダなら、子リソースを持ってもよい
		 * ので、存在するものとして返す */
		res->exists     = 1;
		res->collection = 1;
	}
	/*
	 * 特殊フォルダを表していた場合
	 */
	else if (u_spec->uritype & DIVY_URITYPE_SPECIAL) {
		char *err_str   = NULL;
		res->exists     = 0;
		res->collection = 0;
		switch (u_spec->infotype) {

			/* グループフォルダ */
			case DIVY_INFOTYPE_group:
				/* 親が仮想フォルダなら、子リソースを持ってもよい
				 * ので、存在するものとして返す */
				res->exists     = 1;
				res->collection = 1;
				break;

			/* DIVY ユーザ */
			case DIVY_INFOTYPE_m_user_e:
				if (divy_rdbo_get_user_property(r,
							u_spec->final_path_segment, &rdb_r->usr_pr))
					err_str = "user";

				if (rdb_r->usr_pr) {
					res->exists     = 1;
					res->collection = 1;
				}
				break;

			/* DIVY グループ */
			case DIVY_INFOTYPE_m_group_e:
				if (divy_rdbo_get_hierarchy_group_property(r,
							u_spec->other_part, 0, &rdb_r->grp_pr, NULL))
					err_str = "group";

				if (rdb_r->grp_pr) {
					res->exists     = 1;
					res->collection = 1;
				}
				break;

			/* DIVY SQL */
			case DIVY_INFOTYPE_m_sql_e:
				if (divy_rdbo_get_sql_property(r,
							u_spec->other_part, &rdb_r->sql_pr))
					err_str = "sql";

				if (rdb_r->sql_pr) {
					res->exists     = 1;
					res->collection = 1;
				}
				break;

#ifdef DIVY_SUPPORT_PLUGIN
			/* 個々のプラグイン */
			case DIVY_INFOTYPE_plugin_e:
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, ""); 
#endif	/* DIVY_SUPPORT_PLUGIN */

			/* 
			 * 以下は親として指定され得ないリソースのはず.
			 * 指定された場合には、恐らく指定ミスなので、それを
			 * 教えてあげます。(エラーですが。)
			 *
			 *   更新クライアント , システムメッセージ ,
			 *   DBMS 接続情報 , 統計情報
			 *   ユーザリレーション , SQLリレーション
			 *   グループリレーション, 送信メール,
			 *   システムCGI, プラグインCGI
			 */
			case DIVY_INFOTYPE_m_update_e:
			case DIVY_INFOTYPE_m_msg_e:
			case DIVY_INFOTYPE_m_dbms_e:
			case DIVY_INFOTYPE_m_status_e:
			case DIVY_INFOTYPE_m_group_ruser:
			case DIVY_INFOTYPE_m_group_rsql:
			case DIVY_INFOTYPE_m_user_rgrp:
			case DIVY_INFOTYPE_m_sql_rgrp:
			case DIVY_INFOTYPE_mail_e:
#ifdef DAV_SUPPORT_POST_HANDLING
			case DIVY_INFOTYPE_u_update_e:
			case DIVY_INFOTYPE_cgibin_e:
#endif	/* DAV_SUPPORT_POST_HANDLING */
#ifdef DIVY_SUPPORT_PLUGIN
			case DIVY_INFOTYPE_plugin_cgibin_e:
#endif	/* DIVY_SUPPORT_PLUGIN */

				ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The resource of a specifed uri should be "
					"single hierachy."
					"(uri = %s, uritype = %d, infotype = %d)",
					parent_uri, u_spec->uritype, u_spec->infotype);

				return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");

			/* 上記以外 */
			default:
				/* 事前に防いでいるはずなので恐らくバグ */
				ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Unexpected infotype for get parent resource. "
					"(uri = %s, uritype = %d, infotype = %d)",
					parent_uri, u_spec->uritype, u_spec->infotype);
				return dav_new_error(p,
					HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				break;
		}

		if (err_str) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to read parent resource property of "
				"\"%s\".", err_str);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		if (!res->collection) {
			rdb_r->resourcetype = DIVY_TYPE_RESOURCE;
		}
	}
	/*
	 * 親URI が処理場所を示すURI の場合
	 */
	else if (u_spec->uritype & DIVY_URITYPE_ARBITER) {
		/* 親が仮想フォルダの属性を持たずに処理場所を
		 * 表しているのなら、それは存在しないと返却する */
		res->exists     = 0;
		res->collection = 1;
	}
	/* 
	 * URI が正しくない
	 */
	else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"We are Not permitted to do this operation.(uri = %s)",
			parent_uri);
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* 値をセットして終了 */
	*parent_resource = res;

	/* 取得したres をresource->info->parent_resource にも
	 * 入れておく(性能向上対応) */
	resource->info->parent_resource = res;

	return NULL;
}

/**
 * 指定された２つのリソースが同一かどうかを判定して返却する。
 *
 * [ ２つが同じであるとみなされる条件 ]
 * (1) 同じrepository プロバイダを使用していること。
 * (2) rsid があれば、同じrsidを持っていること。
 * (3) rsid がないならば、同じURI文字列であること
 *
 * (note)
 * 	final path segment 末尾のスラッシュはあっても無くても同じものと
 * 	みなします。
 * 	(例)	"/aaaa/" "/aaaa" は同じURIであると判断します。
 *
 * @param res1 dav_resource *
 * @param res2 dav_resource *
 * @return int 0: 異なる / 1: 等しい
 */
static int dav_divy_is_same_resource(const dav_resource *res1, 
                                     const dav_resource *res2)
{
	apr_pool_t *p = res1->pool;

	if (res1 == NULL || res2 == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to compare operation, because "
			"the resource of the comparison object is NULL.");
		return 0;
	}
	
	/* 同じリポジトリプロバイダか？*/
	if (res1->hooks != res1->hooks) return 0;

	/* 同じrsid か ? */
	if (IS_FILLED(res1->info->rdb_r->rsid) &&
	    IS_FILLED(res2->info->rdb_r->rsid)) {
		if (strcmp(res1->info->rdb_r->rsid,
			   res2->info->rdb_r->rsid) != 0) return 0;
	}
	else {
		/* URIによる比較(final path segment末尾のスラッシュは無視 */ 
		if (strcmp(dav_divy_remove_endslash(p, res1->uri),
			   dav_divy_remove_endslash(p, res2->uri)) != 0) return 0;
	}

	return 1;
}

/**
 * 指定された res1 が res2 の直接または間接的に親リソース(コレクション) と
 * なっているかどうかを判定する。
 *
 * [ 親コレクションとみなされる条件 ]
 *
 * (1) 同じrepository プロバイダを使用していること。
 * (2) res2->uri がres1->uri の部分要素として現れること。
 *     但し、部分要素はres1->uri[0]から始まってres1->uri[strlen(res2->uri)] まで
 *     一致している必要があります。
 * (例1)
 *   res1->uri = /divy/aaa/bbb/ccc
 *   res2->uri = /divy/aaa/bbb/ccc/ddd/eee 
 *   --> res1->uri の先頭から"/divy/aaa/bbb/ccc"が部分一致しているので、
 *       res1 はres2 の親リソースである。(間接的な親)
 *
 * (例2)
 *   res1->uri = /divy/aaa/bbb/ccc
 *   res2->uri = /divy2/divy/aaa/bbb/ccc/ddd
 *   --> res1->uri の先頭からres2->uriとは一致しないので、親リソースではない
 *
 * @param res1 dav_resource * 比較対象親リソース
 * @param res2 dav_resource * 比較対象の子リソース
 * @return int 0: 異なる / 1: 等しい
 */
static int dav_divy_is_parent_resource(const dav_resource *res1,
                                       const dav_resource *res2)
{
	apr_size_t len1,len2;

	if (res1 == NULL || res2 == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to compare operation, because "
			"the resource of the comparison object is NULL.");
		return 0;
	}

	/*
	 * (1) 同じリポジトリプロバイダか？
	 */
	if (res1->hooks != res1->hooks) return 0;

	/*
	 * (2) res2->uri がres1->uri の部分要素として現れているか？
	 *
	 * res1->uriの先頭からstrlen(res2->uri)までが一致しなければ
	 * ならない
	 */
	len1 = (IS_FILLED(res1->uri)) ? strlen(res1->uri) : 0;
	len2 = (IS_FILLED(res2->uri)) ? strlen(res2->uri) : 0;
	
	return (len2 > len1 &&
		memcmp(res1->uri, res2->uri, len1) == 0 &&
		res2->uri[len1] == '/');
}

/**
 * 指定されたresourceの出力用(PUT用)Streamを、指定されたmodeで開く。
 * 開かれたStreamを表すオブジェクト(構造体)は、引数*streamにセットされ
 * 返却される。
 *
 * @param resource dav_resource * 1つのリソースを表す構造体へのポインタ
 * @param mode dav_stream_mode streamを開く際のモード
 *              DAV_MODE_WRITE_TRUNC    : 上書きモード(存在すれば消してしまう)
 *              DAV_MODE_WRITE_SEEKABLE : ランダムアクセスが可能なモード
 *                                        (dav_divy_seek_stream関数が使用可能)
 * @param stream dav_stream ** Streamを表す構造体へのポインタ
 * @return DAVエラー (500)
 *
 */
static dav_error * dav_divy_open_stream(const dav_resource *resource,
                                        dav_stream_mode mode,
                                        dav_stream **stream)
{
	dav_error *err = NULL;
	dav_stream *d_stream = NULL;
	apr_int32_t f_flags;

	divy_rdbo_resource *rdb_r = resource->info->rdb_r;
	request_rec *r            = resource->info->r;
	apr_pool_t *p             = resource->pool;
	const char *tenc          = apr_table_get(r->headers_in, "Transfer-Encoding");

	/* dav_stream (PUTファイルの情報を保持する構造体)の生成 */
	*stream = d_stream = _create_dav_stream(p, resource);

	/*
	 * PUT時のレスポンスセキュリティヘッダを設定する
	 */
	_set_enhance_security_headers(r);

	/* 親コレクションをチェックする */
	if ((err = _validate_parent_collection(p, resource)) != NULL) {
		return err;
	}

	/* 表示名称の禁則文字チェック
	 * (note) 親URIは既にチェック済みのはず */
	if (divy_validate_pathsegment(p, DIVY_IGNORE_COLON | DIVY_IGNORE_ASTERISK, rdb_r->displayname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The part of uri has invalid char." );
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* Chunk PUT の判定 */
	if (IS_FILLED(tenc) && strcasecmp(tenc, "chunked") == 0) {
		/* サポートしているか? */
		if (divy_support_chunkput(r)) {
			d_stream->is_chunkput = 1;	/* チャンク転送を使う */
		}
		else {
			/* 未サポートのときのchunk PUT は許可しない.
			 * RFC2616 に従って HTTP_NOT_IMPLEMENTED を返却します */
			ERRLOG0(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"The chunked PUT operation is not supported.");

			return dav_new_error(p, HTTP_NOT_IMPLEMENTED, 0, 0, "");
		}
	}
	/* else はHTTP 1.1 では未定義なので書きません */

	/* (note)
	 * ファイル書き込みが途中でキャンセルされたり、異常終了する場合が
	 * あるので、全てが完了するまでは一時ファイルに保存する.
	 * 終わったら一時ファイルを正式名称にrename します.
	 */

	/* modeによるf_flagsの設定 */
	switch (mode) {
	
		/* 上書きモード */
		case DAV_MODE_WRITE_TRUNC:
			f_flags = APR_WRITE | APR_CREATE | APR_TRUNCATE | APR_BINARY;
			
			break;

		/* 差分書き込み(追記)モード (Content-Rangeが指定された場合) */
		case DAV_MODE_WRITE_SEEKABLE:

			/*
			 * 2003/05/07 Wed
			 * Content-Range ヘッダによる差分PUTはサポートしません。
			 * 既存resource が、今まさにPUTされてきたresourceと
			 * 同一であることを検証＆保証できないためです。mod_dav.cは
			 * Content-Rangeヘッダの整合性しかチェックしてくれません。
			 */
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");

		/* Read-Only モード */
		default:
			/* Read-Onlyモードにおけるopen_streamの使用は停止された
			   はず。ここにくるのはおかしい。*/
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
				"You should use new-reading api. open_stream "
				"is not recommended using read-file."
				"(uri = %s, userid = %s)",
				resource->uri, divy_get_userid(r));
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	if (resource->exists == 0) {
		rdb_r->physicalpath = NULL;
	}

	/* 一時ファイルを開く */
	if ((err = _open_empty_physical_resource(r, f_flags, rdb_r->physicalpath,
								&d_stream->pfile, &d_stream->f)) != NULL) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to open empty physical resource. (uri = %s, userid = %s)",
				resource->uri, divy_get_userid(r));
		return err;
	}

	/* ウィルス検索機能の準備 */
	if ((err = _setup_virusscan(r, d_stream)) != NULL) {
		return err;
	}

	return NULL;
}

/**
 * 指定されたstreamを閉じる。
 *
 * (note)
 * close時のエラーの可能性について注意すること。エラーを生じさせるような
 * 種類の"commit"をcloseで実行することは完全に可能であるためだ。
 * commitフラグは、readのためにOpenしている場合には無視することが可能です。
 * (ただ、現状、readにはフィルタアーキテクチャを使用しているので、このclose
 *  が使われることはありませんが。)
 *
 * @param stream dav_stream * Streamを表す構造体へのポインタ
 * @param commit int ファイルを保持すべきかどうか
 *                   0: commitしない(保持しない) / 0以外(1): commitする
 * @return DAVエラー (500)
 */
static dav_error * dav_divy_close_stream(dav_stream *stream, int commit)
{
	apr_status_t rv;
	dav_error *err       = NULL;
	int ret;
	const char *filepath = NULL, *formal_filepath = NULL;

	apr_pool_t *p                = stream->p;
	dav_resource *resource       = stream->resource;
	divy_rdbo_resource *rdb_r    = stream->resource->info->rdb_r;
	request_rec *r               = stream->resource->info->r;
	time_t short_time;
	char *contenttype, *contentlanguage;
	apr_int64_t	slength = 0;
	int inspection_result = 0;
#ifdef DIVY_SUPPORT_PLUGIN
	tfsp_ccipher_file *cfile = NULL;
#endif	/* DIVY_SUPPORT_PLUGIN */
#ifdef TF_SUPPORT_CIPHER_STORAGE
	const char *encryptfile=NULL;
	dav_divy_dir_conf *conf   = dav_divy_get_dir_config(stream->resource->info->r);
#endif

	TRACE(p);

	/*
	 * HTTP ヘッダの値を再取得
	 * (note)
	 *   上書きPUTの場合、rdb_r の値は旧リースの値. だから更新は必須.
	 *   新規PUTの場合には、実は正しい値がrdb_r に入っているが無視して上書き.
	 */
	if (stream->is_chunkput) {
		/* Mac OS X 10.5対応 特殊ヘッダを取得する */
		slength = divy_get_x_expected_entity_length(r);
		if (slength) {
			/* OS/XのX-Expected-Entity-Length にはContent-Lengthの代わりが入っている */
			rdb_r->getcontentlength = stream->clength = slength;
		}
		else {
			/* 一般的なChunkPUT */
			/* 読み込みサイズをContent-Lengthの値に合わせる */
			rdb_r->getcontentlength = stream->clength = stream->read_size;
		}
	}
	else {
		rdb_r->getcontentlength = stream->clength = divy_get_content_length(r);
	}

	/* contentlanguage は、存在しているときだけUPDATEする(専用の仕様) */
	contentlanguage = (char *) divy_get_content_language(r);
	if (IS_FILLED(contentlanguage)) {
		rdb_r->getcontentlanguage = contentlanguage;
	}

	/* contenttype は、存在しているときだけUPDATEする(専用の仕様) */
	contenttype = (char *) divy_get_content_type(r);
	if (IS_FILLED(contenttype)) {
		rdb_r->getcontenttype = contenttype;
	}

	/* 一時ファイルを閉じる */
	if (stream->f) {
		/* ファイルパスの取得 */
		filepath = divy_pfile_get_fullpath(stream->pfile);

		/* 閉じる */
		rv = apr_file_close(stream->f);
		if (rv != APR_SUCCESS) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to close temprary file. (filepath = %s, "
				"code = %d)", filepath, rv);
			/*
			 * NFS経由の場合はWriteではなくcloseでディスクFULLの
			 * 反応してしまいます。
			 * ディスク溢れのエラーがここで発生した場合においてのみ
			 * 処理を中断するようにしました。
			 */
			if (APR_STATUS_IS_ENOSPC(rv)) {
				ERRLOG0(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to close resource. because no space "
				"left on physical disk.");
				return dav_new_error(stream->p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
			}
			/* ディスク溢れ以外は失敗しても先に進みます */
		}
	}

	/*
	 * アップロードポリシーをサポートしていれグループフォルダ配下のリソースは
	 * 検閲されます
	 * 本当はopen_streamでやりたかったのだが、エラーを出すとクライアントが
	 * びっくりしてしまい、LOCKが残ってしまった。。
	 */
	if (divy_support_upload_policy(r)
			&& rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {

		inspection_result = divy_resource_inspection(r, resource);
		if (inspection_result == 2) {
			/* * 問題があったが通知は正常にします
			 * エラーメッセージはdivy_resource_inspection()で出力済み */
			return NULL;
		}
		else if (inspection_result == 1) {
			/* 拒否されます */
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	/*
	 * 反映する必要がない or データ不整合が生じていた場合
	 */
	/*
	 * ##### TODO content-length以外の方法でデータが切れてしまっていることを
	 * 	検出できないか? chunkモードを安全に受け入れることが出来ない。
	 * ただしMacOS/XではExpected-Entity-Lengthによって分かる 
	 */
	if (commit == 0 || (!stream->is_chunkput && stream->read_size != stream->clength)
					|| ( stream->is_chunkput && divy_get_x_expected_entity_length(r) 
							&& stream->clength != stream->read_size) ) {

		/* 一時ファイルを消去して終了する */
		if (stream->f &&
		    (rv = apr_file_remove(filepath, p)) != APR_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to remove temprary file. (code = %d) "
				"Please check temprary file. (filepath = %s)", rv, filepath);
			/* 失敗しても進む */
		}

		/* 途中で終わってしまったのでウィルス検索処理はabort */
		if (stream->vscsession != NULL) {

			divy_vsc_abortStreamScan(stream->vscsession, stream->sfile);
			/* ウィルス感染を検出していた場合 */
			if (stream->is_infected) {
				if (divy_ml_enable_virul_detection_alert(r)) {
					/* メールを送信する */
					divy_ml_notify_virusdetection(r, DIVY_VSC_ST_INFECTED, rdb_r);
				}

				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}
			/* セッションを閉じる */
			divy_vsc_closeSession(stream->vscsession);
			stream->vscsession = NULL;
		}

		/* Content-Length の値と実際に書き込んだ値が等しくない時 */
		/* is_chunkput == 1 の場合でdivy_get_x_expected_entity_length(r) > 0
		 * ならば、stream->read_size（実際の読み込んだバイト）とstream->clength
		 * (Content-Length相当）が同じでなければエラーにしてもよい
		 * divy_get_x_expected_entity_length(r)はMacOS/Xしかないが、通常の
		 * ChunkPUTではこの値がない。 = 0である。
		 */
		if ((!stream->is_chunkput && stream->read_size != stream->clength) || 
			( stream->is_chunkput && divy_get_x_expected_entity_length(r) 
			  								&& stream->read_size != stream->clength) ) {

			/* アップロードの途中で止めてしまうとここにくることがある.
			 * エラーではなくワーニング扱い */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"Content-Length and readed data size were "
				"diffrent.(content-length = %"APR_INT64_T_FMT
				", read_size = %"APR_INT64_T_FMT") "
				"Maybe client stop to upload",
				stream->clength, stream->read_size);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* commit == 0 即ち、データの反映は不要であると上層から渡された時 */
		if (commit == 0) {
			/* 2006/10/27 Fri takehara
			 *
			 * DAVプロバイダとしてはエラーではありませんが、mod_dav君が
			 * 正しいエラーログ出力をしないので、ここで吐いておきます */
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Could not process request body (r->status = %d)", r->status);
			return NULL;
		}
	}

	/*
	 * クライアント主体のメール送信
	 * メールフォルダ($root/.mail/) に対するPUT はメール送信動作
	 * (note)
	 * 	メール情報はDBに書き込んではならない。書き込み終わったら一時
	 * 	ファイルを消さなければならない。メール送信に失敗しても正常と返却
	 */
	if (rdb_r->u_spec->infotype == DIVY_INFOTYPE_mail_e) {
		/* メール送信の実施 */
		ret = divy_ml_send_clientmail(r, filepath);
		if (ret != ML_SUCCESS) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to send mail from \"%s\" user.(code = %d)",
				divy_get_userid(r), ret);
		}
		err = NULL;	/* 何があろうと正常終了 */
		goto cleanup_temporary_file;
	}

	/*
	 * ウィルス検出処理
	 */
	err = _scanvirus(r, stream, filepath);
	if (err) {
		goto cleanup_temporary_file;
	}

	/*
	 * 格納されるファイルを暗号化する
	 */
#ifdef TF_SUPPORT_CIPHER_STORAGE
	if (conf->cipherstorage == DIVY_CIPHER_STORAGE_ON) {
		encryptfile = tf_encrypt_file(r, filepath, conf->cipherstoragetype, conf->cipherstoragekey, NULL);
		if (encryptfile != NULL) {
			/* 暗号化されたファイルパスに置き換える */
			rv = apr_file_rename(encryptfile, filepath, p);
			if (rv != APR_SUCCESS) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
						"Failed to rename encription file (file = %s, code = %d)",
						encryptfile, rv);

				(void)apr_file_remove(encryptfile, p);
				goto cleanup_temporary_file;
			}
		}
	}
#endif

#ifdef DIVY_SUPPORT_PLUGIN
	/*
	 * プラグイン専用アップロードファイルはDBに書かない
	 */
	if (rdb_r->u_spec->infotype == DIVY_INFOTYPE_plugin_uploads_e) {
		err = NULL;

		/* uri -> path を算出 */
		formal_filepath  = divy_pi_uri2path(r, rdb_r->u_spec);
		if (formal_filepath == NULL) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA, "Invalid plugin uri found. ");
			err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			goto cleanup_temporary_file;
		}

		/* 一時ファイルを正式ファイルにリネームする */
		rv = apr_file_rename(filepath, formal_filepath, p);
		if (rv != APR_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to rename file.(dist = %s, src = %s, code = %d)",
					formal_filepath, filepath, rv);
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		goto cleanup_temporary_file;	/* 以降の処理は不要 */
	}
#endif	/* DIVY_SUPPORT_PLUGIN */

#ifdef DIVY_SUPPORT_PLUGIN
	/*
	 * 実体暗号プラグイン処理
	 */
	ret = divy_pi_ccipher(r, rdb_r, filepath, &cfile);
	if (ret == DIVY_PL_ST_OK) {
		char *uri;

		stream->clength       = cfile->ofsize;
		filepath              = cfile->opath;
		rdb_r->displayname    = cfile->oname;
		rdb_r->getcontenttype = (char *) cfile->octype;

		/* URIの変換 */
		uri = apr_psprintf(p, "%s/%s", divy_get_parenturi(p, rdb_r->uri),
						   rdb_r->displayname);
		ap_no2slash(uri);

		/* URI が変更された場合 */
		if (strcmp(uri, rdb_r->uri) != 0) {
			dav_resource c_resource    = { 0 };
			divy_rdbo_resource c_rdb_r = { 0 };
			divy_rdbo_lock_resource *lp= NULL;

			/* 新しいURIでDBを再度検索してみる */
			c_rdb_r.uri          = uri;
			c_rdb_r.depth        = divy_count_dirs(uri);
			c_rdb_r.resourcetype = DIVY_TYPE_RESOURCE;
			divy_parse_uri(p, dav_divy_get_root_uri(r), c_rdb_r.uri, &c_rdb_r.u_spec);

			if (divy_rdbo_get_hierarchy_property(r, &c_rdb_r, 0, 0, NULL)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get resource information.(uri = %s)", uri);
				err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				goto cleanup_temporary_file;
			}

			/* 新しいURIリソースのアクセス権限をチェックする */
			c_resource.collection  = 0;
			c_resource.exists      = IS_FILLED(c_rdb_r.rsid) ? 1 : 0;
			c_resource.info        = apr_pcalloc(p, sizeof(dav_resource_private));
			c_resource.info->rdb_r = &c_rdb_r;
			err = divy_validate_request_resource(r, &c_resource);
			if (err) {
				goto cleanup_temporary_file;
			}

			if (IS_FILLED(c_rdb_r.rsid)) {

				resource->exists = 1;
				rdb_r->uri                 = c_rdb_r.uri;
				rdb_r->rsid                = c_rdb_r.rsid;
				rdb_r->lastmodifier_userid = c_rdb_r.lastmodifier_userid;
				rdb_r->physicalpath        = c_rdb_r.physicalpath;
				rdb_r->resourcetype        = c_rdb_r.resourcetype;
				rdb_r->depth               = c_rdb_r.depth;

				/* リソースのロックを調べる */
				if (divy_rdbo_get_lock_record(r, &lp, rdb_r->uri, 0, 0)) {
					ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
							"Failed to get lock resource. (uri = %s)", rdb_r->uri);
					err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
					goto cleanup_temporary_file;
				}
				if (lp != NULL && apr_hash_count(lp->token_list_hash) > 0) {
					divy_rdbo_lock *lock = apr_hash_get(lp->token_list_hash,
											rdb_r->uri, APR_HASH_KEY_STRING);
					/* 他人のロックが見つかったらNG */
					if (lock != NULL && strcmp(lock->userid, divy_get_userid(r)) != 0) {
						err = dav_new_error(p, HTTP_UNAUTHORIZED, 0, 0, "");
						goto cleanup_temporary_file;
					}
				}
			}
			else {
				resource->exists = 0;
				rdb_r->uri = c_rdb_r.uri;

				/* リソースのNULL ロックを調べる */
				if (divy_rdbo_get_lock_record(r, &lp, rdb_r->uri, 0, 0)) {
					ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
							"Failed to get NULL-lock resource. (uri = %s)", rdb_r->uri);
					err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
					goto cleanup_temporary_file;
				}
				if (lp != NULL && apr_hash_count(lp->token_list_hash) > 0) {
					divy_rdbo_lock *lock = apr_hash_get(lp->token_list_hash,
											rdb_r->uri, APR_HASH_KEY_STRING);
					/* 他人のNULLロックが見つかったらNG */
					if (lock != NULL && lock->is_null &&
						strcmp(lock->userid, divy_get_userid(r)) != 0) {
						err = dav_new_error(p, HTTP_UNAUTHORIZED, 0, 0, "");
						goto cleanup_temporary_file;
					}
				}
			}
			r->uri = uri;
			stream->resource->uri = uri;
		}
	}
	/* プラグインがなかった / 何もしなかった / ライセンスが無くて動作しなかった場合 */
	else if (ret == DIVY_PL_ST_NOPLUGIN || ret == DIVY_PL_ST_NOPROCESS || ret == DIVY_PL_ST_NOLICENSE) {
		/* 何もしない */
	}
	else {
		err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		goto cleanup_temporary_file;
	}
#endif	/* DIVY_SUPPORT_PLUGIN */

	/* 更新日付を取得 */
	short_time = dav_divy_get_now_epoch();

	/**
	 * アクション拡張コマンドの実行(PUT)
	 *
	 * サポートしていて且つ条件にマッチしたリクエストだけ実行できる
	 */
	if (divy_extcmd_support(r) && divy_extcmd_match(r)) {
		divy_extcmd_resource ecres = { 0 };
		int exitcode               = DIVY_EXTCMD_BITNONE;

		/* 引き渡し用のリソースを生成 */
		ecres.uri                  = rdb_r->uri;
		ecres.displayname          = rdb_r->displayname;
		ecres.getcontenttype       = rdb_r->getcontenttype;
		ecres.getlastmodified      = short_time;
		ecres.lastmodifier_userid  = apr_pstrdup(p, divy_get_userid(r));
		ecres.creator_userid       = (resource->exists == 0) ? NULL : ecres.lastmodifier_userid;
		ecres.u_spec               = rdb_r->u_spec;
		ecres.flen                 = stream->clength;
		ecres.pfile                = stream->pfile;

		/* 実行 */
		if (divy_extcmd_execute(r, &ecres, &exitcode)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to execute action-command.");
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			goto cleanup_temporary_file;
		}

		/* exitcode による判定 */
		if (DIVY_EXTCMD_IS_FAIL(exitcode)) {
			/* アクション拡張コマンドからエラーを戻された場合 -> ログに出すだけ */
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
					"The action-command raised error (code = %d)", exitcode);
		}

		if (DIVY_EXTCMD_IS_NOCONTINUE(exitcode)) {
			/* 処理を続行しない場合 */
			if (DIVY_EXTCMD_IS_NOTIFYFAIL(exitcode)) {
				/* クライアントに異常発生を通知する */
				err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}
			else {
				/* クライアントに異常発生を通知しない */
				/* (note)
				 *   mod_dav に err をNULL で返してしまうと、正常だと思って
				 *   ロックを正式ロックに昇格してしまう. これはまずい.
				 *   仕方がないので、意図的に201や204を返すようにして辻褄を合わせている */
				if (resource->exists == 0) {
					err = dav_new_error(p, HTTP_CREATED, 0, 0, NULL);
				}
				else {
					err = dav_new_error(p, HTTP_NO_CONTENT, 0, 0, NULL);
				}
				/* Location ヘッダを付ける
				 * (note) 本当はmod_dav がやるのだが、201や204をerr に入れて返すと
				 * やってくれない. 仕方ない */
				apr_table_setn(r->headers_out, "Location", ap_construct_url(p, resource->uri, r));
			}
			goto cleanup_temporary_file;
		}
		else {
			/* 処理を続行する場合 */
			if (DIVY_EXTCMD_IS_DELETEFILE(exitcode)) {
				/* ファイルを残さない場合 */
				/* (note) 上と同じ理由でerr を生成する */
				if (resource->exists == 0) {
					err = dav_new_error(p, HTTP_CREATED, 0, 0, NULL);
				}
				else {
					err = dav_new_error(p, HTTP_NO_CONTENT, 0, 0, NULL);
				}
				apr_table_setn(r->headers_out, "Location", ap_construct_url(p, resource->uri, r));
				goto cleanup_temporary_file;
			}
			else {
				/* ファイルを残す場合 -> 何もしないで続行*/
			}
		}
	}

	/* 新規PUTならばリソースIDを採番する */
	if (resource->exists == 0) {
		if (divy_rdbo_create_rsid(r, &rdb_r->rsid, NULL) || IS_EMPTY(rdb_r->rsid)) {
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			goto cleanup_temporary_file;
		}
	}

	/* DB への登録＆更新に必要な情報をセットする(共通) */
	rdb_r->resourcetype    = DIVY_TYPE_RESOURCE;
	rdb_r->getlastmodified = short_time;
	rdb_r->creator         = NULL;
	rdb_r->lastmodifier    = NULL;
	rdb_r->getcontentlength= stream->clength;
	rdb_r->getetag         = (char *) dav_divy_get_etag_string(p,
					rdb_r->rsid, stream->clength, short_time);
	rdb_r->lastmodifier_userid = apr_pstrdup(p, divy_get_userid(r));

	/* Last-Modified 時刻を更新 */
	apr_time_ansi_put(&r->mtime, rdb_r->getlastmodified);

	/* 登録データの作成 */
	if (resource->exists == 0) {
		/* 登録に必要な値をセット */

		/* name をrsid にリネーム */
		ret = divy_pfile_rename(stream->pfile, rdb_r->rsid);
		if (ret != DIVY_FS_ST_OK) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to rename abstract-file name.(orig filename = %s)",
				divy_pfile_get_fullpath(stream->pfile));
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			goto cleanup_temporary_file;
		}

		formal_filepath       = divy_pfile_get_fullpath(stream->pfile);
		rdb_r->physicalpath   = (char *) divy_pfile_get_relativepath(stream->pfile);
		rdb_r->creationdate   = short_time;
		rdb_r->isversioned    = 0;
		rdb_r->checkin        = -1;
		rdb_r->checkout       = -1;
		rdb_r->creator_userid = rdb_r->lastmodifier_userid;
	}
	else {
		/* 相対パスから抽象パスを生成する */
		ret = divy_pfile_relativemove(stream->pfile, rdb_r->physicalpath);
		if (ret != DIVY_FS_ST_OK) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to create pfile info."
				"(relativepath = %s)",rdb_r->physicalpath);
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			goto cleanup_temporary_file;
		}
		formal_filepath = divy_pfile_get_fullpath(stream->pfile);
	}

	/* 物理ファイルとDBの変更を確定する */
	if (err == NULL) {
		err = _commit_physical_resource(r, formal_filepath, filepath, rdb_r,
										(resource->exists ? 1 : 0));
	}

	/* サムネイルファイルを削除する */
	if (err == NULL && resource->exists) {
		(void) divy_thumbnail_remove(p, r, rdb_r->physicalpath);
	}

	/* サーバ主体のメール送信を実施するのか ? */
	if (err == NULL && divy_ml_enable_svrsendmail(r, rdb_r)) {
		/* サーバ主体のメール送信処理を行う */
		divy_ml_send_servermail(r, rdb_r, NULL);
	}

	filepath = NULL;	/* 一時ファイル削除ロジックをスキップする */

#ifdef DIVY_SUPPORT_PRINTER
	/* プリンタ連携機能が有効であればプリンタに出力する */
	_print_file(r, rdb_r, formal_filepath);
#endif	/* DIVY_SUPPORT_PRINTER */

/*pitstop連携が有効である時 */
#ifdef DIVY_SUPPORT_PREFLIGHT_PLUGIN
	_send_preflightfolder(r, rdb_r->u_spec, formal_filepath, rdb_r->displayname); 
#endif	/* DIVY_SUPPORT_PREFLIGHT_PLUGIN */

cleanup_temporary_file:
	if (IS_FILLED(filepath)) {
		/* 一時ファイルを削除 */
		rv = apr_file_remove(filepath, p);
		if (rv != APR_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to remove physical file. "
				"(filepath = %s, code = %d) ", filepath, rv);
		}
	}

	/* ウィルス検索プロバイダとの接続を明示的に切断する */
	if (stream->vscsession != NULL) {
		divy_vsc_closeSession(stream->vscsession);
		stream->vscsession = NULL;
	}

	return err;
}

/**
 * 指定されたstreamに、指定されたbufの全て(bufsize分)を書き込む。
 *
 * @param stream dav_stream * Streamを表す構造体へのポインタ
 * @param buf const void * 書き込むデータを持つバッファ
 * @param bufsize apr_size_t 書き込むデータの長さ
 * @return DAVエラー (500, 507)
 */
static dav_error * dav_divy_write_stream(dav_stream *stream, 
                                         const void *buf, 
                                         apr_size_t bufsize)
{
	apr_status_t rv;

	/* 一時ファイルにバッファを書き込む */
	rv = apr_file_write_full(stream->f, buf, bufsize, NULL);
	if (APR_STATUS_IS_ENOSPC(rv)) {
		ERRLOG1(stream->p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"There is not enough storage to write to this "
			"resource.(bufsize = %"APR_SIZE_T_FMT ")", bufsize);
		/* 書き込みスペースが足りなかった */
		return dav_new_error(stream->p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
	}
	else if (rv != APR_SUCCESS) {
		ERRLOG1(stream->p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"An error occurred while writing to a resource."
			"(code = %d)", rv);
		return dav_new_error(stream->p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * ストリームベースのウィルス検索を実施
	 */
	if (stream->sfile != NULL) {
		int ret = divy_vsc_writeStreamBytes(stream->vscsession,
						stream->sfile, buf, bufsize);
		/* ウィルスに感染していた */
		if (ret == DIVY_VSC_ST_INFECTED) {
			stream->is_infected = 1;
			return dav_new_error(stream->p, HTTP_FORBIDDEN, 0, 0, "");
		}
		else if (ret != DIVY_VSC_ST_OK) {
			return dav_new_error(stream->p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	/* 今までに読み込んだバイト数を記録しておく */
	stream->read_size += bufsize;
	return NULL;
}

/*
 * 指定されたstream(のファイル)の書き込みオフセット位置を指定された
 * abs_positionまで移動する.
 * (参考)
 *  Content-Rangeヘッダ(メタ情報)があるときに呼ばれます。
 *
 * @param stream dav_stream * Streamを表す構造体へのポインタ
 * @param abs_position apr_off_t 移動するバイト数
 * @return DAVエラー(500)
 */
static dav_error * dav_divy_seek_stream(dav_stream *stream, 
                                        apr_off_t abs_position)
{
	/*
	 * 2003/05/07 Wed
	 * Content-Range: ヘッダによる差分PUTは、危険が伴うため、
	 * サポートされないことになりました。
	 * 必ず400 Bad Request が返却されるようになります。
	 */
	ERRLOG0(stream->p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
		"The server does not support seek to specified "
		"position in the resource.(Content-Range PUT)");
	return dav_new_error(stream->p, HTTP_BAD_REQUEST, 0, 0, "");
}

/**
 * GETがstreamを使用して処理されるとき、この関数は、リポジトリに対し、
 * レスポンスの中にヘッダをセットする方法を提供する目的で使用される。
 * この関数は、deliver関数とは独立して呼ばれるかもしれない。
 * (GETとSEARCHで使用されています)
 *
 * @param r request_rec * リクエスト構造体へのポインタ
 * @param resource const dav_resource * 1つのリソースを表す構造体へのポインタ
 * @return dav_error ただし、エラーが返ることはありません。
 */
static dav_error * dav_divy_set_headers(request_rec *r, 
                                        const dav_resource *resource)
{
	divy_rdbo_resource *rdb_r;
	int m_search = ap_method_number_of("SEARCH");
	int mnum     = divy_get_method_number(r);
	apr_hash_t *parsed_h = NULL;
	char *ext = NULL, *last = NULL;
	apr_hash_t *mime_map  = NULL;
	const divy_cmap_elem *map_e = NULL;

	/* リソースがなかったとき */
	if (!resource->exists) return NULL;

	/*
	 * ショートコードの場合は再度ショートコード先のrdb_r情報を取得する
	 * 必要があります
	 */
	if (resource->info->rdb_r->u_spec->infotype == DIVY_INFOTYPE_shorten) {
		rdb_r = divy_convert_shorten(r->pool, r);
		if (rdb_r == NULL) return NULL;
	}
	else {
		rdb_r = resource->info->rdb_r;
	}

	/* SEARCH ならば以下は不要 */
	if (mnum == m_search) {
		return NULL;
	}

	/*
	 * コレクションに対するGET/POSTリクエストには通常の
	 * ファイルGETと同じレスポンスヘッダを付けてはならない
	 */
	if (resource->collection && (mnum == M_GET || mnum == M_POST)) {
		/* Accept-Ranges ヘッダ(byte-range リクエストは許可しない) */
		apr_table_setn(r->headers_out, "Accept-Ranges", "none");

		/* mtime を現在の時刻にする.
		 * (note)
		 *   Last-Modified やETagを返却せずとも、一部のクライアント(Mac)は
		 *   自分のローカル日付を使ってIf-Modified-Since してくる.
		 *   この場合、r->mtimeよりもクライアントのアクセス時間の方が未来なので
		 *   必ず304 になってしまう(関数ap_meets_conditions)
		 *   これを避ける目的で、r->mtime をクライアント時刻よりも未来に
		 *   ずらすことにしました. 特に問題はないでしょう.
		 *   (リソースに対してこの対応を適用してはなりません)
		 *   */
		apr_time_ansi_put(&r->mtime, dav_divy_get_now_epoch());
	}
	/* ファイルへのGET */
	else {
		/* Last-Modified ヘッダ */
		if (r->mtime) {
			ap_set_last_modified(r);
		}

		/* ETagヘッダ */
		if (IS_FILLED(rdb_r->getetag)) {
			apr_table_setn(r->headers_out, "ETag", rdb_r->getetag);
		}

		/* CGIファイルに対するレスポンスには Accept-Range: bytes を付けない */
		if (divy_cgi_executable(r, rdb_r->u_spec)) {
			/* Accept-Ranges ヘッダ(byte-range リクエストは許可しない) */
			apr_table_setn(r->headers_out, "Accept-Ranges", "none");
		}
		else {
			/* Accept-Ranges ヘッダ */
			apr_table_setn(r->headers_out, "Accept-Ranges", "bytes");
		}
	}

	/* Content-Typeヘッダ */
	if (IS_FILLED(rdb_r->getcontenttype)) {

		/*
		 * ここはすでにcontent-typeがtext/xml; charset="utf-8"で登録されて
		 * しまっているOffice系のファイルをしかるべき形のContent-Typeで
		 * 返す為のロジックです。
		 * この登録版はutil.cのdivy_get_content_type()で行っています。
		 */
		if (strncmp(rdb_r->getcontenttype, "text/xml; charset=\"utf-8\"", 25)
																		== 0) {
			if (IS_FILLED(rdb_r->displayname)) {
				last = divy_strrchr(rdb_r->displayname, '.');
				ext = (last != NULL) ? ++last : NULL;

				if (IS_FILLED(ext)) {
					if ( (strncmp(ext, "xls", 3) == 0)  ||
						 (strncmp(ext, "xlsx", 4) == 0) ||
						 (strncmp(ext, "doc", 3) == 0)  || 
						 (strncmp(ext, "docx", 4) == 0) ||
						 (strncmp(ext, "ppt", 3) == 0)  || 
						 (strncmp(ext, "pptx", 4) == 0)) {

						/* 拡張子、Content-Typeマップを読み込む */
						mime_map = divy_read_mimetype_map(r);
						/* 拡張子からマッピング情報を取得する */
						map_e = divy_get_extmapinfo(r->pool,
													mime_map, NULL, ext);
						if (map_e != NULL && IS_FILLED(map_e->ctype)) {
							/* 書き換えます */
							rdb_r->getcontenttype =
									apr_pstrdup(r->pool, map_e->ctype);
						}
					}
				}
			}

		} /* if (strncmp(text/xml; charset="utf-8" == 0)  */

		ap_set_content_type(r, rdb_r->getcontenttype);
	}
	else {
		ap_set_content_type(r, "application/tfoctet-stream");
	}

	/* URLパラメータをパースする */
	if (!resource->collection && rdb_r->resourcetype != DIVY_TYPE_COLLECTION) {
		(void) dav_divy_parse_urlparam(rdb_r->p, r->args, &parsed_h);
		if (parsed_h != NULL) {
			divy_linkedlist_t *ll =
					apr_hash_get(parsed_h, DIVY_URIPARAM_DOWNLOAD_ONLY, APR_HASH_KEY_STRING);

			/* ダウンロード専用リンクが付いていた場合 */
			if (ll != NULL && IS_FILLED(ll->val) &&
			    strcmp(ll->val, DIVY_URIPARAM_DOWNLOAD_ONLY_VAL) == 0) {
				/* 強制ダウンロードを実現するためmimetype を偽装する */
				ap_set_content_type(r, "application/octet-stream");
			}

			/* Content-Dispositionヘッダをつけて添付を明示する
			 * 2.2.0-21まではダウンロード専用リンクがある場合だけつけていた
			 * デフォルトでつけてしまうと利便性が下がるという理由であったが
			 * IEからOffice系を開くと勝手にDAVの接続を試みて(結果失敗）しまい
			 * ログイン画面（POST時）を表示させてしまう問題が発生します。
			 *
			 * 結局救いようがないのでブラウザでのダウンロードでは直接
			 * 開けないように措置を行います。
			 * 
			 * 2.2.0-23ではCGIパスを対象外としました。autodeleteなども影響を
			 * 受けてダウンロードしてしまうからです。
			 *
			 * file*を追加してRFC-5987に基づきエンコードしたファイル名を設定
			 * します。filenameよりfilename*のほうが優先です。
			 */
			if (!divy_cgi_executable(r, rdb_r->u_spec)) {
				apr_table_setn(r->headers_out, "Content-Disposition",
						apr_psprintf(r->pool,
							"attachement; filename=\"%s\"; filename*=UTF-8''%s",
								rdb_r->displayname,
								divy_url_encode(r->pool, rdb_r->displayname)
						)
				);
			}
		}
	}

	/* (2005.6.15 Wed takehara)
	 *   以下を有効にするとページ遷移を行う度にリクエストを発行するので
	 *   Autoindex表示においてシーケンスパラメータをつけなくても良くなるのだが、
	 *  「戻る」ボタンを押しても再読込してしまうので、例えば親フォルダが無くなっていた
	 *  場合などに問題となる。(二度と戻れないので)
	 *  利便性が損なわれるのでコメントアウト */
#if 0
	if (rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
		apr_table_setn(r->headers_out, "Cache-Control", "no-cache");
		apr_table_setn(r->headers_out, "Pragma", "no-cache");
	}
#endif

	/* (2005.6.15 Wed takehara)
	 *   以下を有効にすると、HTTPS環境においてIEによるダウンロードが出来なくなります。
	 *   これはIEのバグなのですが、修正するには全クライアントのレジストリを触らなければ
	 *   なりません。これは厳しいので、サーバ側で付けないようにするしかありません。
	 */
#if 0
	if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
		apr_table_setn(r->headers_out, "Cache-Control", "no-cache");
		apr_table_setn(r->headers_out, "Pragma", "no-cache");
	}
#endif

	/* IE sniffing XSS 対策 */
	/* ファイルの中身を勝手に調べてしまわないようにするためのヘッダ */
	apr_table_setn(r->headers_out, "X-Content-Type-Options", "nosniff");
	apr_table_setn(r->headers_out, "X-XSS-Protection", "1; mode=block");

	/* IEで直接開くことが出来なくなります。Edgeは開ける */
	apr_table_setn(r->headers_out, "X-Download-options", "noopen");

	/* Frame上でTeamFileが見れるかを決めるセキュリティオプション 
	 * SAMEORIGNで自分のサイトだけとなります
	 * このオプションはBOXの時に変化します。詳しくはtf_autoindex.cを参照
	 * してください。
	 */
	apr_table_setn(r->headers_out, "X-Frame-options", "SAMEORIGIN");

	return NULL;
}

/**
 * 指定されたフィルタ(ap_filter_t)に指定されたリソース(dav_resource)を
 * deliver(配送、送信)する。
 * 基本的に、これは、GETメソッドに対するレスポンスである。
 * このコールは、コレクションに含まれている全てのリソースのためにCallされる
 * ことに注意しなければならない。
 * この関数がコールされる前に、dav_divy_set_headers関数が呼ばれるだろう。
 * プロバイダは、dav_divy_set_headers関数を使って、適切なレスポンスヘッダを
 * 設定できる。
 *
 * @param resource dav_resource* 1つのリソースを表す構造体へのポインタ
 * @param output ap_filter_t* フィルタchainを表す構造体へのポインタ(util_filter.h)
 * @return DAVエラー
 */
static dav_error * dav_divy_deliver(const dav_resource *resource,
                                    ap_filter_t *output)
{
	const char *filepath      = NULL;
	divy_rdbo_resource *rdb_r = resource->info->rdb_r;
	request_rec *r            = resource->info->r;
	apr_pool_t *p             = resource->pool;
	divy_fstorage_t *fstorage = NULL;
	int ret;
	divy_uri_spec *u_spec     = rdb_r->u_spec;
	divy_cgi_property *cgi_pr = NULL;
#ifdef TF_SUPPORT_CIPHER_STORAGE
	const char* decryptfile = NULL;
	dav_divy_dir_conf *conf   = dav_divy_get_dir_config(resource->info->r);
#endif

	TRACE(p);

	/*
	 * resourcetype を判定する
	 */
	if (resource->type != DAV_RESOURCE_TYPE_REGULAR &&
	    resource->type != DAV_RESOURCE_TYPE_VERSION &&
	    resource->type != DAV_RESOURCE_TYPE_WORKING) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"Cannot GET this type of resource.");
		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/* リクエストの委譲
	 *
	 * [ 優先順位 ]
	 *   1. トップCGI
	 *   2. CGI
	 *   3. login
	 *   4. SAML
	 *   5. 短縮URL(shorten)
	 *   6. Autoindex
	 *   7. ファイルのストリーム送信
	 */
	/* トップCGIを処理する? */
	if (u_spec && u_spec->infotype == DIVY_INFOTYPE_root &&
		divy_cgi_support_topcgi(r)) {
		cgi_pr = NULL;

		/* トップCGIのプロパティ取得 */
		ret = divy_cgi_get_property(r, u_spec, &cgi_pr);
		if (ret) {
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get top-CGI property.");
			return dav_new_error(r->pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* トップCGIを実行 */
		/* CGI の呼び出し */
		ret = divy_cgi_run(r, cgi_pr);
		if (ret) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to run top-CGI. (cgi = %s)", cgi_pr->path);
			return dav_new_error(r->pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		return NULL;
	}
	/* CGI リクエストか? */
	else if (divy_cgi_executable(r, u_spec)) {
		cgi_pr = resource->info->list.cgi_pr;

		/* CGIリクエストの処理 */
		ret = divy_cgi_run(r, cgi_pr);
		if (ret) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to run CGI. (cgi = %s)", cgi_pr->path);
			return dav_new_error(r->pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		return NULL;
	}
	/* ログイン画面ではないか? */
	else if (divy_enable_login(r, resource)) {
		return divy_do_login(resource, output);
	}
	/* SAML画面ではないか? */
	else if (divy_enable_saml(r, resource)) {
		return divy_do_saml_login(resource, output);
	}
	/* BOX用の短縮URLではないか? */
	else if (divy_enable_box_shorten(r, resource)) {
		return divy_do_shorten(resource, output);
	}
	/* autoindex 機能は有効か? */
	else if (divy_enable_autoindex(r, resource)) {
		/* autoindex 処理の実行 */
		return divy_do_autoindex(resource, NULL, output);
	}
	
	/* autoindex がOFFでコレクションの場合 */
	else if (resource->collection) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"There is no default response to GET "
			"for a collection.");
 		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/* ストレージを開く */
	ret = divy_fstorage_open(r, p, &fstorage);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to open storage.(uri = %s, userid = %s)",
			rdb_r->uri, divy_get_userid(r));
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/**
	 * アクション拡張コマンドの実行(GET)
	 */
	if (divy_extcmd_support(r) && divy_extcmd_match(r)) {

		divy_pfile_t *pfile;
		divy_extcmd_resource ecres = { 0 };
		int exitcode               = DIVY_EXTCMD_BITNONE;
		divy_pfile_create(fstorage, p, rdb_r->physicalpath, &pfile);

		ecres.uri                  = rdb_r->uri;
		ecres.displayname          = rdb_r->displayname;
		ecres.getcontenttype       = rdb_r->getcontenttype;
		ecres.getlastmodified      = rdb_r->getlastmodified;
		ecres.lastmodifier_userid  = rdb_r->lastmodifier_userid;
		ecres.creator_userid       = rdb_r->creator_userid;
		ecres.u_spec               = rdb_r->u_spec;
		ecres.flen                 = rdb_r->getcontentlength;
		ecres.pfile                = pfile;

		/* 実行 */
		if (divy_extcmd_execute(r, &ecres, &exitcode)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to execute action-command.");
			/* ストレージを閉じる */
			(void) divy_fstorage_close(fstorage);

			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* exitcode による判定 */
		if (DIVY_EXTCMD_IS_FAIL(exitcode)) {
			/* アクション拡張コマンドからエラーを戻された場合 -> ログに出すだけ */
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
					"The action-command raised error (code = %d)", exitcode);
		}

		/*
		 * エラーコードによるクライアントへ返す値を調査する
		 *
		 * ファイルの取得はサーバから見るとファイルを渡すだけです。
		 * その為、サーバ側で抑制できることはファイルを渡さないこと
		 * しかできません。
		 * PUT側ではいろいろと処理が行えますが、GETの場合はファイルを
		 * 渡すか渡さないかの二点だけであることを注意してください
		 *
		 * また、 PUTの場合と異なり失敗とした時の戻りはアクセス拒否とします
		 */
		if (DIVY_EXTCMD_IS_NOCONTINUE(exitcode)) {
			/* 処理を続行しない場合 */
			if (DIVY_EXTCMD_IS_NOTIFYFAIL(exitcode)) {
				/* ストレージを閉じる */
				(void) divy_fstorage_close(fstorage);
				/* クライアントに異常発生を通知する */
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0,"");
			}
			else {
				/* クライアントに異常発生を通知しない */
			}
		}
		else {
			/* 処理を続行する場合は何もしない */
		}
	}

	/* ファイルパスの作成 */
	filepath = divy_make_fullpath(fstorage, rdb_r->physicalpath);

	/* ストレージを閉じる */
	(void) divy_fstorage_close(fstorage);

	/*
	 * 暗号化ファイルを復号化する
	 */
#ifdef TF_SUPPORT_CIPHER_STORAGE
	/* ディレクティブの確認 */
	if (conf->cipherstorage == DIVY_CIPHER_STORAGE_ON) {
		decryptfile = tf_decrypt_file(r, filepath,
										conf->cipherstoragetype,
										conf->cipherstoragekey, NULL);
		if (decryptfile != NULL) {
			/* パスを置き換える */
			filepath = decryptfile;
		}
	}
#endif
	/*
	 * 物理ファイルをストリームに流す
	 */
	ret = divy_sendfile2stream(r, output, rdb_r->getcontentlength, filepath);
	if (ret != OK) {
		return dav_new_error(p, ret, 0, 0, "");
	}

	/* 開封通知を送る */
	if (divy_support_confirmreading(r)) {
		(void) divy_ml_notify_confirmreading(r, rdb_r);
	}

	/* サーバ主体のメール送信を実施するのか ? */
	if (divy_ml_enable_svrsendmail(r, rdb_r)) {
		/* サーバ主体のメール送信処理を行う */
		divy_ml_send_servermail(r, rdb_r, NULL);
	}

	return NULL;
}

/**
 * 指定されたresource の情報を元にコレクションを作成し、作成した
 * ものがコレクションであることと、そのコレクションが存在している
 * ことをresource オブジェクトに反映する。
 * 
 * 対応コレクションは存在していてはなりません。この制約は
 * 呼び出し側で保証する必要があります。(see mod_dav.h)
 *
 * @param resource dav_resource *
 * @return dav_error エラー
 */
static dav_error * dav_divy_create_collection(dav_resource *resource)
{
	dav_error *err;
	divy_rdbo_resource *rdb_r = resource->info->rdb_r;
	apr_pool_t *p             = resource->pool;
	request_rec *r            = resource->info->r;
	divy_uri_spec *u_spec     = rdb_r->u_spec;
	int ret;

	TRACE(p);

	/*
	 * MKCOL時のレスポンスセキュリティヘッダを設定する
	 */
	_set_enhance_security_headers(r);

	/*
	 * ロックオペレーションが失敗していなかったかどうかを確認
	 * (note) 本来であればロックオペレーションに失敗していればmod_dav が
	 * 	ここをコールしないようにすべきであるが、実際にはエラーでも
	 * 	コールされてしまう。この場合、以下の処理をやってはいけないので
	 * 	ここはさっさと抜けてしまうことにする。
	 */
	if ((ret = dav_divy_lock_close_status(r)) != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Could not create collection, because the operation "
			"of LOCK had been failed.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * 親コレクションをチェックする
	 */
	if ((err = _validate_parent_collection(p, resource)) != NULL) {
		return err;
	}

	/*
	 * 通常コレクションの作成
	 */
	if (u_spec->uritype & DIVY_URITYPE_REGULAR) {

		/*
		 * 表示名称の禁則文字チェック
		 * (note) 親URIは既にチェック済みのはず
		 */
		if (divy_validate_pathsegment(p, DIVY_IGNORE_COLON | DIVY_IGNORE_ASTERISK,
							rdb_r->displayname)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The part of uri has invalid char." );
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* コレクションの生成 */	
		err = _create_regular_collection(r, rdb_r);
		if (err) return err;
	}
	/*
	 * 特殊フォルダの場合
	 */
	else if (u_spec->uritype & DIVY_URITYPE_SPECIAL) {
		int status;
		apr_xml_doc *doc = NULL;

		/* Body のデータを取得する */
		if ((status = ap_xml_parse_input(r, &doc)) != OK) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"Failed to parse body content for MKCOL."
				"(uri = %s)", resource->uri);
			return dav_new_error(p, status, 0, 0, "");
		}

		/* コレクション種別による分岐 */
		switch (u_spec->infotype) {

			/*
			 * linkdbsearch 結果フォルダの作成
			 */
			case DIVY_INFOTYPE_dbfolder_e:
				err = _create_dbfolder(r, rdb_r, doc);
				break;
			/*
			 * 共有コレクションの作成
			 */
			case DIVY_INFOTYPE_dbshfolder_e:
				err = _create_sharedcollection(r, rdb_r, doc);
				break;
			/*
			 * ユーザ(in 管理者フォルダ) の作成
			 */
			case DIVY_INFOTYPE_m_user_e:
				err = _create_user(r, rdb_r, doc);
				break;
			/*
			 * グループ(in 管理者フォルダ) の作成
			 */
			case DIVY_INFOTYPE_m_group_e:
				err = _create_group(r, rdb_r, doc);
				break;
			/*
			 * SQL(in 管理者フォルダ) の作成
			 */
			case DIVY_INFOTYPE_m_sql_e:
				err = _create_sql(r, rdb_r , doc);
				break;
			/*
			 * システムメッセージ(in 管理者フォルダ) の作成
			 */
			case DIVY_INFOTYPE_m_msg_e:
				err = _create_sysmsg(r, rdb_r, doc);
				break;

			/*
			 * DBMS 情報(in 管理者フォルダ) の作成
			 */
			case DIVY_INFOTYPE_m_dbms_e:
				err = _create_dbms(r, rdb_r, doc);
				break;

			/* 上記の何れにも該当しなかった場合 */
			default:
				/* 事前に防いでいるはずなので恐らくバグ */
				ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Unexpected infotype for mkcol. "
					"(uri = %s, uritype = %d, infotype = %d)",
					resource->uri, u_spec->uritype, u_spec->infotype);
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		
		/* エラーが発生していたか？*/
		if (err) return err;
	}
	/* 予期しないタイプ */
	else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The server is not permitted to do MKCOL operation."
			"(uri = %s)", resource->uri);
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* resource の情報を更新する */
	resource->collection = 1;
	resource->exists     = 1;

	return NULL;
}

/**
 * src が表すコレクションまたはリソースを、dst にコピーする。
 * Live, Dead プロパティもコピーします。
 *
 * (note)
 * mod_dav.c は COPY メソッドのリクエスト時にBodyをパースしてくれません。
 * それどころかBodyデータを捨ててしまっており、プロバイダ側では受け取ること
 * すら出来ません。
 * RFCに従えば、propertybehavior XMLエレメントを見てコピーするプロパティの
 * 制御を行うべきですが、異なるサーバ間におけるCOPYをmod_dav.c 側で塞いで
 * しまっている以上、あまり意味がありません。
 * 以上から、このサーバでは、propertybehavior エレメントで
 * 特定のプロパティがkeepalive 指定されたものとして動作します。
 *
 * @param src const dav_resource *
 * @param dst dav_resource *
 * @param depth int Copy メソッドのヘッダで指定されたDepth 値(0/DAV_INFINITY)
 * @param response dav_response **
 * @return DAV エラー (409, 500)
 */
static dav_error * dav_divy_copy_resource(const dav_resource *src,
                                          dav_resource *dst,
					  int depth,
					  dav_response **response)
{
	dav_error *err                = NULL;
	request_rec *r                = src->info->r;
	apr_pool_t *p                 = src->pool;
	divy_rdbo_resource *src_rdb_r = src->info->rdb_r;
	divy_rdbo_resource *dst_rdb_r = dst->info->rdb_r;
	divy_uri_spec *src_u_spec     = src_rdb_r->u_spec;
	divy_uri_spec *dst_u_spec     = dst_rdb_r->u_spec;
	int ret;

	TRACE(p);

	*response = NULL;	/* 初期化 */

	/*
	 * MOVE時のレスポンスセキュリティヘッダを設定する
	 */
	_set_enhance_security_headers(r);

	/*
	 * ロックオペレーションが失敗していなかったかどうかを確認
	 */
	if ((ret = dav_divy_lock_close_status(r)) != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Could not copy resource, because the operation "
			"of LOCK had been failed.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * コピー先(dst)の親コレクションをチェックする
	 */
	if ((err = _validate_parent_collection(p, dst)) != NULL) {
		return err;
	}

	/*
	 * 通常コレクションの場合
	 */
	if (src_u_spec->uritype & DIVY_URITYPE_REGULAR) {

		/* コピーリソースuri の検証 */
		err = divy_validate_copy_entity(r, src_u_spec, dst_u_spec);
		if (err) return err;

		/* DBと物理ファイルのコピー */
		err = _copy_regular_resource(r, depth, src_rdb_r, dst_rdb_r);
		if (err) return err;
	}
	/*
	 * 特殊フォルダを表していた場合
	 */
	else if (src_u_spec->uritype & DIVY_URITYPE_SPECIAL) {
		int dst_infotype = dst_u_spec->infotype;

		switch (src_u_spec->infotype) {

			/*
			 * ユーザエンティティのコピー
			 * ユーザリレーションの作成
			 */
			case DIVY_INFOTYPE_m_user_e:

				/* src とdst のinfotype が同じならコピー */
				if (dst_infotype == DIVY_INFOTYPE_m_user_e) {

					/* コピーリソースuri の検証 */
					err = divy_validate_copy_entity(r, src_u_spec, dst_u_spec);
					if (err) return err;

					/* コピー処理の実施 */
					err = _copy_user(r, src_rdb_r, dst_rdb_r);
					if (err) return err;

					break;
				}

				/* ユーザリレーションの作成 */
				if (dst_infotype == DIVY_INFOTYPE_m_group_ruser) {

					err = _create_ruser(r, src, dst);
					if (err) return err;
				}
				/* Destination のuri が不正だった場合 */
				else {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The destination uri must be "
						"user relation or user entity "
						"uri for copy operation.");
					return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
				}
				break;

			/*
			 * グループエンティティのコピー
			 * グループリレーションの作成
			 */
			case DIVY_INFOTYPE_m_group_e:

				/* src とdst のinfotype が同じならコピー */
				if (dst_infotype == DIVY_INFOTYPE_m_group_e) {

					/* コピーリソースuri の検証 */
					err = _validate_copy_group(r, src, dst);
					if (err) return err;

					/* コピー処理の実施 */
					err = _copy_group(r, depth, src_rdb_r, dst_rdb_r);
					if (err) return err;

					break;
				}

				/* グループリレーションの作成 */
				if (dst_infotype == DIVY_INFOTYPE_m_user_rgrp ||
				    dst_infotype == DIVY_INFOTYPE_m_sql_rgrp) {

					err = _create_rgrp(r, src, dst);
					if (err) return err;
				}
				/* Destination のuri が不正だった場合 */
				else {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The destination uri must be "
						"group relation or group entity "
						"uri for copy operation.");
					return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
				}
				break;

			/*
			 * SQLエンティティのコピー
			 * SQLリレーションの作成
			 */
			case DIVY_INFOTYPE_m_sql_e:

				/* SQL エンティティのコピーは出来ません */
				if (dst_infotype == DIVY_INFOTYPE_m_sql_e) {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"COPY for SQL not supported.");
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0,"");
				}
				/* SQLリレーションの作成 */
				else if (dst_infotype == DIVY_INFOTYPE_m_group_rsql) {
					err = _create_rsql(r, src, dst);
					if (err) return err;
				}
				/* Destination のuri が不正だった場合 */
				else {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The destination uri must be "
						"sql relation or sql entity "
						"uri for copy operation.");
					return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
				}
				break;

			/*
			 * システムメッセージのコピー
			 */
			case DIVY_INFOTYPE_m_msg_e:

				err = _copy_sysmsg(r, src, dst);
				if (err) return err;

				break;
			/*
			 * DBMS のコピー
			 */
			case DIVY_INFOTYPE_m_dbms_e:

				/* コピーリソースuri の検証 */
				err = divy_validate_copy_entity(r, src_u_spec, dst_u_spec);
				if (err) return err;

				/* コピー処理の実施 */
				err = _copy_dbms(r, src_rdb_r, dst_rdb_r);
				if (err) return err;
				
				break;
			/*
			 * ユーザリレーションのコピー
			 */
			case DIVY_INFOTYPE_m_group_ruser:

				err = _copy_ruser(r, src, dst);
				if (err) return err;

				break;
			/*
			 * SQL リレーションのコピー
			 */
			case DIVY_INFOTYPE_m_group_rsql:

				err = _copy_rsql(r, src, dst, response);
				if (err) return err;

				break;
			/*
			 * グループリレーションのコピー
			 */
			/* $userid/.RG_xxxx */
			case DIVY_INFOTYPE_m_user_rgrp:

				err = _copy_rgrp(r, src, dst, response);
				if (err) return err;

				break;

			/* $sqlname/.RG_xxxx */
			case DIVY_INFOTYPE_m_sql_rgrp:

				err = _copy_rgrp(r, src, dst, response);
				if (err) return err;

				break;

			/* グループコレクション */
			case DIVY_INFOTYPE_group_e:
				/* コピーリソースuri の検証 */
				err = divy_validate_copy_entity(r, src_u_spec, dst_u_spec);
				if (err) return err;
				
				/* DBと物理ファイルのコピー */
				err = _copy_regular_resource(r, depth, src_rdb_r, dst_rdb_r);
				if (err) return err;

				break;

			/* 上記以外が指定された場合 */
			default:
				ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Unexpected infotype for \"COPY\" operation."
					"(src = %s, dst = %s, src_infotype = %d)",
					src->uri, dst->uri,src_u_spec->infotype);
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}
	/* この種類のリソースに対するCOPY オペレーションは認められていない */
	else {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The server is not permitted to do COPY operation."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* dst は存在するようになった */
	dst->exists     = 1;
	dst->collection = src->collection;

	return NULL;
}

/**
 * src が表すコレクションまたはリソースを、dst に移動する。
 *
 * @param src dav_resource * ソースリソース
 * @param dst dav_resource * ディスティネーションリソース
 * @param response dav_response ** レスポンス(今は常にNULL)
 * @return DAV エラー (400, 403, 409, 500)
 */
static dav_error * dav_divy_move_resource(dav_resource *src,
                                          dav_resource *dst,
                                          dav_response **response)
{
	dav_error *err                = NULL;
	request_rec *r                = src->info->r;
	apr_pool_t *p                 = src->pool;
	divy_rdbo_resource *src_rdb_r = src->info->rdb_r;
	divy_rdbo_resource *dst_rdb_r = dst->info->rdb_r;
	divy_uri_spec *src_u_spec     = src_rdb_r->u_spec;
	divy_uri_spec *dst_u_spec     = dst_rdb_r->u_spec;
	int ret;

	TRACE(p);

	*response = NULL;	/* 初期化 */

	/*
	 * MOVE時のレスポンスセキュリティヘッダを設定する
	 */
	_set_enhance_security_headers(r);

	/*
	 * ロックオペレーションが失敗していなかったかどうかを確認
	 */
	if ((ret = dav_divy_lock_close_status(r)) != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Could not move resource, because the operation "
			"of LOCK had been failed.");
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto finalize_move_resource;
	}

	/*
	 * 親コレクションをチェックする
	 */
	if ((err = _validate_parent_collection(p, dst)) != NULL) {
		goto finalize_move_resource;
	}

	/*
	 * 通常コレクション,
	 * $root/.dbshfolder/$name,
	 * $root/.management/UPDATE/$name,
	 * ユーザごみ箱の下,
	 * グループごみ箱の下
	 */
	if (src_u_spec->uritype & DIVY_URITYPE_REGULAR           ||
	    src_u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e   ||
	    src_u_spec->infotype == DIVY_INFOTYPE_m_update_e     ||
	    src_u_spec->infotype == DIVY_INFOTYPE_user_trash_e0  ||
	    src_u_spec->infotype == DIVY_INFOTYPE_user_trash_ex  ||
	    src_u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
	    src_u_spec->infotype == DIVY_INFOTYPE_group_trash_ex) {

		/* 移動リソースuri の検証 */
		err = divy_validate_move_entity(r, src_u_spec, dst_u_spec);
		if (err) goto finalize_move_resource;

		/* 移動処理の実施 */
		err = _move_regular_resource(r, src_rdb_r, dst_rdb_r);
		if (err) goto finalize_move_resource;
	}
	/*
	 * 特殊フォルダを表していた場合
	 */
	else if (src_u_spec->uritype & DIVY_URITYPE_SPECIAL) {
		switch (src_u_spec->infotype) {

			/*
			 * ユーザ(in 管理者フォルダ) のリネーム
			 */
			case DIVY_INFOTYPE_m_user_e:

				/* 移動リソースuri の検証 */
				err = _validate_move_user(r, src, dst, response);
				if (err) goto finalize_move_resource;

				/* 移動処理の実施 */
				err = _move_user(r, src_rdb_r, dst_rdb_r);
				if (err) goto finalize_move_resource;

				break;
			/*
			 * グループ(in 管理者フォルダ) の移動、リネーム
			 */
			case DIVY_INFOTYPE_m_group_e:

				/* 移動リソースuri の検証 */
				err = _validate_move_group(r, src, dst, response);
				if (err) goto finalize_move_resource;

				/* 移動処理の実施 */
				err = _move_group(r, src_rdb_r, dst_rdb_r);
				if (err) goto finalize_move_resource;

				break;
			/*
			 * SQL(in 管理者フォルダ) のリネーム
			 */
			case DIVY_INFOTYPE_m_sql_e:

				/* 移動リソースuri の検証 */
				err = divy_validate_move_entity(r, src_u_spec, dst_u_spec);
				if (err) goto finalize_move_resource;
				
				/* 移動処理の実施 */
				err = _move_sql(r, src_rdb_r, dst_rdb_r);
				if (err) goto finalize_move_resource;
				
				break;
			/*
			 * DBMS 情報(in 管理者フォルダ) のリネーム
			 */
			case DIVY_INFOTYPE_m_dbms_e:

				/* 移動リソースuri の検証 */
				err = divy_validate_move_entity(r, src_u_spec, dst_u_spec);
				if (err) goto finalize_move_resource;

				/* 移動処理の実施 */
				err = _move_dbms(r, src_rdb_r, dst_rdb_r);
				if (err) goto finalize_move_resource;

				break;
			/*
			 * ユーザリレーションの移動
			 */
			case DIVY_INFOTYPE_m_group_ruser:

				err = _move_ruser(r, src, dst); 
				if (err) goto finalize_move_resource;

				break;
			/*
			 * SQL リレーションの移動
			 */
			case DIVY_INFOTYPE_m_group_rsql:

				err = _move_rsql(r, src, dst, response);
				if (err) goto finalize_move_resource;

				break;
			/*
			 * グループリレーションの移動
			 */
			/* $userid/.RG_xxxx */
			case DIVY_INFOTYPE_m_user_rgrp:

				err = _move_rgrp(r, src, dst, response); 
				if (err) goto finalize_move_resource;

				break;
			/* $sqlname/.RG_xxxx */
			case DIVY_INFOTYPE_m_sql_rgrp:

				err = _move_rgrp(r, src, dst, response); 
				if (err) goto finalize_move_resource;

				break;

			/* 上記以外(MOVE がサポートされていないリソース) */
			default:
				ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Unexpected infotype for \"MOVE\" operation."
					"(src = %s, dst = %s, src_infotype = %d)",
					src->uri, dst->uri, src_u_spec->infotype);
				err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				goto finalize_move_resource;
		}
	}
	/* この種類のリソースに対するMOVEオペレーションは認められていない */
	else {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The server is not permitted to do MOVE operation."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		goto finalize_move_resource;
	}

	/*
	 * ソースは移動されたのでなくなり、
	 * ディスティネーションは存在するようになったということ
	 */
	src->exists     = 0;
	src->collection = 0;
	dst->exists     = 1;
	dst->collection = 1;

finalize_move_resource:
	/*
	 * (2007/08/31 Fri takehara)
	 * エラー終了した場合、dst に残っているかもしれないロックを削除する.
	 * (note)
	 *    mod_dav のフレームワークの不備を回避するためにこのコードがある.
	 *    DELETEメソッドでは、ロックされたリソースに対してunlock してからremove ハンドラが
	 *    コールされる.この場合、正しくロックは削除される.
	 *
	 *    ところが、MOVEでは、Overwrite: Tの場合、dstをremove ハンドラで削除してから
	 *    move_resource しようとする. しかも、dst のロックをunlockしないでだ.
	 *    move_resource はロックの有無を考慮していないハンドラなのにだ.
	 *    そのため、remove ハンドラの呼び出しが成功すると、ロックは残っているのに実体が
	 *    切れてしまうというデータ不整合が起きる.というか、そうなるしかない.
	 *
	 *    move_resource ハンドラが直に成功すれば、また実体が生成されるのでOKだが、
	 *    万が一失敗すると、不整合のままになる.
	 *    これ以降は、この不整合リソースはNULLリソースのような違うような不思議な状態になり
	 *    完全に動作が未定義になります.
	 *    これを避けるため、MOVE でエラーがでていたら、src リソースのロックを解除します.
	 *    越権行為ですが、仕方ありません.
	 *    解除できるロックかどうかは既に調べてあるのでいいでしょう.
	 *
	 *    なお、ここに来る頃にはsrc のリソースは既にないので、注意すること.
	 *
	 */
	if (err != NULL) {
		/* 強制的にロックを削除します -> ロックプロバイダに委譲 */
		const dav_hooks_locks *lock_hocks = dav_get_lock_hooks(r);

		/* ロックがサポートされていたら実行 */
		if (lock_hocks != NULL) {
			dav_error *err_lock;
			dav_lockdb *lockdb = NULL;

			/* (note) dst->info->r でないとdst のロックは消えません... */
			err_lock = lock_hocks->open_lockdb(dst->info->r, 0/* rw */, 1/* force */, &lockdb);
			if (err_lock == NULL) {
				(void) lock_hocks->remove_lock(lockdb, dst, NULL);
			}

			lock_hocks->close_lockdb(lockdb);
		}
	}

	return err;
}

/**
 * 指定されたresource のコレクションまたはリソースとそのプロパティを削除する。
 * コレクションが指定された場合には、そのコレクション以下に存在する全ての
 * コレクションとリソースを削除する。(Depth は指定出来ません。)
 * 削除が成功したら、引数resource に該当するリソースまたはコレクションが
 * 既に存在しないという状態を反映します。(see mod_dav.h)
 * 
 * @param resource dav_resource * 削除対象のリソース
 * @param response dav_response ** レスポンス
 * @return DAVエラー (403, 500)
 */
static dav_error * dav_divy_remove_resource(dav_resource *resource,
                                            dav_response **response)
{
	dav_error *err             = NULL;
	dav_resource_private *info = resource->info;
	apr_pool_t  *p             = resource->pool;
	request_rec *r             = info->r;
	divy_rdbo_resource *rdb_r  = info->rdb_r;
	divy_uri_spec *u_spec      = rdb_r->u_spec;
	int ret;

	TRACE(p);

	*response = NULL;	/* 初期化 */

	/*
	 * DELETE時のレスポンスセキュリティヘッダを設定する
	 */
	_set_enhance_security_headers(r);

	/*
	 * ロックオペレーションが失敗していなかったかどうかを確認
	 */
	if ((ret = dav_divy_lock_close_status(r)) != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Could not remove resource, because the operation "
			"of LOCK had been failed.");
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto finalize_remove_resource;
	}

	/*
	 * 通常コレクション,
	 * $root/.dbfolder/$keys,
	 * $root/.dbshfolder/$keys,
	 * $root/.management/UPDATE/xxxxx,
	 * $root/.mail/$uuid
	 * ユーザごみ箱,
	 * グループごみ箱	の場合	
	 */
	if (u_spec->uritype & DIVY_URITYPE_REGULAR ||
	    u_spec->infotype == DIVY_INFOTYPE_dbfolder_e     ||
	    u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e   ||
	    u_spec->infotype == DIVY_INFOTYPE_m_update_e     ||
	    u_spec->infotype == DIVY_INFOTYPE_mail_e         ||
	    u_spec->infotype == DIVY_INFOTYPE_user_trash_e0  ||
	    u_spec->infotype == DIVY_INFOTYPE_user_trash_ex  ||
	    u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
	    u_spec->infotype == DIVY_INFOTYPE_group_trash_ex) {

		err = _remove_regular_resource(r, rdb_r);
		if (err) goto finalize_remove_resource;
	}
	/*
	 * 特殊フォルダを表していた場合
	 */
	else if (u_spec->uritype & DIVY_URITYPE_SPECIAL) {
		char *err_str = NULL;
		switch (u_spec->infotype) {
			/*
			 * ユーザ(in 管理者フォルダ) の削除
			 */
			case DIVY_INFOTYPE_m_user_e:
				/* ユーザを削除できるかどうか検証 */
				err = _validate_remove_user(r, resource,
							rdb_r->usr_pr, response);
				if (err) goto finalize_remove_resource;

				/* ユーザの削除 */
				err = _remove_user(r, rdb_r->usr_pr);
				if (err) goto finalize_remove_resource;

				break;
			/*
			 * グループ(in 管理者フォルダ) の削除
			 */
			case DIVY_INFOTYPE_m_group_e:
				/* グループを削除できるかどうか検証 */
				err = _validate_remove_group(r, resource,
							rdb_r->grp_pr, response);
				if (err) goto finalize_remove_resource;

				/* グループの削除 */
				err = _remove_group(r, rdb_r->grp_pr);
				if (err) goto finalize_remove_resource;

				break;

			/*
			 * SQLの削除
			 */
			case DIVY_INFOTYPE_m_sql_e:
				/* SQLを削除できるかどうか検証 */
				err = _validate_remove_sql(r, resource,
							rdb_r->sql_pr, response);
				if (err) goto finalize_remove_resource;

				/* DB からの削除 */
				if (divy_rdbo_remove_sql_property(r, rdb_r->sql_pr))
					err_str = "sql";
				break;

			/*
			 * システムメッセージ
			 */
			case DIVY_INFOTYPE_m_msg_e:
				if (divy_rdbo_remove_sysmsg_property(r,
							info->list.sysmsg_pr))
					err_str = "sysmsg";
				break;

			/*
			 * DBMS の削除
			 */
			case DIVY_INFOTYPE_m_dbms_e:
				/* SQLを削除できるかどうか検証 */
				err = _validate_remove_dbms(r, resource,
							rdb_r->dbms_pr, response);
				if (err) goto finalize_remove_resource;

				/* DB からの削除 */
				if (divy_rdbo_remove_dbms_property(r, rdb_r->dbms_pr))
					err_str = "dbms";
				break;

			/*
			 * ユーザリレーションの削除
			 */
			case DIVY_INFOTYPE_m_group_ruser:
				if (divy_rdbo_remove_rusr(r, info->list.rusr_pr))
					err_str = "user relation";
				break;

			/*
			 * グループリレーションの削除
			 */
			case DIVY_INFOTYPE_m_user_rgrp:
				if (divy_rdbo_remove_rgrp(r, info->list.rgrp_pr))
					err_str = "group relation";
				break;
			case DIVY_INFOTYPE_m_sql_rgrp:

				/* SQL使用権の検証 */
				err = _validate_inheritsql(r, resource, response);
				if (err) goto finalize_remove_resource;

				if (divy_rdbo_remove_rgrp(r, info->list.rgrp_pr))
					err_str = "group relation";
				break;

			/*
			 * SQLリレーションの削除
			 */
			case DIVY_INFOTYPE_m_group_rsql:
				
				/* SQL使用権の検証 */
				err = _validate_inheritsql(r, resource, response);
				if (err) goto finalize_remove_resource;

				if (divy_rdbo_remove_rsql(r, info->list.rsql_pr))
					err_str = "sql relation";
				break;

			/* 上記以外が指定された場合 */
			default:
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Unexpected infotype for \"DELETE\" operation."
					"(uri = %s, infotype = %d)",
					resource->uri, u_spec->infotype);
				err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				goto finalize_remove_resource;
		}

		/* エラーが発生していたか？*/
		if (err_str) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to DELETE resource.(uri = %s)", resource->uri);
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			goto finalize_remove_resource;
		}
	}
	/*
	 * 分類されていないフォルダ
	 */
	else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The server is not permitted to do DELETE operation."
			"(uri = %s)", resource->uri);
		err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		goto finalize_remove_resource;
	}

	/* 削除されたことを設定する */
	resource->exists = 0;
	*response = NULL;

finalize_remove_resource:

	return err;
}

/**
 * 指定されたコレクション以下のリソースまたはコレクションを階層的に
 * 辿って(walk して)、レスポンスを作成する。
 *
 * @param params dav_walk_params *
 * @param depth int Depthヘッダの値
 * @param response dav_response ** レスポンス構造体へのポインタ
 */
static dav_error * dav_divy_walk(const dav_walk_params *params, 
                                 int depth, dav_response **response)
{
	dav_error *err;
	apr_pool_t *p                = params->pool;
	apr_pool_t *wp               = NULL;
	dav_walker_ctx *ctx          = params->walk_ctx;
	const dav_resource *resource = params->root;
	dav_resource_private *info   = resource->info;
	divy_rdbo_resource *rdb_r    = info->rdb_r;
	request_rec *r               = info->r;
	divy_uri_spec *u_spec        = rdb_r->u_spec;

	int is_lock_nullresource     = 0;	/* lock-null resource かどうか */
	int is_skip_nullresource     = 0;	/* lock-null resourceの検索を飛ばすかどうか */
	apr_uint32_t propflag        = 0;

	TRACE(p);

	/* 
	 * NULL 初期化する
	 * (意味) 
	 * *params->func 関数 (ポイント先はdav_propfind_walker関数)は、
	 * *response のリストを作るためにdav_add_response 関数をCallします。
	 * この関数は、*response->next メンバに *params->func の第１引数
	 * wres のメンバ response (wres->response) を設定します。
	 * そのため、*response を最初に初期化しておかなければ、ゴミが詰まった
	 * *response が *response->next メンバに設定されてしまいます。
	 * これを防ぐ目的で、初期値の *response をNULLにします。
	 */
	*response = NULL;

	/* リクエストの間だけ有効な作業用のプールを取得する */
	wp = divy_get_request_temporary_pool(r);
	if (wp == NULL) wp = r->pool;

	/* 
	 * rdb_r->uri 以下に存在するリソースまたはコレクションの一覧を取得する。
	 */
	if (!resource->collection && depth != 0) {
		depth = 0;	/* リソースならば、depth = 0 にする */
	}

	/* propflag を組み立てる */
	propflag = _build_propflag(r, ctx, depth, rdb_r);

	/*  ネームスペースのハッシュを取得する */
	if (propflag &&
	    divy_rdbo_build_nsmap_hash(r, &info->ns_id_hash, &info->ns_uri_hash)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to create namespace hash.(uri = %s)",
			resource->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * リソースのプロパティを取得する
	 */
	switch (u_spec->infotype) {
		/* デフォルトの動作 */
		default:
			/* "処理場所" かつ"仮想フォルダ"に該当する場合 */
			if (u_spec->uritype & DIVY_URITYPE_ARBITER &&
			    u_spec->uritype & DIVY_URITYPE_VIRTUAL) {
				/* 必ず存在するものとして返却する */

				err = _get_virtual_entiry(r, rdb_r);
				if (err) return err;
				is_skip_nullresource = 1;       /* lock null は調べない */
				break;
			}

			/* DB から情報を取得する */
			if (divy_rdbo_get_hierarchy_property(r, rdb_r, depth, propflag, NULL)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get hierarchy resource.");
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}

			/* View属性がないと駄目だと言っているのにView属性の付いていない
			 * リソースしかなかった場合 */
			if ((propflag & DIVY_DEL_PROP_noviewattr) &&
				u_spec->infotype != DIVY_INFOTYPE_group_e && rdb_r->rstate_pr == NULL) {
				/* 下を探すだけ無駄(外部仕様) */
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* リソースを取得出来ていたかどうか判定する */
			if (IS_EMPTY(rdb_r->rsid)) {
				/* このリソースは、lock-null resource の可能性が高い */
				is_lock_nullresource = 1;
			}
			/* rsid が取得できていて、かつこれがリソースの場合 */
			else if (IS_FILLED(rdb_r->rsid) && !resource->collection) {
				is_skip_nullresource = 1;	/* lock null は調べない */
			}

			break;
		/* $root フォルダ */
		case DIVY_INFOTYPE_root:
			err = _get_rootfolder(r, depth, rdb_r);
			if (err) return err;
			is_skip_nullresource = 1;	/* lock null は調べない */

			break;

		/* $root/Group Folder フォルダ */
		case DIVY_INFOTYPE_group:
			err = _get_groupfolder(r, rdb_r);
			if (err) return err;
			is_skip_nullresource = 1;	/* lock null は調べない */

			break;

		/* ユーザフォルダのユーザ */
		case DIVY_INFOTYPE_m_user_e:
			if (rdb_r->usr_pr == NULL) {
				is_lock_nullresource = 1;
				break;
			}
			err = _get_special_entity(r, rdb_r->usr_pr->registdt,
						rdb_r->usr_pr->updatedt, rdb_r);
			if (err) return err;
			is_skip_nullresource = 1;	/* lock null は調べない */
			break;

		/* グループフォルダのグループ */
		case DIVY_INFOTYPE_m_group_e:
			if (rdb_r->grp_pr == NULL) {
				is_lock_nullresource = 1;
				break;
			}
			err = _get_special_entity(r, rdb_r->grp_pr->registdt,
						rdb_r->grp_pr->updatedt, rdb_r);
			if (err) return err;
			is_skip_nullresource = 1;	/* lock null は調べない */
			break;

		/* SQLフォルダのSQL */
		case DIVY_INFOTYPE_m_sql_e:
			if (rdb_r->sql_pr == NULL) {
				is_lock_nullresource = 1;
				break;
			}
			err = _get_special_entity(r, rdb_r->sql_pr->registdt,
						rdb_r->sql_pr->updatedt, rdb_r);
			if (err) return err;
			is_skip_nullresource = 1;	/* lock null は調べない */
			break;

		/* システムメッセージフォルダのメッセージ */
		case DIVY_INFOTYPE_m_msg_e:
			if (info->list.sysmsg_pr == NULL) {
				is_lock_nullresource = 1;
				break;
			}
			err = _get_special_entity(r, info->list.sysmsg_pr->registdt,
						info->list.sysmsg_pr->updatedt, rdb_r);
			if (err) return err;
			is_skip_nullresource = 1;	/* lock null は調べない */
			break;

		/* DBMSフォルダのDBMS */
		case DIVY_INFOTYPE_m_dbms_e:
			if (rdb_r->dbms_pr == NULL) {
				is_lock_nullresource = 1;
				break;
			}
			err = _get_special_entity(r, rdb_r->dbms_pr->registdt,
						rdb_r->dbms_pr->updatedt, rdb_r);
			if (err) return err;
			is_skip_nullresource = 1;	/* lock null は調べない */
			break;

		/* ユーザリレーション */
		case DIVY_INFOTYPE_m_group_ruser:
			if (info->list.rusr_pr == NULL) {
				is_lock_nullresource = 1;
				break;
			}
			err = _get_special_relation(r, DIVY_PREFIX_RUSER, rdb_r);
			if (err) return err;
			is_skip_nullresource = 1;	/* lock null は調べない */
			break;

		/* SQL リレーション */
		case DIVY_INFOTYPE_m_group_rsql:
			if (info->list.rsql_pr == NULL) {
				is_lock_nullresource = 1;
				break;
			}
			err = _get_special_relation(r, DIVY_PREFIX_RSQL, rdb_r);
			if (err) return err;
			is_skip_nullresource = 1;	/* lock null は調べない */
			break;

		/* グループリレーション */
		case DIVY_INFOTYPE_m_user_rgrp:
		case DIVY_INFOTYPE_m_sql_rgrp:
			if (info->list.rgrp_pr == NULL) {
				is_lock_nullresource = 1;
				break;
			}
			err = _get_special_relation(r, DIVY_PREFIX_RGROUP, rdb_r);
			if (err) return err;
			is_skip_nullresource = 1;	/* lock null は調べない */
			break;

		/* 送信メール   */
		case DIVY_INFOTYPE_mail_e:
			/* 送信メールはDB に格納されないので絶対にnot found */
			return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}

	/*
	 * DB から取得した値を使って、レスポンス用XMLを作成する。
	 * レスポンスXMLの作成処理は、この関数の呼び出し元(mod_dav)が
	 * 実装しているwalker 関数に任せることになっています。
	 */
	if (!is_lock_nullresource) {
		/* rdb_r 以下のリソースを順番に辿ってレスポンスを作成する */
		err = _walk_resource(r, wp, params, rdb_r, 0, response);
		if (err) return err;
	}

	/*
	 * walk_type としてnull resource の取得が指示されていて、
	 * look-null resource を検索する必要があるなら特別なWalk を行う.
	 */
	if (params->walk_type & DAV_WALKTYPE_LOCKNULL && is_skip_nullresource == 0) {
		int status;
		divy_rdbo_resource *lock_null_rdb_r = NULL;

		/* lock-null resource だけを取得する */
		if (is_lock_nullresource) {
			status = divy_rdbo_get_null_resource_member(r, resource->uri,
				  			0, 0, &lock_null_rdb_r);
		}
		else {
			status = divy_rdbo_get_null_resource_member(r, resource->uri,
					depth, resource->collection, &lock_null_rdb_r);
		}
		if (status) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get null resource at this uri."
				"(uri = %s, depth = %d)", resource->uri, depth);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* lock_null_rdb_r 以下のリソースを順番に辿ってレスポンスを作成する */
		if (lock_null_rdb_r) {
			err = _walk_resource(r, wp, params, lock_null_rdb_r, 1, response);
			if (err) return err;
		}
	}

	return NULL;
}

/**
 * Etag を作成して返却する。
 *
 * (note) resource->info->rdb_r 必須項目
 * 	rsid, getcontentlength, lastmodified
 *
 * @param resource const dav_resource *
 * @return const char* Etag を表す文字列
 *                     対象となるresource が存在しなかったり、
 *                     dav_resource_private, divy_rdbo_resource が
 *                     ない場合には、空文字列を返却する。
 */
static const char * dav_divy_getetag(const dav_resource *resource)
{
	divy_rdbo_resource *rdb_r = NULL;

	if (resource->exists == 0 || resource->info == NULL ||
	    resource->info->rdb_r == NULL ||
	    IS_EMPTY(resource->info->rdb_r->rsid)) {
		/* (note)
		 * リソースIDが存在しないリソース/コレクションはETagを
		 * 持っていないので、ETagを「""」にして返却する。
		 * 「""」にするのはmod_davがCoreDumpするのを防ぐため。
		 * またETagが存在しないリソース/コレクションに対して
		 * このハンドラが呼び出されてしまうのはmod_davの実装の問題.
		 */
		return "\"\"";
	}

	rdb_r = resource->info->rdb_r;

	return dav_divy_get_etag_string(resource->pool,
					rdb_r->rsid,
					rdb_r->getcontentlength,
					rdb_r->getlastmodified);
}

static request_rec *dav_divy_get_request_rec(const dav_resource *resource)
{
	return resource->info->r;
}

static const char *dav_divy_pathname(const dav_resource *resource)
{
	return NULL;
}

#ifdef DAV_SUPPORT_POST_HANDLING
/**
 * resource が示すリソースに対し、POSTメソッドをサポートしているかどうか.
 *
 * @param resource const dav_resource * 対象リソース
 * @return int (1: サポートしている / 0: 未サポート)
 */
static int dav_divy_is_posthandling_supported(const dav_resource *resource)
{
	divy_uri_spec *u_spec = NULL;
	apr_pool_t *p;
	request_rec *r;
	int ret;

	if (resource == NULL || resource->info == NULL) {
		return 0;	/* 未サポート */
	}
	r = resource->info->r;
	p = (r != NULL) ? r->pool : resource->pool;

	/* URI の種類を取得 */
	u_spec = (resource->info->rdb_r != NULL) ? resource->info->rdb_r->u_spec : NULL;
	if (u_spec == NULL) {
		ret = divy_parse_uri(p, dav_divy_get_root_uri(r), resource->uri, &u_spec);
		if (ret) {
			return 0;   /* 未サポート */
		}
		resource->info->rdb_r->u_spec = u_spec;
	}

	/*
	 * URI の種類による判定
	 *
	 * [ POST が有効なURI ]
	 *   * $root
	 *   * プラグイン専用アップロードファイル
	 *   * プライベートフォルダ自身とそれ以下のリソース
	 *   * グループフォルダの親
	 *   * グループフォルダ自身とそれ以下のリソース
	 *   * CGI
	 *   * 自動削除
	 *   * 開封通知
	 *   * ログイン
	 *   * 短縮URL
	 */
	if (u_spec->infotype == DIVY_INFOTYPE_root     ||
#ifdef DIVY_SUPPORT_PLUGIN
		u_spec->infotype == DIVY_INFOTYPE_plugin_uploads_e ||
#endif	/* DIVY_SUPPORT_PLUGIN */
		u_spec->infotype == DIVY_INFOTYPE_user_e   ||
		u_spec->infotype == DIVY_INFOTYPE_user_e_regular  ||
		u_spec->infotype == DIVY_INFOTYPE_group    ||
		u_spec->infotype == DIVY_INFOTYPE_group_e  ||
		u_spec->infotype == DIVY_INFOTYPE_group_e_regular ||
		divy_cgi_executable(r, u_spec) ||
		u_spec->infotype == DIVY_INFOTYPE_autodelete ||
		u_spec->infotype == DIVY_INFOTYPE_confirmreading ||
		u_spec->infotype == DIVY_INFOTYPE_login || 
		u_spec->infotype == DIVY_INFOTYPE_shorten || 
		u_spec->infotype == DIVY_INFOTYPE_saml) {

		return 1;	/* サポートしている */
	}

	ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_SYNTAX,
			"Unsupported POST in this resource. (uri = %s)", resource->uri);

	return 0;	/* 未サポート */
}
#endif	/* DAV_SUPPORT_POST_HANDLING */

#ifdef DAV_SUPPORT_POST_HANDLING
/**
 * resource が示すリソースに対しPOSTリクエストが行われる時、
 * 渡して欲しいデータ形式を返却する.
 *
 * @param resource const dav_resource * 対象リソース
 * @return dav_postdata_format
 */
static dav_postdata_format dav_divy_get_desired_format(const dav_resource *resource)
{
	divy_uri_spec *u_spec = NULL;

	TRACE(resource->info->r->pool);

	if (resource == NULL || resource->info == NULL ||
		resource->info->rdb_r == NULL || resource->info->rdb_r->u_spec == NULL) {
		return DAV_POSTDATA_ROW;
	}

	u_spec = resource->info->rdb_r->u_spec;

	/* CGI はRAWデータで渡してもらう */
	if (u_spec->infotype == DIVY_INFOTYPE_cgibin_e
#ifdef DIVY_SUPPORT_PLUGIN
		|| u_spec->infotype == DIVY_INFOTYPE_plugin_cgibin_e
#endif	/* DIVY_SUPPORT_PLUGIN */
	   ) {
		return DAV_POSTDATA_ROW;
	}

	/* パースした状態で返してもらう */
	return DAV_POSTDATA_PARSED;
}
#endif	/* DAV_SUPPORT_POST_HANDLING */

#ifdef DAV_SUPPORT_POST_HANDLING
/**
 * POST リクエストを処理する
 *
 * @param resource dav_resource * 対象リソース
 * @param data dav_postdata * POSTデータへのポインタ
 * @param response dav_response ** 取得したレスポンスへのポインタ
 * @return dav_error エラー
 */
static dav_error * dav_divy_post_resource(dav_resource *resource,
                                          dav_postdata *data,
                                          dav_response **response)
{
	divy_rdbo_resource *rdb_r  = resource->info->rdb_r;
	request_rec *r             = resource->info->r;
	divy_uri_spec *u_spec      = rdb_r->u_spec;
	divy_cgi_property *cgi_pr  = NULL;
	int ret;

	TRACE(r->pool);

	*response = NULL;	/* 初期化 */

	/* 共通処理
	 * POSTのレスポンスセキュリティヘッダを追加する
	 */
	/* コンテンツタイプSniffing対策 BugTrack-TeamFile_Server/1036 */
	apr_table_setn(r->headers_out, "X-Content-Type-Options", "nosniff");

	/* XSS対策 BugTrack-TeamFile_Server/1036 */
	apr_table_setn(r->headers_out, "X-XSS-Protection", "1; mode=block");

	/* IEで直接開くことが出来なくなります。Edgeは開ける 
	 * BugTrack-TeamFile_Server/1036
	 */
	apr_table_setn(r->headers_out, "X-Download-options", "noopen");

	/* Frame上でTeamFileが見れるかを決めるセキュリティオプション 
	 * SAMEORIGNで自分のサイトだけとなります
	 * このオプションはBOXの時に変化します。詳しくはtf_autoindex.cを参照
	 * してください。
	 */
	apr_table_setn(r->headers_out, "X-Frame-options", "SAMEORIGIN");

	/* リクエストの委譲
	 *
	 * [ 優先順位 ]
	 *   1. トップCGI
	 *   2. CGI
	 *   3. Autoindex
	 *   4. 自動削除 (他とは独立していますが)
	 */
	/* トップCGIを処理する? */
	if (u_spec && u_spec->infotype == DIVY_INFOTYPE_root &&
		divy_cgi_support_topcgi(r)) {
		cgi_pr = NULL;

		/* トップCGIのプロパティ取得 */
		ret = divy_cgi_get_property(r, u_spec, &cgi_pr);
		if (ret) {
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get top-CGI property.");
			return dav_new_error(r->pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* トップCGIを実行 */
		/* CGI の呼び出し */
		ret = divy_cgi_run(r, cgi_pr);
		if (ret) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to run top-CGI. (cgi = %s)", cgi_pr->path);
			return dav_new_error(r->pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}
	/* CGI リクエストか? */
	else if (divy_cgi_executable(r, u_spec)) {
		cgi_pr  = resource->info->list.cgi_pr;

		/* CGI の呼び出し */
		ret = divy_cgi_run(r, cgi_pr);
		if (ret) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to run CGI. (cgi = %s)", cgi_pr->path);
			return dav_new_error(r->pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}
	/* ログイン画面ではないか? */
	else if (u_spec->infotype == DIVY_INFOTYPE_login) {
		return divy_do_post_login(resource, data, r->output_filters);
	}
	/* SAML画面ではないか? */
	else if (u_spec->infotype == DIVY_INFOTYPE_saml) {
		return divy_do_saml_acs_login(resource, data, r->output_filters);
	}
	/* 短縮URLではないか? */
	else if (divy_enable_box_shorten(r, resource)) {
		return divy_do_post_shorten(resource ,data, r->output_filters);
	}
	/* autoindex 機能は有効か? */
	else if (divy_enable_autoindex(r, resource)) {
		/* ##### TODO Autoindex コマンド用hidden を見る */

		/* autoindex 処理の実行 */
		return divy_do_autoindex(resource, data, r->output_filters);
	}
	/* 自動削除の場合 */
	else if (u_spec && u_spec->infotype == DIVY_INFOTYPE_autodelete) {
		ret = divy_autodel_enable(r, u_spec);
		if (ret > 0) {
			/* 自動削除処理の実行 */
			return divy_autodel_execute(r, resource);
		}
		else if (ret < 0) {
			ERRLOG1(r->pool, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The server does not accept \"%s\" this request "
					"on this autodelete-process resource.", divy_get_method_name(r));
			return dav_new_error(r->pool, HTTP_FORBIDDEN, 0, 0, "");
		}
		else {
			/* 単に無効化されていただけなので、一々ログに出さない
			 * ステータスコードは仕様通り */
			return dav_new_error(r->pool, HTTP_SERVICE_UNAVAILABLE, 0, 0, NULL);
		}
	}
	/* 開封通知の自動解除の場合 */
	else if (u_spec && u_spec->infotype == DIVY_INFOTYPE_confirmreading) {
		ret = divy_confirmreading_enable(r, u_spec);
		if (ret > 0) {
			/* 自動削除処理の実行 */
			return divy_confirmreading_release(r, resource);
		}
		else if (ret < 0) {
			ERRLOG1(r->pool, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The server does not accept \"%s\" this request "
					"on this releasing-confirmreading resource.", divy_get_method_name(r));
			return dav_new_error(r->pool, HTTP_FORBIDDEN, 0, 0, "");
		}
		else {
			return dav_new_error(r->pool, HTTP_SERVICE_UNAVAILABLE, 0, 0, NULL);
		}
	}
	/* autoindex がOFFでコレクションの場合 */
	else if (resource->collection) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"There is no default response to POST for a collection.");
		return dav_new_error(r->pool, HTTP_CONFLICT, 0, 0, "");
	}

	return NULL;
}
#endif	/* DAV_SUPPORT_POST_HANDLING */

#ifdef DAV_SUPPORT_POST_HANDLING
/**
 * POSTのトークンを検証する
 *
 * @param resource dav_resource * 対象リソース
 * @return dav_error エラー
 */
static dav_error * dav_divy_validate_post_token(const dav_resource *resource,
												const char* token)
{
	char *sid = NULL;
	char *passwd = NULL;
	divy_auth_session session = { 0 };
	request_rec *r = resource->info->r;
	dav_divy_dir_conf   *dconf = dav_divy_get_dir_config(r);

	TRACE(r->pool);

	if (divy_support_session(r) == 0) {
		/* セッション未使用だと調べないでエラーにする */
		return dav_new_error(r->pool, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* Cookieがあるか調べる */
	if (divy_util_auth_parse_cookie(r, &passwd, &sid)
								 == DIVY_AUTH_SESSION_ERROR) {
		return NULL;
	}

	ERRLOG1(r->pool, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"sid = %s", sid);

	session.sid = sid;
	(void)(divy_util_auth_get_memcache_userinfo(r, &session));

	ERRLOG2(r->pool, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"sent CSRF TOKEN = %s / uniqueid = %s", token, session.uniqueid);

	if (IS_FILLED(token) && strcmp(token, session.uniqueid) == 0 ) {
		/* すべて一致しているなら正常とする */
		return NULL;
	}

	return dav_new_error(r->pool, HTTP_FORBIDDEN, 0, 0, "");
}
#endif	/* DAV_SUPPORT_POST_HANDLING */

/*--------------------------------------------------------------
  Define functions
  --------------------------------------------------------------*/
/**
 * Etag フォーマットの文字列を作成して返却する。
 */
DIVY_DECLARE(const char *) dav_divy_get_etag_string(apr_pool_t *p, char *rsid,
				      apr_int64_t getcontentlength,
				      time_t lastmodified)
{
	char *sp;
	unsigned int hash = 0;

	if (rsid == NULL) {
		/* 呼び出し元が保障するのだが、CoreDump防止のための
		 * 保険コードも入れておく */
		return "\"\"";
	}

	/* (note) フォーマット
	 * rsid"-"hex(getcontentlength + hash)"-"hex(lastmodified + hash)
	 */
	/* ハッシュ値の計算 */
	for (sp = rsid; *sp; sp++) {
		hash = hash * 33 + *sp;
	}

	/* 組み立て */
	return apr_psprintf(p, "\"%s-%"APR_UINT64_T_HEX_FMT
				"-%"APR_UINT64_T_HEX_FMT"\"",
				rsid, (getcontentlength + hash),
				(apr_int64_t)lastmodified + hash);
}

/*------------------------------------------------------------------------------
  Define private functions 
  ----------------------------------------------------------------------------*/
/**
 * dav_stream 構造体のインスタンスを生成する。
 *
 * @param p apr_pool_t *
 * @param resource const dav_resource *
 * @return dav_stream *
 */
static dav_stream * _create_dav_stream(apr_pool_t *p, const dav_resource *resource)
{
	dav_stream *d_stream = apr_pcalloc(p, sizeof(dav_stream));
	d_stream->f          = NULL;
	d_stream->resource   = (dav_resource *) resource;
	d_stream->read_size  = APR_INT64_C(0);
	d_stream->clength    = APR_INT64_C(0);
	d_stream->pfile      = NULL;
	d_stream->p          = p;
	d_stream->vscds      = NULL;
	d_stream->vscsession = NULL;
	d_stream->sfile      = NULL;
	d_stream->is_infected= 0;
	d_stream->is_chunkput= 0;

	return d_stream;
}

/**
 * 通常コレクションを作成するのに必要なdivy_rdbo_resource の情報を
 * rdb_r のメンバに設定する。
 * rdb_r はNULLであってはなりません。
 * (note)
 * 	rsid がNULLになることがあります。主としてDBエラーによって引き起こされます。
 * 	そのときは呼び出し側にてエラーを判定して下さい。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource *
 */
static void _fill_default_collection_info(request_rec *r, divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p     = r->pool;
	time_t short_time = dav_divy_get_now_epoch();

	divy_rdbo_create_rsid(r, &rdb_r->rsid, NULL);
	rdb_r->getcontentlength    = APR_INT64_C(0);
	rdb_r->getcontenttype      = apr_pstrdup(p, DIR_MAGIC_TYPE);
	rdb_r->resourcetype        = DIVY_TYPE_COLLECTION;  
	rdb_r->physicalpath        = NULL;
	rdb_r->creationdate        = short_time;
	rdb_r->getlastmodified     = short_time;
	rdb_r->getetag             = (char *) dav_divy_get_etag_string(p,
						rdb_r->rsid, 0, short_time);
	rdb_r->isversioned         = 0;
	rdb_r->checkin             = -1;
	rdb_r->checkout            = -1;
	rdb_r->creator             = NULL;
	rdb_r->lastmodifier        = NULL;
	rdb_r->creator_userid      = apr_pstrdup(p, divy_get_userid(r));
	rdb_r->lastmodifier_userid = rdb_r->creator_userid;

	return;
}

/**
 * uri が示すリソースのdav_resource の雛型を組み立てて返却する.
 *
 * @param p apr_pool_t * 作業用プール
 * @param uri const char * リソースURI(not null)
 * @param u_spec divy_uri_spec *
 * @param r request_rec *
 * @param is_collection int コレクションかどうか(1/0)
 * @param use_header int HTTP ヘッダ値を取得するかどうか(1/0)
 * @return dav_resource *
 */
static dav_resource * _create_empty_resource(apr_pool_t *p,
					const char *uri, divy_uri_spec *u_spec, request_rec *r,
					int is_collection, int use_header)
{
	dav_resource *res;
	dav_resource_private *info;
	divy_rdbo_resource *rdb_r;

	if (p == NULL || IS_EMPTY(uri) || r == NULL) return NULL;

	/* divy_rdbo_resource 構造体の初期化  */
	rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
	rdb_r->rsid         = NULL;
	rdb_r->uri          = apr_pstrdup(p, uri);
	rdb_r->displayname  = dav_divy_extract_finalpath_segment(p, uri); 
	rdb_r->depth        = divy_count_dirs(uri);
	rdb_r->u_spec       = u_spec;
	if (rdb_r->u_spec == NULL) {
		(void) divy_parse_uri(p, dav_divy_get_root_uri(r), uri, &rdb_r->u_spec);
	}
	if (use_header) {
		rdb_r->getcontentlanguage = (char *) divy_get_content_language(r);
		rdb_r->getcontentlength   = divy_get_content_length(r);
		rdb_r->getcontenttype     = (char *) divy_get_content_type(r);
	}
	else {
		rdb_r->getcontentlanguage = NULL;
		rdb_r->getcontentlength   = APR_INT64_C(0);
		rdb_r->getcontenttype     = NULL;
	}
	if (is_collection) {
		rdb_r->resourcetype   = DIVY_TYPE_COLLECTION;
	}
	else {
		rdb_r->resourcetype   = DIVY_TYPE_RESOURCE;
	}
	rdb_r->p                  = p;
	rdb_r->next               = NULL;

	/* dav_resource_private 構造体の初期化 */
	info = apr_pcalloc(p, sizeof(dav_resource_private));
	info->rdb_r           = rdb_r;
	info->parent_resource = NULL;
	info->liveprop_ts_ctx = NULL;
	info->r               = r;

	/* dav_resource 構造体の初期化 */
	res = apr_pcalloc(p, sizeof(dav_resource));
	res->type       = DAV_RESOURCE_TYPE_REGULAR;	/* 通常リソース */
	res->exists     = 0;	/* 存在しない (仮) */
	res->uri        = uri;
	res->info       = info;
	res->hooks      = &dav_divy_hooks_repository;
	res->pool       = p;
	res->collection = is_collection;
#ifdef DIVY_SUPPORT_VERSIONING
	res->versioned = 0;
	res->baselined = 0;
	res->working   = 0;
#endif

	return res;
}

/**
 * sharedcollectioncreate エレメントをパースして、エレメントの整合性を検証し
 * その結果をshsrccol_pr に格納して返却する。
 *
 * (note)
 * 	doc がNULL であればエラーを返却します。それを避けたければ
 * 	呼び出し元にて、NULL チェックをして下さい。
 *
 * (note) チェック成功条件
 * ・sharedcollectioncreate エレメントが存在し、空ではないこと
 * ・sharedsrccollectioninfo エレメントが存在し空ではないこと
 * ・sharedsrccollection エレメントが存在し空ではないこと
 *
 * @param p apr_poot_t *
 * @param doc const apr_xml_doc * Doc オブジェクト
 * @param shsrccol_pr divy_rdbo_shsrccol ** 解析したsharedcollectioncreate 
 * 						タグの内容
 * @return dav_error (NULL: 検証に成功した / 
 * 		      NULL以外: 検証に失敗した or 検証エラー発生)
 * 		      400 を返します。
 */
static dav_error * _parse_sharedcollectioncreate_elem(apr_pool_t *p,
					const apr_xml_doc *doc,
					divy_rdbo_shsrccol **shsrccol_pr)
{
	const apr_xml_elem *shinfo_elem, *shsrc_elem, *elem;
	divy_rdbo_shsrccol *_shsrccol_pr = NULL;
	*shsrccol_pr = NULL;

	if (doc == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The request body does not contain any element. ");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * sharedcollectioncreate タグのチェック
	 * (note)
	 * <!ELEMENT sharedcollectioncreate (sharedsrccollectioninfo) >
	 * <!ELEMENT sharedsrccollectioninfo (sharedsrccollection+) >
	 * <!ELEMENT sharedsrccollection (name, srcuri) >
	 * <!ELEMENT name       (#PCDATA) >
	 * <!ELEMENT srcuri     (#PCDATA) >
	 */
	if (strcmp(doc->root->name, "sharedcollectioncreate") != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The request body does not contain a "
			"\"sharedcollectioncreate\" element for MKCOL.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * sharedsrccollectioninfo タグのチェック
	 */
	shinfo_elem = doc->root->first_child;
	if (shinfo_elem == NULL ||
	    strcmp(shinfo_elem->name, "sharedsrccollectioninfo") != 0) {

		/* haredsrccollectioninfoエレメントが正しく設定されていない */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The \"sharedsrccollectioninfo\" element is not "
			"specified in the \"sharedcollectioncreate\" element.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * sharedsrccollection タグのチェックとデータ取得
	 */
	if (shinfo_elem->first_child == NULL) {

		/* sharedsrccollectionエレメントが１つも存在しない */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The \"sharedsrccollection\" element is missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	/* divy_rdbo_shsrccol の生成 */
	for (shsrc_elem = shinfo_elem->first_child; shsrc_elem;
				shsrc_elem = shsrc_elem->next) {
		if (*shsrccol_pr == NULL) {
			_shsrccol_pr = apr_pcalloc(p, sizeof(divy_rdbo_shsrccol));
			*shsrccol_pr = _shsrccol_pr;
		}
		else {
			_shsrccol_pr->next = apr_pcalloc(p, sizeof(divy_rdbo_shsrccol));
			_shsrccol_pr = _shsrccol_pr->next;
		}

		/* タグの検証 */
		if (strcmp(shsrc_elem->name, "sharedsrccollection") != 0) {

			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"Invalid element is specified "
				"in the \"sharedsrccollectioninfo\" element.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* name, srcuri の取得 */
		for (elem = shsrc_elem->first_child; elem; elem = elem->next) {
			/* name の取得 */
			if (strcmp(elem->name, "name") == 0) {
				_shsrccol_pr->name = (char *) divy_xml_get_cdata(elem, p, 1);
			}
			else if (strcmp(elem->name, "srcuri") == 0) {
				/* 後ろについたスラッシュを抜き取る */
				_shsrccol_pr->uri = dav_divy_remove_endslash(p,
							divy_xml_get_cdata(elem, p, 1));
			}
			else {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"Invalid element is specified in the "
					"\"sharedsrccollection\".(elem->name = %s)",
					elem->name);
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
		}
	}

	return NULL;
}

/**
 * dav_divy_walk が呼び出す関数divy_rdbo_get_hierarchy_property に与える
 * propflag を各種パラメータの状態に応じて設定して返却する。(dav_divy_walk専用)
 * (note)
 * 	ほとんどサブルーチンなのでdav_divy_walk 以外から呼び出すことは不可能です。
 * 	propflag を入念に作る最大の目的は不要なプロパティ検索を少しでも減らして
 * 	パフォーマンスを上げようというものなので、他の関数は興味すらないはず。
 *
 * 	なお、rdb_r はPROPFINDしたときのリクエストURIのリソースになります.
 * 	メンバではないことに注意.
 *
 * @param r request_rec *
 * @param ctx dav_walker_ctx *
 * @param depth int
 * @param rdb_r divy_rdbo_resource *
 * @return apr_uint32_t 組み立てられたpropflag
 */
static apr_uint32_t _build_propflag(request_rec *r, dav_walker_ctx *ctx,
					int depth, divy_rdbo_resource *rdb_r)
{
	divy_uri_spec *u_spec = rdb_r->u_spec;
	int propfind_type     = ctx->propfind_type;
	apr_uint32_t propflag = 0;
	divy_rdbo_extstatus *grp_extstatus = NULL;

	/*
	 * (note)
	 * 	以下の方法を用いて余計なプロパティを検索しないようにしよう
	 * 	という試み自体はmod_dav フレームワークの思想から逸脱するので
	 * 	mod_dav の変更に非常にsensitiveです。頑張って追従して下さい。
	 */

	/* allprop の場合 */
	if (propfind_type == DAV_PROPFIND_IS_ALLPROP) {
		/* 更新者, Deadプロパティ検索は必須 */
		propflag |= DIVY_GET_PROP_fullname;
		propflag |= DIVY_GET_PROP_dead;

		/* クライアント主体のメール監視が有効ならプロパティは必須 */
		if (divy_ml_enable_clientsendmail(r, rdb_r->uri)) {
			propflag |= DIVY_GET_PROP_mailwatch;
		}

		/* linkdbsearch結果フォルダであればプロパティは必須 */
		if (u_spec && u_spec->infotype == DIVY_INFOTYPE_dbfolder_e) {
			propflag |= DIVY_GET_PROP_shared;
		}
	}
	/* propname の場合 */
	else if (propfind_type == DAV_PROPFIND_IS_PROPNAME) {
		/* (note) 更新者や作成者は値をとらなくてもname の確認くらい可能 */
		propflag |= DIVY_GET_PROP_dead;

		/* クライアント主体のメール監視が有効ならプロパティは必要 */
		if (divy_ml_enable_clientsendmail(r, rdb_r->uri)) {
			propflag |= DIVY_GET_PROP_mailwatch;
		}

		/* linkdbsearch結果フォルダであれば検索が必要 */
		if (u_spec && u_spec->infotype == DIVY_INFOTYPE_dbfolder_e) {
			propflag |= DIVY_GET_PROP_shared;
		}
	}
	/* prop の場合 */
	else if (propfind_type == DAV_PROPFIND_IS_PROP) {
		/* 指定されたLiveプロパティを調べてみる */
		apr_xml_elem *elem = dav_find_child(ctx->doc->root, "prop");

		propflag |= DIVY_GET_PROP_dead;	/* Dead は常に取るしかない */

		/* (note) liveprop.c のdav_divy_props との整合性を維持せよ */
		for (elem = elem->first_child; elem; elem = elem->next) {
			if (IS_EMPTY(elem->name)) continue;

			/* (note)
			 * 以下のロジックは正確ではありません。
			 * 正しくはネームスペースも参照しなければならないからです。
			 * だが、mod_dav フレームワークに従っていない以上、ネーム
			 * スペースの参照は不可能であるため、情報を取得する方に倒して
			 * 問題が起きないようにしておくことにします。*/
			if (*elem->name == 'm' &&
					strcmp(elem->name, "mailwatch") == 0) {
				if (divy_ml_enable_clientsendmail(r, rdb_r->uri)) {
					propflag |= DIVY_GET_PROP_mailwatch;
				}
			}
			else if (*elem->name == 'c' &&
					strcmp(elem->name, "creator") == 0) {
				propflag |= DIVY_GET_PROP_fullname;
			}
			else if (*elem->name == 'l' &&
					strcmp(elem->name, "lastmodifier") == 0) {
				propflag |= DIVY_GET_PROP_fullname;
			}
			else if (*elem->name == 's' &&
					strcmp(elem->name, "sharedcollection") == 0) {
				if (u_spec && u_spec->infotype == DIVY_INFOTYPE_dbfolder_e) {
					propflag |= DIVY_GET_PROP_shared;
				}
			}
		}
	}
	else {
	 	/* (仮定) * propfind_type が設定されていればPROPFINDリクエスト */
		/* PROPFIND以外なら特殊なステータスは要らないでしょ? (以下を除いては) */
	}

	/* ごみ箱フォルダ専用のプロパティは必要か？ */
	if (divy_support_trashfolder(r) &&
	    ((depth != 0 && (u_spec->infotype == DIVY_INFOTYPE_user_trash ||
			     u_spec->infotype == DIVY_INFOTYPE_group_trash)) ||
	     (u_spec->infotype == DIVY_INFOTYPE_user_trash_e0 ||
	      u_spec->infotype == DIVY_INFOTYPE_group_trash_e0))) {

		propflag |= DIVY_GET_PROP_trash;
	}

	/* ごみ箱フォルダ自身を除外するかどうか */
	if ((!divy_enable_trashfolder_access(r) || !divy_support_trashfolder(r)) &&
			(u_spec->infotype == DIVY_INFOTYPE_user_e ||
			 u_spec->infotype == DIVY_INFOTYPE_group_e)) {
		propflag |= DIVY_GET_PROP_notrashfolder;
	}

	/* ユーザ拡張ステータスが有効でPROPFIND の場合 */
	if (divy_support_extenduserstatus(r) && divy_get_method_number(r) == M_PROPFIND) {
		if (rdb_r->u_spec == NULL) {
			divy_parse_uri(r->pool, dav_divy_get_root_uri(r), rdb_r->uri, &rdb_r->u_spec);
		}

		/* グループコレクションよりも下のリソース/コレクション または、
		 * グループコレクション以下のリソース取得要求の時だけ有効 */
		if (rdb_r->u_spec &&
			(rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e_regular ||
			 (rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e && depth != 0))) {
			propflag |= DIVY_GET_PROP_resourcestate;

			/* 制限ユーザであればView属性の付いていないリソースは削り落とす */
			if (!divy_rdbo_is_trusty_user(divy_get_extstatus(r))) {
				propflag |= DIVY_DEL_PROP_noviewattr;
			}
		}

		/*
		 * グループコレクション直下のリソース取得の場合でBOX機能が有効な
		 * グループはゴミ箱フォルダの利用をさせない（外部仕様)
		 */
		if (rdb_r->u_spec &&
			(rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e)) {

			(void) divy_get_cached_availablegroupextstatus(r, rdb_r->uri,
														  		&grp_extstatus);
			if (divy_rdbo_is_box_group(grp_extstatus)) {
				propflag |= DIVY_GET_PROP_notrashfolder;
			}
		}
	}

	/* グループリーダ機能がサポートされていた場合、
	 * グループ情報(groupstate) を取得するかどうか */
	if (divy_support_groupleader(r)) {
		/*
		 * グループコレクション自身(depth = 0), グループコレクショ以下(depth = 1)
		 * のときだけ取得すればいい */
		if (rdb_r->u_spec && rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e) {
			propflag |= DIVY_GET_PROP_groupstate;
		}
	}

	/* 開封通知機能がサポートされていた場合、
	 * confirmreading を取得するかどうか */
	if (divy_support_confirmreading(r)) {
		/* グループコレクション以下への(depth = 0,1) のときだけ取得すればいい */
		if (rdb_r->u_spec &&
			(rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e ||
			 rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e_regular)) {
			propflag |= DIVY_GET_PROP_confirmreading;
		}
	}

	return propflag;
}

/**
 * oldpath があればこれを利用して、fflags のモードで空のファイルを
 * オープンしてそのファイルの抽象パス表現及び物理ファイル情報を
 * 返却する.
 *
 * @param r request_rec *
 * @param fflag int ファイルのopenモード
 * @param oldpath const char * 古いファイルパス(なければNULLを指定すること)
 * @param pfile divy_pfile_t ** 抽象パス表現へのポインタ
 * @param fd apr_file_t ** 物理ファイルへのポインタ
 * @return dav_error * DAVエラー
 */
static dav_error * _open_empty_physical_resource(request_rec *r,
					int fflags, const char *oldpath,
					divy_pfile_t **pfile, apr_file_t **fd)
{
	apr_pool_t *p = r->pool;
	divy_fstorage_t *fstorage = NULL;
	int ret;
	apr_status_t rv;
	char *filepath;

	/* 初期化 */
	*pfile = NULL;
	*fd    = NULL;

	/* ストレージのopen */
	ret = divy_fstorage_open(r, p, &fstorage);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to open storage.(code = %d)", ret);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* 旧ファイルパスが無い場合 -> 新たに開く */
	if (IS_EMPTY(oldpath)) {
		/* 一時ファイルの抽象パスを算出 */
		ret = divy_pfile_mktemp(fstorage, p, DIVY_TMP_FILE_TEMPLATE_NAME, pfile);
		if (ret != DIVY_FS_ST_OK) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to make temporary file path.(code = %d) ", ret);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}
	/* 旧ファイルパスありの場合 -> 既存ファイルパスを利用して新たに開く */
	else {
		ret = divy_pfile_create(fstorage, p, oldpath, pfile);
		if (ret != DIVY_FS_ST_OK) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to create file path.(code = %d)", ret);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* テンプレート付き名称に変更する */
		ret = divy_pfile_rename(*pfile,
			divy_make_temporaryname(fstorage, DIVY_TMP_FILE_TEMPLATE_NAME));
		if (ret != DIVY_FS_ST_OK) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to rename file. (code = %d)", ret);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	/* 一時物理ファイルを開く
	 * (note) apr_file_mktemp の第2引数は書き換えられるのでコピーする */
	filepath = apr_pstrdup(p, divy_pfile_get_fullpath(*pfile));

	rv = apr_file_mktemp(fd, filepath, fflags, p);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to open temporary file for put operation. (code = %d)", rv);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* リネーム (一時ファイルのパスをpfile に反映する) */
	ret = divy_pfile_rename(*pfile, divy_extract_finalpath(p, filepath));
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to rename file. (filepath = %s, code = %d)", filepath, ret);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	(void) divy_fstorage_close(fstorage); /* ストレージを閉じる */

	return NULL;
}

/**
 * ts_ctx のトランザクションの中で、path の物理リソースにnewpath のリソースを
 * 安全に適用する.
 *
 * @param r request_rec *
 * @param path const char * 既存の物理パスリソース
 * @param newpath const char * 新しい物理パスリソース
 * @param rdb_r divy_rdbo_resource * 対象リソースのDB表現
 * @param is_update int ファイルの上書きアップロード処理かどうか
 * @return dav_error * DAVエラー
 */
static dav_error * _commit_physical_resource(request_rec *r,
					const char *path, const char *newpath,
					divy_rdbo_resource *rdb_r, int is_update)
{
	dav_error *err = NULL;
	apr_status_t rv;
	int ret;
	apr_pool_t *p  = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;
	const char *bak_filepath = NULL, *garbage_f = NULL;

	/* トランザクションコンテキストの作成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to create transaction context for put resource.");
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		garbage_f = newpath;	/* 新しいファイルはごみになった */
		goto cleanup_garbage;
	}

	/* 更新の場合 */
	if (is_update) {
		ret = divy_rdbo_update_property(r, rdb_r, ts_ctx);
	}
	/* 新規登録の場合 */
	else {
		ret = divy_rdbo_insert_property(r, rdb_r, ts_ctx);
	}

	/* Quotaオーバー */
	if (ret == DIVY_STCODE_QUOTAOVER) {
		err = dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
	}
	/* 親が無かった / 更新時に自分が無かった */
	else if (ret == DIVY_STCODE_PARENT_NOT_EXIST ||
			(ret == DIVY_STCODE_NOT_EXIST && is_update)) {
		/* 正しくはconflict だが外部仕様としてはNOT Foundにした方が分かりやすい */
		err = dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}
	else if (ret) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to insert / update property for PUT resource.");
		err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 更新の失敗に備え既存ファイルを一時退避する */
	if (is_update && err == NULL) {
		bak_filepath = apr_psprintf(p, "%s.%"APR_PID_T_FMT".%s",
									path, getpid(), DIVY_BAKFILE_SUFFIX);
		rv = apr_file_rename(path, bak_filepath, p);
		if (rv != APR_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to backup old file.(dist = %s, src = %s, code = %d)",
					bak_filepath, path, rv);
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	if (err != NULL) {
		/* トランザクションのロールバックを実施 */
		if (divy_db_rollback_transaction(ts_ctx)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to rollback transaction while processing PUT."
					"Maybe, repository DB server has problem.");
		}
		garbage_f = newpath;	/* 新しいファイルはごみになった */
		goto cleanup_garbage;
	}

	/* newfile をpath にリネーム */
	rv = apr_file_rename(newpath, path, p);
	if (rv != APR_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to rename temp-file.(dist = %s, src = %s, "
				"code = %d)", path, newpath, rv);
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		garbage_f = newpath;	/* 新しいファイルはごみになった */
	}

	/* エラーが発生していた場合 */
	if (err != NULL) {

		/* トランザクションのロールバックを実施 */
		if (divy_db_rollback_transaction(ts_ctx)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to rollback transaction while processing PUT."
				"Maybe, repository DB server has problem.");
		}

		if (IS_FILLED(bak_filepath)) {
			/* バックアップファイルを書き戻す(正式名称への変更に失敗していた場合) */
			rv = apr_file_rename(bak_filepath, path, p);
			if (rv != APR_SUCCESS) {
				/*
				 * オリジナル物理ファイルを書き戻そうとしたのだが失敗
				 * (note)
				 *   オリジナルファイルは別名で退避されたままになっている。
				 *   そのためDBエントリとの整合性が取れない状態にある。
				 *   先ほどはリネームできたのに今度できないというのは
				 *   極めて稀なケースだと思われるので、ここは手動で
				 *   修正してもらうしかないだろう。
				 */
				ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
						"Failed to restore original file."
						"Ths old physical file still has been renamed."
						"Please write back is manually."
						"(renamed file = %s, formal name = %s, code = %d)",
						bak_filepath, path, rv);
				/* バックアップファイルは残しておく */
			}
		}
	}
	/* 成功していた場合 */
	else {
		/* トランザクションのコミットを実施 */
		if (divy_db_commit_transaction(ts_ctx)) {
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");

			/* 上書きの場合 */
			if (is_update) {
				/*
				 * 元の物理ファイルは上書きされたのに、DBエントリを
				 * UPDATE出来なかった場合(データ不正)
				 */
				/* バックアップから書き戻す */
				rv = apr_file_rename(bak_filepath, path, p);
				if (rv != APR_SUCCESS) {
					/*
					 * バックアップからの書き戻しにも失敗した
					 * (note)
					 *   残念ながら、オリジナルファイルは更新され、
					 *   DB エントリは古いままになってしまった。
					 *   このような状況において、DB更新をリトライしても
					 *   恐らくまた失敗する可能性が高い。よって何もしない。
					 */
					ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
							"Failed to commit transaction while processing PUT."
							"The old physical file was overwritten and broken."
							"so, we saved old physical file, Please write back "
							"it manually.(brokenfile = %s, savedfile = %s, "
							"code = %d)", path, bak_filepath, rv);
					/* バックアップファイルは残しておく */
				}
			}
			/* 新規登録の場合 */
			else {
				/* 新しい物理ファイルは正しく書けたがDBを追加できなかった */
				ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to commit transaction while processing PUT."
						"The physical file is OK, but DB entry is missing."
						"so we delete physical file, because it become "
						"a garbage. (brokenfile = %s)", path);
				garbage_f = path;	/* ファイルはごみになった */
			}
		}
		/* commit に成功 */
		else {
			garbage_f = bak_filepath;	/* バックアップファイルは不要 */
		}
	}

cleanup_garbage:

	/* ごみファイルの削除 */
	if (IS_FILLED(garbage_f)) {
		rv = apr_file_remove(garbage_f, p);
		if (rv != APR_SUCCESS && rv != APR_ENOENT) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
					"Failed to remove file. (code = %d) "
					"Please check this file. (filepath = %s)", rv, garbage_f);
		}
	}

	return err;
}

/**
 * resource の親リソースが子リソースを持つことが出来るかどうかを検証する。
 *
 * (note) 子リソースを持てる条件
 * 	・存在していること
 * 	・コレクションであることを
 *
 * @param p apr_pool_t *
 * @param resource const dav_resource *
 * @return DAVエラー
 * 	NULL	: 問題なし
 */
static dav_error * _validate_parent_collection(apr_pool_t *p,
					const dav_resource *resource)
{
	dav_error *err       = NULL;
	dav_resource *parent = NULL;
	char *method   = divy_get_method_name(resource->info->r);

	/* 親コレクションの取得 */
	err = dav_divy_get_parent_resource(resource, &parent);
	if (err != NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"The server could not fetch a "
			"parent resource info. (method = %s)", method);

		return err;
	}

	/* 親コレクションは存在するか？ */
	if (!parent->exists) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Parent collection does not exists. "
			"(method = %s, own = %s, parent = %s)",
			method, resource->uri, parent->uri);

		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/* 親コレクションはコレクションか？ */
	if (!parent->collection) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Parent resource is not COLLECTION. "
			"(method = %s, own = %s, parent = %s)",
			method, resource->uri, parent->uri);

		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	return NULL;
}

/**
 * 指定されたdoc が持つpropertyupdate エレメントがMKCOLの仕様に
 * あっているかどうかを検証する。
 * (note)
 * 	doc がNULL であればエラーを返却します。それを避けたければ
 * 	呼び出し元にて、NULL チェックをして下さい。
 *
 * (note) チェック成功条件
 *   * propertypdate エレメントが存在し、空ではないこと
 *   * set エレメントがただ１つだけ存在すること(removeなどとの混合はNG)
 *   * prop エレメントの中身が存在していること
 *   * prop エレメントで指定されるプロパティは１つだけであること。
 *
 * @param p apr_pool_t *
 * @aram allow_multiprop divy_allow_prop_num prop の中に複数のプロパティを
 * 				許容できるかどうか。
 * @param doc const apr_xml_doc * Doc オブジェクト
 * @return dav_error (NULL: 検証に成功した / 
 * 	HTTP_BAD_REQUEST : XMLの検証エラー
 */
static dav_error * _validate_mkcol_propertyupdate(apr_pool_t *p,
					divy_allow_prop_num allow_multiprop,
					const apr_xml_doc *doc)
{
	const apr_xml_elem *set_elem, *prop_elem;

	/* doc が存在しなかった */
	if (doc == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The request body does not contain any element.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * propertyupdate タグのチェック
	 */
	if (!dav_validate_root(doc, "propertyupdate")) {

		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The request body does not contain a "
			"\"propertyupdate\" element or has invalid namespace uri "
			"for MKCOL.(propname = %s, nsid = %d)",
			doc->root->name, doc->root->ns);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * set エレメントのチェック
	 */
	set_elem = doc->root->first_child;
	if (set_elem == NULL || set_elem->ns != APR_XML_NS_DAV_ID ||
				strcmp(set_elem->name, "set") != 0) {

		/* set エレメントが正しく設定されていない */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The \"set\" element is missing or has invalid "
			"namespece uri in the \"propertyupdate\" element.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * prop エレメントのチェック
	 */
	prop_elem = set_elem->first_child;
	if (prop_elem == NULL || prop_elem->ns != APR_XML_NS_DAV_ID ||
				 strcmp(prop_elem->name, "prop") != 0) {

		/* prop エレメントが正しく設定されていない */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The \"prop\" element is missing or has invalid "
			"namespece uri in the \"propertyupdate\" element.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* prop エレメントのChildが存在するかどうか */
	if (prop_elem->first_child == NULL) {
		/* child が存在しない */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The child element of \"prop\" is missing "
			"in the \"propertyupdate\" element.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* prop の中に一つのプロパティしか許されていない場合 */
	if (allow_multiprop == DIVY_ALLOW_SINGLE_PROP) {
		if (prop_elem->next || (prop_elem->first_child &&
					prop_elem->first_child->next)) {

			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"prop\" element is specified with "
				"the several for MKCOL.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * 指定されたkeylist_pr が MKCOLの仕様にあっているかどうかを検証する。
 * (note)
 * 	keylist_pr がNULL であればエラーを返却します。それを避けたければ
 * 	呼び出し元にて、NULL チェックをして下さい。
 *
 * (note) チェック成功条件
 * ・keylist_pr がNULLではないこと
 * ・rdb_r->diplayname がNULL または空文字でないこと
 * ・index が連番であること
 * ・index が1から10までの数値であること
 * ・key がNULL ではないこと
 *
 * @param p apr_pool_t *
 * @param rdb_r const divy_rdbo_resource * リクエストされたリソース
 * @param keylist_pr const divy_rdbo_keylist * keylist の内容
 * @return dav_error (NULL: 検証に成功した / 
 * 		      NULL以外: 検証に失敗した or 検証エラー発生)
 * 		      400 を返します。
 */
static dav_error * _validate_mkcol_dbfolder(apr_pool_t *p,
					const divy_rdbo_resource *rdb_r,
					const divy_rdbo_keylist *keylist_pr)
{
	int i, cnt;
	const divy_rdbo_keylist *keylist;
	const divy_rdbo_keylist *keys[] = {NULL, NULL, NULL, NULL, NULL,
					   NULL, NULL, NULL, NULL, NULL}; 

	if (keylist_pr == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"keyvalue\" element is missing."
			"(uri = %s)", rdb_r->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (IS_EMPTY(rdb_r->displayname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"displayname\" element MUST be specified "
			"in the \"prop\" element for MKCOL.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* keylist_pr をindexの昇順になるよう並び替える */
	cnt = 0;
	for (keylist = keylist_pr; keylist; keylist = keylist->next) {
		/* index の最大値である10を超えていないか？*/
		if (keylist->index > 10) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"id\" attribute of \"keylist\" element "
				"is bigger than number of 10."
				"Must be smaller than 10.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
		else if (keylist->index <= 0) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"id\" attribute of \"keylist\" element "
				"is invalid.(id = %d)", keylist->index);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* keylist の メンバ key はNULLではないか？*/
		if (keylist->key == NULL) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The value  of \"keylist\" element is empty.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* keys に記録する */
		keys[keylist->index - 1] = keylist;
		cnt++;
	}

	/* keylist_pr のindex に歯抜けがないかどうかをチェックする */
	for (i = 0; i < cnt; i++) {
		if (keys[i] == NULL) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"id\" attribute of \"keylist\" element "
				"MUST be a serial number.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * 指定されたshsrccol_pr が MKCOLの仕様にあっているかどうかを検証する。
 * (note)
 * 	shsrccol_pr がNULL であればエラーを返却します。それを避けたければ
 * 	呼び出し元にて、NULL チェックをして下さい。
 *
 * (note) チェック成功条件
 * ・shsrccol_pr がNULLではないこと
 * ・name, srcuri がNULLではないこと
 * ・name, srcuri に不正な文字が含まれていないこと
 *
 * @param r request_rec *
 * @param rdb_r const divy_rdbo_resource * リクエストされたリソース
 * @param keylist_pr const divy_rdbo_keylist * keylist の内容
 * @return dav_error (NULL: 検証に成功した / 
 * 		      NULL以外: 検証に失敗した or 検証エラー発生)
 * 		      400 を返します。
 */
static dav_error * _validate_mkcol_dbshfolder(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					const divy_rdbo_shsrccol *shsrccol_pr)
{
	const char *uri    = rdb_r->uri;
	apr_pool_t *p      = r->pool;
	char *uri_top_part = NULL, *tmp;
	const divy_rdbo_shsrccol *shsrc = NULL;

	if (shsrccol_pr == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sharedcollectioncreate\" element is missing."
			"(uri = %s)", uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 共有コレクションのソースURI(linkdbsearch結果フォルダURI)
	 * の上部を作成する */
	uri_top_part = divy_build_dbfolder_uri(p, dav_divy_get_root_uri(r), "/");

	/* name と uri のチェック */
	for (shsrc = shsrccol_pr; shsrc; shsrc = shsrc->next) {
		/* 表示名 がない */
		if (IS_EMPTY(shsrc->name)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"name\"(\"sharedsrccollection\" child) "
				"element is missing. (uri = %s)", uri);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
		/* 禁則文字が含まれている */
		if (divy_validate_pathsegment(p, 0, shsrc->name)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"name\"(\"sharedsrccollection\" child) "
				"element has invalid char.(name = %s)",
				shsrc->name);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* uri がない */
		if (IS_EMPTY(shsrc->uri)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"srcuri\"(\"sharedsrccollection\" child) "
				"element is missing.(uri = %s)", uri);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		tmp = divy_strstr(shsrc->uri, uri_top_part);
		if (tmp == NULL || tmp != shsrc->uri) {
			/* shlink_pr->uri の先頭がshuri_top_part ではない */
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The content of \"linkuri\"(\"sharedcollection\" "
				"child) element does not have a valid "
				"linkdbsearch folder uri.(uri = %s)",
				shsrc->uri);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* uri のfinal path segment に禁則文字が含まれている */
		tmp = divy_strrchr(shsrc->uri, '/');
		if (divy_validate_pathsegment(p, 0, ++tmp)) {

			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"linkuri\"(\"sharedccollection\" child) "
				"element has invalid char.(uri = %s)",
				shsrc->uri);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * usr_pr の示すユーザが削除可能かどうかを検証する。
 *
 * [ 検証成功条件 ]
 *	* 管理者が自分自身のユーザプロパティを削除しないこと
 * 	* プライベートコレクション以下にごみ箱以外のリソースが存在しないこと
 * (2005/04/01) ロックの有無は削除条件から除外されました。
 *
 * @param r request_rec *
 * @param resource const dav_resource * 削除リソースの情報
 * @param usr_pr const divy_rdbo_usr * ユーザプロパティ
 * @param response dav_response **
 * @return DAV エラー(403)
 */
static dav_error * _validate_remove_user(request_rec *r,
					const dav_resource *resource,
					const divy_rdbo_usr *usr_pr,
					dav_response **response)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_resource usr_rdb_r = { 0 };
	dav_response *new_response   = NULL, *first = NULL;
	divy_rdbo_resource *rdb_r    = NULL, *n_rdb_r;
	int found_resource = 0;
	int status = HTTP_OK;
	char *trash_uri;
	int force_delete = 0;
	int support_groupleader = divy_support_groupleader(r);

	if (usr_pr == NULL) return NULL;

	/* 自分自身のユーザプロパティを削除することは出来ない */
	if (IS_FILLED(usr_pr->usrid) && strcmp(usr_pr->usrid, divy_get_userid(r)) == 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"You could not delete own user property.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* 削除対象ユーザがグループリーダの場合
	 * 管理下においているユーザが存在すれば削除はNG */
	if (support_groupleader && divy_rdbo_is_groupleader(usr_pr->extstatus)) {
		int has_user = 0;

		if (divy_rdbo_has_owner_users(r, usr_pr->usrid, &has_user)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get owner-user information.");
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		if (has_user) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"Could not delete this groupleader because this user have owner-user. "
					"(target user = %s)", usr_pr->usrid);

			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NODELETE_GROUPLEADER);	/* ヘッダに設定 */
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	/* 自身がグループリーダの場合、Otherユーザを削除できてはならない */
	if (support_groupleader && divy_rdbo_is_groupleader(divy_get_extstatus(r)) && usr_pr->is_otheruser) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"Could not delete this user because target user is other-user."
				"(target user = %s)", usr_pr->usrid);
		divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NODELETE_OTHERUSER);	/* ヘッダに設定 */
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* 強制削除フラグの判定 */
	force_delete = divy_validate_is_forcedelete(r);
	if (force_delete) {
		/* 強制削除が指示された場合、内包リソースの存在チェックは無意味 */
		return NULL;
	}

	/* 以下は、強制削除指示がされていない場合 */

	/*
	 * ごみ箱以外のリソースが存在するかどうかの確認
	 */
	usr_rdb_r.uri          = usr_pr->prvcol_uri;
	usr_rdb_r.depth        = divy_count_dirs(usr_rdb_r.uri);
	usr_rdb_r.resourcetype = DIVY_TYPE_COLLECTION;
	usr_rdb_r.rsid         = NULL;
	usr_rdb_r.next         = NULL;

	/* (note)
	 * 	本来であれば depth = DAV_INFINITY で取得すべきであるが、
	 * 	大量のリソースを抱えていた場合、著しい性能劣化に繋がる。
	 * 	なので、depth = 1 にして直下のリソースだけを調べることにした。
	 */
	if (divy_rdbo_get_hierarchy_property(r, &usr_rdb_r, 1, 0, NULL)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get resource information."
			"(userid = %s)", usr_pr->usrid);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* ごみ箱でもトップフォルダでもないリソースがあったか？ */
	trash_uri = divy_build_user_trash_uri(p, dav_divy_get_root_uri(r),
							usr_pr->usrid);
	for (rdb_r = &usr_rdb_r; rdb_r; rdb_r = rdb_r->next) {
		if (IS_FILLED(rdb_r->uri) &&
		    strcmp(rdb_r->uri, trash_uri) != 0 &&
		    strcmp(rdb_r->uri, usr_pr->prvcol_uri) != 0) {
			/* (note)
			 * 	例えごみ箱機能をサポートしていなくても
			 * 	ごみ箱フォルダに対するオペレーションは常に制限
			 * 	されているのでここでごみ箱フォルダは無視しないと
			 * 	このユーザは一生削除出来なくなってしまう */
			found_resource = 1;
			break;
		}
	}

	/* リソースが存在していたらNG */
	if (found_resource) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The server could not DELETE user. "
			"Because private collection have some resources."
			"(prvcol_uri = %s)", usr_rdb_r.uri);

		status = HTTP_OK;
		rdb_r  = usr_rdb_r.next;
		goto remove_user_failed_dependency;
	}

	return NULL;

remove_user_failed_dependency:

	/* Failed Dependency レスポンスの生成 */
	for (n_rdb_r = rdb_r; n_rdb_r; n_rdb_r = n_rdb_r->next) {
		if (first == NULL) {
			first = new_response =
					apr_pcalloc(p, sizeof(dav_response));
		}
		else {
			new_response->next =
					apr_pcalloc(p, sizeof(dav_response));
			new_response = new_response->next;
		}

		if (n_rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
			new_response->href = apr_psprintf(p, "%s/", n_rdb_r->uri);
		}
		else {
			new_response->href = n_rdb_r->uri;
		}
		new_response->status = status;
	}

	/* *response に繋げる */
	new_response->next = *response;
	*response = first;

	return dav_new_error(p, HTTP_FAILED_DEPENDENCY, 0, 0, "");
}

/**
 * grp_pr の示すグループが削除可能かどうか検証する。
 *
 * [ 検証成功条件 ]
 * 	* 削除するグループ以下にごみ箱以外のリソースが存在しないこと
 * (2005/04/01) ロックの有無は削除条件から除外されました。
 *
 * @param r request_rec *
 * @param resource const dav_resource *
 * @param sql_pr const divy_rdbo_sql * SQLプロパティ
 * @param response dav_response **
 * @return DAVエラー (HTTP_FAILED_DEPENDENCY, HTTP_INTERNAL_SERVER_ERROR)
 */
static dav_error * _validate_remove_group(request_rec *r,
					const dav_resource *resource,
					const divy_rdbo_grp *grp_pr,
					dav_response **response)
{
	apr_pool_t *p = r->pool;
	dav_response *new_response = NULL, *first = NULL;
	divy_rdbo_resource *rdb_r  = NULL, *n_rdb_r;
	divy_rdbo_grp *grp_pr_list = NULL, *n_grp_pr;
	divy_rdbo_resource grp_rdb_r = { 0 };
	char *trash_uri;
	int found_resource = 0;
	int status = HTTP_OK;
	int force_delete = 0;

	/* 強制削除フラグの判定 */
	force_delete = divy_validate_is_forcedelete(r);
	if (force_delete) {
		/* 強制削除が指示された場合、内包リソースの存在チェックは無意味 */
		return NULL;
	}
	/* 以下は、強制削除指示がされていない場合 */

	/*
	 * 指定されたグループと下位階層グループのグループ情報を取得する
	 */
	if (divy_rdbo_get_hierarchy_group_property(r, grp_pr->relativeuri,
					DAV_INFINITY, &grp_pr_list, NULL)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get hierarchy group information."
			"(relativeuri = %s)", grp_pr->relativeuri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * グループコレクション以下に、リソースとロックリソースが
	 * 存在しないこと
	 */
	for (n_grp_pr = grp_pr_list; n_grp_pr; n_grp_pr = n_grp_pr->next) {
		/*
		 * リソースの存在確認(実際に検索してみる)
		 */
		grp_rdb_r.uri          = n_grp_pr->grpcol_uri;
		grp_rdb_r.depth        = divy_count_dirs(grp_rdb_r.uri);
		grp_rdb_r.resourcetype = DIVY_TYPE_COLLECTION;
		grp_rdb_r.rsid         = NULL;
		grp_rdb_r.next         = NULL;

		/* (note) 性能劣化を防ぐため depth = 1 で取ります。
		 * 	本来はDAV_INFINITY で取るべきなのですが */
		if (divy_rdbo_get_hierarchy_property(r, &grp_rdb_r, 1, 0, NULL)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get resource information."
				"(grpcol_uri = %s)", n_grp_pr->grpcol_uri);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* ごみ箱でもトップフォルダでもないリソースがあったか？ */
		(void) divy_extract_trashfolder_uri(p, dav_divy_get_root_uri(r),
						n_grp_pr->grpcol_uri, &trash_uri);

		for (rdb_r = &grp_rdb_r; rdb_r; rdb_r = rdb_r->next) {
			if (IS_FILLED(rdb_r->uri) &&
			    strcmp(rdb_r->uri, trash_uri) != 0 &&
			    strcmp(rdb_r->uri, n_grp_pr->grpcol_uri) != 0) {
				/* (note)
				 * 	例えごみ箱機能をサポートしていなくても
				 * 	ごみ箱フォルダに対するオペレーションは常に制限
				 * 	されているのでここでごみ箱フォルダは無視しないと
				 * 	このグループは一生削除出来なくなってしまう */
				found_resource = 1;
				break;
			}
		}

		/* リソースが存在していたらNG */
		if (found_resource) {
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The server could not DELETE group. "
				"Because group collection have some resources."
				"(grpcol_uri = %s)", n_grp_pr->grpcol_uri);

			status = HTTP_OK;
			rdb_r  = grp_rdb_r.next;
			goto remove_group_failed_dependency;
		}
	}

	return NULL;

remove_group_failed_dependency:

	/* Failed Dependency レスポンスの生成 */
	for (n_rdb_r = rdb_r; n_rdb_r; n_rdb_r = n_rdb_r->next) {
		if (first == NULL) {
			first = new_response =
					apr_pcalloc(p, sizeof(dav_response));
		}
		else {
			new_response->next =
					apr_pcalloc(p, sizeof(dav_response));
			new_response = new_response->next;
		}

		if (n_rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
			new_response->href = apr_psprintf(p, "%s/", n_rdb_r->uri);
		}
		else {
			new_response->href = n_rdb_r->uri;
		}
		new_response->status = status;
	}

	/* *response に繋げる */
	new_response->next = *response;
	*response = first;

	return dav_new_error(p, HTTP_FAILED_DEPENDENCY, 0, 0, "");

}

/**
 * sql_pr の示すSQLが削除可能かどうか検証する。
 * 依存SQLが存在し削除できない場合には、reponse にその情報を格納して返却します。
 *
 * @param r request_rec *
 * @param resource const dav_resource *
 * @param sql_pr const divy_rdbo_sql * SQLプロパティ
 * @param response dav_response **
 * @return DAVエラー (HTTP_FAILED_DEPENDENCY, HTTP_INTERNAL_SERVER_ERROR)
 */
static dav_error * _validate_remove_sql(request_rec *r,
					const dav_resource *resource,
					const divy_rdbo_sql *sql_pr,
					dav_response **response)
{
	int ret;
	divy_sql_parser *parser         = NULL;
	divy_rdbo_sqldepend *dependlist = NULL;
	divy_cset_t *usingsql_set       = NULL;
	apr_pool_t *p                   = r->pool;
	dav_response *new_response      = NULL, *first = NULL;
	char **namelist                 = NULL;
	char *baseuri;
	divy_rdbo_sql *sql_uris = NULL, *tmp_sql = NULL;

	/*
	 * このSQLは依存ノードに参加できるSQLか？
	 */
	if (sql_pr->type == DIVY_SQL_TYPE_NORMAL ||
	    sql_pr->type == DIVY_SQL_TYPE_REPOSITORY) {
		return NULL;	/* 参加できないノードでした */
	}

	/*
	 * 依存関係リストの取得
	 */
	if (divy_rdbo_get_sqldepend_list(r, NULL, &dependlist, NULL)) {
		
		/* 予期しないエラー発生 */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get sql-depend list."
			"(sqlno = %s, sqlname = %s)",
			REPLACE_NULL(sql_pr->sqlid),
			REPLACE_NULL(sql_pr->labelname));

		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * このSQLを利用するSQLが無いかどうかを検証
	 */

	/* SQLパーサの作成 */
	(void) divy_sql_parser_create(p, &parser);

	/* 依存集合の取得 */
	ret = divy_sql_parser_find_usingsql(parser, sql_pr->subname,
						dependlist, &usingsql_set);
	if (ret == DIVY_SQLP_ST_USINGSQL_FOUND) {

		/* 依存SQLが見つかった場合 --> 424 Failed Dependency
		 * (起き得るので警告にしておく) */
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"Failed to delete this sql, because the sql was "
			"used by another sql.(sqlno = %s, subname = %s)",
			REPLACE_NULL(sql_pr->sqlid), sql_pr->subname);

		baseuri = divy_build_m_sql_uri(p, dav_divy_get_root_uri(r), NULL);
		
		/* subname -> SQL のuriを取得 */
		namelist = divy_cset_toarray(usingsql_set);
		if (divy_rdbo_get_sqluri_by_subname(r,
					(const char **) namelist, &sql_uris)) {

			/* 予期しないエラー発生 */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get sqlname list.(sqlno = %s, "
				"sqlname = %s)", REPLACE_NULL(sql_pr->sqlid),
				REPLACE_NULL(sql_pr->labelname));
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* divy_sqldependテーブルとdivy_sqlテーブルの間に矛盾が生じていた */
		if (sql_uris == NULL) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to sql uri list, bacause the set of "
				"divy_sqldepend is different from divy_sql."
				"Maybe divy_sqldepend table have some "
				"unnecessary data. "
				"Please check divy_sqldepend and divy_sql and "
				"correct these relationship.(sqlno = %s,"
				"sqlname = %s)", REPLACE_NULL(sql_pr->sqlid),
				REPLACE_NULL(sql_pr->labelname));
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* Failed Dependency 用multistatus ボディの作成 */
		for (tmp_sql = sql_uris; tmp_sql; tmp_sql = tmp_sql->next) {
			if (first == NULL) {
				first = new_response =
					apr_pcalloc(p, sizeof(dav_response));
			}
			else {
				new_response->next = apr_pcalloc(p, sizeof(dav_response));
				new_response = new_response->next;
			}

			new_response->href = apr_psprintf(p, "%s/",
						dav_divy_make_uri(p,
							(const char *) baseuri,
							(const char *) tmp_sql->relativeuri,
							NULL));
			new_response->status = HTTP_LOCKED;
		}

		/* *response に繋げる */
		new_response->next = *response;
		*response = first;

		return dav_new_error(p, HTTP_FAILED_DEPENDENCY, 0, 0, "");
	}
	else if (ret != DIVY_SQLP_ST_OK) {

		/* 予期しないエラー発生 */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to find usingsql.(sqlno = %s, sqlname = %s)",
			REPLACE_NULL(sql_pr->sqlid),
			REPLACE_NULL(sql_pr->labelname));

		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * dbms_pr の示すDBMSが削除可能かどうか検証する。
 * このDBMSを利用しているSQLが存在する場合には、reponse にその情報を格納して返却します。
 *
 * @param r request_rec *
 * @param resource const dav_resource *
 * @param dbms_pr const divy_rdbo_dbms * DBMSプロパティ
 * @param response dav_response **
 * @return DAVエラー (HTTP_FAILED_DEPENDENCY, HTTP_INTERNAL_SERVER_ERROR)
 */
static dav_error * _validate_remove_dbms(request_rec *r,
					const dav_resource *resource,
					const divy_rdbo_dbms *dbms_pr,
					dav_response **response)
{
	apr_pool_t *p               = r->pool;
	dav_response *new_response  = NULL, *first = NULL;
	apr_hash_t *usingdbms_set   = NULL;
	apr_hash_index_t *hi;
	const char *sql_uri;

	/*
	 * このDBMSを利用するSQLが無いかどうかを検証
	 */
	if (divy_rdbo_find_sql_usingdbms(r, dbms_pr->name, &usingdbms_set)) {
		/* 予期しないエラー発生 */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to find usingdbms.(dbmsname = %s)",
			dbms_pr->name);

		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	else if (usingdbms_set && apr_hash_count(usingdbms_set) > 0) {
		/* 依存SQLが見つかった場合 --> 424 Failed Dependency
		 * (起き得るので警告にしておく) */
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"Failed to delete this dbms, because the dbms was "
			"used by sql.(dbmsname = %s)", dbms_pr->name);

		/* Failed Dependency 用multistatus ボディの作成 */
		for (hi = apr_hash_first(p, usingdbms_set); hi; hi = apr_hash_next(hi)) {

			apr_hash_this(hi, (const void **) &sql_uri, NULL, NULL);

			if (first == NULL) {
				first = new_response =
					apr_pcalloc(p, sizeof(dav_response));
			}
			else {
				new_response->next = apr_pcalloc(p, sizeof(dav_response));
				new_response = new_response->next;
			}

			new_response->href = sql_uri;
			new_response->status = HTTP_LOCKED;
		}

		/* *response に繋げる */
		new_response->next = *response;
		*response = first;

		return dav_new_error(p, HTTP_FAILED_DEPENDENCY, 0, 0, "");
	}

	return NULL;
}

/**
 * 指定されたsrc が表すリレーションをdst として作成可能かどうかを
 * 検証する.
 *
 * (note) 検証にパスする条件
 * 　・src がエンティティで、dst がエンティティのリレーションであること。
 * 　・dst のfinal path segment に禁則文字が含まれていないこと
 * 　・src と dst の整合性を確かめる
 *   ・SQLリレーションの場合、そのエンティティがRequiredSQLまたは
 *     名前付きバインド変数の時、リレーションの作成はできない
 *
 * @param r request_rec *
 * @param src const dav_resource * ソース
 * @param dst const dav_resource * ディスティネーション
 * @return dav_error * DAVエラーを返す(400, 403, 409)
 */
static dav_error * _validate_create_relation(request_rec *r,
					const dav_resource *src,
					const dav_resource *dst)
{
	apr_pool_t *p = r->pool;

	divy_uri_spec *src_u_spec = src->info->rdb_r->u_spec;
	divy_uri_spec *dst_u_spec = dst->info->rdb_r->u_spec;

	/*
	 * dst のfinal path segment に禁則が含まれていないことを検証する
	 */
	if (divy_validate_pathsegment(p, 0, dst_u_spec->final_path_segment)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The part of uri has invalid char." );
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * src と dst のURIの整合性を検証する.
	 *
	 * [ ユーザ -> ユーザリレーション]
	 * 	src: $root/.management/USER/$userid
	 * 	dst: $root/.management/GROUP/$grp_uri/.RU_$userid
	 *
	 * [ SQL -> SQLリレーション ]
	 * 	src: $root/.management/SQL/$sqlname
	 * 	dst: $root/.management/GROUP/$grp_uri/.RS_$sqlname
	 *
	 * [ グループ -> グループリレーション ]
	 * 	src: $root/.management/GROUP/$grp_uri
	 * 	dst: $root/.management/USER/$userid/.RG_$grpid
	 * 	     $root/.management/SQL/$sqlname/.RG_$grpid
	 */
	if (src_u_spec->infotype == DIVY_INFOTYPE_m_user_e &&
	    dst_u_spec->infotype == DIVY_INFOTYPE_m_group_ruser) {
		divy_rdbo_rusr *rusr = NULL;

		/* dst->uri のパース */
		(void) divy_rdbo_parse_rusr_uri(r, dst->uri, &rusr);
		if (rusr && strcmp(rusr->usrid, src_u_spec->final_path_segment)) {
			/* final path segment のuserid 部分が一致しなかった */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The src final path segment is different from "
				"the userid part of dst final path segment."
				"(src = %s, dst = %s)", src->uri, dst->uri);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}
	else if (src_u_spec->infotype == DIVY_INFOTYPE_m_sql_e &&
		 dst_u_spec->infotype == DIVY_INFOTYPE_m_group_rsql) {
		divy_rdbo_rsql *rsql = NULL;

		/* dst->uri のパース */
		(void) divy_rdbo_parse_rsql_uri(r, dst->uri, &rsql);
		if (rsql && strcmp(rsql->labelname, src_u_spec->final_path_segment) != 0) {
			/* final path segment の表示名称部分が一致しなかった */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The src final path segment is different from "
				"the sqlname part of dst final path segment."
				"(src = %s, dst = %s)", src->uri, dst->uri);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* RequiredSQL及び名前付きバインド変数は、グループに
		 * 使用権を譲渡してはならない */
		if (src->info->rdb_r->sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
		    src->info->rdb_r->sql_pr->type == DIVY_SQL_TYPE_BIND) {

			ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Could not make the RequiredSQL and NamedBind "
				"belong to a group."
				"(src = %s, dst = %s, sqltype = %d)",
				src->uri, dst->uri, src->info->rdb_r->sql_pr->type);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}
	else if ((src_u_spec->infotype == DIVY_INFOTYPE_m_group_e &&
		  dst_u_spec->infotype == DIVY_INFOTYPE_m_user_rgrp) ||
		 (src_u_spec->infotype == DIVY_INFOTYPE_m_group_e &&
		  dst_u_spec->infotype == DIVY_INFOTYPE_m_sql_rgrp)) {
		divy_rdbo_rgrp *rgrp = NULL;
		divy_rdbo_grp  *src_grp_pr = src->info->rdb_r->grp_pr;

		/* dst->uri のパース */
		(void) divy_rdbo_parse_rgrp_uri(r, dst->uri, &rgrp);
		if (rgrp && src_grp_pr && src_grp_pr->grpid  &&
			strcmp(src_grp_pr->grpid, rgrp->grpid) != 0) {

			/* src のグループURIから検索したグループIDと
			 * dst のfinal path segment のグループID部分が一致しなかった */
			ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The src group is different from "
				"the groupid part of dst final path segment."
				"(src = %s, src_grpid = %s, dst = %s)",
				src->uri, src_grp_pr->grpid, dst->uri);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}
	/* 上記のどれにも一致しなかった場合 */
	else {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"When creating relation, the type of src must be "
			"entity-resource and the type of dst must be"
			"relation-resource.(src = %s, dst = %s)",
			src->uri, dst->uri);

		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	return NULL;
}

/**
 * SQLをチェックし、そのSQLに対する使用権が親から継承されてきた
 * ものであれば、オペレーションが可能かどうか判断する.
 *
 * (note) チェックにパスする条件
 *	・SQL使用権を継承していないこと
 *	・DELETE, MOVE, COPY ではないこと
 *
 * @param r request_rec *
 * @param resource const dav_resource * チェック対象リソース
 * @return dav_error * DAVエラーを返す(424 & 409)
 */
static dav_error * _validate_inheritsql(request_rec *r,
					const dav_resource *resource,
					dav_response **response)
{
	apr_pool_t *p          = r->pool;
	divy_rdbo_rsql *rsql   = NULL;
	divy_rdbo_rgrp *rgrp   = NULL;
	divy_rdbo_grp *grp_pr  = NULL;
	divy_cset_t *grpid_set = divy_cset_make(p);
	char *parent_grpid     = NULL;
	dav_response *new_response = NULL;
	int mnum               = divy_get_method_number(r);

	divy_uri_spec *u_spec = resource->info->rdb_r->u_spec;

	/* SQL リレーションの場合 */
	if (u_spec->infotype == DIVY_INFOTYPE_m_group_rsql) {
		rsql = resource->info->list.rsql_pr;

		/* 使用権を継承していた場合 */
		if (rsql && rsql->inheritsql == DIVY_SQL_INHERIT_TRUE) {
			parent_grpid  = rsql->owner_grpid;
			divy_cset_set(grpid_set, rsql->owner_grpid);

			/* グループのURIを取得 */
			if (divy_rdbo_get_grpuri_by_grpid(r, grpid_set, &grp_pr)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get group uri using groupid."
					"(uri = %s)", resource->uri);
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}
		}
	}
	/* グループリレーションの場合 */
	else if (u_spec->infotype == DIVY_INFOTYPE_m_sql_rgrp) {
		rgrp = resource->info->list.rgrp_pr;

		/* 使用権を継承していた場合 */
		if (rgrp && rgrp->inheritsql == DIVY_SQL_INHERIT_TRUE) {
			parent_grpid  = rgrp->owner_grpid;
		}
	}
	else {
		/* 処理できません */
		return NULL;
	}

	if (parent_grpid == NULL) return NULL;

	/*
	 * メソッドの判定
	 */
	if (mnum != M_COPY && mnum != M_MOVE && mnum != M_DELETE) {
		return NULL;	/* これも正常 */
	}

	/* エラーメッセージを出す(起き得るので警告にしておく) */
	ERRLOG3(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
		"Failed to \"%s\" this resource, because the resource was "
		"owner by another one.(access uri = %s, owner grp = %s)",
		divy_get_method_name(r), resource->uri, parent_grpid);

	/*
	 * メソッドがCOPY, MOVE, DELETE で継承したSQLに対する
	 * オペレーションは禁止
	 */

	/* レスポンスの作成 */
	new_response = apr_pcalloc(p, sizeof(dav_response)); 
	if (u_spec->infotype == DIVY_INFOTYPE_m_group_rsql) { 
		new_response->href   = dav_divy_make_uri(p,
					u_spec->root,
					u_spec->special_part,
					grp_pr->relativeuri,
					u_spec->final_path_segment,
					NULL);
	}
	else {
		new_response->href   = dav_divy_make_uri(p,
					u_spec->root,
					u_spec->special_part,
					rgrp->labelname,
					apr_psprintf(p, DIVY_PREFIX_RGROUP"%s", parent_grpid),
					NULL);

	}
	new_response->status = HTTP_CONFLICT;
	new_response->next   = *response;

	*response = new_response;
	return dav_new_error(p, HTTP_FAILED_DEPENDENCY, 0, 0, "");
}

/**
 * 指定されたsrc が表すグループをdst のグループとして
 * コピー可能かどうかを検証する。
 *
 * (note) 検証にパスする条件
 *   ・(divy_validate_copy_entity が実施しているもの)
 *   ・syncgrpuri がONのとき、複数階層のグループ作成を許可しない
 *
 * @param r request_rec *
 * @param src const dav_resource * ソース
 * @param dst const dav_resource * ディスティネーション
 * @return dav_error * DAVエラー(400, 409). 問題なければNULL
 */
static dav_error * _validate_copy_group(request_rec *r,
					const dav_resource *src,
					const dav_resource *dst)
{
	dav_error *err = NULL;
	apr_pool_t *p  = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

	divy_uri_spec *src_u_spec = src->info->rdb_r->u_spec;
	divy_uri_spec *dst_u_spec = dst->info->rdb_r->u_spec;

	/* エンティティの検証 */
	err = divy_validate_copy_entity(r, src_u_spec, dst_u_spec);
	if (err) return err;

	/* syncgrpuri がONならば複数階層のグループ作成を許可しない */
	if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON &&
		(divy_count_dirs(src_u_spec->other_part) > 1 ||
		 divy_count_dirs(dst_u_spec->other_part) > 1)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"At present, you could not copy a group "
			"under another group. Please consider change of "
			"TeamFile setting.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	return NULL;
}

/**
 * 指定されたsrc が表すユーザをdst のユーザとして
 * 移動可能かどうかを検証する。
 *
 * (note) 検証にパスする条件
 *	* (関数divy_validate_move_entityと同じ)
 *	* プライベートコレクションの下にロックがないこと
 *
 * @param r request_rec *
 * @param src const dav_resource * ソース
 * @param dst const dav_resource * ディスティネーション
 * @param response dav_response ** エラーレスポンス
 * @return dav_error 問題なければNULL, 問題があればエラーを返す(409)
 */
static dav_error * _validate_move_user(request_rec *r,
					const dav_resource *src,
					const dav_resource *dst,
					dav_response **response)
{
	dav_error *err = NULL;
	apr_pool_t *p             = r->pool;
	divy_rdbo_usr *src_usr_pr = src->info->rdb_r->usr_pr;

	dav_response *new_response = NULL, *first = NULL;
	divy_rdbo_resource *rdb_r  = NULL, *null_rdb_r = NULL, *n_rdb_r;

	/* エンティティの検証 */
	err = divy_validate_move_entity(r, src->info->rdb_r->u_spec,
						dst->info->rdb_r->u_spec);
	if (err) return err;

	/*
	 * プライベートコレクション以下にロックが無いこと
	 */
	if (divy_rdbo_get_hierarchy_lockedresource(r,
					src_usr_pr->prvcol_uri, &rdb_r)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get locked resource."
			"(prvcol_uri = %s)", src_usr_pr->prvcol_uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * LOCK NULLリソースの確認
	 */
	if (divy_rdbo_get_null_resource_member(r, src_usr_pr->prvcol_uri,
					DAV_INFINITY, 1, &null_rdb_r)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get lock-null resource."
			"(prvcol_uri = %s)", src_usr_pr->prvcol_uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	if (rdb_r != NULL && null_rdb_r != NULL) {
		for (n_rdb_r = rdb_r; n_rdb_r->next; n_rdb_r = n_rdb_r->next);
		n_rdb_r->next = null_rdb_r;
	}
	else if (rdb_r == NULL && null_rdb_r != NULL) {
		rdb_r = null_rdb_r;
	}

	/* ロックされたリソース、またはLOCK NULLリソースがあったらNG */
	if (rdb_r != NULL) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The server could not MOVE user. "
			"Because private collection have some locked resources."
			"(prvcol_uri = %s)", src_usr_pr->prvcol_uri);

		/* Failed Dependency レスポンスの生成 */
		for (n_rdb_r = rdb_r; n_rdb_r; n_rdb_r = n_rdb_r->next) {
			if (first == NULL) {
				first = new_response =
					apr_pcalloc(p, sizeof(dav_response));
			}
			else {
				new_response->next =
					apr_pcalloc(p, sizeof(dav_response));
				new_response = new_response->next;
			}

			if (n_rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
				new_response->href = apr_psprintf(p, "%s/",
						n_rdb_r->uri);
			}
			else {
				new_response->href = n_rdb_r->uri;
			}
			new_response->status = HTTP_LOCKED;
		}
		/* *response に繋げる */
		new_response->next = *response;
		*response = first;

		return dav_new_error(p, HTTP_FAILED_DEPENDENCY, 0, 0, "");
	}

	return NULL;

}

/**
 * 指定されたsrc が表すグループをdst のグループとして
 * 移動可能かどうかを検証する。
 *
 * (note) 検証にパスする条件
 *   ・(関数divy_validate_move_entityと同じ)
 *   ・[ syncgrpuri = on ] 複数階層のグループ作成を許可しない
 *   ・syncgrpuri を運用中に変更したと思われるオペレーションを許可しない
 *   ・[ syncgrpuri = on || nameとfinal path が異なる]
 *   		グループコレクションの下にロックがないこと
 *
 * @param r request_rec *
 * @param src const dav_resource * ソース
 * @param dst const dav_resource * ディスティネーション
 * @param response dav_response ** エラーレスポンス
 * @return dav_error 問題なければNULL, 問題があればエラーを返す(409)
 */
static dav_error * _validate_move_group(request_rec *r,
					const dav_resource *src,
					const dav_resource *dst,
					dav_response **response)
{
	dav_error *err = NULL;
	apr_pool_t *p             = r->pool;
	dav_divy_dir_conf *dconf  = dav_divy_get_dir_config(r);
	divy_uri_spec *src_u_spec = src->info->rdb_r->u_spec;
	divy_uri_spec *dst_u_spec = dst->info->rdb_r->u_spec;
	divy_rdbo_grp *src_grp_pr = src->info->rdb_r->grp_pr;

	dav_response *new_response = NULL, *first = NULL;
	divy_rdbo_resource *rdb_r  = NULL, *n_rdb_r;
	char *src_grpcol_name;

	/* エンティティの検証 */
	err = divy_validate_move_entity(r, src_u_spec, dst_u_spec);
	if (err) return err;

	/* syncgrpuri がON ならば複数階層のグループ作成を許可しない */
	if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON &&
		(divy_count_dirs(src_u_spec->other_part) > 1 ||
		 divy_count_dirs(dst_u_spec->other_part) > 1)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"At present, you could not move a group "
			"under another group. Please consider change of "
			"TeamFile setting.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}
	/* syncgrpuri が MIDDLE ならば、同名の表示名を持つグループの存在を許可しない */
	else if (dconf->syncgrpuri == DIVY_SYNCGRPURI_MIDDLE) {
		divy_rdbo_grp *grp = NULL;
		if (divy_rdbo_get_group_property_by_name(r, dst_u_spec->final_path_segment, &grp)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get group property by name.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* 同名のグループが存在 && そのグループは src のグループではない -> NG
		   (note) src のグループを除外しないと階層移動のMOVEに失敗する */
		if (grp != NULL && strcmp(src->uri,
				divy_build_m_group_uri(p, dav_divy_get_root_uri(r), grp->relativeuri)) != 0) {
			ERRLOG0(p, APLOG_NOTICE, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"We found a same group at another hierarchy."
				"This setting could not allow to make same group name.");
			return dav_new_error(p, HTTP_PRECONDITION_FAILED, 0, 0, "");
		}
	}

	/* グループコレクションURIのfinal path segment を取得する */
	src_grpcol_name = dav_divy_extract_finalpath_segment(p, src_grp_pr->grpcol_uri);

	/*
	 * syncgrpuri を運用中に変更したと思われるオペレーションをブロックする
	 * (ブロック条件)
	 * 	[ syncgrpuri == ON  ]
	 * 	    グループコレクションがグループ名称をベースに作られていない
	 * 	[ syncgrpuri != ON ]
	 * 	    グループコレクションがグループ名称をベースに作成されており、
	 * 	    それはグループIDをベースに作成したものではない 
	 *
	 * ブロックするのは、グループ名にグループIDと同じ名称を利用してしまう
	 * ことで発生するDB不整合を未然に防止するためです。この不正が起きると
	 * 修正することは可能ですが、全てのグループを手動で修正しなければ
	 * なりません。これは事実上不可能です。
	 */
	if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON &&
			strcmp(src_grpcol_name, src_grp_pr->name) != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"We refushed to change this group uri, because we "
			"could not maintain the consistency of internal data structure."
			"Maybe the value of \"TfSyncGroupUri\" was changed to \"ON\" "
			"by administrator."
			"Please check configuration or repair the data.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	if ((dconf->syncgrpuri != DIVY_SYNCGRPURI_ON &&
	     dconf->syncgrpuri != DIVY_SYNCGRPURI_MIDDLE) &&
			strcmp(src_grpcol_name, src_grp_pr->name) == 0 &&
			strcmp(src_grpcol_name, src_grp_pr->grpid) != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"We refushed to change this group uri, because we "
			"could not maintain the consistency of internal data structure."
			"Maybe the value of \"TfSyncGroupUri\" was changed to \"OFF\" "
			"by administrator."
			"Please check configuration or repair the data.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/*
	 * グループコレクションのURIを変更しなければならないとき、グループ
	 * コレクション以下にロックが無いこと
	 * (URI変更条件)
	 * 	* グループURIとグループ名を同期化する (syncgrpuri == ON/MIDDLE)
	 * 	* リネームの場合 (src_grp_pr->name != dst->info->rdb_r->displayname)
	 * (note)
	 * 	上記の変更条件は、先のブロック条件があることで初めて成立するもの。
	 * 	ブロック条件を変更したら見直す必要があります。
	 */
	if ((dconf->syncgrpuri == DIVY_SYNCGRPURI_ON ||
	     dconf->syncgrpuri == DIVY_SYNCGRPURI_MIDDLE)&&
		strcmp(src_grp_pr->name, dst->info->rdb_r->displayname) != 0) {
		divy_rdbo_resource *null_rdb_r = NULL;

		if (divy_rdbo_get_hierarchy_lockedresource(r,
					src_grp_pr->grpcol_uri, &rdb_r)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get locked resource."
				"(grpcol_uri = %s)", src_grp_pr->grpcol_uri);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/*
		 * LOCK NULLリソースの確認
		 */
		if (divy_rdbo_get_null_resource_member(r, src_grp_pr->grpcol_uri,
					DAV_INFINITY, 1, &null_rdb_r)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get lock-null resource."
				"(grpcol_uri = %s)", src_grp_pr->grpcol_uri);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		if (rdb_r != NULL && null_rdb_r != NULL) {
			for (n_rdb_r = rdb_r; n_rdb_r->next; n_rdb_r = n_rdb_r->next);
			n_rdb_r->next = null_rdb_r;
		}
		else if (rdb_r == NULL && null_rdb_r != NULL) {
			rdb_r = null_rdb_r;
		}

		/* ロックされたリソースまたはLOCK NULLリソースがあったらNG */
		if (rdb_r != NULL) {
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The server could not MOVE group. "
				"Because group collection have some locked resources."
				"(grpcol_uri = %s)", src_grp_pr->grpcol_uri);

			/* Failed Dependency レスポンスの生成 */
			for (n_rdb_r = rdb_r; n_rdb_r; n_rdb_r = n_rdb_r->next) {
				if (first == NULL) {
					first = new_response =
						apr_pcalloc(p, sizeof(dav_response));
				}
				else {
					new_response->next =
						apr_pcalloc(p, sizeof(dav_response));
					new_response = new_response->next;
				}

				if (n_rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
					new_response->href = apr_psprintf(p, "%s/",
							n_rdb_r->uri);
				}
				else {
					new_response->href = n_rdb_r->uri;
				}
				new_response->status = HTTP_LOCKED;
			}
			/* *response に繋げる */
			new_response->next = *response;
			*response = first;

			return dav_new_error(p, HTTP_FAILED_DEPENDENCY, 0, 0, "");
		}
	}

	/*
	 * グループリーダによるグループ操作には一定の制限が加えられている
	 * -> mod_dav のフレームワーク不良から divy_validate_request_resource() にこれを移動しました
	 */

	return NULL;
}

/**
 * rdb_r が示す通常コレクションを作成する。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource * 作成するコレクション情報
 * @return DAV エラー(500)
 */
static dav_error * _create_regular_collection(request_rec *r,
					divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p = r->pool;
	int ret;

	/* 足りない情報を rdb_r にセットする */
	_fill_default_collection_info(r, rdb_r);
	if (rdb_r->rsid == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get rsid.(uri = %s)", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* DB にコレクションのエントリをinsert する */
	ret = divy_rdbo_insert_property(r, rdb_r, NULL);
	if (ret == DIVY_STCODE_PARENT_NOT_EXIST) {
		/* 正しくはconflict だが外部仕様としては
		 * NOT Foundにした方が分かりやすい */
		return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}
	else if (ret) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to create collection.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * docとrdb_r が示すlinkdbsearch 結果フォルダを作成する。
 * (note)
 * 	doc が NULL であればエラーとなります。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource * １つのdbfolderを表すリソース構造体
 * @param doc apr_xml_doc * dbfolder 情報を保持するXMLエレメントオブジェクト
 * @return dav_error 	doc が NULL だった、パースエラーなど。
 */
static dav_error * _create_dbfolder(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc)
{
	dav_error *err                = NULL;
	const apr_xml_elem *prop_elem = NULL;
	apr_pool_t *p                 = r->pool;
	divy_rdbo_keylist *keylist_pr = NULL;
	divy_rdbo_keylist *sorted_keylist = NULL;
	divy_uri_spec *u_spec         = rdb_r->u_spec;
	int ret;

	/* XML エレメントが無かったとき */
	if (doc == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Failed to create linkdb folder. The \"displayname\" "
			"and \"keylist\" element MUST be specified. (uri = %s)",
			rdb_r->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* final path segment の禁則チェック */
	if (divy_validate_pathsegment(p, 0, u_spec->final_path_segment)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The part of uri has invalid char." );
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* propertyupdate タグの解析＆検証 */
	err =_validate_mkcol_propertyupdate(p, DIVY_ALLOW_MULTI_PROP, doc);
	if (err) return err;

	prop_elem = doc->root->first_child->first_child;

	/* displayname の取り出し */
	rdb_r->displayname = (char *) divy_xml_get_cdata(prop_elem->first_child, p, 1);
	if (IS_EMPTY(rdb_r->displayname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"displayname\" property is missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* keylist の取り出し */
	if (divy_parse_keylist_property(p, prop_elem->first_child->next, &keylist_pr)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"propertyupdate\" element is not valid for MKCOL method."
			"(displayname or keylist is missing)");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 取得値の検証 */
	err = _validate_mkcol_dbfolder(p, rdb_r, keylist_pr);
	if (err) return err;

	/* keystlist を並べ替える */
	_sort_mkcol_keylist_elem(p, keylist_pr, &sorted_keylist);

	/* 足りない情報を rdb_r にセットする */
	_fill_default_collection_info(r, rdb_r);
	if (rdb_r->rsid == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get rsid.(uri = %s)", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	rdb_r->keylist_pr = sorted_keylist;

	/*
	 * DB にコレクションのエントリをinsert する
	 */
	ret = divy_rdbo_insert_property(r, rdb_r, NULL);
	if (ret == DIVY_STCODE_PARENT_NOT_EXIST) {
		/* DBフォルダの場合、これが発生するのはDB不正である */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The parent of resource was missing."
			"Maybe it means the data was broken."
			"Please check it.(uri = %s)", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	else if (ret) {
		/* 予期しないエラー発生 */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert %s property.(uri = %s)",
			"linkdb folder", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * docとresource が示す共有コレクションを作成する。
 * (note)
 * 	doc が NULL であればエラーとなります。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource * １つの共有コレクションを表すリソース構造体
 * @param doc apr_xml_doc * 共有コレクション情報を保持するXMLエレメントオブジェクト
 * @return dav_error 	doc が NULL だった、パースエラーなど。
 */
static dav_error * _create_sharedcollection(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc)
{
	dav_error *err                = NULL;
	divy_rdbo_shsrccol *shsrccol  = NULL;
	apr_pool_t *p                 = r->pool;

	/* XML エレメントが無かったとき */
	if (doc == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Failed to create shared collection. The \""
			"sharedcollectioncreate\" element MUST be specified."
			"(uri = %s)", rdb_r->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* sharedcollectioncreate タグの解析＆検証 */
	if ((err = _parse_sharedcollectioncreate_elem(p, doc, &shsrccol)) != NULL)
		return err;

	/* 取得値の検証 */
	if ((err = _validate_mkcol_dbshfolder(r, rdb_r, shsrccol)) != NULL)
		return err;

	/* 足りない情報を rdb_r にセットする */
	_fill_default_collection_info(r, rdb_r);
	if (rdb_r->rsid == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get rsid.(uri = %s)", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* 共有コレクションを作成する */
	if (divy_rdbo_insert_sharedsrccollection_info(r, rdb_r, shsrccol)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert %s property.(uri = %s)",
			"sharedcollection", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * docとresource が示すユーザを作成する。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource * １人のユーザを表すリソース構造体
 * @param usr_pr divy_rdbo_usr * 登録対象のユーザ
 * @return dav_error
 */
static dav_error * _create_user(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc)
{
	dav_error *err                = NULL;
	const apr_xml_elem *prop_elem = NULL;
	divy_rdbo_usr *usr_pr         = NULL;
	apr_pool_t *p                 = r->pool;
	divy_uri_spec *u_spec         = rdb_r->u_spec;
	divy_user_iscreen *iscreen    = NULL;
	int support_groupleader       = divy_support_groupleader(r);
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	int support_access_control    = divy_support_access_control(r);
	
	/* XML エレメントが指定されていたとき */
	if (doc) {
		/* propertyupdate タグの検証 */
		err =_validate_mkcol_propertyupdate(p, DIVY_ALLOW_SINGLE_PROP, doc);
		if (err) return err;

		/* prop エレメントの取り出し */
		prop_elem = doc->root->first_child->first_child;

		/* prop エレメントの中に指定されたタグを解析する */
		if (divy_parse_user_property(p, prop_elem->first_child, &iscreen)) {

			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The \"propertyupdate\" element is not valid for MKCOL method.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}
	/* Body がなければデフォルト値を設定 */
	else {
		iscreen = apr_pcalloc(p, sizeof(divy_user_iscreen));

		/* URI のfinal path segment がuserid になる */
		iscreen->usrid      = (char *) u_spec->final_path_segment;
		iscreen->password   = iscreen->usrid;
		iscreen->fullname   = iscreen->usrid;
		iscreen->adminmode  = DIVY_ADMINMODE_NORMAL;	/* 一般ユーザ */
		iscreen->smaxst     = apr_psprintf(p, "%"APR_INT64_T_FMT, DIVY_QUOTA_MAX_STORAGE);
		iscreen->smaxres    = apr_psprintf(p, "%"APR_INT64_T_FMT, DIVY_QUOTA_MAX_RESOURCE);
		iscreen->accessdeny = NULL;	/* アクセス制限なし */
		if (divy_support_extenduserstatus(r)) {
			iscreen->extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_USR); /* デフォルト */
		}
		else {
			iscreen->extstatus = NULL;
		}
		if (support_groupleader) {
			iscreen->smaxusercreation = "0";	/* デフォルト値 */
		}
		if (support_access_control) {
			iscreen->allowhosts = NULL;			/* デフォルト値 */
		}
	}
	iscreen->u_spec     = u_spec;
	iscreen->src_usr_pr = NULL;	/* 昔の値はない */

	/* 値の検証 */
	err = divy_validate_user_property(r, DIVY_VALIDATE_INSERT, iscreen);
	if (err) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"userdiscovery\" property is wrong. (uri = %s)", rdb_r->uri);
		return err;
	}

	/* iscreen -> rdbo */
	(void) divy_map_user_property(p, iscreen, &usr_pr);

#ifdef DIVY_SUPPORT_PASSPOLICY
	/* ユーザのパスワードポリシー状態を作る */
	usr_pr->passpolicy_status = apr_pcalloc(p, sizeof(divy_rdbo_passpolicystatus));
	usr_pr->passpolicy_status->policyid        = DIVY_DEFAULT_POLICY_ID;
	usr_pr->passpolicy_status->usrid           = usr_pr->usrid;
	usr_pr->passpolicy_status->lastchange      = 0L;	/* 自動的にいれます */
	usr_pr->passpolicy_status->sendexpiredmail = 0L;
	usr_pr->passpolicy_status->firstlogin      = 0L;
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* (2007/06/21 Thu) 新規作成時には、サポートしていなければ「無視しない」が正しい */
	/* グループ制約属性をサポートしていなければ、制約属性の無視フラグは「無視しない」にする */
	if (!divy_support_grpconstraints(r)) {
		divy_rdbo_set_user_groupconstraints_ignore(usr_pr->extstatus, 0);
	}

	/* グループ管理者機能をサポートしている場合 (クライアントの新旧は影響しない) */
	if (support_groupleader) {
		/* 自身がグループリーダであればownerid を入れる */
		if (divy_rdbo_is_groupleader(own_extstatus)) {
			usr_pr->ownerid = apr_pstrdup(p, divy_get_userid(r));
		}
	}
	else {
		usr_pr->maxusercreation = 0;							/* 最大ユーザ作成数をデフォルトにする */
		divy_rdbo_set_groupleader(usr_pr->extstatus, 0);		/* グループリーダではない */
		usr_pr->ownerid = NULL;									/* オーナなし */
		divy_rdbo_set_control_otheruser(usr_pr->extstatus, 0);	/* Otherユーザを管理下におけない */
	}

	/*
	 * 対象ユーザがグループリーダ以外の場合、"管理ユーザ数"や"ユーザを管理下に置くことができる"
	 * と設定されていても無意味なので、デフォルト状態に戻します.
	 */
	if (support_groupleader && !divy_rdbo_is_groupleader(usr_pr->extstatus)) {
		usr_pr->maxusercreation = 0;	/* 無制限 */
		divy_rdbo_set_control_otheruser(usr_pr->extstatus, 0);	/* 制御できない */
	}

	/*
	 * ユーザ情報を記録する
	 */
	if (divy_rdbo_insert_user_property(r, usr_pr, NULL)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to insert %s property.(uri = %s)",
				"user", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * 作成されたユーザに作成通知メールを送信する
	 */
	if (divy_ml_enable_notify_usercreation(r, usr_pr)) {
		/* グループリーダが作成者であれば、リーダの名前を入れておく */
		if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus)) {
			usr_pr->ownername = apr_pstrdup(p, divy_get_fullname(r));
		}
		divy_ml_notify_usercreation(r, usr_pr);
	}

	return NULL;
}

/**
 * docとresource が示すグループを作成する。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource * １つのグループを表すリソース構造体
 * @param doc apr_xml_doc * グループ情報を保持するXMLエレメントオブジェクト
 * @return dav_error
 */
static dav_error * _create_group(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc)
{
	dav_error *err                = NULL;
	const apr_xml_elem *prop_elem = NULL;
	divy_rdbo_grp *grp_pr         = NULL;
	apr_pool_t *p                 = r->pool;
	divy_uri_spec *u_spec         = rdb_r->u_spec;
	divy_group_iscreen *iscreen   = NULL;
	int support_groupleader       = divy_support_groupleader(r);

	/* XML エレメントが指定されていたとき */
	if (doc) {
		/* propertyupdate タグの検証 */
		err =_validate_mkcol_propertyupdate(p, DIVY_ALLOW_SINGLE_PROP, doc);
		if (err) return err;

		/* prop エレメントの取り出し */
		prop_elem = doc->root->first_child->first_child;

		/* パース */
		if (divy_parse_group_property(p, prop_elem->first_child, &iscreen)) {

			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"propertyupdate\" element is not valid for MKCOL method.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* groupmailwatch が設定されていれば値を補完する */
		if (iscreen != NULL && iscreen->mailwatch_pr != NULL) {
			iscreen->mailwatch_pr->uri   = rdb_r->uri;
			iscreen->mailwatch_pr->usrid = (char *) divy_get_userid(r);
		}
	}
	/* Body がないとき */
	else {
		iscreen = apr_pcalloc(p, sizeof(divy_group_iscreen));
		iscreen->name         = (char *) u_spec->final_path_segment;
		iscreen->mailwatch_pr = NULL;	/* groupmailwatch なし */
		iscreen->grp_extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_GRP);
	}
	iscreen->u_spec = u_spec;

	/* 値の検証 */
	err = divy_validate_group_property(r, DIVY_VALIDATE_INSERT, iscreen);
	if (err) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"groupdiscovery\" property is wrong. (uri = %s)",
			rdb_r->uri);
		return err;
	}

	/* iscreen -> rdbo */
	(void) divy_map_group_property(p, iscreen, &grp_pr);

	/* groupdiscovery にはuri がないので入れる */
	grp_pr->relativeuri = (char *) u_spec->other_part;
	rdb_r->grp_pr = grp_pr;	

	/* グループ管理者機能をサポートしている場合 */
	if (support_groupleader) {
		/* グループリーダであればownerid を入れる */
		if (divy_rdbo_is_groupleader(divy_get_extstatus(r))) {
			grp_pr->ownerid = apr_pstrdup(p, divy_get_userid(r));
		}
		/* グループリーダでなければトップグループのオーナIDを継承する */
		else {
			divy_rdbo_grp *top_grp = NULL;

			if (divy_rdbo_get_topgroup_property_by_grpuri(r, grp_pr->relativeuri, &top_grp, NULL)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get top-group information. (uri = %s)",
						grp_pr->relativeuri);
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}

			if (top_grp != NULL) {
				grp_pr->ownerid = top_grp->ownerid;
			}
			else {
				/* トップグループが存在しない = 新規作成top グループ -> ownerid は存在しない */
				grp_pr->ownerid = NULL;
			}
		}
	}
	else {
		grp_pr->ownerid = NULL;	/* オーナなし */
	}

	/* グループ情報を記録する */
	if (divy_rdbo_insert_group_property(r, grp_pr, NULL)) {

		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert %s property.(uri = %s)", "group", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * docとresource が示すSQL を作成する。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource * １つのSQLを表すリソース構造体
 * @param doc apr_xml_doc * SQL情報を保持するXMLエレメントオブジェクト
 * @return dav_error
 */
static dav_error * _create_sql(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc)
{
	dav_error *err                = NULL;
	const apr_xml_elem *prop_elem = NULL;
	divy_rdbo_sql *sql_pr         = NULL;
	apr_pool_t *p                 = r->pool;
	divy_uri_spec *u_spec         = rdb_r->u_spec;
	divy_sql_iscreen iscreen;

	/* XML エレメントが無かったとき */
	if (doc == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Failed to create SQL. The \"propertyupdate\" "
			"element MUST be specified. (uri = %s)", rdb_r->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* propertyupdate タグの検証 */
	err =_validate_mkcol_propertyupdate(p, DIVY_ALLOW_SINGLE_PROP, doc);
	if (err) return err;

	/* prop エレメントの取り出し */
	prop_elem = doc->root->first_child->first_child;

	/* sqldiscovery のパース */
	if (divy_parse_sql_property(p, prop_elem->first_child, &sql_pr)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"propertyupdate\" element is not valid for MKCOL method.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	iscreen.u_spec    = u_spec;
	iscreen.sql_pr    = sql_pr;
	iscreen.db_sql_pr = NULL;	/* 古い値は存在しない */

	/* 値の検証 */
	err = divy_validate_sql_property(r, DIVY_VALIDATE_INSERT, &iscreen);
	if (err) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"sqldiscovery\" property is wrong. (uri = %s)", rdb_r->uri);
		return err;
	}

	/* sqldiscovery にはuri がないので入れる */
	sql_pr->relativeuri = (char *) u_spec->other_part;

	/*
	 * SQL 情報を記録する
	 */
	if (divy_rdbo_insert_sql_property(r, sql_pr)) {
		/* 予期しないエラー */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert %s property.(uri = %s)",
			"sql", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * docとresource が示すシステムメッセージ を作成する。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource * １つのメッセージを表すリソース構造体
 * @param doc apr_xml_doc * メッセージ情報を保持するXMLエレメントオブジェクト
 * @return dav_error
 */
static dav_error * _create_sysmsg(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc)
{
	dav_error *err                = NULL;
	const apr_xml_elem *prop_elem = NULL;
	divy_rdbo_sysmsg *sysmsg_pr   = NULL;
	apr_pool_t *p                 = r->pool;
	divy_uri_spec *u_spec         = rdb_r->u_spec;
	divy_sysmsg_iscreen *iscreen  = NULL;
	int support_extendsysmsg      = divy_support_extendsysmsg(r);

	/* XML エレメントが指定されていたとき */
	if (doc) {
		/* propertyupdate タグの検証 */
		err =_validate_mkcol_propertyupdate(p, DIVY_ALLOW_SINGLE_PROP, doc);
		if (err) return err;

		/* prop エレメントの取り出し */
		prop_elem = doc->root->first_child->first_child;

		/* パース */
		if (divy_parse_sysmsg_property(p, prop_elem->first_child, &iscreen)) {

			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"propertyupdate\" element is not valid for MKCOL method.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}
	/* Body がないとき */
	else {
		iscreen = apr_pcalloc(p, sizeof(divy_sysmsg_iscreen));
		iscreen->msgid  = (char *) u_spec->final_path_segment;
		iscreen->msg    = NULL;
		iscreen->active = 0;
		if (support_extendsysmsg) {
			iscreen->subject = NULL;
			iscreen->lang    = NULL;
			iscreen->scope   = NULL;
		}
	}

	/* 値の検証 */
	iscreen->u_spec = u_spec;

	err = divy_validate_sysmsg_property(r, DIVY_VALIDATE_INSERT, iscreen);
	if (err) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"sysmsgdiscovery\" property is wrong. (uri = %s)", rdb_r->uri);
		return err;
	}

	/* iscreen -> rdbo */
	(void) divy_map_sysmsg_property(p, iscreen, &sysmsg_pr);

	/* システムメッセージ情報を記録する */
	if (divy_rdbo_insert_sysmsg_property(r, sysmsg_pr)) {

		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert %s property.(uri = %s)", "sysmsg", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * docとresource が示すDBMS メッセージ を作成する。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource * １つのDBMSを表すリソース構造体
 * @param doc apr_xml_doc * DBMS情報を保持するXMLエレメントオブジェクト
 * @return dav_error *
 */
static dav_error * _create_dbms(request_rec *r,
					divy_rdbo_resource *rdb_r,
					apr_xml_doc *doc)
{
	dav_error *err                = NULL;
	const apr_xml_elem *prop_elem = NULL;
	divy_rdbo_dbms *dbms_pr       = NULL;
	apr_pool_t *p                 = r->pool;
	divy_dbms_iscreen *iscreen    = NULL;

	/* XML エレメントが無かったとき */
	if (doc == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Failed to create DBMS entry. The \"propertyupdate\" "
			"element MUST be specified. (uri = %s)", rdb_r->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* propertyupdate タグの検証 */
	err =_validate_mkcol_propertyupdate(p, DIVY_ALLOW_SINGLE_PROP, doc);
	if (err) return err;

	/* prop エレメントの取り出し */
	prop_elem = doc->root->first_child->first_child;

	/* prop エレメントの中に指定されたタグを解析する */
	if (divy_parse_dbms_property(p, prop_elem->first_child, &iscreen)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"propertyupdate\" element is not valid for MKCOL method.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	iscreen->u_spec = rdb_r->u_spec;

	/* 値の検証 */
	err = divy_validate_dbms_property(r, DIVY_VALIDATE_INSERT, iscreen);
	if (err) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"dbmsdiscovery\" property is wrong. (uri = %s)", rdb_r->uri);
		return err;
	}
	/* iscreen -> rdbo */
	(void) divy_map_dbms_property(p, iscreen, &dbms_pr);
	rdb_r->dbms_pr = dbms_pr;

	/* DBMS 情報を記録する */
	if (divy_rdbo_insert_dbms_property(r, dbms_pr, NULL)) {

		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert %s property.(uri = %s)", "dbms", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * ユーザリレーション src_rdb_r の情報を使って、新しいユーザリレーション
 * dst_rdb_r を作成する。
 *
 * @param r request_rec *
 * @param src const dav_resource *
 * @param dst dav_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _create_ruser(request_rec *r,
					const dav_resource *src,
					dav_resource *dst)
{
	dav_error *err;
	apr_pool_t *p = r->pool;
	divy_rdbo_rusr *rusr_pr = NULL;

	/* 作成リソースの検証 */
	err = _validate_create_relation(r, src, dst);
	if (err) return err;

	/*
	 * プロパティ rusr_pr を生成
	 */
	/* Desitination uri のパース */
	if (divy_rdbo_parse_rusr_uri(r, dst->uri, &rusr_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse destionation uri of "
			"user-relation resource. (uri = %s)", dst->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * リレーションを作成する
	 */
	if (divy_rdbo_insert_rusr(r, rusr_pr, NULL)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy user-relation resource."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * グループリレーション src_rdb_r の情報を使って、新しいグループリレーション
 * dst_rdb_r を作成する。
 *
 * @param r request_rec *
 * @param src const dav_resource *
 * @param dst dav_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _create_rgrp(request_rec *r,
					const dav_resource *src,
					dav_resource *dst)
{
	dav_error *err;
	apr_pool_t *p = r->pool;
	divy_rdbo_rgrp *rgrp_pr = NULL;

	/* 作成リソースの検証 */
	err = _validate_create_relation(r, src, dst);
	if (err) return err;

	/*
	 * プロパティ rgrp_pr を生成
	 */
	/* Desitination uri のパース */
	if (divy_rdbo_parse_rgrp_uri(r, dst->uri, &rgrp_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse destionation uri of "
			"group-relation resource. (uri = %s)", dst->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * rgrp_pr のリレーションを作成する
	 */
	if (divy_rdbo_insert_rgrp(r, rgrp_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy group-relation resource."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}
/**
 * SQLリレーション src_rdb_r の情報を使って、新しいSQLリレーション
 * dst_rdb_r を作成する。
 *
 * @param r request_rec *
 * @param src const dav_resource *
 * @param dst dav_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _create_rsql(request_rec *r,
					const dav_resource *src,
					dav_resource *dst)
{
	dav_error *err;
	apr_pool_t *p = r->pool;
	divy_rdbo_rsql *rsql_pr = NULL;

	/* 作成リソースの検証 */
	err = _validate_create_relation(r, src, dst);
	if (err) return err;

	/*
	 * プロパティ rsql_pr を生成
	 */
	/* Desitination uri のパース */
	if (divy_rdbo_parse_rsql_uri(r, dst->uri, &rsql_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse destionation uri of "
			"sql-relation resource. (uri = %s)", dst->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * リレーションを作成する
	 */
	if (divy_rdbo_insert_rsql(r, rsql_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy sql-relation resource."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * src_rdb_r が示すリソースまたはコレクションをdst_rdb_r が示す
 * リソースまたはコレクションとしてコピーする。
 * この関数は、通常のコレクション・リソースに対して使用できます。
 * 特殊コレクションのコピーには使用できません。
 *
 * @param r request_rec *
 * @param src_rdb_r divy_rdbo_resource * ソース
 * @param dst_rdb_r divy_rdbo_resource * ディスティネーション
 * @return dav_error
 */
static dav_error * _copy_regular_resource(request_rec *r,
					int depth,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r)
{
	dav_error *err = NULL;
	apr_status_t rv;
	divy_db_transaction_ctx *ts_ctx = NULL;
	apr_pool_t *p            = r->pool;
	const char *src_path, *dst_path;
	divy_copyfile_info *copyinfo = NULL, *cpinfo;
	int ret, i;

	/* 
	 * 全ての物理ファイルがコピーされるまでDB の更新は待つ必要がある。
	 * そのため、トランザクションコンテキストを作成して、
	 * トランザクションの確定を保留出来るようにする。
	 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to create transaction context for copy resource.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* dst_rdb_r に必要な値を設定する */
	dst_rdb_r->resourcetype   = src_rdb_r->resourcetype;
	dst_rdb_r->creator        = NULL;
	dst_rdb_r->lastmodifier   = NULL;
	dst_rdb_r->creator_userid = apr_pstrdup(p, divy_get_userid(r));
	dst_rdb_r->isversioned    = 0;
	dst_rdb_r->checkin        = -1;
	dst_rdb_r->checkout       = -1;
	dst_rdb_r->lastmodifier_userid = dst_rdb_r->creator_userid;

	/* DB エントリのコピー */
	if (depth == 0 || src_rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
		ret = divy_rdbo_copy_resource_without_member(r,
				src_rdb_r, dst_rdb_r, &copyinfo, ts_ctx);
	}
	else {
		ret = divy_rdbo_copy_resource(r,
				src_rdb_r, dst_rdb_r, &copyinfo, ts_ctx);
	}

	/* src が実は存在していなかった場合(オペレーション競合が原因) */
	if (ret == DIVY_STCODE_NOT_EXIST) {
		/* トランザクションのロールバック */
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		/* ステータスはmod_dav のcopymove にあわせます */
		return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}
	/* Quotaオーバー */
	else if (ret == DIVY_STCODE_QUOTAOVER) {
		/* トランザクションのロールバック */
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"There is not enough storage to write to "
			"this resource for quota limit.");
		return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
	}
	/* 予期しないエラー */
	else if (ret) {
		/* トランザクションのロールバック */
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy db entries of resource.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* 物理ファイルのコピー */
	for (cpinfo = copyinfo, i = 0; cpinfo; cpinfo = cpinfo->next,i++) {
		/* コレクションならば物理ファイルのコピーは不要 */
		if (!cpinfo->isfile) continue;

		/* サムネイルファイルのコピー */
		ret = divy_thumbnail_copy(p, r, divy_pfile_get_relativepath(cpinfo->src_pfile),
									divy_pfile_get_relativepath(cpinfo->dst_pfile));
		/* エラーが出ていて、サムネイル機能がサポートされていたらエラー扱い */
		if (ret != DIVY_THMNL_ST_OK && ret != DIVY_THMNL_ST_NOTSUPPORTED) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to copy thumbnail.");
			err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			goto cleanup_copyfile;
		}

		/* 物理パスの算出 */
		src_path = divy_pfile_get_fullpath(cpinfo->src_pfile);
		dst_path = divy_pfile_get_fullpath(cpinfo->dst_pfile);

		/* ファイルのコピー */
		rv = apr_file_copy(src_path, dst_path, APR_FILE_SOURCE_PERMS, p);
		
		if (rv != APR_SUCCESS) {
			/* 書き込みスペースが足りなかった場合 */
			if (APR_STATUS_IS_ENOSPC(rv)) {
				ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to copy resource, because no space "
					"left on physical disk. we rollback "
					"this operation. (src = %s, dst = %s, code = %d)",
					src_path, dst_path, rv);
				err = dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
			}
			/* その他のエラー */
			else {
				ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to copy resource. so we rollback "
					"this operation. (src = %s, dst = %s, code = %d)",
					src_path, dst_path, rv);
				err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}

cleanup_copyfile:

			/* 今までコピーしたファイルを削除する */ 
			for (cpinfo = copyinfo; cpinfo && i >= 0;
						cpinfo = cpinfo->next, i--) {
				/* コレクションならば何もしない */
				if (!cpinfo->isfile) continue;

				/* サムネイルファイルの削除 */
				(void) divy_thumbnail_remove(p, r, divy_pfile_get_relativepath(cpinfo->dst_pfile));

				/* パスの作成 */
				dst_path = divy_pfile_get_fullpath(cpinfo->dst_pfile);
				/* 削除 */
				rv = apr_file_remove(dst_path, p);
				if (APR_STATUS_IS_ENOENT(rv)) {
					/* 先のCOPYに失敗したもの。無視 */
				}
				else if (rv != APR_SUCCESS) {
					ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_FS,
						"Failed to remove copy-file. "
						"This file is a garbage."
						"(path = %s, code = %d)", dst_path, rv);
					/* でも続ける */
				}
			}

			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);

			return err;
		}
#ifdef DIVY_SUPPORT_PREFLIGHT_PLUGIN
		_send_preflightfolder(r, dst_rdb_r->u_spec, dst_path, cpinfo->dispname);
#endif	/* DIVY_SUPPORT_PREFLIGHT_PLUGIN */
	}

	/* 反映を確定する */
	divy_db_commit_transaction(ts_ctx);
		
	/* サーバ主体のメール送信を実施するのか ? */
	if (divy_ml_enable_svrsendmail(r, src_rdb_r) ||
	    divy_ml_enable_svrsendmail(r, dst_rdb_r)) {
		/* サーバ主体のメール送信処理を行う */
		divy_ml_send_servermail(r, src_rdb_r, dst_rdb_r);
	}

	return NULL;
}

/**
 * ユーザを表すプロパティ src_rdb_r をユーザdst_rdb_r としてコピーする。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 *
 * @param r request_rec *
 * @param src_rdb_r divy_rdbo_resource *
 * @param dst_rdb_r divy_rdbo_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _copy_user(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r)
{
	dav_error *err = NULL;
	apr_pool_t *p             = r->pool;
	divy_user_iscreen iscreen = { 0 };
	divy_rdbo_usr *src_usr_pr = src_rdb_r->usr_pr;
	divy_rdbo_usr *dst_usr_pr = NULL;
	int support_groupleader   = divy_support_groupleader(r);
	int support_access_control = divy_support_access_control(r);

	/* dst_usr に src のユーザから情報を取得してiscreen に値をつめる.
	 * (note)
	 * 	srd_usr_pr から代入された幾つかの値は、直接は使われない。SQL文でコピーしてしまうので.
	 * 	単にvalidator を通す目的で代入しているだけです. */
	iscreen.usrid       = (char *) dst_rdb_r->u_spec->final_path_segment;
	iscreen.password    = src_usr_pr->password;
	iscreen.fullname    = src_usr_pr->fullname;
	iscreen.mailaddr    = src_usr_pr->mailaddr;
	iscreen.adminmode   = src_usr_pr->adminmode;
	iscreen.smaxst      = apr_psprintf(p, "%"APR_INT64_T_FMT, src_usr_pr->maxst);
	iscreen.smaxres     = apr_psprintf(p, "%"APR_INT64_T_FMT, src_usr_pr->maxres);
	iscreen.accessdeny  = src_usr_pr->accessdeny;
	iscreen.comment     = src_usr_pr->comment;
	divy_format_time_t(p, src_usr_pr->expiration, DIVY_TIME_STYLE_ISO8601, &iscreen.sexpiration);
	iscreen.extstatus   = src_usr_pr->extstatus;
	iscreen.u_spec      = dst_rdb_r->u_spec;
	iscreen.is_old_client    = 0;	/* 古くはない */
	if (support_groupleader) {
		iscreen.smaxusercreation = apr_psprintf(p, "%d", src_usr_pr->maxusercreation);
	}
	else {
		iscreen.smaxusercreation = "0";	/* 初期値 */

		if (divy_rdbo_is_groupleader(iscreen.extstatus)) {
			/* ユーザの種類は一般に格下げ */
			divy_rdbo_set_groupleader(iscreen.extstatus, 0);

			/* Otherユーザをコントロールできない */
			divy_rdbo_set_control_otheruser(iscreen.extstatus, 0);
		}
	}
	/* (note) 2007/10/10 Tue
	 * 	COPY 時にグループリーダはOtherユーザをコピーできないいことを
	 *  チェックしなければならないため、src_ur_pr 情報が必要になりました */
	iscreen.src_usr_pr  = src_usr_pr;

	if (support_access_control) {
		iscreen.allowhosts = src_usr_pr->allowhosts;
	}

	/* userdiscovery の検証 */
	err = divy_validate_user_property(r, (DIVY_VALIDATE_INSERT | DIVY_VALIDATE_COPY), &iscreen);
	if (err) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"userdiscovery\" property is wrong. (uri = %s)", dst_rdb_r->uri);
		return err;
	}

	/* iscreen -> rdbo */
	(void) divy_map_user_property(p, &iscreen, &dst_usr_pr);

#ifdef DIVY_SUPPORT_PASSPOLICY
	/* ユーザのパスワードポリシー状態を作る */
	dst_usr_pr->passpolicy_status = apr_pcalloc(p, sizeof(divy_rdbo_passpolicystatus));
	dst_usr_pr->passpolicy_status->policyid        = DIVY_DEFAULT_POLICY_ID;
	dst_usr_pr->passpolicy_status->usrid           = dst_usr_pr->usrid;
	dst_usr_pr->passpolicy_status->lastchange      = 0L;	/* 自動的にいれます */
	dst_usr_pr->passpolicy_status->sendexpiredmail = 0L;
	dst_usr_pr->passpolicy_status->firstlogin      = 0L;
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* (2007/06/21 Thu) off かつコピーの場合には、例えsrc が制約属性を持っていても
	 * それはコピーしない. さもなければ、on にしたときに、dst のユーザ権限が
	 * 突然昇格してしまいます. 運用上、コントロール出来なくなるでしょう. */
	/* グループ制約属性をサポートしていなければ、制約属性の無視フラグは「無視しない」にする */
	if (!divy_support_grpconstraints(r)) {
		divy_rdbo_set_user_groupconstraints_ignore(dst_usr_pr->extstatus, 0);
	}

	/* グループ管理者機能をサポートしている場合 (クライアントの新旧は影響しない) */
	if (support_groupleader) {
		/* 自身がグループリーダであればownerid を入れる */
		if (divy_rdbo_is_groupleader(divy_get_extstatus(r))) {
			dst_usr_pr->ownerid = apr_pstrdup(p, divy_get_userid(r));
		}
	}
	else {
		dst_usr_pr->maxusercreation = 0;							/* 最大ユーザ作成数をデフォルトにする */
		divy_rdbo_set_groupleader(dst_usr_pr->extstatus, 0);		/* グループリーダではない */
		dst_usr_pr->ownerid = NULL;									/* オーナなし */
		divy_rdbo_set_control_otheruser(dst_usr_pr->extstatus, 0);	/* Otherユーザを管理下におけない */
	}
	/*
	 * src_rdb_r のレコードをdst_rdb_r にコピーする
	 * (note)
	 * 	dst_usr_pr に代入された多くは無視されてSQL文でコピーされてしまいます.
	 */
	if (divy_rdbo_copy_user_property(r, src_usr_pr, dst_usr_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy user resource.(src = %s, dst = %s)",
			src_rdb_r->uri, dst_rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * グループを表すプロパティ src_rdb_r をグループdst_rdb_r としてコピーする。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 *
 * @param r request_rec *
 * @param depth int コピー時のDepth (0, DAV_INFINITY)
 * @param src_rdb_r divy_rdbo_resource *
 * @param dst_rdb_r divy_rdbo_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _copy_group(request_rec *r,
					int depth,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r)
{
	apr_pool_t *p = r->pool;

	/*
	 * プロパティ dst_rdb_r->grp_pr を生成
	 */
	dst_rdb_r->grp_pr = apr_pcalloc(p, sizeof(divy_rdbo_grp));
	dst_rdb_r->grp_pr->name = (char *) dst_rdb_r->u_spec->final_path_segment;
	dst_rdb_r->grp_pr->relativeuri = (char *) dst_rdb_r->u_spec->other_part; 

	/*
	 * src_rdb_r のレコードをdst_rdb_r にコピーする
	 */
	if (divy_rdbo_copy_group_property(r, depth, src_rdb_r->grp_pr,
							dst_rdb_r->grp_pr)) { 
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy group resource."
			"(src = %s, dst = %s, depth = %d)",
			src_rdb_r->uri, dst_rdb_r->uri, depth);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * システムメッセージを表すプロパティ src_rdb_r をシステムメッセージdst_rdb_r 
 * としてコピーする。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 *
 * @param r request_rec *
 * @param src const dav_resource *
 * @param dst dav_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _copy_sysmsg(request_rec *r,
					const dav_resource *src,
					dav_resource *dst)
{
	dav_error *err;
	apr_pool_t *p = r->pool;
	divy_rdbo_sysmsg *src_sysmsg_pr = src->info->list.sysmsg_pr;
	divy_rdbo_sysmsg dst_sysmsg_pr = { 0 };

	/* コピーリソースuri の検証 */
	err = divy_validate_copy_entity(r, src->info->rdb_r->u_spec,
						dst->info->rdb_r->u_spec);
	if (err) return err;

	/* プロパティ sysmsg_pr を生成 */
	dst_sysmsg_pr.msgid = (char *) dst->info->rdb_r->u_spec->final_path_segment;
	
	/*
	 * src_rdb_r のレコードをdst_rdb_r にコピーする
	 */
	if (divy_rdbo_copy_sysmsg_property(r, src_sysmsg_pr, &dst_sysmsg_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy sysmsg resource."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * DBMS を表すプロパティ src_rdb_r をdst_rdb_r としてコピーする。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 *
 * @param r request_rec *
 * @param src_rdb_r divy_rdbo_resource *
 * @param dst_rdb_r divy_rdbo_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _copy_dbms(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_dbms *dbms_pr;

	/*
	 * プロパティ dst_rdb_r->dbms_pr を生成
	 */
	dbms_pr = apr_pcalloc(p, sizeof(divy_rdbo_dbms));
	dbms_pr->name = (char *) dst_rdb_r->u_spec->final_path_segment;  

	/*
	 * src_rdb_r のレコードをdst_rdb_r にコピーする
	 */
	if (divy_rdbo_copy_dbms_property(r, src_rdb_r->dbms_pr, dbms_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy dbms resource."
			"(src = %s, dst = %s)",
			src_rdb_r->uri, dst_rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * ユーザリレーションを表すプロパティ src_rdb_r をユーザリレーション
 * dst_rdb_r としてコピーする。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 *
 * @param r request_rec *
 * @param src const dav_resource *
 * @param dst dav_resource *
 * @return dav_error * DAVエラー(400, 500). 成功したらNULL.
 */
static dav_error * _copy_ruser(request_rec *r,
					const dav_resource *src,
					dav_resource *dst)
{
	dav_error *err;
	apr_pool_t *p = r->pool;
	divy_rdbo_rusr *rusr_pr = NULL;

	/* コピーリソースuri の検証 */
	err = divy_validate_copy_relation(r, src->info->rdb_r->u_spec,
						dst->info->rdb_r->u_spec);
	if (err) return err;

	/*
	 * プロパティ rusr_pr を生成
	 */
	/* Desitination uri のパース */
	if (divy_rdbo_parse_rusr_uri(r, dst->uri, &rusr_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse destionation uri of "
			"user-relation resource. (uri = %s)", dst->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * dst_rdb_r のリレーションを作成する
	 * (note) COPY とは言いつつも、リレーションの場合には、
	 * 	src から受け継ぐものは殆ど無いので、新規作成している.
	 */
	if (divy_rdbo_insert_rusr(r, rusr_pr, NULL)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy user-relation resource."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * SQLリレーションを表すプロパティ src_rdb_r をSQLリレーション
 * dst_rdb_r としてコピーする。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 *
 * @param r request_rec *
 * @param src const dav_resource *
 * @param dst dav_resource *
 * @param response dav_response ** レスポンス
 * @return dav_error * DAVエラー(400, 500). 成功したらNULL.
 */
static dav_error * _copy_rsql(request_rec *r,
					const dav_resource *src,
					dav_resource *dst,
					dav_response **response)
{
	dav_error *err;
	apr_pool_t *p = r->pool;
	divy_rdbo_rsql *rsql_pr = NULL;

	/* コピーリソースuri の検証 */
	err = divy_validate_copy_relation(r, src->info->rdb_r->u_spec,
						dst->info->rdb_r->u_spec);
	if (err) return err;

	/* SQL使用権の検証 */
	err = _validate_inheritsql(r, src, response);
	if (err) return err;

	/*
	 * プロパティ dst_rdb_r->rsql_pr を生成
	 */
	/* Desitination uri のパース */
	if (divy_rdbo_parse_rsql_uri(r, dst->uri, &rsql_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse destionation uri of "
			"sql-relation resource. (uri = %s)", dst->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 *  dst のリレーションを作成する
	 * (note) COPY とは言いつつも、リレーションの場合には、
	 * 	src から受け継ぐものは殆ど無いので、新規作成している.
	 */
	if (divy_rdbo_insert_rsql(r, rsql_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy sql-relation resource."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * グループリレーションを表すプロパティ src_rdb_r をグループリレーション
 * dst_rdb_r としてコピーする。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 *
 * @param r request_rec *
 * @param src const dav_resource *
 * @param dst dav_resource *
 * @param response dav_response ** レスポンス
 * @return dav_error * DAVエラー(400, 500). 成功したらNULL.
 */
static dav_error * _copy_rgrp(request_rec *r,
					const dav_resource *src,
					dav_resource *dst,
					dav_response **response)
{
	dav_error *err;
	apr_pool_t *p = r->pool;
	divy_rdbo_rgrp *rgrp_pr = NULL;

	/* コピーリソースuri の検証 */
	err = divy_validate_copy_relation(r, src->info->rdb_r->u_spec,
						dst->info->rdb_r->u_spec);
	if (err) return err;

	/* SQL使用権の検証 */
	if (src->info->rdb_r->u_spec->infotype == DIVY_INFOTYPE_m_sql_rgrp) {
		err = _validate_inheritsql(r, src, response);
		if (err) return err;
	}

	/*
	 * プロパティ rgrp_pr を生成
	 */
	/* Desitination uri のパース */
	if (divy_rdbo_parse_rgrp_uri(r, dst->uri, &rgrp_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse destionation uri of "
			"group-relation resource. (uri = %s)", dst->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (divy_rdbo_insert_rgrp(r, rgrp_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy group-relation resource."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}


/**
 * 通常リソース src_rdb_r を通常リソースdst_rdb_r として移動する。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 *
 * @param r request_rec *
 * @param src_rdb_r divy_rdbo_resource *
 * @param dst_rdb_r divy_rdbo_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _move_regular_resource(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r)
{
	apr_pool_t *p = r->pool;
	int ret;

	/* dst_rdb_r 更新に必要な値を設定する */
	dst_rdb_r->resourcetype = src_rdb_r->resourcetype;

	/*
	 * src_rdb_r のレコードをdst_rdb_r に移動する
	 * (DB レコードの変更)
	 */
	ret = divy_rdbo_move_resource(r, src_rdb_r, dst_rdb_r, NULL);
	if (ret == DIVY_STCODE_NOT_EXIST) {
		/* src_rdb_r が存在しなかった場合には
		 * not found とする(mod_davにあわせる) */
		return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}
	else if (ret != 0) {

		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to move regular resource."
			"(src = %s, dst = %s)",
			src_rdb_r->uri, dst_rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* サーバ主体のメール送信を実施するのか ? */
	if (divy_ml_enable_svrsendmail(r, src_rdb_r) ||
	    divy_ml_enable_svrsendmail(r, dst_rdb_r)) {
		/* サーバ主体のメール送信処理を行う */
		divy_ml_send_servermail(r, src_rdb_r, dst_rdb_r);
	}

	return NULL;
}

/**
 * ユーザを表すプロパティ src_rdb_r をユーザdst_rdb_r として移動する。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 * (note)
 * 	ユーザは階層を持たないので、ユーザの移動とは、ユーザIDの
 * 	リネームのことになります。
 *
 * @param r request_rec *
 * @param src_rdb_r divy_rdbo_resource *
 * @param dst_rdb_r divy_rdbo_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _move_user(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r)
{
	apr_pool_t *p = r->pool;

	/*
	 * 移動ユーザのプロパティ dst_rdb_r->usr_pr を生成
	 */
	dst_rdb_r->usr_pr = apr_pcalloc(p, sizeof(divy_rdbo_usr));
	dst_rdb_r->usr_pr->usrid = (char *) dst_rdb_r->u_spec->final_path_segment;

	/*
	 * src_rdb_r のレコードをdst_rdb_r に移動する
	 * (DB レコードの変更)
	 */
	if (divy_rdbo_move_user_property(r,
				src_rdb_r->usr_pr, dst_rdb_r->usr_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to move(rename) user resource."
			"(src = %s, dst = %s)",
			src_rdb_r->uri, dst_rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * グループを表すプロパティ src_rdb_r をグループdst_rdb_r として移動する。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 *
 * @param r request_rec *
 * @param src_rdb_r divy_rdbo_resource *
 * @param dst_rdb_r divy_rdbo_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _move_group(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r)
{
	apr_pool_t *p = r->pool;

	/*
	 * プロパティ dst_rdb_r->grp_pr を生成
	 */
	dst_rdb_r->grp_pr = apr_pcalloc(p, sizeof(divy_rdbo_grp));
	dst_rdb_r->grp_pr->name = (char *) dst_rdb_r->u_spec->final_path_segment;
	dst_rdb_r->grp_pr->relativeuri = (char *) dst_rdb_r->u_spec->other_part; 

	/*
	 * src_rdb_r のレコードをdst_rdb_r に移動する
	 * (DB レコードの変更)
	 */
	if (divy_rdbo_move_group_property(r,
				src_rdb_r->grp_pr, dst_rdb_r->grp_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to move group resource."
			"(src = %s, dst = %s)",
			src_rdb_r->uri, dst_rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * SQLを表すプロパティ src_rdb_r をSQL dst_rdb_r として移動する。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 * (note)
 * 	SQLは階層を持たないので、SQLの移動とは、表示名のリネームの
 * 	ことになります。
 *
 * @param r request_rec *
 * @param src_rdb_r divy_rdbo_resource *
 * @param dst_rdb_r divy_rdbo_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _move_sql(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r)
{
	apr_pool_t *p = r->pool;

	/*
	 * dst_rdb_r->sql_pr を生成
	 */
	dst_rdb_r->sql_pr = apr_pcalloc(p, sizeof(divy_rdbo_sql));
	dst_rdb_r->sql_pr->labelname   = (char *) dst_rdb_r->u_spec->final_path_segment;
	dst_rdb_r->sql_pr->relativeuri = (char *) dst_rdb_r->u_spec->other_part; 

	/*
	 * src_rdb_r のレコードをdst_rdb_r に移動する
	 * (DB レコードの変更)
	 */
	if (divy_rdbo_move_sql_property(r,
				src_rdb_r->sql_pr, dst_rdb_r->sql_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to move(rename) sql resource."
			"(src = %s, dst = %s)",
			src_rdb_r->uri, dst_rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * DBMSを表すプロパティ src_rdb_r をSQL dst_rdb_r として移動する。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 * (note)
 * 	DBMSは階層を持たないので、DBMSの移動とは、DBMS識別名称のリネームの
 * 	ことになります。
 *
 * @param r request_rec *
 * @param src_rdb_r divy_rdbo_resource *
 * @param dst_rdb_r divy_rdbo_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _move_dbms(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dst_rdb_r)
{
	apr_pool_t *p = r->pool;

	/*
	 * dst_rdb_r->dbms_pr を生成
	 */
	dst_rdb_r->dbms_pr = apr_pcalloc(p, sizeof(divy_rdbo_dbms));
	dst_rdb_r->dbms_pr->name   = (char *) dst_rdb_r->u_spec->final_path_segment;

	/*
	 * src_rdb_r のレコードをdst_rdb_r に移動する
	 * (DB レコードの変更)
	 */
	if (divy_rdbo_move_dbms_property(r,
				src_rdb_r->dbms_pr, dst_rdb_r->dbms_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to move(rename) dbms resource."
			"(src = %s, dst = %s)",
			src_rdb_r->uri, dst_rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * ユーザリレーションを表すプロパティ src_rdb_r をユーザリレーションdst_rdb_r
 * として移動する。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 * (note)
 * 	ユーザリレーションの移動とは、あるグループから別のグループに
 * 	ユーザエンティティを所属させ直すという動作と同値になります。
 * 	ユーザリレーション自身は仮想的なコレクションですので、
 * 	名称を変更することは出来ません。
 *
 * @param r request_rec *
 * @param src_rdb_r divy_rdbo_resource *
 * @param dst_rdb_r divy_rdbo_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _move_ruser(request_rec *r,
					const dav_resource *src,
					dav_resource *dst)
{
	dav_error *err;
	apr_pool_t *p = r->pool;
	divy_rdbo_rusr *rusr_pr = NULL;

	/* 移動リソースuri の検証 */
	err = divy_validate_move_relation(r, src->info->rdb_r->u_spec,
						dst->info->rdb_r->u_spec);
	if (err) return err;

	/*
	 * プロパティ rusr_pr を生成
	 */
	/* Desitination uri のパース */
	if (divy_rdbo_parse_rusr_uri(r, dst->uri, &rusr_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to parse destionation uri of "
				"user-relation resource. (uri = %s)", dst->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 任命済みグループリーダは移動できない -> divy_validate_request_resource() */

	/*
	 * src のレコードをdst に移動する
	 * (DB レコードの変更)
	 */
	if (divy_rdbo_move_rusr(r, src->info->list.rusr_pr, rusr_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to move user-relation resource."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * SQLリレーションを表すプロパティ src_rdb_r をSQLリレーションdst_rdb_r
 * として移動する。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 * (note)
 * 	SQLリレーションの移動とは、あるグループから別のグループに
 * 	SQLエンティティの所有権を譲渡するという動作と同値になります。
 * 	SQLリレーション自身は仮想的なコレクションですので、
 * 	名称を変更することは出来ません。
 *
 * @param r request_rec *
 * @param src_rdb_r divy_rdbo_resource *
 * @param dst_rdb_r divy_rdbo_resource *
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _move_rsql(request_rec *r,
					const dav_resource *src,
					dav_resource *dst,
					dav_response **response)
{
	dav_error *err;
	apr_pool_t *p = r->pool;
	divy_rdbo_rsql *rsql_pr = NULL;

	/* 移動リソースuri の検証 */
	err = divy_validate_move_relation(r, src->info->rdb_r->u_spec,
						dst->info->rdb_r->u_spec);
	if (err) return err;

	/* SQL使用権の検証 */
	err = _validate_inheritsql(r, src, response);
	if (err) return err;

	/*
	 * プロパティ rsql_pr を生成
	 */
	/* Desitination uri のパース */
	if (divy_rdbo_parse_rsql_uri(r, dst->uri, &rsql_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse destionation uri of "
			"sql-relation resource. (uri = %s)", dst->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 移動処理 */
	if (divy_rdbo_move_rsql(r, src->info->list.rsql_pr, rsql_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to move sql-relation resource."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * グループリレーションを表すプロパティ src_rdb_r をグループリレーション
 * dst_rdb_r として移動する。
 * (note)
 * 	値の検証は、この関数呼び出しの前に終わっているものとする。
 * (note)
 * 	グループリレーションの移動とは、あるSQLから別のSQLに、または
 * 	あるユーザから別のユーザにSQLまたはユーザエンティティの譲渡する
 * 	という動作と同値になります。
 * 	グループリレーション自身は仮想的なコレクションですので、
 * 	名称を変更することは出来ません。
 *
 * @param r request_rec *
 * @param src const dav_resource *
 * @param dst dav_resource *
 * @param response dav_response ** レスポンス
 * @return dav_error * DAVエラー(500). 成功したらNULL.
 */
static dav_error * _move_rgrp(request_rec *r,
					const dav_resource *src,
					dav_resource *dst,
					dav_response **response)
{
	dav_error *err;
	apr_pool_t *p = r->pool;
	divy_rdbo_rgrp *rgrp_pr = NULL;

	/* 移動リソースuri の検証 */
	err = divy_validate_move_relation(r, src->info->rdb_r->u_spec,
						dst->info->rdb_r->u_spec);
	if (err) return err;

	/* SQL使用権の検証 */
	if (src->info->rdb_r->u_spec->infotype == DIVY_INFOTYPE_m_sql_rgrp) {
		err = _validate_inheritsql(r, src, response);
		if (err) return err;
	}

	/* 任命済みグループリーダは移動できない -> divy_validate_request_resource() */

	/*
	 * プロパティ rgrp_pr を生成
	 */
	/* Desitination uri のパース */
	if (divy_rdbo_parse_rgrp_uri(r, dst->uri, &rgrp_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse destionation uri of "
			"grp-relation resource. (dst = %s)", dst->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");

	}

	/* 移動処理 */
	if (divy_rdbo_move_rgrp(r, src->info->list.rgrp_pr, rgrp_pr)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to move group-relation resource."
			"(src = %s, dst = %s)", src->uri, dst->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * 指定されたrdb_r のDBエントリと物理ファイルを削除する。
 * (note)
 * この関数は、通常リソースまたはコレクションの削除のみに使用できます。
 * 特殊コレクションに対しては使用できません。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource *
 * @return dav_error
 */
static dav_error * _remove_regular_resource(request_rec *r,
					divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p = r->pool;
	divy_linkedlist_t *del_filelist = NULL;
	int ret;

	/*
	 * DB からの削除
	 */
	ret = divy_rdbo_remove_resource(r, rdb_r, &del_filelist, NULL);
	if (ret == DIVY_STCODE_NOT_EXIST) {
		/* rdb_r->uri を再取得したら、実は存在していなかった場合
		 * mod_dav に合わせてnot found を返す */
		return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}
	else if (ret) {
		/* 予期しないエラー */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to delete db entry.(uri = %s)", rdb_r->uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* 物理ファイルをストレージから削除する */	
	if (divy_remove_physical_files(r, del_filelist)) {
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* サーバ主体のメール送信を実施するのか ? */
	if (divy_ml_enable_svrsendmail(r, rdb_r)) {
		/* サーバ主体のメール送信処理を行う */
		divy_ml_send_servermail(r, rdb_r, NULL);
	}

	return NULL;
}

/**
 * 指定されたsrc が示すdivy_rdbo_keylist のListを、メンバindex の
 * 昇順になるよう並べ替えて、dst として返却する。
 * (note)
 * 	正しく並べ替えるためには、関数_validate_mkcol_dbfolder
 * 	による整合性の検証を行ってこれにパスしていなければなりません。
 *
 * @param p apr_pool_t *
 * @param src const divy_rdbo_keylist *
 * @param dst divy_rdbo_keylist **
 */
static void _sort_mkcol_keylist_elem(apr_pool_t *p,
					const divy_rdbo_keylist *src,
					divy_rdbo_keylist **dst)
{
	int i, cnt;
	divy_rdbo_keylist *keylist;
	divy_rdbo_keylist *keys[] = {NULL, NULL, NULL, NULL, NULL,
				     NULL, NULL, NULL, NULL, NULL}; 
	
	*dst = NULL;

	/* src をindexの昇順になるよう並び替える */
	cnt = 0;
	for (keylist = (divy_rdbo_keylist *) src;
					keylist; keylist = keylist->next) {
		
		/* keys にコピーする */
		keys[keylist->index - 1] = apr_pcalloc(p, sizeof(*keylist));
		keys[keylist->index - 1]->index = keylist->index;
		keys[keylist->index - 1]->key   = apr_pstrdup(p, keylist->key);
		cnt++;
	}

	/* List 構造を作る */
	for (i = 0; i < cnt; i++) {
		if (*dst == NULL) {
			keylist = keys[i];
			*dst    = keylist;
		}
		else {
			keylist->next = keys[i];
			keylist       = keylist->next;
		}
	}
}

/**
 * usr_pr が示すユーザ(in 管理者フォルダ) を削除する。
 *
 * @param r request_rec *
 * @param usr_pr divy_rdbo_usr * 削除対象ユーザのプロパティ
 * @return dav_error * DAV エラー
 */
static dav_error * _remove_user(request_rec *r, divy_rdbo_usr *usr_pr)
{
	apr_pool_t *p = r->pool;
	divy_linkedlist_t *del_filelist = NULL;
	divy_db_transaction_ctx *ts_ctx = NULL;

	/* トランザクション開始 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to create transaction context for copy resource.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * ユーザプロパティ及びプライベートコレクションの削除
	 */
	if (divy_rdbo_remove_user_property(r, usr_pr, &del_filelist, ts_ctx)) {
		/* トランザクションのロールバック */
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to delete user.(userid = %s)", usr_pr->usrid);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * 対象ユーザが持っていたロックを全て解除
	 */
	if (divy_lock_remove_user_lock(r, usr_pr->usrid, ts_ctx)) {
		/* トランザクションのロールバック */
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to delete lock resource.(userid = %s)", usr_pr->usrid);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* DBへの反映を確定する */
	divy_db_commit_transaction(ts_ctx);

	/*
	 * 物理ファイルをストレージから削除する
	 */
	if (divy_remove_physical_files(r, del_filelist)) {
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * grp_pr が示すグループ(in 管理者フォルダ) を削除する。
 * grp_pr が複数のグループをChildグループとして保持していたのなら
 * それらもまとめて削除します。
 *
 * @param r request_rec *
 * @param grp_pr divy_rdbo_grp * 削除対象グループのプロパティ
 * @return dav_error * DAV エラー
 */
static dav_error * _remove_group(request_rec *r, divy_rdbo_grp *grp_pr)
{
	apr_pool_t *p = r->pool;
	divy_linkedlist_t *del_filelist = NULL;

	/*
	 * グループプロパティ及びグループコレクションの削除
	 */
	if (divy_rdbo_remove_group_property(r, grp_pr, &del_filelist, NULL)) {
		/* 予期しないエラー */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to delete group.(group_name = %s)", grp_pr->name);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * 物理ファイルをストレージから削除する
	 */
	if (divy_remove_physical_files(r, del_filelist)) {
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * $root に対するPROPFINDを処理する。
 *
 * (note) 返却するコレクションの種類
 * 	・$root (rdb_r->rsid が存在しない場合だけ検索を実施)
 * 	・$root/$userid
 * 	・$root/Group Folder (特殊ディレクティブが存在する場合)
 *
 * @param r request_rec *
 * @param depth int 深さ (0 / 1 のみ)
 * @param rdb_r divy_rdbo_resource
 * @return dav_error * DAVエラー (HTTP_INTERNAL_SERVER_ERROR)
 * 			エラーがなければNULL
 */
static dav_error * _get_rootfolder(request_rec *r,
					int depth,
					divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p = r->pool;
	const char *userid   = divy_get_userid(r);
	const char *root_uri = dav_divy_get_root_uri(r);
	divy_rdbo_resource *_rdb_r = NULL;
#ifdef DIVY_SUPPORT_EXTENDQUOTA
	const char *authkey;
	dav_divy_server_conf *sconf;
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

	/* depth は 0 / 1 のみ */
	if (depth != 0 && depth != 1) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The depth must be \"0\" or \"1\"");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * $root の検索
	 * (note)
	 * 	$root自身(rdb_r) には、特殊プロパティが付かないので
	 * 	rdb_r->rsid が存在する限り、何もしないことにしました。
	 */
	if (IS_EMPTY(rdb_r->rsid) &&
		divy_rdbo_get_hierarchy_property(r, rdb_r, 0, 0, NULL)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get $root resource.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

#ifdef DIVY_SUPPORT_EXTENDQUOTA
	/* システムQuota情報の取得 */
	authkey = apr_table_get(r->headers_in, DIVY_HEADER_AUTHKEY);
	sconf   = dav_divy_get_server_config(r->server);
	if (IS_FILLED(authkey) && strcmp(sconf->svrlicensekey, authkey) == 0 &&
	    divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN) {
		if (divy_rdbo_get_systemquota_property(r, &rdb_r->sysquota_pr, NULL)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get systemquota.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* 後方互換性に関する考慮 */
		if (rdb_r->sysquota_pr == NULL) {

			/* 全て0にする。設定されていないことを明示するためである */
			rdb_r->sysquota_pr = apr_pcalloc(p, sizeof(divy_rdbo_diskquota));
			rdb_r->sysquota_pr->uri     = apr_pstrdup(p, root_uri);
			rdb_r->sysquota_pr->type    = DIVY_QTTYPE_SYSTEM;
			rdb_r->sysquota_pr->usedst  = APR_INT64_C(0);
			rdb_r->sysquota_pr->maxst   = APR_INT64_C(0);
			rdb_r->sysquota_pr->usedres = APR_INT64_C(0);
			rdb_r->sysquota_pr->maxres  = APR_INT64_C(0);
			rdb_r->sysquota_pr->sfs     = NULL; /* 取得しない */
		}
	}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

	if (depth == 0) {
		rdb_r->next = NULL;	/* 後方メンバをアンリンク */
		return NULL;		/* もうすることはない */
	}

	/*
	 * $root/$userid の検索
	 * (note) 但しプライベートフォルダを見せてもよいと記述されたときだけ。
	 */
	if (divy_enable_userfolderview(r)) {

		/* $root/$userid で検索させる */
		_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource)); 
		_rdb_r->uri   = divy_build_user_uri(p, root_uri, userid);
		_rdb_r->depth = divy_count_dirs(_rdb_r->uri);
		_rdb_r->p     = p;
		_rdb_r->next  = NULL;

		/* DB から情報を取得する */
		if (divy_rdbo_get_hierarchy_property(r, _rdb_r, 0, 0, NULL)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get $root hierarchy resource.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		rdb_r->next = _rdb_r;
		rdb_r = rdb_r->next;
	}

	/*
	 * $root/Group Folder の検索(情報の作成)
	 * (note) 但しグループフォルダを見せてもよいと記述されたときだけ。
	 */
	if (divy_enable_groupview(r)) {

		/* グループトップフォルダの取得 */
		_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
		_rdb_r->uri          = divy_build_group_uri(p, root_uri, NULL);
		_rdb_r->displayname  = divy_get_sfolder_name(p, DIVY_FOLDER_ID_group);
		_rdb_r->depth        = divy_count_dirs(_rdb_r->uri);
		_rdb_r->resourcetype = DIVY_TYPE_COLLECTION;
		_rdb_r->getcontenttype = apr_pstrdup(p, DIR_MAGIC_TYPE);

		/* (note) 2005/07/26 takehara
			プライベートフォルダの更新日時を使用する理由
			getlastmodified が存在しないとXMLエレメントのパースに失敗して
			トップフォルダを空にしてしまう種類のWebDAVクライアントが存在する。
			だが、グループトップフォルダは元々正しいタイムスタンプを持っていないので
			そもそも返すことができない。
			仕方がないので、プライベートフォルダのタイムスタンプを設定しておく
			また、Accessdenyでプライベートフォルダを非許可にしている場合は
			このプライベートフォルダの情報が取れません。その場合は固定値に
			しています（ epoch = 1 = 1970/1/1 )
		 */
		if (rdb_r->getlastmodified != 0) {
			_rdb_r->getlastmodified = rdb_r->getlastmodified;
		}
		else {
			_rdb_r->getlastmodified = 1;
		}
		_rdb_r->p            = p;
		_rdb_r->next         = NULL;

		rdb_r->next = _rdb_r;
		rdb_r = rdb_r->next;
	}

	return NULL;
}

/**
 * $root/Group Folder に対するPROPFINDを処理する。
 *
 * (note) 返却するコレクションの種類
 * 	* $root/Group Folder/$groupid (特殊ディレクティブが存在する場合)
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource
 * @return dav_error * 
 * 	HTTP_INTERNAL_SERVER_ERROR: 内部エラー
 * 	HTTP_CONTINUE : (limit が正だった時、limit件以上のデータが存在した。正常)
 * 	エラーがなければNULL
 */
static dav_error * _get_groupfolder(request_rec *r,
					divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p = r->pool;
	const char *userid = divy_get_userid(r);
	divy_rdbo_resource *grp_rdb_r = NULL;

	if (rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rdb_r is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* $root/Group Folder の検索 --> 取得済みなのでdisplayname を調整する */
	rdb_r->displayname = divy_get_sfolder_name(p, DIVY_FOLDER_ID_group);
	rdb_r->getcontenttype = apr_pstrdup(p, DIR_MAGIC_TYPE);

	/*
	 * $root/Group Folder/$groupid の検索
	 * (note) 但しグループフォルダを見せてもよいと記述されたときだけ。
	 */
	if (divy_enable_groupview(r)) {
		int ret;

		/* グループ一覧の取得 */
		ret = divy_rdbo_get_group_resource_by_userid(r, userid, -1, -1,
								&grp_rdb_r);
		if (ret != 0) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get group-folder resources.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	/* rdb_r に取得リストを繋げる */
	rdb_r->next = grp_rdb_r;

	return NULL;
}

/**
 * 特殊フォルダ(エンティティ) に対するPROPFINDを処理する。
 * (note) 対象フォルダ
 * 	$root/.management/USER/$username
 * 	$root/.management/GROUP/$groupname1/.../$groupnamex
 * 	$root/.management/SQL/$lablename
 * 	$root/.management/MSG/$msgid
 * 	$root/.management/DBMS/$dbmsname
 *
 * @param r request_rec *
 * @param registdt const char * 登録日付(ISO8601形式)
 * @param updatedt const char * 更新日付(ISO8601形式)
 * @param rdb_r divy_rdbo_resource
 * @return dav_error * DAVエラー (HTTP_INTERNAL_SERVER_ERROR)
 * 			エラーがなければNULL
 */
static dav_error * _get_special_entity(request_rec *r,
					const char *registdt,
					const char *updatedt,
					divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p = r->pool;

	if (rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rdb_r is null.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	rdb_r->getcontenttype  = apr_pstrdup(p, DIR_MAGIC_TYPE); 
	rdb_r->creationdate    = dav_divy_iso8601totime_t(p, registdt);
	rdb_r->getlastmodified = dav_divy_iso8601totime_t(p, updatedt);

	return NULL;
}

/**
 * 特殊フォルダ(リレーション) に対するPROPFINDを処理する。
 * (note) 対象フォルダ
 * 	$root/.management/GROUP/$groupname1/.../$groupnamex/.RU_$userid
 * 	$root/.management/GROUP/$groupname1/.../$groupnamex/.RS_$labelname
 * 	$root/.management/USER/$username/.RG_$groupid
 * 	$root/.management/SQL/$lablename/.RG_$groupid
 *
 * @param r request_rec *
 * @param prefix const char * リレーション識別プレフィックス
 * @param rdb_r divy_rdbo_resource
 * @return dav_error * DAVエラー (HTTP_INTERNAL_SERVER_ERROR)
 * 			エラーがなければNULL
 */
static dav_error * _get_special_relation(request_rec *r,
					const char *prefix,
					divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p = r->pool;

	if (rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rdb_r is null.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* 表示名称の切り取り */
	rdb_r->displayname = dav_divy_truncate_prefix(p,
					dav_divy_extract_finalpath_segment(p, r->uri),
					prefix);
	rdb_r->getcontenttype  = apr_pstrdup(p, DIR_MAGIC_TYPE); 

	return NULL;
}

/**
 * "仮想フォルダ"かつ"処理場所"に対するPROPFINDプロパティを生成して返却する。
 * (note) 対象フォルダ
 *	$root/Group Folder
 *	$root/.dblink
 *	$root/.reposdblink
 *	$root/.dbfolder
 *	$root/.management/UPDATE
 *	$root/.management/SQL
 *	$root/.management/USER
 *	$root/.management/GROUP
 *	$root/.management/STATUS
 *	$root/.management/MSG
 *	$root/.management/DBMS
 *	$root/.management/EXECSQL
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource
 * @return dav_error * DAVエラー (HTTP_INTERNAL_SERVER_ERROR)
 */
static dav_error * _get_virtual_entiry(request_rec *r,
					divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p = r->pool;

	if (rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rdb_r is null.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	rdb_r->getcontenttype  = apr_pstrdup(p, DIR_MAGIC_TYPE); 

	return NULL;
}

/**
 * rdb_r に格納されたリソースデータを使ってレスポンスを作成し
 * *response に格納して返却する。
 * (note) rdb_r 必須項目 : uri
 *
 * @param r request_rec *
 * @param wp apr_pool_t * 作業用のプール
 * @param rdb_r divy_rdbo_resource * 
 * @param islocknull int Lock-NUllリソースかどうか
 * @param response dav_response ** レスポンス
 * @return dav_error * DAVエラー
 */
static dav_error * _walk_resource(request_rec *r,
					apr_pool_t *wp,
					const dav_walk_params *params,
					divy_rdbo_resource *rdb_r,
					int islocknull,
					dav_response **response)
{
	dav_error *err               = NULL;
	divy_rdbo_resource *n_rdb_r  = NULL;
	dav_walk_resource  wres      = { 0 };
	dav_resource *res            = NULL;
	dav_resource_private *info   = NULL;
	int exists                   = (islocknull ? 0: 1);
	apr_hash_t *ns_id_hash       = params->root->info->ns_id_hash;
	apr_hash_t *ns_uri_hash      = params->root->info->ns_uri_hash;
	int calltype;
	dav_walker_ctx *ctx          = params->walk_ctx;

	TRACE(r->pool);

	/* walker リソースコンテキストの初期化 */
	wres.walk_ctx = params->walk_ctx;
	wres.pool     = wp;	/* 作業用プールを渡す. r->pool ではない(メモリ縮小のため) */

	for (n_rdb_r = rdb_r; n_rdb_r; n_rdb_r = n_rdb_r->next) {

		/* dav_resource_private の生成＆初期化 */
		info = apr_pcalloc(wp, sizeof(dav_resource_private));
		info->rdb_r       = n_rdb_r;
		info->ns_id_hash  = ns_id_hash;
		info->ns_uri_hash = ns_uri_hash;
		info->r           = r;

		/* dav_resource の生成＆初期化 */
		res = apr_pcalloc(wp, sizeof(dav_resource));
		res->type   = params->root->type;
		res->exists = exists;
		if (n_rdb_r->resourcetype == DIVY_TYPE_COLLECTION && !islocknull) {
			res->collection = 1;
			/* 末尾にスラッシュをつける(responseのhrefになるので。) */
			res->uri = dav_divy_append_endslash(wp, n_rdb_r->uri);
		}
		else {
			res->collection = 0;
			res->uri = n_rdb_r->uri;
		}
		res->info  = info;
		res->hooks = &dav_divy_hooks_repository;
		res->pool  = (ctx->scratchpool != NULL) ? ctx->scratchpool : wp;

		/* walker リソースコンテキストへの設定 */
		wres.resource = res;
		wres.response = *response;

		/* walker 関数をコールして、XMLレスポンスを作成してもらう */
		if (islocknull) {
			calltype = DAV_CALLTYPE_LOCKNULL;
		}
		else if (n_rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
			calltype = DAV_CALLTYPE_COLLECTION;
		}
		else if (n_rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
			calltype = DAV_CALLTYPE_MEMBER;
		}
		else {
			ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Unknown resourcetype. (type = %d)\n"
				"Could not process this resource. (uri = %s)", 
				n_rdb_r->resourcetype, n_rdb_r->uri);
			continue;
		}

		/* walker をCallback する */
		err = (*params->func)(&wres, calltype);
		if (err != NULL) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"An error occured in walk.(uri = %s)", n_rdb_r->uri);
			return err;
		}

		/* 取得値の記録 */
		*response = wres.response;
	}

	return NULL;
}

/*------------------------------------------------------------------------------
 * サーバ主体メール送信機能
  ----------------------------------------------------------------------------*/
/**
 * ウィルス検索に必要な環境を用意し、ウィルス検索処理をスタートさせる.
 * なお、実際の検索はファイルストリームを受けてからとなります.
 *
 * @param r request_rec *
 * @param d_stream dav_stream * DAVストリーム
 * @return dav_error * DAVエラー
 */
static dav_error * _setup_virusscan(request_rec *r, dav_stream *d_stream)
{
	int ret;
	apr_pool_t *p = r->pool;
	dav_divy_dir_conf *dconf;

	/* ウィルス検索機能は有効なの? */
	ret = divy_vsc_enable(r);
	if (ret != DIVY_VSC_ST_ACTIVE) {
		return NULL;	/* エラーではない. 何もしないだけ */
	}

	/* * 4GBを超えるファイルはclamavでは処理が行えない為、スキップします */
	if (d_stream->resource->info->rdb_r->getcontentlength >
												DIVY_VSC_LIMIT_FILE_SIZE) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_INFO + DIVY_SST_DATA,
				"WARNING: Content-Length >4GB skip virus provider."
				"(name = %s / content-length = %"APR_INT64_T_FMT")",
				d_stream->resource->info->rdb_r->displayname,
				d_stream->resource->info->rdb_r->getcontentlength);
		return NULL; /* エラーにしない */
	}

	d_stream->is_infected = 0;	/* 未感染で初期化 */
	dconf = dav_divy_get_dir_config(r);

	/* ウィルス検索プロバイダを取得 */
	d_stream->vscds = lookup_vsc_provider(r, dconf->vscan);
	if (d_stream->vscds == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get virus scanning provider.(name = %s)", dconf->vscan);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* ウィルス検索プロバイダとのセッションを取得 */
	d_stream->vscsession = divy_get_vscsession(r, d_stream->vscds);
	if (d_stream->vscsession == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get session for virus scanning provider."
				"(name = %s)", dconf->vscan);
		d_stream->vscds = NULL;	/* 無かったことにする */
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* ファイルストリームの取得 */
	if (dconf->streammethod == DIVY_STREAMMETHOD_ON &&
	    divy_vsc_canScanStream(d_stream->vscds) == DIVY_VSC_ST_SUPPORTED_STREAM) {

		ret = divy_vsc_startStreamScan(d_stream->vscsession,
					DIVY_VSC_ACT_NOTICE, &d_stream->sfile);
		if (ret != DIVY_VSC_ST_OK) {
			ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to open virus scan stream.(name = %s | msg = %s)", dconf->vscan, divy_vsc_getMsg(d_stream->vscsession, DIVY_VSC_TYPE_SESSION));
			d_stream->vscds = NULL;	/* 無かったことにする */
			divy_vsc_closeSession(d_stream->vscsession);
			d_stream->vscsession = NULL;
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * stream のファイルのウィルス感染状況をチェックする。
 *
 * @param r request_rec *
 * @param stream dav_stream * DAVストリーム
 * @param filepath const char * 検索ファイルパス
 * @return dav_error * DAVエラー
 */
static dav_error * _scanvirus(request_rec *r, dav_stream *stream,
					const char *filepath)
{
	dav_error *err = NULL;
	apr_pool_t *p  = r->pool;
	divy_rdbo_resource *rdb_r = stream->resource->info->rdb_r;
	int ret;
	char *msg, *ncd;

	if (stream->vscsession == NULL) return NULL;

	TRACE(p);

	/* ウィルス検索の実施 */
	if (stream->sfile != NULL) {
		/* ストリームベースウィルス検索の実施 */
		ret = divy_vsc_endStreamScan(stream->vscsession, stream->sfile);
	}
	else {
		/* ファイルベースウィルス検索の実施
		 * (修復はしない) */
		ret = divy_vsc_scanFile(stream->vscsession, DIVY_VSC_ACT_NOTICE,
					filepath);
	}

	/* ウィルス感染していなかった場合(何もしない) */
	if (ret == DIVY_VSC_ST_OK) {
		return NULL;
	}

	/* ステータスコード, メッセージ取得 */
	ret = divy_vsc_getCode(stream->vscsession, DIVY_VSC_TYPE_SESSION);
	ncd = divy_vsc_getNativeCode(stream->vscsession, DIVY_VSC_TYPE_SESSION);
	msg = divy_vsc_getMsg(stream->vscsession, DIVY_VSC_TYPE_SESSION);

	/* エラーは常にForbidden */
	err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	//err = dav_new_error(p, HTTP_UNPROCESSABLE_ENTITY, 0, "");

	/* ウィルスに感染していた場合 */
	if (ret == DIVY_VSC_ST_INFECTED) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"We detected virus in file. (name = %s, userid = %s) "
			"so we will delete it.",
			rdb_r->displayname, divy_get_userid(r));
	}
	/* 検索できないファイルサイズだった */
	else if (ret == DIVY_VSC_ST_TOOLARGE_CONTENT) {
		ERRLOG3(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The file length is too long. (name = %s) "
			"so we will delete it.(ncode = %s, msg = %s)",
			rdb_r->displayname, ncd, msg);
	}
	/* 予期せぬエラー発生 */
	else {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"Failed to scan virus in file. (name = %s) "
			"but we could not allow this file. (ncode = %s, msg = %s)",
			rdb_r->displayname, ncd, msg);
	}

	/*
	 * メールを送信する
	 */
	if (divy_ml_enable_virul_detection_alert(r)) {
		divy_ml_notify_virusdetection(r, ret, rdb_r);
	}

	return err;
}

/*
 * ブラウザのセキュリティを強化するためのセキュリティヘッダ群を追加する
 */
static void _set_enhance_security_headers(request_rec *r)
{
	apr_table_setn(r->headers_out, "X-Content-Type-Options", "nosniff");
	apr_table_setn(r->headers_out, "X-XSS-Protection", "1; mode=block");
	apr_table_setn(r->headers_out, "X-Frame-options", "SAMEORIGIN");
}

#ifdef DIVY_SUPPORT_PRINTER
/**
 * filepath のファイルを設定されたプリンタに出力する。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource *
 * @param filepath const char * ファイルパス
 */
static void _print_file(request_rec *r, divy_rdbo_resource *rdb_r, const char *filepath)
{
	apr_pool_t *p            = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	divy_uri_spec *u_spec    = rdb_r->u_spec;
	int jobid, num_options = 0;
#if 0
	ipp_status_t ipp_st;
#endif
	int ipp_st;
	cups_option_t *options = NULL;
	char *jobname;

	/* プリンタ連携機能が無効であればスキップ */
	if (dconf->netprint != DIVY_NETPRINT_ON) {
		return;
	}

	/* グループ以下(ごみ箱を除く) へのリクエストか？ */
	if (u_spec->infotype != DIVY_INFOTYPE_group_e_regular) {
		return;
	}

	/* プリンター名が無ければ印刷できない */
	if (IS_EMPTY(dconf->printername)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The name of printer is missing.");
		return;
	}

	/* 対象グループが判らなければ印刷できない */
	if (IS_EMPTY(dconf->printergroupname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The name of printer-group is missing.");
	}
	else {
		/* このグループは印刷対象グループか?
		 * URL path segment の単純な比較を行う。
		 * 正式にはDBを引かないとならないがサンプルなのでさぼる */
		char *segment = dav_divy_extract_firstpath_segment(p, &u_spec->other_part[1]);
		if (IS_EMPTY(segment) || strcmp(segment, dconf->printergroupname) != 0) {
			return;
		}
	}

	/* ジョブ名称の作成 */
	jobname = apr_psprintf(p, "TfJob(%s)", divy_get_userid(r));

	/* 印刷する */
	jobid = cupsPrintFile(dconf->printername, filepath, jobname, num_options, options);

	/* 実行結果を取得 */
	ipp_st = cupsLastError();
	if (ipp_st == 0) {
		ERRLOG3(p, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"Print OK (JobName: %s, JobID: %d, file: %s)", jobname, jobid, rdb_r->displayname);
	}
	else {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"Failed to print file. (JobName: %s, JobID: %d, code: 0x0%x)",
			jobname, jobid, ipp_st);
	}
}
#endif	/* DIVY_SUPPORT_PRINTER */


#ifdef DIVY_SUPPORT_PREFLIGHT_PLUGIN
/**
 * 論理ファイル名name、物理パスfilepath のファイルをPitStop連携フォルダに送る.
 *
 * @param r request_rec *
 * @param bu_spec divy_uri_spec * ベースフォルダのuri構造体
 * @param filepath const char * 物理ファイルのフルパス
 * @param name const char * ファイルの論理名(表示名称)
 */
static void _send_preflightfolder(request_rec *r, divy_uri_spec *bu_spec,
								const char *filepath, const char *name)
{
	dav_divy_dir_conf  *dconf = dav_divy_get_dir_config(r);
	apr_pool_t *p  = r->pool;
	char *encname  = NULL, *dispname, *pitstop_path, *ext, *grpname, *tmp;
	int rv;
	apr_status_t st;

	if (r == NULL || bu_spec == NULL || IS_EMPTY(filepath) || IS_EMPTY(name) || 
		IS_EMPTY(dconf->plrootpath) || dconf->plgroupset == NULL) {
		return;
	}

	/* 拡張子判定(pdf以外はスキップ) */
	ext = divy_strrchr(name, '.');
	if (ext == NULL || strcasecmp(ext, ".pdf") != 0) {
		return;
	}

	/* グループ判定(対象グループ以外はスキップ) */
	if (bu_spec->infotype != DIVY_INFOTYPE_group_e_regular) {
		return;
	}
	grpname = apr_pstrdup(p, (bu_spec->other_part + 1));
	tmp = divy_strchr(grpname, '/');
	if (tmp != NULL) {
		*tmp = '\0';
	}
	if (!divy_cset_contains(dconf->plgroupset, grpname)) {
		return;
	}

	/* エンコード変換 */
	dispname = apr_pstrdup(p, name);	/* オリジナル文字列を保護 */
	rv = divy_misc_encode_str(p, "UTF-8", "SHIFT-JIS", dispname, &encname);
	if (rv || IS_EMPTY(encname)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to encode filename to SHIFT-JIS for pre-flight. (code : %d)", rv);
		return;
	}

	/* pre-flight フォルダへのコピー */
	pitstop_path = apr_pstrcat(p, dconf->plrootpath, "/", encname, NULL);
	st = apr_file_copy(filepath, pitstop_path, APR_FILE_SOURCE_PERMS, p);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to copy pdf file to pre-flight directory. "
				"(filename = %s, code : %d)", pitstop_path, rv);
	}
}
#endif	/* DIVY_SUPPORT_PREFLIGHT_PLUGIN */


/*--------------------------------------------------------------
  Create & Initialize REPOSITORY provider Hook structure
  --------------------------------------------------------------*/
const dav_hooks_repository dav_divy_hooks_repository = {
	ENABLE_GET_HANDLING,	/* special GET Handling が必要である */
	dav_divy_get_resource,
	dav_divy_get_parent_resource,
	dav_divy_is_same_resource,
	dav_divy_is_parent_resource,
	dav_divy_open_stream,	/* PUT */
	dav_divy_close_stream,	/* PUT */
	dav_divy_write_stream,	/* PUT */
	dav_divy_seek_stream,	/* PUT */
	dav_divy_set_headers,	/* GET, SEARCH */
	dav_divy_deliver,		/* GET */
	dav_divy_create_collection,	/* MKCOL */
	dav_divy_copy_resource,		/* COPY */
	dav_divy_move_resource,		/* MOVE */
	dav_divy_remove_resource,	/* DELETE */
	dav_divy_walk,				/* PROPFIND */
	dav_divy_getetag,
	NULL,
	dav_divy_get_request_rec,
	dav_divy_pathname,
#ifdef DAV_SUPPORT_POST_HANDLING
	dav_divy_is_posthandling_supported, /* POST (1) */
	dav_divy_get_desired_format,        /* POST (2) */
	dav_divy_post_resource,             /* POST (3) */
	dav_divy_validate_post_token,       /* POST (4) */
#endif	/* DAV_SUPPORT_POST_HANDLING */
};

