/***************************************************************************
                          rmmgr.c  -  relocatable memory manager
                             -------------------
    begin                : Fri.  Apr., 2 2004
    copyright            : (C) 2004 Teamfile team
    email                : jiang-lei@8107.co.jp
 ***************************************************************************/

#include "rmmgr.h"

/*
 * value pairs of [bytes, blocks] in case sMaxBlocks
 * is passed to RMCreate() as a negative value
 */
static const RMSIZEPAIR c_RMPrimes[] = {
  {123456789, 10000},
  {12345678, 1000},
  {1234567, 100},
  {123456, 10},
  {0, 0}
};

#define RM_RDLOCK()   if(pRMManager->pfnLockCB)                       \
                        pRMManager->pfnLockCB(RMLTREADLOCK,           \
                                            pRMManager->pvLockData)
#define RM_WRLOCK()   if(pRMManager->pfnLockCB)                       \
                        pRMManager->pfnLockCB(RMLTWRITELOCK,          \
                                            pRMManager->pvLockData)
#define RM_UNLOCK()   if(pRMManager->pfnLockCB)                       \
                        pRMManager->pfnLockCB(RMLTUNLOCK,             \
                                            pRMManager->pvLockData)

RMRESULT
RMCreate (RMMANAGER * pRMManager, void *pBaseAddress,
	  RM_SIZE_T unRawLength, int sMaxBlocks,
	  RMLOCKCB pfnLockCB, void *pvLockData)
{
  const RMSIZEPAIR *pPair;
  RMFIXEDMNGDATA *pFixedData;
  RMBLOCKREC *pAllocatedTop, *pAvailableTop;

  if (!pRMManager || !pBaseAddress || (unRawLength < RM_MIN_RAW_LEN)) {
    return RMRSINVALIDARG;
  }

  if (sMaxBlocks < 0) {
    for (pPair = c_RMPrimes; pPair->unFirst; pPair++) {
      if (pPair->unFirst < unRawLength) {
	sMaxBlocks = pPair->unSecond;
      }
    }
    if (sMaxBlocks < 0) {
      /* max block not found in primes */
      return RMRSINVALIDARG;
    }
  }
  pRMManager->pfnLockCB = pfnLockCB;
  pRMManager->pvLockData = pvLockData;

  pRMManager->enuLastError = RMRSOK;

  RM_WRLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) pBaseAddress;
  pFixedData->nRMMagic = RM_MAGIC;
  pFixedData->sAllocCounter = 0;
  pFixedData->sFreeCounter = 0;
  pFixedData->unMargin = RM_DEFAULT_MARGIN;
  pFixedData->unUserData = 0;

  pFixedData->sRecordStackMax = sMaxBlocks;
  pFixedData->sAllocatedBlockCount = 0;
  pFixedData->sAvailableBlockCount = 0;

  pFixedData->unMngAreaLength =
    sizeof (RMFIXEDMNGDATA) + sMaxBlocks * sizeof (RMBLOCKREC);
  pFixedData->unStgAreaLength = unRawLength - pFixedData->unMngAreaLength;
  pFixedData->unStgUsed = 0;
  pFixedData->sAttachedManagerCount = 1;

  /* initialize mem block records */
  pAllocatedTop = (RMBLOCKREC *) (pFixedData + 1);
  pAvailableTop = pAllocatedTop + sMaxBlocks - 1;
  pAvailableTop->sBlockNo = 0;
  pAvailableTop->unBlockHead = pFixedData->unMngAreaLength;
  pAvailableTop->unBlockLength = pFixedData->unStgAreaLength;
  pFixedData->sAvailableBlockCount = 1;

  /* save info into local structure */
  pRMManager->pBaseAddress = pBaseAddress;
  pRMManager->unSize = unRawLength;
  RM_UNLOCK ();
  return RMRSOK;
}

RMRESULT
RMAttach (RMMANAGER * pRMManager, void *pBaseAddress,
	  RMLOCKCB pfnLockCB, void *pvLockData)
{
  RMFIXEDMNGDATA *pFixedData;

  if (!pRMManager || !pBaseAddress) {
    return RMRSINVALIDARG;
  }

  pRMManager->pfnLockCB = pfnLockCB;
  pRMManager->pvLockData = pvLockData;
  RM_WRLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) pBaseAddress;
  /* check the magic for safty */
  if (pFixedData->nRMMagic != RM_MAGIC) {
    RM_UNLOCK ();
    return RMRSMAGICMISMATCH;
  }

  pFixedData->sAttachedManagerCount++;
  pRMManager->pBaseAddress = pBaseAddress;
  pRMManager->unSize = pFixedData->unStgAreaLength;
  RM_UNLOCK ();
  return RMRSOK;
}

RMRESULT
RMSetUserData (RMMANAGER * pRMManager, RM_USER_DATA_T UserData)
{
  RMFIXEDMNGDATA *pFixedData;

  if (!pRMManager) {
    return RMRSINVALIDARG;
  }

  RM_WRLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  pFixedData->unUserData = UserData;
  RM_UNLOCK ();

  return RMRSOK;
}

RMRESULT
RMGetUserData (RMMANAGER * pRMManager, RM_USER_DATA_T * pUserData)
{
  RMFIXEDMNGDATA *pFixedData;

  if (!pRMManager || !pUserData) {
    return RMRSINVALIDARG;
  }

  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  *pUserData = pFixedData->unUserData;
  RM_UNLOCK ();

  return RMRSOK;
}

void
RMSetLockCallback (RMMANAGER * pRMManager, RMLOCKCB pfnLockCB,
		   void *pvUserData)
{
  if (!pRMManager) {
    return;
  }

  pRMManager->pfnLockCB = pfnLockCB;
  pRMManager->pvLockData = pvUserData;
}

/*
RMRESULT
RMDumpState(RMMANAGER* pRMManager, FILE* pOutput, int bDumpBlockInfo) {
  RMFIXEDMNGDATA* pFixedData;
  RMBLOCKREC *pAllocatedBottom, *pAvailableTop;
  int nIndex;
  
  if(!pRMManager){
    return RMRSINVALIDARG;
  }

  pFixedData = (RMFIXEDMNGDATA*)(pRMManager->pBaseAddress);
  pAllocatedBottom = (RMBLOCKREC*)(pFixedData + 1);
  pAvailableTop = pAllocatedBottom + pFixedData->sRecordStackMax -
                  pFixedData->sAvailableBlockCount;
  fprintf(pOutput, "Dumping relocatable memory status:\n");
  fprintf(pOutput, "====================================================\n");
  fprintf(pOutput, "%20s : %d\n", "Attached Managers",
          (int)(pFixedData->sAttachedManagerCount));
  fprintf(pOutput, "%20s : 0x%x\n", "Base Address",
          (unsigned int)(pRMManager->pBaseAddress));
  fprintf(pOutput, "%20s : %d bytes\n", "Size",
          (unsigned long)(pRMManager->unSize));
  fprintf(pOutput, "%20s : %d bytes\n", "Management Area",
          (unsigned long)pFixedData->unMngAreaLength);
  fprintf(pOutput, "%20s : %d bytes\n", "Storage Area",
          (unsigned long)pFixedData->unStgAreaLength);
  fprintf(pOutput, "%20s : %d bytes\n", "Storage used",
          pFixedData->unStgUsed);
  fprintf(pOutput, "%20s : %u\n", "Max blocks",
          pFixedData->sRecordStackMax);
  fprintf(pOutput, "%20s : %2.2f%%\n", "Efficiency",
          (double)100.0 * (double)(pFixedData->unStgAreaLength) / (double)pRMManager->unSize);
  fprintf(pOutput, "%20s : %d\n", "Used blocks",
          pFixedData->sAllocatedBlockCount + pFixedData->sAvailableBlockCount);
  fprintf(pOutput, "%d block(s) allocated, %d block(s) available\n",
          pFixedData->sAllocatedBlockCount,
          pFixedData->sAvailableBlockCount);

  if(!bDumpBlockInfo)
    return RMRSINVALIDARG;
  fprintf(pOutput, "available blocks:\n");
  fprintf(pOutput, "----------------------------------------------------\n");
  for(nIndex = 0; nIndex < pFixedData->sAvailableBlockCount; nIndex++) {
    fprintf(pOutput, "block #%d\nstart 0x%x\t end 0x%x\t length %d\n",
            nIndex,
            pAvailableTop[nIndex].unBlockHead,
            pAvailableTop[nIndex].unBlockHead + pAvailableTop[nIndex].unBlockLength -1,
            pAvailableTop[nIndex].unBlockLength);
            
  }
  
  fprintf(pOutput, "----------------------------------------------------\n");
  fprintf(pOutput, "allocated blocks:\n");
  fprintf(pOutput, "----------------------------------------------------\n");
  for(nIndex = 0; nIndex < pFixedData->sAllocatedBlockCount ; nIndex++) {
    fprintf(pOutput, "block #%u\nstart 0x%x\t end 0x%x\t length %d\n",
            pAllocatedBottom[nIndex].sBlockNo ,
            pAllocatedBottom[nIndex].unBlockHead,
            pAllocatedBottom[nIndex].unBlockHead + pAllocatedBottom[nIndex].unBlockLength -1,
            pAllocatedBottom[nIndex].unBlockLength);

  }
  return RMRSOK;
}
*/

#ifdef RM_DEBUG
RM_HANDLE
RMMallocDebug (RMMANAGER * pRMManager, RM_SIZE_T unSize, const char *pszFile,
	       int nLine)
{
#else
RM_HANDLE
RMMalloc (RMMANAGER * pRMManager, RM_SIZE_T unSize)
{
#endif /*RM_DEBUG */
  RMFIXEDMNGDATA *pFixedData;
  RMBLOCKREC *pAvailableBlock, *pAvailableTop;
  RMBLOCKREC *pBlockToAllocate, *pNewBlock;
  int     sAvailableBlockCount;
  int     sIndex, sLastIndex;
  RM_SIZE_T unLastSize, unBlockSize;
  RM_OFF_T unBlockHead;

  if (!pRMManager || !unSize) {
    /* invalid argument */
    return 0;
  }

  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);

  RM_WRLOCK ();
  /* let's check record count first */
  if ((pFixedData->sAllocatedBlockCount + pFixedData->sAvailableBlockCount)
      >= pFixedData->sRecordStackMax) {
    /* no more blocks */
    RM_UNLOCK ();
    return 0;
  }

  sAvailableBlockCount = pFixedData->sAvailableBlockCount;
  pAvailableTop = (RMBLOCKREC *) (pFixedData + 1) +
    pFixedData->sRecordStackMax - pFixedData->sAvailableBlockCount;

  /* search for the most suitable block through available list */
  sLastIndex = -1;
  unLastSize = 0;
  for (sIndex = 0; sIndex < sAvailableBlockCount; sIndex++) {
    pAvailableBlock = pAvailableTop + sIndex;
    unBlockSize = pAvailableBlock->unBlockLength;
    if (unBlockSize >= unSize) {
      if (sLastIndex == -1) {
	sLastIndex = sIndex;
	unLastSize = unBlockSize;
      } else if (unBlockSize < unSize) {
	sLastIndex = sIndex;
	unLastSize = unBlockSize;
      }
    }
  }

  if (sLastIndex == -1) {
    /* out of memory */
    RM_UNLOCK ();
    return 0;
  }

  /* we found a block */
  pBlockToAllocate = pAvailableTop + sLastIndex;
  unBlockHead = pBlockToAllocate->unBlockHead;
  unBlockSize = pBlockToAllocate->unBlockLength;

  /* but is it still too large? */
  if (unBlockSize - unSize > pFixedData->unMargin) {
    /* too large, split the block into two */
    pBlockToAllocate->unBlockLength -= unSize;
    pBlockToAllocate->unBlockHead += unSize;
    unBlockSize = unSize;
  } else {
    /* the size is OK, remove it from available list */
    if (sLastIndex) {
      /*  We only need to move memory when block
       *   is not at top of the list. Otherwise
       *   we do nothing.
       */
      void   *pFrom = (void *) pAvailableTop;
      void   *pTo = (void *) (pAvailableTop + 1);
      unsigned long unLen = sLastIndex * sizeof (RMBLOCKREC);
      memmove (pTo, pFrom, unLen);
    }
    pFixedData->sAvailableBlockCount--;
  }

  /* now insert it into the allocated list */
  pFixedData->sAllocCounter++;
  pNewBlock =
    (RMBLOCKREC *) (pFixedData + 1) + pFixedData->sAllocatedBlockCount;
  pNewBlock->sBlockNo = pFixedData->sAllocCounter;
  pNewBlock->unBlockHead = unBlockHead;
  pNewBlock->unBlockLength = unBlockSize;
#ifdef RM_DEBUG
  pNewBlock->pszFile = pszFile;
  pNewBlock->nLine = nLine;
#endif /* RM_DEBUG */
  pFixedData->sAllocatedBlockCount++;
  pFixedData->unStgUsed += unBlockSize;
  RM_UNLOCK ();
  return unBlockHead;
}

RM_SIZE_T
RMGetMargin (RMMANAGER * pRMManager)
{
  RMFIXEDMNGDATA *pFixedData;
  RM_OFF_T unRet;
  if (!pRMManager) {
    /* invalid argument */
    return 0;
  }
  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  unRet = pFixedData->unMargin;
  RM_UNLOCK ();
  return unRet;
}

void
RMSetMargin (RMMANAGER * pRMManager, RM_SIZE_T unNewValue)
{
  RMFIXEDMNGDATA *pFixedData;
  if (!pRMManager) {
    /* invalid argument */
    return;
  }
  RM_WRLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  pFixedData->unMargin = unNewValue;
  RM_UNLOCK ();
}

void   *
RMHandleToPtr (RMMANAGER * pRMManager, RM_HANDLE hMem)
{
  RMFIXEDMNGDATA *pFixedData;
  void   *pRet;
  if (!pRMManager || !hMem) {
    /* invalid argument */
    return 0;
  }
  /* RM_RDLOCK(); */
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  pRet =
    (void *) ((unsigned char *) (pRMManager->pBaseAddress) +
	      (unsigned int) hMem);
  /* RM_UNLOCK(); */
  return pRet;
}

#ifdef RM_DEBUG
RM_HANDLE
RMCallocDebug (RMMANAGER * pRMManager, RM_SIZE_T unSize, const char *pszFile,
	       int nLine)
{
  void   *pData;
  RM_HANDLE hMem = RMMallocDebug (pRMManager, unSize, pszFile, nLine);
#else
RM_HANDLE
RMCalloc (RMMANAGER * pRMManager, RM_SIZE_T unSize)
{
  void   *pData;
  RM_HANDLE hMem = RMMalloc (pRMManager, unSize);
#endif /* RM_DEBUG */
  if (!hMem)
    return 0;
  pData = RMHandleToPtr (pRMManager, hMem);
  if (!pData)
    return 0;
  memset (pData, 0, unSize);
  return hMem;
}

void
RMFree (RMMANAGER * pRMManager, RM_HANDLE hMem)
{
  RMFIXEDMNGDATA *pFixedData;
  RM_OFF_T unHead, unEnd;
  RM_SIZE_T unLen;
  RMBLOCKREC *pAllocatedBottom, *pBlockToFree;
  int     sIndex, sAllocatedCount;
  RM_OFF_T unBlockHead, unBlockEnd;
  int     sAfter, sBefore;
  int     nJoinUpper, nJoinLower;

  if (!pRMManager || !hMem) {
    /* invalid argument */
    return;
  }

  RM_WRLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  pAllocatedBottom = (RMBLOCKREC *) (pFixedData + 1);
  sAllocatedCount = pFixedData->sAllocatedBlockCount;
  unHead = (RM_OFF_T) hMem;

  for (sIndex = 0; sIndex < sAllocatedCount; sIndex++) {
    pBlockToFree = pAllocatedBottom + sIndex;

    if (pBlockToFree->unBlockHead == unHead) {
      void   *pFrom, *pTo;
      unsigned long unRecLen;
      /* mem block found, free it */
      RMBLOCKREC *pAvailableTop;
      int     sAvailableBlockCount;
      unLen = pBlockToFree->unBlockLength;
      unEnd = unHead + unLen;	/* preciseley, 1 byte past end */
      /* move following blocks forward to delete the record */
      pFrom = (void *) (pBlockToFree + 1);
      pTo = (void *) (pBlockToFree);
      unRecLen = (pFixedData->sAllocatedBlockCount - sIndex - 1) *
	sizeof (RMBLOCKREC);
      /* first remove it from allocated list */
      if (unRecLen) {
	memmove (pTo, pFrom, unRecLen);
      }

      pFixedData->sAllocatedBlockCount--;

      /* then add it into the available list.
       *   available list is sorted by address
       *   so we shall check if we can join the
       *   block with adjacent ones.
       */
      sAvailableBlockCount = pFixedData->sAvailableBlockCount;

      pAvailableTop = (RMBLOCKREC *) (pFixedData + 1) +
	pFixedData->sRecordStackMax - pFixedData->sAvailableBlockCount;
      sAfter = -1;
      sBefore = -1;
      nJoinUpper = 0;
      nJoinLower = 0;
      /*try to find the "before" and "after" block */
      for (sIndex = 0; sIndex < sAvailableBlockCount; sIndex++) {
	RMBLOCKREC *pCurrent = pAvailableTop + sIndex;
	unBlockHead = pCurrent->unBlockHead;
	unBlockEnd = unBlockHead + pCurrent->unBlockLength;
	if (unBlockHead > unEnd) {
	  sBefore = sIndex;
	  break;
	} else if (unBlockHead == unEnd) {
	  sBefore = sIndex;
	  nJoinUpper = 1;
	  break;
	}

	if (unBlockEnd < unHead) {
	  sAfter = sIndex;
	} else if (unBlockEnd == unHead) {
	  sAfter = sIndex;
	  nJoinLower = 1;
	}
      }

      /* add the mem block and join bounds if necessary */
      if (sAfter < 0 && sBefore < 0) {
	/*this is the ONE and only block */
	pAvailableTop--;
	pAvailableTop->sBlockNo = 0;
	pAvailableTop->unBlockHead = unHead;
	pAvailableTop->unBlockLength = unLen;
	pFixedData->sAvailableBlockCount++;
      } else if (sAfter < 0) {

	/* at top */
	if (nJoinUpper) {
	  pAvailableTop->unBlockHead = unHead;
	  pAvailableTop->unBlockLength += unLen;
	} else {
	  /* insert at top */
	  pAvailableTop--;
	  pAvailableTop->sBlockNo = 0;
	  pAvailableTop->unBlockHead = unHead;
	  pAvailableTop->unBlockLength = unLen;
	  pFixedData->sAvailableBlockCount++;
	}
      } else if (sBefore < 0) {
	RM_ASSERT (sAvailableBlockCount > 0);
	/* at bottom */
	if (nJoinLower) {
	  pAvailableTop[sAvailableBlockCount - 1].unBlockHead = unHead;
	  pAvailableTop[sAvailableBlockCount - 1].unBlockLength += unLen;
	} else {
	  void   *pFrom = (void *) pAvailableTop;
	  void   *pTo = (void *) (pAvailableTop - 1);
	  unsigned long unRecLen = sAvailableBlockCount * sizeof (RMBLOCKREC);
	  memmove (pTo, pFrom, unRecLen);
	  pAvailableTop[sAvailableBlockCount].sBlockNo = 0;
	  pAvailableTop[sAvailableBlockCount].unBlockHead = unHead;
	  pAvailableTop[sAvailableBlockCount].unBlockLength = unLen;
	  pFixedData->sAvailableBlockCount++;
	}
      } else {
	/* in the middle */
	RM_ASSERT (sBefore > 0 && sAfter >= 0);
	RM_ASSERT (sAvailableBlockCount > 1);
	RM_ASSERT (sBefore != sAfter);

	if (nJoinUpper && nJoinLower) {
	  /* both bounds are joined and
	   * lower block shall be removed */
	  void   *pFrom = (void *) pAvailableTop;
	  void   *pTo = (void *) (pAvailableTop + 1);
	  unsigned long unRecLen = sAfter * sizeof (RMBLOCKREC);
	  pAvailableTop[sBefore].unBlockHead =
	    pAvailableTop[sAfter].unBlockHead;
	  pAvailableTop[sBefore].unBlockLength +=
	    pAvailableTop[sAfter].unBlockLength + unLen;
	  if (unRecLen)
	    memmove (pTo, pFrom, unRecLen);
	  pFixedData->sAvailableBlockCount--;
	} else if (nJoinUpper) {
	  /* upper block shall be joined */
	  pAvailableTop[sBefore].unBlockHead = unHead;
	  pAvailableTop[sBefore].unBlockLength += unLen;
	} else if (nJoinLower) {
	  /* lower block shall be joined */
	  pAvailableTop[sAfter].unBlockLength += unLen;
	} else {
	  /* no join, just insert a block */
	  void   *pFrom = (void *) pAvailableTop;
	  void   *pTo = (void *) (pAvailableTop - 1);
	  unsigned long unRecLen = (sAfter + 1) * sizeof (RMBLOCKREC);
	  memmove (pTo, pFrom, unRecLen);
	  pAvailableTop[sAfter].unBlockHead = unHead;
	  pAvailableTop[sAfter].unBlockLength = unLen;
	  pFixedData->sAvailableBlockCount++;
	}
      }
      pFixedData->unStgUsed -= unLen;
      pFixedData->sFreeCounter++;
      RM_UNLOCK ();
      return;
    }
  }
  RM_UNLOCK ();
  RM_ASSERT (0);
}

RM_HANDLE
RMRealloc (RMMANAGER * pRMManager, RM_HANDLE hMem, RM_SIZE_T unNewSize)
{
  return 0;
}

int
rmGetMaxBlockCount (RMMANAGER * pRMManager)
{
  RMFIXEDMNGDATA *pFixedData;
  int     nRet;

  if (!pRMManager)
    return -1;
  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  nRet = pFixedData->sRecordStackMax;
  RM_UNLOCK ();

  return nRet;
}


int
rmGetAllocatedBlockCount (RMMANAGER * pRMManager)
{
  RMFIXEDMNGDATA *pFixedData;
  int     nRet;

  if (!pRMManager)
    return -1;
  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  nRet = pFixedData->sAllocatedBlockCount;
  RM_UNLOCK ();

  return nRet;
}

RMRESULT
rmGetAllocatedBlock (RMMANAGER * pRMManager, int nIndex, RMBLOCKREC * pRec)
{
  RMFIXEDMNGDATA *pFixedData;
  if (!pRMManager || nIndex < 0 || !pRec)
    return RMRSINVALIDARG;

  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  if (nIndex >= pFixedData->sAllocatedBlockCount) {
    RM_UNLOCK ();
    return RMRSINVALIDARG;
  }
  *pRec = ((RMBLOCKREC *) (pFixedData + 1))[nIndex];
  RM_UNLOCK ();
  return RMRSOK;
}

int
rmGetAvailableBlockCount (RMMANAGER * pRMManager)
{
  RMFIXEDMNGDATA *pFixedData;
  int     nRet;

  if (!pRMManager)
    return -1;
  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  nRet = pFixedData->sAvailableBlockCount;
  RM_UNLOCK ();
  return nRet;
}

RMRESULT
rmGetAvailableBlock (RMMANAGER * pRMManager, int nIndex, RMBLOCKREC * pRec)
{
  RMFIXEDMNGDATA *pFixedData;
  RMBLOCKREC *pAvailableTop;
  if (!pRMManager || nIndex < 0 || !pRec)
    return RMRSINVALIDARG;
  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  if (nIndex >= pFixedData->sAvailableBlockCount) {
    RM_UNLOCK ();
    return RMRSINVALIDARG;
  }
  pAvailableTop = (RMBLOCKREC *) (pFixedData + 1) +
    pFixedData->sRecordStackMax - pFixedData->sAvailableBlockCount;
  *pRec = pAvailableTop[nIndex];
  RM_UNLOCK ();
  return RMRSOK;
}

int
rmGetMngAreaLength (RMMANAGER * pRMManager)
{
  RMFIXEDMNGDATA *pFixedData;
  int     nRet;
  if (!pRMManager)
    return -1;
  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  nRet = pFixedData->unMngAreaLength;
  RM_UNLOCK ();
  return nRet;
}

int
rmGetStgAreaLength (RMMANAGER * pRMManager)
{
  RMFIXEDMNGDATA *pFixedData;
  int     nRet;
  if (!pRMManager)
    return -1;
  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  nRet = pFixedData->unStgAreaLength;
  RM_UNLOCK ();
  return nRet;
}

int
rmGetStgUsage (RMMANAGER * pRMManager)
{
  RMFIXEDMNGDATA *pFixedData;
  int     nRet;
  if (!pRMManager)
    return -1;
  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  nRet = pFixedData->unStgUsed;
  RM_UNLOCK ();
  return nRet;
}

int
rmGetAllocCount (RMMANAGER * pRMManager)
{
  RMFIXEDMNGDATA *pFixedData;
  int     nRet;
  if (!pRMManager)
    return -1;
  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  nRet = pFixedData->sAllocCounter;
  RM_UNLOCK ();
  return nRet;
}

int
rmGetFreeCount (RMMANAGER * pRMManager)
{
  RMFIXEDMNGDATA *pFixedData;
  int     nRet;
  if (!pRMManager)
    return -1;
  RM_RDLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  nRet = pFixedData->sFreeCounter;
  RM_UNLOCK ();
  return nRet;
}

void
RMPurge (RMMANAGER * pRMManager)
{
  RMFIXEDMNGDATA *pFixedData;
  RMBLOCKREC *pAllocatedTop, *pAvailableTop;
  if (!pRMManager)
    return;
  RM_WRLOCK ();
  pFixedData = (RMFIXEDMNGDATA *) (pRMManager->pBaseAddress);
  pFixedData->sAllocCounter = 0;
  pFixedData->sFreeCounter = 0;
  pFixedData->sAllocatedBlockCount = 0;
  pFixedData->sAvailableBlockCount = 1;
  pFixedData->unStgUsed = 0;

  pAllocatedTop = (RMBLOCKREC *) (pFixedData + 1);
  pAvailableTop = pAllocatedTop + pFixedData->sRecordStackMax - 1;
  pAvailableTop->sBlockNo = 0;
  pAvailableTop->unBlockHead = pFixedData->unMngAreaLength;
  pAvailableTop->unBlockLength = pFixedData->unStgAreaLength;
  RM_UNLOCK ();
}

/*
 * Utility functions
 *   string operations
 *
 */

RM_HANDLE
RMUstrdup (RMMANAGER * pRMMgr, const char *pszSrc, char **ppszDst)
{
  RM_HANDLE hRet;
  char   *pszRet = 0;
  int     nLen;
  if (!pRMMgr || !pszSrc)
    return 0;
  nLen = strlen (pszSrc);
  hRet = RMMalloc (pRMMgr, nLen + 1);
  pszRet = RMHandleToPtr (pRMMgr, hRet);
  if (pszRet) {
    strcpy (pszRet, pszSrc);
  }

  if (ppszDst)
    *ppszDst = pszRet;
  return hRet;
}

/*
 * Utility functions
 *   double linked list
 *
 */

/*
 * internal functions
 */

/* get tail position of a list
 * list must not be empty
 */
RMURESULT
rmuGetTailPos (RMULIST * pList, RMULISTWALKER * pWalker)
{
  RMULISTWALKER *pListWalkder;
  RM_HANDLE hCurrent;
  RMULISTNODE *pCurrent;

  RM_ASSERT (pList);
  RM_ASSERT (pWalker);

  pListWalkder = &(pList->stWalker);
  *pWalker = *pListWalkder;

  pCurrent = pWalker->pCurrent;
  hCurrent = pWalker->hCurrent;
  if (!pCurrent) {
    pCurrent = pWalker->pHead;
    hCurrent = pWalker->hHead;
    pWalker->pCurrent = pWalker->pHead;
    pWalker->hCurrent = pWalker->hHead;
  }

  for (; pCurrent->hNext;) {
    hCurrent = pCurrent->hNext;
    pCurrent = (RMULISTNODE *) RMHandleToPtr (pList->pRMMgr, hCurrent);
    if (!pCurrent)
      return RMUDATACORRUPTED;
  }
  pWalker->hTail = hCurrent;
  pWalker->pTail = pCurrent;
  pWalker->hCurrent = hCurrent;
  pWalker->pCurrent = pCurrent;
  return RMUOK;
}

/*
 *  Public functions
 */
RMURESULT
RMUListCreate (RMMANAGER * pRMMgr, RMULIST * pList,
	       RMULINSERTCALLBACK pfnInsertCB,
	       RMULREMOVECALLBACK pfnRemoveCB, RM_HANDLE * phHead)
{
  RM_HANDLE hHead = 0;
  RMULISTNODE *pHead = 0;

  if (!pRMMgr || !pList)
    return RMUINVALIDARG;

  if (!phHead)
    return RMUINVALIDARG;

  pList->pRMMgr = pRMMgr;
  pList->phHead = phHead;
  pList->pfnInsertCB = pfnInsertCB;
  pList->pfnRemoveCB = pfnRemoveCB;
  pList->stWalker.hCurrent = hHead;
  pList->stWalker.hHead = hHead;
  pList->stWalker.pHead = pHead;
  pList->stWalker.hTail = hHead;
  pList->stWalker.pTail = pHead;
  pList->stWalker.hCurrent = hHead;
  pList->stWalker.pCurrent = pHead;
  return RMUOK;
}

RMURESULT
RMUListAttach (RMMANAGER * pRMMgr, RMULIST * pList,
	       RMULINSERTCALLBACK pfnInsertCB,
	       RMULREMOVECALLBACK pfnRemoveCB, RM_HANDLE * phHead)
{
  RMULISTNODE *pHead;
  if (!pRMMgr || !pList || !phHead)
    return RMUINVALIDARG;
  pHead = (RMULISTNODE *) RMHandleToPtr (pRMMgr, *phHead);
  if (!pHead)
    return RMUINVALIDARG;
  pList->pRMMgr = pRMMgr;
  pList->phHead = phHead;
  pList->pfnInsertCB = pfnInsertCB;
  pList->pfnRemoveCB = pfnRemoveCB;
  pList->stWalker.hCurrent = *phHead;
  pList->stWalker.hHead = *phHead;
  pList->stWalker.pHead = pHead;
  pList->stWalker.hTail = 0;
  pList->stWalker.pTail = 0;
  pList->stWalker.hCurrent = *phHead;
  pList->stWalker.pCurrent = pHead;
  return RMUOK;
}

RMURESULT
RMUListDestroy (RMULIST * pList)
{
  RM_HANDLE hLast, hCurrent, hData;
  RMMANAGER *pRMMgr;
  RMULISTNODE *pCurrent;
  if (!pList)
    return RMUINVALIDARG;
  pRMMgr = pList->pRMMgr;
  RM_ASSERT (pRMMgr);

  if (!pList->phHead)
    return RMUOK;		/* already destroyed */

  for (hCurrent = pList->stWalker.hHead; hCurrent;) {
    pCurrent = (RMULISTNODE *) RMHandleToPtr (pRMMgr, hCurrent);
    if (!pCurrent)
      return RMUDATACORRUPTED;
    hLast = hCurrent;
    hCurrent = pCurrent->hNext;
    hData = pCurrent->hData;
    if (pList->pfnRemoveCB) {
      pList->pfnRemoveCB (pRMMgr, hData);
    }
    RMFree (pRMMgr, hLast);
  }
  *(pList->phHead) = 0;
  pList->phHead = 0;
  return RMUOK;
}

int
RMUListIsEmpty (RMULIST * pList)
{
  if (!pList)
    return 1;
  if (!pList->stWalker.pHead || !pList->stWalker.hHead)
    return 1;
  return 0;
}

RMURESULT
RMUListInsertHead (RMULIST * pList, void *pData)
{
  RM_HANDLE hHead;
  RMULISTNODE *pHead;
  RMMANAGER *pRMMgr;
  RMULISTWALKER *pWalker;
  if (!pList)
    return RMUINVALIDARG;
  pRMMgr = pList->pRMMgr;
  pWalker = &(pList->stWalker);
  RM_ASSERT (pRMMgr);

  hHead = RMMalloc (pRMMgr, sizeof (RMULISTNODE));
  pHead = (RMULISTNODE *) RMHandleToPtr (pRMMgr, hHead);
  if (!pHead)
    return RMUMALLOCFAILED;
  *(pList->phHead) = hHead;

  pHead->hNext = pList->stWalker.hHead;
  pHead->hPrev = 0;
  if (pList->pfnInsertCB)
    pHead->hData = pList->pfnInsertCB (pRMMgr, pData);
  else
    pHead->hData = (RM_HANDLE) pData;

  if (pWalker->pHead) {
    /* head exists, modify the link */
    pWalker->pHead->hPrev = hHead;
  } else {
    /* this is the first element, so set the tail */
    pWalker->hTail = hHead;
    pWalker->pTail = pHead;
  }

  pWalker->hHead = hHead;
  pWalker->pHead = pHead;

  return RMUOK;
}

RMURESULT
RMUListInsertTail (RMULIST * pList, void *pData)
{
  RM_HANDLE hTail;
  RMULISTNODE *pTail;
  RMMANAGER *pRMMgr;
  RMULISTWALKER *pWalker;
  if (!pList)
    return RMUINVALIDARG;
  pRMMgr = pList->pRMMgr;
  pWalker = &(pList->stWalker);
  RM_ASSERT (pRMMgr);

  if (!pWalker->pHead) {
    /* this is the first element */
    return RMUListInsertHead (pList, pData);
  } else {
    RMULISTWALKER stWalker;
    RMURESULT enuRet;
    RM_HANDLE hOldTail;
    RMULISTNODE *pOldTail;
    /* allocate a new node */
    hTail = RMMalloc (pRMMgr, sizeof (RMULISTNODE));
    pTail = (RMULISTNODE *) RMHandleToPtr (pRMMgr, hTail);
    if (!hTail)
      return RMUMALLOCFAILED;

    if (pWalker->hTail && pWalker->pTail) {
      /* we have old tail position */
      hOldTail = pWalker->hTail;
      pOldTail = pWalker->pTail;
    } else {
      /* tail position is unknown */
      /* get tail of this list */
      enuRet = rmuGetTailPos (pList, &stWalker);
      if (RMUOK != enuRet) {
	return enuRet;
      }
      hOldTail = stWalker.hTail;
      pOldTail = stWalker.pTail;
    }

    if (pList->pfnInsertCB)
      pTail->hData = pList->pfnInsertCB (pRMMgr, pData);
    else
      pTail->hData = (RM_HANDLE) pData;

    pTail->hPrev = hOldTail;
    pTail->hNext = 0;
    pOldTail->hNext = hTail;

    pWalker->hTail = hTail;
    pWalker->pTail = pTail;

    return RMUOK;
  }
}

RMURESULT
RMUListRemoveHead (RMULIST * pList)
{
  RM_HANDLE hHead;
  RMULISTNODE *pHead;
  RMMANAGER *pRMMgr;
  RMULISTWALKER *pWalker;
  int     bSyncCurrent, bSyncTail;

  if (!pList)
    return RMUINVALIDARG;
  pRMMgr = pList->pRMMgr;
  pWalker = &(pList->stWalker);
  RM_ASSERT (pRMMgr);

  pHead = pWalker->pHead;
  hHead = pWalker->hHead;

  if (!pHead)
    return RMUEMPTYLIST;

  bSyncCurrent = (pWalker->hCurrent == pWalker->hHead);
  bSyncTail = (pWalker->hTail == pWalker->hHead);
  pWalker->hHead = pHead->hNext;
  if (pWalker->hHead) {
    pWalker->pHead = (RMULISTNODE *) RMHandleToPtr (pRMMgr, pWalker->hHead);
    RM_ASSERT (pWalker->pHead);
    pWalker->pHead->hPrev = 0;
  } else {
    pWalker->pHead = 0;
  }

  if (bSyncCurrent) {
    pWalker->hCurrent = pWalker->hHead;
    pWalker->pCurrent = pWalker->pCurrent;
  }

  if (bSyncTail) {
    pWalker->hTail = 0;
    pWalker->pTail = 0;
  }

  if (pList->pfnRemoveCB)
    pList->pfnRemoveCB (pRMMgr, pHead->hData);

  RMFree (pRMMgr, hHead);

  *(pList->phHead) = pWalker->hHead;
  return RMUOK;
}

RMURESULT
RMUListRemoveTail (RMULIST * pList)
{
  return RMUNOTIMPLEMENTED;
}

RMURESULT
RMUListRemoveCurrent (RMULIST * pList)
{
  RM_HANDLE hPrev, hNext, hCurrent;
  RMULISTNODE *pPrev, *pNext, *pCurrent;
  RMMANAGER *pRMMgr;
  RMULISTWALKER *pWalker;

  if (!pList)
    return RMUINVALIDARG;
  pRMMgr = pList->pRMMgr;
  pWalker = &(pList->stWalker);
  RM_ASSERT (pRMMgr);

  hCurrent = pWalker->hCurrent;
  if (!hCurrent)
    return RMUINVALIDPOSITION;
  pCurrent = (RMULISTNODE *) RMHandleToPtr (pRMMgr, hCurrent);
  RM_ASSERT (pCurrent);


  hPrev = pCurrent->hPrev;
  hNext = pCurrent->hNext;

  if (!hPrev) {
    /* this is head node */
    return RMUListRemoveHead (pList);
  }

  pPrev = (RMULISTNODE *) RMHandleToPtr (pRMMgr, hPrev);
  RM_ASSERT (pPrev);
  pPrev->hNext = pCurrent->hNext;

  if (hNext) {
    pNext = (RMULISTNODE *) RMHandleToPtr (pRMMgr, hNext);
    RM_ASSERT (pNext);
    pNext->hPrev = hPrev;
  }

  /* remove data and the node */
  if (pList->pfnRemoveCB)
    pList->pfnRemoveCB (pRMMgr, pCurrent->hData);

  RMFree (pRMMgr, hCurrent);
  return RMUOK;

}


RMURESULT
RMUListGoToHead (RMULIST * pList)
{
  RM_HANDLE hHead;
  RMULISTNODE *pHead;
  if (!pList)
    return RMUINVALIDARG;

  hHead = pList->stWalker.hHead;
  pHead = pList->stWalker.pHead;
  if (!pHead)
    return RMUEMPTYLIST;
  pList->stWalker.hCurrent = hHead;
  pList->stWalker.pCurrent = pHead;
  return RMUOK;
}

RMURESULT
RMUListGoToTail (RMULIST * pList)
{
  RM_HANDLE hHead, hTail;
  RMULISTNODE *pHead, *pTail;
  RMULISTWALKER *pWalker;
  if (!pList)
    return RMUINVALIDARG;
  pWalker = &(pList->stWalker);

  hHead = pWalker->hHead;
  pHead = pWalker->pHead;
  if (!pHead || hHead)
    return RMUEMPTYLIST;

  /* list not empty. But if this list was
   * attached rather than created, hTail
   * and pTail will be NULL
   */
  hTail = pWalker->hTail;
  pTail = pWalker->pTail;

  if (hTail && pTail) {
    /* list was created in this process,
     * just move walker to tail
     */
    pWalker->hCurrent = hTail;
    pWalker->pCurrent = pTail;
    return RMUOK;
  } else {
    /* this list was created in another process.
     * We have to traverse the list to move to the end.
     */
    RMULISTWALKER stWalker;
    RMURESULT enuRet;

    enuRet = rmuGetTailPos (pList, &stWalker);
    if (RMUOK == enuRet) {
      pList->stWalker = stWalker;
    }

    return enuRet;

  }
}

RMURESULT
RMUListGoTo (RMULIST * pList, unsigned int unPos)
{
  return RMUNOTIMPLEMENTED;
}

RMURESULT
RMUListPrev (RMULIST * pList)
{
  RMULISTWALKER *pWalker;
  RM_HANDLE hCurrent;
  RMULISTNODE *pCurrent;
  if (!pList)
    return RMUINVALIDARG;
  pWalker = &(pList->stWalker);
  if (RMUListIsEmpty (pList))
    return RMUNOMOREELEMENT;
  if (!pWalker->pCurrent)
    return RMUNOMOREELEMENT;

  hCurrent = pWalker->pCurrent->hPrev;
  if (!hCurrent)
    return RMUNOMOREELEMENT;

  pCurrent = (RMULISTNODE *) RMHandleToPtr (pList->pRMMgr, hCurrent);
  if (!pCurrent)
    return RMUDATACORRUPTED;

  pWalker->hCurrent = hCurrent;
  pWalker->pCurrent = pCurrent;
  return RMUOK;
}

RMURESULT
RMUListNext (RMULIST * pList)
{
  RMULISTWALKER *pWalker;
  RM_HANDLE hCurrent;
  RMULISTNODE *pCurrent;
  if (!pList)
    return RMUINVALIDARG;
  pWalker = &(pList->stWalker);
  if (RMUListIsEmpty (pList))
    return RMUNOMOREELEMENT;
  if (!pWalker->pCurrent)
    return RMUNOMOREELEMENT;

  hCurrent = pWalker->pCurrent->hNext;
  if (!hCurrent)
    return RMUNOMOREELEMENT;

  pCurrent = (RMULISTNODE *) RMHandleToPtr (pList->pRMMgr, hCurrent);
  if (!pCurrent)
    return RMUDATACORRUPTED;

  pWalker->hCurrent = hCurrent;
  pWalker->pCurrent = pCurrent;
  return RMUOK;
}

RMURESULT
RMUListGetData (RMULIST * pList, RM_HANDLE * phData)
{
  RMULISTWALKER *pWalker;
  if (!pList || !phData)
    return RMUINVALIDARG;
  pWalker = &(pList->stWalker);
  if (RMUListIsEmpty (pList))
    return RMUNOMOREELEMENT;
  if (!pWalker->pCurrent)
    return RMUINVALIDPOSITION;
  *phData = pWalker->pCurrent->hData;
  return RMUOK;
}

RM_HANDLE
RMUListGetHead (RMULIST * pList)
{
  if (!pList)
    return 0;
  if (RMUListIsEmpty (pList))
    return 0;
  return pList->stWalker.hHead;
}
