/**
 * $Id$
 *
 * search.c
 *
 * Search・プロバイダ
 * 
 * (note)
 * 	search.c では管理者のみが使用できるSQL(dbmsinformationsearch など) 
 * 	を実行する際も、権限チェックはしていません。
 * 	権限のチェックはここより前のフェーズで実行することに統一するためです。
 *
 * 変更履歴
 *
 * 2003/03/06 Thu nakayama 実装
 * 2003/02/13 Thu takehara NEW
 *
 */
/* Apache header files */
#include "apr.h"
#include "apr_hash.h"
#include "apr_strings.h"
#include "apr_date.h"		/* for conversion of the date */
#include "apr_time.h"
#include "apr_pools.h"
#include "apr_allocator.h"

#include "httpd.h"
#include "http_config.h"
#include "http_protocol.h"
#include "util_md5.h"
#include "mod_dav.h"

/* document management headers */
#include "mod_dav_tf.h"

#include "util.h"
#include "tf_db.h"
#include "util_db.h"
#include "tf_rdbo.h"
#include "tf_rdbo_linkdb.h"
#include "tf_rdbo_root.h"
#include "tf_rdbo_sql.h"
#include "tf_rdbo_sysmsg.h"
#include "tf_rdbo_dbms.h"
#include "tf_rdbo_group.h"
#include "tf_rdbo_user.h"
#include "search.h"
#include "tf_sqlparser.h"
#include "liveprop.h"
#include "util_ml.h"
#include "repos.h"
#include "tf_folder.h"
#include "tf_validator.h"
#include "tf_plugin.h"
#include "tf_rdbo_util.h"
#include "tf_confirmreading.h"
#include "tf_box.h"


/*------------------------------------------------------------------------------
  Define Fixed values and macro
  ----------------------------------------------------------------------------*/

APLOG_USE_MODULE(dav_tf);

/**
 * サポートしているSEARCH の種類
 */
#define DIVY_SEARCH_TYPE_STR_BASIC	"basicsearch"
#define DIVY_SEARCH_TYPE_STR_LINKDB	"linkdbsearch"
#define DIVY_SEARCH_TYPE_STR_MNGLINKDB	"managementlinkdbsearch"
#define DIVY_SEARCH_TYPE_STR_ROOTTREE	"roottreesearch"
#define DIVY_SEARCH_TYPE_STR_USERINFO	"userinformationsearch"
#define DIVY_SEARCH_TYPE_STR_GROUPINFO	"groupinformationsearch"
#define DIVY_SEARCH_TYPE_STR_SQLINFO	"sqlinformationsearch"
#define DIVY_SEARCH_TYPE_STR_DBMSINFO	"dbmsinformationsearch"
#define DIVY_SEARCH_TYPE_STR_UPDATEINFO	"updateinformationsearch"
#define DIVY_SEARCH_TYPE_STR_STATUSINFO	"statusinformationsearch"
#define DIVY_SEARCH_TYPE_STR_SYSMSGINFO	"sysmsginformationsearch"
#define DIVY_SEARCH_TYPE_STR_TRASHINFO	"trashinformationsearch"

/*
 * メール監視機能をサポートしているかどうかを表すフラグ的な意味合いで
 * 導入された仮想的なSEARCHの種類
 */
#define DIVY_SEARCH_TYPE_STR_MAILINFO	"mailinformationsearch"

/*
 * 拡張ユーザステータス機能をサポートしているかどうかを表すフラグ的な意味合いで
 * 導入された仮称的なSEARCHの種類
 */
#define DIVY_SEARCH_TYPE_STR_EXTENDUSERINFO	"extendeduserinformationsearch"

/**
 * サムネイル機能をサポートしているかどうかを示すフラグ的な意味合いで
 * 導入された仮想的なSEARCHの種類
 */
#define DIVY_SEARCH_TYPE_STR_THUMBNAILINFO	"thumbnailinformationsearch"

/**
 * パスワードポリシー機能をサポートしているかどうかを示すフラグ的な意味合いで
 * 導入された仮想的なSEARCHの種類
 */
#define DIVY_SEARCH_TYPE_STR_PASSWORDPOLICYINFO	"passwordpolicyinformationsearch"


/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
typedef struct search_cntxt	search_cntxt;

/**
 * クエリー作成用構造体
 */
struct search_cntxt {

	char *query;		/* 整形後のクエリー 			     */
	char *select;		/* 解析後のselect句	*/
	char *from;		/* 解析後のfrom句	*/
	char *where;		/* 解析後のwhere句	*/
	char *orderby;		/* 解析後のorderby句	*/
	char *limit;		/* 解析後のlimit句	*/

	int optflg;		/* 検索オプションのフラグ 		     */

	/* basicsearch 用 */
	int escflg;		/* % をエスケープするかどうかのフラグ	  */
	apr_hash_t *bs_table;	/* 参照されるテーブル (key=テーブル識別子 / val=tableinfo) */
	divy_cset_t *bs_selprop;	/* selectされたプロパティ                 */ 
	divy_cset_t *bs_notfprop;	/* selectされているがサポートされていない */
	dav_lockdb *bs_lockdb;	/* lock情報取得用 			  */
	apr_hash_t *bs_bind;	/* バインド情報格納			　*/

	/* linkdbsearch 用 */
	char *ldbs_query_sqlc;	/* linkdbsearchのdivy_sqlcontent用クエリー   */
	divy_search_ldbs_bind *ldbs_value;	/* value の値 */
	divy_rdbo_rsvalue *ldbs_rsvalue;	/* required の値 */

	/* sqlinformationsearch 用 */
	apr_hash_t *sqlis_sqldtopid_h;
				/* divy_sqldepend 検索 topid */
	apr_hash_t *sqlis_cachedreq;
				/* クライアントでcache されているreqiredsql  */ 
	divy_rdbo_rsvalue *rsvalue;
				/* querycontent でクライアントから送られてきた
				 * requiredsql 情報 */
	divy_rdbo_sql *sqlis_analysis_sql;
				/* analysis 対象となるSQL 構造体 */
	int newsqlflg;		/* 更新か新規かのフラグ(1:新規) */

	/* ユーザの入力値を保持する構造体を管理する共用体 */
	union {
		/* userinformationsearch に入力された値を保持する構造体へのポインタ */
		divy_search_useris_iscreen *useris_iscreen;

		/* groupinformationsearch に入力された値を保持する構造体へのポインタ */
		divy_search_grpis_iscreen *grpis_iscreen;

		/* sysmsginformationsearch  に入力された値を保持する構造体へのポインタ */
		divy_search_sysmsgis_iscreen *sysmsgis_iscreen;

		/* updateinformationsearch  に入力された値を保持する構造体へのポインタ */
		divy_search_updateis_iscreen *updateis_iscreen;

		/* dbmsinformationsearch  に入力された値を保持する構造体へのポインタ */
		divy_search_dbmsis_iscreen *dbmsis_iscreen;

		/* linkdbsearchsearch  に入力された値を保持する構造体へのポインタ */
		divy_search_ldbs_iscreen *ldbs_iscreen;
	} screen;

	/* 入力パラメータを保持する構造体へのポインタ */
	divy_rdbo_search_params *params;

	/* 検索結果を格納する構造体へのポインタを持つ構造体 */
	divy_rdbo_search_output *output;
};

/**
 * [ 不完全型の定義 ]
 * SEARCH の種類をそれを処理する関数との対応関係を記録する構造体の定義
 */
struct __divy_search_process {
	/*
	 * SEARCH の種類を表す文字列 (define値 DIVY_SEARCH_TYPE_STR_xxx に該当)
	 */
	const char *search_type;

	/*
	 * type が表すSEARCHを処理する関数へのポインタ
	 *
	 * @param r request_rec *
	 * @param stype_elem apr_xml_elem * SEARCH の種類を表すエレメントへのポインタ
	 * @param scntxt search_cntxt * 検索用コンテキスト構造体へのポインタ
	 * @param res dav_response ** 生成したレスポンス
	 * @return int エラーコード. エラーがなければHTTP_OKを返すこと
	 */
	int (*do_process)(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);

#ifdef DAV_SUPPORT_EXTENDED_SEARCH

#define DIVY_SEARCH_RESPONSE_SKIP 3	/* スキップしなさい */
#define DIVY_SEARCH_RESPONSE_NONE 4	/* レスポンスなし */
	/*
	 * レスポンスの作成 (response の中身を埋めて返却する)
	 *
	 * @param r request_rec *
	 * @param out divy_search_output *
	 * @param responce dav_response * レスポンス(アロケート済み)
	 * @param wp apr_pool_t *
	 * @return int ステータス (0: 正常 / 1: 失敗 /
	 * 		DIVY_SEARCH_RESPONSE_SKIP: このレスポンスはスキップしなさい
	 * 		DIVY_SEARCH_RESPONSE_NONE: ない)
	 */
	int (*get_response)(request_rec *r, divy_rdbo_search_output *output,
				dav_response *response, apr_pool_t *wp);
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
};


/*------------------------------------------------------------------------------
  Define array
  ----------------------------------------------------------------------------*/
/**
 * allprop時取得prop名(basicsearch用)
 * "lockdiscovery","supportedlock"以外
 */
static const char *bs_allprop_names[] = {
	"creationdate",
	"displayname",
	"getcontentlanguage",
	"getcontentlength",
	"getcontenttype",
	"getetag",
	"getlastmodified",
	"resourcetype",
	"creator",
	"lastmodifier",
	NULL
};

/**
 * basicsearch において、どのようなカラムが指定されようとも
 * 必ずSELECTしなければならないLiveプロパティを示す。
 */
static const char *bs_indispensable_columns[] = {
	"resourcetype",
	NULL
};

/**
 * linkdbsearch 時のエラーリターン値とエラーコード・メッセージの関係を
 * 持つ構造体の配列の定義
 * 
 * (note)
 * 最初の２つが通常の linkdbsearch 用コード・メッセージ
 * 次の２つが managementlinkdbsearch 用コード・メッセージ
 */
const divy_search_ldb_err search_ldb_errs[] = {
	{ DIVY_LDB_ST_MISMATCH_BINDVAL,
	  "TF-1001",
	  "Inconsistency has arisen between server and client.",
	  "TF-0001",
	  "Illegal number of arguments for bind."
	},
	{ DIVY_LDB_ST_SQL_NOT_SELECT,
	  "TF-1002",
	  "Wrong SQL statement for linkdbsearch.",
	  "TF-0002",
	  "Must specify <SELECT> statement."
	},
	{ DIVY_LDB_ST_NSQL_TOOFEW_COLUMN,
	  "TF-1002",
	  "Wrong SQL statement for linkdbsearch.",
	  "TF-0003",
	  "Too few columns for normalsql."
	},
	{ DIVY_LDB_ST_REPSQL_SYNTAX_ERR,
	  "TF-1002",
	  "Wrong SQL statement for linkdbsearch.",
	  "TF-0004",
	  "Wrong syntax for repositorydbsearch."
	},
	{ DIVY_LDB_ST_REQSQL_WRONG_COLUMN,
	  "TF-1002",
	  "Wrong SQL statement for linkdbsearch.",
	  "TF-0005",
	  "The number of columns is wrong for requiredsql."
	},
	{ DIVY_LDB_ST_MISMATCH_REQVAL,
	  "TF-1001",
	  "Inconsistency has arisen between server and client.",
	  "TF-0006",
	  "Illegal number of arguments for required."
	},
	{ DIVY_LDB_ST_NSQL_TOOMANY_COLUMN,
	  "TF-1002",
	  "Wrong SQL statement for linkdbsearch.",
	  "TF-0009",
	  "Too many columns for normalsql."
	},
	{ DIVY_LDB_ST_NO_BINDVAL,
	  "TF-1003",
	  "The input of conditions is wrong.",
	  "TF-0010",
	  "Empty bindvalue(s) exists."
	},
	{ DIVY_LDB_ST_EMPTY_DBKEY,
	  "TF-1004",
	  "Failed to getting data of linkdb by data mismatching.",
	  "TF-0012",
	  "Failed to getting data of linkdb. Empty dbkey(s) exists."
	},
	{ DIVY_LDB_ST_NO_REQVAL,
	  "TF-1003",
	  "The input of conditions is wrong.",
	  "TF-0013",
	  "Empty requiredvalue(s) exists."
	},
	{ DIVY_LDB_ST_ERR_DBPROVIDER,
	  "TF-1005",
	  "The database provider returns error status.",
	  "",	/* management は DBMS のコード・メッセージを使用します */
	  ""
	},
	{ -1, NULL, NULL, NULL, NULL }      /* sentinel */
};
#define SEARCH_LDB_ERR_LEN      sizeof(search_ldb_errs)/sizeof(divy_search_ldb_err)

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static const divy_search_process * _find_process_handler(const char *search_type);
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
static int _output_reponse(divy_rdbo_search_params *params, divy_rdbo_search_output *output);
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

/* ディスパッチ用SEARCH 処理の実装 */
static int _bs_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);
static int _ldbs_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);
static int _rts_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);
static int _useris_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);
static int _grpis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);
static int _sqlis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);
static int _dbmsis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);
static int _updateis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);
static int _statusis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);
static int _sysmsgis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);
static int _trashis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res);

/* basic SEARCH */
static int bs_build_query(request_rec *r, apr_xml_elem *stype_elem, 
				search_cntxt *scntxt, 
				divy_rdbo_resource *search_result);
static int bs_parse_select(request_rec *r, search_cntxt *scntxt,
		 		apr_xml_elem *select_elem);
static int bs_parse_from(request_rec *r, search_cntxt *scntxt, 
				divy_rdbo_resource *search_result,
				apr_xml_elem *from_elem);
static int bs_parse_where(request_rec *r, search_cntxt *scntxt,
		                                apr_xml_elem *where_elem); 
static int bs_parse_orderby(request_rec *r, search_cntxt *scntxt,
		                apr_xml_elem *orderby_elem);
static int bs_parse_limit(request_rec *r, search_cntxt *scntxt,
		apr_xml_elem *limit_elem);
static void bs_make_sql_from(request_rec *r, search_cntxt *scntxt);
static void _bs_reftbl_hash_set(apr_pool_t *p,
					search_cntxt *scntxt,
					divy_rdbo_tableinfo *tableinfo);
static char * _bs_build_tableident(apr_pool_t *p,
					const char *tablename,
					const char *refname);
static int bs_build_xml_response(request_rec *r, search_cntxt *scntxt,
					divy_rdbo_resource *search_result,
					dav_response **res);
static dav_response *bs_mkresponse(request_rec *r, divy_rdbo_resource *resource,
		                                search_cntxt *scntxt);
/* linkdb SEARCH */
static int ldbs_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt);
static int ldbs_build_xml_response(request_rec *r,
					divy_rdbo_resource *search_result,
					dav_response **res,
					int optflg, int ldbstcode);
static dav_response *ldbs_mkresponse(request_rec *r,
	   			divy_rdbo_resource *resource);
static dav_response *ldbs_repository_mkresponse(request_rec *r,
	   			divy_rdbo_resource *resource);
static dav_response *ldbs_err_mkresponse(request_rec *r,
	   				divy_rdbo_resource *resource, 
					int optflg, int ldbstcode);
/* ルートツリー SEARCH */
static int rts_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
						search_cntxt *scntxt);
static int rts_build_xml_response(request_rec *r, int searchopt,
					divy_rdbo_resource *search_result, 
					dav_response **res);
static dav_response *rts_spfolder_mkresponse(request_rec *r,
					divy_rdbo_resource *resource, 
					const divy_special_folder_spec *fspec);
static dav_response *rts_grp_mkresponse(request_rec *r, divy_rdbo_grp *grp
#ifdef DIVY_SUPPORT_PLUGIN
		, tfs_hset_t *groupid_set
#endif	/* DIVY_SUPPORT_PLUGIN */
		);
static void _rts_build_groupstate(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_text_header *hdr,
					apr_pool_t *wp);
static dav_response *rts_sqlstmt_mkresponse(request_rec *r, divy_rdbo_sql *sql,
						int execflg);
/* ユーザ SEARCH */
static int useris_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
						search_cntxt *scntxt);
static int useris_build_xml_response(request_rec *r,
					divy_rdbo_usr *usr_pr,
					dav_response **res, int optflg);
static void useris_usr_mkresponse(request_rec *r, divy_rdbo_usr *usr, int optflg,
					dav_response *res, apr_pool_t *wp);
static void useris_grp_mkresponse(request_rec *r,
					divy_rdbo_usr *usr, divy_rdbo_grp *grp,
					dav_response *res, apr_pool_t *wp);
static void _useris_build_extstatus(request_rec *r,
					divy_rdbo_usr *usr,
					char *extra_status,
					apr_text_header *hdr,
					apr_pool_t *wp);
static void _useris_build_groupleader(request_rec *r,
					divy_rdbo_usr *usr,
					apr_text_header *hdr,
					apr_pool_t *wp);
static void _useris_build_accesscontrol(request_rec *r,
					divy_rdbo_usr *usr,
					apr_text_header *hdr,
					apr_pool_t *wp);
#ifdef DIVY_SUPPORT_PASSPOLICY
static void _useris_build_passpolicystatus(request_rec *r,
					divy_rdbo_usr *usr,
					char **ret,
					apr_pool_t *wp);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
static void _grpis_build_grp_extstatus(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_text_header *hdr,
					apr_pool_t *wp);
static void _grpis_build_groupleader(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_text_header *hdr,
					apr_pool_t *wp);
static int _useris_get_response(request_rec *r, divy_rdbo_search_output *output,
					dav_response *response, apr_pool_t *wp);
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
#ifdef DIVY_SUPPORT_GROUPQUOTA
static void _grpis_build_groupquota(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_text_header *hdr,
					apr_pool_t *wp);
#endif /* DIVY_SUPPORT_GROUPQUOTA */

static void _grpis_build_uploadpolicy(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_text_header *hdr,
					apr_pool_t *wp);

/* グループ SEARCH */
static int grpis_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
						search_cntxt *scntxt);
static int grpis_build_xml_response(request_rec *r,
					divy_rdbo_grp *grp_pr,
					dav_response **res,
					int optflg);
static void grpis_grp_mkresponse(request_rec *r, divy_rdbo_grp *grp, int optflg,
					dav_response *res, apr_pool_t *wp);
static void grpis_usr_mkresponse(request_rec *r, divy_rdbo_grp *grp, divy_rdbo_usr *usr,
					dav_response *res, apr_pool_t *wp);
static void grpis_sql_mkresponse(request_rec *r, divy_rdbo_grp *grp, divy_rdbo_sql *sql,
					dav_response *res, apr_pool_t *wp);
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
static int _grpis_get_response(request_rec *r, divy_rdbo_search_output *output,
					dav_response *response, apr_pool_t *wp);
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

/* SQL SEARCH */
static int sqlis_build_query(request_rec *r, apr_xml_elem *stype_elem,
					search_cntxt *scntxt);
static int sqlis_build_xml_response(request_rec *r,
					divy_rdbo_resource *search_result,
					dav_response **res, int analysisstcode, 
					int optflg);
static dav_response *sqlis_sql_mkresponse(request_rec *r, 
					divy_rdbo_resource *resource,
					divy_rdbo_sql *sql,
					int optflg);
static dav_response *sqlis_grp_mkresponse(request_rec *r, divy_rdbo_sql *sql, 
					divy_rdbo_grp *grp);
static dav_response *sqlis_reqsqlinfo_mkresponse(request_rec *r, 
					divy_rdbo_resource *resource,
					int optflg);
static int sqlis_exec_analysis(request_rec *r, divy_rdbo_sql *sql_pr,
				search_cntxt *scntxt,
				divy_rdbo_resource **rdb_r);
static dav_response *sqlis_faileddependency_mkresponse(request_rec *r, 
					divy_rdbo_sql *sql);
static dav_response *sqlis_analysiserr_mkresponse(request_rec *r, 
					divy_rdbo_resource *resource,
					int analysisstcode);
/* DBMS情報 SEARCH */
static int dbmsis_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
						search_cntxt *scntxt);
static int dbmsis_build_xml_response(request_rec *r,
					divy_rdbo_dbms *dbms_pr,
					apr_hash_t *lcsdbms_h,
					dav_response **res, int optflg);
static dav_response *dbmsis_lisenceddbms_mkresponse(request_rec *r,
					apr_hash_t *lcsdbms_h);
static dav_response *dbmsis_mkresponse(request_rec *r, divy_rdbo_dbms *dbms, 
					int optflg);
/* 更新クライアント SEARCH */
static int updateis_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
							search_cntxt *scntxt);
static int updateis_build_xml_response(request_rec *r,
					divy_rdbo_clmodule *clmodule_pr,
					dav_response **res, int optflg);
static dav_response *updateis_mkresponse(request_rec *r,
						divy_rdbo_clmodule *clmodule,
						int optflg);
/* 統計情報 SEARCH */
static int statusis_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
    						search_cntxt *scntxt);
static int statusis_build_xml_response(request_rec *r,
					divy_rdbo_usr *usr_pr,
					dav_response **res);
static void statusis_mkresponse(request_rec *r, divy_rdbo_usr *usr, dav_response *res, apr_pool_t *wp);

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
static int statusis_get_response(request_rec *r, divy_rdbo_search_output *output,
					dav_response *response, apr_pool_t *wp);
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

/* システムメッセージ SEARCH */
static int sysmsgis_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
						search_cntxt *scntxt);
static int sysmsgis_build_xml_response(request_rec *r,
					divy_rdbo_sysmsg *sysmsg_pr,
					dav_response **res, int optflg);
static dav_response *sysmsgis_mkresponse(request_rec *r,
					divy_rdbo_sysmsg *sysmsg, int optflg);
/* ごみ箱 SEARCH */
static int trashis_build_xml_response(request_rec *r,
					divy_rdbo_resource *search_result,
					dav_response **res, int optflg);
static dav_response *trashis_mkresponse(request_rec *r,
					divy_rdbo_resource *rdb_r, int optflg);
static int _make_bs_wherecond(request_rec *r, search_cntxt *scntxt,
				apr_xml_elem *cond_elem);
static dav_error * _validate_root_node(request_rec *r, apr_xml_doc *doc);

/**
 * SEARCH の種類をそれを処理する関数との関係を登録する配列
 * (divy_search_process の配列)
 * 第１メンバ(search_type) の並び順はASCIIの昇順になっています。
 */
static const divy_search_process process_array[] = {
	{	DIVY_SEARCH_TYPE_STR_BASIC,
		_bs_do_process
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,NULL
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	},
	{	DIVY_SEARCH_TYPE_STR_DBMSINFO,
		_dbmsis_do_process
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,NULL
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	},
	{	DIVY_SEARCH_TYPE_STR_GROUPINFO,
		_grpis_do_process
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,_grpis_get_response
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	},
	{	DIVY_SEARCH_TYPE_STR_LINKDB,
		_ldbs_do_process
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,NULL
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	},
	{	DIVY_SEARCH_TYPE_STR_MNGLINKDB,
		_ldbs_do_process	/* linkdbsearch と同じ */
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,NULL
#endif /* DAV_SUPPORT_EXTENDED_SEARCH */
	},
	{	DIVY_SEARCH_TYPE_STR_ROOTTREE,
		_rts_do_process
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,NULL
#endif /* DAV_SUPPORT_EXTENDED_SEARCH */
	},
	{	DIVY_SEARCH_TYPE_STR_SQLINFO,
		_sqlis_do_process
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,NULL
#endif /* DAV_SUPPORT_EXTENDED_SEARCH */
	},
	{	DIVY_SEARCH_TYPE_STR_STATUSINFO,
		_statusis_do_process
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,statusis_get_response
#endif /* DAV_SUPPORT_EXTENDED_SEARCH */
	},
	{	DIVY_SEARCH_TYPE_STR_SYSMSGINFO,
		_sysmsgis_do_process
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,NULL
#endif /* DAV_SUPPORT_EXTENDED_SEARCH */
	},
	{	DIVY_SEARCH_TYPE_STR_TRASHINFO,
		_trashis_do_process
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,NULL
#endif /* DAV_SUPPORT_EXTENDED_SEARCH */
	},
	{	DIVY_SEARCH_TYPE_STR_UPDATEINFO,
		_updateis_do_process
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,NULL
#endif /* DAV_SUPPORT_EXTENDED_SEARCH */
	},
	{	DIVY_SEARCH_TYPE_STR_USERINFO,
		_useris_do_process
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		,_useris_get_response
#endif /* DAV_SUPPORT_EXTENDED_SEARCH */
	},
};
#define PROCESS_ARRAY_LEN	sizeof(process_array) / sizeof(divy_search_process)

/*------------------------------------------------------------------------------
  Define Hook functions
  ----------------------------------------------------------------------------*/
/**
 * MIMEヘッダにsupported methodsを設定する
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @return dav_error * mod_dav.hで定義されたerror構造体
 */
static dav_error * dav_divy_set_option_head(request_rec *r)
{
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	char *opt_str;
	unsigned int optbit;
	int support_groupleader = divy_support_groupleader(r);

	apr_table_addn(r->headers_out, "DASL", "<DAV:"DIVY_SEARCH_TYPE_STR_BASIC">");

	/* データベースコラボレーション機能が購入されている場合 */
	if (sconf->use_db_opt) {
		apr_table_addn(r->headers_out, "DASL", 
						DIVY_NS_DEFAULT_URI
						DIVY_SEARCH_TYPE_STR_LINKDB);
		apr_table_addn(r->headers_out, "DASL", 
						DIVY_NS_DEFAULT_URI
						DIVY_SEARCH_TYPE_STR_MNGLINKDB);
	}

	apr_table_addn(r->headers_out, "DASL", 
					DIVY_NS_DEFAULT_URI
					DIVY_SEARCH_TYPE_STR_ROOTTREE);
	apr_table_addn(r->headers_out, "DASL",
					DIVY_NS_DEFAULT_URI
					DIVY_SEARCH_TYPE_STR_USERINFO);
	apr_table_addn(r->headers_out, "DASL",
					DIVY_NS_DEFAULT_URI
					DIVY_SEARCH_TYPE_STR_GROUPINFO);

	/* データベースコラボレーション機能が購入されている場合 */
	if (sconf->use_db_opt) {
		apr_table_addn(r->headers_out, "DASL",
						DIVY_NS_DEFAULT_URI
						DIVY_SEARCH_TYPE_STR_SQLINFO);
		apr_table_addn(r->headers_out, "DASL",
						DIVY_NS_DEFAULT_URI
						DIVY_SEARCH_TYPE_STR_DBMSINFO);
	}

	apr_table_addn(r->headers_out, "DASL",
					DIVY_NS_DEFAULT_URI
					DIVY_SEARCH_TYPE_STR_UPDATEINFO);
	apr_table_addn(r->headers_out, "DASL",
					DIVY_NS_DEFAULT_URI
					DIVY_SEARCH_TYPE_STR_STATUSINFO);
	apr_table_addn(r->headers_out, "DASL",
					DIVY_NS_DEFAULT_URI
					DIVY_SEARCH_TYPE_STR_SYSMSGINFO);

	/* メール監視機能が購入されている場合 */
	if (sconf->use_mail_opt) {

		/* メール送信機能がONであり(= OFFではないとき)、
		 * クライアント主体かハイブリッド型のメール監視の場合 */
		if (IS_FILLED(dconf->mail) && strcmp(dconf->mail, DIVY_ML_OFF) != 0 &&
			(dconf->mlserversend != DIVY_MLSERVERSEND_ON &&
			 dconf->mlserversend != DIVY_MLSERVERSEND_MIDDLE)) {

			/* (note)
			 * 	このヘッダで宣言しているmailinformationsearch は
			 * 	架空のSEARCHです。クライアント側に通知する目的で
			 * 	無理矢理追加したヘッダです。DASL仕様には明らかに
			 * 	違反していますが、これ以外で実現するのは困難なの
			 * 	で、今は目をつぶっておきます。
			 */
			apr_table_addn(r->headers_out, "DASL",
							DIVY_NS_DEFAULT_URI
							DIVY_SEARCH_TYPE_STR_MAILINFO);
		}
	}

	/* ごみ箱が利用でき、アクセス可能なクライアントの時 */
	if (divy_support_trashfolder(r) && divy_enable_trashfolder_access(r)) {
		apr_table_addn(r->headers_out, "DASL",
						DIVY_NS_DEFAULT_URI
						DIVY_SEARCH_TYPE_STR_TRASHINFO);
	}

	/* 拡張ユーザステータス機能がサポートされている時だけヘッダを追加 */
	if (divy_support_extenduserstatus(r)) {
		apr_table_addn(r->headers_out, "DASL",
						DIVY_NS_DEFAULT_URI
						DIVY_SEARCH_TYPE_STR_EXTENDUSERINFO);
	}

	/* サムネイル機能がサポートされている時だけヘッダを追加 */
	if (dconf->thumbnail == DIVY_THUMBNAIL_ON) {
		apr_table_addn(r->headers_out, "DASL",
						DIVY_NS_DEFAULT_URI
						DIVY_SEARCH_TYPE_STR_THUMBNAILINFO);
	}
#ifdef DIVY_SUPPORT_PASSPOLICY
	/*
	 * パスワードポリシー機能が有効であることを示す.
	 * (note) 条件
	 *   パスワードポリシー機能が「サポート」されていれば
	 *   DASLヘッダは付加されます. 「有効/無効」には影響されません.
	 */
	if (divy_support_passpolicy(r)) {
		apr_table_addn(r->headers_out, "DASL",
						DIVY_NS_DEFAULT_URI
						DIVY_SEARCH_TYPE_STR_PASSWORDPOLICYINFO);
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/*
	 * 機能オプションを表すヘッダの追加
	 */

	/* 自分が設定したメール監視状態の監視を表示出来る(1) / 出来ない(0) */
	optbit = DIVY_SUPPORTEDFUNC_OWNERMLWATCH;

	/* グループ制約 */
	if (divy_support_grpconstraints(r)) {
		optbit |= DIVY_SUPPORTEDFUNC_GROUPCONSTRAINTS;
	}

	/* グループリーダ */
	if (support_groupleader) {
		optbit |= DIVY_SUPPORTEDFUNC_GROUPLEADER;
	}

	/* ユーザ種類(特殊) */
	if (divy_support_extenduserstatus(r)) {
		const divy_rdbo_extstatus *extstatus = divy_get_extstatus(r);
		if (extstatus != NULL) {
			if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN) {
				optbit |= DIVY_SUPPORTEDFUNC_UTYPE_ADMIN;
			}
			else if (divy_support_groupleader(r) && divy_rdbo_is_groupleader(extstatus)) {
				optbit |= DIVY_SUPPORTEDFUNC_UTYPE_GROUPLEADER;
			}
			else if (divy_rdbo_is_trusty_user(extstatus)) {
				/* (note) グループリーダもtrustyではあるが、前文で除外されているから
				 * ここに該当するのは一般ユーザだけとなる.
				 * グループ管理者機能が未サポートであれば、こちらになります. */
				optbit |= DIVY_SUPPORTEDFUNC_UTYPE_NORMAL;
			}
			else {
				optbit |= DIVY_SUPPORTEDFUNC_UTYPE_LIMITED;
			}
		}
	}

	/* ファイルticket機能 -> 現時点では未サポートなのでフラグを立てません */

	/* Other ユーザを管理できるかどうか */
	if (support_groupleader) {
		const divy_rdbo_extstatus *extstatus = divy_get_extstatus(r);
		/* グループリーダで、Otherユーザを管理下に置くことができるユーザのみがこれを持つ */
		if (divy_rdbo_is_groupleader(extstatus) && divy_rdbo_has_control_otheruser(extstatus)) {
			optbit |= DIVY_SUPPORTEDFUNC_CONTROLOTHERUSER;
		}
	}

	/* 開封通知機能はサポートしているか? */
	if (divy_support_confirmreading(r)) {
		optbit |= DIVY_SUPPORTEDFUNC_CONFIRMREADING;
	}

	/* グループクォータをサポートしているか? */
	if (divy_support_groupquota(r)) {
		optbit |= DIVY_SUPPORTEDFUNC_GROUPQUOTA;
	}

	/* アクセス制御機能をサポートしているか? */
	if (divy_support_access_control(r)) {
		optbit |= DIVY_SUPPORTEDFUNC_ACCESS_CONTROL;
	}

	/* BOX機能をサポートしているか? */
	if (divy_support_tfbox(r)) {
		optbit |= DIVY_SUPPORTEDFUNC_BOX;
	}

	/* アップロードポリシー機能をサポートしているか? */
	if (divy_support_upload_policy(r)) {
		optbit |= DIVY_SUPPORTEDFUNC_UPLOADPOLICY;
	}

	/* Two Factor Authentication(2FA)機能をサポートしているか? */
	if (divy_support_2FA(r)) {
		optbit |= DIVY_SUPPORTEDFUNC_2FA;
	}

	/* ヘッダへの追加 */
	opt_str = divy_build_supportedfunc_str(r, optbit);
	if (IS_FILLED(opt_str)) {
		apr_table_addn(r->headers_out, DIVY_HEADER_ANNOUNCE, opt_str);
	}

	return NULL;
}

/**
 * DASL SEARCHハンドラ
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @param dav_response ** davのレスポンス格納構造体
 * @return dav_error * mod_dav.hで定義されたerror構造体
 */
static dav_error * dav_divy_search_resource(request_rec *r, 
						dav_response **res)
{
	apr_pool_t *p            = r->pool;
	dav_error *err           = NULL;
	apr_xml_doc *doc         = NULL;
	search_cntxt scntxt      = { 0 };
	apr_xml_elem *stype_elem = NULL;
	const char *search_type  = NULL;
	int ret;
	const divy_search_process *process;
	divy_rdbo_search_params sparams = { 0 };
	divy_rdbo_search_output soutput = { 0 };

	*res = NULL;	/* 初期化 */

	/*
	 * XMLを解析する
	 */
	if ((ret = ap_xml_parse_input(r, &doc)) != 0) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to parse body content for SEARCH. "
			"(code = %d)", ret);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * SEARCH で利用するXMLの先頭要素の検証
	 */
	if ((err = _validate_root_node(r, doc)) != NULL) {
		return err;
	}

	/*
	 * SEARCH処理の振り分けと実行
	 */

	/* 各種情報の取得 */
	search_type = doc->root->first_child->name;
	stype_elem  = doc->root->first_child->first_child;

	/* SEARCH の種類に対応する処理関数を探す */
	process = _find_process_handler(search_type);

	/* 対応するSEARCH の種類が存在しなかった場合 */
	if (process == NULL || process->do_process == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"This type of SEARCH \"%s\" is not supported.", search_type);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* SEARCH コンテキストの初期化 */
	scntxt.optflg = 0;	/* 検索オプション初期化 */
	scntxt.params = &sparams;
	scntxt.output = &soutput;

	/* DB検索入力パラメータを初期化する */
	sparams.r  = r;
	sparams.wp = divy_get_request_temporary_pool(r);
	if (sparams.wp == NULL) sparams.wp = r->pool;
	sparams.scratchpool       = sparams.wp;	/* scratchpool と wp を同じものとする */
	sparams.optflg            = scntxt.optflg;
	sparams.sqlstr            = NULL;
	sparams.is_streaming_mode = 0;	/* 非ストリーミングモード */
	sparams.dwparam           = NULL;
	sparams.process           = NULL;
	sparams.output_response   = NULL;

	/* DB結果格納構造体を初期化する */
	soutput.optflg       = scntxt.optflg;
	soutput.response_cnt = 0;

	/*
	 * SEARCH処理を委譲する
	 */
	ret = process->do_process(r, stype_elem, &scntxt, res);
	if (ret != HTTP_OK && ret != HTTP_NOT_FOUND) {
		return dav_new_error(p, ret, 0, 0, "");
	}
	else if (ret == HTTP_NOT_FOUND) {
		/* Not Found はログに出さない */
		return dav_new_error(p, ret, 0, 0, NULL);
	}

	return NULL;
}

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
/**
 * 指定されたresource のSEARCHが"ストリーミングモード"をサポートしているかどうか。
 *
 * (note) ストリーミングモードとは？
 *     1つのresponseタグ単位で結果出力を行うモード。オリジナルのSEARCHでは全ての
 *     response タグが揃ってから出ないと結果出力を行いません。結果セットが比較的
 *     小さいうちは高速に動作しますが、増えてくるとプロセスメモリの増大に繋がります。
 *     そのようなSEARCHを行いたくない場合には、ストリーミングモードをサポートすべきです。
 *
 * @param r request_rec *
 * @param resource dav_resource * リクエストURIのリソース
 * @return int サポートしているかどうか(1: サポートしている / 0: 未サポート)
 */
static int dav_divy_support_streaming_mode(request_rec *r, dav_resource *resource)
{
	divy_uri_spec *u_spec;

	if (resource->info == NULL || resource->info->rdb_r == NULL ||
		resource->info->rdb_r->u_spec == NULL ||
		resource->info->rdb_r->u_spec->uritype == DIVY_URITYPE_UNKNOWN) {
		return 0;	/* 未サポートとする */
	}
	u_spec = resource->info->rdb_r->u_spec;

	if (u_spec->infotype == DIVY_INFOTYPE_m_status ||
	    u_spec->infotype == DIVY_INFOTYPE_m_user ||
	    u_spec->infotype == DIVY_INFOTYPE_m_group) {
		return 1;
	}

	return 0;
}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
/**
 * parames が示すSEARCHを処理する。
 *
 * @param params const dav_walk_params *
 * @return dav_error * DAVエラー
 */
static dav_error * dav_divy_process_request(const dav_walk_params *params)
{
	dav_error *err      = NULL;
	dav_walker_ctx *ctx = params->walk_ctx;
	request_rec *r      = ctx->r;
	apr_pool_t *p       = r->pool;
	apr_xml_doc *doc    = NULL;
	const char *search_type;
	int ret;
	const divy_search_process *process;
	dav_response *dummy = NULL;
	search_cntxt scntxt    = { 0 };
	divy_rdbo_search_params sparams = { 0 };
	divy_rdbo_search_output soutput = { 0 };

	/* XMLのパース */
	if ((ret = ap_xml_parse_input(r, &doc)) != 0) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to parse body content for SEARCH. (code = %d)", ret);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* SEARCH で利用するXMLの先頭要素の検証 */
	if ((err = _validate_root_node(r, doc)) != NULL) {
		return err;
	}
	ctx->doc = doc;

	search_type = doc->root->first_child->name;

	/* SEARCH の種類に対応する処理関数を探す */
	process = _find_process_handler(search_type);

	/* 対応するSEARCH の種類が存在しなかった場合 */
	if (process == NULL || process->do_process == NULL || process->get_response == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"This type of SEARCH \"%s\" is not supported.", search_type);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* search_cntxt の初期化 */
	scntxt.optflg     = 0;
	scntxt.params = &sparams;
	scntxt.output = &soutput;

	/* DB検索入力パラメータを初期化する */
	sparams.r  = r;
	sparams.wp = divy_get_request_temporary_pool(r);
	if (sparams.wp == NULL) sparams.wp = r->pool;
	sparams.scratchpool       = ctx->scratchpool;
	sparams.optflg            = scntxt.optflg;
	sparams.sqlstr            = NULL;
	sparams.is_streaming_mode = 1;	/* ストリーミングモード */
	sparams.dwparam           = params;
	sparams.process           = process;
	sparams.output_response   = _output_reponse;

	/* DB結果格納構造体を初期化する */
	soutput.optflg       = scntxt.optflg;
	soutput.response_cnt = 0;

	/*
	 * SEARCH処理を委譲する
	 */
	ret = process->do_process(r, doc->root->first_child->first_child, &scntxt, &dummy);
	if (ret != HTTP_OK && ret != HTTP_NOT_FOUND) {
		return dav_new_error(p, ret, 0, 0, "");
	}
	else if (ret == HTTP_NOT_FOUND) {
		/* Not Found はログに出さない */
		return dav_new_error(p, ret, 0, 0, NULL);
	}

	return err;
}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

/*------------------------------------------------------------------------------
  Define Privete functions
  ----------------------------------------------------------------------------*/
/**
 * search_type に一致するSEARCHを処理するハンドラ構造体を探してそのポインタを
 * 返却する。
 * (note)
 * 	返却されたdivy_search_process * の中身を書き換えてはなりません。
 * 	これはstatic な構造体なので書き換えると後々問題を起こします。
 *
 * @param search_type const char * SEARCHの種類(DIVY_SEARCH_TYPE_STR_xxx のこと)
 * @return const divy_search_process *
 * 	search_type に一致するハンドラ構造体がなければNULLを返します。
 * 	search_type が空またはNULLの場合にもNULLを返します。
 */
static const divy_search_process * _find_process_handler(const char *search_type)
{
	const divy_search_process *process = NULL;
	int low, high, mid, ret;
	apr_size_t len = PROCESS_ARRAY_LEN;	/* SEARCH を処理する関数の総数 */

	if (IS_EMPTY(search_type)) return NULL;

	/*
	 * SEARCH の種類に対応する処理関数を探す(binary サーチ)
	 */
	for (low = 0, high = len; low < high; ) {
		mid = (low + high) / 2;
		ret = strcmp(process_array[mid].search_type, search_type);
		if (ret == 0) {
			process = &process_array[mid];
			break;
		}
		else if (ret < 0) {
			low = mid + 1;
		}
		else {
			high = mid;
		}
	}

	return process;
}

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
/**
 * 1レスポンスをストリーム出力する。
 *
 * @param param divy_rdbo_search_params * 入力パラメータ
 * @param output divy_rdbo_search_output * 出力結果
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
static int _output_reponse(divy_rdbo_search_params *params, divy_rdbo_search_output *output)
{
	int ret;
	dav_error *err;
	dav_response response  = { 0 };
	dav_walk_resource wres = { 0 };
	dav_walker_ctx *ctx    = params->dwparam->walk_ctx;

	if (params == NULL || output == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"params or output is NULL.");
	}

	/* レスポンスを生成させる */
	wres.walk_ctx = ctx;
	wres.pool     = params->scratchpool;
	wres.resource = ctx->w.root;
	wres.response = &response;
	wres.isFirst  = (output->response_cnt == 0) ? 1 : 0;

	/* レスポンスを取得する */
	ret = params->process->get_response(params->r, output, wres.response, wres.pool);
	if (ret == DIVY_SEARCH_RESPONSE_NONE) {
		wres.response = NULL;	/* 最後のレスポンス。レスポンスはなしとする */
	}
	else if (ret == DIVY_SEARCH_RESPONSE_SKIP) {
		apr_pool_clear(wres.pool);
		return 0;	/* レスポンス出力はスキップする */
	}

	/* search_walker を呼び出す */
	err = params->dwparam->func(&wres, DIVY_TYPE_COLLECTION);
	if (err != NULL) {
		ERRLOG1(params->r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"An error occured in search walk.(uri = %s)", wres.resource->uri);
		return 1;
	}
	output->response_cnt++;	/* レスポンスカウントを増やす */
	return 0;
}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

/**
 * basicsearch の実装
 * (see process_array)
 */
static int _bs_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_resource *search_result = NULL;
	int result;

	/*
	 * (2005/05/27 Fri)
	 * SQLインジェクション問題を回避できないため、basicsearchはやむなく
	 * 機能を封じます。直ったら元に戻してください。
	 */
	
	ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
		"basicsearch not supported.");
	return HTTP_BAD_REQUEST;

	/* search結果のリソース情報を格納する領域を確保
	 * basicはcatacombの仕様にあわせて先頭は常にデータがある状態
	 * です。そのためここで領域を確保します。
	 * 他はtf_rdbo.cで確保します。 */
	search_result = apr_pcalloc(p, sizeof(*search_result));

	/* クエリー作成 */
	result = bs_build_query(r, stype_elem, scntxt, search_result);
	if (result != HTTP_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"XML parse error.");
		return result;
	}
	ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"sql = %s", REPLACE_NULL(scntxt->query));

	/* DBを検索する */
	if ((divy_rdbo_basicsearch(r, search_result, scntxt->query,
					scntxt->escflg,
					scntxt->bs_bind)) != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"DBMS_SEARCH error");
		return HTTP_INTERNAL_SERVER_ERROR;
	}
	/* データなし */
	else if (search_result->next == NULL) {
		ERRLOG0(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"basicsearch is not found");
		return HTTP_NOT_FOUND;
	}

	/* ストリームモードであればレスポンス出力は行わずに終了 */
	if (scntxt->params->is_streaming_mode) {
		scntxt->output->list.rdb_r = search_result;
		return HTTP_OK;
	}

	/* XML作成 */
	if ((result = bs_build_xml_response(r, scntxt,
						search_result, res) != HTTP_OK)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"An error occurred while building XML response!");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}

/**
 * linkdbsearch, managementlinkdbsearch の実装
 * (see process_array)
 */
static int _ldbs_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_resource *search_result = NULL;
	int result;

	/* XML のパース */
	result = ldbs_parse_xml(r, stype_elem, scntxt);
	if (result != DIVY_LDB_ST_OK) {
		if (result == DIVY_LDB_ST_ERR) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"XML parse error.");
			return HTTP_BAD_REQUEST;
		} 
	}
	scntxt->params->optflg = scntxt->optflg;
	scntxt->output->optflg = scntxt->optflg;
	scntxt->screen.ldbs_iscreen->ldbs_value   = scntxt->ldbs_value;
	scntxt->screen.ldbs_iscreen->ldbs_rsvalue = scntxt->ldbs_rsvalue;
	scntxt->screen.ldbs_iscreen->retcd        = result;

	/*
	 * DBを検索する 
	 * (note)
	 * クエリー作成時にすでに(エラー情報を返却する)エラーとわかっていても、
	 * URIを取得するために検索を行います
	 */
	if ((result = divy_rdbo_linkdbsearch(scntxt->params,
					scntxt->screen.ldbs_iscreen,
					scntxt->output)) != DIVY_LDB_ST_OK) {

		if (result == DIVY_LDB_ST_ERR_BADREQ) {
			return HTTP_BAD_REQUEST;
		}
		else if (result == DIVY_LDB_ST_ERR_IERR) {
			return HTTP_INTERNAL_SERVER_ERROR;
		}
	}
	search_result = scntxt->output->list.rdb_r;

	/* データなし */
	if (search_result == NULL) {
		return HTTP_NOT_FOUND;
	}

	/* ストリームモードであればレスポンス出力は行わずに終了 */
	if (scntxt->params->is_streaming_mode) {
		return HTTP_OK;
	}

	/* XML作成 */
	if ((result = ldbs_build_xml_response(r, search_result, res,
						scntxt->optflg,
						result) != HTTP_OK)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"An error occurred while building XML response!");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}

/**
 * roottreesearch の実装
 * (see process_array)
 */
static int _rts_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_resource *search_result = NULL;
	int result;
	const int *acdeny;

	/* XMLのパース */
	result = rts_parse_xml(r, stype_elem, scntxt);
	if (result != HTTP_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"XML parse error.");
		return result;
	}

	/* ユーザのアクセス権限を取得する */
	acdeny = divy_get_accessdeny(r);

#ifdef DIVY_SUPPORT_PASSPOLICY
	/* パスワードポリシー機能がサポートされていれば、アクセスできるフォルダを制限する.
	 * [ 制限条件 ]
	 *   * 初回ログインしなければ操作が許されないケース
	 *   * 上記の場合、以下へのアクセスを禁止する. (scntxt->optflg)
	 *
	 *   ** グループコレクション一覧 (DIVY_SEARCH_OPT_GROUPLIST)
	 *   ** データベース             (DIVY_SEARCH_OPT_SQLSTMT)
	 *   ** リポジトリ               (DIVY_SEARCH_OPT_REPOSSQLSTMT)
	 *   ** データベーステスト用     (DIVY_SEARCH_OPT_EXECSQLSTMT)
	 *   ** ユーザ一覧               (ここではチェックできない)
	 */
	if (divy_support_passpolicy(r)) {
		dav_error *err;
		divy_uri_spec *u_spec = NULL;
		divy_reqbody_type type = DIVY_REQBODY_TYPE_UNKNOWN;;

		(void) divy_parse_uri(r->pool, dav_divy_get_root_uri(r), r->uri, &u_spec);

		switch (scntxt->optflg) {
			case DIVY_SEARCH_OPT_ALLLIST:
				type = DIVY_REQBODY_TYPE_SEARCH_ALLLIST;
				break;
			case DIVY_SEARCH_OPT_GROUPLIST:
				type = DIVY_REQBODY_TYPE_SEARCH_GROUPLIST;
				break;
			case DIVY_SEARCH_OPT_SQLSTMT:
				type = DIVY_REQBODY_TYPE_SEARCH_SQLSTMT;
				break;
			case DIVY_SEARCH_OPT_REPOSSQLSTMT:
				type = DIVY_REQBODY_TYPE_SEARCH_REPOSSQLSTMT;
				break;
			case DIVY_SEARCH_OPT_EXECSQLSTMT:
				type = DIVY_REQBODY_TYPE_SEARCH_EXECSQLSTMT;
				break;
		}

		/* パスワードの有効期限をチェック */
		err = divy_validate_password_term(r, divy_get_userid(r), u_spec, type);
		if (err != NULL) {
			return err->status;
		}
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* 
	 * DB 検索
	 * [ 検索条件 ]
	 *     * management 以外である (management はDBではなく内部データで生成するため)
	 *     * 表示権限がある
	 */
	if (scntxt->optflg != DIVY_SEARCH_OPT_MANAGEMENT) {

		/* alllist */
		if (scntxt->optflg == DIVY_SEARCH_OPT_ALLLIST &&
			(!acdeny || (acdeny && acdeny[DIVY_FOLDER_ID_user] == 0))) {

			result = divy_rdbo_roottree_alllist(r, &search_result);
			if (result) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get roottree information (alllist).");
				return HTTP_INTERNAL_SERVER_ERROR;
			}
		}
		/* sqlstmt */
		else if (scntxt->optflg == DIVY_SEARCH_OPT_SQLSTMT &&
				 (!acdeny || (acdeny && acdeny[DIVY_FOLDER_ID_dblink] == 0))) {

			result = divy_rdbo_roottree_sql(r, &search_result);
			if (result) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get roottree information (sqlstmt).");
				return HTTP_INTERNAL_SERVER_ERROR;
			}
		}
		/* reposdbsqlstmt */
		else if (scntxt->optflg == DIVY_SEARCH_OPT_REPOSSQLSTMT && 
				 (!acdeny || (acdeny != NULL && acdeny[DIVY_FOLDER_ID_reposdblink] == 0))) {

			result = divy_rdbo_roottree_reposdbsql(r, &search_result);
			if (result) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get roottree information (reposdbsqlstmt).");
				return HTTP_INTERNAL_SERVER_ERROR;
			}
		}
		/* execsqlstmt */
		else if (scntxt->optflg == DIVY_SEARCH_OPT_EXECSQLSTMT &&
				 (!acdeny || (acdeny != NULL && acdeny[DIVY_FOLDER_ID_m_execsql] == 0))) {

			result = divy_rdbo_roottree_execsql(r, &search_result);
			if (result) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get roottree information (execsqlstmt).");
				return HTTP_INTERNAL_SERVER_ERROR;
			}
		}
		/* grouplist */
		else if ( scntxt->optflg == DIVY_SEARCH_OPT_GROUPLIST &&
				 (!acdeny || (acdeny != NULL && acdeny[DIVY_FOLDER_ID_group] == 0))) {

			result = divy_rdbo_roottree_grouplist(r, &search_result);
			if (result) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get roottree information (grouplist).");
				return HTTP_INTERNAL_SERVER_ERROR;
			}
			
		}
		else {
			/* 無視 (通常ケースでもここに来るのでエラー表示も禁止) */
		}

		/* alllist以外のデータ無しは、ここで終了 */
		if (search_result == NULL && scntxt->optflg != DIVY_SEARCH_OPT_ALLLIST) {
			return HTTP_NOT_FOUND;
		}
	}

	/* ストリームモードであればレスポンス出力は行わずに終了 */
	if (scntxt->params->is_streaming_mode) {
		scntxt->output->list.rdb_r = search_result;
		return HTTP_OK;
	}

	/* XML作成 */
	if ((result = rts_build_xml_response(r, scntxt->optflg,
						search_result, res)) != HTTP_OK) {
		/* roottree は構造体からデータを取る場合があるので
		 * ここでも NOT FOUND があります */
		if (result == HTTP_NOT_FOUND) {
			ERRLOG0(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"You have no right to access to this folder.");
			return result;
		}
		else {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"An error occurred while building XML response!");
			return HTTP_BAD_REQUEST;
		}
	}

	return HTTP_OK;
}

/**
 * userinformationsearch の実装
 * (see process_array)
 */
static int _useris_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_usr *usr_pr;
	int result;
#ifdef DIVY_SUPPORT_PASSPOLICY
	divy_rdbo_passwordpolicy *passpolicy_pr = NULL;
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* XML のパース */
	result = useris_parse_xml(r, stype_elem, scntxt);
	if (result != HTTP_OK) {
		return result;
	}
	scntxt->params->optflg = scntxt->optflg;
	scntxt->output->optflg = scntxt->optflg;

#ifdef DIVY_SUPPORT_PASSPOLICY
	/* パスワードポリシー機能がサポートされていれば、アクセスできるフォルダを制限する.
	 * [ 制限条件 ]
	 *   * 初回ログインしなければ操作が許されないケース
	 *   * 上記の場合、以下へのアクセスを禁止する. (scntxt->optflg)
	 *
	 *   ** グループコレクション一覧 (ここではチェックできない)
	 *   ** データベース             (ここではチェックできない)
	 *   ** リポジトリ               (ここではチェックできない)
	 *   ** データベーステスト用     (ここではチェックできない)
	 *   ** ユーザ一覧               (DIVY_SEARCH_OPT_DETAILLIST)
	 *
	 * [ 条件 ]
	 *   * パスワードポリシー機能がサポートされていること
	 *   * システム実行権限を持っていないこと
	 */
	if (divy_support_passpolicy(r) && !divy_rdbo_has_sysexec_privilege(divy_get_extstatus(r))) {
		dav_error *err;
		divy_uri_spec *u_spec = NULL;
		divy_reqbody_type type = DIVY_REQBODY_TYPE_UNKNOWN;;

		(void) divy_parse_uri(r->pool, dav_divy_get_root_uri(r), r->uri, &u_spec);

		if (scntxt->optflg == DIVY_SEARCH_OPT_DETAILLIST) {
			type = DIVY_REQBODY_TYPE_SEARCH_USERDETAILLIST;
		}
		else if ((scntxt->optflg == DIVY_SEARCH_OPT_CONTENT)) {
			type = DIVY_REQBODY_TYPE_SEARCH_USERCONTENT;
		}

		/* パスワードの有効期限をチェック */
		err = divy_validate_password_term(r, divy_get_userid(r), u_spec, type);
		if (err != NULL) {
			return err->status;
		}
	}

	/* パスワードポリシープロパティを取得する(キャッシュさせるだけ) */
	if (divy_get_cached_passwordpolicy(r, DIVY_DEFAULT_POLICY_ID,
										  0, &passpolicy_pr)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to get passwordpolicy property.");
		return HTTP_INTERNAL_SERVER_ERROR;
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* DBを検索する see tf_rdbo_user.h */
	/* availablegroup */
	if (scntxt->params->optflg == DIVY_SEARCH_OPT_AVAILABLEGROUP) {
		result = divy_rdbo_userinformation_availablegroup(scntxt->params,
						scntxt->screen.useris_iscreen, scntxt->output);
	}
	/* treelist */
	else if (scntxt->params->optflg == DIVY_SEARCH_OPT_TREELIST) {
		result = divy_rdbo_userinformation_treelist(scntxt->params,
						scntxt->screen.useris_iscreen, scntxt->output);
	}
	/* content */
	else if (scntxt->params->optflg == DIVY_SEARCH_OPT_CONTENT) {
		result = divy_rdbo_userinformation_content(scntxt->params,
						scntxt->screen.useris_iscreen, scntxt->output);
	}
	/* detail */
	else if (scntxt->params->optflg == DIVY_SEARCH_OPT_DETAILLIST) {
		result = divy_rdbo_userinformation_detaillist(scntxt->params,
						scntxt->screen.useris_iscreen, scntxt->output);
	}

	if (result != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get user information.");
		return HTTP_INTERNAL_SERVER_ERROR;
	}
	usr_pr = scntxt->output->list.usr_pr;

	/* データなし */
	if (usr_pr == NULL) {
		return HTTP_NOT_FOUND;
	}

	/* ストリームモードであればレスポンス出力は行わずに終了 */
	if (scntxt->params->is_streaming_mode) {
		return HTTP_OK;
	}

	/* XML作成 */
	if ((result = useris_build_xml_response(r, usr_pr, res,
						scntxt->optflg) != HTTP_OK)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"An error occurred while building XML response!");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}

/**
 * groupinformationsearch の実装
 * (see process_array)
 */
static int _grpis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res)
{
	apr_pool_t *p         = r->pool;
	divy_rdbo_grp *grp_pr;
	int result;

	/* XML のパース */
	result = grpis_parse_xml(r, stype_elem, scntxt);
	if (result == HTTP_NOT_FOUND && scntxt->query == NULL) {
		/* 以降何もやらなくてよいケース */
		return HTTP_NOT_FOUND;
	}
	/* アクセスチェックに抵触 */
	else if (result == HTTP_FORBIDDEN) {
		return HTTP_FORBIDDEN;	/* 何も出さない */
	}
	else if (result != HTTP_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"XML parse error.");
		return result;
	}

	scntxt->params->optflg = scntxt->optflg;
	scntxt->output->optflg = scntxt->optflg;

	/* DBを検索する see tf_rdbo_group.h */
	/* availableuser */
	if (scntxt->params->optflg == DIVY_SEARCH_OPT_AVAILABLEUSER) {
		result = divy_rdbo_groupinformation_availableuser(scntxt->params,
						scntxt->screen.grpis_iscreen,
						scntxt->output);
	}
	/* availablesql */
	else if (scntxt->params->optflg == DIVY_SEARCH_OPT_AVAILABLESQL) {
		result = divy_rdbo_groupinformation_availablesql(scntxt->params,
						scntxt->screen.grpis_iscreen,
						scntxt->output);
	}
	/* content, detaillist, treelist */
	else {
		result = divy_rdbo_groupinformationsearch(scntxt->params,
						scntxt->screen.grpis_iscreen,
						scntxt->output);
	}
	if (result != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get group information searth.");
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	grp_pr = scntxt->output->list.grp_pr;

	/* データなし */
	if (grp_pr == NULL) {
		return HTTP_NOT_FOUND;
	}

	/* ストリームモードであればレスポンス出力は行わずに終了 */
	if (scntxt->params->is_streaming_mode) {
		return HTTP_OK;
	}

	/* XML作成 */
	if ((result = grpis_build_xml_response(r, grp_pr, res,
						scntxt->output->optflg) != HTTP_OK)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"An error occurred while building XML response!");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}

/**
 * sqlinformationsearch の実装
 * (see process_array)
 */
static int _sqlis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_resource *search_result = NULL;
	int result;

	/* クエリー作成 */
	result = sqlis_build_query(r, stype_elem, scntxt);
	if (result != HTTP_OK) {
		return result;
	}

	/* analysis は特殊 */
	if (scntxt->optflg == DIVY_SEARCH_OPT_ANALYSIS) {
		result = sqlis_exec_analysis(r, scntxt->sqlis_analysis_sql,
						scntxt, &search_result);
		if (result != DIVY_SQLP_ST_OK) {
			if (result == DIVY_SQLP_ST_ERR) {
				/* 予期しないエラー */
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"An error occured while execute analysis.");
				return HTTP_BAD_REQUEST;
			}
			else if (result == DIVY_SQLP_ST_IERR) {
				/* INTERNAL SERVER ERR */
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"An error occured while execute analysis.");
				return HTTP_INTERNAL_SERVER_ERROR;
			}
			else {
				scntxt->optflg = DIVY_SEARCH_OPT_ANALYSIS_SQLERR;
						/* フラグ設定 */
			}
		}
	}
	else {
		ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"sql = %s", REPLACE_NULL(scntxt->query));

		/* DBを検索する */
		result = divy_rdbo_sqlinformationsearch(r, 
						&search_result,
						scntxt->query,
						scntxt->sqlis_sqldtopid_h,
						scntxt->sqlis_cachedreq,
						scntxt->rsvalue,
						scntxt->optflg);

		/* topsql が取得できなかった */
		if (result == DIVY_STCODE_SQLISDEPENDNOTOPURI) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Top sql statement is not found.");
			return HTTP_BAD_REQUEST;
		}
		/* querycontent で Multi Status & Not Found */
		else if (scntxt->optflg == DIVY_SEARCH_OPT_QUERYCONTENT &&
			 result == DIVY_STCODE_SQLISFAILEDDEPENDENCY) {

			/* フラグを設定しなおす */
			scntxt->optflg = DIVY_SEARCH_OPT_QUERYCONTENT_FD;
		}
		/* 通常のエラー */
		else if (result != 0) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"DBMS_SEARCH error");
			return HTTP_INTERNAL_SERVER_ERROR;
		}
		/* データなし */
		else if (search_result == NULL) {
			ERRLOG0(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"sqlinformationsearch is not found");
			return HTTP_NOT_FOUND;
		}
	}

	/* ストリームモードであればレスポンス出力は行わずに終了 */
	if (scntxt->params->is_streaming_mode) {
		scntxt->output->list.rdb_r = search_result;
		return HTTP_OK;
	}

	/* XML作成 analysis 文法エラーはステータスコードを渡す */
	if (scntxt->optflg == DIVY_SEARCH_OPT_ANALYSIS_SQLERR) {
		result = sqlis_build_xml_response(r, search_result, res,
							result, scntxt->optflg);
	}
	else {
		result = sqlis_build_xml_response(r, search_result, res,
							0, scntxt->optflg);

	} 

	if (result != HTTP_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"An error occurred while building XML response!");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}

/**
 * dbmsinformationsearch の実装
 * (see process_array)
 */
static int _dbmsis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_dbms *dbms_pr = NULL;
	int result;
	apr_hash_t *lcsdbms_h = NULL;

	/* XML のパース */
	result = dbmsis_parse_xml(r, stype_elem, scntxt);
	if (result != HTTP_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"XML parse error.");
		return result;
	}
	scntxt->params->optflg = scntxt->optflg;
	scntxt->output->optflg = scntxt->optflg;

	/* licenseddbms は特殊 */
	if (scntxt->optflg == DIVY_SEARCH_OPT_LICENSEDBMS) {

		/* ライセンス有りで使用可能のdbmsを取得 */
		result = divy_db_get_licensed_dbmstype(r, &lcsdbms_h);
		if (result != DB_SUCCESS) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"An error occured while getting lisenced dbms.");
			return HTTP_INTERNAL_SERVER_ERROR;
		}
		dbms_pr = NULL;	/* 使用しない */
	} 
	/* 通常の検索 */
	else {
		/* DBを検索する see tf_rdbo.h */
		if ((result = divy_rdbo_dbmsinformationsearch(scntxt->params,
						scntxt->screen.dbmsis_iscreen,
						scntxt->output)) != 0) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to search dbms information.");
			return HTTP_INTERNAL_SERVER_ERROR;
		}
		dbms_pr = scntxt->output->list.dbms_pr;

		/* データなし */
		if (dbms_pr == NULL) {
			return HTTP_NOT_FOUND;
		}
	}

	/* ストリームモードであればレスポンス出力は行わずに終了 */
	if (scntxt->params->is_streaming_mode) {
		return HTTP_OK;
	}

	/* XML作成 */
	if ((result = dbmsis_build_xml_response(r, dbms_pr, lcsdbms_h,
						res, scntxt->optflg) != HTTP_OK)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"An error occurred while building XML response!");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}

/**
 * updateinformationsearch の実装
 * (see process_array)
 */
static int _updateis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_clmodule *clmodule_pr = NULL;
	int result;

	/* XML のパース */
	result = updateis_parse_xml(r, stype_elem, scntxt);
	if (result != HTTP_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"XML parse error.");
		return result;
	}
	scntxt->params->optflg = scntxt->optflg;
	scntxt->output->optflg = scntxt->optflg;

	/* DBを検索する see tf_rdbo.h */
	if ((result = divy_rdbo_updateinformationsearch(scntxt->params,
						scntxt->screen.updateis_iscreen,
						scntxt->output)) != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to search updated-client information.");
		return HTTP_INTERNAL_SERVER_ERROR;
	}
	clmodule_pr = scntxt->output->list.clmodule_pr;

	/* データなし */
	if (clmodule_pr == NULL) {
		return HTTP_NOT_FOUND;
	}

	/* ストリームモードであればレスポンス出力は行わずに終了 */
	if (scntxt->params->is_streaming_mode) {
		return HTTP_OK;
	}

	/* XML作成 */
	if ((result = updateis_build_xml_response(r, clmodule_pr, res,
						scntxt->optflg) != HTTP_OK)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"An error occurred while building XML response!");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}

/**
 * statusinformationsearch の実装
 * (see process_array)
 */
static int _statusis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res)
{
	apr_pool_t *p = r->pool;
	int result;
	divy_rdbo_usr *usr_pr;
#ifdef DIVY_SUPPORT_PASSPOLICY
	divy_rdbo_passwordpolicy *passpolicy_pr = NULL;
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* XML のパース */
	result = statusis_parse_xml(r, stype_elem, scntxt);
	if (result != HTTP_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"XML parse error.");
		return result;
	}
	scntxt->params->optflg = scntxt->optflg;
	scntxt->output->optflg = scntxt->optflg;

#ifdef DIVY_SUPPORT_PASSPOLICY
	/* パスワードポリシープロパティを取得する(キャッシュさせるだけ) */
	if (divy_get_cached_passwordpolicy(r, DIVY_DEFAULT_POLICY_ID,
										  0, &passpolicy_pr)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to get passwordpolicy property.");
		return HTTP_INTERNAL_SERVER_ERROR;
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */


	/* DBを検索する see tf_rdbo.h */
	if ((result = divy_rdbo_statusinformationsearch(scntxt->params, scntxt->output)) != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to execute statusinformationsearch.");
		return HTTP_INTERNAL_SERVER_ERROR;
	}
	usr_pr = scntxt->output->list.usr_pr;

	/* データなし */
	if (usr_pr == NULL) {
		return HTTP_NOT_FOUND;
	}

	/* ストリームモードであればレスポンス出力は行わずに終了 */
	if (scntxt->params->is_streaming_mode) {
		return HTTP_OK;
	}

	/* XML作成 */
	if ((result = statusis_build_xml_response(r, usr_pr, res) != HTTP_OK)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"An error occurred while building XML response!");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}

/**
 * sysmsginformationsearch の実装
 * (see process_array)
 */
static int _sysmsgis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_sysmsg *sysmsg_pr = NULL;
	int result;

	/* XML のパース */
	result = sysmsgis_parse_xml(r, stype_elem, scntxt);
	if (result != HTTP_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"XML parse error.");
		return result;
	}
	scntxt->params->optflg = scntxt->optflg;
	scntxt->output->optflg = scntxt->optflg;

	/* DBを検索する see tf_rdbo.h */
	if ((result = divy_rdbo_sysmsginformationsearch(scntxt->params, 
						scntxt->screen.sysmsgis_iscreen,
						scntxt->output)) != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to search sysmsg information.");
		return HTTP_INTERNAL_SERVER_ERROR;
	}
	sysmsg_pr = scntxt->output->list.sysmsg_pr;

	/* データなし */
	if (sysmsg_pr == NULL) {
		return HTTP_NOT_FOUND;
	}

	/* ストリームモードであればレスポンス出力は行わずに終了 */
	if (scntxt->params->is_streaming_mode) {
		return HTTP_OK;
	}

	/* XML作成 */
	if ((result = sysmsgis_build_xml_response(r, sysmsg_pr, res,
						scntxt->optflg) != HTTP_OK)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"An error occurred while building XML response!");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}

/**
 * trashinformationsearch の実装
 * (see process_array)
 */
static int _trashis_do_process(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt, dav_response **res)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_resource *search_result = NULL;
	int result;
	const char *root_uri = dav_divy_get_root_uri(r);
	divy_rdbo_extstatus *grp_extstatus = NULL;

	/* ごみ箱が利用出来ない or
	 * アクセス可能なクライアントではない時、Bad Request (外部仕様) */
	if (!divy_support_trashfolder(r) || !divy_enable_trashfolder_access(r)) {
		return HTTP_BAD_REQUEST;
	}

	search_result = apr_pcalloc(p, sizeof(divy_rdbo_resource));

	/* ごみ箱フォルダのURIを算出する */
	result = divy_extract_trashfolder_uri(p, root_uri, r->uri, &search_result->uri);
	if (result || IS_EMPTY(search_result->uri)) {
		return HTTP_BAD_REQUEST;	/* ログを出さない(よく出るので) */
	}

	/* ユーザ拡張ステータスがサポートされている場合にゴミ箱の表示を
	 * 制御します。
	 */
	if (divy_support_extenduserstatus(r)) {
		divy_parse_uri(p, root_uri, search_result->uri, &search_result->u_spec);

		/* 制限ユーザにはグループゴミ箱を見せない */
		if (!divy_rdbo_is_trusty_user(divy_get_extstatus(r))) {
			if (search_result->u_spec->infotype == DIVY_INFOTYPE_group_trash) {
				return HTTP_NOT_FOUND;	/* 無い事にする */
			}
		}

		/* BOX機能が有効なグループのゴミ箱は教えない */
		if (search_result->u_spec
				&& search_result->u_spec->p_infotype == DIVY_INFOTYPE_group_e) {
			(void)divy_get_cached_availablegroupextstatus(r, search_result->uri,
														  		&grp_extstatus);
			if (divy_rdbo_is_box_group(grp_extstatus)) {
				return HTTP_NOT_FOUND; /* 無い事にする */
			}
		}
	}

	/* DBを検索する(特殊なプロパティは不要) */
	result = divy_rdbo_get_hierarchy_property(r, search_result, 0, 0, NULL);
	if (result) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get trash-folder properties.");
		return HTTP_INTERNAL_SERVER_ERROR;
	} 
	/* データなし */
	else if (IS_EMPTY(search_result->rsid)) {
		return HTTP_NOT_FOUND;
	}

	/* ストリームモードであればレスポンス出力は行わずに終了 */
	if (scntxt->params->is_streaming_mode) {
		scntxt->output->list.rdb_r = search_result;
		return HTTP_OK;
	}

	/* XML作成 */
	if ((result = trashis_build_xml_response(r, search_result, res,
						scntxt->optflg) != HTTP_OK)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"An error occurred while building XML response!");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}



/**
 * XMLからクエリーを作成する(basicsearch)
 *
 * @param  request_rec * apache request_rec構造体
 * @param  stype_elem apr_xml_elem * 解析済みのXML
 * @param  search_cntxt * 作成されたクエリーを格納する
 * @param  divy_rdbo_resource *検索結果
 * @return int HTTPコード
 */
static int bs_build_query(request_rec *r, apr_xml_elem *stype_elem,
		search_cntxt *scntxt, divy_rdbo_resource *search_result)
{
	apr_pool_t *p			= r->pool;
	apr_xml_elem *cur_elem 		= stype_elem;
	apr_xml_elem *select_elem 	= NULL;
	apr_xml_elem *from_elem 	= NULL;
	apr_xml_elem *where_elem 	= NULL;
	apr_xml_elem *orderby_elem	= NULL;
	apr_xml_elem *limit_elem 	= NULL;
	apr_xml_elem *elem;
	int result;

	if (stype_elem == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Requested search grammar not supported."
			" We support only <basicsearch>.");
		return HTTP_BAD_REQUEST;
	}

	/* Find elements */
	/* 
	 * 上記で格納したbasicsearchからnsがDAV、nameが[select]/[from]など
	 * のchildを探して格納する。
	*/
	for (elem = cur_elem; elem; elem = elem->next) {
		if (elem->ns == APR_XML_NS_DAV_ID) {
			if (strcmp(elem->name, "select") == 0) {
				select_elem  = elem;
			}
			else if (strcmp(elem->name, "from") == 0) {
				from_elem    = elem;
			}
			else if (strcmp(elem->name, "where") == 0) {
				where_elem   = elem;
			}
			else if (strcmp(elem->name, "orderby") == 0) {
				orderby_elem = elem;
			}
			else if (strcmp(elem->name, "limit") == 0) {
				limit_elem   = elem;
			}
		}
	}

	/* EEEEEエラーチェック */
	if (select_elem == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"We don't have <select> element.");
		return HTTP_BAD_REQUEST;
	}

	/* 参照されるテーブル,選択されたプロパティ,選択されたがNOT　FOUNDで返す
	 * プロパティのハッシュ作成 */
	scntxt->bs_table    = apr_hash_make(p);
	scntxt->bs_selprop  = divy_cset_make(p);
	scntxt->bs_notfprop = divy_cset_make(p);
    
	/* 
	 * select要素を解析
	 */
	if ((result = bs_parse_select(r, scntxt, select_elem)) != HTTP_OK){
		return result;
	}
	/* EEEEEエラーチェック */
	if (!from_elem){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"We don't have <from> element.");
		return HTTP_BAD_REQUEST;
	}

	/*
	 * from要素を解析
	 */
	if ((result = bs_parse_from(r, scntxt, search_result, from_elem))
								!= HTTP_OK){
		return result;
	}
    
	/* where要素があれば AND を付加し解析する */
	if (where_elem != NULL){
        
		/* EEEEEエラーチェック */
		if (!where_elem->first_child){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"We have <where> element without child.");
			return HTTP_BAD_REQUEST;
		}

		if (IS_FILLED(scntxt->where)) {
		       scntxt->where = apr_pstrcat(p, scntxt->where, 
				       			" AND", NULL);
		} else {
			scntxt->where = apr_pstrdup(p, " AND");
		}

		if ((result = bs_parse_where(r, scntxt, 
					where_elem->first_child)) != HTTP_OK){
			return result;
		}
	}

	/* orderby要素があれば解析する */
	if (orderby_elem != NULL){
		/* EEEEEエラーチェック */
		if (!orderby_elem->first_child){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"We have <orderby> element without child.");
			return HTTP_BAD_REQUEST;
		}
		if ((result = bs_parse_orderby(r, scntxt, orderby_elem))
								!= HTTP_OK){
			return result;
		}
	}

	/* limit要素があれば解析する */
	if (limit_elem != NULL){
		/* EEEEEエラーチェック */
		if (!limit_elem->first_child){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"We have <limit> element without child.");
			return HTTP_BAD_REQUEST;
		}
		if ((result = bs_parse_limit(r, scntxt, limit_elem)) 
							!= HTTP_OK){
			return result;
		}
	}

	/* SQLのFROM(whereの一部含む)句を作成する */
	bs_make_sql_from(r, scntxt);

	/* 各要素を連結 */
	scntxt->query = apr_psprintf(p, "SELECT "
					DIVY_RESOURCE_TABLE_REF".rs_uri_txt,%s"
					" FROM %s"
					" WHERE %s",
					scntxt->select, 
					scntxt->from, scntxt->where);
/* ##### TODO ここのIFブロックの db2 はまだテストが終了していません */
	/* orderbyとlimitはあれば追加 */
	if (orderby_elem != NULL && limit_elem != NULL){
		scntxt->query = apr_pstrcat(p,
					scntxt->query, 
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_DB2) /* postgres / db2 */ 
					scntxt->orderby, 
					scntxt->limit,
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
					" AND",
					scntxt->limit,
					scntxt->orderby,
#endif
					NULL);
	}
	else if (orderby_elem != NULL){
		scntxt->query = apr_pstrcat(p, 
					scntxt->query, scntxt->orderby, NULL);
	}
	else if (limit_elem != NULL){
		scntxt->query = apr_pstrcat(p,
					scntxt->query,
#if defined(DIVY_DBMS_ORACLE) /* oracle */
					" AND",
#endif
					scntxt->limit, NULL);
	}
	return HTTP_OK;
}

/**
 * XMLのselect要素を解析する(basicsearch)
 * (note) select のDTD
 * -----------------------------------------------------------
 * <!ELEMENT select        (allprop | prop) >
 * -----------------------------------------------------------
 *
 * @param  request_rec * apache request_rec構造体
 * @param  search_cntxt * 作成されたクエリーを格納する
 * @param  select_elem apr_xml_elem * select要素XML
 * @return int HTTPコード
 */
static int bs_parse_select(request_rec *r, search_cntxt *scntxt,
		 apr_xml_elem * select_elem)
{
	apr_pool_t *p		= r->pool;
	apr_xml_elem * cur_elem = NULL;
	divy_sbuf *select_buf   = NULL;
	int i;
	divy_rdbo_tableinfo *curpropinfo;	/* see tf_rdbo.h */
	const dav_hooks_locks *hooks = dav_get_lock_hooks(r);
    
	/* 
	 * prop | allprop取得・EEEEEない場合はエラー
	 */
	if (select_elem->first_child){
		cur_elem = select_elem->first_child;
	} else {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"No select element.");
		return HTTP_BAD_REQUEST;
	}
    
	/* EEEEEエラーチェック prop | allprop取得 が空の場合 */
	if (IS_EMPTY(cur_elem->name)){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"No select element name.");
		return HTTP_BAD_REQUEST;
	}

	/* select 句組み立て用バッファの作成 */
	divy_sbuf_create(p, &select_buf, 1024);
	scntxt->select = NULL;

	if (strcmp(cur_elem->name, "allprop") == 0){

		/* 取得するprop数分ループ */
		for (i = 0; bs_allprop_names[i]; i++){

			/* 
			 * カラム名、テーブル名を取得する
			 */
			curpropinfo = divy_rdbo_get_tableinfo(r,
							bs_allprop_names[i]);

			/* 1つ目以外はカンマ挿入 */
			if (i == 0){
				divy_sbuf_appendbyte(select_buf, 1, " ");
			} else {
				divy_sbuf_appendbyte(select_buf, 2, ", ");
			}
	    
			/* select句作成 */
			divy_sbuf_append(select_buf, curpropinfo->refname);
			divy_sbuf_appendbyte(select_buf, 1, ".");
			divy_sbuf_append(select_buf, curpropinfo->colname);

			/* 参照するテーブル・選択されたプロパティを
			 * ハッシュのキーに追加 */
			_bs_reftbl_hash_set(p, scntxt, curpropinfo);

			/* prop で出力するプロパティ一覧に追加 */
			divy_cset_set(scntxt->bs_selprop, bs_allprop_names[i]);
		}
		/* lockプロバイダが存在する時のみlock関連のプロパティを追加 */
		if (hooks != NULL){
			/* lockdiscovery */
			divy_cset_set(scntxt->bs_selprop, "lockdiscovery"); 
			/* supportedlock */
			divy_cset_set(scntxt->bs_selprop, "supportedlock");
		}

	} else if (strcmp(cur_elem->name, "prop") == 0){
		divy_cset_t *indispensable_set;

		/* EEEEEエラーチェック:指定なし又は文法誤り */
		if (!cur_elem->first_child ||
				cur_elem->first_child->first_child){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"No prop information.");
			return HTTP_BAD_REQUEST;
		}

		/* 必ずSELECTしなければならないプロパティの集合を生成 */
		indispensable_set = divy_cset_make(p);
		for (i = 0; bs_indispensable_columns[i] != NULL; i++) {
			divy_cset_set(indispensable_set,
						bs_indispensable_columns[i]);
		}

		i = 0;
		for (cur_elem = cur_elem->first_child; cur_elem; 
					cur_elem = cur_elem->next){
			/* カラム名、テーブル名を取得する */
			if (!(curpropinfo = divy_rdbo_get_tableinfo(r,
							cur_elem->name))){
				/* 取得できなかったがロックプロバイダが
				 * サポートされかつlockdiscoveryまたは
				 * supportedlockの場合 */
				if (hooks != NULL && 
					(strcmp(cur_elem->name, 
						"lockdiscovery") == 0 ||
					strcmp(cur_elem->name, 
						"supportedlock") == 0)) {
					/* 選択されたプロパティに追加 */
					divy_cset_set(scntxt->bs_selprop,
							cur_elem->name);
				} else {
					/* 選択されたがNOT　FOUND */
					divy_cset_set(scntxt->bs_notfprop,
							cur_elem->name);
				}
			}
			/* カラム名、テーブル名を取得できた場合 */
			else {
				/* 1つ目以外はカンマ挿入 */
				if (i == 0){
					divy_sbuf_appendbyte(select_buf, 1, " ");
					i = 1;
				} else {
					divy_sbuf_appendbyte(select_buf, 2, ", ");
				}

				/* select句作成 */
				divy_sbuf_append(select_buf, curpropinfo->refname);
				divy_sbuf_appendbyte(select_buf, 1, ".");
				divy_sbuf_append(select_buf, curpropinfo->colname);

				/* 参照するテーブル,プロパティを
				 * ハッシュのキーに追加 */
				_bs_reftbl_hash_set(p, scntxt, curpropinfo);

				/* prop で出力するプロパティ一覧に追加 */
				divy_cset_set(scntxt->bs_selprop, cur_elem->name);

				/* 必須集合から除去 */
				divy_cset_remove(indispensable_set, cur_elem->name);
			}
		}

		/* 必須項目はまだ残っているか ? */
		if (divy_cset_count(indispensable_set) > 0) {
			char **names = divy_cset_toarray(indispensable_set);
			for ( ; *names != NULL; names++) {
				curpropinfo = divy_rdbo_get_tableinfo(r, *names);

				if (i == 0) {
					divy_sbuf_appendbyte(select_buf, 1, " ");
					i = 1;
				}
				else {
					divy_sbuf_appendbyte(select_buf, 2, ", ");
				}

				/* select句作成 */
				divy_sbuf_append(select_buf, curpropinfo->refname);
				divy_sbuf_appendbyte(select_buf, 1, ".");
				divy_sbuf_append(select_buf, curpropinfo->colname);

				/* 参照するテーブル,プロパティを
				 * ハッシュのキーに追加 */
				_bs_reftbl_hash_set(p, scntxt, curpropinfo);
				/* prop で出力する必要はないので何もしない */
			}
		}
	} else {
		/* EEEEEエラーチェック：DTD違反 */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Unkonw element name(%s) in select."
				 "Use <allprop> or <prop>.",
				 REPLACE_NULL(select_elem->name));
		return HTTP_BAD_REQUEST;
	}

	/* 文字列バッファから文字列への変換 */
	scntxt->select = divy_sbuf_tostring(select_buf);

	return HTTP_OK;
}

/**
 * XMLのfrom要素を解析する(basicsearch)
 * (note) from のDTD
 * -----------------------------------------------------------
 * <!ELEMENT from	(scope+) >
 * <!ELEMENT scope	(href, depth, include-versions?) >
 * <!ELEMENT include-versions EMPTY >
 * -----------------------------------------------------------
 *
 * @param  request_rec * apache request_rec構造体
 * @param  search_cntxt * 作成されたクエリーを格納する
 * @param  from_elem apr_xml_elem * from要素XML
 * @return int HTTPコード
 */
static int bs_parse_from(request_rec *r, search_cntxt *scntxt,
				divy_rdbo_resource *search_result, 
				apr_xml_elem *from_elem)
{
	apr_pool_t *p		= r->pool;
	apr_xml_elem * cur_elem = NULL;
	const char *href 	= NULL;
	const char *depth 	= NULL;
	divy_rdbo_resourcetype urirestype;
	int depthi;

	/* fromにchild(scope)のchild(href)がある場合はそこをカレントにする
	 * ない場合はエラー */
    	if (from_elem->first_child && from_elem->first_child->first_child){
		cur_elem = from_elem->first_child->first_child;
	} else {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"scope\"(\"from\" child) element is missing.");
		return HTTP_BAD_REQUEST;
	}

	/*
	 * href の解析
	 */

	/*  hrefからuriの条件取得(なしの場合はエラー) */
	href = divy_xml_get_cdata(cur_elem, p, 1); 
	if (IS_EMPTY(href)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"href\"(\"scope\" child) element "
			"is missing or empty.");
		return HTTP_BAD_REQUEST;
	}

	/* If it is full URL, we need only path
	 * 取得データがURLの場合(ap_is_urlで0以外/apr_is_url関数はなし)
	 */
	if (ap_is_url(href)){
		apr_uri_t uptr;
		/* URL解析(解析できない、解析後のpathがない場合はエラー) */
		if (apr_uri_parse(p, href, &uptr) != APR_SUCCESS ||
							!uptr.path){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Malformed a value of \"href\" element.");
			return HTTP_BAD_REQUEST;
		}
		/* search結果のURIにpathを格納 */
		search_result->uri = apr_pstrdup(p, uptr.path);
	}
	/* URLでなく'/'で開始の場合はそのまま格納 */
	else if (*href == '/'){	/* absolute href */
		search_result->uri = apr_pstrdup(p, href);
	}
	/* URLでなく'/'で開始しない場合はエラー */
	else {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"href\" element must be full URL "
			"or absolute URI.");
		return HTTP_BAD_REQUEST;
	}

	/* URI最後の'/'を除去する see util_common.h*/
	search_result->uri = dav_divy_remove_endslash(p, search_result->uri);

	/* resourcetypeを取得 see tf_rdbo.h 見つからない場合はNOT FOUND */
	urirestype = divy_rdbo_get_resourcetype(r, search_result);
	if (urirestype == DIVY_TYPE_UNKNOWN){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get resource information.");
		return HTTP_NOT_FOUND;
	}
    
	/*
	 * depth の解析
	 */
	/* カレント要素(href)から次の要素(depth)に移動、ない場合はエラー */
	if (cur_elem->next) {
		cur_elem = cur_elem->next;
	} else {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"depth\"(\"scope\" child) element is missing.");
		return HTTP_BAD_REQUEST;
	}
    
	/* depthを取得 */
	depth = divy_xml_get_cdata(cur_elem, p, 1);
	if (urirestype == DIVY_TYPE_RESOURCE) {
		depthi = 0;	/* リソースならば常にdepth = 0 */
	}
	else if (IS_FILLED(depth) && strcmp(depth, "0") == 0) {
		depthi = 0;
	}
	else if (IS_FILLED(depth) && strcmp(depth, "1") == 0) {
		depthi = 1;
	}
	else {
		depthi = DAV_INFINITY;
	}

	/* 
	 * depth/ファイルタイプによって異なるwhere句を作成する
	 */
	/* depthが0 (自身のみ)またはuriのリソースタイプがファイルの場合 */
	if (depthi == 0) {
		scntxt->where = apr_pstrdup(p,
				"("DIVY_RESOURCE_TABLE_REF".rs_uri_txt = ?)");
		scntxt->escflg = 0;	/* エスケープしない */
	}
	else if (depthi == 1) {
		/* uriの中の'/'をカウントし、+1した位置をdepthに設定 */
		search_result->depth = divy_count_dirs(search_result->uri);
		scntxt->where = apr_psprintf(p,
				"("DIVY_RESOURCE_TABLE_REF".rs_uri_txt = ? "
				"OR ("
				DIVY_RESOURCE_TABLE_REF".rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND "
				DIVY_RESOURCE_TABLE_REF".rs_depth_i = '%d'))",
				search_result->depth + 1);
		scntxt->escflg = 1;	/* エスケープする */
	}
	else {
		scntxt->where = apr_pstrdup(p,
				"("DIVY_RESOURCE_TABLE_REF".rs_uri_txt = ? "
				"OR "
				DIVY_RESOURCE_TABLE_REF".rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE")");
		scntxt->escflg = 1;	/* エスケープする */
	}
	/* lockdiscoveryが選択されたプロパティにあった場合 */
	if (divy_cset_contains(scntxt->bs_selprop, "lockdiscovery")) {
		int ret;
		/* uri,depthを取得した時点で全てのロック情報を取得します
         	 * see lock.c*/
		if ((ret = dav_divy_open_lockdb_private
					(r, &(scntxt->bs_lockdb), 
					search_result->uri, depthi, 0)) != 0){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get lockinfo.");
			return HTTP_BAD_REQUEST;
		}
	}

	/*
	 * include-versions の解析
	 */
	if (cur_elem->next) {
		/* include-versions はバージョニングをサポートするまで未サポート */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"include-versions\" element is not supported.");
		return HTTP_UNPROCESSABLE_ENTITY;
	}

	return HTTP_OK;
}

/**
 * XMLのwhere要素を解析する(basicsearch)
 *
 * (note)
 * 	再帰呼び出しによってchildをたどりながらwhere条件を作成します。
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  search_ctx * 作成したクエリーを格納する構造体
 * @param  apr_xml_elem * 作成元となるXML
 * @return int HTTPコード
 */
static int bs_parse_where(request_rec *r, search_cntxt *scntxt, 
				apr_xml_elem *select_elem)
{
	apr_pool_t *p		= r->pool;
	char *op 		= NULL;
	apr_xml_elem *cur_elem 	= select_elem;
	int result;

	/* 渡されたXMLの1番外のchildのname(演算子)を取得 */
	op = apr_pstrdup(p, cur_elem->name);

	/* 渡されたXMLの1番外のchildのnameが"and"の場合 */
	if (strcmp(op, "and") == 0){
		int andstat = 0;
		/* EEEEEエラーチェック andのchildが２つ以上ない */
		if (!cur_elem->first_child || !cur_elem->first_child->next){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<and> element has too few child.");
			return HTTP_BAD_REQUEST;
		}

		scntxt->where = apr_pstrcat(p, scntxt->where, " (", NULL);
    
		for(cur_elem = cur_elem->first_child; cur_elem;
					cur_elem = cur_elem->next){
			if (andstat == 0){
				bs_parse_where(r, scntxt, cur_elem);
				andstat = 1;
			} else {
				scntxt->where = apr_pstrcat(p, 
						scntxt->where, " AND", NULL);
				bs_parse_where(r, scntxt, cur_elem);
			}
		}

		scntxt->where = apr_pstrcat(p, scntxt->where, " )", NULL);
	}
	/* 渡されたXMLの1番外のchildのnameが"or"の場合 */
	else if (strcmp(op, "or") == 0){
		int orstat = 0;
		/* EEEEEエラーチェック orのchildが２つ以上ない */
		if (!cur_elem->first_child || !cur_elem->first_child->next){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<or> element has too few child.");
			return HTTP_BAD_REQUEST;
		}

		scntxt->where = apr_pstrcat(p, scntxt->where, " (", NULL);

		for(cur_elem = cur_elem->first_child; cur_elem;
					cur_elem = cur_elem->next){
	 		if (orstat == 0){
				bs_parse_where(r, scntxt, cur_elem);
				orstat = 1;
			} else {
				scntxt->where = apr_pstrcat(p, 
						scntxt->where, " OR", NULL);
				bs_parse_where(r, scntxt, cur_elem);
			}
		}
		scntxt->where = apr_pstrcat(p, scntxt->where, " )", NULL);
	}
	/* 渡されたXMLの1番外のchildのnameが"not"の場合 */
	else if (strcmp(op, "not") == 0){
		/* "not"のchildが存在する場合は子へ移動して再帰
		 * 存在しない場合はエラー */
		if (cur_elem->first_child){
			/* EEEEEエラーチェック　notのchildが１つ以上 */
			if (cur_elem->first_child->next){
				ERRLOG0(p, APLOG_ERR, 
					DIVY_FST_IERR + DIVY_SST_DATA,
					"<not> element has too many child.");
				return HTTP_BAD_REQUEST;
			}
	    
			scntxt->where = apr_pstrcat(p, scntxt->where, 
							" NOT (", NULL);
			bs_parse_where(r, scntxt, cur_elem->first_child);
			scntxt->where = apr_pstrcat(p, scntxt->where, " )",NULL);
		} else {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"We have <not> element without child.");
			return HTTP_BAD_REQUEST;
		}
	}
	/* 渡されたXMLの1番外のchildのnameが比較演算子の場合 */
	else if (strcmp(op, "lt") == 0 || strcmp(op, "lte") == 0 || 
		 strcmp(op, "gt") == 0 || strcmp(op, "gte") == 0 || 
		 strcmp(op, "eq") == 0 || strcmp(op, "like") == 0){
		/* バインド変数用構造体とSQL文作成 */
		if ((result = _make_bs_wherecond(r, scntxt, cur_elem)) 
				!= HTTP_OK){
			/* エラーログ出力済み */
			return result;
		}
	}
	/* 渡されたXMLの1番外のchildのnameが"is-collection"場合 */
	else if (strcmp(op, "is-collection") == 0){
		/* コレクションのみを検索 */
		scntxt->where = apr_pstrcat(p, scntxt->where,
				" ("DIVY_RESOURCE_TABLE_REF
				".rs_resourcetype_i = 1)", NULL);
	}
	/* 渡されたXMLの1番外のchildのnameがサポートされていない場合 */
	else if (strcmp(op, "is-defined") == 0 ||
		 strcmp(op, "contains") == 0 ||
		 strcmp(op, "language-defined") == 0 ||
		 strcmp(op, "language-matches") == 0) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"\'%s\' is not supported.", op);
		return HTTP_UNPROCESSABLE_ENTITY;
	}
	/* BAD REQUEST */
	else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"\'%s\' is unknown operator for basicsearch.", op);
		return HTTP_BAD_REQUEST;
	}
	return HTTP_OK;
}

/**
 * XMLのorderby要素を解析する(basicsearch)
 *
 * (note) orderby のDTD
 * -----------------------------------------------------------
 * <!ELEMENT orderby       (order+) >
 * <!ELEMENT order         ((prop | score), (ascending | descending)?)
 * <!ATTLIST order         caseless   (yes|no) >
 * <!ELEMENT ascending     EMPTY>
 * <!ELEMENT descending    EMPTY>
 * -----------------------------------------------------------
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  search_ctx * 作成したクエリーを格納する構造体
 * @param  apr_xml_elem * 作成元となるXML
 * @return int HTTPコード
 */
static int bs_parse_orderby(request_rec *r, search_cntxt *scntxt,
		                                apr_xml_elem *orderby_elem)
{
	apr_pool_t *p		= r->pool;
    	apr_xml_elem *cur_elem 	= NULL;
	divy_rdbo_tableinfo *ordertblinfo = NULL;
	int pauseflg = 0; 
    
	/* 
	 * XMLのchild(order)数分間ループ
	 */
	for (cur_elem = orderby_elem->first_child;
		cur_elem; cur_elem = cur_elem->next){
    
		/* caselessはサポート外 */
		if (cur_elem->attr){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"caseless\" attribute is not supported.");
			return HTTP_UNPROCESSABLE_ENTITY;
		}
		/* EEEEEエラーチェック DTD違反*/
		if (strcmp(cur_elem->name, "order") != 0){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<orderby> elem needs <order>.");
			return HTTP_BAD_REQUEST;
		}
		if (!cur_elem->first_child || 
			strcmp(cur_elem->first_child->name, "prop") != 0){
			/* サポート外 */
			if (strcmp(cur_elem->first_child->name, "score") == 0){
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The \"score\" element is not supported.");
				return HTTP_UNPROCESSABLE_ENTITY;
			}

			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<order> elem needs <prop>.");
			return HTTP_BAD_REQUEST;
		}
		if (cur_elem->first_child->next->next){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<order> element has too many child.");
			return HTTP_BAD_REQUEST;
		}

		/* EEEEEエラーチェック　propのchildがない、または2つ以上 */
		if (!cur_elem->first_child->first_child || 
				cur_elem->first_child->first_child->next){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"<prop> should have one child in <orderby>.");
			return HTTP_BAD_REQUEST;
		}

		/* カラム名、テーブル名を取得する(取得できなかったらエラー) */
		if (!(ordertblinfo = divy_rdbo_get_tableinfo(r, 
				cur_elem->first_child->first_child->name))){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"<order> element has wrong prop information.");
			return HTTP_BAD_REQUEST;
		}
		/* 参照するテーブルをハッシュのキーに追加 */
		_bs_reftbl_hash_set(p, scntxt, ordertblinfo);

		/* 1回目の場合 */
		if (pauseflg == 0){
			/* orderby句に" ORDER BY "を出力 */
			scntxt->orderby = apr_pstrdup(p, " ORDER BY ");
			pauseflg = 1;
		} else {
			/*それ以外の場合はカンマ追加 */
			scntxt->orderby = apr_pstrcat(p, scntxt->orderby,
							", ", NULL);
		}
		/* テーブル名・カラム名追加 */
		scntxt->orderby = apr_pstrcat(p, scntxt->orderby,
						ordertblinfo->refname, ".",
						ordertblinfo->colname, NULL);
		/* 
		 * 昇順/降順 
		 */
		/* 指定ありの場合 */
		if (cur_elem->first_child->next){
			if (strcmp(cur_elem->first_child->next->name,
							"ascending") == 0){
				scntxt->orderby = apr_pstrcat(p,
							scntxt->orderby,
							" ASC", NULL);
			} else if (strcmp(cur_elem->first_child->next->name,
							"descending") == 0){
				scntxt->orderby = apr_pstrcat(p,
							scntxt->orderby,
							" DESC", NULL);
			} else {
				ERRLOG0(p, APLOG_ERR, 
					DIVY_FST_IERR + DIVY_SST_DATA,
					"In <order>, next of <prop> "
					"must be <ascending> or <descending>.");
		}
		/* 指定なしの場合は昇順 */
		} else {
			scntxt->orderby = apr_pstrcat(p, scntxt->orderby, 
							" ASC", NULL);
		}
    
	}
	return HTTP_OK;
}

/**
 * XMLのlimit要素を解析する(basicsearch)
 * 
 * (note) orderby のDTD
 * -----------------------------------------------------------
 * <!ELEMENT limit         (nresults) >
 * <!ELEMENT nresults      (#PCDATA) >
 * -----------------------------------------------------------
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  search_ctx * 作成したクエリーを格納する構造体
 * @param  apr_xml_elem * 作成元となるXML
 * @return int HTTPコード
 */
static int bs_parse_limit(request_rec *r, search_cntxt *scntxt,
				apr_xml_elem *limit_elem)
{
	apr_pool_t *p		=r->pool;
	apr_xml_elem *cur_elem 	= NULL;
	const char *limit;
    
	cur_elem = limit_elem->first_child;

	/* EEEEEエラーチェック DTD違反*/
	if (strcmp(cur_elem->name, "nresults") != 0 ||
			cur_elem->first_child || cur_elem->next){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"<limit> elem should have one child <nresults>.");
		return HTTP_BAD_REQUEST;
	}
	/* リミットの値を取得 */
	limit = divy_xml_get_cdata(cur_elem, p, 1);

/* ##### TODO 
 * ここのIFブロックの db2 はまだテストが終了していません 
 */
	if (IS_FILLED(limit)) {
		/* limit句を作成 */
		scntxt->limit = apr_psprintf(p, 
#if defined(DIVY_DBMS_POSTGRES)	/* postgres */
				" LIMIT %s", 
#elif defined(DIVY_DBMS_DB2)	/* db2 */
				" FETCH FIRST %s ROWS ONLY",
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
				" ROWNUM <= %s",
#endif
				limit);
	}

	return HTTP_OK;
}

/**
 *  SQLのFROM(WHEREの一部を含む)句を作成する(basicsearch)
 *
 * @param  request_rec * apache request_rec構造体
 * @param  search_ctx * 作成したクエリーを格納する構造体
 */
static void bs_make_sql_from(request_rec *r, search_cntxt *scntxt)
{
	apr_pool_t *p		= r->pool;
	apr_hash_index_t *hashind;
	divy_rdbo_tableinfo *tableinfo;
	const char *tableident;
	const char *res_ident;
	char *jointype_str;
#if defined(DIVY_DBMS_ORACLE) /* oracle */
	char *wherestr		= "";
#endif

	scntxt->from = "";	/* 初期化 */

	/* 必須テーブルの識別子の生成 */
	res_ident = _bs_build_tableident(p, DIVY_RESOURCE_TABLE,
					DIVY_RESOURCE_TABLE_REF);
    
	/* 参照テーブルのハッシュの最後までループ */
	for (hashind = apr_hash_first(p, scntxt->bs_table); hashind;
					hashind = apr_hash_next(hashind)) {
		/* 値を取得 */
		apr_hash_this(hashind, (const void **)&tableident,
				NULL,  (void **)&tableinfo);

		/* tableinfo は必須テーブルのものか? */
		if (IS_FILLED(tableident) &&
					strcmp(tableident, res_ident) == 0) {
			/* 後で付加します */
			continue;
		}

		/* JOIN 句の生成 */
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_DB2) /* postgres / db2 */
		switch (tableinfo->jointype) {
			case DIVY_JOIN_INNER:
				jointype_str = "INNER JOIN"; break;
			case DIVY_JOIN_LEFT:
				jointype_str = "LEFT JOIN"; break;
			default:
				continue;	/* 何もする必要なし */
		}

		/* (例) XXXX LEFT JOIN divy_usr u1 ON u1.xxx = rs.yyy */
		scntxt->from = apr_psprintf(p, " %s %s %s ON %s = %s",
					scntxt->from,
					jointype_str, tableident,
					tableinfo->l_joinkey,
					tableinfo->r_joinkey);
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
		scntxt->from = apr_psprintf(p,	"%s, %s",
						scntxt->from, tableident);

		/* 1件目の場合以外は where 条件に AND を付加 */
		if (IS_FILLED(wherestr)) {
			wherestr = apr_psprintf(p, "%s AND", wherestr);
		}

		switch (tableinfo->jointype) {
			case DIVY_JOIN_INNER:
				jointype_str = ""; break;
			case DIVY_JOIN_LEFT:
				jointype_str = "(+)"; break;
			default:
				continue;	/* 何もする必要なし */
		}

		/* (例) XXXX u1.xxx = rs.yyy(+) */
		wherestr = apr_psprintf(p, "%s %s = %s%s",
					wherestr,
					tableinfo->l_joinkey,
					tableinfo->r_joinkey,
					jointype_str);
#endif
	}

	/* scntxt->from があればfirst_from と連結する */
	if (IS_FILLED(scntxt->from)) {
		scntxt->from = apr_pstrcat(p, res_ident, scntxt->from, NULL); 
	}
	else {
		scntxt->from = (char *) res_ident;
	}

#if defined(DIVY_DBMS_ORACLE) /* oracle */
	if (IS_FILLED(wherestr) && IS_FILLED(scntxt->where)) {
		scntxt->where = apr_pstrcat(p, wherestr, " AND ",
						scntxt->where, NULL);
	}
#endif
	return;
}

/**
 * basicsearch で検索するテーブル名を記録する。(basicsearch 専用)
 *
 * @param p apr_pool_t * 作業用プール
 * @param search_cntxt search_cntxt * 記録するハッシュを保持するコンテナ
 * @param tableinfo divy_rdbo_tableinfo * テーブル情報を持つ構造体
 */
static void _bs_reftbl_hash_set(apr_pool_t *p,
					search_cntxt *scntxt,
					divy_rdbo_tableinfo *tableinfo)
{
	char *key;

	if (tableinfo == NULL || scntxt == NULL) return;	/* 何も出来ない */

	key = _bs_build_tableident(p, tableinfo->tablename, tableinfo->refname);

	/* 参照するテーブルをハッシュに追加 */
	apr_hash_set(scntxt->bs_table, key, APR_HASH_KEY_STRING, tableinfo);
}


/**
 * テーブル情報からFROM句に指定されるテーブル識別子を作成する。(basicsearch)
 *
 * @param p apr_pool_t * 文字列が割り当てられるプール
 * @param tablename const char * テーブル名
 * @param refname const char * 参照名
 */
static char * _bs_build_tableident(apr_pool_t *p,
					const char *tablename,
					const char *refname)
{
	return apr_psprintf(p, "%s %s", tablename, refname);
}

/**
 * 検索結果から返信するXMLを作成する(basicsearch)
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @param  search_cntxt * クエリー格納構造体(lock関連の有無フラグを使用)
 * @param  divy_rdbo_resource *検索結果
 * @paramr dav_response ** davのレスポンス格納構造体
 * @return int HTTPコード
 */
static int bs_build_xml_response(request_rec *r, search_cntxt *scntxt,
					divy_rdbo_resource *search_result, 
					dav_response **res)
{
	apr_pool_t *p		= r->pool;
	divy_rdbo_resource *resource;
	dav_response *newres 	= NULL;
	dav_response *firstres 	= NULL;
    
	*res = NULL;

	/* 
	 * 検索結果の2件目から最後までループ
	 * 2件目開始なのはcatacombと同じ仕様です。
	 */
	for (resource = search_result->next; resource; 
				resource = resource->next){
		/* プールをセット */
		resource->p = r->pool;

		/* responseを１件作成 */
		if ((newres = bs_mkresponse(r, resource, scntxt)) == NULL) {
			return HTTP_BAD_REQUEST;
		}

		if (firstres == NULL) {
			firstres = *res = newres;
		}
		else {
			(*res)->next = newres;
			*res = newres;
		}
		(*res)->next = NULL;
	}
	/* lockdiscoveryが選択されたプロパティにあった場合 */
	if (divy_cset_contains(scntxt->bs_selprop, "lockdiscovery")) {
		int ret;

		/* 取得したロック情報を開放 see lock.c*/
		if ((ret = dav_divy_close_lockdb_private(scntxt->bs_lockdb)) 
								!= 0){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to close lockinfo.");
			return HTTP_BAD_REQUEST;
		}
	}
    
	/* responceの先頭を返却 */
	*res = firstres;
    
	return HTTP_OK;
}

/**
 * 検索結果からsearchのresponceを作成する(basicsearch)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_resource * 検索結果
 * @param  search_cntxt * クエリー格納構造体(選択されたプロパティを使用)
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *bs_mkresponse(request_rec *r, 
					divy_rdbo_resource *resource, 
					search_cntxt *scntxt)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	int divynsflg 		= 0;
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	divy_cset_index_t *ci;
	const char *prop;
	const char *s 		= NULL;
	char *datebuf 		= NULL;
	char *tmp;
  
	res->status = HTTP_OK;
	if (resource->resourcetype == DIVY_TYPE_COLLECTION) {
		res->href = dav_divy_make_uri(p, resource->uri, "/", NULL);
	}
	else {
		res->href = apr_pstrdup(p, resource->uri);
	}

	/* 
	 * XML作成 
	 */
	apr_text_append(p, &hdr, "<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF);

	/* selectされたプロパティの最後までループ */    
	for (ci = divy_cset_first(scntxt->bs_selprop); ci;
						ci = divy_cset_next(ci)) {
		s 	= NULL;
		datebuf = NULL;

		/* 選択されたプロパティを取得する */
		divy_cset_this(ci, &prop);

		/* プロパティにより分岐 */
		if (strcmp(prop, "creationdate") == 0){
			divy_format_time_t(p, resource->creationdate, 
					DIVY_TIME_STYLE_ISO8601, &datebuf);
			if (IS_FILLED(datebuf)) {
				s = apr_psprintf(p, "<"DAV_NS_PREFIX":%s>"
					"%s"
					"</"DAV_NS_PREFIX":%s>" CRLF,
					prop, datebuf, prop);
			}
			else {
				s =  apr_psprintf(p,
					"<"DAV_NS_PREFIX":%s/>" CRLF, prop);
			}
		} else if (strcmp(prop, "displayname") == 0){
			s = apr_psprintf(p, "<"DAV_NS_PREFIX":%s>"
					"%s"
					"</"DAV_NS_PREFIX":%s>" CRLF,
					prop, 
					dav_divy_escape_xmlstr(p, 
						resource->displayname, 
						DIVY_XML_T2T_QUOTE), 
					prop);
		} else if (strcmp(prop, "getcontentlanguage") == 0){
			if (IS_FILLED(resource->getcontentlanguage)) {
				s = apr_psprintf(p, "<"DAV_NS_PREFIX":%s>"
					"%s"
					"</"DAV_NS_PREFIX":%s>" CRLF,
					prop, 
					resource->getcontentlanguage,
					prop);
			}
			else {
				s = apr_psprintf(p,
					"<"DAV_NS_PREFIX":%s/>" CRLF, prop);
			}
		} else if (strcmp(prop, "getcontentlength") == 0){
			s = apr_psprintf(p, "<"DAV_NS_PREFIX":%s>"
					"%"APR_INT64_T_FMT
					"</"DAV_NS_PREFIX":%s>" CRLF,
					prop, 
					resource->getcontentlength, 
					prop);
		} else if (strcmp(prop, "getcontenttype") == 0){
			if (IS_FILLED(resource->getcontenttype)) {
				s = apr_psprintf(p, "<"DAV_NS_PREFIX":%s>"
					"%s"
					"</"DAV_NS_PREFIX":%s>" CRLF,
					prop, resource->getcontenttype, prop);
			}
			else {
				s = apr_psprintf(p,
					"<"DAV_NS_PREFIX":%s/>" CRLF, prop);
			}
		} else if (strcmp(prop, "getetag") == 0){
			if (IS_FILLED(resource->getetag)) {
				s = apr_psprintf(p, "<"DAV_NS_PREFIX":%s>"
					"%s"
					"</"DAV_NS_PREFIX":%s>" CRLF,
					prop, resource->getetag, prop);
			}
			else {
				s = apr_psprintf(p,
					"<"DAV_NS_PREFIX":%s/>" CRLF, prop);
			}
		} else if (strcmp(prop, "getlastmodified") == 0){
			divy_format_time_t(p, resource->getlastmodified, 
					DIVY_TIME_STYLE_RFC822, &datebuf);
			if (IS_FILLED(datebuf)) {
				s = apr_psprintf(p, "<"DAV_NS_PREFIX":%s>"
					"%s"
					"</"DAV_NS_PREFIX":%s>" CRLF,
					prop, datebuf, prop);
			}
			else {
				s = apr_psprintf(p,
					"<"DAV_NS_PREFIX":%s/>" CRLF, prop);
			}
		} else if (strcmp(prop, "resourcetype") == 0){
			/* リソースタイプはコレクションの場合のみ設定 */
			if (resource->resourcetype == DIVY_TYPE_COLLECTION){
				s = apr_psprintf(p, "<"DAV_NS_PREFIX":%s>"
						"<"DAV_NS_PREFIX":collection/>"
						"</"DAV_NS_PREFIX":%s>" CRLF,
						prop, prop);
			}
			else {
				s = apr_psprintf(p,
					"<"DAV_NS_PREFIX":%s/>" CRLF, prop);
			}
		} else if (strcmp(prop, "lockdiscovery") == 0){
			/* ロック情報取得 see lock.h */    
			if ((s = dav_divy_lock_get_activelock
					(scntxt->bs_lockdb,
						 resource->uri)) == NULL){
				ERRLOG0(p, APLOG_ERR, 
						DIVY_FST_IERR + DIVY_SST_PROC,
						"An error occured "
						"while getting lockinfo.");
				return NULL;
			}
			s = apr_pstrcat(p, s, CRLF, NULL);
	    
		} else if (strcmp(prop, "supportedlock") == 0){
			tmp = (char *) dav_divy_get_exclusivelock_string();
			if (IS_FILLED(tmp)) {
				s = apr_psprintf(p, "<"DAV_NS_PREFIX":%s>"
					"%s"
					"</"DAV_NS_PREFIX":%s>" CRLF, 
					prop, tmp, prop);
			}
			else {
				s = apr_psprintf(p,
					"<"DAV_NS_PREFIX":%s/>" CRLF, prop);
			}
		} else if (strcmp(prop, "creator") == 0){
			if (IS_FILLED(resource->creator)) {
				s = apr_psprintf(p, "<"DIVY_NS_PREFIX":%s>"
					"%s"
					"</"DIVY_NS_PREFIX":%s>" CRLF,
			    		prop,
					dav_divy_escape_xmlstr(p,
						resource->creator, 
						DIVY_XML_T2T_CDATA), prop);
			}
			else {
				s = apr_psprintf(p,
					"<"DIVY_NS_PREFIX":%s/>" CRLF, prop);
			}
			/* divyのネームスペース取得 */
			divynsflg = 1;
		} else if (strcmp(prop, "lastmodifier") == 0){
			if (IS_FILLED(resource->lastmodifier)) {
				s = apr_psprintf(p, "<"DIVY_NS_PREFIX":%s>"
					"%s"
					"</"DIVY_NS_PREFIX":%s>" CRLF,
					prop, 
					dav_divy_escape_xmlstr(p,
						resource->lastmodifier,
						DIVY_XML_T2T_CDATA), prop);
			}
			else {
				s = apr_psprintf(p,
					"<"DIVY_NS_PREFIX":%s/>" CRLF, prop);
			}
			/* divyのネームスペース取得 */
			divynsflg = 1;
		}

		/* ヘッダに追加 */
		apr_text_append(p, &hdr, s);
	}

	/* クローズ */
	apr_text_append(p, &hdr, "</"DAV_NS_PREFIX":prop>" CRLF
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF 
			"</"DAV_NS_PREFIX":propstat>" CRLF);

	/* selectされたが見つからないプロパティありの場合 NOT FOUNDで追加 */
	if (scntxt->bs_notfprop && divy_cset_count(scntxt->bs_notfprop) > 0){
		/* 開始 */
		apr_text_append(p, &hdr, "<"DAV_NS_PREFIX":propstat>" CRLF 
				"<"DAV_NS_PREFIX":prop>" CRLF);
	    
		for (ci = divy_cset_first(scntxt->bs_notfprop); ci;
						ci = divy_cset_next(ci)) {
			/* 対象プロパティを取得する */
			divy_cset_this(ci, &prop);

			/* XML作成し追加 */
			s = apr_psprintf(p, "<"DAV_NS_PREFIX":%s/>" CRLF, prop);
			apr_text_append(p, &hdr, s);
		}

		/* クローズ */
		apr_text_append(p, &hdr, "</"DAV_NS_PREFIX":prop>" CRLF
			    "<"DAV_NS_PREFIX":status>"
			    "HTTP/1.1 404 Not Found"
			    "</"DAV_NS_PREFIX":status>" CRLF
			    "</"DAV_NS_PREFIX":propstat>" CRLF);
	}

	/* ネームスペースを追加する */
	/* divyが必要な場合 */
	if (divynsflg == 1) {
		apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));
	}
	else {
		apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS));
	}

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;

	return res;

}

/**
 * XMLをパースし、バインド変数のハッシュを作成し、パース結果を返却する (linkdbsearch)
 *
 * @param  request_rec * apache request_rec構造体
 * @param  stype_elem apr_xml_elem * 解析済みのXML
 * @param  search_cntxt * 作成されたクエリー,ハッシュを格納する
 * @return int linkdbsearch エラーコード -> see search.h
 */
static int ldbs_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
				search_cntxt *scntxt)
{
	apr_pool_t *p 		= r->pool;
	divy_uri_spec *u_spec	= NULL;
	apr_xml_elem *cur_elem  = stype_elem, *elem, *elem2;
	const char *cdata, *cdata2, *sqlno = NULL;
	divy_rdbo_rsvalue *rsval = NULL;
	divy_rdbo_rsvalueset *rsvals	= NULL;
	divy_search_ldbs_bind *ldbs_value = NULL;
	char *decode;
	int ret, errcode = DIVY_LDB_ST_OK;

	/* エラーチェック */
	if (stype_elem == NULL || stype_elem->first_child != NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Element of <linkdbsearch> is wrong.");
		return DIVY_LDB_ST_ERR;
	}

	/* 入力値を格納するscreen を作成 */
	scntxt->screen.ldbs_iscreen = apr_pcalloc(p, sizeof(divy_search_ldbs_iscreen));
	scntxt->optflg = 0;

	/* 初期化 */
	scntxt->ldbs_value   = NULL;
	scntxt->ldbs_rsvalue = NULL;

	/* 
	 * uri をパースしてどんな目的で実行されたか設定する
	 */
	if (divy_parse_uri(p, dav_divy_get_root_uri(r), r->uri, &u_spec)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse uri. (uri = %s)", r->uri);
		return DIVY_LDB_ST_ERR;
	}
	switch (u_spec->infotype) {
		case DIVY_INFOTYPE_dblink:
			/* 通常のlinkdbsearch */
			scntxt->optflg = DIVY_SEARCH_LDB_NORMAL;
			break;
		case DIVY_INFOTYPE_reposdblink:
			/* リポジトリ検索 */
			scntxt->optflg = DIVY_SEARCH_LDB_REPOSITORY;
			break;
		case DIVY_INFOTYPE_m_execsql:
			/* SQLの試し実行 */
			scntxt->optflg = DIVY_SEARCH_LDB_MLKDB;
			break;
		default:
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"This type of uri was not supported for linkdbsearch. "
				"(uri = %s)", r->uri);
			return DIVY_LDB_ST_ERR;
	}

	/*
	 * XMLのパース & 検証
	 *
	 * [ DTD ]
	 * 	<!ELEMENT linkdbsearch ( sqlno, value*, rsvalue* ) >
	 * 	<!ELEMENT sqlno (#PCDATA) >
	 * 	<!ELEMENT value (#PCDATA) >
	 * 	<!ATTLIST value id ID #REQUIRD >
	 * 	<!ELEMENT rsvalue ( reqsqlname, reqsqlvalue* ) >
	 * 		<!ELEMENT reqsqlname    (#PCDATA) >
	 * 		<!ELEMENT reqsqlvalue   (#PCDATA) >
	 * 		<!ATTLIST rsvalue id ID #IMPLIED >
	 */
	for (elem = cur_elem; elem; elem = elem->next) {

		if (strcmp(elem->name, "sqlno") == 0) {
			cdata = divy_xml_get_cdata(elem, p, 1);
			sqlno = cdata;

			/* エラーが発生すると、任意の箇所でreturn してしまうため、
			 * sqlno の整合性は予め検証しておく必要があります */
			if (IS_EMPTY(sqlno)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The value of \"sqlno\" element is EMPTY.");
				return DIVY_LDB_ST_ERR;
			}
			/* sqlno は数値でなければならない */
			else if (IS_FILLED(sqlno) && !dav_divy_isdigit_str(sqlno)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The value of \"sqlno\" element must be digit. "
					"(sqlno = %s)", sqlno);
				return DIVY_LDB_ST_ERR;
			}
			scntxt->screen.ldbs_iscreen->sqlid = sqlno;

			/* エラーが見つかっていたら? */
			if (errcode != DIVY_LDB_ST_OK) {
				/* sqlno が取得できたのでもう用はない */
				return errcode;
			}
		}
		else if (strcmp(elem->name, "value") == 0) {

			/* id 取得 */
			cdata = elem->attr->value;
			if (IS_EMPTY(cdata)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"id\" attribute for \"value\" element is missing.");
				return DIVY_LDB_ST_ERR;
			}
			else if (!dav_divy_isdigit_str(cdata)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The value of \"id\" attribute must be digit.");
				return DIVY_LDB_ST_ERR;
			}

			/* value 取得. 入力値をそのまま取得(dav関数使用) */
			cdata2 = dav_xml_get_cdata(elem, p, 0);
			if (IS_EMPTY(cdata2)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The value of \"value\" element is EMPTY.(id = %s)", cdata);
				/* ステータス返却 */
				errcode = DIVY_LDB_ST_NO_BINDVAL;
				/* sqlno が見つかっていなければ続ける */
				if (IS_EMPTY(sqlno)) continue;
				return errcode;
			}

			/* value をURL デコードする */
			ret = dav_divy_unescape_uri(p, cdata2, &decode);
			if (ret != 0) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to URL decode. (id = %s, value = %s)",
					cdata, cdata2);
				return DIVY_LDB_ST_ERR;
			}

			/* ハッシュにセット(key: id / value: url-decoded value) */
			if (scntxt->ldbs_value == NULL) {
				scntxt->ldbs_value = ldbs_value = apr_pcalloc(p, sizeof(divy_search_ldbs_bind));
			}
			else {
				ldbs_value->next = apr_pcalloc(p, sizeof(divy_search_ldbs_bind));
				ldbs_value = ldbs_value->next;
			}
			ldbs_value->id    = atoi(cdata);
			ldbs_value->value = decode;
			ldbs_value->next  = NULL;
		}
		else if (strcmp(elem->name, "rsvalue") == 0) {

			if (scntxt->ldbs_rsvalue == NULL) {
				scntxt->ldbs_rsvalue = rsval = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
			}
			else {
				rsval->next = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
				rsval = rsval->next;
			}
			rsval->next = NULL;

			/* id 取得 */
			cdata = elem->attr->value;
			if (IS_EMPTY(cdata)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"id\" attribute for \"rsvalue\" element is missing.");
				return DIVY_LDB_ST_ERR;
			}
			else if (!dav_divy_isdigit_str(cdata)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The value of \"id\" attribute must be digit.");
				return DIVY_LDB_ST_ERR;
			}
			rsval->id = atoi(cdata);	/* id を記録 */

			/* 次のエレメントが無いのはエラー */
			if (elem->first_child == NULL) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"reqsqlname\" element is missing.");
				return DIVY_LDB_ST_ERR;
			}

			rsval->valueset = NULL;
			rsvals = NULL;
			for (elem2 = elem->first_child; elem2; elem2 = elem2->next) {

				if (strcmp(elem2->name, "reqsqlname") == 0) {
					/* (note) reqsqlname は1つのrsvalue につき
					 * 1つしか存在し得ないことが外部仕様で保証されています */

					cdata = divy_xml_get_cdata(elem2, p, 1);
					if (IS_EMPTY(cdata)) {
						ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"The value of \"reqsqlname\" is EMPTY.");
						return DIVY_LDB_ST_ERR;
					}

					/* URL デコード */
					ret = dav_divy_unescape_uri(p, cdata, &decode);
					if (ret != 0){
						ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
							"Failed to URL decode. (rsname = %s)", cdata);
						return DIVY_LDB_ST_ERR;
					}
					rsval->name = decode;	/* 名前を記録 */
				}
				else if (strcmp(elem2->name, "reqsqlvalue") == 0) {

					/* reqsqlvalue は入力値をそのまま取得 (dav関数使用) */
					cdata = dav_xml_get_cdata(elem2, p, 0);
					if (IS_EMPTY(cdata)) {
						if (IS_FILLED(rsval->name)) {
							ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
								"The value of \"reqsqlvalue\" is EMPTY."
								"(rsname = %s)", rsval->name);
						}
						else {
							ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
								"The value of \"reqsqlvalue\" is EMPTY.");
						}
						/* ステータス返却 */
						errcode = DIVY_LDB_ST_NO_REQVAL;
						/* sqlno が見つかっていなければ続ける */
						if (IS_EMPTY(sqlno)) continue;
						return errcode;
					}

					/* URL デコード */
					ret = dav_divy_unescape_uri(p, cdata, &decode);
					if (ret != 0) {
						ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
							"Failed to URL decode. (rsvalue = %s)", cdata);
						return DIVY_LDB_ST_ERR;
					}

					if (rsval->valueset == NULL) {
						rsval->valueset = rsvals =
							apr_pcalloc(p, sizeof(divy_rdbo_rsvalueset));
					}
					else {
						rsvals->next = apr_pcalloc(p, sizeof(divy_rdbo_rsvalueset));
						rsvals = rsvals->next;
					}
					rsvals->value = decode;
					rsvals->next = NULL;
				}
			}

			/* reqsqlname が存在しない */
			if (IS_EMPTY(rsval->name)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"reqsqlname\" element is missing.");
				return DIVY_LDB_ST_ERR;
			}

			/* reqsqlvalue が１つもない場合はエラー */
			if (rsval->valueset == NULL) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The \"reqsqlvalue\" element is missing."
					"(reqsqlname = %s)", rsval->name);
				/* ステータス返却 */
				errcode = DIVY_LDB_ST_NO_REQVAL;
				/* sqlno が見つかっていなければ続ける */
				if (IS_EMPTY(sqlno)) continue;
				return errcode;
			}
		}
		else {
			/* 未知のエレメントが指定された */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"linkdbsearch", elem->name);
			return DIVY_LDB_ST_ERR;
		}
	}

	/*
	 * 取得結果の検証
	 */
	/* 具体的なSEARCH動作を示す指示エレメントが存在しなかった */
	if (scntxt->optflg == 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The request element of linkdbsearch not found.");
		return DIVY_LDB_ST_ERR;
	}
	/* sqlno は必須 */
	else if (IS_EMPTY(sqlno)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sqlno\" element is missing.");
		return DIVY_LDB_ST_ERR;
	}

	/* 入力値をscreen に記録 */
	scntxt->screen.ldbs_iscreen->sqlid = sqlno;

	return DIVY_LDB_ST_OK;
}

/**
 * 検索結果から返信するXMLを作成する(linkdbsearch)
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_resource *検索結果
 * @param dav_response ** davのレスポンス格納構造体
 * @param int linkdb を実行する目的のフラグ
 * @param int errdiscovery,linkdberrdiscovery を返却する場合のエラーコード
 * @return int HTTPコード
 */
static int ldbs_build_xml_response(request_rec *r,
					divy_rdbo_resource *search_result,
					dav_response **res, 
					int optflg, int ldbstcode)
{
	dav_response *newres 	= NULL;
	dav_response *firstres 	= NULL;
	divy_rdbo_resource *resource;

	*res = NULL;

	/* error 情報を返却する場合 */
	if (ldbstcode != DIVY_LDB_ST_OK){
		/* response を作成(常に1件) */
		*res = ldbs_err_mkresponse(r, search_result, optflg,
						ldbstcode);
		(*res)->next = NULL;

		return HTTP_OK;
	}

	/* SQL の実行目的(おためしの場合は+SQLタイプ)により分岐 */
	if (optflg == DIVY_SEARCH_LDB_NORMAL ||
		(optflg == DIVY_SEARCH_LDB_MLKDB &&
		 (search_result->search_pr->sqltype == DIVY_SQL_TYPE_NORMAL ||
		  search_result->search_pr->sqltype == DIVY_SQL_TYPE_REQUIRED))) {

		/* 通常のlinkdbで表示 */
		for (resource = search_result; resource; resource = resource->next) {
			resource->p = r->pool;

			newres = ldbs_mkresponse(r, resource);

			if (firstres == NULL) {
				firstres = *res = newres;
			}
			else {
				(*res)->next = newres;
				*res = newres;
			}
			(*res)->next = NULL;
		}
	}
	else if (optflg == DIVY_SEARCH_LDB_REPOSITORY ||
		 (optflg == DIVY_SEARCH_LDB_MLKDB &&
		  search_result->search_pr->sqltype == DIVY_SQL_TYPE_REPOSITORY)) {

		/* リポジトリ検索で表示 */
		for (resource = search_result; resource; resource = resource->next) {

			/* プールを設定　*/
			resource->p = r->pool;

			newres = ldbs_repository_mkresponse(r, resource);

			if (firstres == NULL) {
				firstres = *res = newres;
			}
			else {
				(*res)->next = newres;
				*res = newres;
			}
			(*res)->next = NULL;
		}
	}
	*res = firstres;
	return HTTP_OK;
}

/**
 * searchのresponceを作成する(linkdbsearch)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_resource * 検索結果
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *ldbs_mkresponse(request_rec *r,
					divy_rdbo_resource *resource)
{
	apr_pool_t *p = r->pool;
	dav_response *res = apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr = { 0 };
	apr_text_header hdr_ns = { 0 };
	const char *s;
	int i;
	divy_rdbo_resource rdb = { 0 };

	/*
	 * (note)
	 * resource->uriがNULL(即ちDBキーがNULL)時のは
	 * tf_rdbo 側でエラーステータスを返却し
	 * この関数は通らないように修正しました。
	 */
	/*
	 * resource->uri からMD5のダイジェストメッセージを作成して、
	 * uriを作成する。
	 * (linkdbsearch ではuri のfinal path segmentにダイジェスト値を使う)
	 */
	res->href = divy_build_dbfolder_uri(p, dav_divy_get_root_uri(r),
				apr_psprintf(p, "%s/", ap_md5(p, (const unsigned char*)resource->uri)));
	res->status = HTTP_OK;

	/* 
	 * XML作成
	 */
	apr_text_append(p, &hdr, "<"DAV_NS_PREFIX":propstat>" CRLF
				"<"DAV_NS_PREFIX":prop>" CRLF);

	/* (note) displaynameは255サイズに切り捨てます
	 * 		値が空の場合は固定値に置き換えます */
	if (IS_EMPTY(resource->displayname)) {
		apr_text_append(p, &hdr, "<"DAV_NS_PREFIX":displayname>"
				DIVY_SEARCH_LKDB_EMPTYVALUE
				"</"DAV_NS_PREFIX":displayname>" CRLF);
	} else {
		s = apr_psprintf(p, "<"DAV_NS_PREFIX":displayname>"
				"%s"
				"</"DAV_NS_PREFIX":displayname>" CRLF,
				dav_divy_escape_xmlstr(p, 
					(char *)dav_divy_truncate_str(p,
						resource->displayname, 
						DIVY_SEARCH_DISPN_MAXSLEN), DIVY_XML_T2T_QUOTE));
		apr_text_append(p, &hdr, s);
	}

	apr_text_append(p, &hdr, "<"DIVY_NS_PREFIX":keylist>" CRLF);
	for (i=1; resource->search_pr->linkdb_key_values[i-1]; i++) {
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":key id =\"%d\">"
				"%s"
				"</"DIVY_NS_PREFIX":key>" CRLF,
				i, resource->search_pr->linkdb_key_values[i-1]);
		apr_text_append(p, &hdr, s);
	}

	/* リソースタイプ追加 */
	apr_text_append(p, &hdr, "</"DIVY_NS_PREFIX":keylist>" CRLF
				"<"DAV_NS_PREFIX":resourcetype>"
				"<"DAV_NS_PREFIX":collection/>"
				"</"DAV_NS_PREFIX":resourcetype>" CRLF);

	/* オペレーション権限を書き込む(View設定権限は不要) */
	rdb.uri            = (char *) res->href;
	rdb.resourcetype   = DIVY_TYPE_COLLECTION;
	rdb.getcontenttype = NULL;
	rdb.rstate_pr      = NULL;
	divy_insert_prop_user_privilege_grant_set(r, NULL, DIVY_DEL_VIEW_PRIVILEGE, &rdb, &hdr);

	/* クローズ */
	apr_text_append(p, &hdr,"</"DAV_NS_PREFIX":prop>" CRLF 
				"<"DAV_NS_PREFIX":status>"
				"HTTP/1.1 200 OK"
				"</"DAV_NS_PREFIX":status>" CRLF
				"</"DAV_NS_PREFIX":propstat>" CRLF);

	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));
    
	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
    
	return res;
}

/**
 * searchのresponceを作成する(linkdbsearch リポジトリ検索)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_resource * 検索結果
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *ldbs_repository_mkresponse(request_rec *r,
					divy_rdbo_resource *resource)
{
	apr_pool_t *p 		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s		= NULL;
	char *datebuf 		= NULL;
	divy_rdbo_resource rdb  = { 0 };

	/*
	 * URI 作成 collection:最後に'/'、それ以外はなし
	 */
	if (resource->resourcetype == DIVY_TYPE_COLLECTION){
		res->href = apr_psprintf(p, "%s/", resource->uri);
	} else {
		res->href = apr_psprintf(p, "%s", resource->uri);
	}
	res->status = HTTP_OK;

	/* 
	 * XML作成
	 */
	s = apr_psprintf(p, "<"DAV_NS_PREFIX":propstat>" CRLF
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DAV_NS_PREFIX":displayname>"
			"%s"
			"</"DAV_NS_PREFIX":displayname>" CRLF
			"<"DAV_NS_PREFIX":getcontentlength>"
			"%"APR_INT64_T_FMT
			"</"DAV_NS_PREFIX":getcontentlength>" CRLF
			"<"DAV_NS_PREFIX":getcontenttype>"
			"%s"
			"</"DAV_NS_PREFIX":getcontenttype>" CRLF,
			dav_divy_escape_xmlstr(p, resource->displayname, 
						DIVY_XML_T2T_QUOTE),
			resource->getcontentlength, resource->getcontenttype);
	apr_text_append(p, &hdr, s);

	/* getlastmodified 変換 */
	divy_format_time_t(p, resource->getlastmodified,
				DIVY_TIME_STYLE_RFC822, &datebuf);
	if (IS_FILLED(datebuf)) {
		s = apr_psprintf(p, "<"DAV_NS_PREFIX":getlastmodified>"
			"%s"
			"</"DAV_NS_PREFIX":getlastmodified>" CRLF,
			datebuf);
	}
	else {
		s = "<"DAV_NS_PREFIX":getlastmodified/>"CRLF;
	}
	apr_text_append(p, &hdr, s);

	/* resourcetype 設定(collection 以外は空) */
	if (resource->resourcetype == DIVY_TYPE_COLLECTION){
		apr_text_append(p, &hdr, "<"DAV_NS_PREFIX":resourcetype>"
				"<"DAV_NS_PREFIX":collection/>"
				"</"DAV_NS_PREFIX":resourcetype>" CRLF);
	} else {
		apr_text_append(p, &hdr, 
				"<"DAV_NS_PREFIX":resourcetype/>" CRLF);
	}

	/* オプションはあれば追加 */
	if (resource->search_pr->reposdb_selected
			[DIVY_REPOSDBOPT_CREATIONDATE]){
		divy_format_time_t(p, resource->creationdate,
					DIVY_TIME_STYLE_ISO8601, &datebuf);
		if (IS_FILLED(datebuf)) {
			s = apr_psprintf(p, "<"DAV_NS_PREFIX":creationdate>"
				"%s"
				"</"DAV_NS_PREFIX":creationdate>" CRLF,
				datebuf);
		}
		else {
			s = "<"DAV_NS_PREFIX":creationdate/>"CRLF;
		}
		apr_text_append(p, &hdr, s);
	}
	if (resource->search_pr->reposdb_selected
			[DIVY_REPOSDBOPT_CREATOR]){
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":creator>"
				"%s"
				"</"DIVY_NS_PREFIX":creator>" CRLF,
				dav_divy_escape_xmlstr(p, resource->creator,
					DIVY_XML_T2T_CDATA));
		apr_text_append(p, &hdr, s);
	}
	if (resource->search_pr->reposdb_selected
			[DIVY_REPOSDBOPT_LASTMODIFIER]){
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":lastmodifier>"
				"%s"
				"</"DIVY_NS_PREFIX":lastmodifier>" CRLF,
				dav_divy_escape_xmlstr(p,
					resource->lastmodifier, 
					DIVY_XML_T2T_CDATA));
		apr_text_append(p, &hdr, s);
	}

	/* オペレーション権限を書き込む(View設定権限は不要) */
	rdb.uri            = (char *) res->href;
	rdb.resourcetype   = resource->resourcetype;
	rdb.getcontenttype = resource->getcontenttype;
	rdb.rstate_pr      = NULL;
	divy_insert_prop_user_privilege_grant_set(r, NULL, DIVY_DEL_VIEW_PRIVILEGE, &rdb, &hdr);

	/* クローズ */
	apr_text_append(p, &hdr, "</"DAV_NS_PREFIX":prop>" CRLF 
				"<"DAV_NS_PREFIX":status>"
				"HTTP/1.1 200 OK"
				"</"DAV_NS_PREFIX":status>" CRLF
				"</"DAV_NS_PREFIX":propstat>" CRLF);

	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));
    
	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
    
	return res;
}

/**
 * searchのresponceを作成する
 * (linkdbsearch/mamagementlinkdbsearch, error/dbmserr)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_resource * 検索結果
 * @param  int linkdbsearch エラーコード(see search.h)
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *ldbs_err_mkresponse(request_rec *r,
					divy_rdbo_resource *resource,
					int optflg, int ldbstcode)
{
	apr_pool_t *p 		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s		= NULL;
	divy_search_ldb_err *serr = divy_search_get_ldb_err(p, ldbstcode);

	/*
	 * URI 作成
	 */
	if (optflg == DIVY_SEARCH_LDB_MLKDB) {
		res->href = divy_build_m_execsql_uri(p, dav_divy_get_root_uri(r),
				apr_psprintf(p, "%s/", resource->uri));
	}
	else {
		res->href = divy_build_dblink_uri(p, dav_divy_get_root_uri(r), 
				apr_psprintf(p, "%s/", resource->uri));
	}
	res->status = HTTP_OK;

	/* 
	 * XML作成
	 */
	/* 共通部分 */
	apr_text_append(p, &hdr, "<"DAV_NS_PREFIX":propstat>" CRLF
			"<"DAV_NS_PREFIX":prop>" CRLF);
	/* managementlinkdbsearch */
	if (optflg == DIVY_SEARCH_LDB_MLKDB){
		apr_text_append(p, &hdr,
				"<"DIVY_NS_PREFIX":dbmserrdiscovery>" CRLF
				"<"DIVY_NS_PREFIX":dbmserrinfo>" CRLF);
		/* DBMS のエラー情報が取得されていた場合 */
		if (ldbstcode == DIVY_LDB_ST_ERR_DBPROVIDER &&
				resource->search_pr &&
				resource->search_pr->dbmserrcode && 
				resource->search_pr->dbmserrmsg){

			s = apr_psprintf(p, "<"DIVY_NS_PREFIX":dbmserrcode>"
					"%s"
					"</"DIVY_NS_PREFIX":dbmserrcode>" CRLF
					"<"DIVY_NS_PREFIX":dbmserrmsg>"
					"%s"
					"</"DIVY_NS_PREFIX":dbmserrmsg>" CRLF,
					dav_divy_escape_xmlstr(p, 
						resource->search_pr->
							dbmserrcode,
						DIVY_XML_T2T_CDATA),
					dav_divy_escape_xmlstr(p,
						resource->search_pr->
							dbmserrmsg,
						DIVY_XML_T2T_CDATA));
		}
		/* teamfile のエラーを返却する場合 */
		else {
			s = apr_psprintf(p, "<"DIVY_NS_PREFIX":dbmserrcode>"
					"%s"
					"</"DIVY_NS_PREFIX":dbmserrcode>" CRLF
					"<"DIVY_NS_PREFIX":dbmserrmsg>"
					"%s"
					"</"DIVY_NS_PREFIX":dbmserrmsg>" CRLF,
					serr->code_m,
					dav_divy_escape_xmlstr(p, 
						(char *)serr->msg_m,
						DIVY_XML_T2T_CDATA));
		}
		apr_text_append(p, &hdr, s);
		apr_text_append(p, &hdr, "</"DIVY_NS_PREFIX":dbmserrinfo>" CRLF
			"</"DIVY_NS_PREFIX":dbmserrdiscovery>" CRLF);
	}
	/* linkdbsearch */
	else {
		apr_text_append(p, &hdr,
				"<"DIVY_NS_PREFIX":linkdberrdiscovery>" CRLF
				"<"DIVY_NS_PREFIX":linkdberrinfo>" CRLF);
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":linkdberrcode>"
				"%s"
				"</"DIVY_NS_PREFIX":linkdberrcode>" CRLF
				"<"DIVY_NS_PREFIX":linkdberrmsg>"
				"%s"
				"</"DIVY_NS_PREFIX":linkdberrmsg>" CRLF,
				serr->code_l,
				dav_divy_escape_xmlstr(p, (char *)serr->msg_l,
						DIVY_XML_T2T_CDATA));
		apr_text_append(p, &hdr, s);
		apr_text_append(p, &hdr, 
				"</"DIVY_NS_PREFIX":linkdberrinfo>" CRLF
				"</"DIVY_NS_PREFIX":linkdberrdiscovery>" CRLF);
	}

	/* クローズ */
	apr_text_append(p, &hdr, "</"DAV_NS_PREFIX":prop>" CRLF 
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);

	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));
    
	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
    
	return res;
}

/**
 * XMLをパースし、パース結果を返却する (roottreesearch)
 *
 * @param  request_rec * apache request_rec構造体
 * @param  stype_elem apr_xml_elem * 解析済みのXML
 * @param  search_cntxt * 作成されたクエリーを格納する
 * @return int HTTPコード
 */
static int rts_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
						search_cntxt *scntxt)
{
	apr_pool_t *p 		= r->pool;
	apr_xml_elem *cur_elem  = stype_elem, *elem;
	int elem_cnt            = 0;

	/* エラーチェック */
	if (stype_elem == NULL || stype_elem->first_child != NULL || stype_elem->next != NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Element of <roottreesearch> is wrong.");
		return HTTP_BAD_REQUEST;
	}

	scntxt->optflg = 0;
	/*
	 * XMLのパース
	 *
	 * [ DTD ]
	 * 	<!ELEMENT roottreesearch (alllist | sqlstmt | reposdbsqlstmt | grouplist | management | execsqlstmt ) >
	 * 	<!ELEMENT alllist        EMPTY >
	 * 	<!ELEMENT sqlstmt        EMPTY >
	 * 	<!ELEMENT reposdbsqlstmt EMPTY >
	 * 	<!ELEMENT grouplist      EMPTY >
	 * 	<!ELEMENT management     EMPTY >
	 * 	<!ELEMENT execsqlstmt    EMPTY >
	 */
	for (elem = cur_elem; elem; elem = elem->next) {

		if (strcmp(elem->name, "alllist") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_ALLLIST;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "sqlstmt") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_SQLSTMT;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "reposdbsqlstmt") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_REPOSSQLSTMT;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "grouplist") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_GROUPLIST;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "execsqlstmt") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_EXECSQLSTMT;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "management") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_MANAGEMENT;
			elem_cnt++;
		}
		else {
			/* 未知のエレメントが指定された */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"roottreesearch", elem->name);
			return HTTP_BAD_REQUEST;
		}
	}

	/*
	 * 取得結果の検証
	 */
	/* 具体的なSEARCH動作を示す指示エレメントが存在しなかった */
	if (scntxt->optflg == 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The request element (alllist / sqlstmt / reposdbsqlstmt / "
			"grouplist / execsqlstmt / management) "
			"of roottreesearch not found.");
		return HTTP_BAD_REQUEST;
	}
	/* 指示エレメントが2つ以上存在した */
	else if (elem_cnt > 1) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Too many request elements. The number of request element "
			"(alllist / sqlstmt / reposdbsqlstmt / grouplist / execsqlstmt "
			"/ management) must be one.");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}

/**
 * 検索結果から返信するXMLを作成する(roottreesearch)
 * 
 *
 * @param  request_rec * apache request_rec構造体
 * @param  int 検索オプション see search.h
 * @param  divy_rdbo_resource *検索結果
 * @paramr dav_response ** davのレスポンス格納構造体
 * @return int HTTPコード
 */
static int rts_build_xml_response(request_rec *r, int searchopt,
					divy_rdbo_resource *search_result,
					dav_response **res)
{
	dav_response *newres	    = NULL;
	dav_response *firstres 	    = NULL;
	apr_pool_t *p               = r->pool;
	const int *acdeny	    = divy_get_accessdeny(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	const divy_special_folder_spec *fspec = NULL;
	divy_special_folderid_id fid, i;
	divy_rdbo_sql *sql;
	divy_rdbo_grp *grp;
	int mode;
	const divy_rdbo_extstatus *extstatus = divy_get_extstatus(r);
	int support_extstatus   = divy_support_extenduserstatus(r);
	int support_groupleader = divy_support_groupleader(r);
	int is_trusty = divy_rdbo_is_trusty_user(extstatus);
	int is_groupleader = divy_rdbo_is_groupleader(extstatus);
    
	*res = NULL;
    
	/* 管理者モードの取得 */
	mode = divy_get_adminmode(r);

	/* alllist */
	if (searchopt == DIVY_SEARCH_OPT_ALLLIST) {

 		/* 特殊フォルダIDの最後までループ */
		for (i = 0; i < DIVY_FOLDER_ID_END; i++) {

			fspec = divy_get_special_fspec(p, i);
			fid = fspec->id;

			/* DBコラボレーション機能は利用可能か？ */
			if (sconf->use_db_opt != 1 &&
				fspec->show_group & DIVY_SHOW_DB) continue;

			/* alllist の時のみ */
			if (fspec->show_group & DIVY_SHOW_ALLLIST) {

				/* 管理者フォルダはシステム管理者とグループリーダ以外不可 */
				if (fid == DIVY_FOLDER_ID_management && !divy_rdbo_has_administrative_privilege(r)) {
					continue;
				}

				/* 権限がない場合もスキップ
				 * 数が少ないのはDIVY_SHOW_ALLLISTのものだけが対象だからです */
				if (acdeny != NULL &&
					(fid == DIVY_FOLDER_ID_user        ||
					 fid == DIVY_FOLDER_ID_group       ||
					 fid == DIVY_FOLDER_ID_dblink      ||
					 fid == DIVY_FOLDER_ID_reposdblink ||
					 fid == DIVY_FOLDER_ID_management) &&
					acdeny[fid] != 0) {
					continue;
				}

				/* 制限ユーザにはリポジトリ検索フォルダを見せない(外部仕様) */
				if (support_extstatus && !is_trusty && fid == DIVY_FOLDER_ID_reposdblink) {
					continue;
				}

				/* XML作成 */
				newres = rts_spfolder_mkresponse(r, search_result, fspec);

				if (firstres == NULL) {
					firstres = *res = newres;
				}
				else {
					(*res)->next = newres;
					*res = newres;
				}
				(*res)->next = NULL;
			}
		}
	}
	/* sqlstmt, reposdbsqlstmt, execsqlstmt */
	else if (searchopt == DIVY_SEARCH_OPT_SQLSTMT ||
			 searchopt == DIVY_SEARCH_OPT_REPOSSQLSTMT ||
			 searchopt == DIVY_SEARCH_OPT_EXECSQLSTMT) {

		/* 制限ユーザにはリポジトリ検索フォルダを見せない(外部仕様) */
		if (support_extstatus && !is_trusty && searchopt == DIVY_SEARCH_OPT_REPOSSQLSTMT) {
			firstres = NULL;
		}
		/* 管理者のみがデータベース(テスト用)を使用可能 */
		else if (searchopt == DIVY_SEARCH_OPT_EXECSQLSTMT && mode != DIVY_ADMINMODE_ADMIN) {
			firstres = NULL;
		}
		/* 権限ありの場合のみ実行 */
		else if (acdeny == NULL || 
			((searchopt == DIVY_SEARCH_OPT_SQLSTMT      && acdeny[DIVY_FOLDER_ID_dblink] == 0) || 
			 (searchopt == DIVY_SEARCH_OPT_REPOSSQLSTMT && acdeny[DIVY_FOLDER_ID_reposdblink] == 0) ||
			 (searchopt == DIVY_SEARCH_OPT_EXECSQLSTMT  && acdeny[DIVY_FOLDER_ID_m_execsql] == 0))) {

			for (sql = search_result->sql_pr; sql; sql = sql->next){
				/* responseを1件作成 */
				if (searchopt == DIVY_SEARCH_OPT_EXECSQLSTMT) {
					newres = rts_sqlstmt_mkresponse(r, sql, 1);
				}
				else {
					newres = rts_sqlstmt_mkresponse(r, sql, 0);
				}

				if (firstres == NULL) {
					firstres = *res = newres;
				}
				else {
					(*res)->next = newres;
					*res = newres;
				}
				(*res)->next = NULL;
			}
		}
	}
	/* grouplist */
	else if (searchopt == DIVY_SEARCH_OPT_GROUPLIST) {

		/* 権限ありの場合のみ実行 */
		if (acdeny == NULL || acdeny[DIVY_FOLDER_ID_group] == 0) {
#ifdef DIVY_SUPPORT_PLUGIN
			tfs_hset_t *groupid_set = NULL;

			/* セキュリティで保護されたグループかどうかを取得する */
			divy_pi_cciphered_grpset(r, &groupid_set);
#endif	/* DIVY_SUPPORT_PLUGIN */
			for (grp = search_result->grp_pr; grp; grp = grp->next) {

				/* responseを1件作成 */
				newres = rts_grp_mkresponse(r, grp
#ifdef DIVY_SUPPORT_PLUGIN
						,groupid_set
#endif	/* DIVY_SUPPORT_PLUGIN */
						);

				if (firstres == NULL) {
					firstres = *res = newres;
				}
				else {
					(*res)->next = newres;
					*res = newres;
				}
				(*res)->next = NULL;
			}
		}
	}
	/* management */
	else if (searchopt == DIVY_SEARCH_OPT_MANAGEMENT) {

	 	/* 特殊フォルダIDの最後までループ */
		for (i = 0; i < DIVY_FOLDER_ID_END; i++) {

			fspec = divy_get_special_fspec(p, i);
			fid = fspec->id;

			/* DBコラボレーション機能は使用可能か？ */
			if (sconf->use_db_opt != 1 &&
				fspec->show_group & DIVY_SHOW_DB) continue;

			/* management の時のみ */
			if (fspec->show_group & DIVY_SHOW_MANAGEMENT) {

				/* 管理者フォルダはシステム管理者とグループリーダ以外不可 */
				if (!divy_rdbo_has_administrative_privilege(r)) {
					continue;
				}

				/* グループリーダはクライアント更新, DBMS, メッセージ, データベース(テスト用)を見れない */
				if (support_groupleader && is_groupleader &&
					(fid == DIVY_FOLDER_ID_m_update  ||
					 fid == DIVY_FOLDER_ID_m_dbms    ||
					 fid == DIVY_FOLDER_ID_m_msg     ||
					 fid == DIVY_FOLDER_ID_m_execsql)) {
					continue;
				}

				/* 権限がない場合スキップ
				 * 数が少ないのはDIVY_SHOW_ALLLIST
				 * のものだけが対象だからです */
				if (acdeny &&
					(fid == DIVY_FOLDER_ID_m_update ||
					 fid == DIVY_FOLDER_ID_m_sql    ||
					 fid == DIVY_FOLDER_ID_m_group  ||
					 fid == DIVY_FOLDER_ID_m_user   ||
					 fid == DIVY_FOLDER_ID_m_status ||
					 fid == DIVY_FOLDER_ID_m_msg    ||
					 fid == DIVY_FOLDER_ID_m_dbms   ||
					 fid == DIVY_FOLDER_ID_m_execsql) &&
					acdeny[fid] != 0) {
					continue;
				}

				/* XML作成 */
				newres = rts_spfolder_mkresponse(r, search_result, fspec);

				if (firstres == NULL) {
					firstres = *res = newres;
				}
				else {
					(*res)->next = newres;
					*res = newres;
				}
				(*res)->next = NULL;
			}
		}
	}
    
	if (firstres != NULL) {
		/* responceの先頭を返却 */
		*res = firstres; 
		return HTTP_OK;
	}
	else {
		return HTTP_NOT_FOUND;
	}
}

/**
 * searchのresponceを作成する(roottreesearchの特殊フォルダ)
 *
 *
 * @param r request_rec * apache request_rec構造体
 * @param resource divy_rdbo_resource * 検索結果
 * @param fspec const divy_special_folder_spec * 特殊フォルダ構造体
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *rts_spfolder_mkresponse(request_rec *r, 
					divy_rdbo_resource *resource,
					const divy_special_folder_spec *fspec)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;

	/* href作成 */
	if (fspec->id == DIVY_FOLDER_ID_user){
		res->href = apr_psprintf(p, "%s/", resource->uri);
	} else {
		res->href = dav_divy_make_uri(p, dav_divy_get_root_uri(r),
						fspec->relativeuri, "/", NULL);
	}
	res->status = HTTP_OK;

	/* XML作成 */
	apr_text_append(p, &hdr,"<"DAV_NS_PREFIX":propstat>" CRLF 
				"<"DAV_NS_PREFIX":prop>" CRLF);
	s = apr_psprintf(p, "<"DAV_NS_PREFIX":displayname>"
			"%s"
			"</"DAV_NS_PREFIX":displayname>" CRLF
			"<"DAV_NS_PREFIX":resourcetype>"
			"<"DAV_NS_PREFIX":collection/>"
			"</"DAV_NS_PREFIX":resourcetype>" CRLF
			"<"DIVY_NS_PREFIX":foldertype>"
			"<"DIVY_NS_PREFIX":%s/>"
			"</"DIVY_NS_PREFIX":foldertype>" CRLF,
			dav_divy_escape_xmlstr(p,
				(char *)fspec->displayname, 
				DIVY_XML_T2T_QUOTE),
			fspec->foldertype); 
	apr_text_append(p, &hdr, s);

	/* displayorder が定義されていれば追加 */
	if (fspec->displayorder != DIVY_DISPLAYORDER_UNDEFINE) {
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":displayorder>"
				"%d"
				"</"DIVY_NS_PREFIX":displayorder>" CRLF,
					fspec->displayorder);
		apr_text_append(p, &hdr, s);
	}
	apr_text_append(p, &hdr, "</"DAV_NS_PREFIX":prop>" CRLF 
				"<"DAV_NS_PREFIX":status>"
				"HTTP/1.1 200 OK"
				"</"DAV_NS_PREFIX":status>" CRLF
				"</"DAV_NS_PREFIX":propstat>" CRLF);

	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;

	return res;
}

/**
 * searchのresponceを作成する(roottreesearchのグループフォルダ)
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_grp * 取得済みのgrp
 * @return dav_response * davのレスポンス格納構造体
 */ 
static dav_response *rts_grp_mkresponse(request_rec *r, divy_rdbo_grp *grp
#ifdef DIVY_SUPPORT_PLUGIN
		, tfs_hset_t *groupid_set
#endif	/* DIVY_SUPPORT_PLUGIN */
		)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	divy_rdbo_resource rdb = { 0 };
#ifdef DIVY_SUPPORT_PLUGIN
	char *extra_status     = "";
#endif	/* DIVY_SUPPORT_PLUGIN */

	res->href = apr_psprintf(p, "%s/", grp->grpcol_uri);
	res->status = HTTP_OK;

#ifdef DIVY_SUPPORT_PLUGIN
	/* セキュリティで保護されたグループかどうか */
	if (groupid_set != NULL) {
		if (tfs_hset_contains(groupid_set, grp->grpid)) {
			extra_status = "<"DIVY_NS_PREFIX":resourcestate>"
							"<"DIVY_NS_PREFIX":secured/>"
							"</"DIVY_NS_PREFIX":resourcestate>" CRLF;
		}
		else {
			extra_status = "";
		}
	}
#endif	/* DIVY_SUPPORT_PLUGIN */

	/* XML作成 */
	apr_text_append(p, &hdr,"<"DAV_NS_PREFIX":propstat>" CRLF 
				"<"DAV_NS_PREFIX":prop>" CRLF);
	s = apr_psprintf(p, "<"DAV_NS_PREFIX":displayname>"
			"%s"
			"</"DAV_NS_PREFIX":displayname>" CRLF
			"<"DAV_NS_PREFIX":resourcetype>"
			"<"DAV_NS_PREFIX":collection/>"
			"</"DAV_NS_PREFIX":resourcetype>" CRLF
			"<"DIVY_NS_PREFIX":foldertype>"
			"<"DIVY_NS_PREFIX":"DIVY_FOLDER_TYPE_group_elm"/>"
			"</"DIVY_NS_PREFIX":foldertype>" CRLF
			"<"DIVY_NS_PREFIX":groupid>"
			"%s"
			"</"DIVY_NS_PREFIX":groupid>" CRLF
#ifdef DIVY_SUPPORT_PLUGIN
			"%s"
#endif	/* DIVY_SUPPORT_PLUGIN */
			,dav_divy_escape_xmlstr(p, grp->name, DIVY_XML_T2T_QUOTE)
			,grp->grpid
#ifdef DIVY_SUPPORT_PLUGIN
			,extra_status
#endif	/* DIVY_SUPPORT_PLUGIN */
			);
	apr_text_append(p, &hdr, s);

	/* グループフォルダのユーザに紐付くオペレーション権限を書き込む(View設定権限は不要) */
	rdb.uri            = (char *) res->href;
	rdb.resourcetype   = DIVY_TYPE_COLLECTION;
	rdb.getcontenttype = NULL;
	rdb.rstate_pr      = NULL;
	divy_insert_prop_user_privilege_grant_set(r, NULL, DIVY_DEL_VIEW_PRIVILEGE, &rdb, &hdr);

	/* グループ管理者機能が有効な場合、オーナ情報を記録する */
	if (divy_support_groupleader(r)) {
		_rts_build_groupstate(r, grp, &hdr, p);
	}

	/* コメントがある場合はコメントの表示を変更する */
	if (IS_FILLED(grp->comment)) {
		/* 先頭が-(ハイフン）の場合は非表示にする */
		if (strncmp(grp->comment, "-", 1) == 0) {
			grp->comment = NULL;
		}
		/* 先頭が+(プラス）の場合は+だけを非表示にする */
		else if (strncmp(grp->comment, "+", 1) == 0) {
			grp->comment++;
		}
		apr_text_append(p, &hdr, apr_psprintf(p,
			"<"DIVY_NS_PREFIX":comment>"
			"%s"
			"</"DIVY_NS_PREFIX":comment>",
			dav_divy_escape_xmlstr(p, grp->comment, DIVY_XML_T2T_CDATA)));
	}
	else {
		apr_text_append(p, &hdr, "<"DIVY_NS_PREFIX":comment/>");
	}

	apr_text_append(p, &hdr, "</"DAV_NS_PREFIX":prop>" CRLF
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);
	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;

	return res;
}

/**
 * roottreesearch のgroupstate プロパティを組み立てる
 */
static void _rts_build_groupstate(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_text_header *hdr,
					apr_pool_t *wp)
{
	if (hdr == NULL || grp_pr == NULL) return;

	apr_text_append(wp, hdr,
				"<"DIVY_NS_PREFIX":groupstate>");
	/* オーナ名 */
	if (IS_FILLED(grp_pr->ownername)) {
		apr_text_append(wp, hdr, apr_psprintf(wp,
					"<"DIVY_NS_PREFIX":ownername>"
					"%s"
					"</"DIVY_NS_PREFIX":ownername>",
					dav_divy_escape_xmlstr(wp, grp_pr->ownername, DIVY_XML_T2T_CDATA)));

	}
	else {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":ownername/>");

	}

	/* BOXをサポートしているなら属性を調べて追加 */
	if (divy_support_tfbox(r)) {
		/* BOX設定されているグループかどうか */
		if (divy_rdbo_is_box_group(grp_pr->grp_extstatus)) {
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":box/>");
		}
	}

	/* アップロードポリシーをサポートしているなら属性を調べて追加 */
	if (divy_support_upload_policy(r)) {
		if (divy_rdbo_is_uploadpolicy_group(grp_pr->grp_extstatus)) {
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":uploadpolicy/>");
		}
	}

	apr_text_append(wp, hdr,
				"</"DIVY_NS_PREFIX":groupstate>" CRLF);
}

/**
 * searchのresponceを作成する(roottreesearchのsql情報)
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_sql * 取得済みのSQL
 * @param  int EXECSQL フラグ( 0:EXECSQL ではない、1:EXECSQL)
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *rts_sqlstmt_mkresponse(request_rec *r, divy_rdbo_sql *sql,
						int execflg)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns = { 0 };
	const char *s;

	if (execflg == 1) {
		res->href = divy_build_m_execsql_uri(p, dav_divy_get_root_uri(r),
					apr_psprintf(p, "%s/", sql->relativeuri));
	}
	else if (sql->type == DIVY_SQL_TYPE_REPOSITORY) {
		res->href = divy_build_reposdblink_uri(p, dav_divy_get_root_uri(r),
					apr_psprintf(p, "%s/", sql->relativeuri));
	}
	else {
		res->href = divy_build_dblink_uri(p, dav_divy_get_root_uri(r),
					apr_psprintf(p, "%s/", sql->relativeuri));
	}
	res->status = HTTP_OK;

	/* XML作成 */
	apr_text_append(p, &hdr,"<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF);
	s = apr_psprintf(p, "<"DAV_NS_PREFIX":displayname>%s"
			"</"DAV_NS_PREFIX":displayname>" CRLF
			"<"DAV_NS_PREFIX":resourcetype>"
			"<"DAV_NS_PREFIX":collection/>"
			"</"DAV_NS_PREFIX":resourcetype>" CRLF,
			dav_divy_escape_xmlstr(p, sql->labelname, 
				DIVY_XML_T2T_QUOTE));
	apr_text_append(p, &hdr, s);

	/* foldertype は分岐 */
	if (execflg == 1){
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":foldertype>"
				"<"DIVY_NS_PREFIX":%s/>"
				"</"DIVY_NS_PREFIX":foldertype>" CRLF,
				DIVY_FOLDER_TYPE_execsqlstmt);
	} else if (sql->type == DIVY_SQL_TYPE_REPOSITORY){
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":foldertype>"
				"<"DIVY_NS_PREFIX":%s/>"
				"</"DIVY_NS_PREFIX":foldertype>" CRLF,
				DIVY_FOLDER_TYPE_reposdbsqlstmt);
	} else {
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":foldertype>"
				"<"DIVY_NS_PREFIX":%s/>"
				"</"DIVY_NS_PREFIX":foldertype>" CRLF,
				DIVY_FOLDER_TYPE_sqlstmt);
	}
	apr_text_append(p, &hdr, s);

	s = apr_psprintf(p, "<"DIVY_NS_PREFIX":sqlno>"
			"%s"
			"</"DIVY_NS_PREFIX":sqlno>" CRLF, sql->sqlid);
	apr_text_append(p, &hdr, s);
	apr_text_append(p, &hdr, "</"DAV_NS_PREFIX":prop>" CRLF
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);

	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;

	return res;
}

/**
 * XMLをパースし、パース結果を返却する (userinformationsearch)
 *
 * @param  request_rec * apache request_rec構造体
 * @param  stype_elem apr_xml_elem * 解析済みのXML
 * @param  search_cntxt * 作成されたクエリーを格納する
 * @return int HTTPコード
 */
static int useris_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
						search_cntxt *scntxt)
{
	apr_pool_t *p	= r->pool;
	apr_xml_elem *cur_elem = stype_elem, *elem;
	const char *userid     = NULL;
	const int *accessdeny  = divy_get_accessdeny(r);
	int support_extstatus = divy_support_extenduserstatus(r);
	const divy_rdbo_extstatus *extstatus = divy_get_extstatus(r);
	int elem_cnt           = 0;
	divy_rdbo_usr *ac_usr_pr = NULL;

	/* エラーチェック */
	if (stype_elem == NULL || stype_elem->first_child != NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Element of <userinformationsearch> is wrong.");
		return HTTP_BAD_REQUEST;
	}

	/* 入力値を格納するscreen を作成 */
	scntxt->screen.useris_iscreen = apr_pcalloc(p, sizeof(divy_search_useris_iscreen));
	scntxt->optflg = 0;

	/*
	 * XMLのパース
	 *
	 * [ DTD ]
	 * 	<!ELEMENT userinformationsearch (
	 * 		treelist | detaillist | (userid, (content | availablegroup) ) >
	 * 	<!ELEMENT userid        (#PCDATA) >
	 * 	<!ELEMENT treelist       EMPTY >
	 * 	<!ELEMENT detaillist     EMPTY >
	 * 	<!ELEMENT content        EMPTY >
	 * 	<!ELEMENT availablegroup EMPTY >
	 */
	for (elem = cur_elem; elem; elem = elem->next) {

		if (strcmp(elem->name, "userid") == 0) {
			/* (2007.12.21 Fri)
			 * PCDATA の両端に含まれていたスペースを除去するのはNG
			 * (前後に半角スペースの埋まったユーザIDが利用できなくなるため) */
			userid = divy_xml_get_cdata(elem, p, 0);
		}
		else if (strcmp(elem->name, "treelist") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_TREELIST;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "detaillist") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_DETAILLIST;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "content") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_CONTENT;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "availablegroup") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_AVAILABLEGROUP;
			elem_cnt++;
		}
		else {
			/* 未知のエレメントが指定された */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"userinformationsearch", elem->name);
			return HTTP_BAD_REQUEST;
		}
	}

	/*
	 * 取得結果の検証
	 */
	/* 具体的なSEARCH動作を示す指示エレメントが存在しなかった */
	if (scntxt->optflg == 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The request element (treelist / detaillist / content / availablegroup) "
			"of userinformationsearch not found.");
		return HTTP_BAD_REQUEST;
	}
	/* 指示エレメントが2つ以上存在した */
	else if (elem_cnt > 1) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Too many request elements. The number of request element "
			"(treelist / detaillist / content / availablegroup) must be one.");
		return HTTP_BAD_REQUEST;
	}
	/* content, availablegroup ではuserid が必須 */
	else if (IS_EMPTY(userid) &&
			(scntxt->optflg == DIVY_SEARCH_OPT_CONTENT ||
			 scntxt->optflg == DIVY_SEARCH_OPT_AVAILABLEGROUP)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"userid\" element is missing.");
		return HTTP_BAD_REQUEST;
	}
	/* treelist, detaillist ではuserid を指定してはならない */
	else if (IS_FILLED(userid) &&
			(scntxt->optflg == DIVY_SEARCH_OPT_TREELIST ||
			 scntxt->optflg == DIVY_SEARCH_OPT_DETAILLIST)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"userid\" element must be EMPTY.");
		return HTTP_BAD_REQUEST;
	}

	/* アカウントが無効、有効期限切れのユーザには自身のcontent 表示のみ許可する */
	if (support_extstatus &&
		(!divy_rdbo_is_active_user(extstatus) || divy_rdbo_is_expired_user(r))) {
		if (!(scntxt->optflg == DIVY_SEARCH_OPT_CONTENT &&
			strcmp(userid, divy_get_userid(r)) == 0)) {
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The user was inactive or expired. (user = %s)", userid);
			return HTTP_FORBIDDEN;
		}
	}

	/*
	 * (2005/03/03 Thu) 管理者 or グループリーダでないユーザ = 管理者系ユーザには
	 *                  一定の制限を加える
	 * [ 適用条件 ]
	 *  * 管理者ではない
	 *  * グループリーダでもない
	 *
	 * [ 制約条件 ]
	 *  * conent  : アクセスユーザ自身のcontent, availablegroup は許可
	 *  * 上記以外: 全て拒否
	 */
	if (!divy_rdbo_has_administrative_privilege(r)) {

		if (scntxt->optflg == DIVY_SEARCH_OPT_DETAILLIST ||
		    scntxt->optflg == DIVY_SEARCH_OPT_TREELIST) {

			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"You could not get all user list."
					"Need to administrive privilege."
					"(access user = %s)", divy_get_userid(r));

			return HTTP_FORBIDDEN;
		}
		else if ((scntxt->optflg == DIVY_SEARCH_OPT_CONTENT ||
				  scntxt->optflg == DIVY_SEARCH_OPT_AVAILABLEGROUP)&&
				strcmp(userid, divy_get_userid(r)) != 0) {

			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"You could not access other's user property."
					"Need to administrive privilege."
					"(access user = %s)", divy_get_userid(r));

			return HTTP_FORBIDDEN;
		}
	}

	/*
	 * アクセスユーザがグループリーダの場合、
	 * Otherユーザの所属グループは見えないこと.
	 * また、Otherユーザのプロパティも見えないこと.
	 *
	 * -> Other ユーザの availablegroup (例外あり), content(例外あり) を禁止
	 *
	 * (例外の理由)
	 *   自分自身のavailablegroup, content はCGI やパスワード変更で利用しているので許可.
	 *
	 */
	if (divy_rdbo_is_groupleader(extstatus) &&
		(scntxt->optflg == DIVY_SEARCH_OPT_AVAILABLEGROUP ||
		 scntxt->optflg == DIVY_SEARCH_OPT_CONTENT) && strcmp(userid, divy_get_userid(r)) != 0) {

		/* アクセス先ユーザの情報を取得する */
		if (ac_usr_pr == NULL) {
			if (divy_rdbo_get_user_property(r, userid, &ac_usr_pr)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get userinformation.(userid = %s)", userid);
				return HTTP_INTERNAL_SERVER_ERROR;
			}
		}

		if (ac_usr_pr != NULL && ac_usr_pr->is_otheruser) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not get other-user property (leader = %s)", divy_get_userid(r));
			return HTTP_FORBIDDEN;
		}
	}

	/*
	 * (2005/01/18 Tue)
	 * アクセス拒否リストにリストされていてもcontent は許可する
	 * 管理者であってもアクセス拒否リストの影響は受けます。
	 * (content でないリクエストはエラー)
	 */
	if (accessdeny != NULL && accessdeny[DIVY_FOLDER_ID_m_user] &&
					scntxt->optflg != DIVY_SEARCH_OPT_CONTENT) {

		/* 許可しない */
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The server does not accept \"%s\" method "
			"on this uri that is listed on access-deny folder."
			"(access user = %s)", divy_get_method_name(r), divy_get_userid(r));

		return HTTP_FORBIDDEN;
	}

	/*
	 * アクセス先ユーザがシステム実行権限ユーザの場合、アクセスを許可しない
	 * [ 適用条件 ]
	 *  * システム実行権限ユーザである
	 *
	 * [ 制約条件 ]
	 *  * content, availablegroup : 拒否
	 *  * 上記以外 : 許可
	 */
	if (IS_FILLED(userid) &&
		(scntxt->optflg == DIVY_SEARCH_OPT_CONTENT ||
		 scntxt->optflg == DIVY_SEARCH_OPT_AVAILABLEGROUP)) {

		/* アクセス先ユーザの情報を取得する */
		if (ac_usr_pr == NULL) {
			if (divy_rdbo_get_user_property(r, userid, &ac_usr_pr)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get userinformation.(userid = %s)", userid);
				return HTTP_INTERNAL_SERVER_ERROR;
			}
		}

		/* システム実行権限を持つかどうか */
		if (ac_usr_pr != NULL &&
			divy_rdbo_has_sysexec_privilege(ac_usr_pr->extstatus)) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"The server does not accept \"%s\" method "
					"on this system-exec user property. (access user = %s)",
					divy_get_method_name(r), divy_get_userid(r));
			return HTTP_FORBIDDEN;
		}
	}

	/* 入力値をscreen に記録 */
	scntxt->screen.useris_iscreen->userid = userid;

	return  HTTP_OK;
}

/**
 * 検索結果から返信するXMLを作成する(userinformationsearch)
 *
 *
 * @param  r request_rec * apache request_rec構造体
 * @param  usr_pr divy_rdbo_usr * 検索結果
 * @paramr res dav_response ** davのレスポンス格納構造体
 * @param  optflg int 検索オプション
 * @return int HTTPコード
 */
static int useris_build_xml_response(request_rec *r,
					divy_rdbo_usr *usr_pr,
					dav_response **res, int optflg)
{
	dav_response *newres 	= NULL;
	dav_response *firstres  = NULL;
	apr_pool_t *p = r->pool;
    
	*res = NULL;
    
	/* availablegroupはグループでループしrespouse作成 */
	if (optflg == DIVY_SEARCH_OPT_AVAILABLEGROUP){
		divy_rdbo_grp *grp;

		for (grp = usr_pr->grp_pr; grp; grp = grp->next){
			/* responseを１件作成 */
			newres = apr_pcalloc(p, sizeof(dav_response));

			useris_grp_mkresponse(r, usr_pr, grp, newres, p);
			if (firstres == NULL) {
				firstres = *res = newres;
			}
			else {
				(*res)->next = newres;
				*res = newres;
			}
			(*res)->next = NULL;
		}
	}
	/* availablegroup以外はユーザでループしrespouse作成 */
	else {
		divy_rdbo_usr *usr;

		for (usr = usr_pr; usr; usr = usr->next){
			/* responseを１件作成 */
			newres = apr_pcalloc(p, sizeof(dav_response));

			useris_usr_mkresponse(r, usr, optflg, newres, p);
			/* 1件目の場合 */
			if (firstres == NULL) {
				firstres = *res = newres;
			}
			else {
				(*res)->next = newres;
				*res = newres;
			}
			(*res)->next = NULL;
		}
	}
	*res = firstres;
	return HTTP_OK;
}

/**
 * searchのresponceを作成する(userinformationsearchのuser情報)
 *
 * @param r request_rec * apache request_rec構造体
 * @param usr divy_rdbo_usr * 取得済みのuser
 * @param optflg int 検索オプション see search.h
 * @param res dav_response * davのレスポンス格納構造体
 * @param wp apr_pool_t * 作業用のプール
 */
static void useris_usr_mkresponse(request_rec *r, divy_rdbo_usr *usr, int optflg,
					dav_response *res, apr_pool_t *wp)
{
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	int i;
#ifdef DIVY_SUPPORT_PASSPOLICY
	char *extra = NULL;
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	res->href = divy_build_m_user_uri(wp, dav_divy_get_root_uri(r),
				apr_psprintf(wp, "%s/", usr->usrid));
	res->status = HTTP_OK;

	/* XML作成(共通部分) */
	apr_text_append(wp, &hdr,"<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":userdiscovery>" CRLF 
			"<"DIVY_NS_PREFIX":userinfo>" CRLF);
	s = apr_psprintf(wp,
			"<"DIVY_NS_PREFIX":userid>"
			"%s"
			"</"DIVY_NS_PREFIX":userid>" CRLF
			"<"DIVY_NS_PREFIX":name>%s</"DIVY_NS_PREFIX":name>" CRLF,
			dav_divy_escape_xmlstr(wp, usr->usrid, DIVY_XML_T2T_CDATA),
			dav_divy_escape_xmlstr(wp, usr->fullname, DIVY_XML_T2T_CDATA));
	apr_text_append(wp, &hdr, s);

	/*
	 * treelist以外(共通)
	 */
	if (optflg != DIVY_SEARCH_OPT_TREELIST) {
		/* メールアドレス */
		if (IS_FILLED(usr->mailaddr)) {
			s = apr_psprintf(wp,
				"<"DIVY_NS_PREFIX":mailaddr>%s</"DIVY_NS_PREFIX":mailaddr>" CRLF,
				dav_divy_escape_xmlstr(wp, usr->mailaddr, DIVY_XML_T2T_CDATA));
			apr_text_append(wp, &hdr, s);
		}
		else {
			apr_text_append(wp, &hdr, "<"DIVY_NS_PREFIX":mailaddr/>" CRLF);
		}

		/* 作成日付、更新日付 */
		s = apr_psprintf(wp,
				"<"DIVY_NS_PREFIX":creationdt>%s"
				"</"DIVY_NS_PREFIX":creationdt>" CRLF
				"<"DIVY_NS_PREFIX":updatedt>%s"
				"</"DIVY_NS_PREFIX":updatedt>" CRLF,
				usr->registdt, usr->updatedt);
		apr_text_append(wp, &hdr, s);
	}

	/*
	 * detaillist, content の場合
	 */
	if (optflg == DIVY_SEARCH_OPT_DETAILLIST || optflg == DIVY_SEARCH_OPT_CONTENT) {
		/* password, quota情報 */
		s = apr_psprintf(wp,
				"<"DIVY_NS_PREFIX":password>%s"
				"</"DIVY_NS_PREFIX":password>" CRLF
				"<"DIVY_NS_PREFIX":usedstorage>%"APR_INT64_T_FMT
				"</"DIVY_NS_PREFIX":usedstorage>" CRLF
				"<"DIVY_NS_PREFIX":maxstorage>%"APR_INT64_T_FMT
				"</"DIVY_NS_PREFIX":maxstorage>" CRLF
				"<"DIVY_NS_PREFIX":usedresource>%"APR_INT64_T_FMT
				"</"DIVY_NS_PREFIX":usedresource>" CRLF
				"<"DIVY_NS_PREFIX":maxresource>%"APR_INT64_T_FMT
				"</"DIVY_NS_PREFIX":maxresource>" CRLF,
				dav_divy_escape_xmlstr(wp, usr->password, DIVY_XML_T2T_CDATA),
				usr->usedst, usr->maxst, usr->usedres, usr->maxres);
		apr_text_append(wp, &hdr, s);

		/* accessdeny */
		s = NULL;	/* 初期化 */
		if (usr->accessdeny != NULL) {
			const divy_special_folder_spec *fspec;

			for (i = 0; i < DIVY_FOLDER_ID_END; i++) {
				fspec = divy_get_special_fspec(wp, i);

				/* 拒否 かつ 表示フォルダだった場合 */
				if (usr->accessdeny[i] != 0 &&
					(fspec->show_group & DIVY_SHOW_ALLLIST ||
					 fspec->show_group & DIVY_SHOW_MANAGEMENT)) {
					/* 文字列を追加 */
					if (IS_EMPTY(s)) {
						s = apr_pstrdup(wp, fspec->accessdeny_name);
					}
					else {
						s = apr_pstrcat(wp, s, " ", fspec->accessdeny_name, NULL);
					}
				}
			}
			/* タグを付ける(文字列なしは空タグ) */
			if (IS_FILLED(s)) {
				s = apr_pstrcat(wp,
					"<"DIVY_NS_PREFIX":accessdeny>", s,
					"</"DIVY_NS_PREFIX":accessdeny>" CRLF, NULL);
				apr_text_append(wp, &hdr, s);
			}
			else {
				/* エントリありですべて0 
				 * WARNING 出力 */
				ERRLOG0(r->pool, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
					"The accessdeny property exists in database "
					"but all deny flag are 0.");
				apr_text_append(wp, &hdr, "<"DIVY_NS_PREFIX":accessdeny/>" CRLF);
			}
		}
		/* エントリなし 空タグ作成 */
		else {
			apr_text_append(wp, &hdr, "<"DIVY_NS_PREFIX":accessdeny/>" CRLF);
		}

		/* コメント */
		/* 管理者並びにグループリーダーのみがコメントを見れるようになります */
		/* 2.5.0-1からは一般ユーザは先頭-のコメントは見えなくなりました		*/
		if (IS_FILLED(usr->comment)) {
			if ((strncmp(usr->comment, "-", 1) != 0)
						|| (divy_rdbo_has_administrative_privilege(r))) {
				s = apr_psprintf(wp,
					"<"DIVY_NS_PREFIX":comment>%s</"DIVY_NS_PREFIX":comment>" CRLF,
				dav_divy_escape_xmlstr(wp, usr->comment, DIVY_XML_T2T_CDATA));
				apr_text_append(wp, &hdr, s);
			}
		}
		else {
			apr_text_append(wp, &hdr, "<"DIVY_NS_PREFIX":comment/>" CRLF);
		}

		/* 管理者モード(下位互換性のために残してあります) */
		if (usr->adminmode == DIVY_ADMINMODE_ADMIN) {
			apr_text_append(wp, &hdr, "<"DIVY_NS_PREFIX":admin/>" CRLF);
		}

#ifdef DIVY_SUPPORT_PASSPOLICY
		/* ユーザのパスワードポリシー状態を組み立てる */
		if (divy_support_passpolicy(r)) {
			_useris_build_passpolicystatus(r, usr, &extra, wp);
		}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

		/* ユーザ拡張ステータスを入れる */
		if (divy_support_extenduserstatus(r)) {
			_useris_build_extstatus(r, usr, extra, &hdr, wp);
		}

		/* グループリーダの情報を入れる */
		if (divy_support_groupleader(r)) {
			_useris_build_groupleader(r, usr, &hdr, wp);
		}

		/* アクセス制御の情報を入れる */
		if (divy_support_access_control(r)) {
			_useris_build_accesscontrol(r, usr, &hdr, wp);

		}
	}

	/* クローズ */
	apr_text_append(wp, &hdr,
			"</"DIVY_NS_PREFIX":userinfo>" CRLF
			"</"DIVY_NS_PREFIX":userdiscovery>" CRLF 
			"</"DAV_NS_PREFIX":prop>" CRLF 
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);
 
	/* ネームスペース追加 */
	apr_text_append(wp, &hdr_ns, divy_make_liveprop_ns(wp, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
}

/**
 * searchのresponceを作成する(userinformationsearchのgrp情報)
 *
 *
 * @param r request_rec * apache request_rec構造体
 * @param usr divy_rdbo_usr * 取得済みのuser
 * @param grp divy_rdbo_grp * 取得済みのgrp
 * @param res dav_response * davのレスポンス格納構造体
 * @param wp apr_pool_t * 作業用のプール
 */
static void useris_grp_mkresponse(request_rec *r,
					divy_rdbo_usr *usr, divy_rdbo_grp *grp,
					dav_response *res, apr_pool_t *wp)
{
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *rg_grpid	= apr_psprintf(wp, DIVY_PREFIX_RGROUP"%s", grp->grpid);

	res->href = divy_build_m_user_uri(wp, dav_divy_get_root_uri(r),
				apr_psprintf(wp, "%s/%s/", usr->usrid, rg_grpid));
	res->status = HTTP_OK;

	/* XML作成 */
	apr_text_append(wp, &hdr,
		apr_psprintf(wp, "<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":groupdiscovery>" CRLF 
			"<"DIVY_NS_PREFIX":groupinfo>" CRLF
			"<"DIVY_NS_PREFIX":groupid>"
			"%s"
			"</"DIVY_NS_PREFIX":groupid>" CRLF
			"<"DIVY_NS_PREFIX":name>"
			"%s"
			"</"DIVY_NS_PREFIX":name>" CRLF
			"<"DIVY_NS_PREFIX":creationdt>"
			"%s"
			"</"DIVY_NS_PREFIX":creationdt>" CRLF
			"<"DIVY_NS_PREFIX":updatedt>"
			"%s"
			"</"DIVY_NS_PREFIX":updatedt>" CRLF
			"<"DIVY_NS_PREFIX":comment>"
			"%s"
			"</"DIVY_NS_PREFIX":comment>" CRLF,
			grp->grpid, 
			dav_divy_escape_xmlstr(wp, grp->name, DIVY_XML_T2T_CDATA),
			grp->registdt, grp->updatedt,
			dav_divy_escape_xmlstr(wp, grp->comment, DIVY_XML_T2T_CDATA)));

	/* 拡張ユーザステータスをサポートしていれば返却する */
	if (IS_FILLED(grp->grpcol_uri) && divy_support_extenduserstatus(r)) {
		apr_text_append(wp, &hdr, apr_psprintf(wp,
			"<"DIVY_NS_PREFIX":groupcollectionuri>"
			"%s/"
			"</"DIVY_NS_PREFIX":groupcollectionuri>" CRLF,
			dav_divy_escape_uri(wp, grp->grpcol_uri)));
	}

	/* グループ拡張ステータスを追加する */
	/* グループ制約属性を追加する */
	_grpis_build_grp_extstatus(r, grp, &hdr, wp);

	/* オーナー情報を追加する */
	if (divy_support_groupleader(r)) {
		_grpis_build_groupleader(r, grp, &hdr, wp);
	}

	apr_text_append(wp, &hdr,
			"</"DIVY_NS_PREFIX":groupinfo>" CRLF
			"</"DIVY_NS_PREFIX":groupdiscovery>" CRLF
			"</"DAV_NS_PREFIX":prop>" CRLF 
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);
    
	/* ネームスペース追加 */
	apr_text_append(wp, &hdr_ns, divy_make_liveprop_ns(wp, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));
							    
	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
}

/**
 * ユーザ拡張ステータスを表すXMLタグを作ってhdr に詰める。
 */
static void _useris_build_extstatus(request_rec *r,
					divy_rdbo_usr *usr,
					char *extra_status,
					apr_text_header *hdr,
					apr_pool_t *wp)
{
	char *expiration_date = NULL;
	divy_rdbo_extstatus *extstatus = NULL;
	const char *s;
	int support_groupleader = divy_support_groupleader(r);

	if (hdr == NULL || usr == NULL) return;

	/* ユーザ拡張ステータスの取得 */
	extstatus = usr->extstatus;
	if (extstatus == NULL) {
		/* 無ければデフォルト値を使う */
		extstatus = divy_rdbo_create_default_extstatus(wp, EXTSTATUS_TYPE_USR);
	}

	/*
	 * 状態
	 */
	apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":state>");
	/* ユーザが有効かどうか */
	if (divy_rdbo_is_active_user(extstatus)) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":active/>");
	}
	else {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":inactive/>");
	}

	/* 有効期限がリクエスト開始時よりも前だった場合 (expiration が 0の場合は特別扱い) */
	if (usr->expiration != 0 && usr->expiration < apr_time_sec(r->request_time)) {
		/* 有効期限切れ */
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":expired/>");
	}

	/* 特殊ステータスを挿入する */
	if (IS_FILLED(extra_status)) {
		apr_text_append(wp, hdr, extra_status);
	}
	apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":state>" CRLF);

	/*
	 * 有効期限
	 */
	if (usr->expiration != 0) {
		divy_format_time_t(wp, usr->expiration, DIVY_TIME_STYLE_ISO8601, &expiration_date);
		s = apr_psprintf(wp, "<"DIVY_NS_PREFIX":expiration>%s</"DIVY_NS_PREFIX":expiration>" CRLF,
				expiration_date);
		apr_text_append(wp, hdr, s);
	}
	else {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":expiration/>" CRLF);
	}

	/*
	 * 権限
	 */
	apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":user-privilege>");
	if (divy_rdbo_has_readwrite_privilege(extstatus)) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":readwrite/>");
	}
	else if (divy_rdbo_has_upload_privilege(extstatus)) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":upload/>");
	}
	else {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":read/>");
	}

	if (divy_rdbo_has_setview_privilege(extstatus)) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":set-view/>");
	}
	if (divy_support_grpconstraints(r) &&
		divy_rdbo_has_user_groupconstraints_ignore(extstatus)) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":group-constraints-ignore/>");
	}
	if (support_groupleader &&
		divy_rdbo_has_control_otheruser(extstatus)) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":control-other-user/>");
	}
	apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":user-privilege>" CRLF);

	/*
	 * ユーザの種類
	 */
	apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":usertype>");
	if (usr->adminmode == DIVY_ADMINMODE_ADMIN) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":admin/>");
	}
	else {
		if (support_groupleader && divy_rdbo_is_groupleader(extstatus)) {
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":groupleader/>");
		}
		else if (divy_rdbo_is_trusty_user(extstatus)) {
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":normal/>");
		}
		else {
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":limited/>");
		}
	}
	apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":usertype>" CRLF);
}


/**
 * usr が示すグループリーダに関するXMLを組み立ててhdr にappendする.
 */
static void _useris_build_groupleader(request_rec *r,
					divy_rdbo_usr *usr,
					apr_text_header *hdr,
					apr_pool_t *wp)
{
	if (hdr == NULL || usr == NULL) return;

	/*
	 * 最大管理ユーザ数
	 */
	apr_text_append(wp, hdr, apr_psprintf(wp, 
					"<"DIVY_NS_PREFIX":maxusercreation>"
					"%d"
					"</"DIVY_NS_PREFIX":maxusercreation>" CRLF,
					usr->maxusercreation));

	/*
	 * オーナ名称
	 */
	if (IS_FILLED(usr->ownername)) {
		apr_text_append(wp, hdr, apr_psprintf(wp,
						"<"DIVY_NS_PREFIX":ownername>"
						"%s"
						"</"DIVY_NS_PREFIX":ownername>" CRLF,
						dav_divy_escape_xmlstr(wp, usr->ownername, DIVY_XML_T2T_CDATA)));
	}
	else {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":ownername/>" CRLF);
	}

	/*
	 * オーナID
	 */
	if (IS_FILLED(usr->ownerid)) {
		apr_text_append(wp, hdr, apr_psprintf(wp,
						"<"DIVY_NS_PREFIX":ownerid>"
						"%s"
						"</"DIVY_NS_PREFIX":ownerid>" CRLF,
						dav_divy_escape_xmlstr(wp, usr->ownerid, DIVY_XML_T2T_CDATA)));
	}
	else {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":ownerid/>" CRLF);
	}

	/*
	 * Otherユーザかどうか
	 */
	if (usr->is_otheruser) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":otheruser/>" CRLF);
	}

	/*
	 * グループリーダに任命されているかどうか.
	 */
	if (usr->is_appointed_groupleader) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":appointed-groupleader/>" CRLF);
	}
}

/**
 * usr が示すアクセス制御に関するXMLを組み立ててhdr にappendする.
 */
static void _useris_build_accesscontrol(request_rec *r,
					divy_rdbo_usr *usr,
					apr_text_header *hdr,
					apr_pool_t *wp)
{
	if (hdr == NULL || usr == NULL) return;

	/*
	 *  接続可能IPアドレス
	 */
	if (IS_FILLED(usr->allowhosts)) {
		apr_text_append(wp, hdr, apr_psprintf(wp,
						"<"DIVY_NS_PREFIX":allowhosts>"
						"%s"
						"</"DIVY_NS_PREFIX":allowhosts>" CRLF,
						usr->allowhosts));
	}
	else {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":allowhosts/>" CRLF);
	}
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * パスワードポリシー状態を組み立てて返却する.
 */
static void _useris_build_passpolicystatus(request_rec *r,
					divy_rdbo_usr *usr,
					char **ret,
					apr_pool_t *wp)
{
	divy_rdbo_passwordpolicy *passpolicy_pr = NULL;
	time_t now = dav_divy_get_now_epoch();
	divy_rdbo_passpolicystatus *passpolicy_status;
	time_t compdt = 0L;

	*ret = NULL;

	if (usr == NULL || usr->passpolicy_status == NULL) {
		return;
	}
	passpolicy_status = usr->passpolicy_status;

	/* キャッシュからパスワードポリシーを取得する */
	(void) divy_get_cached_passwordpolicy(r, DIVY_DEFAULT_POLICY_ID,
										  1, &passpolicy_pr);

	/* パスワードポリシーが無効であれば終了 */
	if (passpolicy_pr == NULL || passpolicy_pr->status == 0) {
		return;	/* 何もしない */
	}

	/* 初回ログイン時対応(即時変更) が必要で、
	 * 初回ログイン時にのパスワード変更操作が済んでいない場合 */
	if (passpolicy_pr->firstlogin_limitday == 0 &&
		passpolicy_status->firstlogin == 0L) {

		*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":unupdate-password/>");
	}
	/* 初回ログイン時対応が必要で
	 * 初回ログイン時にのパスワード変更操作が済んでいない場合 */
	else if (passpolicy_pr->firstlogin_limitday > 0 &&
		passpolicy_status->firstlogin == 0L) {

		time_t firstlogin_limitday =
			DIVY_PASSPOLICY_PROBATION_SEC(passpolicy_pr->firstlogin_limitday);

		if (passpolicy_status->lastchange != 0L) {
			compdt = passpolicy_status->lastchange;
		}
		else {
			compdt = dav_divy_iso8601totime_t(wp, usr->registdt);
		}

		/* パスワード最終更新日付(or 作成日)よりもポリシー開始日の方が
		 * 大きければ置き換える (see _validate_password_term);
		 */
		if (passpolicy_pr->startdt > 0L && passpolicy_pr->startdt > compdt) {
			compdt = passpolicy_pr->startdt;
		}

		if (now <= compdt + firstlogin_limitday) {
			*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":unupdate-pending-password/>");
		}
		else {
			if (passpolicy_status->specialstart == 0L && usr->lastaccess == 0L) {
				*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":unupdate-pending-password/>");
			}
			else if (passpolicy_status->specialstart != 0L &&
					 passpolicy_status->specialstart > compdt) {
				if (passpolicy_status->specialstart > now) {
					*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":regular-password/>");
				}
				else if (passpolicy_status->specialstart <= now &&
						 now <= passpolicy_status->specialstart + firstlogin_limitday) {
					*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":unupdate-pending-password/>");
				}
				else {
					*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":expired-password/>");
				}
			}
			else {
				*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":expired-password/>");
			}
		}
	}
	else if (passpolicy_pr->change_pw_cycle > 0) {
		time_t lastchange = passpolicy_status->lastchange;
		time_t pw_cycle   = DIVY_PASSPOLICY_PASSPERIOD_SEC(passpolicy_pr->change_pw_cycle);
		time_t probation  = DIVY_PASSPOLICY_PROBATION_SEC(passpolicy_pr->probation);

		/* パスワード最終更新日付(or 作成日)よりもポリシー開始日の方が
		 * 大きければ置き換える */
		if (passpolicy_pr->startdt > 0L && passpolicy_pr->startdt > lastchange) {
			compdt = passpolicy_pr->startdt;
		}
		else {
			compdt = lastchange;	/* 最終更新日を使う */
		}

		if (compdt + pw_cycle < now && now <= compdt + pw_cycle + probation) {
			*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":pending-password/>");
		}
		else if (compdt + pw_cycle + probation < now) {
			if (passpolicy_status->specialstart == 0L &&
				compdt + pw_cycle >= usr->lastaccess) {
				//*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":pending-password/>");
				*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":regular-password/>");
			}
			else if (passpolicy_status->specialstart != 0L &&
					 passpolicy_status->specialstart > compdt) {

				if (passpolicy_status->specialstart > now) {
					*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":regular-password/>");
				}
				else if (passpolicy_status->specialstart <= now &&
						 now <= passpolicy_status->specialstart + probation) {
					*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":pending-password/>");
				}
				else {
					*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":expired-password/>");
				}
			}
			else {
				*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":expired-password/>");
			}
		}
		else {
			*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":regular-password/>");
		}
	}
	else {
		*ret = apr_pstrdup(wp, "<"DIVY_NS_PREFIX":regular-password/>");
	}
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * グループ拡張ステータスを表すXMLタグを作ってhdr に詰める。
 */
static void _grpis_build_grp_extstatus(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_text_header *hdr,
					apr_pool_t *wp)
{
	divy_rdbo_extstatus *grp_extstatus = NULL;

	if (hdr == NULL || grp_pr == NULL) return;

	grp_extstatus = grp_pr->grp_extstatus;
	if (divy_support_grpconstraints(r)) {
		if (grp_extstatus == NULL) {
			/* 無ければグループ制約なしとする */
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":constraints/>" CRLF);
		}
		else {
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":constraints>");
			if (divy_rdbo_has_write_groupconstraints(grp_extstatus)) {
				apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":write/>");
			}
			if (divy_rdbo_has_oplog_groupconstraints(grp_extstatus)) {
				apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":operationlog/>");
			}
			if (divy_rdbo_has_showprop_groupconstraints(grp_extstatus)) {
				apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":showproperty/>");
			}
			apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":constraints>" CRLF);
		}
	}

	if (divy_support_groupleader(r)) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":state>");
		if (grp_extstatus == NULL || divy_rdbo_is_active_group(grp_extstatus)) {
			apr_text_append(wp, hdr,	"<"DIVY_NS_PREFIX":active/>");
		}
		else {
			apr_text_append(wp, hdr,	"<"DIVY_NS_PREFIX":inactive/>");
		}
		apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":state>" CRLF);
	}

	if (divy_support_tfbox(r)) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":box>");
		if (grp_extstatus == NULL || divy_rdbo_is_box_group(grp_extstatus)) {
			apr_text_append(wp, hdr,	"<"DIVY_NS_PREFIX":active/>");
		}
		else {
			apr_text_append(wp, hdr,	"<"DIVY_NS_PREFIX":inactive/>");
		}
		apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":box>" CRLF);
	}

	/* アップロードポリシー */
	if (divy_support_upload_policy(r)) {
		if (grp_pr->policy == NULL) {
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":uploadpolicy/>" CRLF);
		}
		else {
			_grpis_build_uploadpolicy(r, grp_pr, hdr, wp);
		}
	}
}

/**
 * グループのグループリーダに関するXMLエレメントをhdr にappendする.
 *
 * @param r request_rec *r
 * @param grp_pr divy_rdbo_grp *
 * @param hdr apr_text_header *
 * @param wp apr_pool_t *
 */
static void _grpis_build_groupleader(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_text_header *hdr,
					apr_pool_t *wp)
{
	if (hdr == NULL || grp_pr == NULL) return;

	/* オーナ名称 */
	if (IS_FILLED(grp_pr->ownername)) {
		apr_text_append(wp, hdr, apr_psprintf(wp,
						"<"DIVY_NS_PREFIX":ownername>"
						"%s"
						"</"DIVY_NS_PREFIX":ownername>" CRLF,
						dav_divy_escape_xmlstr(wp, grp_pr->ownername, DIVY_XML_T2T_CDATA)));
	}
	else {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":ownername/>" CRLF);
	}

	/* オーナID */
	if (IS_FILLED(grp_pr->ownerid)) {
		apr_text_append(wp, hdr, apr_psprintf(wp,
						"<"DIVY_NS_PREFIX":ownerid>"
						"%s"
						"</"DIVY_NS_PREFIX":ownerid>" CRLF,
						dav_divy_escape_xmlstr(wp, grp_pr->ownerid, DIVY_XML_T2T_CDATA)));
	}
	else {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":ownerid/>" CRLF);
	}
}

#ifdef DIVY_SUPPORT_GROUPQUOTA
/*
 * グループのグループQuotaに関するXMLエレメントをhdr にappendする.
 */
static void _grpis_build_groupquota(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_text_header *hdr,
					apr_pool_t *wp) 
{
	divy_rdbo_grpquota* grp_q;	/* グループQuota */

	if (hdr == NULL || grp_pr == NULL) return;

	grp_q = grp_pr->grp_q;

	/* unsupport */
	if (!grp_q) return;

	apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":groupquota>" CRLF);

	apr_text_append(wp, hdr, apr_psprintf(wp,
					"<"DIVY_NS_PREFIX":used-bytes>"
					"%"APR_INT64_T_FMT
					"</"DIVY_NS_PREFIX":used-bytes>" CRLF,
					grp_q->usedst));

	apr_text_append(wp, hdr, apr_psprintf(wp,
					"<"DIVY_NS_PREFIX":available-bytes>"
					"%"APR_INT64_T_FMT
					"</"DIVY_NS_PREFIX":available-bytes>" CRLF,
					grp_q->maxst));

	apr_text_append(wp, hdr, apr_psprintf(wp,
					"<"DIVY_NS_PREFIX":used-files>"
					"%"APR_INT64_T_FMT
					"</"DIVY_NS_PREFIX":used-files>" CRLF,
					grp_q->usedres));

	apr_text_append(wp, hdr, apr_psprintf(wp,
					"<"DIVY_NS_PREFIX":available-files>"
					"%"APR_INT64_T_FMT
					"</"DIVY_NS_PREFIX":available-files>" CRLF,
					grp_q->maxres));

	apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":groupquota>" CRLF);

}
#endif /* DIVY_SUPPORT_GROUPQUOTA */

/*
 * アップロードポリシー
 */
static void _grpis_build_uploadpolicy(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_text_header *hdr,
					apr_pool_t *wp)
{
	char *datebuf = NULL;
	char *token, *ctx;
	const char *s;
	int isView = 0;

	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);

	if (grp_pr->policy) {
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":uploadpolicy>" CRLF);

		divy_format_time_t(wp, grp_pr->policy->updatedate,
									DIVY_TIME_STYLE_ISO8601, &datebuf);
		if (grp_pr->policy->updatedate != 0 && IS_FILLED(datebuf)) {
			s = apr_psprintf(wp,
					"<"DIVY_NS_PREFIX":updatedate>%s"
					"</"DIVY_NS_PREFIX":updatedate>" CRLF, datebuf);
		}
		else {
			s = "<"DIVY_NS_PREFIX":updatedate/>" CRLF;
		}
		apr_text_append(wp, hdr, s);

		if (grp_pr->policy->hidden == 1) {
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":hidden/>" CRLF);
		}

		/* アクティブ・非アクティブ(フラグ) */
		apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":rulestate>");
		if (grp_pr->policy->active == 1) {
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":active/>");
		}
		else {
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":inactive/>");
		}
		apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":rulestate>" CRLF);

		/* 
		 * ルールは管理者かグループリーダーで且つ自分のグループなら返答を返す
		 * それ以外は、条件(hidden)を教えない
		 */
		if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN) {
			isView = 1;
		}
		else if (divy_support_groupleader(r)
						&& divy_rdbo_is_groupleader(own_extstatus)) {
			isView = 1;
		}

		if (isView == 1) {

			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":rule>" CRLF);

			/* ファイル名ルール */
			if (IS_FILLED(grp_pr->policy->rules_fnames)) {
				while((token = apr_strtok(grp_pr->policy->rules_fnames,
														"/", &ctx)) != NULL) {
					grp_pr->policy->rules_fnames = NULL;

					apr_text_append(wp, hdr, apr_psprintf(wp,
								"<"DIVY_NS_PREFIX":filename>"
								"%s"
								"</"DIVY_NS_PREFIX":filename>" CRLF, token));
				}

			}

			/* 拡張子ルール */
			if (IS_FILLED(grp_pr->policy->rules_suffix)) {
				while((token = apr_strtok(grp_pr->policy->rules_suffix,
														",", &ctx)) != NULL) {
					grp_pr->policy->rules_suffix = NULL;

					apr_text_append(wp, hdr, apr_psprintf(wp,
								"<"DIVY_NS_PREFIX":suffix>"
								"%s"
								"</"DIVY_NS_PREFIX":suffix>" CRLF, token));
				}
			}

			/* 長さルール */
			if (IS_FILLED(grp_pr->policy->rules_length)) {
				apr_text_append(wp, hdr, apr_psprintf(wp,
							"<"DIVY_NS_PREFIX":length>"
							"%s"
							"</"DIVY_NS_PREFIX":length>" CRLF,
							grp_pr->policy->rules_length));
			}

			/* 文字種ルール */
			if (grp_pr->policy->rules_chartype
								!= TF_UPLOAD_POLICY_CHARTYPE_NONE) {
				apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":chartype>");
				if (grp_pr->policy->rules_chartype
									& TF_UPLOAD_POLICY_CHARTYPE_NUM) {
					apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":number/>");
				}
				if (grp_pr->policy->rules_chartype
									&  TF_UPLOAD_POLICY_CHARTYPE_ALP) {
					apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":alphabet/>");
				}
				apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":chartype>" CRLF);
			}
			/* マッチ種類 */
			apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":matchtype>");
			if (grp_pr->policy->rules_matchtype
										== TF_UPLOAD_POLICY_MATCHTYPE_ANY) {
				apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":any/>");
			}
			else {
				apr_text_append(wp, hdr, "<"DIVY_NS_PREFIX":all/>");
			}
			apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":matchtype>" CRLF);

			apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":rule>" CRLF);
		}

		apr_text_append(wp, hdr, "</"DIVY_NS_PREFIX":uploadpolicy>" CRLF);

	}
}

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
/**
 * get_response ハンドラにセットされる関数 (userinformationsearch)
 */
static int _useris_get_response(request_rec *r, divy_rdbo_search_output *output,
					dav_response *response, apr_pool_t *wp)
{
	divy_rdbo_usr *usr_pr = output->list.usr_pr;

	if (usr_pr == NULL) return DIVY_SEARCH_RESPONSE_NONE;

	if (output->optflg == DIVY_SEARCH_OPT_AVAILABLEGROUP) {
		if (usr_pr->grp_pr == NULL) return DIVY_SEARCH_RESPONSE_NONE;
		/* 1つのレスポンス生成 */
		useris_grp_mkresponse(r, usr_pr, usr_pr->grp_pr, response, wp);
	}
	else {
		/* 1つのレスポンス生成 */
		useris_usr_mkresponse(r, usr_pr, output->optflg, response, wp);
	}

	return 0;
}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

/**
 * XMLをパースし、パース結果を返却する (groupinformationsearch)
 *
 * @param  request_rec * apache request_rec構造体
 * @param  stype_elem apr_xml_elem * 解析済みのXML
 * @param  search_cntxt * 作成されたクエリーを格納する
 * @return int HTTPコード
 */
static int grpis_parse_xml(request_rec *r, apr_xml_elem *stype_elem, 
						search_cntxt *scntxt)
{
	apr_pool_t *p          = r->pool;
	apr_xml_elem *cur_elem = stype_elem, *elem;
	const char *grpid      = NULL;
	const char *grpuri     = NULL;
	const char *grpcol_uri = NULL;
	int elem_cnt           = 0;
	const char *data;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

	/* エラーチェック */
	if (stype_elem == NULL || stype_elem->first_child != NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Element of <groupinformationsearch> is wrong.");
		return HTTP_BAD_REQUEST;
	}

	/* 入力値を格納するscreen を作成 */
	scntxt->screen.grpis_iscreen = apr_pcalloc(p, sizeof(divy_search_grpis_iscreen));
	scntxt->optflg = 0;

	/*
	 * XMLのパース
	 *
	 * [ DTD ]
	 * 	<!ELEMENT groupinformationsearch (
	 * 		((groupid | groupuri)?, (treelist | detaillist | availablesql)) |
	 * 		((groupid | groupuri | groupcollectionuri), (content | availableuser))) >
	 *	<!ELEMENT groupid            (#PCDATA) >
	 *	<!ELEMENT groupuri           (#PCDATA) >
	 *	<!ELEMENT groupcollectionuri (#PCDATA) >
	 *	<!ELEMENT treelist      EMPTY >
	 *	<!ELEMENT detaillist    EMPTY >
	 *	<!ELEMENT content       EMPTY >
	 *	<!ELEMENT availableuser EMPTY >
	 *	<!ELEMENT availablesql  EMPTY >
	 */
	for (elem = cur_elem; elem; elem = elem->next) {

		/* PCDATA の両端に含まれていたスペースを除去して取得する */
		data = divy_xml_get_cdata(elem, p, 1);

		if (strcmp(elem->name, "groupid") == 0) {
			grpid = data;
			scntxt->screen.grpis_iscreen->param_type = DIVY_SEARCH_GRPIS_USE_GRPID;
		}
		else if (strcmp(elem->name, "groupuri") == 0) {
			/* uri の正規化 */
			if (data != NULL && strlen(data) != 0) {
				char *s = apr_pstrcat(p, "/", data, NULL);
				s = dav_divy_remove_endslash(p, s);
				ap_no2slash(s);
				grpuri = s;
			}
			scntxt->screen.grpis_iscreen->param_type = DIVY_SEARCH_GRPIS_USE_GRPURI;
		}
		else if (strcmp(elem->name, "groupcollectionuri") == 0) {
			/* uri の正規化 */
			if (data != NULL && strlen(data) != 0) {
				char *s = apr_pstrcat(p, "/", data, NULL);
				s = dav_divy_remove_endslash(p, s);
				ap_no2slash(s);
				grpcol_uri = s;
			}
			scntxt->screen.grpis_iscreen->param_type = DIVY_SEARCH_GRPIS_USE_GRPCOLURI;
		}
		else if (strcmp(elem->name, "treelist") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_TREELIST;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "detaillist") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_DETAILLIST;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "content") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_CONTENT;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "availableuser") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_AVAILABLEUSER;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "availablesql") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_AVAILABLESQL;
			elem_cnt++;
		}
		else {
			/* 未知のエレメントが指定された */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"groupinformationsearch", elem->name);
			return HTTP_BAD_REQUEST;
		}
	}

	/*
	 * 取得結果の検証
	 */
	/* 具体的なSEARCH動作を示す指示エレメントが存在しなかった */
	if (scntxt->optflg == 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The request element (treelist / detaillist / content / availableuser / "
			"availablesql) of groupinformationsearch not found.");
		return HTTP_BAD_REQUEST;
	}
	/* 指示エレメントが2つ以上存在した */
	else if (elem_cnt > 1) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Too many request elements. The number of request element "
			"(treelist / detaillist / content / availableuser / availablesql) must be one.");
		return HTTP_BAD_REQUEST;
	}
	/* groupid , groupuri, groupcollectionuri が同時に指定されていた */
	else if (IS_FILLED(grpid) && IS_FILLED(grpuri) && IS_FILLED(grpcol_uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Could not specify the \"groupid\" and \"groupuri\" and \"groupcollectionuri\" "
			"element at the same time.");
		return HTTP_BAD_REQUEST;
	}
	/* availableuser または content では grpid / groupuri / groupcollectionuri が必須 */
	else if ((IS_EMPTY(grpid) && IS_EMPTY(grpuri) && IS_EMPTY(grpcol_uri)) &&
		(scntxt->optflg == DIVY_SEARCH_OPT_AVAILABLEUSER || scntxt->optflg == DIVY_SEARCH_OPT_CONTENT)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"groupid\" or \"groupuri\" or \"groupcollectionuri\" element is missing.");
		return HTTP_BAD_REQUEST;
	}
	/* groupcollectionuri は availableuser または content でのみ指定可能 */
	else if (IS_FILLED(grpcol_uri) &&
			(scntxt->optflg != DIVY_SEARCH_OPT_AVAILABLEUSER && scntxt->optflg != DIVY_SEARCH_OPT_CONTENT)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Could not specify \"groupcollectionuri\" element except availableuser or content search.");
		return HTTP_BAD_REQUEST;
	}
	/* ユーザ拡張ステータスがサポートされていなければgroupcollectionuri は拒否 */
	else if (IS_FILLED(grpcol_uri) && scntxt->optflg == DIVY_SEARCH_OPT_AVAILABLEUSER &&
			!divy_support_extenduserstatus(r)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Could not specify \"groupcollectionuri\" element.");
		return HTTP_BAD_REQUEST;
	}
	/* availablesql ではgrpid / groupuri が必須 */
	else if ((IS_EMPTY(grpid) && IS_EMPTY(grpuri)) && scntxt->optflg == DIVY_SEARCH_OPT_AVAILABLESQL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"groupid\" or \"groupuri\" element is missing.");
		return HTTP_BAD_REQUEST;
	}

	/* 複数階層のグループを検索するかどうか検証する */
	if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON) {
		/* groupid / groupuri がtreelistまたはdetaillist に指定された場合には何もしない(1階層しか表示しないから) */
		if ((IS_FILLED(grpid) || IS_FILLED(grpuri)) &&
			(scntxt->optflg == DIVY_SEARCH_OPT_TREELIST ||
			 scntxt->optflg == DIVY_SEARCH_OPT_DETAILLIST)) {
			scntxt->query = NULL;
			return HTTP_NOT_FOUND;
		}
	}

	/*
	 * 管理者 or グループリーダでないユーザ = 管理権限を持っていないユーザ群には
	 * availableuser と content (何れもプロパティ表示) 以外を許可しない
	 */
	if (!divy_rdbo_has_administrative_privilege(r) &&
		(scntxt->optflg != DIVY_SEARCH_OPT_AVAILABLEUSER && scntxt->optflg != DIVY_SEARCH_OPT_CONTENT)) {

		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"You could not access group property."
				"Need to administrive privilege. (access user = %s)", divy_get_userid(r));
		return HTTP_FORBIDDEN;
	}

	/*
	 * availableuser で groupcollectionuri が指定された時の処理
	 *
	 * * アクセスユーザが所属していないグループに対するアクセスは禁止する.
	 * * グループ制約のプロパティ表示制約があればアクセス禁止(404扱いにするけど)
	 * (note)
	 *   content && groupcollectionuri のケースはプロパティ表示制約で拒否しない仕様です.
	 *   だからここでは拒否しません.
	 *
	 * (2007/08/10 Fri)
	 *   非アクティブグループであれば、プロパティ表示禁止です.
	 */
	if (scntxt->optflg == DIVY_SEARCH_OPT_AVAILABLEUSER && IS_FILLED(grpcol_uri)) {
		divy_rdbo_grp *grp_pr = NULL;
		const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);

		/* 所属グループを取得する */
		(void) divy_get_cached_availablegroup_by_uri(r, grpcol_uri, &grp_pr);

		/* 所属グループが1つもない or 所属していないグループだった場合
		 * (note)
		 *   このSEARCHはグループコレクションとして見えているグループだけに
		 *   行われる. そのため、グループリーダのように所属していなくても見えてしまう
		 *   ユーザであっても、このSEARCHが流れることはない. だから所属グループだけを
		 *   気にすればいい.
		 * */
		if (grp_pr == NULL) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not access a group.(uri = %s, userid = %s)",
					grpcol_uri, divy_get_userid(r));
			return HTTP_FORBIDDEN;	/* アクセス違反とする */
		}

		/* グループ制約をサポートしていて、管理者 / (グループリーダ&自分自身) の
		 * グループ以外の場合には、プロパティ表示制約の有無を調べる */
		if (divy_support_grpconstraints(r) && divy_get_adminmode(r) != DIVY_ADMINMODE_ADMIN) {

			divy_rdbo_extstatus *grp_extstatus = NULL;
			int use_user_groupconstraints = 0;	/* グループ制約による影響を受けるかどうか(0: 受けない) */
			int skip = 0;

			/* グループリーダで自分自身のグループならば例外とする */
			if (divy_support_groupleader(r) && divy_rdbo_is_groupleader(own_extstatus)) {
				/* グループプロパティを取得 */
				if (IS_FILLED(grp_pr->ownerid) && strcmp(grp_pr->ownerid, divy_get_userid(r)) == 0) {
					skip = 1;
				}
			}

			if (skip == 0) {
				/* 制限ユーザ or グループ制約を"無視できない" ユーザか? */
				if (!divy_rdbo_is_trusty_user(own_extstatus) ||
						!divy_rdbo_has_user_groupconstraints_ignore(own_extstatus)) {
					use_user_groupconstraints = 1;
				}

				/* グループ拡張ステータスを取得 */
				(void) divy_get_cached_availablegroupextstatus(r, grpcol_uri, &grp_extstatus);

				/* グループ制約が適用されるべきユーザであり、プロパティ表示制約を持っているかどうか */
				if (use_user_groupconstraints && divy_rdbo_has_showprop_groupconstraints(grp_extstatus)) {
					return HTTP_NOT_FOUND;	/* 空画面にする */
				}
			}
		}
	}

	/* 入力値をscreen に記録 */
	scntxt->screen.grpis_iscreen->grpid      = grpid;	/* グループID */
	scntxt->screen.grpis_iscreen->grpuri     = grpuri;	/* グループ相対URI */
	scntxt->screen.grpis_iscreen->grpcol_uri = grpcol_uri;	/* グループコレクションURI */

	/* scntxt->optflg の修正 */
	if (IS_EMPTY(grpid) && IS_EMPTY(grpuri)) {
		if (scntxt->optflg == DIVY_SEARCH_OPT_TREELIST) {
			/* トップ検索設定 */
			scntxt->optflg = DIVY_SEARCH_OPT_TREELIST_TOP;
		}
		else if (scntxt->optflg == DIVY_SEARCH_OPT_DETAILLIST) {
			/* トップ検索設定 */
			scntxt->optflg = DIVY_SEARCH_OPT_DETAILLIST_TOP;
		}
	}

	return HTTP_OK;
}

/**
 * 検索結果から返信するXMLを作成する(groupinformationsearch)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  grp_pr divy_rdbo_grp * 検索結果
 * @param  divy_cset_t * 継承SQLのセット(availablesqlで使用)
 * @param  dav_response ** davのレスポンス格納構造体
 * @param  int 検索オプション
 * @return int HTTPコード
 */
static int grpis_build_xml_response(request_rec *r,
					divy_rdbo_grp *grp_pr,
					dav_response **res,
					int optflg)
{
	apr_pool_t *p           = r->pool;
	dav_response *newres 	= NULL;
	dav_response *firstres 	= NULL;

	*res = NULL;
			    
	/* availableuserはユーザでループしrespouse作成 */
	if (optflg == DIVY_SEARCH_OPT_AVAILABLEUSER){
		divy_rdbo_usr *usr;

        	for (usr = grp_pr->usr_pr; usr; usr = usr->next) {
			/* responseを１件作成 */
			newres = apr_pcalloc(p, sizeof(dav_response));

			grpis_usr_mkresponse(r, grp_pr, usr, newres, p);
			if (firstres == NULL) {
				firstres = *res = newres;
			}
			else {
				(*res)->next = newres;
				*res = newres;
			}
			(*res)->next = NULL;
		}
	}
	/* availablesqlはSQLでループしrespouse作成 */
	else if (optflg == DIVY_SEARCH_OPT_AVAILABLESQL){
		divy_rdbo_sql *sql;

		for (sql = grp_pr->sql_pr; sql; sql = sql->next){
			/* responseを１件作成 */
			newres = apr_pcalloc(p, sizeof(dav_response));

			grpis_sql_mkresponse(r, grp_pr, sql, newres, p);
			if (firstres == NULL) {
				firstres = *res = newres;
			}
			else {
				(*res)->next = newres;
				*res = newres;
			}
			(*res)->next = NULL;
		}
	}
	/* availableuser,availablesql以外はグループでループしrespouse作成 */
	else {
		divy_rdbo_grp *grp;

		for (grp = grp_pr; grp; grp = grp->next) {
			/* responseを１件作成 */
			newres = apr_pcalloc(p, sizeof(dav_response));

			grpis_grp_mkresponse(r, grp, optflg, newres, p);
			if (firstres == NULL) {
				firstres = *res = newres;
			}
			else {
				(*res)->next = newres;
				*res = newres;
			}
			(*res)->next = NULL;
		}
	}
	*res = firstres;
	return HTTP_OK;
}

/**
 * searchのresponceを作成する(groupinformationsearchのグループ情報)
 *
 *
 * @param r request_rec * apache request_rec構造体
 * @param grp divy_rdbo_grp * 取得済みのgroup
 * @param optflg int 検索オプション see search.h
 * @param res dav_response * davのレスポンス格納構造体
 * @param wp apr_pool_t * 作業用のプール
 */
static void grpis_grp_mkresponse(request_rec *r, divy_rdbo_grp *grp, int optflg,
					dav_response *res, apr_pool_t *wp)
{
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
						    
	res->href = divy_build_m_group_uri(wp, dav_divy_get_root_uri(r),
				apr_psprintf(wp, "%s/", grp->relativeuri));
	res->status = HTTP_OK;

	/* XML作成(共通部分) */
	s = apr_psprintf(wp, "<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":groupdiscovery>" CRLF 
			"<"DIVY_NS_PREFIX":groupinfo>" CRLF
			"<"DIVY_NS_PREFIX":groupid>"
			"%s"
			"</"DIVY_NS_PREFIX":groupid>" CRLF
			"<"DIVY_NS_PREFIX":name>"
			"%s"
			"</"DIVY_NS_PREFIX":name>" CRLF
			"<"DIVY_NS_PREFIX":creationdt>"
			"%s"
			"</"DIVY_NS_PREFIX":creationdt>" CRLF
			"<"DIVY_NS_PREFIX":updatedt>"
			"%s"
			"</"DIVY_NS_PREFIX":updatedt>" CRLF
			"<"DIVY_NS_PREFIX":comment>"
			"%s"
			"</"DIVY_NS_PREFIX":comment>" CRLF,
			grp->grpid,
			dav_divy_escape_xmlstr(wp, grp->name, DIVY_XML_T2T_CDATA),
			grp->registdt,
			grp->updatedt, 
			dav_divy_escape_xmlstr(wp, grp->comment, DIVY_XML_T2T_CDATA));
	apr_text_append(wp, &hdr, s);

	/* content, detaillist は groupcollectionuri, groupmailwatchを追加 */
	if (optflg == DIVY_SEARCH_OPT_CONTENT    ||
		optflg == DIVY_SEARCH_OPT_DETAILLIST ||
		optflg == DIVY_SEARCH_OPT_DETAILLIST_TOP) {

		divy_rdbo_mailwatch *mw = grp->mailwatch_pr;

		/* 拡張ユーザステータスをサポートしていれば返却する */
		if (IS_FILLED(grp->grpcol_uri) && divy_support_extenduserstatus(r)) {
			s = apr_psprintf(wp,
					"<"DIVY_NS_PREFIX":groupcollectionuri>"
					"%s/"
					"</"DIVY_NS_PREFIX":groupcollectionuri>" CRLF,
					dav_divy_escape_uri(wp, grp->grpcol_uri));
				apr_text_append(wp, &hdr, s);
		}

		if (mw == NULL){
			/* mwなしは空タグ */
			s = apr_pstrcat(wp, 
					"<"DIVY_NS_PREFIX":groupmailwatch>" 
					CRLF
					"<"DIVY_NS_PREFIX":triggermethod/>", 
					CRLF
					"</"DIVY_NS_PREFIX":groupmailwatch>"
					CRLF, NULL);
		}
		else {
			int i;
			s = NULL;	/* 初期化 */
			for (i = 0; i < METHODS; i++){
				if (mw->trigger_methods[i] == 1){
					if (!s){
						s = apr_pstrdup(wp, ap_method_name_of(wp, i));
					} else {
						s = apr_pstrcat(wp, s, " ", ap_method_name_of(wp, i), NULL);
					}
				}
			}
			/* タグをつける */
			if (s){
				s = apr_pstrcat(wp, 
				"<"DIVY_NS_PREFIX":groupmailwatch>" 
					CRLF
				"<"DIVY_NS_PREFIX":triggermethod>", 
				s,
				"</"DIVY_NS_PREFIX":triggermethod>" 
				CRLF
				"</"DIVY_NS_PREFIX":groupmailwatch>"
				CRLF, NULL);
			}
			else {
				/* データはあるがフラグが全て0
				 * (正しくない) 
				 * WARNING 出力 */
				ERRLOG0(r->pool, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
						"The mailwatch property exists in database "
						"but all triggers are 0.");
				s = apr_pstrcat(wp, 
					"<"DIVY_NS_PREFIX":groupmailwatch>" 
					CRLF
					"<"DIVY_NS_PREFIX":triggermethod/>", 
					CRLF
					"</"DIVY_NS_PREFIX":groupmailwatch>"
					CRLF, NULL);
			}
		}
		apr_text_append(wp, &hdr, s);
	}
	/* グループ拡張ステータスを追加する */
	_grpis_build_grp_extstatus(r, grp, &hdr, wp);

	/* グループリーダプロパティを追加する */
	if (divy_support_groupleader(r)) {
		_grpis_build_groupleader(r, grp, &hdr, wp);

		/* treelist の場合、Child グループ数を追加 */
		if (optflg == DIVY_SEARCH_OPT_TREELIST || optflg == DIVY_SEARCH_OPT_TREELIST_TOP) {
			s = apr_psprintf(wp,
					"<"DIVY_NS_PREFIX":childcount>"
					"%d"
					"</"DIVY_NS_PREFIX":childcount>" CRLF, grp->childcount);
			apr_text_append(wp, &hdr, s);
		}
	}

#ifdef DIVY_SUPPORT_GROUPQUOTA
	/* グループQuotaプロパティを追加する */
	if (divy_support_groupquota(r)) {
		_grpis_build_groupquota(r, grp, &hdr,wp);
	}
#endif /* DIVY_SUPPORT_GROUPQUOTA */

	/* クローズ */
	apr_text_append(wp, &hdr, "</"DIVY_NS_PREFIX":groupinfo>" CRLF
			"</"DIVY_NS_PREFIX":groupdiscovery>" CRLF 
			"</"DAV_NS_PREFIX":prop>" CRLF 
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);
	/* ネームスペース追加 */
	apr_text_append(wp, &hdr_ns, divy_make_liveprop_ns(wp, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));
					    
	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
}

/**
 * searchのresponceを作成する(groupinformationsearchのユーザ情報)
 *
 *
 * @param r request_rec * apache request_rec構造体
 * @param grp divy_rdbo_grp * 取得済みのgroup
 * @param usr divy_rdbo_usr * 取得済みのuser
 * @param res dav_response * davのレスポンス格納構造体
 * @param wp apr_pool_t * 作業用のプール
 */
static void grpis_usr_mkresponse(request_rec *r, divy_rdbo_grp *grp, divy_rdbo_usr *usr,
					dav_response *res, apr_pool_t *wp)
{
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	const char *ru_usrid	= apr_psprintf(wp, DIVY_PREFIX_RUSER"%s", usr->usrid);
						    
	res->href = divy_build_m_group_uri(wp, dav_divy_get_root_uri(r), 
			apr_psprintf(wp, "%s/%s/", grp->relativeuri, ru_usrid));
	res->status = HTTP_OK;

	/* XML作成 */
	s = apr_psprintf(wp, "<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":userdiscovery>" CRLF 
			"<"DIVY_NS_PREFIX":userinfo>" CRLF
			"<"DIVY_NS_PREFIX":userid>"
			"%s"
			"</"DIVY_NS_PREFIX":userid>" CRLF
			"<"DIVY_NS_PREFIX":name>%s</"DIVY_NS_PREFIX":name>" CRLF
			"<"DIVY_NS_PREFIX":mailaddr>"
			"%s"
			"</"DIVY_NS_PREFIX":mailaddr>" CRLF
			"<"DIVY_NS_PREFIX":creationdt>"
			"%s"
			"</"DIVY_NS_PREFIX":creationdt>" CRLF
			"<"DIVY_NS_PREFIX":updatedt>"
			"%s""</"DIVY_NS_PREFIX":updatedt>" CRLF,
			dav_divy_escape_xmlstr(wp, usr->usrid,    DIVY_XML_T2T_CDATA),
			dav_divy_escape_xmlstr(wp, usr->fullname, DIVY_XML_T2T_CDATA),
			dav_divy_escape_xmlstr(wp, usr->mailaddr, DIVY_XML_T2T_CDATA),
			usr->registdt, usr->updatedt);
	apr_text_append(wp, &hdr, s);

	/* 管理者の場合はタグ追加 (下位互換性のために残してあります) */
	if (usr->adminmode == DIVY_ADMINMODE_ADMIN){
		apr_text_append(wp, &hdr, "<"DIVY_NS_PREFIX":admin/>" CRLF);
	}

	/* メール監視が存在するなら追加 */
	if (usr->mwatch_pr == NULL) {
		apr_text_append(wp, &hdr, "<"DIVY_NS_PREFIX":groupmailwatch>"
								  "<"DIVY_NS_PREFIX":triggermethod/>"
								  "</"DIVY_NS_PREFIX":groupmailwatch>"
								  CRLF);
	}
	else {
		int i;
		s = NULL; /* 初期化 */
		for (i = 0; i < METHODS; i++) {
			if (usr->mwatch_pr->trigger_methods[i] == 1) {
				if (!s) {
					s = apr_pstrdup(wp, ap_method_name_of(wp, i));
				} else {
					s = apr_pstrcat(wp, s, " ", ap_method_name_of(wp, i), NULL);
				}
			}
		}

		if (s) {
			s = apr_pstrcat(wp, "<"DIVY_NS_PREFIX":groupmailwatch>" 
								CRLF
								"<"DIVY_NS_PREFIX":triggermethod>", 
								s,
								"</"DIVY_NS_PREFIX":triggermethod>" 
								CRLF
								"</"DIVY_NS_PREFIX":groupmailwatch>"
								CRLF, NULL);
		}
		apr_text_append(wp, &hdr, s);
	}

	/* ユーザ拡張ステータスを入れる */
	if (divy_support_extenduserstatus(r)) {
		_useris_build_extstatus(r, usr, NULL, &hdr, wp);
	}

	/* グループリーダの情報を入れる */
	if (divy_support_groupleader(r)) {
		_useris_build_groupleader(r, usr, &hdr, wp);
	}
    
	/* グループクォータの情報を入れる */
	if (divy_support_groupquota(r)) {
		/* FIXME add code group quota*/
	}

	/* クローズ */
	apr_text_append(wp, &hdr, "</"DIVY_NS_PREFIX":userinfo>" CRLF
			"</"DIVY_NS_PREFIX":userdiscovery>" CRLF 
			"</"DAV_NS_PREFIX":prop>" CRLF 
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);
	/* ネームスペース追加 */
	apr_text_append(wp, &hdr_ns, divy_make_liveprop_ns(wp, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));
					    
	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
}

/**
 * searchのresponceを作成する(groupinformationsearchのSQL情報)
 *
 *
 * @param r request_rec * apache request_rec構造体
 * @param grp divy_rdbo_grp * 取得済みのgroup
 * @param sql divy_rdbo_sql * 取得済みのsql
 * @param res dav_response * davのレスポンス格納構造体
 * @param wp apr_pool_t * 作業用のプール
 */
static void grpis_sql_mkresponse(request_rec *r, divy_rdbo_grp *grp, divy_rdbo_sql *sql,
					dav_response *res, apr_pool_t *wp)
{
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	const char *rs_sql	= apr_psprintf(wp, DIVY_PREFIX_RSQL "%s", sql->labelname);
						    
	res->href = divy_build_m_group_uri(wp, dav_divy_get_root_uri(r), 
			apr_psprintf(wp, "%s/%s/", grp->relativeuri, rs_sql));
	res->status = HTTP_OK;

	/* XML作成 */
	s = apr_psprintf(wp, "<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":sqldiscovery>" CRLF 
			"<"DIVY_NS_PREFIX":sqlinfo>" CRLF
			"<"DIVY_NS_PREFIX":sqlno>"
			"%s"
			"</"DIVY_NS_PREFIX":sqlno>" CRLF
			"<"DIVY_NS_PREFIX":name>"
			"%s"
			"</"DIVY_NS_PREFIX":name>" CRLF,
			sql->sqlid,
			dav_divy_escape_xmlstr(wp, sql->labelname,
						DIVY_XML_T2T_CDATA));
	apr_text_append(wp, &hdr, s);
	/* RequiredSQL,名前付きバインド変数はsubname 設定 それ以外は空タグ */
	if (sql->type == DIVY_SQL_TYPE_REQUIRED ||
			sql->type == DIVY_SQL_TYPE_BIND){
		s = apr_psprintf(wp, "<"DIVY_NS_PREFIX":subname>"
				"%s"
				"</"DIVY_NS_PREFIX":subname>" CRLF,
				TO_EMPTY(sql->subname));
		apr_text_append(wp, &hdr, s);
	} else {
		apr_text_append(wp, &hdr, "<"DIVY_NS_PREFIX":subname/>" CRLF);
	}

	/* sqltype 作成 */
	apr_text_append(wp, &hdr,"<"DIVY_NS_PREFIX":sqltype>");
	switch (sql->type){
		case DIVY_SQL_TYPE_NORMAL:
			apr_text_append(wp, &hdr,
					"<"DIVY_NS_PREFIX":normalsql/>");
			break;
		case DIVY_SQL_TYPE_REPOSITORY:
			apr_text_append(wp, &hdr,
					"<"DIVY_NS_PREFIX":reposdbsql/>");
			break;
		case DIVY_SQL_TYPE_REQUIRED:
			apr_text_append(wp, &hdr,
					"<"DIVY_NS_PREFIX":reqsql/>");
			break;
		case DIVY_SQL_TYPE_BIND:
			apr_text_append(wp, &hdr,
					"<"DIVY_NS_PREFIX":namedbind/>");
			break;
		default:
			break;
	}
	apr_text_append(wp, &hdr,"</"DIVY_NS_PREFIX":sqltype>" CRLF);

	s = apr_psprintf(wp, "<"DIVY_NS_PREFIX":usedbmsname>"
			"%s"
			"</"DIVY_NS_PREFIX":usedbmsname>" CRLF
			"<"DIVY_NS_PREFIX":creationdt>"
			"%s"
			"</"DIVY_NS_PREFIX":creationdt>" CRLF
			"<"DIVY_NS_PREFIX":updatedt>"
			"%s"
			"</"DIVY_NS_PREFIX":updatedt>" CRLF
			"<"DIVY_NS_PREFIX":comment>"
			"%s"
			"</"DIVY_NS_PREFIX":comment>" CRLF,
			dav_divy_escape_xmlstr(wp, sql->usedbmsname,
						DIVY_XML_T2T_CDATA),
			sql->registdt, sql->updatedt, 
			dav_divy_escape_xmlstr(wp, sql->comment,
						DIVY_XML_T2T_CDATA));
	apr_text_append(wp, &hdr, s);

	/* inactivesqlの場合はタグ追加 */
	if (sql->active == DIVY_SQL_INACTIVE){
		apr_text_append(wp, &hdr, 
				"<"DIVY_NS_PREFIX":inactivesql/>" CRLF);
	}
   
	/* 敬称されたSQL の場合はタグ追加 */
	if (sql->inheritsql == DIVY_SQL_INHERIT_TRUE){
		apr_text_append(wp, &hdr,
				"<"DIVY_NS_PREFIX":inheritsql/>" CRLF);
	}

	/* クローズ */
	apr_text_append(wp, &hdr, "</"DIVY_NS_PREFIX":sqlinfo>" CRLF
			"</"DIVY_NS_PREFIX":sqldiscovery>" 
			CRLF "</"DAV_NS_PREFIX":prop>" CRLF 
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);
	/* ネームスペース追加 */
	apr_text_append(wp, &hdr_ns, divy_make_liveprop_ns(wp, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));
					    
	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
}

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
/**
 * get_response ハンドラにセットされる関数 (groupinformationsearch)
 */
static int _grpis_get_response(request_rec *r, divy_rdbo_search_output *output,
					dav_response *response, apr_pool_t *wp)
{
	divy_rdbo_grp *grp_pr = output->list.grp_pr;

	if (grp_pr == NULL) return DIVY_SEARCH_RESPONSE_NONE;

	/* availableuser の場合 */
	if (output->optflg == DIVY_SEARCH_OPT_AVAILABLEUSER) {
		if (grp_pr->usr_pr == NULL) return DIVY_SEARCH_RESPONSE_NONE;

		grpis_usr_mkresponse(r, grp_pr, grp_pr->usr_pr, response, wp);
	}
	/* availablesql の場合 */
	else if (output->optflg == DIVY_SEARCH_OPT_AVAILABLESQL) {
		divy_rdbo_sql *sql_pr = grp_pr->sql_pr;

		if (sql_pr == NULL) return DIVY_SEARCH_RESPONSE_NONE;
		grpis_sql_mkresponse(r, grp_pr, sql_pr, response, wp);
	}
	/* 上記以外の場合 */
	else {
		grpis_grp_mkresponse(r, grp_pr, output->optflg, response, wp);
	}

	return 0;
}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

/**
 *  XMLからクエリーを作成する(sqlinformationsearch)
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @param  stype_elem apr_xml_elem * 解析済みのXML
 * @param  search_cntxt * 作成されたクエリーを格納する
 * @return int HTTPコード
 */
static int sqlis_build_query(request_rec *r, apr_xml_elem *stype_elem, 
						search_cntxt *scntxt)
{
	apr_pool_t *p		= r->pool;
	apr_xml_elem *cur_elem  = stype_elem;
	const char *cdata	= NULL;
	apr_xml_elem *tmp_elem;
	divy_rdbo_sql *sql	= NULL;
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);

	/*
	 * XMLのパース
	 *
	 * [ DTD ]
	 * 	<!ELEMENT sqlinformationsearch (
	 * 		treelist | detaillist | analysis |
	 * 		(sqlno, content | availablegroup | querycontent | dependsqllist | sqlpreference) >
	 * 	<!ELEMENT sqlno          (#PCDATA) >
	 * 	<!ELEMENT treelist       EMPTY >
	 * 	<!ELEMENT detaillist     EMPTY >
	 * 	<!ELEMENT analysis       EMPTY >
	 * 	<!ELEMENT content        EMPTY >
	 * 	<!ELEMENT availablegroup EMPTY >
	 * 	<!ELEMENT querycontent   EMPTY >
	 * 	<!ELEMENT dependsqllist  EMPTY >
	 * 	<!ELEMENT sqlpreference  EMPTY >
	 */
    
	/* エラーチェック */
	if (stype_elem == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"<sqlinformationsearch> has no child.");
		return HTTP_BAD_REQUEST;
	}

	/* 1番基本的なtreelistの条件で初期化 */
	scntxt->select = apr_pstrdup(p, "SELECT"
					" sql.sql_id_c"
		    			", sql.sql_label_name_vc");
	scntxt->from = apr_pstrdup(p, " FROM divy_sql sql");
	scntxt->where = NULL;

	/* treelist */
	if (strcmp(cur_elem->name, "treelist") == 0){
		/* 設定済みなのでエラーチェック・フラグ設定のみ */
		if (cur_elem->first_child){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<treelist> elem has child.");
			return HTTP_BAD_REQUEST;
		}
		if (cur_elem->next){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<treelist> elem has next.");
			return HTTP_BAD_REQUEST;
		}

		/* フラグ設定 */
		scntxt->optflg = DIVY_SEARCH_OPT_TREELIST;
	}
	/* detaillist */
	else if (strcmp(cur_elem->name, "detaillist") == 0){
		/* エラーチェック */
		if (cur_elem->first_child || cur_elem->next){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<detaillist> has child or next.");
			return HTTP_BAD_REQUEST;
		}

		/* SQL追加 */
		scntxt->select = apr_pstrcat(p, scntxt->select,
						", sql.sql_sub_name_vc",
						", sql.sql_type_i",
						", sql.sql_active_i",
						", sql.sql_comment_vc",
						", sql.sql_regist_c",
						", sql.sql_update_c",
						", dbms.ds_id_name_vc", NULL);
		scntxt->from = apr_pstrcat(p, scntxt->from,
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_DB2) /* postgres / db2 */
				" INNER JOIN divy_dbms dbms"
				" ON (sql.sql_ds_id_c = dbms.ds_ds_id_c)",
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				", divy_dbms dbms",
#endif
				NULL);

		/* JOIN があるのでoracleのみwhere句を作成 */
#if defined(DIVY_DBMS_ORACLE) /* oracle */
		scntxt->where = apr_pstrdup(p, 
				" WHERE sql.sql_ds_id_c = dbms.ds_ds_id_c");
#endif
		/* フラグ設定 */
		scntxt->optflg = DIVY_SEARCH_OPT_DETAILLIST;
	}
	/* sqlno */
	else if (strcmp(cur_elem->name, "sqlno") == 0){

		/* sqlno取得 取得できない場合はエラー */
		cdata = divy_xml_get_cdata(cur_elem, p, 1);
		if (IS_EMPTY(cdata)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"<sqlno> needs PCDATA.");
			return HTTP_BAD_REQUEST;
		}
		/* 数値でなければエラー */
		else if (!dav_divy_isdigit_str(cdata)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The value of \"sqlno\" element must be digit. "
				"(sqlno = %s)", cdata);
			return HTTP_BAD_REQUEST;
		}

		/* エラーチェック */
		if (cur_elem->first_child){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<sqlno> has child.");
			return HTTP_BAD_REQUEST;
		} 
		if (!cur_elem->next){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<sqlno> must have option.");
			return HTTP_BAD_REQUEST;
		} 
		if (cur_elem->next->next){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<sqlno> has too many options.");
			return HTTP_BAD_REQUEST;
		}
		cur_elem = cur_elem->next;

		/* availablegroup */
		if (strcmp(cur_elem->name, "availablegroup") == 0) {
			/* エラーチェック */
			if (cur_elem->first_child){
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<availablegroup> should not have child.");
				return HTTP_BAD_REQUEST;
			}

			/* SQLが基本と違うので書き直し */
			scntxt->select = apr_pstrdup(p,
					"SELECT"
					" sqlm.sqlm_sql_id_c"
					", sqlm.sqlm_grp_id_c"
					", grp.grp_name_vc"
					", grp.grp_comment_vc"
					", grp.grp_regist_c"
					", grp.grp_update_c"
					", sql.sql_label_name_vc");

			if (support_grpconstraints || support_groupleader) {
				scntxt->select = apr_pstrcat(p, scntxt->select,
					", grp.grp_extended_status_c", NULL);
			}

			if (support_groupleader) {
				scntxt->select = apr_pstrcat(p, scntxt->select,
					", grp.grp_owner_usr_id_vc"
					", usr2.usr_fullname_vc AS ownername", NULL);
			}
			scntxt->from = apr_pstrdup(p, 
				" FROM (divy_sqlmem sqlm"
				" INNER JOIN divy_grp grp"
				" ON (sqlm.sqlm_grp_id_c = grp.grp_grp_id_c))"
				" INNER JOIN divy_sql sql"
				" ON (sqlm.sqlm_sql_id_c = sql.sql_id_c)");

			if (support_groupleader) {
				scntxt->from = apr_pstrcat(p, scntxt->from,
					" LEFT JOIN divy_usr usr2"
					" ON (grp.grp_owner_usr_id_vc = usr2.usr_usr_id_vc)", NULL);
			}

			/* where句作成 */
			scntxt->where = apr_psprintf(p, 
				" WHERE sqlm.sqlm_sql_id_c = '%s'", cdata);
			/* フラグ設定 */
			scntxt->optflg = DIVY_SEARCH_OPT_AVAILABLEGROUP;
		}
		/* content */
		else if (strcmp(cur_elem->name, "content") == 0) {
			/* エラーチェック */
			if (cur_elem->first_child){
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<content> should not have child.");
				return HTTP_BAD_REQUEST;
			}

			/* グループリーダにはcontent を許可しない */
			if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not get sql property for groupleader.");
				return HTTP_FORBIDDEN;
			}

			/* select 句追加 */
			scntxt->select = apr_pstrcat(p, scntxt->select,
					", sql.sql_sub_name_vc",
					", sql.sql_type_i",
					", sql.sql_cachemode_i",
					", sql.sql_stmt_txt",
					", sql.sql_active_i",
					", sql.sql_comment_vc",
					", dbms.ds_id_name_vc",
					", sql.sql_regist_c",
					", sql.sql_update_c",
					", sqlc.sqlc_where_id_i",
					", sqlc.sqlc_where_contype_i",
					", sqlc.sqlc_where_name_vc",
					", sqlc.sqlc_where_colname_vc",
					", sqlc.sqlc_where_fmt_vc",
					", sqlc.sqlc_depsql_sub_name_vc",
					", sqlc.sqlc_depsql_mode_i",
					", sqlc.sqlc_where_sql_position_i",
					NULL);
			/* from句は連結テーブルが増えたため()を入れるので
			 * 書き直し */
			scntxt->from = apr_pstrdup(p, 
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_DB2) /* postgres / db2 */
				" FROM (divy_sql sql"
				" INNER JOIN divy_dbms dbms"
				" ON (sql.sql_ds_id_c = dbms.ds_ds_id_c))"
				" LEFT OUTER JOIN divy_sqlcontent sqlc"
				" ON (sql.sql_id_c = sqlc.sqlc_id_c)"
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				" FROM divy_sql sql, divy_dbms dbms, "
				"divy_sqlcontent sqlc"
#endif
				);
			/* where句新規作成 */
			scntxt->where = apr_psprintf(p,
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_DB2) /* postgres / db2 */
				" WHERE sql.sql_id_c = '%s'"
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				" WHERE sql.sql_ds_id_c = dbms.ds_ds_id_c"
				" AND sql.sql_id_c = sqlc.sqlc_id_c(+)"
				" AND sql.sql_id_c = '%s'"
#endif
				" ORDER BY sqlc.sqlc_where_sql_position_i",
				cdata);
			/* フラグ設定 */
			scntxt->optflg = DIVY_SEARCH_OPT_CONTENT;
		}
		/* querycontent */
		else if (strcmp(cur_elem->name, "querycontent") == 0){
			/* select 句追加 */
			scntxt->select = apr_pstrcat(p, scntxt->select,
					", sql.sql_sub_name_vc",
					", sql.sql_type_i",
					", sql.sql_comment_vc",
					", dbms.ds_id_name_vc",
					", sqlc.sqlc_where_id_i",
					", sqlc.sqlc_where_contype_i",
					", sqlc.sqlc_where_name_vc",
					", sqlc.sqlc_where_fmt_vc",
					", sqlc.sqlc_depsql_sub_name_vc",
					", sqlc.sqlc_depsql_mode_i",
					NULL);
			/* from句は連結テーブルが増えたため()を
			 * 入れるので書き直し */
			scntxt->from = apr_pstrdup(p, 
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_DB2) /* postgres / db2 */
					" FROM (divy_sql sql"
					" INNER JOIN divy_dbms dbms"
					" ON (sql.sql_ds_id_c ="
					" dbms.ds_ds_id_c))"
					" LEFT OUTER JOIN divy_sqlcontent sqlc"
					" ON (sql.sql_id_c = sqlc.sqlc_id_c)"
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
					" FROM divy_sql sql, divy_dbms dbms,"
					" divy_sqlcontent sqlc"
#endif
					);
			/* where句新規作成 */
			/* ##### FIXME ソート順を決める事
			 * 		SQL作成ウィザードに関連します 
			 */
			scntxt->where = apr_psprintf(p,
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_DB2) /* postgres / db2 */
					" WHERE sql.sql_id_c = '%s'"
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
					" WHERE sql.sql_ds_id_c = dbms.ds_ds_id_c"
					" AND sql.sql_id_c = sqlc.sqlc_id_c(+)"
					" AND sql.sql_id_c = '%s'"
#endif
					" AND sql.sql_active_i = 1"
					" ORDER BY sqlc.sqlc_where_contype_i,"
					" sqlc.sqlc_where_id_i", cdata);
			/* sqldepend 検索用ハッシュ作成 
			 * required,名前つきバインドなしの場合は使用しません */
			scntxt->sqlis_sqldtopid_h = apr_hash_make(p);
			apr_hash_set(scntxt->sqlis_sqldtopid_h, cdata,
					APR_HASH_KEY_STRING, "");

			scntxt->rsvalue = NULL;	/* 初期化 */

			/* child あり(クライアントにキャッシュがある)場合 
			 * 構造体を作成します */
			if (cur_elem->first_child){
				apr_xml_elem *rsval_elem;
				divy_rdbo_rsvalue *rs 		= NULL;
				divy_rdbo_rsvalueset *rsvals 	= NULL;


				/* rsvalue をループ */
				for (cur_elem = cur_elem->first_child; cur_elem; cur_elem = cur_elem->next) {
					rsvals = NULL;	/* 初期化 */

					/* エラーチェック */
					if (strcmp(cur_elem->name, "rsvalue") != 0 || !cur_elem->first_child) {

						ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"<rsvalue> have Syntax error.");
						return HTTP_BAD_REQUEST;
					}
					rsval_elem = cur_elem->first_child;

					/* エラーチェック */
					if (strcmp(rsval_elem->name, "reqsqlname") != 0 ||
						rsval_elem->first_child || !rsval_elem->next) {

						ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"<reqsqlname> have syntax error.");
						return HTTP_BAD_REQUEST;
					}
					cdata = NULL;	/* 初期化 */
					/* 
	 				 * reqsqlname 取得(なしの場合はエラー)
					 */
					cdata = divy_xml_get_cdata(rsval_elem, p, 1); 
					if (IS_EMPTY(cdata)) {
						ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"<reqsqlname> must have PCDATA.");
						return HTTP_BAD_REQUEST;
					}

					/* 
	 				 * reqsqlname が 名前付きバインド変数
					 * ($$Bxxx) 以外の場合はエラー
					 * 名前付きバインド変数以外の場合は
					 * displayvalue を返却できないので
					 * この仕様としました。
					 */
					if (!IS_NBIND_NODE(cdata)) {
						ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"<reqsqlname> must have subname of namedbind.");
						return HTTP_BAD_REQUEST;
					}

					/* 領域確保(rsvalue) */
					if (!rs){
						rs = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
						scntxt->rsvalue = rs;
					} else {
						rs->next = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
						rs = rs->next;
					}
					/* name 格納 rsvalueset 初期化 */
					rs->name 	= apr_pstrdup(p, cdata);
					rsvals		= NULL;

					/* reqsqlvalue をループ */
					for (rsval_elem = rsval_elem->next; rsval_elem;
							rsval_elem = rsval_elem->next) {

						/* エラーチェック */
						if (strcmp(rsval_elem->name, "reqsqlvalue") != 0 ||
								rsval_elem->first_child) {

							ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
								"<reqsqlvalue> have syntax error.");
							return HTTP_BAD_REQUEST;
						}
						cdata = NULL;	/* 初期化 */
						/* 
	 				 	 * reqsqlvalue 取得
						 * (なしの場合はエラー)
					 	 */
						/* reqsqlvalue は入力値を
						 * そのまま取得(dav関数使用) */
						cdata = dav_xml_get_cdata(rsval_elem, p, 0); 
						if (IS_EMPTY(cdata)) {
							ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
								"<reqsqlvalue> must have PCDATA.");
							return HTTP_BAD_REQUEST;
						}

						/* 領域確保(rsvalueset) */
						if (!rsvals){
							rsvals = apr_pcalloc(p, sizeof(divy_rdbo_rsvalueset));
							rs->valueset = rsvals;
						} else {
							rsvals->next = apr_pcalloc(p, sizeof(divy_rdbo_rsvalueset));
							rsvals = rsvals->next;
						}
						/* value 格納 */
						rsvals->value = apr_pstrdup(p, cdata);
					}
				}
			}
			/* フラグ設定 */
			scntxt->optflg = DIVY_SEARCH_OPT_QUERYCONTENT;
		}
		/* dependsqllist */
		else if (strcmp(cur_elem->name, "dependsqllist") == 0){
			/* エラーチェック */
			if (cur_elem->first_child){
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<dependsqllist> should not have child.");
				return HTTP_BAD_REQUEST;
			}
			/* SQLが基本と違うので書き直し(トップのURL取得) */
			scntxt->select = apr_pstrdup(p, "SELECT"
					" sql_relative_uri_txt");
			scntxt->from = apr_pstrdup(p, " FROM divy_sql");
			scntxt->where = apr_psprintf(p, " WHERE sql_id_c = '%s'", cdata);
			/* sqldepend 検索用ハッシュ作成 */
			scntxt->sqlis_sqldtopid_h = apr_hash_make(p);
			apr_hash_set(scntxt->sqlis_sqldtopid_h, cdata, APR_HASH_KEY_STRING, "");

			/* フラグ設定 */
			scntxt->optflg = DIVY_SEARCH_OPT_DEPENDSQLLIST;
		}
		/* sqlpreference */
		else if (strcmp(cur_elem->name, "sqlpreference") == 0){

			scntxt->sqlis_cachedreq = NULL;	/* 初期化 */

			/* SQLが基本と違うので書き直し(トップSQL URI取得) */
			scntxt->select = apr_pstrdup(p, "SELECT"
					" sql_relative_uri_txt");
			scntxt->from = apr_pstrdup(p, " FROM divy_sql");
			scntxt->where = apr_psprintf(p, " WHERE sql_id_c = '%s'", cdata);
			/* sqldepend 検索用ハッシュ作成 */
			scntxt->sqlis_sqldtopid_h = apr_hash_make(p);
			apr_hash_set(scntxt->sqlis_sqldtopid_h, cdata,
					APR_HASH_KEY_STRING, "");

			cdata = NULL;	/* 初期化 */
			/* 
			 * reqsqlname ありの場合はcache 済みのreqsql サブ名称の
			 * ハッシュを作成します
			 */
			if (cur_elem->first_child){
				scntxt->sqlis_cachedreq	= apr_hash_make(p);
			}

			for (cur_elem = cur_elem->first_child; cur_elem;
						cur_elem = cur_elem->next){
				/* エラーチェック */
				if (strcmp(cur_elem->name, "reqsqlname") != 0 || cur_elem->first_child) {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
						"Child of <sqlpreference> is wrong.");
					return HTTP_BAD_REQUEST;
				}
				/* 
	 			 * reqsqlname 取得(なしの場合はエラー)
				 */
				cdata = divy_xml_get_cdata(cur_elem, p, 1); 
				if (IS_EMPTY(cdata)) {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
						"<reqsqlname> must have PCDATA.");
					return HTTP_BAD_REQUEST;
				}
				/* ハッシュにセット */
				apr_hash_set(scntxt->sqlis_cachedreq, cdata, APR_HASH_KEY_STRING, "");
			}

			/* フラグ設定 */
			scntxt->optflg = DIVY_SEARCH_OPT_SQLPREFERENCE;
		}
		else {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<%s> is wrong for option of <sqlno>.",
					cur_elem->name);
			return HTTP_BAD_REQUEST;
		}
	}
	/* analysis
	 * (note) analysis のXML 解析では値がない、などの場合も
	 * 	エラーにしません。
	 * 	すべてのチェックは exec_analysis で行いエラー情報を
	 * 	クライアントに返却します
	 */
	else if (strcmp(cur_elem->name, "analysis") == 0){

		if (!cur_elem->first_child ||
			strcmp(cur_elem->first_child->name, "sqldiscovery") != 0 || cur_elem->next) {
			scntxt->sqlis_analysis_sql = NULL;
			return HTTP_OK;
		}
		cur_elem = cur_elem->first_child;
		if (!cur_elem->first_child ||
			strcmp(cur_elem->first_child->name, "sqlinfo") != 0 || cur_elem->first_child->next){
			scntxt->sqlis_analysis_sql = NULL;
			return HTTP_OK;
		}

		/* 領域確保 */
		sql = apr_pcalloc(p, sizeof(divy_rdbo_sql));
		scntxt->sqlis_analysis_sql = sql;

		/* 初期化 */
		sql->active 	  = DIVY_SQL_ACTIVE;
		sql->type 	  = DIVY_SQL_TYPE_UNKNOWN;
		sql->cachemode	  = DIVY_SQL_NOCACHE;
		scntxt->newsqlflg = 0;

		/* 
		 * name,statement,sqltype,(cachemode,subname,inactive) 
		 * タグを探す
		 */
		for (tmp_elem = cur_elem->first_child->first_child; tmp_elem;
				tmp_elem = tmp_elem->next){
			if (strcmp(tmp_elem->name, "name") == 0) {
				/* name 取得 */
				/* name は入力値をそのまま取得のため
				 * strip_whit = 0 */
				cdata = divy_xml_get_cdata(tmp_elem, p, 0); 
				if (IS_FILLED(cdata)) {
					sql->labelname = apr_pstrdup(p, cdata);
				}
			} else if (strcmp(tmp_elem->name, "statement") == 0) {
				/* SQL 文取得 */
				cdata = divy_xml_get_cdata(tmp_elem, p, 1); 
				if (IS_FILLED(cdata)) {
					sql->sql = apr_pstrdup(p, cdata);
				}
			} else if (strcmp(tmp_elem->name, "sqltype") == 0) {
				/* sqltype 取得 */
				if (tmp_elem->first_child){
					if (strcmp(tmp_elem->first_child->name, "normalsql") == 0) {
						sql->type = DIVY_SQL_TYPE_NORMAL;
					} else if (strcmp(tmp_elem->first_child->name, "reposdbsql") == 0) {
						sql->type = DIVY_SQL_TYPE_REPOSITORY;
					} else if (strcmp(tmp_elem->first_child->name, "reqsql") == 0) {
						sql->type = DIVY_SQL_TYPE_REQUIRED;
					} else if (strcmp(tmp_elem->first_child->name, "namedbind") == 0) {
						sql->type = DIVY_SQL_TYPE_BIND;
					} else {
					}
				}
			} else if (strcmp(tmp_elem->name, "cachemode") == 0) {
				/* cachemode 取得 */
				if (tmp_elem->first_child){
					if (strcmp(tmp_elem->first_child->name, "cache") == 0) {
						sql->cachemode = DIVY_SQL_CACHE;
					}
				}
			} else if (strcmp(tmp_elem->name, "subname") == 0) {
				/* subname 取得 */
				cdata = divy_xml_get_cdata(tmp_elem, p, 1); 
				if (IS_FILLED(cdata)) {
					sql->subname = apr_pstrdup(p, cdata);
				}
			} else if (strcmp(tmp_elem->name, "inactivesql") == 0){
				/* inactive 設定 */
				sql->active = DIVY_SQL_INACTIVE;
			} else if (strcmp(tmp_elem->name, "newsql") == 0){
				/* フラグ設定 */
				scntxt->newsqlflg = 1;
			}
		}

		/* フラグ設定 */
		scntxt->optflg = DIVY_SEARCH_OPT_ANALYSIS;
		return HTTP_OK;
	} else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"<%s> is wrong for child of "
				"<sqlinformationsearch>.", cur_elem->name);
		return HTTP_BAD_REQUEST;
	}

	/* クエリーを接続(whereは存在する場合のみ) */
	scntxt->query = apr_pstrcat(p, scntxt->select, scntxt->from, NULL);
	if (scntxt->where){
		scntxt->query = apr_pstrcat(p, scntxt->query, scntxt->where, 
						NULL);
	}

	return HTTP_OK;
}

/**
 * analysis を実行する(sqlinformationsearch)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_sql * 解析されるSQL が格納された構造体
 * @param  divy_rdbo_resource ** 結果格納構造体
 * @return int ステータスコード <- see sqlparser.h
 */
static int sqlis_exec_analysis(request_rec *r, divy_rdbo_sql *sql_pr, 
				search_cntxt *scntxt,
				divy_rdbo_resource **rdb_r)
{
	apr_pool_t 		*p = r->pool;
	divy_sql_parser 	*parser;
	apr_hash_t 		*reqsql_h = NULL;
	divy_rdbo_sqlcontent 	*sqlc = NULL, *sqlc_first;
	apr_hash_t		*namelist_h = NULL;
	apr_hash_index_t 	*hashind;
	const void		*hkey;
	divy_rdbo_clist 	*namelist = NULL;
	divy_rdbo_sqldepend 	*depend, *depend_top, *depend_top_last,
				*depend_db_id, *depend_db_all;
	divy_cset_t		*cset_t;
	divy_rdbo_sql		*old_sql_pr = NULL;
	divy_rdbo_clist		*subname_cl = NULL, *first_subname_cl = NULL;
	apr_hash_t		*sql_pr_h   = NULL;
	const char 		*sql_uri;
	int result, do_inactive, do_change_type, do_delete_defval;
	dav_divy_server_conf	*sconf;

	*rdb_r 		= NULL;

	/*
	 * FIXME analysis とMKCOL & PROPFIND のチェックロジックは
	 * 	コピーコードでなければならないのに、コードがあまりにも
	 * 	違いすぎる。
	 * 	何とか一緒にできないものだろうか？
	 */

	/* sqlparser 作成 */
	if (divy_sql_parser_create(p, &parser) != DIVY_SQLP_ST_OK){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to create sql parser.");
		return DIVY_SQLP_ST_IERR;
	}

	/* sqldiscovery がない(URI が作れないので BAD_REQUEST) */
	if (sql_pr == NULL){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sqldiscovery\" element is missing.");
		return DIVY_SQLP_ST_ERR;
	}

	/* 
	 * SQL表示名称
	 */
	if (IS_EMPTY(sql_pr->labelname)) {
		/* labelnameなし (URI が作れないので BAD_REQUEST) */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"name\" element is missing or empty.");
		return DIVY_SQLP_ST_ERR;
	}

	/* SQL表示名称(禁則) */
	if (IS_FILLED(sql_pr->labelname) &&
	    divy_validate_pathsegment(p, 0, sql_pr->labelname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The \"name\" element has invalid char");
		/* 禁則文字チェックはクライアントが既にやっているのでエラー扱い */
		return DIVY_SQLP_ST_ERR;
	}

	/* 結果格納構造体に割り当て 
	 * エラー時もURI として SQL表示名称 を必要とするので 
	 * 先に割り当てます */
	*rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
	(*rdb_r)->sql_pr = sql_pr;

	sql_uri = apr_psprintf(p, "/%s", sql_pr->labelname);
	/* 
	 * SQL名称が登録済みか？
	 */
	/* 変更前のプロパティを取得 */
	if (divy_rdbo_get_sql_property(r, sql_uri, &old_sql_pr) != 0){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"An error occurred while getting "
			"sqlproperty of srcsql.(sqluri = %s)", sql_uri);
		return DIVY_SQLP_ST_IERR;
	}
	/* 新規で登録済みの場合 */
	if (scntxt->newsqlflg == 1 && old_sql_pr){
		return DIVY_SQLP_ST_NEWSQL_LABELNAME;
	}
	/* 更新で未登録の場合 */
	else if (scntxt->newsqlflg == 0 && !old_sql_pr){
		return DIVY_SQLP_ST_OLDSQL_LABELNAME;
	}

	/* sqltype */
	if (sql_pr->type == DIVY_SQL_TYPE_UNKNOWN) {
		return DIVY_SQLP_ST_SQLTYPE_UNKNOWN;
	}

	/* 名前付きバインド変数構文以外ではSQL文が存在すること */
	if (sql_pr->type != DIVY_SQL_TYPE_BIND && (IS_EMPTY(sql_pr->sql))) {
		return DIVY_SQLP_ST_MISSING_SQLSTMT;
	}

	/* DB機能が利用できない時には"通常SQL"を許可してはならない */
	sconf = dav_divy_get_server_config(r->server);
	if (sconf->use_db_opt != 1 && sql_pr->type == DIVY_SQL_TYPE_NORMAL) {
		return DIVY_SQLP_ST_LACKOF_LICENSE;
	}

	/* サブ名称 */
	result = divy_sql_parser_validate_subname(parser, r, sql_pr->type,
						sql_uri, scntxt->newsqlflg,
						sql_pr->subname);
	if (result != DIVY_SQLP_ST_OK) {
		return result;
	}

	/* cachemode */
	if (sql_pr->cachemode == DIVY_SQL_CACHE) {
		result = divy_sql_parser_validate_cachemode(parser,
					sql_pr->type, sql_pr->subname,
					sql_pr->sql, sql_pr->cachemode);
		if (result != DIVY_SQLP_ST_OK) {
			return result;
		}
	}

	/* SQL 文法チェック */
	result = divy_sql_parser_validate_reqsql(parser, sql_pr->type, 
						sql_pr->sql);
	if (result != DIVY_SQLP_ST_OK){
		return result;
	}
	/* リポジトリ検索SQL のSELECT句は正しいか？ */
	if (sql_pr->type == DIVY_SQL_TYPE_REPOSITORY){
		/* SELECT句の検証 */
		result = divy_sql_parser_validate_selectcol(parser, 
								sql_pr->sql, 
								&namelist_h);
		if (result != DIVY_SQLP_ST_OK) {
			/* MUST なカラムなし */
			if (result == DIVY_SQLP_ST_MUSTCOL_NF && namelist_h) {
				for (hashind = apr_hash_first(p, namelist_h);
					hashind; 
					hashind = apr_hash_next(hashind)){

					/* namelist を作成 */
					apr_hash_this(hashind, &hkey, NULL, NULL);

					if (!namelist){
        					(*rdb_r)->search_pr = 
						    apr_pcalloc(p, 
						    sizeof(divy_rdbo_search));
						namelist = apr_pcalloc(p,
						    sizeof(divy_rdbo_clist));
						(*rdb_r)->search_pr->namelist =
							namelist;
					} else {
						namelist->next = apr_pcalloc(p,
						    sizeof(divy_rdbo_clist));
						namelist = namelist->next;
					}
					namelist->val = apr_pstrdup(p, hkey);
				}
			}
			return result;
		}
	}

	/* where 句プロパティ作成 */
	if (sql_pr->type != DIVY_SQL_TYPE_BIND) {

		result = divy_sql_parser_make_whereprop(parser, sql_pr->sql,
				&sqlc);
		if (result != DIVY_SQLP_ST_OK){
			return result;
		}
	}
	else {
		/* 名前付きバインド変数構文のWHERE句プロパティは存在しない */
		sqlc = NULL;
	}

	/* WHERE句プロパティが存在する場合 */
	if (sqlc) {

		/* SQL 文中のRequiredSQL,名前付きバインド変数を取得する */
		result = divy_sql_parser_extract_reqsql(parser, sql_pr->sql, 
							&reqsql_h);
		if (result != DIVY_SQLP_ST_OK){
			return result;
		}

		/* SQL 文中のRequiredSQL,名前付きバインド変数が
		 * DB にあるかチェック */
		if (reqsql_h){
			result = divy_rdbo_check_required(r, reqsql_h, 
					&namelist_h, 1, NULL);
		}
		else {
			result = 0;
		}
		if (result != 0) {
			/* DB に存在しないreqsql がある */
			if (result == DIVY_STCODE_REQUIRED_NF && namelist_h){
				for (hashind = apr_hash_first(p, namelist_h);
					hashind; 
					hashind = apr_hash_next(hashind)){

					/* namelist を作成 */
					apr_hash_this(hashind, &hkey, NULL,
							NULL);

					if (!namelist){
        					(*rdb_r)->search_pr = 
						    apr_pcalloc(p, 
						    sizeof(divy_rdbo_search));
						namelist = apr_pcalloc(p,
						    sizeof(divy_rdbo_clist));
						(*rdb_r)->search_pr->namelist =
							namelist;
					} else {
						namelist->next = apr_pcalloc(p,
						    sizeof(divy_rdbo_clist));
						namelist = namelist->next;
					}
					namelist->val = apr_pstrdup(p, hkey);
				}
				return DIVY_SQLP_ST_REQSQL_NF;
			} else {
				ERRLOG0(p, APLOG_ERR, 
					DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to check existence of "
					"Required sql.");
				return DIVY_SQLP_ST_IERR;
			}
		}
	}

/* 
 * start dependlist チェック
 * かなり難解になってしまったためどのタイプのSQLに関係なく
 * まずはdependlistを作成し、必要に応じてチェックする方針に変更しました。
 */
	/* 初期化 */
	depend		= NULL;	/* ループに使用するtemp変数 */
	depend_top	= NULL;	/* SQL文中に持つrequiredsqlのdependlist
				   (DB未登録のため仮の値) */
	depend_top_last	= NULL;	/* depend_top の最後部
				   この後にDBから取得したdependlist を接続 */
	depend_db_id	= NULL;	/* DB から取得したdependlist(topid指定) */
	depend_db_all	= NULL;	/* DB から取得したdependlist(すべて) */

	/* SQL文中にrequiredsql がある場合 */
	if (reqsql_h){
		/* SQL文中の requiredsql(divy_dependlist未登録)の
		 * 仮 dependlist 作成 */
		for (hashind = apr_hash_first(p, reqsql_h); hashind; 
				hashind = apr_hash_next(hashind)){

			apr_hash_this(hashind, &hkey, NULL, NULL);

			if (!depend_top){
        			depend = apr_pcalloc(p, 
						sizeof(divy_rdbo_sqldepend));
				depend_top = depend;
			} else {
				depend->next = apr_pcalloc(p,
						sizeof(divy_rdbo_sqldepend));
				depend = depend->next;
			}

			/* sqltype により親subname をつけます */
			if (sql_pr->type == DIVY_SQL_TYPE_REQUIRED) {
				depend->ptsubname = sql_pr->subname;
			} else {
				/* 通常・リポジトリは仮の値 */
				depend->ptsubname = DIVY_SQL_TOPPTVALUE;
			}
			depend->clsubname = apr_pstrdup(p, hkey);
		}

		/* この後にDB から取得したdependlist を接続するため
		 *  最後をとっておく */
		depend_top_last = depend;

		/* 自身が持つrequiredsqlをtopid とするdependlistをdbから取得 */
		result = divy_rdbo_get_sqldepend_list(r, reqsql_h,
							&depend_db_id, NULL);
		if (result != 0){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"An error occurred while getting "
				"sqldepend of RequiredSQL.");
			return DIVY_SQLP_ST_IERR;
		}
	}

	/* DBにある全dependlist 取得 */
	result = divy_rdbo_get_sqldepend_list(r, NULL, &depend_db_all, NULL);
	if (result != 0){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"An error occurred while getting "
			"sqldepend of RequiredSQL.");
		return DIVY_SQLP_ST_IERR;
	}

	/* 
	 * 最大連鎖数のチェック
	 * SQL 文中にrequiredsqlがある場合
	 */
	if (reqsql_h){

		/* 仮topidとdbのdependlist(topid指定)を接続 */
		if (depend_db_id){
			depend_top_last->next = depend_db_id;
		}

		result = divy_sql_parser_validate_maxchain(parser, depend_top);
		if (result != DIVY_SQLP_ST_OK) {
			return result;
		}
	}

	/* 
	 * 閉回路のチェック
	 * SQL 文中にrequiredsqlがある、かつ自身が Required の場合
	 */
	if (reqsql_h && sql_pr->type == DIVY_SQL_TYPE_REQUIRED){

		/* 仮topidとdbのdependlist(topid指定)を接続 */
		if (depend_db_id){
			depend_top_last->next = depend_db_id;
		}

		result = divy_sql_parser_validate_closedpath(parser, depend_top);
		if (result != DIVY_SQLP_ST_OK) {
			return result;
		}
	}

	/* どの種類のチェックをおこなうのか判定する */
	do_inactive      = 0;	/* 非アクティブチェックをするかどうか */
	do_change_type   = 0;	/* sqltype変更チェックをするかどうか  */
	do_delete_defval = 0;	/* デフォルト値チェックをするかどうか */

	if (scntxt->newsqlflg == 0) {
		/* 非アクティブにしようとしていた */
		if (sql_pr->active == DIVY_SQL_INACTIVE &&
		    (sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
		     sql_pr->type == DIVY_SQL_TYPE_BIND)) {
			do_inactive = 1;
		}

		/* sqltype を変えようとしていた */
		if ((sql_pr->type == DIVY_SQL_TYPE_NORMAL ||
		     sql_pr->type == DIVY_SQL_TYPE_REPOSITORY) &&
		    (old_sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
		     old_sql_pr->type == DIVY_SQL_TYPE_BIND)) {
			do_change_type = 1;
		}

		/* デフォルト値を消そうとしていた */
		if (sql_pr->type == DIVY_SQL_TYPE_BIND && (IS_EMPTY(sql_pr->sql))) {
			do_delete_defval = 1;
		}
	}

	/* 
	 * (検証開始条件)
	 *	・更新処理である
	 *	・sql_pr のSQLがRequiredSQL、名前付きバインド変数構文ならば、
	 * 	  アクティブフラグを非アクティブに変更しようとしている
	 * 	  or
	 * 	  sql_pr のSQLが通常SQL・リポジトリSQLならば、sqltype を
	 * 	  RequiredSQLまたは名前付きバインド変数から変更しようとしている
	 * 	  or
	 * 	  sql_pr が名前付きバインド変数構文で、
	 * 	  他のSQLからhidden指定されていて、
	 * 	  デフォルト値を消そうとしている
	 */
	if (scntxt->newsqlflg == 0 &&
		(do_inactive || do_change_type || do_delete_defval)) {

		char *subname;
		if (sql_pr->type == DIVY_SQL_TYPE_NORMAL ||
		    sql_pr->type == DIVY_SQL_TYPE_REPOSITORY) {
			subname = old_sql_pr->subname;	/* 昔の値を利用 */
		}
		else {
			subname = sql_pr->subname;
		}

		/* SQL 文中にrequiredsqlがある場合 */
		if (reqsql_h) {
			/* 仮topidとdbのdependlist(すべて)を接続 */
			if (depend_db_all){
				depend_top_last->next = depend_db_all;
			}
			result = divy_sql_parser_find_usingsql(parser, subname,
							depend_top, &cset_t);
		} else {
			result = divy_sql_parser_find_usingsql(parser, subname,
							depend_db_all, &cset_t);
		}

		/* 他から利用されていた */
		if (result == DIVY_SQLP_ST_USINGSQL_FOUND && cset_t) {
			namelist_h = NULL;
			namelist   = NULL;

			/* 非アクティブに失敗していた or sqltypeの変更に失敗 */
			if (do_inactive || do_change_type) {
				/* 使用SQL ハッシュ(id と subname)から
				 * ラベル名称を取得 */
				if (divy_rdbo_get_sql_labelname_by_idsubname(r,
						divy_cset_tohash(cset_t),
						&namelist_h, NULL) != 0) {

					ERRLOG0(p, APLOG_ERR,
						DIVY_FST_IERR + DIVY_SST_PROC,
						"An error occurred "
						"while getting "
						"sqllabelname with dependency.");
					return DIVY_SQLP_ST_IERR;
				}
			}
			/* バインド変数のデフォルト値をなくす場合 */
			else if (do_delete_defval) {
				/* 使用SQL ハッシュ(id と subname),
				 * sqlcontent の subname,値取得モード
				 * からラベル名称を取得 */
				if (divy_rdbo_get_sql_labelname_by_idsubname_sqlcsubnamemode(r,
						 divy_cset_tohash(cset_t),
						 sql_pr->subname,
						 DIVY_SQL_MODE_HIDDEN, 
						 &namelist_h, NULL) != 0) {

					ERRLOG0(p, APLOG_ERR,
						DIVY_FST_IERR + DIVY_SST_PROC,
						"An error occurred "
						"while getting "
						"sqllname with dependency.");
					return DIVY_SQLP_ST_IERR;
				}

				/* HIDDEN で使用されていた */
				if (namelist_h != NULL) {
					/* ステータス切り替え */
					result = DIVY_SQLP_ST_EMPTY_HIDDEN_NBIND;
				}
			}

			/* 使用SQLがありの場合 */
			if (namelist_h != NULL) {
				/* namelist 作成 */
				for (hashind = apr_hash_first(p, namelist_h);
					hashind; hashind = apr_hash_next(hashind)) {

					/* 名称の取得 */
					apr_hash_this(hashind, &hkey, NULL, NULL);

					if (!namelist) {
						(*rdb_r)->search_pr = apr_pcalloc(p,
							sizeof(divy_rdbo_search));
						namelist = apr_pcalloc(p,
							sizeof(divy_rdbo_clist));
						(*rdb_r)->search_pr->namelist =
							namelist;
					}
					else {
						namelist->next = apr_pcalloc(p,
							sizeof(divy_rdbo_clist));
						namelist = namelist->next;
					}
					namelist->val = apr_pstrdup(p, hkey);
				}
				return result;
			}
		}
		else if (result != DIVY_SQLP_ST_OK) {
			return result;
		}
	}

/*
 * end dependlist チェック
 */
	/*
	 * WHERE句プロパティのSQLモードリストを生成する
	 */
	sqlc_first = sqlc; /* sqlcontent の先頭を取っておく */

	/* 
	 * sqlmodelist 作成 
	 */
	for ( ; sqlc; sqlc = sqlc->next) {

		/* RequiredSQL, 名前付きバインド変数以外はスキップ */
		if (sqlc->contype != DIVY_SQL_CONTYPE_REQUIRED) continue;

		/* WHERE句プロパティとして$$Bxxx が指定されていた場合 */
		if (IS_NBIND_NODE(sqlc->reqsql->rsvalue->name)) {
			/* (note)
			 *  これだけを特別扱いしたのはループの中でSQLを引き
			 *  たくなかったため。まずは集めておいて、後で
			 *  デフォルト値があるかどうか一括して調べる
			 */
			if (subname_cl == NULL) {
				subname_cl = first_subname_cl =
					apr_pcalloc(p, sizeof(divy_rdbo_clist));
			}
			else {
				subname_cl->next =
					apr_pcalloc(p, sizeof(divy_rdbo_clist));
				subname_cl = subname_cl->next;
			}
			subname_cl->val = apr_pstrdup(p, sqlc->reqsql->rsvalue->name);
			subname_cl->next = NULL;
		}
		else {
			/* sqlmodelist の設定 */
			(void) divy_sql_parser_get_sqlmodelist(parser,
						sql_pr->type,
						sqlc->reqsql->rsvalue->name,
						DIVY_SQL_NO_DEFAULT_VAL,
						&sqlc->reqsql->sqlmodelist);
		}
	}

	/*
	 * WHERE句に指定された名前付きバインド変数がデフォルト値を
	 * 持っていたかどうかを調べる
	 */

	/* divy_sql テーブルを検索して
	 * 名前付きバインド変数のSQL文を取得する */
	sql_pr_h = NULL;
	if (first_subname_cl &&
	    divy_rdbo_get_sql_by_subname(r, first_subname_cl,
		    			 &sql_pr_h, NULL) != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"An error occured while getting value "
			"of namedbind.");
		return DIVY_SQLP_ST_IERR;
	}

	/* ループしながら値を取得する */
	for (sqlc = sqlc_first; first_subname_cl && sqlc; sqlc = sqlc->next) {

		/* 名前付きバインド変数以外には興味なし */
		if (sqlc->contype != DIVY_SQL_CONTYPE_REQUIRED ||
		    !IS_NBIND_NODE(sqlc->reqsql->rsvalue->name)) {
			continue;
		}

		/* DB には存在しない名前付きバインド変数が
		 * WHERE句プロパティとして指定されていた(不整合) */
		if (sql_pr_h == NULL ||
		    ((sql_pr = apr_hash_get(sql_pr_h,
					sqlc->reqsql->rsvalue->name,
					APR_HASH_KEY_STRING)) && sql_pr == NULL)) {

			/* ちゃんとエラーも報告できないので
			 * 		大目に見ておきましょう */
			(void) divy_sql_parser_get_sqlmodelist(parser,
						sql_pr->type,
						sqlc->reqsql->rsvalue->name,
						DIVY_SQL_NO_DEFAULT_VAL,
						&sqlc->reqsql->sqlmodelist);
			continue;
		}

		/* SQL文をパースして値があるかどうかを調べる */
		result = divy_sql_parser_is_nbind_setting(parser, sql_pr->sql);
		if (result == DIVY_SQLP_ST_FOUND_NBIND_VAL) {

			/* sqlmodelist の設定 */
			(void) divy_sql_parser_get_sqlmodelist(parser,
						sql_pr->type,
						sqlc->reqsql->rsvalue->name,
						DIVY_SQL_HAS_DEFAULT_VAL,
						&sqlc->reqsql->sqlmodelist);
		}
		/* なければshow */
		else if (result == DIVY_SQLP_ST_NF_NBIND_VAL) {

			/* sqlmodelist の設定 */
			(void) divy_sql_parser_get_sqlmodelist(parser,
						sql_pr->type,
						sqlc->reqsql->rsvalue->name,
						DIVY_SQL_NO_DEFAULT_VAL,
						&sqlc->reqsql->sqlmodelist);
		}
		else {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"An error occured while checking "
				"value of namedbind.");

			/* Internal Server Error */
			return DIVY_SQLP_ST_IERR;
		}

	}

	/* 結果格納構造体に割り当て */
	if (sqlc_first){
		(*rdb_r)->sql_pr->sqlcnt_pr = apr_pcalloc(p,
					sizeof(divy_rdbo_sqlcontent));
		(*rdb_r)->sql_pr->sqlcnt_pr = sqlc_first;
	} else {
		(*rdb_r)->sql_pr->sqlcnt_pr = NULL;
	}
 
	return DIVY_SQLP_ST_OK;
}

/**
 * 検索結果から返信するXMLを作成する(sqlinformationsearch)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_resource *検索結果
 * @paramr dav_response ** davのレスポンス格納構造体
 * @param  int 検索オプション
 * @return int HTTPコード
 */
static int sqlis_build_xml_response(request_rec *r, 
					divy_rdbo_resource *search_result, 
					dav_response **res, int analysisstcode, 
					int optflg)
{
	dav_response *newres 	= NULL;
	dav_response *firstres 	= NULL;
	divy_rdbo_sql *sql;
	        
	*res = NULL;

	/* availablegroupはグループでループしrespouse作成 */
	if (optflg == DIVY_SEARCH_OPT_AVAILABLEGROUP){
		divy_rdbo_grp *grp;

		for (grp = search_result->sql_pr->grp_pr; grp;
						grp = grp->next){
			/* responseを１件作成 */
			newres = sqlis_grp_mkresponse(r, search_result->sql_pr,
							grp);
			if (firstres == NULL) {
				firstres = *res = newres;
			}
			else {
				(*res)->next = newres;
				*res = newres;
			} 
			(*res)->next = NULL;
		}   
	}
	/* dependsqllist,sqlpreference はresource から1件のみ作成 */
	else if (optflg == DIVY_SEARCH_OPT_DEPENDSQLLIST ||
			optflg == DIVY_SEARCH_OPT_SQLPREFERENCE){
		/* responseを１件作成 */
		firstres = sqlis_reqsqlinfo_mkresponse(r, search_result,
							optflg);
		firstres->next = NULL;
	}
	/* querycontent で Failed Dependency は sql でループして
	 * 作成 
	 * 特殊なので他と分けています */
	else if (optflg == DIVY_SEARCH_OPT_QUERYCONTENT_FD){

		/* sql でループ */
		for (sql = search_result->sql_pr; sql; sql = sql->next){
			/* responseを１件作成 */
			newres = sqlis_faileddependency_mkresponse(r, sql);

			if (firstres == NULL) {
				firstres = *res = newres;
			}
			else {
				(*res)->next = newres;
				*res = newres;
			}
			(*res)->next = NULL;
		}
	}
	/* analysis で 文法エラー は １件のみでエラーコードを渡す
	 * 特殊なので他と分けています */
	else if (optflg == DIVY_SEARCH_OPT_ANALYSIS_SQLERR){
		/* responseを１件作成 */
		firstres = sqlis_analysiserr_mkresponse(r, search_result,
							analysisstcode);
		firstres->next = NULL;
	}
	/* 上記以外はSQLでループしrespouse作成 */
	else {

		for (sql = search_result->sql_pr; sql; sql = sql->next){
	    
			/* responseを１件作成 */
			newres = sqlis_sql_mkresponse(r, search_result, sql, 
							optflg);
			if (firstres == NULL) {
				firstres = *res = newres;
			}
			else {
				(*res)->next = newres;
				*res = newres;
				if (optflg == DIVY_SEARCH_OPT_CONTENT ||
				    optflg == DIVY_SEARCH_OPT_ANALYSIS){
					/* 次の構造体に dbms 情報を渡す */
					sql->next->dbms_pr = sql->dbms_pr;
				}
			}
			(*res)->next = NULL;
		}
	}
	*res = firstres;
	return HTTP_OK;
}

/**
 * searchのresponceを作成する(sqlinformationsearchのSQL情報)
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_resource * 検索結果
 * @param  divy_rdbo_sql * 取得済みのSQL
 * @param  int 検索オプション
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *sqlis_sql_mkresponse(request_rec *r, 
					divy_rdbo_resource *resource,
					divy_rdbo_sql *sql, int optflg)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	divy_rdbo_sqlcontent 	*sqlc;
	divy_rdbo_dbms 		*dbms;
	apr_hash_index_t *hashind;
	const char *hkey;
	divy_rdbo_rsvalue 	*rsval;
	divy_rdbo_rsvalueset 	*rsvals;
	int i;

	res->href = divy_build_m_sql_uri(p, dav_divy_get_root_uri(r), 
				apr_psprintf(p, "%s/", sql->labelname));
	res->status = HTTP_OK;

	/* XML作成 共通部分 */
	apr_text_append(p, &hdr, "<"DAV_NS_PREFIX":propstat>" CRLF 
				"<"DAV_NS_PREFIX":prop>" CRLF
				"<"DIVY_NS_PREFIX":sqldiscovery>" CRLF
				"<"DIVY_NS_PREFIX":sqlinfo>" CRLF);

	/* analysis 以外は sqlno 作成 */
	if (optflg != DIVY_SEARCH_OPT_ANALYSIS){
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":sqlno>"
				"%s"
				"</"DIVY_NS_PREFIX":sqlno>" CRLF, 
				sql->sqlid);
		apr_text_append(p, &hdr, s);
	}

	/* 検索オプションによって分岐 */
	if (optflg == DIVY_SEARCH_OPT_TREELIST){
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":name>"
				"%s"
				"</"DIVY_NS_PREFIX":name>" CRLF,
				dav_divy_escape_xmlstr(p, sql->labelname,
					DIVY_XML_T2T_CDATA));
		apr_text_append(p, &hdr, s);
	} else if (optflg == DIVY_SEARCH_OPT_DETAILLIST){
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":name>"
				"%s"
				"</"DIVY_NS_PREFIX":name>" CRLF
				"<"DIVY_NS_PREFIX":subname>"
				"%s"
				"</"DIVY_NS_PREFIX":subname>" CRLF,
				dav_divy_escape_xmlstr(p, sql->labelname,
							DIVY_XML_T2T_CDATA),
				TO_EMPTY(sql->subname));
		apr_text_append(p, &hdr, s);

		/* sqltype 作成 */
		apr_text_append(p, &hdr,"<"DIVY_NS_PREFIX":sqltype>");
		switch (sql->type){
			case DIVY_SQL_TYPE_NORMAL:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":normalsql/>");
				break;
			case DIVY_SQL_TYPE_REPOSITORY:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":reposdbsql/>");
				break;
			case DIVY_SQL_TYPE_REQUIRED:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":reqsql/>");
				break;
			case DIVY_SQL_TYPE_BIND:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":namedbind/>");
				break;
			default:
				break;
		}
		apr_text_append(p, &hdr,"</"DIVY_NS_PREFIX":sqltype>" CRLF);

		/* 非アクティブの場合タグを追加 */
		if (sql->active == DIVY_SQL_INACTIVE){
			apr_text_append(p, &hdr, 
					"<"DIVY_NS_PREFIX":inactivesql/>" CRLF);
		}

		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":usedbmsname>"
				"%s"
				"</"DIVY_NS_PREFIX":usedbmsname>" CRLF
				"<"DIVY_NS_PREFIX":creationdt>"
				"%s"
				"</"DIVY_NS_PREFIX":creationdt>" CRLF
				"<"DIVY_NS_PREFIX":updatedt>"
				"%s"
				"</"DIVY_NS_PREFIX":updatedt>" CRLF
				"<"DIVY_NS_PREFIX":comment>"
				"%s"
				"</"DIVY_NS_PREFIX":comment>" CRLF,
				dav_divy_escape_xmlstr(p, sql->usedbmsname,
							DIVY_XML_T2T_CDATA),
				sql->registdt, sql->updatedt, 
				dav_divy_escape_xmlstr(p, sql->comment,
							DIVY_XML_T2T_CDATA));
		apr_text_append(p, &hdr, s);

	} else if (optflg == DIVY_SEARCH_OPT_CONTENT){
		sqlc 	= sql->sqlcnt_pr;
		dbms	= resource->dbms_pr;
	    
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":name>"
				"%s"
				"</"DIVY_NS_PREFIX":name>" CRLF,
				dav_divy_escape_xmlstr(p, sql->labelname,
					DIVY_XML_T2T_CDATA));
		apr_text_append(p, &hdr, s);

		/* RequiredSQL,名前付きバインド変数はsubname 設定
		 * それ以外は空タグ */
		if (sql->type == DIVY_SQL_TYPE_REQUIRED ||
				sql->type == DIVY_SQL_TYPE_BIND){
			s = apr_psprintf(p, "<"DIVY_NS_PREFIX":subname>"
					"%s"
					"</"DIVY_NS_PREFIX":subname>" CRLF,
					TO_EMPTY(sql->subname));
			apr_text_append(p, &hdr, s);
		} else {
			apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":subname/>" CRLF);
		}

		/* sqltype 作成 */
		apr_text_append(p, &hdr,"<"DIVY_NS_PREFIX":sqltype>");
		switch (sql->type){
			case DIVY_SQL_TYPE_NORMAL:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":normalsql/>");
				break;
			case DIVY_SQL_TYPE_REPOSITORY:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":reposdbsql/>");
				break;
			case DIVY_SQL_TYPE_REQUIRED:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":reqsql/>");
				break;
			case DIVY_SQL_TYPE_BIND:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":namedbind/>");
				break;
			default:
				break;
		}
		apr_text_append(p, &hdr,"</"DIVY_NS_PREFIX":sqltype>" CRLF);

		/* cachemode 作成 */
		apr_text_append(p, &hdr,"<"DIVY_NS_PREFIX":cachemode>");
		if (sql->cachemode == DIVY_SQL_CACHE && 
				sql->type == DIVY_SQL_TYPE_BIND){
			/* DB 値がキャッシュで名前付きバインド変数の場合 */
			apr_text_append(p, &hdr,"<"DIVY_NS_PREFIX":cache/>");
		} else {
			/* 上記以外 */
			apr_text_append(p, &hdr,"<"DIVY_NS_PREFIX":nocache/>");
		}
		apr_text_append(p, &hdr,"</"DIVY_NS_PREFIX":cachemode>" CRLF);

		/* 非アクティブの場合タグを追加 */
		if (sql->active == DIVY_SQL_INACTIVE){
			apr_text_append(p, &hdr, 
					"<"DIVY_NS_PREFIX":inactivesql/>" CRLF);
		}

		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":statement>"
				"%s"
				"</"DIVY_NS_PREFIX":statement>" CRLF
				"<"DIVY_NS_PREFIX":usedbmsname>"
				"%s"
				"</"DIVY_NS_PREFIX":usedbmsname>" CRLF,
				dav_divy_escape_xmlstr(p, sql->sql,
							DIVY_XML_T2T_CDATA),
				dav_divy_escape_xmlstr(p, sql->usedbmsname,
							DIVY_XML_T2T_CDATA));
		apr_text_append(p, &hdr, s);

		/* dbmslist(なしの場合は空タグ) */
		if (resource->dbms_pr){
			apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":dbmslist>" CRLF);
		} else {
			apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":dbmslist/>" CRLF);
		}

		for (dbms = resource->dbms_pr; dbms; dbms = dbms->next){
   			 
			s = apr_psprintf(p, 
				"<"DIVY_NS_PREFIX":dbmsentry>"
				"%s"
				"</"DIVY_NS_PREFIX":dbmsentry>" CRLF,
				dav_divy_escape_xmlstr(p, dbms->name,
							DIVY_XML_T2T_CDATA));
			apr_text_append(p, &hdr, s);
		}
		if (resource->dbms_pr){
			apr_text_append(p, &hdr,
					"</"DIVY_NS_PREFIX":dbmslist>" CRLF);
		}

		/* usingsqllist(なしの場合は空タグ) */
		if (sql->usingsql_hash){
			apr_text_append(p, &hdr,
				"<"DIVY_NS_PREFIX":usingsqllist>" CRLF);
			for (hashind = apr_hash_first(p, sql->usingsql_hash);
					hashind;
					hashind = apr_hash_next(hashind)){
				/* キー(依存しているSQLのlabelname)を取得 */
				apr_hash_this(hashind, (const void **)&hkey, NULL, NULL);

				s = apr_psprintf(p, 
					"<"DIVY_NS_PREFIX":usingsqlname>"
					"%s"
					"</"DIVY_NS_PREFIX":usingsqlname>" CRLF,
					hkey);
				apr_text_append(p, &hdr, s);
			}
			apr_text_append(p, &hdr,
				"</"DIVY_NS_PREFIX":usingsqllist>" CRLF);
		} else {
			apr_text_append(p, &hdr,
				"<"DIVY_NS_PREFIX":usingsqllist/>" CRLF);
		}

		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":creationdt>"
				"%s"
				"</"DIVY_NS_PREFIX":creationdt>" CRLF
				"<"DIVY_NS_PREFIX":updatedt>"
				"%s"
				"</"DIVY_NS_PREFIX":updatedt>" CRLF
				"<"DIVY_NS_PREFIX":comment>"
				"%s"
				"</"DIVY_NS_PREFIX":comment>" CRLF,
				sql->registdt, sql->updatedt,
				dav_divy_escape_xmlstr(p, sql->comment,
							DIVY_XML_T2T_CDATA));
		apr_text_append(p, &hdr, s);

		for (sqlc = sql->sqlcnt_pr; sqlc; sqlc = sqlc->next){
			/* id なしの場合はループ終了 */
			if (!sqlc->id) break;
    
			s = apr_psprintf(p, 
				"<"DIVY_NS_PREFIX":keyvalue id=\"%d\">" CRLF
				"<"DIVY_NS_PREFIX":columnname>"
				"%s"
				"</"DIVY_NS_PREFIX":columnname>" CRLF
				"<"DIVY_NS_PREFIX":dispcolumnname>"
				"%s"
				"</"DIVY_NS_PREFIX":dispcolumnname>" CRLF,
				sqlc->id,
				dav_divy_escape_xmlstr(p, sqlc->colname,
							DIVY_XML_T2T_CDATA),
				dav_divy_escape_xmlstr(p, sqlc->name,
							DIVY_XML_T2T_CDATA));
			apr_text_append(p, &hdr, s);

			/* required、名前付きバインド変数 */
			if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED){
				s = apr_psprintf(p,
					"<"DIVY_NS_PREFIX":reqsqlname>"
					"%s"
					"</"DIVY_NS_PREFIX":reqsqlname>" CRLF,
					sqlc->reqsql->rsvalue->name);
				apr_text_append(p, &hdr, s);

				/* sqlmode,sqlmodelist 作成 */
				apr_text_append(p, &hdr, 
						"<"DIVY_NS_PREFIX":sqlmode>");
				switch (sqlc->reqsql->sqlmode){
					case DIVY_SQL_MODE_SHOW:
						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":show/>");
						break;
					case DIVY_SQL_MODE_HIDDEN:
						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":hidden/>");
						break;
					case DIVY_SQL_MODE_SELECTED:
						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":selected/>");
						break;
					case DIVY_SQL_MODE_MULTISELECTED:
						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":multiselected/>");
						break;
					default:
						break;
				}
				apr_text_append(p, &hdr,
					"</"DIVY_NS_PREFIX":sqlmode>" CRLF
					"<"DIVY_NS_PREFIX":sqlmodelist>" CRLF);
				for (i = 0; sqlc->reqsql->sqlmodelist &&
					    sqlc->reqsql->sqlmodelist[i] != 
							DIVY_SQL_MODE_UNKNOWN;
						i++){
					apr_text_append(p, &hdr,
						"<"DIVY_NS_PREFIX":sqlmode>");

					if (sqlc->reqsql->sqlmodelist[i] == 
							DIVY_SQL_MODE_SHOW){
						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":show/>");
					} else if (sqlc->reqsql->sqlmodelist[i]
						== DIVY_SQL_MODE_HIDDEN){

						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":hidden/>");
					} else if (sqlc->reqsql->sqlmodelist[i]
						== DIVY_SQL_MODE_SELECTED){

						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":selected/>");
					} else if (sqlc->reqsql->sqlmodelist[i] 
						== DIVY_SQL_MODE_MULTISELECTED){

						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":multiselected/>");
					}
					apr_text_append(p, &hdr,
						"</"DIVY_NS_PREFIX":sqlmode>" 
						CRLF);
				}
				apr_text_append(p, &hdr,
					"</"DIVY_NS_PREFIX":sqlmodelist>" 
					CRLF);
			}
			/* required、名前付きバインド変数 以外の場合
			 * sqlmode 空タグ */
			else {
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":sqlmode/>" CRLF
					"<"DIVY_NS_PREFIX":sqlmodelist/>" CRLF);
			}

			if (sqlc->fmt && strlen(sqlc->fmt) > 0){
				s = apr_psprintf(p, 
						"<"DIVY_NS_PREFIX":datafmt>"
						"%s"
						"</"DIVY_NS_PREFIX":datafmt>" 
						CRLF, 
						sqlc->fmt);
				apr_text_append(p, &hdr, s);
			} else {
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":datafmt/>" CRLF);
			}

	    		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":sqlposition>"
					"%d"
					"</"DIVY_NS_PREFIX":sqlposition>"
					CRLF,
					sqlc->sqlposition);
			apr_text_append(p, &hdr, s);
			apr_text_append(p, &hdr, 
					"</"DIVY_NS_PREFIX":keyvalue>" CRLF);
		}
	} else if (optflg == DIVY_SEARCH_OPT_QUERYCONTENT){
		/* sqltype 作成 */
		apr_text_append(p, &hdr,"<"DIVY_NS_PREFIX":sqltype>");
		switch (sql->type){
			case DIVY_SQL_TYPE_NORMAL:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":normalsql/>");
				break;
			case DIVY_SQL_TYPE_REPOSITORY:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":reposdbsql/>");
				break;
			case DIVY_SQL_TYPE_REQUIRED:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":reqsql/>");
				break;
			case DIVY_SQL_TYPE_BIND:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":namedbind/>");
				break;
			default:
				break;
		}
		apr_text_append(p, &hdr,"</"DIVY_NS_PREFIX":sqltype>" CRLF);
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":displaycontent>" CRLF
				"<"DIVY_NS_PREFIX":sqlcontent>" CRLF
				"<"DIVY_NS_PREFIX":sqlno>"
				"%s"
				"</"DIVY_NS_PREFIX":sqlno>" CRLF
				"<"DIVY_NS_PREFIX":name>"
				"%s"
				"</"DIVY_NS_PREFIX":name>" CRLF,
				sql->sqlid,
				dav_divy_escape_xmlstr(p, sql->labelname,
							DIVY_XML_T2T_CDATA));
		apr_text_append(p, &hdr, s);

		/* RequiredSQL,名前付きバインド変数はsubname 設定
		 * それ以外は空タグ */
		if (sql->type == DIVY_SQL_TYPE_REQUIRED ||
				sql->type == DIVY_SQL_TYPE_BIND){
			s = apr_psprintf(p, "<"DIVY_NS_PREFIX":subname>"
					"%s"
					"</"DIVY_NS_PREFIX":subname>" CRLF,
					TO_EMPTY(sql->subname));
			apr_text_append(p, &hdr, s);
		} else {
			apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":subname/>" CRLF);
		}

		/* sqltype 作成 */
		apr_text_append(p, &hdr,"<"DIVY_NS_PREFIX":sqltype>");
		switch (sql->type){
			case DIVY_SQL_TYPE_NORMAL:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":normalsql/>");
				break;
			case DIVY_SQL_TYPE_REPOSITORY:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":reposdbsql/>");
				break;
			case DIVY_SQL_TYPE_REQUIRED:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":reqsql/>");
				break;
			case DIVY_SQL_TYPE_BIND:
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":namedbind/>");
				break;
			default:
				break;
		}
		apr_text_append(p, &hdr,"</"DIVY_NS_PREFIX":sqltype>" CRLF);

		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":usedbmsname>"
				"%s"
				"</"DIVY_NS_PREFIX":usedbmsname>" CRLF
				"<"DIVY_NS_PREFIX":comment>"
				"%s"
				"</"DIVY_NS_PREFIX":comment>" CRLF,
				dav_divy_escape_xmlstr(p, sql->usedbmsname,
							DIVY_XML_T2T_CDATA),
				dav_divy_escape_xmlstr(p, sql->comment,
							DIVY_XML_T2T_CDATA));
		apr_text_append(p, &hdr, s);

		for (sqlc = sql->sqlcnt_pr; sqlc; sqlc = sqlc->next){
			/* contentなしの場合はループ終了 */
			if (!sqlc->id) break;
	    
			/* 通常バインド変数[?] */
			if (sqlc->contype == DIVY_SQL_CONTYPE_BIND){
				s = apr_psprintf(p, 
					"<"DIVY_NS_PREFIX":keyvalue id=\"%d\">"
				       	CRLF
					"<"DIVY_NS_PREFIX":dispcolumnname>"
					"%s"
					"</"DIVY_NS_PREFIX":dispcolumnname>" 
					CRLF,
					sqlc->id,
					dav_divy_escape_xmlstr(p, sqlc->name,
							DIVY_XML_T2T_CDATA));
				apr_text_append(p, &hdr, s);

				/* fmt は値がない場合は規定値・空タグ */
				if (!sqlc->fmt || strlen(sqlc->fmt) == 0){
					s = apr_pstrdup(p, 
						"<"DIVY_NS_PREFIX":datafmt/>" 
						CRLF);
				} else {
					s = apr_psprintf(p, 
						"<"DIVY_NS_PREFIX":datafmt>"
						"%s"
						"</"DIVY_NS_PREFIX":datafmt>" 
						CRLF,
						sqlc->fmt);
				}
				apr_text_append(p, &hdr, s);
				apr_text_append(p, &hdr, 
					"</"DIVY_NS_PREFIX":keyvalue>" CRLF);
			}
			/* required、名前付きバインド変数[$$....] */
			else if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED){
				rsval 	= sqlc->reqsql->rsvalue;
				s = apr_psprintf(p,
					"<"DIVY_NS_PREFIX":rskeyvalue "
					"id=\"%d\">" CRLF
					"<"DIVY_NS_PREFIX":dispcolumnname>"
					"%s"
					"</"DIVY_NS_PREFIX":dispcolumnname>"
					CRLF
					"<"DIVY_NS_PREFIX":reqsqlname>"
					"%s"
					"</"DIVY_NS_PREFIX":reqsqlname>"
					CRLF,
					sqlc->id,
					dav_divy_escape_xmlstr(p, sqlc->name,
							DIVY_XML_T2T_CDATA),
					rsval->name);
				apr_text_append(p, &hdr, s);

				/* valueset をループしてreqsqlset 作成 */
				for (rsvals = rsval->valueset; rsvals;
						rsvals = rsvals->next){
					s = apr_psprintf(p,
						"<"DIVY_NS_PREFIX
						":reqsqlvalueset>" CRLF
						"<"DIVY_NS_PREFIX
						":reqsqlvalue>"
						"%s"
						"</"DIVY_NS_PREFIX
						":reqsqlvalue>" CRLF
						"<"DIVY_NS_PREFIX
						":displayvalue>"
						"%s"
						"</"DIVY_NS_PREFIX
						":displayvalue>" CRLF
						"</"DIVY_NS_PREFIX
						":reqsqlvalueset>" CRLF,
						rsvals->value,
						dav_divy_escape_xmlstr(p,
							rsvals->dispvalue,
							DIVY_XML_T2T_QUOTE));
					apr_text_append(p, &hdr, s);
				}

				/* sqlmode 作成 */
				apr_text_append(p, &hdr,
						"<"DIVY_NS_PREFIX":sqlmode>");
				switch (sqlc->reqsql->sqlmode){
					case DIVY_SQL_MODE_SHOW:
						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":show/>");
						break;
					case DIVY_SQL_MODE_HIDDEN:
						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":hidden/>");
						break;
					case DIVY_SQL_MODE_SELECTED:
						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":selected/>");
						break;
					case DIVY_SQL_MODE_MULTISELECTED:
						apr_text_append(p, &hdr,
							"<"DIVY_NS_PREFIX
							":multiselected/>");
						break;
					default:
						break;
				}
				apr_text_append(p, &hdr,
					"</"DIVY_NS_PREFIX":sqlmode>" CRLF);

				/* fmt は値がない場合は規定値・空タグ */
				if (!sqlc->fmt || strlen(sqlc->fmt) == 0){
					s = apr_pstrdup(p, 
						"<"DIVY_NS_PREFIX":datafmt/>" 
						CRLF);
				} else {
					s = apr_psprintf(p, 
						"<"DIVY_NS_PREFIX":datafmt>"
						"%s"
						"</"DIVY_NS_PREFIX":datafmt>" 
						CRLF,
						sqlc->fmt);
				}
				apr_text_append(p, &hdr, s);
				apr_text_append(p, &hdr, 
					"</"DIVY_NS_PREFIX":rskeyvalue>" CRLF);
			}
		}
		apr_text_append(p, &hdr, "</"DIVY_NS_PREFIX":sqlcontent>" CRLF
				"</"DIVY_NS_PREFIX":displaycontent>" CRLF);
	} else if (optflg == DIVY_SEARCH_OPT_ANALYSIS){
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":name>"
				"%s"
				"</"DIVY_NS_PREFIX":name>" CRLF
				"<"DIVY_NS_PREFIX":statement>"
				"%s"
				"</"DIVY_NS_PREFIX":statement>" CRLF,
				dav_divy_escape_xmlstr(p, sql->labelname,
							DIVY_XML_T2T_CDATA),
				dav_divy_escape_xmlstr(p, sql->sql,
							DIVY_XML_T2T_CDATA));
		apr_text_append(p, &hdr, s);

		for (sqlc = sql->sqlcnt_pr; sqlc; sqlc = sqlc->next){
			/* contentなしの場合はループ終了 */
			if (!sqlc->id) break;
	    
			s = apr_psprintf(p, 
					"<"DIVY_NS_PREFIX":keyvalue id=\"%d\">"
				       	CRLF,
					sqlc->id);
			apr_text_append(p, &hdr, s);
			if (sqlc->colname){
				s = apr_psprintf(p,
					"<"DIVY_NS_PREFIX":columnname>"
					"%s"
					"</"DIVY_NS_PREFIX":columnname>" 
					CRLF,
					dav_divy_escape_xmlstr(p, 
							sqlc->colname,
							DIVY_XML_T2T_CDATA));
			} else {
				s = apr_psprintf(p,
					"<"DIVY_NS_PREFIX":columnname/>" CRLF);
			}
			apr_text_append(p, &hdr, s);
			/* required、名前付きバインド変数[$$....] */
			if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED){
				rsval 	= sqlc->reqsql->rsvalue;
				s = apr_psprintf(p,
					"<"DIVY_NS_PREFIX":reqsqlname>"
					"%s"
					"</"DIVY_NS_PREFIX":reqsqlname>"
					CRLF,
					rsval->name);
				apr_text_append(p, &hdr, s);

				/* sqlmode 作成 */
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":sqlmode>"
					"<"DIVY_NS_PREFIX":show/>"
					"</"DIVY_NS_PREFIX":sqlmode>" CRLF
					"<"DIVY_NS_PREFIX":sqlmodelist>" CRLF);
				for (i = 0; sqlc->reqsql->sqlmodelist[i] != 
						DIVY_SQL_MODE_UNKNOWN;
						i++){
					apr_text_append(p, &hdr,
						"<"DIVY_NS_PREFIX":sqlmode>");

					if (sqlc->reqsql->sqlmodelist[i] 
						== DIVY_SQL_MODE_SHOW){

						apr_text_append(p, &hdr,
						    "<"DIVY_NS_PREFIX
						    ":show/>");
					} else if (sqlc->reqsql->sqlmodelist[i]
						== DIVY_SQL_MODE_HIDDEN){
						apr_text_append(p, &hdr,
						    "<"DIVY_NS_PREFIX
						    ":hidden/>");
					} else if (sqlc->reqsql->sqlmodelist[i]
						== DIVY_SQL_MODE_SELECTED){

						apr_text_append(p, &hdr,
						    "<"DIVY_NS_PREFIX
						    ":selected/>");
					} else if (sqlc->reqsql->sqlmodelist[i] 
						== DIVY_SQL_MODE_MULTISELECTED){
						apr_text_append(p, &hdr,
						    "<"DIVY_NS_PREFIX
						    ":multiselected/>");
					}
					apr_text_append(p, &hdr,
						"</"DIVY_NS_PREFIX":sqlmode>"
						CRLF);
				}
				apr_text_append(p, &hdr,
					"</"DIVY_NS_PREFIX":sqlmodelist>"
					CRLF);
			} 
			/* required、名前付きバインド変数 以外の場合
			 * sqlmode 空タグ */
			else {
				apr_text_append(p, &hdr,
					"<"DIVY_NS_PREFIX":sqlmode/>" CRLF
					"<"DIVY_NS_PREFIX":sqlmodelist/>" 
					CRLF);
			} 
			s = apr_psprintf(p,
					"<"DIVY_NS_PREFIX":sqlposition>"
					"%d"
					"</"DIVY_NS_PREFIX":sqlposition>"
					CRLF,
					sqlc->sqlposition);
			apr_text_append(p, &hdr, s);
			apr_text_append(p, &hdr, 
					"</"DIVY_NS_PREFIX":keyvalue>" CRLF);
		}
	}
    
	/* クローズ */
	apr_text_append(p, &hdr, "</"DIVY_NS_PREFIX":sqlinfo>" CRLF
				"</"DIVY_NS_PREFIX":sqldiscovery>" CRLF
				"</"DAV_NS_PREFIX":prop>" CRLF
				"<"DAV_NS_PREFIX":status>"
				"HTTP/1.1 200 OK"
				"</"DAV_NS_PREFIX":status>" CRLF
				"</"DAV_NS_PREFIX":propstat>" CRLF);
	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;

	return res;
}

/**
 * searchのresponceを作成する(sqlinformationsearchのグループ情報)
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_sql * 取得済みのSQL
 * @param  divy_rdbo_grp * 取得済みのgrp
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *sqlis_grp_mkresponse(request_rec *r, divy_rdbo_sql *sql, 
					divy_rdbo_grp *grp)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	const char *rg_grpid	= apr_psprintf(p, DIVY_PREFIX_RGROUP"%s", 
						grp->grpid);

	res->href = divy_build_m_sql_uri(p, dav_divy_get_root_uri(r),
			apr_psprintf(p, "%s/%s/", sql->labelname, rg_grpid));
	res->status = HTTP_OK;

	/* XML作成 */
	s = apr_psprintf(p, "<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":groupdiscovery>" CRLF 
			"<"DIVY_NS_PREFIX":groupinfo>" CRLF
			"<"DIVY_NS_PREFIX":groupid>"
			"%s"
			"</"DIVY_NS_PREFIX":groupid>" CRLF
			"<"DIVY_NS_PREFIX":name>"
			"%s"
			"</"DIVY_NS_PREFIX":name>" CRLF
			"<"DIVY_NS_PREFIX":creationdt>"
			"%s"
			"</"DIVY_NS_PREFIX":creationdt>" CRLF
			"<"DIVY_NS_PREFIX":updatedt>"
			"%s"
			"</"DIVY_NS_PREFIX":updatedt>" CRLF
			"<"DIVY_NS_PREFIX":comment>"
			"%s"
			"</"DIVY_NS_PREFIX":comment>" CRLF,
			grp->grpid, 
			dav_divy_escape_xmlstr(p, grp->name,
						DIVY_XML_T2T_CDATA), 
			grp->registdt, grp->updatedt,
			dav_divy_escape_xmlstr(p, grp->comment,
						DIVY_XML_T2T_CDATA));
	apr_text_append(p, &hdr, s);

	/* グループ拡張ステータスを追加する */
	/* グループ制約属性を追加する */
	_grpis_build_grp_extstatus(r, grp, &hdr, p);

	apr_text_append(p, &hdr,
			"</"DIVY_NS_PREFIX":groupinfo>" CRLF 
			"</"DIVY_NS_PREFIX":groupdiscovery>" CRLF
			"</"DAV_NS_PREFIX":prop>" CRLF
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);

	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;

	return res;
}

/**
 * searchのresponceを作成する(sqlinformationsearch のrequiredsql に関する情報)
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_resource * 検索結果
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *sqlis_reqsqlinfo_mkresponse(request_rec *r, 
					divy_rdbo_resource *resource,
					int optflg)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	apr_hash_index_t *hashind;
	const char *hkey;
	divy_rdbo_sql *sql;

	/* 
	 * dblink,reposdblink,exec などから発行されても
	 * sqlinformationsearch の場合、response のURI の infotype は
	 * /.management/SQL
	 * で統一します。
	 */
	res->href = divy_build_m_sql_uri(p, dav_divy_get_root_uri(r), 
				apr_psprintf(p, "%s/", resource->uri));
	res->status = HTTP_OK;

	/* XML作成 */
	apr_text_append(p, &hdr, "<"DAV_NS_PREFIX":propstat>" CRLF
			"<"DAV_NS_PREFIX":prop>" CRLF);

	/* 検索オプションにより分岐 */
	if (optflg == DIVY_SEARCH_OPT_DEPENDSQLLIST){
		apr_text_append(p, &hdr, 
				"<"DIVY_NS_PREFIX":dependsqldiscovery>" CRLF 
				"<"DIVY_NS_PREFIX":dependsqlinfo>" CRLF);
		/* requiredsqlname のハッシュでループしてreqsqlname 作成 */
		for (hashind = apr_hash_first(p, resource->search_pr->reqsqlname_hash); 
				hashind; hashind = apr_hash_next(hashind)){
			/* キー(reqsqlname)を取得 */
			apr_hash_this(hashind, (const void **)&hkey, NULL, NULL);

			s = apr_psprintf(p, "<"DIVY_NS_PREFIX":reqsqlname>"
					"%s"
					"</"DIVY_NS_PREFIX":reqsqlname>" CRLF,
					hkey);
			apr_text_append(p, &hdr, s);
		}
		apr_text_append(p, &hdr, 
				"</"DIVY_NS_PREFIX":dependsqlinfo>" CRLF 
				"</"DIVY_NS_PREFIX":dependsqldiscovery>" CRLF);
	} 
	else if (optflg == DIVY_SEARCH_OPT_SQLPREFERENCE){
		apr_text_append(p, &hdr,
			"<"DIVY_NS_PREFIX":sqlpreferencediscovery>" CRLF);
		/* sql 構造体でループして requiredsql 情報作成 */
		for(sql = resource->sql_pr; sql; sql = sql->next){
			s = apr_psprintf(p, 
				"<"DIVY_NS_PREFIX":sqlpreferenceinfo>" CRLF
				"<"DIVY_NS_PREFIX":dispcolumnname>"
				"%s"
				"</"DIVY_NS_PREFIX":dispcolumnname>" CRLF
				"<"DIVY_NS_PREFIX":reqsqlname>"
				"%s"
				"</"DIVY_NS_PREFIX":reqsqlname>" CRLF,
				sql->labelname, TO_EMPTY(sql->subname));
			apr_text_append(p, &hdr, s);

			/* cachemode 作成 */
			apr_text_append(p, &hdr,"<"DIVY_NS_PREFIX":cachemode>");
			if (sql->cachemode == DIVY_SQL_CACHE && 
					IS_NBIND_NODE(sql->subname)){
				/* DB 値がキャッシュ で 名前付きバインド変数
				 * の場合 */
				apr_text_append(p, &hdr,
						"<"DIVY_NS_PREFIX":cache/>");
			} else {
				/* 上記以外 */
				apr_text_append(p, &hdr,
						"<"DIVY_NS_PREFIX":nocache/>");
			}
			apr_text_append(p, &hdr,
					"</"DIVY_NS_PREFIX":cachemode>" CRLF);

			s = apr_psprintf(p, "<"DIVY_NS_PREFIX":comment>"
				"%s"
				"</"DIVY_NS_PREFIX":comment>" CRLF
				"</"DIVY_NS_PREFIX":sqlpreferenceinfo>" CRLF,
				dav_divy_escape_xmlstr(p, sql->comment,
							DIVY_XML_T2T_CDATA));
			apr_text_append(p, &hdr, s);
		}
		apr_text_append(p, &hdr,
			"</"DIVY_NS_PREFIX":sqlpreferencediscovery>" CRLF);
	}

	/* クローズ */
	apr_text_append(p, &hdr, "</"DAV_NS_PREFIX":prop>" CRLF
				"<"DAV_NS_PREFIX":status>"
				"HTTP/1.1 200 OK"
				"</"DAV_NS_PREFIX":status>" CRLF
				"</"DAV_NS_PREFIX":propstat>" CRLF);

	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;

	return res;
}

/**
 * searchのresponceを作成する(sqlinformationsearch の Failed Dependency)
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_sql * 検索結果のSQL構造体
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *sqlis_faileddependency_mkresponse(request_rec *r, 
					divy_rdbo_sql *sql)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr	= { 0 };
	apr_text_header hdr_ns 	= { 0 };

	res->href = divy_build_m_sql_uri(p, dav_divy_get_root_uri(r), 
				apr_psprintf(p, "%s/", sql->relativeuri));
	/* status 作成 */
	apr_text_append(p, &hdr, "<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 404  Not Found"
			"</"DAV_NS_PREFIX":status>" CRLF);

	res->status = HTTP_OK;

	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;

	return res;
}


/**
 * searchのresponceを作成する(sqlinformationsearch の analysis 文法エラー)
 *
 *
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_resource * 検索結果
 * @param  divy_rdbo_sql * 検索結果のSQL構造体
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *sqlis_analysiserr_mkresponse(request_rec *r, 
					divy_rdbo_resource *resource,
					int analysisstcode)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	divy_sql_parser_err *perr = divy_sql_parser_get_parser_err(p, 
							analysisstcode);
	divy_rdbo_clist *clist;

	res->href = divy_build_m_sql_uri(p, dav_divy_get_root_uri(r), 
			apr_psprintf(p, "%s/", resource->sql_pr->labelname));
	res->status = HTTP_OK;

	apr_text_append(p, &hdr, "<"DAV_NS_PREFIX":propstat>" CRLF
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":analysiserrdiscovery>" CRLF
			"<"DIVY_NS_PREFIX":errdescription>" CRLF);

	s = apr_psprintf(p, "<"DIVY_NS_PREFIX":code>"
			"%s"
			"</"DIVY_NS_PREFIX":code>" CRLF
			"<"DIVY_NS_PREFIX":msg>"
			"%s"
			"</"DIVY_NS_PREFIX":msg>" CRLF,
			perr->code, 
			dav_divy_escape_xmlstr(p, (char *)perr->msg, 
				DIVY_XML_T2T_CDATA));
	apr_text_append(p, &hdr, s);

	/* namelist 作成 */
	if (resource->search_pr && resource->search_pr->namelist){
		apr_text_append(p, &hdr, "<"DIVY_NS_PREFIX":namelist>" CRLF);
		for (clist = resource->search_pr->namelist; clist; 
				clist = clist->next){
			s = apr_psprintf(p, "<"DIVY_NS_PREFIX":nameentry>"
					"%s"
					"</"DIVY_NS_PREFIX":nameentry>" CRLF,
					clist->val);
			apr_text_append(p, &hdr, s);
		}
		apr_text_append(p, &hdr, "</"DIVY_NS_PREFIX":namelist>" CRLF);
	}

	apr_text_append(p, &hdr, "</"DIVY_NS_PREFIX":errdescription>" CRLF
			"</"DIVY_NS_PREFIX":analysiserrdiscovery>" CRLF
			"</"DAV_NS_PREFIX":prop>" CRLF
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);


	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;

	return res;
}

/**
 * XMLをパースし、パース結果を返却する (dbmsinformationsearch)
 *
 * @param  request_rec * apache request_rec構造体
 * @param  stype_elem apr_xml_elem * 解析済みのXML
 * @param  search_cntxt * 作成されたクエリーを格納する
 * @return int HTTPコード
 */
static int dbmsis_parse_xml(request_rec *r, apr_xml_elem *stype_elem, 
						search_cntxt *scntxt)
{
	apr_pool_t *p          = r->pool;
	apr_xml_elem *cur_elem = stype_elem, *elem;
	const char *data, *dbmsid = NULL;
	int elem_cnt           = 0;

	/* エラーチェック */
	if (stype_elem == NULL || stype_elem->first_child != NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Element of <dbmsinformationsearch> is wrong.");
		return HTTP_BAD_REQUEST;
	}	        

	/* 入力値を格納するscreen を作成 */
	scntxt->screen.dbmsis_iscreen = apr_pcalloc(p, sizeof(divy_search_dbmsis_iscreen));
	scntxt->optflg = 0;

	/*
	 * XMLのパース
	 *
	 * [ DTD ]
	 * 	<!ELEMENT dbmsinformationsearch ((dbmsid, content) | detaillist | activedbms | licenseddbms) >
	 * 	<!ELEMENT dbmsid        (#PCDATA) >
	 * 	<!ELEMENT detaillist     EMPTY >
	 * 	<!ELEMENT content        EMPTY >
	 * 	<!ELEMENT activedbms     EMPTY >
	 * 	<!ELEMENT licenseddbms   EMPTY >
	 */
	for (elem = cur_elem; elem; elem = elem->next) {
		/* PCDATA の両端に含まれていたスペースを除去して取得する */
		data = divy_xml_get_cdata(elem, p, 1);

		if (strcmp(elem->name, "dbmsid") == 0) {
			dbmsid = data;
		}
		else if (strcmp(elem->name, "detaillist") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_DETAILLIST;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "content") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_CONTENT;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "activedbms") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_ACTIVEDBMS;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "licenseddbms") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_LICENSEDBMS;
			elem_cnt++;
		}
		else {
			/* 未知のエレメントが指定された */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"dbmsinformationsearch", elem->name);
			return HTTP_BAD_REQUEST;
		}
	}

	/*
	 * 取得結果の検証
	 */
	/* 具体的なSEARCH動作を示す指示エレメントが存在しなかった */
	if (scntxt->optflg == 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The request element (detaillist / content / activedbms / licenseddbms) "
			"of dbmsinformationsearch not found.");
		return HTTP_BAD_REQUEST;
	}
	/* 指示エレメントが2つ以上存在した */
	else if (elem_cnt > 1) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Too many request elements. The number of request element "
			"(detaillist / content / activedbms / licenseddbms) must be one.");
		return HTTP_BAD_REQUEST;
	}
	/* content ではdbmsidが指定されなければならない */
	else if (IS_EMPTY(dbmsid) && scntxt->optflg == DIVY_SEARCH_OPT_CONTENT) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"dbmsid\" element must be specified.");
		return HTTP_BAD_REQUEST;
	}
	/* dbmsid は数値でなければならない */
	else if (IS_FILLED(dbmsid) && !dav_divy_isdigit_str(dbmsid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The value of \"dbmsid\" element must be digit.");
		return HTTP_BAD_REQUEST;
	}

	/* 入力値をscreen に記録 */
	scntxt->screen.dbmsis_iscreen->dbmsid = dbmsid;

	return HTTP_OK;
}

/**
 * 検索結果から返信するXMLを作成する(dbmsinformationsearch)
 *
 * 
 * @param r request_rec *
 * @param dbms_pr divy_rdbo_dbms * 検索結果
 * @param lcsdbms_h apr_hash_t * ライセンス構造体を持つハッシュ
 * @param res dav_response ** レスポンス格納構造体
 * @return int ステータスコード
 */
static int dbmsis_build_xml_response(request_rec *r,
					divy_rdbo_dbms *dbms_pr,
					apr_hash_t *lcsdbms_h,
					dav_response **res, int optflg)
{
	dav_response *newres 	= NULL;
	dav_response *firstres 	= NULL;
	divy_rdbo_dbms *dbms;
	*res = NULL;

	/* lisenceddbms は ハッシュから1件のみ作成 */
	if (optflg == DIVY_SEARCH_OPT_LICENSEDBMS) {
		firstres = dbmsis_lisenceddbms_mkresponse(r, lcsdbms_h);
		firstres->next = NULL;
	}
	else {
		for (dbms = dbms_pr; dbms; dbms = dbms->next) {

			/* responseを１件作成 */
			newres = dbmsis_mkresponse(r, dbms, optflg);

			if (firstres == NULL) {
				firstres = *res = newres;
			}
			else {
				(*res)->next = newres;
				*res = newres;
			}
			(*res)->next = NULL;
		}
	}

	*res = firstres;
	return HTTP_OK;
}

/**
 * searchのresponceを作成する(dbmsinformationsearch)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_dbms * 取得済みのdbms
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *dbmsis_mkresponse(request_rec *r, divy_rdbo_dbms *dbms, 
					int optflg)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	char *active_str;

	res->href = divy_build_m_dbms_uri(p, dav_divy_get_root_uri(r),
					apr_psprintf(p, "%s/", dbms->name));
	res->status = HTTP_OK;

	if (dbms->active == DIVY_DBMS_ACTIVE) {
		active_str = "<"DIVY_NS_PREFIX":active/>"CRLF;
	}
	else {
		active_str = "";
	}

	/* XML作成 */
	s = apr_psprintf(p, "<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":dbmsdiscovery>" CRLF 
			"<"DIVY_NS_PREFIX":dbmsinfo>" CRLF
			"<"DIVY_NS_PREFIX":dbmsid>"
			"%s"
			"</"DIVY_NS_PREFIX":dbmsid>" CRLF
			"<"DIVY_NS_PREFIX":name>"
			"%s"
			"</"DIVY_NS_PREFIX":name>" CRLF,
			dbms->dbmsid,
			dav_divy_escape_xmlstr(p, dbms->name,
						DIVY_XML_T2T_CDATA));
	apr_text_append(p, &hdr, s);

	/* activedbms 以外(detaillist,content) */
	if (optflg != DIVY_SEARCH_OPT_ACTIVEDBMS){
		s = apr_psprintf(p,
			"<"DIVY_NS_PREFIX":type>"
			"%s"
			"</"DIVY_NS_PREFIX":type>" CRLF 
			"<"DIVY_NS_PREFIX":hostname>"
			"%s"
			"</"DIVY_NS_PREFIX":hostname>" CRLF
			"<"DIVY_NS_PREFIX":hostport>"
			"%d"
			"</"DIVY_NS_PREFIX":hostport>" CRLF
			"<"DIVY_NS_PREFIX":dbname>"
			"%s"
			"</"DIVY_NS_PREFIX":dbname>" CRLF
			"<"DIVY_NS_PREFIX":username>"
			"%s"
			"</"DIVY_NS_PREFIX":username>" CRLF
			"<"DIVY_NS_PREFIX":password>"
			"%s"
			"</"DIVY_NS_PREFIX":password>" CRLF
			"<"DIVY_NS_PREFIX":creationdt>"
			"%s"
			"</"DIVY_NS_PREFIX":creationdt>" CRLF
			"<"DIVY_NS_PREFIX":updatedt>"
			"%s"
			"</"DIVY_NS_PREFIX":updatedt>" CRLF
			"<"DIVY_NS_PREFIX":comment>"
			"%s"
			"</"DIVY_NS_PREFIX":comment>" CRLF
			"%s",
			dbms->type, dbms->hostname, dbms->port,
			dav_divy_escape_xmlstr(p, dbms->dbname,
							DIVY_XML_T2T_CDATA),
			dav_divy_escape_xmlstr(p, dbms->username,
							DIVY_XML_T2T_CDATA),
			dav_divy_escape_xmlstr(p, dbms->password,
							DIVY_XML_T2T_CDATA),
			dbms->registdt, dbms->updatedt,
			dav_divy_escape_xmlstr(p, dbms->comment,
							DIVY_XML_T2T_CDATA),
			active_str);
		apr_text_append(p, &hdr, s);
	}
	/* クローズ */
	apr_text_append(p, &hdr,
			"</"DIVY_NS_PREFIX":dbmsinfo>" CRLF
			"</"DIVY_NS_PREFIX":dbmsdiscovery>" CRLF 
			"</"DAV_NS_PREFIX":prop>" CRLF 
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);

	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
									    
	return res;
}

/**
 * searchのresponceを作成する(dbmsinformationsearch の lisenceddbms)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  apr_hash_t *ライセンスがあって使用可能な DBMS のハッシュ
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *dbmsis_lisenceddbms_mkresponse(request_rec *r,
					apr_hash_t *lcsdbms_h)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	apr_hash_index_t *hi;
	const char *hkey;
									    
	res->href   = divy_build_m_dbms_uri(p, dav_divy_get_root_uri(r), "/");
	res->status = HTTP_OK;

	/* XML作成 */
	apr_text_append(p, &hdr, "<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":dbmsdiscovery>" CRLF 
			"<"DIVY_NS_PREFIX":dbmsinfo>" CRLF
			"<"DIVY_NS_PREFIX":typelist>" CRLF);

	/* ライセンスがあって使用可能な DBMS のハッシュでループ */
	for (hi = apr_hash_first(p, lcsdbms_h); hi; hi = apr_hash_next(hi)){
		apr_hash_this(hi, (const void **)&hkey, NULL, NULL);

		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":type>"
				"%s"
				"</"DIVY_NS_PREFIX":type>" CRLF, 
				hkey);
		apr_text_append(p, &hdr, s);
	}
	/* クローズ */
	apr_text_append(p, &hdr,
			"</"DIVY_NS_PREFIX":typelist>" CRLF
			"</"DIVY_NS_PREFIX":dbmsinfo>" CRLF
			"</"DIVY_NS_PREFIX":dbmsdiscovery>" CRLF 
			"</"DAV_NS_PREFIX":prop>" CRLF 
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);

	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));
    
	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
									    
	return res;
}

/**
 * XMLをパースし、パース結果を返却する (updateinformationsearch)
 *
 * @param  request_rec * apache request_rec構造体
 * @param  stype_elem apr_xml_elem * 解析済みのXML
 * @param  search_cntxt * 作成されたクエリーを格納する
 * @return int HTTPコード
 */
static int updateis_parse_xml(request_rec *r, apr_xml_elem *stype_elem, 
						search_cntxt *scntxt)
{
	apr_pool_t *p          = r->pool;
	apr_xml_elem *cur_elem = stype_elem, *elem;
	const char *data, *lineup = NULL;
	int elem_cnt           = 0;

	/* エラーチェック */
	if (stype_elem == NULL ||
	    stype_elem->first_child != NULL || stype_elem->next != NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Element of <updateinformationsearch> is wrong.");
		return HTTP_BAD_REQUEST;
	}

	/* 入力値を格納するscreen を作成 */
	scntxt->screen.updateis_iscreen = apr_pcalloc(p, sizeof(divy_search_updateis_iscreen));
	scntxt->optflg = 0;

	/*
	 * XMLのパース
	 *
	 * [ DTD ]
	 * 	<!ELEMENT updateinformationsearch (detaillist | lineup) >
	 * 	<!ELEMENT detaillist  EMPTY >
	 * 	<!ELEMENT lineup (#PCDATA) >
	 */
	for (elem = cur_elem; elem; elem = elem->next) {
		/* PCDATA の両端に含まれていたスペースを除去して取得する */
		data = divy_xml_get_cdata(elem, p, 1);

		if (strcmp(elem->name, "lineup") == 0) {
			lineup = data;
			scntxt->optflg = DIVY_SEARCH_OPT_LINEUP;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "detaillist") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_DETAILLIST;
			elem_cnt++;
		}
		else {
			/* 未知のエレメントが指定された */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"updateinformationsearch", elem->name);
			return HTTP_BAD_REQUEST;
		}
	}

	/*
	 * 取得結果の検証
	 */
	/* 具体的なSEARCH動作を示す指示エレメントが存在しなかった */
	if (scntxt->optflg == 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The request element (detaillist / lineup) "
			"of updateinformationsearch not found.");
		return HTTP_BAD_REQUEST;
	}
	/* 指示エレメントが2つ以上存在した */
	else if (elem_cnt > 1) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Too many request elements. The number of request element "
			"(detaillist / lineup) must be one.");
		return HTTP_BAD_REQUEST;
	}
	/* lineup では文字列が指定されなければならない */
	else if (IS_EMPTY(lineup) && scntxt->optflg == DIVY_SEARCH_OPT_LINEUP) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"lineup\" element must be specified.");
		return HTTP_BAD_REQUEST;
	}

	/* 入力値をscreen に記録 */
	scntxt->screen.updateis_iscreen->lineup = lineup;

	return HTTP_OK;
}

/**
 * 検索結果から返信するXMLを作成する(updateinformationsearch)
 *
 * @param r request_rec *
 * @param clmodule_pr divy_rdbo_clmodule * 検索結果を保持するリスト
 * @param res dav_response ** davのレスポンス格納構造体
 * @param optflg int 検索オプション
 * @return int ステータスコード
 */
static int updateis_build_xml_response(request_rec *r,
					divy_rdbo_clmodule *clmodule_pr,
					dav_response **res, int optflg)
{
	dav_response *newres 	= NULL;
	dav_response *firstres 	= NULL;
	divy_rdbo_clmodule *clmodule;
    
	*res = NULL;
									    
	for (clmodule = clmodule_pr; clmodule; clmodule = clmodule->next) {

		/* responseを１件作成 */
		newres = updateis_mkresponse(r, clmodule, optflg);

		if (firstres == NULL) {
			firstres = *res = newres;
		}
		else {
			(*res)->next = newres;
			*res = newres;
		}
		(*res)->next = NULL;
	}
	*res = firstres;
	return HTTP_OK;
}

/**
 * searchのresponceを作成する(updateinformationsearch)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_clmodule * 取得済みのクライアントモジュール情報
 * @param  int 検索オプション see search.h
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *updateis_mkresponse(request_rec *r,
						divy_rdbo_clmodule *clmodule, 
						int optflg)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;												    
	res->href = divy_build_m_update_uri(p, dav_divy_get_root_uri(r),
							clmodule->name);
	res->status = HTTP_OK;

	/* XML作成(共通部分) */
	s = apr_psprintf(p, "<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":updatediscovery>" CRLF 
			"<"DIVY_NS_PREFIX":updateinfo>" CRLF
			"<"DIVY_NS_PREFIX":name>"
			"%s"
			"</"DIVY_NS_PREFIX":name>" CRLF
			"<"DIVY_NS_PREFIX":version>"
			"%s"
			"</"DIVY_NS_PREFIX":version>" CRLF
			"<"DIVY_NS_PREFIX":lineup>"
			"%s"
			"</"DIVY_NS_PREFIX":lineup>" CRLF
			"<"DIVY_NS_PREFIX":digest>"
			"%s"
			"</"DIVY_NS_PREFIX":digest>" CRLF,
			dav_divy_escape_xmlstr(p, clmodule->name,
						DIVY_XML_T2T_CDATA),
			dav_divy_escape_xmlstr(p, clmodule->version,
						DIVY_XML_T2T_CDATA),
			clmodule->lineup,
			clmodule->digest);
	apr_text_append(p, &hdr, s);
											/* detaillistはさらに追加 */
	if (optflg == DIVY_SEARCH_OPT_DETAILLIST){
		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":creationdt>"
				"%s"
				"</"DIVY_NS_PREFIX":creationdt>" CRLF
				"<"DIVY_NS_PREFIX":updatedt>"
				"%s"
				"</"DIVY_NS_PREFIX":updatedt>" CRLF, 
				clmodule->registdt, clmodule->updatedt);
		apr_text_append(p, &hdr, s);
	}
    
	/* updatediscovery クローズ */
	apr_text_append(p, &hdr, "</"DIVY_NS_PREFIX":updateinfo>" CRLF
			"</"DIVY_NS_PREFIX":updatediscovery>" CRLF);

	/* getcontentlength 追加 */
	s = apr_psprintf(p, "<"DAV_NS_PREFIX":getcontentlength>"
			"%"APR_INT64_T_FMT
			"</"DAV_NS_PREFIX":getcontentlength>" CRLF,
			clmodule->getcontentlength);
	apr_text_append(p, &hdr, s);

	/* クローズ */
	apr_text_append(p, &hdr, "</"DAV_NS_PREFIX":prop>" CRLF 
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);

	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;

	return res;
}

/**
 * XMLをパースし、パース結果を返却する (statusinformationsearch)
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  stype_elem apr_xml_elem * 解析済みのXML
 * @param  search_cntxt * 作成されたクエリーを格納する
 * @return int HTTPコード
 */
static int statusis_parse_xml(request_rec *r, apr_xml_elem *stype_elem,
						search_cntxt *scntxt)
{
	apr_pool_t *p           = r->pool;
	apr_xml_elem *cur_elem  = stype_elem, *elem;
	int elem_cnt  = 0;

	/* エラーチェック */
	if (stype_elem == NULL || stype_elem->first_child != NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Element of <statusinformationsearch> is wrong.");
		return HTTP_BAD_REQUEST;
	}
	scntxt->optflg = 0;

	/*
	 * XMLのパース
	 *
	 * [ DTD ]
	 * 	<!ELEMENT statusinformationsearch (detaillist) >
	 * 	<!ELEMENT detaillist     EMPTY >
	 */
	for (elem = cur_elem; elem; elem = elem->next) {

		if (strcmp(elem->name, "detaillist") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_DETAILLIST;
			elem_cnt++;
		}
		else {
			/* 未知のエレメントが指定された */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"userinformationsearch", elem->name);
			return HTTP_BAD_REQUEST;
		}
	}

	/*
	 * 取得結果の検証
	 */
	/* 具体的なSEARCH動作を示す指示エレメントが存在しなかった */
	if (scntxt->optflg == 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The request element (detaillist) "
			"of statusinformationsearch not found.");
		return HTTP_BAD_REQUEST;
	}
	/* 指示エレメントが2つ以上存在した */
	else if (elem_cnt > 1) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Too many request elements. The number of request element "
			"(detaillist) must be one.");
		return HTTP_BAD_REQUEST;
	}

	return HTTP_OK;
}

/**
 * 検索結果から返信するXMLを作成する(statusinformationsearch)
 * 
 * 
 * @param  r request_rec * apache request_rec構造体
 * @param  usr_pr divy_rdbo_usr * *検索結果
 * @paramr res dav_response ** davのレスポンス格納構造体
 * @return int HTTPコード
 */
static int statusis_build_xml_response(request_rec *r,
					divy_rdbo_usr *usr_pr,
					dav_response **res)
{
	dav_response *newres = NULL;
	dav_response *firstres = NULL;
	apr_pool_t *p = r->pool;
		    
	*res = NULL;

	for (; usr_pr; usr_pr = usr_pr->next) {
		/* responseを１件作成 */
		newres = apr_pcalloc(p, sizeof(dav_response));

		statusis_mkresponse(r, usr_pr, newres, p);

		if (firstres == NULL) {
			firstres = *res = newres;
		}
		else {
			(*res)->next = newres;
			*res = newres;
		}
		(*res)->next = NULL;
	}
	*res = firstres;
	return HTTP_OK;
}

/**
 * searchのresponceを作成する(statusinformationsearch)
 *
 *
 * @param r request_rec * apache request_rec構造体
 * @param usr divy_rdbo_usr * 取得済みのユーザ情報
 * @param res dav_response * davのレスポンス格納構造体
 * @param wp apr_pool_t * メモリアロケータ
 */
static void statusis_mkresponse(request_rec *r, divy_rdbo_usr *usr, dav_response *res, apr_pool_t *wp)
{
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	char *datebuf = NULL;
	char *extra   = NULL;

	res->href = divy_build_m_status_uri(wp, dav_divy_get_root_uri(r),
					apr_psprintf(wp, "%s/", usr->usrid));
	res->status = HTTP_OK;

	/* XML作成 */
	apr_text_append(wp, &hdr,
			"<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":statusdiscovery>" CRLF 
			"<"DIVY_NS_PREFIX":statusinfo>" CRLF);

	s = apr_psprintf(wp,
			"<"DIVY_NS_PREFIX":userid>%s"
			"</"DIVY_NS_PREFIX":userid>" CRLF
			"<"DIVY_NS_PREFIX":name>%s"
			"</"DIVY_NS_PREFIX":name>" CRLF
			"<"DIVY_NS_PREFIX":creationdt>%s"
			"</"DIVY_NS_PREFIX":creationdt>" CRLF
			"<"DIVY_NS_PREFIX":updatedt>%s"
			"</"DIVY_NS_PREFIX":updatedt>" CRLF
			"<"DIVY_NS_PREFIX":usedstorage>%"APR_INT64_T_FMT
			"</"DIVY_NS_PREFIX":usedstorage>" CRLF
			"<"DIVY_NS_PREFIX":maxstorage>%"APR_INT64_T_FMT
			"</"DIVY_NS_PREFIX":maxstorage>" CRLF
			"<"DIVY_NS_PREFIX":usedresource>%"APR_INT64_T_FMT
			"</"DIVY_NS_PREFIX":usedresource>" CRLF
			"<"DIVY_NS_PREFIX":maxresource>%"APR_INT64_T_FMT
			"</"DIVY_NS_PREFIX":maxresource>" CRLF,
			dav_divy_escape_xmlstr(wp, usr->usrid, DIVY_XML_T2T_CDATA),
			dav_divy_escape_xmlstr(wp, usr->fullname, DIVY_XML_T2T_CDATA),
			usr->registdt, usr->updatedt,
			usr->usedst, usr->maxst, usr->usedres, usr->maxres);
	apr_text_append(wp, &hdr, s);

	/* 日付を変換 */
	divy_format_time_t(wp, usr->lastaccess, DIVY_TIME_STYLE_ISO8601, &datebuf);
	if (IS_FILLED(datebuf)) {
		s = apr_psprintf(wp,
			"<"DIVY_NS_PREFIX":lastaccessdt>%s"
			"</"DIVY_NS_PREFIX":lastaccessdt>" CRLF, datebuf);
	}
	else {
		s = "<"DIVY_NS_PREFIX":lastaccessdt/>" CRLF;
	}
	apr_text_append(wp, &hdr, s);

	/* 最終アクセスクライアント */
	if (IS_FILLED(usr->lastaccesscl)) {
		s = apr_psprintf(wp,
			"<"DIVY_NS_PREFIX":lastaccessclient>%s"
			"</"DIVY_NS_PREFIX":lastaccessclient>" CRLF,
			dav_divy_escape_xmlstr(wp, usr->lastaccesscl, DIVY_XML_T2T_CDATA));
		apr_text_append(wp, &hdr, s);
	}
	else {
		apr_text_append(wp, &hdr, "<"DIVY_NS_PREFIX":lastaccessclient/>" CRLF);
	}

	/* メールアドレス */
	if (IS_FILLED(usr->mailaddr)) {
		s = apr_psprintf(wp,
			"<"DIVY_NS_PREFIX":mailaddr>%s"
			"</"DIVY_NS_PREFIX":mailaddr>" CRLF,
			dav_divy_escape_xmlstr(wp, usr->mailaddr, DIVY_XML_T2T_CDATA));
		apr_text_append(wp, &hdr, s);
	}
	else {
		apr_text_append(wp, &hdr, "<"DIVY_NS_PREFIX":mailaddr/>" CRLF);
	}

	/* コメント */
	if (IS_FILLED(usr->comment)) {
		s = apr_psprintf(wp,
			"<"DIVY_NS_PREFIX":comment>%s</"DIVY_NS_PREFIX":comment>" CRLF,
			dav_divy_escape_xmlstr(wp, usr->comment, DIVY_XML_T2T_CDATA));
		apr_text_append(wp, &hdr, s);
	}
	else {
		apr_text_append(wp, &hdr, "<"DIVY_NS_PREFIX":comment/>" CRLF);
	}
    
#ifdef DIVY_SUPPORT_PASSPOLICY
	/* ユーザのパスワードポリシー状態を組み立てる */
	if (divy_support_passpolicy(r)) {
		_useris_build_passpolicystatus(r, usr, &extra, wp);
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* ユーザ拡張ステータスを入れる */
	if (divy_support_extenduserstatus(r)) {
		_useris_build_extstatus(r, usr, extra, &hdr, wp);
	}

	/* グループリーダの情報を入れる */
	if (divy_support_groupleader(r)) {
		_useris_build_groupleader(r, usr, &hdr, wp);
	}

	/* クローズ */
	apr_text_append(wp, &hdr, "</"DIVY_NS_PREFIX":statusinfo>" CRLF
			"</"DIVY_NS_PREFIX":statusdiscovery>" CRLF 
			"</"DAV_NS_PREFIX":prop>" CRLF
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF);
	/* ネームスペース追加 */
	apr_text_append(wp, &hdr_ns, divy_make_liveprop_ns(wp, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));

	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
}


#ifdef DAV_SUPPORT_EXTENDED_SEARCH
/**
 * get_response ハンドラにセットされる関数 (statusinformationsearch)
 */
static int statusis_get_response(request_rec *r, divy_rdbo_search_output *output,
					dav_response *response, apr_pool_t *wp)
{
	divy_rdbo_usr *usr_pr = output->list.usr_pr;

	if (usr_pr == NULL) return DIVY_SEARCH_RESPONSE_NONE;

	/* 1つのレスポンス生成 */
	statusis_mkresponse(r, usr_pr, response, wp);

	return 0;
}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

/**
 * XMLをパースし、パース結果を返却する (sysmsginformationsearch)
 *
 * @param  request_rec * apache request_rec構造体
 * @param  stype_elem apr_xml_elem * 解析済みのXML
 * @param  search_cntxt * 作成されたクエリーを格納する
 * @return int HTTPコード
 */
static int sysmsgis_parse_xml(request_rec *r, apr_xml_elem *stype_elem, 
						search_cntxt *scntxt)
{
	apr_pool_t *p          = r->pool;
	apr_xml_elem *cur_elem = stype_elem, *elem;
	const char *data, *msgid = NULL;
	int elem_cnt           = 0;
	    		    
	/* エラーチェック */
	if (stype_elem == NULL || stype_elem->first_child != NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Element of <sysmsginformationsearch> is wrong.");
		return HTTP_BAD_REQUEST;
	}

	/* 入力値を格納するscreen を作成 */
	scntxt->screen.sysmsgis_iscreen = apr_pcalloc(p, sizeof(divy_search_sysmsgis_iscreen));
	scntxt->optflg = 0;

	/*
	 * XMLのパース
	 *
	 * [ DTD ]
	 * 	<!ELEMENT sysmsginformationsearch (
	 * 		detaillist | currentmsg | (msgid, content) ) >
	 * 	<!ELEMENT msgid       (#PCDATA) >
	 * 	<!ELEMENT detaillist  EMPTY >
	 * 	<!ELEMENT currentmsg  EMPTY >
	 * 	<!ELEMENT content     EMPTY >
	 */
	for (elem = cur_elem; elem; elem = elem->next) {
		/* PCDATA の両端に含まれていたスペースを除去して取得する */
		data = divy_xml_get_cdata(elem, p, 1);

		if (strcmp(elem->name, "msgid") == 0) {
			msgid = data;
		}
		else if (strcmp(elem->name, "currentmsg") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_CURRENTMSG;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "detaillist") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_DETAILLIST;
			elem_cnt++;
		}
		else if (strcmp(elem->name, "content") == 0) {
			scntxt->optflg = DIVY_SEARCH_OPT_CONTENT;
			elem_cnt++;
		}
		else {
			/* 未知のエレメントが指定された */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"sysmsginformationsearch", elem->name);
			return HTTP_BAD_REQUEST;
		}
	}

	/*
	 * 取得結果の検証
	 */
	/* 具体的なSEARCH動作を示す指示エレメントが存在しなかった */
	if (scntxt->optflg == 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The request element (currentmsg / detaillist / content) "
			"of sysmsginformationsearch not found.");
		return HTTP_BAD_REQUEST;
	}
	/* 指示エレメントが2つ以上存在した */
	else if (elem_cnt > 1) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Too many request elements. The number of request element "
			"(currentmsg / detaillist / content) must be one.");
		return HTTP_BAD_REQUEST;
	}
	/* currentmsg, detaillist ではmsgid を指定してはならない */
	else if (IS_FILLED(msgid) &&
			(scntxt->optflg == DIVY_SEARCH_OPT_CURRENTMSG ||
			 scntxt->optflg == DIVY_SEARCH_OPT_DETAILLIST)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"msgid\" element must be EMPTY.");
		return HTTP_BAD_REQUEST;
	}
	/* content ではmsgid が必須 */
	else if (IS_EMPTY(msgid) && scntxt->optflg == DIVY_SEARCH_OPT_CONTENT) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"msgid\" element must be specified.");
		return HTTP_BAD_REQUEST;
	}
	/* msgid は数値でなければならない */
	else if (IS_FILLED(msgid) && !dav_divy_isdigit_str(msgid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The value of \"msgid\" element must be digit.");
		return HTTP_BAD_REQUEST;
	}

	/* 入力値をscreen に記録 */
	if (IS_FILLED(msgid)) {
		scntxt->screen.sysmsgis_iscreen->msgid = atoi(msgid);
	}

	return  HTTP_OK;
}

/**
 * 検索結果から返信するXMLを作成する(sysmsginformationsearch)
 * 
 * 
 * @param  r request_rec * apache request_rec構造体
 * @param  sysmsg_pr divy_rdbo_sysmsg * 検索結果
 * @paramr res dav_response ** レスポンス格納構造体
 * @param  optflg int 検索オプション
 * @return int ステータスコード
 */
static int sysmsgis_build_xml_response(request_rec *r,
					divy_rdbo_sysmsg *sysmsg_pr,
					dav_response **res, int optflg)
{
	dav_response *newres 	= NULL;
	dav_response *firstres 	= NULL;
	divy_rdbo_sysmsg *sysmsg;

	*res = NULL;

	for (sysmsg = sysmsg_pr; sysmsg; sysmsg = sysmsg->next){
		/* responseを１件作成 */
		newres = sysmsgis_mkresponse(r, sysmsg, optflg);

		if (firstres == NULL) {
			firstres = *res = newres;
		}
		else {
			(*res)->next = newres;
			*res = newres;
		}
		(*res)->next = NULL;
	}
	*res = firstres;
	return HTTP_OK;
}

/**
 * searchのresponceを作成する(sysmsginformationsearch)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_sysmsg * 取得済みのsysmsg
 * @param  int 検索オプション see search.h
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *sysmsgis_mkresponse(request_rec *r,
					divy_rdbo_sysmsg *sysmsg, int optflg)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;
	int support_extstatus   = divy_support_extenduserstatus(r);

	res->href = divy_build_m_msg_uri(p, dav_divy_get_root_uri(r),
					apr_psprintf(p, "%s/", sysmsg->msgid));
	res->status = HTTP_OK;

	/* XML作成(共通部分) */
	s = apr_psprintf(p, "<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":sysmsgdiscovery>" CRLF 
			"<"DIVY_NS_PREFIX":sysmsginfo>" CRLF
			"<"DIVY_NS_PREFIX":msgid>"
			"%s"
			"</"DIVY_NS_PREFIX":msgid>" CRLF
			"<"DIVY_NS_PREFIX":msgtxt>"
			"%s"
			"</"DIVY_NS_PREFIX":msgtxt>" CRLF,
			sysmsg->msgid,
			dav_divy_escape_xmlstr(p, sysmsg->msg,
						DIVY_XML_T2T_CDATA));
	apr_text_append(p, &hdr, s);

	/* 拡張メッセージ追加 */
	if (support_extstatus) {
		s = apr_psprintf(p, 
					"<"DIVY_NS_PREFIX":lang>"
					"%s"
					"</"DIVY_NS_PREFIX":lang>"
					"<"DIVY_NS_PREFIX":subject>"
					"%s"
					"</"DIVY_NS_PREFIX":subject>",
					dav_divy_escape_xmlstr(p, sysmsg->lang, DIVY_XML_T2T_CDATA),
					dav_divy_escape_xmlstr(p, sysmsg->subject, DIVY_XML_T2T_CDATA)
					);

		apr_text_append(p, &hdr, s);

		s = apr_psprintf(p, "<"DIVY_NS_PREFIX":creationdt>"
				"%s"
				"</"DIVY_NS_PREFIX":creationdt>" CRLF,
				sysmsg->registdt);
		apr_text_append(p, &hdr, s);
	}

	/* currentmsg以外はさらに追加(detaillist,content) */
	if (optflg != DIVY_SEARCH_OPT_CURRENTMSG){
		/* アクティブの場合タグ追加 */
		if (sysmsg->active == 1){
			apr_text_append(p, &hdr, 
					"<"DIVY_NS_PREFIX":active/>" CRLF);
		}
	}
    
	/* 最後の共通部分,クローズ */
	s = apr_psprintf(p, "<"DIVY_NS_PREFIX":updatedt>"
			"%s"
			"</"DIVY_NS_PREFIX":updatedt>" CRLF
			"</"DIVY_NS_PREFIX":sysmsginfo>" CRLF
			"</"DIVY_NS_PREFIX":sysmsgdiscovery>" CRLF 
			"</"DAV_NS_PREFIX":prop>" CRLF 
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF, 
			sysmsg->updatedt);
	apr_text_append(p, &hdr, s);
	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));
					    
	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
    
	return res;
}

/**
 * 検索結果から返信するXMLを作成する(trashinformationsearch)
 * 
 * 
 * @param  request_rec * apache request_rec構造体
 * @param  divy_rdbo_resource *検索結果
 * @paramr dav_response ** davのレスポンス格納構造体
 * @param  int 検索オプション
 * @return int HTTPコード
 */
static int trashis_build_xml_response(request_rec *r,
					divy_rdbo_resource *search_result,
					dav_response **res, int optflg)
{
	dav_response *newres 	= NULL;
	dav_response *firstres 	= NULL;

	*res = NULL;

	for ( ; search_result; search_result = search_result->next) {
		/* responseを１件作成 */
		newres = trashis_mkresponse(r, search_result, optflg);

		if (firstres == NULL) {
			firstres = *res = newres;
		}
		else {
			(*res)->next = newres;
			*res = newres;
		}
		(*res)->next = NULL;
	}

	*res = firstres;

	return HTTP_OK;
}

/**
 * searchのresponceを作成する(trashinformationsearch)
 * 
 * 
 * @param  r request_rec * apache request_rec構造体
 * @param  rdb_r divy_rdbo_resource * 取得済みのsysmsg
 * @param  optflg int 検索オプション see search.h
 * @return dav_response * davのレスポンス格納構造体
 */
static dav_response *trashis_mkresponse(request_rec *r,
					divy_rdbo_resource *rdb_r, int optflg)
{
	apr_pool_t *p		= r->pool;
	dav_response *res 	= apr_pcalloc(p, sizeof(*res));
	apr_text_header hdr 	= { 0 };
	apr_text_header hdr_ns 	= { 0 };
	const char *s;

	res->href = dav_divy_make_uri(p, rdb_r->uri, "/", NULL);
	res->status = HTTP_OK;

	/* XML作成(共通部分) */
	s = apr_psprintf(p,
			"<"DAV_NS_PREFIX":propstat>" CRLF 
			"<"DAV_NS_PREFIX":prop>" CRLF
			"<"DIVY_NS_PREFIX":trashdiscovery>" CRLF 
			"<"DIVY_NS_PREFIX":trashinfo>" CRLF
			"<"DIVY_NS_PREFIX":name>"
			"%s"
			"</"DIVY_NS_PREFIX":name>" CRLF
			"</"DIVY_NS_PREFIX":trashinfo>" CRLF
			"</"DIVY_NS_PREFIX":trashdiscovery>" CRLF 
			"</"DAV_NS_PREFIX":prop>" CRLF 
			"<"DAV_NS_PREFIX":status>"
			"HTTP/1.1 200 OK"
			"</"DAV_NS_PREFIX":status>" CRLF
			"</"DAV_NS_PREFIX":propstat>" CRLF,
			rdb_r->displayname);
	apr_text_append(p, &hdr, s);
	/* ネームスペース追加 */
	apr_text_append(p, &hdr_ns, divy_make_liveprop_ns(p, DIVY_GET_DAV_NS | DIVY_GET_DIVY_NS));
					    
	/* 返却値を設定 */
	res->propresult.propstats = hdr.first;
	res->propresult.xmlns = hdr_ns.first;
    
	return res;
}


/**
 * where条件を解析しバインド変数用の構造体のハッシュ
 * where条件文を作成する(basicsearch)
 *
 * (note)
 * lt,lte,gt,gte,eq,like のいずれかのエレメントを渡してください。
 * 
 * @param  request_rec *
 * @param  search_cntxt * 作成したバインド情報を格納
 * @param  apr_xml_elem * 作成元となるXML
 * @return int HTTPコード
 */
static int _make_bs_wherecond(request_rec *r, search_cntxt *scntxt,
				apr_xml_elem *cond_elem)
{
	apr_pool_t *p			= r->pool;
	const char *op			= cond_elem->name;
	const char *prop		= NULL;
	const char *literal		= NULL;
	divy_rdbo_tableinfo *tblinfo 	= NULL;
	char *comop			= NULL;
	int escflg			= 0;
	divy_search_bs_bind *bindinfo	= apr_pcalloc(p, 
						sizeof(divy_search_bs_bind)); 
	char *hkey;

	/* caseless はサポート外 */
	if (cond_elem->attr){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"caseless\" attribute is not supported.");
		return HTTP_UNPROCESSABLE_ENTITY;
	}

	/* 
	 * XMLから値を取得する
	 */
	/* prop */
	if (cond_elem->first_child && 
			strcmp(cond_elem->first_child->name, "prop") == 0 &&
			cond_elem->first_child->first_child){
		prop = cond_elem->first_child->first_child->name;
	} else {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"DTD grammar mistake was found around <prop>.");
		return HTTP_BAD_REQUEST;
	}
	/* literal */
	if (cond_elem->first_child->next &&
	    strcmp(cond_elem->first_child->next->name, "literal") == 0) {
		literal = divy_xml_get_cdata(cond_elem->first_child->next, p, 1);
		if (IS_EMPTY(literal)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"<literal> needs PCDATA.");
			return HTTP_BAD_REQUEST;
		}
	}
	else if (cond_elem->first_child->next &&
			strcmp(cond_elem->first_child->next->name, "typed-literal")
									== 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"typed-literal\" element is not supported.");
		return HTTP_UNPROCESSABLE_ENTITY;
	}
	else {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"DTD grammar mistake was found around <literal>.");
		return HTTP_BAD_REQUEST;
	}

	/*
	 * prop からカラム名・テーブル名を取得する
	 */
	if (!(tblinfo = divy_rdbo_get_tableinfo(r, prop))){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The specified property(child of <prop>) is wrong.");
		return HTTP_BAD_REQUEST;
	}

	/* 参照するテーブルをハッシュのキーに追加 */
	_bs_reftbl_hash_set(p, scntxt, tblinfo);

	/* 
	 * バインドの対象となるwhere条件を作成
	 */
	/* オペレータにより演算子を指定 */
	if (strcmp(op, "lt") == 0){
		comop = apr_pstrdup(p, "<");
	} 
	else if (strcmp(op, "lte") == 0){
		comop = apr_pstrdup(p, "<=");
	}
	else if (strcmp(op, "gt") == 0){
		comop = apr_pstrdup(p, ">");
	}
	else if (strcmp(op, "gte") == 0){
		comop = apr_pstrdup(p, ">=");
	}
	else if (strcmp(op, "eq") == 0){
		comop = apr_pstrdup(p, "=");
	}
	else if (strcmp(op, "like") == 0){
		comop = apr_pstrdup(p, "like");
		/* like の場合はエスケープされるワイルドカードを
		 * DB にあわせて変換 */
		literal = divy_rdbo_escape_wildcard(p, literal);
		escflg = 1;
	}

	if (escflg){	/* like の場合はエスケープ句を追加します */
		scntxt->where = apr_psprintf(p, "%s (%s.%s %s ? "
					DIVY_DBFUNC_ESCAPE_CLAUSE")",
					scntxt->where,
					tblinfo->refname,
					tblinfo->colname,
					comop);
	} else {
		scntxt->where = apr_psprintf(p, "%s (%s.%s %s ?)",
					scntxt->where,
					tblinfo->refname,
					tblinfo->colname,
					comop);

	}

	/* 
	 * バインド情報作成 
	 */
	/* 指定されたプロパティにより分岐 */
	if (strcmp(prop, "creationdate") == 0){
		bindinfo->type = DIVY_SEARCH_BIGLONG_TT;
		/* 値は time_t に変換 */
		bindinfo->ttval = dav_divy_iso8601totime_t(p, literal);
	} 
	else if (strcmp(prop, "getlastmodified") == 0){
		bindinfo->type = DIVY_SEARCH_BIGLONG_TT;
		/* 値は time_t に変換 */
		bindinfo->ttval = apr_time_sec(apr_date_parse_rfc(literal));
	}
	else if (strcmp(prop, "getcontentlength") == 0) {
		bindinfo->type = DIVY_SEARCH_BIGLONG_I64;
		/* 数値かどうかチェック */
		if (!dav_divy_isdigit_str(literal)){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The conditions of \'getcontentlength\' "
				"should be numerical values.");
			return HTTP_BAD_REQUEST;
		}
		/* 値は int64 に変換 */
		bindinfo->i64val = apr_atoi64(literal);
	}
	else if (strcmp(prop, "creator") == 0 || 
			strcmp(prop, "displayname") == 0 ||
			strcmp(prop, "getcontentlanguage") == 0 ||
			strcmp(prop, "getcontenttype") == 0 ||
			strcmp(prop, "getetag") == 0 ||
			strcmp(prop, "lastmodifier") == 0){
		bindinfo->type = DIVY_SEARCH_STRING;
		bindinfo->strval = apr_pstrdup(p, literal);
	} 
	else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"\'%s\' property cannot be include in "
			"where condition.", prop);
		return HTTP_BAD_REQUEST;
	}
	/* ハッシュにセットする */
	if (!scntxt->bs_bind){
		scntxt->bs_bind = apr_hash_make(p);
	}
	hkey = apr_psprintf(p, "%d", apr_hash_count(scntxt->bs_bind) + 1);
	apr_hash_set(scntxt->bs_bind, hkey, APR_HASH_KEY_STRING, bindinfo);

 	return HTTP_OK;
}

/**
 * 指定されたdoc が表すSEARCH のDocument ノードを検証する。
 * [ 検証に合格する条件 ]
 *   * XML エレメントのBody が存在すること
 *   * searchrequest エレメントが存在すること
 *   * searchrequest はroot ノードであること
 *   * SEARCH の種類を表すエレメント(ex, linkdbsearch, userinformationsearch) が存在すること
 *   * また上記は１つだけであること
 *
 * @param r request_rec *
 * @param doc apr_xml_doc *
 * @return dav_error * エラーがなければNULL
 * 	HTTP_BAD_REQUEST
 */
static dav_error * _validate_root_node(request_rec *r, apr_xml_doc *doc)
{
	apr_pool_t *p = r->pool;

	/* ドキュメントドが存在しなかった */
	if (doc == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Failed to process search request. "
			"the document node is NULL.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* ルートノードが存在しなかった */
	if (doc->root == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Failed to process search request. "
			"the root node is NULL.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* searchrequest エレメントは存在するか？ */
	if (IS_EMPTY(doc->root->name) ||
	    strcmp(doc->root->name, "searchrequest") != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"searchrequest\" element was missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* searchrequest はルートノードか？ */
	if (doc->root->next != NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"searchrequest\" element MUST be root node.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* SEARCH の種類を表すエレメントは存在するか？ */
	if (doc->root->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The element which represented type of SEARCH "
			"was missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* SEARCH の種類を表すエレメントはただ１つだけであるか？ */
	if (doc->root->first_child->next != NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There were some elements which represented "
			"type of SEARCH. It MUST be single.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	return NULL;
}


/*--------------------------------------------------------------
  Declare public functions
  --------------------------------------------------------------*/
/**
 * linkdbsearch の戻り値retcd から、詳細なエラー情報(divy_search_ldb_err)を
 * 取得して返却する。
 *
 * 
 */
DIVY_DECLARE(divy_search_ldb_err *) divy_search_get_ldb_err(apr_pool_t *p, 
								int retcd)
{
	divy_search_ldb_err *err = NULL;
	int i, len = SEARCH_LDB_ERR_LEN;

	for (i = 0; i < len; i++) {
		err = (divy_search_ldb_err *) &search_ldb_errs[i];
		if (err && err->retcd == retcd) {
			return err;
		}
	}
	return NULL;
}

/*--------------------------------------------------------------
  Create & Initialize SEARCH provider Hook structure
  --------------------------------------------------------------*/
const dav_hooks_search dav_divy_hooks_search = {
	dav_divy_set_option_head,
	dav_divy_search_resource,
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	"_extended_method_",	/* バイナリ互換性を保つための考慮 */
	dav_divy_support_streaming_mode,
	dav_divy_process_request,
#else	/* !DAV_SUPPORT_EXTENDED_SEARCH */
	NULL
#endif
};


