%{
/**
 * $Id$
 *
 * sqlparser.y
 *
 * TeamFile で使用するSQL文の構文解析を行う関数
 */
#include "httpd.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_hash.h"

#include "mod_dav_tf.h"
#include "tf_sqlparser.h"
#include "util_common.h"

/*------------------------------------------------------------------------------
  Define fixed value and macros 
  ----------------------------------------------------------------------------*/

/* キャストしたp_ctx を表すマクロ (ルール部でのみ使用可能) */
#define P_CTX	((divy_sql_parser_ctx *)p_ctx)

/* プール取得用マクロ (ルール部でのみ使用可能) */
#define POOL	(((divy_sql_parser_ctx *)p_ctx)->p)

/* エラーコード設定用マクロ (ルール部でのみ使用可能) */
#define SET_ERRCD(cd)	(((divy_sql_parser_ctx *)p_ctx)->errcd = cd)

/*
 * 構文解析パーサが出力するエラーメッセージから取り除く
 * エラーメッセージパートの文字列とその長さ
 */
#define _REMOVE_ERR_MSG_PART	"parse error,"
#define _REMOVE_ERR_MSG_PART_EN (strlen(_REMOVE_ERR_MSG_PART))

/*
 * 詳細な解析エラーの表示を指示する
 */
#define YYERROR_VERBOSE

/*--------------------------------------------------------------
  Declare private functions
  --------------------------------------------------------------*/
void yyerror(divy_sql_parser_ctx *p_ctx, const char *str);
static divy_sql_elemnode * _makeNode(divy_sql_parser_ctx *p_ctx,
					divy_sql_nodetype type, char *value);
static divy_sql_elemnode * _makeExpr(divy_sql_parser_ctx *p_ctx,
					divy_sql_nodetype type,
					char *op,
					divy_sql_elemnode *lnode,
					divy_sql_elemnode *rnode);
static divy_sql_elemnode * _appendExpr(divy_sql_elemnode *l1,
						divy_sql_elemnode *l2);
static void _incrementSelectColCnt(divy_sql_parser_ctx *p_ctx);
static void _resetSelectColCnt(divy_sql_parser_ctx *p_ctx);
static void _resetMaxSelectColCnt(divy_sql_parser_ctx *p_ctx);
static int _validate_lval(apr_pool_t *p, char *op, divy_sql_elemnode *lnode);
static int _enable_making_expr(char *op, divy_sql_elemnode *lnode,
				divy_sql_elemnode *rnode);

%}

/*
 * yyparse がグローバル変数を使用しないようにする(Bison専用)
 * (構文解析パーサをリエントラントにする)
 */
%pure-parser
/*
 * yyparse の引数の書き換え
 * (note)
 * 	divy_sql_parser_ctx を引数に渡す
 */
%parse-param { divy_sql_parser_ctx *p_ctx }

/*
 * yylex の引数の置き換え
 * (note)
 *      divy_sql_parser_ctx を引数に渡す
 */
%lex-param { divy_sql_parser_ctx *p_ctx }

/*
 * 開始構文の指定
 */
%start sql

/*
 * 字句解析パーサとのやり取りに使用する共用体の定義
 */
%union
	{
		char *str;			/* 色々..           */
		char *keyword;			/* SQLキーワード    */
		divy_sql_elemnode *node;	/* 解析途中のノード */
	}
%{
/*------------------------------------------------------------------------------
  Define fixed value and macros for bison
  ----------------------------------------------------------------------------*/
/*
 * yyparse の引数の書き換え
 * (note)
 * 	divy_sql_parser_ctx を引数に渡す
 */
#define YYPARSE_PARAM p_ctx

/*
 * yylex の引数の置き換え
 * (note)
 *      divy_sql_parser_ctx を引数に渡す
 */
#define YYLEX_PARAM p_ctx
/*
 * yylex の置き換え
 * (note)
 *	緩衝ルーチンを使用したyylex を構文解析パーサでは使用する.
 */
#ifndef yylex
#define yylex tf_inter_yylex
#endif	/* yylex */

/* 緩衝ルーチンの外部宣言 */
extern int tf_inter_yylex(YYSTYPE *p_yylval, divy_sql_parser_ctx *p_ctx);

%}

/*
 * SQL92またはPostgreSQL、Oracle、Sybaseで予約されたトークンの定義
 */
%token  <keyword>
		ABORT_TRANS ABS ABSOLUTE ACCESS ACOS ACTION ADD ADDMONTHS AFTER
		AGGREGATE ALL ALTER ANALYSE ANALYZE AND ANY AS ASC
		ASSERTION ASSIGNMENT AT AUTHORIZATION AVG

		BACKWARD BEFORE BEGIN_TRANS BETWEEN BIGINT BINARY 
		BIT BIT_LENGTH BLOCK BOOLEAN BOTH BY

		CACHE CALLED CASCADE CASE CAST CHAIN CHAR_P
		CHARACTER CHARACTERISTICS CHARACTER_LENGTH CHECK CHECKPOINT
		CLASS CLOSE CLUSTER COALESCE COLLATE COLUMN COMMENT COMMIT
		COMMITTED CONNECT CONSTRAINT CONSTRAINTS CONVERSION_P CONVERT
		COPY COUNT CREATE CREATEDB CREATEUSER CROSS CS CUBE CURRENT_DATE
		CURRENT_TIME CURRENT_TIMESTAMP CURRENT_USER CURSOR CYCLE

		DATABASE DATE_P DAY_P DEALLOCATE DEC DECIMAL DECLARE DEFAULT
		DEFERRABLE DEFERRED DEFINER DELETE_P DELIMITER DELIMITERS
		DESC DISTINCT DO DOMAIN_P DOUBLE DROP

		EACH ELSE ENCODING ENCRYPTED END_TRANS ESCAPE EXCEPT
		EXCLUSIVE EXECUTE EXISTS EXPLAIN EXTERNAL EXTRACT

		FALSE_P FETCH FIRST FETCH_FIRST FLOAT_P FOR FORCE FOREIGN FORWARD
		FREEZE FROM FULL FUNCTION

		GET GLOBAL GRANT GROUP_P GROUPING

		HANDLER HAVING HOUR_P

		ILIKE IMMEDIATE IMMUTABLE IMPLICIT_P INCREMENT
		INDEX INHERITS INITIALLY INNER_P INOUT IN_P INPUT
		INSENSITIVE INSERT INSTEAD INT INTEGER INTERSECT
		INTERVAL INTO INVOKER IS ISNULL ISOLATION

		JOIN

		KEY

		LANCOMPILER LANGUAGE LAST LEADING LEFT LEVEL LIKE
		LIMIT LISTEN LOAD LOCAL LOCALTIME LOCALTIMESTAMP LOCATION
		LOWER LOCK_P
		/* LIKEC LIKE2 LIKE4 --> 何れもLIKE に丸めました (for Oracle) */

		MATCH MAX MAXVALUE MIN MINUS_P MINUTE_P MINVALUE MODE MODULE
		MONTH_P MOVE

		NAMES NATIONAL NATURAL NCHAR NEW NEXT NO NOCREATEDB
		NOCREATEUSER NONE NOT NOTHING NOTIFY NOTNULL NOWAIT
		NULL_P NULLIF NULLS NUMERIC

		OCTET_LENGTH OF OFF OFFSET OIDS OLD ON ONLY OPERATOR OPTION OR
		ORDER OUT_P OUTER_P OVERLAPS OVERLAY OWNER

		PARTIAL PARTITION PARTITION_DA PASSWORD PATH_P PENDANT PLACING
		POSITION PRECISION PREPARE PRIMARY PRIOR PRIVILEGES PROCEDURAL PROCEDURE

		READ REAL RECHECK REFERENCES REINDEX RELATIVE RENAME REPLACE
		RESET RESTRICT RETURNS REVOKE RIGHT ROLLBACK ROLLUP ROW ROWID
		ROWNUM ROWS RR RS RULE

		SAMPLE SAMPLE_BLOCK SAMPLE_DA SCHEMA SCN SCROLL SECOND_P SECURITY SELECT
		SEQUENCE SERIALIZABLE SESSION SESSION_USER SET SETOF SETS SHARE
		SHOW SIBLINGS SIMILAR SIMPLE SMALLINT SOME STABLE START
		STATEMENT STATISTICS STDIN STDOUT STORAGE STRICT SUBPARTITION SUBPARTITION_DA
		SUBSTRING SUM SYSID

		TABLE TEMP TEMPLATE TEMPORARY THEN TIME TIMESTAMP
		TO TOAST TRAILING TRANSLATE TRANSACTION TREAT TRIGGER TRIM
		TRUE_P TRUNCATE TRUSTED TYPE_P

		UNENCRYPTED UNION UNION_JOIN UNIQUE UNKNOWN UNLISTEN UNTIL
		UPDATE UPPER UR USAGE USER USING

		VACUUM VALID VALIDATOR VALUES VARCHAR VARCHAR2 VARYING
		VERBOSE VERSION VIEW VOLATILE

		WAIT WHEN WHERE WITH WITH_CS WITH_RR WITH_RS WITH_UR WITHOUT WORK WRITE

		YEAR_P

		ZONE

%token	<str>	OP_NOTEQ OP_GT OP_LT OP_CONCAT OP
%token	<str>	IDENT BIND NAMEDBIND S_REQUIREDSQL M_REQUIREDSQL
		NCONST SCONST EOF_SCONST QT_IDENT EOF_QT_IDENT EOF_COMMENT
%type	<node>	sql
		subquery_factoring_clause
		subquery_factoring_list subquery_factoring_sub_list
		select_sentence select_no_parens select_with_parens
		select_stmt select_stmt_set set_quantifier
		select_col_list select_col_sub_list
		from_clause from_list from_sub_list where_clause
		table_reference table_function column_alias_list column_alias_sub_list
		column_alias_list_with_parens joined_table join_specification
		join_column_list join_column_sub_list
		groupby_clause having_clause
		orderby_clause orderby_list orderby_sub_list
		limit_clause forupdate_clause
		fetch_first_clause opt_fetch_first
		with_clause
		hierachical_query_clause

		a_expr simple_expr compound_expr function_expr

		case_expr simple_case_expr searched_case_expr
		result_expr else_expr simple_when_list simple_when_sub_list
		searched_case_list searched_case_sub_list
		expr_list in_expr
		bind_expr colname_expr single_reqsql_expr multi_reqsql_expr
		between_expr rval_list_expr rval_expr
		cast_target_expr
		opt_limit opt_offset
		sample_clause partition_clause
%type	<str>	all_symbol_op math_symbol_op separate_op
%type	<str>	funcname tablename table_alias_name eof_idents

%type	<keyword>
		opt_query_append into_clause opt_orderby opt_orderby_nulls
		reserved_funcname restricted_sentence
		join_type outer_join_type sub_type
		unreserved_query_keyword
		character_string_type national_character_string_type
		bit_string_type numeric_type scale_numeric_type
		datetime_type interval_type paren_length
		interval_qualifier start_field non_second_datetime_field
		end_field single_datetime_field
		fetch_first_row

/*
 * 字句解析パーサで使用するステータスコード
 * (note)
 *	トークンの値は構文解析パーサが勝手に決めてしまうので、字句解析パーサを
 *	単独で呼び出したときのエラー戻り値がバッティングしないかどうか保証できない。
 *	仕方が無いので、ここで定義して、絶対にバッティングしないようにしている。
 *	(構文解析パーサにエラーコードも採番してもらうということ)
 */
%token		ST_YYLEX_OK ST_YYLEX_MISSING_ID ST_YYLEX_MISMATCH_ID ST_YYLEX_MISSING_TOKEN

/**
 * 演算子の結合順序規則
 * (note)
 *	優先度が低い順に並びます。同列は同じ優先度であることを示します。
 */
%left           UNION EXCEPT
%left           INTERSECT MINUS_P
%left           OR
%left           AND
%right          NOT
%right          '='
%nonassoc       '<' '>'
%nonassoc       LIKE ILIKE SIMILAR
%nonassoc       ESCAPE
%nonassoc       OVERLAPS
%nonassoc       BETWEEN
%nonassoc       IN_P
%right		LOW_PRIORITY
%left           OP_CONCAT OP OPERATOR OP_NOTEQ OP_GT OP_LT
%nonassoc       NOTNULL
%nonassoc       ISNULL
%nonassoc       IS NULL_P TRUE_P FALSE_P UNKNOWN
%left           '+' '-'
%left           '*' '/' '%'
%left           '^'
%left           AT ZONE
%right          UMINUS
%left           '[' ']'
%left           '(' ')'
%left           COLLATE
%left           TYPECAST
%left           '.'
%left		JOIN UNION_JOIN CROSS LEFT FULL RIGHT INNER_P NATURAL

%%

/*
 * SQL解析文法
 *
 * SELECT構文だけを構文解析するのに必要な文法を記述しています。
 *
 * (note) 構文解析パーサの役割
 *-----------------------------
 *	TeamFileにおいては、RequiredSQLまたは通常・名前付きバインド変数を識別する
 *	目的のためだけにSQLの構文解析が必要となります。
 *	従って、これらの変数が指定され得る"式"に関する構文記述が殆どになります。
 *	SQLの構造を理解しなければならない部分があったため、SQL構文と関係のある
 *	識別子を使って文法を記述していますが、RequiredSQLに関係しない構文は
 *	"正しく無視する"ために記述しているに過ぎません。
 *
 *	一般的なSQLパーサのように意味解析の前段階として構文解析を行う訳ではない
 *	SQL文の構造が殆ど無視されたSyntax Treeを生成します。
 *	TeamFileの目的においては十分ですが、将来、意味解析までやりたいのでしたら
 *	根本的に作り直す必要があります。
 */

sql:
		select_sentence
			{
				P_CTX->sqlenode = $1;
			}
		| subquery_factoring_clause select_sentence	/* Oracle 方言 */
			{
				P_CTX->sqlenode = _appendExpr($1, $2);
			}
		| restricted_sentence
			{
				/* 禁止された構文(DDL, 更新DML, DCL) が指定された */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not use DDL or update-DML or DCL sentence");
				SET_ERRCD(DIVY_SQLP_ST_FOUND_UPDATESTMT);
				YYABORT;
			}
		| IDENT
			{
				/* 先頭が有効な開始識別子(SELECT句)ではなかった */
				ERRLOG1(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The first keyword must be \"SELECT\" or \"WITH\"."
					"(keyword = %s)", $1);
				SET_ERRCD(DIVY_SQLP_ST_NONSELECT);
				YYABORT;
			}
		| EOF_COMMENT
			{
				/* コメントが正しく閉じられていなかった */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The c-style comment is not closed.");
				SET_ERRCD(DIVY_SQLP_ST_UNQUOTED_CTSTR);
				YYABORT;
			}
		| error
			{
				/* 解析に失敗した */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not understand this sentence.");
				SET_ERRCD(DIVY_SQLP_ST_UNKNOWN_STMT);
				YYABORT;
			}
	;
/*
 * select 構文とはどのようなものなのかを定義する構文
 * (note)
 *	SELECT構文の前後に指定された括弧を、括弧の無いSELECT構文と同一視
 *	するのに必要な規則です
 */
select_sentence:
		select_no_parens		%prec UMINUS
				{ $$ = $1; }
		| select_with_parens		%prec UMINUS
				{ $$ = $1; }
	;
/*
 * 括弧付きselect 構文
 */
select_with_parens:
		'(' select_no_parens ')'	{ $$ = $2; }
		| '(' select_with_parens ')'	{ $$ = $2; }
	;
/*
 * 括弧なしselect 構文
 */
select_no_parens:
		select_stmt orderby_clause limit_clause fetch_first_clause with_clause forupdate_clause
			{
				/* SELECT 文を表すノードの生成 */
				$$ = _makeNode(P_CTX, DIVY_NT_SELECT, "SELECT");
				$$->first_child = _appendExpr($1, $2);
				$$->first_child = _appendExpr($$->first_child, $3);
				$$->first_child = _appendExpr($$->first_child, $4);
			}
	;
/*
 * select 構文
 */
select_stmt:
		SELECT	{ P_CTX->select_id++; } set_quantifier select_col_list
		into_clause from_clause where_clause
		hierachical_query_clause groupby_clause having_clause
			{
				/* reduceされるのでSELECT句識別IDを1つ前に戻す */
				P_CTX->select_id--;

				$$ = _appendExpr($4, $6);	/* $2 はSELECTのアクション */
				$$ = _appendExpr($$, $7);
				$$ = _appendExpr($$, $8);
				$$ = _appendExpr($$, $9);
			}
		| select_stmt_set UNION { _resetSelectColCnt(p_ctx); }
		  opt_query_append select_stmt_set
			{
				$$ = _appendExpr($1, $5);
			}
		| select_stmt_set EXCEPT { _resetSelectColCnt(p_ctx); }
		  opt_query_append select_stmt_set
			{
				$$ = _appendExpr($1, $5);
			}
		| select_stmt_set INTERSECT { _resetSelectColCnt(p_ctx); }
		  opt_query_append select_stmt_set
			{
				$$ = _appendExpr($1, $5);
			}
		| select_stmt_set MINUS_P { _resetSelectColCnt(p_ctx); }
		  opt_query_append select_stmt_set
			{
				$$ = _appendExpr($1, $5);
			}
	;

/*
 * 副問合せファクタリング構文
 * (note) SQL99あるWITH句とは多少異なるようです。Oracleの方言です。
 */
subquery_factoring_clause:
		WITH subquery_factoring_list
			{
				$$ = $2;

				/* 副問い合わせファクタリングにおける
						カウント数は使用しない */
				_resetSelectColCnt(p_ctx);
				_resetMaxSelectColCnt(p_ctx);
			}
	;
subquery_factoring_list:
		subquery_factoring_sub_list
					{ $$ = $1; }
		| subquery_factoring_list ',' subquery_factoring_sub_list
					{ $$ = _appendExpr($1, $3); }
	;
subquery_factoring_sub_list:
		/* ここのselect_stmt にはDIVY_NT_SELECT_CLAUSEを付けない */
		IDENT AS '(' select_stmt ')'	{ $$ = $4; }
		| unreserved_query_keyword AS '(' select_stmt ')'
						{ $$ = $4; }
	;

/* select_col_list を修飾する構文 */
set_quantifier:
		DISTINCT		{ $$ = NULL; }
		| ALL			{ $$ = NULL; }
		| UNIQUE		{ $$ = NULL; }		/* Oracle 方言 */
		| DISTINCT ON '(' expr_list ')'	{ $$ = $4; }	/* PostgreSQL方言 */
		| /* empty */		{ $$ = NULL; }
	;

/* UION, INTERSECT,MINUS_P, EXCEPT と共に使用される識別子 */
opt_query_append:
		ALL
		| /* empty */		{ $$ = NULL; }
	;
/*
 * SELECT文とみなせる集合の定義
 * (note)
 *	括弧付きSELECT文の列挙に必要な構文(技巧的)
 */
select_stmt_set:
		select_stmt		{ $$ = $1; }
		| select_with_parens	{ $$ = $1; }
	;
/*
 * SELECT句にカンマで区切って並べられる式の構文
 */
select_col_list:
		select_col_sub_list
			{
				$$ = $1;
				/* SELECT句カラムのカウント */
				_incrementSelectColCnt(p_ctx);
			}
		| select_col_list ',' select_col_sub_list
			{
				$$ = _appendExpr($1, $3);	

				/* SELECT句カラムのカウント */
				_incrementSelectColCnt(p_ctx);
			}
	;
select_col_sub_list:
		a_expr
			{
				$$ = $1;
				if ($$) $$->belong_nodetype = DIVY_NT_SELECT_CLAUSE;
			}
		| a_expr AS IDENT
			{
				/* エイリアス名をカラム名にする */
				$$ = _makeNode(P_CTX, DIVY_NT_COLUMN, $3);
				if ($$) $$->belong_nodetype = DIVY_NT_SELECT_CLAUSE;
			}
		| a_expr AS QT_IDENT
			{
				/* エイリアス名をカラム名にする */
				$$ = _makeNode(P_CTX, DIVY_NT_COLUMN, $3);
				if ($$) $$->belong_nodetype = DIVY_NT_SELECT_CLAUSE;
			}
		| a_expr AS unreserved_query_keyword
			{
				/* エイリアス名をカラム名にする */
				$$ = _makeNode(P_CTX, DIVY_NT_COLUMN, $3);
				if ($$) $$->belong_nodetype = DIVY_NT_SELECT_CLAUSE;
			}
		| a_expr AS error
			{
				/* 正しくないエイリアスが指定された */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The wrong column alias name is specified.");
				SET_ERRCD(DIVY_SQLP_ST_INVALID);
				YYABORT;
			}
		| '*'
			{
				/* SELECT 句に'*'を指定することは出来ない */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not use '*' column in \"SELECT\" clause.");
				SET_ERRCD(DIVY_SQLP_ST_FOUND_ASTERISK);
				YYABORT;
			}
	;
/*
 * INTO構文
 */
into_clause:
		INTO
			{
				/* SELECT ... INTO 句は使用できない */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not use \"SELECT ... INTO\" sentence.");
				SET_ERRCD(DIVY_SQLP_ST_FOUND_UPDATESTMT);
				YYABORT;
			}
		| /* empty */		{ $$ = NULL; }
	;

/*
 * FROM句の構文
 */
from_clause:
		FROM from_list		{ $$ = $2; }
		| /* empty */		{ $$ = NULL; }
	;
from_list:
		from_sub_list		{ $$ = $1; }
		| from_list ',' from_sub_list
					{ $$ = _appendExpr($1, $3); }
	;
from_sub_list:
		table_reference		{ $$ = $1; }
	;
/*
 * 結合されるテーブルの規則
 */
table_reference:
		tablename		{ $$ = NULL; }
		| tablename table_alias_name column_alias_list_with_parens
					{ $$ = $3; }
		| tablename AS table_alias_name column_alias_list_with_parens
					{ $$ = $4; }
		| ONLY tablename	{ $$ = NULL; }		/* Oracle, PostgreSQL方言 */
		| ONLY tablename table_alias_name column_alias_list_with_parens
					{ $$ = $4; }		/* Oracle, PostgreSQL方言 */
		| ONLY tablename AS table_alias_name column_alias_list_with_parens
					{ $$ = $5; }		/* Oracle, PostgreSQL方言 */
		| table_function	{ $$ = $1; }		/* PostgreSQL方言 */
		| table_function table_alias_name column_alias_list_with_parens 
			{
				$$ = _appendExpr($1, $3);	/* PostgreSQL方言 */
			}
		| table_function AS table_alias_name column_alias_list_with_parens 
			{
				$$ = _appendExpr($1, $4);	/* PostgreSQL方言 */
			}
		| select_with_parens table_alias_name column_alias_list_with_parens
			{
				$$ = _appendExpr($1, $3);
			}
		| select_with_parens AS table_alias_name column_alias_list_with_parens
			{
				$$ = _appendExpr($1, $4);
			}
		| joined_table		{ $$ = $1; }
	;
tablename:
		IDENT
		| unreserved_query_keyword
					{ $$ = NULL; }
		| IDENT sample_clause	{ $$ = NULL; }	/* Oracle */
		| unreserved_query_keyword sample_clause
					{ $$ = NULL; }	/* Oracle */
		| IDENT partition_clause		/* Oracle */
					{ $$ = NULL; }
		| unreserved_query_keyword partition_clause	/* Oracle */
					{ $$ = NULL; }
		| MODULE '.' IDENT	{ $$ = NULL; }	/* SQL92 */
		| MODULE '.' unreserved_query_keyword
					{ $$ = NULL; }
	;
sample_clause:
		SAMPLE_DA '(' NCONST ')'
			{
				/* SAMPLE_DAは、SAMPLEというトークンが
				   tablenameと衝突するのを防止する目的で導入したもの。
				   括弧の対応が取れていないのは意図的です。*/
					$$ = NULL;
			}
		| SAMPLE_BLOCK '(' NCONST ')'
					{ $$ = NULL; }
	;
partition_clause:
		PARTITION_DA '(' IDENT ')'	{ $$ = NULL; }
		| SUBPARTITION_DA '(' IDENT ')'
					{ $$ = NULL; }
		| PARTITION_DA '(' IDENT ')' sample_clause
					{ $$ = NULL; }
		| SUBPARTITION_DA '(' IDENT ')' sample_clause
					{ $$ = NULL; }
	;
table_function:
		function_expr		{ $$ = $1; }
	;
table_alias_name:
		IDENT
		| QT_IDENT
		| unreserved_query_keyword { $$ = NULL; }
	;
column_alias_list_with_parens:
		'(' column_alias_list ')' { $$ = $2; }
		| /* empty */		  { $$ = NULL; }
	;
column_alias_list:
		column_alias_sub_list	{ $$ = $1; }
		| column_alias_list ',' column_alias_sub_list
					{ $$ = _appendExpr($1, $3); }
	;
column_alias_sub_list:
		colname_expr		{ $$ = $1; }
		| single_reqsql_expr	{ $$ = $1; }
		| bind_expr		{ $$ = $1; }
		| error
			{
				/* 多分持てない。。 */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not have this value for column_alias.");
				SET_ERRCD(DIVY_SQLP_ST_INVALID);
				YYABORT;
			}
	;
joined_table:
		table_reference CROSS JOIN table_reference
					{ $$ = _appendExpr($1, $4); }
		| table_reference UNION_JOIN table_reference
					{ $$ = _appendExpr($1, $3); }
		| table_reference join_type table_reference join_specification
					{ $$ = _appendExpr($1, $4); }
		| table_reference NATURAL join_type table_reference join_specification
					{ $$ = _appendExpr($1, $5); }
	;
join_type:
		JOIN
		| INNER_P JOIN
		| outer_join_type JOIN
		| outer_join_type OUTER_P JOIN
	;
outer_join_type:
		LEFT
		| RIGHT
		| FULL
	;	
join_specification:
		ON a_expr		{ $$ = $2; }
		| USING '(' join_column_list ')'
					{ $$ = $3; }
	;
join_column_list:
		join_column_sub_list	{ $$ = $1; }
		| join_column_list ',' join_column_sub_list
					{ $$ = _appendExpr($1, $3); }
	;
join_column_sub_list:
		colname_expr		{ $$ = NULL; }
		| single_reqsql_expr	{ $$ = $1; }
		| bind_expr		{ $$ = $1; }
		| error
			{
				/* 多分持てない。。 */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not have this value for join_column_list.");
				SET_ERRCD(DIVY_SQLP_ST_INVALID);
				YYABORT;
			}
	;

/*
 * WHERE句の構文
 */
where_clause:
		WHERE a_expr		{ $$ = $2; }
		| /* empty */		{ $$ = NULL; }
	;
/*
 * 階層問合せの構文(Oracle)
 */
hierachical_query_clause:
		START WITH a_expr	{ $$ = $3; }
		| START WITH a_expr CONNECT BY a_expr
					{ $$ = _appendExpr($3, $6); }
		| /* empty */		{ $$ = NULL; }
	;

/*
 * GROUP BY句の構文
 */
groupby_clause:
		GROUP_P BY expr_list	{ $$ = $3; }
		| /* empty */		{ $$ = NULL; }
	;

/*
 * HAVING句 構文
 */
having_clause:
		HAVING a_expr		{ $$ = $2; }
		| /* empty */   	{ $$ = NULL; }
	;	

/*
 * ORDER BY構文
 */
orderby_clause:
		ORDER BY orderby_list	{ $$ = $3; }
		| ORDER SIBLINGS BY orderby_list	/* Oracle 方言 */
					{ $$ = $4; }
		| ORDER error
			{
				/* 構文に誤り発見 */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The syntax of \"ORDER BY\" is wrong.");
				SET_ERRCD(DIVY_SQLP_ST_INVALID);
				YYABORT;
			}
		| /* empty */		{ $$ = NULL; }
	;
orderby_list:
		orderby_sub_list	{ $$ = $1; }
		| orderby_list ',' orderby_sub_list
					{ $$ = _appendExpr($1, $3); }
	;
orderby_sub_list:
		a_expr			{ $$ = $1; }
		| a_expr opt_orderby 	{ $$ = $1; }
		| a_expr opt_orderby_nulls
					{ $$ = $1; }	/* Oracle 方言 */
		| a_expr opt_orderby opt_orderby_nulls
					{ $$ = $1; }	/* Oracle 方言 */
	;
opt_orderby:
		ASC
		| DESC
		| USING all_symbol_op
	;
opt_orderby_nulls:
		NULLS FIRST		/* Oracle 方言 */
		| NULLS LAST		/* Oracle 方言 */
	;

/*
 * LIMIT 構文 (PostgreSQL)
 */
limit_clause:
		LIMIT opt_limit OFFSET opt_offset
					{ $$ = _appendExpr($2, $4); }
		| /* empty */		{ $$ = NULL; }
	;
opt_limit:
		NCONST			{ $$ = NULL; }
		| bind_expr		{ $$ = $1; }
		| ALL			{ $$ = NULL; }
		| error
			{
				/* LIMITの後ろには数値か"ALL"だけ */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not have this value for \"LIMIT\" clause.");
				SET_ERRCD(DIVY_SQLP_ST_INVALID);
				YYABORT;
			}
	;
opt_offset:
		NCONST			{ $$ = NULL; }
		| bind_expr		{ $$ = $1; }
		| error
			{
				/* 数値以外はNG */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not have this value for \"OFFSET\" clause.");
				SET_ERRCD(DIVY_SQLP_ST_INVALID);
				YYABORT;
			}
	;
/*
 * FETCH FIRST 構文 (DB2)
 */
fetch_first_clause:
		FETCH_FIRST opt_fetch_first fetch_first_row ONLY
					{ $$ = $2; }
		| /* empty */		{ $$ = NULL; }
	;
opt_fetch_first:
		NCONST			{ $$ = NULL; }
		| bind_expr
					{ $$ = $1; }
		| error	{
				/* 数値以外はNG */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not have this value for \"FETCH FIRST\" clause.");
				SET_ERRCD(DIVY_SQLP_ST_INVALID);
				YYABORT;
			}
	;
fetch_first_row:
		ROW
		| ROWS
	;
/*
 * WITH 句 (DB2) トランザクションの分離レベルの指定に使用する
 */
with_clause:
		WITH_RR			{ $$ = NULL; }
		| WITH_RS		{ $$ = NULL; }
		| WITH_CS		{ $$ = NULL; }
		| WITH_UR		{ $$ = NULL; }
		| /* empty */		{ $$ = NULL; }
	;

/*
 * FOR UPDATE 構文
 */
forupdate_clause:
		FOR UPDATE
			{
				/* 禁止された構文(更新DML, DCL) が指定された */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not use \"SELECT ... FOR UPDATE\" sentence");
				SET_ERRCD(DIVY_SQLP_ST_FOUND_UPDATESTMT);
				YYABORT;
			}
		| /* empty */		{ $$ = NULL; }
	;

/*-----------------------------------------------------------------------------
 * ある１つの"式"(単項、複合)を表す構文
 *---------------------------------------------------------------------------*/
a_expr:
		simple_expr		{ $$ = $1; }	/* 単純式 */
		| compound_expr		{ $$ = $1; }	/* 複合式 */
		| case_expr		{ $$ = $1; }	/* CASE式 */
		| function_expr		{ $$ = $1; }	/* ファンクション式 */
		| a_expr separate_op a_expr				%prec LOW_PRIORITY
			{
				/* 左辺値のチェック */
				if (_validate_lval(POOL, $2, $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				/* $$SMxxx が IN句以外の演算子の右辺値 */
				if ($3 && $3->nodetype == DIVY_NT_M_REQUIREDSQL) {
					ERRLOG3(POOL, APLOG_WARNING,
						DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The RequiredSQL \"%s\" was specified "
						"as rvalue operand of \"%s\" operator."
						"(lvalue = %s)", $3->name, $2, $1->name);
					SET_ERRCD(DIVY_SQLP_ST_WRONG_M_PREF);
					YYABORT;
				}

				if (_enable_making_expr($2, $1, $3)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR, $2, $1, $3);
				}
				else {
					$$ = _appendExpr($1, $3);
				}
			}
		| a_expr separate_op sub_type select_with_parens
			{
				/* 左辺値のチェック */
				if (_validate_lval(POOL, $2, $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				$$ = _appendExpr($1, $4);
			}
		| a_expr IN_P in_expr
			{
				/* 左辺値のチェック */
				if (_validate_lval(POOL, $2, $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				if (_enable_making_expr($2,$1,$3)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR, $2, $1, $3);
				}
				else {
					$$ = _appendExpr($1, $3);
				}
			}
		| a_expr NOT IN_P in_expr
			{
				/* 左辺値のチェック */
				if (_validate_lval(POOL, "not in", $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				if (_enable_making_expr($2, $1, $4)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR, "not in", $1, $4);
				}
				else {
					$$ = _appendExpr($1, $4);
				}
			}
		| a_expr BETWEEN between_expr AND between_expr
			{
				/* 左辺値のチェック */
				if (_validate_lval(POOL, $2, $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}
				/* ANDの前の式にANDの後ろの式を連結する */
				$$ = _appendExpr($3, $5);

				if ($$ && _enable_making_expr($2, $1, $$)) {
					/* BETWEEN 式の生成 */
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR, $2, $1, $$);
				}
				else {
					$$ = _appendExpr($1, $$);
				}
			}
		| a_expr NOT BETWEEN between_expr AND between_expr
			{
				/* 左辺値のチェック */
				if (_validate_lval(POOL, "not between", $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}
				/* ANDの前の式にANDの後ろの式を連結する */
				$$ = _appendExpr($4, $6);
				if ($$ && _enable_making_expr($2,$1,$$)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR, "not in", $1, $$);	
				}
				else {
					$$ = _appendExpr($1, $$);
				}
			}
		| a_expr BETWEEN between_expr error
			{
				/* 構文エラー */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Unsupported sql statement.(\"BETWEEN\" operator)");
				SET_ERRCD(DIVY_SQLP_ST_INVALID);
				YYABORT;
			}
		| a_expr NOT BETWEEN between_expr error
			{
				/* 構文エラー */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Unsupported sql statement.(\"NOT BETWEEN\" operator)");
				SET_ERRCD(DIVY_SQLP_ST_INVALID);
				YYABORT;
			}
		| a_expr LIKE a_expr
			{ 
				if (_validate_lval(POOL, $2, $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				/* $$SMxxx が IN句以外の演算子の右辺値 */
				if ($3 && $3->nodetype == DIVY_NT_M_REQUIREDSQL) {
					ERRLOG3(POOL, APLOG_WARNING,
						DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The RequiredSQL \"%s\" was specified "
						"as rvalue operand of \"%s\" operator."
						"(lvalue = %s)", $3->name, $2, $1->name);
					SET_ERRCD(DIVY_SQLP_ST_WRONG_M_PREF);
					YYABORT;
				}

				if (_enable_making_expr($2, $1, $3)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR, $2, $1, $3);	
				}
				else {
					$$ = _appendExpr($1, $3);
				}
			}
		| a_expr LIKE a_expr ESCAPE SCONST
			{
				/* 左辺値のチェック */
				if (_validate_lval(POOL, $2, $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				/* $$SMxxx が IN句以外の演算子の右辺値 */
				if ($3 && $3->nodetype == DIVY_NT_M_REQUIREDSQL) {
					ERRLOG3(POOL, APLOG_WARNING,
						DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The RequiredSQL \"%s\" was specified "
						"as rvalue operand of \"%s\" operator."
						"(lvalue = %s)", $3->name, $2, $1->name);
					SET_ERRCD(DIVY_SQLP_ST_WRONG_M_PREF);
					YYABORT;
				}

				if (_enable_making_expr($2,$1,$3)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR, $2, $1, $3);	
				}
				else {
					$$ = _appendExpr($1, $3);
				}
			}
		| a_expr NOT LIKE a_expr
			{
				/* 左辺値のチェック */
				if (_validate_lval(POOL, "not like", $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				/* $$SMxxx が IN句以外の演算子の右辺値 */
				if ($4 && $4->nodetype == DIVY_NT_M_REQUIREDSQL) {
					ERRLOG3(POOL, APLOG_WARNING,
						DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The RequiredSQL \"%s\" was specified "
						"as rvalue operand of \"%s\" operator."
						"(lvalue = %s)", $4->name, "not like", $1->name);
					SET_ERRCD(DIVY_SQLP_ST_WRONG_M_PREF);
					YYABORT;
				}

				if (_enable_making_expr("not like",$1,$4)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR,
									"not like", $1, $4);	
				}
				else {
					$$ = _appendExpr($1, $4);
				}
			}
		| a_expr NOT LIKE a_expr ESCAPE SCONST
			{
				/* 左辺値のチェック */
				if (_validate_lval(POOL, "not like", $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				/* $$SMxxx が IN句以外の演算子の右辺値 */
				if ($4 && $4->nodetype == DIVY_NT_M_REQUIREDSQL) {
					ERRLOG3(POOL, APLOG_WARNING,
						DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The RequiredSQL \"%s\" was specified "
						"as rvalue operand of \"%s\" operator."
						"(lvalue = %s)", $4->name, "not like", $1->name);
					SET_ERRCD(DIVY_SQLP_ST_WRONG_M_PREF);
					YYABORT;
				}

				if (_enable_making_expr("not like",$1,$4)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR,
									"not like", $1, $4);	
				}
				else {
					$$ = _appendExpr($1, $4);
				}
			}
		| a_expr SIMILAR TO a_expr				%prec SIMILAR
			{
				if (_validate_lval(POOL, $2, $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				/* $$SMxxx が IN句以外の演算子の右辺値 */
				if ($4 && $4->nodetype == DIVY_NT_M_REQUIREDSQL) {
					ERRLOG3(POOL, APLOG_WARNING,
						DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The RequiredSQL \"%s\" was specified "
						"as rvalue operand of \"%s\" operator."
						"(lvalue = %s)", $4->name, $2, $1->name);
					SET_ERRCD(DIVY_SQLP_ST_WRONG_M_PREF);
					YYABORT;
				}

				if (_enable_making_expr($2, $1, $4)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR, $2, $1, $4);	
				}
				else {
					$$ = _appendExpr($1, $4);
				}
			}
		| a_expr SIMILAR TO a_expr ESCAPE SCONST
			{
				if (_validate_lval(POOL, $2, $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				/* $$SMxxx が IN句以外の演算子の右辺値 */
				if ($4 && $4->nodetype == DIVY_NT_M_REQUIREDSQL) {
					ERRLOG3(POOL, APLOG_WARNING,
						DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The RequiredSQL \"%s\" was specified "
						"as rvalue operand of \"%s\" operator."
						"(lvalue = %s)", $4->name, $2, $1->name);
					SET_ERRCD(DIVY_SQLP_ST_WRONG_M_PREF);
					YYABORT;
				}

				if (_enable_making_expr($2, $1, $4)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR, $2, $1, $4);	
				}
				else {
					$$ = _appendExpr($1, $4);
				}
			}
		| a_expr NOT SIMILAR TO a_expr				%prec SIMILAR
			{
				if (_validate_lval(POOL, "not similar to", $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				/* $$SMxxx が IN句以外の演算子の右辺値 */
				if ($5 && $5->nodetype == DIVY_NT_M_REQUIREDSQL) {
					ERRLOG3(POOL, APLOG_WARNING,
						DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The RequiredSQL \"%s\" was specified "
						"as rvalue operand of \"%s\" operator."
						"(lvalue = %s)", $5->name, "not similar to",
						$1->name);
					SET_ERRCD(DIVY_SQLP_ST_WRONG_M_PREF);
					YYABORT;
				}

				if (_enable_making_expr($2, $1, $5)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR, $2, $1, $5);	
				}
				else {
					$$ = _appendExpr($1, $5);
				}
			}
		| a_expr NOT SIMILAR TO a_expr ESCAPE SCONST
			{
				if (_validate_lval(POOL, "not similar to", $1)) {
					SET_ERRCD(DIVY_SQLP_ST_WRONG_LVAL);
					YYABORT;
				}

				/* $$SMxxx が IN句以外の演算子の右辺値 */
				if ($5 && $5->nodetype == DIVY_NT_M_REQUIREDSQL) {
					ERRLOG3(POOL, APLOG_WARNING,
						DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The RequiredSQL \"%s\" was specified "
						"as rvalue operand of \"%s\" operator."
						"(lvalue = %s)", $5->name, "not similar to",
						$1->name);
					SET_ERRCD(DIVY_SQLP_ST_WRONG_M_PREF);
					YYABORT;
				}

				if (_enable_making_expr($2, $1, $5)) {
					$$ = _makeExpr(P_CTX, DIVY_NT_OPERATOR, $2, $1, $5);	
				}
				else {
					$$ = _appendExpr($1, $5);
				}
			}
	;
/*
 * 単純式を表す構文
 */
simple_expr:
		colname_expr		{ $$ = $1; }
		| SCONST		{ $$ = _makeNode(P_CTX, DIVY_NT_COLUMN, $1); }
		| NCONST		{ $$ = _makeNode(P_CTX, DIVY_NT_COLUMN, $1); }
		| ROWID			{ $$ = _makeNode(P_CTX, DIVY_NT_COLUMN, $1); }	/* Oracle */
		| ROWNUM		{ $$ = _makeNode(P_CTX, DIVY_NT_COLUMN, $1); }	/* Oracle */
		| NULL_P		{ $$ = _makeNode(P_CTX, DIVY_NT_COLUMN, $1); }
		| bind_expr		{ $$ = $1; }
		| single_reqsql_expr	{ $$ = $1; }
		| multi_reqsql_expr	{ $$ = $1; }
		| select_with_parens				%prec UMINUS
					{ $$ = $1; }
		| EXISTS select_with_parens { $$ = $2; }
		| CAST '(' a_expr AS cast_target_expr ')'
					{ $$ = $3; }
		| eof_idents		{ $$ = NULL; }
	;
/*
 * 複合式を表す構文
 */
compound_expr:
		'(' a_expr ')'		{ $$ = $2; }
		| '+' a_expr				%prec UMINUS
					{ $$ = $2; }
		| '-' a_expr				%prec UMINUS
					{ $$ = $2; }
		| '%' a_expr		{ $$ = $2; }
		| '^' a_expr		{ $$ = $2; }
		| a_expr '*' a_expr	{ $$ = _appendExpr($1, $3); }
		| a_expr '/' a_expr	{ $$ = _appendExpr($1, $3); }
		| a_expr '+' a_expr	{ $$ = _appendExpr($1, $3); }
		| a_expr '-' a_expr	{ $$ = _appendExpr($1, $3); }
		| a_expr '%' a_expr	{ $$ = _appendExpr($1, $3); }
		| a_expr '^' a_expr	{ $$ = _appendExpr($1, $3); }
		| a_expr OP_CONCAT a_expr
					{ $$ = _appendExpr($1, $3); }
		| a_expr AND a_expr 	{ $$ = _appendExpr($1, $3); }
		| a_expr OR a_expr	{ $$ = _appendExpr($1, $3); }
		| NOT a_expr		{ $$ = $2; }
		| a_expr OP a_expr						%prec OP
					{ $$ = _appendExpr($1, $3); }
		| OP a_expr							%prec OP
					{ $$ = $2; }
		| a_expr IS NULL_P	{ $$ = $1; }
		| a_expr ISNULL		{ $$ = $1; }
		| a_expr IS NOT NULL_P	{ $$ = $1; }
		| a_expr NOTNULL	{ $$ = $1; }
		| a_expr IS UNKNOWN	{ $$ = $1; }
		| a_expr IS NOT UNKNOWN	{ $$ = $1; }
		| a_expr IS TRUE_P	{ $$ = $1; }
		| a_expr IS NOT TRUE_P	{ $$ = $1; }
		| a_expr IS FALSE_P	{ $$ = $1; }
		| a_expr IS NOT FALSE_P	{ $$ = $1; }
	;

/*
 * ファンクション式を表す構文
 */
function_expr:
		funcname '(' ')'	{ $$ = NULL; }
		| funcname '(' '*' ')'	{ $$ = NULL; }
		| funcname '(' rval_list_expr ')'
					{ $$ = $3; }
		| funcname '(' ALL rval_list_expr ')'
					{ $$ = $4; }		/* PostgreSQL ? */
		| funcname '(' DISTINCT rval_list_expr ')'	/* PostgreSQL ? */
					{ $$ = $4; }
	;

/* 一般的なDB 関数を定義する構文 */
funcname:
		IDENT			{ $$ = $1; }
		| reserved_funcname	{ $$ = $1; }
	;

/* 予め理解している関数の名前一覧 */
reserved_funcname:
		ABS
		| ACOS
		| ADDMONTHS
		| AVG
		| BIT_LENGTH
		| CHARACTER_LENGTH
		| COLLATE
		| CONVERT
		| COUNT
		| CURRENT_DATE
		| CURRENT_TIME
		| CURRENT_TIMESTAMP
		| EXTRACT
		| LOWER
		| MAX
		| MIN
		| OCTET_LENGTH
		| POSITION
		| SUBSTRING
		| REPLACE
		| SUM
		| TRANSLATE
		| TRIM
		| UPPER
	;

/*
 * CASE 式を表す構文
 */
case_expr:
		CASE simple_case_expr END_TRANS		{ $$ = $2; }
		| CASE simple_case_expr ELSE else_expr END_TRANS
			{
				$$ = _appendExpr($2, $4);
			}
		| CASE searched_case_expr END_TRANS	{ $$ = $2; }
		| CASE searched_case_expr ELSE else_expr END_TRANS
			{
				$$ = _appendExpr($2, $4);
			}
	;
simple_case_expr:
		a_expr simple_when_list	{ $$ = $2; }
	;
searched_case_expr:
		searched_case_list	{ $$ = $1; }
	;
simple_when_list:
		simple_when_sub_list	{ $$ = $1; }
		| simple_when_list simple_when_sub_list
			{
				$$ = _appendExpr($1, $2);
			}
	;
/* ##### FIXME comparison_expr が無い(WHENの後) */
simple_when_sub_list:
		WHEN IDENT THEN result_expr	{ $$ = NULL; }
	;
/* ##### FIXME condition が無い(WHENの後) */
searched_case_list:
		searched_case_sub_list	{ $$ = $1; }
		| searched_case_list searched_case_sub_list
			{
				$$ = _appendExpr($1, $2);
			}
	;
searched_case_sub_list:
		WHEN NCONST THEN result_expr	{ $$ = NULL; }
	;
/* ##### FIXME 正しく書くこと */
else_expr:
		IDENT			{ $$ = NULL; }
		| NCONST		{ $$ = NULL; }
		| SCONST		{ $$ = NULL; }
	;
/* ##### FIXME 正しく書くこと */
result_expr:
		IDENT			{ $$ = NULL; }
		| NCONST		{ $$ = NULL; }
		| SCONST		{ $$ = NULL; }
	;

/*
 * IN 演算子の右辺値に指定する式
 */
in_expr:
		select_with_parens	{ $$ = $1; }
		| '(' expr_list ')'	{ $$ = $2; }
		| multi_reqsql_expr	{ $$ = $1; } 
	;

/*
 * BETWEEN 演算子の右辺値に指定する式
 */
between_expr:
		IDENT			{ $$ = NULL; }
		| NCONST		{ $$ = NULL; }
		| SCONST		{ $$ = NULL; }
		| unreserved_query_keyword
					{ $$ = NULL; }
		| NULL_P		{ $$ = NULL; }
		| ROWID			{ $$ = NULL; }	/* Oracle 方言 */
		| ROWNUM		{ $$ = NULL; }	/* Oracle 方言 */
		| bind_expr		{ $$ = $1; }
		| single_reqsql_expr	{ $$ = $1; }
		| multi_reqsql_expr
			{	/* $$SMxxx がBETWEEN 演算子と共に使用された */
				ERRLOG1(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The RequiredSQL \"%s\" was specified "
					"as BETWEEN parameter." , $1->name);
				SET_ERRCD(DIVY_SQLP_ST_WRONG_M_PREF);
				YYABORT;
			}
		| select_with_parens	{ $$ = $1; }
		| function_expr		{ $$ = $1; }	/* ファンクション式 */
		| eof_idents		{ $$ = NULL; } 
	;
/*
 * 式の集合
 */
expr_list:
		a_expr			{ $$ = $1; }
		| expr_list ',' a_expr	{ $$ = _appendExpr($1, $3); }
	;

/*
 * 演算子の右辺値に並ぶ複数個の構成要素を表す式
 */
rval_list_expr:
		rval_expr		{ $$ = $1; }
		| rval_list_expr ',' rval_expr
					{ $$ = _appendExpr($1, $3); }
	;
rval_expr:
		colname_expr		{ $$ = NULL; }
		| NCONST		{ $$ = NULL; }
		| SCONST		{ $$ = NULL; }
		| NULL_P		{ $$ = NULL; }
		| ROWID			{ $$ = NULL; }	/* Oracle 方言 */
		| ROWNUM		{ $$ = NULL; }	/* Oracle 方言 */
		| '*'			{ $$ = NULL; }
		| bind_expr		{ $$ = $1; }
		| single_reqsql_expr	{ $$ = $1; }
		| multi_reqsql_expr
			{ /* $$SMxxx が関数の中に指定された場合 */
				ERRLOG1(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The RequiredSQL \"%s\" was specified "
					"as function parameter.", $1->name);
				SET_ERRCD(DIVY_SQLP_ST_WRONG_M_PREF);
				YYABORT;
			}
		| funcname '(' rval_list_expr ')'
						{ $$ = $3; }
		| CAST '(' a_expr AS cast_target_expr ')'
					{ $$ = $3; }
		| eof_idents		{ $$ = NULL; } 
	;

/**
 * CAST 式のターゲットを記述する構文
 */
cast_target_expr:
		character_string_type	{ $$ = NULL; }
		|character_string_type CHARACTER SET IDENT
					{ $$ = NULL; }
		| national_character_string_type
					{ $$ = NULL; }
		| bit_string_type	{ $$ = NULL; }
		| numeric_type		{ $$ = NULL; }
		| datetime_type		{ $$ = NULL; }
		| interval_type		{ $$ = NULL; }
	;
character_string_type:
		CHARACTER paren_length
		| CHAR_P paren_length
		| CHARACTER VARYING '(' NCONST ')'
		| CHAR_P VARYING '(' NCONST ')'
		| VARCHAR '(' NCONST ')'
		| VARCHAR2 '(' NCONST ')'
	;
national_character_string_type:
		NATIONAL CHARACTER paren_length
		| NATIONAL CHAR_P paren_length
		| NCHAR paren_length
		| NATIONAL CHARACTER VARYING '(' NCONST ')'
		| NATIONAL CHAR_P VARYING '(' NCONST ')'
		| NCHAR VARYING '(' NCONST ')'
	;
bit_string_type:
		BIT paren_length
		| BIT VARYING '(' NCONST ')'
	;
numeric_type:
		scale_numeric_type
		| scale_numeric_type '(' NCONST ')'
		| scale_numeric_type '(' NCONST '.' NCONST ')'
		| INTEGER
		| INT
		| SMALLINT
	;
scale_numeric_type:
		NUMERIC
		| DECIMAL
		| DEC
	;
datetime_type:
		DATE_P
		| TIME paren_length
		| TIME paren_length WITH TIME ZONE
		| TIMESTAMP paren_length
		| TIMESTAMP paren_length WITH TIME ZONE
	;
interval_type:
		INTERVAL interval_qualifier
	;
interval_qualifier:
		start_field TO end_field
		| single_datetime_field
	;
start_field:
		non_second_datetime_field paren_length
	;
non_second_datetime_field:
		YEAR_P
		| MONTH_P
		| DAY_P
		| HOUR_P
		| MINUTE_P
	;
end_field:
		non_second_datetime_field
		| SECOND_P paren_length
	;
single_datetime_field:
		non_second_datetime_field paren_length
		| SECOND_P
		| SECOND_P '(' NCONST ')'
		| SECOND_P '(' NCONST ',' NCONST ')'
	;
paren_length:
		'(' NCONST ')'		{ $$ = NULL; }
		| /* empty */		{ $$ = NULL; }
	;

/*-----------------------------------------------------------------------------
 * "式"を構成する"要素"を定義する構文 
 *---------------------------------------------------------------------------*/
/*
 * キーワードとしては予約されているが、
 * SELECT文中のカラム名やエイリアス名として利用できるキーワード一覧
 */
unreserved_query_keyword:
		ABSOLUTE
		| ACTION
		| ADD
		| AFTER
		| AGGREGATE
		| ASSIGNMENT
		| AT
		| BACKWARD
		| BEFORE
		| BLOCK
		| BY
		| CALLED
		| CASCADE
		| CHAIN
		| CHARACTERISTICS
		| CHECKPOINT
		| CLASS
		| CLOSE
		| CLUSTER
		| COMMENT
		| COMMIT
		| COMMITTED
		| CONSTRAINTS
		| COPY
		| CREATEDB
		| CREATEUSER
		| CUBE
		| CS
		| CURSOR
		| DAY_P
		| DECLARE
		| DEFERRED
		| DEFINER
		| DELETE_P
		| DELIMITER
		| DELIMITERS
		| DOMAIN_P
		| DOUBLE
		| DROP
		| EACH
		| ENCRYPTED
		| ESCAPE
		| EXECUTE
		| EXTERNAL
		| FETCH
		| FIRST
		| FORWARD
		| FUNCTION
		| GET
		| GLOBAL
		| GROUPING
		| HANDLER
		| HOUR_P
		| IMMEDIATE
		| IMMUTABLE
		| INHERITS
		| INOUT
		| INPUT
		| INSENSITIVE
		| INSERT
		| INSTEAD
		| INVOKER
		| ISOLATION
		| KEY
		| LAST
		| LEVEL
		| LOCAL
		| MATCH
		| MINUTE_P
		| MONTH_P
		| MOVE
		| NAMES
		| NATIONAL
		| NEXT
		| NO
		| NOCREATEDB
		| NOCREATEUSER
		| NOTHING
		| NOTIFY
		| NULLS
		| OF
		| OIDS
		| OPERATOR
		| OPTION
		| OUT_P
		| PARTIAL
		| PARTITION
		| PASSWORD
		| PATH_P
		| PENDANT
		| PRECISION
		| PRIOR
		| PRIVILEGES
		| PROCEDURE
		| READ
		| RECHECK
		| RELATIVE
		| RESET
		| RESTRICT
		| RETURNS
		| REVOKE
		| ROLLUP
		| ROW
		| ROWS
		| RR
		| RS
		| RULE
		| SAMPLE
		| SCHEMA
		| SCN
		| SCROLL
		| SECOND_P
		| SECURITY
		| SESSION
		| SERIALIZABLE
		| SET
		| SETS
		| SHOW
		| SIBLINGS
		| SIMPLE
		| STABLE
		| STATEMENT
		| STATISTICS
		| STDIN
		| STDOUT
		| STORAGE
		| STRICT
		| SUBPARTITION
		| SYSID
		| TEMP
		| TEMPORARY
		| TOAST
		| TRANSACTION
		| TRIGGER
		| TRUNCATE
		| TYPE_P
		| UNENCRYPTED
		| UNKNOWN
		| UNTIL
		| UPDATE
		| UR
		| USAGE
		| VACUUM
		| VALID
		| VALUES
		| VARYING
		| VERSION
		| VOLATILE
		| WAIT
		| WITH
		| WITHOUT
		| WRITE
		| YEAR_P
		| ZONE
	;

sub_type:
		ALL
		| SOME
		| ANY
	;
/*
 * 生成値が１つのRequiredSQLを定義する構文
 */
single_reqsql_expr:
		NAMEDBIND	{ $$ = _makeNode(P_CTX, DIVY_NT_NAMEDBIND,     $1); }
		| S_REQUIREDSQL	{ $$ = _makeNode(P_CTX, DIVY_NT_S_REQUIREDSQL, $1); }
	;
/*
 * 生成値が複数のRequiredSQLを定義する構文
 */
multi_reqsql_expr:
		M_REQUIREDSQL	{ $$ = _makeNode(P_CTX, DIVY_NT_M_REQUIREDSQL, $1); }
	;
/*
 * 通常バインド変数を定義する構文
 */
bind_expr:
		BIND		{ $$ = _makeNode(P_CTX, DIVY_NT_BIND, $1); }
	;

/*
 * DB カラム名になる識別子を定義する
 */
colname_expr:
		IDENT		{ $$ = _makeNode(P_CTX, DIVY_NT_COLUMN, $1); } 
		| QT_IDENT	{ $$ = _makeNode(P_CTX, DIVY_NT_COLUMN, $1); }
		| unreserved_query_keyword
				{ $$ = _makeNode(P_CTX, DIVY_NT_COLUMN, $1); }
	;

/*
 * 終端マーカが存在しなかったためエラーとなり得る構文の一覧
 */
eof_idents:
		EOF_SCONST
			{
				/* シングルクオートが正しく閉じられていなかった */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The single-quote is not closed.");
				SET_ERRCD(DIVY_SQLP_ST_UNQUOTED_STR);
				YYABORT;
			}
		| EOF_QT_IDENT
			{
				/* ダブルクオートが正しく閉じられていなかった */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The dobule-quote is not closed.");
				SET_ERRCD(DIVY_SQLP_ST_UNQUOTED_DQSTR);
				YYABORT;
			}
		| EOF_COMMENT
			{
				/* コメントが正しく閉じられていなかった */
				ERRLOG0(POOL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The c-style comment is not closed.");
				SET_ERRCD(DIVY_SQLP_ST_UNQUOTED_CTSTR);
				YYABORT;
			}	
	;

/*-----------------------------------------------------------------------------
 * "式"を構成する演算子を定義する構文 
 *---------------------------------------------------------------------------*/
/*
 * 全ての記号演算子を表す構文
 */
all_symbol_op:
		OP			{ $$ = $1; }
		| math_symbol_op	{ $$ = $1; }
	;
/*
 * 算術演算に使用される演算子一覧
 */
math_symbol_op:
		'+'			{ $$ = "+"; }
		| '-'			{ $$ = "-"; }
		| '*'			{ $$ = "*"; }
		| '/'			{ $$ = "/"; }
		| '%'			{ $$ = "%"; }
		| '^'			{ $$ = "^"; }
		| '<'			{ $$ = "<"; }
		| '>'			{ $$ = ">"; }
		| '='			{ $$ = "="; }
	;

/*
 * 式から左辺値を識別するために使用する演算子の一覧
 * (note) BETWEENは外しています
 */
separate_op:
		OP_NOTEQ	{ $$ = $1;  }
		| OP_GT		{ $$ = $1;  }
		| OP_LT		{ $$ = $1;  }
		| '<'		{ $$ = "<"; }
		| '>'		{ $$ = ">"; }
		| '='		{ $$ = "="; }
	;

/*-----------------------------------------------------------------------------
 * エラー構文の定義
 *---------------------------------------------------------------------------*/
/*
 * 禁止された構文(DDL, 更新系DML, DCL)を記述する
 * (note)
 *	文の先頭に表示される可能性があるトークンだけを記述すればよい。
 */
restricted_sentence:
		ABORT_TRANS
		| ALTER
		| ANALYZE
		| BEGIN_TRANS
		| CHECKPOINT
		| CLOSE
		| CLUSTER	/* for PostgreSQL */
		| COMMENT
		| COMMIT
		| COPY
		| CREATE
		| DEALLOCATE
		| DECLARE
		| DELETE_P
		| DROP
		| END_TRANS	/* for PostgreSQL */
		| EXECUTE
		| EXPLAIN	/* for PostgreSQL */
		| FETCH
		| GRANT
		| INSERT
		| LISTEN	/* for PostgreSQL */
		| LOAD
		| LOCK_P
		| MOVE		/* for PostgreSQL */
		| NOTIFY	/* for PostgreSQL */
		| PREPARE
		| REINDEX	/* for PostgreSQL */
		| RESET		/* for PostgreSQL */
		| REVOKE
		| ROLLBACK
		| SET		/* for PostgreSQL */
		| SHOW		/* for PostgreSQL */
		| START		/* for PostgreSQL */
		| TRUNCATE
		| UNLISTEN	/* for PostgreSQL */
		| UPDATE
		| VACUUM	/* for PostgreSQL */
	;
%%

/**
 * パーサで発生したエラーを報告する。
 * (note)
 *	構文解析パーサが自動的に呼び出すエラーハンドラです。
 * (note)
 *	構文解析パーサは、error構文によって予め逃がしていたエラーに対しても
 *	容赦なくエラーログ関数yyerror を出力してしまいます。
 *	ただ、重要なメッセージもあるため、informationという形で出力
 *	することにします。そのためにも、_REMOVE_ERR_MSG_PART で定義された
 *	エラーを匂わすメッセージを切り取ります。
 *
 * @param str const char *
 */
void yyerror(divy_sql_parser_ctx *p_ctx, const char *str)
{
	const char *errmsg;

	/*
	 * エラーを匂わすエラーメッセージ文字列を削り取る
	 */
	if (str && (errmsg = divy_strstr_c((char *)str, _REMOVE_ERR_MSG_PART)) != NULL &&
							str == errmsg) {
		/* _REMOVE_ERR_MSG_PART 以降の文字列を取り出す */
		errmsg = (const char *) &str[_REMOVE_ERR_MSG_PART_EN];
	}
	else {
		errmsg = str;
	}

	/*
	 * メッセージの出力
	 */
	ERRLOG1(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_SYNTAX,
		"[sqlparser report]: %s.", errmsg);
	return;
}

/**
 * typeとvalue を持つdivy_sql_elemnode を生成して返却する.
 *
 * @param p_ctx divy_sql_parser_ctx * パーサコンテキスト
 * @param type divy_sql_nodetype
 * @param value char *
 * @return divy_sql_elemnode * 生成したノード
 */
static divy_sql_elemnode * _makeNode(divy_sql_parser_ctx *p_ctx,
					divy_sql_nodetype type, char *value)
{
	divy_sql_elemnode *node = apr_palloc(p_ctx->p, sizeof(divy_sql_elemnode));

	node->nodetype    = type;
	node->name        = apr_pstrdup(p_ctx->p, value);	/* 名前に割り当てる */
	node->next        = NULL;
	node->first_child = NULL;
	node->belong_nodetype = DIVY_NT_UNKNOWN;	/* 初期値 */
	node->pos         = p_ctx->sqlposition;

	return node;
}

/**
 * 式を表すノードを生成する
 *
 * @param p_ctx divy_sql_parser_ctx * パーサコンテキスト
 * @param type divy_sql_nodetype
 * @param op char * 演算子を表す文字列
 * @param lnode divy_sql_elemnode * 左辺値 (Lvalue)
 * @param rnode divy_sql_elemnode * 右辺値 (Rvalue)
 * @return divy_sql_elemnode * 生成したノード
 */
static divy_sql_elemnode * _makeExpr(divy_sql_parser_ctx *p_ctx,
					divy_sql_nodetype type,
					char *op,
					divy_sql_elemnode *lnode,
					divy_sql_elemnode *rnode)
{
	divy_sql_elemnode *node = apr_palloc(p_ctx->p, sizeof(divy_sql_elemnode));

	node->nodetype  = type;
	node->name      = apr_pstrdup(p_ctx->p, op);	/* 名前に割り当てる */
	node->next      = NULL;
	node->belong_nodetype = DIVY_NT_UNKNOWN;	/* 初期値 */

	node->first_child = lnode;
	if (node->first_child) {
		node->first_child->next = rnode;
	}
	else {
		/* op が単項演算子の場合 */
		node->first_child = rnode;
	}

	return node;
}

/**
 * l1 の末尾にl2を繋げる
 * (note)
 * 	l1がNULLならば、l2を返却します
 *	l2がNULLならば、l1をそのまま返却します
 *
 * @param l1 divy_sql_elemnode * 連結先SQLノード
 * @param l2 divy_sql_elemnode * 連結SQLノード
 * @return divy_sql_elemnode * 連結したSQLノード
 */
static divy_sql_elemnode * _appendExpr(divy_sql_elemnode *l1,
						divy_sql_elemnode *l2)
{
	divy_sql_elemnode *list;

	if (l1 == NULL) return l2;
	if (l2 == NULL) return l1;

	for (list = l1; list->next; list = list->next);
	list->next = l2;

	return l1;
}

/**
 * SELECT句が持つカラム数のカウンターをインクリメントする
 * (note)
 *	最初のSELECT句に現れるカラムの個数だけを数える
 *	(例)
 *		SELECT id,id2, (SELECT id3 FROM bbb
 *				WHERE id = (SELECT id4 FROM c)) as A FROM ...
 *		--> カウンタは３. id3,id4 は数えない
 */
static void _incrementSelectColCnt(divy_sql_parser_ctx *p_ctx)
{
	/* 最初に現れるカラムだけ対象 */
	if (p_ctx->select_id == 1) {
		p_ctx->select_colcnt++;

		/* 大きい方の値を持っておく */
		if (p_ctx->select_colcnt > p_ctx->max_select_colcnt) {
			p_ctx->max_select_colcnt = p_ctx->select_colcnt;
		}
	}
}

/**
 * SELECT句カラム数カウンタをクリアする
 */
static void _resetSelectColCnt(divy_sql_parser_ctx *p_ctx)
{
	p_ctx->select_colcnt = 0;
}

static void _resetMaxSelectColCnt(divy_sql_parser_ctx *p_ctx)
{
	p_ctx->max_select_colcnt = 0;
}

/**
 * lnode がop の左辺値として指定可能かどうかチェックする
 *
 * @param p apr_pool_t *
 * @param lnode divy_sql_elemnode * 左辺値
 * @return int (1: 指定できない / 0: 指定できる)
 */
static int _validate_lval(apr_pool_t *p, char *op, divy_sql_elemnode *lnode)
{
	if (lnode == NULL) return 0;

	/* RequiredSQL($$SS, $$SM)、名前付きバインド変数($$B) は
	   左辺値になれない */
	if (lnode->nodetype == DIVY_NT_NAMEDBIND ||
	    lnode->nodetype == DIVY_NT_S_REQUIREDSQL ||
	    lnode->nodetype == DIVY_NT_M_REQUIREDSQL) {

		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The variable \"%s\" was specified "
			"as left value operand of \"%s\" operator.",
			REPLACE_NULL(lnode->name), REPLACE_NULL(op));
		return 1;
	}

	return 0;
}

/**
 * 指定されたオペレータop でlnodeを左辺値にrnodeを右辺値とする
 * 式と表現してよいかどうかを判定する。
 *
 * (note) 作成可能な条件
 *	これはSQL文の仕様ではなく、TeamFileの画面表示仕様を満たす
 *	目的で実施するチェックです。
 *	左辺値がDBカラムまたは通常バインド変数の時のみ作成可能と
 *	判断します。
 *
 * @param op char * オペレータ
 * @param lnode divy_sql_elemnode * 左辺値
 * @param rnode divy_sql_elemnode * 右辺値
 * @return int (0: 作成できない / 1: 作成できる)
 */
static int _enable_making_expr(char *op, divy_sql_elemnode *lnode,
				divy_sql_elemnode *rnode)
{
	/* (note) op は予約された引数。今は使っていません */
	if (lnode == NULL || rnode == NULL) return 0;

	/* 左辺値がDBカラム、通常バインド変数ならOK */
	if (lnode->nodetype == DIVY_NT_COLUMN ||
	    lnode->nodetype == DIVY_NT_BIND) {
		return 1;
	}

	return 0;
}

