%{
/**
 * $Id$
 *
 * sqlscan.l
 *
 * TeamFile で使用するSQL文の字句解析を行う関数
 */
#include "httpd.h"
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_lib.h"

#include "tfr.h"
#include "mod_dav_tf.h"
#include "tf_sqlparser.h"
#include "sqlparser.h"
#include "util_common.h"

/*--------------------------------------------------------------
  Define special values
  --------------------------------------------------------------*/
/*
 * (note) static 変数が存在する訳
 * flex が出力するコードには、apr_pool_t などの機構を全く
 * 利用できないため、malloc やfreeを使用しなければならない。
 * また、割り当てたメモリ領域を何処かに保持しておく必要すら
 * 生じている。
 * これが、static 変数の存在する理由である。
 */

/*
 * バッファに対するハンドル
 */
static YY_BUFFER_STATE scanbufhandle;

/*--------------------------------------------------------------
  Define structs
  --------------------------------------------------------------*/
/**
 * 予約されたあるSQLキーワードを表す構造体
 */
struct __sqlkeyword {
	char *name;	/* キーワードの名前 */
	int token;	/* トークン */

#define FLG_NONE	0x00	/* 特別なフラグを持たない             */
#define FLG_RESERVE	0x01	/* キーワードとして予約されている     */
#define FLG_NOTUSING	0x02	/* キーワード構文解析に不必要         */
#define FLG_LOOKAHEAD	0x04	/* 先読みのさらに先読みトークンが必要 */
#define FLG_COMBTOKEN	0x08	/* 組合せトークンが必要               */
	int flag;	/* 汎用フラグ */
};
typedef struct __sqlkeyword	sqlkeyword;

/* p_ctx に記録されたsqlkeywordをNULLクリアする */
#define CLEAR_KEYWORD(p_ctx)	(p_ctx->keyword = NULL)

/**
 * "コンビネーショントークン" を表す構造体
 */
struct __combtoken {
	int combination;	/* コンビネーショントークンの値 */
	int token1;		/* コンビネーショントークンの１個目 */
	int token2;		/* コンビネーショントークンの２個目 */
};
typedef struct __combtoken	combtoken;

/**
 * "ダブル先読みトークン" を表す構造体
 */
struct __doubleaheadtoken {
	int token;		/* ダブル先読みトークンの値  */
	int token1;		/* 先読みトークン            */
	int token2;		/* token1 の更に先のトークン */
};
typedef struct __doubleaheadtoken	doubleaheadtoken;

/*------------------------------------------------------------------------------
  Declare functions
  lex が用意しているデフォルト関数を置き換えています。(apache対応)
  引数の数は、yaccに依存して決まってきてしまします。
  ----------------------------------------------------------------------------*/
#define YY_USE_PROTOS

/* YY_DECL の置き換え */
#undef  YY_DECL
#define YY_DECL int yylex (YYSTYPE *p_yylval, divy_sql_parser_ctx *p_ctx)


/* YY_FATAL_ERROR の置き換え(exit してしまうので。。) */
#undef  YY_FATAL_ERROR
#define YY_FATAL_ERROR(msg) tf_yy_fatal_error( msg )
static void tf_yy_fatal_error ( yyconst char msg[] );

/* tf_inter_yylex 用デバッグログマクロ */
//#define YYLEX_DEBUG

#ifdef YYLEX_DEBUG
#define _TOKEN_DEBUG0(p,f) (ERRLOG0(p, APLOG_DEBUG, \
				DIVY_FST_INFO + DIVY_SST_DEBUG, "  "f))
#define _TOKEN_DEBUG2(p,i,s,f) (ERRLOG2(p, APLOG_DEBUG, \
				DIVY_FST_INFO + DIVY_SST_DEBUG, "  (%d)  [%s]"f,i,s))
#define _TOKEN_DEBUG3(p,i,s,d,f) (ERRLOG3(p, APLOG_DEBUG, \
				DIVY_FST_INFO + DIVY_SST_DEBUG, "  (%d)  [%s]<%d>"f,i,s,d))
#else
#define _TOKEN_DEBUG0(p,f)
#define _TOKEN_DEBUG2(p,i,s,f)
#define _TOKEN_DEBUG3(p,i,s,d,f)
#endif	/* YYLEX_DEBUG */

/*--------------------------------------------------------------
  Declare private functions
  --------------------------------------------------------------*/
static sqlkeyword * _lookup_sqlkeyword(apr_pool_t *p, char *text);
static sqlkeyword * _lookup_compound_operator(apr_pool_t *p, char *text);
static sqlkeyword * _binarysearch(apr_pool_t *p, char *text,
					int len, const sqlkeyword tbl[]);
static int _replace_reqsqlvalue(divy_sql_parser_ctx *p_ctx, int type, char *text);
static int _check_token_combination(int token1, int token2);
static int _check_token_lookdoubleahead(int token1, int token2);

/*--------------------------------------------------------------
  Define values
  --------------------------------------------------------------*/
/**
 * 予約されたSQL 識別子を定義する
 * (note)
 *	識別子はPostgreSQLとOracleから抽出したものです
 * (note)
 *	このテーブルの検索にはバイナリサーチを利用するため、
 *	name は必ずASCIIの昇順に並んでいなければなりません。
 */
static const sqlkeyword SqlKeyWordTbl[] = {
	{"abort",	ABORT_TRANS,	FLG_RESERVE},
	{"abs",		ABS,		FLG_RESERVE},	/* Oracle */
	{"absolute",	ABSOLUTE,	FLG_RESERVE},
	{"access",	ACCESS,		FLG_RESERVE},
	{"acos",	ACOS,		FLG_RESERVE},	/* Oracle */
	{"action",	ACTION,		FLG_RESERVE},
	{"add",		ADD,		FLG_RESERVE},
	{"add_months",	ADDMONTHS,	FLG_RESERVE},	/* Oracle */
	{"after",	AFTER,		FLG_RESERVE},
	{"aggregate",	AGGREGATE,	FLG_RESERVE},
	{"all",		ALL,		FLG_RESERVE},
	{"alter",	ALTER,		FLG_RESERVE},
	{"analyse",	ANALYSE,	FLG_RESERVE},	/* PostgreSQL */
	{"analyze",	ANALYZE,	FLG_RESERVE},	/* PostgreSQL */
	{"and",		AND,		FLG_RESERVE},
	{"any",		ANY,		FLG_RESERVE},
	{"as",		AS,		FLG_RESERVE},
	{"asc",		ASC,		FLG_RESERVE},
	{"assertion",	ASSERTION,	FLG_RESERVE},
	{"assignment",	ASSIGNMENT,	FLG_RESERVE},
	{"at",		AT,		FLG_RESERVE},
	{"authorization",AUTHORIZATION,	FLG_RESERVE},
	{"avg",		AVG,		FLG_RESERVE},	/* SQL92 */
	{"backward",	BACKWARD,	FLG_RESERVE},
	{"before",	BEFORE,		FLG_RESERVE},
	{"begin",	BEGIN_TRANS,	FLG_RESERVE},
	{"between",	BETWEEN,	FLG_RESERVE},
	{"bigint",	BIGINT,		FLG_RESERVE},	/* PostgreSQL */
	{"binary",	BINARY,		FLG_RESERVE},
	{"bit",		BIT,		FLG_RESERVE},
	{"bit_length",	BIT_LENGTH,	FLG_RESERVE},
	{"block",	BLOCK,		FLG_RESERVE},	/* Oracle */
	{"boolean",	BOOLEAN,	FLG_RESERVE},
	{"both",	BOTH,		FLG_RESERVE},
	{"by",		BY,		FLG_RESERVE},
	{"cache",	CACHE,		FLG_RESERVE},
	{"called",	CALLED,		FLG_RESERVE},
	{"cascade",	CASCADE,	FLG_RESERVE},
	{"case",	CASE,		FLG_RESERVE},
	{"cast",	CAST,		FLG_RESERVE},
	{"chain",	CHAIN,		FLG_RESERVE},
	{"char",	CHAR_P,		FLG_RESERVE},
	{"character",	CHARACTER,	FLG_RESERVE},
	{"characteristics",  CHARACTERISTICS,  FLG_RESERVE},
	{"character_length", CHARACTER_LENGTH, FLG_RESERVE},
	{"check",	CHECK,		FLG_RESERVE},
	{"checkpoint",	CHECKPOINT,	FLG_RESERVE},
	{"class",	CLASS,		FLG_RESERVE},
	{"close",	CLOSE,		FLG_RESERVE},
	{"cluster",	CLUSTER,	FLG_RESERVE},
	{"coalesce",	COALESCE,	FLG_RESERVE},
	{"collate",	COLLATE,	FLG_RESERVE},
	{"column",	COLUMN,		FLG_RESERVE},
	{"comment",	COMMENT,	FLG_RESERVE},
	{"commit",	COMMIT,		FLG_RESERVE},
	{"committed",	COMMITTED,	FLG_RESERVE},
	{"connect",	CONNECT,	FLG_RESERVE},	/* Oracle用 */
	{"constraint",	CONSTRAINT,	FLG_RESERVE},
	{"constraints",	CONSTRAINTS,	FLG_RESERVE},
	{"conversion",	CONVERSION_P,	FLG_RESERVE},
	{"convert",	CONVERT,	FLG_RESERVE},
	{"copy",	COPY,		FLG_RESERVE},
	{"count",	COUNT,		FLG_RESERVE},
	{"create",	CREATE,		FLG_RESERVE},
	{"createdb",	CREATEDB,	FLG_RESERVE},
	{"createuser",	CREATEUSER,	FLG_RESERVE},
	{"cross",	CROSS,		FLG_RESERVE},
	{"cs",		CS,		FLG_RESERVE},	/* DB2 */
	{"cube",	CUBE,		FLG_RESERVE},	/* Oracle */
	{"current_date",CURRENT_DATE,	FLG_RESERVE},
	{"current_time",CURRENT_TIME,	FLG_RESERVE},
	{"current_timestamp", CURRENT_TIMESTAMP, FLG_RESERVE},
	{"current_user",CURRENT_USER,	FLG_RESERVE},
	{"cursor",	CURSOR,		FLG_RESERVE},
	{"cycle",	CYCLE,		FLG_RESERVE},
	{"database",	DATABASE,	FLG_RESERVE},
	{"date",	DATE_P,		FLG_RESERVE},
	{"day",		DAY_P,		FLG_RESERVE},
	{"deallocate",	DEALLOCATE,	FLG_RESERVE},
	{"dec",		DEC,		FLG_RESERVE},
	{"decimal",	DECIMAL,	FLG_RESERVE},
	{"declare",	DECLARE,	FLG_RESERVE},
	{"default",	DEFAULT,	FLG_RESERVE},
	{"deferrable",	DEFERRABLE,	FLG_RESERVE},
	{"deferred",	DEFERRED,	FLG_RESERVE},
	{"definer",	DEFINER,	FLG_RESERVE},
	{"delete",	DELETE_P,	FLG_RESERVE},
	{"delimiter",	DELIMITER,	FLG_RESERVE},
	{"delimiters",	DELIMITERS,	FLG_RESERVE},
	{"desc",	DESC,		FLG_RESERVE},
	{"distinct",	DISTINCT,	FLG_RESERVE},
	{"do",		DO,		FLG_RESERVE},
	{"domain",	DOMAIN_P,	FLG_RESERVE},
	{"double",	DOUBLE,		FLG_RESERVE},
	{"drop",	DROP,		FLG_RESERVE},
	{"each",	EACH,		FLG_RESERVE},
	{"else",	ELSE,		FLG_RESERVE},
	{"encoding",	ENCODING,	FLG_RESERVE},
	{"encrypted",	ENCRYPTED,	FLG_RESERVE},
	{"end",		END_TRANS,	FLG_RESERVE},
	{"escape",	ESCAPE,		FLG_RESERVE},
	{"except",	EXCEPT,		FLG_RESERVE},
	{"exclusive",	EXCLUSIVE,	FLG_RESERVE},
	{"execute",	EXECUTE,	FLG_RESERVE},
	{"exists",	EXISTS,		FLG_RESERVE},
	{"explain",	EXPLAIN,	FLG_RESERVE},
	{"external",	EXTERNAL,	FLG_RESERVE},
	{"extract",	EXTRACT,	FLG_RESERVE},
	{"false",	FALSE_P,	FLG_RESERVE},
	{"fetch",	FETCH,		FLG_RESERVE | FLG_COMBTOKEN},
	{"first",	FIRST,		FLG_RESERVE},	/* Oracle */
	{"float",	FLOAT_P,	FLG_RESERVE},
	{"for",		FOR,		FLG_RESERVE},
	{"force",	FORCE,		FLG_RESERVE},
	{"foreign",	FOREIGN,	FLG_RESERVE},
	{"forward",	FORWARD,	FLG_RESERVE},
	{"freeze",	FREEZE,		FLG_RESERVE},
	{"from",	FROM,		FLG_RESERVE},
	{"full",	FULL,		FLG_RESERVE},
	{"function",	FUNCTION,	FLG_RESERVE},
	{"get",		GET,		FLG_RESERVE},
	{"global",	GLOBAL,		FLG_RESERVE},
	{"grant",	GRANT,		FLG_RESERVE},
	{"group",	GROUP_P,	FLG_RESERVE},
	{"grouping",	GROUPING,	FLG_RESERVE},	/* Oracle */
	{"handler",	HANDLER,	FLG_RESERVE},
	{"having",	HAVING,		FLG_RESERVE},
	{"hour",	HOUR_P,		FLG_RESERVE},
	{"ilike",	ILIKE,		FLG_RESERVE},
	{"immediate",	IMMEDIATE,	FLG_RESERVE},
	{"immutable",	IMMUTABLE,	FLG_RESERVE},
	{"implicit",	IMPLICIT_P,	FLG_RESERVE},
	{"in",		IN_P,		FLG_RESERVE},
	{"increment",	INCREMENT,	FLG_RESERVE},
	{"index",	INDEX,		FLG_RESERVE},
	{"inherits",	INHERITS,	FLG_RESERVE},
	{"initially",	INITIALLY,	FLG_RESERVE},
	{"inner",	INNER_P,	FLG_RESERVE},
	{"inout",	INOUT,		FLG_RESERVE},
	{"input",	INPUT,		FLG_RESERVE},
	{"insensitive",	INSENSITIVE,	FLG_RESERVE},
	{"instead",	INSTEAD,	FLG_RESERVE},
	{"int",		INT,		FLG_RESERVE},
	{"integer",	INTEGER,	FLG_RESERVE},
	{"intersect",	INTERSECT,	FLG_RESERVE},
	{"interval",	INTERVAL,	FLG_RESERVE},
	{"into",	INTO,		FLG_RESERVE},
	{"invoker",	INVOKER,	FLG_RESERVE},
	{"is",		IS,		FLG_RESERVE},
	{"isnull",	ISNULL,		FLG_RESERVE},
	{"isolation",	ISOLATION,	FLG_RESERVE},
	{"join",	JOIN,		FLG_RESERVE},
	{"key",		KEY,		FLG_RESERVE},
	{"lancompiler",	LANCOMPILER,	FLG_RESERVE},
	{"language",	LANGUAGE,	FLG_RESERVE},
	{"last",	LAST,		FLG_RESERVE},	/* Oracle */
	{"leading",	LEADING,	FLG_RESERVE},
	{"left",	LEFT,		FLG_RESERVE},
	{"level",	LEVEL,		FLG_RESERVE},
	{"like",	LIKE,		FLG_RESERVE},
	{"like2",	LIKE,		FLG_RESERVE},	/* Oracle, LIKEと同じとする */
	{"like4",	LIKE,		FLG_RESERVE},	/* Oracle, LIKEと同じとする */
	{"likec",	LIKE,		FLG_RESERVE},	/* Oracle, LIKEと同じとする */
	{"limit",	LIMIT,		FLG_RESERVE},	/* PostgreSQL */
	{"listen",	LISTEN,		FLG_RESERVE},
	{"load",	LOAD,		FLG_RESERVE},
	{"local",	LOCAL,		FLG_RESERVE},
	{"localtime",	LOCALTIME,	FLG_RESERVE},
	{"localtimestamp", LOCALTIMESTAMP, FLG_RESERVE},
	{"location",	LOCATION,	FLG_RESERVE},
	{"lock",	LOCK_P,		FLG_RESERVE},
	{"lower",	LOWER,		FLG_RESERVE},	/* SQL92 */
	{"match",	MATCH,		FLG_RESERVE},
	{"max",		MAX,		FLG_RESERVE},
	{"maxvalue",	MAXVALUE,	FLG_RESERVE},
	{"min",		MIN,		FLG_RESERVE},
	{"minus",	MINUS_P,	FLG_RESERVE},
	{"minute",	MINUTE_P,	FLG_RESERVE},
	{"minvalue",	MINVALUE,	FLG_RESERVE},
	{"mode",	MODE,		FLG_RESERVE},
	{"module",	MODULE,		FLG_RESERVE},	/* SQL92 */
	{"month",	MONTH_P,	FLG_RESERVE},
	{"move",	MOVE,		FLG_RESERVE},
	{"names",	NAMES,		FLG_RESERVE},
	{"national",	NATIONAL,	FLG_RESERVE},
	{"natural",	NATURAL,	FLG_RESERVE},
	{"nchar",	NCHAR,		FLG_RESERVE},
	{"new",		NEW,		FLG_RESERVE},
	{"next",	NEXT,		FLG_RESERVE},
	{"no",		NO,		FLG_RESERVE},
	{"nocreatedb",	NOCREATEDB,	FLG_RESERVE},
	{"nocreateuser",NOCREATEUSER,	FLG_RESERVE},
	{"none",	NONE,		FLG_RESERVE},
	{"not",		NOT,		FLG_RESERVE},
	{"nothing",	NOTHING,	FLG_RESERVE},
	{"notify",	NOTIFY,		FLG_RESERVE},
	{"notnull",	NOTNULL,	FLG_RESERVE},
	{"nowait",	NOWAIT,		FLG_RESERVE},	/* Oracle */
	{"null",	NULL_P,		FLG_RESERVE},
	{"nullif",	NULLIF,		FLG_RESERVE},
	{"nulls",	NULLS,		FLG_RESERVE},	/* Oracle */
	{"numeric",	NUMERIC,	FLG_RESERVE},
	{"octet_length",OCTET_LENGTH,	FLG_RESERVE},	/* SQL92 */
	{"of",		OF,		FLG_RESERVE},
	{"off",		OFF,		FLG_RESERVE},
	{"offset",	OFFSET,		FLG_RESERVE},	/* PostgreSQL */
	{"oids",	OIDS,		FLG_RESERVE},	/* PostgreSQL */
	{"old",		OLD,		FLG_RESERVE},	/* PostgreSQL */
	{"on",		ON,		FLG_RESERVE},
	{"only",	ONLY,		FLG_RESERVE},
	{"operator",	OPERATOR,	FLG_RESERVE},
	{"option",	OPTION,		FLG_RESERVE},
	{"or",		OR,		FLG_RESERVE},
	{"order",	ORDER,		FLG_RESERVE},
	{"out",		OUT_P,		FLG_RESERVE},
	{"outer",	OUTER_P,	FLG_RESERVE},
	{"overlaps",	OVERLAPS,	FLG_RESERVE},
	{"overlay",	OVERLAY,	FLG_RESERVE},
	{"owner",	OWNER,		FLG_RESERVE},
	{"partial",	PARTIAL,	FLG_RESERVE},
	{"partition",	PARTITION,	FLG_RESERVE | FLG_LOOKAHEAD},	/* Oracle */
	{"password",	PASSWORD,	FLG_RESERVE},
	{"path",	PATH_P,		FLG_RESERVE},
	{"pendant",	PENDANT,	FLG_RESERVE},
	{"placing",	PLACING,	FLG_RESERVE},
	{"position",	POSITION,	FLG_RESERVE},
	{"precision",	PRECISION,	FLG_RESERVE},
	{"prepare",	PREPARE,	FLG_RESERVE},
	{"primary",	PRIMARY,	FLG_RESERVE},
	{"prior",	PRIOR,		FLG_RESERVE},
	{"privileges",	PRIVILEGES,	FLG_RESERVE},
	{"procedural",	PROCEDURAL,	FLG_RESERVE},
	{"procedure",	PROCEDURE,	FLG_RESERVE},
	{"read",	READ,		FLG_RESERVE},
	{"real",	REAL,		FLG_RESERVE},
	{"recheck",	RECHECK,	FLG_RESERVE},
	{"references",	REFERENCES,	FLG_RESERVE},
	{"reindex",	REINDEX,	FLG_RESERVE},
	{"relative",	RELATIVE,	FLG_RESERVE},
	{"rename",	RENAME,		FLG_RESERVE},
	{"replace",	REPLACE,	FLG_RESERVE},
	{"reset",	RESET,		FLG_RESERVE},
	{"restrict",	RESTRICT,	FLG_RESERVE},
	{"returns",	RETURNS,	FLG_RESERVE},
	{"revoke",	REVOKE,		FLG_RESERVE},
	{"right",	RIGHT,		FLG_RESERVE},
	{"rollback",	ROLLBACK,	FLG_RESERVE},
	{"rollup",	ROLLUP,		FLG_RESERVE},	/* Oracle */
	{"row",		ROW,		FLG_RESERVE},
	{"rowid",	ROWID,		FLG_RESERVE},	/* Oracle */
	{"rownum",	ROWNUM,		FLG_RESERVE},	/* Oracle */
	{"rows",	ROWS,		FLG_RESERVE},	/* DB2 */
	{"rr",		RR,		FLG_RESERVE},	/* DB2 */
	{"rs",		RS,		FLG_RESERVE},	/* DB2 */
	{"rule",	RULE,		FLG_RESERVE},
	{"sample",	SAMPLE,		FLG_RESERVE | FLG_LOOKAHEAD | FLG_COMBTOKEN},	/* Oracle */
	{"schema",	SCHEMA,		FLG_RESERVE},
	{"scn",		SCN,		FLG_RESERVE},	/* Oracle */
	{"scroll",	SCROLL,		FLG_RESERVE},
	{"second",	SECOND_P,	FLG_RESERVE},
	{"security",	SECURITY,	FLG_RESERVE},
	{"select",	SELECT,		FLG_RESERVE},
	{"sequence",	SEQUENCE,	FLG_RESERVE},
	{"serializable",SERIALIZABLE,	FLG_RESERVE},
	{"session",	SESSION,	FLG_RESERVE},
	{"session_user",SESSION_USER,	FLG_RESERVE},
	{"set",		SET,		FLG_RESERVE},
	{"setof",	SETOF,		FLG_RESERVE},
	{"sets",	SETS,		FLG_RESERVE},	/* Oracle */
	{"share",	SHARE,		FLG_RESERVE},
	{"show",	SHOW,		FLG_RESERVE},
	{"siblings",	SIBLINGS,	FLG_RESERVE},	/* Oracle */
	{"similar",	SIMILAR,	FLG_RESERVE},
	{"simple",	SIMPLE,		FLG_RESERVE},
	{"smallint",	SMALLINT,	FLG_RESERVE},
	{"some",	SOME,		FLG_RESERVE},
	{"stable",	STABLE,		FLG_RESERVE},
	{"start",	START,		FLG_RESERVE},
	{"statement",	STATEMENT,	FLG_RESERVE},
	{"statistics",	STATISTICS,	FLG_RESERVE},
	{"stdin",	STDIN,		FLG_RESERVE},
	{"stdout",	STDOUT,		FLG_RESERVE},
	{"storage",	STORAGE,	FLG_RESERVE},
	{"strict",	STRICT,		FLG_RESERVE},
	{"subpartition",SUBPARTITION,	FLG_RESERVE | FLG_LOOKAHEAD},	/* Oracle */
	{"substr",	SUBSTRING,	FLG_RESERVE},	/* DB2 (substring にマッピングします) */
	{"substring",	SUBSTRING,	FLG_RESERVE},
	{"sum",		SUM,		FLG_RESERVE},
	{"sysid",	SYSID,		FLG_RESERVE},
	{"table",	TABLE,		FLG_RESERVE},
	{"temp",	TEMP,		FLG_RESERVE},
	{"template",	TEMPLATE,	FLG_RESERVE},
	{"temporary",	TEMPORARY,	FLG_RESERVE},
	{"then",	THEN,		FLG_RESERVE},
	{"time",	TIME,		FLG_RESERVE},
	{"timestamp",	TIMESTAMP,	FLG_RESERVE},
	{"to",		TO,		FLG_RESERVE},
	{"toast",	TOAST,		FLG_RESERVE},
	{"trailing",	TRAILING,	FLG_RESERVE},
	{"transaction", TRANSACTION,	FLG_RESERVE},
	{"translate",	TRANSLATE,	FLG_RESERVE},
	{"treat",	TREAT,		FLG_RESERVE},
	{"trigger",	TRIGGER,	FLG_RESERVE},
	{"trim",	TRIM,		FLG_RESERVE},
	{"true",	TRUE_P,		FLG_RESERVE},
	{"truncate",	TRUNCATE,	FLG_RESERVE},
	{"trusted",	TRUSTED,	FLG_RESERVE},
	{"type",	TYPE_P,		FLG_RESERVE},
	{"unencrypted",	UNENCRYPTED,	FLG_RESERVE},
	{"union",	UNION,		FLG_RESERVE | FLG_COMBTOKEN},
	{"unique",	UNIQUE,		FLG_RESERVE},
	{"unknown",	UNKNOWN,	FLG_RESERVE},
	{"unlisten",	UNLISTEN,	FLG_RESERVE},
	{"until",	UNTIL,		FLG_RESERVE},
	{"update",	UPDATE,		FLG_RESERVE},
	{"upper",	UPPER,		FLG_RESERVE},	/* SQL92 */
	{"ur",		UR,		FLG_RESERVE},	/* DB2 */
	{"usage",	USAGE,		FLG_RESERVE},
	{"user",	USER,		FLG_RESERVE},
	{"using",	USING,		FLG_RESERVE},
	{"vacuum",	VACUUM,		FLG_RESERVE},	/* PostgreSQL */
	{"valid",	VALID,		FLG_RESERVE},
	{"validator",	VALIDATOR,	FLG_RESERVE},
	{"values",	VALUES,		FLG_RESERVE},
	{"varchar",	VARCHAR,	FLG_RESERVE},
	{"varchar2",	VARCHAR2,	FLG_RESERVE},	/* Oracle */
	{"varying",	VARYING,	FLG_RESERVE},
	{"verbose",	VERBOSE,	FLG_RESERVE},
	{"version",	VERSION,	FLG_RESERVE},
	{"view",	VIEW,		FLG_RESERVE},
	{"volatile",	VOLATILE,	FLG_RESERVE},
	{"wait",	WAIT,		FLG_RESERVE},	/* Oracle */
	{"when",	WHEN,		FLG_RESERVE},
	{"where",	WHERE,		FLG_RESERVE},
	{"with",	WITH,		FLG_RESERVE | FLG_COMBTOKEN},
	{"without",	WITHOUT,	FLG_RESERVE},
	{"work",	WORK,		FLG_RESERVE},
	{"write",	WRITE,		FLG_RESERVE},
	{"year",	YEAR_P,		FLG_RESERVE},
	{"zone",	ZONE,		FLG_RESERVE},
};
#define SQLKEYWORDTBL_LEN	sizeof(SqlKeyWordTbl) / sizeof(sqlkeyword)

/**
 * 認識している複合記号演算子の集合を定義する構造体の配列
 * (note)
 *	第１メンバの並び順はASCIIの昇順になっています。
 */
static const sqlkeyword CompoundOpTbl[] = {
	{"!<",	OP_LT,		FLG_RESERVE},	/* SQL Server, Sybase */
	{"!=",	OP_NOTEQ,	FLG_RESERVE},
	{"!>",	OP_GT,		FLG_RESERVE},	/* SQL Server, Sybase */
	{"^=",	OP_NOTEQ,	FLG_RESERVE},	/* Oracle */
	{"||",	OP_CONCAT,	FLG_RESERVE},	/* Oracle */
	{"<=",	OP_GT,		FLG_RESERVE},
	{"<>",	OP_NOTEQ,	FLG_RESERVE},
	{">=",	OP_LT,		FLG_RESERVE}
};
#define COMPOUNDOPTBL_LEN		sizeof(CompoundOpTbl) / sizeof(sqlkeyword)

/**
 * コンビネーショントークンの集合
 */
static const combtoken CombTokenTbl[] = {
	{UNION_JOIN,	UNION, JOIN},
	{FETCH_FIRST,	FETCH, FIRST},
	{WITH_RR,	WITH, RR},
	{WITH_RS,	WITH, RS},
	{WITH_CS,	WITH, CS},
	{WITH_UR,	WITH, UR},
	{SAMPLE_BLOCK,	SAMPLE, BLOCK},
};
#define COMBTOKENTBL_LEN	sizeof(CombTokenTbl) / sizeof(combtoken)

/**
 * ダブル先読みトークンの集合
 */
static const doubleaheadtoken DoubleAheadTokenTbl[] = {
	{SAMPLE_DA,		SAMPLE, '('},
	{PARTITION_DA,		PARTITION, '('},
	{SUBPARTITION_DA,	SUBPARTITION_DA, '('}
};
#define DOUBLEAHEADTOKENTBL_LEN	sizeof(DoubleAheadTokenTbl) / sizeof(doubleaheadtoken)

%}

/* 8ビットコードを有効にする */
%option 8bit

/* yywrap という関数を無効にする */
%option noyywrap

/* unput という関数を無効にする */
%option nounput

/* "対話的"ではないパーサの作成を指示する */
%option never-interactive

/* プレフィックスを"tf_yy" に変更する */
%option prefix="tf_yy"

/*
 * 開始条件(排他的開始条件)の設定
 *
 * <xrp> RequiredSQL、名前付きバインド変数の置き換え処理モード
 * <xq>  シングルクオート文字
 * <xd>  ダブルクオート文字
 */
%x xrp
%x xq
%x xd
%x xc

/*
 * シングルクオートと囲まれた文字列を認識するパターン
 */
quote			'
xqdouble		{quote}{quote}
xqinside		[^']+
xqstart			{quote}
xqstop			{quote}

/*
 * ダブルクオートと囲まれた文字列を認識するパターン
 */
dquote			\"
xddouble		{dquote}{dquote}
xdinside		[^"]+
xdstart			{dquote}
xdstop			{dquote}

/*
 * PostgreSQLのコメントやOracleのヒントを認識するパターン
 */
xcstart			\/\*{op_chars}*
xcstop			\*+\/
xcinside		[^*/]+

/**
 * 数値, 文字列, 数値 & 文字列
 * (note)
 *	ドット(.) は、SQL構文解析において、一般的には一般的な文字に
 *	分類されることはありません。SQLにおいて意味を持つからです。
 *	ですが、我々は、エンドユーザに示すカラム名や関数名にドットで
 *	区切られた識別子として示した方が好ましく、またドットを解釈して
 *	何かをする訳ではないので、一般的な文字としてドットを扱うことに
 *	しています。
 * (note)
 *	\200-\377 は8進数であり、128-255 (nonascii)のこと。
 */
digit                   [0-9]
letter                  [\200-\377_A-Za-z.]
letter_or_digit		[\200-\377_A-Za-z.0-9]

/**
 * 識別名称
 * SQLの予約語も、そうでないもの(カラム名など)もここに分類されます。
 * % は切り離されてもTeamFileでは困ってしまうので、そのままにしています。
 */
identifier              {letter}{letter_or_digit}*[\%]*

/**
 * 記号演算子を識別する正規表現.
 *
 *
 * self     : 単体で１つの記号演算子として機能する演算子のトークン
 * operator : ２個以上の記号の組み合わせで、機能する演算子のトークン
 *
 * (note)
 *   $?% は名前付きバインド変数やRequiredSQLと干渉するので、operatorからは
 *   外してあります。元々TeamFile自身がSQLの正規表現をサポートする必要は
 *   全くないのでこれは正当な行為です。
 *
 * (note) self_ignore について
 *	我々の仕様を満たすパーサを作成するにあたり、解析が不要であった
 *	演算子のトークン。本来ならば、self に分類されるもの。
 */
self                    [,()\<\>\=\*\+\-\/\%\^$]
self_ignore             [\[\].;\:]
op_chars                [\~\!\@\#\^\&\|\`\+\-\*\/\<\>\=]
operator                {op_chars}+
op_ora_outerjoin	(\+)

/* 数値 */
integer                 {digit}+

/* 小数点を含む数値 */
decimal                 (({digit}*\.{digit}+)|({digit}+\.{digit}*))

/* 指数表示を含む数値 */
real                    ((({digit}*\.{digit}+)|({digit}+\.{digit}*)|({digit}+))([Ee][-+]?{digit}+))

/*
 * DB 上でストアードプロシージャに引数を渡す時に使用される変数。
 * ですが、TeamFileでは用なしなので別途捨てるために定義しています。
 */
param                   \${integer}

/*
 * Oracle dblinkを表す識別子
 * "@"は演算子として予約されているため、特別にマッチング構文を用意しました。
 */
dblink			\@{letter}{letter_or_digit}*

/* Oracle の$ 付きView */
orasys_viewname		{letter}+\${letter_or_digit}+

space                   [ \t\n\r\f]
horiz_space             [ \t\f]
newline                 [\n\r]
non_newline             [^\n\r]
comment                 ("--"{non_newline}*)
whitespace              ({space}+|{comment})

horiz_whitespace        ({horiz_space}|{comment})
whitespace_with_newline ({horiz_whitespace}*{newline}{whitespace}*)

/**
 * bind        : バインド変数
 * namedbind   : 名前付きバインド変数
 * requiredsql : RequiredSQL
 */
bind			[\?]
namedbind		\$\$B{letter_or_digit}+
s_requiredsql		\$\$SS{letter_or_digit}+
m_requiredsql		\$\$SM{letter_or_digit}+

/**
 * その他分類されなかったもの
 */
other                   .

/**
 * RequiredSQL、名前付きバインド変数置き換えモード
 *
 * xrpmatch: RequiredSQL, 名前付きバインド変数以外にマッチする
 * xrpmatch だけでは曖昧ですので、namedbind、s_requiredsql、
 * m_requiredsqlなどをマッチ条件に含めています。
 */
xrpmatch		[^$$]|[^B]

%%
	/* 特殊な開始条件を呼び出した呼び出し元の開始条件を表す */
	int xstart_caller = INITIAL;
	int retcd = 0;
	sqlkeyword *keyword = NULL;

	/* 作業用に使用するプール */
	apr_pool_t *p     = p_ctx->p;

	/* 作業用バッファ */
	divy_sbuf *sbuf = NULL;
	divy_sbuf_create(p, &sbuf, 256);

	/* キーワードのクリア */
	CLEAR_KEYWORD(p_ctx);

{whitespace}    { /* ignore */ }

<INITIAL,xrp>{xqstart}	{
			CLEAR_KEYWORD(p_ctx);

			/* シングルクオート文字が現われた場合 */
			xstart_caller = YYSTATE;
			BEGIN(xq);

			/* 置換モードの時には、new_sql_bufバッファに追加 */
			if (xstart_caller == xrp) {
				divy_sbuf_append(p_ctx->new_sql_buf, yytext);
			}
			else {
				divy_sbuf_append(sbuf, yytext);
			}
		}
<xq>{xqinside}	{
			CLEAR_KEYWORD(p_ctx);

			if (xstart_caller == xrp) {
				divy_sbuf_append(p_ctx->new_sql_buf, yytext);
			}
			else {
				divy_sbuf_append(sbuf, yytext);
			}
		}
<xq>{xqstop}	{
			CLEAR_KEYWORD(p_ctx);

			/* 呼び出し元に戻る */
			BEGIN(xstart_caller);

			if (xstart_caller == xrp) {
				divy_sbuf_append(p_ctx->new_sql_buf, yytext);
			}
			else {
				divy_sbuf_append(sbuf, yytext);
				p_yylval->str = apr_pstrdup(p, divy_sbuf_tostring(sbuf));

				return SCONST;
			}
		}
<xq><<EOF>>	{
			CLEAR_KEYWORD(p_ctx);

			/* xq モードを解除して処理を終わらせてしまう */
			BEGIN(INITIAL);
			return EOF_SCONST;
		}

<INITIAL,xrp>{xdstart}	{
			CLEAR_KEYWORD(p_ctx);

			/* ダブルクオート文字が現われた場合 */
			xstart_caller = YYSTATE;
			BEGIN(xd);

			/* 置換モードの時には、new_sql_bufバッファに追加 */
			if (xstart_caller == xrp) {
				divy_sbuf_append(p_ctx->new_sql_buf, yytext);
			}
			else {
				divy_sbuf_append(sbuf, yytext);
			}
		}
<xd>{xdinside}	{
			CLEAR_KEYWORD(p_ctx);

			if (xstart_caller == xrp) {
				divy_sbuf_append(p_ctx->new_sql_buf, yytext);
			}
			else {
				divy_sbuf_append(sbuf, yytext);
			}
		}
<xd>{xdstop}	{
			CLEAR_KEYWORD(p_ctx);

			/* 呼び出し元に戻る */
			BEGIN(xstart_caller);

			if (xstart_caller == xrp) {
				divy_sbuf_append(p_ctx->new_sql_buf, yytext);
			}
			else {
				divy_sbuf_append(sbuf, yytext);
				p_yylval->str = apr_pstrdup(p, divy_sbuf_tostring(sbuf));

				return QT_IDENT;
			}
		}
<xd><<EOF>>	{
			CLEAR_KEYWORD(p_ctx);

			/* xd モードを解除して処理を終わらせてしまう */
			BEGIN(INITIAL);
			return EOF_QT_IDENT;
		}
<INITIAL,xrp>{xcstart}	{
			CLEAR_KEYWORD(p_ctx);

			/* コメント開始文字が現われた場合 */
			xstart_caller = YYSTATE;
			BEGIN(xc);

			/* xrp モード以外ではコメントは構文解析に不要なので無視 */
			if (xstart_caller == xrp) {
				/* new_sql_bufバッファに追加 */
				divy_sbuf_append(p_ctx->new_sql_buf, yytext);
			}
		}
<xc>{xcinside}	{
			CLEAR_KEYWORD(p_ctx);

			/* xpr モード以外ではコメントは構文解析に不要なので無視 */
			if (xstart_caller == xrp) {
				divy_sbuf_append(p_ctx->new_sql_buf, yytext);
			}
		}
<xc>{xcstop}	{
			CLEAR_KEYWORD(p_ctx);

			/* 呼び出し元に戻る */
			BEGIN(xstart_caller);

			/* xpr モード以外ではコメントは構文解析に不要なので無視 */
			if (xstart_caller == xrp) {
				divy_sbuf_append(p_ctx->new_sql_buf, yytext);
			}
		}
<xc><<EOF>>	{
			CLEAR_KEYWORD(p_ctx);

			/* xc モードを解除して処理を終わらせてしまう */
			BEGIN(INITIAL);
			return EOF_COMMENT;
		}
<xrp>{bind} {
			CLEAR_KEYWORD(p_ctx);

			/* バインド変数を置き換える */
			retcd = _replace_reqsqlvalue(p_ctx, BIND, yytext); 
			if (retcd != ST_YYLEX_OK) return retcd;
		}
<xrp>{xrpmatch}	{
			CLEAR_KEYWORD(p_ctx);

			divy_sbuf_append(p_ctx->new_sql_buf, yytext);
		}

<xrp>{namedbind} {
			CLEAR_KEYWORD(p_ctx);

			/* 名前付きバインド変数を置き換える */
			retcd = _replace_reqsqlvalue(p_ctx, NAMEDBIND, yytext); 
			if (retcd != ST_YYLEX_OK) return retcd;
		}

<xrp>{s_requiredsql} {
			CLEAR_KEYWORD(p_ctx);

			/* RequiredSQLを値で置き換える */
			retcd = _replace_reqsqlvalue(p_ctx, S_REQUIREDSQL, yytext); 
			if (retcd != ST_YYLEX_OK) return retcd;
		}

<xrp>{m_requiredsql} {
			CLEAR_KEYWORD(p_ctx);

			/* RequiredSQLを値で置き換える */
			retcd = _replace_reqsqlvalue(p_ctx, M_REQUIREDSQL, yytext); 
			if (retcd != ST_YYLEX_OK) return retcd;
		}

<xrp><<EOF>>	{
			CLEAR_KEYWORD(p_ctx);

			BEGIN(INITIAL);
			/* 終了 */
			yyterminate();
		}

{param}		{
			CLEAR_KEYWORD(p_ctx);

			p_yylval->str = apr_pstrdup(p, yytext);
			return SCONST;
		}

{integer}	{
			CLEAR_KEYWORD(p_ctx);

			p_yylval->str = apr_pstrdup(p, yytext);
			return NCONST;
		}

{decimal}	{
			CLEAR_KEYWORD(p_ctx);

			p_yylval->str = apr_pstrdup(p, yytext);
			return NCONST;
		}

{real}		{
			CLEAR_KEYWORD(p_ctx);

			p_yylval->str = apr_pstrdup(p, yytext);
			return NCONST;
		}
{dblink}	{ /* ignore */ }
{orasys_viewname} {
			CLEAR_KEYWORD(p_ctx);
			p_yylval->str = apr_pstrdup(p, yytext);
			return IDENT;
		}
{identifier}	{
			keyword = NULL;
			CLEAR_KEYWORD(p_ctx);

			/* yytext から予約されたSQLキーワードを探す */
			keyword = _lookup_sqlkeyword(p, yytext);
			if (keyword) {
				p_yylval->keyword = apr_pstrdup(p, yytext);
				p_ctx->keyword = keyword; /* 一時保存 */
				return keyword->token;
			}
			else {
				p_yylval->str = apr_pstrdup(p, yytext);
				return IDENT;
			}
		}

{op_ora_outerjoin} { /* ignore */ }
{self}		{
			CLEAR_KEYWORD(p_ctx);

			/* 文字をそのまま渡す (文字コードが数値として返却される) */
			return yytext[0];
		}

{self_ignore}	{ /* ignore */
			/* self 以外の単項演算子の場合 */
			CLEAR_KEYWORD(p_ctx);
		}

{operator}	{
			keyword = NULL;
			CLEAR_KEYWORD(p_ctx);

			/* 値をコピーする */
			p_yylval->str = apr_pstrdup(p, yytext);

			/* yytext が予約された演算してあるかどうか */
			keyword = _lookup_compound_operator(p, yytext);
			if (keyword) {
				return keyword->token;
			}
			else {
				/* 未定義の複合演算子の場合 */
				return OP;
			}
		}

{bind}		{
			CLEAR_KEYWORD(p_ctx);

			p_yylval->str = apr_pstrdup(p, yytext);
			return BIND;
		}

{namedbind}     {
			CLEAR_KEYWORD(p_ctx);

			p_yylval->str = apr_pstrdup(p, yytext);
			return NAMEDBIND;
		}

{s_requiredsql}   {
			CLEAR_KEYWORD(p_ctx);

			p_yylval->str = apr_pstrdup(p, yytext);
			return S_REQUIREDSQL;
                }

{m_requiredsql}   {
			CLEAR_KEYWORD(p_ctx);

			p_yylval->str = apr_pstrdup(p, yytext);
			return M_REQUIREDSQL;
                }

{other}		{ /* ignore */
			CLEAR_KEYWORD(p_ctx);
#ifdef YYLEX_DEBUG
			ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"[other]:%s", yytext);
#endif	/* YYLEX_DEBUG */
		}
<<EOF>>		{
#ifdef YYLEX_DEBUG
			ERRLOG0(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "[string eof]");
#endif	/* YYLEX_DEBUG */
			/* YY_NULL (= 0) を返却するマクロを呼ぶ */
			yyterminate();
		}
%%

/*--------------------------------------------------------------
  Define functions
  --------------------------------------------------------------*/
/**
 * 解析しなければならないSQLをセットしてscanner(字句解読パーサ)を
 * 初期化する。
 * (note)
 *	この関数はマルチスレッド対応ではありません。
 *	この関数が呼び出されてからtf_scanner_finishが呼ばれるまでの間
 *	複数のスレッドから使用されると、Coreダンプ等の原因になります。
 *
 * (note) マルチスレッド対応にならない理由
 *	yacc 経由でyylex という関数の
 *
 * @param p apr_pool_t *
 * @param p_ctx divy_sql_parser_ctx *
 * @return 処理ステータス
 *	DIVY_SQLP_ST_OK  : 成功	
 *	DIVY_SQLP_ST_ERR : 失敗
 */
DIVY_DECLARE(int) tf_scanner_init(divy_sql_parser_ctx *p_ctx)
{
	apr_size_t len;
	apr_pool_t *p = p_ctx->p;

	TRACE(p);

	if (IS_EMPTY(p_ctx->str)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"p_ctx->str is empty.");
		return DIVY_SQLP_ST_ERR;
	}

	len = strlen(p_ctx->str);

	/* バッファが存在しているかどうか */
	if (YY_CURRENT_BUFFER) {
		/* 古いバッファを破棄する */
		yy_delete_buffer(YY_CURRENT_BUFFER);
	}

	p_ctx->scanbuf = apr_pcalloc(p, len + 2);
	memcpy(p_ctx->scanbuf, p_ctx->str, len);
	p_ctx->scanbuf[len] =
		p_ctx->scanbuf[len + 1] = YY_END_OF_BUFFER_CHAR;

	/* スキャン用バッファの作成 */
	scanbufhandle = yy_scan_buffer(p_ctx->scanbuf, len + 2);

	/*
	 * 開始条件の設定
	 */
	if (p_ctx->xstart == SCANNER_START_MODE_REPLACE) {
		/* RequiredSQL, 名前付きバインド変数
			置き換え処理のための開始条件 */
		BEGIN(xrp);
	}
	else {
		/* デフォルトの開始条件を使用する */
		BEGIN(INITIAL);
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * scanner の終了処理
 *
 * @param p_ctx divy_sql_parser_ctx *
 * @return 処理ステータス
 *	DIVY_SQLP_ST_OK  : 成功	
 */
DIVY_DECLARE(int) tf_scanner_finish(divy_sql_parser_ctx *p_ctx)
{
	TRACE(p_ctx->p);

	/* バッファを破棄する */
        yy_delete_buffer(scanbufhandle);

	return DIVY_SQLP_ST_OK;
}

/*--------------------------------------------------------------
  Replace functions 
  --------------------------------------------------------------*/
/**
 * LEX の致命的なエラーを表示するための関数
 * YY_FATAL_ERROR を置き換えるための定義
 * (note)
 *	YY_FATAL_ERROR のデフォルト実装は、exitを発行してしまうため、
 *	仕方なく置き換えました。
 * (note)
 *	置き換えてはいるものの、呼び出し規則は決まっているので、
 *	apr_pool_t を新たに追加することはできませんでした。
 */
#ifdef YY_USE_PROTOS
static void tf_yy_fatal_error(yyconst char msg[])
#else
static void tf_yy_fatal_error( msg )
char msg[];
#endif
{
	ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
		"Failed to parse sql. Reason: %s", msg);
	return;
}

/*--------------------------------------------------------------
  Define private function
  --------------------------------------------------------------*/
/**
 * text が示す文字列のトークンを探して返却する
 * (note) 一旦小文字にしてから実行する
 *
 * @param p apr_pool_t *
 * @param text char *   検索対象のキーワード
 * @return sqlkeyword * 検索されたキーワード
 *			見つからなかった場合にはNULLを返却
 */
static sqlkeyword * _lookup_sqlkeyword(apr_pool_t *p, char *text)
{
	return _binarysearch(p, text, SQLKEYWORDTBL_LEN, SqlKeyWordTbl); 	
}

/**
 * 式に指定された複合記号オペレータを探して、sqlkeywordを返却する
 * (note)
 *
 * @param p apr_pool_t *
 * @param text char *   検索対象のキーワード
 * @return sqlkeyword * 検索されたキーワード
 */
static sqlkeyword * _lookup_compound_operator(apr_pool_t *p, char *text)
{
	return _binarysearch(p, text, COMPOUNDOPTBL_LEN, CompoundOpTbl);
}

/**
 * 長さlen の指定されたキーワードテーブルtblからtext の名前をもつsqlkeyword
 * を２分検索し、検索結果を返却する.
 *
 * @param p apr_pool_t *
 * @param text char *   検索対象のキーワード
 * @param len int テーブルのサイズ
 * @param tbl const sqlkeyword [] キーワードテーブル
 * @return sqlkeyword * 検索されたキーワード
 *			見つからなかった場合にはNULLを返却
 */
static sqlkeyword * _binarysearch(apr_pool_t *p, char *text,
					int len, const sqlkeyword tbl[])
{
	int low, high, mid, ret;
	char *str;

	if (len == 0) return NULL;

	str = apr_pstrdup(p, text);
	ap_str_tolower(str);	/* 小文字にする */

	for (low = 0, high = len; low < high; ) {
		mid = (low + high) / 2;
		ret = strcmp(tbl[mid].name, str);
		if (ret == 0) {
			return (sqlkeyword *) &tbl[mid];
		}
		else if (ret < 0) {
			low = mid + 1;
		}
		else {
			high = mid;
		}
	}

	return NULL;
}

/**
 * text が示すRequiredSQLまたは名前付きバインド変数を"?"で置換し、
 * p_ctx->bindvals に値を設定する
 *
 * @param p_ctx divy_sql_parser_ctx *
 * @param type int 種類 (NAMEDBIND, S_REQUIREDSQL, M_REQUIREDSQL, BIND)
 * @param text char *
 * @return int 処理ステータス
 *	ST_YYLEX_OK          : 正常
 *	ST_YYLEX_MISSING_ID  : ipos に対応するIDがrsvalueの中に無かった
 *	ST_YYLEX_MISMATCH_ID : ipos位置のRequiredSQL名称がSQL文の中のそれと異なっていた
 */
static int _replace_reqsqlvalue(divy_sql_parser_ctx *p_ctx, int type, char *text)
{
	apr_pool_t *p = p_ctx->p;
	int first     = 1;

	if (IS_EMPTY(text)) {
		return ST_YYLEX_OK;
	}

	/* $$Bxxx, $$SSxxx, $$SMxxx */
	if (type == NAMEDBIND || type == S_REQUIREDSQL || type ==M_REQUIREDSQL) {
		divy_rdbo_rsvalue *rsvalue;
		divy_rdbo_rsvalueset *valueset;

		/* p_ctx->reqsql_value から出現位置の値を取り出す */
		p_ctx->ipos++;		/* 出現位置をカウントアップ */
		for (rsvalue = p_ctx->reqsql_value; rsvalue; rsvalue = rsvalue->next) {
			if (rsvalue->id == p_ctx->ipos) break;
		}

		if (rsvalue == NULL) {
			/* ipos に対応するIDがrsvalue の中に無かった場合 */
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The id \"%d\" in the resolve-value is missing.",
				p_ctx->ipos);

			return ST_YYLEX_MISSING_ID;
		}
		else if (rsvalue->name == NULL || strcmp(text, rsvalue->name) != 0) {
			/* ipos位置のRequiredSQL名称がSQL文の中のそれと異なっていた */	
			ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The subname \"%s\" in the resolve-value data is different "
				"from the subname \"%s\" in the sql.(condition-postion = %d)",
				REPLACE_NULL(rsvalue->name), text, p_ctx->ipos);

			return ST_YYLEX_MISMATCH_ID;
		}

		for (valueset = rsvalue->valueset; valueset; valueset = valueset->next) {

			if (p_ctx->bindvals == NULL) {
				p_ctx->bindvals = divy_array_make(p, 10);
			}

			/* $$Bxxx, $$SSxxx */
			if (type == NAMEDBIND || type == S_REQUIREDSQL) {
				divy_sbuf_appendbyte(p_ctx->new_sql_buf, 1, "?");
				divy_array_add(p_ctx->bindvals, valueset->value);
				break;	/* 取得値は１つだけ */
			}
			/* $$SMxxx */
			else {
				if (first) {
					divy_sbuf_appendbyte(p_ctx->new_sql_buf, 1, "(");
					first = 0;
				}
				if (valueset->next != NULL) {
					divy_sbuf_appendbyte(p_ctx->new_sql_buf, 2, "?,");
				}
				else {
					divy_sbuf_appendbyte(p_ctx->new_sql_buf, 2, "?)");
				}
				divy_array_add(p_ctx->bindvals, valueset->value);
			}
		}
	}
	/* バインド変数 */
	else {
		divy_search_ldbs_bind *bvalue;
		p_ctx->ibpos++;
		for (bvalue = p_ctx->bvalue; bvalue; bvalue = bvalue->next) {
			if (bvalue->id == p_ctx->ibpos) break;
		}

		if (bvalue == NULL) {
			/* ipos に対応するIDがrsvalue の中に無かった場合 */
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The id \"%d\" in the value element is missing.",
				p_ctx->ibpos);

			return ST_YYLEX_MISSING_ID;
		}
		divy_sbuf_appendbyte(p_ctx->new_sql_buf, 1, "?");

		if (p_ctx->bindvals == NULL) {
			p_ctx->bindvals = divy_array_make(p, 10);
		}
		divy_array_add(p_ctx->bindvals, bvalue->value);
	}

	return ST_YYLEX_OK;
}


/**
 * 与えられた２つのトークン値 token1 と token2 の組合せが
 * "コンビネーショントークン"として登録されていないかどうか調べる。
 *
 * @param token1 int
 * @param token2 int
 * @return int コンビネーショントークンの値 /
 		ST_YYLEX_MISSING_TOKEN (見つからなかった場合)
 */
static int _check_token_combination(int token1, int token2)
{
	int i, len = COMBTOKENTBL_LEN;
	const combtoken *ctoken;

	/* コンビネーショントークンの集合から探す */
	for (i = 0; i < len; i++) {
		/* 1個の組合せを取り出す */
		ctoken = &CombTokenTbl[i];

		if (token1 == ctoken->token1 && token2 == ctoken->token2) {
		    return ctoken->combination;
		}
	}

	return ST_YYLEX_MISSING_TOKEN;	/* 見つからなかった */
}

/**
 * 与えられた２つのトークン値 token1 と token2 の組合せが
 * "ダブル先読みトークン"として登録されていないかどうか調べる。
 *
 * (note) "ダブル先読みトークン"とは?
 *	YACCパーサは、LALR(1)という構文解析方法に基づいて状態表を作成するため、
 *	1つ先のトークン(lookahead token)しか参照出来ない。ところが、構文の
 *	中にはもう1つ先のトークンを見なければ遷移先が分からないようなものが
 *	存在している。そのような時に、先読みトークンのさらに先を見て
 *	そのトークンの状態によってトークン値を変更すればYACCパーサは
 *	状態遷移できるようになる。このように、先読みのさらに先を見た結果
 *	得られたトークンを"ダブル先読みトークン"と呼ぶことにした。(造語)
 *	なお、先読みトークンとその更に先のトークンを1つのトークンに再編成
 *	してしまうには、関数_check_token_combination を使ってください。
 *	このようなトークンのことを"コンビネーショントークン"(造語)と呼んでいます。
 *
 * @param token1 int
 * @param token2 int
 * @return int コンビネーショントークンの値 /
 		ST_YYLEX_MISSING_TOKEN (見つからなかった場合)
 */
static int _check_token_lookdoubleahead(int token1, int token2)
{
	int i, len = DOUBLEAHEADTOKENTBL_LEN;
	const doubleaheadtoken *dtoken;

	/* ダブル先読みトークンの集合から探す */
	for (i = 0; i < len; i++) {
		/* 1個の組合せを取り出す */
		dtoken = &DoubleAheadTokenTbl[i];

		if (token1 == dtoken->token1 && token2 == dtoken->token2) {
		    return dtoken->token;
		}
	}

	return ST_YYLEX_MISSING_TOKEN;	/* 見つからなかった */
}

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * 構文解析パーサ用緩衝ルーチン
 * 構文解析パーサが利用するyylex のラップバージョンです。
 *
 * (note) この関数導入の動機
 *	構文解析パーサに渡されるトークンの情報を隠す目的で導入しました。
 *	SELECT文を解析する上で不要なトークンが構文解析パーサ側に渡されると
 *	それに対応した構文を記述しなければならなくなりますので、コードを
 *	単純化する役割を持っています。

 *	当初は、SQL文中で構文を形成する殆どのキーワードを理解しなくてもカラム名
 *	やSQL関数などと共に使用されるRequiredSQLや通常・名前付きバインド変数を
 *	解析できると考えていたため、この緩衝ルーチンを使って構文解析パーサ側
 *	から殆どのキーワードが見えないように隠す目的で使用していた。
 *	だが、結局、SQL文に現れる多くのキーワードは認識しなければならなかった
 *	ことが判明したため、この緩衝ルーチンは、構文解析パーサに本当に不要な
 *	構文を隠す役割になりました。

 * @param p_yylval YYSTYPE *
 * @param p_ctx divy_sql_parser_ctx * パーサコンテキストへのポインタ
 * @return int yylex からの戻り値
 */
int tf_inter_yylex(YYSTYPE *p_yylval, divy_sql_parser_ctx *p_ctx)
{
	int ret = 0;
	sqlkeyword *keyword = NULL;

	/* トークンの取得 */
	ret = yylex(p_yylval, p_ctx);
	keyword = (sqlkeyword *) p_ctx->keyword;

	/*
	 * SQL解析位置の取得
	 * (note) キーワード自身間の長さも含んでいます
	 */
	p_ctx->sqlposition = (int) (yytext - p_ctx->scanbuf + yyleng);

	/*
	 * 構文解析パーサでは必要のないトークンの読み飛ばし
	 */
	if (keyword) {
		/* 必要なトークンが見つかるまで読み飛ばす */
		if (keyword->flag & FLG_RESERVE &&
		    keyword->flag & FLG_NOTUSING) {
			_TOKEN_DEBUG2(p_ctx->p, ret, yytext, "(notusing) ->");
			return tf_inter_yylex(p_yylval, p_ctx); /* 再帰 */
		}
		/* 2トークン読み替え処理(コンビネーショントークンの処理) */
		else if (keyword->flag & FLG_LOOKAHEAD ||
			 keyword->flag & FLG_COMBTOKEN) {
			int ret2, token;

			/* 最初のトークン情報を一時退避 */
			char *bk_buf = apr_pstrdup(p_ctx->p, p_yylval->keyword);
			sqlkeyword *bk_keyword = p_ctx->keyword;

			_TOKEN_DEBUG3(p_ctx->p, ret, p_yylval->keyword,
						p_ctx->sqlposition,"(token1)");

			/* 次のトークンを読んでみる */
			ret2 = yylex(p_yylval, p_ctx);

			/* コンビネーショントークンの検索 */
			if (keyword->flag & FLG_COMBTOKEN) {

				token = _check_token_combination(ret, ret2);
				if (token != ST_YYLEX_MISSING_TOKEN) {
					/* SQL 解析位置の補正 */
					p_ctx->sqlposition = (int) (yytext -
							p_ctx->scanbuf + yyleng);

					_TOKEN_DEBUG3(p_ctx->p, token, p_yylval->keyword,
							p_ctx->sqlposition,"(ctoken)");
					return token;	/* コンビネーショントークンを返却 */
				}
				else if (token == ST_YYLEX_MISSING_TOKEN &&
					!(keyword->flag & FLG_LOOKAHEAD)) {
					/* 次のトークン読み込みを無かったことにする */
					yyless(0);
					p_yylval->keyword = bk_buf;
					p_ctx->keyword = bk_keyword;

					_TOKEN_DEBUG3(p_ctx->p, ret, p_yylval->keyword,
							p_ctx->sqlposition, "(token1_return)");
					return ret;
				}
			}

			/* ダブル先読みトークンの検索 */
			if (keyword->flag & FLG_LOOKAHEAD) {

				token = _check_token_lookdoubleahead(ret, ret2);

				/* 見つかってもそうでなくても解析位置,キーワードは前に戻す */
				yyless(0);
				p_yylval->keyword = bk_buf;
				p_ctx->keyword = bk_keyword;

				if (token != ST_YYLEX_MISSING_TOKEN) {

					/* 解析位置の補正は必要なし */
					_TOKEN_DEBUG3(p_ctx->p, token, p_yylval->keyword,
							p_ctx->sqlposition,"(dtoken)");
					return token;	/* ダブル先読みトークンを返却 */
				}
				else {
					_TOKEN_DEBUG3(p_ctx->p, ret, p_yylval->keyword,
							p_ctx->sqlposition,"(token1_return)");
					return ret;
				}
			}
		}
	}

	_TOKEN_DEBUG3(p_ctx->p, ret, yytext, p_ctx->sqlposition, "");
	return ret;
}

