/**
 * $Id$
 *
 * 自動削除周りの操作を行う関数、構造体の定義
 *
 */
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_hash.h"
#include "apr_time.h"
#include "apr_xml.h"
#include "apr_file_info.h"
#include "apr_file_io.h"
#include "httpd.h"

#include "util_common.h"
#include "tf_autodelete.h"
#include "tf_validator.h"
#include "tf_thumbnail.h"
#include "tf_rdbo.h"
#include "tf_rdbo_group.h"

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare and Define structure
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static char * _build_rb_configpath(apr_pool_t *wp, const char *location);
static char * _build_glist_configpath(apr_pool_t *wp, const char *location);
static char * _build_dellist_configpath(apr_pool_t *wp, const char *location, const char *groupid);
static int _parse_rb_config(request_rec *r, divy_autodel_rb_conf **rb_conf);
static int _parse_glist(request_rec *r, divy_autodel_glist **glist);
static int _parse_deletedlist(request_rec *r, const char *groupid, divy_autodel_dresource **del_res);
static int _save_glist_xml(request_rec *r, divy_autodel_glist *glist);
static int _save_deletedlist_xml(request_rec *r, divy_autodel_dresource *del_res,
                                 const char *groupid, int delperiod, apr_time_t now);

/*------------------------------------------------------------------------------
  Define array
  ----------------------------------------------------------------------------*/
/**
 * 関数divy_autodel_encipher_groupid で使用する文字マップテーブル.
 * (note)
 *    ここを変えるのなら、システムCGI(autodelhistory)も変えること
 */
static const char _mpgtbl[] = { 'F','J','w','m','E','h','c','k','a','z' };

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * 自動削除機能がサポートされているかどうか.
 * (note)
 *   サポートされているかどうかは、ライセンスとディレクティブの有効/無効化設定で
 *   決まります.
 *
 */
DIVY_DECLARE(int) divy_support_autodelete(request_rec *r)
{
	dav_divy_server_conf *sconf = NULL;
	dav_divy_dir_conf    *dconf = NULL;

	if (r == NULL) return 0;

	sconf = dav_divy_get_server_config(r->server);
	dconf = dav_divy_get_dir_config(r);

	/* ライセンスが有効 かつ ディレクティブで有効になっている */
	if (sconf->use_autodel_opt && dconf->autodelete == DIVY_AUTODELETE_ON) {
		return 1;
	}

	return 0;
}

/**
 * 指定されたu_spec, r が示すリソースが自動削除処理リクエストを受けるのに
 * 適切かどうか.
 *
 */
DIVY_DECLARE(int) divy_autodel_enable(request_rec *r, divy_uri_spec *u_spec)
{
	if (r == NULL || u_spec == NULL) {
		return 0;
	}

	/* POST オペレーションをサポートしていなければ常に有効ではない */
#ifndef DAV_SUPPORT_POST_HANDLING
	return 0;
#endif	/* DAV_SUPPORT_POST_HANDLING */

	/*
	 * [ 有効となる条件 ]
	 *
	 *  * 自動削除機能のライセンスがあること(ディレクティブ)
	 *  * 自動削除機能そのものが有効であること(ディレクティブ)
	 *
	 *  * infotype が DIVY_INFOTYPE_autodelete
	 *  * POST リクエストであること
	 *  * 127.0.0.1 or ディレクティブアドレスへのリクエストであること
	 *  * 対象ロケーションで自動削除機能がアクティブであること(XMLファイル)
	 */
	if (!divy_support_autodelete(r)) {
		return 0;
	}

	if (u_spec->infotype == DIVY_INFOTYPE_autodelete &&
		divy_get_method_number(r) == M_POST) {

		apr_status_t rv;
		apr_ipsubnet_t *ipsub         = NULL;
		divy_autodel_rb_conf *rb_conf = NULL;
		dav_divy_dir_conf *dconf      = dav_divy_get_dir_config(r);

		/* アドレスパターンのチェック */
		if (dconf->autodel_allowipaddrs != NULL) {
			int i, len = divy_array_getlength(dconf->autodel_allowipaddrs);
			for (i = 0; i < len; i++) {
				ipsub = divy_array_get(dconf->autodel_allowipaddrs, i);
				if (ipsub == NULL || !apr_ipsubnet_test(ipsub, r->useragent_addr)) {
					return -1;	/* 不正アクセス */ 
				}
			}
		}
		/* 省略された場合には、ローカルIPからのアクセスのみを許可する */
		else {
			rv = apr_ipsubnet_create(&ipsub, "127.0.0.1", "255.255.255.255", r->pool);
			if (rv != APR_SUCCESS || ipsub == NULL ||
				!apr_ipsubnet_test(ipsub, r->useragent_addr)) {
				return -1;	/* 不正アクセス */
			}
		}

		/* 設定ファイルをパースする */
		if (_parse_rb_config(r, &rb_conf) || rb_conf == NULL || !rb_conf->isActive) {
			return 0;
		}
		return 1;
	}
	else {
		return -1;	/* 不正アクセス */
	}

	return 0;
}

/**
 * 自動削除処理のメイン
 *
 */
DIVY_DECLARE(dav_error *) divy_autodel_execute(request_rec *r, const dav_resource *resource)
{
	apr_pool_t *p                   = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;
	divy_autodel_glist *glist       = NULL;
	apr_time_t now                  = apr_time_now();
	divy_autodel_rb_conf *rb_conf   = NULL;
	divy_linkedlist_t *del_filelist;
	divy_autodel_dresource *del_res;
	apr_time_t bordertime;
	divy_rdbo_grp *grp_pr;
	int ret;

	/*
	 * (1) 自動削除バッチの設定を取得する
	 */
	if (_parse_rb_config(r, &rb_conf) || rb_conf == NULL || !rb_conf->isActive) {
		/* ここのステージでコンフィグが取れない等の問題が起きたらエラーにする */
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * (2) 自動削除対象グループ一覧を取得する
	 */
	if (_parse_glist(r, &glist)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to parse autodelete list XML.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* 1件もグループリストが無ければ何もしない */
	if (glist == NULL) {
		return NULL;
	}

	/*
	 * (3) グループ毎に処理を実施 (グループ単位のトランザクションとする)
	 */
	for (; glist != NULL; glist = glist->next) {
		/* 境界値を算出する(glist->pperiod が0 の場合もあるが気にしない) */
		bordertime = now - apr_time_from_sec(glist->pperiod * 24 * 60 * 60);

		/* 保存日数が今まで経過したエポックタイムよりも長かった場合 */
		if (bordertime < APR_TIME_C(0)) {
			continue;	/* これらのファイルは決して自動削除対象には成り得ないのでスキップ */
		}

		ts_ctx = NULL;
		/* トランザクション開始
		 * (note) トランザクションはグループ単位で. */
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to create transaction context for autodelete operation. "
					"(groupid = %s)", glist->groupid);
			continue;	/* 一応次へ */
		}

		/* グループプロパティの取得 */
		if (divy_rdbo_get_group_property_by_groupid(r, glist->groupid, &grp_pr, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get group-property. (groupid = %s)", glist->groupid);
			divy_db_rollback_transaction(ts_ctx);
			continue;
		}
		/* グループ情報が取得できなければ削除の必要なし. 終わったことにする. */
		if (grp_pr == NULL) {
			divy_db_commit_transaction(ts_ctx);
			continue;
		}

		del_filelist = NULL; del_res = NULL;
		/*
		 * 条件に一致するファイル / フォルダを全て削除する
		 * [ 対象 ]
		 *  * ファイル/フォルダ (dav_resource, divy_usr, divy_usrdiskquota, divy_diskquota)
		 *  * ごみ箱プロパティ  (divy_trash_info)
		 *  * Deadプロパティ    (dav_dead_property)
		 *  * 公開・非公開状態  (divy_resourcestate)
		 *  * ロック状態        (dav_lock)
		 *  * サムネイル        (後で)
		 *  * ファイルの実体    (後で)
		 */
		ret = divy_rdbo_remove_old_groupresources(r, grp_pr, bordertime,
						(glist->spaction & DIVY_AUTODEL_ACTION_DELFOLDER) ? 1 : 0,
						&del_filelist, &del_res, ts_ctx);
		if (ret) {
			/* トランザクションのロールバック */
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);

			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to auto-delete old group resources.(groupid = %s)", glist->groupid);
			continue;
		}

		/* DBへの反映を確定する
		 * (ファイルの削除は待ちません。待ってもリカバリーできないから) */
		divy_db_commit_transaction(ts_ctx);

		/*
		 * 物理ファイルとサムネイルをストレージから削除する
		 * (note) エラーを無視して進める
		 */
		(void) divy_remove_physical_files(r, del_filelist);

		/*
		 * 削除リストをファイルに書き出す
		 */
		(void) _save_deletedlist_xml(r, del_res, glist->groupid, rb_conf->delperiod, now);
	}

	return NULL;
}

/**
 * 自動削除対象からグループIDの集合groupid_set のグループ群を対象外とする.
 *
 * @param r request_rec *
 * @param groupid_set divy_cset_t グループIDを要素として持つ集合
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_autodel_excludeTargets(request_rec *r, divy_cset_t *groupid_set)
{
	apr_pool_t *p  = r->pool;
	divy_autodel_glist *glist = NULL, *prev = NULL, *g = NULL;
	int found_grp = 0;
	char *fdellist, *groupid;
	const char *root_uri = dav_divy_get_root_uri(r);
	divy_cset_index_t *ci;

	if (groupid_set == NULL) {
		return 0;	/* 何もしない */
	}

	/*
	 * (1) 自動削除対象グループ(glist) があるかどうかに関わらず、groupid_set に含まれる
	 *     グループの自動削除済みリストは削除する.
	 *     何故なら、そのグループは嘗て、削除済みリストを持っていたかもしれないから.
	 */
	for (ci = divy_cset_first(groupid_set); ci != NULL; ci = divy_cset_next(ci)) {
		divy_cset_this(ci, (const char **)&groupid);

		if (IS_FILLED(groupid)) {
			/* 自動削除済みリストを削除 */
			fdellist = _build_dellist_configpath(p, root_uri, groupid);
			if (IS_FILLED(fdellist)) {
				apr_file_remove(fdellist, p);
			}
		}
	}

	/*
	 * (2) 自動削除対象グループ一覧を取得する
	 */
	if (_parse_glist(r, &glist)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to parse autodelete list XML.");
		return 1;
	}

	/* 1件もグループリストが無ければ何もしない */
	if (glist == NULL) {
		return 0;
	}

	/*
	 * (3) groupid_set に含まれるグループをglist から外す
	 */
	prev = NULL;
	for (g = glist; g != NULL; g = g->next) {
		if (divy_cset_contains(groupid_set, g->groupid)) {
			if (prev == NULL) {
				glist = g->next;
			}
			else {
				prev->next = g->next;
			}
			found_grp = 1;
		}
		else {
			prev = g;
		}
	}

	/*
	 * (3) 残ったglist からコンフィグファイルを再構築する
	 */
	if (found_grp && _save_glist_xml(r, glist)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to build autodelete list XML.");
		return 1;
	}

	return 0;
}

/**
 * 指定されたグループID 文字列を自動削除済みリストアクセス用のグループIDに符号化して返却する.
 *
 */
DIVY_DECLARE(char *) divy_autodel_encipher_groupid(apr_pool_t *p, const char *groupid)
{
	int i, len;
	const char *ch;
	char *str, *first;

	if (IS_EMPTY(groupid)) return "";

	first = str = apr_pcalloc(p, strlen(groupid) + 1);

	/*
	 * (方式)
	 *   数値文字を決められたアルファベット文字に変換する.
	 * (note)
	 *   ここで決められた符号化方式は、システムCGIでも意識しているので
	 *   変更するなら一緒に変えなさい
	 */
	len = strlen(_mpgtbl);
	for (ch = groupid; *ch != '\0'; ++ch) {
		i = *ch - '0';
		if (i < 0 || i >= len) {
			/* groupid に数値以外が含まれていた場合には、暗号化を止めてそのまま返す */
			return apr_pstrdup(p, groupid);
		}
		*str++ = _mpgtbl[i];
	}
	*str = '\0';

	return first;
}

/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * 定期バッチ用設定ファイルのパスを組み立てて返却する.
 *
 * [ ファイル名規則 ]
 * 	(ロケーション名) + ".set.xml"
 *
 * @param wp apr_pool_t *
 * @param location const char * ロケーション名
 * @return char * 組み立てたコンフィグファイルのパス
 */
static char * _build_rb_configpath(apr_pool_t *wp, const char *location)
{
	char *fullpath = ap_make_full_path(wp, DIVY_AUTODELETE_CONFIGROOT,
			apr_psprintf(wp, "%s.set.xml", location));

	ap_no2slash(fullpath);
	return fullpath;
}

/**
 * 自動削除対象グループ一覧を記述するファイルのパスを組み立てて返却する.
 *
 * [ ファイル名規則 ]
 * 	(ロケーション名) + ".list.xml"
 *
 * @param wp apr_pool_t *
 * @param location const char * ロケーション名
 * @return char * 組み立てたコンフィグファイルのパス
 */
static char * _build_glist_configpath(apr_pool_t *wp, const char *location)
{
	char *fullpath = ap_make_full_path(wp, DIVY_AUTODELETE_CONFIGROOT,
			apr_psprintf(wp, "%s.list.xml", location));

	ap_no2slash(fullpath);
	return fullpath;
}

/**
 * 自動削除済みファイル一覧を記述するファイルのパスを組み立てて返却する.
 *
 * [ ファイル名規則 ]
 * 	(ロケーション名) + ".del" + (グループID) + ".xml"
 *
 * @param wp apr_pool_t *
 * @param location const char * ロケーション名
 * @param groupid const char * グループID
 * @return char * 組み立てたコンフィグファイルのパス
 */
static char * _build_dellist_configpath(apr_pool_t *wp, const char *location, const char *groupid)
{
	char *fullpath = ap_make_full_path(wp, DIVY_AUTODELETE_CONFIGROOT,
			apr_psprintf(wp, "%s.del.%s.xml", location, groupid));

	ap_no2slash(fullpath);
	return fullpath;
}

/**
 * 定期バッチ用設定ファイルをパースする.
 *
 */
static int _parse_rb_config(request_rec *r, divy_autodel_rb_conf **rb_conf)
{
	apr_status_t rv;
	apr_pool_t *p          = r->pool;
	apr_file_t *fd         = NULL;
	apr_xml_parser *parser = NULL;
	apr_xml_doc	   *doc    = NULL;
	apr_xml_elem   *elem   = NULL;
	char *fconfig = _build_rb_configpath(p, dav_divy_get_root_uri(r));
	char *data;

	*rb_conf = NULL;	/* 初期化 */

	/*
	 * 対象ロケーションで自動削除機能がアクティブかどうか
	 */
	rv = apr_file_open(&fd, fconfig, APR_READ | APR_BINARY, 0, p);
	/* コンフィグファイルが存在しないのはエラーにはしない */
	if (rv == APR_ENOENT) {
		if (fd != NULL) apr_file_close(fd);
		return 0;	/* 正常だけど*rb_conf は空 */
	}
	else if (rv != APR_SUCCESS) {
		if (fd != NULL) apr_file_close(fd);
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to read autodelete settings XML. (file = %s, code = %d)",
				fconfig, rv);
		return 0;	/* エラーログには出すが、単にサポートしていないと出すだけ */
	}

	/* XML ファイルをパースする */

	/* [ DTD ]
	 * <!ELEMENT autodeletesetting (state, starttime, logdeleteperiod) >
	 * <!ELEMENT state (active | inactive) >
	 * <!ELEMENT active   EMPTY >
	 * <!ELEMENT inactive EMPTY >
	 * <!ELEMENT starttime (hour, minutes) > ; 定期バッチの開始時刻
	 * <!ELEMENT hour    (#PCDATA) >
	 * <!ELEMENT minutes (#PCDATA) >
	 * <!ELEMENT logdeleteperiod (#PCDATA) > ; 自動削除履歴を保持する日数
	 */
	rv = apr_xml_parse_file(p, &parser, &doc, fd, 2048);
	if (rv != APR_SUCCESS) {
		char errbuf[512] = { 0 };
		if (parser != NULL) {
			apr_xml_parser_geterror(parser, errbuf, sizeof(errbuf));

			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"Failed to parse autodelete settings XML. (file = %s)"
					"Reason: (%s)", fconfig, errbuf);
		}
		else {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"Failed to parse autodelete settings XML. (file = %s)", fconfig);
		}
		apr_file_close(fd);
		return 1;	/* 失敗 */
	}
	apr_file_close(fd);

	if (doc == NULL || doc->root == NULL || doc->root->first_child == NULL) {
		return 0;	/* 文法エラーだが、大目に見る(即ち無効) */
	}

	*rb_conf = apr_pcalloc(p, sizeof(divy_autodel_rb_conf));
	(*rb_conf)->isActive  = 0;	/* 無効 */
	(*rb_conf)->hour      = -1;	/* 未設定 */
	(*rb_conf)->min       = -1;	/* 未設定 */
	(*rb_conf)->delperiod = -1;	/* 未設定 */

	for (elem = doc->root->first_child; elem != NULL; elem = elem->next) {
		if (strcmp(elem->name, "state") == 0) {
			apr_xml_elem *state_child = elem->first_child;

			if (state_child != NULL && strcmp(state_child->name, "active") == 0) {
				(*rb_conf)->isActive = 1;
			}
			else {
				(*rb_conf)->isActive = 0;
			}
		}
		else if (strcmp(elem->name, "starttime") == 0) {
			apr_xml_elem *time_child = elem->first_child;
			for (; time_child; time_child = time_child->next) {
				data = (char *) divy_xml_get_cdata(time_child, p, 1);	/* white space は除去する */
				if (strcmp(time_child->name, "hour") == 0 && IS_FILLED(data)) {
					(*rb_conf)->hour = atoi(data);
				}
				else if (strcmp(time_child->name, "minutes") == 0 && IS_FILLED(data)) {
					(*rb_conf)->min = atoi(data);
				}
				else {
					/* 無視 */
				}
			}
		}
		else if (strcmp(elem->name, "logdeleteperiod") == 0) {
			data = (char *) divy_xml_get_cdata(elem, p, 1);	/* white space は除去する */
			if (IS_FILLED(data)) {
				(*rb_conf)->delperiod = atoi(data);
			}
		}
		else {
			/* 無視 */
		}
	}

	return 0;
}


/**
 * 自動削除対象グループ一覧を表す設定ファイルをパースする
 *
 * @return int 処理ステータス(0: 正常 / 1: 失敗)
 */
static int _parse_glist(request_rec *r, divy_autodel_glist **glist)
{
	apr_status_t rv;
	apr_pool_t *p          = r->pool;
	apr_file_t *fd         = NULL;
	apr_xml_parser *parser = NULL;
	apr_xml_doc	   *doc    = NULL;
	apr_xml_elem   *elem   = NULL, *child;
	char *fconfig = _build_glist_configpath(p, dav_divy_get_root_uri(r));
	char *data;
	divy_autodel_glist *g, *wk;

	*glist = NULL;	/* 初期化 */

	/*
	 * 設定ファイルは存在するのか？
	 */
	rv = apr_file_open(&fd, fconfig, APR_READ | APR_BINARY, 0, p);
	/* コンフィグファイルが存在しないのはエラーにはしない */
	if (rv == APR_ENOENT) {
		if (fd != NULL) apr_file_close(fd);
		return 0;	/* 正常だけど*glist は空 */
	}
	else if (rv != APR_SUCCESS) {
		if (fd != NULL) apr_file_close(fd);
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to read autodelete grouplist XML. (file = %s, code = %d)",
				fconfig, rv);
		return 0;	/* エラーログには出すが、単にサポートしていないと出すだけ */
	}

	/* XML ファイルをパースする */

	/* [ DTD ]
	 * <!ELEMENT autodelete (autodeletefolder+) >
	 * <!ELEMENT autodeletefolder (groupid, specialaction, preservationperiod) >
	 * <!ELEMENT groupid (#PCDATA) >             ; グループID
	 * <!ELEMENT specialaction (deletefolder?) > ; 特殊な操作を表す
	 * <!ELEMENT deletefolder EMPTY >            ; フォルダを削除する(無ければ削除しない)
	 * <!ELEMENT preservationperiod (#PCDATA) >  ; 保持日数(数値)
	 */
	rv = apr_xml_parse_file(p, &parser, &doc, fd, 2048);
	if (rv != APR_SUCCESS) {
		char errbuf[512] = { 0 };
		if (parser != NULL) {
			apr_xml_parser_geterror(parser, errbuf, sizeof(errbuf));

			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"Failed to parse autodelete grouplist XML. (file = %s)"
					"Reason: (%s)", fconfig, errbuf);
		}
		else {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"Failed to parse autodelete grouplist XML. (file = %s)", fconfig);
		}
		apr_file_close(fd);
		return 1;	/* 失敗 */
	}
	apr_file_close(fd);

	if (doc == NULL || doc->root == NULL || doc->root->first_child == NULL) {
		return 0;	/* 文法エラーだが、大目に見る(即ち無効) */
	}

	g = NULL;
	for (elem = doc->root->first_child; elem != NULL; elem = elem->next) {

		wk = apr_pcalloc(p, sizeof(divy_autodel_glist));
		wk->groupid  = NULL;
		wk->spaction = DIVY_AUTODEL_ACTION_NOTHING;
		wk->pperiod  = 0;	/* 0 日(デフォルト) */
		wk->next     = NULL;

		for (child = elem->first_child; child != NULL; child = child->next) {
			data = (char *) divy_xml_get_cdata(child, p, 1);	/* white space は除去する */
			if (strcmp(child->name, "groupid") == 0) {
				wk->groupid = data;	/* white space は除去する */
			}
			else if (strcmp(child->name, "specialaction") == 0) {
				if (child->first_child != NULL &&
					strcmp(child->first_child->name, "deletefolder") == 0) {
					wk->spaction |= DIVY_AUTODEL_ACTION_DELFOLDER;
				}
			}
			else if (strcmp(child->name, "preservationperiod") == 0) {
				if (IS_FILLED(data)) {
					wk->pperiod = atoi(data);
				}
			}
			else {
				/* 無視 */
			}
		}

		/* 値が正しく取得できていたものだけをリストに繋げる */
		if (IS_FILLED(wk->groupid) && wk->pperiod >= 0) {
			if (*glist == NULL) {
				*glist = g = wk;
			}
			else {
				g->next = wk;
				g = g->next;
			}
		}
	}

	return 0;
}

/**
 * 自動削除済みリストを読み込んでパースする
 *
 * @return int 処理ステータス(0: 正常 / 1: 失敗)
 */
static int _parse_deletedlist(request_rec *r, const char *groupid, divy_autodel_dresource **del_res)
{
	apr_status_t rv;
	apr_pool_t *p          = r->pool;
	apr_file_t *fd         = NULL;
	apr_xml_parser *parser = NULL;
	apr_xml_doc	   *doc    = NULL;
	apr_xml_elem   *elem   = NULL, *child;
	char *fconfig = _build_dellist_configpath(p, dav_divy_get_root_uri(r), groupid);
	char *data;
	divy_autodel_dresource *d, *wk;

	*del_res = NULL;	/* 初期化 */

	/*
	 * 設定ファイルは存在するのか？
	 */
	rv = apr_file_open(&fd, fconfig, APR_READ | APR_BINARY, 0, p);
	/* コンフィグファイルが存在しないのはエラーにはしない */
	if (rv == APR_ENOENT) {
		if (fd != NULL) apr_file_close(fd);
		return 0;	/* 正常だけど*del_res は空 */
	}
	else if (rv != APR_SUCCESS) {
		if (fd != NULL) apr_file_close(fd);
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to read autodeleted filelist XML. (file = %s, code = %d)",
				fconfig, rv);
		return 0;	/* エラーログには出すが、単にサポートしていないと出すだけ */
	}

	/* XML ファイルをパースする */

	/* [ DTD ]
	 * <!ELEMENT autodeletedlist (filelist?) >
	 * <!ELEMENT filelist (displayname, getcontentlength, getcontenttype,
	 *                     lastmodifier, getlastmodified, deletion) >
	 * <!ELEMENT displayname      (#PCDATA) > ; 名称 (表示名)
	 * <!ELEMENT getcontentlength (#PCDATA) > ; サイズ
	 * <!ELEMENT getcontenttype   (#PCDATA) > ; 種類
	 * <!ELEMENT lastmodifier     (#PCDATA) > ; 最終更新者
	 * <!ELEMENT getlastmodified  (#PCDATA) > ; 最終更新日 (エポックタイム表現, micro sec)
	 * <!ELEMENT deletion         (#PCDATA) > ; 削除日     (エポックタイム表現, micro sec)
	 */
	rv = apr_xml_parse_file(p, &parser, &doc, fd, 8192);
	if (rv != APR_SUCCESS) {
		char errbuf[512] = { 0 };
		if (parser != NULL) {
			char *tmp_file;
			apr_xml_parser_geterror(parser, errbuf, sizeof(errbuf));
			apr_file_close(fd);

			/* もしデータが壊れていた場合には、2度と開けないので、古い削除済みリストは退避してしまう */
			tmp_file = apr_psprintf(p, "%s.broken", fconfig);
			(void) apr_file_rename(fconfig, tmp_file, p);

			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"Failed to parse autodeleted filelist XML. (file = %s) "
					"We rename this file to %s. "
					"Reason: (%s)", fconfig, tmp_file, errbuf);
		}
		else {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"Failed to parse autodeleted filelist XML. (file = %s)", fconfig);
			apr_file_close(fd);
		}
		return 1;	/* 失敗 */
	}
	apr_file_close(fd);

	if (doc == NULL || doc->root == NULL || doc->root->first_child == NULL) {
		return 0;	/* 文法エラーだが、大目に見る(即ち無効) */
	}

	d = NULL;
	for (elem = doc->root->first_child; elem != NULL; elem = elem->next) {

		wk = apr_pcalloc(p, sizeof(divy_autodel_dresource));
		wk->next = NULL;

		for (child = elem->first_child; child != NULL; child = child->next) {
			if (strcmp(child->name, "displayname") == 0) {
				data = (char *) divy_xml_get_cdata(child, p, 0);	/* white space は除去しない */
				wk->displayname = data;
			}
			else if (strcmp(child->name, "getcontentlength") == 0) {
				data = (char *) divy_xml_get_cdata(child, p, 1);
				if (IS_FILLED(data)) {
					wk->getcontentlength = apr_atoi64(data);
				}
			}
			else if (strcmp(child->name, "getcontenttype") == 0) {
				data = (char *) divy_xml_get_cdata(child, p, 1);
				wk->getcontenttype = data;
			}
			else if (strcmp(child->name, "lastmodifier") == 0) {
				data = (char *) divy_xml_get_cdata(child, p, 0);	/* white space は除去しない */
				wk->lastmodifier = data;
			}
			else if (strcmp(child->name, "getlastmodified") == 0) {
				data = (char *) divy_xml_get_cdata(child, p, 1);
				if (IS_FILLED(data)) {
					wk->getlastmodified = apr_atoi64(data);
				}
			}
			else if (strcmp(child->name, "deletion") == 0) {
				data = (char *) divy_xml_get_cdata(child, p, 1);
				if (IS_FILLED(data)) {
					wk->deletion = apr_atoi64(data);
				}
			}
			else {
				/* 無視 */
			}
		}

		/* 値が正しく取得できていたものだけをリストに繋げる */
		if (IS_FILLED(wk->displayname)) {
			if (*del_res == NULL) {
				*del_res = d = wk;
			}
			else {
				d->next = wk;
				d = d->next;
			}
		}
	}

	return 0;
}

/**
 * glist が示す自動削除対象リストから自動設定用ファイルを生成する.
 */
static int _save_glist_xml(request_rec *r, divy_autodel_glist *glist)
{
	apr_status_t rv;
	apr_pool_t *p   = r->pool;
	apr_file_t *fd  = NULL;
	divy_sbuf *sbuf = NULL;
	char *fconfig   = _build_glist_configpath(p, dav_divy_get_root_uri(r));
	char *tempfile, *backfile;

	/* glist がNULL、つまり自動削除対象グループが1つもなければ、
	 * 古いコンフィグをリネームして終了する */
	if (glist == NULL) {
		backfile = apr_psprintf(p, "%s.bak", fconfig);
		rv = apr_file_rename(fconfig, backfile, p);
		if (rv != APR_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to backup old autodelete xml file. (file = %s, code = %d)",
					fconfig, rv);
			return 1;
		}

		return 0;
	}

	/* 以下はglist が存在する場合 */

	/* 一時ファイル名を作成する */
	tempfile = apr_psprintf(p, "%s.XXXXXX", fconfig);
	rv = apr_file_mktemp(&fd, tempfile, APR_WRITE | APR_CREATE | APR_TRUNCATE | APR_BINARY, p);
	if (rv != APR_SUCCESS || fd == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to open temp file.(code = %d)", rv);
	}

	divy_sbuf_create(p, &sbuf, 1024);
	divy_sbuf_append(sbuf,
			"<?xml version=\"1.0\" encoding=\"utf-8\" standalone=\"yes\"?>\n"
			"<autodelete>");

	for (; glist != NULL; glist = glist->next) {
		divy_sbuf_append(sbuf,
				"<autodeletefolder>"
				"<groupid>");
		divy_sbuf_append(sbuf, glist->groupid);
		divy_sbuf_append(sbuf,
				"</groupid>");
		if (glist->spaction & DIVY_AUTODEL_ACTION_DELFOLDER) {
			divy_sbuf_append(sbuf,
				"<specialaction><deletefolder/></specialaction>");
		}
		else {
			divy_sbuf_append(sbuf,
				"<specialaction/>");
		}
		divy_sbuf_append(sbuf,
				"<preservationperiod>");
		divy_sbuf_append(sbuf, apr_psprintf(p, "%d", glist->pperiod));
		divy_sbuf_append(sbuf,
				"</preservationperiod>"
				"</autodeletefolder>\n");

		rv = apr_file_write_full(fd, divy_sbuf_tostring(sbuf), divy_sbuf_getlength(sbuf), NULL);
		if (rv != APR_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to write autodelete xml file (file = %s, code = %d)",
					tempfile, rv);
			/* 後始末 */
			(void) apr_file_close(fd);
			(void) apr_file_remove(tempfile, p);
			return 1;
		}
		divy_sbuf_clear(sbuf);
	}

	divy_sbuf_append(sbuf,
			"</autodelete>");
	rv = apr_file_write_full(fd, divy_sbuf_tostring(sbuf), divy_sbuf_getlength(sbuf), NULL);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to write autodelete xml file (file = %s, code = %d)",
				tempfile, rv);
		/* 後始末 */
		(void) apr_file_close(fd);
		(void) apr_file_remove(tempfile, p);
		return 1;
	}
	(void) apr_file_close(fd);

	/* 古いファイルをリネームしてバックアップしておく */
	backfile = apr_psprintf(p, "%s.bak", fconfig);
	rv = apr_file_rename(fconfig, backfile, p);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to backup old autodelete xml file. (file = %s, code = %d)",
				fconfig, rv);
		(void) apr_file_remove(tempfile, p);
		return 1;
	}

	/* 正式ファイル名にリネームする */
	rv = apr_file_rename(tempfile, fconfig, p);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to rename autodelete xml file. (file = %s, code = %d)",
				tempfile, rv);
		(void) apr_file_remove(tempfile, p);
		return 1;
	}

	return 0;
}

/**
 * del_res が示す削除済みリストから自動削除済みXMLファイルを生成する.
 * (note)
 *   delperiod が0 ならば、自動削除済みXMLファイルは削除します!!
 *
 * @param r request_rec *
 * @param del_res divy_autodel_dresource * 削除済みファイル・フォルダ一覧情報
 * @param groupid const char * 対象グループID
 * @param delperiod int 自動削除履歴の保持日数(日)
 * @param now apr_time_t 削除を行っている時刻
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _save_deletedlist_xml(request_rec *r, divy_autodel_dresource *del_res,
                                 const char *groupid, int delperiod, apr_time_t now)
{
	apr_status_t rv;
	apr_pool_t *p   = r->pool;
	apr_file_t *fd  = NULL;
	divy_sbuf *sbuf = NULL;
	char *fconfig   = _build_dellist_configpath(p, dav_divy_get_root_uri(r), groupid);
	char *tempfile, *backfile;
	divy_autodel_dresource *old_del_res, *dres, *prev;
	apr_time_t deltime;

	/* 保持日数が0であれば消去する。また、以下のXML作成処理もやらない */
	if (delperiod == 0) {
		/* 無いかどうかに関わらず実施するのでエラーは見ない */
		(void) apr_file_remove(fconfig, p);

		return 0;	/* 成功とする */
	}

	/* (1) 古い自動削除済みファイルリストを取得する */
	(void) _parse_deletedlist(r, groupid, &old_del_res);

	/* 古い自動削除済みファイルリストから古くなってしまったエントリを削り落とす */
	deltime = now - apr_time_from_sec(delperiod * 24 * 60 * 60);
	prev = NULL;
	for (dres = old_del_res; dres != NULL; dres = dres->next) {
		if (dres->deletion < deltime) {
			if (prev == NULL) {
				old_del_res = dres->next;
			}
			else {
				prev->next = dres->next;
			}
		}
		else {
			prev = dres;
		}
	}

	/* 削除したばかりのリソースエントリの削除日付にnow を入れる */
	if (del_res != NULL) {
		prev = NULL;
		for (dres = del_res; dres != NULL; dres = dres->next) {
			dres->deletion = now;
			if (dres->next == NULL) {
				prev = dres;
				break;
			}
		}
		if (prev != NULL) {
			prev->next = old_del_res;	/* 旧データを末尾に追加 */
		}
	}
	else {
		del_res = old_del_res;
	}

	/* (2) 自動削除済みファイルリストの生成 */

	/* 一時ファイル名を作成する */
	tempfile = apr_psprintf(p, "%s.XXXXXX", fconfig);
	rv = apr_file_mktemp(&fd, tempfile, APR_WRITE | APR_CREATE | APR_TRUNCATE | APR_BINARY, p);
	if (rv != APR_SUCCESS || fd == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to open temp file.(code = %d)", rv);
	}

	divy_sbuf_create(p, &sbuf, 1024);
	divy_sbuf_append(sbuf,
			"<?xml version=\"1.0\" encoding=\"utf-8\" standalone=\"yes\"?>\n"
			"<autodeletedlist>");

	for (; del_res != NULL; del_res = del_res->next) {
		divy_sbuf_append(sbuf,
				"<filelist>"
				"<displayname>");
		divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, del_res->displayname, DIVY_XML_T2T_QUOTE));
		divy_sbuf_append(sbuf,
				"</displayname>"
				"<getcontentlength>");
		divy_sbuf_append(sbuf, apr_psprintf(p, "%"APR_INT64_T_FMT, del_res->getcontentlength));
		divy_sbuf_append(sbuf,
				"</getcontentlength>"
				"<getcontenttype>");
		divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, del_res->getcontenttype, DIVY_XML_T2T_QUOTE));
		divy_sbuf_append(sbuf,
				"</getcontenttype>"
				"<lastmodifier>");
		divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, del_res->lastmodifier, DIVY_XML_T2T_QUOTE));
		divy_sbuf_append(sbuf,
				"</lastmodifier>"
				"<getlastmodified>");
		divy_sbuf_append(sbuf, apr_psprintf(p, "%"APR_INT64_T_FMT, del_res->getlastmodified));
		divy_sbuf_append(sbuf,
				"</getlastmodified>"
				"<deletion>");
		divy_sbuf_append(sbuf, apr_psprintf(p, "%"APR_INT64_T_FMT, del_res->deletion));
		divy_sbuf_append(sbuf,
				"</deletion>"
				"</filelist>\n");

		/* ファイルに吐き出す */
		rv = apr_file_write_full(fd, divy_sbuf_tostring(sbuf), divy_sbuf_getlength(sbuf), NULL);
		if (rv != APR_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to write autodelete xml file (file = %s, code = %d)",
					tempfile, rv);
			/* 後始末 */
			(void) apr_file_close(fd);
			(void) apr_file_remove(tempfile, p);
			return 1;
		}
		divy_sbuf_clear(sbuf);
	}
	divy_sbuf_append(sbuf,
			"</autodeletedlist>");
	rv = apr_file_write_full(fd, divy_sbuf_tostring(sbuf), divy_sbuf_getlength(sbuf), NULL);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to write autodelete xml file (file = %s, code = %d)",
				tempfile, rv);
		/* 後始末 */
		(void) apr_file_close(fd);
		(void) apr_file_remove(tempfile, p);
		return 1;
	}
	(void) apr_file_close(fd);

	/* 古いファイルをリネームしてバックアップしておく(無いかもしれないが気にしない) */
	backfile = apr_psprintf(p, "%s.bak", fconfig);
	(void) apr_file_rename(fconfig, backfile, p);

	/* 正式ファイル名にリネームする */
	rv = apr_file_rename(tempfile, fconfig, p);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to rename autodelete xml file. (file = %s, code = %d)",
				tempfile, rv);
		(void) apr_file_remove(tempfile, p);
		return 1;
	}

	return 0;
}

