/**
 * $Id$
 *
 * Autoindex 及びブラウザインターフェースを実現する構造体、関数の宣言＆定義
 */
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_hash.h"
#include "apr_time.h"
#include "apr_network_io.h"
#include "apr_uuid.h"
#include "apreq2/apreq_cookie.h"

#include <libxml/xmlmemory.h>
#include <libxml/debugXML.h>
#include <libxml/HTMLtree.h>
#include <libxml/xmlIO.h>
#include <libxml/xinclude.h>
#include <libxml/catalog.h>
#include <libxml/parserInternals.h>
#include <libxslt/xslt.h>
#include <libxslt/xsltInternals.h>
#include <libxslt/transform.h>
#include <libxslt/xsltutils.h>

#include "httpd.h"
#include "http_protocol.h"
#include "mod_dav_tf.h"
#include "liveprop.h"
#include "repos.h"

#include "util_common.h"
#include "util.h"
#include "util_auth.h"
#include "util_ml.h"
#include "util_saml.h"
#include "tf_autoindex.h"
#include "tf_confirmreading.h"
#include "tf_rdbo.h"
#include "tf_rdbo_sysmsg.h"
#include "tf_rdbo_user.h"
#include "tf_rdbo_group.h"
#include "tf_rdbo_box.h"
#include "tf_folder.h"
#include "tf_validator.h"
#include "tf_plugin.h"
#include "tf_extmap.h"
#include "tf_memcache.h"

APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * XMLバッファの内容をXMLファイルに書き出す閾値(byte)
 * 小さくし過ぎると頻繁にファイルI/Oが発生し、性能が劣化するでしょう。
 */
#define XMLBUF_BOUNDARY_SIZE	8192

/*------------------------------------------------------------------------------
  Declare and Define structure
  ----------------------------------------------------------------------------*/
typedef struct __divy_strconvertion	divy_strconvertion;
typedef struct __divy_autoindex_screen	divy_autoindex_screen;
typedef struct __divy_outputxml		divy_outputxml;

/**
 * autoindex 画面表示用Screen構造体
 *
 * autoindex リクエストデータ及びレスポンスデータを保持しています。
 */
struct __divy_autoindex_screen {
	/* リスクエストパラメータ */
	char *cmd;		/* コマンドパラメータ */
	char *crsid;		/* 暗号化されたrsid */
	int ipageno;		/* リクエストされたページ番号 */
	char *ticket;		/* チケット文字列 */

	/* Apache 環境変数 */
	const char *encoding;		/* 出力エンコーディング */
	const char *s_stylesheet;	/* 概要表示用スタイルシートファイル名 */
	const char *d_stylesheet;	/* 詳細表示用スタイルシートファイル名 */

	/* 拡張系 */
	int isRow;					/* データの加工を施していない(日付とか) */
	const char *extend_css;		/* 独自のCSS(xsltのスタイルシートではない */
	int with_rss;				/* RSSへのリンクを一緒に表示できるか */
	int require2fa;				/* 二段階認証が必要 */

	/* 算出されたパラメータ */
	divy_rdbo_resource *r_rdb_r;	/* リクエストパラメータ       */
	divy_uri_spec *r_u_spec;	/* リクエストuri の種別       */
	int found_extra_data;		/* 続きページがあるかどうか   */
	apr_int32_t offset; 		/* 表示開始位置(初期値は0)    */
	apr_int32_t limit;		/* 画面表示件数(mobileのみ)   */
	char *rsid;			/* 復号されたrsid             */
	const char *seq;		/* キャッシュ防止用パラメータ */

	/* レスポンスパラメータ */
	divy_rdbo_resource *p_rdb_r;	/* 親リソース               */
	divy_rdbo_resource *c_rdb_r;	/* カレントフォルダリソース */
	divy_rdbo_resource *m_rdb_r;	/* メンバリソース           */

	/* おおもととなったリクエスト情報 */
	request_rec *r;
};

/**
 * _replace_specialstr で使用する特殊文字列と置換先文字列の関係を表す構造体
 */
struct __divy_strconvertion {
	const char *from;
	const char *to;
};

/**
 * divy_outputxml で使用される列挙型。
 * どのような形式でXMLデータを保持しているかどうかを示す。
 */
enum {
	DIVY_XMLOUTPUT_TYPE_UNKNOWN= 0,	/* (未定義)           */
	DIVY_XMLOUTPUT_TYPE_STRING,	/* 文字列形式で保持   */
	DIVY_XMLOUTPUT_TYPE_FILE	/* ファイル形式で保持 */
};

/**
 * screen から生成されたXMLデータを保持する構造体
 */
struct __divy_outputxml {
	/* どのような形式でXMLデータを保持しているか */
	int type;

	/* 出力データ
	 * type により示す値の内容が異なります。*/
	union {
		divy_sbuf *sbuf; /* DIVY_XMLOUTPUT_TYPE_STRING の場合: 文字列バッファ */
		char *filepath;  /* DIVY_XMLOUTPUT_TYPE_FILE の場合: ファイルパス */
	} d;
};

/*------------------------------------------------------------------------------
  Define array
  ----------------------------------------------------------------------------*/
/* _encipher_rsid, _encipher_rsid 用 マッピングテーブル */
static const char _mptbl0[] = { '7','q','4','2','Z','8','a','1','E','v' }; /* 重複不可 */
static const char _mptbl1[] = { 'G','i','6','3','1','f','w','d','b','7' };
static const char _mptbl2[] = { 'T','e','a','5','f','i','l','3','J','S' };
static const char _mptbl3[] = { '7','X','C','p','4','o','n','8','1','5' };
static const char _mptbl4[] = { 'A','v','M','3','5','9','L','k','F','1' };
static const char _mptbl5[] = { '4','V','n','6','2','Y','8','e','N','j' };
static const char _mptbl6[] = { 'o','B','Z','9','3','T','7','O','p','Q' };
static const char _mptbl7[] = { 'R','0','H','f','s','2','b','8','7','3' };
static const char _mptbl8[] = { 'V','l','9','2','M','c','z','5','S','u' };
static const char _mptbl9[] = { 'Y','q','3','o','1','6','a','C','K','0' };
static const char * _mptbls[] = { _mptbl0, _mptbl1, _mptbl2, _mptbl3, _mptbl4,
				  _mptbl5, _mptbl6, _mptbl7, _mptbl8, _mptbl9 };

/* _encipher_rsid, _encipher_rsid 用 シフトテーブル */
static const int _sftbl[] = { 7,4,2,8,1,5,6,3,9,2,5,9,4,6,9,1,7,5,2 };
static const char _i2a[]  = { '0','1','2','3','4','5','6','7','8','9' };

/*
 * _replace_specialstr 用 特殊文字列とその文字列の置換先文字列のマッピングテーブル
 */
static const divy_strconvertion _special2normal[] = {
	{"\xe3\x88\xb1", "(\xe6\xa0\xaa)"},	/* (株) */
	{"\xc3\x97", "X"},			/* X    */
	{"\xe3\x88\xb2", "(\xe6\x9c\x89)"},	/* (有) */
	{"\xe3\x88\xb9", "(\xe4\xbb\xa3)"},	/* (代) */
	{NULL, NULL}
};

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static divy_autoindex_screen * _create_autoindex_screen(apr_pool_t *p,
						request_rec *r,
						divy_rdbo_resource *rdb_r);
static dav_error * _get_autoindex_data(apr_pool_t *p,
					divy_autoindex_screen *screen);
static dav_error * _get_autoindex_startuppage(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r);
static dav_error * _get_autoindex_groupfolder(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r);
static dav_error * _get_autoindex_utilityfolder(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r);
static dav_error * _get_autoindex_updateclient(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r);
static dav_error * _get_autoindex_sysmsg(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r);
static dav_error * _get_autoindex_list(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r);
static dav_error * _get_autoindex_mobile_list(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r);
static dav_error * _screen2xml(apr_pool_t *p,
					divy_fstorage_t *fstorage,
					divy_autoindex_screen *screen,
					divy_outputxml *outputxml);
static dav_error * _xml2html(apr_pool_t *p,
					divy_fstorage_t *fstorage,
					divy_autoindex_screen *screen,
					const char *stylesheet,
					divy_outputxml *outputxml,
					const char **filepath);
static dav_error* _samlxml2html(apr_pool_t *p, divy_samlinfo_iscreen *screen,
								const char *stylesheet, char **filepath);
static char * _get_stylesheet_path(apr_pool_t *wp,
					const char *prefix,
					request_rec *r);
static const char * _encipher_rsid(apr_pool_t *p, int seed, const char *rsid);
static const char * _decipher_rsid(apr_pool_t *p, const char *cipherrsid);
static char * _replace_specialstr(apr_pool_t *p, char *str);
static char * _encode_property(apr_pool_t *p, const char *encoding, char *src, char *failedconv_str);
static void _build_response_url(apr_pool_t *p, divy_sbuf *sbuf, divy_infotype infotype,
                                divy_rdbo_resource *rdb_r,
                                const char *seq, int is_hideuri, int is_downloadonly);
static int _remove_browser_session(request_rec *r);

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * Autoindex 処理を行う.
 *
 */
DIVY_DECLARE(dav_error *) divy_do_autoindex(const dav_resource *resource,
										dav_postdata *data, ap_filter_t *output)
{
	apr_status_t rv;
	dav_error *err            = NULL;
	apr_pool_t *wp            = NULL;
	char *stylesheet          = NULL;
	request_rec *r            = resource->info->r;
	apr_pool_t *p             = resource->pool;
	apr_pool_t *log_p         = r->pool;
	divy_rdbo_resource *rdb_r = resource->info->rdb_r;
	const char *filepath      = NULL;
	divy_autoindex_screen *screen = NULL;
	divy_fstorage_t *fstorage = NULL;
	int ret;
	divy_outputxml outputxml;
	apr_uint32_t propflag     = 0;

	TRACE(p);

	apr_table_addn(r->err_headers_out, "Cache-Control", "no-store no-cache, must-revalidate, post-check=0, precheck=0, max-age=0");
	apr_table_addn(r->err_headers_out, "Pragma", "no-cache");

	/* URIのパース */
	if (rdb_r->u_spec == NULL) {
		divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &rdb_r->u_spec);
	}

	/* outputxml のtype を初期化 */
	outputxml.type = DIVY_XMLOUTPUT_TYPE_UNKNOWN;

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 短縮URLへのリクエストの場合
	 * ここに短縮URLで来るケースはBOXではない（パラメータにcmd=boxがない）
	 * だけである。
	 * この関数では、必ず短縮URLの処理を継続はできないため、必ず
	 * 正規URLでのリダイレクト指示をします
	 */
	if (rdb_r->u_spec->infotype == DIVY_INFOTYPE_shorten) {

		divy_rdbo_resource *tmp_rdb_r = NULL;

		/* 正しいURLを調べる。ない場合はNOTFOUND扱い */
	 	if ((tmp_rdb_r = divy_convert_shorten(p, r)) == NULL) {
			err = dav_new_error(log_p, HTTP_NOT_FOUND, 0, 0, "");
			goto cleanup_autoindex_resource;
		}

		if (IS_FILLED(r->user)) {
			apr_table_setn(r->headers_out, "Location", tmp_rdb_r->uri);
		}
		else {
			apr_table_setn(r->headers_out, "Location", r->uri);
		}
		err = dav_new_error(log_p, HTTP_MOVED_TEMPORARILY, 0, 0, "");
		goto cleanup_autoindex_resource;

	} 

	/* screen の生成 */
	screen = _create_autoindex_screen(wp, r, rdb_r);
	if (screen == NULL) {
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_autoindex_resource;
	}

	/* 論理ストレージのopen */
	ret = divy_fstorage_open(r, wp, &fstorage);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to open storage.(uri = %s, userid = %s)",
			r->uri, divy_get_userid(r));
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_autoindex_resource;
	}

	/* モバイルURIへのリクエストの場合 */
	if (rdb_r->u_spec->infotype == DIVY_INFOTYPE_mobile) {

		/* ダウンロードするかどうか */
		if (IS_FILLED(screen->cmd) && strcmp(screen->cmd, DIVY_CMD_DOWNLOAD) == 0) {
			const char *dlfilepath;
			dav_resource _resource     = { 0 };
			dav_resource_private _info = { 0 };
			divy_uri_spec *_u_spec     = NULL, *swap_u_spec = NULL;

			/* ダウンロードファイルのプロパティを取得(状態・属性プロパティを含む) */
			rdb_r->rsid = screen->rsid;	/* 入れ替え */

			if (divy_support_extenduserstatus(r)) {
				propflag = DIVY_GET_PROP_resourcestate;	/* 状態プロパティも必要(かもしれない) */
			}

			if (divy_rdbo_get_property_by_rsid(r, rdb_r, propflag)) {
				ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get resource by rsid.(rsid = %s)",
					screen->rsid);
				err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				goto cleanup_autoindex_resource;
			}

			/* (note) rdb_r->u_spec は潰してはならない. 後で使うので */
			swap_u_spec = rdb_r->u_spec;	/* 一時退避 */
			rdb_r->u_spec = _u_spec;
			divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &rdb_r->u_spec);

			/* rdb_r->uri の最小アクセス権限のチェック */
			err = divy_validate_minimum_access_guarantee(r, rdb_r->u_spec);
			if (err != NULL) {
				if (fstorage != NULL) (void) divy_fstorage_close(fstorage);
				if (wp != NULL) {
					apr_pool_destroy(wp);
					 wp = NULL;
				}
				return err;
			}
			/* アクセス権のフルチェック */
			_resource.exists      = 1;
			_resource.info        = &_info;
			_resource.info->rdb_r = rdb_r;
			_resource.collection  = (rdb_r->resourcetype == DIVY_TYPE_COLLECTION) ? 1 : 0;

			err = divy_validate_request_resource(r, &_resource);
			if (err != NULL) {
				if (fstorage != NULL) (void) divy_fstorage_close(fstorage);
				if (wp != NULL) {
					apr_pool_destroy(wp);
				 	wp = NULL;
				}
				return err;
			}
			rdb_r->u_spec = swap_u_spec;	/* 元に戻す */

			/* ステータスを設定する */
			r->status = HTTP_OK;

			/* ファイルパスの作成 */
			dlfilepath = divy_make_fullpath(fstorage, rdb_r->physicalpath);

			/* ファイルをストリームに流す */
			ret = divy_sendfile2stream(r, output, rdb_r->getcontentlength, dlfilepath);
			if (ret != OK) {
				err = dav_new_error(log_p, ret, 0, 0, "");
			}

			/* 論理ストレージを閉じる */
			if (fstorage != NULL) (void) divy_fstorage_close(fstorage);
			if (wp != NULL) {
				apr_pool_destroy(wp);
				wp = NULL;
			}
			return err;	/* (note) 本質的にダウンロードとは異なるので、ロジックを共用してはならない */
		}

		/* 詳細表示 or 概要表示の決定とスタイルシートパスの生成 */
		if (IS_FILLED(screen->cmd) && strcmp(screen->cmd, DIVY_CMD_SHOW_DETAIL) == 0) {

			/* ブラウザから直接モバイル画面を開いた場合 */
			if (IS_EMPTY(screen->d_stylesheet)) {
				ERRLOG1(log_p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The stylesheet for mobile is EMPTY."
					"This request may have been send from other than "
					"the mobile terminal.(User-Agent = %s)",
					dav_divy_get_user_agent(r));
				err = dav_new_error(log_p, HTTP_BAD_REQUEST, 0, 0, "");
				goto cleanup_autoindex_resource;
			}
			stylesheet = _get_stylesheet_path(wp, screen->d_stylesheet, r);
		}
		else {
			stylesheet = _get_stylesheet_path(wp, screen->s_stylesheet, r);
		}
	}
	/* 一般的なブラウザからのリクエスト */
	else {
		stylesheet = _get_stylesheet_path(wp, screen->s_stylesheet, r);
	}

	/* screen データの取得 */
	err = _get_autoindex_data(wp, screen);
	if (err) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get autoindex data.");
		goto cleanup_autoindex_resource;
	}

	/* screen -> XMLファイル */
	err = _screen2xml(wp, fstorage, screen, &outputxml);
	if (err) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to build xml string.");
		goto cleanup_autoindex_resource;
	}

	/* XMLファイル -> HTMLファイル */
	err = _xml2html(wp, fstorage, screen, stylesheet, &outputxml, &filepath);
	if (err) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to convert xml to html.");
		goto cleanup_autoindex_resource;
	}

	/* ステータスを設定する */
	r->status = HTTP_OK;
	if (IS_FILLED(screen->encoding)) {
		ap_set_content_type(r, apr_psprintf(wp,
				"text/html; charset=\"%s\"", screen->encoding));
	}
	else {
		ap_set_content_type(r, "text/html; charset=\"utf-8\"");
	}

	/* HTMLファイルをストリームに流す */
	ret = divy_sendfile2stream(r, output, APR_INT64_C(-1), filepath);
	if (ret != OK) {
		err = dav_new_error(log_p, ret, 0, 0, "");
		goto cleanup_autoindex_resource;
	}

cleanup_autoindex_resource:

	/* 論理ストレージを閉じる */
	if (fstorage != NULL) {
		(void) divy_fstorage_close(fstorage);
	}

	/* HTMLファイルを削除する */
	if (IS_FILLED(filepath)) {
		rv = apr_file_remove(filepath, p);
		if (!APR_STATUS_IS_ENOENT(rv) && rv != APR_SUCCESS) {
			ERRLOG2(log_p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to remove temporary html file for autoindex."
				"(filepath = %s, code = %d)", filepath, rv);
		}
	}

	/* XMLファイルを削除する */
	if (outputxml.type == DIVY_XMLOUTPUT_TYPE_FILE && IS_FILLED(outputxml.d.filepath)) {
		rv = apr_file_remove(outputxml.d.filepath, p);
		if (!APR_STATUS_IS_ENOENT(rv) && rv != APR_SUCCESS) {
			ERRLOG2(log_p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to remove temporary xml file for autoindex."
				"(filepath = %s, code = %d)", outputxml.d.filepath, rv);
		}
	}

	/* サブプールを破棄する */
	if (wp != NULL) {
		apr_pool_destroy(wp);
	 	wp = NULL;
	}

	return err;
}

/**
 * GET 要求に対するAutoIndex 機能をアクティブにするかどうかを判定する.
 *
 */
DIVY_DECLARE(int) divy_enable_autoindex(request_rec *r, const dav_resource *resource)
{
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	divy_uri_spec *u_spec;

	if (resource == NULL || resource->info->rdb_r == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"resource or rdb_r is NULL.");
		return 0;
	}

	if (resource->info->rdb_r->u_spec == NULL) {
		divy_parse_uri(r->pool, dav_divy_get_root_uri(r),
				resource->uri, &u_spec);
		resource->info->rdb_r->u_spec = u_spec;
	}
	else {
		u_spec = resource->info->rdb_r->u_spec;
	}

	/* 旧スタイルのAutoindexアクセスも許可する */
	if (resource->collection &&
	    conf->autoindex != DIVY_AUTOINDEX_OFF &&
	    apr_table_get(r->subprocess_env, DIVY_APENV_STYLESHEET) != NULL &&
	    IS_FILLED(conf->stylesheetroot) &&
	    (u_spec->infotype == DIVY_INFOTYPE_mobile          ||
		 u_spec->infotype == DIVY_INFOTYPE_fileviewer      ||
		 u_spec->infotype == DIVY_INFOTYPE_root            ||
	     u_spec->infotype == DIVY_INFOTYPE_user_e          ||
	     u_spec->infotype == DIVY_INFOTYPE_user_trash      ||
	     u_spec->infotype == DIVY_INFOTYPE_user_e_regular  ||
	     u_spec->infotype == DIVY_INFOTYPE_group           ||
	     u_spec->infotype == DIVY_INFOTYPE_group_trash     ||
	     u_spec->infotype == DIVY_INFOTYPE_group_e         ||
	     u_spec->infotype == DIVY_INFOTYPE_group_e_regular ||
	     u_spec->infotype == DIVY_INFOTYPE_utility         || 
	     u_spec->infotype == DIVY_INFOTYPE_u_update        ||
	     u_spec->infotype == DIVY_INFOTYPE_u_msg           ||
	     u_spec->infotype == DIVY_INFOTYPE_ticket)) {
		return 1;	/* 許可する */
	}

	if (u_spec->infotype == DIVY_INFOTYPE_shorten) 
		return 1;

	return 0;
}

/**
 * ログイン画面が有効にするか判定する
 *
 */
DIVY_DECLARE(int) divy_enable_login(request_rec *r, const dav_resource *resource)
{
	divy_uri_spec *u_spec = resource->info->rdb_r->u_spec;

	if (resource == NULL || resource->info->rdb_r == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"resource or rdb_r is NULL.");
		return 0;
	}

	if (u_spec == NULL)
		return 0;

	/* sessionがOnでなければならない */
	if (divy_support_session(r)
			&& u_spec->infotype == DIVY_INFOTYPE_login ) {
		return 1; /* 許可する */
	}

	return 0;

}

/**
 * 二要素認証のコードを入力する画面を出力する
 *
 */
DIVY_DECLARE(dav_error *)
divy_do_2FA(const dav_resource *resource, ap_filter_t *output)
{
	request_rec *r = resource->info->r;
	apr_pool_t *pool = resource->info->r->pool;
	dav_error *err = NULL;
	char *filepath = NULL;
	divy_samlinfo_iscreen *iscreen;
	dav_divy_dir_conf *dconf  = dav_divy_get_dir_config(r);

	char *stylesheet = NULL;
	char *xsl_path = "default_2FA.xsl";
	int ret;

	TRACE(pool);

	iscreen = apr_pcalloc(pool, sizeof(divy_samlinfo_iscreen));
	iscreen->r = r;
	iscreen->actionurl = r->uri;
	iscreen->browserstyle = dconf->tfbrowserstyle;
	iscreen->uniqueid = dav_divy_replace_str(pool,
				apr_table_get(r->subprocess_env, "UNIQUE_ID"), "@", "-", NULL);

	stylesheet = _get_stylesheet_path(pool, xsl_path, r);
	if (IS_EMPTY(stylesheet)) {
		/* スタイルシートがない */
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_2FA;
	}

	err = _samlxml2html(pool, iscreen, stylesheet, &filepath);

	/* セキュリティ Content Security Policy (CSP) */
	char *policy_str = apr_psprintf(pool, "default-src 'self'; "
			"script-src 'self' 'nonce-%s'", iscreen->uniqueid);
	apr_table_addn(r->err_headers_out, "Content-Security-Policy", policy_str);

	ap_set_content_type(r, "text/html; charset=\"utf-8\"");
	ret = divy_sendfile2stream(r, output, APR_INT64_C(-1), filepath);
	if (ret != OK) {
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

cleanup_2FA:

	return err;
}

/**
 * SAML選択画面を出力する
 * SAMLの場合はこれが一番始めの画面になります
 */
DIVY_DECLARE(dav_error *)
divy_do_saml_idp_select(const dav_resource *resource, ap_filter_t *output)
{
	apr_status_t rv;
	request_rec *r = resource->info->r;
	apr_pool_t *pool = resource->info->r->pool;
	apr_hash_t *mime_map  = NULL;
	const divy_cmap_elem *map_e = NULL;
	dav_error *err = NULL;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	const char *root_uri = dav_divy_get_root_uri(r);
	char *stylesheet = NULL;
	char *filepath = NULL;
	char *brandname = NULL;
	int ret;
	char *next = NULL;
	char *statuscode = NULL;
	int need_unlock = 0, length;
	apr_hash_t *parsed_h 	  = NULL;
	divy_linkedlist_t *ll	  = NULL;
	divy_pfile_t *pfile       = NULL;
	apr_file_t *fd            = NULL;
	apr_os_file_t osfd;
	const char *params[]      = { NULL };
	divy_fstorage_t *fstorage = NULL;
	apr_uuid_t uuid;
    char char_uuid[APR_UUID_FORMATTED_LENGTH + 1];
	char *token;

	xmlNsPtr ns_p			  = NULL;
	xmlDocPtr doc_p           = NULL;
	xmlDocPtr html_p          = NULL;
	xmlNodePtr root_ele		  = NULL;
	xmlNodePtr tf_ele		  = NULL;
	xsltStylesheetPtr cur     = NULL;
	xmlCharEncodingHandlerPtr enc_p = NULL;
	xmlOutputBufferPtr obuf   = NULL;

	TRACE(pool);
	/*
	 * メタデータが準備されていない場合はエラーとする
	 */
	if (conf->idpmeta == NULL || IS_EMPTY(conf->idpmeta->filepath)) {
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "IdP meta data not found. Please check config file.");
		goto cleanup_saml_xml2html;
	}

	dav_divy_parse_urlparam(pool, r->args, &parsed_h);
	if (parsed_h != NULL) {
		/* nextがパラメーターを調べてあれば記録 */
		ll = apr_hash_get(parsed_h, "next", APR_HASH_KEY_STRING);
		if (ll != NULL) {
			next = apr_pstrdup(pool, ll->val);
		}

		/* st(ステータスコード）があれば記録 */
		ll = apr_hash_get(parsed_h, "st", APR_HASH_KEY_STRING);
		if (ll != NULL) {
			statuscode = apr_pstrdup(pool, ll->val);
		}
	}

	/* nextがない場合はTOPフォルダとする */
	if (next == NULL) {
		next = (char *)dav_divy_get_root_uri(r);
	}
	/*
	 * SAML選択画面はユーザがまだログインしていない状態の為r->userに値はない
	 * その為ゲストユーザ扱いとして更にユーザ情報をキャッシュします
	 * これをおこなわないとdivy_fstorage_openができない
	 */
	r->user = DIVY_GUEST_ID;
	(void)divy_rdbo_cache_userinfo(r, 1 /* no cache */);

	/* 論理ストレージのopen */
	ret = divy_fstorage_open(r, pool, &fstorage);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG2(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to open storage.(uri = %s, userid = %s)",
			r->uri, divy_get_userid(r));
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_saml_xml2html;
	}

	/*
	 * HTML出力用一時ファイルの作成
	 */
	ret = divy_pfile_mkfixedtemp(fstorage, pool, DIVY_TMP_FILE_TEMPLATE_NAME, &pfile);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG2(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to make a temporary file.(uri = %s, "
			"userid = %s)", r->uri, divy_get_userid(r));
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_saml_xml2html;
	}

	/* 一時ファイルを開く */
	filepath = apr_pstrdup(pool, divy_pfile_get_fullpath(pfile));
	rv = apr_file_mktemp(&fd, (char *) filepath,
			APR_WRITE | APR_CREATE | APR_TRUNCATE | APR_BINARY, pool);
	if (rv != APR_SUCCESS || fd == NULL) {
		ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to open temp file.(code = %d)", rv);
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_saml_xml2html;
	}

	/* OSのファイルディスクリプタを取得する */
	rv = apr_os_file_get(&osfd, fd);
	if (rv != APR_SUCCESS) {
		ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to get OS fd. (code = %d)", rv);
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_saml_xml2html;
	}

	/* ロックを掛ける */
	rv = apr_thread_mutex_lock(autoindex_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to lock for open file.(code = %d)", rv);
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_saml_xml2html;
	}
	need_unlock = 1;	/* 兎に角最後にはunlockが必要であることを明示 */

	/* SAML選択画面のスタイルシートを設定する */
	stylesheet = _get_stylesheet_path(pool, "saml_selectidp.xsl", r);

	xmlSubstituteEntitiesDefault(1);
	xmlLoadExtDtdDefaultValue = 1;

	/* IdP META XMLファイルを使って xmlctx_p を生成 */
	doc_p = xmlParseFile(conf->idpmeta->filepath);

	if (doc_p == NULL) {
		ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to parse xml. [SAML IdP file = %s]", conf->idpmeta->filepath);
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_saml_xml2html;
	}

	/*
	 * IdPから入手したXMLMETAデータに独自に渡したい項目をXMLエレメントとして
	 * 追加します。
	 */
	root_ele = xmlDocGetRootElement(doc_p);

	ns_p = xmlNewNs(NULL,
				(xmlChar *)"http://www.teamfile.com/DTD/TF/", (xmlChar*)"TF");
	tf_ele = xmlNewNode(NULL, BAD_CAST "teamfile");
	xmlSetNs(tf_ele, ns_p);
	xmlAddNextSibling(root_ele, tf_ele);

	/* brandname を付加 */
	if (IS_EMPTY(sconf->brandname)) {
		/* デフォルトのプロダクト名を入れる */
		brandname = apr_pstrdup(pool, DIVY_PRODUCT_NAME);
	}
	else {
		brandname = apr_pstrdup(pool, sconf->brandname);
	}
	xmlNodePtr newNode = xmlNewNode(NULL, (xmlChar *)"brandname");
	xmlNodePtr newText = xmlNewText((xmlChar *)dav_divy_escape_xmlstr(pool,
												brandname, DIVY_XML_T2T_QUOTE));
	xmlSetNs(newNode, ns_p);
	xmlAddChild(newNode, newText);
	xmlAddChild(tf_ele, newNode);

	/* 拡張子、Content-Typeマップを読み込む */
	mime_map = divy_read_mimetype_map(r);

	/* ブランド画像を取得 */
	map_e = divy_get_extmapinfo(pool, mime_map,
			"application/x-tf-toppage", NULL);
	if (map_e != NULL && IS_FILLED(map_e->img_path)) {
		newNode = xmlNewNode(NULL, (xmlChar *)"brandimg");
		newText = xmlNewText((xmlChar *)map_e->img_path);

		xmlSetNs(newNode, ns_p);
		xmlAddChild(newNode, newText);
		xmlAddChild(tf_ele, newNode);
	}

	/* ブランド画像を表示するかしないか
	 *
	 * 2.4以前はログイン画面にブランドが表示されていなかった為、特に
	 * 製品名をログインなしでわかる手段はありませんでした
	 * invisiblebrandが設定されると従来通りブランドを表示しません
	 */
	if (conf->tfbrowserstyle & DIVY_BROWSER_STYLE_INVISIBLE_BRAND) {
		newNode = xmlNewNode(NULL, (xmlChar *)"invisiblebrand");
		xmlSetNs(newNode, ns_p);
		xmlAddChild(tf_ele, newNode);
	}

	if (IS_FILLED(statuscode)) {
		newNode = xmlNewNode(NULL, (xmlChar *)"statuscode");
		xmlNodePtr newText = xmlNewText((xmlChar *)dav_divy_escape_xmlstr(pool,
											statuscode, DIVY_XML_T2T_QUOTE));
		xmlSetNs(newNode, ns_p);
		xmlAddChild(newNode, newText);
		xmlAddChild(tf_ele, newNode);
	}

	char *samluri = apr_psprintf(pool, "%s?next=%s",
								divy_build_saml_uri(pool, root_uri), next);

	/*
	 * SAMLが有効になっていると通常ログインするdivy_do_login()は
	 * (urlはloginuriを利用します） 常にSAML選択画面へ移動してしまいます。
	 * tokenパラメーターがあれば通常ログインの画面を表示させるようにしますが
	 * tokenパラメーターはmemcachedで有効期限付きで管理されるようにして
	 * 通常ログイン画面のお気に入り追加を防止します
	 */
	apr_uuid_get(&uuid);
	apr_uuid_format(char_uuid, &uuid);
	divy_saml_save_relaystate(r, char_uuid, IDP_REFRESH_TIME + 5, &token);

	char *loginuri = apr_psprintf(pool, "%s?next=%s&token=%s",
								divy_build_login_uri(pool, root_uri), next,
								token);

	root_ele = xmlDocGetRootElement(doc_p);
	newNode = xmlNewNode(NULL, (xmlChar *)"samluri");
	newText = xmlNewText((xmlChar *)samluri);
	xmlAddChild(newNode, newText);
	xmlAddChild(root_ele, newNode);

	newNode = xmlNewNode(NULL, (xmlChar *)"loginuri");
	newText = xmlNewText((xmlChar *)loginuri);
	xmlAddChild(newNode, newText);
	xmlAddChild(root_ele, newNode);

	/*
	 * XSLT プロセッサを使ってXMLをHTMLに変換
	 */
	cur = xsltParseStylesheetFile((const xmlChar*)stylesheet);
	if (cur == NULL) {
		ERRLOG2(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to parse stylesheetfile.(stylesheet = %s, "
			"filepath = %s)", stylesheet, conf->idpmeta->filepath);
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_saml_xml2html;
	}

	html_p = xsltApplyStylesheet(cur, doc_p, params);
	if (html_p == NULL) {
		ERRLOG2(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to apply stylesheet [SAML].(stylesheet = %s, "
			"filepath = %s)", stylesheet, filepath);
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_saml_xml2html;
	}

	enc_p = xmlGetCharEncodingHandler(XML_CHAR_ENCODING_UTF8);
	obuf = xmlOutputBufferCreateFd(osfd, enc_p);
	length = xsltSaveResultTo(obuf, html_p, cur);
	if (length == -1) {
		ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to save SAML html to file."
			"(filepath = %s, code = -1)", filepath);
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_saml_xml2html;
	}
	xmlOutputBufferFlush(obuf);	/* バッファをフラッシュする */

	/* ステータスを設定する */
	r->status = HTTP_OK;
	ap_set_content_type(r, "text/html");

	/* キャッシュを行わない */
	apr_table_addn(r->headers_out, "Cache-Control", "no-store");
	apr_table_addn(r->err_headers_out, "Cache-Control", "no-store");

	/* 定期的に更新させる see util_saml.h */
	apr_table_setn(r->err_headers_out, "Refresh",
							apr_psprintf(pool, "%d", IDP_REFRESH_TIME));

	/* HTMLファイルをストリームに流す */
	ret = divy_sendfile2stream(r, output, APR_INT64_C(-1), filepath);
	if (ret != OK) {
		err = dav_new_error(pool, ret, 0, 0, "");
		goto cleanup_saml_xml2html;
	}

cleanup_saml_xml2html:

	/* 使用資源を解放 */
	if (obuf != NULL) xmlOutputBufferClose(obuf);
	if (cur != NULL) xsltFreeStylesheet(cur);
	if (html_p != NULL) xmlFreeDoc(html_p);
	if (doc_p != NULL)  xmlFreeDoc(doc_p);

	if (need_unlock) {
		/* ロック後に使用されるのでこのタイミングでクリーンアップできる */
		xsltCleanupGlobals();
		xmlCleanupParser();

		/* ロックを解除 */
		rv = apr_thread_mutex_unlock(autoindex_mutex);
		if (rv != APR_SUCCESS) {
			ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to unlock for open file.(code = %d)", rv);
			/* 続ける */
		}
	}

	/* 一時ファイルを閉じる */
	if (fd != NULL && (rv = apr_file_close(fd)) != APR_SUCCESS) {
		ERRLOG1(pool, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to close tempory file for SAML IdP List."
			"(code = %d)", rv);
		/* 続ける */
	}

	/* 一時ファイルを削除する */
	if (err && filepath) {
		rv = apr_file_remove(filepath, pool);
		if (rv != APR_SUCCESS) {
			ERRLOG2(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to remove temporary file for autoindex."
				"(filepath = %s, code = %d)", filepath, rv);
		}
		filepath = NULL;	/* 無かったことにする */
	}
	return err;
}

/**
 * SAML画面を出力する
 * SAML実行は即IdPへリダイレクトを行います。
 * この画面ではユーザー情報が必要ではないが仕組み上r->userは
 * ゲストユーザーを設定するようにします
 */
DIVY_DECLARE(dav_error *)
divy_do_saml_login(const dav_resource *resource, ap_filter_t *output)
{
	request_rec *r = resource->info->r;
	dav_error *err = NULL;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	apr_pool_t *pool = resource->info->r->pool;
	apr_hash_t *parsed_h = NULL;
	divy_saml_idp_metadata *idpmeta = NULL;
	divy_saml_idp_metadata *foundmeta = NULL;
	divy_samlinfo_iscreen *iscreen = NULL;
	divy_linkedlist_t *ll;
	char *entityid = NULL;
	char *stylesheet = NULL;
	//char *xml = NULL;
	const char* next = NULL;
	char *filepath = NULL;
	int ret;
	//divy_pfile_t *pfile       = NULL;
	apr_file_t *fd            = NULL;
	//apr_os_file_t osfd;
	int need_unlock = 0;
	apr_status_t rv;
	xmlDocPtr doc_p           = NULL;
	xsltStylesheetPtr cur     = NULL;
	xmlDocPtr html_p          = NULL;
	//const char *params[]      = { NULL };
	//xmlCharEncodingHandlerPtr enc_p = NULL;
	//xmlOutputBufferPtr obuf   = NULL;

	apr_table_addn(r->headers_out, "Cache-Control", "no-store");
	apr_table_addn(r->err_headers_out, "Cache-Control", "no-store");
	/*
	 * SAMLのログイン画面はentityidによってどのIdPを利用するかを
	 * 決めます。無い場合はエラーとして処理されます。
	 * 無い場合というケースはほとんどは直リンクです
	 */
	dav_divy_parse_urlparam(pool, r->args, &parsed_h);
	if (parsed_h != NULL) {
		ll = apr_hash_get(parsed_h, "entityid", APR_HASH_KEY_STRING);
		if (ll != NULL) {
			entityid = apr_pstrdup(pool, ll->val);
		}

		ll = apr_hash_get(parsed_h, "next", APR_HASH_KEY_STRING);
		if (ll != NULL) {
			next = apr_pstrdup(pool, ll->val);
		}
	}

	if (entityid != NULL) {
		for (idpmeta = conf->idpmeta; idpmeta != NULL;
												idpmeta = idpmeta->next) {
			if (idpmeta->entityID != NULL &&
							strcmp(idpmeta->entityID, entityid) == 0) {
				foundmeta = idpmeta;	/* 見つかった */
				break;
			}
		}
	}
	else {
		return divy_do_login(resource, output);
	}

	/* 指定されたentityidがなかった場合はログイン画面へリダイレクト */
	if (foundmeta == 0) {
		apr_table_setn(r->headers_out, "Location",
			divy_build_login_uri(pool, dav_divy_get_root_uri(r)));

		ERRLOG1(pool, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
				"Not found entityID. Please check IdP metadata."
				"(entityid=%s)", entityid);

		return dav_new_error(pool, HTTP_MOVED_TEMPORARILY, 0, 0, "");
	}

	if (foundmeta->signon->http_post == NULL) {
		/* POST先がないのはエラーとすべし */
		ERRLOG1(pool, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
				"failed to post binding. Please check IdP metadata."
				"(entityid=%s)", entityid);
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_saml_login;
	}

	/* XML用の情報を作成 */
	iscreen = apr_pcalloc(pool, sizeof(divy_samlinfo_iscreen));
	iscreen->r = r;
	iscreen->actionurl = apr_pstrdup(pool, foundmeta->signon->http_post);
	iscreen->next = apr_pstrdup(pool, next);
	iscreen->x509path = apr_pstrdup(pool, conf->saml_x509certificatepath);
	iscreen->usr_pr = NULL;
	iscreen->selectedtfuser = NULL;
	iscreen->statuscode = NULL;
	iscreen->msg = NULL;
	iscreen->token = NULL;
	iscreen->browserstyle = conf->tfbrowserstyle;

	/* スタイルシートを設定する */
	stylesheet = _get_stylesheet_path(pool, "saml_authnrequest.xsl", r);
	if (IS_EMPTY(stylesheet)) {
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_saml_login;
	}

	err = _samlxml2html(pool, iscreen, stylesheet, &filepath);
	if (err != NULL) {
		goto cleanup_saml_login;
	}

	/* ステータスを設定する */
	r->status = HTTP_OK;
	ap_set_content_type(r, "text/html");

	/* HTMLファイルをストリームに流す */
	ret = divy_sendfile2stream(r, output, APR_INT64_C(-1), filepath);
	if (ret != OK) {
		err = dav_new_error(pool, ret, 0, 0, "");
		goto cleanup_saml_login;
	}

	/* リダイレクト先の設定 */
/*
	apr_table_setn(r->headers_out, "Location",
						conf->idpmeta->signon->http_post);
*/
	ap_set_content_type(r, "text/html; charset=\"utf-8\"");

cleanup_saml_login:

	if (cur != NULL) xsltFreeStylesheet(cur);
	if (html_p != NULL) xmlFreeDoc(html_p);
	if (doc_p != NULL)  xmlFreeDoc(doc_p);

	if (need_unlock) {
		/* ロック後に使用されるのでこのタイミングでクリーンアップできる */
		xsltCleanupGlobals();
		xmlCleanupParser();

		/* ロックを解除 */
		rv = apr_thread_mutex_unlock(autoindex_mutex);
		if (rv != APR_SUCCESS) {
			ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to unlock for open file.(code = %d)", rv);
			/* 続ける */
		}
	}

	/* 一時ファイルを閉じる */
	if (fd != NULL && (rv = apr_file_close(fd)) != APR_SUCCESS) {
		ERRLOG1(pool, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to close tempory file for SAML Request xml."
			"(code = %d)", rv);
		/* 続ける */
	}

	/* 一時ファイルを削除する */
	if (err && filepath) {
		rv = apr_file_remove(filepath, pool);
		if (rv != APR_SUCCESS) {
			ERRLOG2(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to remove temporary file for autoindex."
				"(filepath = %s, code = %d)", filepath, rv);
		}
		filepath = NULL;	/* 無かったことにする */
	}

	/* 失敗した場合は標準のログイン画面へ遷移する */
	/* ここは標準に行くべきか悩むところ */
	/* TODO エラーメッセージはいらないのか？*/
	//return divy_do_login(resource, output);
	//
	/* 正常の場合は移動する */
	if (err == NULL)
		err = dav_new_error(pool, HTTP_MOVED_TEMPORARILY, 0, 0, "");

	return err;
}

/**
 * SAML アサーション・コンシューマー・サービス(ACS)へのリダイレクトPOSTを
 * 処理する
 * この関数に来るケースは主にIdPからのリダイレクトです。
 */
DIVY_DECLARE(dav_error *)
divy_do_saml_acs_login(const dav_resource *resource, const dav_postdata *pdata, ap_filter_t *output)
{
	request_rec *r = resource->info->r;
	apr_pool_t *p  = resource->pool;
	dav_error *err = NULL;
	char *samlresponse = NULL;
	char *relaystate = NULL;
	char *tfuser = NULL;
	divy_saml_assertion *assertion = NULL;
	char *next = NULL;
	int parsed = 1;	/* パース結果失敗 = 1 / 0 は成功 */
	int ret = 0;
	divy_rdbo_usr	  *usr_pr = NULL;
	char *userid = NULL;
	char *stylesheet;
	apreq_cookie_t *cookie;
	divy_samlinfo_iscreen *iscreen;
	divy_saml_tfuser* tfusr_pr = NULL;
	char *filepath      = NULL;
	char *saml_uri = NULL;
	int count = 0;
	apreq_cookie_t *cookie_authtype;
	dav_divy_dir_conf *dconf  = dav_divy_get_dir_config(r);
	apr_uuid_t uuid;
    char char_uuid[APR_UUID_FORMATTED_LENGTH + 1];
	char *token;
	const char *header = NULL;
	apr_table_t* jar;

	divy_auth_session session = {0};

	/* acsに対してCookieが入っている場合はそのCookieがTeamFileが発行した
	 * 可能性があります。（ブラウザの戻るボタン）
	 * この画面ではまだユーザーは特定していないことが条件であるため
	 * また始めの画面に強制的に戻します
	 */
	header = apr_table_get(r->headers_in, "Cookie");
	if (IS_FILLED(header)) {
		jar = apr_table_make(p, APREQ_DEFAULT_NELTS);
		if (apreq_parse_cookie_header(p, jar, header) != APR_SUCCESS) {
			/* パース失敗 */
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"Failed to parse cookie. cookie value = %s", header);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		if (apr_table_get(jar, TF_COOKIE_SES_NAME) != NULL) {
			/* ログアウトさせる */
			(void)_remove_browser_session(r);
			apr_table_setn(r->headers_out, "Location",
						apr_psprintf(p, "%s",
						divy_build_login_uri(p, dav_divy_get_root_uri(r))));
			return dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, "");
		}
	}

	if (pdata && pdata->urlparams) {
		samlresponse = (char*)apr_table_get(pdata->urlparams, "SAMLResponse");
		relaystate   = (char*)apr_table_get(pdata->urlparams, "RelayState");
		tfuser       = (char*)apr_table_get(pdata->urlparams, "TFUser");

		ERRLOG2(p, APLOG_DEBUG, DIVY_FST_IERR + DIVY_SST_DEBUG,
				"post relaystate = [%s] / tfuser = [%s]", relaystate, tfuser);

	}

	if (samlresponse == NULL) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
				"failed to parse samlresponse. SAMLResponse not found.");

		/* SAMLResponseがないと処理はできない */
		apr_table_setn(r->headers_out, "Location",
				divy_construct_url2(p, DIVY_URLTYPE_PUBLIC,
										dav_divy_get_root_uri(r), r, NULL));
		return dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, "");
	}
	else {
		parsed = divy_saml_parse_samlresponse(p, samlresponse, &assertion);
	}

	/* パースが成功した場合は、中身が正しいかをチェックします */
	if (parsed == 0) {
		err = divy_saml_validate_samlresponse(r, assertion, relaystate);
		if (err != NULL) return err;
	}
	else {
		/* パースが失敗 */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"failed to parse saml response");

		apr_table_setn(r->headers_out, "Location",
				divy_construct_url2(p, DIVY_URLTYPE_PUBLIC,
										dav_divy_get_root_uri(r), r, NULL));

		return dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, "");
	}

	/* ここに来るときは*使える*アサーションデータであるべきです */

	/* アサーションデーターのNameIDをマッチングに選択 */
	userid = assertion->subject->nameid;
	ERRLOG1(p, APLOG_INFO, DIVY_FST_INFO + DIVY_SST_DATA,
			"assertion NameID = %s", assertion->subject->nameid);

	/*
	 * マッチングをメールアドレスにする
	 * 複数ヒットする可能性があります。
	 *
	 * 一人も該当しない場合はそのままスルーしてユーザがいないときの
	 * 処理に任せます
	 */
	if (dconf->saml_matchedmailaddr == DIVY_SAML_MATCHED_MAIL_ON) {

		/*
		 * パラメーターにtfuserが含まれている場合そのIDはユーザによって選択
		 * されたユーザIDである
		 */
		if (IS_FILLED(tfuser)) {
			count = 1; /* 一人に絞り込めた */
			userid = tfuser;
		}
		else {
			if (divy_rdbo_get_userlist_from_mail(r, userid, &usr_pr, &count)) {
				/* DBの検索ができなかった */
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to search user userid = %s", userid);
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}

			/* countが1の場合はメールアドレスをキーにして検索した結果
			 * 絞り込めたことになる 
			 * useridはNameID（メールアドレス）として検索していたため
			 * TeamFileのユーザIDと置き換える
			 */
			if (count == 1) {
				userid = usr_pr->usrid;
			}
		}

		/* ユーザー発見 */
		if (count == 1) {

			/* divy_rdbo_get_userlist_from_mail()は情報が少ないため
			 * 再度divy_rdbo_get_user_property()ですべての項目を埋めます
			 */
			if (divy_rdbo_get_user_property(r, userid, &usr_pr)) {
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}
		}

		/* 該当ユーザーが複数見つかった */
		if (count > 1) {
			iscreen = apr_pcalloc(p, sizeof(divy_samlinfo_iscreen));
			for (; usr_pr != NULL; usr_pr = usr_pr->next) {
				if (iscreen->usr_pr == NULL) {
					iscreen->usr_pr = tfusr_pr = apr_pcalloc(p,
							sizeof(divy_saml_tfuser));
				}
				else {
					tfusr_pr->next = apr_pcalloc(p, sizeof(divy_saml_tfuser));
					tfusr_pr = tfusr_pr->next;
				}
				tfusr_pr->usrid    = apr_pstrdup(p, usr_pr->usrid);
				tfusr_pr->fullname = apr_pstrdup(p, usr_pr->fullname);
			}
			iscreen->r = r;

			saml_uri = apr_psprintf(p, "%s%s",
					dav_divy_get_root_uri(r), "/.saml/acs");
			iscreen->actionurl = divy_construct_url2(p,
					DIVY_URLTYPE_PUBLIC, saml_uri, r, NULL);
			iscreen->next = NULL;
			iscreen->relaystate   = apr_pstrdup(p, relaystate);
			iscreen->samlxml = apr_pstrdup(p, samlresponse);
			iscreen->statuscode = NULL;
			iscreen->msg = NULL;
			iscreen->token = NULL;
			iscreen->browserstyle = dconf->tfbrowserstyle;

			ERRLOG1(p, APLOG_INFO, DIVY_FST_INFO + DIVY_SST_DATA,
					"user found count = %d", count);

			apr_table_addn(r->err_headers_out, "Cache-Control",
					"no-store no-cache, must-revalidate,"
					"post-check=0, precheck=0");
			apr_table_addn(r->err_headers_out, "Pragma", "no-cache");

			/* スタイルシートを設定する */
			stylesheet = _get_stylesheet_path(p, "saml_selectuser.xsl", r);
			if (IS_EMPTY(stylesheet)) {
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}

			err = _samlxml2html(p, iscreen, stylesheet, &filepath);

			/* ステータスを設定する */
			r->status = HTTP_OK;
			ap_set_content_type(r, "text/html");

			/* HTMLファイルをストリームに流す */
			ret = divy_sendfile2stream(r, output, APR_INT64_C(-1), filepath);
			if (ret != OK) {
				return dav_new_error(p, ret, 0, 0, "");
			}

			/* 二人以上いた場合はここで終わる */
			return NULL;
		} /* count > 1 */

	} 
	else {
		/* ユーザIDとマッチングさせる */
		if (divy_rdbo_get_user_property(r, userid, &usr_pr)) {
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	/* SAMLのユーザーがTeamFileにいませんでした */
	if (usr_pr == NULL) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
				"The SAML Assertion user [%s] not found.", 
				assertion->subject->nameid);

		/*
		 * TeamFileのユーザー調べても存在しなかった場合は、
		 * TeamFileの今までのログインと同じ仕組みに移行します
		 * relaystateの内容がもし合った場合は、再利用する必要も
		 * 無いため、読み取ってからmemcachedから削除をします。
		 */

		/* TODO ユーザがいない場合、作ってしまう仕組みを考えなさい */
		divy_saml_provisioning_account(r, assertion);

		userid = "";	/* ユーザーIDはクリアする */

		/*
		 * relaystateよりnextのURLを取得してみる
		 * 無かった場合はロケーションのTOP（rooturi)を設定する
		 */
		if (IS_FILLED(relaystate)) {
			divy_saml_load_relaystate(r, relaystate,
					0 /* 読んだら削除 */, &next);
		}

		apr_uuid_get(&uuid);
		apr_uuid_format(char_uuid, &uuid);
		divy_saml_save_relaystate(r, char_uuid, 60 * 10, &token);

		if (IS_EMPTY(next)) {
			next = (char *)dav_divy_get_root_uri(r);
		}

		apr_table_setn(r->headers_out, "Location",
				apr_psprintf(p, "%s?next=%s&token=%s&st=404",
					divy_build_login_uri(p, dav_divy_get_root_uri(r)),
					next, token));

		return dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, "");

	} 

	session.uid		 = apr_pstrdup(p, userid);
	session.password = apr_pstrdup(p, usr_pr->password);
	session.validmail= NULL;

	if (divy_util_auth_set_memcache_session(r, &session,
												1 /* 一時的なキャッシュ */)) {
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* SAMLの認証であることをCookieに記録します */
	cookie_authtype = apreq_cookie_make(r->pool, "authtype", 8, "saml", 4);
	cookie_authtype->path = (char*)dav_divy_get_root_uri(r);
	apr_table_add(r->err_headers_out, "Set-Cookie",
									apreq_cookie_as_string(cookie_authtype, p));

	cookie = apreq_cookie_make(r->pool, TF_COOKIE_SES_NAME, strlen(TF_COOKIE_SES_NAME),
											session.sid, strlen(session.sid));
	cookie->path = (char*)dav_divy_get_root_uri(r);
	apr_table_add(r->err_headers_out, "Set-Cookie",
										apreq_cookie_as_string(cookie, p));

	/*
	 * relaystateがある場合は、つぎ進むべきURLが入っているかもしれない
	 */
	if (IS_FILLED(relaystate)) {
		divy_saml_load_relaystate(r, relaystate, 0 /* 読んだら削除 */, &next);
	}

	if (IS_EMPTY(next)) {
		apr_table_setn(r->headers_out, "Location",
							divy_construct_url2(p, DIVY_URLTYPE_PUBLIC,
					   						dav_divy_get_root_uri(r), r, NULL));
	}
	else {
		apr_table_setn(r->headers_out, "Location",
					divy_construct_url2(p, DIVY_URLTYPE_PUBLIC, next, r, NULL));
	}


	return dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, "");
}

/**
 * ログイン画面を出力する
 */
DIVY_DECLARE(dav_error *)
divy_do_login(const dav_resource *resource, ap_filter_t *output)
{

	request_rec *r = resource->info->r;
	apr_pool_t *pool = resource->info->r->pool;
	dav_divy_dir_conf *dconf  = dav_divy_get_dir_config(r);
	apr_hash_t *parsed_h = NULL;
	divy_linkedlist_t *ll;
	divy_samlinfo_iscreen *iscreen;
	char *token = NULL;

	char *stylesheet = NULL;
	char *filepath = NULL;
	char *xsl_path = NULL;

	dav_error *err = NULL;
	int ret;
	int tflogin = 0;	/* TeamFileへのログインか？ */

	TRACE(pool);

	/* SAML情報の初期化 */
	iscreen = apr_pcalloc(pool, sizeof(divy_samlinfo_iscreen));
	iscreen->r = r;
	iscreen->actionurl = r->uri;
	iscreen->browserstyle = dconf->tfbrowserstyle;
	iscreen->uniqueid = dav_divy_replace_str(pool,
				apr_table_get(r->subprocess_env, "UNIQUE_ID"), "@", "-", NULL);

	dav_divy_parse_urlparam(pool, r->args, &parsed_h);
	if (parsed_h != NULL) {
		/* logoutがパラメータについた場合は無条件にクッキー他情報を削除 */
		ll = apr_hash_get(parsed_h, "logout", APR_HASH_KEY_STRING);
		if (ll != NULL) {
			/* 成功失敗問わない */
			(void)_remove_browser_session(r);
		}

		/* ステータスコード（エラーコード）がある場合はセットする */
		ll = apr_hash_get(parsed_h, "st", APR_HASH_KEY_STRING);
		if (ll != NULL) {
			iscreen->statuscode = apr_pstrdup(pool, ll->val);
		}

		/*
		 * tokenがパラメータであった場合、意図的にTeamFileのログインを
		 * 行なうことになります。
		 * このtokenがあった時は内部のmemcachedで記録しているかを調べます。
		 * memcachedでは有効期限付きで記録されているので無いこともあります
		 */
		ll = apr_hash_get(parsed_h, "token", APR_HASH_KEY_STRING);
		tflogin = 0;
		if (ll != NULL) {
			/* パラメータにあった */
			iscreen->token = apr_pstrdup(pool, ll->val);
			(void)divy_saml_load_relaystate(r, ll->val,
									SAML_LOCAL_LOGIN_INTERVAL - 1, &token);
			/* SAMLの時のログイン画面はリフレッシュさせます see util_saml.h */
			apr_table_setn(r->err_headers_out, "Refresh",
							apr_psprintf(pool, "%d",SAML_LOCAL_LOGIN_INTERVAL));
			if (IS_FILLED(token)) tflogin = 1;
		}

		/* nextが設定されていることがある */
		ll = apr_hash_get(parsed_h, "next", APR_HASH_KEY_STRING);
		if (ll != NULL) {
			iscreen->actionurl = apr_psprintf(pool, "%s?next=%s",
										iscreen->actionurl, ll->val);
		}

	}

	/*
	 * SAMLをサポートして有効の場合はログイン画面を出さず
	 * SAML選択画面へ遷移する
	 */
	if (tflogin == 0 && divy_enable_saml(r, resource)) {
		err = divy_do_saml_idp_select(resource, output);
		goto cleanup_login;
	}

	/* ログイン画面はキャッシュを行わないようにする */
//	apr_table_addn(r->headers_out, "Cache-Control", "no-store");
	apr_table_addn(r->err_headers_out, "Cache-Control", "no-store no-cache, must-revalidate, post-check=0, precheck=0");
	apr_table_addn(r->err_headers_out, "Pragma", "no-cache");

	if (dconf->tfbrowserstyle & DIVY_BROWSER_STYLE_OLDLOGINPAGE) {
		xsl_path = "default_login.html";
	}
	else {
		xsl_path = "default_login.xsl";
	}

	stylesheet = _get_stylesheet_path(pool, xsl_path, r);
	if (IS_EMPTY(stylesheet)) {
		/* スタイルシートがない */
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_login;
	}

	err = _samlxml2html(pool, iscreen, stylesheet, &filepath);
	if (err != NULL) {
		goto cleanup_login;
	}

	/* セキュリティ Content Security Policy (CSP) */
	char *policy_str = apr_psprintf(pool, "default-src 'self'; "
			"script-src 'self' 'nonce-%s'", iscreen->uniqueid);
	apr_table_addn(r->err_headers_out, "Content-Security-Policy", policy_str);

	ap_set_content_type(r, "text/html; charset=\"utf-8\"");
	ret = divy_sendfile2stream(r, output, APR_INT64_C(-1), filepath);
	if (ret != OK) {
		err = dav_new_error(pool, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

cleanup_login:

	return err; 
}
/**
 * ログインのPOSTを処理する
 * (note)
 * ここに来るときは*まだ*認証が終わっていません。
 * memcacheに設定は行うがこの時点ではまだそのパスワードが正しいかどうか
 * はわかりません。実際にはauth.cのdav_divy_authenticate_user()関数でなければ
 * ユーザー認証は行われません。
 *
 * == ここではブラウザのCookieを設定するだけしかできません ==
 */
DIVY_DECLARE(dav_error *)
divy_do_post_login(const dav_resource *resource, const dav_postdata* pdata, ap_filter_t *output)
{
	request_rec *r = resource->info->r;
	apr_pool_t *p  = resource->pool;
	char *id , *pw, *lang, *token, *action;
	apreq_cookie_t *cookie, *cookie_lang;
	int clear_cookie = 0;
	divy_auth_session session = {0};
	apr_hash_t *parsed_h = NULL;
	divy_linkedlist_t *ll;
	char *next = NULL;
	char *testnext = NULL;

	TRACE(p);

	id = pw = NULL;
	if (pdata && pdata->urlparams) {

		id = (char *)apr_table_get(pdata->urlparams, "id");
		pw = (char *)apr_table_get(pdata->urlparams, "password");
		lang=(char *)apr_table_get(pdata->urlparams, "lang");
		token=(char *)apr_table_get(pdata->urlparams, "token");
		action=(char*)apr_table_get(pdata->urlparams, "action");

		if (IS_FILLED(apr_table_get(pdata->urlparams, "return"))) {
			apr_table_setn(r->headers_out, "Location",
						apr_psprintf(p, "%s",
						divy_build_login_uri(p, dav_divy_get_root_uri(r))));
			return dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, "");
		}	

		/* chglngパラメーターがある場合は言語だけの更新 */
		if (IS_FILLED(action) && strncmp(action, "chglng", 6) == 0) {

			/*
			 * デフォルト（ブラウザ）を選択するとlangが空になってしまうため
			 * デフォルトの言語を取得します 
			 */
			if (IS_EMPTY(lang)) {
				lang = (char*)divy_get_language_param(r);
				ERRLOG1(p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
						"lang = %s", lang);
				clear_cookie = 1; /* langを記録しているcookieは消す */
			}

			cookie_lang = apreq_cookie_make(r->pool, 
									"lang", 4, lang, strlen(lang));
			cookie_lang->path = (char*)dav_divy_get_root_uri(r);

			if (clear_cookie) {
				apreq_cookie_expires(cookie_lang, "-3y");
			}

			apr_table_add(r->err_headers_out, "Set-Cookie",
									apreq_cookie_as_string(cookie_lang, p));
			apr_table_setn(r->headers_out, "Location",
						apr_psprintf(p, "%s&token=%s",
						divy_build_login_uri(p, dav_divy_get_root_uri(r)), token));
			return dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, "");
		}

		if (IS_EMPTY(id) || IS_EMPTY(pw)) {
			apr_table_setn(r->headers_out, "Location",
						apr_psprintf(p, "%s&token=%s&st=401",
						divy_build_login_uri(p, dav_divy_get_root_uri(r)), token));
			/* id,pw どちらか空白は再認証画面へリダイレクト */
			//return divy_do_login(resource, output);
			return dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, "");
		}
	}
	else {
		/* POSTデータがない */
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* セッションデータを構築する */
	session.uid      = apr_pstrdup(p, id);
	session.password = apr_pstrdup(p, pw);
	session.sid      = apr_pstrdup(p, token);
	session.validmail= NULL;
	session.uniqueid = (char*)apr_table_get(r->subprocess_env, "UNIQUE_ID");

	if (divy_util_auth_set_memcache_session(r, &session,
												1 /* 一時的なキャッシュ */)) {
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* Cookieを作る */
	cookie = apreq_cookie_make(r->pool, TF_COOKIE_SES_NAME, strlen(TF_COOKIE_SES_NAME), session.sid, strlen(session.sid));
	cookie->path = (char*)dav_divy_get_root_uri(r);
	apreq_cookie_expires(cookie, "+1y");
	
	/* 言語が設定されていたらCookie作成追加 */
	cookie_lang = apreq_cookie_make(r->pool, "lang", 4, lang, strlen(lang));
	cookie_lang->path = (char*)dav_divy_get_root_uri(r);
	/* langが入っていない場合は、クッキーの有効期限をつける(過去日付で削除) */
	if (IS_EMPTY(lang)) {
		apreq_cookie_expires(cookie_lang, "-3y");
	}
	apr_table_add(r->err_headers_out, "Set-Cookie",
									apreq_cookie_as_string(cookie_lang, p));

	apr_table_add(r->err_headers_out, "Set-Cookie",
										apreq_cookie_as_string(cookie, p));

	/*
	 * ログインができる状態になったのでリダイレクトで移動する先をパラメータ
	 * より取得します。nextが存在しない場合はTOPフォルダとします
	 */
	dav_divy_parse_urlparam(p, r->args, &parsed_h);
	if (parsed_h != NULL) {
		ll = apr_hash_get(parsed_h, "next", APR_HASH_KEY_STRING);
		if (ll != NULL) {
			/* nextが存在してもloginの場合は遷移はしない */
			char *loginuri = divy_build_login_uri(p, dav_divy_get_root_uri(r));
			if (IS_FILLED(loginuri) || IS_FILLED(ll->val)) {
				if (strcmp(loginuri, ll->val) != 0) {
					next = apr_pstrdup(p, ll->val);
				}
			}
		}
	}

	if (next == NULL) {
		/* nextがなかったらロケーションのトップを設定 */
		next = divy_construct_url2(p, DIVY_URLTYPE_PUBLIC,
								dav_divy_get_root_uri(r), r, NULL);
	}
	else {

		next = dav_divy_remove_endslash(p, next);

	 	/* https://などのスキーマがある場合は、他のサイトに飛ばされるかも
		 * しれないTeamFileではこれを許さない。トップへ移動
	 	 */
		testnext = divy_tolower_str(p, next);
		if (strncmp(testnext, "http://", 7) == 0
								|| strncmp(testnext, "https://", 8) == 0) {
			/* ロケーションのトップを設定 */
			next = divy_construct_url2(p, DIVY_URLTYPE_PUBLIC,
								dav_divy_get_root_uri(r), r, NULL);
		}
		else {
			ap_no2slash(next);
		}
	}

	/* next=に入っているURLへ移動させる */
	apr_table_setn(r->headers_out, "Location", next);

	return dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, "");
}

/* リクエストのURIの短縮URLもとにリソースのプロパティを設定します
 * 短縮URL戻りがおかしい場合はNULLを返却します
 */
DIVY_DECLARE(divy_rdbo_resource*) divy_convert_shorten(apr_pool_t* pool, request_rec* r)
{
	divy_rdbo_resource *rdb_r;
	char *shorten = NULL;
	char *rid = NULL;
	apr_uint32_t propflag = 0;
	apr_hash_t *parsed_h = NULL;
	divy_linkedlist_t *ll;

	if (dav_divy_parse_urlparam(pool, r->args, &parsed_h)) {
		return NULL;
	}

	if (parsed_h != NULL) {
		ll = apr_hash_get(parsed_h, DIVY_URIPARAM_SHORTEN, APR_HASH_KEY_STRING);
		if (ll != NULL) {
			/* パラメータのst=から短縮文字列を取得します */
			shorten = dav_divy_extract_finalpath_segment(pool, ll->val);
		}
	}

	if (shorten == NULL) {
		/* URIのファイナルパスセグメントより短縮文字列を取得します */
		shorten = dav_divy_extract_finalpath_segment(pool, r->uri);
	}

	/* 短縮文字列からrsidを導きます */
	rid = divy_get_shorten2rid(pool, shorten);
	if (IS_EMPTY(rid)) return NULL;

	rdb_r = apr_pcalloc(pool, sizeof(divy_rdbo_resource));
	rdb_r->p      = r->pool;
	rdb_r->rsid   = rid;

	/* ridよりプロパティを設定します。
	 * 取得できない場合はNULLを返します
	 */
	propflag |= DIVY_GET_PROP_fullname;		/* フルネームは必要 */
	propflag |= DIVY_GET_PROP_resourcestate;
	if (divy_rdbo_get_property_by_rsid(r, rdb_r, propflag) != 0) {
		return NULL;
	}
	/* IS_EMPTY()チェックはdivy_rdbo_get_property_by_rsidの要求 */
	if (IS_EMPTY(rdb_r->uri)) return NULL;

	return rdb_r;

}

/**
 * 短縮URLを出力する
 */
DIVY_DECLARE(dav_error *)
divy_do_shorten(const dav_resource *resource, ap_filter_t *output)
{
	apr_status_t rv;
	request_rec *r            = resource->info->r;
	dav_divy_dir_conf *dconf  = dav_divy_get_dir_config(r);
	apr_pool_t *p             = resource->pool;
	divy_rdbo_box *box        = NULL;
	char *stylesheet          = NULL;
	int ret;
	dav_error *err            = NULL;
	divy_rdbo_resource *rdb_r = NULL;

	apr_pool_t *log_p         = r->pool;
	apr_pool_t *wp            = NULL;
	divy_autoindex_screen *screen = NULL;
	divy_fstorage_t *fstorage = NULL;
	divy_outputxml outputxml;
	const char *filepath      = NULL;
	apr_hash_t *parsed_h      = NULL;
	divy_linkedlist_t *ll;
	divy_linkedlist_t *ll2;
	int auth_success= 1; /* default OK */
	int has_param = 0;

	char *refresh = NULL;
	char *style = NULL;
	char *format = NULL;
	char *xsl = "box_default.xsl";	/* デフォルトのXSL */
	char *ctype = "text/html";	/* デフォルトのcontent-type */
	char *typeline = NULL, *token = NULL, *token_ctx = NULL;
	char *pos = NULL;
	int with_rss = 1;	/* RSSも同時表示できる 1:できる / 0:できない */
	char *origin = NULL;	/* Originヘッダ (CORS)*/
	int isRow = 0;		/* データを加工しないようにする */
	int isAdmin = 0;
	divy_rdbo_usr *usr_pr = NULL;	/* 管理者チェック用 */
	time_t now;
	int validmail = 0; /* メールアドレスのチェック結果 */
	int pass2fa = 0;
	char *next = NULL;

	TRACE(p);

	now = dav_divy_get_now_epoch();

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/*
	 * BOXで利用されるユーザをセットする
	 * ここでユーザ情報が設定される為、ユーザの情報を一切まだシステムが
	 * 理解できていない為一度すべてのユーザ情報をキャッシュします
	 */
	r->user = DIVY_GUEST_ID;
	(void)divy_rdbo_cache_userinfo(r, 1 /* no cache */);
	
	/* 短縮URLより正規のリソース情報を取得する */
	rdb_r = divy_convert_shorten(wp, r);
	if (rdb_r == NULL) {
		err = dav_new_error(log_p, HTTP_NOT_FOUND, 0, 0, "");
		goto cleanup_box_resource;
	}

	/* uriのパース */
	if (divy_parse_uri(wp, dav_divy_get_root_uri(r),
								rdb_r->uri, &rdb_r->u_spec)) {
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_box_resource;
	}

	/* リソース状態がないときもある */
	if (rdb_r->rstate_pr == NULL) {
		err = dav_new_error(log_p, HTTP_NOT_FOUND, 0, 0, "");
		goto cleanup_box_resource;
	}

	/*
	 * BOX情報を取得 
	 * BOXは器があっても中身がない場合があります
	 */
	box = rdb_r->rstate_pr->optional.box_pr;
	if (box == NULL || (box != NULL && box->uri == NULL)) {
		ERRLOG1(log_p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
				"The BOX shorten request. but not found box registerd. (uri=%s)"
				, r->uri);
		err = dav_new_error(log_p, HTTP_NOT_FOUND, 0, 0, "");
		goto cleanup_box_resource;
	}

	/*
	 * 親フォルダがグループフォルダの場合でここに来るケースは
	 * 必ずBOXである。このBOXを利用される前にリクエストとBOXに
	 * 登録されている短縮URL（shorten)を比較しなければなりません。
	 */
	if (rdb_r->u_spec->p_infotype == DIVY_INFOTYPE_group_e) {
		if (rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
			if (strcmp(dav_divy_extract_finalpath_segment(wp, r->uri),
														box->shorten) != 0) {
				ERRLOG2(log_p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
					"Mismatch Request shorten uri. uri = %s, shorten = %s",
					r->uri, box->shorten);
				err = dav_new_error(log_p, HTTP_NOT_FOUND, 0, 0, "");
				goto cleanup_box_resource;

			}
		}
	}

	err = divy_validate_minimum_access_guarantee(r, rdb_r->u_spec);
	if (err != NULL) {
		goto cleanup_box_resource;
	}

	/*
	 * 公開が有効になっている場合だけ次に進めます
	 */
	if ((box->flag & BOX_FLAG_OPEN) == 0) {
		ERRLOG1(log_p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
				"The box is disable. shorten = %s", box->shorten);
		err = dav_new_error(wp, HTTP_NOT_FOUND, 0, 0, "");
		goto cleanup_box_resource;
	}

	/*
	 * allowd_originの役割
	 * クロスサイトでJavaScriptのXMLRequestが行えるために導入されたが、
	 * そうではない単にiframeへの貼り付けなどを行う場合、張り付けるサイトを
	 * 限定させる役割も担っています。
	 * 
	 * allowed_originの値が入っており且つhttpで始まらないものが対象です。
	 * httpから始まるものは単にAccess-control-allow-originヘッダをつければ
	 * いいだけであとはブラウザがやってくれます。
	 *
	 * 比較するのはリモートホストからのDNSLookupです。
	 * 名前で比較するのは、iframeなどで貼り付けが行える為にIPアドレスでは
	 * 特定が困難なためです。名前、つまりドメインで管理を行えば張り付ける
	 * 事に関しては抑制ができます。
	 * 
	 */
	if (IS_FILLED(box->allowed_origin) && strncmp(box->allowed_origin, "http", 4) != 0) {

		int match = 0;   /* default mismatch = 0 */
		char *remote_host;
		char *allow = apr_pstrdup(p, box->allowed_origin);

		/* リモートホストを取得する */
		if (APR_SUCCESS
			== apr_getnameinfo(&remote_host, r->useragent_addr, 0)) {

			if (IS_FILLED(remote_host)) {

				char *token, *token_cntx;
				while((token = apr_strtok(allow, ",", &token_cntx)) != NULL) {
					allow = NULL;

					if (token == NULL) break;
					token = (char *) dav_divy_trim_white(p, token);

					if (strcmp(token, remote_host) == 0) {
						/* match !!! */
						match = 1;
						break;
					}
				}
			}

		}
		else {
			/* DNS lookup failed */
			ERRLOG0(log_p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_NET,
					"failed to allow origin(host). DNS lookup failed");
			err = dav_new_error(log_p, HTTP_NOT_FOUND, 0, 0, "");
			goto cleanup_box_resource;
		}

		if (match != 1) {
			/* allow origin mismatch */
			ERRLOG1(log_p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_NET,
					"Failed to allow origin(host)."
					"mismatch host name. remote host = (%s)", remote_host);
			err = dav_new_error(log_p, HTTP_NOT_FOUND, 0, 0, "");
			goto cleanup_box_resource;
		}
	}

	/* 期限切れならNOTFOUND */
	if (box->expirationdate != 0 && box->expirationdate < now) {
		ERRLOG1(log_p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
				"Failed to box is expired. box shorten=%s", box->shorten);
		err = dav_new_error(log_p, HTTP_NOT_FOUND, 0, 0, "");
		goto cleanup_box_resource;
	}

	/* URLパラメータのパース */
	if (dav_divy_parse_urlparam(wp, r->args, &parsed_h) == 0) {
		if (parsed_h != NULL)
			has_param = 1;	/* URLパラメータを持っている */
	}

	/* パラメータを持っている場合は内容に応じて処理を行います */
	if (has_param) {
		/* refreshパラメータ - ヘッダの付与 */
		ll = apr_hash_get(parsed_h, "refresh", APR_HASH_KEY_STRING);
		if (ll != NULL) refresh = apr_pstrdup(p, ll->val);

		/* スタイルシート(CSS)パラメータ - スタイルの指定 */
		ll = apr_hash_get(parsed_h, "style", APR_HASH_KEY_STRING);
		if (ll != NULL) {
			style = apr_pstrdup(p, ll->val);
		}

		/*
		 * format=???により出力に利用するスタイルシートを決定します
		 * xslにスタイルシートを設定して後で利用します。
		 * スタイルは複数はできません。どれか一つになるように実装すべきです
		 * またここで設定されるスタイルシートはコレクションだけしか利用しない
		 * 
		 * list
		 *   リスト形式です。主に組み込み一覧を行うために最適化されています
		 *   出力Content-TypeはHTMLとします
		 *
		 * rss
		 *   RSS形式です。出力ContentTypeはxmlとなります。
		 *
		 * xml
		 *   XML形式です。簡素なフォーマットにしてxmlに出力します
		 *
		 * json
		 *   JSON形式です。java scriptで最適でしょうね
		 */
		ll = apr_hash_get(parsed_h, "format", APR_HASH_KEY_STRING);
		if (ll != NULL) {
			/*
			 * 管理者が作成したBOXかを調査する
			 * 管理者が作ったBOXの場合はフォーマット抑制機能tfboxdisableformat
			 * は有効にならない。
			 */
			divy_rdbo_get_user_property(r, box->creator_usr_id, &usr_pr);
			if (usr_pr != NULL && usr_pr->adminmode == DIVY_ADMINMODE_ADMIN) {
				isAdmin = 1;
			}

			format = apr_pstrdup(p, ll->val);
			if (format != NULL) {
				if (strcmp(format, "list") == 0 && (!(dconf->tfboxdisableformat & DIVY_BOX_DISABLE_FORMAT_LIST) || isAdmin)) {
					xsl = "box_list.xsl";
					ctype = "text/html";
				}
				else 
				if (strcmp(format, "rss") == 0 && (!(dconf->tfboxdisableformat & DIVY_BOX_DISABLE_FORMAT_RSS) || isAdmin)) {
					xsl = "box_rss.xsl";
					ctype = "text/xml";
					isRow = 1;	/* 生データ */
				}
				else 
				if (strcmp(format, "xml") == 0 && (!(dconf->tfboxdisableformat & DIVY_BOX_DISABLE_FORMAT_XML) || isAdmin)) {
					xsl = "box_xml.xsl";
					ctype = "text/xml";
					isRow = 1;	/* 生データ */
				}
				else 
				if (strcmp(format, "json") == 0 && ((!(dconf->tfboxdisableformat & DIVY_BOX_DISABLE_FORMAT_JSON)) || isAdmin)) {
					xsl = "box_json.xsl";
					ctype = "application/json";
					isRow = 1;	/* 生データ */
				}
				else {
					xsl = "box_default.xsl";	/* 落ち葉拾い */
					ctype = "text/html";
				}
			}

		}

		/* タイプパラメータ - 独自パラメータの指定*/
		ll = apr_hash_get(parsed_h, "type", APR_HASH_KEY_STRING);
		if (ll != NULL) {
			typeline = apr_pstrdup(p, ll->val);
			while ((token = apr_strtok(typeline, "|", &token_ctx)) != NULL) {
				if (token == NULL) break;
				typeline = NULL;

				token = (char*)dav_divy_trim_white(p, token);
				ERRLOG1(log_p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA, "type = %s", token);
			}
		}

		/* ログアウトパラメータ - ログアウトをする */
		ll = apr_hash_get(parsed_h, "logout", APR_HASH_KEY_STRING);
		if (ll != NULL) {

			/* セッションを削除する */
			(void)_remove_browser_session(r);

			/*
			 * r->argsパラメータからlogoutの部分を切り取りリダイレクト先
			 * を作成する
			 */
			pos = divy_strstr(r->args, "&logout");
			apr_table_setn(r->headers_out, "Location",
								apr_psprintf(p, "%s?%s", r->uri,
											apr_pstrndup(p , r->args, 
											strlen(r->args) - strlen(pos))));

			if (wp != NULL) {
				apr_pool_destroy(wp);
				wp = NULL;
			}

			/* HTTP_MOVED_PERMANENTLY(HTTP 301)を利用しない
			 * IE9（これしか試していない）では301を返すと一回目はうまくいくが
			 * 二回目からは動作がおかしくなるため。
			 */
			return dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, "");
		}

	}

	/* screen の生成 */
	screen = _create_autoindex_screen(wp, r, rdb_r);
	if (screen == NULL) {
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_box_resource;
	}

	/* 追加パラメータの設定 */
	if (IS_FILLED(style))
		screen->extend_css = style;

	/* データ加工フラグを設定 */
	screen->isRow = isRow;

	screen->with_rss = with_rss;

	/* BOXは親の表示が不要 */
	//screen->p_rdb_r = NULL;

	/* screen データの取得 */
	err = _get_autoindex_data(wp, screen);
	if (err) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get autoindex data.");
		goto cleanup_box_resource;
	}

	/* BOXのスタイルシートを設定する */
	stylesheet = _get_stylesheet_path(wp, xsl, r);

	/* 論理ストレージのopen */
	ret = divy_fstorage_open(r, wp, &fstorage);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to open storage.(uri = %s, userid = %s)",
			r->uri, divy_get_userid(r));
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_box_resource;
	}

	/*
	 * パスワードを要求しているならリクエストをチェックして
	 * スタイルシートを変更します
	 */
	if (IS_FILLED(box->password)) {
		if (divy_util_box_authenticate_user(r, box->password, 0) == 1) {
			/* エラーの場合はログイン画面をだす */
			stylesheet = _get_stylesheet_path(wp, "box_login.xsl", r);
			ctype = "text/html"; /* ログイン画面はHTML固定 */
			auth_success = 0;	/* 0=失敗 */
			/* ヘッダに設定 */
			divy_set_resultstatus(r,
					DIVY_RESULTSTATUS_TYPE_BOX_AUTHENTICATION_FAILED);
		}
		with_rss = 0;	/* パスワードが必須なBOXはRSS表示はできない */
	}

	/*
	 * パスワードが問題なかった場合は
	 * 2FAが有効の場合はメールアドレスがBOXに指定されているメールアドレスに
	 * 含まれているかを調べます
	 */
	if (auth_success &&  divy_support_2FA(r) && IS_FILLED(box->tomailaddr)) {
		validmail = divy_util_box_validate_user(r, NULL, box->tomailaddr);

		if (validmail == 1) {
			/* 許可されたメールアドレスではなかった */
			stylesheet = _get_stylesheet_path(wp, "box_login.xsl", r);
			ctype = "text/html"; /* ログイン画面はHTML固定 */

			/* ヘッダに設定 */
			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_BOX_INVALID_USER);
			auth_success = 0;	/* 認証失敗 */
		}
		else if (validmail == -1) {
			/* 利用者のメールアドレスが空だった */
			stylesheet = _get_stylesheet_path(wp, "box_login.xsl", r);
			ctype = "text/html"; /* ログイン画面はHTML固定 */
			auth_success = 0;

			/* セッションを削除する */
			(void)_remove_browser_session(r);
		}
		else {
			/* メールが正しいなら二段階認証を行っているかを調べる */
			pass2fa = divy_util_confirm_2FA_passed(r, NULL);
			if (pass2fa == 1) {
				screen->require2fa = 0;	/* 2FA認証しない */
			}
			else if (pass2fa == 0) {
				stylesheet = _get_stylesheet_path(wp, "box_login.xsl", r);
				ctype = "text/html"; /* ログイン画面はHTML固定 */
				divy_set_resultstatus(r,
						DIVY_RESULTSTATUS_TYPE_2FA_AUTHENTICAITON_REQUIRED);
				screen->require2fa = 1;	/* 2FA認証要求 */
			}
			else {
				/* すべて期限切れ扱い */
				stylesheet = _get_stylesheet_path(wp, "box_login.xsl", r);
				ctype = "text/html"; /* ログイン画面はHTML固定 */
				auth_success = 0;	/* 認証失敗にする */
				screen->require2fa = 0;	/* 2FA認証しない */

				/* セッションを削除する */
				(void)_remove_browser_session(r);
			}
		}

		with_rss = 0;	/* メールアドレスが必須なBOXはRSS表示はできない */
	}

	if (auth_success) {
		/* 認証成功 */

		char *sid, *keycode;
		if ((divy_util_auth_parse_cookie(r, &keycode, &sid))
											== DIVY_AUTH_SESSION_OK) {
			/* セッションの有効期限を更新する */
			divy_util_auth_refresh_session(r, sid);
		}

		/* 通常ファイルの取り扱い */
		if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
			if (dav_divy_parse_urlparam(p, r->args, &parsed_h) == 0) {

				ll = apr_hash_get(parsed_h, DIVY_URIPARAM_PARENT,
														APR_HASH_KEY_STRING);
				if ((ll != NULL) && strcmp(ll->val, box->shorten) == 0) {
					const char *dlpath = divy_make_fullpath(fstorage, rdb_r->physicalpath);
					/* content-typeが空っぽになっている場合がある
					 * 通常のファイルダウンロードでは対策していました。
					 *
					 * 例: IEからのアップロード
					 */
					if (IS_FILLED(rdb_r->getcontenttype)) {
						ap_set_content_type(r, rdb_r->getcontenttype);
					}
					else {
						ap_set_content_type(r, "application/tfoctet-stream");
					}
					/* Content-Dispositonヘッダを追加 */
					 apr_table_setn(r->headers_out, 
							 	"Content-Disposition", "attachment");
					if (OK != divy_sendfile2stream(r, output,
								rdb_r->getcontentlength, dlpath)) {
						err = dav_new_error(log_p, ret, 0, 0, "");
					}

					/* 論理ストレージを閉じる */
					if (fstorage != NULL) (void) divy_fstorage_close(fstorage);

					/* 開封通知を送る */
					if (divy_support_confirmreading(r)) {
						(void) divy_ml_notify_confirmreading(r, rdb_r);
					}

					if (wp != NULL) {
						apr_pool_destroy(wp);
						wp = NULL;
					}

					/* ダウンロードを許可します */
					return err;
				}
			}

			err =  dav_new_error(wp, HTTP_FORBIDDEN, 0, 0, "");
			goto cleanup_box_resource;
		}

		/* collection(フォルダ)はスルー */
	}
	else {
		/* 失敗 */
		if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
			if (has_param) {
				ll = apr_hash_get(parsed_h, DIVY_URIPARAM_PARENT,
														APR_HASH_KEY_STRING);
				ll2 = apr_hash_get(parsed_h, DIVY_URIPARAM_CMD,
														APR_HASH_KEY_STRING);
				if (ll != NULL) {

					stylesheet = _get_stylesheet_path(wp, "box_login.xsl", r);
					ctype = "text/html"; /* ログイン画面はHTML固定 */

					next = apr_psprintf(wp, "%s/%s",
										divy_build_shorten_uri(wp,
										dav_divy_get_root_uri(r)), ll->val);
					if (ll2 != NULL) {
						next = apr_psprintf(wp, "%s?%s=%s",
									next, DIVY_URIPARAM_CMD, ll2->val);

						ERRLOG1(wp, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
								"next location = %s", next);
					}

					apr_table_setn(r->headers_out, "Location", next);

					/* セッションを削除する */
					(void)_remove_browser_session(r);

					err =  dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, "");

					/* ストレージを閉じる */
					if (fstorage != NULL) (void) divy_fstorage_close(fstorage);
					if (wp != NULL) {
						apr_pool_destroy(wp);
						wp = NULL;
					}

					return  dav_new_error(p, HTTP_MOVED_PERMANENTLY, 0, 0, "");
				}
			}
			else {
				goto cleanup_box_resource;
			}
		}
		/* リソースでなければそのまま通過 */
	}

	/* screen -> XMLファイル */
	err = _screen2xml(wp, fstorage, screen, &outputxml);
	if (err) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to build xml string.");
		goto cleanup_box_resource;
	}

	/* XMLファイル -> HTMLファイル */
	err = _xml2html(wp, fstorage, screen, stylesheet, &outputxml, &filepath);
	if (err) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to convert xml to html.");
		goto cleanup_box_resource;
	}

	/* ステータスを設定する */
	r->status = HTTP_OK;
	if (IS_FILLED(screen->encoding)) {
		ap_set_content_type(r, apr_psprintf(wp,
				"%s; charset=\"%s\"", ctype, screen->encoding));
	}
	else {
		ap_set_content_type(r, apr_psprintf(wp,
				"%s; charset=\"utf-8\"", ctype));
	}

	/* ブラウザにキャッシュさせない 
	 * この処理をいれないとIE11でキャッシュがしつこく残る問題が発生しました */
	apr_table_addn(r->err_headers_out, "Cache-Control", "no-store");

	/* リフレッシュヘッダが必要なら追加する */
	if (auth_success && IS_FILLED(refresh)) {
		apr_table_setn(r->headers_out, RFC_HEADER_REFRESH, refresh);
	}

	/* Cross-Origin Resource Sharing (CORS) */
	origin = (char *)apr_table_get(r->headers_in, RFC_ORIGIN);
	if (IS_FILLED(origin)) {
		apr_table_setn(r->headers_out, RFC_ACCESS_CONTROL_ALLOW_ORIGIN
														,IS_FILLED(box->allowed_origin) ? box->allowed_origin : "*");
		apr_table_setn(r->headers_out, RFC_ACCESS_CONTROL_ALLOW_METHODS
														,"GET, POST, OPTIONS");
		apr_table_setn(r->headers_out, RFC_ACCESS_CONTROL_ALLOW_HEADERS ,"*");
	}
	else {
		/* originヘッダがない場合は、X-Frame-optionの中身をboxのallowd_origin
		 * へ変更する
		 */
		if (IS_FILLED(box->allowed_origin)) {
			apr_table_set(r->headers_out, "X-Frame-options",
					apr_psprintf(wp, "ALLOW-FROM %s", box->allowed_origin));
		}
		else {
			/* 外す */
			apr_table_unset(r->headers_out, "X-Frame-options");
		}
	}

	/* HTMLファイルをストリームに流す */
	ret = divy_sendfile2stream(r, output, APR_INT64_C(-1), filepath);
	if (ret != OK) {
		err = dav_new_error(log_p, ret, 0, 0, "");
		goto cleanup_box_resource;
	}

	/*
	 * 認証が正常（アクセスコード無しも同意）の場合は表示した回数を
	 * 記録します。
	 * BOXそのもののリクエストのみカウントを行います
	 * BOX配下のフォルダはアクセスカウントしない
	 */
	if (auth_success && strcmp(dav_divy_extract_finalpath_segment(p, r->uri),
													box->shorten) == 0) {
		if (divy_support_2FA(r) && IS_FILLED(box->tomailaddr) && pass2fa != 1) {
			goto cleanup_box_resource;
		}

		/* BOX参照カウントを増やす */
		if (divy_rdbo_box_update_viewcount(r, box->shorten) == 0) {
			/* 追加成功
			 * 始めの一回だけメールを送信します(外部仕様）
			 * 宅ファイル便が始めの一回だけやっているから同じにしました
			 */
			if (box->viewcount == 0) {
				/* メール送信 結果はどうでもいい */
				(void)divy_ml_box(r, DIVY_ML_BOX_TYPE_OPEN_NOTIFY, box);
			}
		}
	}

cleanup_box_resource:

	/* 論理ストレージを閉じる */
	if (fstorage != NULL) {
		(void) divy_fstorage_close(fstorage);
	}

	/* HTMLファイルを削除する */
	if (IS_FILLED(filepath)) {
		rv = apr_file_remove(filepath, p);
		if (!APR_STATUS_IS_ENOENT(rv) && rv != APR_SUCCESS) {
			ERRLOG2(log_p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to remove temporary html file for autoindex."
				"(filepath = %s, code = %d)", filepath, rv);
		}
	}

	/* XMLファイルを削除する */
	if (outputxml.type == DIVY_XMLOUTPUT_TYPE_FILE && IS_FILLED(outputxml.d.filepath)) {
		rv = apr_file_remove(outputxml.d.filepath, p);
		if (!APR_STATUS_IS_ENOENT(rv) && rv != APR_SUCCESS) {
			ERRLOG2(log_p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to remove temporary xml file for autoindex."
				"(filepath = %s, code = %d)", outputxml.d.filepath, rv);
		}
	}

	/* サブプールを破棄する */
	if (wp != NULL) {
		apr_pool_destroy(wp);
		wp = NULL;
	}

	return err;
}

/**
 * 短縮URLのポスト処理を行う
 */
DIVY_DECLARE(dav_error *)
divy_do_post_shorten(const dav_resource *resource, const dav_postdata* pdata, ap_filter_t *output)
{
	request_rec *r= resource->info->r;
	apr_pool_t *p = r->pool;

	char *keycode = NULL;
	char *param_authcode = NULL;
	char *sid = NULL;
	divy_rdbo_resource *rdb_r;
	divy_rdbo_box *box;
	divy_auth_session session = { 0 };
	int has_session = 0;	/* セッションを持っていなかった */
	int update_session = 0;	/* アップデートしない */

	if ((rdb_r = divy_convert_shorten(p, r)) == NULL) {
		return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}

	TRACE(p);

	/* BOXの情報がない場合はそのままエラーとして処理 */
	box = rdb_r->rstate_pr->optional.box_pr;

	if (box == NULL) {
		return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}

	/* cookieがあるかもしれない */
	if (divy_util_auth_parse_cookie(r, &keycode, &sid)
										== DIVY_AUTH_SESSION_OK) {
		has_session = 1;	/* セッション見つかった */
		session.sid = sid;
		(void)(divy_util_auth_get_memcache_userinfo(r, &session));
	}

	if (has_session == 0) {
		/* セッションがないときは新規作成 */
		session.password  = (char *)apr_table_get(pdata->urlparams, "keycode");
		session.uid       = DIVY_GUEST_ID; /* guestユーザ */
		session.lang      = (char *)apr_table_get(pdata->urlparams, "lang");
		session.validmail = (char *)apr_table_get(pdata->urlparams, "mailaddr");
		session.uniqueid  = (char*)apr_table_get(r->subprocess_env, "UNIQUE_ID");
		update_session    = 1;
	}
	else {
		/* セッションがある時はauthcodeパラメータがあるかを調べてあれば追加 */
		param_authcode = (char *)apr_table_get(pdata->urlparams, "authcode");
		/* ここにはうまく来ている */

		if (IS_FILLED(param_authcode)) {

			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
												"validmail=%s", session.validmail);

			session.authcode = apr_pstrdup(p, param_authcode);
			update_session = 1;
		}
	}

	if (update_session) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC, "update session");
		divy_util_auth_remove_session(r, session.sid);
		/* セッション(memcache)とCookieの登録 */
		divy_util_box_set_cookie(r, &session);
	}

	apr_table_setn(r->headers_out, "Location", r->unparsed_uri);
	return dav_new_error(p, HTTP_MOVED_PERMANENTLY, 0, 0, "");

}

/**
 * SAML画面を有効にするか判定する
 *
 */
DIVY_DECLARE(int) divy_enable_saml(request_rec *r, const dav_resource *resource)
{
	divy_uri_spec *u_spec = resource->info->rdb_r->u_spec;

	if (resource == NULL || resource->info->rdb_r == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"resource or rdb_r is NULL.");
		return 0;
	}

	if (u_spec == NULL)
		return 0;

	/* sessionがOnで且つSAMLも有効でなければならない */
	if (divy_support_session(r) && divy_support_saml(r) 
			&& (u_spec->infotype == DIVY_INFOTYPE_saml
					|| u_spec->infotype == DIVY_INFOTYPE_login) ) {
		return 1; /* 許可する */
	}

	return 0;
}

/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * リクエストパラメータを取得して、autoindex 用screen を生成する.
 *
 * @param p apr_pool_t * screen が割り当てられる領域
 * @param r request_rec * リクエストパラメータ
 * @param rdb_r divy_rdbo_resource * リクエストリソース
 * @return divy_autoindex_screen *
 */
static divy_autoindex_screen * _create_autoindex_screen(apr_pool_t *p,
						request_rec *r,
						divy_rdbo_resource *rdb_r)
{
	divy_autoindex_screen *screen;
	apr_hash_t *parsed_h = NULL;
	divy_linkedlist_t *ll;

	if (rdb_r == NULL) return NULL;

	screen = apr_pcalloc(p, sizeof(divy_autoindex_screen));

	/* 出力エンコーディング */
	screen->encoding = apr_table_get(r->subprocess_env, DIVY_APENV_HTMLENCODE);

	/* 概要表示用スタイルシートファイル名 */
	screen->s_stylesheet = apr_table_get(r->subprocess_env, DIVY_APENV_STYLESHEET);

	/* 詳細用スタイルシートファイル名 */
	screen->d_stylesheet = apr_table_get(r->subprocess_env, DIVY_APENV_DSTYLESHEET);

	/* 拡張CSSはNULL */
	screen->extend_css = NULL;

	/* リクエスト対象となったリソースの情報 */
	screen->r_rdb_r  = rdb_r;
	screen->r_u_spec = rdb_r->u_spec;

	/* キャッシュ防止用パラメータ */
	screen->seq = divy_get_urlseq(p);

	/* リクエスト構造体をキャッシュする */
	screen->r = r;

#if 0
	/* 短縮URL経由のリクエストの場合 */
	if (rdb_r->u_spec->infotype == DIVY_INFOTYPE_shorten) {

		/* URI パラメータのパース */
		if (dav_divy_parse_urlparam(p, r->args, &parsed_h)) {
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to parse URL params.");
			return NULL;
		}

		screen->s_stylesheet = "box_default.xsl.ja";
		screen->limit   = -1;	/* 制限なし     */
		screen->offset  = -1;	/* offset無視   */
		screen->cmd     = r->args;	/* ここは使うがパース前のデータとする */
		screen->ipageno = 0;	/* (使用しない) */
		screen->crsid   = dav_divy_extract_finalpath_segment(r->pool, r->uri);	/* (短縮URI) */
		screen->rsid    = NULL;	/* (使用しない) */
	}
	else 
#endif
	/* モバイル経由のリクエストの場合 */
	if (rdb_r->u_spec->infotype == DIVY_INFOTYPE_mobile) {

		/* 画面表示件数(mobile のみ) */
		screen->limit = DIVY_AUTOINDEX_DISPCNT;

		screen->offset = -1;

		/* URI パラメータのパース */
		if (dav_divy_parse_urlparam(p, r->args, &parsed_h)) {
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to parse URL params.");
			return NULL;
		}
		if (parsed_h != NULL) {
			/* コマンド */
			ll = apr_hash_get(parsed_h, DIVY_URIPARAM_CMD, APR_HASH_KEY_STRING);
			screen->cmd = (ll != NULL) ? ll->val : NULL;

			/* リクエストページ番号 */
			ll = apr_hash_get(parsed_h, DIVY_URIPARAM_REQPAGE, APR_HASH_KEY_STRING);
			screen->ipageno = (ll != NULL) ? atoi(ll->val) : 1;
			screen->ipageno = (screen->ipageno == 0) ? 1 : screen->ipageno; /* ページ0は不正 */

			/* 暗号化されたrsid */
			ll  = apr_hash_get(parsed_h, DIVY_URIPARAM_CRSID, APR_HASH_KEY_STRING);
			screen->crsid = (ll != NULL) ? ll->val : NULL;

			/* 復号されたrsid */
			screen->rsid = (IS_FILLED(screen->crsid)) ? (char *) _decipher_rsid(p, screen->crsid) : NULL;

			/* 表示開始位置(mobile のみ) */
			screen->offset = screen->limit * (screen->ipageno - 1);
		}
	}
	/* チケットリクエストの場合 */
	else if (rdb_r->u_spec->infotype == DIVY_INFOTYPE_ticket) {

		/* URI パラメータのパース */
		if (dav_divy_parse_urlparam(p, r->args, &parsed_h)) {
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to parse URL params.");
			return NULL;
		}
		if (parsed_h != NULL) {
			/* チケット */
			ll = apr_hash_get(parsed_h, DIVY_URIPARAM_TICKET_URI, APR_HASH_KEY_STRING);
			screen->ticket = (ll != NULL) ? ll->val : NULL;
		}

		screen->limit   = -1;	/* 制限なし     */
		screen->offset  = -1;	/* offset無視   */
		screen->cmd     = NULL;	/* (使用しない) */
		screen->ipageno = 0;	/* (使用しない) */
		screen->crsid   = NULL;	/* (使用しない) */
		screen->rsid    = NULL;	/* (使用しない) */
		screen->extend_css = NULL;	/* 未設定	*/
		screen->with_rss= 0;	/* 未設定		*/
		screen->isRow	= 0;	/* 未設定		*/
		screen->require2fa = 0;	/* not require	*/
	}
	/* モバイル、チケットリクエスト以外の場合 */
	else {
		screen->limit   = -1;	/* 制限なし     */
		screen->offset  = -1;	/* offset無視   */
		screen->cmd     = NULL;	/* (使用しない) */
		screen->ipageno = 0;	/* (使用しない) */
		screen->crsid   = NULL;	/* (使用しない) */
		screen->rsid    = NULL;	/* (使用しない) */
		screen->extend_css = NULL;	/* 未設定	*/
		screen->with_rss= 0;	/* 未設定		*/
		screen->isRow	= 0;	/* 未設定		*/
		screen->require2fa = 0;	/* not require	*/
	}

	return screen;
}

/**
 * 指定されたrdb_r->uri 以下のリソースを取得する。
 * (note)
 * 	Apache環境変数により出力文字列をエンコードします。
 *
 * @param p apr_pool_t * 作業用プール
 * @param screen divy_autoindex_screen * 画面入力情報を持つ構造体
 * @return dav_error *
 */
static dav_error * _get_autoindex_data(apr_pool_t *p,
					divy_autoindex_screen *screen)
{
	request_rec *r              = NULL;
	apr_pool_t *log_p           = NULL;
	dav_error *err              = NULL;
	divy_rdbo_resource *n_rdb_r = NULL;
	const char *tmp;
	int i;
	divy_infotype infotype;
	const char *root_uri;
	divy_uri_spec *t_u_spec     = NULL;	/* for .ticket */

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"screen is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	r     = screen->r;
	log_p = r->pool;
	root_uri = dav_divy_get_root_uri(r);

	/* uri 種別を取得 */
	infotype = screen->r_u_spec->infotype;

	/*
	 * 各特殊フォルダ毎の処理
	 */
	screen->p_rdb_r = NULL;	/* 初期化 */
	screen->c_rdb_r = NULL;
	screen->m_rdb_r = NULL;

	/* $root/.ticket フォルダ (特別扱い) */
	if (infotype == DIVY_INFOTYPE_ticket) {
		/* チケットの復号 */
		dav_resource resource     = { 0 };
		dav_resource_private info = { 0 };
		char *parsed_ticket       = divy_parse_ticket_str(p, screen->ticket);
		apr_uint32_t propflag     = 0;
		divy_rdbo_resource *tmp_rdb_r = NULL;

		if (IS_EMPTY(parsed_ticket)) {
			ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to parse ticket.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* 復号されたチケット(=uri)をパース */
		divy_parse_uri(p, root_uri, parsed_ticket, &t_u_spec);

		/* parsed_ticket が示すURIが必要最低限のアクセス許可条件を
		 * 満たしているかどうかを検証 */
		err = divy_validate_minimum_access_guarantee(r, t_u_spec);
		if (err) return err;

		screen->c_rdb_r = NULL;	/* ファイル or フォルダが判るまでは態度を保留 */

		/* parsed_ticket のリソースを検索 */
		tmp_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
		tmp_rdb_r->uri   = parsed_ticket;
		tmp_rdb_r->depth = divy_count_dirs(tmp_rdb_r->uri);
		tmp_rdb_r->p     = p;
		tmp_rdb_r->u_spec= t_u_spec;
		propflag = DIVY_GET_PROP_fullname;
		if (divy_support_extenduserstatus(r)) {
			propflag |= DIVY_GET_PROP_resourcestate;	/* 状態プロパティも必要 */
		}

		/* アクセス権限チェックを行なうためにカレントフォルダの情報を取得 */
		if (divy_rdbo_get_hierarchy_property(r, tmp_rdb_r, 0, propflag, NULL)) {
			ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get ticket resource.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		else if (IS_EMPTY(tmp_rdb_r->rsid)) {
			return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
		}

		/* このリソースへのアクセスが正当かどうかを検証 */
		resource.exists = 1;
		resource.info   = &info;
		resource.info->rdb_r = tmp_rdb_r;
		if (tmp_rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
			resource.collection = 1;
		}
		else {
			resource.collection = 0;
		}
		err = divy_validate_request_resource(r, &resource);
		if (err) return err;

		/*
		 * チケットが示すファイルの種類に応じて処理の分岐
		 */
		if (tmp_rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
			screen->c_rdb_r = tmp_rdb_r;	/* そのままカレントフォルダ情報として利用する */
		}
		else {
			/* まだ未サポートです */
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/*
		 * チケット貼り付け可能なリソース
		 *  $root/Group Folder/$name
		 *  $root/Group Folder/$name/xxxx,
		 *  グループごみ箱
		 *  $root/$userid
		 *  $root/$userid/xxx
		 *  ユーザごみ箱
		 */
		if (!(t_u_spec->infotype == DIVY_INFOTYPE_group_e         ||
			  t_u_spec->infotype == DIVY_INFOTYPE_group_e_regular ||
			  t_u_spec->infotype == DIVY_INFOTYPE_group_trash     ||
			  t_u_spec->infotype == DIVY_INFOTYPE_user_e          ||
			  t_u_spec->infotype == DIVY_INFOTYPE_user_e_regular  ||
			  t_u_spec->infotype == DIVY_INFOTYPE_user_trash)) {

			ERRLOG0(log_p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Can not access this resource for this ticket");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}
	/* (note) ticket 解析結果を以下に反映します. だからreturn しません */

	/* $root/.st フォルダ|ファイル (特別扱い) */
	if (infotype == DIVY_INFOTYPE_shorten) {

		divy_rdbo_resource *tmp_rdb_r = NULL;

		tmp_rdb_r = divy_convert_shorten(p, r);
		if (tmp_rdb_r == NULL) 
			return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");

		divy_parse_uri(p, root_uri, tmp_rdb_r->uri, &t_u_spec);

		screen->c_rdb_r = tmp_rdb_r;

	}

	/* Autoindex スタートアップページ */
	if (infotype == DIVY_INFOTYPE_root || infotype == DIVY_INFOTYPE_fileviewer) {

		/* 新スタートアップページの場合 */
		if (infotype == DIVY_INFOTYPE_fileviewer) {
			/* uri を$root に入れ替える */
			screen->r_rdb_r->uri = apr_pstrdup(p, root_uri);
		}

		/* 表示データの取得 */
		err = _get_autoindex_startuppage(p, screen, screen->r_rdb_r);
		if (err) return err;

		screen->m_rdb_r = screen->r_rdb_r->next;
		screen->c_rdb_r = screen->r_rdb_r;
		screen->c_rdb_r->next = NULL;	/* 後方メンバとのリンクを切る */
		screen->p_rdb_r = NULL;	/* 親は存在しない */
	}
	/* $root/Group Folder フォルダ */
	else if (infotype == DIVY_INFOTYPE_group) {

		/* 表示データの取得 */
		screen->limit  = -1;
		screen->offset = -1;

		err = _get_autoindex_groupfolder(p, screen, screen->r_rdb_r);
		if (err) return err;

		screen->m_rdb_r = screen->r_rdb_r->next;
		screen->c_rdb_r = screen->r_rdb_r;
		screen->c_rdb_r->next = NULL;

		/* 自分自身のフォルダのContent-Typeを置き換える */
		screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_GRPCOLFOLDER;

		/* 親は表示しない */
		screen->p_rdb_r = NULL;
	}
	/* Utility Folder フォルダ */
	else if (infotype == DIVY_INFOTYPE_utility) {

		/* 表示データの取得 */
		err = _get_autoindex_utilityfolder(p, screen, screen->r_rdb_r);
		if (err) return err;

		screen->m_rdb_r = screen->r_rdb_r->next;
		screen->c_rdb_r = screen->r_rdb_r;
		screen->c_rdb_r->next = NULL;

		/* 親は表示しない */
		screen->p_rdb_r = NULL;
	}
	/* Updated Client フォルダ,
	 * System Message フォルダ */
	else if (infotype == DIVY_INFOTYPE_u_update || infotype == DIVY_INFOTYPE_u_msg) {

		/* 表示データの取得 */
		if (infotype == DIVY_INFOTYPE_u_update) {
			err = _get_autoindex_updateclient(p, screen, screen->r_rdb_r);
		}
		else {
			err = _get_autoindex_sysmsg(p, screen, screen->r_rdb_r);
		}
		if (err) return err;

		screen->m_rdb_r = screen->r_rdb_r->next;
		screen->c_rdb_r = screen->r_rdb_r;
		screen->c_rdb_r->next = NULL;

		/* 親は表示しない */
		screen->p_rdb_r = NULL;
	}
	/* ユーザ/グループごみ箱の場合 */
	else if (infotype == DIVY_INFOTYPE_user_trash ||
			 infotype == DIVY_INFOTYPE_group_trash ||
			 (t_u_spec != NULL && t_u_spec->infotype == DIVY_INFOTYPE_group_trash)) {
		/*
		 * (note) ごみ箱が有効だろうがそうでなかろうが、常に内包リソースは
		 * 	空にすることにします。
		 * 	問題があれば divy_support_trashfolder(r) を上記if 文に加えて。
		 */

		screen->m_rdb_r = NULL;	/* メンバなし */

		/* ごみ箱情報の取得 */
		if (t_u_spec == NULL || screen->c_rdb_r == NULL) {
			/* (note) ticket アクセスではc_rdb_r を取得済み */
			screen->c_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
			screen->c_rdb_r->uri         = apr_pstrdup(p, screen->r_rdb_r->uri);
			screen->c_rdb_r->displayname = apr_pstrdup(p, DIVY_TRASHFOLDER_NAME);
			screen->c_rdb_r->p           = p;
			screen->c_rdb_r->next        = NULL;
		}

		/* 親フォルダの取得 */
		screen->p_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
		screen->p_rdb_r->uri = divy_get_parenturi(p, screen->c_rdb_r->uri);

		/* DBを検索 */
		if (divy_rdbo_get_hierarchy_property(r, screen->p_rdb_r, 0, 0, NULL)) {
			ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get hierarchy resource.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}
	/* $root/.mobile フォルダ */
	else if (infotype == DIVY_INFOTYPE_mobile) {

		/* トップページを表示 */
		if (IS_EMPTY(screen->cmd)) {
			screen->c_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
			screen->c_rdb_r->uri = (char *) root_uri;	/* URIの入れ替え */
			screen->c_rdb_r->p   = p;

			/* Autoindexスタートアップページの情報を取得 */
			err = _get_autoindex_startuppage(p, screen, screen->c_rdb_r);
			if (err) return err;

			screen->m_rdb_r = screen->c_rdb_r->next;
			screen->c_rdb_r->next = NULL;	/* 後方メンバとのリンクを切る */
			screen->p_rdb_r = NULL;	/* 親は存在しない */

			/* uri の書き換え */
			for (n_rdb_r = screen->m_rdb_r; n_rdb_r; n_rdb_r = n_rdb_r->next) {
				/* $root/$userid */
				if (IS_FILLED(n_rdb_r->rsid)) {
					tmp = _encipher_rsid(p, 1, n_rdb_r->rsid);
					n_rdb_r->uri = divy_build_mobile_uri(p, root_uri,
							DIVY_CMD_SHOW_ULIST, tmp,
							DIVY_AUTOINDEX_FIRSTOFFSET);
				}
				/* $root/Group Folder */
				else {
					n_rdb_r->uri = divy_build_mobile_uri(p, root_uri,
							DIVY_CMD_SHOW_GLIST, "0",
							DIVY_AUTOINDEX_FIRSTOFFSET);
				}
			}
		}
		/* グループフォルダ以下の一覧表示 */
		else if (strcmp(screen->cmd, DIVY_CMD_SHOW_GLIST) == 0) {

			screen->c_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
			screen->c_rdb_r->uri = divy_build_group_uri(p, root_uri, NULL);
			screen->c_rdb_r->p   = p;

			/* $root/Group Folder 以下の検索 */
			err = _get_autoindex_groupfolder(p, screen, screen->c_rdb_r);
			if (err && err->status != HTTP_CONTINUE) {
				return err;
			}
			else if (err && err->status == HTTP_CONTINUE) {
				screen->found_extra_data = 1;	/* 続きがあった */
			}

			screen->m_rdb_r = screen->c_rdb_r->next;
			screen->c_rdb_r->next = NULL;

			/* 自分自身のフォルダのContent-Typeを置き換える */
			screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_GRPCOLFOLDER;

			/* 親は表示しない */
			screen->p_rdb_r = NULL;

			/* uri の書き換え */
			i = 0;
			for (n_rdb_r = screen->m_rdb_r; n_rdb_r; n_rdb_r = n_rdb_r->next) {
				tmp = _encipher_rsid(p, (i++ % 10), n_rdb_r->rsid); 
				n_rdb_r->uri = divy_build_mobile_uri(p, root_uri,
							DIVY_CMD_SHOW_SUMMARY, tmp,
							DIVY_AUTOINDEX_FIRSTOFFSET);
			}
		}
		/* 通常の一覧表示, 詳細表示,プライベートフォルダ一覧表示 */
		else if (strcmp(screen->cmd, DIVY_CMD_SHOW_SUMMARY) == 0 ||
			 strcmp(screen->cmd, DIVY_CMD_SHOW_DETAIL) == 0 ||
			 strcmp(screen->cmd, DIVY_CMD_SHOW_ULIST) == 0) {

			/* リソースIDのチェック */
			if (IS_EMPTY(screen->rsid)) {
				ERRLOG1(log_p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to decipher rsid. Maybe invalid access.(crsid = %s)",
					REPLACE_NULL(screen->crsid));
				return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
			}

			/* rdb_r の内容rsid のリソースが表す情報に入れ替える */
			screen->c_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
			screen->c_rdb_r->rsid = screen->rsid;

			/* 一覧 or 詳細データの取得 */
			err = _get_autoindex_mobile_list(p, screen, screen->c_rdb_r);
			if (err) return err;

			screen->m_rdb_r = screen->c_rdb_r->next;
			screen->c_rdb_r->next = NULL;

			if (strcmp(screen->cmd, DIVY_CMD_SHOW_ULIST) == 0) {
				/* 自分自身のフォルダのContent-Typeを置き換える */
				screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_PRIVCOL;
			}

			screen->p_rdb_r = NULL;		/* 親は不要 */

			/* ランダム数の取得 */
			i = (int) (10.0*rand()/(RAND_MAX+1.0));

			/* uri の書き換え */
			if (strcmp(screen->cmd, DIVY_CMD_SHOW_DETAIL) == 0 &&
					screen->c_rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
				tmp = _encipher_rsid(p, i, screen->c_rdb_r->rsid);

				/* 次のリンクはダウンロード */
				screen->c_rdb_r->uri = divy_build_mobile_uri(p, root_uri,
								DIVY_CMD_DOWNLOAD, tmp,
								DIVY_AUTOINDEX_FIRSTOFFSET);
			}

			for (n_rdb_r = screen->m_rdb_r; n_rdb_r; n_rdb_r = n_rdb_r->next) {
				/* rsid の暗号化 */
				tmp = _encipher_rsid(p, (++i % 10), n_rdb_r->rsid);

				if (n_rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
					/* 次は詳細表示 */
					n_rdb_r->uri = divy_build_mobile_uri(p, root_uri,
								DIVY_CMD_SHOW_DETAIL, tmp,
								DIVY_AUTOINDEX_FIRSTOFFSET);
				}
				else {
					/* 常に概要表示 */
					n_rdb_r->uri = divy_build_mobile_uri(p, root_uri,
								DIVY_CMD_SHOW_SUMMARY, tmp,
								DIVY_AUTOINDEX_FIRSTOFFSET);
				}
			}
		}
		/* 不正なコマンド */
		else {
			ERRLOG1(log_p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
				"Invalid cmd (cmd = %s)", screen->cmd);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}
	/* プライベートコレクション */
	else if (infotype == DIVY_INFOTYPE_user_e ||
			(t_u_spec != NULL && t_u_spec->infotype == DIVY_INFOTYPE_user_e)) {

		if (t_u_spec == NULL || screen->c_rdb_r == NULL) {
			screen->c_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
			screen->c_rdb_r->uri   = apr_pstrdup(p, screen->r_rdb_r->uri);
			screen->c_rdb_r->depth = divy_count_dirs(screen->c_rdb_r->uri);
			screen->c_rdb_r->p     = p;
		}

		/* 一覧データの取得 */
		err = _get_autoindex_list(p, screen, screen->c_rdb_r);
		if (err) return err;

		/* 自分自身のフォルダのContent-Typeを置き換える */
		screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_PRIVCOL;

		screen->m_rdb_r = screen->c_rdb_r->next;
		screen->c_rdb_r->next = NULL;

		/* 親フォルダの情報は不要 */
		screen->p_rdb_r = NULL;
	}
	/* グループコレクション */
	else if (infotype == DIVY_INFOTYPE_group_e ||
			 (t_u_spec != NULL && t_u_spec->infotype == DIVY_INFOTYPE_group_e)) {
#ifdef DIVY_SUPPORT_PLUGIN
			tfs_hset_t *groupid_set = NULL;
			divy_rdbo_grp *_grp_pr  = NULL;

			/* セキュリティで保護されたグループかどうかを取得する */
			divy_pi_cciphered_grpset(r, &groupid_set);
#endif	/* DIVY_SUPPORT_PLUGIN */

		if (t_u_spec == NULL || screen->c_rdb_r == NULL) {
			screen->c_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
			screen->c_rdb_r->uri   = apr_pstrdup(p, screen->r_rdb_r->uri);
			screen->c_rdb_r->depth = divy_count_dirs(screen->c_rdb_r->uri);
			screen->c_rdb_r->p     = p;
		}

		/* 一覧データの取得 */
		err = _get_autoindex_list(p, screen, screen->c_rdb_r);
		if (err) return err;

#ifdef DIVY_SUPPORT_PLUGIN
		/* screen->c_rdb_r->uri -> グループプロパティ */
		(void) divy_rdbo_get_group_property_by_resuri(r, screen->c_rdb_r->uri, &_grp_pr);
		/* _screen2xmlで利用するために保持しておきます */
		screen->c_rdb_r->grp_pr = _grp_pr;
		if (groupid_set != NULL) {
			if (_grp_pr != NULL && tfs_hset_contains(groupid_set, _grp_pr->grpid)) {
				screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_SECGRPCOLFOLDER;
			}
			else {
				/* BOXならContent-Typeを変更 */
				if (divy_rdbo_is_box_group(_grp_pr->grp_extstatus)) {
					screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_BOXGRPCOLFOLDER;
				}
				else {
					screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_GRPCOL;
				}
			}
		}
		else {
			/* BOXならContent-Typeを変更 */
			if (divy_rdbo_is_box_group(_grp_pr->grp_extstatus)) {
				screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_BOXGRPCOLFOLDER;
			}
			else {
				screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_GRPCOL;
			}
		}
#else
		/* 自分自身のフォルダのContent-Typeを置き換える */
		screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_GRPCOL;
#endif	/* DIVY_SUPPORT_PLUGIN */

		/* 書き込み制約による影響を受けるか? */
		if (divy_rdbo_has_read_privilege(divy_get_extstatus(r)) ||
			divy_rdbo_has_write_constraints_on_uri(r, screen->c_rdb_r->uri, DIVY_INFOTYPE_group_e)) {
			if (divy_rdbo_is_box_group(_grp_pr->grp_extstatus)) {
				screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_ROBOXGRPCOLFOLDER;
			}
			else {
				/* セキュリティフォルダよりも、グループフォルダよりも最優先される */
				screen->c_rdb_r->getcontenttype = DIVY_CONTTYPE_ROGRPCOLFOLDER;
			}
		}

		screen->m_rdb_r = screen->c_rdb_r->next;
		screen->c_rdb_r->next = NULL;

		/* 親フォルダの取得 */
		screen->p_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
		screen->p_rdb_r->uri =  divy_get_parenturi(p, screen->c_rdb_r->uri);
		screen->p_rdb_r->p   = p;
		screen->p_rdb_r->displayname    = divy_get_sfolder_name(p, DIVY_FOLDER_ID_group);
		screen->p_rdb_r->resourcetype   = DIVY_TYPE_COLLECTION;
		screen->p_rdb_r->getcontenttype = DIVY_CONTTYPE_GRPCOLFOLDER;
		screen->p_rdb_r->next = NULL;

		/* URI の解析 */
		divy_parse_uri(p, root_uri, screen->p_rdb_r->uri, &screen->p_rdb_r->u_spec);
	}
	/* 上記以外のフォルダ */
	else {
		if (t_u_spec == NULL || screen->c_rdb_r == NULL) {
			screen->c_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
			screen->c_rdb_r->uri   = apr_pstrdup(p, screen->r_rdb_r->uri);
			screen->c_rdb_r->depth = divy_count_dirs(screen->c_rdb_r->uri);
			screen->c_rdb_r->p     = p;
		}

		/* 一覧データの取得 */
		err = _get_autoindex_list(p, screen, screen->c_rdb_r);
		if (err) return err;

		screen->m_rdb_r = screen->c_rdb_r->next;
		screen->c_rdb_r->next = NULL;

		/* 親フォルダの取得 */
		screen->p_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
		screen->p_rdb_r->uri =  divy_get_parenturi(p, screen->c_rdb_r->uri);
		screen->p_rdb_r->p   = p;

		/* URI の解析 */
		divy_parse_uri(p, root_uri, screen->p_rdb_r->uri, &screen->p_rdb_r->u_spec);

		/* 親フォルダが$root -> スタートアップページに差し替える */
		if (screen->p_rdb_r->u_spec->infotype == DIVY_INFOTYPE_root) {
			screen->p_rdb_r->uri = divy_build_autoindex_startup_uri(p, root_uri);
			screen->p_rdb_r->displayname  = dav_divy_extract_finalpath_segment(p, root_uri);
			screen->p_rdb_r->resourcetype = DIVY_TYPE_COLLECTION;
		}
		/* 親がプライベートコレクションであれば、Content-Typeを入れ替える */
		else if (screen->p_rdb_r->u_spec->infotype == DIVY_INFOTYPE_user_e) {
			screen->p_rdb_r->getcontenttype = DIVY_CONTTYPE_PRIVCOL;
			screen->p_rdb_r->resourcetype   = DIVY_TYPE_COLLECTION;
		}
		else {
			/* (note) Autoindex のスタートアップページは階層化されないので
			   ここに来ることはありません */
			/* DBを検索 */
			if (divy_rdbo_get_hierarchy_property(r, screen->p_rdb_r, 0, 0, NULL)) {
				ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get hierarchy resource.");
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}
		}
	}

	return NULL;
}

/**
 * Autoindex スタートアップページに対するGETを処理する。
 *
 * [ 返却するコレクションの種類 ]
 * 	・$root (rdb_r->rsid が存在しない場合だけ検索を実施)
 * 	・$root/$userid
 * 	・$root/Group Folder (特殊ディレクティブが存在する場合)
 * 	・$root/Utility Folder
 *
 * @param p apr_pool_t * 作業用のプール
 * @param screen divy_autoindex_screen * 画面入力情報を持つ構造体
 * @param rdb_r divy_rdbo_resource * 検索条件とその結果を保持するリソース
 * @return dav_error * DAVエラー (HTTP_INTERNAL_SERVER_ERROR)
 * 			エラーがなければNULL
 */
static dav_error * _get_autoindex_startuppage(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r)
{
	request_rec *r    = NULL;
	apr_pool_t *log_p = NULL;
	divy_rdbo_resource *_rdb_r = NULL;

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	r     = screen->r;
	log_p = screen->r->pool;

	/*
	 * $root の検索
	 * (note)
	 * 	$root自身(rdb_r) には、特殊プロパティが付かないので
	 * 	rdb_r->rsid が存在する限り、何もしないことにしました。
	 */
	if (IS_EMPTY(rdb_r->rsid) &&
		divy_rdbo_get_hierarchy_property(r, rdb_r, 0, 0, NULL)) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get $root resource.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * $root/$userid の情報作成
	 * (note) 但しプライベートフォルダを見せてもよいと記述されたときだけ。
	 */
	if (divy_enable_userfolderview(r)) {
		_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource)); 
		_rdb_r->uri = divy_build_user_uri(p, dav_divy_get_root_uri(r),
						divy_get_userid(r));
		_rdb_r->p   = p;

		/* DB から情報を取得する */
		if (divy_rdbo_get_hierarchy_property(r, _rdb_r, 0, 0, NULL) ||
				IS_EMPTY(_rdb_r->rsid)) {
			ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get private-folder resource."
					"(userid = %s)", divy_get_userid(r));
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		/* 自分自身のフォルダのContent-Typeを置き換える */
		_rdb_r->getcontenttype = DIVY_CONTTYPE_PRIVCOL;

		/* リストに繋げる */
		rdb_r->next = _rdb_r;
		rdb_r = rdb_r->next;
	}

	/*
	 * $root/Group Folder の情報の作成
	 * (note) 但しグループフォルダを見せてもよいと記述されたときだけ。
	 */
	if (divy_enable_groupview(r)) {

		/* グループトップフォルダの取得 */
		_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
		_rdb_r->uri            = divy_build_group_uri(p, dav_divy_get_root_uri(r), NULL);
		_rdb_r->displayname    = divy_get_sfolder_name(p, DIVY_FOLDER_ID_group);
		_rdb_r->resourcetype   = DIVY_TYPE_COLLECTION;
		_rdb_r->getcontenttype = DIVY_CONTTYPE_GRPCOLFOLDER;
		_rdb_r->p              = p;

		/* リストに繋げる */
		rdb_r->next = _rdb_r;
		rdb_r = rdb_r->next;
	}

	/* 2006/04/26 Wed 以下は不要になっていたことを忘れていました */
#if 0
	/*
	 * $root/Utility Folder の情報作成
	 */
	if (screen->r_u_spec && screen->r_u_spec->infotype != DIVY_INFOTYPE_mobile) {

		_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
		_rdb_r->uri = divy_build_utility_uri(p, dav_divy_get_root_uri(r));
		_rdb_r->displayname    = divy_get_sfolder_name(p, DIVY_FOLDER_ID_utility);
		_rdb_r->resourcetype   = DIVY_TYPE_COLLECTION;
		_rdb_r->getcontenttype = DIR_MAGIC_TYPE;
		_rdb_r->p            = p;

		/* リストに繋げる */
		rdb_r->next = _rdb_r;
		rdb_r = rdb_r->next;
	}
#endif

	return NULL;
}

/**
 * $root/Group Folder に対するGETを処理する。
 *
 * screen->limit > 0 の時には、screen->offset から screen->limit 個の
 * グループコレクションのみを取得する
 *
 * (note) 返却するコレクションの種類
 * 	* offset は 1から始まります
 * 	* $root/Group Folder/$groupid (特殊ディレクティブが存在する場合)
 *
 * @param p apr_pool_t * 作業用プール
 * @param screen divy_autoindex_screen * 画面入力情報を持つ構造体
 * @param rdb_r divy_rdbo_resource
 * @return dav_error * 
 * 	HTTP_INTERNAL_SERVER_ERROR: 内部エラー
 * 	HTTP_CONTINUE : (limit が正だった時、limit件以上のデータが存在した。正常)
 * 	エラーがなければNULL
 */
static dav_error * _get_autoindex_groupfolder(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r)
{
	request_rec *r    = NULL;
	apr_pool_t *log_p = NULL;
	divy_rdbo_resource *grp_rdb_r = NULL, *g;
	int supported_box = 0; /* unsupport */

	if (screen == NULL || rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen or rdb_r is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	r     = screen->r;
	log_p = r->pool;

	/* BOXのサポートを確認する */
	supported_box = divy_support_tfbox(r);

	/* $root/Group Folder の検索 --> 取得済みなのでdisplayname を調整する */
	rdb_r->displayname = divy_get_sfolder_name(p, DIVY_FOLDER_ID_group);

	/*
	 * $root/Group Folder/$groupid の検索
	 * (note) 但しグループフォルダを見せてもよいと記述されたときだけ。
	 */
	if (divy_enable_groupview(r)) {
		int ret;
		const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
#ifdef DIVY_SUPPORT_PLUGIN
			tfs_hset_t *groupid_set = NULL;

			/* セキュリティで保護されたグループかどうかを取得する */
			divy_pi_cciphered_grpset(r, &groupid_set);
#endif	/* DIVY_SUPPORT_PLUGIN */

		/* グループ一覧の取得 */
		ret = divy_rdbo_get_group_resource_by_userid(r, divy_get_userid(r),
				screen->offset, screen->limit, &grp_rdb_r);
		if (ret == DIVY_STCODE_REMAIN_DATA) {
			/* 残りのデータが存在した場合 */
			rdb_r->next = grp_rdb_r;
			return dav_new_error(p, HTTP_CONTINUE, 0, 0, ""); /* 正常 */
		}
		else if (ret != 0) {
			ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get group hierarchy resource.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		/* Content-Type を強制的に置き換える */
		for (g = grp_rdb_r; g != NULL; g = g->next) {
#ifdef DIVY_SUPPORT_PLUGIN
			if (groupid_set != NULL &&
				tfs_hset_contains(groupid_set, g->grp_pr->grpid)) {
				/* セキュリティで保護されたフォルダ用のContent-Typeを指定 */
				g->getcontenttype = DIVY_CONTTYPE_SECGRPCOLFOLDER;
			}
			else {
				/* BOXをサポートしているならContent-Typeを変更 */
				if (supported_box &&
							divy_rdbo_is_box_group(g->grp_pr->grp_extstatus)) {
					g->getcontenttype = DIVY_CONTTYPE_BOXGRPCOLFOLDER;
				}
				else {
					g->getcontenttype = DIVY_CONTTYPE_GRPCOL;
				}
			}
#else
			/* BOXをサポートしているならContent-Typeを変更 */
			if (supported_box &&
						divy_rdbo_is_box_group(g->grp_pr->grp_extstatus)) {
				g->getcontenttype = DIVY_CONTTYPE_BOXGRPCOLFOLDER;
			}
			else {
				g->getcontenttype = DIVY_CONTTYPE_GRPCOL;
			}
#endif	/* DIVY_SUPPORT_PLUGIN */

			/* 書き込み制約による影響を受けるか? */
			if (divy_rdbo_has_read_privilege(own_extstatus) ||
				divy_rdbo_has_write_constraints_on_uri(r, g->uri, DIVY_INFOTYPE_group_e)) {
				/* セキュリティフォルダよりも、グループフォルダよりも最優先される */
				if (supported_box &&
							divy_rdbo_is_box_group(g->grp_pr->grp_extstatus)) {
					g->getcontenttype = DIVY_CONTTYPE_ROBOXGRPCOLFOLDER;
				}
				else {
					g->getcontenttype = DIVY_CONTTYPE_ROGRPCOLFOLDER;
				}
			}
		}
	}

	/* rdb_r に取得リストを繋げる */
	rdb_r->next = grp_rdb_r;

	return NULL;
}

/**
 * $root/Utility Folder に対するGETを処理する。
 *
 * (note) 返却するコレクションの種類
 * 	* $root/Utility Folder
 * 	* $root/Utility Folder/Updated Client
 * 	* $root/Utility Folder/System Message
 *
 * @param p apr_pool_t * 作業用プール
 * @param depth int 検索深さ (0: Utility Folderのみ / 1: メンバも含む)
 * @param screen divy_autoindex_screen * 画面入力情報を持つ構造体
 * @param rdb_r divy_rdbo_resource
 * @return dav_error * 
 */
static dav_error * _get_autoindex_utilityfolder(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r)
{
	request_rec *r    = NULL;
	apr_pool_t *log_p = NULL;
	divy_rdbo_resource *_rdb_r = NULL;
	int msgcnt        = 0;

	if (screen == NULL || rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen or rdb_r is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	r     = screen->r;
	log_p = r->pool;

	/* $root/Utility Folder のdisplayname を調整する */
	rdb_r->displayname    = divy_get_sfolder_name(p, DIVY_FOLDER_ID_utility);
	rdb_r->getcontenttype = DIR_MAGIC_TYPE;

	/*
	 * $root/Utility Folder/Updated Client の情報を作成
	 */
	_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
	_rdb_r->uri = divy_build_u_update_uri(p, dav_divy_get_root_uri(r));
	_rdb_r->displayname    = divy_get_sfolder_name(p, DIVY_FOLDER_ID_u_update);
	_rdb_r->resourcetype   = DIVY_TYPE_COLLECTION;
	_rdb_r->getcontenttype = DIVY_CONTTYPE_UPDATEDCL;
	_rdb_r->p            = p;

	/* リストに繋げる */
	rdb_r->next = _rdb_r;
	rdb_r = rdb_r->next;

	/*
	 * $root/Utility Folder/System Message の情報を作成
	 *
	 */
	/* アクティブなシステムメッセージ数を取得する */
	if (divy_rdbo_count_active_sysmsg(r, &msgcnt)) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get system message information.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	_rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
	_rdb_r->uri = divy_build_u_msg_uri(p, dav_divy_get_root_uri(r));
	_rdb_r->displayname    = apr_psprintf(p, "%s ( %d )",
			divy_get_sfolder_name(p, DIVY_FOLDER_ID_u_msg), msgcnt);
	_rdb_r->resourcetype   = DIVY_TYPE_COLLECTION;
	_rdb_r->getcontenttype = DIVY_CONTTYPE_SYSMSGFOLDER;
	_rdb_r->p            = p;

	/* リストに繋げる */
	rdb_r->next = _rdb_r;
	rdb_r = rdb_r->next;

	return NULL;

}

/**
 * $root/Utility Folder/Updated Client に対するGETを処理する。
 *
 * (note) 返却するコレクションの種類
 * 	* $root/Utility Folder/Updated Client
 * 	* $root/Utility Folder/Updated Client/$modulename
 *
 * @param p apr_pool_t * 作業用プール
 * @param screen divy_autoindex_screen * 画面入力情報を持つ構造体
 * @param rdb_r divy_rdbo_resource
 * @return dav_error * 
 */
static dav_error * _get_autoindex_updateclient(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r)
{
	request_rec *r    = NULL;
	apr_pool_t *log_p = NULL;
	apr_uint32_t propflag = 0;
	char *base_uri;
	divy_rdbo_sortorder sort  = { 0 };

	if (screen == NULL || rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen or rdb_r is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	r     = screen->r;
	log_p = r->pool;

	/*
	 * Updated Client 以下の情報取得する
	 * --> $root/.management/UPDATE 以下の情報を取得すると等価
	 */
	rdb_r->uri   = divy_build_m_update_uri(p, dav_divy_get_root_uri(r), NULL);
	rdb_r->depth = divy_count_dirs(rdb_r->uri);
	rdb_r->p     = p;
	rdb_r->next  = NULL;

	/* DBを検索する */
	propflag |= DIVY_GET_PROP_fullname;	/* 作成者名を取得する */

	sort.key   = DIVY_SORTKEY_RESTYPE;
	sort.order = DIVY_SORTORDER_ASC;

	if (divy_rdbo_get_hierarchy_property(r, rdb_r, 1, propflag, &sort)) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get updated client list.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * URI, 表示名を変換する(作り直す)
	 *
	 * $root/.management/UPDATE/ --> $root/Utility Folder/Updated Client/
	 */
	base_uri = divy_build_u_update_uri(p, dav_divy_get_root_uri(r));

	/* 先頭がUpdated Client (divy_rdbo_get_hierarchy_property が保証しています) */
	rdb_r->uri            = base_uri;
	rdb_r->displayname    = divy_get_sfolder_name(p, DIVY_FOLDER_ID_u_update);
	rdb_r->depth          = divy_count_dirs(rdb_r->uri);
	rdb_r->getcontenttype = DIVY_CONTTYPE_UPDATEDCL;
	for (rdb_r = rdb_r->next; rdb_r; rdb_r = rdb_r->next) {
		rdb_r->uri   = dav_divy_make_uri(p, base_uri,
						rdb_r->displayname, NULL);
		rdb_r->depth = divy_count_dirs(rdb_r->uri);
	}

	return NULL;
}

/**
 * $root/Utility Folder/System Message に対するGETを処理する。
 *
 * (note) 返却するコレクションの種類
 * 	* $root/Utility Folder/System Message
 * 	* $root/Utility Folder/System Message/$modulename
 *
 * @param p apr_pool_t * 作業用プール
 * @param screen divy_autoindex_screen * 画面入力情報を持つ構造体
 * @param rdb_r divy_rdbo_resource
 * @return dav_error * 
 */
static dav_error * _get_autoindex_sysmsg(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r)
{
	request_rec *r    = NULL;
	apr_pool_t *log_p = NULL;
	divy_rdbo_sysmsg *sysmsg_pr = NULL;
	char *base_uri;

	if (screen == NULL || rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen or rdb_r is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	r     = screen->r;
	log_p = r->pool;

	/* System Message のdisplayname を調整する */
	rdb_r->displayname = divy_get_sfolder_name(p, DIVY_FOLDER_ID_u_msg);

	/*
	 * System Message 以下の情報取得する
	 */
	if (divy_rdbo_get_active_sysmsg(r, &sysmsg_pr)) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get system message list.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * sysmsg_pr を rdb_r に焼き直す
	 */
	base_uri = divy_build_u_msg_uri(p, dav_divy_get_root_uri(r));

	/* 先頭がSystem Message (divy_rdbo_get_hierarchy_property が保証しています) */
	rdb_r->uri            = base_uri;
	rdb_r->displayname    = divy_get_sfolder_name(p, DIVY_FOLDER_ID_u_msg);
	rdb_r->depth          = divy_count_dirs(rdb_r->uri);
	rdb_r->getcontenttype = DIVY_CONTTYPE_SYSMSGFOLDER;

	for (; sysmsg_pr ; sysmsg_pr = sysmsg_pr->next) {
		rdb_r->next = apr_pcalloc(p, sizeof(divy_rdbo_resource));
		rdb_r = rdb_r->next;

		rdb_r->uri = dav_divy_make_uri(p, base_uri, sysmsg_pr->msgid, NULL);
		rdb_r->displayname     = sysmsg_pr->msg;
		rdb_r->resourcetype    = DIVY_TYPE_RESOURCE;
		rdb_r->getcontenttype  = DIVY_CONTTYPE_SYSMSG;
		rdb_r->getlastmodified = dav_divy_iso8601totime_t(p, sysmsg_pr->updatedt);

		rdb_r->next = NULL;
	}

	return NULL;
}

/**
 * 一覧＆詳細の表示に対するGETを処理する。
 *
 * (note) 返却するコレクションの種類
 *
 * @param p apr_pool_t * 作業用プール
 * @param screen divy_autoindex_screen * 画面入力情報を持つ構造体
 * @param rdb_r divy_rdbo_resource *
 * @return dav_error * 
 */
static dav_error * _get_autoindex_list(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r)
{
	request_rec *r            = NULL;
	apr_pool_t *log_p         = NULL;
	apr_uint32_t propflag     = 0;
	divy_rdbo_sortorder sort  = { 0 };

	if (screen == NULL || rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen or rdb_r is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	r     = screen->r;
	log_p = r->pool;

	/* フルネームを取得する */
	propflag |= DIVY_GET_PROP_fullname;

	if (rdb_r->u_spec == NULL) {
		if (divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &rdb_r->u_spec)) {
			ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to parse uri.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	/* 状態・属性プロパティを取得するかどうか */
	if (divy_support_extenduserstatus(r) &&
			(rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e_regular ||
			 rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e)) {
		propflag |= DIVY_GET_PROP_resourcestate;
		propflag |= DIVY_GET_PROP_mailwatch;

		/* 制限ユーザであればView属性の付いていないリソースは削り落とす */
		if (!divy_rdbo_is_trusty_user(divy_get_extstatus(r))) {
			propflag |= DIVY_DEL_PROP_noviewattr;
		}
	}

	/* ごみ箱フォルダ自身を除外するかどうか */
	if ((!divy_enable_trashfolder_access(r) || !divy_support_trashfolder(r)) &&
			(rdb_r->u_spec->infotype == DIVY_INFOTYPE_user_e ||
			 rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e)) {
		propflag |= DIVY_GET_PROP_notrashfolder;
	}

	sort.key   = DIVY_SORTKEY_RESTYPE;
	sort.order = DIVY_SORTORDER_ASC;

	/* リソース一覧を取得 */
	if (divy_rdbo_get_hierarchy_property(r, rdb_r, 1, propflag, &sort)) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get hierarchy resource.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/**
 * $root/.mobile (一覧＆詳細の表示) に対するGETを処理する。
 *
 * (note) 返却するコレクションの種類
 *
 * @param p apr_pool_t * 作業用プール
 * @param screen divy_autoindex_screen * 画面入力情報を持つ構造体
 * @param rdb_r divy_rdbo_resource *
 * @return dav_error * 
 */
static dav_error * _get_autoindex_mobile_list(apr_pool_t *p,
					divy_autoindex_screen *screen,
					divy_rdbo_resource *rdb_r)
{
	dav_error *err            = NULL;
	request_rec *r            = NULL;
	apr_pool_t *log_p         = NULL;
	apr_uint32_t propflag     = 0, propflag0 = 0;
	dav_resource resource     = { 0 };
	dav_resource_private info = { 0 };
	int is_detail             = 0, ret;
	divy_infotype infotype;

	if (screen == NULL || rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen or rdb_r is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	r     = screen->r;
	log_p = r->pool;

	/* 詳細表示か? */
	if (strcmp(screen->cmd, DIVY_CMD_SHOW_DETAIL) == 0) {
		propflag |= DIVY_GET_PROP_fullname;
		is_detail = 1;
	}

	/* (note) rsid しかわからないのでフラグを付けていいかどうかはきちんと判定できません */
	propflag0 |= DIVY_GET_PROP_fullname;		/* フルネームは必要 */
	if (divy_support_extenduserstatus(r)) {
		propflag0 |= DIVY_GET_PROP_resourcestate;	/* 状態プロパティも必要(かもしれない) */
	}

	/*
	 * モバイル経由のリクエスト(.mobile?cmd=xxx) をWebDAV経由のリクエスト(/xxx/yyyy) に変換する
	 * (note)
	 *   モバイル経由のリクエストは特殊であるため、あたかもWebDAV経由で
	 *   アクセスされたかのようにし、検証ロジックの再利用を図る
	 */
	/* (1) rsid -> uri 変換 (rsid のリソース取得) */
	if (divy_rdbo_get_property_by_rsid(r, rdb_r, propflag0)) {
		ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get resource by rsid.(rsid = %s)", rdb_r->rsid);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	else if (IS_EMPTY(rdb_r->uri)) {
		return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}

	/* (2) uri のパース */
	if (divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &rdb_r->u_spec)) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse uri.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* (3) uri が必要最低限のアクセス許可条件を満たしているかどうかを検証 */
	err = divy_validate_minimum_access_guarantee(r, rdb_r->u_spec);
	if (err) return err;

	/* (4) このリソースへのアクセスが正当かどうか検証 */
	resource.exists = 1;
	resource.info   = &info;
	resource.info->rdb_r = rdb_r;
	if (rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
		resource.collection = 1;
	}
	err = divy_validate_request_resource(r, &resource);
	if (err) return err;

	/* rdb_r->uri の"下"のリソースを取得 */
	if (!is_detail &&
		(rdb_r->u_spec->infotype == DIVY_INFOTYPE_user_trash ||
		 rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_trash)) {
		/*
		 * (note) ごみ箱が有効だろうがそうでなかろうが、常に
		 * 	内包リソースは空にすることにします。問題があれば
		 * 	divy_support_trashfolder(r) を上記if 文に加えて。
		 */
		/* ごみ箱の中身は空 */
		rdb_r->next = NULL;	/* メンバは表示しない */
	}
	else if (!is_detail) {
		infotype = rdb_r->u_spec->infotype;

		if (divy_support_extenduserstatus(r) &&
			(infotype == DIVY_INFOTYPE_group_e_regular || infotype == DIVY_INFOTYPE_group_e)) {
			propflag |= DIVY_GET_PROP_resourcestate;

			/* 制限ユーザであればView属性の付いていないリソースは削り落とす */
			if (!divy_rdbo_is_trusty_user(divy_get_extstatus(r))) {
				propflag |= DIVY_DEL_PROP_noviewattr;
			}
		}

		/* リソース一覧の取得 */
		ret = divy_rdbo_get_autoindex_data(r, rdb_r, screen->offset, screen->limit, propflag);
		if (ret == DIVY_STCODE_REMAIN_DATA) {
			screen->found_extra_data = 1;	/* 続きがある */
		}
		else if (ret != 0) {
			ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get autoindex data.(uri = %s, depth = %d)",
				rdb_r->uri, rdb_r->depth);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	return NULL;
}


/**
 * screen の内容を元に、XMLタグが書き込まれたXMLファイルを出力し
 * そのファイルへのパスを返却する。
 *
 * @param p apr_pool_t *
 * @param fstorage divy_fstorage_t * 論理ストレージ
 * @param screen divy_autoindex_screen *
 * @param outputxml divy_outputxml * 生成したXMLファイルパスまたはXML文字列を格納する入れ物
 * @return DAVエラー
 * 	エラー発生時ならデータ部(d.sbuf / d.filepath) はNULLです。
 */
static dav_error * _screen2xml(apr_pool_t *p,
					divy_fstorage_t *fstorage,
					divy_autoindex_screen *screen,
					divy_outputxml *outputxml)
{
	apr_status_t rv;
	dav_error *err              = NULL;
	request_rec *r              = NULL;
	apr_pool_t *log_p           = NULL;
	divy_rdbo_resource *n_rdb_r = NULL;
	divy_sbuf *sbuf             = NULL;
	char *datebuf               = NULL;
	int ret, is_detail, hide_uri = 0;
	int i = 0;
	int first = 0;
	divy_infotype infotype;
	divy_rdbo_resource *c_rdb_r, *p_rdb_r;
	divy_pfile_t *pfile         = NULL;
	apr_file_t *fd              = NULL;
	apr_size_t sbuf_len;
	char *filepath              = NULL;
	char *brandname, *ext, *last;
	apr_hash_t *mime_map  = NULL;
	const divy_cmap_elem *map_e = NULL;
	dav_divy_server_conf *sconf;
	dav_divy_dir_conf *dconf;
	const divy_rdbo_extstatus *extstatus;
	int is_write_constraints = 0;
	char *ticketurl;
	char *boxmessage;
	apr_hash_t *parsed_h = NULL;
	divy_linkedlist_t *ll;
	divy_rdbo_box *box_pr = NULL;
#ifdef DIVY_SUPPORT_EXTENDQUOTA
	divy_rdbo_diskquota squota = { 0 };	/* システムQuota */
#endif
	char *sesspw, *sessid;
	divy_auth_session session = { 0 };

	if (screen == NULL || outputxml == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"screen is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	r     = screen->r;
	log_p = r->pool;
	dconf = dav_divy_get_dir_config(r);
	sconf = dav_divy_get_server_config(r->server);
	extstatus = divy_get_extstatus(r);
	const char* e_uri = NULL;

	if (screen->c_rdb_r == NULL) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"The current folder information is EMPTY.");
		return dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* アップロードユーザでグループフォルダ以下ならばhrefを隠す */
	if (divy_support_extenduserstatus(r) && divy_rdbo_has_upload_privilege(extstatus)) {
		if (screen->c_rdb_r->u_spec == NULL) {
			divy_parse_uri(p, dav_divy_get_root_uri(r),
					screen->c_rdb_r->uri, &screen->c_rdb_r->u_spec);
		}

		if (screen->c_rdb_r->u_spec &&
			(screen->c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e ||
			 screen->c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e_regular)) {
			hide_uri = 1;
		}
	}

	infotype = screen->r_rdb_r->u_spec->infotype;
	outputxml->type = DIVY_XMLOUTPUT_TYPE_UNKNOWN;	/* 初期化 */

	/*
	 * XML出力用一時ファイルの論理パスを作成
	 */
	ret = divy_pfile_mkfixedtemp(fstorage, p, DIVY_TMP_FILE_TEMPLATE_NAME, &pfile);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to make a temporary xml file.(uri = %s, "
			"userid = %s)", r->uri, divy_get_userid(r));
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_screen2xml_resource;
	}
	/* パス文字列のテンプレートを作成
	 * (note) apr_file_mktemp の第2引数は書き換えられるのでコピーする */
	filepath = apr_pstrdup(p, divy_pfile_get_fullpath(pfile));

	/* バッファの生成 (ファイル出力の閾値よりも少し大きくしておかないと性能が落ちる) */
	divy_sbuf_create(p, &sbuf, XMLBUF_BOUNDARY_SIZE + 1024);

	/* エンコードを取得 */
	if (IS_FILLED(screen->encoding)) {
		divy_sbuf_append(sbuf, "<?xml version=\"1.0\" encoding=\"");
		divy_sbuf_append(sbuf, screen->encoding);
		divy_sbuf_append(sbuf, "\"?>"CRLF);
		divy_sbuf_append(sbuf,
			"<?xml-stylesheet type=\"text/xsl\"?>"CRLF
			"<"DIVY_NS_PREFIX":propdiscovery "
			"xmlns:"DAV_NS_PREFIX"=\"DAV:\" "
			"xmlns:TF=\""DIVY_NS_DEFAULT_URI"\">"CRLF);
	}
	/* デフォルトはUTF-8 */
	else {
		divy_sbuf_append(sbuf,
			DAV_XML_HEADER CRLF
			"<?xml-stylesheet type=\"text/xsl\"?>"CRLF
			"<"DIVY_NS_PREFIX":propdiscovery "
			"xmlns:"DAV_NS_PREFIX"=\"DAV:\" "
			"xmlns:TF=\""DIVY_NS_DEFAULT_URI"\">"CRLF);

	}

	/* 詳細表示かどうかの判定 */
	is_detail = 0;
	if (IS_FILLED(screen->cmd) && strcmp(screen->cmd, DIVY_CMD_SHOW_DETAIL) == 0) {
		is_detail = 1;
	}

	/* 拡張子、Content-Typeマップを読み込む */
	mime_map = divy_read_mimetype_map(r);

	/*
	 * カレントフォルダ用XMLの生成
	 */
	c_rdb_r = screen->c_rdb_r;

	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":currentinfo>" CRLF);
	if (IS_FILLED(screen->encoding)) {
		c_rdb_r->displayname = _encode_property(p, screen->encoding,
							c_rdb_r->displayname,
							DIVY_AUTOINDEX_FAILEDCONV_STR);
		c_rdb_r->uri = _encode_property(p, screen->encoding,
						c_rdb_r->uri, DIVY_AUTOINDEX_FAILEDCONV_URI);

		if (is_detail) {
			c_rdb_r->lastmodifier = _encode_property(p, screen->encoding,
								c_rdb_r->lastmodifier,
								DIVY_AUTOINDEX_FAILEDCONV_STR);
		}
	}

	if (c_rdb_r->u_spec == NULL) {
		divy_parse_uri(p, dav_divy_get_root_uri(r), c_rdb_r->uri, &c_rdb_r->u_spec);
	}

	/* contenttype データを取得 */
	last = divy_strrchr_c(c_rdb_r->displayname, '.');
	ext = (last != NULL) ? ++last : NULL;

	map_e = divy_get_extmapinfo(p, mime_map, c_rdb_r->getcontenttype, ext);
	if (map_e != NULL && IS_FILLED(map_e->ctype_name)) {
		if (IS_FILLED(screen->encoding)) {
			c_rdb_r->getcontenttype = _encode_property(p, screen->encoding,
								map_e->ctype_name,
								DIVY_AUTOINDEX_FAILEDCONV_URI);
		}
		else {
			c_rdb_r->getcontenttype = map_e->ctype_name;
		}
	}

	/* 表示名 */
	divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":displayname>");
	/* プライベートコレクション, Group Folder, Updated Client, System Message,
	 * モバイル(プライベートコレクション、グループコレクション) の場合
	 * Content-Typeの種類名称を名前に使う(現CGI構成に依存した特殊な処理) */
	if (c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_user_e   ||
		c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group    ||
		c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_u_update ||
		c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_u_msg    ||
		(c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_mobile &&
		 IS_FILLED(screen->cmd) &&
		 (strcmp(screen->cmd, DIVY_CMD_SHOW_ULIST) == 0 ||
		  strcmp(screen->cmd, DIVY_CMD_SHOW_GLIST) == 0))) {
		divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, c_rdb_r->getcontenttype, DIVY_XML_T2T_QUOTE));
	}
	else {
		divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, c_rdb_r->displayname, DIVY_XML_T2T_QUOTE));
	}
	divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":displayname>"CRLF);


	/* resourceタイプ(フォルダの種類) */

	if (c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_u_msg) {
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype>"
						"<"DIVY_NS_PREFIX":sysmsgcollection/>"
						"</"DAV_NS_PREFIX":resourcetype>"CRLF);
	}
	/* 更新クライアントフォルダ */
	else if (c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_u_update) {
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype>"
						"<"DIVY_NS_PREFIX":updateclientcollection/>"
						"</"DAV_NS_PREFIX":resourcetype>"CRLF);
	}
	/* グループコレクション格納フォルダ */
	else if (c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group) {
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype>"
						"<"DIVY_NS_PREFIX":groupfoldercollection/>"
						"</"DAV_NS_PREFIX":resourcetype>"CRLF);
	}
	/* グループコレクション直下のフォルダ */
	else if (c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e) {
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype>"
						"<"DIVY_NS_PREFIX":groupcollection/>"
						"</"DAV_NS_PREFIX":resourcetype>"CRLF);
	}
	/* グループコレクション以下のフォルダ */
	else if (c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_trash ||
			 c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
			 c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_trash_ex ||
			 c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype>"
						"<"DIVY_NS_PREFIX":groupcollection1/>"
						"</"DAV_NS_PREFIX":resourcetype>"CRLF);
	}
	/* Autoindex スタートアップページ */
	else if (c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_root ||
			 c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_fileviewer) {
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype>"
						"<"DIVY_NS_PREFIX":rootcollection/>"
						"</"DAV_NS_PREFIX":resourcetype>"CRLF);
	}
	/* Utility Folder (互換性のため) */
	else if (c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_utility) {
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype>"
						"<"DIVY_NS_PREFIX":utilitycollection/>"
						"</"DAV_NS_PREFIX":resourcetype>"CRLF);
	}
	/* 通常コレクション */
	else if (c_rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype>"
					"<"DAV_NS_PREFIX":collection/>"
					"</"DAV_NS_PREFIX":resourcetype>"CRLF);
	}
	else {
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype/>"CRLF);
	}

	/* URL */
	divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":href>");
	_build_response_url(p, sbuf, infotype, c_rdb_r, screen->seq, hide_uri, 0);
	divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":href>"CRLF);

	/* contenttype */
	if (IS_FILLED(c_rdb_r->getcontenttype)) {
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":getcontenttype>");
		divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p,
				c_rdb_r->getcontenttype, DIVY_XML_T2T_QUOTE));
		divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":getcontenttype>"CRLF);
	}

	/* summaryicon */
	if (map_e != NULL && IS_FILLED(map_e->img_path)) {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":summaryicon>");
				divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p,
				map_e->img_path, DIVY_XML_T2T_QUOTE));
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":summaryicon>");
	}

	if (is_detail) {
		/* サイズ */
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":getcontentlength>");
		divy_sbuf_append(sbuf, divy_byte2displaysize(p, c_rdb_r->getcontentlength, 1));
		divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":getcontentlength>"CRLF);

		/* 更新日付 */
		if (c_rdb_r->getlastmodified > 0) {
			divy_format_time_t(p, c_rdb_r->getlastmodified,
						DIVY_TIME_STYLE_JAPANESE, &datebuf);

			divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":getlastmodified>");
			divy_sbuf_append(sbuf, datebuf);
			divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":getlastmodified>"CRLF);
		}

		/* lastmodifier */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":lastmodifier>");
		divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p,
				c_rdb_r->lastmodifier, DIVY_XML_T2T_QUOTE));
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":lastmodifier>"CRLF);
	}

	/* locationuri と absoluteuri はモバイルには付けない(使わないし有害である) */
	if (infotype != DIVY_INFOTYPE_mobile) {

		/* Location URI */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":locationuri>");
		divy_sbuf_append(sbuf, dav_divy_escape_uri(p, dav_divy_get_root_uri(r)));
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":locationuri>"CRLF);

		/* Location URI(JavaScript Escaped) */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":locationuriJS>");
		e_uri = dav_divy_escape_uri(p, dav_divy_get_root_uri(r));
		e_uri = dav_divy_replace_str(p, e_uri, "/", "\\/", NULL);
		divy_sbuf_append(sbuf, e_uri);
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":locationuriJS>"CRLF);

		/* 絶対URL */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":absoluteuri>");
		divy_sbuf_append(sbuf, dav_divy_escape_uri(p,
					divy_construct_url2(p, DIVY_URLTYPE_PUBLIC,  c_rdb_r->uri, r, NULL)));
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":absoluteuri>"CRLF);
	}

	/* グループ制約による書き込み制約の影響を受けるかどうか. */
	is_write_constraints = divy_rdbo_has_write_constraints_on_uri(r, c_rdb_r->uri, infotype);

	if (divy_support_extenduserstatus(r)) {
		int is_trusty = divy_rdbo_is_trusty_user(extstatus);

		/* ユーザの権限 */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":user-privilege>");
		if (divy_rdbo_has_readwrite_privilege(extstatus)) {
			if (is_write_constraints) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":read/>");
			}
			else {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":readwrite/>");
			}
			
		}
		else if (divy_rdbo_has_upload_privilege(extstatus)) {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":upload/>");
			if (is_write_constraints) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":read/>");
			}
		}
		else if (divy_rdbo_has_read_privilege(extstatus)) {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":read/>");
		}
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":user-privilege>");

		/* ユーザの種類 */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":usertype>");
		if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN) {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":admin/>");
		}
		else {
			if (is_trusty) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":normal/>");
			}
			else {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":limited/>");
			}
		}
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":usertype>");
	}
	/* チケットURLの生成.
	 * (特殊エンコードルール) 
	 * 	% -> %25, & -> %26, ? -> %3f */
	ticketurl = divy_construct_oneclick_url(r, p, c_rdb_r);
	ticketurl = (char *) dav_divy_replace_str(p, ticketurl, "%", "%25", NULL);
	ticketurl = (char *) dav_divy_replace_str(p, ticketurl, "&", "%26", NULL);
	ticketurl = (char *) dav_divy_replace_str(p, ticketurl, "?", "%3f", NULL);

	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":ticketurl>");
	divy_sbuf_append(sbuf, ticketurl);
	divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":ticketurl>");

	/* グループ管理者機能のサポート状況 */
	if (divy_support_groupleader(r)) {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":supported-groupleader/>");
	}

	/* 短縮URLのサポート状況 */
	if (divy_support_shorten(r)) {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":supported-shortenurl/>");
	}

	/* ユーザのクォータ */
	/* 使用容量 */
	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":usedbytes>");
	divy_sbuf_append(sbuf, divy_byte2displaysize(p, divy_get_userusquota(r), 1));
	divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":usedbytes>");

	/* 最大利用容量 */
	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":availablebytes>");
	divy_sbuf_append(sbuf, divy_byte2displaysize(p, divy_get_usermsquota(r), 1));
	divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":availablebytes>");

#ifdef DIVY_SUPPORT_EXTENDQUOTA
	/* システムクォータ */
	squota.uri = apr_pstrdup(p, dav_divy_get_root_uri(r));
	squota.type = DIVY_QTTYPE_SYSTEM;
	if (divy_rdbo_get_systemquota(r, &squota, 0, NULL) == 0) {
		/* 使用容量 */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":systemusedbytes>");
		divy_sbuf_append(sbuf, divy_byte2displaysize(p, squota.usedst, 1));
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":systemusedbytes>");

		/* 最大利用容量 */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":systemavailablebytes>");
		divy_sbuf_append(sbuf, divy_byte2displaysize(p, squota.maxst, 1));
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":systemavailablebytes>");
	}
#endif
	/* グループの直下のフォルダの場合、カレントのフォルダがBOX機能があるかを
	 * 調べます */
	if (c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e) {
		if (divy_support_tfbox(r) && c_rdb_r->grp_pr != NULL
				&& divy_rdbo_is_box_group(c_rdb_r->grp_pr->grp_extstatus)) {
					divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":boxinfo/>");
		}
	}
	
	if (divy_support_tfbox(r) && c_rdb_r->rstate_pr != NULL
					&& c_rdb_r->rstate_pr->type == DIVY_RSTATE_TYPE_BOX) {

		box_pr = c_rdb_r->rstate_pr->optional.box_pr;

		if (box_pr != NULL) {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":boxinfo>");
			if (IS_FILLED(box_pr->allowed_origin)) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":allowedorigin>");
				divy_sbuf_append(sbuf, box_pr->allowed_origin);
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":allowedorigin>");
			}
			if (IS_FILLED(box_pr->shorten)) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":shorten>");
				divy_sbuf_append(sbuf, box_pr->shorten);
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":shorten>");
			}
			if (box_pr->flag & BOX_FLAG_OPEN) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":sealingwax/>");
			}
			if (box_pr->flag & BOX_FLAG_PRIV) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":privacy/>");
			}
			if (box_pr->flag & BOX_FLAG_LOCK) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":lockurl/>");
			}
			if (box_pr->flag & BOX_FLAG_DELE) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":autodelete/>");
			}
			if (IS_FILLED(box_pr->password)) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":password>");
				//divy_sbuf_append(sbuf, box_pr->password);
				divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, box_pr->password, DIVY_XML_T2T_CDATA));
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":password>");
			}
			if (box_pr->creationdate > 0) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":creationdate>");
				divy_format_time_t(p, box_pr->creationdate,
						DIVY_TIME_STYLE_ISO8601, &datebuf);

				divy_sbuf_append(sbuf, datebuf);
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":creationdate>");
			}
			if (box_pr->expirationdate > 0) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":expirationdate>");
				divy_format_time_t(p, box_pr->expirationdate,
						DIVY_TIME_STYLE_ISO8601, &datebuf);

				divy_sbuf_append(sbuf, datebuf);
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":expirationdate>");
			}

			/* プライバシーモードの場合は名前は表示させなくなる */
			if (box_pr->flag & BOX_FLAG_PRIV || IS_EMPTY(box_pr->creator_usr_id)) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":creator_user/>");
			}
			else {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":creator_user>");
				divy_sbuf_append(sbuf, box_pr->creator);
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":creator_user>");
			}

			if (IS_FILLED(box_pr->greeting)) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":greeting>");
				divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, box_pr->greeting, DIVY_XML_T2T_CDATA));
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":greeting>");
			}
			if (IS_FILLED(box_pr->message)) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":message>");
				boxmessage = box_pr->message;
				divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, boxmessage, DIVY_XML_T2T_QUOTE));
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":message>");
			}
			/* 2FA */
			if (IS_FILLED(box_pr->tomailaddr)) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":tomailaddr>");
				divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, box_pr->tomailaddr, DIVY_XML_T2T_QUOTE));
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":tomailaddr>");
			}

			divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":boxinfo>");
		}
		else {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":boxinfo/>");
		}

	}

	/* メール監視 */
	/* WEBDAV_METHOD=の比較はPUTとDELETEはURLパラメータの始め(?の次）
	 * からWEBDAV_METHOD=で始まる為です。
	 */
	if (dconf->mlserversend == DIVY_MLSERVERSEND_MIDDLE && c_rdb_r->mwatch_pr) {

		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":mailwatch>");

		/* パラメータのチェックは実際に処理を行ったらチェックをする */
		if (IS_FILLED(r->args)
				&& (divy_strstr(r->args, "WEBDAV_METHOD=") == r->args) ) {
			/* URI パラメータのパース */
			if (dav_divy_parse_urlparam(p, r->args, &parsed_h)) {
				ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to parse URL params.");
				return NULL;
			}
			if (parsed_h != NULL) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":parammethod>");
				ll = apr_hash_get(parsed_h, "WEBDAV_METHOD", APR_HASH_KEY_STRING);
				if (ll != NULL) divy_sbuf_append(sbuf, ll->val);
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":parammethod>");

				/* triggerとなったファイル
				 * PUTはURLパラメータにname=????という形で戻されるが
				 * DELETEはそのままURIだけの為、最後のFinalPathセグメント
				 * を利用するようにしています。
				 */
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":triggerfile>");
				if (ll != NULL && strncmp(ll->val, "DELETE", 6) == 0)  {
					divy_sbuf_append(sbuf,
							dav_divy_extract_finalpath_segment(p, r->uri));
				} else {
					/* DELETE メソッド以外 */
					ll = apr_hash_get(parsed_h, "name", APR_HASH_KEY_STRING);
					if (ll != NULL) divy_sbuf_append(sbuf, ll->val);
				}
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":triggerfile>");
			}
		}

		/* トリガーとして設定されているメソッドを全て列挙する */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":triggermethod>");
		first = 1;
		for (i = 0; i < METHODS; i++) {
			if (c_rdb_r->mwatch_pr->trigger_methods[i]) {
				if (!first) {
					divy_sbuf_append(sbuf, " ");
				}
				first = 0;
				divy_sbuf_append(sbuf, ap_method_name_of(p, i));
			}
		}

		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":triggermethod>");
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":mailwatch>");
	}
	else {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":mailwatch/>");
	}

	divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":currentinfo>" CRLF);

	/*
	 * 親フォルダ用XMLの生成
	 */
	p_rdb_r = screen->p_rdb_r;

	if (p_rdb_r != NULL) {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":parentinfo>" CRLF);

		if (p_rdb_r->u_spec == NULL) {
			divy_parse_uri(p, dav_divy_get_root_uri(r), p_rdb_r->uri, &p_rdb_r->u_spec);
		}

		if (IS_FILLED(screen->encoding)) {
				/* URIはモバイルでない時だけエンコードする */
			if (infotype != DIVY_INFOTYPE_mobile) {
				p_rdb_r->uri = _encode_property(p, screen->encoding,
								p_rdb_r->uri,
								DIVY_AUTOINDEX_FAILEDCONV_URI);
			}
		}

		/* プライベートフォルダ、グループフォルダの場合には、
		 * Content-Typeマップから手に入れた種類名を名前として利用する */
		if (p_rdb_r->u_spec->infotype == DIVY_INFOTYPE_user_e ||
			p_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group) {

			/* contenttype データを取得 */
			map_e = divy_get_extmapinfo(p, mime_map, p_rdb_r->getcontenttype, NULL);
			if (map_e != NULL && IS_FILLED(map_e->ctype_name)) {
				if (IS_FILLED(screen->encoding)) {
					p_rdb_r->getcontenttype = _encode_property(p, screen->encoding,
									map_e->ctype_name,
									DIVY_AUTOINDEX_FAILEDCONV_URI);
				}
				else {
					p_rdb_r->getcontenttype = map_e->ctype_name;
				}
				p_rdb_r->displayname = p_rdb_r->getcontenttype;	/* 無理矢理おきかえ */
			}
		}
		else {
			if (IS_FILLED(screen->encoding)) {
				p_rdb_r->displayname = _encode_property(p, screen->encoding,
									p_rdb_r->displayname,
									DIVY_AUTOINDEX_FAILEDCONV_STR);
			}
		}
		/* URL */
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":href>");
		_build_response_url(p, sbuf, infotype, p_rdb_r, screen->seq, hide_uri, 0);
		divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":href>"CRLF);

		/* 表示名 */
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":displayname>" CRLF);
		divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, p_rdb_r->displayname, DIVY_XML_T2T_QUOTE|DIVY_XML_T2T_CDATA));
		divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":displayname>" CRLF);

		/* resourceタイプ */
		if (p_rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
			divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype>"
						"<"DAV_NS_PREFIX":collection/>"
						"</"DAV_NS_PREFIX":resourcetype>"CRLF);
		}
		else {
			divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype/>"CRLF);
		}
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":parentinfo>" CRLF);
	}


	/*
	 * メンバのXML生成
	 */
	for (n_rdb_r = screen->m_rdb_r; n_rdb_r; n_rdb_r = n_rdb_r->next) {
		int change_displayname = 0;

		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":propinfo>" CRLF);

		/* Autoindex スタートアップページ、ユーティリティフォルダ、
		 * モバイル(トップ)であれば、タイプに応じて変更する */
		if (infotype == DIVY_INFOTYPE_root || infotype == DIVY_INFOTYPE_fileviewer ||
			infotype == DIVY_INFOTYPE_utility ||
			(infotype == DIVY_INFOTYPE_mobile && IS_EMPTY(screen->cmd))) {
			/* プライベートコレクション、グループコレクションの親、
			 * クライアント更新、システムメッセージは置き換え */
			if (strcmp(n_rdb_r->getcontenttype, DIVY_CONTTYPE_PRIVCOL) == 0 ||
				strcmp(n_rdb_r->getcontenttype, DIVY_CONTTYPE_GRPCOLFOLDER) == 0 ||
				strcmp(n_rdb_r->getcontenttype, DIVY_CONTTYPE_UPDATEDCL) == 0 ||
				strcmp(n_rdb_r->getcontenttype, DIVY_CONTTYPE_SYSMSGFOLDER) == 0) {
				change_displayname = 1;
				n_rdb_r->displayname = n_rdb_r->getcontenttype;
			}
		}

		/* グループ直下のフォルダの場合のみコレクションなら
		 * BOXになっているかを調べます。
		 * BOXの場合は画像を変えるためにContent-Typeを変更する */
		if (infotype == DIVY_INFOTYPE_group_e &&
				n_rdb_r->resourcetype == DIVY_TYPE_COLLECTION && 
				divy_support_tfbox(r) && n_rdb_r->rstate_pr != NULL 
					&& n_rdb_r->rstate_pr->type == DIVY_RSTATE_TYPE_BOX) {

			/* BOXのフォルダになる条件
			 * 1. 公開されている
			 * 2. BOXの期限が無期限若しくは切れていない
			 */
			if (n_rdb_r->rstate_pr->optional.box_pr->flag & BOX_FLAG_OPEN
					&& (n_rdb_r->rstate_pr->optional.box_pr->expirationdate == 0
					|| (n_rdb_r->rstate_pr->optional.box_pr->expirationdate != 0
					&& n_rdb_r->rstate_pr->optional.box_pr->expirationdate > dav_divy_get_now_epoch()))
					) {
				n_rdb_r->getcontenttype = DIVY_CONTTYPE_BOXFOLDER;
			}
		}

		/* contenttype, summaryicon データを取得 */
		last = divy_strrchr_c(n_rdb_r->displayname, '.');
		ext = (last != NULL) ? ++last : NULL;

		map_e = divy_get_extmapinfo(p, mime_map, n_rdb_r->getcontenttype, ext);

		if (map_e != NULL && IS_FILLED(map_e->ctype_name)) {
			n_rdb_r->getcontenttype = map_e->ctype_name;
			if (change_displayname) {
				n_rdb_r->displayname = n_rdb_r->getcontenttype;
			}
		}

		/* エンコードが必要ならURI、表示名、更新者、Content-Typeをエンコードする */
		if (IS_FILLED(screen->encoding)) {

			/* URIはモバイルでない時だけエンコードする */
			if (infotype != DIVY_INFOTYPE_mobile) {
				n_rdb_r->uri = _encode_property(p, screen->encoding,
								n_rdb_r->uri,
								DIVY_AUTOINDEX_FAILEDCONV_URI);
			}

			n_rdb_r->displayname = _encode_property(p, screen->encoding,
								n_rdb_r->displayname,
								DIVY_AUTOINDEX_FAILEDCONV_STR);

			n_rdb_r->lastmodifier = _encode_property(p, screen->encoding,
								n_rdb_r->lastmodifier,
								DIVY_AUTOINDEX_FAILEDCONV_STR);

			if (IS_FILLED(n_rdb_r->getcontenttype)) {
				n_rdb_r->getcontenttype = _encode_property(p, screen->encoding,
									n_rdb_r->getcontenttype,
									DIVY_AUTOINDEX_FAILEDCONV_URI);
			}
		}

		/* URL */
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":href>");
		_build_response_url(p, sbuf, infotype, n_rdb_r, screen->seq, hide_uri, 0);
		divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":href>"CRLF);

		/* ダウンロード専用リンク */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":dwldhref>");
		_build_response_url(p, sbuf, infotype, n_rdb_r, screen->seq, hide_uri, 1);
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":dwldhref>"CRLF);

		/* 表示名 */
		/*
		 * libxsltを利用するが故のcross site scripting問題
		 * TeamFileではXMLをサーバサイドでstylesheetのフォーマットを利用して
		 * HTMLを出力する仕組みを採用しています。
		 * 表示名はそのままブラウザで見える必要がある為、HTML特殊文字は
		 * エスケープします。
		 * しかし、'(シングルクォート）だけは最終的にHTML化されるとき
		 * 展開されてしまい、'が文字とされてしまいました。
		 * &などは実体参照のままとなりうまくいく。
		 *
		 * その為にHTML化されても'は&#39;にしておくために次のコードとしました。
		 * (エスケープ(DIVY_XML_T2T_QUOTE)してCDATAで囲む(DIVY_XML_T2T_CDATA))
		 * 更にスタイルシート側でも
		 * <xsl:value-of disable-output-escaping="yes" select="D:displayname" />
		 * とすることにより実現しています。
		 * 親フォルダも同じように実装を変更しています
		 */
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":displayname>");
		divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p,
				n_rdb_r->displayname, DIVY_XML_T2T_QUOTE|DIVY_XML_T2T_CDATA));
		divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":displayname>"CRLF);

		/* サイズ */
		divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":getcontentlength>");
		/* APIリクエストの場合はサイズ表示を加工しない */
		if (screen->isRow) {
			divy_sbuf_append(sbuf, apr_psprintf(p, "%"APR_INT64_T_FMT, n_rdb_r->getcontentlength));
		}
		else {
			divy_sbuf_append(sbuf, divy_byte2displaysize(p, n_rdb_r->getcontentlength, 1));
		}
		divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":getcontentlength>"CRLF);

		/* 更新日付 */
		if (n_rdb_r->getlastmodified > 0) {
			/* APIリクエストの場合はフォーマットがISO8601へなります */
			divy_format_time_t(p, n_rdb_r->getlastmodified, (screen->isRow)?DIVY_TIME_STYLE_ISO8601:DIVY_TIME_STYLE_JAPANESE, &datebuf);
			divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":getlastmodified>");
			divy_sbuf_append(sbuf, datebuf);
			divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":getlastmodified>"CRLF);
		}

		/* contenttype */
		if (IS_FILLED(n_rdb_r->getcontenttype)) {
			divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":getcontenttype>");
			divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p,
						n_rdb_r->getcontenttype, DIVY_XML_T2T_QUOTE));
			divy_sbuf_append(sbuf, "</"DAV_NS_PREFIX":getcontenttype>"CRLF);
		}

		/* summaryicon */
		if (map_e != NULL && IS_FILLED(map_e->img_path)) {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":summaryicon>");
				divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p,
					map_e->img_path, DIVY_XML_T2T_QUOTE));
			divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":summaryicon>");
		}

		/* resourceタイプ */
		if (c_rdb_r->u_spec->infotype == DIVY_INFOTYPE_u_msg) {
			/* カレントがDIVY_INFOTYPE_u_msg ならばメンバも同じタイプ */
			divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype>"
						"<"DIVY_NS_PREFIX":sysmsgcollection/>"
						"</"DAV_NS_PREFIX":resourcetype>"CRLF);
		}
		else if (n_rdb_r->resourcetype == DIVY_TYPE_COLLECTION) {
			divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype>"
						"<"DAV_NS_PREFIX":collection/>"
						"</"DAV_NS_PREFIX":resourcetype>"CRLF);
		}
		else {
			divy_sbuf_append(sbuf, "<"DAV_NS_PREFIX":resourcetype/>"CRLF);
		}

		/* 更新者(mobile では要らない) */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":lastmodifier>");
		divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p,
				n_rdb_r->lastmodifier, DIVY_XML_T2T_QUOTE));
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":lastmodifier>"CRLF);

		/* 符号化されたグループID (存在する場合のみ入れる) */
		if (n_rdb_r->grp_pr != NULL && IS_FILLED(n_rdb_r->grp_pr->grpid)) {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":groupid>");
			divy_sbuf_append(sbuf, n_rdb_r->grp_pr->grpid);
			divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":groupid>");

			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":encgroupid>");
			divy_sbuf_append(sbuf, divy_autodel_encipher_groupid(p, n_rdb_r->grp_pr->grpid));
			divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":encgroupid>");

			/* コメントの先頭文字で表示を制御します */
			if (IS_FILLED(n_rdb_r->grp_pr->comment)) {

				/*
				 * 先頭が -（ハイフン）の場合は表示させない（仕様）
				 */
				if (strncmp(n_rdb_r->grp_pr->comment, "-", 1) == 0) {
					n_rdb_r->grp_pr->comment = NULL;
				}
				/*
				 * 先頭が +（プラス）の場合は先頭の+を除いて表示する（仕様）
				 */
				else if (strncmp(n_rdb_r->grp_pr->comment, "+", 1) == 0) {
					n_rdb_r->grp_pr->comment++;
				}

				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":groupcomment>");
				divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, n_rdb_r->grp_pr->comment, DIVY_XML_T2T_QUOTE));
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":groupcomment>");
			}
		}

		/* 短縮URL */
		if (divy_support_shorten(r)) {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":shortenuri>");
			divy_sbuf_append(sbuf, apr_psprintf(p, "%s/%s", divy_build_shorten_uri(p, dav_divy_get_root_uri(r)), divy_get_rid2shorten(p, n_rdb_r->rsid, NULL)));
			divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":shortenuri>");
		}

		/* BOXフォルダ */
		if (divy_support_tfbox(r) && n_rdb_r->rstate_pr != NULL 
					&& n_rdb_r->rstate_pr->type == DIVY_RSTATE_TYPE_BOX) {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":box/>");
		}

    	/* グループフォルダの場合、そのグループにアップロードポリシーが
		 * 設定されているかを調べます */
       	if (divy_support_upload_policy(r) && n_rdb_r->grp_pr != NULL
			&& divy_rdbo_is_uploadpolicy_group(n_rdb_r->grp_pr->grp_extstatus)){
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":uploadpolicyrule/>");
    	}
    
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":propinfo>" CRLF);

		/* 閾値(XMLBUF_BOUNDARY_SIZE)を超えていたら書き出す */
		sbuf_len = divy_sbuf_getlength(sbuf);

		if (sbuf_len >= XMLBUF_BOUNDARY_SIZE) {

			/* 一時ファイルを開く */
			if (fd == NULL) {
				rv = apr_file_mktemp(&fd, filepath,
						APR_WRITE | APR_CREATE | APR_TRUNCATE | APR_BINARY, p);
				if (rv != APR_SUCCESS || fd == NULL) {
					ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
						"Failed to open temp file.(code = %d)", rv);
					err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
					goto cleanup_screen2xml_resource;
				}
				/* 一時ファイルの作成に成功したらタイプを変える */
				outputxml->type = DIVY_XMLOUTPUT_TYPE_FILE; /* ファイル形式である */
				outputxml->d.filepath = filepath;
			}

			/* ファイルに書き出す */
			rv = apr_file_write_full(fd, divy_sbuf_tostring(sbuf), sbuf_len, NULL);
			if (APR_STATUS_IS_ENOSPC(rv)) {
				ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"There is not enough storage to write to this fille."
					"(bufsize = %"APR_SIZE_T_FMT ")", sbuf_len);
				/* 書き込みスペースが足りなかった */
				err = dav_new_error(log_p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
				goto cleanup_screen2xml_resource;
			}
			else if (rv != APR_SUCCESS) {
				ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"An error occurred while writing to a resource."
					"(code = %d)", rv);
				err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				goto cleanup_screen2xml_resource;
			}

			/* バッファをクリアする */
			divy_sbuf_clear(sbuf);
		}
	}

	/*
	 * ページ情報XMLの生成
	 */
	divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":pageinfo>");

	/* root URI */
	divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":rooturi>");
	divy_sbuf_append(sbuf, divy_construct_url2(p, DIVY_URLTYPE_PUBLIC,
							"", r, NULL));
	divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":rooturi>");

	/* mobile & 続きがある & 概要表示であれば次ページ遷移情報を付加 */
	if (infotype == DIVY_INFOTYPE_mobile && screen->found_extra_data && !is_detail) {

		char *nexthref;

		if (IS_FILLED(screen->cmd) && strcmp(screen->cmd, DIVY_CMD_SHOW_GLIST) == 0) {
			/* グループフォルダ以下の表示が次ページ */
			nexthref = divy_build_mobile_uri(p, dav_divy_get_root_uri(r),
					DIVY_CMD_SHOW_GLIST, "0", (screen->ipageno+1));
		}
		else {
			/* 概要表示が次ページ */
			nexthref = divy_build_mobile_uri(p, dav_divy_get_root_uri(r),
					DIVY_CMD_SHOW_SUMMARY, screen->crsid, (screen->ipageno+1));
		}
		divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":nexthref>");
		/* モバイルのURIには'&'があるのでXSLパーサがエラーを出す
		 * これを避ける目的でXMLエスケープだけを実施する */
		divy_sbuf_append(sbuf, apr_xml_quote_string(p,
					apr_psprintf(p, "%s&%s", nexthref, screen->seq), 0));
		divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":nexthref>");
	}

	/* brandname を付加 */
	if (IS_EMPTY(sconf->brandname)) {
		/* デフォルトのプロダクト名を入れる */
		brandname = apr_pstrdup(p, DIVY_PRODUCT_NAME);
	}
	else {
		brandname = apr_pstrdup(p, sconf->brandname);
	}

	divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":brandname>");
	divy_sbuf_append(sbuf,	dav_divy_escape_xmlstr(p, brandname, DIVY_XML_T2T_QUOTE));
	divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":brandname>");

	/* ブランド画像 */
	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":brandimg>");
	map_e = divy_get_extmapinfo(p, mime_map, "application/x-tf-toppage", NULL);
	if (map_e != NULL && IS_FILLED(map_e->img_path)) {
		divy_sbuf_append(sbuf,
			dav_divy_escape_xmlstr(p, map_e->img_path, DIVY_XML_T2T_QUOTE));
	}
	divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":brandimg>");

	/* TeamFileバージョン 2.4.0-7 / これは必ず存在する */
	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":tfversion>");
	divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(p, (char*)sconf->tfversion, DIVY_XML_T2T_QUOTE));
	divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":tfversion>");

	/* ユーザID */
	if (divy_get_userid(r) != NULL) {
		divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":userid>");
		divy_sbuf_append(sbuf,	dav_divy_escape_xmlstr(p, (char*)divy_get_userid(r), DIVY_XML_T2T_QUOTE));
		divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":userid>");
		/* ユーザ名 */
		divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":username>");
		divy_sbuf_append(sbuf,	dav_divy_escape_xmlstr(p, (char*)divy_get_fullname(r), DIVY_XML_T2T_QUOTE));
		divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":username>");
	}

	/* 機能オプションの有効/無効化状態を付加する */
	divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":availablefunction>");
	if (divy_support_autodelete(r)) {
		/* 自動削除機能の有効/無効状態を付加 */
		divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":autodelete/>");
	}

	if (divy_disable_webfolerlink(r) == 0) {
		/* Webフォルダリンクが有効なら追加（デフォルト）*/
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":webfolderlink/>");
	}

	if (divy_support_session(r) == 1) {
		/* ブラウザセッションが有効なら追加 */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":usesession/>");
	}

	if (divy_support_upload_policy(r) == 1) {
		/* アップロードポリシーが有効なら追加 */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":uploadpolicy/>");
	}

	if (divy_support_2FA(r) == 1) {
		/* 二段階認証が有効なら追加 */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":twofactorauthentiaction/>");
	}

	divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":availablefunction>");

	/* 拡張したいCSSがあれば追加 */
	if (screen->extend_css != NULL) {
		divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":csshref>");
		divy_sbuf_append(sbuf, apr_xml_quote_string(p,
					dav_divy_escape_uri(p, screen->extend_css), 0));
		divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":csshref>");
	}

	/* RSS表示をするページならタグ追加 */
	if (screen->with_rss) {
		divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":withrss/>");
	}

	if (screen->r->args != NULL) {
		divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":args>");
		divy_sbuf_append(sbuf, apr_xml_quote_string(p, screen->r->args, 0));
		divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":args>");
	}

	/* 二段階認証の要求 */
	if (screen->require2fa) {
		divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":require2fa/>");
	}

	/* ResultStateの取得 */
	if (apr_table_get(r->subprocess_env, DIVY_CGIENV_RESULTSTATUS) != NULL) {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":resultstate>");
		divy_sbuf_append(sbuf, apr_table_get(r->subprocess_env,
											DIVY_CGIENV_RESULTSTATUS));
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":resultstate>");
	}
	else {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":resultstate/>");
	}

	/* UNIQUEID */
	if (divy_support_session(r) == 1) {
		if (divy_util_auth_parse_cookie(r, &sesspw, &sessid) == 0) {
			session.sid = sessid;
			(void)(divy_util_auth_get_memcache_userinfo(r, &session));
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":token>");
			divy_sbuf_append(sbuf, session.uniqueid);
			divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":token>");
		}
		else {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":token/>");
		}
	}
	else {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":token/>");
	}

	divy_sbuf_append(sbuf,
				"</"DIVY_NS_PREFIX":pageinfo>"
				"</"DIVY_NS_PREFIX":propdiscovery>");

	/*
	 * ファイルに書き出す
	 * [ 書き出し条件 ]
	 *   * 既にファイルに出力済みである or
	 *   * ファイルには出力していないが閾値を超えている
	 */
	sbuf_len = divy_sbuf_getlength(sbuf);
	if (outputxml->type == DIVY_XMLOUTPUT_TYPE_FILE || sbuf_len >= XMLBUF_BOUNDARY_SIZE) {
		/* まだ開いていなければ一時ファイルを開く */
		if (fd == NULL) {
			rv = apr_file_mktemp(&fd, filepath,
					APR_WRITE | APR_CREATE | APR_TRUNCATE | APR_BINARY, p);
			if (rv != APR_SUCCESS || fd == NULL) {
				ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to open temp file.(code = %d)", rv);
				err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				goto cleanup_screen2xml_resource;
			}
			/* 一時ファイルの作成に成功したらタイプを変える */
			outputxml->type = DIVY_XMLOUTPUT_TYPE_FILE; /* ファイル形式である */
			outputxml->d.filepath = filepath;
		}

		rv = apr_file_write_full(fd, divy_sbuf_tostring(sbuf), sbuf_len, NULL);
		if (APR_STATUS_IS_ENOSPC(rv)) {
			ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"There is not enough storage to write to this fille."
				"(bufsize = %"APR_SIZE_T_FMT ")", sbuf_len);
			/* 書き込みスペースが足りなかった */
			err = dav_new_error(log_p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
			goto cleanup_screen2xml_resource;
		}
		else if (rv != APR_SUCCESS) {
			ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"An error occurred while writing to a resource.(code = %d)", rv);
			err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			goto cleanup_screen2xml_resource;
		}
	}
	else {
		outputxml->type   = DIVY_XMLOUTPUT_TYPE_STRING;	/* 文字列形式 */
		outputxml->d.sbuf = sbuf;
	}

cleanup_screen2xml_resource:

	/* XML出力用ファイルを閉じる */
	if (fd != NULL && (rv = apr_file_close(fd)) != APR_SUCCESS) {
		ERRLOG1(log_p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to close tempory xml file for autoindex."
			"(code = %d)", rv);
		/* 続ける */
	}

	/* エラー時にはXML出力用ファイルを削除する */
	if (err && outputxml->type == DIVY_XMLOUTPUT_TYPE_FILE && IS_FILLED(outputxml->d.filepath)) {
		rv = apr_file_remove(outputxml->d.filepath, p);
		if (rv != APR_SUCCESS) {
			ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to remove temporary file for autoindex."
				"(filepath = %s, code = %d)", outputxml->d.filepath, rv);
		}
		outputxml->type = DIVY_XMLOUTPUT_TYPE_UNKNOWN;	/* 未定義に戻す */
		outputxml->d.filepath = NULL;	/* 無かったことにする */
	}

	return err;
}

/**
 * XMLファイルxml_filepath をstylesheet が示すスタイルシートファイルを使って
 * HTMLに変換する。変換されたHTMLはファイル出力される。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param fstorage divy_fstorage_t * 論理ストレージ
 * @param screen divy_autoindex_screen * 画面入力情報を持つ構造体
 * @param stylesheet const char * スタイルシートへのフルパス名
 * @param outputxml divy_outputxml * XML情報
 * @param filepath const char ** HTMLファイルのパス文字列へのポインタ
 * @return DAVエラー
 */
static dav_error * _xml2html(apr_pool_t *p,
					divy_fstorage_t *fstorage,
					divy_autoindex_screen *screen,
					const char *stylesheet,
					divy_outputxml *outputxml,
					const char **filepath)
{
	apr_status_t rv;
	dav_error *err            = NULL;
	apr_pool_t *log_p         = NULL;
	request_rec *r            = NULL;
	xmlParserCtxtPtr ctxt_p   = NULL;
	xmlDocPtr doc_p           = NULL;
	xmlDocPtr html_p          = NULL;
	xmlCharEncoding enc;
	xsltStylesheetPtr cur     = NULL;
	apr_os_file_t osfd;
	const char *params[]      = { NULL };
	divy_pfile_t *pfile       = NULL;
	apr_file_t *fd            = NULL;
	int need_unlock           = 0, length, ret;
	xmlOutputBufferPtr obuf   = NULL;
	xmlCharEncodingHandlerPtr enc_p = NULL;

	*filepath = NULL;	/* 初期化 */

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	r     = screen->r;
	log_p = r->pool;

	/* このタイプのXMLは処理できません(バグ以外ありえないけど) */
	if (outputxml->type == DIVY_XMLOUTPUT_TYPE_UNKNOWN) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The type of xml is unknown.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * HTML出力用一時ファイルの作成
	 */
	ret = divy_pfile_mkfixedtemp(fstorage, p, DIVY_TMP_FILE_TEMPLATE_NAME, &pfile);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to make a temporary file.(uri = %s, "
			"userid = %s)", r->uri, divy_get_userid(r));
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_xml2html_resource;
	}

	/* 一時ファイルを開く
	 * (note) apr_file_mktemp の第2引数は書き換えられるのでコピーする */
	*filepath = apr_pstrdup(p, divy_pfile_get_fullpath(pfile));

	rv = apr_file_mktemp(&fd, (char *) *filepath,
			APR_WRITE | APR_CREATE | APR_TRUNCATE | APR_BINARY, p);
	if (rv != APR_SUCCESS || fd == NULL) {
		ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to open temp file.(code = %d)", rv);
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_xml2html_resource;
	}

	/* OSのファイルディスクリプタを取得する */
	rv = apr_os_file_get(&osfd, fd);
	if (rv != APR_SUCCESS) {
		ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to get OS fd. (code = %d)", rv);
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_xml2html_resource;
	}

	/* ロックを掛ける */
	rv = apr_thread_mutex_lock(autoindex_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to lock for open file.(code = %d)", rv);
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_xml2html_resource;
	}
	need_unlock = 1;	/* 兎に角最後にはunlockが必要であることを明示 */

	/*
	 * DOM ツリーに変換
	 */
	xmlSubstituteEntitiesDefault(1);
	xmlLoadExtDtdDefaultValue = 1;
	if (IS_FILLED(screen->encoding)) {
		/* エンコードする必要があるのなら、パーサコンテキストを生成して
		 * エンコード変換が必要 */
		if (outputxml->type == DIVY_XMLOUTPUT_TYPE_FILE) {
			/* XMLファイルを使ってctxt_p を生成 */
			ctxt_p = xmlCreateFileParserCtxt(outputxml->d.filepath);
		}
		else {
			/* XML文字列を使ってctxt_p を生成 */
			ctxt_p = xmlCreateMemoryParserCtxt(divy_sbuf_tostring(outputxml->d.sbuf),
							divy_sbuf_getlength(outputxml->d.sbuf));
		}

		if (ctxt_p == NULL) {
			ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to create parser context.");
			err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			goto cleanup_xml2html_resource;
		}

		enc = xmlParseCharEncoding(screen->encoding);
		if (enc != XML_CHAR_ENCODING_ERROR) {
			/* コンテキストのエンコーディングをスイッチ */
			xmlSwitchEncoding(ctxt_p, enc);

			/* コンテキストをパース */
			ret = xmlParseDocument(ctxt_p);
			if (ret < 0) {
				ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to parse document(code = %d)", ret);
				err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				goto cleanup_xml2html_resource;
			}
			doc_p = ctxt_p->myDoc;		/* doc_p を取得   */
		}
		else {
			/* screen->encoding が利用出来なかった場合 */
			ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Could not understand encoding name.(enc = %s)",
				screen->encoding);
			err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			goto cleanup_xml2html_resource;
		}
	}
	else {
		/* doc_p を生成する */
		if (outputxml->type == DIVY_XMLOUTPUT_TYPE_FILE) {
			/* XMLファイルを使ってdoc_p を生成 */
			doc_p = xmlParseFile(outputxml->d.filepath);
		}
		else {
			/* XML文字列を使ってdoc_p を生成 */
			doc_p = xmlParseMemory(divy_sbuf_tostring(outputxml->d.sbuf),
						divy_sbuf_getlength(outputxml->d.sbuf));
		}

		if (doc_p == NULL) {
			ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to parse xml.");
			err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			goto cleanup_xml2html_resource;
		}
	}

	/*
	 * XSLT プロセッサを使ってXMLをHTMLに変換
	 */
	cur = xsltParseStylesheetFile((const xmlChar*)stylesheet);
	if (cur == NULL) {
		ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to parse stylesheetfile.(stylesheet = %s, "
			"filepath = %s)", stylesheet, *filepath);
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_xml2html_resource;
	}

	html_p = xsltApplyStylesheet(cur, doc_p, params);
	if (html_p == NULL) {
		ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to apply stylesheet.(stylesheet = %s, "
			"filepath = %s)", stylesheet, *filepath);
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_xml2html_resource;
	}

	/* エンコードする必要はあるのか? */
	if (IS_FILLED(screen->encoding)) {
		enc_p = xmlFindCharEncodingHandler(screen->encoding);
	}
	else {
		enc_p = xmlGetCharEncodingHandler(XML_CHAR_ENCODING_UTF8);
	}

	obuf = xmlOutputBufferCreateFd(osfd, enc_p);
	length = xsltSaveResultTo(obuf, html_p, cur);
	if (length == -1) {
		ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to save html to file."
			"(filepath = %s, code = -1)", *filepath);
		err = dav_new_error(log_p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_xml2html_resource;
	}
	xmlOutputBufferFlush(obuf);	/* バッファをフラッシュする */

cleanup_xml2html_resource:

	/* 使用資源を解放 */
	if (obuf != NULL) xmlOutputBufferClose(obuf);
	if (cur != NULL) xsltFreeStylesheet(cur);
	if (html_p != NULL) xmlFreeDoc(html_p);
	if (doc_p != NULL)  xmlFreeDoc(doc_p);
	if (ctxt_p != NULL) xmlFreeParserCtxt(ctxt_p);

	if (need_unlock) {
		/* ロック後に使用されるのでこのタイミングでクリーンアップできる */
		xsltCleanupGlobals();
		xmlCleanupParser();

		/* ロックを解除 */
		rv = apr_thread_mutex_unlock(autoindex_mutex);
		if (rv != APR_SUCCESS) {
			ERRLOG1(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to unlock for open file.(code = %d)", rv);
			/* 続ける */
		}
	}

	/* 一時ファイルを閉じる */
	if (fd != NULL && (rv = apr_file_close(fd)) != APR_SUCCESS) {
		ERRLOG1(log_p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to close tempory file for autoindex."
			"(code = %d)", rv);
		/* 続ける */
	}

	/* 一時ファイルを削除する */
	if (err && *filepath) {
		rv = apr_file_remove(*filepath, p);
		if (rv != APR_SUCCESS) {
			ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to remove temporary file for autoindex."
				"(filepath = %s, code = %d)", *filepath, rv);
		}
		*filepath = NULL;	/* 無かったことにする */
	}

	return err;
}

static dav_error* _samlxml2html(apr_pool_t *p, divy_samlinfo_iscreen *screen,
										const char *stylesheet, char **filepath)
{

	apr_status_t rv;
	dav_error *err            = NULL;
	*filepath = NULL;	/* 初期化 */
	int ret, need_unlock, length;
	request_rec *r;
	char *xml                 = NULL;
	divy_fstorage_t *fstorage = NULL;
	divy_pfile_t *pfile       = NULL;
	apr_file_t *fd            = NULL;
	apr_os_file_t osfd;
	xmlDocPtr doc_p           = NULL;
	xsltStylesheetPtr cur     = NULL;
	xmlDocPtr html_p          = NULL;
	const char *params[]      = { NULL };
	xmlCharEncodingHandlerPtr enc_p = NULL;
	xmlOutputBufferPtr obuf   = NULL;
	
	r = screen->r;

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"SAML screen is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * divy_fstorage_openはr->userがないといけない
	 * まだユーザーが確定していない為r->userにはゲストを使います
	 */
	r->user = DIVY_GUEST_ID;
	(void)divy_rdbo_cache_userinfo(r, 1 /* no cache */);

	/* 論理ストレージのopen */
	ret = divy_fstorage_open(r, p, &fstorage);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to open storage.(uri = %s, userid = %s)",
			r->uri, divy_get_userid(r));
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_samlxml2html;
	}

	/*
	 * HTML出力用一時ファイルの作成
	 */
	ret = divy_pfile_mkfixedtemp(fstorage, p, DIVY_TMP_FILE_TEMPLATE_NAME, &pfile);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to make a temporary file.(uri = %s, "
			"userid = %s)", r->uri, divy_get_userid(r));
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_samlxml2html;
	}

	/* 一時ファイルを開く
	 * (note) apr_file_mktemp の第2引数は書き換えられるのでコピーする */
	*filepath = apr_pstrdup(p, divy_pfile_get_fullpath(pfile));

	rv = apr_file_mktemp(&fd, (char *) *filepath,
			APR_WRITE | APR_CREATE | APR_TRUNCATE | APR_BINARY, p);
	if (rv != APR_SUCCESS || fd == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to open temp file.(code = %d)", rv);
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_samlxml2html;
	}

	/* OSのファイルディスクリプタを取得する */
	rv = apr_os_file_get(&osfd, fd);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to get OS fd. (code = %d)", rv);
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_samlxml2html;
	}

	/* ロックを掛ける */
	rv = apr_thread_mutex_lock(autoindex_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to lock for open file.(code = %d)", rv);
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_samlxml2html;
	}
	need_unlock = 1;	/* 兎に角最後にはunlockが必要であることを明示 */

	/* SAMLRequestのXMLを作成する */
	if (divy_saml_build_autoindex_xml(r, screen, &xml)) {
		/* エラー */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
											"Failed to build SAML XML data.");
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_samlxml2html;
	}

	/*
	 * DOM ツリーに変換
	 */
	xmlSubstituteEntitiesDefault(1);
	xmlLoadExtDtdDefaultValue = 1;

	doc_p = xmlParseMemory(xml, strlen(xml));
	if (doc_p == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
											"Failed to parse xml buffer.");
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_samlxml2html;
	}

	/*
	 * XSLT プロセッサを使ってXMLをHTMLに変換
	 */
	cur = xsltParseStylesheetFile((const xmlChar*)stylesheet);
	if (cur == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to parse stylesheetfile. (stylesheet = %s)", stylesheet);
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_samlxml2html;
	}

	html_p = xsltApplyStylesheet(cur, doc_p, params);
	if (html_p == NULL) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to apply stylesheet.(stylesheet = %s, "
			"filepath = %s)", stylesheet, *filepath);
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_samlxml2html;
	}

	enc_p = xmlGetCharEncodingHandler(XML_CHAR_ENCODING_UTF8);
	obuf = xmlOutputBufferCreateFd(osfd, enc_p);

	length = xsltSaveResultTo(obuf, html_p, cur);
	if (length == -1) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to save html to file."
			"(filepath = %s, code = -1)", *filepath);
		err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		goto cleanup_samlxml2html;
	}
	xmlOutputBufferFlush(obuf);	/* バッファをフラッシュする */
cleanup_samlxml2html:

	if (cur != NULL) xsltFreeStylesheet(cur);
	if (html_p != NULL) xmlFreeDoc(html_p);
	if (doc_p != NULL)  xmlFreeDoc(doc_p);

	/* ストレージを閉じる */
	if (fstorage != NULL) (void) divy_fstorage_close(fstorage);

	if (need_unlock) {
		/* ロック後に使用されるのでこのタイミングでクリーンアップできる */
		xsltCleanupGlobals();
		xmlCleanupParser();

		/* ロックを解除 */
		rv = apr_thread_mutex_unlock(autoindex_mutex);
		if (rv != APR_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to unlock for open file.(code = %d)", rv);
			/* 続ける */
		}
	}

	/* 一時ファイルを閉じる */
	if (fd != NULL && (rv = apr_file_close(fd)) != APR_SUCCESS) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to close tempory file for SAML Request xml."
			"(code = %d)", rv);
		/* 続ける */
	}

	/* 一時ファイルを削除する */
	if (err && filepath) {
		rv = apr_file_remove(*filepath, p);
		if (rv != APR_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to remove temporary file for autoindex."
				"(filepath = %s, code = %d)", *filepath, rv);
		}
		*filepath = NULL;	/* 無かったことにする */
	}

	return err;
}


/*
 * 以下はユーティリティ
 */

/**
 * rsid を暗号化して返却する
 * (note)
 * 	復号化には、_decipher_rsid を使用して下さい。
 *
 * @param p apr_pool_t * 暗号化文字列を確保する文字列
 * @param seed int 0 から 9までの数値
 * @param rsid const char * 暗号化されていないrsid
 * @return const char * 暗号化されたrsid
 */
static const char * _encipher_rsid(apr_pool_t *p, int seed, const char *rsid)
{
	char *ret;
	const char *mtbl;
	int i, e;
	apr_size_t len;

	if (IS_EMPTY(rsid)) return NULL;

	len = strlen(rsid);
	ret = apr_pcalloc(p, sizeof(char) * (len + 2));
	mtbl = _mptbls[seed];	/* テーブルを取得 */

	for (i = 0; i < len; i++) {
		/* シフト数を算出 */
		e = (_sftbl[i] + rsid[i] - '0') % 10;
		ret[i] = mtbl[e];	/* シフトを実施 */
	}
	ret[len] = _mptbl0[seed];	/* seed を暗号化 */
	ret[len + 1] = '\0';

	return ret;
}

/**
 * 暗号化されたrsid を復号して返却する
 * (note)
 * 	暗号化には _encipher_rsid を使用して下さい。
 *
 * @param p apr_pool_t * 複合化された文字列を確保する文字列
 * @param cipherrsid const char * 暗号化されたrsid
 * @return const char * 複合化されたrsid
 */
static const char * _decipher_rsid(apr_pool_t *p, const char *cipherrsid)
{
	char *ret, c;
	const char *mtbl;
	int i, j, e, seed = -1;
	apr_size_t len;

	if (IS_EMPTY(cipherrsid)) return NULL;

	len = strlen(cipherrsid);
	ret = apr_pcalloc(p, sizeof(char) * len);

	for (i = 0; i < 10; i++) {
		c = _mptbl0[i];
		if (c == cipherrsid[len-1]) {
			seed = i;
			break;
		}
	}
	/* 改竄されたid */
	if (seed < 0) return NULL;

	mtbl = _mptbls[seed];	/* 変換テーブルを取得 */

	/* 元に戻す */
	for (i = 0; i < len - 1; i++) {
		for (j = 0; j < 10; j++) {
			if (cipherrsid[i] == mtbl[j]) {
				e = j - _sftbl[i];
				if (e >= 0) {
					ret[i] = _i2a[e];
				}
				else {
					e = 10 + e;
					ret[i] = _i2a[e];
				}
			}
		}
		if (ret[i] == '\0') {
			return NULL;	/* 改竄されていた */
		}
	}
	ret[len - 1] = '\0';

	return ret;
}

/**
 * 正当なUTF-8ではあるのだが、他のエンコードに変換できない特別な文字列を
 * 置換して返却する (autoindex 用)
 * なお、置換すべき文字が1つもなければ、str をそのまま返却します(同じポインタ)
 *
 * @param p apr_pool_t *
 * @param str const char *
 * @return char * 置換後の文字列
 */
static char * _replace_specialstr(apr_pool_t *p, char *str)
{
	const divy_strconvertion *entry;
	int i;
	char *ch = NULL;

	if (IS_EMPTY(str)) return str;	/* 何もしなくていい */

	ch = str;
	for (i = 0, entry = &_special2normal[0];
			entry->from != NULL; i++, entry = &_special2normal[i]) {
		/* entry->from の文字列をentry->to で置き換える */
		ch = (char *) dav_divy_replace_str(p, ch,
						entry->from, entry->to, NULL);
	}

	return ch;
}

/**
 * src が示す文字列を encoding でエンコード変換して返却する。
 * エンコードに失敗したら DIVY_AUTOINDEX_FAILEDCONV_STR で定義された文字列が戻されます。
 * src のエンコードはUTF-8であると仮定しています。(内部専用)
 *
 * @param p apr_pool_t *
 * @param encoding const char * エンコード名
 * @param src char * オリジナル文字列(変換後には内容が変更される可能性あり)
 * @param failedconv_str char * 変換失敗時に利用する文字列
 * @return char * 変換された文字列
 */
static char * _encode_property(apr_pool_t *p, const char *encoding, char *src, char *failedconv_str)
{
	char *dst = NULL, *replace_str;

	if (IS_EMPTY(src)) return "";

	/* エンコード変換すら出来ない文字列を最初に何とかしておく */
	replace_str = _replace_specialstr(p, src);

	/* エンコード変換を実施 */
	if (divy_misc_encode_str(p, "UTF-8", encoding, replace_str, &dst)) {
		/* エンコードに失敗したら置き換える */
		dst = failedconv_str;
	}

	return dst;
}

/**
 * 指定されたuri を組み立ててAutoindex用URLをつくりsbuf に追加する.
 */
static void _build_response_url(apr_pool_t *p, divy_sbuf *sbuf, divy_infotype infotype,
                                divy_rdbo_resource *rdb_r,
                                const char *seq, int is_hideuri, int is_downloadonly)
{

	const char* e_uri = NULL;

	if (sbuf == NULL || rdb_r == NULL || IS_EMPTY(rdb_r->uri) || IS_EMPTY(seq)) return;

	/* hide_uri && ファイルであればリンクをつけない */
	if (is_hideuri && rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
		divy_sbuf_append(sbuf, "#");
	}
	/* ブラウザ経由のリクエスト(モバイルリクエストではない) */
	else if (infotype != DIVY_INFOTYPE_mobile) {

		/* XXX
		 * シングルクォータ以外すべてURLエンコードされます
		 */
		e_uri = dav_divy_escape_uri(p, rdb_r->uri);

		/*
		 * ファイルでダウンロード用である場合、そのURLはautoindexの
		 * javascriptの関数に設定されます。
		 * javascriptに埋める為のURLはJavascript変数としてのエスケープ
		 * 処理を行う必要があります。
		 */
		if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE && is_downloadonly) {
			e_uri = dav_divy_replace_str(p, e_uri, "/", "\\/", NULL);
		}

		/*
		 * ブラウザのURLは'（シングルクォート）はcross site scripting問題
		 * がある為URLエンコードしなければなりません。
		 *
		 * しかし、Apacheではシングルクォートは対象になっていません。
		 * (上の (XXX) ポイントの処理を指します）
		 * さらにXML的には問題がなかったため、今まで放置されてきました。
		 * 既に作られているURIを展開しファイル名はエンコードして
		 * 親フォルダは足りないシングルクォートだけをエンコードして戻します
		 *
		 * このエンコード処理は表示名の実体参照のようにxsltの影響を
		 * 受けません。つまり勝手に変換はされない為、スタイルシートの
		 * 変更はありません。
		 *
		 */
		divy_sbuf_append(sbuf, dav_divy_replace_str(p, e_uri, "'", "%27", NULL));

		divy_sbuf_appendbyte(sbuf, 1, "?");
		divy_sbuf_append(sbuf, seq);

		/* ファイルの場合 */
		if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
			/* WebDAV メソッドGW用の識別子(飾り)をつける */
			divy_sbuf_append(sbuf, "&amp;WEBDAV_METHOD=GET");

			if (is_downloadonly) {
				/* ダウンロード専用であることを示すフラグをつける */
				divy_sbuf_append(sbuf, "&amp;"DIVY_URIPARAM_DOWNLOAD_ONLY"="
										DIVY_URIPARAM_DOWNLOAD_ONLY_VAL);
			}

			/* リソースの場合だけ"name=(final path semgne)"というURLパラメータを付ける
			 * (note)
			 *   IEはURL文字列の末尾とfinal path segment が異なる場合、拡張子を切り捨てる
			 *   ことがあります。これを防止する目的でURLパラメータの最後を
			 *   final path segment にします。
			 */
			divy_sbuf_append(sbuf, "&amp;name=");
			divy_sbuf_append(sbuf, divy_url_encode(p, rdb_r->displayname));
		}
		else {
			/* WebDAV メソッドGW用の識別子(飾り)をつける */
			divy_sbuf_append(sbuf, "&amp;WEBDAV_METHOD=PROPFIND");
		}
	}
	/* モバイルリクエスト */
	else {
		/* モバイルのURIには'&'があるのでXSLパーサがエラーを出す
		 * これを避ける目的でXMLエスケープだけを実施する */
		divy_sbuf_append(sbuf, apr_xml_quote_string(p,
				apr_psprintf(p, "%s&%s", rdb_r->uri, seq), 0));
	}
}

static int _remove_browser_session(request_rec *r) 
{
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	apr_pool_t *p      = r->pool;
	const char *header = NULL;
	char *cookie_value = NULL;
	apr_table_t* jar;

	/* Cookieヘッダを削除する */
	(void)divy_set_delete_session_cookie(r);

	/* Cookieヘッダを取得する */
	header = apr_table_get(r->headers_in, "Cookie");
	if (IS_FILLED(header)) {
		/* Cookie パース */
		jar = apr_table_make(p, APREQ_DEFAULT_NELTS);
		if (apreq_parse_cookie_header(p, jar, header) != APR_SUCCESS) {
			/* パース失敗 */
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"Failed to parse cookie. cookie value = %s", header);
			return 1;
		}
		/* セッションを削除する */
		cookie_value = (char *)apr_table_get(jar, TF_COOKIE_SES_NAME);
		if (IS_FILLED(cookie_value)) {
			/* 削除できたか否かは調べない */
			if (dconf->usememcache == DIVY_TFMEMCACHE_ON) {
				apr_status_t st;
				apr_size_t length = 0;
				apr_uint16_t flags = 0;
				char *value = NULL;

				/* 誰のセッションが削除されたか調べる */
				st = divy_memcache_get(p, dconf->memd, (char*)dav_divy_get_root_uri(r), cookie_value, &value, &length, &flags);

				/* memcacheより削除 */
				st = divy_memcache_delete(p, dconf->memd,
							 (char*)dav_divy_get_root_uri(r), cookie_value, 0);

				if (st == APR_SUCCESS) {
					ERRLOG1(p, APLOG_INFO, DIVY_FST_INFO + DIVY_SST_DATA,
						"delete browser session uid = %s", ap_getword_nulls(p, (const char**)&value, ':'));
				}
			}
			else {
				(void)divy_rdbo_delete_session(r, NULL, NULL, cookie_value);
			}
		}
	}

	return 0;
}

/**
 * スタイルシート(XSLT)のパスを組み立てて返却する.
 *
 * @param wp apr_pool_t *
 * @param prefix const char * スタイルシートプレフィックス名
 * @param r request_rec *
 * @return char * 組み立てたスタイルシートパス
 */
static char * _get_stylesheet_path(apr_pool_t *wp, const char *prefix, request_rec *r)
{
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char *lang = divy_get_language_param(r);
	char *path = NULL;
	apr_status_t rv;
	apr_file_t *fd = NULL;

	path = ap_make_full_path(wp, dconf->stylesheetroot,
									apr_psprintf(wp, "%s.%s", prefix, lang));

	/* ファイルの存在を確かめる */
	rv = apr_file_open(&fd, path, APR_READ | APR_BINARY, 0, wp);
	if (fd != NULL) apr_file_close(fd);
	if (rv == APR_SUCCESS) {
		return path;
	}

	/* lang パラメータをつけない */
	path = ap_make_full_path(wp, dconf->stylesheetroot, prefix);

	return path;
}

