/**
 * $Id$
 *
 * TeamFile CGI プログラム呼び出し用インターフェースの定義
 *
 * [ 役割 ]
 *   * TeamFile で使用するシステムCGI、プラグインCGI用コードの共通化
 */
#include "httpd.h"
#include "http_request.h"	/* for ap_internal_redirect_handler */
#include "util_script.h"
#include "util_filter.h"

#include "apr.h"
#include "ap_mpm.h"
#include "apr_file_info.h"
#include "apr_strings.h"
#include "apr_poll.h"
#include "apr_buckets.h"

#include "tf_cgi.h"
#include "tf_env.h"
#include "tf_logger.h"
#include "tf_folder.h"
#include "mod_dav_tf.h"
#include "tf_rdbo.h"
#include "util.h"
#include "util_ldap.h"
#include "util_auth.h"
#include "tf_plugin.h"
#include "tf_autoindex.h"
#include "tf_extmap.h"
#include "auth.h"
#include "tf_confirmreading.h"

APLOG_USE_MODULE(dav_tf);

#ifdef DAV_SUPPORT_POST_HANDLING
/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
/**
 *
 */
typedef struct divy_cgi_bucket_data	divy_cgi_bucket_data;
struct divy_cgi_bucket_data {
	apr_pollset_t *pollset;
	request_rec *r;
};

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static int _run_cgi(request_rec *r, divy_cgi_property *cgi_pr);
static void _cgi_loggger(apr_pool_t *p, apr_status_t rv, const char *desc);
static void _discard_script_output(apr_bucket_brigade *bb);
#if APR_FILES_AS_SOCKETS
static apr_bucket * _cgi_bucket_create(request_rec *r, apr_file_t *out,
                                       apr_file_t *err, apr_bucket_alloc_t *list);
static apr_bucket * _cgi_bucket_dup(divy_cgi_bucket_data *data, apr_bucket_alloc_t *list);
static apr_status_t _cgi_read_stdout(apr_bucket *a, apr_file_t *out,
                                     const char **str, apr_size_t *len);
static apr_status_t _cgi_bucket_read(apr_bucket *b, const char **str,
                                     apr_size_t *len, apr_read_type_e block);
static apr_status_t _cgi_log_error(request_rec *r, apr_file_t *script_err);
#endif	/* APR_FILES_AS_SOCKETS */
static void _cgi_setup_user_envvars(request_rec *r);
static void _cgi_setup_extmap_envvars(request_rec *r, const char *lang);
#ifdef DIVY_SUPPORT_PLUGIN
static void _cgi_setup_plugin_envvars(request_rec *r, const char *lang, divy_cgi_property *cgi_pr);
#endif	/* DIVY_SUPPORT_PLUGIN */
static void _cgi_setup_rproxy_envvars(request_rec *r);
static void _cgi_setup_notifyurl_envvars(request_rec *r);
static void _cgi_setup_cgiconinfo_envvars(request_rec *r);
static void _cgi_setup_remoteclient_envvars(request_rec *r);

/*------------------------------------------------------------------------------
  Define static values
  ----------------------------------------------------------------------------*/
#if APR_FILES_AS_SOCKETS
/**
 * CGI のstdoutとstderr を読み込むbucket の定義(生成)
 * read だけ定義できれば十分.
 */
static const apr_bucket_type_t bucket_type_cgi = {
	"TF_CGI", 5, APR_BUCKET_DATA,
	apr_bucket_destroy_noop,		/* 未実装 */
	_cgi_bucket_read,
	apr_bucket_setaside_notimpl,	/* 未実装 */
	apr_bucket_split_notimpl,		/* 未実装 */
	apr_bucket_copy_notimpl			/* 未実装 */
};
#endif	/* APR_FILES_AS_SOCKETS */

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * u_spec が示すリソースはCGI実行が可能かどうかを判定する.
 *
 */
DIVY_DECLARE(int) divy_cgi_executable(request_rec *r, divy_uri_spec *u_spec)
{
	if (u_spec == NULL) {
		(void) divy_parse_uri(r->pool, dav_divy_get_root_uri(r), r->uri, &u_spec);
		if (u_spec == NULL) {
			return 0;
		}
	}

	/*
	 * [ 有効なCGI ]
	 *
	 *  * システムCGI
	 *  * プラグインCGI
	 */
	if (u_spec->infotype == DIVY_INFOTYPE_cgibin_e
#ifdef DIVY_SUPPORT_PLUGIN
		|| u_spec->infotype == DIVY_INFOTYPE_plugin_cgibin_e
#endif	/* DIVY_SUPPORT_PLUGIN */
		) {
		return 1;
	}
	else {
		return 0;
	}
}

/**
 * DIVY_SYSCGI_TOP_NAME が示すトップCGIが有効かどうか.
 *
 */
DIVY_DECLARE(int) divy_cgi_support_topcgi(request_rec *r)
{
	/* 今は常に有効. 将来、ディレクティブ等で判定が必要になるかもしれない */
	return 1;
}

/**
 * u_spec が示すURIパートのCGIプログラムのプロパティを取得する.
 * u_spec が$root を示す場合、トップCGIのプロパティを取得します.
 *
 */
DIVY_DECLARE(int) divy_cgi_get_property(request_rec *r, divy_uri_spec *u_spec,
                                        divy_cgi_property **cgi_pr)
{
	apr_pool_t *p = r->pool;
	dav_divy_server_conf *sconf;
	apr_status_t rv;
	divy_uri_spec *cgi_u_spec;
	const char *root_uri = dav_divy_get_root_uri(r);

	if (u_spec == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"u_spec was EMPTY.");
		return 1;
	}

	*cgi_pr = NULL;	/* 初期化 */

	/* トップCGI($root) または CGIリソース以外の場合 */
	if (!(u_spec->infotype == DIVY_INFOTYPE_root
		|| u_spec->infotype == DIVY_INFOTYPE_cgibin_e
#ifdef DIVY_SUPPORT_PLUGIN
		|| u_spec->infotype == DIVY_INFOTYPE_plugin_cgibin_e
#endif	/* DIVY_SUPPORT_PLUGIN */
		)) {
		return 0;	/* CGIプロパティを持ち得ないリソースである */
	}

	/* トップCGIは特別扱い */
	if (u_spec->infotype == DIVY_INFOTYPE_root) {
		(void) divy_parse_uri(p, root_uri,
				 	divy_build_topcgi_uri(p, root_uri), &cgi_u_spec);
	}
	else {
		cgi_u_spec = u_spec;
	}

	*cgi_pr = apr_pcalloc(p, sizeof(divy_cgi_property));
	(*cgi_pr)->p            = p;
	(*cgi_pr)->u_spec       = cgi_u_spec;
	(*cgi_pr)->name         = apr_pstrdup(p, cgi_u_spec->final_path_segment);
	(*cgi_pr)->relativepath = apr_pstrdup(p, cgi_u_spec->other_part);
	if (strncmp((*cgi_pr)->name, "nph-", 4) == 0) {
		(*cgi_pr)->is_nph = 1;	/* NPH CGIである */
	}
	else {
		(*cgi_pr)->is_nph = 0;
	}
	(*cgi_pr)->cgi_in  = NULL;
	(*cgi_pr)->cgi_out = NULL;
	(*cgi_pr)->cgi_err = NULL;

	/*
	 * cgi_u_spec から物理パスへの変換
	 */
	/* システムCGIの場合 */
	if (cgi_u_spec->infotype == DIVY_INFOTYPE_cgibin_e) {
		sconf = dav_divy_get_server_config(r->server);

		/* 自分自身の絶対パスを算出 */
		if (IS_FILLED(sconf->syscgirootpath)) {
			(*cgi_pr)->path = ap_make_full_path(p, sconf->syscgirootpath,
												(*cgi_pr)->relativepath);
		}
		else {
			(*cgi_pr)->path = ap_make_full_path(p, DIVY_SYSCGI_DEFAULT_ROOTPATH,
												(*cgi_pr)->relativepath);
		}
		ap_no2slash((char *)(*cgi_pr)->path);

		/* 親パスを算出 */
		(*cgi_pr)->ppath = ap_make_dirstr_parent(p, (*cgi_pr)->path);
		ap_no2slash((char *)(*cgi_pr)->ppath);
	}
#ifdef DIVY_SUPPORT_PLUGIN
	/* プラグインCGIの場合 */
	else {
		/* ファイルパスパターン : $plugin_root/$plugin_name/cgi-bin/xxxx
		 * URLパターン          : $root/.plugin/$plugin_name/cgi-bin/xxxx  */
		char *uri_prefix = dav_divy_make_uri(p, dav_divy_get_root_uri(r),
		                                     DIVY_PLUGINFOLDER_NAME, NULL);
		apr_size_t len   = strlen(uri_prefix);
		char *uri_suffix;
		const char *plugin_root = divy_pi_get_pluginroot(r);

		/* 特殊フォルダより下のパスを切り出す */
		if (strncmp(cgi_u_spec->uri, uri_prefix, len) != 0) {
			/* URIの先頭が一致していなかった (ありえないが一応) */
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Invalid uri found. uri prefix was diffrent from uri parts.");
			return 1;
		}
		uri_suffix = apr_pstrdup(p, &cgi_u_spec->uri[len]);

		/* (仮定) URIのpath segment 区切り文字とファイルの区切り文字は一緒 */
		(*cgi_pr)->path  = ap_make_full_path(p, plugin_root, uri_suffix);
		ap_no2slash((char *)(*cgi_pr)->path);
		(*cgi_pr)->ppath = ap_make_dirstr_parent(p, (*cgi_pr)->path);
		ap_no2slash((char *)(*cgi_pr)->ppath);
	}
#endif	/* DIVY_SUPPORT_PLUGIN */

	/* CGIファイルの状態を調べる */
	if (IS_FILLED((*cgi_pr)->path)) {
		rv = apr_stat(&(*cgi_pr)->finfo, (*cgi_pr)->path, APR_FINFO_NORM, p);
		if (rv != APR_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to get system-cgi stat.(path = %s, code : %d)",
					(*cgi_pr)->path, rv);
			*cgi_pr = NULL;	/* 無かったことにする */
			return 1;
		}

		/* ファイルではなかった */
		if ((*cgi_pr)->finfo.filetype != APR_REG) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The system cgi file was not regular file (file = %s).",
					(*cgi_pr)->path);
			*cgi_pr = NULL;
			return 1;
		}
	}

	return 0;
}

/**
 * cgi_pr で指定されたCGIプログラムを実行する.
 *
 */
DIVY_DECLARE(int) divy_cgi_run(request_rec *r, divy_cgi_property *cgi_pr)
{
	apr_status_t rv;
	apr_bucket_brigade *bb;
	int seen_eos, child_stopped_reading;
	const char *data;
	apr_size_t len;
	apr_pool_t *p = r->pool;
	conn_rec *c   = r->connection;
#if APR_FILES_AS_SOCKETS
	apr_bucket *b;
#else
	apr_bucket *b, *be;
#endif

	if (cgi_pr == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"cgi_pr was EMPTY.");
		return 1;
	}

	/* CGI に渡す環境変数を設定する */
	divy_cgi_setup_lc_envvars(r, cgi_pr);

	/* CGI の実行 */
	if (_run_cgi(r, cgi_pr)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to execute CGI commandline.");
		return 1;
	}

	/* 起動したCGI(Child プロセス) にデータを流し込む.
	 * pipe は既に接続済みです(see _run_cgi) */
	bb = apr_brigade_create(p, c->bucket_alloc);
	seen_eos = 0;
	child_stopped_reading = 0;

	do {
		apr_bucket *bucket;

		rv = ap_get_brigade(r->input_filters, bb, AP_MODE_READBYTES,
							APR_BLOCK_READ, HUGE_STRING_LEN);
		if (rv != APR_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to get brigade.(code = %d)", rv);
			return 1;
		}

		bucket = APR_BRIGADE_FIRST(bb);
		while(bucket != APR_BRIGADE_SENTINEL(bb)) {

			if (APR_BUCKET_IS_EOS(bucket)) {
				seen_eos = 1;
				break;
			}

			/* We can't do much with this. */
			if (APR_BUCKET_IS_FLUSH(bucket)) {
				continue;
			}
	
			/* Childプロセスが停止したら、EOSを読まなければならない */
			if (child_stopped_reading) {
				continue;
			}

			/* ストリームからのデータ読み込み */
			apr_bucket_read(bucket, &data, &len, APR_BLOCK_READ);

			/* CGI にデータを書き込む */
			rv = apr_file_write_full(cgi_pr->cgi_out, data, len, NULL);
			if (rv != APR_SUCCESS) {
				child_stopped_reading = 1;
			}

			bucket = APR_BUCKET_NEXT(bucket);
		}
		apr_brigade_cleanup(bb);
	} while (!seen_eos);

	/* CGIへの出力口をフラッシュして閉じる */
	apr_file_flush(cgi_pr->cgi_out);
	apr_file_close(cgi_pr->cgi_out);
    apr_brigade_cleanup(bb);

	/* CGIからの出力を取得する */
#if APR_FILES_AS_SOCKETS
	/* (note) Linux だとこっちが動きます */
	apr_file_pipe_timeout_set(cgi_pr->cgi_in, 0);
	apr_file_pipe_timeout_set(cgi_pr->cgi_err, 0);

	b = _cgi_bucket_create(r, cgi_pr->cgi_in, cgi_pr->cgi_err, c->bucket_alloc);
	APR_BRIGADE_INSERT_TAIL(bb, b);
#else
	/* (note) こっちが動く場合、stderr へのログ出力が行われなくなり、
	 * stdout と共に画面出力されるようになります. */
	b = apr_bucket_pipe_create(cgi_pr->cgi_in, c->bucket_alloc);
	APR_BRIGADE_INSERT_TAIL(bb, b);

	be = apr_bucket_pipe_create(cgi_pr->cgi_err, c->bucket_alloc);
	APR_BRIGADE_INSERT_TAIL(bb, be);
#endif
	/* 終了bucket を挿入する */
	b = apr_bucket_eos_create(c->bucket_alloc);
	APR_BRIGADE_INSERT_TAIL(bb, b);

	/* NPH CGI の場合 */
    if (cgi_pr->is_nph) {
		struct ap_filter_t *cur = r->proto_output_filters;

		while (cur && cur->frec->ftype < AP_FTYPE_CONNECTION) {
			cur = cur->next;
		}

		r->output_filters = r->proto_output_filters = cur;
		rv = ap_pass_brigade(r->output_filters, bb);
	}
	/* NPH CGIでは無い場合 */
	else {
		const char *location;
		char sbuf[MAX_STRING_LEN];
		int ret;

		/* CGI から出力されたヘッダとボディデータ及びstderr を読み取るための準備をする */
		if ((ret = ap_scan_script_header_err_brigade(r, bb, sbuf))) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to get script's header and body. (code = %d)", ret);
			return 1;
		}

		/* 200 OK でLocation ヘッダがあれば、リダイレクトする */
		location = apr_table_get(r->headers_out, "Location");
		if (location && location[0] == '/' && r->status == 200) {
			/* CGI スクリプトの出力を読み捨てる */
			_discard_script_output(bb);
			apr_brigade_destroy(bb);
			apr_file_pipe_timeout_set(cgi_pr->cgi_err, r->server->timeout);

			r->method = apr_pstrdup(r->pool, "GET");
			r->method_number = M_GET;

			/* Content-Length ヘッダを除去する */
			apr_table_unset(r->headers_in, "Content-Length");
			ap_internal_redirect_handler(location, r);

			return 0;
		}
		/* 200 OK でLocation ヘッダがあっても、Locationの始まりが'/' でなければ読み捨て */
		else if (location && r->status == 200) {
			/* CGI スクリプトの出力を読み捨てる */
			_discard_script_output(bb);
			apr_brigade_destroy(bb);
			return 0;
		}

		/* 次のoutput フィルターに渡す */
		rv = ap_pass_brigade(r->output_filters, bb);
	}

	if (rv == APR_SUCCESS && !r->connection->aborted) {
		apr_file_pipe_timeout_set(cgi_pr->cgi_err, r->server->timeout);
    }
	apr_file_close(cgi_pr->cgi_err);

	return 0;
}

/**
 * TeamFile 独自の"グローバル"CGI 環境変数をセットアップする.
 *
 */
DIVY_DECLARE(void) divy_cgi_setup_gl_envvars(request_rec *r)
{
	apr_table_t *e = r->subprocess_env;
	apr_pool_t  *p = r->pool;
	const char  *str;
	const char  *lang = divy_get_language_param(r);
	const char  *default_lang;
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

#ifdef DAV_SUPPORT_POST_HANDLING
	/* システムCGIルート */
	str = sconf->syscgirootpath;
	if (IS_FILLED(str)) {
		apr_table_setn(e, DIVY_CGIENV_SYSCGI_ROOT, ap_escape_html(p, str));
	}
#endif	/* DAV_SUPPORT_POST_HANDLING */

	/* サポートしている言語タグ一覧 */
	if (sconf->supported_languages != NULL) {
		char *supported_lang = NULL;
		divy_cset_index_t *idx;

		for (idx = divy_cset_first(sconf->supported_languages);
				idx != NULL; idx = divy_cset_next(idx)) {
			divy_cset_this(idx, &str);
			if (IS_EMPTY(supported_lang)) {
				supported_lang = apr_pstrdup(p, str);
			}
			else {
				supported_lang = apr_psprintf(p, "%s %s", supported_lang, str);
			}
		}
		if (IS_FILLED(supported_lang)) {
			apr_table_setn(e, DIVY_CGIENV_SUPPORTED_LANGTAG, supported_lang);
		}
	}

	/* デフォルト言語タグ */
	default_lang = divy_get_default_languagetag(r);
	if (IS_FILLED(default_lang)) {
		apr_table_setn(e, DIVY_CGIENV_DEFAULT_LANGTAG, default_lang);
	}

	/* 言語パラメータ(ja/en) */
	if (IS_FILLED(lang)) {
		apr_table_setn(e, DIVY_CGIENV_LANGPARAM, lang);

		/* 言語パラメータをTeamFileが判別したものにする(Apacheのネゴシエーションを無効化)
		 * --> 環境変数 prefer-language を設定 */
		apr_table_setn(r->subprocess_env, "prefer-language", apr_pstrdup(p, lang));
	}
	else if (IS_FILLED(default_lang)) {
		/* lang が無いのならdefault_lang を使う */
		apr_table_setn(r->subprocess_env, "prefer-language", apr_pstrdup(p, default_lang));
	}
}

/**
 * TeamFile 独自の"ローカル"CGI環境変数をセットアップする.
 *
 * @param r request_rec *
 * @param cgi_pr divy_cgi_property *
 */
DIVY_DECLARE(void) divy_cgi_setup_lc_envvars(request_rec *r, divy_cgi_property *cgi_pr)
{
	apr_table_t *e = r->subprocess_env;
	apr_pool_t *p  = r->pool;
	const char *root_uri = dav_divy_get_root_uri(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	dav_divy_dir_conf    *dconf = dav_divy_get_dir_config(r);
	const char *str;
	const char *lang = divy_get_language_param(r);
	divy_cset_t	*disablefmt = NULL;
	divy_cset_index_t *idx;
	const char *val = NULL;
	char *format = NULL;

	/* CGIに渡す共通の環境変数をセットする */
	ap_add_common_vars(r);

	/* HTTP/1.1 に必要な環境変数をセット */
	ap_add_cgi_vars(r);

	/*
	 * ap_add_cgi_vars(r) で正しく設定されない値を置き換える.
	 * (note) 正しく設定されないのは、TeamFile が支配するLocationだからです.
	 */
	if (cgi_pr != NULL) {
		/* CGI スクリプトの絶対パス(PATH_INFOを含む) */
		apr_table_setn(e, "PATH_TRANSLATED", apr_pstrdup(p, cgi_pr->path));

		/* CGI プログラムルート以降のURI (/cgi-bin/zzz/xxxx) */
		apr_table_setn(e, "SCRIPT_NAME",
			divy_build_cgi_scriptname(p, cgi_pr->u_spec->infotype, cgi_pr->relativepath));

		/* CGI プログラムの絶対パス */
		apr_table_setn(e, "SCRIPT_FILENAME", apr_pstrdup(p, cgi_pr->path));
	}

	/*
	 * TeamFile 専用の環境変数を設定する
	 */

	/* アクセスユーザの環境変数をセットアップする */
	_cgi_setup_user_envvars(r);
	
	/* ロケーションURI */
	apr_table_setn(e, DIVY_CGIENV_LOCATION, ap_escape_uri(p, root_uri));

	/* プライベートフォルダURI */
	apr_table_setn(e, DIVY_CGIENV_PRIVATEFOLDER_URI,
					ap_escape_uri(p, divy_build_user_uri(p, root_uri, divy_get_userid(r))));

	/* グループフォルダURI */
	apr_table_setn(e, DIVY_CGIENV_GROUPFOLDER_URI,
					ap_escape_uri(p, divy_build_group_uri(p, root_uri, NULL)));

	/* ユーティリティフォルダURI */
	apr_table_setn(e, DIVY_CGIENV_UTILITYFOLDER_URI,
					ap_escape_uri(p, divy_build_utility_uri(p, root_uri)));

	/* システムメッセージ(in ユーティリティフォルダ)のURI */
	apr_table_setn(e, DIVY_CGIENV_UTILSYSMSG_URI,
					ap_escape_uri(p, divy_build_u_msg_uri(p, root_uri)));

	/* クライアント更新(in ユーティリティフォルダ)のURI */
	apr_table_setn(e, DIVY_CGIENV_UTILUPDATEDCLIENT_URI,
					ap_escape_uri(p, divy_build_u_update_uri(p, root_uri)));

	/* ブランド名称 */
	str = (IS_EMPTY(sconf->brandname)) ? DIVY_PRODUCT_NAME : sconf->brandname;
	apr_table_setn(e, DIVY_CGIENV_BRANDNAME, ap_escape_html(p, str));

	/* サーババージョン */
	str = apr_psprintf(p, "%s", sconf->tfversion);
	apr_table_setn(e, DIVY_CGIENV_TFVERSION, ap_escape_html(p, str));

	/* メールテンプレートのルートパス */
	if (IS_FILLED(dconf->mltemplateroot)) {
		apr_table_setn(e, DIVY_CGIENV_MLTEMPLATE_ROOT, ap_escape_html(p, dconf->mltemplateroot));
	}
	/* サーバ主体メールかどどうか */
	if (dconf->mlserversend == DIVY_MLSERVERSEND_ON) {
		apr_table_setn(e, DIVY_CGIENV_MLSERVERSEND, "on");
	}
	else if (dconf->mlserversend == DIVY_MLSERVERSEND_MIDDLE) {
		apr_table_setn(e, DIVY_CGIENV_MLSERVERSEND, "middle");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_MLSERVERSEND, "off");
	}

	/* メール編集機能のON/OFF */
	if (dconf->mledit == DIVY_MLEDITL_ON) {
		apr_table_setn(e, DIVY_CGIENV_MLEDIT, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_MLEDIT, "off");
	}

	/* (note) 以下のCGI環境変数は divy_cgi_setup_gl_envvars(r) で設定することにした.
	 * DIVY_CGIENV_SUPPORTED_LANGTAG, DIVY_CGIENV_DEFAULT_LANGTAG */
	if (divy_support_session(r)) {
		/*
		 * DIVY_CGIENV_LANGPARAM はdivy_cgi_setup_gl_envvars(r)でも設定は
		 * 行うがセッションの場合はブラウザから再提出される可能性もある為
		 * 二重で管理します
		 */
		apr_table_setn(e, DIVY_CGIENV_LANGPARAM, divy_get_language_param(r));
	}

	/* ログ取得機能の有無 */
#ifdef DIVY_SUPPORT_LOGACCESS
	if (dconf->logaccess == DIVY_LOGACCESS_ON) {
		apr_table_setn(e, DIVY_CGIENV_TF_LOGACCESS, "on");
	}
	else if (dconf->logaccess == DIVY_LOGACCESS_ONALL) {
		apr_table_setn(e, DIVY_CGIENV_TF_LOGACCESS, "on-all");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_LOGACCESS, "off");
	}
	/* ログファイルの格納ディレクトリ(APACHE_INSTALL_ROOT はMakefile で定義) */
	apr_table_setn(e, DIVY_CGIENV_LOGFILEROOT, DIVY_LOGFILEROOT);
#endif	/* DIVY_SUPPORT_LOGACCESS */

	/* 自動削除機能の有無 */
	if (divy_support_autodelete(r)) {
		apr_table_setn(e, DIVY_CGIENV_TF_AUTODELETE, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_AUTODELETE, "off");
	}

	/* 操作ログ機能の有無 */
	if (divy_support_logoperation(r)) {
		apr_table_setn(e, DIVY_CGIENV_TF_LOGOPERATION, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_LOGOPERATION, "off");
	}

	/* グループ制約機能の有無 */
	if (divy_support_grpconstraints(r)) {
		apr_table_setn(e, DIVY_CGIENV_TF_GROUPCONSTRAINTS, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_GROUPCONSTRAINTS, "off");
	}

	/* ライセンスアップローダ機能の有無 */
	if (divy_support_licenseuploader(r)) {
		apr_table_setn(e, DIVY_CGIENV_TF_LICENSEUPLOADER, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_LICENSEUPLOADER, "off");
	}
	apr_table_setn(e, DIVY_CGIENV_BIN_ROOT, DIVY_BINROOT);
	apr_table_setn(e, DIVY_CGIENV_SBIN_ROOT, DIVY_SBINROOT);

	/* グループリーダ機能の有無 */
	if (divy_support_groupleader(r)) {
		apr_table_setn(e, DIVY_CGIENV_TF_GROUPLEADER, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_GROUPLEADER, "off");
	}

	/* 開封通知機能の有無と自動解除までの日数 */
	if (divy_support_confirmreading(r)) {
		apr_table_setn(e, DIVY_CGIENV_TF_CONFIRMREADING, "on");
		apr_table_setn(e, DIVY_CGIENV_TF_CONFIRMREADING_RESERVEDDAY,
							apr_psprintf(p, "%d", dconf->cr_reservedday));
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_CONFIRMREADING, "off");
		apr_table_setn(e, DIVY_CGIENV_TF_CONFIRMREADING_RESERVEDDAY, "0");
	}

	/* 拡張システムメッセージの有無 */
	if (divy_support_extendsysmsg(r)) {
		apr_table_setn(e, DIVY_CGIENV_TF_EXTENDSYSMSG, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_EXTENDSYSMSG, "off");
	}

	/* ブラウザセッションの有無 */
	if (divy_support_session(r)) {
		apr_table_setn(e, DIVY_CGIENV_TF_USESESSION, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_USESESSION, "off");
	}

	if (divy_util_ldap_use_ldap(r)) {
		apr_table_setn(e, DIVY_CGIENV_TF_DISABLE_CHANGE_PASSWORD, "on");
	}
	else {
		if (divy_disable_change_password(r)) {
			apr_table_setn(e, DIVY_CGIENV_TF_DISABLE_CHANGE_PASSWORD, "on");
		}
		else {
			apr_table_setn(e, DIVY_CGIENV_TF_DISABLE_CHANGE_PASSWORD, "off");
		}
	}

	/* BOXのポリシー有効期限 */
	apr_table_setn(e, DIVY_CGIENV_TFBOXPOLICYLIMIT, 
					(divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN) ? 0:
						apr_psprintf(p, "%"APR_INT64_T_FMT, dconf->tfboxpolicylimit));

	/* BOXに登録しているメールアドレスに送信をするか? */
	if (dconf->boxwithoutmail == BOX_WITHOUTMAIL_ALL) {
		apr_table_setn(e, DIVY_CGIENV_TF_BOXWITHOUTMAIL, "on");
	}
	else
	if ((dconf->boxwithoutmail == BOX_WITHOUTMAIL_USER)
			&& (divy_get_adminmode(r) != DIVY_ADMINMODE_ADMIN)) {
		apr_table_setn(e, DIVY_CGIENV_TF_BOXWITHOUTMAIL, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_BOXWITHOUTMAIL, "off");
	}

	/* BOXの利用不可能フォーマット */
	disablefmt = divy_cset_make(p);
	if (dconf->tfboxdisableformat & DIVY_BOX_DISABLE_FORMAT_LIST) {
		divy_cset_set(disablefmt, "list");
	}
	if (dconf->tfboxdisableformat & DIVY_BOX_DISABLE_FORMAT_XML) {
		divy_cset_set(disablefmt, "xml");
	}
	if (dconf->tfboxdisableformat & DIVY_BOX_DISABLE_FORMAT_JSON) {
		divy_cset_set(disablefmt, "json");
	}
	if (dconf->tfboxdisableformat & DIVY_BOX_DISABLE_FORMAT_RSS) {
		divy_cset_set(disablefmt, "rss");
	}

	for(idx = divy_cset_first(disablefmt);
					idx != NULL; idx = divy_cset_next(idx)) {
		divy_cset_this(idx, &val);
		if (format != NULL) {
			format = apr_pstrcat(p, format, " ", val, NULL);
		}
		else {
			format = apr_pstrdup(p, val);
		}
	}

	/* アップロードポリシーの有無 */
	if (divy_support_upload_policy(r)) {
		apr_table_setn(e, DIVY_CGIENV_TF_UPLOADPOLICY, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_UPLOADPOLICY, "off");
	}

	/* 二段階認証の有無 */
	if (divy_support_2FA(r)) {
		apr_table_setn(e, DIVY_CGIENV_TF_TWO_FACTOR_AUTHENTICATION, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_TF_TWO_FACTOR_AUTHENTICATION, "off");
	}

	apr_table_setn(e, DIVY_CGIENV_TFBOXDISABLE_FORMAT, format);

	/* 自動削除用コンフィグファイルパス */
	apr_table_setn(e, DIVY_CGIENV_AUTODELETE_CONFIGROOT, DIVY_AUTODELETE_CONFIGROOT);

	/* Apache コンフィグルートディレクトリパス */
	apr_table_setn(e, DIVY_CGIENV_APCONFIGROOTPATH, DIVY_APCONFIGROOT);

	/* TeamFile のコンフィグルートディレクトリパス */
	apr_table_setn(e, DIVY_CGIENV_CONFIGROOTPATH, DIVY_CONFIGROOT);

	/* クライアント通知用NotifyUrl関連の環境変数のセットアップ */
	_cgi_setup_notifyurl_envvars(r);

	/* システムCGI用の環境変数のセットアップ */
	_cgi_setup_cgiconinfo_envvars(r);

	/* リモートアクセスクライアントの接続情報をセットアップ */
	_cgi_setup_remoteclient_envvars(r);

#ifdef DIVY_SUPPORT_PLUGIN
	/* プラグイン用環境変数のセットアップ */
	_cgi_setup_plugin_envvars(r, lang, cgi_pr);
#endif	/* DIVY_SUPPORT_PLUGIN */

	/* 拡張子マッピング用環境変数のセットアップ */
	_cgi_setup_extmap_envvars(r, lang);

	/* リバースプロキシ対応用環境変数のセットアップ */
	_cgi_setup_rproxy_envvars(r);
}


/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * cgi_pr が示すCGIを別のChildプロセスで実行する.
 */
static int _run_cgi(request_rec *r, divy_cgi_property *cgi_pr)
{
	apr_status_t rv;
	apr_procattr_t *procattr = NULL;
	const char * const *env  = NULL;
	apr_proc_t *procnew;
	apr_pool_t *p = cgi_pr->p;
	const char *command = cgi_pr->path;

	/* Child プロセスに渡す環境変数テーブルの生成 */
	env = (const char * const *)ap_create_environment(p, r->subprocess_env);

	/* process の環境を作成 */
	if ((rv = apr_procattr_create(&procattr, p)) != APR_SUCCESS) {
		ERRLOG1(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to create process attribute.(code = %d)", rv);
		return 1;
	}

	/* 作成するChildのパイプに親プロセスのstdin, stdout, stderr をどう繋ぐか */
	if ((rv = apr_procattr_io_set(procattr, APR_CHILD_BLOCK, APR_CHILD_BLOCK,
	                              APR_CHILD_BLOCK)) != APR_SUCCESS) {
		ERRLOG1(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to set child io_set.(code = %d)", rv);
		return 1;
	}

	/* CGIが動作するカレントディレクトリをppath に移動する */
	if ((rv = apr_procattr_dir_set(procattr, cgi_pr->ppath)) != APR_SUCCESS) {
		ERRLOG2(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to change directory. (code = %d, dir = %s)", rv, cgi_pr->ppath);
		return 1;
	}

	/* コマンドタイプ設定 */
	if ((rv = apr_procattr_cmdtype_set(procattr, APR_PROGRAM)) != APR_SUCCESS) {
		ERRLOG2(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to set cmdtype.(code = %d, cmdtype = %d)", rv, APR_PROGRAM);
		return 1;
	}

	/* CGIプログラムをデタッチ状態では生成しない */
	if ((rv = apr_procattr_detach_set(procattr, 0)) != APR_SUCCESS) {
		ERRLOG1(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to set detach mode.(code = %d, detach = 0)", rv);
		return 1;
	}

	/* 親プロセスのアドレス空間を使ってCGIを起動する */
	if ((rv = apr_procattr_addrspace_set(procattr, 0)) != APR_SUCCESS) {
		ERRLOG1(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to set addspace mode.(code = %d, mode = 0)", rv);
		return 1;
	}

	/* Childプロセスのエラーを出力するように設定 */
	if ((rv = apr_procattr_child_errfn_set(procattr, _cgi_loggger)) != APR_SUCCESS) {
		ERRLOG1(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to set error function.(code = %d)", rv);
		return 1;
	}

	/* プロセス構造体初期化 */
	procnew = apr_pcalloc(p, sizeof(apr_proc_t));

	/* process 実行 */
	rv = ap_os_create_privileged_process(r, procnew, command, NULL, env, procattr, p);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to create child process. (code = %d, file = %s)",
				rv, cgi_pr->relativepath);
		return 1;
	}

	/* タイムアウトの設定 */
	apr_pool_note_subprocess(p, procnew, APR_KILL_AFTER_TIMEOUT);
	cgi_pr->cgi_in = procnew->out;
	if (cgi_pr->cgi_in == NULL) {
		ERRLOG0(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Bad file number.(cgi_in is NULL)");
		return 1;
	}
	apr_file_pipe_timeout_set(cgi_pr->cgi_in, r->server->timeout);

	cgi_pr->cgi_out = procnew->in;
	if (cgi_pr->cgi_out == NULL) {
		ERRLOG0(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Bad file number.(cgi_out is NULL)");
		return 1;
	}
	apr_file_pipe_timeout_set(cgi_pr->cgi_out, r->server->timeout);

	cgi_pr->cgi_err = procnew->err;
	if (cgi_pr->cgi_err == NULL) {
		ERRLOG0(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Bad file number.(cgi_err is NULL)");
		return 1;
	}
	apr_file_pipe_timeout_set(cgi_pr->cgi_err, r->server->timeout);

	return 0;
}

/**
 * 起動されるChildプロセス(CGIプロセス)のロガー
 * (note)
 *   主として、CGIの実行そのものに失敗した場合に利用されます.
 */
static void _cgi_loggger(apr_pool_t *p, apr_status_t rv, const char *desc)
{
	ERRLOG2(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Child process return code. (code = %d, desc = %s)", rv, desc);
}

/**
 * bb のデータを読み捨てる.
 */
static void _discard_script_output(apr_bucket_brigade *bb)
{
	apr_bucket *e;
	const char *buf;
	apr_size_t len;
	apr_status_t rv;

	e = APR_BRIGADE_FIRST(bb);
	while (e != APR_BRIGADE_SENTINEL(bb)) {
		if (APR_BUCKET_IS_EOS(e)) {
			break;
		}
		rv = apr_bucket_read(e, &buf, &len, APR_BLOCK_READ);
		if (rv != APR_SUCCESS) {
			break;
		}

		e = APR_BUCKET_NEXT(e);
	}
}

#if APR_FILES_AS_SOCKETS
/**
 * CGIからのstdout, stderr を受けるbucketを生成する.
 */
static apr_bucket * _cgi_bucket_create(request_rec *r, apr_file_t *out,
                                       apr_file_t *err, apr_bucket_alloc_t *list)
{
	apr_pool_t *p = r->pool;
	apr_bucket *b = apr_bucket_alloc(sizeof(apr_bucket), list);
	apr_status_t rv;
	apr_pollfd_t fd;
	divy_cgi_bucket_data *data = apr_palloc(p, sizeof(divy_cgi_bucket_data));

	APR_BUCKET_INIT(b);
	b->free = apr_bucket_free;
	b->list = list;
	b->type = &bucket_type_cgi;
	b->length = (apr_size_t)(-1);
	b->start = -1;

	/* Create the pollset */
	rv = apr_pollset_create(&data->pollset, 2, p, 0);
	AP_DEBUG_ASSERT(rv == APR_SUCCESS);

	fd.desc_type = APR_POLL_FILE;
	fd.reqevents = APR_POLLIN;
	fd.p = p;
	fd.desc.f = out; /* script's stdout */
	fd.client_data = (void *)1;
	rv = apr_pollset_add(data->pollset, &fd);
	AP_DEBUG_ASSERT(rv == APR_SUCCESS);

	fd.desc.f = err; /* script's stderr */
	fd.client_data = (void *)2;
	rv = apr_pollset_add(data->pollset, &fd);
	AP_DEBUG_ASSERT(rv == APR_SUCCESS);
 
	data->r = r;
	b->data = data;

	return b;
}
#endif	/* APR_FILES_AS_SOCKETS */

#if APR_FILES_AS_SOCKETS
/**
 * CGI のbucket を複製する.
 */
static apr_bucket * _cgi_bucket_dup(divy_cgi_bucket_data *data, apr_bucket_alloc_t *list)
{
	apr_bucket *b = apr_bucket_alloc(sizeof(apr_bucket), list);

	APR_BUCKET_INIT(b);
	b->free = apr_bucket_free;
	b->list = list;
	b->type = &bucket_type_cgi;
	b->length = (apr_size_t)(-1);
	b->start = -1;
	b->data = data;

	return b;
}
#endif	/* APR_FILES_AS_SOCKETS */

#if APR_FILES_AS_SOCKETS
/**
 * CGI のstderr からデータを読み込んで出力する.
 * より厳密には読み込んだデータをheap bucket に渡しています.
 *
 */
static apr_status_t _cgi_read_stdout(apr_bucket *a, apr_file_t *out,
                                     const char **str, apr_size_t *len)
{
	char *buf;
	apr_status_t rv;

	*str = NULL;
	*len = APR_BUCKET_BUFF_SIZE;
	buf = apr_bucket_alloc(*len, a->list); /* XXX: check for failure? */

	/* stdout からlen バイトのデータを読み込む */
	rv = apr_file_read(out, buf, len);

	if (rv != APR_SUCCESS && rv != APR_EOF) {
		apr_bucket_free(buf);
		return rv;
	}

	if (*len > 0) {
		divy_cgi_bucket_data *data = a->data;
		apr_bucket_heap *h;

		/* Change the current bucket to refer to what we read */
		a = apr_bucket_heap_make(a, buf, *len, apr_bucket_free);
		h = a->data;
		h->alloc_len = APR_BUCKET_BUFF_SIZE; /* note the real buffer size */
		*str = buf;
		APR_BUCKET_INSERT_AFTER(a, _cgi_bucket_dup(data, a->list));
	}
	else {
		apr_bucket_free(buf);
		a = apr_bucket_immortal_make(a, "", 0);
		*str = a->data;
	}

	return rv;
}
#endif	/* APR_FILES_AS_SOCKETS */

#if APR_FILES_AS_SOCKETS
/**
 * CGI のstdout またはstderr からデータを読み込む.
 *
 */
static apr_status_t _cgi_bucket_read(apr_bucket *b, const char **str,
                                     apr_size_t *len, apr_read_type_e block)
{
	divy_cgi_bucket_data *data = b->data;
	apr_interval_time_t timeout;
	apr_status_t rv;
	int gotdata = 0;

	timeout = block == APR_NONBLOCK_READ ? 0 : data->r->server->timeout;

	do {
		const apr_pollfd_t *results;
		apr_int32_t num;

		/* CGI のstdout, stderr が変化するまで待つ(poll) */
		rv = apr_pollset_poll(data->pollset, timeout, &num, &results);
		if (APR_STATUS_IS_TIMEUP(rv)) {
			return timeout == 0 ? APR_EAGAIN : rv;
		}
		else if (APR_STATUS_IS_EINTR(rv)) {
			continue;
		}
		else if (rv != APR_SUCCESS) {
			ERRLOG0(data->r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to poll CGI child");
			return rv;
		}

		for (; num; num--, results++) {
			if (results[0].client_data == (void *)1) {
				/* CGI のstdout からデータを読んで出力する */
				rv = _cgi_read_stdout(b, results[0].desc.f, str, len);
				if (APR_STATUS_IS_EOF(rv)) {
					rv = APR_SUCCESS;
				}
				gotdata = 1;	/* stdout を読み終えたら終了 */
			}
			else {
				/* CGI のstderr からデータを読んでログに書く */
				apr_status_t rv2 = _cgi_log_error(data->r, results[0].desc.f);
				if (APR_STATUS_IS_EOF(rv2)) {
					apr_pollset_remove(data->pollset, &results[0]);
				}
				/* stderr からの読み込みは終了条件に当たらないので続ける */
			}
		}
	}
	while (!gotdata);	/* stdout を読み込むまで終わらない */

	return rv;
}
#endif	/* APR_FILES_AS_SOCKETS */

#if APR_FILES_AS_SOCKETS
/**
 * CGI のstderr からデータを読み込んでログファイルに出力する.
 */
static apr_status_t _cgi_log_error(request_rec *r, apr_file_t *script_err)
{
	char buf[HUGE_STRING_LEN];
	char *newline;
	apr_status_t rv;

	/*
	 * 1行単位でCGIからのstderr を読み込む
	 * (note)
	 *   関数 apr_file_gets() は'\n' にであうか、指定サイズになるまで読んでくれます
	 */
	while ((rv = apr_file_gets(buf, HUGE_STRING_LEN, script_err)) == APR_SUCCESS) {

		/* apr_file_gets は'\n' を付けてしまうので、それを除去する
		 * (後ろから探したほうが速い) */
		newline = divy_strrchr(buf, '\n');
		if (newline != NULL) {
			*newline = '\0';
		}
		ERRLOG1(r->pool, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC, "[CGI] %s", buf);
	}

	return rv;
}
#endif	/* APR_FILES_AS_SOCKETS */

#else
	/* Nothing to do */

#endif	/* DAV_SUPPORT_POST_HANDLING */

/**
 * r->user の環境変数をセットする.
 *
 * @param r request_rec *
 */
static void _cgi_setup_user_envvars(request_rec *r)
{
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	apr_table_t *e = r->subprocess_env;
	apr_pool_t *p  = r->pool;
	const char *str;
	apr_int64_t len;
	apr_int32_t len32;
	const int *accessdeny = NULL;
	apr_hash_t *grp_pr_h = NULL;
#ifdef DIVY_SUPPORT_EXTENDQUOTA
	divy_rdbo_diskquota squota = { 0 };	/* システムQuota */
#endif
	divy_auth_session session = {0};

	/* ユーザ名 */
	str = divy_get_fullname(r);
	if (IS_FILLED(str)) {
		apr_table_setn(e, DIVY_CGIENV_USERNAME, ap_escape_html(p, str));
	}

	/* ユーザパスワード */
	str = divy_get_password(r);
	if (IS_FILLED(str)) {
		apr_table_setn(e, DIVY_CGIENV_USERPASSWORD, ap_escape_html(p, str));
	}

	/* ユーザ平文パスワード */
	/* NOTE
	 * divy_parse_authorization_header(r)はAuthenticationヘッダをパース
	 * してユーザパスワードを取得するが、POST認証の場合このヘッダはクライアント
	 * から提出されません。
	 * その為に、内部で保持している値を利用するようにします
	 * さらに暗号化されている場合は、ユーザ情報から（divy_get_password(r))から
	 * 取得が出来ません。（ユーザテーブルには暗号化された情報しかないから）
	 * その為、ここで取得しなければならない平文パスワードはdivy_sessionテーブル
	 * から取得するようにしています。
	 * 
	 * また、Authenticationヘッダとセッションは別ヘッダの為に存在するかも
	 * しれません。
	 * - 例 -
	 * Authorization:	Basic cm86MDk4NzY1
	 * Cookie       :   tf-session=ec33ee0bc649ab1e44b4c5bcdb996586
	 * この状態になる場合も可能性としてはあります。Cookieが入っている場合は
	 * Basic認証は無視するようにしています。
	 */
	str = divy_parse_authorization_header(r);
	if (divy_support_session(r) == 0 && IS_FILLED(str)) {
		/* 実体参照に変換してはならない */
		apr_table_setn(e, DIVY_CGIENV_USERPLAINPASSWORD, str);
	}
	else {
		/* ここに来るのはPOST認証の場合のみ */
		session.sid = divy_pcache_get_data(r->pool, DIVY_PCACHE_DAT_SES_SID);
		if (divy_util_auth_get_memcache_userinfo(r, &session) != 0) {
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA, "Failed to password get error. [enc pass=On | memcache = Off]");
		}

		if (conf->encryptpassword == DIVY_ENCRYPTPASSWORD_ON) {
			/* 暗号パスワードの場合 */
			str = apr_pstrdup(p, session.password);
		}
		else {
			/* 暗号化されていない場合 */
			str = divy_get_password(r);
		}
		if (IS_FILLED(str)) {
			/* 実体参照に変換してはならない */
			apr_table_setn(e, DIVY_CGIENV_USERPLAINPASSWORD, str);
		}

		/* トークンを取得 */
		apr_table_setn(e, DIVY_CGIENV_TF_TOKEN, session.uniqueid);
		
		ERRLOG1(r->pool, APLOG_DEBUG, DIVY_FST_IERR + DIVY_SST_DEBUG, "token     =%s", session.uniqueid);
	}

	/* ユーザのメールアドレス */
	str = divy_get_mailaddr(r);
	if (IS_FILLED(str)) {
		apr_table_setn(e, DIVY_CGIENV_USERMAILADDR, ap_escape_html(p, str));
	}

	/* ユーザのコメント */
	str = divy_get_usercomment(r);
	if (IS_FILLED(str)) {
		apr_table_setn(e, DIVY_CGIENV_USERCOMMENT, ap_escape_html(p, str));
	}

	/* ユーザの最大サイズQuota */
	len = divy_get_usermsquota(r);
	apr_table_setn(e, DIVY_CGIENV_USERMSQUOTA, apr_psprintf(p, "%"APR_INT64_T_FMT, len));

	/* ユーザの最大ファイル数Quota */
	len = divy_get_usermfquota(r);
	apr_table_setn(e, DIVY_CGIENV_USERMFQUOTA, apr_psprintf(p, "%"APR_INT64_T_FMT, len));

	/* ユーザの使用サイズQuota */
	len = divy_get_userusquota(r);
	apr_table_setn(e, DIVY_CGIENV_USERUSQUOTA, apr_psprintf(p, "%"APR_INT64_T_FMT, len));

	/* ユーザの使用ファイル数Quota */
	len = divy_get_userufquota(r);
	apr_table_setn(e, DIVY_CGIENV_USERUFQUOTA, apr_psprintf(p, "%"APR_INT64_T_FMT, len));

#ifdef DIVY_SUPPORT_EXTENDQUOTA
	squota.uri = apr_pstrdup(p, dav_divy_get_root_uri(r));
	squota.type = DIVY_QTTYPE_SYSTEM;
	if (divy_rdbo_get_systemquota(r, &squota, 0, NULL) == 0) {
		apr_table_setn(e, DIVY_CGIENV_SYSTEMMSQUOTA, apr_psprintf(p, "%"APR_INT64_T_FMT, squota.maxst));
		apr_table_setn(e, DIVY_CGIENV_SYSTEMUSQUOTA, apr_psprintf(p, "%"APR_INT64_T_FMT, squota.usedst));
	}
#endif
	/* 拡張ユーザステータスの設定 */
	if (divy_support_extenduserstatus(r)) {
		const divy_rdbo_extstatus *extstatus = divy_get_extstatus(r);
		time_t expiration = divy_get_expiration(r);
		char *user_privilege = NULL, *user_state = NULL;

		/* ユーザの種類 */
		if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN) {
			apr_table_setn(e, DIVY_CGIENV_USERTYPE, apr_pstrdup(p, "admin"));
		}
		else {
			if (divy_rdbo_is_groupleader(extstatus)) {
				apr_table_setn(e, DIVY_CGIENV_USERTYPE, apr_pstrdup(p, "groupleader"));
			}
			else if (divy_rdbo_is_trusty_user(extstatus)) {
				apr_table_setn(e, DIVY_CGIENV_USERTYPE, apr_pstrdup(p, "normal"));
			}
			else {
				apr_table_setn(e, DIVY_CGIENV_USERTYPE, apr_pstrdup(p, "limited"));
			}
		}

		/* ユーザのアクセス権限 */
		if (divy_rdbo_has_readwrite_privilege(extstatus)) {
			user_privilege = "readwrite";
		}
		else if (divy_rdbo_has_upload_privilege(extstatus)) {
			user_privilege = "upload";
		}
		else {
			user_privilege = "read";
		}
		if (divy_rdbo_has_setview_privilege(extstatus)) {
			user_privilege = apr_pstrcat(p, user_privilege, ",set-view", NULL);
		}

		if (divy_support_grpconstraints(r) &&
			divy_rdbo_has_user_groupconstraints_ignore(extstatus)) {
			user_privilege = apr_pstrcat(p, user_privilege, ",group-constraints-ignore", NULL);
		}

		if (divy_support_groupleader(r) &&
			divy_rdbo_has_control_otheruser(extstatus)) {
			user_privilege = apr_pstrcat(p, user_privilege, ",control-other-user", NULL);
		}

		apr_table_setn(e, DIVY_CGIENV_USERPRIVILEGE, apr_pstrdup(p, user_privilege));


		/* ユーザの状態 */
		if (divy_rdbo_is_active_user(extstatus)) {
			user_state = "active";
		}
		else {
			user_state = "inactive";
		}

		/* ユーザの有効期限切れ状態 */
		if (expiration != 0 && expiration < apr_time_sec(r->request_time)) {
			user_state = apr_pstrcat(p, user_state, ",expired", NULL);
			apr_table_setn(e, DIVY_CGIENV_USERSTATE, user_state);
		}
		else {
			apr_table_setn(e, DIVY_CGIENV_USERSTATE, apr_pstrdup(p, user_state));
		}
	}

	/* ユーザフォルダを表示できるかどうか */
	accessdeny = divy_get_accessdeny(r);
	if (accessdeny == NULL || accessdeny[DIVY_FOLDER_ID_user] == 0) {
		apr_table_setn(e, DIVY_CGIENV_USERSHOWUSER, apr_pstrdup(p, "on"));
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_USERSHOWUSER, apr_pstrdup(p, "off"));
	}

	/* グループフォルダを表示できるかどうか */
	if (accessdeny == NULL || accessdeny[DIVY_FOLDER_ID_group] == 0) {
		apr_table_setn(e, DIVY_CGIENV_USERSHOWGROUP, apr_pstrdup(p, "on"));
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_USERSHOWGROUP, apr_pstrdup(p, "off"));
	}

	/* オーナID */
	str = divy_get_userownerid(r);
	if (IS_FILLED(str)) {
		apr_table_setn(e, DIVY_CGIENV_USEROWNERID, ap_escape_html(p, str));
	}


	/* 最大管理ユーザ数 */
	len32 = divy_get_usermaxusercreation(r);
	apr_table_setn(e, DIVY_CGIENV_USERMAXUSERCREATION, apr_psprintf(p, "%d", len32));

	/* 所属グループ一覧 */
	(void) divy_get_cached_availablegroup(r, &grp_pr_h);
	if (grp_pr_h != NULL)
	{
		apr_hash_index_t *hidx;
		const char *key;
		divy_sbuf *sbuf = NULL;
		int i=0;

		/* バッファの作成 */
		divy_sbuf_create(p, &sbuf, 512);
		for (hidx = apr_hash_first(p, grp_pr_h); hidx; hidx = apr_hash_next(hidx)) {

			apr_hash_this(hidx, (const void**)&key, NULL, NULL);
			if (i>0) divy_sbuf_appendbyte(sbuf, 1, ",");
			divy_sbuf_append(sbuf, key);

			i++;
		}

		apr_table_setn(e, DIVY_CGIENV_USERAVAILABLEGROUPURI, divy_sbuf_tostring(sbuf));

	}
}

/**
 * 拡張子マッピングの環境変数をセットする.
 *
 * @param r request_rec *
 * @param lang const char * 言語環境を表す変数
 */
static void _cgi_setup_extmap_envvars(request_rec *r, const char *lang)
{
	apr_table_t *e = r->subprocess_env;
	apr_pool_t *p  = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char *str;
	apr_hash_t *map;

	/* デフォルト拡張子マッピングファイルのパス */
	apr_table_setn(e, DIVY_CGIENV_DEFAULT_EXTMAPPATH,
					divy_build_default_extmappath(p, dconf->stylesheetroot, lang));

	/* 拡張子マッピングファイルのパス */
	apr_table_setn(e, DIVY_CGIENV_EXTMAPPATH,
					divy_build_extmappath(p, dconf->stylesheetroot, lang));


	/* 拡張子マッピングの取得 */
	map = divy_read_mimetype_map(r);
	if (map != NULL) {
		/* プライベートコレクションの表示名 */
		str = divy_get_mimetype_dispname(p, DIVY_CONTTYPE_PRIVCOL, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_USER, ap_escape_html(p, str));
		}

		/* グループコレクションの表示名 */
		str = divy_get_mimetype_dispname(p, DIVY_CONTTYPE_GRPCOLFOLDER, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_GROUP, ap_escape_html(p, str));
		}

		/* クライアント更新の表示名 */
		str = divy_get_mimetype_dispname(p, DIVY_CONTTYPE_UPDATEDCL, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_M_UPDATE, ap_escape_html(p, str));
		}

		/* システムメッセージの表示名 */
		str = divy_get_mimetype_dispname(p, DIVY_CONTTYPE_SYSMSGFOLDER, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_M_SYSMSG, ap_escape_html(p, str));
		}

		/* ユーザ(管理者フォルダ)の表示名 */
		str = divy_get_mimetype_dispname(p, DIVY_CONTTYPE_MUSERFOLDER, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_M_USER, ap_escape_html(p, str));
		}

		/* メール編集の表示名 */
		str = divy_get_mimetype_dispname(p, DIVY_CONTTYPE_EDITMAIL, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_EDITMAIL, ap_escape_html(p, str));
		}

		/* ログ取得機能の表示名 */
#ifdef DIVY_SUPPORT_LOGACCESS
		str =  divy_get_mimetype_dispname(p, DIVY_CONTTYPE_LOGACCESS, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_LOGACCESS, ap_escape_html(p, str));
		}
#endif	/* DIVY_SUPPORT_LOGACCESS */

		/* 自動削除機能の表示名 */
		str =  divy_get_mimetype_dispname(p, DIVY_CONTTYPE_AUTODELETE, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_AUTODELETE, ap_escape_html(p, str));
		}

		/* 操作ログ機能の表示名 */
		str =  divy_get_mimetype_dispname(p, DIVY_CONTTYPE_LOGOPERATION, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_LOGOPERATION, ap_escape_html(p, str));
		}

		/* レポート設定の表示名 */
		str =  divy_get_mimetype_dispname(p, DIVY_CONTTYPE_REPORTSETTINGS, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_REPORTSETTINGS, ap_escape_html(p, str));
		}

		/* ライセンスアップローダ機能の表示名 */
		str =  divy_get_mimetype_dispname(p, DIVY_CONTTYPE_LICENSEUPLOADER, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_LICENSEUPLOADER, ap_escape_html(p, str));
		}

		/* グループ(in 管理者) の表示名 */
		str =  divy_get_mimetype_dispname(p, DIVY_CONTTYPE_MGRPFOLDER, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_M_GROUP, ap_escape_html(p, str));
		}

		/* プライベートコレクションのイメージパス */
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_PRIVCOL, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_USER_IMG, ap_escape_html(p, str));
		}

		/* グループコレクションのイメージパス */
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_GRPCOLFOLDER, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_GROUP_IMG, ap_escape_html(p, str));
		}

		/* クライアント更新のイメージパス */
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_UPDATEDCL, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_M_UPDATE_IMG, ap_escape_html(p, str));
		}

		/* システムメッセージのイメージパス */
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_SYSMSGFOLDER, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_M_SYSMSG_IMG, ap_escape_html(p, str));
		}

		/* ユーザ(管理者フォルダ)のイメージパス */
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_MUSERFOLDER, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_M_USER_IMG, ap_escape_html(p, str));
		}

		/* メール編集のイメージパス */
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_EDITMAIL, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_EDITMAIL_IMG, ap_escape_html(p, str));
		}

		/* トップページロゴのイメージパス
		 * ロケーション単位のロゴがあるか調べてあれば使う
		 * なければ従来通りファイルを調べて設定
		 */
		str = divy_get_custom_image_path(r, "logo.png");
		if (IS_EMPTY(str)) {
			str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_TOPPAGE, map);
		}
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_TOPPAGE, ap_escape_html(p, str));
		}

		/* ログ取得機能のイメージパス */
#ifdef DIVY_SUPPORT_LOGACCESS
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_LOGACCESS, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_LOGACCESS_IMG, ap_escape_html(p, str));
		}
#endif	/* DIVY_SUPPORT_LOGACCESS */

		/* 自動削除機能のイメージパス */
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_AUTODELETE, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_AUTODELETE_IMG, ap_escape_html(p, str));
		}

		/* レポート設定のイメージパス */
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_REPORTSETTINGS, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_REPORTSETTINGS_IMG, ap_escape_html(p, str));
		}

		/* グループ(in 管理者) のイメージパス */
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_MGRPFOLDER, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_M_GROUP_IMG, ap_escape_html(p, str));
		}

		/* 操作ログ機能のイメージパス */
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_LOGOPERATION, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_LOGOPERATION_IMG, ap_escape_html(p, str));
		}

		/* ライセンスアップローダ機能のイメージパス */
		str = divy_get_mimetype_imgpath(p, DIVY_CONTTYPE_LICENSEUPLOADER, map);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_FOLDER_LICENSEUPLOADER_IMG, ap_escape_html(p, str));
		}
	}
}

#ifdef DIVY_SUPPORT_PLUGIN
/**
 * プラグイン用の環境変数をセットする.
 *
 * @param r request_rec *
 * @param lang const char * 言語環境を表す変数
 * @param cgi_pr divy_cgi_property * CGIプロパティ
 */
static void _cgi_setup_plugin_envvars(request_rec *r, const char *lang, divy_cgi_property *cgi_pr)
{
	apr_table_t *e = r->subprocess_env;
	apr_pool_t *p  = r->pool;
	const char *str;
	char *plugin_name     = NULL;
	tfs_varray_t *m_items = NULL;

	/* プラグインルートパス */
	str = divy_pi_get_pluginroot(r);
	if (IS_FILLED(str)) {
		apr_table_setn(e, DIVY_CGIENV_PLUGINROOT, ap_escape_html(p, str));
	}

	/* プラグイン名称 */
	if (cgi_pr != NULL) {
		(void) divy_extract_plugin_name(p, cgi_pr->u_spec, (char **)&plugin_name);
		if (IS_FILLED(plugin_name)) {
			apr_table_setn(e, DIVY_CGIENV_PLUGINNAME, ap_escape_html(p, plugin_name));
		}
	}

	if (IS_FILLED(plugin_name)) {
		/* プラグインインストールパス */
		str = divy_pi_build_plugin_path(r, plugin_name);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_PLUGINPATH, ap_escape_html(p, str));
		}

		/* プラグインコンフィグパス */
		str = divy_pi_build_plugin_confpath(r, plugin_name);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_PLUGINCONFPATH, ap_escape_html(p, str));
		}
	}

	/* 拡張メニューの取得 */
	if (divy_pi_get_locationmenu_items(r, &m_items) == DIVY_PL_ST_OK && m_items != NULL) {
		int size = tfs_varray_size(m_items), i;
		char *envname, *envvar, *displayname;
		divy_lmenu_item *item;

		apr_table_setn(e, DIVY_CGIENV_EXTENDEDSPEC_N, apr_psprintf(p, "%d", size));
		for (i = 0; i < size; i++) {
			/* キーの生成 */
			envname = apr_psprintf(p, "%s%d", DIVY_CGIENV_EXTENDEDSPEC_PREFIX, i);

			item = tfs_varray_get(m_items, i);
			if (IS_EMPTY(lang) || strcmp(lang, "ja") == 0 || IS_EMPTY(item->displayname_en)) {
				displayname = item->displayname;
			}
			else {
				displayname = item->displayname_en;
			}
			/* 値の生成 */
			envvar = apr_psprintf(p, "%s,%s,%s,%s,%s",
					item->screentype, ap_escape_uri(p, item->uri),
					ap_escape_html(p, item->iconpath), ap_escape_html(p, displayname),
					ap_escape_html(p, item->usertype));

			/* セットする */
			apr_table_setn(e, envname, envvar);
		}
	}
}
#endif	/* DIVY_SUPPORT_PLUGIN */

/**
 * リバースプロキシ対応用の環境変数をセットする.
 *
 * @param r request_rec *
 */
static void _cgi_setup_rproxy_envvars(request_rec *r)
{
	apr_table_t *e = r->subprocess_env;
	apr_pool_t *p  = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char *str;

	/* リバースプロキシ対応の有無 */
	if (dconf->rproxyuse == DIVY_RPROXYUSE_ON) {
		apr_table_setn(e, DIVY_CGIENV_RPROXY_USE, "on");
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_RPROXY_USE, "off");
	}

	/* TfRProxyMatch の値 */
	if (IS_FILLED(dconf->rproxymatch)) {
		apr_table_setn(e, DIVY_CGIENV_RPROXY_MATCH, ap_escape_html(p, dconf->rproxymatch));
	}

	/* TfRProxySchema の値 */
	str = "";
	if (dconf->rproxyschema == DIVY_RPROXY_SCHEMA_HTTP) {
		str = "http";
	}
	else if (dconf->rproxyschema == DIVY_RPROXY_SCHEMA_HTTPS) {
		str = "https";
	}
	else {
		/* auto または未設定の場合にはリクエストベースのスキーマを使用する */
		int port = -1;
		char *schema = NULL;

		(void) divy_get_request_conninfo(r, &schema, &port);
		if (IS_FILLED(schema) && strcasecmp(schema, "http") == 0) {
			str = "http";
		}
		else if (IS_FILLED(schema) && strcasecmp(schema, "https") == 0) {
			str = "https";
		}
	}
	apr_table_setn(e, DIVY_CGIENV_RPROXY_SCHEMA, str);

	/* TfRProxyHttpPort の値 */
	if (dconf->rproxyhttpport != DIVY_INT32_C_UNSET) {
		apr_table_setn(e, DIVY_CGIENV_RPROXY_HTTP_PORT, apr_psprintf(p, "%u", dconf->rproxyhttpport));
	}

	/* TfRProxyHttpsPort の値 */
	if (dconf->rproxyhttpsport != DIVY_INT32_C_UNSET) {
		apr_table_setn(e, DIVY_CGIENV_RPROXY_HTTPS_PORT, apr_psprintf(p, "%u", dconf->rproxyhttpsport));
	}
}

/**
 * クライアントに示すNotifyUrl関連の環境変数をセットする.
 *
 * @param r request_rec *
 */
static void _cgi_setup_notifyurl_envvars(request_rec *r)
{
	apr_table_t *e = r->subprocess_env;
	apr_pool_t *p  = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char *str;
	int cl_port = -1;
	char *cl_schema = NULL;

	/* TfNotifyServerName の値 */
	if (IS_FILLED(dconf->notifyservername)) {
		str = ap_escape_html(p, dconf->notifyservername);
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVERNAME, str);	/* (note) 互換性維持のために残してあります */
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVER_NAME, str);
	}

	/* クライアントがリクエストしてきたポート番号とスキーマ名を取得する */
	(void) divy_get_request_conninfo(r, &cl_schema, &cl_port);

	str = "";
	/* TfNotifyServerPort の値 */
	if (dconf->notifyserverport == DIVY_NOTIFYSERVER_PORT_AUTO ||
		dconf->notifyserverport == DIVY_INT32_C_UNSET) {
		str = apr_psprintf(p, "%u", cl_port);	/* 自動検出したポート番号を使う */
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVER_PORT_ORIG, "auto");	/* そのまま入れる */
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVER_PORT, str);
	}
	else {
		str = apr_psprintf(p, "%u", dconf->notifyserverport);
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVER_PORT_ORIG, str);
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVER_PORT, str);
	}

	/* TfNotifyServerSchema の値 */
	str = "";
	if (dconf->notifyserverschema == DIVY_NOTIFYSERVER_SCHEMA_HTTP) {
		str = "http";
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVER_SCHEMA, str);
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVER_SCHEMA_ORIG, str);
	}
	else if (dconf->notifyserverschema == DIVY_NOTIFYSERVER_SCHEMA_HTTPS) {
		str = "https";
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVER_SCHEMA, str);
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVER_SCHEMA_ORIG, str);
	}
	else {
		/* auto または未設定の場合にはリクエストベースのスキーマを使用する */
		str = cl_schema;
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVER_SCHEMA, str);
		apr_table_setn(e, DIVY_CGIENV_NOTIFYSERVER_SCHEMA_ORIG, "auto");	/* そのまま入れる */
	}
}

/**
 * システムCGI内部接続用の環境変数をセットする.
 *
 * @param r request_rec *
 */
static void _cgi_setup_cgiconinfo_envvars(request_rec *r)
{
	apr_table_t *e = r->subprocess_env;
	apr_pool_t *p  = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char *str;
	int cl_port = -1;
	char *cl_schema = NULL;

	/* クライアントがリクエストしてきたポート番号とスキーマ名を取得する */
	(void) divy_get_request_conninfo(r, &cl_schema, &cl_port);

	str = "";
	/* TfSysCgiConnectPort の値 */
	if (dconf->syscgiconnectport == DIVY_SYSCGI_CONNECT_PORT_AUTO ||
		dconf->syscgiconnectport == DIVY_INT32_C_UNSET) {
		str = apr_psprintf(p, "%u", cl_port);	/* 自動検出したポート番号を使う */
	}
	else {
		str = apr_psprintf(p, "%u", dconf->syscgiconnectport);
	}
	apr_table_setn(e, DIVY_CGIENV_SYSCGI_CONNECT_PORT, str);

	/* TfSysCgiConnectSchema の値 */
	str = "";
	if (dconf->syscgiconnectschema == DIVY_SYSCGI_CONNECT_SCHEMA_HTTP) {
		str = "http";
	}
	else if (dconf->syscgiconnectschema == DIVY_SYSCGI_CONNECT_SCHEMA_HTTPS) {
		str = "https";
	}
	else {
		/* auto または未設定の場合にはリクエストベースのスキーマを使用する */
		str = cl_schema;
	}
	apr_table_setn(e, DIVY_CGIENV_SYSCGI_CONNECT_SCHEMA, str);
}

/**
 * 外部クライアントがURLに打ち込んだと推測されるURL情報の環境変数をセットする.
 *
 * @param r request_rec *
 */
static void _cgi_setup_remoteclient_envvars(request_rec *r)
{
	apr_table_t *e = r->subprocess_env;
	apr_pool_t *p  = r->pool;
	divy_buildurl_segment url_segment = { 0 };

	/* クライアントが接続したURLの推測を行う
	 * (note)
	 *     リバースプロキシなどが中間にあれば推測が正しくないかもしれない */
	(void) divy_construct_url2(p, DIVY_URLTYPE_PROTECTED, "", r, &url_segment);

	if (IS_FILLED(url_segment.schema)) {
		apr_table_setn(e, DIVY_CGIENV_CLIENT_SCHEMA, url_segment.schema);
	}

	if (IS_FILLED(url_segment.host)) {
		apr_table_setn(e, DIVY_CGIENV_CLIENT_HOST, ap_escape_html(p, url_segment.host));
	}

	if (url_segment.port > 0) {
		apr_table_setn(e, DIVY_CGIENV_CLIENT_PORT, apr_psprintf(p, "%u", url_segment.port));
	}
}


