/**
 * $Id$
 *
 * 開封通知周りの操作を行う関数、構造体の定義
 *
 */
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_hash.h"
#include "apr_time.h"
#include "apr_network_io.h"
#include "httpd.h"

#include "tf_folder.h"
#include "util_common.h"
#include "tf_confirmreading.h"
#include "tf_validator.h"
#include "tf_rdbo.h"

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare and Define structure
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define array
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * 開封通知機能をサポートしているかどうか.
 *
 */
DIVY_DECLARE(int) divy_support_confirmreading(request_rec *r)
{
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

	/* 開封通知機能が有効なときだけサポート */
	if (dconf->confirmreading == DIVY_CONFIRMREADING_ON) {
		return 1;
	}

	return 0;
}

/**
 * 開封通知の有効期限(日)を取得する.
 * 
 */
DIVY_DECLARE(int) divy_get_confirmreading_reservedday(request_rec *r)
{
	int day = DIVY_CONFIRMREADING_DEFAULT_RESERVEDDAY;	/* デフォルトは無制限 */

	if (divy_support_confirmreading(r)) {
		dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

		if (dconf->cr_reservedday != DIVY_INT32_C_UNSET) {
			day = dconf->cr_reservedday;
		}
	}

	return day;
}

/**
 * 指定されたu_spec, r が示すリソースが開封通知の自動解除用リクエストを受けるのに
 * 適切かどうか.
 *
 */
DIVY_DECLARE(int) divy_confirmreading_enable(request_rec *r, divy_uri_spec *u_spec)
{
	if (r == NULL || u_spec == NULL) {
		return 0;
	}

	/* POST オペレーションをサポートしていなければ常に有効ではない */
#ifndef DAV_SUPPORT_POST_HANDLING
	return 0;
#endif	/* DAV_SUPPORT_POST_HANDLING */

	/*
	 * [ 有効となる条件 ]
	 *
	 *  * 開封通知機能をサポートしていること (ディレクティブ)
	 *
	 *  * infotype が DIVY_INFOTYPE_confirmreading
	 *  * POST リクエストであること
	 *  * 127.0.0.1 アドレスへのリクエストであること
	 */
	if (divy_support_confirmreading(r) &&
		u_spec->infotype == DIVY_INFOTYPE_confirmreading &&
		divy_get_method_number(r) == M_POST) {
		apr_pool_t *p = r->pool;
		apr_status_t rv;
		apr_ipsubnet_t *ipsub = NULL;

		rv = apr_ipsubnet_create(&ipsub, "127.0.0.1", "255.255.255.255", p);
		if (rv != APR_SUCCESS || ipsub == NULL ||
			!apr_ipsubnet_test(ipsub, r->useragent_addr)) {
			return -1;	/* 不正アクセス */
		}
		return 1;	/* サポートしている */
	}

	return 0;	/* 未サポート */
}

/**
 * resource が示すリクエストに基づき、開封通知の自動解除を実行する.
 *
 * @param r request_rec *
 * @param resource const dav_resource *
 * @return dav_error *
 */
DIVY_DECLARE(dav_error *) divy_confirmreading_release(request_rec *r, const dav_resource *resource)
{
	apr_pool_t *p = r->pool;
	int day;

	if (resource == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"resource is EMPTY.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* 保持日数を取得 */
	day = divy_get_confirmreading_reservedday(r);

	/*
	 * 保持日数が無制限であれば何もしない
	 */
	if (day == DIVY_CONFIRMREADING_INFINITY_RESERVEDDAY) {
		return NULL;	/* 正常終了したとする */
	}

	/*
	 * 開封通知の削除
	 */
	if (divy_rdbo_release_expired_confirmreading(r, day, NULL)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to release confirmreading.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}

/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/

