/**
 * $Id$
 *
 * TeamFile アクション拡張コマンド呼び出し用関数群の定義
 *
 */
#include "httpd.h"
#include "util_script.h"
#include "apr.h"
#include "apr_file_info.h"
#include "apr_strings.h"
#include "apr_poll.h"
#include "apr_buckets.h"
#include "apr_tables.h"
#include "ap_mpm.h"

#include "tf_extcmd.h"
#include "tf_env.h"
#include "tf_logger.h"
#include "tf_folder.h"
#include "mod_dav_tf.h"
#include "tf_rdbo.h"
#include "tf_rdbo_group.h"
#include "util.h"
#include "tf_storage.h"

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
typedef struct divy_extcmd_property	divy_extcmd_property;
struct divy_extcmd_property {
	const char *path;	/* フルパス */
	const char *ppath;	/* 親パス   */

	/* 作業用のプール */
	apr_pool_t *p;
};

typedef struct divy_extcmd_bucket_data	divy_extcmd_bucket_data;
struct divy_extcmd_bucket_data {
	apr_pollset_t *pollset;
	request_rec *r;
};

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static void _setup_file_envvars(request_rec *r, const divy_extcmd_resource *res);
static int _create_extcmd_property(request_rec *r, divy_extcmd_property **extcmd_pr);
static int _run_extcmd(request_rec *r, divy_extcmd_property *extcmd_pr, int *exitcode);
static int _create_procattr(request_rec *r, divy_extcmd_property *extcmd_pr, apr_procattr_t **proc);
static void _extcmd_loggger(apr_pool_t *p, apr_status_t rv, const char *desc);
static apr_bucket * _extcmd_bucket_create(request_rec *r, apr_file_t *err, apr_bucket_alloc_t *list);
static apr_status_t _extcmd_bucket_read(apr_bucket *b, const char **str,
                                        apr_size_t *len, apr_read_type_e block);
static apr_status_t _extcmd_log(request_rec *r, apr_file_t *script_err);

/*------------------------------------------------------------------------------
  Define static values
  ----------------------------------------------------------------------------*/
#if APR_FILES_AS_SOCKETS
/**
 * アクション拡張コマンドのstdoutとstderr を読み込むbucket の定義(生成)
 * read だけ定義できれば十分.
 */
static const apr_bucket_type_t bucket_type_extcmd = {
	"TF_EXTCMD", 5, APR_BUCKET_DATA,
	apr_bucket_destroy_noop,		/* 未実装 */
	_extcmd_bucket_read,
	apr_bucket_setaside_notimpl,	/* 未実装 */
	apr_bucket_split_notimpl,		/* 未実装 */
	apr_bucket_copy_notimpl			/* 未実装 */
};
#endif	/* APR_FILES_AS_SOCKETS */

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * アクション拡張コマンド機構をサポートしているかどうか.
 *
 */
DIVY_DECLARE(int) divy_extcmd_support(request_rec *r)
{
	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	/* ライセンスがあって、ON かつコマンドがあればサポートされている */
	if (sconf && dconf && sconf->use_extcmd_opt &&
		dconf->execengine == DIVY_EXECENGINE_ON && IS_FILLED(dconf->execoncmd)) {
		return 1;
	}
	return 0;
}

/**
 * アクション拡張コマンドを実行するユーザID
 *
 */
DIVY_DECLARE(int) divy_extcmd_match(request_rec *r)
{

	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);

	ap_regex_t *preg;
	int		ret = 0;	/* マッチしない */

	/* 設定されていない場合は実行可能である */
	if (IS_EMPTY(dconf->execmatch)) return 1;

	preg = ap_pregcomp(r->pool, dconf->execmatch, (AP_REG_EXTENDED | AP_REG_NOSUB));
	/* preg の NULLチェックは起動時に行っている為しない */

	/* パターンを調べる */
	ret = !ap_regexec(preg, divy_get_userid(r), 0, NULL, 0);

	ap_pregfree(r->pool, preg);

	return ret;
}


/**
 * res が示すファイルに対するアクション拡張コマンドを実行する.
 *
 */
DIVY_DECLARE(int) divy_extcmd_execute(request_rec *r,
						const divy_extcmd_resource *res, int *exitcode)
{
	apr_pool_t *p = r->pool;
	divy_extcmd_property *extcmd_pr = NULL;

	*exitcode = DIVY_EXTCMD_BITNONE; /* 初期化 */

	if (res == NULL) return 0;

	/* プロパティの生成 */
	if (_create_extcmd_property(r, &extcmd_pr)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to create action-command property.");
		return 1;
	}

	/* コマンドライン実行環境の設定 */
	divy_cgi_setup_lc_envvars(r, NULL);
	
	/* ファイルに関する環境変数の設定 */
	_setup_file_envvars(r, res);

	/* アクション拡張コマンドを実行 */
	if (_run_extcmd(r, extcmd_pr, exitcode)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to execute action-command.");
		return 1;
	}

	return 0;
}


/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * res が示すリソース用環境変数を設定する.
 *
 * @param r request_rec *
 * @param res const divy_extcmd_resource *
 */
static void _setup_file_envvars(request_rec *r, const divy_extcmd_resource *res)
{
	apr_table_t *e           = r->subprocess_env;
	apr_pool_t *p            = r->pool;
	char *str;
	int ret;

	/* 物理パス */
	str = (char *) divy_pfile_get_fullpath(res->pfile);
	if (IS_FILLED(str)) {
		apr_table_setn(e, DIVY_CGIENV_F_PHYSICALPATH, ap_escape_html(p, str));
	}

	/* ファイル名 */
	if (IS_FILLED(res->displayname)) {
		apr_table_setn(e, DIVY_CGIENV_F_NAME, ap_escape_html(p, res->displayname));
	}

	/* URI */
	if (IS_FILLED(res->uri)) {
		apr_table_setn(e, DIVY_CGIENV_F_URI, ap_escape_html(p, res->uri));
	}

	/* URL */
	if (IS_FILLED(res->uri)) {
		/* (note) 公開URLを渡します(限定公開URLではありません) */
		str = divy_construct_url2(p, DIVY_URLTYPE_PUBLIC, res->uri, r, NULL);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_F_URL, ap_escape_html(p, str));
		}
	}

	/* サイズ */
	apr_table_setn(e, DIVY_CGIENV_F_SIZE, apr_psprintf(p, "%"APR_INT64_T_FMT, res->flen));

	/* 種類 */
	if (IS_FILLED(res->getcontenttype)) {
		apr_table_setn(e, DIVY_CGIENV_F_TYPE, ap_escape_html(p, res->getcontenttype));
	}

	/* ファイルの最終更新日時 */
	if (res->getlastmodified > 0L) {
		divy_format_time_t(p, res->getlastmodified, DIVY_TIME_STYLE_ISO8601, &str);
		if (IS_FILLED(str)) {
			apr_table_setn(e, DIVY_CGIENV_F_LASTMODIFIED, ap_escape_html(p, str));
		}
	}

	/* 新規かどうか */
	if (IS_EMPTY(res->creator_userid)) {
		apr_table_setn(e, DIVY_CGIENV_F_ISNEW, apr_pstrdup(p, "1"));
	}
	else {
		apr_table_setn(e, DIVY_CGIENV_F_ISNEW, apr_pstrdup(p, "0"));
	}

	/* フォルダの種類 */
	if (res->u_spec != NULL) {
		/* ユーザフォルダ以下の場合 */
		if (res->u_spec->infotype == DIVY_INFOTYPE_user_e_regular ||
			res->u_spec->infotype == DIVY_INFOTYPE_user_trash_e0  ||
			res->u_spec->infotype == DIVY_INFOTYPE_user_trash_ex) {

			apr_table_setn(e, DIVY_CGIENV_F_FOLDERTYPE, apr_pstrdup(p, "user"));
		}
		/* グループフォルダ以下の場合 */
		else if (res->u_spec->infotype == DIVY_INFOTYPE_group_e_regular ||
				 res->u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
				 res->u_spec->infotype == DIVY_INFOTYPE_group_trash_ex){

			apr_table_setn(e, DIVY_CGIENV_F_FOLDERTYPE, apr_pstrdup(p, "group"));
		}
		/* その他のケースでは不明とする */
		else {
			apr_table_setn(e, DIVY_CGIENV_F_FOLDERTYPE, apr_pstrdup(p, "unknown"));
		}
	}

	/* グループフォルダ名 */
	if (res->u_spec != NULL &&
		(res->u_spec->infotype == DIVY_INFOTYPE_group_e_regular ||
		 res->u_spec->infotype == DIVY_INFOTYPE_group_trash_e0  ||
		 res->u_spec->infotype == DIVY_INFOTYPE_group_trash_ex)) {

		/* リソースURIから所属グループを検索する */
		divy_rdbo_grp *grp_pr = NULL;
		ret = divy_rdbo_get_group_property_by_resuri(r, res->uri, &grp_pr);
		if (ret == 0 && grp_pr != NULL) {
			apr_table_setn(e, DIVY_CGIENV_F_GRPNAME, ap_escape_html(p, grp_pr->name));
		}
	}
}

/**
 * アクション拡張コマンド用プロパティを生成する
 */
static int _create_extcmd_property(request_rec *r, divy_extcmd_property **extcmd_pr)
{
	apr_pool_t *p            = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	char *cmd;

	*extcmd_pr = NULL;	/* 初期化 */

	/* コマンドラインパラメータのチェック */
	if (IS_EMPTY(dconf->execoncmd)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to get action-execute command");
		return 1;
	}

	*extcmd_pr = apr_pcalloc(p, sizeof(divy_extcmd_property));
	cmd = dav_divy_remove_endslash(p, dconf->execoncmd);	/* 末尾の/ を除去 */
	ap_no2slash(cmd);	/* 2つ以上のスラッシュをシングルにする */

	(*extcmd_pr)->path  = cmd;
	(*extcmd_pr)->ppath = ap_make_dirstr_parent(p, cmd);
	(*extcmd_pr)->p     = p;

	return 0;
}

/**
 * アクション拡張コマンドを起動する.
 * (note)
 *  * stdout, stdin は親プロセスとは接続しません
 *  * stderr は親プロセスのストリームと接続するのでその準備を行います(実接続は行わない)
 *    ただし、APR_FILES_AS_SOCKETS が定義されていた場合のみ
 */
static int _run_extcmd(request_rec *r, divy_extcmd_property *extcmd_pr, int *exitcode)
{
	apr_status_t rv;
	apr_procattr_t *procattr  = NULL;
	const char * const *env   = NULL;
	const char * const * argv = NULL;
	apr_proc_t *procnew       = NULL;
	apr_pool_t *p             = r->pool;
	int ret                   = 0;
	apr_exit_why_e exitwhy;
#if APR_FILES_AS_SOCKETS
	apr_bucket_brigade *bb;
	apr_size_t len;
	conn_rec *c   = r->connection;
	apr_bucket *b;
	const char *buf;
#endif	/* APR_FILES_AS_SOCKETS */

	/* Child プロセスに渡す環境変数テーブルの生成 */
	env = (const char * const *)ap_create_environment(p, r->subprocess_env);

	/* Child プロセスのattribute を生成 */
	if (_create_procattr(r, extcmd_pr, &procattr)) {
		return 1;
	}

	/* プロセス構造体初期化 */
	procnew = apr_pcalloc(p, sizeof(apr_proc_t));

	/* 引数のパース */
	argv = (const char * const *) divy_parse_commandline(p, extcmd_pr->path);

	/* process 実行 */
	rv = apr_proc_create(procnew, extcmd_pr->path, argv, env, procattr, p);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to create child process. (code = %d, file = %s)",
				rv, extcmd_pr->path);
		return 1;
	}

#if APR_FILES_AS_SOCKETS
	/* タイムアウトの設定とChildプロセスのストリームを記録 */
	if (procnew->err == NULL) {
		ERRLOG0(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Bad file number. (procnew->err is NULL)");
		ret = 1;
		goto waitfor_child;
	}
	apr_file_pipe_timeout_set(procnew->err, r->server->timeout);

	/*
	 * ストリームの取得
	 * * Child プロセスへのストリーム入力(in),出力(out) は無し.
	 * * エラー出力ストリームをログに向ける
	 *
	 * pipe は既に接続済みです(see _run_extcmd)
	 */

	/* アクションコマンドからのstderr を取得する */
	bb = apr_brigade_create(p, c->bucket_alloc);
	apr_file_pipe_timeout_set(procnew->err, 0);

	b = _extcmd_bucket_create(r, procnew->err, c->bucket_alloc);
	APR_BRIGADE_INSERT_TAIL(bb, b);

	/* 終了bucket を挿入する */
	b = apr_bucket_eos_create(c->bucket_alloc);
	APR_BRIGADE_INSERT_TAIL(bb, b);

	/* データを読む (ここのbuf は利用しない) */
	b = APR_BRIGADE_FIRST(bb);
	while (b != APR_BRIGADE_SENTINEL(bb)) {
		if (APR_BUCKET_IS_EOS(b)) {
			break;
		}
		rv = apr_bucket_read(b, &buf, &len, APR_BLOCK_READ);
		if (rv != APR_SUCCESS) {
			break;
		}

		b = APR_BUCKET_NEXT(b);
	}
	apr_file_close(procnew->err);
#endif

waitfor_child:
	/*
	 * Child プロセスの終了を待つ
	 */
	rv = apr_proc_wait(procnew, exitcode, &exitwhy, APR_WAIT);
	if (rv != APR_CHILD_DONE){
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to wait for child process. (code = %d, exitwhy = %d)", rv, exitwhy);
		return 1;
	} 

	return ret;
}


/**
 * Child プロセスのattribute を生成する.
 *
 */
static int _create_procattr(request_rec *r, divy_extcmd_property *extcmd_pr, apr_procattr_t **proc)
{
	apr_status_t rv;
	apr_pool_t *p = r->pool;

	*proc = NULL;	/* 初期化 */

	/* process の環境を作成 */
	if ((rv = apr_procattr_create(proc, p)) != APR_SUCCESS) {
		ERRLOG1(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to create process attribute.(code = %d)", rv);
		return 1;
	}

	/* 作成するChildのパイプに親プロセスのstdin, stdout, stderr をどう繋ぐか */
	if ((rv = apr_procattr_io_set(*proc, APR_PARENT_BLOCK, APR_PARENT_BLOCK,
#if APR_FILES_AS_SOCKETS
								APR_CHILD_BLOCK
#else
								APR_PARENT_BLOCK
#endif
								  )) != APR_SUCCESS) {
		ERRLOG1(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to set child io_set.(code = %d)", rv);
		return 1;
	}

	/* CGIが動作するカレントディレクトリをppath に移動する */
	if ((rv = apr_procattr_dir_set(*proc, extcmd_pr->ppath)) != APR_SUCCESS) {
		ERRLOG2(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to change directory. (code = %d, dir = %s)", rv, extcmd_pr->ppath);
		return 1;
	}

	/* コマンドタイプ設定 */
	if ((rv = apr_procattr_cmdtype_set(*proc, APR_PROGRAM)) != APR_SUCCESS) {
		ERRLOG2(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to set cmdtype.(code = %d, cmdtype = %d)", rv, APR_PROGRAM);
		return 1;
	}

	/* プログラムをデタッチ状態では生成しない */
	if ((rv = apr_procattr_detach_set(*proc, 0)) != APR_SUCCESS) {
		ERRLOG1(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to set detach mode.(code = %d, detach = 0)", rv);
		return 1;
	}

	/* Childプロセスのエラーを出力するように設定 */
	if ((rv = apr_procattr_child_errfn_set(*proc, _extcmd_loggger)) != APR_SUCCESS) {
		ERRLOG1(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to set error function. (code = %d)", rv);
		return 1;
	}

	return 0;
}

/**
 * 起動されるChildプロセスのロガー
 * (note)
 *   主として、Childプロセスの起動そのものに失敗した場合に利用されます.
 */
static void _extcmd_loggger(apr_pool_t *p, apr_status_t rv, const char *desc)
{
	ERRLOG2(p, LOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"The action-command process return code. (code = %d, desc = %s)", rv, desc);
}

#if APR_FILES_AS_SOCKETS
/**
 * 起動プロセスからの stderr を受けるbucketを生成する.
 */
static apr_bucket * _extcmd_bucket_create(request_rec *r, apr_file_t *err, apr_bucket_alloc_t *list)
{
	apr_pool_t *p = r->pool;
	apr_bucket *b = apr_bucket_alloc(sizeof(apr_bucket), list);
	apr_status_t rv;
	apr_pollfd_t fd;
	divy_extcmd_bucket_data *data = apr_palloc(p, sizeof(divy_extcmd_bucket_data));

	APR_BUCKET_INIT(b);
	b->free = apr_bucket_free;
	b->list = list;
	b->type = &bucket_type_extcmd;
	b->length = (apr_size_t)(-1);
	b->start = -1;

	/* Create the pollset */
	rv = apr_pollset_create(&data->pollset, 1, p, 0);
	AP_DEBUG_ASSERT(rv == APR_SUCCESS);

	fd.desc_type = APR_POLL_FILE;
	fd.reqevents = APR_POLLIN;
	fd.p = p;

	fd.desc.f = err;
	fd.client_data = NULL;
	rv = apr_pollset_add(data->pollset, &fd);
 
	data->r = r;
	b->data = data;

	return b;
}
#endif	/* APR_FILES_AS_SOCKETS */

#if APR_FILES_AS_SOCKETS
/**
 * アクション拡張コマンドのstdout またはstderr からデータを読み込む.
 *
 */
static apr_status_t _extcmd_bucket_read(apr_bucket *b, const char **str,
                                        apr_size_t *len, apr_read_type_e block)
{
	divy_extcmd_bucket_data *data = b->data;
	apr_interval_time_t timeout;
	apr_status_t rv;
	int gotdata = 0;

	timeout = block == APR_NONBLOCK_READ ? 0 : data->r->server->timeout;

	do {
		const apr_pollfd_t *results;
		apr_int32_t num;

		/* stderr が変化するまで待つ(poll) */
		rv = apr_pollset_poll(data->pollset, timeout, &num, &results);
		if (APR_STATUS_IS_TIMEUP(rv)) {
			return timeout == 0 ? APR_EAGAIN : rv;
		}
		else if (APR_STATUS_IS_EINTR(rv)) {
			continue;
		}
		else if (rv != APR_SUCCESS) {
			ERRLOG0(data->r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to poll child process.");
			return rv;
		}

		/* stderr からデータを読んでログに書く */
		rv = _extcmd_log(data->r, results[0].desc.f);
		if (APR_STATUS_IS_EOF(rv)) {
			apr_pollset_remove(data->pollset, &results[0]);
			rv = APR_SUCCESS;

			gotdata = 1;	/* stderr を読み終えたら終了 */
		}
	}
	while (!gotdata);	/* stdout を読み込むまで終わらない */

	return rv;
}
#endif	/* APR_FILES_AS_SOCKETS */

#if APR_FILES_AS_SOCKETS
/**
 * アクション拡張コマンドのstderr からデータを読み込んでログファイルに出力する.
 */
static apr_status_t _extcmd_log(request_rec *r, apr_file_t *script_err)
{
	char buf[HUGE_STRING_LEN];
	char *newline;
	apr_status_t rv;

	/*
	 * 1行単位でstderr を読み込む
	 * (note)
	 *   関数 apr_file_gets() は'\n' にであうか、指定サイズになるまで読んでくれます
	 */
	while ((rv = apr_file_gets(buf, HUGE_STRING_LEN, script_err)) == APR_SUCCESS) {

		/* apr_file_gets は'\n' を付けてしまうので、それを除去する
		 * (後ろから探したほうが速い) */
		newline = divy_strrchr(buf, '\n');
		if (newline != NULL) {
			*newline = '\0';
		}
		ERRLOG1(r->pool, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC, "[EXTCMD] %s", buf);
	}

	return rv;
}
#endif	/* APR_FILES_AS_SOCKETS */


