/**
 * $Id$
 *
 * 拡張子マッピングを扱うための関数の定義、データの定義などを集めたファイル
 *
 */
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_hash.h"
#include "apr_time.h"
#include "apr_file_info.h"
#include "apr_file_io.h"

#include "httpd.h"

#include "mod_dav_tf.h"
#include "util_common.h"
#include "tf_extmap.h"
#include "tf_valuecache.h"
#include "util.h"

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare and Define structure
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static void _read_mimetype_mapfile(apr_pool_t *cp, request_rec *r,
                                  const char *path, apr_hash_t **map_h);

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * Content-Type、拡張子マップを読み込む.
 *
 */
DIVY_DECLARE(apr_hash_t *) divy_read_mimetype_map(request_rec *r)
{
	apr_status_t rv1, rv2;
	apr_hash_t *map          = NULL;
	apr_pool_t *p            = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char *lang         = divy_get_language_param(r);
	apr_pool_t *cp           = r->server->process->pool;
	apr_finfo_t finfo1       = { 0 }, finfo2 = { 0 };
	char *path1, *path2;
	divy_cmap *cmap;
	apr_time_t new_timestamp;

	/* キャッシュからMapを取得してみる */
	cmap = divy_pcache_vget_data(cp, DIVY_PCACHE_DAT_GL_CMAP, lang, NULL);

	/* ファイルパスの組み立て */
	path1 = divy_build_default_extmappath(p, dconf->stylesheetroot, lang);
	path2 = divy_build_extmappath(p, dconf->stylesheetroot, lang);

	/* ファイルのタイムスタンプを取得する */
	rv1 = apr_stat(&finfo1, path1, APR_FINFO_MTIME, p);
	if (rv1 == APR_ENOENT) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"The default Content-Type map file was missing. "
				"(filepath = %s)", path1);
	}
	else if (rv1 != APR_SUCCESS) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to open Content-Type map file. "
				"(filepath = %s, code = %d)", path1, rv1);
	}

	/* ユーザ定義のマッピングファイルは無いことがある */
	rv2 = apr_stat(&finfo2, path2, APR_FINFO_MTIME, p);
	if (rv2 != APR_ENOENT && rv2 != APR_SUCCESS) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to open Content-Type map file. "
				"(filepath = %s, code = %d)", path2, rv2);
	}

	if (rv1 == APR_SUCCESS && rv2 == APR_SUCCESS) {
		new_timestamp = (finfo1.mtime > finfo2.mtime) ? finfo1.mtime : finfo2.mtime;
	}
	else if (rv1 == APR_SUCCESS && rv2 != APR_SUCCESS) {
		new_timestamp = finfo1.mtime;
	}
	else if (rv1 != APR_SUCCESS && rv2 == APR_SUCCESS) {
		new_timestamp = finfo2.mtime;
	}
	else {
		new_timestamp = APR_TIME_C(0);	/* ファイルが読めなかった */

		/* キャッシュもない */
		if (cmap == NULL) {
			return NULL;	/* ファイルも無いし、キャッシュもないので何もできない */
		}
	}

	/* キャッシュがない or ファイルが更新されていたら読み込む */
	if (cmap == NULL || cmap->timestamp < new_timestamp) {

		map = NULL;	/* 初期化 */
		if (rv1 == APR_SUCCESS) {
			_read_mimetype_mapfile(cp, r, path1, &map);
		}
		if (rv2 == APR_SUCCESS) {
			_read_mimetype_mapfile(cp, r, path2, &map);
		}

		if (map != NULL) {
			/* マップに記録する */
			cmap = apr_pcalloc(cp, sizeof(divy_cmap));
			cmap->map_h = map;
			cmap->timestamp = new_timestamp;

			/* キャッシュに登録 */
			divy_pcache_vset_data(cp, cmap, DIVY_PCACHE_DAT_GL_CMAP, apr_pstrdup(cp, lang), NULL);
		}
	}
	else if (cmap->timestamp >= new_timestamp) {
		map = cmap->map_h;	/* キャッシュ値を利用する */
	}

	return map;
}

/**
 * Content-Type 値 ctype または 拡張子ext に一致する拡張子マッピング情報を
 * 取得する.
 *
 */
DIVY_DECLARE(const divy_cmap_elem *) divy_get_extmapinfo(apr_pool_t *p, apr_hash_t *map_h,
														const char *ctype, const char *ext)
{
	if (map_h == NULL || (IS_EMPTY(ctype) && IS_EMPTY(ext))) {
		return NULL;
	}

	if (IS_FILLED(ctype)) {
		return apr_hash_get(map_h, ctype, APR_HASH_KEY_STRING);
	}

	if (IS_FILLED(ext)) {
		return apr_hash_get(map_h, ext, APR_HASH_KEY_STRING);
	}

	return NULL;
}

/**
 * Content-Type値 ctype から対応する表示名称を取得する.
 *
 */
DIVY_DECLARE(char *) divy_get_mimetype_dispname(apr_pool_t *p,
                                                const char *ctype, apr_hash_t *map)
{
	divy_cmap_elem *elem;

	if (map == NULL || IS_EMPTY(ctype)) {
		return NULL;
	}

	elem = apr_hash_get(map, ctype, APR_HASH_KEY_STRING);
	if (elem == NULL || IS_EMPTY(elem->ctype_name)) {
		return NULL;
	}

	return apr_pstrdup(p, elem->ctype_name);	/* コピーして渡す */
}

/**
 * Content-Type値 ctype から対応するイメージパスを取得する.
 *
 */
DIVY_DECLARE(char *) divy_get_mimetype_imgpath(apr_pool_t *p,
                                               const char *ctype, apr_hash_t *map)
{
	divy_cmap_elem *elem;

	if (map == NULL || IS_EMPTY(ctype)) {
		return NULL;
	}

	elem = apr_hash_get(map, ctype, APR_HASH_KEY_STRING);
	if (elem == NULL || IS_EMPTY(elem->img_path)) {
		return NULL;
	}

	return apr_pstrdup(p, elem->img_path);	/* コピーして渡す */

}

/**
 * デフォルト拡張子マッピングファイルのパスを取得する. (ユーティリティ関数)
 * 
 */
DIVY_DECLARE(char *) divy_build_default_extmappath(apr_pool_t *p, const char *stylesheetroot, const char *lang)
{
	if (p == NULL || IS_EMPTY(stylesheetroot) || IS_EMPTY(lang)) {
		return NULL;
	}

	return ap_make_full_path(p, stylesheetroot, apr_psprintf(p, "%s.%s", DIVY_MIMEMAP_FNAME_DEFAULT, lang));
}

/**
 * ユーザが書き換えた拡張子マッピングファイルのパスを取得する. (ユーティリティ関数)
 * 
 */
DIVY_DECLARE(char *) divy_build_extmappath(apr_pool_t *p, const char *stylesheetroot, const char *lang)
{
	if (p == NULL || IS_EMPTY(stylesheetroot) || IS_EMPTY(lang)) {
		return NULL;
	}

	return ap_make_full_path(p, stylesheetroot, apr_psprintf(p, "%s.%s", DIVY_MIMEMAP_FNAME, lang));
}

/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * path のファイルからContent-Type,拡張子マッピングを読み込み *map_h に
 * 格納して返却する.
 *
 * @param cp apr_pool_t * キャッシュ用のプール
 * @param r request_rec *
 * @param path const char * ファイルパス
 * @param map_h apr_hash_t **
 */
static void _read_mimetype_mapfile(apr_pool_t *cp, request_rec *r,
                                  const char *path, apr_hash_t **map_h)
{
	apr_status_t rv;
	apr_file_t *fd = NULL;
	char *buf, *token, *token_cntx = NULL, *str;
	int ret = 0, i;
	divy_cmap_elem *elem = NULL;
	apr_pool_t *p = r->pool;

	/* ファイルを開く */
	rv = apr_file_open(&fd, path, APR_READ | APR_BINARY, 0, p);
	if (rv != APR_SUCCESS) {
		if (rv == APR_ENOENT) {
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
					"Content-Type map file was missing. (filepath = %s)", path);
		}
		else {
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
					"Failed to open Content-Type map file. (filepath = %s)", path);
		}
		return;
	}

	/* Content-Type、拡張子マッピングファイルを読む */
	while ((ret = divy_file_readline(p, fd, &buf)) != DIVY_UTIL_COMMON_ERR) {
		if (IS_EMPTY(buf)) {
			if (ret == DIVY_UTIL_COMMON_EOF) break;
			continue;	/* 空行は読み飛ばし */
		}

		/* コメント行は読み飛ばし */
		if (buf[0] == '#') continue;

		if (*map_h == NULL) {
			*map_h = apr_hash_make(cp);
		}

		/* "," で分解する */
		i = 0;
		elem = NULL;
		while ((token = apr_strtok(buf, ",", &token_cntx)) != NULL) {
			buf = NULL;
			/* Content-Type(NULLあり), 拡張子(NULLあり),イメージパス,種類名 */
			if (token != NULL && *token != '-') {
				str = apr_pstrdup(cp, token);
			}
			else {
				str = NULL;
			}

			switch (i) {
				case 0:
					elem = apr_pcalloc(cp, sizeof(divy_cmap_elem));
					elem->ctype = str;
					break;
				case 1:
					elem->ext = str;
					break;
				case 2:
					elem->img_path = str;
					break;
				case 3:
					elem->ctype_name = str;
					break;
			}
			i++;
		}

		if (elem != NULL) {

			/* Content-Typeで登録する */
			if (IS_FILLED(elem->ctype)) {
				apr_hash_set(*map_h, elem->ctype, APR_HASH_KEY_STRING, elem);
			}

			/* 拡張子で登録する */
			if (IS_FILLED(elem->ext)) {
				divy_cmap_elem *elem2;

				str = elem->ext;
				while ((token = apr_strtok(str, " ", &token_cntx)) != NULL) {
					str = NULL;
					elem2 = apr_pcalloc(cp, sizeof(divy_cmap_elem));
					elem2->ctype      = elem->ctype;
					elem2->ext        = apr_pstrdup(cp, token);
					elem2->img_path   = elem->img_path;
					elem2->ctype_name = elem->ctype_name;

					apr_hash_set(*map_h, elem2->ext, APR_HASH_KEY_STRING, elem2);
				}
			}
		}

		/* ファイルの終了まで読んだ場合 */
		if (ret == DIVY_UTIL_COMMON_EOF) break;
	}

	/* 読み込みエラー発生 */
	if (ret == DIVY_UTIL_COMMON_ERR) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to read Content-Type map file.(code = %d)", ret);
		goto close_mimemap;
	}

close_mimemap:
	if (fd != NULL) {
		apr_file_close(fd);
	}
}

