/**
 * $Id$
 *
 * tf_folder.c
 *
 * "フォルダ"、"URI"を管理する構造体、配列、関数定義するファイル。
 *
 * (note)
 * 	このヘッダにはTeamFileモジュールおよびApache本体固有のヘッダをinclude
 * 	してはなりません。(mod_davやhttpd.hなど)
 * 	includeしてよいものは、apr_xxx とTeamFile"モジュール"に依存しない共通
 * 	ヘッダ(tfr.hなど)だけです。
 *
 * 2004/10/14 Thu takehara NEW
 */
#include "apr.h"
#include "apr_pools.h"
#include "apr_hash.h"
#include "apr_strings.h"
#include "apr_base64.h"

#if APR_HAVE_STRING_H
#include <string.h>
#endif	/* APR_HAVE_STRING_H */

#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif  /* APR_HAVE_STDLIB_H */

#include "util_common.h"
#include "tf_folder.h"
#include "tf_valuecache.h"

/*------------------------------------------------------------------------------
  Define static values and macro
  ----------------------------------------------------------------------------*/

/**
 * _match_special_folder が使用する戻り値
 */
#define DIVY_RET_NOTMATCH	0
#define DIVY_RET_FULLMATCH	1
#define DIVY_RET_PARENTMATCH	2
#define DIVY_RET_PARTMATCH	3

/**
 * リレーションプレフィックスの集合を表す配列
 */
static const char * relation_prefix_set[] = {
	DIVY_PREFIX_RUSER, DIVY_PREFIX_RGROUP, DIVY_PREFIX_RSQL
};
#define RELATION_PREFIX_SET_LEN	sizeof(relation_prefix_set)/sizeof(char *)

/* 64進数を表す配列 */
static unsigned const char _shortentbl[] = {'W','m','C','5','u','P','t','3','B','8','X','i','R','F','s','d','G','7','g','w','O','c','6','n','y','I','j','J','q','U','L','0','V','b','r','4','Z','v','M','e','D','z','h','Q','f','2','o','S','l','E','K','9','p','x','A','T','1','a','H','k','N','Y','-','='}; /* 重複不可 */
#define SHORTEN_TBL_LEN sizeof(_shortentbl)

/**
 * URIを組み立てる (divy_build_xxxx_uri 関数群で使用する)
 */
#define DIVY_BUILD_GENERAL_URI(p,ruri,fid,segment)	\
	(dav_divy_make_uri(p, ruri,	\
		_special_folders[fid].relativeuri, \
		segment, NULL))

/*------------------------------------------------------------------------------
  Declare private prototype function
  ----------------------------------------------------------------------------*/
static void _no2slash(char *str);
static apr_status_t _pattern_cleanup(void *data);
static ap_regex_t * _pattern_pregcomp(apr_pool_t *p, const char *pattern);
static int _pattern_regexec(ap_regex_t *preg, const char *str);
static int _match_special_folder(const char *str, int tbl_len,
				const divy_special_folderid_id *folder_ids,
				divy_special_folderid_id *fid);
static divy_uri_spec * _clone_uri_spec(apr_pool_t *p, divy_uri_spec *u_spec);
static char * _extract_groupcol_uri(apr_pool_t *p, divy_uri_spec *u_spec);

/*------------------------------------------------------------------------------
  Define array
  ----------------------------------------------------------------------------*/
/**
 * divy_special_folder_spec 構造体の配列
 * (note)
 * * auth の中で、
 *   "privatefolder" の長さ < "management/USER" の長さ
 *   "Group Folder" の長さ < "management/GROUP" の長さ
 *    という前提のもと処理を書いている箇所があります。該当フォルダの
 *    名称を変更したければ、auth も変更しなければなりません。
 *
 * * この配列の要素位置は、列挙型divy_special_folderid_id の値と一致
 *   させています。変更するなら両方変えて下さい。
 *
 * * relativeuri の値を変えるときには、_top_folders及び_second_folders 配列
 *   の並び順も変えてください。また、クライアント側のコードも変更する必要が
 *   あるでしょう。(一回決めたものは変えないことがベストです。)
 */
static divy_special_folder_spec _special_folders[] =
{
	{ DIVY_FOLDER_ID_user,		DIVY_INFOTYPE_user_e,
	  "User Folder",		"",
	  DIVY_FOLDER_TYPE_user,	1,
	  DIVY_SHOW_ALLLIST,		"USER", 0
	},
	{ DIVY_FOLDER_ID_group,		DIVY_INFOTYPE_group,
	  "Group Folder",		"/Group Folder",
	  DIVY_FOLDER_TYPE_group,	2,
	  DIVY_SHOW_ALLLIST,		"GROUP", 0
	},
	{ DIVY_FOLDER_ID_dbfolder,	DIVY_INFOTYPE_dblink,
	  "",				"/.dbfolder",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
	{ DIVY_FOLDER_ID_dbshfolder,	DIVY_INFOTYPE_reposdblink,
	  "",				"/.dbshfolder",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
	{ DIVY_FOLDER_ID_management,	DIVY_INFOTYPE_m_management,
	  "Management",			"/.management",
	  DIVY_FOLDER_TYPE_manage,	5,
	  DIVY_SHOW_ALLLIST,		"MANAGEMENT", 0
	},
	{ DIVY_FOLDER_ID_m_update,	DIVY_INFOTYPE_m_update,
	  "UPDATE",			"/.management/UPDATE",
	  DIVY_FOLDER_TYPE_manage_elm,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_MANAGEMENT,		"M_UPDATE", 0
	},
	{ DIVY_FOLDER_ID_m_sql,		DIVY_INFOTYPE_m_sql,
	  "SQL",			"/.management/SQL",
	  DIVY_FOLDER_TYPE_manage_elm,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_MANAGEMENT,		"M_SQL", 0
	},
	{ DIVY_FOLDER_ID_m_group,	DIVY_INFOTYPE_m_group,
	  "GROUP",			"/.management/GROUP",
	  DIVY_FOLDER_TYPE_manage_elm,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_MANAGEMENT,		"M_GROUP", 0
	},
	{ DIVY_FOLDER_ID_m_user,	DIVY_INFOTYPE_m_user,
	  "USER",			"/.management/USER",
	  DIVY_FOLDER_TYPE_manage_elm,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_MANAGEMENT,		"M_USER", 0
	},
	{ DIVY_FOLDER_ID_m_status,	DIVY_INFOTYPE_m_status,
	  "STATUS",			"/.management/STATUS",
	  DIVY_FOLDER_TYPE_manage_elm,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_MANAGEMENT,		"M_STATUS", 0
	},
	{ DIVY_FOLDER_ID_m_msg,		DIVY_INFOTYPE_m_msg,
	  "MSG",			"/.management/MSG",
	  DIVY_FOLDER_TYPE_manage_elm,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_MANAGEMENT,		"M_MSG", 0
	},
	{ DIVY_FOLDER_ID_m_dbms,	DIVY_INFOTYPE_m_dbms,
	  "DBMS",			"/.management/DBMS",
	  DIVY_FOLDER_TYPE_manage_elm,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_MANAGEMENT,		"M_DBMS", 0
	},
	{ DIVY_FOLDER_ID_m_execsql,	DIVY_INFOTYPE_m_execsql,
	  "EXECSQL",			"/.management/EXECSQL",
	  DIVY_FOLDER_TYPE_manage_elm,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_MANAGEMENT,		"M_EXECSQL", 0
	},
	{ DIVY_FOLDER_ID_userinfo,	DIVY_INFOTYPE_userinfo,
	  "userinfo",			"/userinfo",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN, "", 0
	},
	{ DIVY_FOLDER_ID_dblink,	DIVY_INFOTYPE_dblink,
	  "Database",			"/.dblink",
	  DIVY_FOLDER_TYPE_dblink,	3,
	  DIVY_SHOW_ALLLIST | DIVY_SHOW_DB, "DBLINK", 0
	},
	{ DIVY_FOLDER_ID_reposdblink,	DIVY_INFOTYPE_reposdblink,
	  "RepositoryDB",		"/.reposdblink",
	  DIVY_FOLDER_TYPE_reposdblink,	4,
	  DIVY_SHOW_ALLLIST,		"REPOSDBLINK", 0
	},
	{ DIVY_FOLDER_ID_mail,		DIVY_INFOTYPE_mail,
	  "Mail",			"/.mail",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
	{ DIVY_FOLDER_ID_mobile,	DIVY_INFOTYPE_mobile,
	  "Mobile",			"/.mobile",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
	{ DIVY_FOLDER_ID_login,     DIVY_INFOTYPE_login,
	  "Login",          "/.login",
	  DIVY_FOLDER_TYPE_UNDEFINE,    DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,     "", 0
	},
	{ DIVY_FOLDER_ID_shorten,	DIVY_INFOTYPE_shorten,
	  "Shorten",		"/.st",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
	{ DIVY_FOLDER_ID_saml,		DIVY_INFOTYPE_saml,
	  "SAML",			"/.saml",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
	{ DIVY_FOLDER_ID_utility,	DIVY_INFOTYPE_utility,
	  "Utility Folder",		"/Utility_Folder",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
	{ DIVY_FOLDER_ID_u_update,	DIVY_INFOTYPE_u_update,
	  "Updated Client",		"/Utility_Folder/Updated_Client",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
	{ DIVY_FOLDER_ID_u_msg,		DIVY_INFOTYPE_u_msg,
	  "System Message",		"/Utility_Folder/System_Message",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
	{ DIVY_FOLDER_ID_ticket,	DIVY_INFOTYPE_ticket,
	  "Ticket",			"/.ticket",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
	{ DIVY_FOLDER_ID_fileviewer,	DIVY_INFOTYPE_fileviewer,
	  "FileViewer",			"/.fileviewer",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
#ifdef DAV_SUPPORT_POST_HANDLING
	{ DIVY_FOLDER_ID_cgibin,	DIVY_INFOTYPE_cgibin,
	  "cgi-bin",		"/"DIVY_SYSCGIFOLDER_NAME,
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
#endif	/* DAV_SUPPORT_POST_HANDLING */
#ifdef DIVY_SUPPORT_PLUGIN
	{ DIVY_FOLDER_ID_plugin,	DIVY_INFOTYPE_plugin,
	  "plugin",			"/"DIVY_PLUGINFOLDER_NAME,
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
#endif	/* DIVY_SUPPORT_PLUGIN */
	{ DIVY_FOLDER_ID_autodelete,	DIVY_INFOTYPE_autodelete,
	  "AutoDelete",			"/.autodelete",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
	{ DIVY_FOLDER_ID_confirmreading,	DIVY_INFOTYPE_confirmreading,
	  "ConfirmReading",			"/.confirmreading",
	  DIVY_FOLDER_TYPE_UNDEFINE,	DIVY_DISPLAYORDER_UNDEFINE,
	  DIVY_SHOW_HIDDEN,		"", 0
	},
};

/* 配列_special_folders の要素数 */
#define DIVY_SPECIAL_FOLDERS_LEN sizeof(_special_folders)/sizeof(divy_special_folder_spec)

/**
 * divy_uri_pattern の配列
 * (note)
 * 	* この配列は外から参照することも可能ですが、極力避けてください。そもそも
 * 	  ヘッダファイルに置くこと自体、避けたかったのですが、以下の理由で仕方なく
 * 	  置いてあります。
 *
 * 	** _special_folders がこのヘッダにあるので、URI文字列パターンを
 * 	   複数のファイルに書きたくなかった。
 * 	** ヘッダにあるのはsearch が_special_folders を直接参照するため
 *
 * 	* 以下のURIパターンはfid が一致する要素を上から順にパターンマッチング
 * 	  されることを前提とした正規表現になっています。マッチングロジックの
 * 	  実装にはこのことをよく理解した上で使用して下さい。
 */
static divy_uri_pattern _uri_regex_patterns[] =
{
	{ "^\\/userinfo$",
	  DIVY_INFOTYPE_userinfo,		DIVY_FOLDER_ID_userinfo,
	  DIVY_URITYPE_ARBITER,			DIVY_INFOTYPE_root,	NULL
	},
	/* グループ */
	{ "^\\/Group[ ]Folder$",
	  DIVY_INFOTYPE_group,			DIVY_FOLDER_ID_group,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/Group[ ]Folder\\/[^\\/]+\\/"DIVY_TRASHFOLDER_NAME"$",
	  DIVY_INFOTYPE_group_trash,		DIVY_FOLDER_ID_group,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_group_e,	NULL
	},
	{ "^\\/Group[ ]Folder\\/[^\\/]+\\/"DIVY_TRASHFOLDER_NAME"\\/[^\\/]+$",
	  DIVY_INFOTYPE_group_trash_e0,		DIVY_FOLDER_ID_group,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_group_trash,	NULL
	},
	{ "^\\/Group[ ]Folder\\/[^\\/]+\\/"DIVY_TRASHFOLDER_NAME"\\/[^\\/]+\\/[^\\/]+$",
	  DIVY_INFOTYPE_group_trash_ex,		DIVY_FOLDER_ID_group,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_group_trash_e0,	NULL
	},
	{ "^\\/Group[ ]Folder\\/[^\\/]+\\/"DIVY_TRASHFOLDER_NAME"\\/[^\\/]+\\/[^\\/]+\\/",
	  DIVY_INFOTYPE_group_trash_ex,		DIVY_FOLDER_ID_group,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_group_trash_ex,	NULL
	},
	{ "^\\/Group[ ]Folder\\/[^\\/]+$",
	  DIVY_INFOTYPE_group_e,		DIVY_FOLDER_ID_group,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_group,	NULL
	},
	{ "^\\/Group[ ]Folder\\/[^\\/]+\\/[^\\/]+$",	/* グループフォルダ直下 */
	  DIVY_INFOTYPE_group_e_regular,	DIVY_FOLDER_ID_group,
	  DIVY_URITYPE_REGULAR,			DIVY_INFOTYPE_group_e,	NULL
	},
	{ "^\\/Group[ ]Folder\\/[^\\/]+\\/[^\\/]+\\/",	/* グループフォルダ直下よりも下 */
	  DIVY_INFOTYPE_group_e_regular,	DIVY_FOLDER_ID_group,
	  DIVY_URITYPE_REGULAR,			DIVY_INFOTYPE_group_e_regular,	NULL
	},
	{ "^\\/\\.dblink$",
	  DIVY_INFOTYPE_dblink,			DIVY_FOLDER_ID_dblink,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\.dblink\\/[0-9]+$",
	  DIVY_INFOTYPE_dblink_e,		DIVY_FOLDER_ID_dblink,
	  DIVY_URITYPE_VIRTUAL,			DIVY_INFOTYPE_dblink,	NULL
	},
	{ "^\\/\\.dblink\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_dblink,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.reposdblink$",
	  DIVY_INFOTYPE_reposdblink,		DIVY_FOLDER_ID_reposdblink,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\.reposdblink\\/[0-9]+$",
	  DIVY_INFOTYPE_reposdblink_e,		DIVY_FOLDER_ID_reposdblink,
	  DIVY_URITYPE_VIRTUAL,			DIVY_INFOTYPE_reposdblink,	NULL
	},
	{ "^\\/\\.reposdblink\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_reposdblink,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/.dbfolder$",
	  DIVY_INFOTYPE_dbfolder,		DIVY_FOLDER_ID_dbfolder,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\.dbfolder\\/[0-9a-zA-Z]+$",
	  DIVY_INFOTYPE_dbfolder_e,		DIVY_FOLDER_ID_dbfolder,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_dbfolder,		NULL
	},
	{ "^\\/\\.dbfolder\\/[0-9a-zA-Z]+\\/[^\\/]+$",	/* linkdbsearch結果フォルダ直下 */
	  DIVY_INFOTYPE_dbfolder_e_regular,	DIVY_FOLDER_ID_dbfolder,
	  DIVY_URITYPE_REGULAR,			DIVY_INFOTYPE_dbfolder_e, NULL
	},
	{ "^\\/\\.dbfolder\\/[0-9a-zA-Z]+\\/[^\\/]+\\/",/* linkdbsearch結果フォルダ直下よりも下 */
	  DIVY_INFOTYPE_dbfolder_e_regular,	DIVY_FOLDER_ID_dbfolder,
	  DIVY_URITYPE_REGULAR,			DIVY_INFOTYPE_dbfolder_e_regular, NULL
	},
	{ "^\\/\\.dbfolder\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_dbfolder,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.dbshfolder$",
	  DIVY_INFOTYPE_dbshfolder,		DIVY_FOLDER_ID_dbshfolder,
	  DIVY_URITYPE_VIRTUAL,			DIVY_INFOTYPE_root,		NULL
	},
	{ "^\\/\\.dbshfolder\\/[^\\/]+$",
	  DIVY_INFOTYPE_dbshfolder_e,		DIVY_FOLDER_ID_dbshfolder,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_dbshfolder,	NULL
	},
	{ "^\\/\\.dbshfolder\\/[^\\/]+\\/[^\\/]+$",	/* 共有コレクション直下 */
	  DIVY_INFOTYPE_dbshfolder_e_regular,	DIVY_FOLDER_ID_dbshfolder,
	  DIVY_URITYPE_REGULAR,			DIVY_INFOTYPE_dbshfolder_e,	NULL
	},
	{ "^\\/\\.dbshfolder\\/[^\\/]+\\/[^\\/]+\\/",	/* 共有コレクション直下よりも下 */
	  DIVY_INFOTYPE_dbshfolder_e_regular,	DIVY_FOLDER_ID_dbshfolder,
	  DIVY_URITYPE_REGULAR,			DIVY_INFOTYPE_dbshfolder_e_regular,	NULL
	},
	/* 管理者フォルダ */
	{ "^\\/\\.management$",
	  DIVY_INFOTYPE_m_management,		DIVY_FOLDER_ID_management,
	  DIVY_URITYPE_VIRTUAL,			DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\.management\\/UPDATE$",
	  DIVY_INFOTYPE_m_update,		DIVY_FOLDER_ID_m_update,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_m_management,	NULL
	},
	{ "^\\/\\.management\\/UPDATE\\/[^\\/]+$",
	  DIVY_INFOTYPE_m_update_e,		DIVY_FOLDER_ID_m_update,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_m_update,	NULL
	},
	{ "^\\/\\.management\\/UPDATE\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_update,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/USER$",
	  DIVY_INFOTYPE_m_user,			DIVY_FOLDER_ID_m_user,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_m_management,	NULL
	},
	{ "^\\/\\.management\\/USER\\/[^\\/]+\\/\\"DIVY_PREFIX_RGROUP"[0-9]+$",
	  DIVY_INFOTYPE_m_user_rgrp,		DIVY_FOLDER_ID_m_user,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_m_user_e,	NULL
	},
	{ "^\\/\\.management\\/USER\\/\\"DIVY_PREFIX_RGROUP,
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_user,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/USER\\/[^\\/]+$",
	  DIVY_INFOTYPE_m_user_e,		DIVY_FOLDER_ID_m_user,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_m_user,	NULL
	},
	{ "^\\/\\.management\\/USER\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_user,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/GROUP$",
	  DIVY_INFOTYPE_m_group,		DIVY_FOLDER_ID_m_group,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_m_management,	NULL
	},
	{ "^\\/\\.management\\/GROUP\\/\\"DIVY_PREFIX_RUSER,
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_group,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/GROUP\\/\\"DIVY_PREFIX_RSQL,
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_group,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/GROUP\\/.*/\\"DIVY_PREFIX_RUSER"[^\\/]+$",
	  DIVY_INFOTYPE_m_group_ruser,		DIVY_FOLDER_ID_m_group,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_m_group_e,	NULL
	},
	{ "^\\/\\.management\\/GROUP\\/.*/\\"DIVY_PREFIX_RSQL"[^\\/]+$",
	  DIVY_INFOTYPE_m_group_rsql,		DIVY_FOLDER_ID_m_group,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_m_group_e,	NULL
	},
	{ "^\\/\\.management\\/GROUP\\/.*/\\"DIVY_PREFIX_RUSER"[^\\/]+\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_group,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/GROUP\\/.*/\\"DIVY_PREFIX_RSQL"[^\\/]+\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_group,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/GROUP\\/",
	  DIVY_INFOTYPE_m_group_e,		DIVY_FOLDER_ID_m_group,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_m_management,	NULL
	},
	{ "^\\/\\.management\\/SQL$",
	  DIVY_INFOTYPE_m_sql,			DIVY_FOLDER_ID_m_sql,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_m_management,	NULL
	},
	{ "^\\/\\.management\\/SQL\\/[^\\/]+\\/\\"DIVY_PREFIX_RGROUP"[0-9]+$",
	  DIVY_INFOTYPE_m_sql_rgrp,		DIVY_FOLDER_ID_m_sql,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_m_sql_e,	NULL
	},
	{ "^\\/\\.management\\/SQL\\/\\"DIVY_PREFIX_RGROUP,
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_sql,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/SQL\\/[^\\/]+$",
	  DIVY_INFOTYPE_m_sql_e,		DIVY_FOLDER_ID_m_sql,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_m_sql,	NULL
	},
	{ "^\\/\\.management\\/SQL\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_sql,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/STATUS$",
	  DIVY_INFOTYPE_m_status,		DIVY_FOLDER_ID_m_status,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_m_management,	NULL
	},
	{ "^\\/\\.management\\/STATUS\\/[^\\/]+$",
	  DIVY_INFOTYPE_m_status_e,		DIVY_FOLDER_ID_m_status,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_m_status,	NULL
	},
	{ "^\\/\\.management\\/STATUS\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_status,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/MSG$",
	  DIVY_INFOTYPE_m_msg,			DIVY_FOLDER_ID_m_msg,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_m_management,	NULL
	},
	{ "^\\/\\.management\\/MSG\\/[0-9]+$",
	  DIVY_INFOTYPE_m_msg_e,		DIVY_FOLDER_ID_m_msg,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_m_msg,	NULL
	},
	{ "^\\/\\.management\\/MSG\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_msg,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/DBMS$",
	  DIVY_INFOTYPE_m_dbms,			DIVY_FOLDER_ID_m_dbms,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_m_management,	NULL
	},
	{ "^\\/\\.management\\/DBMS\\/[^\\/]+$",
	  DIVY_INFOTYPE_m_dbms_e,		DIVY_FOLDER_ID_m_dbms,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_m_dbms,	NULL
	},
	{ "^\\/\\.management\\/DBMS\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_dbms,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/EXECSQL$",
	  DIVY_INFOTYPE_m_execsql,		DIVY_FOLDER_ID_m_execsql,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_m_management,	NULL
	},
	{ "^\\/\\.management\\/EXECSQL\\/[^\\/]+$",
	  DIVY_INFOTYPE_m_execsql_e,		DIVY_FOLDER_ID_m_execsql,
	  DIVY_URITYPE_VIRTUAL,			DIVY_INFOTYPE_m_execsql,	NULL
	},
	{ "^\\/\\.management\\/EXECSQL\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_m_execsql,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/\\.management\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_management,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	/* メール */
	{ "^\\/\\.mail$",
	  DIVY_INFOTYPE_mail,			DIVY_FOLDER_ID_mail,
	  DIVY_URITYPE_VIRTUAL,			DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\.mail\\/[^\\/]+$",
	  DIVY_INFOTYPE_mail_e,			DIVY_FOLDER_ID_mail,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_mail,	NULL
	},
	{ "^\\/\\.mail\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_mail,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	/* モバイル */
	{ "^\\/\\.mobile$",
	  DIVY_INFOTYPE_mobile,			DIVY_FOLDER_ID_mobile,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\.mobile\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_mobile,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	/* ログイン */
    { "^\\/\\.login$",
	  DIVY_INFOTYPE_login,          DIVY_FOLDER_ID_login,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER, DIVY_INFOTYPE_root, NULL
	},
	{ "^\\/\\.login\\/",
	  DIVY_INFOTYPE_unclassified,       DIVY_FOLDER_ID_login,
	  DIVY_URITYPE_UNKNOWN,         DIVY_INFOTYPE_unclassified, NULL
	},
	/* SAML */
    { "^\\/\\.saml$",
	  DIVY_INFOTYPE_saml,          DIVY_FOLDER_ID_saml,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER, DIVY_INFOTYPE_root, NULL
	},
	{ "^\\/\\.saml\\/acs$",
      DIVY_INFOTYPE_saml,       DIVY_FOLDER_ID_saml,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER, DIVY_INFOTYPE_root, NULL
	},
	{ "^\\/\\.saml\\/acs\\/[^\\/]+$",
      DIVY_INFOTYPE_unclassified,       DIVY_FOLDER_ID_saml,
	  DIVY_URITYPE_UNKNOWN,         DIVY_INFOTYPE_unclassified, NULL
	},
	/* 短縮URL */
	{ "^\\/\\.st\\/[^\\/]+$",
	  DIVY_INFOTYPE_shorten,		DIVY_FOLDER_ID_shorten,
	  DIVY_URITYPE_SPECIAL,         DIVY_INFOTYPE_shorten, NULL
	},
	{ "^\\/\\.st\\/",
	  DIVY_INFOTYPE_unclassified,       DIVY_FOLDER_ID_shorten,
	  DIVY_URITYPE_UNKNOWN,         DIVY_INFOTYPE_unclassified, NULL
	},
	/* Ticket */
	{ "^\\/\\.ticket$",
	  DIVY_INFOTYPE_ticket,			DIVY_FOLDER_ID_ticket,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\.ticket\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_ticket,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	/* ユーティリティ */
	{ "^\\/Utility_Folder$",
	  DIVY_INFOTYPE_utility,		DIVY_FOLDER_ID_utility,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/Utility_Folder\\/Updated_Client$",
	  DIVY_INFOTYPE_u_update,		DIVY_FOLDER_ID_u_update,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_utility,	NULL
	},
	{ "^\\/Utility_Folder\\/Updated_Client\\/[^\\/]+$",
	  DIVY_INFOTYPE_u_update_e,		DIVY_FOLDER_ID_u_update,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_u_update,		NULL
	},
	{ "^\\/Utility_Folder\\/Updated_Client\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_u_update,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/Utility_Folder\\/System_Message$",
	  DIVY_INFOTYPE_u_msg,			DIVY_FOLDER_ID_u_msg,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_utility,	NULL
	},
	{ "^\\/Utility_Folder\\/System_Message\\/[^\\/]+$",
	  DIVY_INFOTYPE_u_msg_e,		DIVY_FOLDER_ID_u_msg,
	  DIVY_URITYPE_VIRTUAL,			DIVY_INFOTYPE_u_msg,		NULL
	},
	{ "^\\/Utility_Folder\\/System_Message\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_u_msg,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/Utility_Folder\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_utility,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	/* Autoindex */
	{ "^\\/\\.fileviewer$",
	  DIVY_INFOTYPE_fileviewer,		DIVY_FOLDER_ID_fileviewer,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\.fileviewer\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_fileviewer,
	  DIVY_URITYPE_UNKNOWN,		DIVY_INFOTYPE_unclassified,	NULL
	},
#ifdef DAV_SUPPORT_POST_HANDLING
	/* CGI */
	{ "^\\/\\"DIVY_SYSCGIFOLDER_NAME"$",
	  DIVY_INFOTYPE_cgibin,		DIVY_FOLDER_ID_cgibin,
	  DIVY_URITYPE_VIRTUAL,		DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\"DIVY_SYSCGIFOLDER_NAME"\\/",	 /* システムCGIよりも下 */
	  DIVY_INFOTYPE_cgibin_e,	DIVY_FOLDER_ID_cgibin,
	  DIVY_URITYPE_SPECIAL,		DIVY_INFOTYPE_cgibin,	NULL
	},
#endif	/* DAV_SUPPORT_POST_HANDLING */
#ifdef DIVY_SUPPORT_PLUGIN
	/* プラグイン関連 */
	{ "^\\/\\"DIVY_PLUGINFOLDER_NAME"$",
	  DIVY_INFOTYPE_plugin,		DIVY_FOLDER_ID_plugin,
	  DIVY_URITYPE_VIRTUAL,		DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\"DIVY_PLUGINFOLDER_NAME"\\/[^\\/]+$",
	  DIVY_INFOTYPE_plugin_e,		DIVY_FOLDER_ID_plugin,
	  DIVY_URITYPE_SPECIAL,		DIVY_INFOTYPE_plugin,	NULL
	},
	/* プラグイン専用アップロード領域 */
	{ "^\\/\\"DIVY_PLUGINFOLDER_NAME"\\/[^\\/]+\\/uploads$",
	  DIVY_INFOTYPE_plugin_uploads,		DIVY_FOLDER_ID_plugin,
	  DIVY_URITYPE_VIRTUAL,		DIVY_INFOTYPE_plugin_e,	NULL
	},
	{ "^\\/\\"DIVY_PLUGINFOLDER_NAME"\\/[^\\/]+\\/uploads\\/[^\\/]+$",
	  DIVY_INFOTYPE_plugin_uploads_e,	DIVY_FOLDER_ID_plugin,
	  DIVY_URITYPE_SPECIAL,				DIVY_INFOTYPE_plugin_uploads,	NULL
	},
	/* プラグインのCGI */
	{ "^\\/\\"DIVY_PLUGINFOLDER_NAME"\\/[^\\/]+\\/"DIVY_PLUGINCGIFOLDER_NAME"$",
	  DIVY_INFOTYPE_plugin_cgibin,		DIVY_FOLDER_ID_plugin,
	  DIVY_URITYPE_VIRTUAL,		DIVY_INFOTYPE_plugin_e,	NULL
	},
	{ "^\\/\\"DIVY_PLUGINFOLDER_NAME"\\/[^\\/]+\\/"DIVY_PLUGINCGIFOLDER_NAME"\\/",	/* プラグインCGIよりも下 */
	  DIVY_INFOTYPE_plugin_cgibin_e,	DIVY_FOLDER_ID_plugin,
	  DIVY_URITYPE_SPECIAL,				DIVY_INFOTYPE_plugin_cgibin,	NULL
	},
	/* プラグイン専用コンフィグ */
	{ "^\\/\\"DIVY_PLUGINFOLDER_NAME"\\/[^\\/]+\\/conf$",
	  DIVY_INFOTYPE_plugin_conf,		DIVY_FOLDER_ID_plugin,
	  DIVY_URITYPE_VIRTUAL,		DIVY_INFOTYPE_plugin_e,	NULL
	},
	{ "^\\/\\"DIVY_PLUGINFOLDER_NAME"\\/[^\\/]+\\/conf\\/[^\\/]+$",
	  DIVY_INFOTYPE_plugin_conf_e,		DIVY_FOLDER_ID_plugin,
	  DIVY_URITYPE_SPECIAL,				DIVY_INFOTYPE_plugin_conf,	NULL
	},
	/* (未定義) */
	{ "^\\/\\"DIVY_PLUGINFOLDER_NAME"\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_plugin,
	  DIVY_URITYPE_UNKNOWN,		DIVY_INFOTYPE_unclassified,	NULL
	},
#endif	/* DIVY_SUPPORT_PLUGIN */
	/* 自動削除 */
	{ "^\\/\\.autodelete$",
	  DIVY_INFOTYPE_autodelete,		DIVY_FOLDER_ID_autodelete,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\.autodelete\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_autodelete,
	  DIVY_URITYPE_UNKNOWN,		DIVY_INFOTYPE_unclassified,	NULL
	},
	/* 開封通知の自動解除 */
	{ "^\\/\\.confirmreading$",
	  DIVY_INFOTYPE_confirmreading,		DIVY_FOLDER_ID_confirmreading,
	  DIVY_URITYPE_VIRTUAL | DIVY_URITYPE_ARBITER,	DIVY_INFOTYPE_root,	NULL
	},
	{ "^\\/\\.confirmreading\\/",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_confirmreading,
	  DIVY_URITYPE_UNKNOWN,		DIVY_INFOTYPE_unclassified,	NULL
	},
	/* プライベート */
	{ "^\\.[^\\/]+",
	  DIVY_INFOTYPE_unclassified,		DIVY_FOLDER_ID_user,
	  DIVY_URITYPE_UNKNOWN,			DIVY_INFOTYPE_unclassified,	NULL
	},
	{ "^\\/[^\\/]+\\/"DIVY_TRASHFOLDER_NAME"$",
	  DIVY_INFOTYPE_user_trash,		DIVY_FOLDER_ID_user,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_user_e,		NULL
	},
	{ "^\\/[^\\/]+\\/"DIVY_TRASHFOLDER_NAME"\\/[^\\/]+$",
	  DIVY_INFOTYPE_user_trash_e0,		DIVY_FOLDER_ID_user,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_user_trash,	NULL
	},
	{ "^\\/[^\\/]+\\/"DIVY_TRASHFOLDER_NAME"\\/[^\\/]+\\/[^\\/]+$",
	  DIVY_INFOTYPE_user_trash_ex,		DIVY_FOLDER_ID_user,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_user_trash_e0,	NULL
	},
	{ "^\\/[^\\/]+\\/"DIVY_TRASHFOLDER_NAME"\\/[^\\/]+\\/[^\\/]+\\/",
	  DIVY_INFOTYPE_user_trash_ex,		DIVY_FOLDER_ID_user,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_user_trash_ex,	NULL
	},
	{ "^\\/[^\\/]+\\/[^\\/]+$",		/* プライベートフォルダ直下 */
	  DIVY_INFOTYPE_user_e_regular,		DIVY_FOLDER_ID_user,
	  DIVY_URITYPE_REGULAR,			DIVY_INFOTYPE_user_e,	NULL
	},
	{ "^\\/[^\\/]+\\/[^\\/]+\\/",		/* プライベートフォルダ直下よりも下 */
	  DIVY_INFOTYPE_user_e_regular,		DIVY_FOLDER_ID_user,
	  DIVY_URITYPE_REGULAR,			DIVY_INFOTYPE_user_e_regular,	NULL
	},
	{ "^\\/[^\\/]",
	  DIVY_INFOTYPE_user_e,			DIVY_FOLDER_ID_user,
	  DIVY_URITYPE_SPECIAL,			DIVY_INFOTYPE_root,	NULL
	},
};

/* 配列 _uri_regex_patterns の要素数 */
#define URI_REGEX_PATTERNS_LEN	sizeof(_uri_regex_patterns)/sizeof(divy_uri_pattern)


/*
 * 第１階層に配置される特殊フォルダのフォルダID 配列(ソート用)
 * (note)
 * 	この配列はパフォーマンス改善を目的として導入されました。
 * 	特殊フォルダの相対URI(relativeuri) が変更されたときには、それらが
 * 	ASCIIの昇順になるよう手動でこの配列を並び替えてください。
 */
static divy_special_folderid_id _top_folders[] =
{
	DIVY_FOLDER_ID_autodelete,
#ifdef DAV_SUPPORT_POST_HANDLING
	DIVY_FOLDER_ID_cgibin,
#endif	/* DAV_SUPPORT_POST_HANDLING */
	DIVY_FOLDER_ID_confirmreading,
	DIVY_FOLDER_ID_dbfolder,
	DIVY_FOLDER_ID_dblink,
	DIVY_FOLDER_ID_dbshfolder,
	DIVY_FOLDER_ID_fileviewer,
	DIVY_FOLDER_ID_login,
	DIVY_FOLDER_ID_mail,
	DIVY_FOLDER_ID_management,
	DIVY_FOLDER_ID_mobile,
#ifdef DIVY_SUPPORT_PLUGIN
	DIVY_FOLDER_ID_plugin,
#endif	/* DIVY_SUPPORT_PLUGIN */
	DIVY_FOLDER_ID_reposdblink,
	DIVY_FOLDER_ID_saml,
	DIVY_FOLDER_ID_shorten,
	DIVY_FOLDER_ID_ticket,
	DIVY_FOLDER_ID_group,
	DIVY_FOLDER_ID_utility,
	DIVY_FOLDER_ID_userinfo,
};
/* 配列 _top_folders の要素数 */
#define TOP_FOLDERS_LEN		sizeof(_top_folders)/sizeof(divy_special_folderid_id)

/*
 * 第２階層に配置される特殊フォルダのフォルダID 配列(ソート用)
 * (note)
 * 	この配列はパフォーマンス改善を目的として導入されました。
 * 	特殊フォルダの相対URI(relativeuri) が変更されたときには、それらが
 * 	ASCIIの昇順になるよう手動でこの配列を並び替えてください。
 */
static divy_special_folderid_id _second_folders[] =
{
	DIVY_FOLDER_ID_m_dbms,
	DIVY_FOLDER_ID_m_execsql,
	DIVY_FOLDER_ID_m_group,
	DIVY_FOLDER_ID_m_msg,
	DIVY_FOLDER_ID_m_sql,
	DIVY_FOLDER_ID_m_status,
	DIVY_FOLDER_ID_m_update,
	DIVY_FOLDER_ID_m_user,
	DIVY_FOLDER_ID_u_msg,
	DIVY_FOLDER_ID_u_update,
};

/* 配列 _second_folders の要素数 */
#define SECOND_FOLDERS_LEN	sizeof(_second_folders)/sizeof(divy_special_folderid_id)

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * 配列 _special_folders が持つURIパターン文字列をコンパイルして
 * 正規表現ライブラリから利用できるようにする. (パフォーマンスを考慮)
 *
 */
DIVY_DECLARE(void) divy_precomp_uri_pattern(apr_pool_t *p)
{
	int i, len;
	divy_uri_pattern *pt;
	divy_special_folder_spec *fspec;

	/* パターンのコンパイル */
	len = URI_REGEX_PATTERNS_LEN;
	for (i = 0; i < len; i++) {
		pt = &_uri_regex_patterns[i];

		/* パターン文字列が設定されていれば、コンパイルする */
		if (IS_FILLED(pt->uri_pt)) {
			pt->preg = _pattern_pregcomp(p, pt->uri_pt);
		}
	}

	/* 特殊フォルダの相対URIの長さを取得 */
	len = DIVY_SPECIAL_FOLDERS_LEN;
	for (i = 0; i < len; i++) {
		fspec = &_special_folders[i];
		if (fspec->relativeuri != NULL) {
			fspec->uri_len = strlen(fspec->relativeuri);
		}
	}
}

/**
 * 特殊フォルダID fid が示す divy_special_folder_spec 構造体へのポインタを
 * 取得して返却する。
 *
 */
DIVY_DECLARE(const divy_special_folder_spec *) divy_get_special_fspec(
					apr_pool_t *p,
					divy_special_folderid_id fid)
{
	/* (仮定) 数値fid を配列 _special_folders の要素数として
	 * 	取得されるdivy_special_folder_spec * のid はfid と
	 * 	等しいこと。
	 */
	return &_special_folders[fid];
}

/**
 * プライベートコレクションまたはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_user_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_user,segment);
}

/**
 * グループコレクションまたはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_group_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_group,segment);
}

/**
 * linkdbsearch 結果格納フォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_dbfolder_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_dbfolder,segment);
}

/**
 * 共有コレクションまたはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_dbshfolder_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_dbshfolder,segment);
}

/**
 * 管理者フォルダのURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_management_uri(apr_pool_t *p, const char *root_uri)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_management,NULL);
}

/**
 * 更新クライアントフォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_m_update_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_m_update,segment);
}

/**
 * SQL フォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_m_sql_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_m_sql,segment);
}

/**
 * グループフォルダ(in 管理者フォルダ) またはそのフォルダの下位階層の
 * URIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_m_group_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_m_group,segment);
}

/**
 * ユーザフォルダ(in 管理者フォルダ) またはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_m_user_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_m_user,segment);
}

/**
 * 統計情報(in 管理者フォルダ) またはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_m_status_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_m_status,segment);
}

/**
 *  システムメッセージフォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_m_msg_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_m_msg,segment);
}

/**
 * DBMS フォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_m_dbms_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_m_dbms,segment);
}

/**
 * SQL ためし実行フォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_m_execsql_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_m_execsql,segment);
}

/**
 * roottreesearch 受け付けURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_userinfo_uri(apr_pool_t *p, const char *root_uri)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_userinfo,NULL);
}

/**
 * linkdbsearch 受け付けURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_dblink_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_dblink,segment);
}

/**
 * リポジトリ検索受け付けURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_reposdblink_uri(apr_pool_t *p, const char *root_uri,
							const char *segment)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_reposdblink,segment);
}

/**
 * メール送信リクエスト受け付けURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_mail_uri(apr_pool_t *p, const char *root_uri)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_mail,NULL);
}

/**
 * モバイルアクセス用URIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_mobile_uri(apr_pool_t *p, const char *root_uri,
					const char *cmd, const char *rsid, int rp)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return apr_psprintf(p,
			"%s?"DIVY_URIPARAM_CMD"=%s"
			"&"DIVY_URIPARAM_CRSID"=%s"
			"&"DIVY_URIPARAM_REQPAGE"=%d",
			DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_mobile,NULL),
			cmd, rsid, rp);
}

/**
 * ユーティリティフォルダのURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_utility_uri(apr_pool_t *p, const char *root_uri)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_utility,NULL);
}

/**
 * 更新クライアント(in ユーティリティ) フォルダのURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_u_update_uri(apr_pool_t *p, const char *root_uri)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_u_update,NULL);
}

/**
 * システムメッセージ(in ユーティリティ) フォルダのURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_u_msg_uri(apr_pool_t *p, const char *root_uri)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_u_msg,NULL);
}

/**
 * グループの下のごみ箱フォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_group_trash_uri(apr_pool_t *p, const char *root_uri,
							const char *grp_segment)
{
	if (IS_EMPTY(root_uri) || IS_EMPTY(grp_segment)) return NULL;

	return dav_divy_make_uri(p, root_uri,
			_special_folders[DIVY_FOLDER_ID_group].relativeuri,
			grp_segment, DIVY_TRASHFOLDER_NAME, NULL);

	return NULL;
}

/**
 * プライベートコレクションの下のごみ箱フォルダまたはそのフォルダの下位階層の
 * URIを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_user_trash_uri(apr_pool_t *p, const char *root_uri,
							const char *userid)
{
	if (IS_EMPTY(root_uri) || IS_EMPTY(userid)) return NULL;

	return dav_divy_make_uri(p, root_uri, userid, DIVY_TRASHFOLDER_NAME, NULL);
}

/**
 * 公開チケット用URLを組み立てる。
 *
 */
DIVY_DECLARE(char *) divy_build_ticket_url(apr_pool_t *p, const char * root_uri,
							const char *ticket)
{
	char *base;

	if (IS_EMPTY(root_uri) || IS_EMPTY(ticket)) return NULL;

	base = dav_divy_make_uri(p, root_uri,
				_special_folders[DIVY_FOLDER_ID_ticket].relativeuri, NULL);

	return apr_psprintf(p, "%s?"DIVY_URIPARAM_TICKET_URI"=%s", base, ticket);
}

/**
 * 公開用短縮URLを生成する [ 相対パス ]
 *
 */
DIVY_DECLARE(char *) divy_build_shorten_url(apr_pool_t *p, const char *root_uri,
							const char *shorten)
{
	if (IS_EMPTY(root_uri) || IS_EMPTY(shorten)) return NULL;

	return apr_psprintf(p, "%s/%s", divy_build_shorten_uri(p, root_uri), shorten);

}

/**
 * Autoindex スタートアップページ用URIを組み立てる.
 *
 */
DIVY_DECLARE(char *) divy_build_autoindex_startup_uri(apr_pool_t *p,
                                                      const char * root_uri)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return dav_divy_make_uri(p, root_uri,
				_special_folders[DIVY_FOLDER_ID_fileviewer].relativeuri, NULL);
}

/**
 * ログイン用URIを組み立てる.
 *
 */
DIVY_DECLARE(char *) divy_build_login_uri(apr_pool_t *p,
                                                      const char * root_uri)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_login,NULL);

}

DIVY_DECLARE(char *) divy_build_saml_uri(apr_pool_t *p,
                                                      const char * root_uri)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_saml,NULL);
}

DIVY_DECLARE(char *) divy_build_shorten_uri(apr_pool_t *p,
                                                      const char * root_uri)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return DIVY_BUILD_GENERAL_URI(p,root_uri,DIVY_FOLDER_ID_shorten,NULL);
}

#ifdef DAV_SUPPORT_POST_HANDLING
/**
 * トップCGI用URIを組み立てる.
 *
 */
DIVY_DECLARE(char *) divy_build_topcgi_uri(apr_pool_t *p, const char * root_uri)
{
	if (IS_EMPTY(root_uri)) return NULL;

	return dav_divy_make_uri(p, root_uri,
					DIVY_SYSCGIFOLDER_NAME, DIVY_SYSCGI_TOP_NAME, NULL);
}
#endif	/* DAV_SUPPORT_POST_HANDLING */

#ifdef DIVY_SUPPORT_PLUGIN
/**
 * プラグイン名 plugin_name、相対URI uri のURIパスを組み立てて返却する.
 *
 */
DIVY_DECLARE(char *) divy_build_plugin_uri(apr_pool_t *p, const char * root_uri,
                                           const char *plugin_name, const char *uri)
{
	if (IS_EMPTY(root_uri) || IS_EMPTY(plugin_name)) return NULL;

	return dav_divy_make_uri(p, root_uri, DIVY_PLUGINFOLDER_NAME, plugin_name, uri, NULL);
}
#endif	/* DIVY_SUPPORT_PLUGIN */

/**
 * 指定されたfid の特殊フォルダが持つ表示名(displayname) を取得して返却する。
 *
 */
DIVY_DECLARE(char *) divy_get_sfolder_name(apr_pool_t *p,
						divy_special_folderid_id fid)
{
	const divy_special_folder_spec *fspec = &_special_folders[fid];
	if (fspec == NULL) return NULL;

	/* 表示名をコピーして返却する */
	return apr_pstrdup(p, fspec->displayname);
}

/**
 * 指定されたfid の特殊フォルダが持つ相対URIのfinal path segment を
 * 取得して返却する。
 *
 */
DIVY_DECLARE(char *) divy_get_sfolder_finalpathsegment(apr_pool_t *p,
						divy_special_folderid_id fid)
{
	const divy_special_folder_spec *fspec = &_special_folders[fid];
	if (fspec == NULL) return NULL;

	/* final path segment を相対URIから切り出す */
	return dav_divy_extract_finalpath_segment(p, fspec->relativeuri);
}

/**
 * 指定されたfid の特殊フォルダが持つ"アクセス拒否メソッド名" を取得して返却する。
 *
 */
DIVY_DECLARE(char *) divy_get_sfolder_accessdeny_name(apr_pool_t *p,
						divy_special_folderid_id fid)
{
	const divy_special_folder_spec *fspec = &_special_folders[fid];
	if (fspec == NULL) return 0;

	/* final path segment を相対URIから切り出す */
	return apr_pstrdup(p, fspec->accessdeny_name);
}

/**
 * pcol_name が示すプライベートコレクションの名称が特殊フォルダの名称と
 * バッティングしていないかどうか判定する。
 *
 */
DIVY_DECLARE(int) divy_validate_different_sfolder_name(apr_pool_t *p,
							const char *pcol_name)
{
	int i;
	char *ch;
	const divy_special_folder_spec *fspec;

	/* 特殊フォルダと名称がバッティングしないことの確認 */
	for (i = 0; i < DIVY_FOLDER_ID_END; i++) {

		fspec = &_special_folders[i];
		if (fspec == NULL) continue;

		if (divy_count_dirs(fspec->relativeuri) != 1) continue;

		/* final path segment の取得 */
		ch = divy_strrchr(fspec->relativeuri, '/');
		if (ch != NULL && strcmp(ch, pcol_name) == 0) {
			return 1;	/* 名前がバッティングしていた */
		}
	}

	return 0;
}

/**
 * エンティティの名称として正しいかどうかを検証する。
 *
 */
DIVY_DECLARE(int) divy_validate_having_entity_prefix(apr_pool_t *p,
							const char *name)
{
	int i, len = RELATION_PREFIX_SET_LEN;

	if (IS_EMPTY(name)) return 0;

	for (i = 0; i < len; i++) {
		if (strncmp(name, relation_prefix_set[i],
					strlen(relation_prefix_set[i])) == 0) {
			return 1;	/* リレーションプレフィックスだった */
		}
	}

	return 0;
}

/**
 * 指定されたuri を解析して、divy_uri_spec 構造体を生成する。
 *
 */
DIVY_DECLARE(int) divy_parse_uri(apr_pool_t *p, const char *root_uri,
					const char *uri, divy_uri_spec **u_spec)
{
	char *uri_str = NULL, *uri_special_part = NULL;
	int i, cmp, ret;
	apr_size_t root_len, len;
	divy_uri_pattern *pt  = NULL;
	divy_special_folder_spec *fspec;
	divy_special_folderid_id fid;
	divy_uri_spec *cached_u_spec = NULL;

	*u_spec = NULL;	/* 初期化 */

	if (p == NULL || IS_EMPTY(uri)) return 1;


	/* キャッシュされていたか?
	 *
	 * (note) uri パースロジックは低速なので結果をキャッシュすることで
	 * 	パフォーマンス劣化を(多少)防いでいる */
	cached_u_spec = divy_pcache_vget_data(p, DIVY_PCACHE_DAT_URI_USPEC, uri, NULL);
	if (cached_u_spec != NULL) {
		/* キャッシュ値をコピーして返却(改変を避けるため) */
		*u_spec = _clone_uri_spec(p, cached_u_spec);
		return 0;
	}

	*u_spec = apr_pcalloc(p, sizeof(divy_uri_spec));
	(*u_spec)->uritype    = DIVY_URITYPE_UNKNOWN;
	(*u_spec)->infotype   = DIVY_INFOTYPE_unclassified;
	(*u_spec)->uri        = apr_pstrdup(p, uri);	/* オリジナルURI */
	(*u_spec)->p_infotype = DIVY_INFOTYPE_unclassified;

	/* 不要なスラッシュ文字を取り除く */
	uri_str = dav_divy_remove_endslash(p, uri);
	_no2slash(uri_str);

	/*
	 * root uri を取得する
	 */
	(*u_spec)->root = apr_pstrdup(p, root_uri);
	if ((*u_spec)->root == NULL) {
		/* divy のディレクティブ以外からの呼び出し */
		(*u_spec)->special_part = "";
		(*u_spec)->other_part   = uri_str;
		goto cache_result;
	}

	root_len = strlen((*u_spec)->root);

	/*
	 * root uri が "/"かどうか
	 */
	(*u_spec)->is_default_root = ((*u_spec)->root[0] == '/') && ((*u_spec)->root[1] == '\0');

	/* final path segment を取得 (先頭にスラッシュはつかない) */
	(*u_spec)->final_path_segment = dav_divy_extract_finalpath_segment(p, uri_str);

	/* root uri の整合性チェック */
	cmp = strcmp(uri_str, (*u_spec)->root);
	if (cmp < 0) {
		/* 扱えないURIだった時 */
		(*u_spec)->special_part = "";
		(*u_spec)->other_part   = uri_str;
		goto cache_result;
	}
	/* $root */
	else if (cmp == 0) {
		(*u_spec)->special_part = "";
		(*u_spec)->other_part   = "";
		(*u_spec)->uritype      = DIVY_URITYPE_SPECIAL;
		(*u_spec)->infotype     = DIVY_INFOTYPE_root;
		goto cache_result;
	}
	else {
		/* $root != "/" の時、uri の前半部と$root/ は一致しているか？*/
		if (!(*u_spec)->is_default_root) {
			ret = strncmp(uri_str, (*u_spec)->root, root_len);
			if (ret != 0 || uri_str[root_len] != '/') {
				(*u_spec)->special_part = "";	/* 何もいれない */
				(*u_spec)->other_part   = uri_str;
				goto cache_result;
			}
		}
	}

	/*
	 * is_default_root が 1ならば、uri_str の先頭に "/"をつける
	 * (note)
	 * 	URI解析ロジックを単純にするために、$root = "/" の場合、
	 * 	$rootは長さ0の名称を持つものとして扱います。
	 */
	if ((*u_spec)->is_default_root) {
		uri_str = apr_pstrcat(p, "/", uri_str, NULL);
	}

	/*
	 * 特殊フォルダを表すURI 部分を取得する
	 * $root を飛ばす (先頭に'/'あり)
	 */
	uri_special_part = &uri_str[root_len];

	/* トップ階層をバイナリサーチ */
	ret = _match_special_folder(uri_special_part, TOP_FOLDERS_LEN, _top_folders, &fid);
	if (ret == DIVY_RET_FULLMATCH || ret == DIVY_RET_PARENTMATCH) {
		/* 完全一致 or 親要素として部分一致 */
		char ch = uri_special_part[_special_folders[fid].uri_len];

		if (ch == '/' && (fid == DIVY_FOLDER_ID_management || fid == DIVY_FOLDER_ID_utility)) {

			/* ２階層目をバイナリサーチ */
			ret = _match_special_folder(uri_special_part, SECOND_FOLDERS_LEN, _second_folders, &fid);
			/* 部分一致しなかった場合 */
			if (ret == DIVY_RET_NOTMATCH || ret == DIVY_RET_PARTMATCH) {
				(*u_spec)->uritype  = DIVY_URITYPE_UNKNOWN;
				(*u_spec)->infotype = DIVY_INFOTYPE_unclassified;
			}
		}

		if (ret == DIVY_RET_FULLMATCH || ret == DIVY_RET_PARENTMATCH) {
			len = URI_REGEX_PATTERNS_LEN;

			/* パターンマッチする */
			for (i = 0; i < len; i++) {
				pt = &_uri_regex_patterns[i];

				if (pt->fid != fid) continue;

				ret = _pattern_regexec(pt->preg, uri_special_part);
				if (ret == 0) break;
			}

			if (pt != NULL) {
				fspec = &_special_folders[pt->fid];

				(*u_spec)->special_part = apr_pstrdup(p, fspec->relativeuri);
				(*u_spec)->other_part   = &uri_special_part[fspec->uri_len];
				(*u_spec)->uritype      = pt->uritype;
				(*u_spec)->infotype     = pt->infotype;
				(*u_spec)->p_infotype   = pt->p_infotype;
			}
		}
	}
	/* ユーザフォルダの可能性を調査する */
	else {
		len = URI_REGEX_PATTERNS_LEN;

		/* パターンマッチする */
		for (i = 0; i < len; i++) {
			pt = &_uri_regex_patterns[i];

			if (pt->fid != DIVY_FOLDER_ID_user) continue;

			ret = _pattern_regexec(pt->preg, uri_special_part);
			if (ret == 0) break;
		}
		if (pt != NULL) {
			fspec = &_special_folders[pt->fid];

			(*u_spec)->special_part = "";
			(*u_spec)->other_part   = uri_special_part;
			(*u_spec)->uritype      = pt->uritype;
			(*u_spec)->infotype     = pt->infotype;
			(*u_spec)->p_infotype   = pt->p_infotype;
		}
	}

cache_result:
	/* 結果をキャッシュする (ただしコピー) */
	divy_pcache_vset_data(p, _clone_uri_spec(p, *u_spec), DIVY_PCACHE_DAT_URI_USPEC, (*u_spec)->uri, NULL);

	return 0;
}

/**
 * 指定されたuri からメール監視状態を設定可能なリソースのuri を切り出す。
 *
 */
DIVY_DECLARE(int) divy_extract_mailwatch_uri(apr_pool_t *p, const char *root_uri,
						const char *uri,char **mwatch_uri)
{
	divy_uri_spec *u_spec = NULL;

	/* 初期化 */
	*mwatch_uri = NULL;

	if (IS_EMPTY(uri) || IS_EMPTY(root_uri)) return 1;

	/* URIパース */
	if (divy_parse_uri(p, root_uri, uri, &u_spec)) return 1;

	/*
	 * 渡されたURIがメール監視状態設定可能コレクションの場合はそのまま
	 * (note)
	 * 	管理フォルダの"グループ"エンティティも監視状態可能という
	 * 	ステータスにしておく。(厳密には違うのだけど)
	 */
	if (u_spec->infotype == DIVY_INFOTYPE_group_e ||
	    u_spec->infotype == DIVY_INFOTYPE_dbfolder_e ||
	    u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e ||
	    u_spec->infotype == DIVY_INFOTYPE_m_group_e) {

		*mwatch_uri = apr_pstrdup(p, uri);
		return 0;
	}
	/*
	 * メール監視状態になり得ないコレクションの場合
	 *
	 * $root/Group Folder/$groupid, $root/.dbfolder/$keys,
	 * $root/.dbshfolder/$keys を含まないコレクションは却下
	 */
	else if (strcmp(_special_folders[DIVY_FOLDER_ID_group].relativeuri,
			u_spec->special_part) &&
		 strcmp(_special_folders[DIVY_FOLDER_ID_dbfolder].relativeuri,
			u_spec->special_part) &&
		 strcmp(_special_folders[DIVY_FOLDER_ID_dbshfolder].relativeuri,
			u_spec->special_part)) {

			return 0;	/* 一応正常としておく */
	}
	/*
	 * メール監視状態の設定が可能な子リソースの場合
	 */
	else {
		/* メール監視状態設定可能コレクションのuri を切り出す */
		char *slapos 	= (char *) (u_spec->other_part);
		char *top_other;
		int  len;

		/* ２つ目のスラッシュの位置を調べる */
		if ((slapos = divy_strchr(&slapos[1], '/')) == NULL) {
			return 1;
		}

		/* トップのパス最終部の長さを取得 */
		len = slapos - u_spec->other_part;

		/* 領域確保->格納(１は終端の分) */
		top_other = apr_pcalloc(p, sizeof(char) * (len + 1));
		(void) apr_cpystrn(top_other, u_spec->other_part, len + 1);

		*mwatch_uri = dav_divy_make_uri(p, u_spec->root,
						u_spec->special_part,
						top_other, NULL);
		return 0;
	}
}

/**
 * 指定されたグループコレクション以下のリソースuri から
 * グループコレクションURIを算出して返却する。
 *
 */
DIVY_DECLARE(int) divy_extract_groupcol_uri(apr_pool_t *p, const char *root_uri,
						const char *uri, char **grpcol_uri)
{
	divy_uri_spec *u_spec = NULL;
	divy_infotype infotype;

	*grpcol_uri = NULL;

	if (IS_EMPTY(root_uri) || IS_EMPTY(uri)) return 1;
	uri = dav_divy_remove_endslash(p, uri);	/* 末尾のスラッシュを除去 */

	/* URIのパース */
	if (divy_parse_uri(p, root_uri, uri, &u_spec)) return 1;
	infotype = u_spec->infotype;

	/* グループコレクション以下のリソースの場合 */
	if (infotype == DIVY_INFOTYPE_group_e        ||
		infotype == DIVY_INFOTYPE_group_trash    ||	/* (note) ごみ箱そのものも対象にしました */
		infotype == DIVY_INFOTYPE_group_trash_e0 ||
		infotype == DIVY_INFOTYPE_group_trash_ex ||
		infotype == DIVY_INFOTYPE_group_e_regular) {

		*grpcol_uri = _extract_groupcol_uri(p, u_spec);
	}
	/* 上記以外の場合 */
	else {
		/* 何もしない. エラーにはしませんがURIは無しです */
	}
	return 0;
}

#ifdef DIVY_SUPPORT_PLUGIN
/**
 * 指定されたURI情報u_spec からプラグイン名称を抽出する.
 *
 */
DIVY_DECLARE(int) divy_extract_plugin_name(apr_pool_t *p, divy_uri_spec *u_spec,
						char **name)
{
	*name = NULL;

	if (u_spec == NULL) return 1;

	if (u_spec->infotype == DIVY_INFOTYPE_plugin_e) {
		*name = apr_pstrdup(p, u_spec->final_path_segment);
	}
	else if (u_spec->infotype == DIVY_INFOTYPE_plugin_uploads   ||
			 u_spec->infotype == DIVY_INFOTYPE_plugin_uploads_e ||
			 u_spec->infotype == DIVY_INFOTYPE_plugin_cgibin    ||
			 u_spec->infotype == DIVY_INFOTYPE_plugin_cgibin_e  ||
			 u_spec->infotype == DIVY_INFOTYPE_plugin_conf      ||
			 u_spec->infotype == DIVY_INFOTYPE_plugin_conf_e) {

		/* プラグインのURIパートの切り出し */
		char *slapos 	= (char *) (u_spec->other_part);
		int  len;

		/* ２つ目のスラッシュの位置を調べる */
		if ((slapos = divy_strchr(&slapos[1], '/')) == NULL) {
			return 1;
		}

		/* トップのパス最終部の長さを取得 */
		len = slapos - u_spec->other_part;

		/* 領域確保->格納(1 は終端の分) */
		*name = apr_pcalloc(p, sizeof(char) * (len + 1));
		(void) apr_cpystrn(*name, &u_spec->other_part[1], len);
	}
	/* 上記以外の場合 */
	else {
		/* 何もしない. エラーにはしませんがURIは無しです */
	}
	return 0;
}
#endif	/* DIVY_SUPPORT_PLUGIN */

/**
 * 指定されたuri のリソースが捨てられる時、そのリソースのごみ箱となる
 * フォルダのURIを算出して返却する。
 *
 */
DIVY_DECLARE(int) divy_extract_trashfolder_uri(apr_pool_t *p, const char *root_uri,
						const char *uri, char **trash_uri)
{
	divy_uri_spec *u_spec = NULL;
	char *segment, *str;

	/* 初期化 */
	*trash_uri = NULL;

	if (IS_EMPTY(uri) || IS_EMPTY(root_uri)) return 1;	/* 何も出来ない */

	/* 不要なスラッシュ文字を取り除く */
	str = dav_divy_remove_endslash(p, uri);
	_no2slash(str);

	/* uri をパースする */
	if (divy_parse_uri(p, root_uri, str, &u_spec)) return 1;

	/* 渡されたURIがごみ箱そのものだった場合 */
	if (u_spec->infotype == DIVY_INFOTYPE_user_trash ||
	    u_spec->infotype == DIVY_INFOTYPE_group_trash) {

		/* ごみ箱のURIをそのまま返す */
		*trash_uri = apr_pstrdup(p, str);
		return 0;
	}

	/* 渡されたURIがユーザごみ箱の下、ユーザフォルダまたはその下 */
	if (u_spec->infotype == DIVY_INFOTYPE_user_trash_e0  ||
	    u_spec->infotype == DIVY_INFOTYPE_user_trash_ex  ||
	    u_spec->infotype == DIVY_INFOTYPE_user_e         ||
	    u_spec->infotype == DIVY_INFOTYPE_user_e_regular) {

		/* 先頭のpath segment を切り出す --> userid を取る */
		segment = dav_divy_extract_firstpath_segment(p, u_spec->other_part);
		*trash_uri = divy_build_user_trash_uri(p, root_uri, segment);
		return 0;
	}

	/* 渡されたURIがグループごみ箱の下、グループフォルダまたはそのの下 */
	if (u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
	    u_spec->infotype == DIVY_INFOTYPE_group_trash_ex ||
	    u_spec->infotype == DIVY_INFOTYPE_group_e        ||
	    u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {

		/* 先頭のpath segment を切り出す */
		segment = dav_divy_extract_firstpath_segment(p, u_spec->other_part);
		*trash_uri = divy_build_group_trash_uri(p, root_uri, segment);
		return 0;
	}

	return 1;	/* 残りは全部駄目。ごみ箱を持っていません */
}

/**
 * str から公開チケット用文字列を生成して返却する
 *
 */
DIVY_DECLARE(char *) divy_create_ticket_str(apr_pool_t *p, const char *str)
{
	char *base64_str, *ticket;
	apr_size_t len, base64_len;
	const unsigned char *s;
	unsigned char c, *d;

	if (IS_EMPTY(str)) return NULL;

	/* (1) Base64エンコーディングする */
	len = strlen(str);
	base64_str = apr_pcalloc(p, sizeof(char) * (apr_base64_encode_len(len) + 1));
	base64_len = apr_base64_encode(base64_str, str, len);
	base64_str[base64_len + 1] = '\0';

	/* (2) 文字の置き換え(問題を起こすので)
	 * 	'+' -> '*', '=' -> '%' */
	ticket = apr_pcalloc(p, sizeof(char) * (base64_len + 1));
	s = (const unsigned char *) base64_str;
	d = (unsigned char *) ticket;

	while ((c = *s)) {
		if (c == '+') {
			*d++ = '*';
		}
		else if (c == '=') {
			*d++ = '%';
		}
		else {
			*d++ = c;
		}
		++s;
	}
	*d = '\0';

	return ticket;
}

/**
 * 公開チケット用文字列ticket からオリジナル文字列に復号化して返却する。
 *
 */
DIVY_DECLARE(char *) divy_parse_ticket_str(apr_pool_t *p, const char *ticket)
{
	char *url, *url1;
	apr_size_t base64_len;
	const unsigned char *s;
	unsigned char c, *d;

	if (IS_EMPTY(ticket)) return NULL;

	/* (1) 文字列の置き換え
	 * 	'*' -> '+', '%' -> '=' */
	url1 = apr_pcalloc(p, sizeof(char) * (strlen(ticket) + 1));
	s = (const unsigned char *) ticket;
	d = (unsigned char *) url1;
	while ((c = *s)) {
		if (c == '*') {
			*d++ = '+';
		}
		else if (c == '%') {
			*d++ = '=';
		}
		else {
			*d++ = c;
		}
		++s;
	}
	*d = '\0';

	/* (2) Base64デコード */
	url = apr_pcalloc(p, sizeof(char) * (apr_base64_decode_len(url1) + 1));
	base64_len = apr_base64_decode(url, url1);
	url[base64_len + 1] = '\0';

	return url;
}

/**
 * リソースIDを元に短縮文字列を生成します。
 *
 * rid  = 000003228573 (12桁)
 * salt = 82           ランダム文字 (2桁)
 * base = 82000003228573 => Ujenskgx (62進数)
 *
 * @param p     apr_pool_t * プール
 * @param rid   const char * リソースID
 * @return 短縮文字列 char * / 失敗: NULL
 */
DIVY_DECLARE(char *) divy_get_rid2shorten(apr_pool_t *pool, const char *rid, const char *old) 
{
	char *str;
	apr_int64_t base;
	char t[8], *p = t;
	char *ret, *s, *first;
	int count = 0;
	
	ret = s = first = apr_pcalloc(pool, 24);

	do {
		ret = s = first; /* initialize */
		/* ランダムな数字二桁(salt)とリソースIDを連結する */
		str = apr_psprintf(pool, "%d%s", 10 + rand() % (99-10+1), rid);
		base = apr_atoi64(str);

		do *p++ = base % SHORTEN_TBL_LEN; while(base /= SHORTEN_TBL_LEN);
		do *s++ = _shortentbl[(int)(*--p)]; while(p != t);
		*s = '\0';
	} while(count<10 && old != NULL && strcmp(old, ret) == 0); 

	return ret;
}

/**
 * 短縮文字列をリソースIDへ戻します
 *
 * @param p      apr_pool_t * プール
 * @param shorten const char * リソースID
 * @return リソースID char * / 失敗: NULL
 */
DIVY_DECLARE(char*) divy_get_shorten2rid(apr_pool_t *pool, const char *shorten)
{

	int i;
	unsigned char *p = (unsigned char*)shorten;
	char tbl[256];	// ascii table
	char *rid;
	apr_int64_t v = APR_INT64_C(0);

	/* テーブルの初期化 */
	for (i=0; i<SHORTEN_TBL_LEN; i++) tbl[_shortentbl[i]] = i;
	while(*p) v = v * SHORTEN_TBL_LEN + tbl[*p++];

	rid = apr_psprintf(pool, "%"APR_INT64_T_FMT, v);

	/* 先頭のsaltを取り除く */
	return rid + 2; 
}

#ifdef DAV_SUPPORT_POST_HANDLING
/**
 * CGIの"SCRIPT_NAME"パスを生成する(CGI専用)
 *
 */
DIVY_DECLARE(char *) divy_build_cgi_scriptname(apr_pool_t *p, divy_infotype infotype,
                                               const char *scriptpath)
{
	const char *prefix = NULL;
	if (IS_EMPTY(scriptpath)) return NULL;

	/* システムCGIの場合 */
	if (infotype == DIVY_INFOTYPE_cgibin_e) {
		prefix = DIVY_SYSCGIFOLDER_NAME;
	}
#ifdef DIVY_SUPPORT_PLUGIN
	else if (infotype == DIVY_INFOTYPE_plugin_cgibin_e) {
		prefix = DIVY_PLUGINCGIFOLDER_NAME;
	}
#endif	/* DIVY_SUPPORT_PLUGIN */

	if (prefix != NULL) {
		if (*scriptpath == '/') {
			return apr_psprintf(p, "/%s%s", prefix, scriptpath);
		}
		else {
			return apr_psprintf(p, "/%s/%s", prefix, scriptpath);
		}
	}

	return NULL;
}
#endif	/* DAV_SUPPORT_POST_HANDLING */

/**
 * ユーザリレーションURI($root/.management/GROUP/$gname/.RU_xxxx) を組み立てて
 * 返却する.
 *
 */
DIVY_DECLARE(char *) divy_build_rusr_uri(apr_pool_t *p, const char *guri, const char *userid)
{
	if (IS_EMPTY(guri) || IS_EMPTY(userid)) {
		return "";
	}

	return dav_divy_make_uri(p, guri, apr_psprintf(p, DIVY_PREFIX_RUSER"%s", userid), NULL);
}


/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
/**
 * 指定された文字列 str に含まれる多重スラッシュを単一スラッシュに変える。
 * str の内容が書き換えられてしまいますので、注意して下さい。
 *
 * @param str char *
 */
static void _no2slash(char *str)
{
	char *s, *d;

	if (IS_EMPTY(str)) return;	/* 何もしない */

	s = d = str;
	while (*s) {
		if ((*d++ = *s) == '/') {
			do {
				++s;
			} while (*s == '/');
		}
		else {
			++s;
		}
	}
	*d = '\0';
}

/**
 * パターンマッチングで使用したregex_t の割り当て領域を解放する。
 *
 * @param data void * regex_t へのポインタが入っています
 * @return apr_status_t 常にAPR_SUCCESS
 */
static apr_status_t _pattern_cleanup(void *data)
{
	if (data != NULL) {
		ap_regfree((ap_regex_t *) data);
	}

	return APR_SUCCESS;
}

/**
 * パターン文字列pattern をコンパイルする。
 *
 * @param p apr_pool_t * 作業用のプール。ap_regex_t へのポインタも格納される
 * @param pattern const char * パターン文字列
 * @return ap_regex_t * コンパイルされたオブジェクト。コンパイルに失敗したらNULL
 * 		またpattern がNULLであってもNULLを返却します。
 */
static ap_regex_t * _pattern_pregcomp(apr_pool_t *p, const char *pattern)
{
	ap_regex_t *preg;
	apr_status_t rv;
	
	if (pattern == NULL) return NULL;

	/* パターンオブジェクト(パターンバッファ)の生成 */
	preg = apr_palloc(p, sizeof(ap_regex_t));

	/* コンパイルする */
	if ((rv = ap_regcomp(preg, pattern, 0)) != APR_SUCCESS) {
		return NULL;
	}

	/* preg をp のクリーンアップハンドラに登録する */
	apr_pool_cleanup_register(p, (void *) preg, _pattern_cleanup,
					apr_pool_cleanup_null);
	return preg;
}

/**
 * string がコンパイルされたパターンpreg と一致するかどうか調べる。
 *
 * @param preg reqex_t * コンパイルされたパターンオブジェクト
 * @param str const char * パターンマッチングする文字列
 * @return int マッチングの結果
 * 	マッチングの成功時には 0 を返し、失敗時には REG_NOMATCH (!= 0) を返す
 * 	str がNULLであってもREG_NOMATCH を返します。
 */
static int _pattern_regexec(ap_regex_t *preg, const char *str)
{
	if (preg == NULL || str == NULL) return AP_REG_NOMATCH;

	return ap_regexec(preg, str, 0, NULL, 0);
}

/**
 * 長さtbl_len のテーブルfolder_ids (ソートされた特殊フォルダIDの集合) の
 * フォルダIDから特殊フォルダの相対URIを算出し、そのURIがstr と部分一致するか
 * どうか判定する。
 *
 * @return int 処理ステータス
 * 	DIVY_RET_NOTMATCH   : 一致しなかった
 * 	DIVY_RET_FULLMATCH  : 完全一致した
 * 	DIVY_RET_PARENTMATCH: 部分一致し、一致した相対URIはstr の親フォルダ要素
 * 	DIVY_RET_PARTMATCH  : 部分一致し、一致した相対URIはstr の親フォルダ要素ではない
 */
static int _match_special_folder(const char *str, int tbl_len,
				const divy_special_folderid_id *folder_ids,
				divy_special_folderid_id *fid)
{
	int ret = 1, low, high, mid;
	divy_special_folder_spec *fspec;

	for (low = 0, high = tbl_len; low < high; ) {
		mid = (low + high) / 2;
		fspec = &_special_folders[folder_ids[mid]];

		ret = strncmp(fspec->relativeuri, str, fspec->uri_len);
		/* 部分一致した */
		if (ret == 0) {
			/* 最後の要素をstr から抽出 */
			char ch = str[fspec->uri_len];
			*fid = fspec->id;

			if (ch == '\0') {
				return DIVY_RET_FULLMATCH;	/* 完全一致 */
			}
			else if (ch == '/') {
				return DIVY_RET_PARENTMATCH;	/* 相対URIは親要素だった */
			}
			else {
				return DIVY_RET_PARTMATCH;	/* 単に部分一致しただけ */
			}
		}
		else if (ret < 0) {
			low = mid + 1;
		}
		else {
			high = mid;
		}
	}

	return DIVY_RET_NOTMATCH;	/* 部分一致しなかった */
}


/**
 * 指定されたu_spec のdeep コピーを作成する。
 *
 * @param p apr_pool_t * コピーされるu_spec の領域を確保するプール
 * @param u_spec divy_uri_spec * コピー元
 * @return divy_uri_spec * コピーされたdivy_uri_spec へのポインタ
 * 	引数のu_spec がNULLならばNULLを返します
 */
static divy_uri_spec * _clone_uri_spec(apr_pool_t *p, divy_uri_spec *u_spec)
{
	divy_uri_spec *cloned_u_spec = NULL;

	if (u_spec == NULL) return NULL;	/* NULLにはNULLを */

	cloned_u_spec = apr_palloc(p, sizeof(divy_uri_spec));

	/* コピー(deep コピーなので自力で実装した) */
	cloned_u_spec->uri                = apr_pstrdup(p, u_spec->uri);
	cloned_u_spec->root               = apr_pstrdup(p, u_spec->root);
	cloned_u_spec->is_default_root    = u_spec->is_default_root;
	cloned_u_spec->special_part       = apr_pstrdup(p, u_spec->special_part);
	cloned_u_spec->other_part         = apr_pstrdup(p, u_spec->other_part);
	cloned_u_spec->final_path_segment = apr_pstrdup(p, u_spec->final_path_segment);
	cloned_u_spec->uritype            = u_spec->uritype;
	cloned_u_spec->infotype           = u_spec->infotype;
	cloned_u_spec->p_infotype         = u_spec->p_infotype;

	return cloned_u_spec;
}

/**
 * u_spec が示すURIからグループコレクションURIを組み立てて(切り出して)返す.
 *
 * @param p apr_pool_t *
 * @param uri const char *
 * @param u_spec divy_uri_spec *
 * @return char * グループコレクションURI
 */
static char * _extract_groupcol_uri(apr_pool_t *p, divy_uri_spec *u_spec)
{
	divy_infotype infotype;	

	if (u_spec == NULL) {
		return NULL;
	}
	infotype = u_spec->infotype;

	/* グループコレクション自身の場合 */
	if (infotype == DIVY_INFOTYPE_group_e) {
		return apr_pstrdup(p, u_spec->uri);	/* コピーして返す */
	}
	/* グループコレクションよりも下の場合 */
	if (infotype == DIVY_INFOTYPE_group_trash    ||
		infotype == DIVY_INFOTYPE_group_trash_e0 ||
		infotype == DIVY_INFOTYPE_group_trash_ex ||
		infotype == DIVY_INFOTYPE_group_e_regular) {

		/* グループコレクションURIパートの切り出し(再合成) */
		char *slapos 	= (char *) (u_spec->other_part);
		char *top_other;
		int  len;

		/* ２つ目のスラッシュの位置を調べる */
		if ((slapos = divy_strchr(&slapos[1], '/')) == NULL) {
			return NULL;
		}

		/* トップのパス最終部の長さを取得 */
		len = slapos - u_spec->other_part;

		/* 領域確保->格納(1 は終端の分) */
		top_other = apr_pcalloc(p, sizeof(char) * (len + 1));
		(void) apr_cpystrn(top_other, u_spec->other_part, len + 1);

		return dav_divy_make_uri(p, u_spec->root, u_spec->special_part, top_other, NULL);
	}
	/* 上記以外 */
	else {
		return NULL;
	}
}


