/**
 * $Id$
 *
 * 定期的な処理を行う
 *
 * watchdog timer
 * 
 */

#include "mpm_common.h"
#include "httpd.h"
#include "http_log.h"
#include "apr.h"
#include "apr_file_io.h"
#include "tf_logger.h"
#include "mod_dav_tf.h"
#include "tf_array.h"
#include "tf_jobs.h"

#include "util_common.h"



APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Define structure
  ----------------------------------------------------------------------------*/
/* watchdog 名 */
#define TF_WATCHDOG_NAME "_TEAMFILE_"

/* ジョブ間隔 */
#define TF_JOB_DEFAULT_INTERVAL 60

typedef struct {

	/* server rec */
	server_rec *s;

	/* work pool */
	apr_pool_t *p;

	ap_watchdog_t *watchdog;

	const char *cmdpath;

	divy_array_t *cmds;

	apr_int64_t  count;

} divy_watchdog_t;

/*------------------------------------------------------------------------------
  Declare Privete Functions
  ----------------------------------------------------------------------------*/
static APR_OPTIONAL_FN_TYPE(ap_watchdog_get_instance) *divy_wd_get_instance;
static APR_OPTIONAL_FN_TYPE(ap_watchdog_register_callback) *divy_wd_register_callback;
static APR_OPTIONAL_FN_TYPE(ap_watchdog_set_callback_interval) *divy_wd_set_interval;

static apr_status_t divy_run_watchdog(int state, void *baton, apr_pool_t *ptemp);
/*------------------------------------------------------------------------------
  Define Public Functions
  ----------------------------------------------------------------------------*/
/*
 * post_configステージで呼ばれる用途です。
 */
DIVY_DECLARE(apr_status_t) dav_divy_start_watchdog(apr_pool_t *pconf, apr_pool_t *plog, apr_pool_t *ptemp, server_rec *s)
{

	int flag;
	apr_allocator_t *allocator;
	apr_pool_t *wdp;
	apr_status_t rv;
	divy_watchdog_t *wd;
	apr_dir_t *job_dir     = NULL;
	apr_finfo_t finfo      = { 0 };

	/* mod_watchdogが有効化されているならばap_watchdog_xxx系の関数を
	 * 持ってくることができます
	 */
	divy_wd_get_instance = APR_RETRIEVE_OPTIONAL_FN(ap_watchdog_get_instance);
	divy_wd_register_callback = APR_RETRIEVE_OPTIONAL_FN(ap_watchdog_register_callback);
	divy_wd_set_interval = APR_RETRIEVE_OPTIONAL_FN(ap_watchdog_set_callback_interval);

	if (!divy_wd_get_instance || !divy_wd_register_callback || !divy_wd_set_interval) {
			ERRLOG0(pconf, APLOG_CRIT, DIVY_FST_INFO + DIVY_SST_PROC, "tf_watchdog is required");
			return !OK;
	}

	/* watchdogで動作するのはRequestではないサーバ構造体をTRACEで利用する */
	TRACE_S(s)

	/* 一回目はスキップします */
	flag = divy_pcache_get_flag(s->process->pool, DIVY_PCACHE_FLG_GL_JOBS);
	if (!flag) {
		divy_pcache_set_flag(s->process->pool, 1, DIVY_PCACHE_FLG_GL_JOBS);
		return OK;
	}

	/*
	 * watchdog内部で保持したいデータ用にプールを作ります
	 */
	apr_allocator_create(&allocator);
	apr_allocator_max_free_set(allocator, ap_max_mem_free);
	rv = apr_pool_create_ex(&wdp, pconf, NULL, allocator);
	if (rv != APR_SUCCESS) {
		ERRLOG0(pconf, APLOG_CRIT, DIVY_FST_INFO + DIVY_SST_PROC, "tf_watchdog: create pool");
		return rv;
	}
	apr_allocator_owner_set(allocator, wdp);
	apr_pool_tag(wdp, "tf_watchdog");
 
	wd = apr_pcalloc(wdp, sizeof(*wd));
	wd->s = s;
	wd->p = wdp;
	wd->cmdpath = TF_DEFAULT_JOB_PATH;
	wd->cmds = divy_array_make(wd->p, 3); /* 三個以上作られる予定はないかも */
	wd->count = 0;

	rv = divy_wd_get_instance(&wd->watchdog, TF_WATCHDOG_NAME, 0, 1, wd->p);
	if (APR_SUCCESS != rv) {
		return rv;
	}
 
	/* JOBSがあるフォルダから実行ファイルを取得します */
	rv = apr_dir_open(&job_dir, wd->cmdpath, wd->p);
	if (rv != APR_SUCCESS)
		return rv;

	while (apr_dir_read(&finfo, 0, job_dir) != APR_ENOENT) {
		if (finfo.filetype == APR_DIR) {
			/* . と .. を除外する */
			if (finfo.name[0] == '.'
					&& (finfo.name[1] == '\0'
						|| (finfo.name[1] == '.' && finfo.name[2] == '\0'))) {
				continue;
			}
		}
		divy_array_add(wd->cmds, (char*)finfo.name);
	}

	rv = apr_dir_close(job_dir);

	/*
	 * 定期的に実行する関数の登録
	 */
	rv = divy_wd_register_callback(wd->watchdog, 0, wd, divy_run_watchdog);

	return rv;

}

/*------------------------------------------------------------------------------
  Define private functions 
  ----------------------------------------------------------------------------*/
/**
 *	watchdogから呼び出されるコールバック関数
 *
 *	@param state	int	状態
 *	@param baton	void* データ
 *	@param ptemp	この関数内で利用できるプール
 *
 *	@return apr_status_t
 */
static apr_status_t divy_run_watchdog(int state, void *baton, apr_pool_t *ptemp)
{

	divy_watchdog_t *wd = baton;
	apr_time_t now, nextrun;
	int i;
	char *cmd;
	void **data;
	int exitcode = 0;
	apr_int32_t length = 0;
	apr_status_t rv;

	TRACE_S(wd->s)

	switch (state) {
		case AP_WATCHDOG_STATE_STARTING:
			ERRLOG0(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_PROC,
														"divy watchdog start");
			now = apr_time_now();
			nextrun = now + apr_time_from_sec(TF_JOB_DEFAULT_INTERVAL);
			divy_wd_set_interval(wd->watchdog, nextrun - now, wd, divy_run_watchdog);
			break;

		case AP_WATCHDOG_STATE_RUNNING:
			if ((wd->count++ % 60) == 0) {
				ERRLOG1(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_PROC,
				"divy watchdog running. %"APR_INT64_T_FMT" H", wd->count / 60);
			}
			if (wd->cmds) {

				length = divy_array_getlength(wd->cmds);
				data = divy_array_getelements(wd->cmds);

				for (i = 0; i < length; i++) {
					cmd = data[i];
					const char *pargv[] = {cmd, NULL};
					/* jobの実行 */
					rv = divy_execute_cmd(ptemp, cmd, pargv, NULL,
									APR_PROGRAM, wd->cmdpath, &exitcode, NULL);
					if (rv != APR_SUCCESS) {
						ERRLOG2(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
														"[%s] : result=[%d]", cmd ,rv);
					}
				}
			}
			break;

		case AP_WATCHDOG_STATE_STOPPING:
			ERRLOG0(NULL, APLOG_NOTICE, DIVY_FST_IERR + DIVY_SST_PROC, "divy watchdog stopping");
			break;
	}

	return APR_SUCCESS;
}

