/**
 * $Id$
 *
 * tf_memcache.c
 *
 * memcacheを利用する為の関数・構造体を宣言
 */
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_memcache.h"

#if APR_HAVE_UNISTD_H
#include <unistd.h>     /* for getpid */
#endif	/* APR_HAVE_UNISTD_H */

#include "ap_mpm.h"		/* ap_mpm_query */
#include "util_common.h"
#include "tfr.h"
#include "tf_cset.h"
#include "tf_memcache.h"
#include "tf_logger.h"
#include "tf_valuecache.h"

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

#define TF_MC_DEFAULT_SERVER	"127.0.0.1"
#define TF_MC_DEFAULT_PORT		11211
#define TF_MC_SERVER_MIN		0
#define TF_MC_SERVER_SMAX		1
#define TF_MC_SERVER_TTL		600

#define TF_MC_KEY_FORMAT		"%s:%s@%s"

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
/* apr_memcacheを隠蔽する不完全型 */
struct __divy_memcache_t {
	const char *myhostname;
	divy_cset_t *servers;
};

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * 新規でmemcacheの構造体を作成します
 * @param pool	apr_pool_t *
 * @@aram myhostname const char* 
 * @param memd  divy_memcache_t **
 */
DIVY_DECLARE(void) divy_memcache_create(apr_pool_t* cpool, const char* myhostname, divy_memcache_t** memd)
{
	divy_memcache_t *mt;

	mt = apr_pcalloc(cpool, sizeof(struct __divy_memcache_t));
	mt->myhostname = apr_pstrdup(cpool, myhostname);
	mt->servers = NULL;

	*memd = mt;
}

/**
 * memcachedサーバを登録します
 *
 * @param cpool		apr_pool_t* 		cmd pool
 * @param memd		divy_memcache_t *
 * @param hostport	const char* 		ホストとポートが連結された文字列
 * @return int 0: 成功 / 1:失敗
 */
DIVY_DECLARE(int) divy_memcache_add_server(apr_pool_t *cpool, divy_memcache_t *memd, const char* hostport)
{

	if (memd == NULL || IS_EMPTY(hostport)) return 1;

	if (memd->servers == NULL) {
		memd->servers = divy_cset_make(cpool);
	}

	divy_cset_set(memd->servers, hostport);

	return 0;
}

/**
 * 設定されたdivy_memcache_tの内容をもとにmemcachedとの接続を行います。
 * リトライを3回します。
 *
 * @param pool		apr_pool_t* 
 * @param  memd divy_memcache_t *
 * @param memcache apr_memcache_t**
 * @return apr_status_t
 */
DIVY_DECLARE(apr_status_t) divy_memcache_connect(apr_pool_t* pool, divy_memcache_t* memd, apr_memcache_t** memcache)
{
	divy_cset_index_t *ci;
	apr_status_t st = APR_SUCCESS;
	int thread_limit = 0;
	char *hostport = NULL;
	int retry = 0;
	char *host_str = NULL;
	char *scope_id = NULL;
	apr_memcache_server_t *server;
	apr_port_t port;
	int connected = 0;
	if (memd == NULL) return APR_EINIT;

	divy_memcache_open(pool, memd, memcache);

	ap_mpm_query(AP_MPMQ_HARD_LIMIT_THREADS, &thread_limit);

	do {
		for(ci = divy_cset_first(memd->servers); ci != NULL;
						ci = divy_cset_next(ci)) {
			divy_cset_this(ci, (const char **)&hostport);
			st = apr_parse_addr_port(&host_str, &scope_id, &port, hostport, pool);
			if (st != APR_SUCCESS) {
				return st;
			}

			/* デフォルト設定 */
			if (port == 0)
				port = TF_MC_DEFAULT_PORT;

			st = apr_memcache_server_create(pool, host_str, port,
														TF_MC_SERVER_MIN,
														TF_MC_SERVER_SMAX,
														thread_limit,
														TF_MC_SERVER_TTL,
														&server);
			if (st != APR_SUCCESS) {
				   ERRLOG1(pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_NET,
				   "Failed to connected memcache server. [%s]", hostport);
				return st;
			}

			st = apr_memcache_add_server(*memcache, server);
			if (st != APR_SUCCESS) {
				ERRLOG1(pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_NET,
		 		"Failed to connected memcache server. [%s]", hostport);
				return st;
			}

			connected = 1;	/* 接続しました */
		}

		retry++;

		if (retry > 2) break;

	} while (connected == 0);

	return st;
}


/**
 * 設定されたmemcacheサーバに対してデータを設定します
 */
DIVY_DECLARE(int) divy_memcache_set(apr_pool_t *pool, divy_memcache_t *memd,
									const char* prefix, const char* key,
									char* value, const apr_size_t value_size,
									apr_uint32_t timeout, apr_uint16_t flags)
{
	apr_status_t st;
	apr_memcache_t* memcache = NULL;

	if (memd == NULL || IS_EMPTY(key) || IS_EMPTY(value)) return 1;

	st = divy_memcache_connect(pool, memd, &memcache);
	if (st != APR_SUCCESS) return 1;	/* エラー */

	/*
	 * memcacheはグローバルな領域であるから、keyの投入はユニークな文字列
	 * を入れる必要があります。そうでなければ他のロケーションのデータを覗けて
	 * しまう可能性があります。そういう利用もできる
	 */
	char *memkey = apr_psprintf(pool, TF_MC_KEY_FORMAT,
										memd->myhostname,
										IS_FILLED(prefix)?prefix:"",
										key);


	st = apr_memcache_set(memcache, memkey, value, value_size, timeout, flags);
	if (st != APR_SUCCESS) 
		return 1;

	return 0;
}

/**
 * memcacheよりデータを取得します
 */
DIVY_DECLARE(apr_status_t) divy_memcache_get(apr_pool_t *pool,
									divy_memcache_t *memd, const char* prefix,
									const char* key, char **value,
									apr_size_t *length, apr_uint16_t *flags)
{
	apr_status_t st;
	apr_memcache_t *memcache = NULL;

	if (memd == NULL || IS_EMPTY(key)) return 1;

	char *memkey = apr_psprintf(pool, TF_MC_KEY_FORMAT,
										memd->myhostname,
										IS_FILLED(prefix)?prefix:"",
										key);

	if ((st = divy_memcache_connect(pool, memd, &memcache)) == APR_SUCCESS) {
		st = apr_memcache_getp(memcache, pool, memkey, value, length, flags);
	}

	return st;

}

/**
 * memcacheから指定のキーを削除する
 */
DIVY_DECLARE(apr_status_t) divy_memcache_delete(apr_pool_t *pool,
									 divy_memcache_t *memd, const char* prefix,
									 const char *key, apr_uint32_t timeout)
{
	apr_status_t st;
	apr_memcache_t *memcache = NULL;

	if (memd == NULL || IS_EMPTY(key)) return 1;

	char *memkey = apr_psprintf(pool, TF_MC_KEY_FORMAT,
										memd->myhostname,
										IS_FILLED(prefix)?prefix:"",
										key);

	if ((st = divy_memcache_connect(pool, memd, &memcache)) == APR_SUCCESS) {
		st = apr_memcache_delete(memcache, memkey, timeout);
	}

	return st;
}

/**
 * 設定されたdivy_memcache_tの内容をもとにmemcachedを構築します
 * @param cpool		apr_pool_t* 
 * @param  memd divy_memcache_t *
 * @param memcache apr_memcache_t**
 * @return int 0: 成功 / 1:失敗
 */
DIVY_DECLARE(int) divy_memcache_open(apr_pool_t* pool, divy_memcache_t *memd, apr_memcache_t **memcache)
{
	apr_status_t st;

	if (memd == NULL) return 1;

	int nserver = divy_cset_count(memd->servers);

	/* サーバーが一件もない場合は正常終了 */
	if (nserver == 0) return 0;

	st = apr_memcache_create(pool, nserver, 0, memcache);
	if (st != APR_SUCCESS) {
		return 1;
	}

	return 0;
}
