/**
 * $Id$
 *
 * LiveプロパティなどのXMLをパースする関数を集めたファイル
 * パース結果(xxx_iscreen) からrdbo オブジェクトへの変換する関数群も
 * 保持しています.
 */
#include "http_protocol.h"
#include "apr.h"
#include "apr_strings.h"
#include "apr_pools.h"

#include "mod_dav_tf.h"
#include "tf_rdbo.h"
#include "util.h"
#include "tf_folder.h"
#include "tf_parser.h"

/*------------------------------------------------------------------------------
  Declare private function 
  ----------------------------------------------------------------------------*/
static int _parse_one_keyvalue_elem(apr_pool_t *p,
					const apr_xml_elem *keyvalue_elem,
					divy_rdbo_sqlcontent *sqlcnt_pr);

static int _divy_parse_state_box(apr_pool_t *pool,
				const apr_xml_elem *elem, divy_resourcestate_iscreen *iscreen);

static int _divy_parse_upload_policy(apr_pool_t *pool,
				const apr_xml_elem *elem, divy_group_iscreen *iscreen);
/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * mailwatch プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_mailwatch_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_rdbo_mailwatch **mwatch_pr)
{
	const apr_xml_elem *elem1, *elem2;
	divy_rdbo_mailwatch *_mwatch_pr = NULL;
	char *all_method_str, *token, *method_cntx;
	int method_num;

	*mwatch_pr = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"mailwatch\"element MUST not be empty.");
		return 1;
	}

	/* 初期化（トリガは全て0に設定されます） */
	_mwatch_pr = apr_pcalloc(p, sizeof(divy_rdbo_mailwatch));
	memset(_mwatch_pr->trigger_methods, 0, METHODS);

	/*
	 * mailwatch エレメントの解析
	 * (note)
	 * <!ELEMENT mailwatch (watchowner, triggermethod, notification) >
	 */
	elem1 = elem->first_child;

	/*
	 * watchowner エレメントの解析
	 */
	if (strcmp(elem1->name, "watchowner") != 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"watchowner\"element MUST be specified.");
		return 1;
	}

	elem1 = elem1->next;	/* 次のエレメントに進める */
	if (elem1 == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"triggermethod\"element MUST be specified.");
		return 1;
	}

	/*
	 * triggermethod エレメントの解析
	 */
	/* PCDATA の両端に含まれていたスペースを除去して取得する */
	all_method_str = (char *) divy_xml_get_cdata(elem1, p, 1);
	if (IS_FILLED(all_method_str)) {
		/* メソッドをスペースでパースしてトリガフラグを設定する */
		while ((token = apr_strtok(all_method_str, " ", &method_cntx)) != NULL) {
			all_method_str = NULL;

			/* メソッド番号を求める */
			method_num = ap_method_number_of(token);
			if (method_num != M_INVALID) {
				_mwatch_pr->trigger_methods[method_num] = 1;
			}
			else {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The \"triggermethod\" cdata MUST be "
						"valid method. (name = %s)", method_cntx);
				return 1;
			}
		}
	}

	/*
	 * notification エレメントの解析
	 */
	elem1 = elem1->next;	/* 次のエレメントに進める */
	if (elem1 == NULL) {
		/* 古いクライアントはnotification を送信しないのでmailに倒す */
		_mwatch_pr->notification = DIVY_NOTICE_MAIL;
	}
	else {
		_mwatch_pr->notification = DIVY_NOTICE_UNKNOWN;

		for (elem2 = elem1->first_child; elem2; elem2 = elem2->next) {
			if (strcmp(elem2->name, "mail") == 0) {
				if (_mwatch_pr->notification == DIVY_NOTICE_UNKNOWN) {
					_mwatch_pr->notification = DIVY_NOTICE_MAIL;
				}
				else if (_mwatch_pr->notification == DIVY_NOTICE_MSG) {
					_mwatch_pr->notification = DIVY_NOTICE_BOTH;
				}
			}
			else if (strcmp(elem2->name, "msg") == 0) {
				if (_mwatch_pr->notification == DIVY_NOTICE_UNKNOWN) {
					_mwatch_pr->notification = DIVY_NOTICE_MSG;
				}
				else if (_mwatch_pr->notification == DIVY_NOTICE_MAIL) {
					_mwatch_pr->notification = DIVY_NOTICE_BOTH;
				}
			}
		}
	}

	*mwatch_pr = _mwatch_pr;
	return 0;
}

/**
 * sharedcollection プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_sharedcollection_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_rdbo_shlink **shlink_pr)
{
	const apr_xml_elem *elem1, *elem2;
	divy_rdbo_shlink *first = NULL, *_shlink_pr = NULL;

	*shlink_pr = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sharedcollection\"element MUST not be empty.");
		return 1;
	}

	for (elem1 = elem->first_child; elem1; elem1 = elem1->next) {
		/* shlink_pr の作成 */
		if (first == NULL) {
			_shlink_pr = apr_pcalloc(p, sizeof(divy_rdbo_shlink));
			first = _shlink_pr;	/* 先頭を記録しておく */
		}
		else {
			_shlink_pr->next = apr_pcalloc(p, sizeof(divy_rdbo_shlink));
			_shlink_pr = _shlink_pr->next;
		}

		/* sharedcollectioninfo は存在するか？ */
		if (strcmp(elem1->name, "sharedcollectioninfo")) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"sharedcollectioninfo\"element "
				"MUST be specified.");
			 return 1;
		}

		/* sharedcollectioninfo エレメントの下にエレメントがあるのか */
		if (elem1->first_child == NULL) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"sharedcollectioninfo\"element "
				"MUST not be empty.");
			 return 1;
		}

		/* shname タグ */
		elem2 = elem1->first_child;
		if (IS_FILLED(elem2->name)) {
			_shlink_pr->displayname = (char *) divy_xml_get_cdata(elem2, p, 1);
		}

		/* linkuri タグ */
		elem2 = elem1->first_child->next;
		if (elem2 != NULL && IS_FILLED(elem2->name)) {
			_shlink_pr->uri = (char *) divy_xml_get_cdata(elem2, p, 1);

			/* uri の要素から２重スラッシュと../や./を解決する */
			ap_no2slash(_shlink_pr->uri);
			ap_getparents(_shlink_pr->uri);

			/* uri の末尾からスラッシュと取り除く */
			_shlink_pr->uri = dav_divy_remove_endslash(p, _shlink_pr->uri);
		}
	}

	*shlink_pr = first;
	return 0;
}

/**
 * updatediscovery プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_updatediscovery_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_rdbo_clmodule **clmodule_pr)
{
	const apr_xml_elem *updateinfo_elem, *child_elem;
	divy_rdbo_clmodule *_clmodule_pr;
	char *data;

	*clmodule_pr = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"updatediscovery\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	_clmodule_pr = apr_pcalloc(p, sizeof(divy_rdbo_clmodule));

	/*
	 * updateinfo タグの解析
	 * (note)
	 * <!ELEMENT updatediscovery (updateinfo) >
	 * <!ELEMENT updateinfo (name, version, lineup, digest,
	 * 					creationdt, updatedt) >
	 */
	updateinfo_elem = elem->first_child;
	/* updateinfo タグの中身が無かった */
	if (updateinfo_elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"updateinfo\"element MUST not be empty.");
		return 1;
	}


	/* updateinfo エレメントの中にあるエレメントを全て解析する */
	for (child_elem = updateinfo_elem->first_child; child_elem;
					child_elem = child_elem->next) {

		/* 文字列データを取得する */
		data = (char *) divy_xml_get_cdata(child_elem, p, 1);

		if (strcmp(child_elem->name, "name") == 0) {
			_clmodule_pr->name = data;
		}
		else if (strcmp(child_elem->name, "version") == 0) {
			_clmodule_pr->version = data;
		}
		else if (strcmp(child_elem->name, "lineup") == 0) {
			_clmodule_pr->lineup = data;
		}
		else if (strcmp(child_elem->name, "digest") == 0) {
			_clmodule_pr->digest = data;
		}
		else if (strcmp(child_elem->name, "creationdt") == 0) {
			_clmodule_pr->registdt = data;
		}
		else if (strcmp(child_elem->name, "updatedt") == 0) {
			_clmodule_pr->updatedt = data;
		}
		else {
			/* 未知のエレメントが指定された(でも無視) */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"updateinfo", child_elem->name);
		}
	}

	*clmodule_pr = _clmodule_pr;
	return 0;
}

/**
 * keylist プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_keylist_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_rdbo_keylist **keylist_pr)
{
	const apr_xml_elem *key_elem;
	apr_xml_attr *attr;
	divy_rdbo_keylist *_keylist_pr = NULL;

	*keylist_pr = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"keylist\"element MUST not be empty.");
		return 1;
	}

	/*
	 * key タグの解析
	 * (note)
	 * <!ELEMENT keylist (key*) >
	 * <!ELEMENT key (#PCDATA)
	 * <!ATTLIST key id ID #REQUIRD >
	 */
	for (key_elem = elem->first_child; key_elem; key_elem = key_elem->next) {

		/* 未知のエレメントが指定された(でも無視) */
		if (strcmp(key_elem->name, "key")) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified "
				"in the element of \"%s\"."
				"(elem->name = %s). We ignore this.",
				"keylist", key_elem->name);
			continue;
		}

		if (*keylist_pr == NULL) {
			_keylist_pr = apr_pcalloc(p, sizeof(divy_rdbo_keylist));
			 *keylist_pr = _keylist_pr;
		}
		else {
			_keylist_pr->next = apr_pcalloc(p, sizeof(divy_rdbo_keylist));
			_keylist_pr       = _keylist_pr->next;
		}
	

		/* 属性値の取得 */
		attr = key_elem->attr;
		if (attr == NULL || attr->value == NULL) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"id\" attribute of \"key\" element "
				"is missing. Must be specified.");
			return 1;
		}
		/* 値は数値か? */
		if (!dav_divy_isdigit_str(attr->value)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"id\" attribute value of \"key\" "
				"MUST be digit.(data = %s)", attr->value);
			return 1;
		}

		/* 値の設定 */
		_keylist_pr->index = atoi(attr->value);
		_keylist_pr->key   = (char *) divy_xml_get_cdata(key_elem, p, 1);
	}

	return 0;
}


/**
 * userdiscovery プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_user_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_user_iscreen **user_iscreen)
{
	const apr_xml_elem *userinfo_elem, *child;
	divy_user_iscreen *iscreen;
	char *data;
	int new_client_found = 0;

	*user_iscreen = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"userdiscovery\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	iscreen = apr_pcalloc(p, sizeof(divy_user_iscreen));
	iscreen->adminmode = DIVY_ADMINMODE_NORMAL;	/* EMPTY タグなので初期化が必要 */
	iscreen->extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_USR);	/* デフォルトステータス */
	iscreen->is_old_client = 0;	/* 新しいクライアントであるとまずは仮定する */

	/*
	 * userinfo タグの解析
	 *
	 * [ DTD ]
	 * <!ELEMENT userdiscovery (userinfo) >
	 * <!ELEMENT userinfo (userid, name, mailaddr?, creationdt?, updatedt?,
	 * 			password?, admin?, usedstorage?, maxstorage,
	 * 			usedresource?, maxresource, accessdeny?, comment?,
	 * 			state?, expiration?, user-privilege?, usertype?
	 * 			maxusercreation?, ownername?, ownerid?, otheruser?, appointed-groupleader? allowhosts?) >
	 *
	 * <!ELEMENT state          ((active | inactive), expired?) >
	 * <!ELEMENT user-privilege ((read | upload | readwrite),
	 *                          set-view?, group-constraints-ignore?, control-other-user?) >
	 * <!ELEMENT usertype       (admin | normal | limited | groupleader) >
	 */
	userinfo_elem = elem->first_child;
	/* userinfo タグの中身が無かった */
	if (userinfo_elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"userinfo\"element MUST not be empty.");
		return 1;
	}

	/* userinfo エレメントの中にあるエレメントを全て解析する */
	for (child = userinfo_elem->first_child; child; child = child->next) {

		/* PCDATA の両端に含まれていたスペースを除去して取得する */
		if (strcmp(child->name, "password") == 0) {
			data = (char *) divy_xml_get_cdata(child, p, 0); /* 両端のスペースは取らない */
		}
		else {
			data = (char *) divy_xml_get_cdata(child, p, 1);
		}

		if (strcmp(child->name, "userid") == 0) {
			iscreen->usrid = data;
		}
		else if (strcmp(child->name, "name") == 0) {
			iscreen->fullname = data;
		}
		else if (strcmp(child->name, "mailaddr") == 0) {
			iscreen->mailaddr = data;
		}
		else if (strcmp(child->name, "password") == 0) {
			iscreen->password = data;
		}
		else if (strcmp(child->name, "admin") == 0) {
			/* 互換性維持のために残してあります */
			iscreen->adminmode = DIVY_ADMINMODE_ADMIN;
		}
		else if (strcmp(child->name, "maxstorage") == 0) {
			iscreen->smaxst = data;
		}
		else if (strcmp(child->name, "maxresource") == 0) {
			iscreen->smaxres = data;
		}
		else if (strcmp(child->name, "accessdeny") == 0) {
			char *token, *token_ctx, *orig_data;
			const char *deny_name;
			int i, found = 0;

			/* 空タグが指定された */
			if (IS_EMPTY(data)) {
				iscreen->accessdeny = NULL;	/* なかった */
				continue;
			}
			orig_data = apr_pstrdup(p, data);

			/* accessdeny 配列を作成 */
			iscreen->accessdeny = apr_pcalloc(p, sizeof(int) * DIVY_FOLDER_ID_END);
			
			/* 識別文字列をスペースでパースしてフラグを立てる */
			while ((token = apr_strtok(data, " ", &token_ctx)) != NULL) {
				data = NULL;
				
				/* 名称からフォルダIDを求める */
				for (i = 0; i < DIVY_FOLDER_ID_END; i++) {
					deny_name = divy_get_sfolder_accessdeny_name(p,i);
					if (IS_FILLED(deny_name) &&
					    strcmp(token, deny_name) == 0) {
						/* フラグを立てる */
						iscreen->accessdeny[i] = 1;
						found = 1;
						break;
					}
				}
			}

			/* 何らかの値が指定されていたようだが、それが全て正しくなかった */
			if (found == 0) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The value of \"accessdeny\"(\"userdiscovery\" child) "
					"element is invalid.(data = %s)", orig_data);
				return 1;
			}
		}
		else if (strcmp(child->name, "creationdt") == 0) {
			iscreen->registdt = data;
		}
		else if (strcmp(child->name, "updatedt") == 0) {
			iscreen->updatedt = data;
		}
		else if (strcmp(child->name, "usedstorage") == 0) {
			iscreen->susedst = data;
		}
		else if (strcmp(child->name, "usedresource") == 0) {
			iscreen->susedres = data;
		}
		else if (strcmp(child->name, "comment") == 0) {
			iscreen->comment = data;
		}
		/* (note) ユーザ拡張ステータスサポートの有無に関わらずパースはしておきます */
		else if (strcmp(child->name, "state") == 0 && child->first_child) {
			apr_xml_elem *state_child = child->first_child;

			for (; state_child; state_child = state_child->next) {
				if (strcmp(state_child->name, "active") == 0) {
					divy_rdbo_set_active_user(iscreen->extstatus, 1);
				}
				else if (strcmp(state_child->name, "inactive") == 0) {
					divy_rdbo_set_active_user(iscreen->extstatus, 0);
				}
				else if (strcmp(state_child->name, "expired") == 0) {
					/* 無視します(大目に見るということ) */
				}
				else {
					ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
						"The child of \"state\" element is invalid."
						"(name = %s)", state_child->name);
					return 1;
				}
			}
			new_client_found = 1;	/* 新しいクライアントだった */
		}
		else if (strcmp(child->name, "expiration") == 0) {
			iscreen->sexpiration = data;
			new_client_found = 1;	/* 新しいクライアントだった */
		}
		else if (strcmp(child->name, "user-privilege") == 0 && child->first_child) {
			apr_xml_elem *privilege_child = child->first_child;

			/* 権限フラグだけをステータスをリセットする */
			divy_rdbo_reset_userprivilege(iscreen->extstatus);

			/* readwrite / upload / read 権限 */
			for (; privilege_child; privilege_child = privilege_child->next) {

				if (strcmp(privilege_child->name, "readwrite") == 0) {
					divy_rdbo_set_readwrite_privilege(iscreen->extstatus, 1);
				}
				else if (strcmp(privilege_child->name, "upload") == 0) {
					divy_rdbo_set_upload_privilege(iscreen->extstatus, 1);
				}
				else if (strcmp(privilege_child->name, "read") == 0) {
					divy_rdbo_set_read_privilege(iscreen->extstatus, 1);
				}
				else if (strcmp(privilege_child->name, "set-view") == 0) {
					divy_rdbo_set_setview_privilege(iscreen->extstatus, 1);
				}
				else if (strcmp(privilege_child->name, "group-constraints-ignore") == 0) {
					divy_rdbo_set_user_groupconstraints_ignore(iscreen->extstatus, 1);
				}
				else if (strcmp(privilege_child->name, "control-other-user") == 0) {
					divy_rdbo_set_control_otheruser(iscreen->extstatus, 1);
				}
				else {
					ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
						"The child of \"user-privilege\" element is invalid."
						"(name = %s)", privilege_child->name);
					return 1;
				}
			}
			new_client_found = 1;	/* 新しいクライアントだった */
		}
		else if (strcmp(child->name, "usertype") == 0 && child->first_child) {
			if (strcmp(child->first_child->name, "admin") == 0) {
				divy_rdbo_set_trusty_user(iscreen->extstatus, 1);
				iscreen->adminmode = DIVY_ADMINMODE_ADMIN;
			}
			else if (strcmp(child->first_child->name, "normal") == 0) {
				divy_rdbo_set_trusty_user(iscreen->extstatus, 1);
				iscreen->adminmode = DIVY_ADMINMODE_NORMAL;
			}
			else if (strcmp(child->first_child->name, "limited") == 0) {
				divy_rdbo_set_trusty_user(iscreen->extstatus, 0);
				iscreen->adminmode = DIVY_ADMINMODE_NORMAL;
			}
			else if (strcmp(child->first_child->name, "groupleader") == 0) {
				divy_rdbo_set_groupleader(iscreen->extstatus, 1);
				iscreen->adminmode = DIVY_ADMINMODE_NORMAL;
			}
			else {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The child of \"usertype\" element is invalid."
					"(name = %s)", child->first_child->name);
				return 1;
			}
			new_client_found = 1;	/* 新しいクライアントだった */
		}
		else if (strcmp(child->name, "maxusercreation") == 0) {
			iscreen->smaxusercreation = data;
		}
		else if (strcmp(child->name, "allowhosts") == 0) {
			iscreen->allowhosts = data;
		}
		else {
			/* 未知のエレメントが指定された(でも無視) */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"userinfo", child->name);
		}
	}

	/* 新たに定義されたエレメントが少なくとも1つは存在しなかった */
	if (!new_client_found) {
		iscreen->is_old_client = 1;	/* 古いクライアントです(ユーザ拡張ステータス以前のクライアント) */
	}

	*user_iscreen = iscreen;
	return 0;
}

/**
 * groupdiscovery プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_group_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_group_iscreen **group_iscreen)
{
	const apr_xml_elem *grpinfo_elem, *child;
	divy_group_iscreen *iscreen;

	*group_iscreen = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"groupdiscovery\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	iscreen = apr_pcalloc(p, sizeof(divy_group_iscreen));
	iscreen->grp_extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_GRP);	/* デフォルト */
	iscreen->upload_policy = apr_pcalloc(p, sizeof(divy_rdbo_upload_policy)); 
	/*
	 * groupinfo タグの解析
	 *
	 * [ DTD ]
	 * <!ELEMENT groupdiscovery       (groupinfo) >
	 * <!ELEMENT groupinfo            (groupid?, name?, creationdt?, updatedt?, comment?, groupmailwatch?,
	 *                                 groupcolletionuri, constraints, ownername?, ownerid?, state?, box?, uploadpolicyrule?, groupquota?) >
	 *
	 * <!ELEMENT constraints (write?, operationlog?, showproperty?) >
	 * <!ELEMENT state      (active | inactive) >
	 * <!ELEMENT box         (active | inactive) >
	 * <!ELEMENT uploadpolicyrule(filename*, suffix*, length?, chartype?) >
	 */
	grpinfo_elem = elem->first_child;
	/* groupinfo タグの中身が無かった */
	if (grpinfo_elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"groupinfo\"element MUST not be empty.");
		return 1;
	}

	/* １つのgroupinfo エレメントをパース */
	for (child = grpinfo_elem->first_child; child; child = child->next) {

		/* PCDATA の両端に含まれていたスペースを除去して取得する */

		if (strcmp(child->name, "groupid") == 0) {
			iscreen->grpid = (char *) divy_xml_get_cdata(child, p, 1);
		}
		else if (strcmp(child->name, "name") == 0) {
			iscreen->name = (char *) divy_xml_get_cdata(child, p, 1);
		}
		else if (strcmp(child->name, "comment") == 0) {
			/* 前後の空白記号は除去しない */
			iscreen->comment = (char *) divy_xml_get_cdata(child, p, 0);
		}
		else if (strcmp(child->name, "creationdt") == 0) {
			iscreen->registdt = (char *) divy_xml_get_cdata(child, p, 1);
		}
		else if (strcmp(child->name, "updatedt") == 0) {
			iscreen->updatedt = (char *) divy_xml_get_cdata(child, p, 1);
		}
		else if (strcmp(child->name, "groupmailwatch") == 0) {
			divy_rdbo_mailwatch *mw = NULL;
			char *all_method_str, *token, *method_cntx;
			int method_num;	
			const apr_xml_elem *elem1, *ntc_elem, *elem2;

			elem1 = child->first_child;	/* triggermethod / notification */

			/* groupmailwatch がEMPTYかどうかを判定 */
			if (elem1 == NULL) {
				/* NULL にしてしまう */
				iscreen->mailwatch_pr = NULL;
				continue;
			}
			else if (strcmp(elem1->name, "triggermethod") != 0) {
				/* groupmailwatch がEMPTYでなければ、triggermethod が
				 * なければならない */
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The content of \'groupmailwatch\' is EMPTY."
					"if you set groupmailwatch element, Must set "
					"\"triggermehod\" element.");
				return 1;
			}

			mw = apr_pcalloc(p, sizeof(divy_rdbo_mailwatch));
			memset(mw->trigger_methods, 0, METHODS);

			/*
			 * triggermethod エレメントの解析
			 */

			/* PCDATA の両端に含まれていたスペースを除去して取得する */
			all_method_str = (char *) divy_xml_get_cdata(elem1, p, 1);
			if (IS_FILLED(all_method_str)) {
				/* メソッドをスペースでパースしてトリガを設定する */
				while ((token = apr_strtok(all_method_str, " ", &method_cntx)) != NULL) {
					all_method_str = NULL;

					/* メソッド番号を求める */
					method_num = ap_method_number_of(token);
					if (method_num != M_INVALID) {
						mw->trigger_methods[method_num] = 1;
					}
					else {
						ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
							"The \"triggermethod\" must contain valid method. "
							"(name = %s)", token);
						return 1;
					}
				}
			}

			/*
			 * notification エレメントの解析
			 */
			ntc_elem = elem1->next;	/* 次のエレメントに進める */
			if (ntc_elem == NULL) {
				/* 古いクライアントはnotification 
				 * を送信しないのでmailに倒す */
				mw->notification = DIVY_NOTICE_MAIL;
			}
			else {
				mw->notification = DIVY_NOTICE_UNKNOWN;

				for (elem2 = ntc_elem->first_child; elem2; elem2 = elem2->next) {
					if (strcmp(elem2->name, "mail") == 0) {
						if (mw->notification == DIVY_NOTICE_UNKNOWN) {
							mw->notification = DIVY_NOTICE_MAIL;
						}
						else if (mw->notification == DIVY_NOTICE_MSG) {
							mw->notification = DIVY_NOTICE_BOTH;
						}
					}
					else if (strcmp(elem2->name, "msg") == 0) {
						if (mw->notification == DIVY_NOTICE_UNKNOWN) {
							mw->notification = DIVY_NOTICE_MSG;
						}
						else if (mw->notification == DIVY_NOTICE_MAIL) {
							mw->notification = DIVY_NOTICE_BOTH;
						}
					}
				}
			}
		       	iscreen->mailwatch_pr = mw;
		}
		else if (strcmp(child->name, "groupcollectionuri") == 0) {
			/* 本来はエラーなのだが、後付けなので寛容にあつかいます。
			 * という訳で無視します */
		}
		else if (strcmp(child->name, "constraints") == 0) {
			apr_xml_elem *constraints_child = child->first_child;

			for (; constraints_child; constraints_child = constraints_child->next) {

				if (strcmp(constraints_child->name, "write") == 0) {
					divy_rdbo_set_write_groupconstraints(iscreen->grp_extstatus, 1);
				}
				else if (strcmp(constraints_child->name, "operationlog") == 0) {
					divy_rdbo_set_oplog_groupconstraints(iscreen->grp_extstatus, 1);
				}
				else if (strcmp(constraints_child->name, "showproperty") == 0) {
					divy_rdbo_set_showprop_groupconstraints(iscreen->grp_extstatus, 1);
				}
				else {
					ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
						"The child of \"constraints\" element is invalid."
						"(name = %s)", constraints_child->name);
					/* 無視します(今後の拡張のため) */
				}
			}
		}
		else if (strcmp(child->name, "state") == 0) {
			apr_xml_elem *state_child = child->first_child;

			if (state_child != NULL) {
				if (strcmp(state_child->name, "active") == 0) {
					divy_rdbo_set_active_group(iscreen->grp_extstatus, 1);
				}
				else if (strcmp(state_child->name, "inactive") == 0) {
					divy_rdbo_set_active_group(iscreen->grp_extstatus, 0);
				}
				else {
					ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
							"The child of \"state\" element is invalid."
							"(name = %s)", state_child->name);
					/* 無視します(今後の拡張のため) */
				}
			}
		}
		else if (strcmp(child->name, "box") == 0) {
			apr_xml_elem *box_child = child->first_child;

			/* BOX機能が有効となるグループは強制的に
			 * 他の制約の設定を受けます
			 */
			if (box_child != NULL) {
				if (strcmp(box_child->name, "active") == 0) {
					divy_rdbo_set_box_group(iscreen->grp_extstatus, 1);
					/* 書き込み制約はOff */
					divy_rdbo_set_write_groupconstraints(iscreen->grp_extstatus, 0);
					/* ログ制約はOn */
					divy_rdbo_set_oplog_groupconstraints(iscreen->grp_extstatus, 1);
					/* プロパティ制約はOn */
					divy_rdbo_set_showprop_groupconstraints(iscreen->grp_extstatus, 1);
				}
				else if (strcmp(box_child->name, "inactive") == 0) {
					divy_rdbo_set_box_group(iscreen->grp_extstatus, 0);
					/* 書き込み制約はOff */
					divy_rdbo_set_write_groupconstraints(iscreen->grp_extstatus, 0);
					/* ログ制約はOff */
					divy_rdbo_set_oplog_groupconstraints(iscreen->grp_extstatus, 0);
					/* プロパティ制約はOff */
					divy_rdbo_set_showprop_groupconstraints(iscreen->grp_extstatus, 0);
				}
				else {
					ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
							"The child of \"box\" element is invalid."
							"(name = %s)", box_child->name);
					/* 無視します(今後の拡張のため) */
				}
			}
		}
		/* アップロードポリシールール */
		else if (strcmp(child->name, "uploadpolicyrule") == 0) {
			if (_divy_parse_upload_policy(p, child, iscreen)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
						"The child of \"uploadpolicyrule\" is invalid.");
				/* パースが失敗 */
				return 1;
			}
		}
		else if (strcmp(child->name, "groupquota") == 0) {
			apr_xml_elem *quota_child = child->first_child;

			if (quota_child != NULL) {
#ifdef DIVY_SUPPORT_GROUPQUOTA
				char *data;
				divy_rdbo_grpquota *grp_q = apr_pcalloc(p, sizeof(divy_rdbo_grpquota));

				for (; quota_child; quota_child = quota_child->next) {

					/* PCDATA の両端に含まれていたスペースを除去して取得 */
					data = (char *) divy_xml_get_cdata(quota_child, p, 1);

					if (strcmp(quota_child->name, "used-bytes") == 0) {
						grp_q->usedst = data ? apr_atoi64(data) : APR_INT64_C(0);
					}
					else if (strcmp(quota_child->name, "available-bytes") == 0) {
						grp_q->maxst = data ? apr_atoi64(data) : APR_INT64_C(0);
					}
					else if (strcmp(quota_child->name, "used-files") == 0) {
						grp_q->usedres = data ? apr_atoi64(data) : APR_INT64_C(0);
					}
					else if (strcmp(quota_child->name, "available-files") == 0) {
						grp_q->maxres = data ? apr_atoi64(data) : APR_INT64_C(0);
					}
					else {
						ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
								"The Child of \"groupquota\" element is invalid."
								"(name = %s)", quota_child->name);
						/* 無視します (今後の拡張の為) */
					}

					iscreen->grp_q = grp_q;
				}
#endif /* DIVY_SUPPORT_GROUPQUOTA */
			}
		}
		else {
			/* 未知のエレメントが指定された(でも無視) */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Invalid element is specified in the element "
					"of \"%s\".(elem->name = %s). We ignore this.",
					"groupinfo", child->name);
		}
	}

	*group_iscreen = iscreen;
	return 0;
}

/**
 * sqldiscovery プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_sql_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_rdbo_sql **sql_pr)
{
	const apr_xml_elem *sqlinfo_elem, *child;
	divy_rdbo_sql *_sql_pr          = NULL;
	divy_rdbo_sqlcontent *sqlcnt_pr = NULL;
	int found_sqltype = 0;

	*sql_pr = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sqldiscovery\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	_sql_pr = apr_pcalloc(p, sizeof(divy_rdbo_sql));

	/* inactiveの初期化 */
	_sql_pr->active = DIVY_SQL_ACTIVE;

	/*
	 * sqlinfo タグの解析
	 * (note)
	 * <!ELEMENT sqldiscovery (sqlinfo) >
	 * <!ELEMENT sqlinfo (sqlno?, name?, subname?, sqltype?, inactivesql?, statement?,
	 * 			usedbmsname?, dbmslist?, creationdt?, updatedt?, comment?,
	 * 			keyvalue*, displaycontent?, inheritsql?) >
	 */
	sqlinfo_elem = elem->first_child;
	/* sqlinfo タグの中身が無かった */
	if (sqlinfo_elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sqlinfo\"element MUST not be empty.");
		return 1;
	}

	_sql_pr->cachemode = DIVY_SQL_NOCACHE;	/* 初期化 */
	
	/* sqlinfo エレメントの中にあるエレメントを全て解析する */
	for (child = sqlinfo_elem->first_child; child; child = child->next) {

		if (strcmp(child->name, "sqlno") == 0) {
			_sql_pr->sqlid = (char *) divy_xml_get_cdata(child, p, 1);
		}
		else if (strcmp(child->name, "name") == 0) {
			_sql_pr->labelname = (char *) divy_xml_get_cdata(child, p, 1);
		}
		else if (strcmp(child->name, "subname") == 0) {
			_sql_pr->subname = (char *) divy_xml_get_cdata(child, p, 1);
		}
		else if (strcmp(child->name, "sqltype") == 0) {
			const char *typename = NULL;
			found_sqltype = 1;
			if (child->first_child && child->first_child->name) {
				typename = child->first_child->name;
				if (strcmp(typename, "normalsql") == 0) {
					_sql_pr->type = DIVY_SQL_TYPE_NORMAL; 
				}
				else if (strcmp(typename, "reposdbsql") == 0) {
					_sql_pr->type = DIVY_SQL_TYPE_REPOSITORY;
				}
				else if (strcmp(typename, "reqsql") == 0) {
					_sql_pr->type = DIVY_SQL_TYPE_REQUIRED;
				}
				else if (strcmp(typename, "namedbind") == 0) {
					_sql_pr->type = DIVY_SQL_TYPE_BIND;
				}
				else {
					/* 不正なタグが指定された */
					ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The child of \"sqltype\" is invalid."
						"(name = %s)", typename);
					return 1;
				}
			}
			else {
				/* 値が無かった */
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The \"sqltype\" element must be specified.");
				return 1;
			}
		}
		else if (strcmp(child->name, "cachemode") == 0) {
			if (child->first_child && 
					strcmp(child->first_child->name, 
						"cache") == 0) {
				_sql_pr->cachemode = DIVY_SQL_CACHE; 
			}
		}
		else if (strcmp(child->name, "inactivesql") == 0) {
			_sql_pr->active = DIVY_SQL_INACTIVE;
		}
		else if (strcmp(child->name, "statement") == 0) {
			_sql_pr->sql = (char *) divy_xml_get_cdata(child, p, 0);
		}
		else if (strcmp(child->name, "usedbmsname") == 0) {
			_sql_pr->usedbmsname = (char *) divy_xml_get_cdata(child, p, 1);
		}
		else if (strcmp(child->name, "dbmslist") == 0) {
			if (child->first_child == NULL) {
				_sql_pr->dbms_pr = NULL;
			}
			else {
				const apr_xml_elem *dbms_elm;
				divy_rdbo_dbms *first;
				for (dbms_elm = child->first_child; dbms_elm;
							dbms_elm = dbms_elm->next) {
					if (_sql_pr->dbms_pr == NULL) {
						first = _sql_pr->dbms_pr =
							apr_pcalloc(p, sizeof(divy_rdbo_dbms));
					}
					else {
						_sql_pr->dbms_pr->next =
							apr_pcalloc(p, sizeof(divy_rdbo_dbms));
						_sql_pr->dbms_pr = _sql_pr->dbms_pr->next;
					}
					_sql_pr->dbms_pr->name = (char *)
							divy_xml_get_cdata(dbms_elm, p, 1);
					_sql_pr->dbms_pr->next = NULL;
				}
			}
		}
		else if (strcmp(child->name, "comment") == 0) {
			/* 空白記号は取り除かない */
			_sql_pr->comment = (char *) divy_xml_get_cdata(child, p, 0);
		}
		else if (strcmp(child->name, "keyvalue") == 0) {

			/* sqlcnt_pr を作成する */
			if (_sql_pr->sqlcnt_pr == NULL) {
				sqlcnt_pr = apr_pcalloc(p, sizeof(divy_rdbo_sqlcontent));
				_sql_pr->sqlcnt_pr = sqlcnt_pr;
			}
			else {
				sqlcnt_pr->next = apr_pcalloc(p, sizeof(divy_rdbo_sqlcontent));
				sqlcnt_pr = sqlcnt_pr->next;
			}

			/* 1つのkeyvalue エレメントをパースする */
			if (_parse_one_keyvalue_elem(p, child, sqlcnt_pr)) {
				return 1;
			}
		}
		else if (strcmp(child->name, "creationdt") == 0) {
			_sql_pr->registdt = (char *) divy_xml_get_cdata(child, p, 1);
		}
		else if (strcmp(child->name, "updatedt") == 0) {
			_sql_pr->updatedt = (char *) divy_xml_get_cdata(child, p, 1);
		}
		else {
			/* 未知のエレメントが指定された(でも無視) */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"sqlinfo", child->name);
		}
	}

	/* sqltype が存在するかどうか */
	if (found_sqltype == 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sqltype\" element must be specified.");
		return 1;
	}

	*sql_pr = _sql_pr;
	return 0;
}

/**
 * sysmsgdiscovery プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_sysmsg_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_sysmsg_iscreen **sysmsg_iscreen)
{
	const apr_xml_elem *sysmsginfo_elem, *child_elem;
	divy_sysmsg_iscreen *iscreen;

	*sysmsg_iscreen = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sysmsgdiscovery\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	iscreen = apr_pcalloc(p, sizeof(divy_sysmsg_iscreen));
	iscreen->active = 0;	/* デフォルト値(非アクティブ) */

	/*
	 * sysmsginfo タグの解析
	 * subjectとlangとscopeは拡張メッセージで利用する
	 *
	 * (note)
	 * <!ELEMENT sysmsgdiscovery (sysmsginfo) >
	 * <!ELEMENT sysmsginfo (msgid, msgtxt, subject ,lang, scope, active, creationdt, updatedt) >
	 */
	sysmsginfo_elem = elem->first_child;
	/* sysmsginfo タグの中身が無かった */
	if (sysmsginfo_elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sysmsginfo\"element MUST not be empty.");
		return 1;
	}


	/* sysmsginfo エレメントの中にあるエレメントを全て解析する */
	for (child_elem = sysmsginfo_elem->first_child; child_elem; child_elem = child_elem->next) {

		if (strcmp(child_elem->name, "msgid") == 0) {
			iscreen->msgid = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else if (strcmp(child_elem->name, "msgtxt") == 0) {
			/* 前後の空白記号は取ってはならない */
			iscreen->msg = (char *) divy_xml_get_cdata(child_elem, p, 0);
		} 
		else if (strcmp(child_elem->name, "subject") == 0) {
			/* 前後の空白記号は取ってはならない */
			iscreen->subject = (char *) divy_xml_get_cdata(child_elem, p, 0);
		}
		else if (strcmp(child_elem->name, "lang") == 0) {
			iscreen->lang = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}	
		else if (strcmp(child_elem->name, "scope") == 0) {
			iscreen->scope = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}	
		else if (strcmp(child_elem->name, "active") == 0) {
			iscreen->active = 1;
		}
		else if (strcmp(child_elem->name, "creationdt") == 0) {
			iscreen->registdt = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else if (strcmp(child_elem->name, "updatedt") == 0) {
			iscreen->updatedt = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else {
			/* 未知のエレメントが指定された(でも無視) */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"sysmsginfo", child_elem->name);
		}
	}

	*sysmsg_iscreen = iscreen;
	return 0;
}

/**
 * dbmsdiscovery プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_dbms_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_dbms_iscreen **dbms_iscreen)
{
	const apr_xml_elem *dbmsinfo_elem, *child_elem;
	divy_dbms_iscreen *iscreen;

	*dbms_iscreen = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"dbmsdiscovery\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	iscreen = apr_pcalloc(p, sizeof(divy_dbms_iscreen));
	iscreen->active = DIVY_DBMS_INACTIVE;	/* デフォルト値(非アクティブ) */

	/*
	 * dbmsinfo タグの解析
	 * (note)
	 * <!ELEMENT dbmsdiscovery (dbmsinfo) >
	 * <!ELEMENT dbmsinfo (dbmsid, name, type?, hostname?, hostport?, dbname?,
	 *                     username?, password?, comment?, active?,
	 *                     creationdt?, updatedt?) >
	 */
	dbmsinfo_elem = elem->first_child;
	/* dbmsinfo タグの中身が無かった */
	if (dbmsinfo_elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"dbmsinfo\"element MUST not be empty.");
		return 1;
	}


	/* dbmsinfo エレメントの中にあるエレメントを全て解析する */
	for (child_elem = dbmsinfo_elem->first_child; child_elem; child_elem = child_elem->next) {

		if (strcmp(child_elem->name, "dbmsid") == 0) {
			iscreen->dbmsid = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else if (strcmp(child_elem->name, "name") == 0) {
			iscreen->name = (char *) divy_xml_get_cdata(child_elem, p, 1);
		} 
		else if (strcmp(child_elem->name, "type") == 0) {
			iscreen->type = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else if (strcmp(child_elem->name, "hostname") == 0) {
			iscreen->hostname = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else if (strcmp(child_elem->name, "hostport") == 0) {
			iscreen->sport = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else if (strcmp(child_elem->name, "dbname") == 0) {
			iscreen->dbname = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else if (strcmp(child_elem->name, "username") == 0) {
			iscreen->username = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else if (strcmp(child_elem->name, "password") == 0) {
			iscreen->password = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else if (strcmp(child_elem->name, "comment") == 0) {
			/* 前後の空白記号は取ってはならない */
			iscreen->comment = (char *) divy_xml_get_cdata(child_elem, p, 0);
		}
		else if (strcmp(child_elem->name, "active") == 0) {
			iscreen->active = DIVY_DBMS_ACTIVE;
		}
		else if (strcmp(child_elem->name, "creationdt") == 0) {
			iscreen->registdt = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else if (strcmp(child_elem->name, "updatedt") == 0) {
			iscreen->updatedt = (char *) divy_xml_get_cdata(child_elem, p, 1);
		}
		else {
			/* 未知のエレメントが指定された(でも無視) */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"dbmsinfo", child_elem->name);
		}
	}
	*dbms_iscreen = iscreen;

	return 0;
}

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * systemquota プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_systemquota_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_sysquota_iscreen **sysquota_iscreen)
{
	const apr_xml_elem *sysquota_elem, *child_elem;
	divy_sysquota_iscreen *iscreen;
	char *data;

	*sysquota_iscreen = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"systemquota\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	iscreen = apr_pcalloc(p, sizeof(divy_sysquota_iscreen));
	iscreen->ignore_assigned_quota = 0;
	iscreen->type = DIVY_QTTYPE_SYSTEM;
	iscreen->sfs  = NULL;	/* ここでは入れない */

	/*
	 * systemquota タグの解析
	 * (note)
	 * <!ELEMENT systemquota (
	 * 		used-bytes, available-bytes, assigned-bytes?, free-bytes?,
	 *		used-files, available-files, assigned-files?, free-files?,
	 *		ignore-assigned-quota?) >) >
	 */
	sysquota_elem = elem;
	/* systemquotaタグの中身が無かった */
	if (sysquota_elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"systemquota\"element is EMPTY.");
		return 1;
	}

	/* systemquota エレメントの中にあるエレメントを全て解析する */
	for (child_elem = sysquota_elem->first_child; child_elem; child_elem = child_elem->next) {

		data = (char *) divy_xml_get_cdata(child_elem, p, 1);

		if (strcmp(child_elem->name, "used-bytes") == 0) {
			iscreen->s_usedst = data;
		}
		else if (strcmp(child_elem->name, "available-bytes") == 0) {
			iscreen->s_maxst = data;
		} 
		else if (strcmp(child_elem->name, "assigned-bytes") == 0) {
			iscreen->s_total_bytes = data;
		}
		else if (strcmp(child_elem->name, "free-bytes") == 0) {
			iscreen->s_avail_bytes = data;
		}
		else if (strcmp(child_elem->name, "used-files") == 0) {
			iscreen->s_usedres = data;
		}
		else if (strcmp(child_elem->name, "available-files") == 0) {
			iscreen->s_maxres = data;
		}
		else if (strcmp(child_elem->name, "assigned-files") == 0) {
			iscreen->s_total_files = data;
		}
		else if (strcmp(child_elem->name, "free-files") == 0) {
			iscreen->s_avail_files = data;
		}
		else if (strcmp(child_elem->name, "ignore-assigned-quota") == 0) {
			iscreen->ignore_assigned_quota = 1;
		}
		else {
			/* 未知のエレメントが指定された(でも無視) */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"systemquota", child_elem->name);
		}
	}

	*sysquota_iscreen = iscreen;

	return 0;
}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

/**
 * resourcestate プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_resourcestate_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_resourcestate_iscreen **rstate_iscreen)
{
	const apr_xml_elem *rstate_elem, *child_elem;
	divy_resourcestate_iscreen *iscreen = NULL;

	*rstate_iscreen = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"resourcestate\" element MUST not be empty.");
		return 1;
	}

	/*
	 * resourcestate タグの解析
	 * (note)
	 * <!ELEMENT resourcestate (published? box?> >
	 */
	rstate_elem = elem;

	/* resourcestate タグの中身が無かった */
	if (rstate_elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"resourcestate\"element is EMPTY.");
		return 1;
	}

	/* resourcestate エレメントの中にあるエレメントを全て解析する */
	for (child_elem = rstate_elem->first_child; child_elem; child_elem = child_elem->next) {

		if (strcmp(child_elem->name, "published") == 0) {
			if (*rstate_iscreen == NULL) {
				*rstate_iscreen = iscreen =
					apr_pcalloc(p, sizeof(divy_resourcestate_iscreen));
			}
			else {
				iscreen->next = apr_pcalloc(p, sizeof(divy_resourcestate_iscreen));
				iscreen = iscreen->next;
			}
			iscreen->uri  = NULL;	/* ここでは判別不能 */
			iscreen->type = DIVY_RSTATE_TYPE_VIEW;
			iscreen->state.published = 1;
			iscreen->next = NULL;
		}
		else if (strcmp(child_elem->name, "box") == 0) {
			if (*rstate_iscreen == NULL) {
				*rstate_iscreen = iscreen =
					apr_pcalloc(p, sizeof(divy_resourcestate_iscreen));
			}
			else {
				iscreen->next = apr_pcalloc(p, sizeof(divy_resourcestate_iscreen));
				iscreen = iscreen->next;
			}
			if (_divy_parse_state_box(p, child_elem, iscreen) != 0) {
				/* ### TODO ERRLOGを書きなさい */
				/* ただ、このパース関数はエラーの戻りがない。どうしませう */
				return 1;
			}

			iscreen->uri = NULL;
			iscreen->type = DIVY_RSTATE_TYPE_BOX;
			iscreen->state.box = 1;	/* 公開 */
			iscreen->next = NULL;
		}
		else {
			/* 未知のエレメントが指定された(でも無視) */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"resourcestate", child_elem->name);
		}
	}

	return 0;
}

/**
 * thumbnail プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_thumbnail_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_thumbnail_iscreen **thumbnail_iscreen)
{
	char *data;

	*thumbnail_iscreen = apr_pcalloc(p, sizeof(divy_thumbnail_iscreen));
	(*thumbnail_iscreen)->data = NULL;

	/*
	 * thumbnail タグの解析
	 * (note)
	 * <!ELEMENT thumbnail (#PCDATA) >
	 */
	data = (char *) divy_xml_get_cdata(elem, p, 1);
	if (IS_FILLED(data)) {
		(*thumbnail_iscreen)->data = data;
	}

	return 0;
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passwordpolicy プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_passwordpolicy_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_passwordpolicy_iscreen **passpolicy_iscreen)
{
	const apr_xml_elem *passpolicy_elem, *child_elem;
	divy_passwordpolicy_iscreen *iscreen;
	char *data;

	*passpolicy_iscreen = NULL;	/* 初期化 */

	/* トップのタグしかなかった場合 */
	if (elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"passwordpolicy\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	iscreen = apr_pcalloc(p, sizeof(divy_passwordpolicy_iscreen));
	iscreen->on                   = 0;
	iscreen->sminlen              = NULL;
	iscreen->schange_pw_cycle     = NULL;
	iscreen->sprobation           = NULL;
	iscreen->shistory_pw_num      = NULL;
	iscreen->ngword               = NULL;
	iscreen->denyUseridString     =  0;
	iscreen->isSymbolChrNeeded    =  0;
	iscreen->isUpAlphabet         =  0;
	iscreen->isLoAlphabet         =  0;
	iscreen->isNumericNeeded      =  0;
	iscreen->denyCycleChr         =  0;
	iscreen->sfirstlogin_limitday = NULL;

	/*
	 * passwordpolicy タグの解析
	 * (note)
	 * <!ELEMENT passwordpolicy ((policy-on | policy-off),
	 *                 minimum-password-length?,
	 *                 change-password-cycle?,
	 *                 password-probation?,
	 *                 change-history-number?,
	 *                 ngwords?,
	 *                 deny-userid-string?,
	 *                 must-include-symbol?,
	 *                 must-include-upper-alphabet?,
	 *                 must-include-lower-alphabet?,
	 *                 must-include-numeric?,
	 *                 deny-cycle-character?,
	 *                 firstlogin-limitday) >
	 */
	passpolicy_elem = elem;
	/* passwordpolicyタグの中身が無かった */
	if (passpolicy_elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"passwordpolicy\"element is EMPTY.");
		return 1;
	}

	/* passwordpolicy エレメントの中にあるエレメントを全て解析する */
	for (child_elem = passpolicy_elem->first_child; child_elem; child_elem = child_elem->next) {

		data = (char *) divy_xml_get_cdata(child_elem, p, 1);

		if (strcmp(child_elem->name, "policy-on") == 0) {
			iscreen->on = 1;
		}
		else if (strcmp(child_elem->name, "policy-off") == 0) {
			iscreen->on = 0;
		}
		else if (strcmp(child_elem->name, "minimum-password-length") == 0) {
			iscreen->sminlen = data;
		}
		else if (strcmp(child_elem->name, "change-password-cycle") == 0) {
			iscreen->schange_pw_cycle = data;
		} 
		else if (strcmp(child_elem->name, "password-probation") == 0) {
			iscreen->sprobation = data;
		} 
		else if (strcmp(child_elem->name, "change-history-number") == 0) {
			iscreen->shistory_pw_num = data;
		}
		else if (strcmp(child_elem->name, "ngwords") == 0) {
			iscreen->ngword = data;
		}
		else if (strcmp(child_elem->name, "deny-userid-string") == 0) {
			iscreen->denyUseridString = 1;
		}
		else if (strcmp(child_elem->name, "must-include-symbol") == 0) {
			iscreen->isSymbolChrNeeded = 1;
		}
		else if (strcmp(child_elem->name, "must-include-upper-alphabet") == 0) {
			iscreen->isUpAlphabet = 1;
		}
		else if (strcmp(child_elem->name, "must-include-lower-alphabet") == 0) {
			iscreen->isLoAlphabet = 1;
		}
		else if (strcmp(child_elem->name, "must-include-numeric") == 0) {
			iscreen->isNumericNeeded = 1;
		}
		else if (strcmp(child_elem->name, "deny-cycle-character") == 0) {
			iscreen->denyCycleChr = 1;
		}
		else if (strcmp(child_elem->name, "firstlogin-limitday") == 0) {
			iscreen->sfirstlogin_limitday = data;
		}
		else {
			/* 未知のエレメントが指定された(でも無視) */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"passwordpolicy", child_elem->name);
		}
	}

	*passpolicy_iscreen = iscreen;

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * changepassword プロパティをパースする。
 * [ 有効性 ]
 *   パスワードポリシー機能が「サポート/未サポート」「有効/無効」に関わらず
 *   定義済みとなります.
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param changepassword_iscreen divy_changepassword_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_changepassword_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_changepassword_iscreen **changepassword_iscreen)
{
	divy_changepassword_iscreen *iscreen;
	char *data;

	*changepassword_iscreen = NULL;	/* 初期化 */

	/* トップのタグがなかった場合 */
	if (elem == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"changepassword\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	iscreen = apr_pcalloc(p, sizeof(divy_changepassword_iscreen));
	iscreen->password  = NULL;

	/*
	 * changepassword タグの解析
	 * (note)
	 *   <!ELEMENT changepassword (#PCDATA) >
	 */
	data = (char *) divy_xml_get_cdata(elem, p, 0);	/* スペースは取らない */
	if (strcmp(elem->name, "changepassword") == 0) {
		iscreen->password = data;
	}
	else {
		/* 未知のエレメントが指定された(でも無視) */
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified. (elem->name = %s). "
				"We ignore this.", elem->name);
	}

	*changepassword_iscreen = iscreen;

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * changeowner プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_changeowner_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_changeowner_iscreen **changeowner_iscreen)
{
	divy_changeowner_iscreen *iscreen;

	*changeowner_iscreen = NULL;	/* 初期化 */

	/* トップのタグがなかった場合 */
	if (elem == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"changeowner\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	iscreen = apr_pcalloc(p, sizeof(divy_changeowner_iscreen));

	/*
	 * changeowner タグの解析
	 * (note)
	 *   <!ELEMENT changeowner EMPTY >
	 */
	if (strcmp(elem->name, "changeowner") == 0) {
		/* 値が無いので何もしなくていい */
	}
	else {
		/* 未知のエレメントが指定された(でも無視) */
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified. (elem->name = %s). "
				"We ignore this.", elem->name);
	}

	*changeowner_iscreen = iscreen;

	return 0;
}

/**
 * changeleader プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_changeleader_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_changeleader_iscreen **changeleader_iscreen)
{
	const apr_xml_elem *e;
	divy_changeleader_iscreen *iscreen;
	char *data;

	*changeleader_iscreen = NULL;	/* 初期化 */

	/* トップのタグがなかった / トップタグしかなかった場合 */
	if (elem == NULL || elem->first_child == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"changeleader\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	iscreen = apr_pcalloc(p, sizeof(divy_changeleader_iscreen));
	iscreen->ownerid = NULL;
	iscreen->action  = DIVY_CHLEADER_ACTION_UNKNOWN;	/* 未定義 */

	/*
	 * changeleader タグの解析
	 * [ DTD ]
	 * 	<!ELEMENT changeleader (ownerid, action) >
	 * 	<!ELEMENT ownerid (#PCDATA) >
	 * 	<!ELEMENT action  (appoint | dismiss) >
	 * 	<!ELEMENT appoint EMPTY >
	 * 	<!ELEMENT dismiss EMPTY >
	 */
	for (e = elem->first_child; e; e = e->next) {
		data = (char *) divy_xml_get_cdata(e, p, 1);
		if (strcmp(e->name, "ownerid") == 0) {
			iscreen->ownerid = data;
		}
		else if (strcmp(e->name, "action") == 0) {
			apr_xml_elem *child = e->first_child;

			if (child != NULL) {
				if (strcmp(child->name, "appoint") == 0) {
					iscreen->action = DIVY_CHLEADER_ACTION_APPOINT;	/* 任命 */
				}
				else if (strcmp(child->name, "dismiss") == 0) {
					iscreen->action = DIVY_CHLEADER_ACTION_DISMISS;	/* 解任 */
				}
				else {
					ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
							"The child of \"action\" element is invalid."
							"(name = %s)", child->name);
					/* 無視します(今後の拡張のため) */
				}
			}
		}
		else {
			/* 未知のエレメントが指定された(でも無視) */
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Invalid element is specified. (elem->name = %s). "
					"We ignore this.", e->name);
		}
	}

	*changeleader_iscreen = iscreen;

	return 0;
}

/**
 * confirmreading プロパティをパースする。
 *
 */
DIVY_DECLARE(int) divy_parse_confirmreading_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_confirmreading_iscreen **confirmreading_iscreen)
{
	divy_confirmreading_iscreen *iscreen;

	*confirmreading_iscreen = NULL;	/* 初期化 */

	/* トップのタグがなかった場合 */
	if (elem == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"confirmreading\"element MUST not be empty.");
		return 1;
	}

	/* 生成 */
	iscreen = apr_pcalloc(p, sizeof(divy_confirmreading_iscreen));

	/*
	 * confirmreading タグの解析
	 * (note)
	 *   <!ELEMENT confirmreading (own?, other?) >
	 *   ただし、own と other はクライアントから提出されることはない
	 */
	if (strcmp(elem->name, "confirmreading") == 0) {
		/* 値が無いので何もしなくていい */
	}
	else {
		/* 未知のエレメントが指定された(でも無視) */
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified. (elem->name = %s). "
				"We ignore this.", elem->name);
	}

	*confirmreading_iscreen = iscreen;

	return 0;
}


/*------------------------------------------------------------------------------
  map iscreen to rdbo object
  ----------------------------------------------------------------------------*/
/**
 * user_iscreen を grp_pr にマップする。
 *
 */
DIVY_DECLARE(int) divy_map_user_property(apr_pool_t *p,
					const divy_user_iscreen *user_iscreen,
					divy_rdbo_usr **usr_pr)
{
	*usr_pr = apr_pcalloc(p, sizeof(divy_rdbo_usr));
	(*usr_pr)->usrid      = user_iscreen->usrid;
	(*usr_pr)->password   = user_iscreen->password;
	(*usr_pr)->fullname   = user_iscreen->fullname;
	(*usr_pr)->mailaddr   = user_iscreen->mailaddr;
	(*usr_pr)->adminmode  = user_iscreen->adminmode;
	(*usr_pr)->usedst     = APR_INT64_C(0);
	(*usr_pr)->maxst      = (user_iscreen->smaxst) ?
					apr_atoi64(user_iscreen->smaxst) : APR_INT64_C(0);
	(*usr_pr)->usedres    = APR_INT64_C(0);
	(*usr_pr)->maxres     = (user_iscreen->smaxres) ?
					apr_atoi64(user_iscreen->smaxres) : APR_INT64_C(0);
	(*usr_pr)->accessdeny = user_iscreen->accessdeny;
	(*usr_pr)->registdt   = user_iscreen->registdt;
	(*usr_pr)->updatedt   = user_iscreen->updatedt;
	(*usr_pr)->comment    = user_iscreen->comment;
	if (IS_FILLED(user_iscreen->sexpiration)) {
		(*usr_pr)->expiration = dav_divy_iso8601totime_t(p, user_iscreen->sexpiration);
	}
	else {
		(*usr_pr)->expiration = (time_t) 0;	/* 無制限 */
	}
	(*usr_pr)->extstatus  = user_iscreen->extstatus;
#ifdef DIVY_SUPPORT_PASSPOLICY
	(*usr_pr)->passpolicy_status = NULL;	/* ここではいれません */
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	if (IS_FILLED(user_iscreen->smaxusercreation)) {
		(*usr_pr)->maxusercreation = atoi(user_iscreen->smaxusercreation);
	}
	else {
		(*usr_pr)->maxusercreation = 0;
	}
	if (IS_FILLED(user_iscreen->allowhosts)) {
		(*usr_pr)->allowhosts = user_iscreen->allowhosts;
	}

	(*usr_pr)->loginfailedcount = 0;

	(*usr_pr)->next       = NULL;

	return 0;
}

/**
 * group_iscreen を grp_pr にマップする。
 *
 */
DIVY_DECLARE(int) divy_map_group_property(apr_pool_t *p,
					const divy_group_iscreen *group_iscreen,
					divy_rdbo_grp **grp_pr)
{
	*grp_pr = apr_pcalloc(p, sizeof(divy_rdbo_grp));

	(*grp_pr)->grpid        = group_iscreen->grpid;
	(*grp_pr)->name         = group_iscreen->name;
	(*grp_pr)->comment      = group_iscreen->comment;
	(*grp_pr)->registdt     = group_iscreen->registdt;
	(*grp_pr)->updatedt     = group_iscreen->updatedt;
	(*grp_pr)->mailwatch_pr = group_iscreen->mailwatch_pr;
	(*grp_pr)->grp_extstatus= group_iscreen->grp_extstatus;
	(*grp_pr)->policy		= group_iscreen->upload_policy;
	(*grp_pr)->next         = NULL;

	return 0;
}

/**
 * sysmsg_iscreen を sysmsg_pr にマップする。
 *
 */
DIVY_DECLARE(int) divy_map_sysmsg_property(apr_pool_t *p,
					const divy_sysmsg_iscreen *sysmsg_iscreen,
					divy_rdbo_sysmsg **sysmsg_pr)
{
	*sysmsg_pr = apr_pcalloc(p, sizeof(divy_rdbo_sysmsg));
	(*sysmsg_pr)->msgid    = sysmsg_iscreen->msgid;
	(*sysmsg_pr)->msg      = sysmsg_iscreen->msg;
	(*sysmsg_pr)->subject  = sysmsg_iscreen->subject;
	(*sysmsg_pr)->lang     = sysmsg_iscreen->lang;
	(*sysmsg_pr)->scope    = sysmsg_iscreen->scope;
	(*sysmsg_pr)->active   = sysmsg_iscreen->active;
	(*sysmsg_pr)->registdt = sysmsg_iscreen->registdt;
	(*sysmsg_pr)->updatedt = sysmsg_iscreen->updatedt;
	(*sysmsg_pr)->next     = NULL;

	return 0;
}

/**
 * dbms_iscreen を dbms_pr にマップする。
 *
 */
DIVY_DECLARE(int) divy_map_dbms_property(apr_pool_t *p,
					const divy_dbms_iscreen *dbms_iscreen,
					divy_rdbo_dbms **dbms_pr)
{
	*dbms_pr = apr_pcalloc(p, sizeof(divy_rdbo_dbms));

	(*dbms_pr)->dbmsid         = dbms_iscreen->dbmsid;
	(*dbms_pr)->type           = dbms_iscreen->type;
	(*dbms_pr)->name           = dbms_iscreen->name;
	(*dbms_pr)->hostname       = dbms_iscreen->hostname;
	(*dbms_pr)->port           = atoi(dbms_iscreen->sport);
	(*dbms_pr)->dbname         = dbms_iscreen->dbname;
	(*dbms_pr)->username       = dbms_iscreen->username;
	(*dbms_pr)->password       = dbms_iscreen->password;
	(*dbms_pr)->comment        = dbms_iscreen->comment;
	(*dbms_pr)->dbpool         = 0;	/* 現時点では利用しないことになっている */
	(*dbms_pr)->dbminspareconn = 0;
	(*dbms_pr)->dbmaxspareconn = 0;
	(*dbms_pr)->active         = dbms_iscreen->active;
	(*dbms_pr)->registdt       = dbms_iscreen->registdt;
	(*dbms_pr)->updatedt       = dbms_iscreen->updatedt;
	(*dbms_pr)->next           = NULL;

	return 0;
}

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * sysquota_iscreen を sysquota_pr にマップする。
 *
 */
DIVY_DECLARE(int) divy_map_systemquota_property(apr_pool_t *p,
					const divy_sysquota_iscreen *sysquota_iscreen,
					divy_rdbo_diskquota **sysquota_pr)
{
	*sysquota_pr = apr_pcalloc(p, sizeof(divy_rdbo_diskquota));

	(*sysquota_pr)->type    = sysquota_iscreen->type;
	(*sysquota_pr)->maxst   = apr_atoi64(sysquota_iscreen->s_maxst);
	(*sysquota_pr)->maxres  = apr_atoi64(sysquota_iscreen->s_maxres);
	(*sysquota_pr)->sfs     = sysquota_iscreen->sfs;

	return 0;
}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

/**
 * rstate_iscreen を rstate_pr にマップする。
 *
 */
DIVY_DECLARE(int) divy_map_resourcestate_property(apr_pool_t *p,
					const divy_resourcestate_iscreen *rstate_iscreen,
					divy_rdbo_resourcestate **rstate_pr)
{
	const divy_resourcestate_iscreen *iscreen;
	divy_rdbo_resourcestate *rstate = NULL;
	divy_rdbo_box *rbox = NULL;

	*rstate_pr = NULL;

	for (iscreen = rstate_iscreen; iscreen; iscreen = iscreen->next) {
		if (*rstate_pr == NULL) {
			*rstate_pr = rstate = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
		}
		else {
			rstate->next = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
			rstate = rstate->next;
		}

		rstate->uri   = iscreen->uri;
		rstate->depth = divy_count_dirs(iscreen->uri);
		rstate->type  = iscreen->type;
		if (iscreen->type == DIVY_RSTATE_TYPE_VIEW) {
			rstate->hierarchy = 1;	/* View属性は階層的に波状する */
			if (iscreen->state.published) {
				rstate->effective = 1;
			}
			else {
				rstate->effective = 0;
			}
		}
		else if (iscreen->type == DIVY_RSTATE_TYPE_BOX) {
			rstate->hierarchy = 1;	/* BOX属性は階層的に波状する */
			if (iscreen->state.box == 1) {
				rstate->effective = 1;
			}
			else {
				rstate->effective = 0;
			}
			rbox = apr_pcalloc(p, sizeof(divy_rdbo_box));
			/* TODO ### 不要か検証しなさい */
			rbox->uri = iscreen->uri;
			rbox->allowed_origin = iscreen->optional.boxinfo->allowed_origin;
			/*
			 * 設定済の公開されているBOX情報が存在する場合(UPDATE)には短縮URL
			 * の更新は行いません。（外部仕様）
			 *
			 * 短縮URLが変更されるのは次の通り
			 * 公開がされたときです。(flag & 0x01の時）
			 * URLロックではないとき (flag & 0x04 == 0) ##TODO(未実装2013/5/17)
			 */
			if (iscreen->src_rstate_pr != NULL &&
					iscreen->src_rstate_pr->optional.box_pr->flag & BOX_FLAG_OPEN) {
				rbox->shorten = iscreen->src_rstate_pr->optional.box_pr->shorten;
			}
			else {
				/* 短縮URLの変更 */
				rbox->shorten = divy_get_rid2shorten(p, iscreen->optional.boxinfo->rsid, (iscreen->src_rstate_pr != NULL)?iscreen->src_rstate_pr->optional.box_pr->shorten:NULL);

				/* 新規フラグを設定 */
				rbox->flag |= BOX_FLAG_NEW;
			}
			rbox->flag |= iscreen->optional.boxinfo->flag;
			rbox->password  = iscreen->optional.boxinfo->password;
			rbox->creationdate = dav_divy_get_now_epoch();
			rbox->expirationdate = dav_divy_iso8601totime_t(p, iscreen->optional.boxinfo->expiration);
			if (rbox->expirationdate == (time_t)0) {
				rbox->flag &= ~(BOX_FLAG_DELE);
			}
			rbox->creator_usr_id = iscreen->optional.boxinfo->creator;
			rbox->greeting = iscreen->optional.boxinfo->greeting;
			rbox->message = iscreen->optional.boxinfo->message;
			/* 2FA */
			rbox->tomailaddr = iscreen->optional.boxinfo->tomailaddr;

			rstate->optional.box_pr = rbox;
		}

		rstate->next = NULL;
	}

	return 0;
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passpolicy_iscreen をpasspolicy_pr にマップする.
 *
 */
DIVY_DECLARE(int) divy_map_passwordpolicy_property(apr_pool_t *p,
					const divy_passwordpolicy_iscreen *passpolicy_iscreen,
					divy_rdbo_passwordpolicy **passpolicy_pr)
{
	*passpolicy_pr = apr_pcalloc(p, sizeof(divy_rdbo_passwordpolicy));

	/* ポリシーID */
	(*passpolicy_pr)->policyid = DIVY_DEFAULT_POLICY_ID;

	/* ステータス */
	(*passpolicy_pr)->status   = passpolicy_iscreen->on;

	/* 開始日(有効にした日時) */
	(*passpolicy_pr)->startdt  = 0L;	/* ここでは設定しない */

	/* 最小長さ */
	if (passpolicy_iscreen->sminlen != NULL) {
		(*passpolicy_pr)->minlen   = atoi(passpolicy_iscreen->sminlen);
	}
	else {
		(*passpolicy_pr)->minlen   = 0;	/* 0: 制限なし */
	}

	/* パスワード有効期限(日) */
	if (passpolicy_iscreen->schange_pw_cycle != NULL) {
		(*passpolicy_pr)->change_pw_cycle = atoi(passpolicy_iscreen->schange_pw_cycle);
	}
	else {
		(*passpolicy_pr)->change_pw_cycle = 0;	/* 0: 無制限 */
	}

	/* 猶予期間(日) */
	if (passpolicy_iscreen->sprobation != NULL) {
		(*passpolicy_pr)->probation = atoi(passpolicy_iscreen->sprobation);
	}
	else {
		(*passpolicy_pr)->probation = 0;	/* 0: 即日 */
	}

	/* 禁則文字 */
	if (IS_FILLED(passpolicy_iscreen->ngword)) {
		char *ch;
		divy_cset_t *nsword_set = divy_cset_make(p);

		(*passpolicy_pr)->ngword = apr_pcalloc(p, sizeof(divy_rdbo_passngword));
		(*passpolicy_pr)->ngword->policyid = (*passpolicy_pr)->policyid;
		(*passpolicy_pr)->ngword->ngword_set = nsword_set;

		ch = passpolicy_iscreen->ngword;
		while (*ch != '\0') {
			divy_cset_set(nsword_set, apr_psprintf(p, "%c", *ch));
			ch++;
		}
	}
	else {
		(*passpolicy_pr)->ngword = NULL;	/* NULL でNGワードなし */
	}

	/* 履歴数 */
	if (passpolicy_iscreen->shistory_pw_num != NULL) {
		(*passpolicy_pr)->history_pw_num = atoi(passpolicy_iscreen->shistory_pw_num);
	}
	else {
		(*passpolicy_pr)->history_pw_num = 0;	/* 0: 履歴無し */
	}

	/* ユーザID文字列禁止 */
	(*passpolicy_pr)->denyUseridString    = passpolicy_iscreen->denyUseridString;

	/* 記号必須 */
	(*passpolicy_pr)->isSymbolChrNeeded   = passpolicy_iscreen->isSymbolChrNeeded;

	/* 大文字アルファベット必須 */
	(*passpolicy_pr)->isUpAlphabet        = passpolicy_iscreen->isUpAlphabet;

	/* 小文字アルファベット必須 */
	(*passpolicy_pr)->isLoAlphabet        = passpolicy_iscreen->isLoAlphabet;

	/* 数値必須 */
	(*passpolicy_pr)->isNumericNeeded     = passpolicy_iscreen->isNumericNeeded;

	/* 連続文字禁止 */
	(*passpolicy_pr)->denyCycleChr        = passpolicy_iscreen->denyCycleChr;

	/* 初回ログイン制限 */
	if (passpolicy_iscreen->sfirstlogin_limitday != NULL) {
		/* 0: パスワードを変えない限りログインできない / n: n日以内に変更必要 */
		(*passpolicy_pr)->firstlogin_limitday =
			atoi(passpolicy_iscreen->sfirstlogin_limitday);
	}
	else {
		(*passpolicy_pr)->firstlogin_limitday = -1;	/* 初回ログイン制約なし */
	}

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */


/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
/**
 * ある１つのkeyvalue エレメントをパースして、その結果をsqlcnt_pr に
 * 格納して返却する。
 * (note)
 *   sqlcnt_pr には領域が割り当てられていなければなりません。
 *   取得したデータは、sqlcnt_pr のメンバにセットします。sqlcnt_pr->next
 *   には触りません。
 * (note)
 * 	sqlmode 及びreqsqlnameとの検証は実装の都合からここで
 * 	やってしまっています。
 *
 * @param p apr_poot_t *
 * @param keyvalue_elem const apr_xml_elem * keyvalue エレメント
 * @param sqlcnt_pr divy_rdbo_sqlcontent * 解析したある１つのkeyvalue
 * 						エレメントの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
static int _parse_one_keyvalue_elem(apr_pool_t *p,
					const apr_xml_elem *keyvalue_elem,
					divy_rdbo_sqlcontent *sqlcnt_pr)
{
	const apr_xml_elem *elem = NULL;
	apr_xml_attr *attr;
	char *data;
	int found_sqlmode = 0;

	/* sqlcnt_pr がNULLだった時 */
	if (sqlcnt_pr == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"sqlcnt_pr is NULL. Must be allocate memory "
			"before this call.(elem->name = %s", elem->name);
		return 1;
	}

	/* sqlcnt_pr の初期化 */
	sqlcnt_pr->contype     = DIVY_SQL_CONTYPE_BIND;	/* 通常バインド変数とする */ 
	sqlcnt_pr->sqlposition = -1;			/* 見つからなかった */

	/* 属性値の取得 */
	attr = keyvalue_elem->attr;
	if (attr == NULL || attr->value == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"id\" attribute of \"keyvalue\" element "
			"is missing. Must be specified.");
		return 1;
	}

	/* 値は数値か? */
	data = (char *) attr->value;
	if (!dav_divy_isdigit_str(data)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"id\" attribute value of \"keyvalue\" "
			"MUST be digit.(data = %s)", data);
		return 1;
	}
	sqlcnt_pr->id = atoi(attr->value);

	/*
	 * keyvalue エレメントの中にあるエレメントを全て解析する
	 * (note) keyvalue エレメントの構造
	 * 	<!ELEMENT keyvalue (columnname, dispcolumname?, reqsqlname?,
	 * 	                    sqlmode?, datafmt, sqlposition) >
	 */
	for (elem = keyvalue_elem->first_child; elem; elem = elem->next) {

		/* PCDATA の両端に含まれていたスペースを除去して取得する */
		data = (char *) divy_xml_get_cdata(elem, p, 1);

		if (strcmp(elem->name, "columnname") == 0) {
			sqlcnt_pr->colname = data;
		}
		else if (strcmp(elem->name, "dispcolumnname") == 0) {
			sqlcnt_pr->name = data;
		}
		else if (strcmp(elem->name, "reqsqlname") == 0) {
			if (sqlcnt_pr->reqsql == NULL) {
				sqlcnt_pr->reqsql =
					apr_pcalloc(p, sizeof(divy_rdbo_reqsql));
			}

			if (sqlcnt_pr->reqsql->rsvalue == NULL) {
				sqlcnt_pr->reqsql->rsvalue =
					apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
			}

			sqlcnt_pr->reqsql->rsvalue->name = data;

			/* reqsqlname エレメントが見つかったらRequiredSQL
			 * または名前付きバインド変数が指定されたということ */
			sqlcnt_pr->contype = DIVY_SQL_CONTYPE_REQUIRED;
		}
		else if (strcmp(elem->name, "sqlmode") == 0) {
			const char *sqlmode_str;
			if (sqlcnt_pr->reqsql == NULL) {
				sqlcnt_pr->reqsql = apr_pcalloc(p, sizeof(divy_rdbo_reqsql));
			}

			if (elem->first_child && elem->first_child->name) {
				sqlmode_str = elem->first_child->name;
				found_sqlmode = 1;

				if (strcmp(sqlmode_str, "show") == 0) {
					sqlcnt_pr->reqsql->sqlmode = DIVY_SQL_MODE_SHOW; 
				}
				else if (strcmp(sqlmode_str, "hidden") == 0) {
					sqlcnt_pr->reqsql->sqlmode = DIVY_SQL_MODE_HIDDEN;
				}
				else if (strcmp(sqlmode_str, "selected") == 0) {
					sqlcnt_pr->reqsql->sqlmode = DIVY_SQL_MODE_SELECTED;
				}
				else if (strcmp(sqlmode_str, "multiselected") == 0) {
					sqlcnt_pr->reqsql->sqlmode = DIVY_SQL_MODE_MULTISELECTED;
				}
				else {
					/* 未知のモードが指定されていた */
					ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The child of \"sqlmode\" is invalid."
						"(name = %s)", sqlmode_str);
					return 1;
				}
			}
			else {
				/* このタグは必須 */
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The \"sqlmode\" element must be specified.");
				return 1;
			}
		}
		else if (strcmp(elem->name, "datafmt") == 0) {
			sqlcnt_pr->fmt = data;
		}
		else if (strcmp(elem->name, "sqlposition") == 0) {
			if (!dav_divy_isdigit_str(data)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The value of \"sqlposition\"(\"keyvalue\" child) "
					"element must be digit. (str = %s)", data);
				return 1;
			}
			sqlcnt_pr->sqlposition = atoi(data);
		}
		else {
			/* 未知のエレメントが指定された(でも無視) */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Invalid element is specified in the element "
				"of \"%s\".(elem->name = %s). We ignore this.",
				"keyvalue", elem->name);
		}
	}

	/* reqsqlname エレメントはあったのに、sqlmodeが無かった場合 */
	if (sqlcnt_pr->reqsql && found_sqlmode == 0) {
		/* 判定はここではやりません */
		sqlcnt_pr->reqsql->sqlmode = DIVY_SQL_MODE_UNKNOWN;
	}

	return 0;
}

/**
 * resourcestate中のboxプロパティをパースする
 *
 * @param pool apr_pool_t *
 * @param elem apr_xml_elem *
 * @param iscreen divy_resourcestate_iscreen *iscreen
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
static int _divy_parse_state_box(apr_pool_t *pool,
				const apr_xml_elem *elem, divy_resourcestate_iscreen *iscreen)
{
	const apr_xml_elem *child_elem;
	char *data;
	divy_boxinfo *_boxinfo;
	char *token, *token_ctx;
	divy_sbuf *buff = NULL;
	int first = 0;

	/* 生成 */
	_boxinfo = apr_pcalloc(pool, sizeof(divy_boxinfo));

	/* BOX内部のエレメントをすべて解析する */
	for (child_elem = elem->first_child; child_elem;
								child_elem = child_elem->next) {

		/* 文字列データを取得する */
		data = (char *) divy_xml_get_cdata(child_elem, pool, 1);

		if (strcmp(child_elem->name, "allowedorigin") == 0) {
			_boxinfo->allowed_origin = data;
		}
		else if (strcmp(child_elem->name, "sealingwax") == 0) {
			_boxinfo->flag |= BOX_FLAG_OPEN;	/* 公開 */
		} else if (strcmp(child_elem->name, "privacy") == 0) {
			_boxinfo->flag |= BOX_FLAG_PRIV;	/* プライバシーモード */
		} else if (strcmp(child_elem->name, "lockurl") == 0) {
			_boxinfo->flag |= BOX_FLAG_LOCK;	/* URLのロック*/
		} else if (strcmp(child_elem->name, "autodelete") == 0) {
			_boxinfo->flag |= BOX_FLAG_DELE;	/* 自動削除 */
		} else if (strcmp(child_elem->name, "greeting") == 0) {
			_boxinfo->greeting = data;
		}
		else if (strcmp(child_elem->name, "message") == 0) {
			_boxinfo->message = data;
		}
		else if (strcmp(child_elem->name, "expiration") == 0) {
			_boxinfo->expiration = data;
		}
		else if (strcmp(child_elem->name, "password") == 0) {
			_boxinfo->password = data;
		}
		else if (strcmp(child_elem->name, "tomailaddr") == 0) {
			/* カンマ区切りを整理する */
			divy_sbuf_create(pool, &buff, 1024);	/* バッファ作成 */
			while ((token = apr_strtok(data, ",", &token_ctx)) != NULL) {
				if (token == NULL) break;
				data = NULL;

				/* 両端のwhitespace を飛ばす */
				token = (char *)dav_divy_trim_white(pool, token);
				if (IS_EMPTY(token)) continue;

				if (first) {
					divy_sbuf_append(buff, ",");
				}

				divy_sbuf_append(buff, token);
				first = 1;
			}

			_boxinfo->tomailaddr = divy_sbuf_tostring(buff); 
		}
	}
	
	iscreen->optional.boxinfo = _boxinfo;

	return 0;	/* 正常 */
}

/*
 * groupinfo中のuploadpolicyruleプロパティをパースする
 *
 * @param pool apr_pool_t *
 * @param elem apr_xml_elem *
 * @param iscreen divy_resourcestate_iscreen *iscreen
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
static int _divy_parse_upload_policy(apr_pool_t *pool,
				const apr_xml_elem *elem, divy_group_iscreen *iscreen) {

	char *val;
	apr_xml_elem *child = elem->first_child;
	divy_sbuf *sbuf_name   = NULL;
	divy_sbuf *sbuf_suffix = NULL;

	divy_sbuf_create(pool, &sbuf_name, 100);
	divy_sbuf_create(pool, &sbuf_suffix, 100);

	/* 初期化 */
	iscreen->upload_policy->rules_fnames = "";
	iscreen->upload_policy->rules_suffix = "";
	iscreen->upload_policy->rules_length = "";

	for (; child; child = child->next) {

		/* アクティブ非アクティブ */
		if (strcmp(child->name, "rulestate") == 0) {
			apr_xml_elem *status = child->first_child;
			if (strcmp(status->name, "inactive") == 0) {
				/* 非アクティブ */
				iscreen->upload_policy->active = 0;
			}
			else {
				/* アクティブ */
				iscreen->upload_policy->active = 1;
			}
		}
		/* ファイル名 ルール */
		else if (strcmp(child->name, "filename") == 0) {
			val = (char *) divy_xml_get_cdata(child, pool, 1);
			if (IS_EMPTY(val)) continue;

			if (divy_sbuf_getlength(sbuf_name) > 0) {
				divy_sbuf_append(sbuf_name, "/");
			}
			divy_sbuf_append(sbuf_name, val);
		}
		/* 拡張子ルール */
		else if (strcmp(child->name, "suffix") == 0) {
			val = (char *) divy_xml_get_cdata(child, pool, 1);
			if (IS_EMPTY(val)) continue;

			if (divy_sbuf_getlength(sbuf_suffix) > 0) {
				divy_sbuf_append(sbuf_suffix, ",");
			}
			divy_sbuf_append(sbuf_suffix, val);
		}
		/* 長さルール */
		else if (strcmp(child->name, "length") == 0) {
			/* lengthはカンマ区切り */
			iscreen->upload_policy->rules_length =
				(char *)dav_divy_replace_str(pool,
							divy_xml_get_cdata(child, pool, 1), " ", ",", NULL);
		}
		/* 文字種ルール */
		else if (strcmp(child->name, "chartype") == 0) {
			apr_xml_elem *type = child->first_child;
			for (; type; type = type->next) {
				if (strcmp(type->name, "alphabet") == 0) {
					/* 英字 */
					iscreen->upload_policy->rules_chartype
									|= TF_UPLOAD_POLICY_CHARTYPE_ALP;
				}
				else if (strcmp(type->name, "number") == 0) {
					/* 数字 */
					iscreen->upload_policy->rules_chartype
									|= TF_UPLOAD_POLICY_CHARTYPE_NUM;
				}
			}
		}
		else if (strcmp(child->name, "matchtype") == 0) {
			apr_xml_elem *mtype = child->first_child;
			if (mtype != NULL) {
				if (strcmp(mtype->name, "all") == 0) {
					/* すべて一致 */
					iscreen->upload_policy->rules_matchtype
									= TF_UPLOAD_POLICY_MATCHTYPE_ALL;
				} else if (strcmp(mtype->name, "any") == 0) {
					/* どれか一致 */
					iscreen->upload_policy->rules_matchtype
									= TF_UPLOAD_POLICY_MATCHTYPE_ANY;
				}
			}
		}
		else if (strcmp(child->name, "hidden") == 0) {
			/* ルール非公開 */
			iscreen->upload_policy->hidden = 1;
		}
		else {
			ERRLOG1(pool, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"The child of \"uploadpolicyrule\" element is invalid."
					"(name = %s)", child->name);
			/* 無視します(今後の拡張のため) */
		}

	}

	/* ファイル名に値があれば設定する */
	if (divy_sbuf_getlength(sbuf_name) > 0) {
		iscreen->upload_policy->rules_fnames = divy_sbuf_tostring(sbuf_name);
	}
	/* 拡張子に値があれば設定する */
	if (divy_sbuf_getlength(sbuf_suffix) > 0) {
		iscreen->upload_policy->rules_suffix = divy_sbuf_tostring(sbuf_suffix);
	}

	/* 日付を更新する */
	if (iscreen->upload_policy->active == 1) {
		iscreen->upload_policy->updatedate = dav_divy_get_now_epoch();
	}
	else {
		/* 無効化のときは適用日をクリア */
		iscreen->upload_policy->updatedate = 0;
	}


	/* uploadpolicyruleエレメントの確認 */
	divy_rdbo_set_uploadpolicy_group(iscreen->grp_extstatus,
											(iscreen->upload_policy->active));

	return 0;
}

