/**
 * $Id$
 *
 * TeamFile プラグインインターフェース (共通)
 *
 * [ 役割 ]
 *   * TeamFile 本体からプラグイン呼び出しコードの共通化
 */

#ifdef DIVY_SUPPORT_PLUGIN

#include "httpd.h"
#include "apr_pools.h"
#include "apr_strings.h"

#if APR_HAVE_STDARG_H
#include <stdarg.h>
#endif	/* APR_HAVE_STDARG_H */

#include "mod_dav_tf.h"
#include "util.h"
#include "tf_rdbo.h"
#include "tf_rdbo_group.h"
#include "tf_logger.h"
#include "tf_plugin.h"

/* Server Plugin Headers */
#include "tfs_errlog.h"
#include "tfsp_loader.h"
#include "tfsp_ihdr.h"
#include "tfs_pools.h"

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static apr_status_t _cleanup_loader(void *data);
static apr_status_t _cterminate(void *data);
static apr_status_t _rterminate(void *data);
static tfsp_ctx_loader * _lookup_loader(apr_pool_t *pconf);
static void _plugin_logger(void *data, int level, const char * desc);
static void _set_pi_req_pool(request_rec *r, tfs_pool_t *p);
static tfs_pool_t * _get_pi_req_pool(request_rec *r);
static void _add_version_component(apr_pool_t *pconf, tfsp_ctx_loader *loader);

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * TeamFile プラグインローダーの初期化.
 *
 */
DIVY_DECLARE(int) divy_pi_init(const char *pluginroot, apr_pool_t *pconf)
{
	tfs_error_t *err = NULL, *plugin_err = NULL;
	tfsp_ctx_loader *loader = NULL;

	/* ローダーコンテキストの生成 */
	if ((err = tfsp_loader_create(pluginroot, &loader)) != NULL) {
		tfs_error_dumpall(err, NULL, _plugin_logger);
		tfs_error_destroy(err);
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to create plugin-loader context");

		/* ローダーコンテキストを破棄する */
		tfsp_loader_destroy(loader);
		return 1;
	}

	/* 全プラグインをロードする */
	err = tfsp_loader_loadAll(loader, &plugin_err);
	if (plugin_err != NULL) {
		tfs_error_dumpall(plugin_err, NULL, _plugin_logger);
		tfs_error_destroy(plugin_err);
		/* 続ける */
	}

	if (err != NULL) {
		tfs_error_dumpall(err, NULL, _plugin_logger);
		tfs_error_destroy(err);
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to load all-plugin.");

		/* ローダーコンテキストを破棄する */
		tfsp_loader_destroy(loader);
		return 1;
	}

	/* プラグインローダーを登録する */
	divy_pcache_set_data(pconf, loader, DIVY_PCACHE_DAT_GL_PLLOADER);

	/* loader のクリーンアップ登録 */
	apr_pool_cleanup_register(pconf, loader, _cleanup_loader, apr_pool_cleanup_null);

	/* Apache 用のサーバヘッダにプラグイン識別文字列を追加する */
	_add_version_component(pconf, loader);

	return 0;
}

/**
 * プラグインのChildプロセス生成時の初期化処理をコールバックする.
 *
 */
DIVY_DECLARE(void) divy_pi_cinit(apr_pool_t *pchild, server_rec *s)
{
	tfsp_ctx_loader *loader;
	tfs_error_t *err = NULL, *plugin_err = NULL;

	/* プラグインローダーを取得 */
	if ((loader = _lookup_loader(s->process->pconf)) == NULL) {
		return;
	}

	/* cinit の呼び出し */
	err = tfsp_loader_cinit(loader, &plugin_err);
	if (err != NULL) {
		tfs_error_dumpall(err, NULL, _plugin_logger);
		tfs_error_destroy(err);
	}
	if (plugin_err != NULL) {
		tfs_error_dumpall(plugin_err, NULL, _plugin_logger);
		tfs_error_destroy(plugin_err);
	}

	/* Cleanup にコールバック関数を登録する */
	apr_pool_cleanup_register(pchild, loader, _cterminate,
								apr_pool_cleanup_null);
}

/**
 * プラグインのリクエストレベルの初期化処理.
 *
 */
DIVY_DECLARE(void) divy_pi_rinit(request_rec *r)
{
	tfs_pool_t *pp = NULL;

	/* プラグイン用のリクエストプールを生成する */
	tfs_pool_create(&pp);

	/* 生成したプールをキャッシュする */
	_set_pi_req_pool(r, pp);

	/* Cleanup にコールバック関数を登録する */
	apr_pool_cleanup_register(r->pool, pp, _rterminate,
								apr_pool_cleanup_null);
}

/**
 * プラグイン種類ptype, 名称pname のプラグインインターフェースを取得する.
 * pname がNULLであれば、同一のプラグイン種類の内、プラグインチェインの先頭にある
 * プラグインインターフェースを取得する.
 *
 */
DIVY_DECLARE(void *) divy_pi_lookupIf(request_rec *r, tfsp_type_plugin ptype,
                                      const char *pname)
{
	const tfsp_plugin_module *pm = divy_pi_lookupPm(r, ptype, pname);
	if (pm == NULL) return NULL;

	return tfsp_pm_get_interface(pm);
}

/**
 * プラグイン種類ptype, 名称pname のプラグイン構造体へのポインタを取得する.
 * pname がNULLであれば、同一のプラグイン種類の内、プラグインチェインの先頭にある
 * プラグインイン構造体を取得する.
 *
 */
DIVY_DECLARE(const tfsp_plugin_module *) divy_pi_lookupPm(request_rec *r,
                                      tfsp_type_plugin ptype, const char *pname)
{
	const tfsp_plugin_module *pm = NULL;
	tfsp_ctx_loader *loader;
	tfs_error_t *err = NULL;

	/* プラグインローダーを取得 */
	if ((loader = _lookup_loader(r->server->process->pconf)) == NULL) {
		return NULL;
	}

	/* インタフェースを取得する */
	err = tfsp_loader_lookupIf(_get_pi_req_pool(r), loader, ptype, pname, &pm);
	if (err != NULL) {
		tfs_error_dumpall(err, r, _plugin_logger);
		tfs_error_destroy(err);

		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to lookup specified plugin's interface.");
	}

	return pm;
}

/**
 * この関数呼び出し時点で利用されているプラグインローダーのプラグインルート
 * ディレクトリパスを取得する.
 */
DIVY_DECLARE(const char *) divy_pi_get_pluginroot(request_rec *r)
{
	tfsp_ctx_loader *loader;
	tfs_pool_t *pp = NULL;

	if (r == NULL) return NULL;

	/* プラグインローダーを取得 */
	if ((loader = _lookup_loader(r->server->process->pconf)) == NULL) {
		return NULL;
	}
	pp = _get_pi_req_pool(r);

	return apr_pstrdup(r->pool, tfsp_loader_get_rootpath(pp, loader));
}

/**
 * 各プラグイン毎のインストールパス名を組み立てて返却する.
 *
 */
DIVY_DECLARE(char *) divy_pi_build_plugin_path(request_rec *r, char *plugin_name)
{
	const char *plugin_root;

	if (IS_EMPTY(plugin_name)) {
		return NULL;
	}

	/* プラグインルートパスを取得 */
	plugin_root = divy_pi_get_pluginroot(r);

	return ap_make_full_path(r->pool, plugin_root, plugin_name);
}

/**
 * 各プラグイン毎のコンフィグパス名を組み立てて返却する.
 *
 */
DIVY_DECLARE(char *) divy_pi_build_plugin_confpath(request_rec *r, char *plugin_name)
{
	char *plugin_path;

	/* 各プラグインのインストールパスを組み立てる */
	plugin_path = divy_pi_build_plugin_path(r, plugin_name);
	if (IS_EMPTY(plugin_path)) {
		return NULL;
	}

	return ap_make_full_path(r->pool, plugin_path, TFSP_PLUGIN_CONFIG_DIR);
}

/**
 * プラグイン以下へのアクセスURI から対象ファイルまたはディレクトリの物理パスを算出する.
 *
 */
DIVY_DECLARE(char *) divy_pi_uri2path(request_rec *r, divy_uri_spec *u_spec)
{
	apr_pool_t *p = r->pool;
	apr_size_t len;
	char *uri_prefix, *uri_suffix;

	if (u_spec == NULL) return NULL;
	/* 
	 * URIのパターン
	 * $root/.plugin/$plugin_name
	 * $root/.plugin/$plugin_name/uploads
	 * $root/.plugin/$plugin_name/uploads/xxxx
	 * $root/.plugin/$plugin_name/cgi-bin
	 * $root/.plugin/$plugin_name/cgi-bin/xxx
	 * $root/.plugin/$plugin_name/conf
	 * $root/.plugin/$plugin_name/conf/xxx
	 */
	if (!(u_spec->infotype == DIVY_INFOTYPE_plugin_e         ||
		  u_spec->infotype == DIVY_INFOTYPE_plugin_uploads   ||
		  u_spec->infotype == DIVY_INFOTYPE_plugin_uploads_e ||
		  u_spec->infotype == DIVY_INFOTYPE_plugin_cgibin    ||
		  u_spec->infotype == DIVY_INFOTYPE_plugin_cgibin_e  ||
		  u_spec->infotype == DIVY_INFOTYPE_plugin_conf      ||
		  u_spec->infotype == DIVY_INFOTYPE_plugin_conf_e)) {
		return NULL;
	}

	uri_prefix  = dav_divy_make_uri(p, dav_divy_get_root_uri(r), DIVY_PLUGINFOLDER_NAME, NULL);
	len         = strlen(uri_prefix);

	/* 特殊フォルダより下のパスを切り出す(uri_prefix 以下を切り出す) */
	if (strncmp(u_spec->uri, uri_prefix, len) != 0) {
		/* URIの先頭が一致していなかった (ありえないが一応) */
		return NULL;
	}
	uri_suffix = apr_pstrdup(p, &u_spec->uri[len]);

	/* (仮定) URIのpath segment 区切り文字とファイルの区切り文字は一緒 */
	return ap_make_full_path(p, divy_pi_get_pluginroot(r), uri_suffix);
}


/**
 * infotype をtfsp_type_folder に変換する.
 *
 */
DIVY_DECLARE(tfsp_type_folder) divy_pi_infotype2foldertype(divy_infotype infotype, int is_dst)
{
	tfsp_type_folder ftype;

	switch (infotype) {
		case DIVY_INFOTYPE_group_e_regular:
			ftype = (!is_dst) ? TFSP_FLTYPE_GROUP1 : TFSP_FLTYPE_GROUP2;
			break;
		case DIVY_INFOTYPE_group_trash_e0:
		case DIVY_INFOTYPE_group_trash_ex:
			ftype = (!is_dst) ? TFSP_FLTYPE_GROUPTRASH1 : TFSP_FLTYPE_GROUPTRASH2;
			break;
		case DIVY_INFOTYPE_user_e_regular:
			ftype = (!is_dst) ? TFSP_FLTYPE_USER1 : TFSP_FLTYPE_USER2;
			break;
		case DIVY_INFOTYPE_user_trash_e0:
		case DIVY_INFOTYPE_user_trash_ex:
			ftype = (!is_dst) ? TFSP_FLTYPE_USERTRASH1 : TFSP_FLTYPE_USERTRASH2;
			break;
		case DIVY_INFOTYPE_dbfolder_e_regular:
			ftype = (!is_dst) ? TFSP_FLTYPE_DB1 : TFSP_FLTYPE_DB2;
			break;
		case DIVY_INFOTYPE_dbshfolder_e:
			ftype = (!is_dst) ? TFSP_FLTYPE_DBSH1 : TFSP_FLTYPE_DBSH2;
			break;
		default:
			ftype = TFSP_FLTYPE_UNKNOWN;
			break;
	}

	return ftype;
}

/**
 * method をtfsp_type_operation に変換する.
 *
 */
DIVY_DECLARE(tfsp_type_operation) divy_pi_method2operation(request_rec *r,
                                        divy_rdbo_resource *src_rdb_r,
                                        divy_rdbo_resource *dst_rdb_r, int is_update)
{
	tfsp_type_operation op;
	int mnum = divy_get_method_number(r);
	apr_pool_t *p = r->pool;

	if (dst_rdb_r == NULL && (mnum == M_MOVE || mnum == M_COPY)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA, "dst resource is EMPTY.");
		return TFSP_OPTYPE_UNKNOWN;
	}

	switch (mnum) {
		case M_PUT:
			if (is_update) {
				op = TFSP_OPTYPE_UPDATEPUT;
			}
			else {
				op = TFSP_OPTYPE_NEWPUT;
			}
			break;
		case M_MOVE:
		case M_COPY:
		{
			char *psrc_uri = divy_get_parenturi(p, src_rdb_r->uri);
			char *pdst_uri = divy_get_parenturi(p, dst_rdb_r->uri);

			if (IS_FILLED(psrc_uri) && IS_FILLED(pdst_uri) &&
				strcmp(psrc_uri, pdst_uri) == 0) {
				op = (mnum == M_MOVE) ? TFSP_OPTYPE_RENAME : TFSP_OPTYPE_DUPLICATE;
			}
			else {
				op = (mnum == M_MOVE) ? TFSP_OPTYPE_MOVE : TFSP_OPTYPE_COPY;
			}
			break;
		}
		default:
			op = TFSP_OPTYPE_UNKNOWN;
			break;
	}

	return op;
}

/**
 * rdb_r が示すリソースの実体暗号化を行なう.(実体暗号化プラグインラッパー)
 *
 */
DIVY_DECLARE(int) divy_pi_ccipher(request_rec *r, divy_rdbo_resource *rdb_r,
                                  const char *filepath, tfsp_ccipher_file **pcfile)
{
	tfs_status_t st;
	apr_pool_t *p = r->pool;
	void *sym     = NULL;
	tfsp_ihdr_ccipher *handle;
	tfsp_ccipher_file cfile = { 0 };
	tfsp_ctx_ccipher ctx    = { 0 };
	divy_rdbo_grp *grp_pr   = NULL;
	tfs_error_t *err = NULL;
	const tfsp_plugin_module *pm = NULL;
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	if (rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC, "rdb_r is EMPTY.");
		return DIVY_PL_ST_ERR;
	}

	/* 実体暗号化プラグインライセンスの確認 */
	if (sconf != NULL && !sconf->use_enccipher_opt) {
		return DIVY_PL_ST_NOLICENSE;	/* ライセンスが無かった */
	}

	/* 実体暗号化プラグインインターフェースの取得 */
	pm = divy_pi_lookupPm(r, TFSP_PTYPE_CCIPHER, NULL);
	if (pm == NULL) {
		return DIVY_PL_ST_NOPLUGIN;	/* 対象プラグインはなかった(エラーではない) */
	}
	sym = tfsp_pm_get_interface(pm);
	if (sym == NULL) {
		return DIVY_PL_ST_NOPLUGIN;	/* 対象プラグインはなかった(エラーではない) */
	}
	handle = (tfsp_ihdr_ccipher *) sym;

	if (rdb_r->u_spec) {
		cfile.srctype = divy_pi_infotype2foldertype(rdb_r->u_spec->infotype, 0);
	}
	cfile.op = divy_pi_method2operation(r, rdb_r, NULL, IS_FILLED(rdb_r->rsid));
	cfile.dsttype = TFSP_FLTYPE_UNKNOWN;
	cfile.path    = filepath;
	cfile.name    = rdb_r->displayname;
	cfile.ctype   = rdb_r->getcontenttype;

	/* グループID の算出 */
	if (divy_rdbo_get_group_property_by_resuri(r, rdb_r->uri, &grp_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get group information.(uri = %s)", rdb_r->uri);
		return DIVY_PL_ST_ERR;
	}
	cfile.groupid = (grp_pr != NULL) ? grp_pr->grpid : NULL;

	/* 実体暗号化コンテキスト生成 */
	ctx.child    = tfsp_pm_get_cctx(pm);
	ctx.location = dav_divy_get_root_uri(r);

	/* 実体暗号化の準備 */
	err = handle->prepare(_get_pi_req_pool(r), &ctx);
	if (err != NULL) {
		tfs_error_dumpall(err, NULL, _plugin_logger);
		tfs_error_destroy(err);
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to prepare ccipher.");
		handle->release(&ctx);	/* リソースを開放して終了 */
		return DIVY_PL_ST_ERR;
	}

	/* 実体暗号化を行なう */
	err = handle->ccipher(&ctx, &cfile);
	if (err != NULL) {
		st = tfs_error_getstatus(err);
		if (st == TFSP_NOPROCESS || st == TFSP_DECLINE) {
			tfs_error_destroy(err);

			/* 何も処理しなかった */
			handle->release(&ctx);	/* リソースを開放して終了 */
			return DIVY_PL_ST_NOPROCESS;
		}
		else if (st != TFSP_SUCCESS) {
			tfs_error_dumpall(err, NULL, _plugin_logger);
			tfs_error_destroy(err);

			handle->release(&ctx);	/* リソースを開放して終了 */
			return DIVY_PL_ST_ERR;
		}
	}

	/* cfile の内容を*pcfile にコピーする */
	*pcfile = apr_pcalloc(p, sizeof(tfsp_ccipher_file));
	(*pcfile)->path    = apr_pstrdup(p, cfile.path);
	(*pcfile)->name    = apr_pstrdup(p, cfile.name);
	(*pcfile)->ctype   = apr_pstrdup(p, cfile.ctype);
	(*pcfile)->op      = cfile.op;
	(*pcfile)->srctype = cfile.srctype;
	(*pcfile)->dsttype = cfile.dsttype;
	(*pcfile)->groupid = apr_pstrdup(p, cfile.groupid);
	(*pcfile)->opath   = apr_pstrdup(p, cfile.opath);
	(*pcfile)->oname   = apr_pstrdup(p, cfile.oname);
	(*pcfile)->octype  = apr_pstrdup(p, cfile.octype);
	(*pcfile)->ofsize  = cfile.ofsize;

	/* リソースを開放して終了 */
	handle->release(&ctx);

	return DIVY_PL_ST_OK;
}

/**
 * ロケーションメニューに表示する表示項目を取得する
 *
 */
DIVY_DECLARE(int) divy_pi_get_locationmenu_items(request_rec *r,
                                                 tfs_varray_t **items)
{
	tfs_varray_t *plugins = NULL;
	tfs_error_t *err = NULL;
	tfs_pool_t *pp   = _get_pi_req_pool(r);
	tfsp_ctx_loader *loader;
	tfsp_plugin_cpage *cpage;
	tfsp_plugin_module *pm;
	divy_lmenu_item *item;
	int len, i;
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	*items = NULL;	/* 初期化 */

	/* プラグインローダーを取得 */
	if ((loader = _lookup_loader(r->server->process->pconf)) == NULL) {
		return DIVY_PL_ST_NOPLUGIN;
	}

	/* プラグイン情報を取得する */
	err = tfsp_loader_lookupAllIf(pp, loader, &plugins);
	if (err != NULL) {
		tfs_error_dumpall(err, r, _plugin_logger);
		tfs_error_destroy(err);

		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to lookup all plugin's information.");
		return DIVY_PL_ST_ERR;
	}

	if (plugins == NULL) {
		return DIVY_PL_ST_NOPLUGIN;
	}
	len = tfs_varray_size(plugins);

	/* コンフィグページ情報を生成する */
	tfs_varray_create(pp, len, items);
	for (i = 0; i < len; i++) {
		pm = tfs_varray_get(plugins, i);
		if (pm == NULL || pm->cpageinfo == NULL || pm->cpageinfo->l_cpuri == NULL) {
			continue;
		}

		/* プラグインライセンスの確認 */
		if (pm->if_info != NULL && pm->if_info->itype == TFSP_PTYPE_CCIPHER &&
			!sconf->use_enccipher_opt) {
			continue;	/* 実体暗号化プラグインのライセンスがなければ列挙しない */
		}

		item = apr_pcalloc(r->pool, sizeof(divy_lmenu_item));

		cpage = pm->cpageinfo->l_cpuri;

		/* URI の組み立て */
		item->uri = divy_build_plugin_uri(r->pool, dav_divy_get_root_uri(r), pm->name, cpage->uri);

		item->iconpath = (IS_FILLED(cpage->iconpath)) ?
				apr_pstrdup(r->pool, cpage->iconpath) : "";
		item->displayname = (IS_FILLED(cpage->displayname)) ?
				apr_pstrdup(r->pool, cpage->displayname) : apr_pstrdup(r->pool, pm->name);
		item->displayname_en = (IS_FILLED(cpage->displayname_en)) ?
				apr_pstrdup(r->pool, cpage->displayname_en) : apr_pstrdup(r->pool, pm->name);

		if (cpage->screentype == TFSP_SCREENTYPE_FILE) {
			item->screentype = "FILE";
		}
		else if (cpage->screentype == TFSP_SCREENTYPE_PERSONAL) {
			item->screentype = "PERSONAL";
		}
		else {
			item->screentype = "UTILITY";	/* デフォルト値です */
		}

		item->usertype = NULL;
		if (cpage->use_admin) {
			item->usertype = "admin";
		}

		if (cpage->use_normal) {
			item->usertype = (item->usertype != NULL) ?
				apr_psprintf(r->pool, "%s-%s", item->usertype, "normal") : "normal";
		}

		if (cpage->use_limited) {
			item->usertype = (item->usertype != NULL) ?
				apr_psprintf(r->pool, "%s-%s", item->usertype, "limited") : "limited";
		}

		if (cpage->use_groupleader) {
			item->usertype = (item->usertype != NULL) ?
				apr_psprintf(r->pool, "%s-%s", item->usertype, "groupleader") : "groupleader";
		}

		/* 配列に追加 */
		tfs_varray_add(*items, item);
	}

	return DIVY_PL_ST_OK;
}

/**
 * 現在のロケーションのグループのうち、実体暗号化対象になっている
 * グループのグループID一覧を取得する. (実体暗号化プラグインラッパー)
 *
 */
DIVY_DECLARE(int) divy_pi_cciphered_grpset(request_rec *r, tfs_hset_t **groupid_set)
{
	tfs_status_t st;
	apr_pool_t *p = r->pool;
	void *sym     = NULL;
	tfsp_ihdr_ccipher *handle;
	tfsp_ctx_ccipher ctx    = { 0 };
	tfs_error_t *err = NULL;
	const tfsp_plugin_module *pm = NULL;
	tfs_hset_t *gset = NULL;
	tfs_hset_index_t *idx;
	const char *groupid;
	tfs_pool_t *pp = _get_pi_req_pool(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	/* 実体暗号化プラグインライセンスの確認 */
	if (sconf != NULL && !sconf->use_enccipher_opt) {
		return DIVY_PL_ST_NOLICENSE;	/* ライセンスが無かった */
	}

	/* 実体暗号化プラグインインターフェースの取得 */
	pm = divy_pi_lookupPm(r, TFSP_PTYPE_CCIPHER, NULL);
	if (pm == NULL) {
		return DIVY_PL_ST_NOPLUGIN;	/* 対象プラグインはなかった(エラーではない) */
	}
	sym = tfsp_pm_get_interface(pm);
	if (sym == NULL) {
		return DIVY_PL_ST_NOPLUGIN;	/* 対象プラグインはなかった(エラーではない) */
	}
	handle = (tfsp_ihdr_ccipher *) sym;

	/* 実体暗号化コンテキスト生成 */
	ctx.child    = tfsp_pm_get_cctx(pm);
	ctx.location = dav_divy_get_root_uri(r);

	/* 実体暗号化の準備 */
	err = handle->prepare(_get_pi_req_pool(r), &ctx);
	if (err != NULL) {
		tfs_error_dumpall(err, NULL, _plugin_logger);
		tfs_error_destroy(err);
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to prepare ccipher.");
		handle->release(&ctx);	/* リソースを開放して終了 */
		return DIVY_PL_ST_ERR;
	}

	/* グループ一覧を取得する */
	err = handle->cipheredGroupSet(&ctx, &gset);
	if (err != NULL) {
		st = tfs_error_getstatus(err);
		if (st == TFSP_NOPROCESS || st == TFSP_DECLINE) {
			tfs_error_destroy(err);

			/* 何も処理しなかった */
			handle->release(&ctx);	/* リソースを開放して終了 */
			return DIVY_PL_ST_NOPROCESS;
		}
		else if (st != TFSP_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get ccipherd group list.(code = %d)", st);
			tfs_error_dumpall(err, NULL, _plugin_logger);
			tfs_error_destroy(err);

			handle->release(&ctx);	/* リソースを開放して終了 */
			return DIVY_PL_ST_ERR;
		}
	}

	if (gset == NULL) {
		handle->release(&ctx);
		return DIVY_PL_ST_OK;
	}

	/* gset を groupid_set にコピーする */
	tfs_hset_create(pp, groupid_set);
	for (idx = tfs_hset_first(gset); idx != NULL; idx = tfs_hset_next(idx)) {
		tfs_hset_getvalue(idx, &groupid);
		tfs_hset_add(*groupid_set, tfs_pstrdup(pp, groupid));
	}

	/* リソースを開放して終了 */
	handle->release(&ctx);

	return DIVY_PL_ST_OK;
}

/**
 * 指定されたグループIDが示すグループが実体暗号化対象のグループであれば
 * プラグインに対象グループが削除されることを通知する.
 *
 */
DIVY_DECLARE(int) divy_pi_excludeTarget(request_rec *r, const char * groupid)
{
	tfs_status_t st;
	apr_pool_t *p = r->pool;
	void *sym     = NULL;
	tfsp_ihdr_ccipher *handle;
	tfsp_ctx_ccipher ctx    = { 0 };
	tfs_error_t *err = NULL;
	const tfsp_plugin_module *pm = NULL;

	/* 実体暗号化プラグインインターフェースの取得 */
	pm = divy_pi_lookupPm(r, TFSP_PTYPE_CCIPHER, NULL);
	if (pm == NULL) {
		return DIVY_PL_ST_NOPLUGIN;	/* 対象プラグインはなかった(エラーではない) */
	}
	sym = tfsp_pm_get_interface(pm);
	if (sym == NULL) {
		return DIVY_PL_ST_NOPLUGIN;	/* 対象プラグインはなかった(エラーではない) */
	}
	handle = (tfsp_ihdr_ccipher *) sym;

	/* 実体暗号化コンテキスト生成 */
	ctx.child    = tfsp_pm_get_cctx(pm);
	ctx.location = dav_divy_get_root_uri(r);

	/* 実体暗号化の準備 */
	err = handle->prepare(_get_pi_req_pool(r), &ctx);
	if (err != NULL) {
		tfs_error_dumpall(err, NULL, _plugin_logger);
		tfs_error_destroy(err);
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to prepare ccipher.");
		handle->release(&ctx);	/* リソースを開放して終了 */
		return DIVY_PL_ST_ERR;
	}

	/* グループの削除を通知する */
	err = handle->excludeTarget(&ctx, groupid);
	if (err != NULL) {
		st = tfs_error_getstatus(err);
		if (st == TFSP_NOPROCESS || st == TFSP_DECLINE) {
			tfs_error_destroy(err);

			/* 何も処理しなかった */
			handle->release(&ctx);	/* リソースを開放して終了 */
			return DIVY_PL_ST_NOPROCESS;
		}
		else if (st != TFSP_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove ccipherd group.(code = %d)", st);
			tfs_error_dumpall(err, NULL, _plugin_logger);
			tfs_error_destroy(err);

			handle->release(&ctx);	/* リソースを開放して終了 */
			return DIVY_PL_ST_ERR;
		}
	}

	/* リソースを開放して終了 */
	handle->release(&ctx);

	return DIVY_PL_ST_OK;
}

/**
 * r にキャッシュされているリクエストレベルプラグインプールを取得する.
 *
 */
DIVY_DECLARE(tfs_pool_t *) divy_get_pi_req_pool(request_rec *r)
{
	return _get_pi_req_pool(r);
}


/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
static apr_status_t _cleanup_loader(void *data)
{
	/* プラグインローダーを破棄する */
	if (data != NULL) {
		tfsp_loader_destroy(data);
	}

	return APR_SUCCESS;
}

/**
 * プラグインのChildプロセスダウン時の終了処理をコールバックする.
 *
 */
static apr_status_t _cterminate(void *data)
{
	tfsp_ctx_loader *loader = data;

	if (loader != NULL) {
		tfs_error_t *err = NULL;

		/* 終了ハンドラをコール */
		tfsp_loader_cterminate(loader, &err);
		if (err != NULL) {
			tfs_error_dumpall(err, NULL, _plugin_logger);
			tfs_error_destroy(err);
		}
	}

	return APR_SUCCESS;
}

/**
 * プラグインのリクエストレベルの終了処理.
 *
 */
static apr_status_t _rterminate(void *data)
{
	tfs_pool_t *pp = data;

	if (pp != NULL) {
		tfs_pool_destroy(pp);
	}

	return APR_SUCCESS;
}

/**
 * pconf にキャッシュされたプラグインローダをインスタンスを取得する.
 */
static tfsp_ctx_loader * _lookup_loader(apr_pool_t *pconf)
{
	if (pconf == NULL) return NULL;

	return divy_pcache_get_data(pconf, DIVY_PCACHE_DAT_GL_PLLOADER);
}

/**
 * エラーレベルlevel のエラーログdesc をApacheのエラーログに出力する.
 * (note)
 *   data にはrequest_rec * またはNULLを渡してください.
 *   NULL の場合には、ログレベルの調整が出来なくなります.
 * 
 * @param data void * 必ずrequest_rec * へのポインタを渡してください
 * @param int level エラーレベル
 * @param desc const char * エラーメッセージ
 */
static void _plugin_logger(void *data, int level, const char * desc)
{
	request_rec *r = data;

	/* 専用のロガーを使って出力する */
	divy_logger_outmsg(r, level, desc);
}

/**
 * 生成されたリクエストレベルのプラグインプール p を
 * キャッシュする.
 */
static void _set_pi_req_pool(request_rec *r, tfs_pool_t *p)
{
	apr_pool_t *req_p = (r->main != NULL) ? r->main->pool : r->pool;

	if (p == NULL) return;

	/* 作ったプールをr->pool にキャッシュしておく */
	divy_pcache_set_data(req_p, p, DIVY_PCACHE_DAT_REQ_PLPOOL);
}

/**
 * r にキャッシュされていたリクエストレベルプラグインプールを
 * 取得する.
 */
static tfs_pool_t * _get_pi_req_pool(request_rec *r)
{
	apr_pool_t *req_p = (r->main != NULL) ? r->main->pool : r->pool;

	return divy_pcache_get_data(req_p, DIVY_PCACHE_DAT_REQ_PLPOOL);
}

/**
 * Apache 用のサーバヘッダにプラグイン識別文字列を追加する.
 *
 * @param pconf apr_pool_t *
 * @param loader tfsp_ctx_loader * プラグインローダー
 */
static void _add_version_component(apr_pool_t *pconf, tfsp_ctx_loader *loader)
{
	tfs_varray_t *plugins = NULL;
	tfs_error_t *err = NULL;
	tfsp_plugin_module *pm;
	int len, i;

	if (pconf == NULL || loader == NULL) {
		return;
	}

	/* プラグイン情報を取得する */
	err = tfsp_loader_lookupAllIf(NULL, loader, &plugins);
	if (err != NULL) {
		tfs_error_dumpall(err, NULL, _plugin_logger);
		tfs_error_destroy(err);

		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to lookup all plugin's information.");
		return;
	}
	len = tfs_varray_size(plugins);

	/* コンフィグページ情報を生成する */
	for (i = 0; i < len; i++) {
		pm = tfs_varray_get(plugins, i);
		if (pm == NULL) {
			continue;
		}

		/* バージョン文字列を組み立てる
		 * (名称)/(バージョン番号)
		 */
		ap_add_version_component(pconf,
				apr_psprintf(pconf, "%s/%s", pm->name, pm->version));
	}
}

#else
	/* Noting to do */

#endif	/* DIVY_SUPPORT_PLUGIN */



