/**
 * $Id$
 *
 * tf_rdbo.c
 *
 * RDBのデータ構造をDAVプロバイダ(repos.c, livepro.cなど)から
 * 隠蔽する役割を持つヘッダファイル(tf_rdbo.h)の実装
 *
 * 変更履歴
 *
 * 2003/03/06 Thu takehara NEW
 *
 */
#include "httpd.h"
#include "http_protocol.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"
#include "apr_uuid.h"
#include "apr_lib.h"
#include "apr_general.h"

#include "mod_dav_tf.h"
#include "tf_db.h"
#include "util_db.h"
#include "tf_rdbo.h"
#include "tf_rdbo_group.h"
#include "lock.h"
#include "util.h"
#include "search.h"
#include "repos.h"
#include "util_ldap.h"
#include "util_ml.h"
#include "tf_storage.h"
#include "util_md5.h"
#include "tf_folder.h"
#include "tf_valuecache.h"
#include "tf_linkedlist.h"
#include "tf_plugin.h"
#include "tf_autodelete.h"
#include "tf_rdbo_util.h"
#include "tf_confirmreading.h"
#include "tf_box.h"
#include "util_auth.h"

APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Fixed value 
  ----------------------------------------------------------------------------*/
/*
 * テーブルデータの断片化を防ぐ目的で挿入するEtagのダミーデータ
 */
#define DIVY_DUMMY_ETAG  "'xxxxxxxxx1xxxxxxxxx2xxxxxxxxx3xxxxxxxxx4xxxxxxxxx5'"

/**
 * 関数divy_rdbo_remove_old_groupresources で利用する
 * 削除対象フォルダに関する情報を表す構造体
 */
typedef struct __divy_rdbo_delfolder	divy_rdbo_delfolder;
struct __divy_rdbo_delfolder {
	char *rsid;                   /* フォルダのリソースID */
	char *uri;                    /* フォルダのURI        */
	char *getcontenttype;         /* フォルダの種類       */

	/* 以下は作業用 */
	char *displayname;            /* 名前         */
	apr_int64_t getcontentlength; /* サイズ       */
	apr_time_t getlastmodified;   /* 最終更新日時 */
	char *lastmodifier_userid;    /* 最終更新者ID */
	char *lastmodifier;           /* 最終更新者名 */

	divy_rdbo_delfolder *next;    /* 次のリストへ */
};

/*------------------------------------------------------------------------------
  Define static values
  ----------------------------------------------------------------------------*/
/**
 * 関数_get_hierarchy_resourcestate_property で利用されるresourcestate の種類からなる配列
 */
static const int _resourcestate_types[] = {
	DIVY_RSTATE_TYPE_VIEW,
	DIVY_RSTATE_TYPE_BOX
};
#define RESOURCESTATE_TYPES_LEN		sizeof(_resourcestate_types)/sizeof(int)


/*--------------------------------------------------------------
  Declare private prototype functions 
  --------------------------------------------------------------*/
static int _remove_hierarchy_dead_property(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					divy_db_transaction_ctx *ts_ctx);
static int _remove_hierarchy_sharedcollection_property(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					divy_db_transaction_ctx *ts_ctx);
/* _remove_mailwatch_property のdelmode の値 */
#define DIVY_REMWATCH_NORMAL      0x00	/* 物理削除 & usrid とuriを使用して削除 */
#define DIVY_REMWATCH_LOGICAL     0x01	/* 論理削除 & usrid とuriを使用して更新 */
#define DIVY_REMWATCH_IGNORE_USER 0x02	/* usrid を削除/更新条件から外す        */
#define DIVY_REMWATCH_IGNORE_URI  0x04	/* uri を削除/更新条件から外す          */
static int _remove_mailwatch_property(request_rec *r,
					const char *uri,
					const char *usrid,
					int delmode,
					divy_db_transaction_ctx *ts_ctx);
static int _clean_gargage_mailwatch_property(request_rec *r,
					const char *uri,
					divy_db_transaction_ctx *ts_ctx);
static int _get_hierarchy_property(request_rec *r,
					   apr_pool_t *wp,
					   divy_rdbo_resource *rdb_r, 
					   int depth,
					   apr_uint32_t propflag,
					   divy_rdbo_sortorder *sort,
					   divy_db_transaction_ctx *ts_ctx);
static int _copy_dav_dead_property(request_rec *r,
					const divy_rdbo_resource *src_rdb_r,
					const divy_rdbo_resource *dist_rdb_r,
					int depth,
					divy_db_transaction_ctx *ts_ctx);
static int _get_dav_dead_property(request_rec *r,
					const char *uri,
					int depth,
					apr_hash_t **dp_list_hash,
					divy_db_transaction_ctx *ts_ctx);
static int _insert_dead_property(request_rec *r,
					const divy_rdbo_dproperty *d_pr,
					divy_db_transaction_ctx *ts_ctx);
static int _remove_dead_property(request_rec *r,
					const divy_rdbo_dproperty *d_pr,
					divy_db_transaction_ctx *ts_ctx);
static int _insert_property(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					divy_db_transaction_ctx *ts_ctx);
static int _get_merge_mailwatch_property(request_rec *r, 
					const char *uri,
					const char *usrid, 
					divy_rdbo_mailwatch **mw,
					divy_db_transaction_ctx *ts_ctx);
static int _remove_clmodule_property(request_rec *r,
					const char *uri,
					divy_db_transaction_ctx *ts_ctx);
static int _get_sharedcollection_property(request_rec *r,
					const char *uri,
					divy_rdbo_shlink **shlink_pr,
					divy_db_transaction_ctx *ts_ctx);
static int _insert_mailwatch_property(request_rec *r,
					const divy_rdbo_mailwatch *mwatch_pr,
					divy_db_transaction_ctx *ts_ctx);
static int _update_mailwatch_property(request_rec *r,
					const divy_rdbo_mailwatch *mwatch_pr,
					divy_db_transaction_ctx *ts_ctx);
static int _insert_grpmem(request_rec *r,
					const char *grpid,
					const char *usrid,
					divy_db_transaction_ctx *ts_ctx);
static int _insert_rusr(request_rec *r,
					const divy_rdbo_rusr *rusr,
					divy_db_transaction_ctx *ts_ctx);
static int _remove_grpmem(request_rec *r,
					const char *grpid,
					const char *usrid,
					divy_db_transaction_ctx *ts_ctx);
static int _insert_sqlmem_using_name(request_rec *r,
					const char *grpid,
					const char *sqlname,
					divy_db_transaction_ctx *ts_ctx);
static int _insert_rsql(request_rec *r,
					const divy_rdbo_rsql *rsql,
					divy_db_transaction_ctx *ts_ctx);
static int _remove_sqlmem(request_rec *r,
					const char *grpid,
					const char *sqlid,
					divy_db_transaction_ctx *ts_ctx);
static int _insert_divy_grpmem(request_rec *r,
					const char *grpid,
					const char *usrid,
					divy_db_transaction_ctx *ts_ctx);
static void _fill_default_collection_property(request_rec *r,
					divy_rdbo_resource *rdb_r);
static int _change_used_userquota(request_rec *r,
					divy_rdbo_usr *usr_pr,
					divy_db_transaction_ctx *ts_ctx);
#ifdef DIVY_SUPPORT_EXTENDQUOTA
static int _change_used_sysquota(request_rec *r,
					divy_rdbo_diskquota *sysquota_pr,
					divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
#ifdef DIVY_SUPPORT_GROUPQUOTA
static int _change_used_grpquota(request_rec *r,
					divy_rdbo_grpquota *grpquota_pr,
					divy_db_transaction_ctx *tx_ctx);
#endif /* DIVY_SUPPORT_GROUPQUOTA */
static int _lock_resource_entry(request_rec *r,
					divy_rdbo_resource *rdb_r,
					divy_db_transaction_ctx *ts_ctx);
static int _get_trash_property(request_rec *r,
					const char *uri,
					int depth,
					apr_hash_t **trash_pr_hash,
					divy_db_transaction_ctx *ts_ctx);
static int _insert_trash_property(request_rec *r,
					const char *rsid,
					time_t deletion_bi,
					const char *deleter_id,
					divy_db_transaction_ctx *ts_ctx);
static int _remove_trash_property(request_rec *r,
					const char *rsid,
					divy_db_transaction_ctx *ts_ctx);
static int _remove_hierarchy_trash_property(request_rec *r,
					const char *trash_uri,
					divy_db_transaction_ctx *ts_ctx);

static int _get_hierarchy_resourcestate_property(request_rec *r,
					const char *uri,
					int depth,
					apr_hash_t **rstate_pr_hash,
					divy_db_transaction_ctx *ts_ctx);
static int _get_parent_resourcestate_property(request_rec *r,
					const char *uri,
					int type,
					divy_rdbo_resourcestate **rstate_pr,
					divy_db_transaction_ctx *ts_ctx);
static int _get_resourcestate_property(request_rec *r,
					const char *uri,
					int type,
					divy_rdbo_resourcestate **rstate_pr,
					divy_db_transaction_ctx *ts_ctx);
static int _get_resourcestate_property_with_rsid(request_rec *r,
					const char *uri,
					divy_rdbo_resourcestate **rstate_pr,
					divy_db_transaction_ctx *ts_ctx);
static int _get_inherited_resourcestate_property_with_rsid(request_rec *r,
					const char *uri,
					int type,
					divy_rdbo_resourcestate **rstate_pr,
					divy_db_transaction_ctx *ts_ctx);
static int _get_inherited_resourcestate_property_with_rsid_by_hash(request_rec *r,
					const char *uri,
					int type,
					apr_hash_t **rstate_pr_hash,
					divy_db_transaction_ctx *ts_ctx);
static int _remove_resourcestate_property(request_rec *r,
					int depth,
					const char *uri,
					int type,
					divy_db_transaction_ctx *ts_ctx);
static int _insert_resourcestate_property(request_rec *r,
					divy_rdbo_resourcestate *rstate_pr,
					divy_db_transaction_ctx *ts_ctx);
static int _remove_box_property(request_rec *r,
					const char *uri,
					divy_db_transaction_ctx *ts_ctx);
static int _remove_old_groupfolders(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_time_t bordertime_sec,
					divy_autodel_dresource **del_res,
					divy_db_transaction_ctx *ts_ctx);
static int _get_confirmreading(request_rec *r, const char *uri,
					const char *userid, int depth, apr_hash_t **cr_pr_hash,
					divy_db_transaction_ctx *ts_ctx);
static char* _make_session_sid(apr_pool_t *p);

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * 指定されたrdb_r のリソースタイプを取得する。
 */
DIVY_DECLARE(divy_rdbo_resourcetype) divy_rdbo_get_resourcetype(request_rec *r,
                                                  divy_rdbo_resource *rdb_r)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	divy_rdbo_resourcetype resourcetype;

	if (rdb_r == NULL || rdb_r->uri == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rdb_r or uri is NULL.");
		return DIVY_TYPE_UNKNOWN;
	}

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.(uri = %s)", rdb_r->uri);
		return DIVY_TYPE_UNKNOWN;
	}

	dbconn->startTrans(dbconn, 0);
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT rs_resourcetype_i "
				"FROM dav_resource "
				"WHERE rs_uri_txt = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s) "
			"Reason: %s", rdb_r->uri, stmt->getMsg(stmt));

		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return DIVY_TYPE_UNKNOWN;
	}

	stmt->setString(stmt, 1, dav_divy_remove_endslash(p, rdb_r->uri));
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(uri = %s) "
			"Reason: %s", rdb_r->uri, rset->getMsg(rset));

		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return DIVY_TYPE_UNKNOWN;
	}
	/* 結果の取得 */
	if (rset->next(rset) == DB_TRUE) {
		resourcetype = (divy_rdbo_resourcetype) rset->getInt(rset, 1);
	}
	else {
		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		/* 取得出来なかった */
		return DIVY_TYPE_UNKNOWN;
	}

	dbconn->commit(dbconn);
	/* 使用した領域を解放 (dbconn は何もしてはならない) */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return (resourcetype ==0 ? DIVY_TYPE_RESOURCE : DIVY_TYPE_COLLECTION);
}

/**
 * 指定されたdepth (Depth ヘッダの値)で、divy_rdbo_resource 以下に存在する全ての
 * リソースまたはコレクションのプロパティを取得する。
 * 引数 rdb_r に設定されていた値はつぶれてしまうかもしれません。
 */
DIVY_DECLARE(int) divy_rdbo_get_hierarchy_property(request_rec *r,
						divy_rdbo_resource *rdb_r,
						int depth,
						apr_uint32_t propflag, divy_rdbo_sortorder *sort)
{
	int status;
	apr_pool_t *wp;

	/* 作業用のプールを取得する(メモリ縮小対策) */
	wp = divy_get_request_temporary_pool(r);
	if (wp == NULL) wp = r->pool;

	/* 指定されたdepth 値で rdb_r->uri 以下のリソース＆コレクションを取得 */
	status = _get_hierarchy_property(r, wp, rdb_r, depth, propflag, sort, NULL);
	if (status) {
		ERRLOG3(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get properties. (uri = %s, depth = %d,"
			"propflag = %d)", rdb_r->uri, depth, propflag);
		return status;
	}

	return 0;
}

/**
 * 指定された新規リソースrdb_r のプロパティを DB に書き込む。(新規)
 * rdb_r がリソースの場合、"更新者"のQuotaを消費します。
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_property(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					divy_db_transaction_ctx *ts_ctx)
{
	int ret;
	int iscommit  = 0;
	apr_pool_t *p = r->pool;
	divy_rdbo_resource p_rdb_r = { 0 };	/* 親リソース */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;

	/* 親リソースのロックと情報の取得 */
	p_rdb_r.uri = divy_get_parenturi(p, rdb_r->uri);
	ret = _lock_resource_entry(r, &p_rdb_r, ts_ctx);
	if (ret == DIVY_STCODE_NOT_EXIST) {
		/* 恐らく処理中に親フォルダが削除されてしまった(レアケース) */
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The resource of parent is missing. "
			"Maybe it was deleted before this operation. "
			"so we give up insert this entry."
			"(parent = %s, own = %s)", p_rdb_r.uri, rdb_r->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return DIVY_STCODE_PARENT_NOT_EXIST;
	}
	else if (ret) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"Failed to get parent resource information."
			"(parent = %s, own = %s)", p_rdb_r.uri, rdb_r->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/*
	 * dav_resourceテーブルへの新規追加
	 * rsid,etagは作成済みの前提です
	 */
	if (_insert_property(r, rdb_r, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert dav_resource. (uri = %s)", rdb_r->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * システムQuotaとユーザQuotaの変更
	 * [ 処理開始条件 ]
	 * 	リソースであること
	 *
	 * [ ルール ]
	 * 	システムQuotaと"更新者"のユーザQuotaを消費させる
	 */
	if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
		divy_rdbo_usr uquota       = { 0 };
#ifdef DIVY_SUPPORT_EXTENDQUOTA
		divy_rdbo_diskquota squota = { 0 };

		/* システムQuotaの更新 */
		squota.usedst  = rdb_r->getcontentlength;
		squota.usedres = APR_INT64_C(1);

		ret = _change_used_sysquota(r, &squota, ts_ctx);
		if (ret) {
			if (ret != DIVY_STCODE_QUOTAOVER) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update system quota."
					"(root_uri = %s, userid = %s)",
					REPLACE_NULL(squota.uri),
					rdb_r->lastmodifier_userid);
			}
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return ret;
		}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

		/* ユーザQuotaの更新 */
		uquota.usrid   = rdb_r->lastmodifier_userid;	/* 更新者のユーザID */
		uquota.usedst  = rdb_r->getcontentlength;
		uquota.usedres = APR_INT64_C(1);
		uquota.next    = NULL;

		ret = _change_used_userquota(r, &uquota, ts_ctx);
		if (ret) {
			if (ret != DIVY_STCODE_QUOTAOVER) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update user quota."
					"(owner = %s)", uquota.usrid);
			}
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return ret;
		}
	}

	/*
	 * 残留しているかもしれない開封通知を削除する
	 * (note)
	 *   運用中に開封通知フラグをon -> off -> on すると、開封通知が残留する可能性あり.
	 *   そのため、残留しているかもしれない開封通知を削除しておきます
	 */
	if (divy_support_confirmreading(r) && rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
		divy_uri_spec *u_spec = rdb_r->u_spec;

		if (u_spec == NULL) {
			(void) divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &u_spec);
		}

		if (u_spec != NULL && u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {
			if (divy_rdbo_remove_confirmreading(r, rdb_r->uri, NULL/*ユーザID無視*/, 0, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to remove confirmreading property. (uri = %s)", rdb_r->uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}
	}

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 既に存在していたrdb_r が示すリソースまたはコレクションのプロパティを
 * DB に書き込む。(更新)
 * rdb_r がリソースの場合、"作成者"に以前消費していたリソースのQuotaを返し、
 * "更新者"のQuotaを消費します。
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_property(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int change_stowner, ret;
	divy_rdbo_resource o_rdb_r = { 0 };	/* 自身のリソース */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* 自身のリソースのロックと情報の取得 */
	o_rdb_r.uri = rdb_r->uri;

	ret = _lock_resource_entry(r, &o_rdb_r, ts_ctx);
	if (ret == DIVY_STCODE_NOT_EXIST) {
		/* 恐らく処理中に自分が削除されてしまった(レアケース) */
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The resource is missing. "
			"Maybe it was deleted before this operation. "
			"so we give up insert this entry."
			"(uri = %s)", rdb_r->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return DIVY_STCODE_NOT_EXIST;
	}
	else if (ret) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"Failed to get resource information."
			"(uri = %s)", rdb_r->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/*
	 * 開封通知の削除
	 * (note) 上書きしたら開封通知は消えます(外部仕様)
	 */
	if (divy_support_confirmreading(r)) {
		divy_uri_spec *u_spec = rdb_r->u_spec;
		if (u_spec == NULL) {
			(void) divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &u_spec);
		}

		if (u_spec && u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {
			if (divy_rdbo_remove_confirmreading(r, rdb_r->uri,
											NULL/*ユーザID無視*/, 0/*自分自身*/, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to remove confirmreading property. (uri = %s)", rdb_r->uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}
	}

	/*
	 * dav_resource テーブルの更新
	 * rsid,etag は作成済みの前提です
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"UPDATE dav_resource "
				"SET "
				"rs_get_cont_lang_vc = ?,"
				"rs_get_cont_len_bi = ?,"
				"rs_get_cont_type_vc = ?,"
				"rs_get_etag_txt = ?,"
				"rs_get_lastmodified_bi = ?,"
				"rs_isversioned_i = ?,"
				"rs_checkin_i = ?,"
				"rs_checkout_i = ?, "
				"rs_lastmodifier_usr_id_vc = ? "
				"WHERE rs_rs_id_c = ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for dav_resource update. "
			"(rsid = %s, uri = %s) Reason: %s", 
			rdb_r->rsid, rdb_r->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	stmt->setString(stmt, 1, REPLACE_EMPTYSTR(rdb_r->getcontentlanguage));
	stmt->setBigInt(stmt, 2, rdb_r->getcontentlength);
	stmt->setString(stmt, 3, REPLACE_EMPTYSTR(rdb_r->getcontenttype));
	stmt->setString(stmt, 4, rdb_r->getetag);
	stmt->setBigInt(stmt, 5, rdb_r->getlastmodified);
	stmt->setInt(stmt,    6, rdb_r->isversioned);
	stmt->setInt(stmt,    7, rdb_r->checkin);
	stmt->setInt(stmt,    8, rdb_r->checkout);
	stmt->setString(stmt, 9, rdb_r->lastmodifier_userid);
	stmt->setString(stmt,10, rdb_r->rsid);

	/* SQL文の実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to executeUpdate for dav_resource update. "
			"(rsid = %s, uri = %s) Reason: %s", 
			rdb_r->rsid, rdb_r->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* 使い終わった資源の解放 */
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * システムQuotaとユーザQuotaの変更
	 * [ 処理開始条件 ]
	 * 	* リソースであること and
	 * 	* 旧"更新者"と新"更新者"が変わっていること or
	 * 	  ファイルサイズが変わっている
	 * [ ルール ]
	 * 	旧"更新者"には消費していたQuotaを返し
	 * 	新"更新者"のQuotaを消費させる。
	 */
	if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE &&
	    strcmp(rdb_r->lastmodifier_userid, o_rdb_r.lastmodifier_userid) != 0) {
		change_stowner = 1;	/* ストレージのOwnerが変わった */
	}
	else {
		change_stowner = 0;	/* 変わらない */
	}

#ifdef DIVY_SUPPORT_EXTENDQUOTA
	/* システムQuotaの更新 */
	if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE &&
	    rdb_r->getcontentlength != o_rdb_r.getcontentlength) {

		divy_rdbo_diskquota squota = { 0 };

		squota.usedst  = rdb_r->getcontentlength - o_rdb_r.getcontentlength;
		squota.usedres = APR_INT64_C(0);

		ret = _change_used_sysquota(r, &squota, ts_ctx);
		if (ret) {
			if (ret != DIVY_STCODE_QUOTAOVER) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update system quota."
					"(root_uri = %s, userid = %s)",
					REPLACE_NULL(squota.uri),
					rdb_r->lastmodifier_userid);
			}
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return ret;
		}
	}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

	/* ユーザQuotaの更新 */
	if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE && (change_stowner ||
	     rdb_r->getcontentlength != o_rdb_r.getcontentlength)) {

		divy_rdbo_usr uquota = { 0 };

		/* 作成者と更新者が異なる */
		if (change_stowner) {
			divy_rdbo_usr double_uquota[2] = { {0},{0} };

			/* "旧更新者"にQuotaを返し、"新更新者"のQuotaを消費する */

			/* "旧更新者"情報 */
			double_uquota[0].usrid   = o_rdb_r.lastmodifier_userid;
			double_uquota[0].usedst  = o_rdb_r.getcontentlength * (-1);
			double_uquota[0].usedres = APR_INT64_C(-1);
			double_uquota[0].next    = &double_uquota[1];

			/* "新更新者"情報 */
			double_uquota[1].usrid   = rdb_r->lastmodifier_userid;
			double_uquota[1].usedst  = rdb_r->getcontentlength;
			double_uquota[1].usedres = APR_INT64_C(1);
			double_uquota[1].next    = NULL;

			ret = _change_used_userquota(r, double_uquota, ts_ctx);
			if (ret) {
				if (ret != DIVY_STCODE_QUOTAOVER) {
					ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to update user quota."
						"(old owner = %s, new owner = %s)",
						double_uquota[0].usrid,
						double_uquota[1].usrid);
				}
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);

				return ret;
			}
		}
		/* 作成者と更新者が同じ時 */
		else {
			uquota.usrid   = rdb_r->lastmodifier_userid;
			uquota.usedst  = rdb_r->getcontentlength -
					 o_rdb_r.getcontentlength;
			uquota.usedres = APR_INT64_C(0);	/* ファイル数は不変 */
			uquota.next    = NULL;

			ret = _change_used_userquota(r, &uquota, ts_ctx);
			/* 更新者が存在しないのは許容できない */
			if (ret) {
				if (ret != DIVY_STCODE_QUOTAOVER) {
					ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to update user quota."
						"(owner = %s)", uquota.usrid);
				}
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);

				return ret;
			}
		}
	}

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * リポジトリDBを検索する(basicsearch)
 *
 * 
 */ 
DIVY_DECLARE(int) divy_rdbo_basicsearch(request_rec *r,
					divy_rdbo_resource *rdb_r, 
					const char *sqlstr,
					int escflg,
					apr_hash_t *bindinfo_h)
{
	DbConn *dbconn = NULL;
	DbPreparedStmt *stmt = NULL;
	DbResultSet *rset = NULL;
	apr_pool_t *p = r->pool;
	int i;
	char *colname;
	divy_rdbo_resource *res_cur;
	int bindcnt;
	const char *hkey;
	apr_hash_index_t *hi;
	divy_search_bs_bind *bindinfo;
    
	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbConn.");
		return 1;
	}

	/* 接続 */
	dbconn->startTrans(dbconn, 0);

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, sqlstr, p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", 
				REPLACE_NULL(stmt->getMsg(stmt)));
		dbconn->rollback(dbconn);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, rdb_r->uri);
	bindcnt = 1;
	/* エスケープする場合 */
	if (escflg){
		stmt->setString(stmt, 2, apr_pstrcat(p, stmt->escWildCard(stmt,
						rdb_r->uri), "/%", NULL));
		bindcnt = 2;
	}
	/* where 指定があった場合はバインド */
	if (bindinfo_h){
		for (hi = apr_hash_first(p, bindinfo_h), bindcnt = bindcnt + 1;
				hi; hi = apr_hash_next(hi), bindcnt++){
			/* バインド情報の構造体を取得 */
			apr_hash_this(hi, (const void **)&hkey, NULL, 
					(void **)&bindinfo);

			switch (bindinfo->type){
				case DIVY_SEARCH_BIGLONG_TT:
					stmt->setBigInt(stmt, bindcnt, 
							bindinfo->ttval);
					break;
				case DIVY_SEARCH_BIGLONG_I64:
					stmt->setBigInt(stmt, bindcnt,
							bindinfo->i64val);
					break;
				case DIVY_SEARCH_STRING:
					stmt->setString(stmt, bindcnt,
							bindinfo->strval);
					break;
			}
		}
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. Reason: %s", 
				REPLACE_NULL(rset->getMsg(rset)));
		dbconn->rollback(dbconn);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	res_cur = rdb_r;

	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE){

		res_cur->next = apr_pcalloc(p, sizeof(divy_rdbo_resource));
		res_cur = res_cur->next;

		/* カラム数分ループ */
		for (i = 1; i <= rset->getColumnCount(rset); i++){

			colname = rset->getColumnName(rset, i);

			/* 比較のため小文字に変換する */
			ap_str_tolower(colname);

			/* 列名により分岐 */
			if (strcmp(colname, "rs_uri_txt") == 0){
				res_cur->uri = rset->getString(rset, i);
			} else if (strcmp(colname, "rs_create_bi") == 0){
				res_cur->creationdate =
					(time_t) rset->getBigInt(rset, i);
			} else if (strcmp(colname, "rs_dispname_vc") == 0){
				res_cur->displayname = rset->getString(rset, i);
			} else if (strcmp(colname, "rs_get_cont_lang_vc") == 0){
				res_cur->getcontentlanguage = rset->getString
								(rset, i);
			} else if (strcmp(colname, "rs_get_cont_len_bi") == 0){
				res_cur->getcontentlength = rset->getBigInt(rset, i);
			} else if (strcmp(colname, "rs_get_cont_type_vc") == 0){
				res_cur->getcontenttype = rset->getString
								(rset, i);
			} else if (strcmp(colname, "rs_get_etag_txt") == 0){
				res_cur->getetag = rset->getString(rset, i);
			} else if (strcmp(colname, "rs_get_lastmodified_bi") == 0){
				res_cur->getlastmodified =
					(time_t) rset->getBigInt(rset, i);
			} else if (strcmp(colname, "rs_resourcetype_i") == 0){
				res_cur->resourcetype = rset->getInt(rset, i);
			}
			else if (strcmp(colname, "rs_creator_vc") == 0){
				res_cur->creator = rset->getString(rset, i);
			}
			else if (strcmp(colname, "rs_lastmodifier_vc") == 0){
				res_cur->lastmodifier = rset->getString(rset, i);
			}
    	    
		}
	}
	dbconn->commit(dbconn);
    
	/* 使用した領域を解放 (dbconn は何もしてはならない) */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 更新クライアントの表示情報を取得する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_updateinformationsearch(divy_rdbo_search_params *params,
						divy_search_updateis_iscreen *screen,
						divy_rdbo_search_output *output)
{
	request_rec *r          = params->r;
	apr_pool_t *p		= params->r->pool;
	apr_pool_t *wp          = params->wp;
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	char *sqlstr            = NULL;

	divy_rdbo_clmodule *clmodule = NULL;
	divy_db_transaction_ctx *ts_ctx   = NULL;

	output->list.clmodule_pr = NULL;  /* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * SQL文の作成
	 */
	sqlstr =	"SELECT "
			" rs.rs_dispname_vc"
			", rs.rs_get_cont_len_bi"
			", clm.cm_ver_vc"
			", clm.cm_lineup_vc"
			", clm.cm_digest_vc";

	/* detaillist */
	if (params->optflg == DIVY_SEARCH_OPT_DETAILLIST) {
		sqlstr = apr_pstrcat(wp, sqlstr,
				", rs.rs_create_bi"
				", rs.rs_get_lastmodified_bi"
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_DB2) /* postgres / db2 */
				" FROM divy_clmodule clm"
				" INNER JOIN dav_resource rs"
				" ON clm.cm_rs_id_c = rs.rs_rs_id_c",
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				" FROM divy_clmodule clm, dav_resource rs"
				" WHERE clm.cm_rs_id_c = rs.rs_rs_id_c",
#endif
				NULL);
	}
	/* lineup */
	else if (params->optflg == DIVY_SEARCH_OPT_LINEUP) {
		sqlstr = apr_pstrcat(wp, sqlstr,
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_DB2) /* postgres / db2 */
				" FROM divy_clmodule clm"
				" INNER JOIN dav_resource rs"
				" ON clm.cm_rs_id_c = rs.rs_rs_id_c"
				" WHERE clm.cm_lineup_vc = ?",
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				" FROM divy_clmodule clm, dav_resource rs"
				" WHERE clm.cm_rs_id_c = rs.rs_rs_id_c"
				" AND clm.cm_lineup_vc = ?",
#endif
				NULL);
	}
	else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
			"Unknown optflg found. (code = %d)", params->optflg);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, sqlstr, wp);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s",
			stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	if (params->optflg == DIVY_SEARCH_OPT_LINEUP) {
		stmt->setString(stmt, 1, screen->lineup);
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE) {
		/* 領域確保 */
		if (output->list.clmodule_pr == NULL) {
			output->list.clmodule_pr = clmodule =
					apr_pcalloc(wp, sizeof(divy_rdbo_clmodule));
		}
		else {
			clmodule->next 	= apr_pcalloc(wp, sizeof(divy_rdbo_clmodule));
			clmodule = clmodule->next;
		}

		/* データを取得 */
		clmodule->name			= rset->getString(rset, 1);
		clmodule->getcontentlength	= rset->getBigInt(rset, 2);
		clmodule->version	 	= rset->getString(rset, 3);
		clmodule->lineup 		= rset->getString(rset, 4);
		clmodule->digest 		= rset->getString(rset, 5);

		/* detaillistの場合はさらに取得 */
		if (params->optflg == DIVY_SEARCH_OPT_DETAILLIST) {
			divy_format_time_t(p, rset->getBigInt(rset, 6),
				DIVY_TIME_STYLE_ISO8601, &clmodule->registdt);
			divy_format_time_t(p, rset->getBigInt(rset, 7),
				DIVY_TIME_STYLE_ISO8601, &clmodule->updatedt);
		}
	}

	divy_db_commit_transaction(ts_ctx);

	/* 使用した領域を解放 (dbconn は何もしてはならない) */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたロックレコードを削除する。
 * 関数名はロックnullとなっているが、これはMOVEやDELETE時にそのURIが
 * ロックされていた場合でもこの関数が呼ばれます。
 *
 * lock.c側でLOCKTOKENが存在しないURIの削除は全てこの関数で
 * 処理を行うようにしています。
 * 対象のURIのロックレコードが削除できなくても目的達成（削除処理）
 * の為エラーを出力しません。
 *
 * @param  r 		request_rec *
 * @param  lockrec	divy_rdbo_lock *
 * @return status 	int (0: 正常 / 1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_locknull_member(request_rec * r,
						divy_rdbo_lock *lockrec)
{

	DbConn		*dbconn  = NULL;
	DbPreparedStmt	*stmt	 = NULL;
	char		*uri	 = NULL;

	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.");
		return 1;
	}

	dbconn->startTrans(dbconn, 0);

	TRACE(r->pool);

	/* ロックレコード削除用のSQLを準備 */
	stmt = dbconn->prepareStatement(dbconn,
				"DELETE FROM dav_lock "
				"WHERE lk_uri_txt like ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE
				"OR    lk_uri_txt = ?", r->pool);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s",
			stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	for (; lockrec != NULL; lockrec = lockrec->next) {

		uri = dav_divy_remove_endslash(r->pool, lockrec->uri);

		if (IS_EMPTY(uri)) {
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Delete record URI is NULL or 0 length.");
			dbconn->rollback(dbconn);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		/*
		 * uriを直接削除する。
		 * NULLLOCKでなくても削除を行います。
		 * これはDELETEやMOVEの副作用で対象のURIのロックが残ってはなら
		 * ないとの理由からです。(RFC2518)
		 * コレクションのダイレクトロックは同じURIでスラッシュの有り無し
		 * のレコードを持っているため、どちらも消すように考慮する
		 * しかし、目的は削除なので消せなくても（result count = 0)エラー
		 * を表示することは行いません。
		 */
		stmt->setString(stmt, 1, apr_pstrcat(r->pool,
					stmt->escWildCard(stmt, uri), "/%", NULL));
		stmt->setString(stmt, 2, uri);

		(void)stmt->executeUpdate(stmt, r->pool);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to executeUpdate for delete."
				"(uri = %s) Reason: %s",
			       	uri , stmt->getMsg(stmt));

			dbconn->rollback(dbconn);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

	}

	dbconn->commit(dbconn);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたユーザID userid のユーザが持っていた全てのロック及び
 * ロックNULLリソースを削除する。
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 *
 * @param r request_rec *
 * @param userid const char * 対象ユーザ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 /1: 失敗)
 * 	削除すべきロックやロックNULLリソースが存在しなくても成功扱いとします。
 */
DIVY_DECLARE(int) divy_rdbo_remove_user_lock(request_rec *r, const char *userid,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;

	if (IS_EMPTY(userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"userid is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * LOCKのAuthユーザと比べて削除する
	 * 削除件数にかかわらず処理は正常とする
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"DELETE FROM dav_lock "
				"WHERE lk_auth_usr_id_vc = ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
		      "Failed to get DbPreparedStmt. Reason: %s",
		      stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (stmt != NULL) stmt->close(stmt);
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	stmt->setString(stmt, 1, userid);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete dav_lock. (userid = %s) Reason: %s",
			userid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたdivy_rdbo_resource のリソースまたはコレクションをDB から削除し、
 * そのリソースまたはコレクション以下のリソースが消費していたQuotaを
 * "更新者"に返します。
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_resource(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					divy_linkedlist_t **del_filelist,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn         *dbconn = NULL;
	DbPreparedStmt *stmt   = NULL;
	DbResultSet    *rset   = NULL;
	apr_pool_t     *p      = r->pool;
	int iscommit           = 0;
	int ret;
	int support_trash      = divy_support_trashfolder(r);
	int support_extstatus  = divy_support_extenduserstatus(r);
	const char *trash_uri  = NULL;
	divy_uri_spec  *u_spec = rdb_r->u_spec;
	divy_rdbo_mailwatch *mw    = NULL;
	divy_rdbo_resource o_rdb_r = { 0 };	/* 自身のリソース */
	divy_rdbo_usr *p_uquota    = NULL;
	divy_linkedlist_t *clist   = NULL;
	divy_rdbo_usr uquota       = { 0 };
#ifdef DIVY_SUPPORT_EXTENDQUOTA
	divy_rdbo_diskquota squota = { 0 };
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
#ifdef DIVY_SUPPORT_GROUPQUOTA
	divy_rdbo_grpquota gquota = { 0 };
#endif  /* DIVY_SUPPORT_GROUPQUOTA */

	if (rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rdb_r is NULL.");
		return 1;
	}

	if (u_spec == NULL) {
		divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &u_spec);
		if (u_spec == NULL) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to parse uri. (uri = %s)", rdb_r->uri);
			return 1;
		}
	}

	*del_filelist = NULL;	/* 初期化 */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * リソースのロックと情報の取得
	 */
	o_rdb_r.uri = rdb_r->uri;

	/* 削除対象がリソースの場合 */
	if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {

		ret = _lock_resource_entry(r, &o_rdb_r, ts_ctx);
		if (ret == DIVY_STCODE_NOT_EXIST) {
			/* 恐らく処理中に自分が削除されてしまった(レアケース) */
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The resource is missing. "
				"Maybe it was deleted before this operation. "
				"so we give up delete this entry."
				"(uri = %s)", o_rdb_r.uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return DIVY_STCODE_NOT_EXIST;
		}
		else if (ret) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"Failed to get resource information."
				"(uri = %s)", o_rdb_r.uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return 1;
		}

		/* 減算するユーザQuotaの算出 */
		uquota.usrid   = o_rdb_r.lastmodifier_userid;	/* 更新者 */
		uquota.usedst  = APR_INT64_C(-1) * o_rdb_r.getcontentlength;
		uquota.usedres = APR_INT64_C(-1);
		uquota.next    = NULL;

#ifdef DIVY_SUPPORT_EXTENDQUOTA
		/* 減算するシステムQuotaの算出 */
		squota.usedst  = uquota.usedst;
		squota.usedres = uquota.usedres;
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

#ifdef DIVY_SUPPORT_GROUPQUOTA
		/* FIXME 
		   ここではまだグループフォルダ配下なのか分かってませんよ
		 */
		/* 減算するグループQuotaの算出 */
		gquota.usedst  = uquota.usedst;
		gquota.usedres = gquota.usedres;
#endif /* DIVY_SUPPORT_GROUPQUOTA */
		/* 削除する物理ファイルのパスを取得 */
		*del_filelist = apr_pcalloc(p, sizeof(divy_linkedlist_t));
		(*del_filelist)->val = o_rdb_r.physicalpath;
	}
	/* 削除対象がコレクションの場合 */
	else {
		divy_rdbo_resourcetype resourcetype;
		apr_int64_t u_usedst, u_usedres;
		char *userid;
		int found_resource = 0;

#ifdef DIVY_SUPPORT_EXTENDQUOTA
		squota.usedst  = APR_INT64_C(0);
		squota.usedres = APR_INT64_C(0);
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

		/*
		 * o_rdb_r.uri 以下の全てをロック
		 */
		stmt = dbconn->prepareStatement(dbconn, 
				"SELECT "
				"rs_resourcetype_i,"
				"rs_physical_path_txt "
#ifdef DIVY_SUPPORT_EXTENDQUOTA
				",rs_get_cont_len_bi "
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
				"FROM dav_resource "
				"WHERE rs_uri_txt = ? "
				"OR rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE" "
				"FOR UPDATE", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (uri = %s) "
				"Reason: %s", o_rdb_r.uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		/* バインド */
		stmt->setString(stmt, 1, o_rdb_r.uri);
		/* uri に含まれるワイルドカード文字をエスケープする */
		stmt->setString(stmt, 2, apr_pstrcat(p,
				stmt->escWildCard(stmt, o_rdb_r.uri), "/%", NULL));

		/* SQLの実行 */
		rset = stmt->executeQueryForUpdate(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to execute select dav_resource for quota. "
				"(uri = %s) Reason: %s",
				o_rdb_r.uri, rset->getMsg(rset));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* 削除容量を取得する */
		while (rset->next(rset) == DB_TRUE) {

			resourcetype = rset->getInt(rset, 1);
			if (resourcetype == DIVY_TYPE_RESOURCE) {
				if (*del_filelist == NULL) {
					clist = *del_filelist = apr_pcalloc(p, sizeof(divy_linkedlist_t));
				}
				else {
					clist->next = apr_pcalloc(p, sizeof(divy_linkedlist_t));
					clist = clist->next;
				}
				/* 相対パスの記録 */
				clist->val = rset->getString(rset, 2);
#ifdef DIVY_SUPPORT_EXTENDQUOTA
				/* 削除リソースのSUMをとる(システムQuotaありの場合) */
				squota.usedst += rset->getBigInt(rset, 3);
				squota.usedres++;
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
			}
			found_resource = 1;	/* リソースが見つかった */
		}
		if (rset != NULL) rset->close(rset);	rset = NULL;
		if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

		/* 恐らく処理中に削除されてしまった(レアケース) */
		if (!found_resource) {
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The resource is missing. "
				"Maybe it was deleted before this operation. "
				"so we give up delete this entry."
				"(uri = %s)", o_rdb_r.uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return DIVY_STCODE_NOT_EXIST;
		}
#ifdef DIVY_SUPPORT_EXTENDQUOTA
		/* 減算する値にシステムQuotaを変形 */
		squota.usedst  = APR_INT64_C(-1) * squota.usedst;
		squota.usedres = APR_INT64_C(-1) * squota.usedres;
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

		/*
		 * o_rdb_r.uri 以下のユーザ毎の消費Diskとファイル数を求める
		 */
		stmt = dbconn->prepareStatement(dbconn, 
				"SELECT "
				"rs_lastmodifier_usr_id_vc,"
				"SUM(rs_get_cont_len_bi) AS len, "
				"COUNT(rs_rs_id_c) AS cnt "
				"FROM dav_resource "
				"WHERE rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND rs_resourcetype_i=0 "
				"GROUP BY rs_lastmodifier_usr_id_vc", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (uri = %s) "
				"Reason: %s", o_rdb_r.uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
		/* uri に含まれるワイルドカード文字をエスケープする */
		stmt->setString(stmt, 1, apr_pstrcat(p,
				stmt->escWildCard(stmt, rdb_r->uri), "/%", NULL));

		rset = stmt->executeQuery(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to execute select for quota. (uri = %s) "
				"Reason: %s", o_rdb_r.uri, rset->getMsg(rset));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* 削除容量を取得する */
		while (rset->next(rset) == DB_TRUE) {

			/* 値の取得 */
			userid    = rset->getString(rset, 1);
			u_usedst  = rset->getBigInt(rset, 2); 
			u_usedres = rset->getBigInt(rset, 3);

			/* 0バイト, 0個だった場合 削除処理はやらなくていい */
			if (u_usedst == APR_INT64_C(0) &&
			    u_usedres == APR_INT64_C(0)) {
				continue;	/* 次へ */
			}

			if (p_uquota == NULL) {
				p_uquota = &uquota;
			}
			else {
				p_uquota->next =
					apr_pcalloc(p, sizeof(divy_rdbo_usr));
				p_uquota = p_uquota->next;
				p_uquota->next = NULL;
			}
			p_uquota->usrid   = userid;
			p_uquota->usedst  = APR_INT64_C(-1) * u_usedst;
			p_uquota->usedres = APR_INT64_C(-1) * u_usedres;
		}

		if (rset != NULL) rset->close(rset);	rset = NULL;
		if (stmt != NULL) stmt->close(stmt);	stmt = NULL;
	}

	/* 
	 * 共有コレクションへのリンクを削除する。
	 * (linkdbsearch結果フォルダ or 共有コレクションのみ)
	 * (dav_resource を消すよりも前にやること)
	 */
	if (u_spec->infotype == DIVY_INFOTYPE_dbfolder_e ||
	    u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e) {
		/* 削除 */
		if (_remove_hierarchy_sharedcollection_property(r, rdb_r, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete sharedcollection property."
				"(uri = %s)", rdb_r->uri);

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1; 
		}
	}

	/*
	 * Dead プロパティを削除する。
	 * (dav_resource を消すよりも前にやること)
	 */
	if (_remove_hierarchy_dead_property(r, rdb_r, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to delete Dead property.(uri = %s)", rdb_r->uri);

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * mailwatch プロパティを論理削除する。(共有コレクションのトップのみ)
	 */
	if (u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e) {

		/* メール監視フラグを取得する */
		if (_get_merge_mailwatch_property(r, rdb_r->uri,
					divy_get_userid(r), &mw, ts_ctx)) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get mailwatch property.(uri = %s, "
				"userid = %s)", rdb_r->uri, divy_get_userid(r));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return 1;
		}

		if (mw && mw->trigger_methods[M_DELETE]) {
			/* 論理削除 */
			ret = _remove_mailwatch_property(r, rdb_r->uri, NULL,
						DIVY_REMWATCH_LOGICAL | 
						DIVY_REMWATCH_IGNORE_USER, ts_ctx);
		}
		else if (mw && !mw->trigger_methods[M_DELETE]) {
			/* 物理削除 */
			ret = _remove_mailwatch_property(r, rdb_r->uri, NULL,
						DIVY_REMWATCH_IGNORE_USER, ts_ctx);
		}
		else {
			ret = 0;
		}

		if (ret) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to delete mailwatch property."
					"(uri = %s)", rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * クライアントモジュールのプロパティ情報を削除する
	 */
	if (u_spec->infotype == DIVY_INFOTYPE_m_update_e) {
		if (_remove_clmodule_property(r, rdb_r->uri, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete clmodule property.(uri = %s)",
				rdb_r->uri);

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * ごみ箱プロパティの削除
	 */
	if (support_trash &&
	    (u_spec->infotype == DIVY_INFOTYPE_user_trash_e0 ||
	     u_spec->infotype == DIVY_INFOTYPE_group_trash_e0)) {
		/* ごみ箱直下のリソース / コレクションの削除 */

		if (_remove_trash_property(r, rdb_r->rsid, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete trashinformation property."
				"(uri = %s)", rdb_r->uri);

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
	else if (support_trash && u_spec->infotype == DIVY_INFOTYPE_user_e) {
		/*  ユーザコレクションの以下のごみ箱プロパティを削除 */
		trash_uri = divy_build_user_trash_uri(p,
				dav_divy_get_root_uri(r),
				dav_divy_extract_finalpath_segment(p, rdb_r->uri));	

		if (_remove_hierarchy_trash_property(r, trash_uri, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete user's trashinformation property."
				"(uri = %s)", rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
	else if (support_trash && u_spec->infotype == DIVY_INFOTYPE_group_e) {
		/*  グループコレクションの以下のごみ箱プロパティを削除 */
		trash_uri = divy_build_group_trash_uri(p,
				dav_divy_get_root_uri(r),
				dav_divy_extract_finalpath_segment(p, rdb_r->uri));

		if (_remove_hierarchy_trash_property(r, trash_uri, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete group's trashinformation property."
				"(uri = %s)", rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * 状態/属性プロパティの削除
	 */
	if (support_extstatus &&
		(u_spec->infotype == DIVY_INFOTYPE_group_e_regular ||
		 u_spec->infotype == DIVY_INFOTYPE_group_e)) {

		if (_remove_resourcestate_property(r, DAV_INFINITY, rdb_r->uri, 0, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete resource state property.(uri = %s)",
				rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * BOXフォルダの削除
	 * (note) リソースが消えたらBOX設定も消えます（外部仕様）
	 */
	if (divy_support_tfbox(r) &&
		(u_spec->infotype == DIVY_INFOTYPE_group_e_regular ||
		 u_spec->infotype == DIVY_INFOTYPE_group_e)) {

		if (_remove_box_property(r, rdb_r->uri, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete box property.(uri = %s)",
				rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * 開封通知の削除
	 * (note) リソースが消えたら開封通知も消えます(外部仕様)
	 */
	if (divy_support_confirmreading(r) &&
		(u_spec->infotype == DIVY_INFOTYPE_group_e ||
		 u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
		 u_spec->infotype == DIVY_INFOTYPE_group_trash_ex ||
		 u_spec->infotype == DIVY_INFOTYPE_group_e_regular)) {

		int depth = 0;
		if (rdb_r->resourcetype != DIVY_TYPE_RESOURCE) {
			depth = DAV_INFINITY;
		}

		if (divy_rdbo_remove_confirmreading(r, rdb_r->uri,
											NULL/*ユーザID無視*/, depth, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove confirmreading property. (uri = %s)", rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/* 
	 * dav_resource テーブルのレコードを削除する。
	 */
	if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
		stmt = dbconn->prepareStatement(dbconn, 
			"DELETE FROM dav_resource "
			"WHERE rs_rs_id_c = ?", p);
	}
	else {
		stmt = dbconn->prepareStatement(dbconn, 
			"DELETE FROM dav_resource "
			"WHERE rs_rs_id_c = ? "
			"OR rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE, p);
	}

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s) Reason: %s",
			rdb_r->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	stmt->setString(stmt, 1, rdb_r->rsid);
	if (rdb_r->resourcetype != DIVY_TYPE_RESOURCE) {
		/* uri に含まれるワイルドカード文字をエスケープする */
		stmt->setString(stmt, 2, apr_pstrcat(p,
				stmt->escWildCard(stmt, rdb_r->uri), "/%", NULL));
	}

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to execute DELETE sql. (uri = %s) Reason: %s",
			rdb_r->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* 使用した資源の片付け */
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DIVY_SUPPORT_EXTENDQUOTA
	/* システムQuotaの更新 */
	ret = _change_used_sysquota(r, &squota, ts_ctx);
	if (ret) {
		if (ret != DIVY_STCODE_QUOTAOVER) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to update system quota.(root_uri = %s, "
				"userid = %s)", REPLACE_NULL(squota.uri),
				o_rdb_r.lastmodifier_userid);
		}
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return ret;
	}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

	/*
	 * ユーザQuotaの更新
	 * "更新者"にQuota を返す(ディスク使用量と使用ファイル数を減らす)
	 */
	/* 削除対象がリソースの場合 */
	if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {

		ret = _change_used_userquota(r, &uquota, ts_ctx);
		if (ret) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to reduce disk quota. (owner = %s)",
				uquota.usrid);

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return ret;
		}
	}
	/* 削除対象がコレクションで1つ以上の内包リソースが存在していた場合 */
	else if (rdb_r->resourcetype == DIVY_TYPE_COLLECTION &&
		 IS_FILLED(uquota.usrid)) {

		/* 全ユーザのQuota情報を更新する(Quotaを返してもらう) */
		ret = _change_used_userquota(r, &uquota, ts_ctx);
		if (ret) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to reduce disk quota.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return ret;
		}
	}

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * depth = infinity として指定されたsrc_rdb_r をdist_rdb_r としてコピーする。
 * ディスティネーションの作成者は、COPY操作者になります。
 *
 */
DIVY_DECLARE(int) divy_rdbo_copy_resource(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dist_rdb_r,
					divy_copyfile_info **copyinfo,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	int iscommit            = 0;
	int insert_cnt          = 0;
	char *userid            = apr_pstrdup(p, divy_get_userid(r));
	time_t now_time         = dav_divy_get_now_epoch();
	int ret;

	/* src_rdb_r, dist_rdb_r は後で潰れてしまうので、uri を保存しておきます */
	char *src_root_uri      = apr_pstrdup(p, src_rdb_r->uri);
	char *dist_root_uri     = apr_pstrdup(p, dist_rdb_r->uri);
	divy_rdbo_resourcetype resourcetype;
	apr_int64_t cp_usedst = APR_INT64_C(0), cp_usedres = APR_INT64_C(0);
	int found_resource = 0;
	const char *etag;
	divy_copyfile_info *cpinfo = NULL;
	divy_fstorage_t *fstorage  = NULL;

	TRACE(p);

	/* 入力値のチェック */
	if (IS_EMPTY(userid) || src_rdb_r == NULL || dist_rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"userid or src_rdb_r or dist_rdb_r is NULL.");
		return 1;
	}

	*copyinfo = NULL;	/* 初期化 */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * リソースのロックと情報の取得
	 * src_rdb_r->uri 以下全てをロックする
	 */
	stmt = dbconn->prepareStatement(dbconn, 
			"SELECT "
			"rs_resourcetype_i,"
			"rs_get_cont_len_bi "
			"FROM dav_resource "
			"WHERE rs_uri_txt = ? "
			"OR rs_uri_txt LIKE ? "
			DIVY_DBFUNC_ESCAPE_CLAUSE" "
			"FOR UPDATE", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (src_uri = %s) "
			"Reason: %s", src_rdb_r->uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, src_rdb_r->uri);
	/* uri に含まれるワイルドカード文字をエスケープする */
	stmt->setString(stmt, 2, apr_pstrcat(p,
				stmt->escWildCard(stmt, src_rdb_r->uri), "/%", NULL));

	/* SQLの実行 */
	rset = stmt->executeQueryForUpdate(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to execute select dav_resource for quota. "
			"(src_uri = %s) Reason: %s",
			src_rdb_r->uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* コピー容量を取得する */
	found_resource = 0;	/* コピーリソースは見つかっていない */
	while (rset->next(rset) == DB_TRUE) {

		resourcetype = rset->getInt(rset, 1);
		if (resourcetype == DIVY_TYPE_RESOURCE) {
			/* コピーリソースのSUMをとる(システムQuotaありの場合) */
			cp_usedst += rset->getBigInt(rset, 2);
			cp_usedres++;
		}
		found_resource = 1;	/* 見つかった(コレクションでもいい) */
	}

	if (rset != NULL) rset->close(rset);	rset = NULL;
	if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

	/* コピー対象のリソースが1つも見つからなかった */
	if (!found_resource) {
		/* 恐らく同時アクセスにより削除 or 移動されてしまったのだろう */
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get dav_resource entry. "
			"Maybe this resource was deleted or moved "
			"before this operation.(src_uri = %s, dist_uri = %s) ",
			src_rdb_r->uri, dist_rdb_r->uri);

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return DIVY_STCODE_NOT_EXIST;
	}

	/*
	 * リソース、プロパティのコピー (dav_resource)
	 * (note)
	 * 	rs_physical_path_txt をコピーしますが、後で入れ替えます。
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"INSERT INTO dav_resource "
			"(rs_rs_id_c,"
			"rs_uri_txt,"
			"rs_dispname_vc,"
			"rs_create_bi,"
			"rs_get_cont_lang_vc,"
			"rs_get_cont_len_bi,"
			"rs_get_cont_type_vc,"
			"rs_get_etag_txt,"
			"rs_get_lastmodified_bi,"
			"rs_resourcetype_i,"
			"rs_depth_i,"
			"rs_isversioned_i, rs_checkin_i, rs_checkout_i,"
			"rs_physical_path_txt, "
			"rs_creator_usr_id_vc, "
			"rs_lastmodifier_usr_id_vc) "
			"(SELECT "
			DIVY_DBFUNC_CREATE_RES_SEQ","
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
			DIVY_DBFUNC_SUBSTRING("rs_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")","
			"rs_dispname_vc,"
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_BIGINT)","
			"rs_get_cont_lang_vc,"
			"rs_get_cont_len_bi,"
			"rs_get_cont_type_vc,"
			DIVY_DUMMY_ETAG","
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_BIGINT)","
			"rs_resourcetype_i,"
			"rs_depth_i + "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_INTEGER)","
			"0, -1, -1,"
			"rs_physical_path_txt, "
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_USRID_LEN))","
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_USRID_LEN))" "
			"FROM dav_resource "
			"WHERE rs_uri_txt = ? OR rs_uri_txt LIKE ? "
			DIVY_DBFUNC_ESCAPE_CLAUSE")" , p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt (copy dav_resource)."
			"(src = %s, dst = %s) Reason: %s",
			src_root_uri, dist_root_uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 物理ファイルのコピーに必要な情報を入れておく */
	dist_rdb_r->creationdate     = now_time;
	dist_rdb_r->getlastmodified  = now_time;

	/* 物理パスやEtag は後でUPDATEします */
	stmt->setString(stmt, 1, dist_root_uri);
	stmt->setString(stmt, 2, src_root_uri);
	stmt->setBigInt(stmt, 3, dist_rdb_r->creationdate);
	stmt->setBigInt(stmt, 4, dist_rdb_r->getlastmodified);
	stmt->setInt(stmt,    5, dist_rdb_r->depth - src_rdb_r->depth);
	stmt->setString(stmt, 6, dist_rdb_r->creator_userid);
	stmt->setString(stmt, 7, dist_rdb_r->lastmodifier_userid);
	stmt->setString(stmt, 8, src_root_uri);
	/* uri に含まれるワイルドカード文字をエスケープする */
	stmt->setString(stmt, 9, apr_pstrcat(p,
			stmt->escWildCard(stmt, src_root_uri), "/%", NULL));

	/* SQL の実行 */
	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to executeUpdate for dav_resource insert. "
			"(src_uri = %s, dist_uri = %s) Reason: %s", 
			src_root_uri, dist_root_uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}
	if (insert_cnt == 0) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"Status is OK, but the count of inserting is 0."
			"Please check dav_resource.(src_uri = %s, "
			"dist_uri = %s)", src_root_uri, dist_root_uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * Dead property をコピーする
	 * (note) dav_resource のコピーが終わった後にCallして下さい。
	 */
	if (_copy_dav_dead_property(r, src_rdb_r, dist_rdb_r, DAV_INFINITY, ts_ctx)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy dead property. (src = %s, dist = %s, "
			"depth = infinity", src_root_uri, dist_root_uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}


	/* dist_rdb_r の rs_dispname_vc を更新する */
	if (strcmp(src_rdb_r->displayname, dist_rdb_r->displayname) != 0) {

		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE dav_resource "
				"SET rs_dispname_vc = ? "
				"WHERE rs_uri_txt = ?", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt "
				"(update dav_resource). (src = %s) Reason: %s",
				src_root_uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		stmt->setString(stmt, 1, dist_rdb_r->displayname);
		stmt->setString(stmt, 2, dist_root_uri);

		/* SQL文の実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update displayname for dav_resource."
				"(dist = %s, displayname = %s) Reason: %s",
				dist_root_uri, dist_rdb_r->displayname,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		/* 使い終わった領域の解放 */
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/*
	 * メンバリソースに対し、正しくない値で更新してしまっていた項目
	 * (rs_dispname_vc, rs_get_etag_txt, rs_physical_path_txt) を
	 * 正しい値で更新する。
	 */

	/*
	 * Destinationパス以下のリソースを取得
	 */ 
	stmt = dbconn->prepareStatement(dbconn, 
			"SELECT"
			" rs_rs_id_c,"
			" rs_get_cont_len_bi,"
			" rs_resourcetype_i,"
			" rs_physical_path_txt "
#ifdef DIVY_SUPPORT_PREFLIGHT_PLUGIN
			",rs_dispname_vc "
#endif	/* DIVY_SUPPORT_PREFLIGHT_PLUGIN */
			"FROM dav_resource "
			"WHERE rs_uri_txt = ? "
			"OR rs_uri_txt LIKE ? "
			DIVY_DBFUNC_ESCAPE_CLAUSE, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (dist_uri = %s) "
			"Reason: %s", dist_root_uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	stmt->setString(stmt, 1, dist_root_uri);
	/* uri に含まれるワイルドカード文字をエスケープする */
	stmt->setString(stmt, 2, apr_pstrcat(p,
			stmt->escWildCard(stmt, dist_root_uri), "/%", NULL));

	/* SQL文の実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet. (uri = %s) Reason: %s",
			dist_root_uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 結果の取得 */
	while (rset->next(rset) == DB_TRUE) {
		if (*copyinfo == NULL) {
			*copyinfo = cpinfo =
				apr_pcalloc(p, sizeof(divy_copyfile_info));
		}
		else {
			cpinfo->next = apr_pcalloc(p, sizeof(divy_copyfile_info));
			cpinfo = cpinfo->next;
		}

		/* 値の取り出し */
		cpinfo->rsid               = rset->getString(rset,  1);
		cpinfo->getcontentlength   = rset->getBigInt(rset,  2);
		resourcetype  = rset->getInt(rset,3);
		if (resourcetype == DIVY_TYPE_RESOURCE) {
			cpinfo->isfile = 1;
		}
		cpinfo->src_relativepath   = rset->getString(rset,  4);
		cpinfo->dst_relativepath   = NULL;
#ifdef DIVY_SUPPORT_PREFLIGHT_PLUGIN
		cpinfo->dispname           = rset->getString(rset, 5);
#endif	/* DIVY_SUPPORT_PREFLIGHT_PLUGIN */
		cpinfo->next               = NULL;
	}
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* ストレージを開く */
	ret = divy_fstorage_open(r, p, &fstorage);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to open storage.");
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/* 物理パスディレクトリ(dst_relativepath)の取得 */
	ret = divy_pfile_createByCopyinfo(fstorage, p, *copyinfo);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get relativepath.");
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}
	/* ストレージを閉じる */
	(void) divy_fstorage_close(fstorage);

	/* Etag, 物理パスを更新するSQLの準備 */
	stmt = dbconn->prepareStatement(dbconn,
			"UPDATE dav_resource "
			"SET "
			"rs_get_etag_txt = ?,"
			"rs_physical_path_txt = ? "
			"WHERE rs_rs_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt (update dav_resource). "
			"Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 全てのDeistination のリソースに対して実施 */
	for (cpinfo = *copyinfo; cpinfo; cpinfo = cpinfo->next) {
		/* Etag の取得 */
		etag = dav_divy_get_etag_string(p, cpinfo->rsid,
						cpinfo->getcontentlength,
						now_time);
		stmt->setString(stmt, 1, etag);
		stmt->setString(stmt, 2, cpinfo->dst_relativepath);
		stmt->setString(stmt, 3, cpinfo->rsid);

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update dav_resource. "
				"(rsid = %s ) Reason: %s", 
				cpinfo->rsid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * システムQuotaとユーザQuotaの変更
	 *
	 * [ 処理開始条件 ]
	 * 	* 1つ以上の物理ファイルが存在したこと
	 * [ ルール ]
	 * 	* システムQuotaを消費する
	 * 	* 新更新者のユーザQuotaを消費する
	 */
#ifdef DIVY_SUPPORT_EXTENDQUOTA
	/* システムQuotaの更新 */
	if (cp_usedst > 0 || cp_usedres > 0) {
		divy_rdbo_diskquota squota = { 0 };
		squota.usedst  = cp_usedst;
		squota.usedres = cp_usedres;

		ret = _change_used_sysquota(r, &squota, ts_ctx);
		if (ret) {
			if (ret != DIVY_STCODE_QUOTAOVER) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update system quota."
					"(root_uri = %s, userid = %s)",
					REPLACE_NULL(squota.uri), userid);
			}
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return ret;
		}
	}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

	/* ユーザQuotaの更新 */
	if (cp_usedst > 0 || cp_usedres > 0) {
		divy_rdbo_usr uquota = { 0 };

		uquota.usrid   = userid;
		uquota.usedst  = cp_usedst;
		uquota.usedres = cp_usedres;
		uquota.next    = NULL;

		ret = _change_used_userquota(r, &uquota, ts_ctx);
		if (ret) {
			if (ret != DIVY_STCODE_QUOTAOVER) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update user quota."
					"(owner = %s)", uquota.usrid);
			}

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return ret;
		}
	}

	/*
	 * 残留しているかもしれない開封通知を削除する
	 * (note)
	 *   運用中に開封通知フラグをon -> off -> on すると、開封通知が残留する可能性あり.
	 *   そのため、残留しているかもしれない開封通知を削除しておきます.
	 *   src ではなく、dst の方です.
	 */
	if (divy_support_confirmreading(r)) {
		if (dist_rdb_r->u_spec == NULL) {
			(void) divy_parse_uri(p, dav_divy_get_root_uri(r), dist_rdb_r->uri, &dist_rdb_r->u_spec);
		}

		if (dist_rdb_r->u_spec != NULL && dist_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {

			if (divy_rdbo_remove_confirmreading(r, dist_rdb_r->uri,
											NULL/*ユーザID無視*/, DAV_INFINITY, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to remove confirmreading property. (dst_uri = %s)", dist_rdb_r->uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * depth = 0 でソースとして指定されたsrc_rdb_r を dist_rdb_r として
 * コピーする。
 * 内部に含まれるリソースやコレクションはコピーされません。
 */
DIVY_DECLARE(int) divy_rdbo_copy_resource_without_member(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dist_rdb_r,
					divy_copyfile_info **copyinfo,
					divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t      *p      = r->pool;
	int iscommit            = 0;
	const char *userid      = divy_get_userid(r);
	time_t now_time         = dav_divy_get_now_epoch();
	char *src_root_uri      = src_rdb_r->uri;
	char *dist_root_uri     = dist_rdb_r->uri;
	int ret;
	divy_rdbo_resource o_rdb_r = { 0 };
	divy_fstorage_t *fstorage = NULL;

	TRACE(p);

	/* 入力値のチェック */
	if (IS_EMPTY(userid) || src_rdb_r == NULL || dist_rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"userid or src_rdb_r or dist_rdb_r is NULL.");
		return 1;
	}

	*copyinfo = NULL;	/* 初期化 */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;

	/*
	 * 自身のリソースのロックと情報の取得
	 */
	o_rdb_r.uri = src_rdb_r->uri;

	ret = _lock_resource_entry(r, &o_rdb_r, ts_ctx);
	if (ret == DIVY_STCODE_NOT_EXIST) {
		/* 恐らく処理中に自分が削除されてしまった(レアケース) */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The resource is missing. "
			"Maybe it was deleted before this operation. "
			"so we give up copy this entry."
			"(uri = %s)", src_rdb_r->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return DIVY_STCODE_NOT_EXIST;
	}
	else if (ret) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"Failed to get resource information."
			"(uri = %s)", src_rdb_r->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/* リソースID の取得 */
	if (divy_rdbo_create_rsid(r, &dist_rdb_r->rsid, ts_ctx)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * リソースの追加 (新規追加)
	 */
	dist_rdb_r->creationdate       = now_time;
	dist_rdb_r->getcontentlanguage = o_rdb_r.getcontentlanguage;
	dist_rdb_r->getcontentlength   = o_rdb_r.getcontentlength;
	dist_rdb_r->getcontenttype     = o_rdb_r.getcontenttype;
	dist_rdb_r->getetag            = (char *) dav_divy_get_etag_string(p,
							dist_rdb_r->rsid,
							o_rdb_r.getcontentlength,
							now_time);
	dist_rdb_r->getlastmodified    = now_time;
	dist_rdb_r->resourcetype       = o_rdb_r.resourcetype;

	/* 物理パス情報の作成 */
	*copyinfo = apr_pcalloc(p, sizeof(divy_copyfile_info));
	(*copyinfo)->rsid               = dist_rdb_r->rsid;
	(*copyinfo)->getcontentlength   = dist_rdb_r->getcontentlength;
	if (dist_rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
		(*copyinfo)->isfile = 1;
	}
	(*copyinfo)->src_relativepath   = o_rdb_r.physicalpath;
#ifdef DIVY_SUPPORT_PREFLIGHT_PLUGIN
	(*copyinfo)->dispname           = dist_rdb_r->displayname;
#endif	/* DIVY_SUPPORT_PREFLIGHT_PLUGIN */
	(*copyinfo)->next               = NULL;

	if (dist_rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {

		/* ストレージのopen */
		ret = divy_fstorage_open(r, p, &fstorage);
		if (ret != DIVY_FS_ST_OK) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to open storage.(uri = %s)",
				src_rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/* 物理パスの取得 */
		ret = divy_pfile_createByCopyinfo(fstorage, p, *copyinfo);
		if (ret != DIVY_FS_ST_OK) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to make physical file path.(uri = %s,"
				"rsid = %s)", dist_rdb_r->uri, dist_rdb_r->rsid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
		dist_rdb_r->physicalpath = (char *) (*copyinfo)->dst_relativepath;

		/* ストレージを閉じる */
		(void) divy_fstorage_close(fstorage);
	}
	/* コレクションの場合 */
	else {
		dist_rdb_r->physicalpath = NULL;
		(*copyinfo)->dst_relativepath = NULL;	/* 不要 */
	}

	/* 新規INSERT */
	if (_insert_property(r, dist_rdb_r, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert dav_resource. (uri = %s)",
			dist_rdb_r->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * Dead property をコピーする
	 * (note) dav_resource のコピーが終わった後にCallして下さい。
	 */
	if (_copy_dav_dead_property(r, src_rdb_r, dist_rdb_r, 0, ts_ctx)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy dead property. (src = %s, dist = %s, "
			"depth = 0)", src_root_uri, dist_root_uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * システムQuotaとユーザQuotaの変更
	 *
	 * [ 処理開始条件 ]
	 * 	* リソースであること
	 * [ ルール ]
	 * 	* システムQuotaを消費する
	 * 	* 新更新者のユーザQuotaを消費する
	 */
#ifdef DIVY_SUPPORT_EXTENDQUOTA
	/* システムQuotaの更新 */
	if (o_rdb_r.resourcetype == DIVY_TYPE_RESOURCE) {
		divy_rdbo_diskquota squota = { 0 };
		squota.usedst  = o_rdb_r.getcontentlength;
		squota.usedres = APR_INT64_C(1);

		ret = _change_used_sysquota(r, &squota, ts_ctx);
		if (ret) {
			if (ret != DIVY_STCODE_QUOTAOVER) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update system quota."
					"(root_uri = %s, userid = %s)",
					REPLACE_NULL(squota.uri), userid);
			}
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return ret;
		}
	}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

	/* ユーザQuotaの更新 */
	if (o_rdb_r.resourcetype == DIVY_TYPE_RESOURCE) {
		divy_rdbo_usr uquota = { 0 };

		uquota.usrid   = (char *) userid;
		uquota.usedst  = o_rdb_r.getcontentlength;
		uquota.usedres = APR_INT64_C(1);
		uquota.next    = NULL;

		ret = _change_used_userquota(r, &uquota, ts_ctx);
		if (ret) {
			if (ret != DIVY_STCODE_QUOTAOVER) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update user quota."
					"(owner = %s)", uquota.usrid);
			}

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return ret;
		}
	}

	/*
	 * 残留しているかもしれない開封通知を削除する
	 * (note)
	 *   運用中に開封通知フラグをon -> off -> on すると、開封通知が残留する可能性あり.
	 *   そのため、残留しているかもしれない開封通知を削除しておきます
	 */
	if (divy_support_confirmreading(r) && dist_rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
		if (dist_rdb_r->u_spec == NULL) {
			(void) divy_parse_uri(p, dav_divy_get_root_uri(r), dist_rdb_r->uri, &dist_rdb_r->u_spec);
		}

		if (dist_rdb_r->u_spec != NULL && dist_rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {

			if (divy_rdbo_remove_confirmreading(r, dist_rdb_r->uri,
											NULL/*ユーザID無視*/, 0, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to remove confirmreading property. (dst_uri = %s)", dist_rdb_r->uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * ソースとして指定されたdivy_rdbo_resource のリソースまたはコレクションを
 * ディスティネーションのdivy_rdbo_resource が示すURI下に移動する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_move_resource(request_rec *r, 
                            const divy_rdbo_resource *src_rdb_r,
                            const divy_rdbo_resource *dist_rdb_r,
			    divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t      *p      = r->pool;
	int iscommit            = 0;
	int update_cnt          = 0;
	int is_rename           = 0;
	int support_trash       = divy_support_trashfolder(r);
	int support_extstatus   = divy_support_extenduserstatus(r);
	int support_confirmreading = divy_support_confirmreading(r);
	divy_uri_spec *src_u_spec  = src_rdb_r->u_spec;
	divy_uri_spec *dst_u_spec  = dist_rdb_r->u_spec;
	divy_rdbo_mailwatch *mw = NULL;
	char *src_parent_uri, *dst_parent_uri;
	int depth;

	TRACE(p);

	if (IS_EMPTY(src_rdb_r->uri) || IS_EMPTY(src_rdb_r->rsid) ||
	    IS_EMPTY(src_rdb_r->displayname) || IS_EMPTY(dist_rdb_r->uri) ||
	    IS_EMPTY(dist_rdb_r->displayname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There are empty value.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* uri のパース */
	if (src_u_spec == NULL) {
		(void) divy_parse_uri(p, dav_divy_get_root_uri(r),
				      src_rdb_r->uri, &src_u_spec);
	}
	if (dst_u_spec == NULL) {
		(void) divy_parse_uri(p, dav_divy_get_root_uri(r),
				      dist_rdb_r->uri, &dst_u_spec);
	}

	/* 階層移動なのかリネームなのかの判定 */
	src_parent_uri = divy_get_parenturi(p, src_rdb_r->uri);
	dst_parent_uri = divy_get_parenturi(p, dist_rdb_r->uri);
	if (strcmp(src_parent_uri, dst_parent_uri) == 0) {
		is_rename = 1;	/* 親が同じならリネーム */
	}

	/**
	 * ごみ箱プロパティの処理
	 */
	/* 通常リソース   --> ごみ箱(1 階層),
	 * ごみ箱(n 階層) --> ごみ箱(1 階層) */
	if (support_trash &&
	    (src_u_spec->uritype == DIVY_URITYPE_REGULAR ||
	     (src_u_spec->infotype == DIVY_INFOTYPE_user_trash_ex ||
	      src_u_spec->infotype == DIVY_INFOTYPE_group_trash_ex)) &&
	    (dst_u_spec->infotype == DIVY_INFOTYPE_user_trash_e0 ||
	     dst_u_spec->infotype == DIVY_INFOTYPE_group_trash_e0)) {

		/* ごみ箱プロパティを付加する */
		if (_insert_trash_property(r, src_rdb_r->rsid,
			dav_divy_get_now_epoch(), divy_get_userid(r), ts_ctx)) {

			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to insert trashinformation property."
				"(src_uri = %s, dist_uri = %s)",
				src_rdb_r->uri, dist_rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
	/* ごみ箱(1 階層) --> 通常リソース,
	 * ごみ箱(1 階層) --> ごみ箱(n 階層) */
	else if (support_trash &&
		 (src_u_spec->infotype == DIVY_INFOTYPE_user_trash_e0 ||
		  src_u_spec->infotype == DIVY_INFOTYPE_group_trash_e0) &&
		 (dst_u_spec->uritype == DIVY_URITYPE_REGULAR ||
		  (dst_u_spec->infotype == DIVY_INFOTYPE_user_trash_ex ||
		   dst_u_spec->infotype == DIVY_INFOTYPE_group_trash_ex))) {

		/* ごみ箱プロパティを除去する */
		if (_remove_trash_property(r, src_rdb_r->rsid, ts_ctx)) {

			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to remove trashinformation property."
				"(src_uri = %s, dist_uri = %s)",
				src_rdb_r->uri, dist_rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
	/* ごみ箱を使えない,
	 * 通常リソース   --> ごみ箱(n 階層),
	 * ごみ箱(n 階層) --> 通常リソース   */
	else {
		/* ごみ箱プロパティの付加は必要なし。何もしない */
	}

	/*
	 * 状態/属性プロパティの処理
	 *
	 * [ 処理開始条件 ]
	 *   * ユーザ拡張プロパティ機能が有効である
	 *   * グループコレクションよりも下の通常リソース/通常コレクションである
	 *
	 * destination の親が持っている状態/属性プロパティを継承する.
	 * 実作業としてはsrc の状態/属性プロパティを何とかすればdestination には
	 * 何もする必要なし(親から自動的に継承されるので)
	 */
	if (support_extstatus) {
		/*
		 * src_rdb_r->uri 以下の状態/属性プロパティURIを変更する
		 * [ 処理開始条件 ]
		 *   * リネーム
		 *   * src_rdb_r はグループより下の通常リソース or 通常コレクション
		 */
		if (is_rename && src_u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {
		 	/*
			 * src_rdb_r->uri 以下の状態/属性プロパティのURIを変更する
			 */
			if (divy_rdbo_move_resourcestate_property(r, src_rdb_r->uri, dist_rdb_r->uri, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update resource state property.(uri = %s)",
					src_rdb_r->uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}
		else {
			/*
			 * src_rdb_r->uri 以下の状態/属性プロパティを全てクリアする
			 */
			if (_remove_resourcestate_property(r, DAV_INFINITY, src_rdb_r->uri, 0, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to delete resource state property.(uri = %s)",
					src_rdb_r->uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}
	}

	/*
	 * 開封通知プロパティの削除
	 * (note) リネーム、階層移動何れかでもsrc の開封通知は削除する
	 */
	if (support_confirmreading && src_u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {

		depth = 0;
		if (src_rdb_r->resourcetype != DIVY_TYPE_RESOURCE) {
			depth = DAV_INFINITY;
		}

		if (divy_rdbo_remove_confirmreading(r, src_rdb_r->uri,
											NULL/*ユーザID無視*/, depth, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove confirmreading property. (src_uri = %s)", src_rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * 残留しているかもしれない開封通知を削除する
	 * (note)
	 *   運用中に開封通知フラグをon -> off -> on すると、開封通知が残留する可能性あり.
	 *   そのため、残留しているかもしれない開封通知を削除しておきます
	 */
	if (support_confirmreading && dst_u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {
		depth = 0;
		if (dist_rdb_r->resourcetype != DIVY_TYPE_RESOURCE) {
			depth = DAV_INFINITY;
		}

		if (divy_rdbo_remove_confirmreading(r, dist_rdb_r->uri,
											NULL/*ユーザID無視*/, depth, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove confirmreading property. (dst_uri = %s)", dist_rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * リソースの場合
	 * rs_uri_txt, rs_dispname_vc, rs_depth_i を更新する
	 */
	if (src_rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
		/*
		 * src_rdb_r のURI, depth, displaynameを変更して、
		 * dist_rdb_r になるようにする
		 */
		stmt = dbconn->prepareStatement(dbconn,
					"UPDATE dav_resource "
					"SET rs_uri_txt = ?,"
					"rs_dispname_vc = ?,"
					"rs_depth_i = ? "
					"WHERE rs_uri_txt = ?", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (src_uri = %s) "
				"Reason: %s", src_rdb_r->uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		stmt->setString(stmt, 1, dist_rdb_r->uri);
		stmt->setString(stmt, 2, dist_rdb_r->displayname);
		stmt->setInt(stmt,    3, dist_rdb_r->depth);
		stmt->setString(stmt, 4, src_rdb_r->uri);

		update_cnt = stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update dav_resource table. "
				"(src_uri = %s, dist_uri = %s) Reason: %s", 
				src_rdb_r->uri, dist_rdb_r->uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		/* リソースが削除されたもしくは移動された場合 */
		if (update_cnt == 0) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to update dav_resource table. "
				"Maybe this resource was deleted or moved "
				"before this operation."
				"(src_uri = %s, dist_uri = %s) ", 
				src_rdb_r->uri, dist_rdb_r->uri);

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return DIVY_STCODE_NOT_EXIST;
		}

		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/*
		 * リソースがクライアントモジュールの場合
		 * クライアントモジュール領域外にMOVEされるのなら、
		 * モジュールプロパティを消す.
		 * (note) 既にdav_resource のuriは変更してしまたので、
		 * 	dist_rdb_r->uri を使ってプロパティを削除する必要あり。
		 */ 
		if (src_rdb_r->u_spec->infotype == DIVY_INFOTYPE_m_update_e &&
		    dist_rdb_r->u_spec->infotype != DIVY_INFOTYPE_m_update_e) {
			/* プロパティの削除 */
			if (_remove_clmodule_property(r, dist_rdb_r->uri, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to delete clmodule property."
					"(uri = %s)", src_rdb_r->uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);

				return 1;
			}
		}

		/* リソースならば、もうすることはありません */
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		return 0;
	}

	/*
	 * コレクションの場合
	 */
	/* 表示名が異なれば、変更する必要あり */
	if (is_rename) {
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE dav_resource "
				"SET rs_dispname_vc = ? "
				"WHERE rs_uri_txt = ?", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (src_uri = %s) "
				"Reason: %s", src_rdb_r->uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		stmt->setString(stmt, 1, dist_rdb_r->displayname);
		stmt->setString(stmt, 2, src_rdb_r->uri);

		update_cnt = stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update dav_resource table. "
				"(src_uri = %s, dist_uri = %s) Reason: %s", 
				src_rdb_r->uri, dist_rdb_r->uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		/* select してからupdate するまでの間に削除されてしまった。。*/
		if (update_cnt == 0) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to update dav_resource table. "
				"Maybe this resource was deleted or moved "
				"before this operation.(src_uri = %s, "
				"dst_uri = %s)", src_rdb_r->uri, dist_rdb_r->uri);

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return DIVY_STCODE_NOT_EXIST;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/*
	 * 共有コレクションの場合 (dav_resource の削除よりも前に実施すべし)
	 * * 通常コレクションへの移動
	 * 	dav_link_collection を削除
	 * 	* メール監視フラグが存在し、MOVEトリガあり
	 * 		divy_mailwatch のエントリを論理削除
	 * 	* メール監視フラグが存在し、MOVEトリガなし
	 * 		divy_mailwatch のエントリを削除
	 * * 名称変更
	 * 	dav_link_collection はそのまま
	 * 	divy_mailwatch のURIを変更
	 */
	/* 通常コレクション以下への階層移動ならばdav_link_collection を削除 */
	if (src_u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e &&
	    dst_u_spec->uritype  == DIVY_URITYPE_REGULAR) {

		/* SQLの準備 */
		stmt = dbconn->prepareStatement(dbconn,
				"DELETE FROM dav_link_collection "
				"WHERE lc_share_rs_id_c ="
				"(SELECT rs_rs_id_c FROM dav_resource"
				" WHERE rs_uri_txt = ?)", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (src_uri = %s) "
				"Reason: %s", src_rdb_r->uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		stmt->setString(stmt, 1, src_rdb_r->uri);

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to delete dav_link_collection for MOVE. "
				"(src_uri = %s, dist_uri = %s) Reason: %s", 
				src_rdb_r->uri, dist_rdb_r->uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/* src のメール監視フラグを取得 */
	if (src_u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e &&
	    _get_merge_mailwatch_property(r, src_rdb_r->uri, NULL, &mw, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get merged mailwatch. (uri = %s) ",
			src_rdb_r->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* src にメール監視フラグが存在し、MOVE トリガが立っていて、
	 * 移動先は通常コレクションの場合 */
	if (mw != NULL && mw->trigger_methods[M_MOVE] &&
	    dst_u_spec->uritype == DIVY_URITYPE_REGULAR) {

		/* divy_mailwatch のエントリを論理削除する */
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_mailwatch "
				"SET mw_l_del_flag = 1 "
				"WHERE mw_rs_uri_txt = ?", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (src_uri = %s) "
				"Reason: %s", src_rdb_r->uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		stmt->setString(stmt, 1, src_rdb_r->uri);

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to UPDATE divy_mailwatch for MOVE. "
				"(src_uri = %s, dist_uri = %s) Reason: %s", 
				src_rdb_r->uri, dist_rdb_r->uri,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}
	/* src にメール監視フラグが存在し、MOVE トリガが立っていなくて、
	 * 移動先は通常コレクションの場合 */
	else if (mw != NULL && !mw->trigger_methods[M_MOVE] &&
		 dst_u_spec->uritype  == DIVY_URITYPE_REGULAR) {

		/* divy_mailwatch のエントリを削除 */
		if (_remove_mailwatch_property(r, src_rdb_r->uri, NULL,
						DIVY_REMWATCH_IGNORE_USER, ts_ctx)) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to DELETE divy_mailwatch for MOVE. "
				"(src_uri = %s, dist_uri = %s) ",
				src_rdb_r->uri, dist_rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
	/* src にメール監視フラグが存在し、名称変更の場合 */
	else if (mw != NULL && is_rename &&
		 dst_u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e) {

		/*
		 * ゴミメール監視フラグエントリ対応
		 * もしdst にゴミ化したメール監視エントリがあれば最初に削除する
		 */
		if (_clean_gargage_mailwatch_property(r, dist_rdb_r->uri, ts_ctx)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to cleanup divy_mailwatch entry for garbage.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/* divy_mailwatch の監視URIを変更する */
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_mailwatch "
				"SET mw_rs_uri_txt = "
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
				DIVY_DBFUNC_SUBSTRING("mw_rs_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")" "
				"WHERE mw_rs_uri_txt = ?", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (src_uri = %s) "
				"Reason: %s", src_rdb_r->uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		stmt->setString(stmt, 1, dist_rdb_r->uri);
		stmt->setString(stmt, 2, src_rdb_r->uri);
		stmt->setString(stmt, 3, src_rdb_r->uri);

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to UPDATE divy_mailwatch for MOVE. "
				"(src_uri = %s, dist_uri = %s) Reason: %s", 
				src_rdb_r->uri, dist_rdb_r->uri,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}
	/* src にはメール監視フラグは無いが、名称変更の場合 */
	else if (mw == NULL && is_rename &&
		 dst_u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e) {
		/*
		 * ゴミメール監視フラグエントリ対応
		 * もしdst にゴミ化したメール監視エントリがあれば最初に削除する
		 */
		if (_clean_gargage_mailwatch_property(r, dist_rdb_r->uri, ts_ctx)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to cleanup divy_mailwatch entry for garbage.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
	else {
		/* 何もしなくていい */
	}

	/*
	 * dav_resource の更新
	 * src_rdb_r->uri 以下のリソースとコレクションのuriとdepth を更新
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"UPDATE dav_resource "
			"SET rs_uri_txt = "
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
			DIVY_DBFUNC_SUBSTRING("rs_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")","
			"rs_depth_i = rs_depth_i + "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_INTEGER)" "
			"WHERE rs_uri_txt = ? OR rs_uri_txt LIKE ? "
			DIVY_DBFUNC_ESCAPE_CLAUSE, p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (src_uri = %s) "
			"Reason: %s", src_rdb_r->uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, dist_rdb_r->uri);
	stmt->setString(stmt, 2, src_rdb_r->uri);
	stmt->setInt(stmt,    3, dist_rdb_r->depth - src_rdb_r->depth);
	stmt->setString(stmt, 4, src_rdb_r->uri);
	/* uri に含まれるワイルドカード文字をエスケープする */
	stmt->setString(stmt, 5, apr_pstrcat(p,
			stmt->escWildCard(stmt, src_rdb_r->uri), "/%", NULL));

	/* SQL文の実行 */
	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to executeUpdate for dav_resource update. "
			"(src_uri = %s, dist_uri = %s) Reason: %s", 
			src_rdb_r->uri, dist_rdb_r->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* select してからupdate するまでの間に削除されてしまった。。*/
	if (update_cnt == 0) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to update dav_resource table. "
			"Maybe this resource was deleted or moved "
			"before this operation.(src_uri = %s)",
			src_rdb_r->uri);

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return DIVY_STCODE_NOT_EXIST;
	}

	/* 資源の解放 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * ネームスペースURI(ネームスペース名) とネームスペースID の関係を以下の
 * ハッシュに格納して返却する。
 *
 * ns_id_hash : key = ネームスペースID(char*),  value = ネームスペースURI(char*)
 * ns_uri_hash: key = ネームスペースURI(char*), value = ネームスペースID(char *)
 */ 
DIVY_DECLARE(int) divy_rdbo_build_nsmap_hash(request_rec *r,
						apr_hash_t **ns_id_hash,
						apr_hash_t **ns_uri_hash)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	char *ns_id, *ns_uri;

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.");
		return 1;
	}
	dbconn->startTrans(dbconn, 0);

	stmt = dbconn->prepareStatement(dbconn,
			"SELECT ns_id_i, ns_name_vc FROM dav_namespace", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for dav_namespace."
			"Reason: %s", stmt->getMsg(stmt));

		dbconn->rollback(dbconn);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet for dav_namespace."
			"Reason: %s", rset->getMsg(rset));

		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/*
	 * 結果の取得
	 * 取得結果は、ハッシュに入れる
	 */
	*ns_id_hash  = apr_hash_make(p);
	*ns_uri_hash = apr_hash_make(p);
	while (rset->next(rset) == DB_TRUE) {
		ns_id  = rset->getString(rset, 1);
		ns_uri = rset->getString(rset, 2);
		apr_hash_set(*ns_id_hash,  ns_id,  APR_HASH_KEY_STRING, ns_uri);
		apr_hash_set(*ns_uri_hash, ns_uri, APR_HASH_KEY_STRING, ns_id);
	}

	/* 終了処理 */
	dbconn->commit(dbconn);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたns_uri のネームスペースをdav_namespace テーブルに格納する。
 * ネームスペースID は、自動的に採番し、引数ns_id_str として返却します。
 *
 */ 
DIVY_DECLARE(int) divy_rdbo_insert_dav_namespace(request_rec *r,
						const char *ns_uri,
						char **ns_id_str)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;

	*ns_id_str              = NULL;	/* 初期化してしまいます */

	if (IS_EMPTY(ns_uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"ns_uri is NULL");
		return 1;
	}

	/* トランザクションコンテキストの作成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* dav_namespaceテーブルにネームスペースURIエントリを追加 */
	stmt = dbconn->prepareStatement(dbconn,
			"INSERT INTO dav_namespace "
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_ORACLE) || defined(DIVY_DBMS_DB2)
			"(ns_id_i, ns_name_vc) "
			"VALUES("DIVY_DBFUNC_CREATE_DP_SEQ", ?)"
#elif defined(DIVY_DBMS_SYBASE)	/* sybase */
			"(ns_name_vc) "
			"VALUES(?)"
#endif
			, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for insert dav_namespace."
			"Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	stmt->setString(stmt, 1, ns_uri);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert dav_namespace."
			"Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * 採番したネームスペースID を取得する
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"SELECT ns_id_i "
			"FROM dav_namespace WHERE ns_name_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for select dav_namespace."
			"Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setString(stmt, 1, ns_uri);

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select dav_namespace."
			"Reason: %s", rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		*ns_id_str = rset->getString(rset, 1);
	}

	if (IS_EMPTY(ns_id_str)) {	/* 念のため */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get new namespace id.");

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL; 
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 終了処理 */
	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたuri の Dead プロパティを取得して、返却する。
 *
 * @param r request_rec *
 * @param uri const char * URI を表す文字列。2重スラッシュ、末尾スラッシュは禁止.
 * @param d_pr divy_rdbo_dproperty ** 取得したDead プロパティ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_dead_property_by_uri(request_rec *r,
						const char *uri, divy_rdbo_dproperty **d_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	divy_rdbo_dproperty *_d_pr = NULL;
	divy_db_transaction_ctx *ts_ctx = NULL;

	*d_pr = NULL;

	/* トランザクションコンテキストの作成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	stmt = dbconn->prepareStatement(dbconn,
			"SELECT "
			"dp_rs_id_c, "
			"dp_ns_id_i, "
			"dp_name_vc, "
			"dp_value_txt, "
			"dp_lang_tag_vc "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2  */
			"FROM dav_dead_property dp"
			" INNER JOIN dav_resource r"
			" ON dp.dp_rs_id_c = r.rs_rs_id_c "
			"WHERE r.rs_uri_txt = ? "
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
			" FROM dav_dead_property dp, dav_resource r"
			" WHERE dp.dp_rs_id_c = r.rs_rs_id_c"
			" AND r.rs_uri_txt = ? "
#endif
			"ORDER BY dp_rs_id_c, dp_ns_id_i", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s)\n "
			"Reason: %s", uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	stmt->setString(stmt, 1, uri);
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(uri = %s) "
			"Reason: %s", uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {
		if (*d_pr == NULL) {
			*d_pr = _d_pr = apr_pcalloc(p, sizeof(divy_rdbo_dproperty));
		}
		else {
			_d_pr->next = apr_pcalloc(p, sizeof(divy_rdbo_dproperty));
			_d_pr = _d_pr->next;
		}

		_d_pr->rsid         = rset->getString(rset, 1);
		_d_pr->ns_id        = rset->getInt(rset,    2);
		_d_pr->name         = rset->getString(rset, 3);
		_d_pr->value        = rset->getString(rset, 4);
		_d_pr->lang_tag     = rset->getString(rset, 5);
		_d_pr->patch_action = DIVY_DP_NOTHING;	/* 初期化 */
		_d_pr->next         = NULL;
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたrsid, ns_id, name の Dead プロパティを取得して返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_dead_property(request_rec *r,
						const char *rsid, apr_int32_t ns_id, const char *name,
						divy_rdbo_dproperty **d_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;

	*d_pr = NULL;

	if (IS_EMPTY(rsid) || IS_EMPTY(name)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"rsid or name is EMPTY");
		return 1;
	}

	/* トランザクションコンテキストの作成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	stmt = dbconn->prepareStatement(dbconn,
			"SELECT "
			"dp_rs_id_c, "
			"dp_ns_id_i, "
			"dp_name_vc, "
			"dp_value_txt, "
			"dp_lang_tag_vc "
			"FROM dav_dead_property "
			"WHERE dp_rs_id_c = ? "
			"AND dp_ns_id_i = ? "
			"AND dp_name_vc = ? ", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (rsid = %s, name = %s) "
			"Reason: %s", rsid, name, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setString(stmt, 1, rsid);
	stmt->setInt(stmt,    2, ns_id);
	stmt->setString(stmt, 3, name);

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(rsid = %s, name = %s) "
			"Reason: %s", rsid, name, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		*d_pr = apr_pcalloc(p, sizeof(divy_rdbo_dproperty));

		(*d_pr)->rsid         = rset->getString(rset, 1);
		(*d_pr)->ns_id        = rset->getInt(rset,    2);
		(*d_pr)->name         = rset->getString(rset, 3);
		(*d_pr)->value        = rset->getString(rset, 4);
		(*d_pr)->lang_tag     = rset->getString(rset, 5);
		(*d_pr)->patch_action = DIVY_DP_NOTHING;	/* 初期化 */
		(*d_pr)->next         = NULL;
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたDead プロパティを新規登録する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_dead_property(request_rec *r,
					const divy_rdbo_dproperty *d_pr)
{
	return _insert_dead_property(r, d_pr, NULL);
}

/**
 * 指定されたDead プロパティの内容を更新する。
 * (note)
 * 	lang_tag は、明示的に指定されない限り、昔の値をそのまま使います。
 */
DIVY_DECLARE(int) divy_rdbo_update_dead_property(request_rec *r,
						const divy_rdbo_dproperty *d_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int update_cnt          = 0;
	int d                   = 0;
	divy_db_transaction_ctx *ts_ctx = NULL;

	if (d_pr == NULL || IS_EMPTY(d_pr->rsid) || IS_EMPTY(d_pr->name)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"d_pr or rsid or name is EMPTY");
		return 1;
	}

	/* トランザクションコンテキストの作成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* dav_dead_property のエントリを削除するかどうか判定する */
	if (IS_FILLED(d_pr->value)) {

		/*
		 * 値が空ではなかったのでdav_dead_property テーブルの更新
		 */
		if (d_pr->lang_tag != NULL) {
			stmt = dbconn->prepareStatement(dbconn, 
					"UPDATE dav_dead_property "
					"SET "
					"dp_value_txt = ?,"
					"dp_lang_tag_vc = ? "
					"WHERE dp_rs_id_c = ? "
					"AND dp_ns_id_i = ? "
					"AND dp_name_vc = ?",p);
		}
		else {
			stmt = dbconn->prepareStatement(dbconn, 
					"UPDATE dav_dead_property "
					"SET "
					"dp_value_txt = ? "
					"WHERE dp_rs_id_c = ? "
					"AND dp_ns_id_i = ? "
					"AND dp_name_vc = ?",p);
		}
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (rsid = %s) Reason: %s",
				d_pr->rsid, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		stmt->setString(stmt, 1, REPLACE_EMPTYSTR(d_pr->value));
		if (d_pr->lang_tag != NULL) {
			stmt->setString(stmt, 2, REPLACE_EMPTYSTR(d_pr->lang_tag));
			d = 1;
		}
		stmt->setString(stmt, d+2, d_pr->rsid);
		stmt->setInt(stmt,    d+3, d_pr->ns_id);
		stmt->setString(stmt, d+4, d_pr->name);

		update_cnt = stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update dav_dead_property. (rsid = %s)"
				"Reason: %s", d_pr->rsid, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* 更新できなかった場合(該当エントリがなかった場合) */
		if (update_cnt == 0) {

			/*
			 * Deadプロパティの値が空だったのでエントリが削除されたか、
			 * select してからupdateするまでの間に削除されたのどちらかなので
			 * INSERTしておく
			 */
			if (_insert_dead_property(r, d_pr, ts_ctx)) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to insert dead property. (rsid = %s, name = %s)",
					d_pr->rsid, d_pr->name);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}
	}
	/* 値が空だった場合 */
	else {
		/* エントリを消去してしまう */
		if (_remove_dead_property(r, d_pr, ts_ctx)) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete dead property for update. "
				"(rsid = %s, name = %s)", d_pr->rsid, d_pr->name);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/* 終了処理 */
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたDead プロパティを削除する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_dead_property(request_rec *r,
						const divy_rdbo_dproperty *d_pr)
{
	return _remove_dead_property(r, d_pr, NULL);
}

/**
 * ユーザの最終更新情報(アクセス時間, User-Agent) を記録する。
 * (認証が成功した場合のみ)
 * 
 */
DIVY_DECLARE(int) divy_rdbo_set_LastAccess(request_rec *r, divy_db_transaction_ctx *ts_ctx)
{
	DbConn *dbconn = NULL;
	DbPreparedStmt *stmt = NULL;
	const char *lastaccesscl;
	int upcnt;			/* アップデート件数 */
	int i;
	int iscommit = 0;
	unsigned char ch;
	apr_size_t clen;
	divy_sbuf *sql_buf = NULL;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションコンテキストの生成 */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	divy_sbuf_create(r->pool, &sql_buf, 1024);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"UPDATE divy_usr "
				"SET "
				"usr_last_access_bi = ?,"
				"usr_last_accesscl_vc = ? ");

	divy_sbuf_append(sql_buf, "WHERE usr_usr_id_vc = ?");

	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), r->pool);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s",
			stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL )  stmt->close(stmt);
		return 1;
	}

	/* r->userが存在しなかった場合はエラーとする */
	if (IS_EMPTY(r->user)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"r->user did not exist.");
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* User-Agent の取得 */
	lastaccesscl = dav_divy_get_user_agent(r);

	/* User-Agent に含まれる文字列が1バイト文字かどうかを検証 */
	clen = (IS_FILLED(lastaccesscl)) ? strlen(lastaccesscl) : 0;
	for (i = 0; i < clen; i++) {
		ch = (unsigned char) lastaccesscl[i];
		if (!apr_isascii(ch)) {
			lastaccesscl = NULL;	/* NULLにしてしまう */
			break;
		}
	}

	/* DIVY_LAST_ACCESSCL_MAXLEN以上あったらトランケートする */
	if (IS_FILLED(lastaccesscl) && clen > DIVY_LAST_ACCESSCL_MAXLEN) {
		lastaccesscl = dav_divy_truncate_str_perbyte(r->pool,
				lastaccesscl, DIVY_LAST_ACCESSCL_MAXLEN);
	}

	stmt->setBigInt(stmt, 1, dav_divy_get_now_epoch());
	stmt->setString(stmt, 2, REPLACE_EMPTYSTR(lastaccesscl));
	stmt->setString(stmt, 3, r->user);

	upcnt = stmt->executeUpdate(stmt, r->pool);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update divy_usr.(r->user = %s) Reason: %s",
			r->user, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	if (upcnt == 0) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"A user did not exist or it has not updated."
			"(r->user = %s)", r->user);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * ユーザがパスワードミスマッチでログインを失敗したことを記録
 * する。指定回数を上回った場合は、ログインを出来ないようにします
 */
DIVY_DECLARE(int) divy_rdbo_set_failed_login(request_rec *r)
{
	dav_divy_dir_conf *conf   = dav_divy_get_dir_config(r);
	DbConn *dbconn = NULL;
	DbPreparedStmt *stmt = NULL;
	divy_db_transaction_ctx *ts_ctx = NULL;
	divy_rdbo_extstatus *extstatus = NULL;
	char *extstatus_str   = NULL;
	int limit = 0; /* ログイン失敗許可回数 */
	apr_int32_t failed = 0;
	time_t now_time  = dav_divy_get_now_epoch();
	time_t last_time = divy_get_lastaccess(r);
	int lockout = 0;
	divy_uri_spec *u_spec = NULL;
	int mnum              = divy_get_method_number(r);
	int m_search          = ap_method_number_of("SEARCH");

	if (!divy_support_failedlogin_lockout(r)) {
		return TF_LOCKOUT_NORMAL;
	}

	if (!divy_support_extenduserstatus(r)) {
		return TF_LOCKOUT_NORMAL;
	}

	TRACE(r->pool);

	/*
	 */
	divy_parse_uri(r->pool, dav_divy_get_root_uri(r), r->uri, &u_spec);
	if ((mnum == m_search && u_spec->infotype == DIVY_INFOTYPE_m_update) ||
	    (mnum == m_search && u_spec->infotype == DIVY_INFOTYPE_m_msg) ||
	    (mnum == M_GET    && u_spec->infotype == DIVY_INFOTYPE_m_update_e)) {
		return TF_LOCKOUT_NORMAL;
	}

	/* 表立って出てこないメソッドはカウント対象にしない
	 * 但し、OPTIONSはTeamFileクライアント以外カウント対象にしない
	 * TeamFileクライアントは必ず一発目はOPTIONSを投げるからです
	 */
	if ((strcmp(r->method, "HEAD")) == 0 ||
			(mnum == M_OPTIONS
			 	&& apr_table_get(r->headers_in, DIVY_HEADER_REQ_CLIENT_INFO)
																== NULL)) {
		return TF_LOCKOUT_NORMAL;
	}

	/* ログインの許容回数を取得する */
	limit = conf->lockoutlimit;

	/* 今までの接続失敗カウントを取得する */
	failed = divy_get_loginfailedcount(r);

	/* ユーザ権限を取得 */
	extstatus = (divy_rdbo_extstatus*)divy_get_extstatus(r);

	/* システム権限のユーザならロックアウトの対象外 */
	if (divy_rdbo_has_sysexec_privilege(extstatus)) {
		return TF_LOCKOUT_NORMAL;
	}

	/*
	 * Office系のUser-Agentである場合は対象外
	 *
	 * OfficeのパスワードはTeamFileクライアントとは別でOS側の資格情報
	 * マネージャに記録されます。
	 * 例えばTeamFile側のパスワードを変更しても、この資格情報マネージャには
	 * 同時には記録できません。
	 * つまり、もしOfficeを開いていた場合は、認証情報は昔のままでアクセスを
	 * してしまいます。従来サーバ側では愚直にパスワードの失敗はカウントを
	 * 行ってきましたが、利用者側すると予期せぬ動きになってしまいます。
	 * ロックアウトを大量に引き起こす原因にもなるため、Officeの場合に限り
	 * カウントしないようにしました。もちろん認証はエラーです。
	 */
	if (divy_is_Office(r)) {
		return TF_LOCKOUT_NORMAL;
	}

	/* ロックアウトされていたら終了 */
	if (extstatus != NULL && divy_rdbo_has_lockout(extstatus)) {
		/* ロックアウトされていたユーザが再チャレンジができるか？ */
		if ((last_time + conf->unlocktime) > now_time) {
			return TF_LOCKOUT_NORMAL;
		}
	}

	/* 再挑戦回数が設定オーバーの場合はなにもできない */
	if (failed / limit >= conf->unlockretrycount) return TF_LOCKOUT_LOCKOUT;

	/* トランザクションコンテキストの生成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return TF_LOCKOUT_FAILED;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return TF_LOCKOUT_FAILED;
	dbconn = ts_ctx->dbconn;

	/* 失敗回数を増加させる */
	failed++;

	/**
	 * 
	 * ロックアウトされていなくて失敗回数がリミットに到達したらロックアウト
	 * ロックアウトされていても ...
	 * ロック回数 = 0 の場合
	 *     失敗回数がリミットを越えているならロックアウトする
	 * ロック回数 > 0 の場合
	 *     失敗回数をリミットの除算と比べて失敗回数と同じならロックアウト
	 */
	if ((!divy_rdbo_has_lockout(extstatus) && (failed >= limit)) ||
		(conf->unlockretrycount == 0 && (failed >= limit)) || 
		(conf->unlockretrycount >  0 && (failed != 0 && (failed % limit == 0) && (failed / limit) <= conf->unlockretrycount))) { 
		/* ロックアウトされたメールを送る */
		divy_ml_notify_failed_login(r);

		/* ロックアウトフラグ */
		lockout = 1;

		/* 強制に更新日付を更新する */
		(void) divy_rdbo_set_LastAccess(r, ts_ctx);

		/* リミット制限を超過 アカウント停止 */
		stmt = dbconn->prepareStatement(dbconn,
							"UPDATE divy_usr "
							"SET "
							"usr_extended_status_c = ?, "
							"usr_failed_access_count_i = ?"
							"WHERE usr_usr_id_vc = ? ", r->pool);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s",
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL )  stmt->close(stmt);
			return TF_LOCKOUT_FAILED;
		}

		divy_rdbo_set_active_user(extstatus, 0);	/* アクティブ */
		divy_rdbo_set_lockout(extstatus, 1);		/* ロックアウト */
		extstatus_str   = divy_rdbo_extstatus2str(extstatus,
												EXTSTATUS_TYPE_USR);
		stmt->setString(stmt, 1, extstatus_str);
		stmt->setInt(stmt,    2, failed);
		stmt->setString(stmt, 3, r->user);
	}
	else {
		/* ログイン失敗カウントのみを設定 */
		stmt = dbconn->prepareStatement(dbconn,
							"UPDATE divy_usr "
							"SET "
							"usr_failed_access_count_i = ?"
							"WHERE usr_usr_id_vc = ? ", r->pool);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s",
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL )  stmt->close(stmt);
			return TF_LOCKOUT_FAILED;
		}

		stmt->setInt(stmt,    1, failed);
		stmt->setString(stmt, 2, r->user);
	}

	/* SQL実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, r->pool);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to UPDATE divy_usr. Reason: %s", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
			return TF_LOCKOUT_FAILED;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	divy_db_commit_transaction(ts_ctx);

	return (lockout) ? TF_LOCKOUT_LOCKOUT : TF_LOCKOUT_NORMAL;
}

/**
 * ロックアウトされているユーザをアクティブユーザへ変更する。
 */
DIVY_DECLARE(int)divy_rdbo_refresh_lockout(request_rec *r)
{
	dav_divy_dir_conf *conf   = dav_divy_get_dir_config(r);
	divy_rdbo_extstatus *extstatus = NULL;
	char *extstatus_str   = NULL;
	time_t now_time  = dav_divy_get_now_epoch();
	time_t last_time = divy_get_lastaccess(r);
	DbConn *dbconn = NULL;
	DbPreparedStmt *stmt = NULL;
	divy_db_transaction_ctx *ts_ctx = NULL;
	int limit = conf->lockoutlimit;
	int unlocktime = conf->unlocktime;
	int retry = conf->unlockretrycount;
	int failed = divy_get_loginfailedcount(r);

	TRACE(r->pool);

	if (!divy_support_failedlogin_lockout(r)) {
		return 0;
	}
	if (!divy_support_extenduserstatus(r)) {
		return 0;
	}

	/* ログイン失敗カウントがあるユーザだけが対象となる */
	if (failed == 0) return 0;

	/* ユーザの状態を取得する */
	extstatus = (divy_rdbo_extstatus*)divy_get_extstatus(r);

	/* ロックアウトされているユーザだけが対象とする　*/
	if (!divy_rdbo_has_lockout(extstatus)) return 0;

	/* 1未満は成功しても処理はできない */
	if (unlocktime < 1) return 1;

	/* 再チャレンジに時間に到達していない場合は処理できない */
	if ((last_time + unlocktime) > now_time) return 1;

	/* 再チャレンジｎ回数が規定の回数に到達していたら処理はできない */
	if (retry > 0 && failed / limit >= retry) return 1;
	
	/* トランザクションコンテキストの生成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* ステータスを正常な状態に戻します */
	divy_rdbo_set_active_user(extstatus, 1);	/* アクティブ */
	divy_rdbo_set_lockout(extstatus, 0);		/* 解除       */
	extstatus_str   = divy_rdbo_extstatus2str(extstatus,
												EXTSTATUS_TYPE_USR);

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;
	stmt = dbconn->prepareStatement(dbconn,
							"UPDATE divy_usr "
							"SET "
							"usr_failed_access_count_i = 0, "
							"usr_extended_status_c = ? "
							"WHERE usr_usr_id_vc = ?" ,r->pool);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s",
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL )  stmt->close(stmt);
		return 1;
	}

	stmt->setString(stmt, 1, extstatus_str);
	stmt->setString(stmt, 2, r->user);

	/* SQL実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, r->pool);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to UPDATE divy_usr. Reason: %s", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
			return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * ユーザのセッションを調べて新規・更新を行う
 * これは認証が成功する前にも記録します。
 * パラメータの*passwordには値がはいっていることもあります。
 *
 * @param r request_rec *
 * @param uid const char ** ユーザID
 * @param sessionid char ** セッションID
 * @param password char ** パスワード
 * @return int (0: 正常 / 1: エラー)
 */
DIVY_DECLARE(int) divy_rdbo_session_manager(request_rec *r, char** uid, char **sessionid, char **password)
{
	dav_divy_dir_conf *conf   = dav_divy_get_dir_config(r);

	apr_pool_t *p = r->pool;
	DbConn *dbconn = NULL;
	DbPreparedStmt *stmt = NULL;
	DbResultSet    *rset = NULL;
	divy_db_transaction_ctx *ts_ctx = NULL;
	time_t now_time = dav_divy_get_now_epoch();
	time_t current_time = 0;
	
	TRACE(r->pool);

	/* トランザクションコンテキストの生成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	if (IS_FILLED(*sessionid)) {
		stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				" ses_sid_c "
				",ses_login_bi "
				",ses_usr_usr_id_vc "
				",ses_usr_password_vc "
				"FROM divy_session "
				"WHERE ses_sid_c = ?" ,p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		stmt->setString(stmt, 1, *sessionid);

		/* SQL実行 失敗時はエラー */
		rset = stmt->executeQuery(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to SELECT divy_session. Reason: %s",
					rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL)   rset->close(rset); rset = NULL;
			if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
			return 1;
		}

		if (rset->next(rset) == DB_TRUE) {
			/* 記録されている時間 */
			current_time = rset->getBigInt(rset, 2); 
			/* セッションに保存されているユーザID */
			*uid         = apr_pstrdup(p, rset->getString(rset, 3));
			/* セッションに保存されているパスワード */
			*password    = apr_pstrdup(p, rset->getString(rset, 4));
		}

		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* 更新もしくは削除 */
		/* 既にあるセッションの有効期限を調べる */
		if ( conf->sessiontimeout != DIVY_SESSION_TIMEOUT_INFINITY && 
				(now_time - current_time) > conf->sessiontimeout ) {
			stmt = dbconn->prepareStatement(dbconn,
					"DELETE "
					" FROM "
					" divy_session "
					" WHERE "
					"ses_sid_c = ?", p);

			if (stmt->getCode(stmt) != DB_SUCCESS) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to DbPreparedStmt. divy_session update."
						"(SID = %s) Reason: %s", *sessionid, stmt->getMsg(stmt));
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt);

				return 1;
			}

			stmt->setString(stmt, 1, *sessionid);

			(void) stmt->executeUpdate(stmt, p);
			if (stmt->getCode(stmt) != DB_SUCCESS) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to DELETE divy_session. (SID = %s)"
						"Reason: %s", *sessionid, stmt->getMsg(stmt));
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt);

				return 1;
			}

			*sessionid = NULL; /* エラーとさせる */
			return 1;
		}
		else {
			/* 更新 */
			stmt = dbconn->prepareStatement(dbconn,
					"UPDATE "
					" divy_session "
					"SET "
					"ses_sid_c = ?,"
					"ses_login_bi = ? "
					" WHERE "
					"ses_sid_c = ? ", p);

			if (stmt->getCode(stmt) != DB_SUCCESS) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to DbPreparedStmt. divy_session update."
						"(SID = %s) Reason: %s", *sessionid, stmt->getMsg(stmt));
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt);

				return 1;
			}

			/* 
			 * ### TODO
			 * divy_util_auth_make_sid(r)をretsidに入れることにより
			 * 毎回新しくSIDを作り設定することができます。
			 * しかし、フレームで動作している現状ではページごとに
			 * SIDを更新してしまうと同じSIDを更新することができません。
			 * フレーム１が更新したらフレーム2はエラーになってしまう。
			 * これはフレーム1がSIDを書き換えてしまうためフレーム2が
			 * 設定しようとしてもそのSIDがない為です。
			 * 今はリクエストと同じsidを利用させるようにしていますが、
			 * 将来的にフレームを利用しなくなる場合はここを改造できる
			 * でしょう。
			 */
			/* retsid = divy_util_auth_make_sid(r); */

			stmt->setString(stmt, 1, *sessionid); /* 新規SID */
			stmt->setBigInt(stmt, 2, now_time);
			stmt->setString(stmt, 3, *sessionid);

			(void) stmt->executeUpdate(stmt, p);
			if (stmt->getCode(stmt) != DB_SUCCESS) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to UPDATE divy_session. (SID = %s)"
						"Reason: %s", *sessionid, stmt->getMsg(stmt));
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt);

				return 1;
			}
		}
	}
	else {
		/* セッションIDが存在しない場合 */
		if (IS_EMPTY(*uid)) {
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);

			/* ユーザIDがないのはエラー */
			return 1;
		}

		/* セッションIDを作成する */
		*sessionid = _make_session_sid(p);

		/* userのセッションをすべて削除する */
		/* コメントを外すと同じIDでログインができなくなります。後が優先 */
		/*
		(void) divy_rdbo_delete_session(r, ts_ctx, *uid, NULL);
		*/

		/* 新規登録 */
		stmt = dbconn->prepareStatement(dbconn,
				"INSERT INTO divy_session "
				"(ses_sid_c, "
				" ses_login_bi, "
				" ses_usr_usr_id_vc, "
				" ses_usr_password_vc) "
				"VALUES (?, ?, ?, ?)", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s",
					stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		/* バインド */
		stmt->setString(stmt, 1, *sessionid);
		stmt->setBigInt(stmt, 2, now_time);
		stmt->setString(stmt, 3, *uid);
		stmt->setString(stmt, 4, *password);

		/* 実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to insert divy_session."
					"(user = %s, SID = %s) Reason: %s",
					r->user, *sessionid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

	}

	/* commit */
	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * セッション情報を取得する
 *
 * @param r request_rec *
 * @param sessionid const char*
 *
 * @return divy_rdbo_session (NULL: データなし / not NULL セッション情報あり)
 *
 */
DIVY_DECLARE(divy_rdbo_session*)
divy_rdbo_get_session(request_rec *r, const char *sid)
{
	divy_rdbo_session *session = NULL, *first = NULL;
	apr_pool_t *p = r->pool;
	DbConn *dbconn		= NULL;
	DbPreparedStmt *stmt = NULL;
	DbResultSet *rset 	= NULL;
	divy_db_transaction_ctx *ts_ctx  = NULL;
	divy_sbuf *sql_buf = NULL;
	int idx = 1;

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return NULL;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return NULL;
	dbconn = ts_ctx->dbconn;

	/* SQL文を組み立てる */
	divy_sbuf_create(p, &sql_buf, 1024);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT "
				" ses_sid_c "
				",ses_login_bi "
				",ses_usr_usr_id_vc "
				",ses_usr_password_vc "
				"FROM divy_session ");

	if (sid != NULL) {
		divy_sbuf_append(sql_buf, "WHERE ses_sid_c = ?");
	}

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return NULL;
	}

	if (sid != NULL) {
		stmt->setString(stmt, 1, sid);
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to SELECT divy_session. Reason: %s",
												rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL)   rset->close(rset); rset = NULL;
			if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
			return NULL;
	}

	while (rset->next(rset) == DB_TRUE) {
		idx = 1;
		if (first == NULL) {
			session = first = apr_pcalloc(p, sizeof(divy_rdbo_session));
		}
		else {
			session->next = apr_pcalloc(p, sizeof(divy_rdbo_session));
			session = session->next;
		}
		session->sid       = rset->getString(rset, idx++);
		session->userid    = rset->getString(rset, idx++);
		session->logindate = rset->getBigInt(rset, idx++); 
		session->password  = rset->getString(rset, idx++);
	}

	if (rset != NULL)   rset->close(rset); rset = NULL;
	if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
	divy_db_commit_transaction(ts_ctx);

	return first;
}

/**
 * ユーザのセッションを削除する
 *
 * @param r request_rec *	リクエスト構造体
 * @param ts_ctx divy_db_transaction_ctx * トランザクション
 * @param user const char*	ユーザID
 * @param sessionid const char* セッションID 
 * (note)
 * userとsessionidの&で削除が行えます。
 * userだけでsessionidがNULLの場合はそのユーザIDをすべてクリア
 * sessionidだけでuserがNULLの場合は、セッションIDだけクリア
 * userとセッションIDどちらもNULLの場合はすべてのセッションをクリア
 */
DIVY_DECLARE(int) divy_rdbo_delete_session(request_rec *r,
						divy_db_transaction_ctx *ts_ctx,
						const char *user,
						const char *sessionid)
{
	apr_pool_t *p = r->pool;
	DbConn *dbconn = NULL;
	DbPreparedStmt *stmt = NULL;
	divy_sbuf *sql_buf = NULL;
	int idx = 0;
	int iscommit = 0;

	TRACE(p);

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	if (ts_ctx == NULL) {
		iscommit = 1;
		/* トランザクションコンテキストの生成 */
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文を組み立てる */
	divy_sbuf_create(p, &sql_buf, 1024);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"DELETE "
				" FROM "
				" divy_session ");
				
	if (IS_FILLED(user)) {
		divy_sbuf_append(sql_buf,
				" WHERE "
				" ses_usr_usr_id_vc = ?");
	}

	if (IS_FILLED(sessionid)) {

		if (IS_FILLED(user))
			divy_sbuf_append(sql_buf, " AND ");
		else
			divy_sbuf_append(sql_buf, " WHERE ");

		divy_sbuf_append(sql_buf,
				" ses_sid_c = ?");
	}

	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	idx = 1;
	if (IS_FILLED(user)) {
		stmt->setString(stmt, idx++, user);
	}
	if (IS_FILLED(sessionid)) {
		stmt->setString(stmt, idx++, sessionid);
	}

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to delete divy_session. "
				"(user %s / sessionid = %s) Reason: %s",
				REPLACE_NULL(user), REPLACE_NULL(sessionid),
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* commit */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL)   stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたソースコレクション( = linkdbsearch の結果フォルダ) のuri が
 * 持っている共有コレクションの情報を取得して返却する。
 */
DIVY_DECLARE(int) divy_rdbo_get_sharedcollection_property(request_rec *r,
						const char *uri,
						divy_rdbo_shlink **shlink_pr)
{
	return _get_sharedcollection_property(r, uri, shlink_pr, NULL);
}

/**
 * セッション情報のパスワードを更新する
 *
 * @param r request_rec *
 * @param ts_ctx divy_db_transaction_ctx *
 * @param sessionid const char * 対象のセッションID(NULLの場合すべてのセッション)
 * @param user const char *		対象のユーザID（NULLの場合すべてのユーザ)
 * @param password const char * 変更するパスワード（NULLの場合エラー)
 *
 */
DIVY_DECLARE(int)
divy_rdbo_update_session_password(request_rec *r, divy_db_transaction_ctx *ts_ctx, const char* sessionid, const char* user, const char* password)
{
	apr_pool_t *p = r->pool;
	DbConn *dbconn = NULL;
	DbPreparedStmt *stmt = NULL;
	divy_sbuf *sql_buf = NULL;
	int idx = 0;
	int iscommit = 0;

	TRACE(p);

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	if (ts_ctx == NULL) {
		iscommit = 1;
		/* トランザクションコンテキストの生成 */
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文を組み立てる */
	divy_sbuf_create(p, &sql_buf, 1024);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"UPDATE divy_session "
				" SET "
				" ses_usr_password_vc = ?");
				
	if (IS_FILLED(user)) {
		divy_sbuf_append(sql_buf,
				" WHERE "
				" ses_usr_usr_id_vc = ?");
	}

	if (IS_FILLED(sessionid)) {

		if (IS_FILLED(user))
			divy_sbuf_append(sql_buf, " AND ");
		else
			divy_sbuf_append(sql_buf, " WHERE ");

		divy_sbuf_append(sql_buf,
				" ses_sid_c = ?");
	}

	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	idx = 1;
	if (IS_FILLED(password)) {
		stmt->setString(stmt, idx++, password);
	}
	if (IS_FILLED(user)) {
		stmt->setString(stmt, idx++, user);
	}
	if (IS_FILLED(sessionid)) {
		stmt->setString(stmt, idx++, sessionid);
	}

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update password divy_session. "
				"(user %s / sessionid = %s) Reason: %s",
				REPLACE_NULL(user), REPLACE_NULL(sessionid),
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* commit */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL)   stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたソースコレクションのuri にshlink_pr が示す共有コレクション
 * へのリンクを追加する。
 */
DIVY_DECLARE(int) divy_rdbo_update_sharedcollection_property(request_rec *r,
					const char *uri,
					const divy_rdbo_shlink *shlink_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn   = NULL;
	DbPreparedStmt  *stmt     = NULL;
	apr_pool_t *p             = r->pool;
	int iscommit = 0;
	int insert_cnt = 0;
	const divy_rdbo_shlink *shlink = NULL;

	TRACE(p);

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"uri is empty !!");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Could not operation. Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* 
	 * ソースコレクションuri が持っていた全ての共有コレクション
	 * 情報を削除する。
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM dav_link_collection "
			"WHERE lc_uri_txt = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt.(uri = %s) "
				"Reason: %s", uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, uri);

	/* sql実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to delete divy_link_collection. "
				"(uri = %s) Reason: %s", 
				uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* SQL文準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, 
			"INSERT INTO dav_link_collection ("
			"lc_uri_txt, "
			"lc_share_rs_id_c) "
			"(SELECT "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))
			", rs_rs_id_c"
			" FROM dav_resource"
			" WHERE rs_uri_txt = ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(src_uri = %s, sh_uri = %s) Reason: %s",
			uri, shlink->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/*
	 * ソースコレクションuri に指定された共有コレクションshlink_pr を
	 * 関連付ける
	 */
	for (shlink = shlink_pr; shlink; shlink = shlink->next) {
		/* 必須項目のチェック */
		if (IS_EMPTY(shlink->uri)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Shlink uri is NULL.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		/* バインド */
		stmt->setString(stmt, 1, uri);
		stmt->setString(stmt, 2, shlink->uri);

		/* sql実行 失敗時はエラー */
		insert_cnt = stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert dav_link_collection. "
				"(src_uri = %s, sh_uri = %s) Reason: %s",
				uri, shlink->uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		/* 挿入データが無かった場合
		 * (共有コレクションが削除されていた場合に発生するので警告) */
		if (insert_cnt == 0) {
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"Failed to insert dav_link_collection. "
				"Maybe the sharedcollection was deleted before "
				"this operation. Please check client cache."
				"(sh_uri = %s)", shlink->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);

			return DIVY_STCODE_SHCOL_NOTFOUND;
		}
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたuriの共有コレクションを作成し、shsrccol_pr が示す
 * ソースコレクションとの関係を記録する。
 */
DIVY_DECLARE(int) divy_rdbo_insert_sharedsrccollection_info(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					const divy_rdbo_shsrccol *shsrccol_pr)
{
	DbConn          *dbconn   = NULL;
	DbPreparedStmt  *stmt     = NULL;
	apr_pool_t *p             = r->pool;
	int insert_cnt            = 0;
	divy_db_transaction_ctx *ts_ctx;
	const divy_rdbo_shsrccol *tmp_shsrccol_pr = NULL;

	TRACE(p);

	if (rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The \"rdb_r\" of sharedcollection is "
				"empty !!");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * ゴミメール監視フラグエントリ対応
	 * ゴミ化したメール監視エントリがあれば最初に削除する
	 */
	if (_clean_gargage_mailwatch_property(r, rdb_r->uri, ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to cleanup divy_mailwatch entry for garbage.");
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * 共有コレクションの新規作成 
	 * rsid, etagは渡される前提です。
	 */
	if (_insert_property(r, rdb_r, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to insert dav_resource.(uri = %s)", 
				rdb_r->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * リンク情報作成(dav_link_collection)
	 */
	for (tmp_shsrccol_pr = shsrccol_pr; tmp_shsrccol_pr; 
			tmp_shsrccol_pr = tmp_shsrccol_pr->next){

		/* 必須項目のチェック */
		if (IS_EMPTY(tmp_shsrccol_pr->uri)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Could not operation. Uri(srccol) is "
					"NULL.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/* SQL文準備 失敗時はエラー */
		stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO dav_link_collection ("
				"lc_uri_txt, "
				"lc_share_rs_id_c) "
				"VALUES (?, ?)", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. "
					"(uri = %s) Reason: %s", 
					rdb_r->uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
		/* バインド */
		stmt->setString(stmt, 1, tmp_shsrccol_pr->uri);
		stmt->setString(stmt, 2, rdb_r->rsid);

		/* sql実行 失敗時はエラー */
		insert_cnt = stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to insert dav_link_collection. "
					"(sharedresourceid = %s) Reason: %s", 
					rdb_r->rsid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
		/* 挿入件数が１件以外はエラー(レアケース) */
		if (insert_cnt != 1){
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Failed to insert dav_link_collection. "
					"Unknown error is occered. Check data. "
					"(sharedresourceid = %s)", rdb_r->rsid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * divy_rdbo_allget_record:
 * resource->uri以下のレコードをすべてハッシュに詰め込む
 *
 * @param r request_rec *
 * @param lp divy_rdbo_resource **
 * @param key const char *		検索するURI（キー)
 * @param depth int			階層の深さ ( 0 / 1 / DAV_INFINITY)
 * @param expire int			タイムアウトチェック( 0: no / 1: yes)
 * @return 結果 ( 0: 成功 / 1:失敗)
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_lock_record(request_rec *r,
						divy_rdbo_lock_resource **lp,
			      			const char *key, int depth,
			      			int expire)
{
	DbConn 		*dbconn   = NULL;
	DbPreparedStmt 	*stmt     = NULL;
	DbResultSet     *rset     = NULL;

	apr_pool_t 	*p        = r->pool;
	divy_rdbo_lock_resource *lp_p;
	divy_rdbo_lock	*lockrec  = NULL;
	char		*hash_key = NULL;
	int		dircnt	  = 0;

	const char *uri;

	char *sqlstmt = "SELECT "
		        " lk_uri_txt"
		        ",lk_locktoken_c"
		        ",lk_lockkbn_i"
		        ",lk_scope_i"
		        ",lk_depth_i"		/* SQL解析用のDEPTH */
		        ",lk_timeout_bi"
		        ",lk_owner_vc"
		        ",lk_lockkey_txt"
		        ",lk_auth_usr_id_vc"
		        ",lk_isnull_i"
			",lk_lock_depth_i"	/* ロックのDEPTH    */
		        " FROM "
		        " dav_lock "
		        " WHERE ";
	divy_rdbo_lock *newlockrec;
	int	        timeout;
	dav_locktoken  *locktoken;

	TRACE(p);

	/* ハッシュテーブルを作成する */
	lp_p = apr_pcalloc(p ,sizeof(divy_rdbo_lock_resource));
	lp_p->token_list_hash = apr_hash_make(p);
	*lp = lp_p;

	uri    = dav_divy_remove_endslash(p, key);

	/* dbconnを取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.");
		return 1;
	}

	/* SQL文を作成する */
	sqlstmt = apr_pstrdup(p, "SELECT "
				 " lk_uri_txt"
				 ",lk_locktoken_c"
				 ",lk_lockkbn_i"
				 ",lk_scope_i"
				 ",lk_depth_i"
				 ",lk_timeout_bi"
				 ",lk_owner_vc"
				 ",lk_lockkey_txt"
				 ",lk_auth_usr_id_vc"
				 ",lk_isnull_i"
				 ",lk_lock_depth_i"
				 " FROM "
				 " dav_lock "
				 " WHERE "
				 " lk_depth_i = ? OR ");

	/* depthを設定
	 *
	 * ここではdepthによって異なるSQLを発行しなければなりません。
	 * しかしながらロックではDEPTH=1が許されていませんがここに
	 * 来る前にdepthのチェックは終了していることを信じてこのルーチンの
	 * depth=1はPROPFIND用として用意してあります。
	 **/
	if (depth == DAV_INFINITY) {
		sqlstmt = apr_pstrcat(p, sqlstmt, "lk_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE, NULL);
	}
	else if (depth < 1) {
		sqlstmt = apr_pstrcat(p, sqlstmt, " lk_uri_txt = ? OR lk_uri_txt = ?" , NULL);
	}
	else {
		/* ロックメソッドではここには来ない */
		sqlstmt = apr_pstrcat(p, sqlstmt, 
					"lk_uri_txt LIKE ? "
					DIVY_DBFUNC_ESCAPE_CLAUSE
					" AND "
					" lk_depth_i IN (?,?)", NULL);
	}

	/* トランザクション開始 */
	dbconn->startTrans(dbconn, 0);

	stmt = dbconn->prepareStatement(dbconn, sqlstmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s)"
			"Reason: %s" , uri, stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* #####
	 * URI以下だけOPENする為に親のコレクションを検索したい場合は
	 * URI以下だけのSQLでは不十分です。
	 * そのため、親のコレクションを取得する為に、URIの階層の
	 * 一つ上(apr_count_dirs(uri) -1 )を余分に取得しています。
	 * しかし、この方法はあまりよくありません。
	 * 本来であれば一つ上に階層のコレクションだけでいいはずです。
	 */
	stmt->setInt(stmt, 1, divy_count_dirs(uri) - 1);

	if (depth == DAV_INFINITY) {
		stmt->setString(stmt, 2, apr_pstrcat(p,
				stmt->escWildCard(stmt, uri), "%", NULL));
	}
	else if (depth < 1) {
		/*
		 * depth0でコレクションをPROPFINDした場合その情報として
		 * 最後に/を含めたものも同時に取得しなければならない
		 *
		 * /a/b  <--- リクエストで来るURI（スラッシュなし）
		 * /a/b/ <--- リクエストで利用しなければならないURI
		 */
		stmt->setString(stmt, 2, uri);
		stmt->setString(stmt, 3, dav_divy_append_endslash(p, uri));
	}
	else {
		dircnt = divy_count_dirs(uri);
		stmt->setString(stmt, 2, apr_pstrcat(p,
				stmt->escWildCard(stmt, uri), "%", NULL));
		stmt->setInt(stmt,    3, dircnt);
		stmt->setInt(stmt,    4, dircnt + 1);
	}

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get LockTable. (uri = %s)"
		        "Reason: %s", uri, rset->getMsg(rset));
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		dbconn->rollback(dbconn);
		return 1;
	}

	while(rset->next(rset) == DB_TRUE) {

		newlockrec = apr_pcalloc(p, sizeof(divy_rdbo_lock));

		/*
		 * タイムアウトを調べてexpireしていた場合はハッシュテーブル
		 * にセットする。
		 * expireのフラグがたっていない場合はタイムアウトのチェックを
		 * 行わない。
		 */
		locktoken       = apr_pcalloc(p, sizeof(*locktoken));
		locktoken->uuid = rset->getString(rset, 2);
		timeout         = rset->getInt(rset, 6);

		newlockrec->uri     = apr_pstrdup(p, rset->getString(rset, 1));
		newlockrec->locktoken  = (dav_locktoken *)locktoken;

		if (expire && dav_divy_lock_expired(timeout)) {
			if ((lockrec = apr_hash_get(lp_p->token_list_hash,
				       	       "EXPIRED_TOKEN",
					       APR_HASH_KEY_STRING)) == NULL) {
				apr_hash_set(lp_p->token_list_hash,
					     "EXPIRED_TOKEN",
					     APR_HASH_KEY_STRING, newlockrec);
			}
			else {
				newlockrec->next = lockrec;
				lockrec 	 = newlockrec;
				apr_hash_set(lp_p->token_list_hash,
					     "EXPIRED_TOKEN",
					     APR_HASH_KEY_STRING, lockrec);
			}
			continue;
		}

		newlockrec->kbn     = rset->getInt(rset, 3);
		newlockrec->scope   = rset->getInt(rset, 4);
		newlockrec->depth   = rset->getInt(rset, 5);
		newlockrec->timeout = timeout;
		newlockrec->owner   = apr_pstrdup(p, rset->getString(rset, 7));
		newlockrec->key     = apr_pstrdup(p, rset->getString(rset, 8));
		newlockrec->userid  = apr_pstrdup(p, rset->getString(rset, 9));
		newlockrec->is_null = rset->getInt(rset, 10);
		newlockrec->lockdepth = rset->getInt(rset,11);

		/*
		 * uriをキーにしてハッシュテーブルを作成する。
		 * 同じuriの場合は内部でリストとして保持する。
		 */

		hash_key = newlockrec->uri;
		if ((lockrec = apr_hash_get(lp_p->token_list_hash, hash_key,
						APR_HASH_KEY_STRING)) == NULL) {
			apr_hash_set(lp_p->token_list_hash, hash_key,
					APR_HASH_KEY_STRING, newlockrec);
		}
		else {
			newlockrec->next = lockrec;
			lockrec          = newlockrec;
			apr_hash_set(lp_p->token_list_hash, hash_key,
						APR_HASH_KEY_STRING, lockrec);
		}
	}

	dbconn->commit(dbconn);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;

}

/**
 * 指定されたロックトークンの時間を更新する。
 * @param r request_rec *
 * @param locktoken dav_lcoktoken
 * @param new_time time_t
 * @return 結果ステータス(0: 正常 / 1: 失敗)
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_locktoken(request_rec *r,
						dav_locktoken *locktoken,
			       			time_t new_time) 
{

	DbConn		*dbconn;
	DbPreparedStmt  *stmt;
	char		*sqlstmt;
	apr_pool_t	*p = r->pool;

	TRACE(p);

	if (locktoken == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"LockToken is Null.");
		return 1;
	}

	/* 指定されたロックトークンのレコードを更新するSQL */
	sqlstmt = "UPDATE "
		  " dav_lock "
		  " SET "
		  " lk_timeout_bi = ?"
		  " WHERE "
		  " lk_locktoken_c = ?";

	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.");
		return 1;
	}

	dbconn->startTrans(dbconn, 0);
	stmt = dbconn->prepareStatement(dbconn, sqlstmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (locktoken = %s)"
			"Reason: %s" , locktoken->uuid, stmt->getMsg(stmt));
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	        dbconn->rollback(dbconn);
	        return 1;
	}

	stmt->setBigInt(stmt, 1, new_time);
	stmt->setString(stmt, 2, locktoken->uuid);

	(void)stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to UPDATE locktoken. (locktoken = %s)"
			"Reason: %s",
			locktoken->uuid, stmt->getMsg(stmt));
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	        dbconn->rollback(dbconn);
	        return 1;
	}

	dbconn->commit(dbconn);
	if (stmt != NULL) stmt->close(stmt);

	return 0;
}

/**
 * NULLロックされているリソースを通常のロックに変更する。
 *
 * @param r request_rec *
 * @param lockrec divy_rdbo_lock *
 * @return 処理ステータス (0: 正常 / 1:失敗)
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_locknull_state(request_rec *r,
						divy_rdbo_lock *lockrec) 
{

	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	char            *sqlstmt = NULL;
	apr_pool_t      *p       = r->pool;
	const char 	*uri     = lockrec->uri;

	TRACE(p);

	if (IS_EMPTY(uri)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to Delete NULLResource. URI is NULL.");
		return 1;
	}

	sqlstmt = "UPDATE "
		  " dav_lock "
		  " SET "
		  " lk_isnull_i = 0"	/* NOT NULL	*/
		  " WHERE "
		  " lk_uri_txt  = ?"
		  " AND "
		  " lk_isnull_i = 1";	/* IS NULL 	*/

	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.");
		return 1;
	}

	dbconn->startTrans(dbconn, 0);
	stmt = dbconn->prepareStatement(dbconn, sqlstmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (locktoken = %s)"
			"Reason: %s" , REPLACE_NULL(uri), stmt->getMsg(stmt));
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		dbconn->rollback(dbconn);
		return 1;
	}

	stmt->setString(stmt, 1, uri);
	(void)stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to Update. NULL LOCK was not able "
			"to be changed into LOCK. (uri = %s)"
		        "Reason: %s",
			REPLACE_NULL(uri), stmt->getMsg(stmt));
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	        dbconn->rollback(dbconn);
		return 1;
	}

	dbconn->commit(dbconn);
	if (stmt != NULL) stmt->close(stmt);

	return 0;
}
/**
 *
 * 指定されたロックトークンを削除する。
 * @param r request_rec
 * @param locktoken dav_locktoken
 * @return int 0: 正常 / 1:失敗
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_locktoken(request_rec *r,
						dav_locktoken *locktoken)
{

	DbConn		*dbconn  = NULL;
	DbPreparedStmt	*stmt    = NULL;
	char		*sqlstmt = NULL;
	apr_pool_t	*p       = r->pool;

	TRACE(p);

	if (locktoken == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"LockToken is Null.");
		return 1;
	}

	/* 指定されたロックトークンのレコードを削除するSQL */
	sqlstmt = "DELETE FROM "
		  " dav_lock "
		  " WHERE "
		  " lk_locktoken_c = ?";

	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.");
		return 1;
	}

	dbconn->startTrans(dbconn, 0);
	stmt = dbconn->prepareStatement(dbconn, sqlstmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (locktoken = %s)"
			"Reason: %s" , REPLACE_NULL(locktoken->uuid),
		       	stmt->getMsg(stmt));
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	        dbconn->rollback(dbconn);
	        return 1;
	}

	stmt->setString(stmt, 1, locktoken->uuid);

	/* 削除件数が0になる可能性もあるが削除したことなのでくよくよしない */
	(void)stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete locktoken. (locktoken = %s)"
			"Reason: %s",
			REPLACE_NULL(locktoken->uuid), stmt->getMsg(stmt));
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		dbconn->rollback(dbconn);
		return 1;
	}

	dbconn->commit(dbconn);
	if (stmt != NULL) stmt->close(stmt);

	return 0;
}

/**
 * 指定されたURIをレコードに追加する。
 * @param r request_rec *
 * @param lockrec divy_rdbo_lock *
 * @return 0: 正常 / 1: 失敗
 *
 */
DIVY_DECLARE(int) divy_rdbo_lock_insert_record(request_rec *r,
						divy_rdbo_lock *lockrec)
{
	DbConn		*dbconn  = NULL;
	DbPreparedStmt 	*stmt    = NULL, *stmt1 = NULL, *stmt2 = NULL;
	apr_pool_t	*p       = r->pool;

	TRACE(p);

	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.");
		return 1;
	}

	dbconn->startTrans(dbconn, 0);
	for (; lockrec != NULL; lockrec = lockrec->next) {

		if (lockrec->scope == 0) {
			if (stmt1 == NULL) {
				stmt1 = dbconn->prepareStatement(dbconn,
					"INSERT INTO dav_lock "
					"(lk_uri_txt,"
					"lk_locktoken_c,"
					"lk_lockkbn_i,"
					"lk_scope_i,"
					"lk_depth_i,"
					"lk_timeout_bi,"
					"lk_owner_vc,"
					"lk_lockkey_txt,"
					"lk_auth_usr_id_vc,"
					"lk_isnull_i,"
					"lk_lock_depth_i)"
					 " VALUES "
					 "(?, ?, ?, NULL, ?, ?, ?, ?, ?, ?, ?)", p);
				if (stmt1->getCode(stmt1) != DB_SUCCESS) {
					ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. (uri = %s)"
						"Reason: %s", REPLACE_NULL(lockrec->uri),
						stmt1->getMsg(stmt1));
					if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
					if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
					dbconn->rollback(dbconn);
					return 1;
				}
			}
			stmt = stmt1;
		}
		else {
			if (stmt2 == NULL) {
				stmt2 = dbconn->prepareStatement(dbconn,
					"INSERT INTO dav_lock "
					"(lk_uri_txt,"
					"lk_locktoken_c,"
					"lk_lockkbn_i,"
					"lk_scope_i,"
					"lk_depth_i,"
					"lk_timeout_bi,"
					"lk_owner_vc,"
					"lk_lockkey_txt,"
					"lk_auth_usr_id_vc,"
					"lk_isnull_i,"
					"lk_lock_depth_i)"
					 " VALUES "
					 "(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)", p);
				if (stmt2->getCode(stmt2) != DB_SUCCESS) {
					ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. (uri = %s)"
						"Reason: %s", REPLACE_NULL(lockrec->uri),
						stmt2->getMsg(stmt2));
					if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
					if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
					dbconn->rollback(dbconn);
					return 1;
				}
			}
			stmt = stmt2;
		}

		stmt->setString(stmt, 1, lockrec->uri);
		stmt->setString(stmt, 2, lockrec->locktoken->uuid);
		stmt->setInt(stmt,    3, lockrec->kbn);
		if (lockrec->scope == 0) {
			stmt->setInt(stmt,    4, lockrec->depth);
			stmt->setBigInt(stmt, 5, lockrec->timeout);
			stmt->setString(stmt, 6, lockrec->owner);
			stmt->setString(stmt, 7, lockrec->key);
			stmt->setString(stmt, 8, lockrec->userid);
			stmt->setInt(stmt,    9, lockrec->is_null);
			stmt->setInt(stmt,   10, lockrec->lockdepth);
		}
		else {
			stmt->setInt(stmt,    4, lockrec->scope);
			stmt->setInt(stmt,    5, lockrec->depth);
			stmt->setBigInt(stmt, 6, lockrec->timeout);
			stmt->setString(stmt, 7, lockrec->owner);
			stmt->setString(stmt, 8, lockrec->key);
			stmt->setString(stmt, 9, lockrec->userid);
			stmt->setInt(stmt,   10, lockrec->is_null);
			stmt->setInt(stmt,   11, lockrec->lockdepth);
		}

		(void)stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Faild to Insert uri. (uri = %s)"
				"Reason: %s", REPLACE_NULL(lockrec->uri),
			       	stmt->getMsg(stmt));
			dbconn->rollback(dbconn);
			/* (note) stmt はリファレンスなのでcloseしません */
			if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
			if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
			return 1;
		}
	}
	/* (note) stmt はリファレンスなのでcloseしません */
	if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
	if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;

	dbconn->commit(dbconn);

	return 0;
}

/**
 * ロックテーブルを検索してnullresource(null lock)を返却する。
 * 
 * @param r        	request_rec *		リクエスト
 * @param uri	   	const char *		調べるURI
 * @param depth    	int			調べる階層(0 | 1 | INFINITY)
 * @param is_collection int			collectionか否か
 * @param resource 	divy_rdbo_resource **	nullresourceのリスト
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_null_resource_member(request_rec *r,
	       			    		const char *uri, int depth,
				    		int is_collection,
				    		divy_rdbo_resource **resource)
{

	DbConn		   *dbconn  = NULL;
	DbPreparedStmt	   *stmt    = NULL;
	DbResultSet	   *rset    = NULL;
	const char	   *sqlstmt = NULL;
	divy_rdbo_resource *tmp_res = NULL;
	apr_pool_t	   *p       = r->pool;
	int		   dircnt   = 0;
	const char	   *key     = NULL;

	*resource = NULL;

	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.");
		return 1;
	}

	dbconn->startTrans(dbconn, 0);

	/* NULLだけを検索するSQLを作成する。 */
	sqlstmt = apr_pstrdup(p, "SELECT"
				 " lk_uri_txt "
				 " FROM "
				 " dav_lock "
				 " WHERE "
				 " lk_isnull_i = 1 "
				 " AND ");

	key = dav_divy_remove_endslash(p ,uri);

	/* uriがcollectionの場合URIの最後に/を付加する */
	if (is_collection) {
		key = dav_divy_append_endslash(p, uri);
	}

	/* depth によって検索するSQLを作成する */
	if (depth == DAV_INFINITY) {
		sqlstmt = apr_pstrcat(p, sqlstmt, "lk_uri_txt LIKE ? "
					DIVY_DBFUNC_ESCAPE_CLAUSE, NULL);
		stmt = dbconn->prepareStatement(dbconn, sqlstmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt."
				"Reason: %s", stmt->getMsg(stmt));
			dbconn->rollback(dbconn);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
		stmt->setString(stmt, 1, apr_pstrcat(p,
				stmt->escWildCard(stmt, key), "%", NULL));
	}
	else if (depth > 0) {
		/* 整形しなおしたkeyではなく引数のuriでカウントをとる。*/
		dircnt  = divy_count_dirs(uri);
		sqlstmt = apr_pstrcat(p, sqlstmt,
					 "lk_uri_txt LIKE ? "
					 DIVY_DBFUNC_ESCAPE_CLAUSE
					 " AND "
					 " lk_depth_i IN (?, ?)"
					 , NULL);

		stmt = dbconn->prepareStatement(dbconn, sqlstmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt."
				"Reason: %s", stmt->getMsg(stmt));
			dbconn->rollback(dbconn);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
		stmt->setString(stmt, 1, apr_pstrcat(p,
				stmt->escWildCard(stmt, key), "%", NULL));
		stmt->setInt(stmt,    2, dircnt);
		stmt->setInt(stmt,    3, dircnt + 1);
	}
	else {
		sqlstmt = apr_pstrcat(p, sqlstmt, "lk_uri_txt = ?", NULL);
		stmt = dbconn->prepareStatement(dbconn, sqlstmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt."
				"Reason: %s", stmt->getMsg(stmt));
			dbconn->rollback(dbconn);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
		stmt->setString(stmt, 1, key);
	}

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet."
			"Reason: %s", rset->getMsg(rset));
		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {
		 *resource        = apr_pcalloc(p, sizeof(divy_rdbo_resource));
		(*resource)->uri  = rset->getString(rset, 1);
		(*resource)->next = tmp_res;
		tmp_res           = *resource;

	}

	/* 結果を確定する */
	dbconn->commit(dbconn);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたメール監視URIに存在するユーザのメールアドレス取得する。
 *
 */
DIVY_DECLARE(divy_rdbo_usr *) divy_rdbo_ml_get_addr(request_rec *r,
							const char *uri,
							int method)
{
	DbConn		*dbconn   = NULL;
	DbPreparedStmt	*stmt = NULL;
	DbResultSet	*rset     = NULL;
	apr_pool_t	*p        = r->pool;

	divy_rdbo_usr	*usr	 = NULL;
	divy_rdbo_usr   *tmp_usr = NULL;
	char            *mw_uri  = NULL;
	int m_search             = ap_method_number_of("SEARCH");
	int support_extstatus    = divy_support_extenduserstatus(r);
	int idx;

	divy_uri_spec	*u_spec	 = NULL;
	divy_sbuf *sql_buf = NULL;
	divy_db_transaction_ctx *ts_ctx  = NULL;

	TRACE(p);

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"URI is NULL or space.");
		return NULL;
	}

	/* メール監視フラグが設定されているリソースのuriを切り出す */
	if (divy_extract_mailwatch_uri(p, dav_divy_get_root_uri(r),
							uri, &mw_uri)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse uri for mailwatch uri."
			"(uri = %s)", uri);
		return NULL;
	}
	else if (mw_uri == NULL) {
		/* mailwatch が存在し得ないコレクションだった場合 */
		return NULL;
	}

	/* 切り出し後のURI をパースする
	 * (トップURIのフォルダタイプを取得) */
	if (divy_parse_uri(p, dav_divy_get_root_uri(r), mw_uri, &u_spec)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Parse error from uri. (uri = %s)", mw_uri);
		return NULL;
	}

	/* SQL文の組み立て */
	divy_sbuf_create(p, &sql_buf, 1024);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT "
				 " u.usr_usr_id_vc"
				 ",u.usr_fullname_vc"
				 ",u.usr_mailaddr_vc");

	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
				",u.usr_expiration_bi"
				",u.usr_extended_status_c");
	}

	divy_sbuf_append(sql_buf,
				 " FROM divy_usr u "
			  	 " INNER JOIN divy_mailwatch m"
				 " ON u.usr_usr_id_vc = m.mw_usr_id_vc");

	/* 
	 * アクセス権限により条件を追加する
	 */
	/* トップがGROUPフォルダだった場合 */
	if (u_spec->infotype == DIVY_INFOTYPE_group_e){
		/* 
		 * (note)
		 * usad_deny_xxxx はLEFTJOIN しているため
		 * <> 1 ではなく = 0 or IS NULL の条件を使用して下さい
		 */
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usraccessdeny uad"
				" ON u.usr_usr_id_vc = uad.usad_usr_id_vc"
				" WHERE m.mw_rs_uri_txt = ?"
				" AND "
				" (uad.usad_deny_group = 0"
				" OR uad.usad_deny_group IS NULL)");
	}
	/* トップが検索結果フォルダ・共有コレクションフォルダの場合 */
	else if (u_spec->infotype == DIVY_INFOTYPE_dbfolder_e ||
			 u_spec->infotype == DIVY_INFOTYPE_dbshfolder_e){
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usraccessdeny uad"
				" ON u.usr_usr_id_vc = uad.usad_usr_id_vc"
				" WHERE m.mw_rs_uri_txt = ?"
				" AND "
				" (uad.usad_deny_dblink = 0"
				" OR uad.usad_deny_dblink IS NULL)");
	} 
	/* 上記以外(権限による条件はなし) */
	else {
		divy_sbuf_append(sql_buf,
				" WHERE m.mw_rs_uri_txt = ?");
	}

	/* method に該当する where 句条件を生成する */
	if (method == M_GET) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_get = ?");
	}
	else if (method == M_PUT) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_put = ?");
	}
	else if (method == M_POST) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_post = ?");
	}
	else if (method == M_DELETE) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_delete = ?");
	}
	else if (method == M_OPTIONS) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_options = ?");
	}
	else if (method == M_PROPFIND) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_propfind = ?");
	}
	else if (method == M_PROPPATCH) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_proppatch = ?");
	}
	else if (method == M_MKCOL) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_mkcol = ?");
	}
	else if (method == M_COPY) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_copy = ?");
	}
	else if (method == M_MOVE) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_move = ?");
	}
	else if (method == M_LOCK) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_lock = ?");
	}
	else if (method == M_UNLOCK) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_unlock = ?");
	}
	else if (method == m_search) {
		divy_sbuf_append(sql_buf, " AND mw_trigger_search = ?");
	}
	else {
		/* 上記以外のメソッドはNG */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Invalid method found (method num = %d)", method);
		return NULL;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return NULL;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return NULL;
	dbconn = ts_ctx->dbconn;

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt.(Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return NULL;
	}

	stmt->setString(stmt, 1, mw_uri);
	stmt->setInt(stmt, 2, 1);	/* method フラグが立っていること */

	/* SQLを実行する */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet."
				"(uri = %s, mailwatch uri = %s) Reason: %s",
				uri, mw_uri, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return NULL;
	}

	while (rset->next(rset) == DB_TRUE) {
		tmp_usr            = apr_pcalloc(p, sizeof(divy_rdbo_usr));

		idx = 1;
		tmp_usr->usrid     = rset->getString(rset, idx++);
		tmp_usr->fullname  = rset->getString(rset, idx++);
		tmp_usr->mailaddr  = rset->getString(rset, idx++);

		if (support_extstatus) {
			tmp_usr->expiration	= (time_t) rset->getBigInt(rset, idx++);
			tmp_usr->extstatus	= divy_rdbo_parse_extstatus(p, rset->getString(rset, idx++), EXTSTATUS_TYPE_USR);

			/* 非アクティブユーザはリストに追加しない */
			if (!divy_rdbo_is_active_user(tmp_usr->extstatus)) {
				continue;
			}

			/* 期限切れユーザもリストに追加しない */
			if (tmp_usr->expiration != 0 && tmp_usr->expiration < apr_time_sec(r->request_time)) {
				continue;
			}
		}
		tmp_usr->next      = usr;
		usr                = tmp_usr;
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return usr;
}

/**
 * divy_userテーブルのエントリ数(現在のユーザ総数)を返却する。
 * 
 */
DIVY_DECLARE(int) divy_rdbo_count_usr(request_rec *r, int *usr_cnt)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;
	int sysexec_cnt = 0;
	divy_rdbo_extstatus *extstatus = NULL;
	char *extstatus_wcstr   = NULL;
	char *extstatus_wcstr2  = NULL;

	*usr_cnt = 0;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* 全ユーザ数を取得する */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT COUNT(usr_usr_id_vc) "
				"FROM divy_usr", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to SELECT divy_usr. Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 結果の取得 */
	if (rset->next(rset) == DB_TRUE) {
		*usr_cnt = rset->getInt(rset, 1);
	}
	else {
		/* これはまずありえないが念のため */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The number of user account is zero. Please check divy_usr table");
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	if (divy_support_extenduserstatus(r)) {
		/*
		 * システム実行権限ユーザ数を調べる
		 */
		extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_USR);	/* デフォルト値で生成 */
		/* システム実行権限ステータス以外を全て非アクティブにする */
		divy_rdbo_reset_extstatus(extstatus, EXTSTATUS_TYPE_USR);
		divy_rdbo_set_sysexec_privilege(extstatus, 1);	/* システム実行権限だけON */

		/* 検索文字列パターンを生成する */
		extstatus_wcstr = divy_rdbo_extstatus2wildcardstr(extstatus, EXTSTATUS_TYPE_USR);

		divy_rdbo_reset_extstatus(extstatus, EXTSTATUS_TYPE_USR);
		divy_rdbo_set_active_user(extstatus, 1);	/* 非アクティブ */

		extstatus_wcstr2 = divy_rdbo_extstatus2wildcardstr(extstatus, EXTSTATUS_TYPE_USR);

		stmt = dbconn->prepareStatement(dbconn,
				"SELECT COUNT(usr_usr_id_vc) "
				"FROM divy_usr "
				"WHERE usr_extended_status_c LIKE ? "
				"OR usr_extended_status_c NOT LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL)   stmt->close(stmt);
			return 1;
		}
		stmt->setString(stmt, 1, extstatus_wcstr);
		stmt->setString(stmt, 2, extstatus_wcstr2);

		/* SQL実行 失敗時はエラー */
		rset = stmt->executeQuery(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to SELECT divy_usr. Reason: %s", rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL)   rset->close(rset); rset = NULL;
			if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* 結果の取得 */
		sysexec_cnt = 0;
		if (rset->next(rset) == DB_TRUE) {
			sysexec_cnt = rset->getInt(rset, 1);
		}
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* システム実行権限ユーザはライセンスカウントに含めてはならない -> 減算する */
		if (sysexec_cnt > 0) {
			*usr_cnt -= sysexec_cnt;
		}
	}

	divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたuri, usrid の メール監視情報を取得して返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_mailwatch_property(request_rec *r,
					const char *uri, const char *usrid,
					divy_rdbo_mailwatch **mwatch_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	divy_rdbo_mailwatch *mw = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int m_search            = ap_method_number_of("SEARCH");
	char *sql               = NULL;
	char *sqlstr            = "SELECT mw_rs_uri_txt"
					", mw_usr_id_vc"
					", mw_trigger_get"
					", mw_trigger_put"
					", mw_trigger_post"
					", mw_trigger_delete"
					", mw_trigger_options"
					", mw_trigger_propfind"
					", mw_trigger_proppatch"
					", mw_trigger_mkcol"
					", mw_trigger_copy"
					", mw_trigger_move"
					", mw_trigger_lock"
					", mw_trigger_unlock"
					", mw_trigger_search"
					", mw_l_del_flag"
#ifdef DIVY_SUPPORT_MESSENGER
					", mw_notification_i"
#endif	/* DIVY_SUPPORT_MESSENGER */
					" FROM divy_mailwatch"
					" WHERE mw_rs_uri_txt = ?";
	char *mwtopuri		= NULL;

	*mwatch_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"URI is empty !!");
		return 1;
	}

	/* usrid があればwhere句追加 */
	if (IS_FILLED(usrid)) {
		sql = apr_pstrcat(p, sqlstr, " AND mw_usr_id_vc = ?", NULL);
	}
	else {
		sql = apr_pstrdup(p, sqlstr);
	}

	/* 
	 * uriから先頭を切り出しトップ階層のURIを作成する 
	 */
	if (divy_extract_mailwatch_uri(p, dav_divy_get_root_uri(r),
							uri, &mwtopuri)) {
		return 1;
	}
	else if (mwtopuri == NULL) {
		return 0;	/* mailwatch が存在しない場合 */
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* sql文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt."
				"(uri = %s, usrid = %s) Reason: %s", 
				uri, usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, mwtopuri);	/* uri ではなくmwtopuriをバインド */
	if (IS_FILLED(usrid)) {
		stmt->setString(stmt, 2, usrid);
	}
	/* sql実行　失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet."
				"(mw_topuri = %s, usrid = %s) Reason: %s",
				mwtopuri, usrid, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}
	/* データがあれば取得(ない場合は初期化時のNULL) */
//	if (rset->next(rset) == DB_TRUE) {
	while (rset->next(rset) == DB_TRUE) {
		/* 領域確保,methods初期化 */
		if (*mwatch_pr == NULL) {
			*mwatch_pr = mw =  apr_pcalloc(p, sizeof(divy_rdbo_mailwatch));
		}
		else {
			mw->next = apr_pcalloc(p, sizeof(divy_rdbo_mailwatch));
			mw       = mw->next;
		}

		memset(mw->trigger_methods, 0, METHODS);
	        
		/* データ格納 */
		mw->uri 	= rset->getString(rset, 1);
		mw->usrid 	= rset->getString(rset, 2);
		mw->trigger_methods[M_GET]       = rset->getInt(rset, 3);
		mw->trigger_methods[M_PUT]       = rset->getInt(rset, 4);
		mw->trigger_methods[M_POST]      = rset->getInt(rset, 5);
		mw->trigger_methods[M_DELETE]    = rset->getInt(rset, 6);
		mw->trigger_methods[M_OPTIONS]   = rset->getInt(rset, 7);
		mw->trigger_methods[M_PROPFIND]  = rset->getInt(rset, 8);
		mw->trigger_methods[M_PROPPATCH] = rset->getInt(rset, 9);
		mw->trigger_methods[M_MKCOL]     = rset->getInt(rset,10);
		mw->trigger_methods[M_COPY]      = rset->getInt(rset,11);
		mw->trigger_methods[M_MOVE]      = rset->getInt(rset,12);
		mw->trigger_methods[M_LOCK]      = rset->getInt(rset,13);
		mw->trigger_methods[M_UNLOCK]    = rset->getInt(rset,14);
		mw->trigger_methods[m_search]    = rset->getInt(rset,15);
		mw->delflag  = rset->getInt(rset, 16);
#ifdef DIVY_SUPPORT_MESSENGER
		mw->notification = rset->getInt(rset, 17);
#else	/* !DIVY_SUPPORT_MESSENGER */
		mw->notification = DIVY_NOTICE_MAIL;
#endif	/* DIVY_SUPPORT_MESSENGER */

		/* 次のmwatch_prはNULL */
		mw->next 	= NULL;

#if 0
		(*mwatch_pr)->uri 	= rset->getString(rset, 1);
		(*mwatch_pr)->usrid 	= rset->getString(rset, 2);
		(*mwatch_pr)->trigger_methods[M_GET]       = rset->getInt(rset, 3);
		(*mwatch_pr)->trigger_methods[M_PUT]       = rset->getInt(rset, 4);
		(*mwatch_pr)->trigger_methods[M_POST]      = rset->getInt(rset, 5);
		(*mwatch_pr)->trigger_methods[M_DELETE]    = rset->getInt(rset, 6);
		(*mwatch_pr)->trigger_methods[M_OPTIONS]   = rset->getInt(rset, 7);
		(*mwatch_pr)->trigger_methods[M_PROPFIND]  = rset->getInt(rset, 8);
		(*mwatch_pr)->trigger_methods[M_PROPPATCH] = rset->getInt(rset, 9);
		(*mwatch_pr)->trigger_methods[M_MKCOL]     = rset->getInt(rset,10);
		(*mwatch_pr)->trigger_methods[M_COPY]      = rset->getInt(rset,11);
		(*mwatch_pr)->trigger_methods[M_MOVE]      = rset->getInt(rset,12);
		(*mwatch_pr)->trigger_methods[M_LOCK]      = rset->getInt(rset,13);
		(*mwatch_pr)->trigger_methods[M_UNLOCK]    = rset->getInt(rset,14);
		(*mwatch_pr)->trigger_methods[m_search]    = rset->getInt(rset,15);
		(*mwatch_pr)->delflag  = rset->getInt(rset, 16);
#ifdef DIVY_SUPPORT_MESSENGER
		(*mwatch_pr)->notification = rset->getInt(rset, 17);
#else	/* !DIVY_SUPPORT_MESSENGER */
		(*mwatch_pr)->notification = DIVY_NOTICE_MAIL;
#endif	/* DIVY_SUPPORT_MESSENGER */

		/* 次のmwatch_prはNULL */
		(*mwatch_pr)->next 	= NULL;
#endif
	}

	/* クローズ */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたmwatch_pr->uri が示すコレクションにmailwatch_pr->trigger_methods 
 * が示すメール監視状態を追加する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_mailwatch_property(request_rec *r,
					const divy_rdbo_mailwatch *mwatch_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	return _insert_mailwatch_property(r, mwatch_pr, ts_ctx);
}

/**
 * 指定されたmwatch_pr->uri が示すコレクションにmailwatch_pr->trigger_methods 
 * が示すメール監視状態を追加する。
 * 既に、mwatch_pr->uri のエントリがdivy_mailwatch に存在するものとします。
 * 純粋にmailwatch_pr->trigger_methods を更新するだけとします。
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_mailwatch_property(request_rec *r,
					const divy_rdbo_mailwatch *mwatch_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	return _update_mailwatch_property(r, mwatch_pr, ts_ctx);
}

/**
 * 指定されたmwatch_pr->uri が示すコレクションとその下位階層からmwatch->usrid 
 * が持つ全てのメール監視状態を削除する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_mailwatch_property(request_rec *r,
					const divy_rdbo_mailwatch *mwatch_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	int delmode = 0;

	if (mwatch_pr->delflag == 1) {
		delmode = DIVY_REMWATCH_LOGICAL;	/* 論理削除 */
	}
	else {
		delmode = DIVY_REMWATCH_NORMAL; 	/* 物理削除 */
	}

	if (IS_EMPTY(mwatch_pr->usrid)) {
		delmode |= DIVY_REMWATCH_IGNORE_USER;
	}
	else if (IS_EMPTY(mwatch_pr->uri)) {
		delmode |= DIVY_REMWATCH_IGNORE_URI;
	}

	/* 削除の実施 */
	return _remove_mailwatch_property(r, mwatch_pr->uri, mwatch_pr->usrid,
						delmode, ts_ctx);
}

/**
 * グループについているグループメール監視状態を全て取得し、
 * 相対URIをキーとしてこの状態をハッシュの値として返却する.
 * (note)
 *   トランザクションが継続中であれば、それを引き継ぎます。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_groupmailwatch_properties(request_rec *r,
							apr_hash_t **gmw_h, divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int m_search            = ap_method_number_of("SEARCH");
	char *sql               =
					"SELECT mw_rs_uri_txt"
					", mw_usr_id_vc"
					", mw_trigger_get"
					", mw_trigger_put"
					", mw_trigger_post"
					", mw_trigger_delete"
					", mw_trigger_options"
					", mw_trigger_propfind"
					", mw_trigger_proppatch"
					", mw_trigger_mkcol"
					", mw_trigger_copy"
					", mw_trigger_move"
					", mw_trigger_lock"
					", mw_trigger_unlock"
					", mw_trigger_search"
					", mw_l_del_flag"
#ifdef DIVY_SUPPORT_MESSENGER
					", mw_notification_i"
#endif	/* DIVY_SUPPORT_MESSENGER */
					" FROM divy_mailwatch"
					" WHERE mw_rs_uri_txt LIKE ? "
					DIVY_DBFUNC_ESCAPE_CLAUSE;
	char *mw_uri, *key;
	int len;
	divy_rdbo_mailwatch *mw = NULL;

	*gmw_h = NULL;	/* 初期化 */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* sql文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	mw_uri = divy_build_m_group_uri(p, dav_divy_get_root_uri(r), NULL);
	len    = strlen(mw_uri);

	/* バインド */
	stmt->setString(stmt, 1, apr_pstrcat(p,
				stmt->escWildCard(stmt, mw_uri), "/%", NULL));

	/* sql実行　失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* データを取得 */
	while (rset->next(rset) == DB_TRUE) {

		/* 領域確保,methods初期化 */
		mw = apr_pcalloc(p, sizeof(divy_rdbo_mailwatch));
		memset(mw->trigger_methods, 0, METHODS);
	        
		/* データ格納 */
		mw->uri                          = rset->getString(rset, 1);
		mw->usrid                        = rset->getString(rset, 2);
		mw->trigger_methods[M_GET]       = rset->getInt(rset, 3);
		mw->trigger_methods[M_PUT]       = rset->getInt(rset, 4);
		mw->trigger_methods[M_POST]      = rset->getInt(rset, 5);
		mw->trigger_methods[M_DELETE]    = rset->getInt(rset, 6);
		mw->trigger_methods[M_OPTIONS]   = rset->getInt(rset, 7);
		mw->trigger_methods[M_PROPFIND]  = rset->getInt(rset, 8);
		mw->trigger_methods[M_PROPPATCH] = rset->getInt(rset, 9);
		mw->trigger_methods[M_MKCOL]     = rset->getInt(rset,10);
		mw->trigger_methods[M_COPY]      = rset->getInt(rset,11);
		mw->trigger_methods[M_MOVE]      = rset->getInt(rset,12);
		mw->trigger_methods[M_LOCK]      = rset->getInt(rset,13);
		mw->trigger_methods[M_UNLOCK]    = rset->getInt(rset,14);
		mw->trigger_methods[m_search]    = rset->getInt(rset,15);
		mw->delflag                      = rset->getInt(rset, 16);
#ifdef DIVY_SUPPORT_MESSENGER
		mw->notification                 = rset->getInt(rset, 17);
#else	/* !DIVY_SUPPORT_MESSENGER */
		mw->notification                 = DIVY_NOTICE_MAIL;
#endif	/* DIVY_SUPPORT_MESSENGER */

		mw->next = NULL;

		if (*gmw_h == NULL) {
			*gmw_h = apr_hash_make(p);
		}

		/* ハッシュに値を入れる */
		key = (len <= strlen(mw->uri)) ? &mw->uri[len] : mw->uri;
		apr_hash_set(*gmw_h, key, APR_HASH_KEY_STRING, mw);
	}

	/* クローズ */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたgrpid のグループが保持するデフォルトメール監視フラグ
 * (groupmailwatch) を所属ユーザusrid に付加する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_copy_groupmailwatch_to_user(request_rec *r,
							const char *grpid,
							const char *usrid,
							int copymode,
							divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn   = NULL;
	DbPreparedStmt  *stmt     = NULL;
	DbResultSet     *rset     = NULL;
	apr_pool_t *p             = r->pool;
	int iscommit              = 0;
	int support_extstatus     = divy_support_extenduserstatus(r);
	char *grpcol_uri, *grp_full_uri;
	char *sql, *extstatus_wcstr = NULL, *sql2;
	char *mw_sql_stmt         =
			"INSERT INTO divy_mailwatch ("
			"mw_rs_uri_txt,"
			"mw_usr_id_vc,"
			"mw_trigger_get,"
			"mw_trigger_put,"
			"mw_trigger_post,"
			"mw_trigger_delete,"
			"mw_trigger_options,"
			"mw_trigger_propfind,"
			"mw_trigger_proppatch,"
			"mw_trigger_mkcol,"
			"mw_trigger_copy,"
			"mw_trigger_move,"
			"mw_trigger_lock,"
			"mw_trigger_unlock,"
			"mw_trigger_search,"
			"mw_l_del_flag) "
			"(SELECT "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))","
			"gmem.grpm_usr_id_vc,"
			"gmw.mw_trigger_get,"
			"gmw.mw_trigger_put,"
			"gmw.mw_trigger_post,"
			"gmw.mw_trigger_delete,"
			"gmw.mw_trigger_options,"
			"gmw.mw_trigger_propfind,"
			"gmw.mw_trigger_proppatch,"
			"gmw.mw_trigger_mkcol,"
			"gmw.mw_trigger_copy,"
			"gmw.mw_trigger_move,"
			"gmw.mw_trigger_lock,"
			"gmw.mw_trigger_unlock,"
			"gmw.mw_trigger_search,"
			"gmw.mw_l_del_flag "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
			"FROM divy_grpmem gmem LEFT JOIN"
			" (SELECT mw_trigger_get,"
			" mw_trigger_put,"
			" mw_trigger_post,"
			" mw_trigger_delete,"
			" mw_trigger_options,"
			" mw_trigger_propfind,"
			" mw_trigger_proppatch,"
			" mw_trigger_mkcol,"
			" mw_trigger_copy,"
			" mw_trigger_move,"
			" mw_trigger_lock,"
			" mw_trigger_unlock,"
			" mw_trigger_search,"
			" mw_l_del_flag"
			" FROM divy_mailwatch"
			" WHERE mw_rs_uri_txt = ?) gmw"
			" ON 1=1 ";
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
			"FROM divy_grpmem gmem,"
			" (SELECT mw_trigger_get,"
			" mw_trigger_put,"
			" mw_trigger_post,"
			" mw_trigger_delete,"
			" mw_trigger_options,"
			" mw_trigger_propfind,"
			" mw_trigger_proppatch,"
			" mw_trigger_mkcol,"
			" mw_trigger_copy,"
			" mw_trigger_move,"
			" mw_trigger_lock,"
			" mw_trigger_unlock,"
			" mw_trigger_search,"
			" mw_l_del_flag"
			" FROM divy_mailwatch"
			" WHERE mw_rs_uri_txt = ?) gmw"
			" WHERE 1=1";
#endif
	TRACE(p);

        /* 必須項目のチェック */
	if (copymode == DIVY_CPMWATCH_NORMAL && (IS_EMPTY(grpid) || IS_EMPTY(usrid))) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"grpid or usrid is EMPTY");
		return 1;
	}
	else if ((copymode & DIVY_CPMWATCH_IGNORE_USER) && IS_EMPTY(grpid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"grpid is EMPTY");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * グループコレクションの絶対URI、グループの絶対URIを生成
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"SELECT"
			" r.rs_uri_txt,"
			" g.grp_relative_uri_txt "
			"FROM divy_grp g "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
			"INNER JOIN dav_resource r"
			" ON g.grp_rs_id_c = r.rs_rs_id_c "
			"WHERE "
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
			",dav_resource r "
			"WHERE g.grp_rs_id_c = r.rs_rs_id_c "
			"AND "
#endif
			"g.grp_grp_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (grpid = %s) "
			"Reason: %s", grpid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, grpid);

	/* SQL 実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_grp and dav_resource.(grpid = %s) "
			"Reason: %s", grpid, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		char *relativeuri;

		grpcol_uri   = rset->getString(rset, 1);
		relativeuri  = rset->getString(rset, 2);
		grp_full_uri = divy_build_m_group_uri(p, dav_divy_get_root_uri(r), relativeuri);
	}
	else {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get a groupuri of group from divy_grp."
			"Maybe this group was deleted before this operation."
			"Please check client cache.(grpid = %s)", grpid);

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 制限ユーザはリストに含めない */
	sql2 = "";
	if (support_extstatus) {
		sql2 =	" AND gmem.grpm_usr_id_vc"
			" IN (SELECT usr_usr_id_vc"
			" FROM divy_usr"
			" WHERE usr_extended_status_c LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE")";
	}


	/* ユーザIDを検索条件に含めない場合 */
	if (copymode & DIVY_CPMWATCH_IGNORE_USER) {
		/* (note)
		 * 	"AND mw_trigger_get IS NOT NULL" はグループのメール監視
		 * 	フラグがdivy_mailwatch になければユーザには追加しない
		 * 	ようにするための細工。(小細工)
		 */
		sql = apr_pstrcat(p, mw_sql_stmt,
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"WHERE gmem.grpm_grp_id_c = ? "
				"AND gmw.mw_trigger_get IS NOT NULL", sql2, ")",
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
				" AND (gmem.grpm_grp_id_c = ?"
				" AND gmw.mw_trigger_get IS NOT NULL", sql2,"))",
#endif
				NULL);
	}
	/* ユーザIDを含める場合 */
	else {
		sql = apr_pstrcat(p, mw_sql_stmt,
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"WHERE gmem.grpm_grp_id_c = ? "
				"AND gmem.grpm_usr_id_vc = ? "
				"AND gmw.mw_trigger_get IS NOT NULL", sql2, ")",
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
				" AND (gmem.grpm_grp_id_c = ?"
				" AND gmem.grpm_usr_id_vc = ?"
				" AND gmw.mw_trigger_get IS NOT NULL", sql2, "))",
#endif
				NULL);
	}

	/*
	 * divy_mailwatch (mailwatch) のコピー
	 */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(grp_uri = %s, usrid = %s) Reason: %s",
			grp_full_uri, usrid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* ユーザ拡張ステータスのワイルドカードパターン文字列を生成する */
	extstatus_wcstr = NULL;
	if (support_extstatus) {
		/* デフォルト値を取得 */
		divy_rdbo_extstatus *extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_USR);

		divy_rdbo_reset_extstatus(extstatus, EXTSTATUS_TYPE_USR);
		divy_rdbo_set_trusty_user(extstatus, 1);	/* ここだけアクティブ化 */

		/* 検索文字列パターンを生成する */
		extstatus_wcstr = divy_rdbo_extstatus2wildcardstr(extstatus, EXTSTATUS_TYPE_USR);
	}

	stmt->setString(stmt, 1, grpcol_uri);
	stmt->setString(stmt, 2, grp_full_uri);
	stmt->setString(stmt, 3, grpid);
	if (!(copymode & DIVY_CPMWATCH_IGNORE_USER)) {
		stmt->setString(stmt, 4, usrid);
		if (IS_FILLED(extstatus_wcstr)) {
			stmt->setString(stmt, 5, extstatus_wcstr);
		}
	}
	else {
		if (IS_FILLED(extstatus_wcstr)) {
			stmt->setString(stmt, 4, extstatus_wcstr);
		}
	}

	/* sql実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_mailwatch for copy groupmailwatch. "
			"(grp_uri = %s, usrid = %s) Reason: %s",
			grp_full_uri, REPLACE_NULL(usrid), stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定された src_grp_uri のグループに所属するユーザの mailwatch 設定を
 * dst_grp_uri の mailwatch 設定としてコピーする。
 *
 */
DIVY_DECLARE(int) divy_rdbo_copy_usermailwatch_to_group(request_rec *r,
							const char *src_grp_uri,
							const char *dst_grp_uri,
							divy_db_transaction_ctx *ts_ctx)
{
	DbConn		*dbconn	= NULL;
	DbPreparedStmt	*stmt	= NULL;
	apr_pool_t 	*p	= r->pool;
	int iscommit		= 0;
	char *sql		=
			"INSERT INTO divy_mailwatch ("
			"mw_rs_uri_txt,"
			" mw_usr_id_vc,"
			" mw_trigger_get,"
			" mw_trigger_put,"
			" mw_trigger_post,"
			" mw_trigger_delete,"
			" mw_trigger_options,"
			" mw_trigger_propfind,"
			" mw_trigger_proppatch,"
			" mw_trigger_mkcol,"
			" mw_trigger_copy,"
			" mw_trigger_move,"
			" mw_trigger_lock,"
			" mw_trigger_unlock,"
			" mw_trigger_search,"
			" mw_l_del_flag) "
			" (SELECT"
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))","
			" mw_usr_id_vc,"
			" mw_trigger_get,"
			" mw_trigger_put,"
			" mw_trigger_post,"
			" mw_trigger_delete,"
			" mw_trigger_options,"
			" mw_trigger_propfind,"
			" mw_trigger_proppatch,"
			" mw_trigger_mkcol,"
			" mw_trigger_copy,"
			" mw_trigger_move,"
			" mw_trigger_lock,"
			" mw_trigger_unlock,"
			" mw_trigger_search,"
			" mw_l_del_flag" 
			" FROM divy_mailwatch"
			" WHERE mw_rs_uri_txt = ?)";
	TRACE(p);

        /* 必須項目のチェック */
	if (IS_EMPTY(src_grp_uri) || IS_EMPTY(dst_grp_uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"There are empty values.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL 文の準備 */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(srcrpuri = %s, dstgrpuri = %s) Reason: %s",
			src_grp_uri, dst_grp_uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, dst_grp_uri);
	stmt->setString(stmt, 2, src_grp_uri);

	/* sql実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_mailwatch for copy "
			"groupmailwatch. (srcgrpuri = %s, dstgrpuri = %s) "
			"Reason: %s",
			src_grp_uri, dst_grp_uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}


/**
 * 指定されたuri のユーザリレーションを取得する
 */
DIVY_DECLARE(int) divy_rdbo_get_rusr(request_rec *r,
					const char *uri,
					divy_rdbo_rusr **rusr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;

	*rusr = NULL;	/* 初期化 */

	/*
	 * uri をパースして、userid とグループURI を取り出す
	 */
	if (divy_rdbo_parse_rusr_uri(r, uri, rusr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse uri. (uri = %s)", uri);
		*rusr = NULL;
		return 1;
	}

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.");
		return 1;
	}

	dbconn->startTrans(dbconn, 0);
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				"grpm_grp_id_c "
				"FROM divy_grpmem "
				"WHERE grpm_usr_id_vc = ? "
				"AND grpm_grp_id_c = "
				"(SELECT grp_grp_id_c FROM divy_grp"
				" WHERE grp_relative_uri_txt = ?)", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.Reason: %s",
			stmt->getMsg(stmt));

		dbconn->rollback(dbconn);
		if (stmt != NULL)  stmt->close(stmt);
		return 1;
	}

	stmt->setString(stmt, 1, (*rusr)->usrid);
	stmt->setString(stmt, 2, (*rusr)->grp_uri);
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_grpmem.(usrid = %s, grp_uri = %s)"
			"Reason: %s", (*rusr)->usrid, (*rusr)->grp_uri,
			rset->getMsg(rset));

		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		(*rusr)->grpid   = rset->getString(rset, 1);
	}
	else {
		*rusr = NULL;	/* 空にする */
	}

	dbconn->commit(dbconn);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたrusr のユーザリレーションを追加し、
 * rusr にrusr->grp_uri のグループが持っていたメール監視状態を設定する。
 * 
 */
DIVY_DECLARE(int) divy_rdbo_insert_rusr(request_rec *r,
			 		const divy_rdbo_rusr *rusr,
					divy_db_transaction_ctx *ts_ctx)
{
	return _insert_rusr(r, rusr, ts_ctx);
}

/**
 * 指定されたrusr のユーザリレーション削除する。
 * 
 */
DIVY_DECLARE(int) divy_rdbo_remove_rusr(request_rec *r,
					const divy_rdbo_rusr *rusr)
{
	return _remove_grpmem(r, rusr->grpid, rusr->usrid, NULL);
}

/**
 * 指定されたsrc_rusr が示すユーザリレーションをdist_rusr の示す
 * ユーザリレーションに移動する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_move_rusr(request_rec *r,
					const divy_rdbo_rusr *src_rusr,
					const divy_rdbo_rusr *dist_rusr)
{
	apr_pool_t *p      = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;

	TRACE(p);

	/* トランザクションコンテキストの作成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* src_rusr を削除する */
	if (_remove_grpmem(r, src_rusr->grpid, src_rusr->usrid, ts_ctx)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/* dist_rusr を新規作成する */
	if (_insert_rusr(r, dist_rusr, ts_ctx)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/* 変更を確定する */
	divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたuri のSQLリレーションを取得する
 *
 * (note)
 * 	親グループに割り当てられたSQLの使用権は、下位グループにも継承される
 * 	という仕様になっているため、下位階層のグループには直接割り当てられて
 * 	いなくても、対応するSQLリレーションを取得して返却します。
 * 
 */
DIVY_DECLARE(int) divy_rdbo_get_rsql(request_rec *r,
					const char *uri,
					divy_rdbo_rsql **rsql)
{
	DbConn          *dbconn    = NULL;
	DbPreparedStmt  *stmt      = NULL;
	DbResultSet     *rset      = NULL;
	apr_pool_t      *p         = r->pool;
	char *sql                  = NULL;
	divy_cset_t     *uri_set   = NULL;
	divy_db_bind_ctx *bindctx = NULL;
	divy_db_bind_ctx_idx *idx;
	divy_linkedlist_t *in_clause = NULL;
	const char *in_clause_str = NULL;
	int i;

	*rsql = NULL;	/* 初期化 */

	/*
	 * uri をパースして、sqlname とグループURI を取り出す
	 */

	if (divy_rdbo_parse_rsql_uri(r, uri, rsql)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse uri. (uri = %s)", uri);
		*rsql = NULL;
		return 1;
	}

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.");
		return 1;
	}

	dbconn->startTrans(dbconn, 0);

	/*
	 * (*rsql)->grp_uri に直接割り当てられた、uri の
	 * SQLリレーションのプロパティを取得する
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				" sqlm_sql_id_c,"
				" sqlm_grp_id_c "
				"FROM divy_sqlmem "
				"WHERE sqlm_sql_id_c = "
				"(SELECT sql_id_c"
				" FROM divy_sql"
				" WHERE sql_label_name_vc = ?) "
				"AND sqlm_grp_id_c = "
				"(SELECT grp_grp_id_c"
				" FROM divy_grp"
				" WHERE grp_relative_uri_txt = ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.Reason: %s",
			stmt->getMsg(stmt));

		dbconn->rollback(dbconn);
		if (stmt != NULL)  stmt->close(stmt);
		return 1;
	}

	stmt->setString(stmt, 1, (*rsql)->labelname);
	stmt->setString(stmt, 2, (*rsql)->grp_uri);
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_sqlmem."
			"(sqlname = %s, grp_uri = %s) Reason: %s",
			(*rsql)->labelname, (*rsql)->grp_uri,
			rset->getMsg(rset));

		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		(*rsql)->sqlid     = rset->getString(rset, 1);
		(*rsql)->grpid     = rset->getString(rset, 2);

		/* 直接割り当てられていたら、親グループにuri のSQLが
		 * 割り当てられているかどうかを探さなくていい */
		(*rsql)->inheritsql  = DIVY_SQL_INHERIT_FALSE;
		(*rsql)->owner_grpid = (*rsql)->grpid;

		dbconn->commit(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 0;
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * (*rsql)->grp_uri の親グループに割り当てられた、uri の
	 * SQLリレーションのプロパティを取得する
	 */

	/* uri を親URIの集合に分解する */
	uri_set = divy_get_parenturi_set(p, (*rsql)->grp_uri);
	if (uri_set == NULL) {
		/* 親URIに分解できない -> 自身が最上位だった */
		*rsql = NULL;	/* リレーションプロパティはなかった */
		dbconn->commit(dbconn);
		return 0;
	}

	/* バインドコンテキスト/インデックスを生成 */
	bindctx = divy_db_bindcontext_make(p, uri_set, -1);
	idx     = divy_db_bindcontext_first(bindctx);

	/* バインド変数文字列の取得 */
	divy_db_bindcontext_get_bindstr(idx, &in_clause_str);

	/* SQL文の作成 (自分のグループIDもとる) */
	sql = apr_psprintf(p, 
				"SELECT "
				" sqlm_sql_id_c,"
				" sqlm_grp_id_c, "
				" own.grp_grp_id_c "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"FROM divy_sqlmem "
				"LEFT JOIN"
				" (SELECT grp_grp_id_c"
				"  FROM divy_grp"
				"  WHERE grp_relative_uri_txt = ?) own"
				" ON 1=1 "
				"WHERE sqlm_sql_id_c = "
				"(SELECT sql_id_c"
				" FROM divy_sql"
				" WHERE sql_label_name_vc = ?) "
				"AND sqlm_grp_id_c IN "
				"(SELECT grp_grp_id_c"
				" FROM divy_grp"
				" WHERE grp_relative_uri_txt IN (%s))",
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
				"FROM divy_sqlmem,"
				" (SELECT grp_grp_id_c"
				" FROM divy_grp"
				" WHERE grp_relative_uri_txt = ?) own"
				" WHERE 1=1"
				" AND (sqlm_sql_id_c = "
				"(SELECT sql_id_c"
				" FROM divy_sql"
				" WHERE sql_label_name_vc = ?) "
				"AND sqlm_grp_id_c IN "
				"(SELECT grp_grp_id_c"
				" FROM divy_grp"
				" WHERE grp_relative_uri_txt IN (%s)))",
#endif
				in_clause_str);
	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.Reason: %s",
			stmt->getMsg(stmt));

		dbconn->rollback(dbconn);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	i = 0;
	stmt->setString(stmt, ++i, (*rsql)->grp_uri);
	stmt->setString(stmt, ++i, (*rsql)->labelname);

	divy_db_bindcontext_get_list(idx, &in_clause);
	for (; in_clause; in_clause = in_clause->next) {
		stmt->setString(stmt, ++i, in_clause->val);
	}

	/* SQL の実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_sqlmem."
			"(sqlname = %s) Reason: %s",
			(*rsql)->labelname, rset->getMsg(rset));

		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 親が同じSQLを複数持っていても情報さえ取れれば問題ない */
	if (rset->next(rset) == DB_TRUE) {
		(*rsql)->sqlid       = rset->getString(rset, 1);
		(*rsql)->owner_grpid = rset->getString(rset, 2);
		(*rsql)->grpid       = rset->getString(rset, 3);

		/* 親グループが持っていたことを明示する */
		(*rsql)->inheritsql  = DIVY_SQL_INHERIT_TRUE;
	}

	/* SQLを持っているグループが１つものなかった or
	 * uri のグループが世の中に存在していなかった場合 */
	if (IS_EMPTY((*rsql)->owner_grpid) || IS_EMPTY((*rsql)->grpid)) {
		/* リレーションプロパティは無かったことにする */
		*rsql = NULL;
	}

	dbconn->commit(dbconn);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたrsql のSQLリレーションを追加する。
 * 
 * rusr 必須項目: labelname, grpid
 */
DIVY_DECLARE(int) divy_rdbo_insert_rsql(request_rec *r,
					const divy_rdbo_rsql *rsql)
{
	return _insert_rsql(r, rsql, NULL);
}

/**
 * 指定されたrsql のSQLリレーションを削除する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_rsql(request_rec *r,
					const divy_rdbo_rsql *rsql)
{
	return _remove_sqlmem(r, rsql->grpid, rsql->sqlid, NULL);
}

/**
 * 指定されたsrc_rsql が示すSQLリレーションをdist_rsql の示す
 * SQLリレーションとして移動する。
 *
 * src_rusr  必須項目: sqlid, grpid
 * dist_rusr 必須項目: labelname, grp_uri
 */
DIVY_DECLARE(int) divy_rdbo_move_rsql(request_rec *r,
					const divy_rdbo_rsql *src_rsql,
					const divy_rdbo_rsql *dist_rsql)
{
	divy_db_transaction_ctx *ts_ctx = NULL;
	apr_pool_t *p = r->pool;

	TRACE(p);

	/* トランザクションコンテキストの作成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* src_rsql を削除する */
	if (_remove_sqlmem(r, src_rsql->grpid, src_rsql->sqlid, ts_ctx)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/* dist_rsql を新規作成する */
	if (_insert_rsql(r, dist_rsql, ts_ctx)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/* 変更を確定する */
	divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたuri のグループリレーションを取得する
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_rgrp(request_rec *r,
					const char *uri,
					divy_rdbo_rgrp **rgrp)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	char *grp_uri           = NULL;

	TRACE(p);

	*rgrp = NULL;	/* 初期化 */

	/*
	 * uri をパースして、userid またはSQL名称とグループID を取り出す
	 */
	if (divy_rdbo_parse_rgrp_uri(r, uri, rgrp)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse uri. (uri = %s)", uri);
		*rgrp = NULL;
		return 1;
	}

	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.");
		return 1;
	}

	dbconn->startTrans(dbconn, 0);

	/* ユーザに対するリレーションの場合 */
	if ((*rgrp)->kind_of_relation == DIVY_RGRP_TO_USER) {
		/* 存在の確認だけで十分 */
		stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				"grpm_grp_id_c "
				"FROM divy_grpmem "
				"WHERE grpm_grp_id_c = ? "
				"AND grpm_usr_id_vc = ?",p);
	}
	/* SQL に対するリレーションの場合には、
	 * SQLシーケンス番号とグループのURIを取る
	 * (リレーションがなければsqlm_sql_id_c はNULL) */
	else {
		stmt = dbconn->prepareStatement(dbconn,
				"SELECT"
				" sm.sqlm_sql_id_c,"
				" grp_relative_uri_txt "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"FROM divy_grp "
				"LEFT JOIN"
				" (SELECT sqlm_sql_id_c"
				"  FROM divy_sqlmem"
				"  WHERE sqlm_grp_id_c = ?"
				"  AND sqlm_sql_id_c ="
				"  (SELECT sql_id_c"
				"   FROM divy_sql"
				"   WHERE sql_label_name_vc = ?)) sm"
				" ON 1=1 "
				"WHERE grp_grp_id_c = ?", 
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
				"FROM divy_grp,"
				" (SELECT sqlm_sql_id_c"
				" FROM divy_sqlmem"
				" WHERE sqlm_grp_id_c = ?"
				" AND sqlm_sql_id_c ="
				" (SELECT sql_id_c"
				" FROM divy_sql"
				" WHERE sql_label_name_vc = ?)) sm"
				" WHERE 1=1"
				" AND grp_grp_id_c = ?",
#endif
				p);
	}

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.Reason: %s",
			stmt->getMsg(stmt));

		dbconn->rollback(dbconn);
		if (stmt != NULL)  stmt->close(stmt); stmt = NULL;
		return 1;
	}

	stmt->setString(stmt, 1, (*rgrp)->grpid);
	if ((*rgrp)->kind_of_relation == DIVY_RGRP_TO_USER) {
		stmt->setString(stmt, 2, (*rgrp)->usrid);
	}
	else {
		stmt->setString(stmt, 2, (*rgrp)->labelname);
		stmt->setString(stmt, 3, (*rgrp)->grpid);
	}

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {

		if ((*rgrp)->kind_of_relation == DIVY_RGRP_TO_USER) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select divy_grpmem."
				"(grpid = %s, userid = %s) Reason: %s",
				(*rgrp)->grpid, (*rgrp)->usrid,
				rset->getMsg(rset));
		}
		else {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select divy_sqlmem."
				"(grpid = %s, sqlname = %s) Reason: %s",
				(*rgrp)->grpid, (*rgrp)->labelname,
				rset->getMsg(rset));
		}

		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* ユーザのグループリレーションの場合 */
	if ((*rgrp)->kind_of_relation == DIVY_RGRP_TO_USER) {

		/* 値が取れなかった場合 */
		if (rset->next(rset) != DB_TRUE) *rgrp = NULL;

		/* 存在の確認だけで十分なので値はいらない */
		dbconn->commit(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 0;
	}

	/* SQLのグループリレーションの場合 */

	/* 少なくとも(*rgrp)->grpid のエントリがあった場合 */
	if (rset->next(rset) == DB_TRUE) {
		(*rgrp)->sqlid = rset->getString(rset, 1);
		grp_uri        = rset->getString(rset, 2);

		/* SQLは(*rgrp)->grpid に直接割り当てられていたか？ */
		if (IS_FILLED((*rgrp)->sqlid)) {
			/* 直接割り当てられていた */
			(*rgrp)->inheritsql  = DIVY_SQL_INHERIT_FALSE;
			(*rgrp)->owner_grpid = (*rgrp)->grpid;
		}
		/* SQLの使用権を親グループから継承していたかどうか探す */
		else {
			divy_cset_t *uri_set;
			char *sql;
			divy_linkedlist_t *in_clause = NULL;
			divy_db_bind_ctx *bindctx = NULL;
			divy_db_bind_ctx_idx *idx;
			const char *in_clause_str = NULL;
			int i;

			/* 資源を解放しておく */
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			/* grp_uri を親URIの集合に分解する */
			uri_set = divy_get_parenturi_set(p, grp_uri);
			if (uri_set == NULL) {
				/* 親URIに分解できない -> 自身が最上位だった */
				*rgrp = NULL;	/* リレーションプロパティは無かった */
				dbconn->commit(dbconn);
				return 0;
			}
			/* バインドコンテキスト/インデックスを生成 */
			bindctx = divy_db_bindcontext_make(p, uri_set, -1);
			idx     = divy_db_bindcontext_first(bindctx);

			/* バインド変数文字列の取得 */
			divy_db_bindcontext_get_bindstr(idx, &in_clause_str);

			/* SQL文の作成 */
			sql = apr_psprintf(p, 
					"SELECT "
					" sqlm_sql_id_c,"
					" sqlm_grp_id_c "
					"FROM divy_sqlmem "
					"WHERE sqlm_sql_id_c = "
					"(SELECT sql_id_c"
					" FROM divy_sql"
					" WHERE sql_label_name_vc = ?) "
					"AND sqlm_grp_id_c IN "
					"(SELECT grp_grp_id_c"
					" FROM divy_grp"
					" WHERE grp_relative_uri_txt IN (%s))",
					in_clause_str);

			/* SQL の実行 */
			stmt = dbconn->prepareStatement(dbconn, sql, p);
			if (stmt->getCode(stmt) != DB_SUCCESS) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt.Reason: %s", stmt->getMsg(stmt));

				dbconn->rollback(dbconn);
				if (stmt != NULL) stmt->close(stmt);
				return 1;
			}

			/* バインド */
			i = 0;
			stmt->setString(stmt, ++i, (*rgrp)->labelname);

			divy_db_bindcontext_get_list(idx, &in_clause);
			for (; in_clause; in_clause = in_clause->next) {
				stmt->setString(stmt, ++i, in_clause->val);
			}

			rset = stmt->executeQuery(stmt, p);
			if (rset->getCode(rset) != DB_SUCCESS) {
				ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to select divy_sqlmem."
					"(sqlname = %s, in_clause = %s) Reason: %s",
					(*rgrp)->labelname, in_clause_str, rset->getMsg(rset));

				dbconn->rollback(dbconn);
				if (rset != NULL) rset->close(rset); rset = NULL;
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}

			/* 親が同じSQLを複数持っていても情報さえ取れれば問題ない */
			if (rset->next(rset) == DB_TRUE) {
				(*rgrp)->sqlid       = rset->getString(rset, 1);
				(*rgrp)->inheritsql  = DIVY_SQL_INHERIT_TRUE;
				(*rgrp)->owner_grpid = rset->getString(rset, 2);
			}

			/* SQLを持っている親グループが１つもなかった場合 */
			if (IS_EMPTY((*rgrp)->sqlid)) {
				/* リレーションプロパティは無かったことにする */
				*rgrp = NULL;
			}
		}
	}
	/* (*rgrp)->grpid のエントリが無かった --> グループがない */
	else {
		*rgrp = NULL;	/* リレーションは無かった */
	}

	dbconn->commit(dbconn);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたrgrp のグループリレーションを追加する。
 * なお、ユーザのグループリレーションの場合、追加対象グループが持っている
 * デフォルトメール監視状態を、ユーザに設定します。
 * 
 */
DIVY_DECLARE(int) divy_rdbo_insert_rgrp(request_rec *r,
					const divy_rdbo_rgrp *rgrp)
{
	if (rgrp->kind_of_relation == DIVY_RGRP_TO_USER) {
		return _insert_grpmem(r, rgrp->grpid, rgrp->usrid, NULL);
	}
	else if (rgrp->kind_of_relation == DIVY_RGRP_TO_SQL) {
		return _insert_sqlmem_using_name(r, rgrp->grpid,
						rgrp->labelname, NULL);
	}

	return 1;
}

/**
 * 指定されたrgrp のグループリレーションを削除する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_rgrp(request_rec *r,
					const divy_rdbo_rgrp *rgrp)
{
	if (rgrp->kind_of_relation == DIVY_RGRP_TO_USER) {
		return _remove_grpmem(r, rgrp->grpid, rgrp->usrid, NULL);
	}
	else if (rgrp->kind_of_relation == DIVY_RGRP_TO_SQL) {
		return _remove_sqlmem(r, rgrp->grpid, rgrp->sqlid, NULL);
	}

	return 1;
}

/**
 * 指定されたsrc_rsql が示すSQLリレーションをdist_rsql の示す
 * SQLリレーションとして移動する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_move_rgrp(request_rec *r,
					const divy_rdbo_rgrp *src_rgrp,
					const divy_rdbo_rgrp *dist_rgrp)
{
	divy_db_transaction_ctx *ts_ctx = NULL;
	apr_pool_t *p = r->pool;
	int kind_of_relation = src_rgrp->kind_of_relation;

	TRACE(p);

	/* トランザクションコンテキストの作成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* ユーザと関係するリレーションの移動 */
	if (kind_of_relation == DIVY_RGRP_TO_USER) {
		/* src_rusr を削除する */
		if (_remove_grpmem(r, src_rgrp->grpid, src_rgrp->usrid, ts_ctx)) {
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);

			return 1;
		}

		/* dist_rusr を新規作成する */
		if (_insert_grpmem(r, dist_rgrp->grpid, dist_rgrp->usrid, ts_ctx)) {
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);

			return 1;
		}
	}
	/* SQLと関係するリレーションの移動 */
	else {
		/* src_rsql を削除する */
		if (_remove_sqlmem(r, src_rgrp->grpid, src_rgrp->sqlid, ts_ctx)) {
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);

			return 1;
		}

		/* dist_rsql を新規作成する */
		 if (_insert_sqlmem_using_name(r, dist_rgrp->grpid,
					 	dist_rgrp->labelname, ts_ctx)) {
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);

			return 1;
		}
	}

	/* 変更を確定する */
	divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたユーザリレーションのuriを解析して、ユーザIDとグループuriを
 * 切り出す。
 * (note) ユーザリレーションuri のフォーマット
 * 	$root/.management/GROUP/$grp_uri/.RU_$userid
 */
DIVY_DECLARE(int) divy_rdbo_parse_rusr_uri(request_rec *r,
					const char *uri, divy_rdbo_rusr **rusr)
{
	apr_pool_t *p = r->pool;
	divy_uri_spec *u_spec = NULL;
	int len, flen;

	*rusr = NULL;	/* 初期化 */

	if (divy_parse_uri(p, dav_divy_get_root_uri(r), uri, &u_spec)) return 1;

	len  = strlen(DIVY_PREFIX_RUSER);
	flen = strlen(u_spec->final_path_segment);
	if (len > flen) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Invalid relation uri of user. The relation name part "
			"must be longger than final path segment."
			"(final path segment = %s, relation name part = %s)",
			u_spec->final_path_segment, DIVY_PREFIX_RUSER);
		return 1;
	}

	*rusr = apr_pcalloc(p, sizeof(divy_rdbo_rusr));
	(*rusr)->usrid   = apr_pstrdup(p, &u_spec->final_path_segment[len]);
	(*rusr)->grp_uri = apr_pstrndup(p, u_spec->other_part,
				strlen(u_spec->other_part) - (flen + 1));
	(*rusr)->grpid   = NULL;

	return 0;
}

/**
 * 指定されたSQLリレーションのuriを解析して、SQL表示名とグループuriを
 * 切り出す。
 * (note) SQLリレーションuri のフォーマット
 *	$root/.management/GROUP/$grp_uri/.RS_$sqlname
 */
DIVY_DECLARE(int) divy_rdbo_parse_rsql_uri(request_rec *r,
					const char *uri, divy_rdbo_rsql **rsql)
{
	apr_pool_t *p = r->pool;
	divy_uri_spec *u_spec = NULL;
	int len, flen;

	*rsql = NULL;	/* 初期化 */

	if (divy_parse_uri(p, dav_divy_get_root_uri(r), uri, &u_spec)) return 1;

	len  = strlen(DIVY_PREFIX_RSQL);
	flen = strlen(u_spec->final_path_segment);
	if (len > flen) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Invalid relation uri of sql. The relation name part "
			"must be longger than final path segment."
			"(final path segment = %s, relation name part = %s)",
			u_spec->final_path_segment, DIVY_PREFIX_RSQL);
		return 1;
	}

	*rsql = apr_pcalloc(p, sizeof(divy_rdbo_rsql));
	(*rsql)->labelname   = apr_pstrdup(p, &u_spec->final_path_segment[len]);
	(*rsql)->grp_uri     = apr_pstrndup(p, u_spec->other_part,
				strlen(u_spec->other_part) - (flen + 1));
	(*rsql)->sqlid       = NULL;
	(*rsql)->inheritsql  = DIVY_SQL_INHERIT_FALSE;	/* デフォルト値 */
	(*rsql)->owner_grpid = NULL;

	return 0;
}

/**
 * 指定されたグループリレーションのuriを解析して、ユーザIDまたはSQL表示名と
 * グループuriを切り出す。
 * (note) グループリレーションuri のフォーマット
 *	$root/.management/USER/$userid/.RG_$grpid
 *	$root/.management/SQL/$sqlname/.RG_$grpid
 */
DIVY_DECLARE(int) divy_rdbo_parse_rgrp_uri(request_rec *r,
					const char *uri, divy_rdbo_rgrp **rgrp)
{
	apr_pool_t *p = r->pool;
	divy_uri_spec *u_spec = NULL;
	int len, flen;

	*rgrp = NULL;	/* 初期化 */

	if (divy_parse_uri(p, dav_divy_get_root_uri(r), uri, &u_spec)) return 1;

	len  = strlen(DIVY_PREFIX_RGROUP);
	flen = strlen(u_spec->final_path_segment);
	if (len > flen) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Invalid relation uri of sql. The relation name part "
			"must be longger than final path segment."
			"(final path segment = %s, relation name part = %s)",
			u_spec->final_path_segment, DIVY_PREFIX_RGROUP);
		return 1;
	}

	*rgrp = apr_pcalloc(p, sizeof(divy_rdbo_rgrp));
	(*rgrp)->grpid = apr_pstrdup(p, &u_spec->final_path_segment[len]);

	/* ユーザに対するリレーションの場合 */
	if (u_spec->infotype == DIVY_INFOTYPE_m_user_rgrp) {
		(*rgrp)->kind_of_relation = DIVY_RGRP_TO_USER;
		(*rgrp)->usrid = apr_pstrndup(p, &u_spec->other_part[1],
					strlen(u_spec->other_part) - (flen + 2));
	}
	/* SQL に対するリレーションの場合 */
	else if (u_spec->infotype == DIVY_INFOTYPE_m_sql_rgrp) {
		(*rgrp)->kind_of_relation = DIVY_RGRP_TO_SQL;
		(*rgrp)->labelname   = apr_pstrndup(p, &u_spec->other_part[1],
					strlen(u_spec->other_part) - (flen + 2));
		(*rgrp)->inheritsql  = DIVY_SQL_INHERIT_FALSE;
		(*rgrp)->owner_grpid = NULL;
	}
	else {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The specified uri is not grp-relation uri."
			"(uri = %s, infotype = %d)", uri, u_spec->infotype);
	}

	(*rgrp)->sqlid = NULL;

	return 0;
}

/**
 * 指定されたグループIDからなる集合grpid_hashを使ってdivy_grpテーブルを検索し、
 * uriとグループIDの組合せを取得。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_grpuri_by_grpid(request_rec *r,
							divy_cset_t *grpid_set,
							divy_rdbo_grp **grp_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	divy_rdbo_grp *new_grp  = NULL;
	char *sql               = NULL;
	divy_linkedlist_t *in_clause = NULL;
	divy_db_bind_ctx *bindctx = NULL;
	divy_db_bind_ctx_idx *idx;
	const char *in_clause_str = NULL;
	int i;
	divy_db_transaction_ctx *ts_ctx = NULL;

	*grp_pr = NULL;	/* 初期化 */

	if (grpid_set == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"grpid_set is NULL.");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* バインドコンテキストを生成 */
	bindctx = divy_db_bindcontext_make(p, grpid_set, 50);
	for (idx = divy_db_bindcontext_first(bindctx); idx; idx = divy_db_bindcontext_next(idx)) {

		/* バインド変数文字列の取得 */
		divy_db_bindcontext_get_bindstr(idx, &in_clause_str);

		/* SQL文の作成 */
		sql = apr_psprintf(p,
				"SELECT "
				"grp_grp_id_c,"
				"grp_relative_uri_txt "
				"FROM divy_grp "
				"WHERE grp_grp_id_c IN (%s)", in_clause_str);

		/* sql文の準備　失敗時はエラー */
		stmt = dbconn->prepareStatement(dbconn, sql, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt.(in_clause = %s) "
				"Reason: %s", in_clause_str, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		/* バインド */
		i = 0;
		divy_db_bindcontext_get_list(idx, &in_clause);
		for (; in_clause; in_clause = in_clause->next) {
			stmt->setString(stmt, ++i, in_clause->val);
		}

		/* sql実行　失敗時はエラー */
		rset = stmt->executeQuery(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select divy_sql.(in_clause = %s) "
				"Reason: %s", in_clause_str, rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		/* 結果の取得 */
		while (rset->next(rset) == DB_TRUE) {

			if (*grp_pr == NULL) {
				*grp_pr = new_grp = apr_pcalloc(p, sizeof(divy_rdbo_grp));
			}
			else {
				new_grp->next = apr_pcalloc(p, sizeof(divy_rdbo_grp));
				new_grp = new_grp->next;
			}

			new_grp->grpid       = rset->getString(rset, 1);
			new_grp->relativeuri = rset->getString(rset, 2);
			new_grp->next = NULL;
		}

		/* クローズ */
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたuri 以下のリソースの内、Directロックされているリソース一覧を取得する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_hierarchy_lockedresource(request_rec *r,
						const char *uri,
						divy_rdbo_resource **rdb_r)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;
	divy_rdbo_resource *n_rdb_r = NULL;

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY");
		return 1;
	}

	*rdb_r = NULL;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT"
				" r.rs_uri_txt,"
				" r.rs_dispname_vc, "
				" r.rs_resourcetype_i "
				"FROM dav_resource r "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"INNER JOIN dav_lock lk "
				"ON r.rs_uri_txt = lk.lk_uri_txt "
				"WHERE lk_lockkbn_i = ? "
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				",dav_lock lk "
				"WHERE r.rs_uri_txt = lk.lk_uri_txt "
				"AND lk_lockkbn_i = ? "
#endif
				"AND (r.rs_uri_txt = ? "
				" OR r.rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE")", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s) "
			"Reason: %s", uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setInt(stmt,    1, DAV_LOCK_DIRECT); /* Direct ロックだけを取る */
	stmt->setString(stmt, 2, uri);
	stmt->setString(stmt, 3, apr_pstrcat(p,
				stmt->escWildCard(stmt, uri), "/%", NULL));

	/* SQLの実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select dav_resource and dav_lock."
			"(prvcol_uri = %s) Reason: %s", uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 
	 * 結果の取得 
	 */
	while (rset->next(rset) == DB_TRUE) {

		/* 値の取得 */
		if (*rdb_r == NULL) {
			*rdb_r = n_rdb_r =
				apr_pcalloc(p, sizeof(divy_rdbo_resource));
		}
		else {
			/* 新規作成 */
			n_rdb_r->next = apr_pcalloc(p, sizeof(divy_rdbo_resource));
			n_rdb_r = n_rdb_r->next;
		}

		n_rdb_r->uri          = rset->getString(rset, 1);
		n_rdb_r->displayname  = rset->getString(rset, 2);
		n_rdb_r->resourcetype = rset->getInt(rset,    3);
		n_rdb_r->next         = NULL;
	}

	/* 終了処理 */
	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたrdb_r->rsid と一致するリソースのプロパティを取得する。
 * 取得した値は、引数rdb_r のプロパティに詰めて返却します。
 * 値が取得できたかどうかは、IS_EMPTY(rdb_r->uri) が真かどうかで判定して下さい。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_property_by_rsid(request_rec *r,
						divy_rdbo_resource *rdb_r,
						apr_uint32_t propflag)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	dav_divy_dir_conf *conf   = dav_divy_get_dir_config(r);
	divy_db_transaction_ctx *ts_ctx = NULL;
	char *sql;
	apr_hash_t *rstate_pr_hash = NULL;	/* rsid - (divy_rdbo_resourcestate *) */
	int ret;

	/* 条件句を含まないSQL文 */
	char *sql_base_stmt     =
				"SELECT"
				" rs.rs_rs_id_c,"
				" rs.rs_uri_txt,"
				" rs.rs_dispname_vc,"
				" rs.rs_create_bi,"
				" rs.rs_get_cont_lang_vc,"
				" rs.rs_get_cont_len_bi,"
				" rs.rs_get_cont_type_vc,"
				" rs.rs_get_etag_txt,"
				" rs.rs_get_lastmodified_bi,"
				" rs.rs_resourcetype_i,"
				" rs.rs_depth_i,"
				" rs.rs_isversioned_i,"
				" rs.rs_checkin_i,"
				" rs.rs_checkout_i,"
				" rs.rs_physical_path_txt,"
				" rs.rs_creator_usr_id_vc,"
				" rs.rs_lastmodifier_usr_id_vc ";

	if (rdb_r == NULL || IS_EMPTY(rdb_r->rsid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rdb_r or rsid is EMPTY.");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	

	/* SQL文の生成 */
	if (propflag & DIVY_GET_PROP_fullname) {
		sql = apr_pstrcat(p, sql_base_stmt,
				",u1.usr_fullname_vc AS rs_creator_vc,"
				" u2.usr_fullname_vc AS rs_lastmodifier_vc "
				"FROM dav_resource rs "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"LEFT JOIN divy_usr u1 "
				"ON rs.rs_creator_usr_id_vc = u1.usr_usr_id_vc "
				"LEFT JOIN divy_usr u2 "
				"ON rs.rs_lastmodifier_usr_id_vc = u2.usr_usr_id_vc "
				"WHERE rs_rs_id_c = ?"
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				", divy_usr u1, divy_usr u2 "
				"WHERE rs.rs_creator_usr_id_vc = u1.usr_usr_id_vc(+) "
				"AND rs.rs_lastmodifier_usr_id_vc = u2.usr_usr_id_vc(+) "
				"AND rs_rs_id_c = ?"
#endif
				, NULL);
	}
	else {
		sql = apr_pstrcat(p, sql_base_stmt,
				"FROM dav_resource rs "
				"WHERE rs_rs_id_c = ?", NULL);
	}

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, sql, p); 
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (rsid = %s) "
			"Reason: %s", rdb_r->rsid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, rdb_r->rsid);

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select dav_resource. (rsid = %s) "
			"Reason: %s", rdb_r->rsid, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {

		/* 値の取り出し */
		rdb_r->rsid               = rset->getString(rset, 1);
		rdb_r->uri                = rset->getString(rset, 2);
		rdb_r->displayname        = rset->getString(rset, 3);
		rdb_r->creationdate       = (time_t) rset->getBigInt(rset, 4);
		rdb_r->getcontentlanguage = rset->getString(rset, 5);
		if (IS_EMPTY(rdb_r->getcontentlanguage)) {
			rdb_r->getcontentlanguage = NULL;
		}
		rdb_r->getcontentlength   = rset->getBigInt(rset, 6);
		rdb_r->getcontenttype     = rset->getString(rset, 7);
		if (IS_EMPTY(rdb_r->getcontenttype)) {
			rdb_r->getcontenttype = NULL;
		}
		rdb_r->getetag            = rset->getString(rset, 8);
		rdb_r->getlastmodified    = (time_t) rset->getBigInt(rset, 9);
		rdb_r->resourcetype       = rset->getInt(rset,   10);
		rdb_r->depth              = rset->getInt(rset,   11);
		rdb_r->isversioned        = rset->getInt(rset,   12);
		rdb_r->checkin            = rset->getInt(rset,   13);
		rdb_r->checkout           = rset->getInt(rset,   14);
		rdb_r->physicalpath       = rset->getString(rset,15);
		rdb_r->creator_userid     = rset->getString(rset,16);
		rdb_r->lastmodifier_userid= rset->getString(rset,17);

		if (propflag & DIVY_GET_PROP_fullname) {
			divy_rdbo_usr ldapusr = { 0 };

			rdb_r->creator      = rset->getString(rset,18);
			rdb_r->lastmodifier = rset->getString(rset,19);

			/* 
			 * LDAPから名前を取得して置換する
			 * 問合せが失敗した場合はそのままの値を利用する
			 */
			ldapusr.fullname = rdb_r->creator;
			if (IS_FILLED(conf->ldapmapfullnameattr) &&
					divy_util_ldap_get_user_property(r, p,
						rdb_r->creator_userid,
						&ldapusr) == TF_LDAP_TRUE) {
					rdb_r->creator = ldapusr.fullname;
			}

			/* 作成者ユーザIDと更新者ユーザID が同じ時 */
			if (rdb_r->creator_userid &&
			    rdb_r->lastmodifier_userid &&
			    strcmp(rdb_r->creator_userid,
				    	rdb_r->lastmodifier_userid) == 0) {
				rdb_r->lastmodifier = rdb_r->creator;
			}
			/* 作成者ユーザIDと更新者ユーザID が異なる時 */
			else {
				ldapusr.fullname = rdb_r->lastmodifier;
				if (IS_FILLED(conf->ldapmapfullnameattr) &&
						divy_util_ldap_get_user_property(r, p,
							rdb_r->lastmodifier_userid,
							&ldapusr) == TF_LDAP_TRUE) {
					rdb_r->lastmodifier = ldapusr.fullname;
				}
			}
		}
		rdb_r->next = NULL;	/* sentinel */
	}
	else {
		/* データが無かった場合 */
		rdb_r->uri = NULL;	/* データが無かったことを明示する */
	}
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * 状態/属性 プロパティの取得
	 */
	if (IS_FILLED(rdb_r->uri) && (propflag & DIVY_GET_PROP_resourcestate)) {
		ret = _get_hierarchy_resourcestate_property(r, rdb_r->uri, 0, &rstate_pr_hash, ts_ctx);
		if (ret) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get resourcestate property.(uri = %s)", rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/* 状態/属性プロパティの取り出し */
		if (rstate_pr_hash != NULL) {
			rdb_r->rstate_pr = apr_hash_get(rstate_pr_hash, rdb_r->rsid, APR_HASH_KEY_STRING);
		}
	}

	/* 終了処理 */
	divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたrdb_r が表すコレクション直下のリソースをoffset から limit まで取得して
 * 返却する。rdb_r 自身のリソースは含みません。(autoindex 用)
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_autoindex_data(request_rec *r,
						divy_rdbo_resource *rdb_r,
						apr_int32_t offset,
						apr_int32_t limit,
						apr_uint32_t propflag)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	dav_divy_dir_conf *conf   = dav_divy_get_dir_config(r);
	apr_pool_t *p           = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;
	apr_int32_t cnt = 0;
	divy_rdbo_resource *n_rdb_r = NULL;
	divy_sbuf *sql_buf    = NULL;
	divy_rdbo_usr ldapusr = { 0 };
	int found_extra_data  = 0;	/* 続きのページデータがあるかどうか */
	apr_int32_t totallimit= 0;
	apr_hash_t *rstate_pr_hash = NULL;	/* rsid - (divy_rdbo_resourcestate *) */
	divy_rdbo_resourcestate *rstate_pr;
	int ret, found_trash = 0, do_skip_trash = 0;
	char *rsid, *displayname;
	divy_rdbo_resourcetype resourcetype;

	/* 条件句を含まないSQL文 */
	char *sql_base_stmt     =
			"SELECT"
			" rs.rs_rs_id_c,"
			" rs.rs_resourcetype_i,"
			" rs.rs_dispname_vc,"
			" rs.rs_uri_txt,"
			" rs.rs_get_cont_len_bi,"
			" rs.rs_get_cont_type_vc,"
			" rs.rs_get_lastmodified_bi,"
			" rs.rs_lastmodifier_usr_id_vc ";

	if (IS_EMPTY(rdb_r->uri) || limit <= 0 || offset < 0) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri or limit or offset was wrong.");
		return 1;
	}

	/* ゴミ箱ならば何もしません */
	if (rdb_r->u_spec == NULL) {
		divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &rdb_r->u_spec);
	}

	if (rdb_r->u_spec->infotype == DIVY_INFOTYPE_user_trash ||
	    rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_trash) {
		rdb_r->next = NULL;
		return 0;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * 状態/属性 プロパティの取得
	 */
	if (propflag & DIVY_GET_PROP_resourcestate) {
		ret = _get_hierarchy_resourcestate_property(r, rdb_r->uri, 1, &rstate_pr_hash, ts_ctx);
		if (ret) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get resourcestate property.(uri = %s)", rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * uri 直下のリソース一覧を取得
	 */
	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf, sql_base_stmt);

	/* (note) OFFSET, LIMIT句は利用できません。非公開リソースの除外をSQL文上で
	 * 	表現出来ないためです。表現できるようになったら使ってください。 */
	if (propflag & DIVY_GET_PROP_fullname) {
		divy_sbuf_append(sql_buf,
				",u1.usr_fullname_vc AS rs_lastmodifier_vc "
				"FROM dav_resource rs "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"LEFT JOIN divy_usr u1 "
				"ON rs.rs_lastmodifier_usr_id_vc = u1.usr_usr_id_vc "
				"WHERE (rs.rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND rs.rs_depth_i = ?) "
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				", divy_usr u1 "
				"WHERE rs.rs_lastmodifier_usr_id_vc = u1.usr_usr_id_vc(+) "
				"AND (rs.rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND rs.rs_depth_i = ?) "
#endif
				"ORDER BY rs.rs_resourcetype_i DESC, rs.rs_uri_txt ASC ");
	}
	/* フルネーム系プロパティを含めない場合 */
	else {
		divy_sbuf_append(sql_buf,
				"FROM dav_resource rs "
				"WHERE rs.rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND rs.rs_depth_i = ? "
				"ORDER BY rs.rs_resourcetype_i DESC, rs.rs_uri_txt ASC ");
	}

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p); 
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s) "
			"Reason: %s", rdb_r->uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1,
		apr_pstrcat(p, stmt->escWildCard(stmt, rdb_r->uri), "/%", NULL));
	stmt->setInt(stmt, 2, rdb_r->depth + 1);

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select dav_resource. (uri = %s) "
			"Reason: %s", rdb_r->uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 残りデータがあることを確認する目的で1行余計に取得する */
	totallimit = limit + 1;

	/* ゴミ箱をスキップする操作を行うか? */
	if (rdb_r->u_spec->infotype == DIVY_INFOTYPE_user_e ||
	    rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e) {
		do_skip_trash = 1;	/* スキップ操作をやります */
	}

	/* 
	 * 結果の取得 
	 */
	while (rset->next(rset) == DB_TRUE) {

		/* 条件判断に必要なデータをまずは取得 */
		rsid         = rset->getString(rset, 1);
		resourcetype = rset->getInt(rset, 2);
		displayname  = rset->getString(rset, 3);

		/* ゴミ箱外し */
		if (do_skip_trash && !found_trash && resourcetype == DIVY_TYPE_COLLECTION &&
					strcmp(DIVY_TRASHFOLDER_NAME, displayname) == 0) {
			found_trash = 1;	/* もう見つかったので今後は無視 */
			continue;
		}

		/* 状態/属性プロパティの取り出し */
		rstate_pr = NULL;
		if (rstate_pr_hash != NULL) {
			rstate_pr = apr_hash_get(rstate_pr_hash, rsid, APR_HASH_KEY_STRING);
		}

		/* View属性が付いていないリソースは要らない場合 */
		if (IS_INEFFECTIVE_STATE(rstate_pr) && (propflag & DIVY_DEL_PROP_noviewattr)) {
			continue;
		}

		++cnt;	/* 取得件数のカウントを開始 */
		/* offset 位置にくるまで飛ばす */
		if (cnt <= offset) {
			totallimit++;	/* 最終行を1つずらす */
			continue;
		}

		if (totallimit == cnt) {
			found_extra_data = 1;	/* 指定されたlimit件以上のデータが存在した */
			break;	/* 最終行のデータは要らない */
		}

		if (n_rdb_r == NULL) {
			/* rdb_r のnext に繋げる */
			rdb_r->next = n_rdb_r = apr_pcalloc(p, sizeof(*n_rdb_r));
		}
		else {
			n_rdb_r->next = apr_pcalloc(p, sizeof(*n_rdb_r));
			n_rdb_r = n_rdb_r->next;
		}
		n_rdb_r->p         = p;
		n_rdb_r->rstate_pr = rstate_pr; /* 状態プロパティの記録 */
		n_rdb_r->next = NULL;	/* sentinel */

		/* 値の取り出し */
		n_rdb_r->rsid                = rsid;
		n_rdb_r->resourcetype        = resourcetype;
		n_rdb_r->displayname         = displayname;
		n_rdb_r->uri                 = rset->getString(rset, 4);
		n_rdb_r->getcontentlength    = rset->getBigInt(rset, 5);
		n_rdb_r->getcontenttype      = rset->getString(rset, 6);
		n_rdb_r->getlastmodified     = (time_t) rset->getBigInt(rset, 7);
		n_rdb_r->lastmodifier_userid = rset->getString(rset, 8);

		if (propflag & DIVY_GET_PROP_fullname) {
			n_rdb_r->lastmodifier = rset->getString(rset,9);

			/* 
			 * LDAPから名前を取得して置換する
			 * 問合せが失敗した場合はそのままの値を利用する
			 */

			ldapusr.fullname = n_rdb_r->lastmodifier;
			if (IS_FILLED(conf->ldapmapfullnameattr)
					&& divy_util_ldap_get_user_property(r, p,
						n_rdb_r->lastmodifier_userid, &ldapusr) == TF_LDAP_TRUE) {
				n_rdb_r->lastmodifier = ldapusr.fullname;
			}
		}
	}

	/* 終了処理 */
	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	if (found_extra_data) {
		return DIVY_STCODE_REMAIN_DATA;	/* まだ取得可能なデータが残っていた */
	}
	else {
		return 0;
	}
}

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * システムQuota情報を取得して返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_systemquota_property(request_rec *r,
					divy_rdbo_diskquota **sysquota_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p              = r->pool;
	const char *root_uri       = dav_divy_get_root_uri(r);
	divy_rdbo_diskquota squota = { 0 };
	dav_divy_dir_conf *dconf   = dav_divy_get_dir_config(r);
	divy_error *derr;
	int ret;

	*sysquota_pr = NULL;	/* 初期化 */

	/*
	 * DB からシステムQuotaを取得する
	 */
	squota.uri  = apr_pstrdup(p, root_uri);
	squota.type = DIVY_QTTYPE_SYSTEM;

	ret = divy_rdbo_get_systemquota(r, &squota, 0, ts_ctx);
	if (ret == DIVY_STCODE_SYSQUOTA_NOT_EXIST) {
		/* システムQuotaテーブルが存在しなかった場合 */
		return 0;	/* 正常であるとする */
	}
	else if (ret) {
		/* 予期しないエラー発生 */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get system-quota information. (location = %s)",
			root_uri);
		return 1;
	}

	/* 
	 * 物理Diskから情報を取得
	 */
	*sysquota_pr = apr_pcalloc(p, sizeof(divy_rdbo_diskquota));
	(*sysquota_pr)->uri     = squota.uri;
	(*sysquota_pr)->type    = squota.type;
	(*sysquota_pr)->usedst  = squota.usedst;
	(*sysquota_pr)->maxst   = squota.maxst;
	(*sysquota_pr)->usedres = squota.usedres;
	(*sysquota_pr)->maxres  = squota.maxres;

	derr = divy_statfs(p, dconf->fsrootpath, &(*sysquota_pr)->sfs);
	if (derr != NULL) {
		DUMP_ERRLOGS(p,derr);
		/* エラーログを出すだけ */
	}

	return 0;
}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * sysquota_pr.uri で指定されたLocation のシステムQuotaを取得する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_systemquota(request_rec *r,
					divy_rdbo_diskquota *sysquota_pr,
					int do_entrylock,
			   		divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;

	/* 入力値のチェック */
	if (sysquota_pr == NULL || IS_EMPTY(sysquota_pr->uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"sysquota_pr is NULL or uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	if (do_entrylock) {
		stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				" dkqt_used_st_bi,"
				" dkqt_max_st_bi,"
				" dkqt_used_res_bi,"
				" dkqt_max_res_bi "
				"FROM divy_diskquota "
				"WHERE dkqt_uri_txt = ? "
				"AND dkqt_type_i = ? "
				"FOR UPDATE", p);
	}
	else {
		stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				" dkqt_used_st_bi,"
				" dkqt_max_st_bi,"
				" dkqt_used_res_bi,"
				" dkqt_max_res_bi "
				"FROM divy_diskquota "
				"WHERE dkqt_uri_txt = ? "
				"AND dkqt_type_i = ?", p);
	}
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (location = %s) "
			"Reason: %s", sysquota_pr->uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, sysquota_pr->uri);
	stmt->setInt(stmt,    2, sysquota_pr->type);

	/* SQL実行 */
	if (do_entrylock) {
		rset = stmt->executeQueryForUpdate(stmt, p);
	}
	else {
		rset = stmt->executeQuery(stmt, p);
	}
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_diskquota. (location = %s) "
			"Reason: %s", sysquota_pr->uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 
	 * 結果の取得 
	 */
	if (rset->next(rset) == DB_TRUE) {
		sysquota_pr->usedst  = rset->getBigInt(rset, 1);
		sysquota_pr->maxst   = rset->getBigInt(rset, 2);
		sysquota_pr->usedres = rset->getBigInt(rset, 3);
		sysquota_pr->maxres  = rset->getBigInt(rset, 4);
	}
	else {
		/* 互換性に関する考慮
		 * 本来このエントリがないのはエラーなのだが、互換性を考慮して
		 * 正常扱いとする。*/
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset);	rset = NULL;
		if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

		return DIVY_STCODE_SYSQUOTA_NOT_EXIST;
	}

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */


#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * 指定された sysquota_pr が示すシステムQuotaを登録/更新する。
 * (note)
 *    訳あって、ここのロジックをmaintenance.sh でもコピーして使用しています。
 *    ここの方式を変更する場合には、maintenance.sh 側も修正して下さい。
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_systemquota_property(request_rec *r,
					divy_rdbo_diskquota *sysquota_pr,
                              		divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int isnew               = 0;	/* 新規登録するかどうか */
	const char *root_uri    = dav_divy_get_root_uri(r);

	TRACE(p);

	if (sysquota_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"sysquota_pr is NULL.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Could not operation. Transaction ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_diskquota テーブルから行ロックしながらエントリを取得
	 */
	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				" dkqt_used_st_bi,"
				" dkqt_max_st_bi,"
				" dkqt_used_res_bi,"
				" dkqt_max_res_bi "
				"FROM divy_diskquota "
				"WHERE dkqt_uri_txt = ? "
				"AND dkqt_type_i = ? "
				"FOR UPDATE", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (location = %s) "
			"Reason: %s", root_uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, root_uri);
	stmt->setInt(stmt,    2, DIVY_QTTYPE_SYSTEM);

	/* SQL実行 */
	rset = stmt->executeQueryForUpdate(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_diskquota. (location = %s) "
			"Reason: %s", root_uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 
	 * 結果の判定
	 */
	if (rset->next(rset) == DB_TRUE) {
		isnew = 0;	/* 更新する */
	}
	else {
		isnew = 1;	/* 新規登録する */
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_diskquota テーブルの更新 / 新規登録
	 */
	/* 新規追加 */
	if (isnew) {
		apr_int64_t isum_usedst, isum_usedres;

		/* divy_usrdiskquota テーブルの全エントリをロックする */
		/* SQL文の準備 */
		stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				"usqt_usr_id_vc "
				"FROM divy_usrdiskquota "
				"FOR UPDATE", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt."
				"Reason: %s", stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* SQL実行 */
		rset = stmt->executeQueryForUpdate(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select divy_usrdiskquota."
				"Reason: %s", rset->getMsg(rset));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* 結果セットは 不要 */
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* divy_usrdiskquota から現在の使用容量を取得する */
		/* SQL文の準備 */
		stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				" SUM(usqt_used_st_bi) AS usedst,"
				" SUM(usqt_used_res_bi) AS usedres "
				"FROM divy_usrdiskquota ", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt."
				"Reason: %s", stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* SQL実行 */
		rset = stmt->executeQuery(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select divy_usrdiskquota."
				"Reason: %s", rset->getMsg(rset));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* 結果の取得 */
		if (rset->next(rset) == DB_TRUE) {
			isum_usedst  = rset->getBigInt(rset, 1);
			isum_usedres = rset->getBigInt(rset, 2);
		}
		else {
			isum_usedst  = APR_INT64_C(0);
			isum_usedres = APR_INT64_C(0);
		}
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* divy_diskquota へのINSERT */
		stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO divy_diskquota "
				"(dkqt_uri_txt,"
				" dkqt_type_i,"
				" dkqt_used_st_bi,"
				" dkqt_max_st_bi,"
				" dkqt_used_res_bi,"
				" dkqt_max_res_bi) "
				"VALUES (?, ?, ?, ?, ?, ?)", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s",
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		/* バインド */
		stmt->setString(stmt, 1, root_uri);
		stmt->setInt(stmt,    2, DIVY_QTTYPE_SYSTEM);
		stmt->setBigInt(stmt, 3, isum_usedst);
		stmt->setBigInt(stmt, 4, sysquota_pr->maxst);
		stmt->setBigInt(stmt, 5, isum_usedres);
		stmt->setBigInt(stmt, 6, sysquota_pr->maxres);

		/* SQLを実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_diskquota."
				"(root_uri = %s, type = %d) Reason: %s",
				root_uri, DIVY_QTTYPE_SYSTEM, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
	}
	/* 更新 */
	else {
		stmt = dbconn->prepareStatement(dbconn, 
				"UPDATE divy_diskquota "
				"SET "
				"dkqt_max_st_bi = ?, "
				"dkqt_max_res_bi = ? "
				"WHERE dkqt_uri_txt = ? "
				"AND dkqt_type_i = ?", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(rooturi = %s, type = %d) Reason: %s",
				root_uri, DIVY_QTTYPE_SYSTEM, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		/* バインド */
		stmt->setBigInt(stmt, 1, sysquota_pr->maxst);
		stmt->setBigInt(stmt, 2, sysquota_pr->maxres);
		stmt->setString(stmt, 3, root_uri);
		stmt->setInt(stmt,    4, DIVY_QTTYPE_SYSTEM);

		/* SQLを実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_diskquota."
				"(root_uri = %s, type = %d) Reason: %s",
				root_uri, DIVY_QTTYPE_SYSTEM, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

#ifdef DIVY_SUPPORT_GROUPQUOTA
/**
 * grpquota_pr.grpid で指定されたグループのグループQuotaを取得する。
 *
 * @param r request_rec *
 * @param grpquota_pr divy_rdbo_grpquota * 対象のグループIDと取得したQuota情報
 * @param do_entrylock int 1: ロックをかけてエントリ取得 / 0: ロックしない
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_groupquota(request_rec *r,
					divy_rdbo_grpquota *grpquota_pr,
					int do_entrylock,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn         *dbconn = NULL;
	DbPreparedStmt *stmt   = NULL;
	DbResultSet    *rset   = NULL;
	int iscommit           = 0;
	apr_pool_t     *p      = r->pool;

	/* 入力値のチェック */
	if (grpquota_pr == NULL || IS_EMPTY(grpquota_pr->grpid)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"grpquota_pr(group quota) is NULL or grpid(group ID) is EMPTY.");
			return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	stmt = dbconn->prepareStatement(dbconn,
					"SELECT "
					" gsqt_used_st_bi,"
					" gsqt_max_st_bi,"
					" gsqt_used_res_bi,"
					" gsqt_max_res_bi "
					"FROM divy_grpdiskquota "
					"WHERE gsqt_grp_id_c = ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (group id = %s) "
			"Reason: %s", grpquota_pr->grpid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, grpquota_pr->grpid);
	
	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_grpquota. (grpid = %s) "
			"Reason: %s", grpquota_pr->grpid, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 
	 * 結果の取得 
	 */
	if (rset->next(rset) == DB_TRUE) {
		grpquota_pr->usedst  = rset->getBigInt(rset, 1);
		grpquota_pr->maxst   = rset->getBigInt(rset, 2);
		grpquota_pr->usedres = rset->getBigInt(rset, 3);
		grpquota_pr->maxres  = rset->getBigInt(rset, 4);
	}
	else {
		/* 結果が無い（エントリ無し）場合でも正常とみなします
		 * 通常の場合必ず存在するはずです. この問題が発生するのは
		 * 互換性を考慮してです。
		 */
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset);	rset = NULL;
		if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

		return DIVY_STCODE_GRPQUOTA_NOT_EXIST;
	}

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}
#endif /* DIVY_SUPPORT_GROUPQUOTA */

/**
 * 指定されたuri の親コレクションに付いている状態、属性を取得する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_parent_resourcestate_property(request_rec *r,
					const char *uri,
					int type,
					divy_rdbo_resourcestate **rstate_pr)
{
	return _get_parent_resourcestate_property(r, uri, type, rstate_pr, NULL);
}

/**
 * 指定されたuri の状態・属性を取得する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_resourcestate_property(request_rec *r,
					const char *uri,
					int type,
					int consider_inherit,
					divy_rdbo_resourcestate **rstate_pr)
{
	apr_pool_t *p           = r->pool;
	divy_rdbo_resourcestate *p_rstate_pr = NULL;
	divy_db_transaction_ctx *ts_ctx = NULL;

	*rstate_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;

	/*
	 * 自身の状態・属性プロパティを取得
	 */
	if (_get_resourcestate_property(r, uri, type, rstate_pr, ts_ctx)) {

		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get own resourcestate.(uri = %s, type = %d)", uri, type);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * 自身に状態が無かったとき親コレクションから継承しているかどうか調べてみる
	 */
	if (*rstate_pr == NULL && consider_inherit) {

		/* 親コレクションの状態を取得 */
		if (_get_parent_resourcestate_property(r, uri, type, &p_rstate_pr, ts_ctx)) {
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
		*rstate_pr = p_rstate_pr;
		if (*rstate_pr != NULL) {
			(*rstate_pr)->inherit = 1;	/* 親から継承しました */
		}
	}

	/* 不要な資源を解放 */
	divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたuri以下に存在する"有効な"type の状態・属性があるかどうかを調べる。
 *
 */
DIVY_DECLARE(int) divy_rdbo_is_resourcestate_effective(request_rec *r,
					const char *uri, int type, int consider_own)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	divy_uri_spec *u_spec   = NULL;
	divy_db_transaction_ctx *ts_ctx  = NULL;
	divy_rdbo_resourcestate *rstate_pr = NULL;
	int found_ineffective   = 0;

	/* 入力不正のチェック */
	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}
	/* uri のチェック */
	divy_parse_uri(p, dav_divy_get_root_uri(r), uri, &u_spec);
	if (u_spec->infotype != DIVY_INFOTYPE_group_e_regular) {
		/* グループコレクションより下でなければ状態・属性プロパティは
		 * 存在しないものとする(グループコレクションにも付いていないとみなす) */
		return DIVY_STCODE_INEFFECTIVE_RSTATE_EXIST;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	if (consider_own) {
		/*
		 * (1) 自分自身に状態・属性プロパティはあるか?
		 */
		if (_get_resourcestate_property(r, uri, type, &rstate_pr, ts_ctx)) {

			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get own resourcestate."
				"(uri = %s, type = %d)", uri, type);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/* 状態プロパティがない / 有効ではない場合 */
		if (IS_INEFFECTIVE_STATE(rstate_pr)) {
			divy_db_commit_transaction(ts_ctx);

			/* 少なくとも1つは有効ではなかった */
			return DIVY_STCODE_INEFFECTIVE_RSTATE_EXIST;
		}
	}

	/*
	 * (2) 有効でない状態・属性プロパティが1つでも存在するかどうか
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"SELECT"
				" rt_uri_txt "
				"FROM divy_resourcestate "
				"WHERE rt_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE" "
				"AND rt_type_i = ? "
				"AND rt_effective_i = 0", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s, type = %d) "
			"Reason: %s", uri, type, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, apr_pstrcat(p, stmt->escWildCard(stmt, uri), "/%", NULL));
	stmt->setInt(stmt,    2, type);

	/* SQL文の実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_resourcestate. (uri = %s, type = %d) "
			"Reason: %s", uri, type, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* プロパティは存在したか? */
	if (rset->next(rset) == DB_TRUE) {
		found_ineffective = 1;	/* 有効でないプロパティが見つかった */
	}

	divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (rset != NULL) rset->close(rset); rset = NULL;

	if (found_ineffective) {
		/* 有効でないプロパティが1つ以上存在した */
		return DIVY_STCODE_INEFFECTIVE_RSTATE_EXIST;
	}
	else {
		/* 有効なプロパティが存在した */
		return DIVY_STCODE_EFFECTIVE_RSTATE_EXIST;
	}
}

/**
 * 指定されたrstate_pr の状態・属性プロパティを作成/更新/削除する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_resourcestate_property(request_rec *r,
					divy_rdbo_resourcestate *rstate_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;
	divy_rdbo_resourcestate *rstate, *p_rstate;
	const char *parent_uri, *root_uri;
	divy_uri_spec *p_u_spec = NULL;

	if (rstate_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rstate_pr is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	root_uri = dav_divy_get_root_uri(r);

	/*
	 * rstate_pr->uri 以下にある全ての
	 * 制限ユーザの開封通知を解除する (外部仕様)
	 * (note)
	 *   公開する時には一々削除する必要はないのだが、データ不整合で残って
	 *   いるかもしれない.これを訂正できるのは、このチャンスくらいなので
	 *   処理しておきます.
	 */
	if (divy_support_confirmreading(r)) {
		divy_rdbo_extstatus *extstatus;
		char *extstatus_wcstr = NULL;

		stmt = dbconn->prepareStatement(dbconn,
					"DELETE FROM divy_confirmreading "
					" WHERE (cr_uri_txt = ?"
					" OR cr_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE")"
					" AND cr_usr_id_vc IN ("
					" SELECT usr_usr_id_vc "
					" FROM divy_usr"
					" WHERE usr_extended_status_c IS NOT NULL"
					" AND usr_extended_status_c NOT LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE")", p);
		/* (note) システム実行権限ユーザも入ってしまうが、ファイルを持てないので影響なし */
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* 検索文字列パターンを作成する */
		extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_USR);

		divy_rdbo_reset_extstatus(extstatus, EXTSTATUS_TYPE_USR);
		divy_rdbo_set_trusty_user(extstatus, 1);	/* ここだけアクティブ化 */

		extstatus_wcstr = divy_rdbo_extstatus2wildcardstr(extstatus, EXTSTATUS_TYPE_USR);

		/* バインド */
		stmt->setString(stmt, 1, rstate_pr->uri);
		stmt->setString(stmt, 2, apr_pstrcat(p,
					stmt->escWildCard(stmt, rstate_pr->uri), "/%", NULL));
		stmt->setString(stmt, 3, extstatus_wcstr);

		/* SQL実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to delete divy_confirmreading. (uri = %s) Reason: %s",
					rstate_pr->uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/* type 毎に順番に処理していく */
	for (rstate = rstate_pr; rstate; rstate = rstate->next) {
		/* 初期化 */
		p_rstate = NULL;
		p_u_spec = NULL;

		/*
		 * (1) 親URIの算出とパース
		 */
		parent_uri = divy_get_parenturi(p, rstate->uri);
		divy_parse_uri(p, root_uri, parent_uri, &p_u_spec);

		if (p_u_spec->infotype != DIVY_INFOTYPE_group_e &&
		    p_u_spec->infotype != DIVY_INFOTYPE_group_e_regular) {

			/* 親がグループフォルダでもその下のコレクションでもないとき */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Found invalid parent uri. Could not insert/update resourcestate."
				"(parent_uri = %s, infotype = %d)", parent_uri, p_u_spec->infotype);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/*
		 * (2) 不整合になっているかもしれない状態を修正
		 * (note)
		 *   大抵のケースでは無駄な作業ですが、不整合を生じやすいテーブル構成
		 *   なのでチャンスがあれば状態を正しい状態に保っておいた方がよいでしょう。
		 */

		/* 状態を"有効"にする場合 */
		if (IS_EFFECTIVE_STATE(rstate)) {
			/*
			 * uri 以下のtypeに一致する"有効な"状態と
			 * 自身の状態を削除
			 */

			/* SQL文の準備 */
			stmt = dbconn->prepareStatement(dbconn,
					"DELETE FROM divy_resourcestate "
					"WHERE (rt_uri_txt = ? AND rt_type_i = ?) "
					"OR (rt_uri_txt LIKE ? "
					DIVY_DBFUNC_ESCAPE_CLAUSE
					" AND rt_type_i = ?"
					" AND rt_effective_i = 1)"
					, p);
			if (stmt->getCode(stmt) != DB_SUCCESS) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s",
					stmt->getMsg(stmt));

				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
			/* バインド */
			stmt->setString(stmt, 1, rstate->uri);
			stmt->setInt(stmt,    2, rstate->type);
			stmt->setString(stmt, 3, apr_pstrcat(p,
					stmt->escWildCard(stmt,  rstate->uri), "/%", NULL));
			stmt->setInt(stmt,    4, rstate->type);

		}
		/* 状態を"無効"にする場合 */
		else {
			/*
			 * uri 以下のtypeに一致する全ての状態を削除
			 */

			/* SQL文の準備 */
			stmt = dbconn->prepareStatement(dbconn,
					"DELETE FROM divy_resourcestate "
					"WHERE (rt_uri_txt = ?"
					" OR rt_uri_txt LIKE ? "
					DIVY_DBFUNC_ESCAPE_CLAUSE") "
					"AND rt_type_i = ?", p);
			if (stmt->getCode(stmt) != DB_SUCCESS) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s",
					stmt->getMsg(stmt));

				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
			/* バインド */
			stmt->setString(stmt, 1, rstate->uri);
			stmt->setString(stmt, 2, apr_pstrcat(p,
					stmt->escWildCard(stmt,  rstate->uri), "/%", NULL));
			stmt->setInt(stmt,    3, rstate->type);
		}

		/* SQL実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to delete divy_resourcestate. (uri = %s, type = %d) "
				"Reason: %s", rstate->uri, rstate->type, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* BOX機能をサポートしているならboxテーブルも削除する */
		if ((rstate->type == DIVY_RSTATE_TYPE_BOX) &&
				(p_u_spec->infotype == DIVY_INFOTYPE_group_e ||
		    	 p_u_spec->infotype == DIVY_INFOTYPE_group_e_regular)) {
			if (_remove_box_property(r, rstate->uri, ts_ctx)) {
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
		}

		/*
		 * (3) 状態・属性プロパティの反映
		 */

		/* 親がグループコレクションだった場合 */
		if (p_u_spec->infotype == DIVY_INFOTYPE_group_e) {

			/* 状態を"有効"にする場合 */
			if (IS_EFFECTIVE_STATE(rstate)) {

				/* "有効な"状態を追加する(さっき消したので) */
				if (_insert_resourcestate_property(r, rstate_pr, ts_ctx)) {
					ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to insert effective resourcestate."
						"(uri = %s, type = %d)", rstate_pr->uri,
						rstate_pr->type);
					ts_ctx->status |= DIVY_TRANS_ABORT;
					if (iscommit) divy_db_rollback_transaction(ts_ctx);
					return 1;
				}
			}
			/* 状態を"無効"にする場合 */
			else {
				continue;	/* もうやることはやった */
			}
		}
		/* 親がグループコレクションの下のコレクションだった場合 */
		else {
			/* 状態を"有効"にする場合 */
			if (IS_EFFECTIVE_STATE(rstate)) {
				/* 以下はコメントアウトしました。詳細は以下のコメントを見てください */
#if 0
				/* 上位コレクションが"有効"な状態を持っていない (状態連続性の破綻) */
				if (IS_INEFFECTIVE_STATE(p_rstate)) {
					/* 本来はエラーにすべきなのだが、キャッシュとタイミングにより
					 * 起き得るのでこの機会にデータ不正を正しておいた方がいい。
					 *
					 * [1] rstate->uri に"有効"な状態があった場合:
					 *  --> そもそもこのソースには"有効"な状態は付いてはならない
					 *      外部仕様。消しておいていい。
					 *
					 * [2] rstate->uri に"無効"な状態があった場合:
					 *  --> 上位コレクションが"無効"な状態であれば無効状態の
					 *      エスカレーションが発生し、このリソースが持っている
					 *      必要は無くなる。(必要がないというよりも、より積極的に
					 *      無効な状態のネストは禁止されている)。消しておいていい
					 *
					 * 以上により、ここでは何もしないことにします。
					 * より厳密に解釈したいのならば、コメントアウトを解除＆
					 * 上位コレクションの持つ状態・属性プロパティを取得してから
					 * 以下のコードを有効にして下さい。
					 */
					ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"The parent resource did not have a effective state."
						"(uri = %s)", rstate->uri);
					ts_ctx->status |= DIVY_TRANS_ABORT;
					if (iscommit) divy_db_rollback_transaction(ts_ctx);
					return 1;
				}
				/* 上位コレクションが"有効"な状態を持っていた場合 */
				else {
					/* もう何もしなくていい(外部仕様) */
				}
#endif
			}
			/* 状態を"無効"にする場合 */
			else {
				/* 上位コレクションが持っているtypeの状態・属性プロパティを取得 */
				if (_get_parent_resourcestate_property(r, rstate->uri,
							rstate->type, &p_rstate, ts_ctx)) {

					ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get resource state of parents. "
						"(uri = %s)", rstate->uri);
					ts_ctx->status |= DIVY_TRANS_ABORT;
					if (iscommit) divy_db_rollback_transaction(ts_ctx);
					return 1;
				}

				/* 上位コレクションが"有効"な状態を持っていない */
				if (IS_INEFFECTIVE_STATE(p_rstate)) {
					/* つまり無効になっているということなので
					 * やりたいことは既に完了していたことになります。
					 * ここは寛容に扱って正常とします。*/

					/* 何もしない */
				}
				/* 上位コレクションが"有効"な状態を持っていた場合 */
				else {
					/* 有効な状態の継承を拒否する */
					if (_insert_resourcestate_property(r, rstate_pr, ts_ctx)) {
						ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
							"Failed to insert ineffective resourcestate."
							"(uri = %s, type = %d)", rstate_pr->uri,
							rstate_pr->type);
						ts_ctx->status |= DIVY_TRANS_ABORT;
						if (iscommit) divy_db_rollback_transaction(ts_ctx);
						return 1;
					}
				}
			}
		}
	}
	
	/* 結果を反映する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * src_uri 以下の状態・属性プロパティをdst_uriのURIになるよう変更(移動)する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_move_resourcestate_property(request_rec *r,
							const char *src_uri,
							const char *dst_uri,
							divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;

	if (IS_EMPTY(src_uri) || IS_EMPTY(dst_uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"src_uri or dst_uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
			"UPDATE divy_resourcestate "
			"SET rt_uri_txt = "
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
			DIVY_DBFUNC_SUBSTRING("rt_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")","
			"rt_depth_i = rt_depth_i + "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_INTEGER)" "
			"WHERE rt_uri_txt = ? OR rt_uri_txt LIKE ? "
			DIVY_DBFUNC_ESCAPE_CLAUSE, p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (src_uri = %s) "
			"Reason: %s", src_uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, dst_uri);
	stmt->setString(stmt, 2, src_uri);
	stmt->setInt(stmt,    3, divy_count_dirs(dst_uri) - divy_count_dirs(src_uri));
	stmt->setString(stmt, 4, src_uri);
	stmt->setString(stmt, 5, apr_pstrcat(p, stmt->escWildCard(stmt, src_uri), "/%", NULL));

	/* SQL文の実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update divy_resourcestate. (src_uri = %s, dist_uri = %s) "
			"Reason: %s", src_uri, dst_uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* BOXテーブル */
	if (divy_support_tfbox(r)) {
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_box "
				"SET box_uri_txt = "
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
				DIVY_DBFUNC_SUBSTRING("box_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")" "
				"WHERE box_uri_txt = ? OR box_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE, p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. (divy_box) (src_uri = %s) "
					"Reason: %s", src_uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* バインド */
		stmt->setString(stmt, 1, dst_uri);
		stmt->setString(stmt, 2, src_uri);
		stmt->setString(stmt, 3, src_uri);
		stmt->setString(stmt, 4, apr_pstrcat(p, stmt->escWildCard(stmt, src_uri), "/%", NULL));

		/* SQL文の実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to update divy_box. (src_uri = %s, dist_uri = %s) "
					"Reason: %s", src_uri, dst_uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
	}

	/* 不要な資源を解放 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * uri 以下にある状態/属性プロパティを削除する。
 *
 * [ 削除対象テーブル ] divy_resourcestate
 *
 * @param r request_rec *
 * @param uri const char * 削除する状態/属性プロパティを持つURI
 * @param type int リソース種別
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_resourcestate_property(request_rec *r,
					const char *uri, int type, divy_db_transaction_ctx *ts_ctx)
{
	return _remove_resourcestate_property(r, DAV_INFINITY, uri, type, NULL);
}

/**
 * 指定したURLのBOX情報を削除する
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_box_property(request_rec *r, const char* uri, divy_db_transaction_ctx *ts_ctx)
{
	return _remove_box_property(r, uri, ts_ctx);
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passwordpolicy プロパティを取得する.
 * [ 有効性 ]
 *   パスワードポリシーがサポートされていなければ *passpolicy_pr はNULLです
 */
DIVY_DECLARE(int) divy_rdbo_get_passwordpolicy_property(request_rec *r,
					int policyid,
					divy_rdbo_passwordpolicy **passpolicy_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p 		= r->pool;
	DbConn *dbconn 		= NULL;
	DbPreparedStmt *stmt= NULL;
	DbResultSet *rset	= NULL;
	int iscommit        = 0;
	int hasNgword       = 0, idx;
	char *str;

	*passpolicy_pr = NULL;	/* 初期化 */

	/* パスワードポリシーはサポートされているか？*/
	if (!divy_support_passpolicy(r)) {
		return 0;	/* 正常に終わったものとする. */
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, 
				"SELECT "
				"pp_status_i,"
				"pp_startdt_bi,"
				"pp_minlen_i,"
				"pp_change_cycle_i,"
				"pp_probation_i,"
				"pp_change_history_i,"
				"pp_ng_userid_i,"
				"pp_ng_word_i,"
				"pp_must_symbol_i,"
				"pp_must_numeric_i,"
				"pp_must_ualpha_i,"
				"pp_must_lalpha_i,"
				"pp_cyclechar_i,"
				"pp_firstlogin_limitday_i "
				"FROM divy_passpolicy "
				"WHERE pp_policy_id_i = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	stmt->setInt(stmt, 1, policyid);

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet. Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	idx = 1;
	if (rset->next(rset) == DB_TRUE) {
		*passpolicy_pr = apr_pcalloc(p, sizeof(divy_rdbo_passwordpolicy));
		(*passpolicy_pr)->policyid            = policyid;
		(*passpolicy_pr)->status              = rset->getInt(rset, idx++);
		(*passpolicy_pr)->startdt             = rset->getBigInt(rset, idx++);
		(*passpolicy_pr)->minlen              = rset->getInt(rset, idx++);
		(*passpolicy_pr)->change_pw_cycle     = rset->getInt(rset, idx++);
		(*passpolicy_pr)->probation           = rset->getInt(rset, idx++);
		(*passpolicy_pr)->history_pw_num      = rset->getInt(rset, idx++);
		(*passpolicy_pr)->denyUseridString    = rset->getInt(rset, idx++);
		hasNgword                             = rset->getInt(rset, idx++);
		(*passpolicy_pr)->isSymbolChrNeeded   = rset->getInt(rset, idx++);
		(*passpolicy_pr)->isNumericNeeded     = rset->getInt(rset, idx++);
		(*passpolicy_pr)->isUpAlphabet        = rset->getInt(rset, idx++);
		(*passpolicy_pr)->isLoAlphabet        = rset->getInt(rset, idx++);
		(*passpolicy_pr)->denyCycleChr        = rset->getInt(rset, idx++);

		str = rset->getString(rset, idx++);
		if (IS_EMPTY(str)) {
			/* DB値がNULLであれば特別扱い */
			(*passpolicy_pr)->firstlogin_limitday = -1;
		}
		else {
			(*passpolicy_pr)->firstlogin_limitday = atoi(str);
		}
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (rset != NULL) rset->close(rset); rset = NULL;

	/* 禁則文字一覧を取得する */
	if (hasNgword) {
		stmt = dbconn->prepareStatement(dbconn, 
					"SELECT "
					"pw_word_vc "
					"FROM divy_passngword "
					"WHERE pw_policy_id_i = ?", p);
		if (stmt->getCode(stmt) != DB_SUCCESS){
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		stmt->setInt(stmt, 1, policyid);

		/* SQL実行 失敗時はエラー */
		rset = stmt->executeQuery(stmt,p);
		if (rset->getCode(rset) != DB_SUCCESS){
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbResultSet. Reason: %s", rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		(*passpolicy_pr)->ngword = NULL;
		while (rset->next(rset) == DB_TRUE) {
			if ((*passpolicy_pr)->ngword == NULL) {
				(*passpolicy_pr)->ngword = apr_pcalloc(p, sizeof(divy_rdbo_passngword));
				(*passpolicy_pr)->ngword->policyid = policyid;
				(*passpolicy_pr)->ngword->ngword_set = divy_cset_make(p);
			}
			divy_cset_set((*passpolicy_pr)->ngword->ngword_set, rset->getString(rset, 1));
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		if (rset != NULL) rset->close(rset); rset = NULL;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passwordpolicy プロパティを登録または更新する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_passwordpolicy_property(request_rec *r,
					divy_rdbo_passwordpolicy *passpolicy_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int idx;

	/* パスワードポリシー機能はサポートされているの? */
	if (!divy_support_passpolicy(r)) {
		return 0;	/* 正常終了したこととする */
	}

	if (passpolicy_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"passpolicy_pr is NULL.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Could not operation. Transaction ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* エントリを一旦削除する */
	if (divy_rdbo_remove_passwordpolicy_property(r,
									passpolicy_pr->policyid, ts_ctx)) {
		return 1;
	}

	/*
	 * divy_passpolicy テーブルへのINSERT
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO divy_passpolicy "
				"(pp_policy_id_i,"
				" pp_status_i,"
				" pp_startdt_bi,"
				" pp_minlen_i,"
				" pp_change_cycle_i,"
				" pp_probation_i,"
				" pp_change_history_i,"
				" pp_ng_userid_i,"
				" pp_ng_word_i,"
				" pp_must_symbol_i,"
				" pp_must_numeric_i,"
				" pp_must_ualpha_i,"
				" pp_must_lalpha_i,"
				" pp_cyclechar_i,"
				" pp_firstlogin_limitday_i) "
				"VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s",
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */
	idx = 1;
	stmt->setInt(stmt, idx++, passpolicy_pr->policyid);
	stmt->setInt(stmt, idx++, passpolicy_pr->status);
	stmt->setBigInt(stmt, idx++, passpolicy_pr->startdt);
	stmt->setInt(stmt, idx++, passpolicy_pr->minlen);
	stmt->setInt(stmt, idx++, passpolicy_pr->change_pw_cycle);
	stmt->setInt(stmt, idx++, passpolicy_pr->probation);
	stmt->setInt(stmt, idx++, passpolicy_pr->history_pw_num);
	stmt->setInt(stmt, idx++, passpolicy_pr->denyUseridString);
	if (passpolicy_pr->ngword != NULL) {
		stmt->setInt(stmt, idx++, 1);
	}
	else {
		stmt->setInt(stmt, idx++, 0);
	}
	stmt->setInt(stmt, idx++, passpolicy_pr->isSymbolChrNeeded);
	stmt->setInt(stmt, idx++, passpolicy_pr->isNumericNeeded);
	stmt->setInt(stmt, idx++, passpolicy_pr->isUpAlphabet);
	stmt->setInt(stmt, idx++, passpolicy_pr->isLoAlphabet);
	stmt->setInt(stmt, idx++, passpolicy_pr->denyCycleChr);
	stmt->setInt(stmt, idx++, passpolicy_pr->firstlogin_limitday);

	/* SQLを実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_passpolicy. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * 禁則文字の登録
	 */
	if (passpolicy_pr->ngword != NULL) {
		const char *val;
		divy_cset_index_t *idx;

		/*
		 * divy_passngword テーブルへのINSERT
		 */
		stmt = dbconn->prepareStatement(dbconn,
						"INSERT INTO divy_passngword "
						"(pw_policy_id_i,"
						" pw_word_vc) "
						"VALUES (?, ?)", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s",
					stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		for (idx = divy_cset_first(passpolicy_pr->ngword->ngword_set);
							idx != NULL; idx = divy_cset_next(idx)) {
			divy_cset_this(idx, &val);

			/* バインド */
			stmt->setInt(stmt,    1, passpolicy_pr->policyid);
			stmt->setString(stmt, 2, val);

			/* SQLを実行 */
			(void) stmt->executeUpdate(stmt, p);
			if (stmt->getCode(stmt) != DB_SUCCESS) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to insert divy_passngword. Reason: %s",
						stmt->getMsg(stmt));
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt);
				return 1;
			}
			/* stmt を再利用する */
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ポリシーID policyid のパスワードポリシープロパティを削除する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_passwordpolicy_property(request_rec *r,
				int policyid, divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;

	/* パスワードポリシー機能はサポートされているの? */
	if (!divy_support_passpolicy(r)) {
		return 0;	/* 正常終了したこととする */
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Could not operation. Transaction ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_passpolicy テーブルのエントリを全て削除する
	 */
	stmt = dbconn->prepareStatement(dbconn, 
					"DELETE FROM divy_passpolicy "
					"WHERE pp_policy_id_i = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt for divy_passpolicy."
				"(policyid = %d) Reason: %s", policyid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setInt(stmt, 1, policyid);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to DELETE divy_passpolicy."
				"(policyid = %d) Reason: %s", policyid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt);

	/*
	 * divy_passngword のエントリを削除
	 */
	stmt = dbconn->prepareStatement(dbconn, 
					"DELETE FROM divy_passngword "
					"WHERE pw_policy_id_i = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt for divy_passngword."
				"(policyid = %d) Reason: %s", policyid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setInt(stmt, 1, policyid);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to DELETE divy_passngword."
				"(policyid = %d) Reason: %s", policyid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* 反映を確定する */
	if (stmt != NULL) stmt->close(stmt);
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ユーザのパスワードを登録または更新する.
 * (note)
 *   ユーザポリシーステータスがNULL ならば、エラーにはしませんが、
 *   ステータスのアップデートは行ないません.
 */
DIVY_DECLARE(int) divy_rdbo_update_userpassword(request_rec *r,
					divy_rdbo_usr *usr_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	char *datebuf           = NULL;
	int support_passpolicy  = divy_support_passpolicy(r);
	time_t updatedt;

	if (usr_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"usr_pr is NULL.");
		return 1;
	}

	TRACE(p);

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Could not operation. Transaction ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_usr テーブルのパスワードと更新日付を更新する
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"UPDATE divy_usr "
				"SET "
				" usr_passwd_vc = ?,"
				" usr_update_c = ? "
				"WHERE usr_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s",
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	if (support_passpolicy && usr_pr->passpolicy_status != NULL) {
		/* パスワードの最終更新日時とユーザプロパティの
		 * 最終更新日時を一致させる */
		updatedt = usr_pr->passpolicy_status->lastchange;
	}
	else {
		updatedt = dav_divy_get_now_epoch();
	}

	if (divy_format_time_t(p, updatedt, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, usr_pr->password);
	stmt->setString(stmt, 2, datebuf);
	stmt->setString(stmt, 3, usr_pr->usrid);

	/* SQLを実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_usr. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * セッションが有効なら更新する
	 * (テーブル: divy_session)
	 */
	if (divy_support_session(r) && divy_enable_memcache(r) == 0) {

		stmt = dbconn->prepareStatement(dbconn,
					"UPDATE divy_session "
					"SET "
					"ses_usr_password_vc = ? "
					"WHERE ses_usr_usr_id_vc = ?", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. (update divy_session) "
					"(userid = %s) Reason: %s",
					usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		if (conf->encryptpassword == DIVY_ENCRYPTPASSWORD_ON) {
			divy_rdbo_session *session = divy_rdbo_get_session(r, divy_pcache_get_data(r->pool, DIVY_PCACHE_DAT_SES_SID));
			if (session != NULL) {
				ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
						"changed password = %s", session->password);
				stmt->setString(stmt, 1, session->password);
			}

		} else {
			stmt->setString(stmt, 1, usr_pr->password);
		}
		stmt->setString(stmt, 2, usr_pr->usrid);

		/* SQL実行 */
		(void)stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to executeUpdate(divy_session) "
					"(userid = %s) Reason: %s",
					usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/* パスワードポリシー機能がサポートされていなければ終わり
	 * (note) "無効" である場合には処理を続ける必要あり */
	if (!support_passpolicy) {
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		return 0;
	}

	/*
	 * ユーザの持つパスワードポリシー状態があれば反映する
	 * (divy_passpolicy テーブルの更新)
	 */
	if (usr_pr->passpolicy_status != NULL) {
		if (divy_rdbo_update_user_passwordpolicy(r, usr_pr->passpolicy_status, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update user-passwordpolicy status.(userid = %s)",
					usr_pr->usrid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * ユーザのパスワード履歴に追加する
	 * (テーブル: divy_passhistory)
	 */
	if (divy_rdbo_update_user_passhistory(r, usr_pr->usrid, usr_pr->password, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to update user-passwordpolicy history.(userid = %s)",
					usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ユーザID userid のユーザが持つパスワード履歴を取得する. 
 * [ 有効性 ]
 *   パスワードポリシーがサポートされていなければ *ph はNULLです
 */
DIVY_DECLARE(int) divy_rdbo_get_passwordhistory(request_rec *r,
					const char *userid,
					divy_rdbo_passhistory **ph,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet *rset	    = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int index;

	*ph = NULL;	/* 初期化 */

	/* パスワードポリシー機能はサポートされているの? */
	if (!divy_support_passpolicy(r)) {
		return 0;	/* 正常終了したこととする */
	}

	if (IS_EMPTY(userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"userid is EMPTY.");
		return 1;
	}


	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * パスワード履歴テーブルを検索する
	 *
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"SELECT "
				"ph_passwd_vc,"
				"ph_passwd_id_i "
				"FROM divy_passhistory "
				"WHERE ph_usr_id_vc = ? ", p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, userid);

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet. Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {
		if (*ph == NULL) {
			*ph = apr_pcalloc(p, sizeof(divy_rdbo_passhistory));
			(*ph)->usrid = (char *) userid;
			memset((*ph)->passhistory, 0, sizeof((*ph)->passhistory));
		}
		index = rset->getInt(rset, 2);
		if (index >= DIVY_MAX_PASSHISTORY_NUM) {
			continue;	/* id が12以上のパスワードは不正なのでスキップ */
		}
		(*ph)->passhistory[index] = rset->getString(rset, 1);
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (rset != NULL) rset->close(rset); rset = NULL;

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ユーザの持つパスワードポリシー状態を登録または更新する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_user_passwordpolicy(request_rec *r,
					divy_rdbo_passpolicystatus *passpolicystatus,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	int update_cnt          = 0;
	apr_pool_t *p           = r->pool;

	/* パスワードポリシー機能はサポートされているの? */
	if (!divy_support_passpolicy(r)) {
		return 0;	/* 正常終了したこととする */
	}

	if (passpolicystatus == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"passpolicystatus is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
			"UPDATE divy_passpolicystatus "
			"SET "
			" ps_last_change_bi = ?,"
			" ps_send_expiredmail_bi = ?,"
			" ps_firstlogin_bi = ?, "
			" ps_special_start_bi = ? "
			"WHERE ps_policy_id_i = ? "
			"AND ps_usr_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setBigInt(stmt, 1, passpolicystatus->lastchange);
	stmt->setBigInt(stmt, 2, passpolicystatus->sendexpiredmail);
	stmt->setBigInt(stmt, 3, passpolicystatus->firstlogin);
	stmt->setBigInt(stmt, 4, passpolicystatus->specialstart);
	stmt->setInt(stmt,    5, passpolicystatus->policyid);
	stmt->setString(stmt, 6, passpolicystatus->usrid);

	/* SQL実行 */
	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_passpolicystatus. "
				"(userid = %s) Reason: %s", passpolicystatus->usrid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 1件もなかった場合にはINSERTしてみる(こういったケースは初期データ移行の関係で起きうるため) */
	if (update_cnt == 0) {
		if (divy_rdbo_insert_user_passwordpolicy(r, passpolicystatus, ts_ctx)) {
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ユーザID src_userid のパスワードポリシーをdst_userid のユーザに
 * 移動する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_move_user_passwordpolicy(request_rec *r,
						const char *src_userid,
						const char *dst_userid,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;

	/* パスワードポリシーはサポートされているか？*/
	if (!divy_support_passpolicy(r)) {
		return 0;	/* 正常に終わったものとする */
	}

	if (IS_EMPTY(src_userid) || IS_EMPTY(dst_userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"src_userid or dst_userid is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
			"UPDATE divy_passpolicystatus "
			"SET ps_usr_usr_id_vc = ? " 
			"WHERE ps_usr_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, dst_userid);
	stmt->setString(stmt, 2, src_userid);

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update divy_passpolicystatus. "
			"(src_userid = %s, dst_uesrid = %s) Reason: %s",
			src_userid, dst_userid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */



#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * userid のユーザのパスワード履歴に追加する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_user_passhistory(request_rec *r,
							const char *userid,
							const char *new_password,
							divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet *rset	    = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	divy_rdbo_passhistory *ph = NULL;
	int index, i;
	char *passwords[DIVY_MAX_PASSHISTORY_NUM] = { 0 };
	divy_rdbo_passwordpolicy *passpolicy_pr = NULL;

	if (!divy_support_passpolicy(r)) {
		return 0;	/* 何もしない */
	}

	if (IS_EMPTY(userid) || IS_EMPTY(new_password)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"userid or new-password is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Could not operation. Transaction ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* パスワードポリシーを取得して
	 * パスワード履歴への追加が必要かどうか判定する */
	if (divy_rdbo_get_passwordpolicy_property(r, DIVY_DEFAULT_POLICY_ID,
							&passpolicy_pr, ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get password-policy.");
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/* パスワードポリシーが登録されていない or 無効である 場合 */
	if (passpolicy_pr == NULL || passpolicy_pr->status == 0) {
		/* もうする事はありません */
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		return 0;
	}

	/* パスワード履歴の処理は不要であるとマークされている場合 */
	if (passpolicy_pr->history_pw_num == 0) {
	 	/* 対象ユーザの履歴を消していおく */
		if (divy_rdbo_remove_user_passhistory(r, userid, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to delete user's password-policy history."
					"(userid = %s)", userid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
		/* もうする事はありません */
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		return 0;
	}

	/*
	 * パスワード履歴テーブルを検索する
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"SELECT "
				"ph_passwd_vc,"
				"ph_passwd_id_i "
				"FROM divy_passhistory "
				"WHERE ph_usr_id_vc = ? "
				"FOR UPDATE", p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, userid);

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQueryForUpdate(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet. Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	ph = NULL;
	while (rset->next(rset) == DB_TRUE) {
		if (ph == NULL) {
			ph = apr_pcalloc(p, sizeof(divy_rdbo_passhistory));
			ph->usrid = (char *) userid;
			memset(ph->passhistory, 0, sizeof(ph->passhistory));
		}
		index = rset->getInt(rset, 2);
		if (index >= DIVY_MAX_PASSHISTORY_NUM) {
			continue;	/* id が12以上のパスワードは不正なのでスキップ */
		}
		ph->passhistory[index] = rset->getString(rset, 1);
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (rset != NULL) rset->close(rset); rset = NULL;

	/*
	 * 対象ユーザのパスワード履歴テーブルエントリを削除
	 */
	if (divy_rdbo_remove_user_passhistory(r, userid, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete user's password-policy history.(userid = %s)",
				userid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * パスワード履歴テーブルへの追加
	 *
	 * 今回追加されるパスワードを履歴の先頭に割り当てる
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO divy_passhistory "
				"(ph_usr_id_vc,"
				" ph_passwd_vc,"
				" ph_passwd_id_i) "
				"VALUES (?, ?, ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s",
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* 0番目を現在のパスワードにして、1番目から11番目までのうち、
	 * passpolicy_pr->history_pw_num個になるまで詰める.
	 * 溢れたものは捨ててしまう */
	index = 0;
	passwords[index++] = (char *)new_password;
	if (ph != NULL) {
		for (i = 0; i < DIVY_MAX_PASSHISTORY_NUM; i++) {
			if (ph->passhistory[i] == NULL) {
				/* 基本的に歯抜け履歴は許されていないのだが、データ上
				 * 持つことは出来てしまう.　仕方が無いのでここで是正する */
				continue;	/* この番号はスキップ */
			}
			if (index + 1 > passpolicy_pr->history_pw_num ||
				index + 1 > DIVY_MAX_PASSHISTORY_NUM) {
				break;	/* ここまでしか要らない(古いのは捨てる) */
			}
			/* パスワードポリシーが無効にされていた場合、
			 * DBエントリにはnew_password と同じパスワードがあるかもしれない.
			 * このケースはアプリケーション側で修正する */
			if (strcmp(new_password, ph->passhistory[i]) == 0) {
				continue;	/* この番号はスキップ */
			}
			passwords[index++] = ph->passhistory[i];
		}
	}

	for (i = 0; i < index; i++) {
		/* バインド */
		stmt->setString(stmt, 1, userid);
		stmt->setString(stmt, 2, passwords[i]);
		stmt->setInt(stmt,    3, i);

		/* SQLを実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to insert divy_passhistory. "
					"Reason: %s", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * userid のユーザのパスワードポリシー履歴を消去する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_user_passhistory(request_rec *r,
							const char *userid, divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;

	if (!divy_support_passpolicy(r)) {
		return 0;	/* 何もしない */
	}

	if (IS_EMPTY(userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"userid is NULL.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * 対象ユーザのパスワード履歴テーブルエントリを削除
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"DELETE FROM divy_passhistory "
				"WHERE ph_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s",
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, userid);

	/* SQLを実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_usr. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * src_userid のパスワードポリシー履歴をdst_userid に移動する.
 *
 * @param r request_rec *
 * @param src_userid cosnt char * src ユーザID
 * @param dst_userid cosnt char * dst ユーザID
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_move_user_passhistory(request_rec *r,
						const char *src_userid,
						const char *dst_userid,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;

	/* パスワードポリシーはサポートされているか？*/
	if (!divy_support_passpolicy(r)) {
		return 0;	/* 正常に終わったものとする */
	}

	if (IS_EMPTY(src_userid) || IS_EMPTY(dst_userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"src_userid or dst_userid is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
			"UPDATE divy_passhistory "
			"SET ph_usr_id_vc = ? " 
			"WHERE ph_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, dst_userid);
	stmt->setString(stmt, 2, src_userid);

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update ph_usr_id_vc. "
			"(src_userid = %s, dst_uesrid = %s) Reason: %s",
			src_userid, dst_userid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */


/**
 * 指定されたgrp_pr が示すグループフォルダ以下からbordertime よりも
 * 古い更新日付を持つファイルとフォルダを削除する.
 * なお、フォルダ内にファイルがある場合には維持します.
 * (note)
 *   ts_ctx のトランザクションを継続します.
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_old_groupresources(request_rec *r,
								divy_rdbo_grp *grp_pr,
								apr_time_t bordertime,
								int isDeleteFolder,
								divy_linkedlist_t **del_filelist,
								divy_autodel_dresource **del_res,
								divy_db_transaction_ctx *ts_ctx)
{
	DbConn         *dbconn     = NULL;
	DbPreparedStmt *stmt       = NULL;
	DbResultSet    *rset       = NULL;
	apr_pool_t     *p          = r->pool;
	int iscommit               = 0;
	dav_divy_dir_conf *dconf   = dav_divy_get_dir_config(r);
	int support_trash          = divy_support_trashfolder(r);
	int support_extstatus      = divy_support_extenduserstatus(r);
	int support_box            = divy_support_tfbox(r);
	const char *trash_uri      = NULL;
	divy_rdbo_usr *p_uquota    = NULL;
	divy_linkedlist_t *clist   = NULL;
	divy_rdbo_usr uquota       = { 0 };
#ifdef DIVY_SUPPORT_EXTENDQUOTA
	divy_rdbo_diskquota squota = { 0 };
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
	divy_rdbo_resourcetype resourcetype;
	apr_int64_t u_usedst, u_usedres;
	int ret, found_folder;
	char *userid;
	apr_time_t bordertime_sec;	/* 秒単位のapr_time_t */
	apr_time_t sec;
	divy_autodel_dresource *res = NULL;
	char *esc_grpcol_uri        = NULL;
	divy_autodel_dresource *dres= NULL;
	divy_rdbo_usr ldapusr       = { 0 };

	if (grp_pr == NULL || bordertime < APR_TIME_C(0)) {
		return 0;	/* 何もやることはない */
	}
	bordertime_sec = apr_time_sec(bordertime);	/* 秒に変換 */

	*del_filelist = NULL;	/* 初期化 */
	*del_res      = NULL;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * (0) 対象データを全てDB ロックする.
	 * (note)
	 *   SELECT FOR UPDATE とJOINを併用することができなかったため
	 *   無駄ですがロックするためだけに更新を行います.
	 *   値は不要ですので取得しません.
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				"1 "
				"FROM dav_resource "
				"WHERE rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE" "
				"AND ((rs_resourcetype_i = 0 AND rs_get_lastmodified_bi < ?) "
				"OR rs_resourcetype_i = 1) " 
				"FOR UPDATE", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* uri に含まれるワイルドカード文字をエスケープする */
	esc_grpcol_uri = apr_pstrcat(p, stmt->escWildCard(stmt, grp_pr->grpcol_uri), "/%", NULL);
	stmt->setString(stmt, 1, esc_grpcol_uri);
	stmt->setBigInt(stmt, 2, bordertime_sec);

	/* SQLの実行 */
	rset = stmt->executeQueryForUpdate(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select dav_resource for quota. Reason: %s", rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* 不要なので閉じる */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * (1) グループコレクション以下のbordertime_sec よりも更新日付が
	 *     古いファイル&フォルダの情報及び全Quota合算値を取得してDB ロック
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				"rs.rs_resourcetype_i,"
				"rs.rs_physical_path_txt,"
				"rs.rs_get_cont_len_bi,"
				"rs.rs_dispname_vc,"
				"rs.rs_get_cont_type_vc,"
				"rs.rs_lastmodifier_usr_id_vc,"
				"rs.rs_get_lastmodified_bi, "
				"u.usr_fullname_vc AS rs_lastmodifier_vc "
				"FROM dav_resource rs "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"LEFT JOIN divy_usr u "
				"ON rs.rs_lastmodifier_usr_id_vc = u.usr_usr_id_vc "
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				", divy_usr u "
#endif
#if defined(DIVY_DBMS_ORACLE) /* oracle */
				"WHERE rs.rs_lastmodifier_usr_id_vc = u.usr_usr_id_vc(+) "
				"AND rs.rs_uri_txt LIKE ? "
#else
				"WHERE rs.rs_uri_txt LIKE ? "
#endif
				DIVY_DBFUNC_ESCAPE_CLAUSE" "
				"AND ((rs.rs_resourcetype_i = 0 AND rs.rs_get_lastmodified_bi < ?) "
				"OR rs.rs_resourcetype_i = 1) ", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setString(stmt, 1, esc_grpcol_uri);
	stmt->setBigInt(stmt, 2, bordertime_sec);

	/* SQLの実行 */
	rset = stmt->executeQueryForUpdate(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select dav_resource for quota. Reason: %s", rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

#ifdef DIVY_SUPPORT_EXTENDQUOTA
	squota.usedst  = APR_INT64_C(0);
	squota.usedres = APR_INT64_C(0);
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
	found_folder = 0;

	while (rset->next(rset) == DB_TRUE) {

		resourcetype = rset->getInt(rset, 1);
		if (resourcetype == DIVY_TYPE_RESOURCE) {
			if (*del_filelist == NULL) {
				clist = *del_filelist = apr_pcalloc(p, sizeof(divy_linkedlist_t));
			}
			else {
				clist->next = apr_pcalloc(p, sizeof(divy_linkedlist_t));
				clist = clist->next;
			}
			/* 相対パスの記録 */
			clist->val = rset->getString(rset, 2);

			/* リソースのプロパティだけ記録する(コレクションは削除リストかどうか不明なので) */
			if (*del_res == NULL) {
				*del_res = res = apr_pcalloc(p, sizeof(divy_autodel_dresource));
			}
			else {
				res->next = apr_pcalloc(p, sizeof(divy_autodel_dresource));
				res = res->next;
			}
			res->getcontentlength    = rset->getBigInt(rset, 3);
			res->displayname         = rset->getString(rset, 4);
			res->getcontenttype      = rset->getString(rset, 5);
			res->lastmodifier_userid = rset->getString(rset, 6);
			sec                      = rset->getBigInt(rset, 7);	/* DB には秒で入っている */
			res->getlastmodified     = apr_time_from_sec(sec);		/* sec -> micro sec */
			res->lastmodifier        = rset->getString(rset, 8);

			/* LDAP から探してみる */
			if (IS_FILLED(dconf->ldapmapfullnameattr) && divy_util_ldap_get_user_property(r, p, res->lastmodifier_userid, &ldapusr) == TF_LDAP_TRUE) {
				res->lastmodifier = ldapusr.fullname;
			}

			res->deletion            = APR_INT64_C(0);	/* 後で */
			res->next                = NULL;
#ifdef DIVY_SUPPORT_EXTENDQUOTA
			/* 削除リソースのSUMをとる(システムQuotaありの場合) */
			squota.usedst += res->getcontentlength;
			squota.usedres++;
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
		}
		else {
			found_folder++;
		}
	}
	if (rset != NULL) rset->close(rset);	rset = NULL;
	if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

	/* 対象ファイルが何もなかった場合 */
	if (*del_res == NULL) {
		/* ファイルもフォルダも全くなければ正常終了する */
		if (found_folder == 0) {
			if (iscommit) divy_db_commit_transaction(ts_ctx);
			return 0;	/* 正常とする */
		}

		/* ファイルが無かった場合、フォルダ削除を行うかどうか判定 */
		if (isDeleteFolder && found_folder) {
			dres = NULL;
			if (_remove_old_groupfolders(r, grp_pr, bordertime_sec, &dres, ts_ctx)) {

				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
			if (dres != NULL) *del_res = dres;
			if (iscommit) divy_db_commit_transaction(ts_ctx);
			return 0;
		}
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		return 0;
	}

	/*
	 * (2) グループコレクション以下のユーザ毎の消費Diskとファイル数を求める
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"SELECT "
				"rs_lastmodifier_usr_id_vc,"
				"SUM(rs_get_cont_len_bi) AS len, "
				"COUNT(rs_rs_id_c) AS cnt "
				"FROM dav_resource "
				"WHERE rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND rs_resourcetype_i=0 "
				" AND rs_get_lastmodified_bi < ? "
				"GROUP BY rs_lastmodifier_usr_id_vc", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setString(stmt, 1, esc_grpcol_uri);
	stmt->setBigInt(stmt, 2, bordertime_sec);

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select for quota. Reason: %s", rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {

		/* 値の取得 */
		userid    = rset->getString(rset, 1);
		u_usedst  = rset->getBigInt(rset, 2); 
		u_usedres = rset->getBigInt(rset, 3);

		/* 0バイト, 0個だった場合 削除処理はやらなくていい */
		if (u_usedst == APR_INT64_C(0) && u_usedres == APR_INT64_C(0)) {
			continue;	/* 次へ */
		}

		if (p_uquota == NULL) {
			p_uquota = &uquota;
		}
		else {
			p_uquota->next = apr_pcalloc(p, sizeof(divy_rdbo_usr));
			p_uquota = p_uquota->next;
		}
		p_uquota->usrid   = userid;
		p_uquota->usedst  = APR_INT64_C(-1) * u_usedst;
		p_uquota->usedres = APR_INT64_C(-1) * u_usedres;
		p_uquota->next    = NULL;
	}
	if (rset != NULL) rset->close(rset);	rset = NULL;
	if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

	/*
	 * (3) Dead プロパティを削除する
	 *     (dav_resource を消すよりも前にやること)
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"DELETE FROM dav_dead_property "
				"WHERE dp_rs_id_c IN ("
				"SELECT rs_rs_id_c FROM dav_resource "
				"WHERE rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND rs_resourcetype_i=0"
				" AND rs_get_lastmodified_bi < ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setString(stmt, 1, esc_grpcol_uri);
	stmt->setBigInt(stmt, 2, bordertime_sec);

	/* 更新処理の実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to DELETE dead property. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * (4) ごみ箱プロパティの削除
	 */
	if (support_trash) {
		/* ごみ箱URIの算出 */
		trash_uri = divy_build_group_trash_uri(p, dav_divy_get_root_uri(r),
					dav_divy_extract_finalpath_segment(p, grp_pr->grpcol_uri));

		stmt = dbconn->prepareStatement(dbconn, 
					"DELETE FROM divy_trash_info "
					"WHERE tr_rs_id_c IN "
					"(SELECT rs_rs_id_c FROM dav_resource "
					" WHERE rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
					" AND rs_resourcetype_i=0"
					" AND rs_get_lastmodified_bi < ?)", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. (trash_uri = %s) "
					"Reason: %s", trash_uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		stmt->setString(stmt, 1, apr_pstrcat(p, stmt->escWildCard(stmt, trash_uri), "/%", NULL));
		stmt->setBigInt(stmt, 2, bordertime_sec);

		/* 更新処理の実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to execute DELETE trash property. (trash_uri = %s) "
					"Reason: %s", trash_uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/*
	 * (5) 状態/属性プロパティの削除
	 */
	if (support_extstatus) {
		stmt = dbconn->prepareStatement(dbconn, 
					"DELETE FROM divy_resourcestate "
					"WHERE rt_uri_txt IN "
					"(SELECT rs_uri_txt FROM dav_resource "
					" WHERE rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
					" AND rs_resourcetype_i=0"
					" AND rs_get_lastmodified_bi < ?)", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		stmt->setString(stmt, 1, esc_grpcol_uri);
		stmt->setBigInt(stmt, 2, bordertime_sec);

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to DELETE resourcestate property. Reason: %s", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* (5-1) BOXの削除 */
		if (support_box) {
			stmt = dbconn->prepareStatement(dbconn,
					"DELETE  FROM divy_box "
					" WHERE box_uri_txt IN "
					"(SELECT rs_uri_txt from dav_resource "
					" WHERE rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
					" AND rs_resourcetype_i=0"
					" AND rs_get_lastmodified_bi < ?)", p);

			if (stmt->getCode(stmt) != DB_SUCCESS) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
			stmt->setString(stmt, 1, esc_grpcol_uri);
			stmt->setBigInt(stmt, 2, bordertime_sec);

			(void) stmt->executeUpdate(stmt, p);
			if (stmt->getCode(stmt) != DB_SUCCESS) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to DELETE box property. Reason: %s", stmt->getMsg(stmt));
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt);
				return 1;
			}
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		}
	}

	/*
	 * (6)ロック情報を削除する
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"DELETE FROM dav_lock "
				"WHERE lk_uri_txt IN "
				"(SELECT rs_uri_txt FROM dav_resource "
				" WHERE rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND rs_resourcetype_i = 0"
				" AND rs_get_lastmodified_bi < ?)", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	stmt->setString(stmt, 1, esc_grpcol_uri);
	stmt->setBigInt(stmt, 2, bordertime_sec);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to DELETE lock information. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * (7) 開封通知を削除する
	 */
	if (divy_support_confirmreading(r)) {
		stmt = dbconn->prepareStatement(dbconn, 
					"DELETE FROM divy_confirmreading "
					"WHERE cr_uri_txt IN "
					"(SELECT rs_uri_txt FROM dav_resource "
					" WHERE rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
					" AND rs_resourcetype_i=0"
					" AND rs_get_lastmodified_bi < ?)", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		stmt->setString(stmt, 1, esc_grpcol_uri);
		stmt->setBigInt(stmt, 2, bordertime_sec);

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to DELETE confirmreading property. Reason: %s", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/* 
	 * (8) リソース情報を削除する
	 */
	stmt = dbconn->prepareStatement(dbconn, 
			"DELETE FROM dav_resource "
			"WHERE rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
			" AND rs_resourcetype_i = 0"
			" AND rs_get_lastmodified_bi < ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setString(stmt, 1, esc_grpcol_uri);
	stmt->setBigInt(stmt, 2, bordertime_sec);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to DELETE resource information. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DIVY_SUPPORT_EXTENDQUOTA
	/*
	 * (9) システムQuotaの更新
	 */
	/* 減算する値にシステムQuotaを変形 */
	squota.usedst  = APR_INT64_C(-1) * squota.usedst;
	squota.usedres = APR_INT64_C(-1) * squota.usedres;

	ret = _change_used_sysquota(r, &squota, ts_ctx);
	if (ret) {
		if (ret != DIVY_STCODE_QUOTAOVER) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to update system quota.(root_uri = %s)", REPLACE_NULL(squota.uri));
		}
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return ret;
	}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

	/*
	 * (10) ユーザQuotaの更新
	 *
	 * "更新者"にQuota を返す(ディスク使用量と使用ファイル数を減らす)
	 * 削除対象がコレクションで1つ以上の内包リソースが存在していた場合
	 */
	if (IS_FILLED(uquota.usrid)) {

		/* 全ユーザのQuota情報を更新する(Quotaを返してもらう) */
		ret = _change_used_userquota(r, &uquota, ts_ctx);
		if (ret) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to reduce disk quota.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return ret;
		}
	}

	/*
	 * (11) フォルダ削除を行うか?
	 */
	if (isDeleteFolder && found_folder) {

		dres = NULL;
		if (_remove_old_groupfolders(r, grp_pr, bordertime_sec, &dres, ts_ctx)) {

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		if (*del_res == NULL) {
			*del_res = dres;
		}
		else {
			for (res = *del_res; res->next != NULL; res = res->next);
			res->next = dres;
		}
	}

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * userid が示すユーザのプライベートコレクションを作成する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_create_private_collection(request_rec *r,
								const char *userid,
								divy_rdbo_resource *rdb_r,
								divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p = r->pool;
	int iscommit  = 0;

	/* 入力不正のチェック */
	if (IS_EMPTY(userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"userid is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;

	/* リソースID の採番 */
	if (divy_rdbo_create_rsid(r, &rdb_r->rsid, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get rsid. (userid = %s)", userid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/* プライベートコレクションの絶対URIを取得 */
	rdb_r->uri = divy_build_user_uri(p, dav_divy_get_root_uri(r), userid);
	rdb_r->displayname = (char *) userid;	/* ユーザID がdisplayname */

	/* 残りの値を詰める */
	_fill_default_collection_property(r, rdb_r);

	/*
	 * dav_resource テーブルへの新規追加
	 */
	if (_insert_property(r, rdb_r, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert dav_resource.(userid = %s)", userid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * grpname が示すグループコレクションを生成する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_create_group_collection(request_rec *r,
								const char *grpname,
								divy_rdbo_resource *rdb_r,
								char **groupid,
								divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p = r->pool;
	int iscommit  = 0;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char *root_uri     = dav_divy_get_root_uri(r);

	if (rdb_r == NULL || IS_EMPTY(grpname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"rdb_r or groupname is EMPTY.");
		return 1;
	}

	*groupid = NULL;	/* 初期化 */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;

	/* リソースID・グループID の取得 */
	if (divy_rdbo_create_rsgrpid(r, &rdb_r->rsid, groupid, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get rsid or grpid. (groupname = %s)", grpname);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/* グループコレクションの絶対URIを取得 */
	if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON ||
	    dconf->syncgrpuri == DIVY_SYNCGRPURI_MIDDLE) {
		rdb_r->uri = divy_build_group_uri(p, root_uri, grpname);
	}
	else {
		/* グループIDを使用 */
		rdb_r->uri = divy_build_group_uri(p, root_uri, *groupid);
	}
	rdb_r->displayname = apr_pstrdup(p, grpname);	/* グループ名 がdisplayname */

	/* 残りの値を詰める */
	_fill_default_collection_property(r, rdb_r);

	/*
	 * dav_resource テーブルへの新規追加
	 */
	if (_insert_property(r, rdb_r, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert dav_resource.(groupname = %s)", grpname);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * uri のごみ箱を作成する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_create_trash_property(request_rec *r,
								const char *uri, divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p = r->pool;
	int iscommit  = 0;
	divy_rdbo_resource trash_rdb_r = { 0 };

	/* 入力不正のチェック */
	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;

	/* リソースIDの採番 */
	if (divy_rdbo_create_rsid(r, &trash_rdb_r.rsid, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get rsid. (uri = %s)", uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	trash_rdb_r.uri         = (char *) uri; 
	trash_rdb_r.displayname = DIVY_TRASHFOLDER_NAME;

	/* 残りの値を詰める */
	_fill_default_collection_property(r, &trash_rdb_r);

	/* Content-Typeをごみ箱のそれに入れ替える */
	trash_rdb_r.getcontenttype = DIVY_TRASHFOLDER_CONTENTTYPE;

	/* dav_resource テーブルへの新規追加 */
	if (_insert_property(r, &trash_rdb_r, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert dav_resource.(uri = %s)", uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passpolicystatus のパスワードポリシーを登録する
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_user_passwordpolicy(request_rec *r,
						divy_rdbo_passpolicystatus *passpolicystatus,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;

	/* パスワードポリシー機能はサポートされているの? */
	if (!divy_support_passpolicy(r)) {
		return 0;	/* 正常終了したこととする */
	}

	if (passpolicystatus == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"passpolicystatus is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
			"INSERT INTO divy_passpolicystatus"
			"(ps_policy_id_i,"
			" ps_usr_usr_id_vc,"
			" ps_send_expiredmail_bi,"
			" ps_last_change_bi,"
			" ps_firstlogin_bi,"
			" ps_special_start_bi) "
			"VALUES (?, ?, ?, ?, ?, ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setInt(stmt,    1, passpolicystatus->policyid);
	stmt->setString(stmt, 2, passpolicystatus->usrid);
	stmt->setBigInt(stmt, 3, passpolicystatus->sendexpiredmail);
	stmt->setBigInt(stmt, 4, passpolicystatus->lastchange);
	stmt->setBigInt(stmt, 5, passpolicystatus->firstlogin);
	stmt->setBigInt(stmt, 6, passpolicystatus->specialstart);

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_passpolicystatus. "
				"(userid = %s) Reason: %s", passpolicystatus->usrid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * userid が示すユーザのパスワードポリシー(状態)を削除する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_user_passwordpolicy(request_rec *r,
						const char *userid, divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;

	/* パスワードポリシー機能はサポートされているの? */
	if (!divy_support_passpolicy(r)) {
		return 0;	/* 正常終了したこととする */
	}

	if (IS_EMPTY(userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"userid is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_passpolicystatus "
			"WHERE ps_usr_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, userid);

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_passpolicystatus. (userid = %s) Reason: %s",
			userid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * confirmreading_pr が表す開封通知を追加する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_confirmreading(request_rec *r,
					const divy_rdbo_confirmreading *cr_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;

	TRACE(p);

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Could not operation. "
				"Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_confirmreading テーブルへの新規追加
	 */
	stmt = dbconn->prepareStatement(dbconn, 
			"INSERT INTO divy_confirmreading ("
			"  cr_uri_txt"
			", cr_usr_id_vc"
			", cr_creation_bi)"
			" VALUES (?, ?, ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt (uri = %s, usrid = %s) Reason: %s",
				cr_pr->uri, cr_pr->userid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1,  cr_pr->uri);
	stmt->setString(stmt, 2,  cr_pr->userid);
	stmt->setBigInt(stmt, 3,  cr_pr->creationdate);

	/* SQLの実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_confirmreading. "
				"(uri = %s, usrid = %s) Reason: %s",
				cr_pr->uri, cr_pr->userid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * uri, userid が示す開封通知を削除する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_confirmreading(request_rec *r,
					const char *uri, const char *userid,
					int uri_scope,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;
	char *sql = "";

	if (IS_EMPTY(uri) && IS_EMPTY(userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"uri and userid is EMPTY.");
		return 1;
	}

	if (uri_scope != 0 && uri_scope != DAV_INFINITY) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Invalid uri_scope found.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	if (IS_EMPTY(uri)) {
		if (IS_FILLED(userid)) {
			sql =		"DELETE FROM divy_confirmreading "
						"WHERE cr_usr_id_vc = ?";
		}
	}
	else {
		if (IS_FILLED(userid)) {
			if (uri_scope == DAV_INFINITY) {
				sql =	"DELETE FROM divy_confirmreading "
						" WHERE cr_usr_id_vc = ?"
						" AND (cr_uri_txt = ?"
						" OR cr_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE")";
			}
			else {
				sql =	"DELETE FROM divy_confirmreading "
						" WHERE cr_usr_id_vc = ?"
						" AND cr_uri_txt = ?";
			}
		}
		else {
			if (uri_scope == DAV_INFINITY) {
				sql =	"DELETE FROM divy_confirmreading "
						" WHERE cr_uri_txt = ?"
						" OR cr_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE;
			}
			else {
				sql =	"DELETE FROM divy_confirmreading "
						" WHERE cr_uri_txt = ?";
			}
		}
	}

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	if (IS_EMPTY(uri)) {
		if (IS_FILLED(userid)) {
			stmt->setString(stmt, 1, userid);
		}
	}
	else {
		if (IS_FILLED(userid)) {
			stmt->setString(stmt, 1, userid);
			stmt->setString(stmt, 2, uri);

			if (uri_scope == DAV_INFINITY) {
				stmt->setString(stmt, 3, apr_pstrcat(p,
							stmt->escWildCard(stmt, uri), "/%", NULL));
			}
		}
		else {
			stmt->setString(stmt, 1, uri);
			if (uri_scope == DAV_INFINITY) {
				stmt->setString(stmt, 2, apr_pstrcat(p,
							stmt->escWildCard(stmt, uri), "/%", NULL));
			}
		}
	}

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_confirmreading. (uri = %s, userid = %s) Reason: %s",
			REPLACE_NULL(uri), REPLACE_NULL(userid), stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * day 日以上設定されていた開封通知を全て解除する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_release_expired_confirmreading(request_rec *r,
					int day, divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;
	apr_time_t bordertime_sec;	/* 秒単位のapr_time_t */
	apr_time_t now          = apr_time_now();

	if (day == 0) {
		return 0;	/* 何もしない */
	}

	/* 境界時間(エポックタイム) の算出 */
	bordertime_sec = apr_time_sec(now) - day * 24 * 3600;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
					"DELETE FROM divy_confirmreading "
					"WHERE cr_creation_bi < ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setBigInt(stmt, 1, bordertime_sec);

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to release divy_confirmreading. Reason: %s",
			stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * uri, userid が示す開封通知を取得する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_confirmreading(request_rec *r,
					const char *uri, const char *userid,
					divy_rdbo_confirmreading **cr_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	divy_sbuf *sql_buf      = NULL;
	int idx;
	divy_rdbo_confirmreading *cr = NULL;

	*cr_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(uri) && IS_EMPTY(userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"uri and userid is EMPTY");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の組み立て */
	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT"
				" cr_uri_txt"
				",cr_usr_id_vc"
				",cr_creation_bi"
				" FROM divy_confirmreading");

	if (IS_EMPTY(uri)) {
		if (IS_FILLED(userid)) {
			divy_sbuf_append(sql_buf,
						" WHERE cr_usr_id_vc = ?");
		}
		else {	/* このケースは既に除外してあります */
		}
	}
	else {
		divy_sbuf_append(sql_buf,
					" WHERE cr_uri_txt = ?");
		if (IS_FILLED(userid)) {
			divy_sbuf_append(sql_buf,
					" AND cr_usr_id_vc = ?");
		}
	}

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt for divy_confirmreading. "
				"(uri = %s, userid = %s) Reason: %s",
				REPLACE_NULL(uri), REPLACE_NULL(userid), stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* バインド */
	idx = 1;
	if (IS_EMPTY(uri)) {
		if (IS_FILLED(userid)) {
			stmt->setString(stmt, idx++, userid);
		}
	}
	else {
		stmt->setString(stmt, idx++, uri);
		if (IS_FILLED(userid)) {
			stmt->setString(stmt, idx++, userid);
		}
	}

	/* SQLの実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get divy_confirmreading table. "
			"(uri = %s, userid = %s) Reason: %s",
			REPLACE_NULL(uri), REPLACE_NULL(userid), rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 
	 * 結果の取得
	 */
	while (rset->next(rset) == DB_TRUE) {
		if (*cr_pr == NULL) {
			*cr_pr = cr = apr_pcalloc(p, sizeof(divy_rdbo_confirmreading));
		}
		else {
			cr->next = apr_pcalloc(p, sizeof(divy_rdbo_confirmreading));
			cr       = cr->next;
		}

		cr->uri          = rset->getString(rset, 1);
		cr->userid       = rset->getString(rset, 2);
		cr->creationdate = (time_t) rset->getBigInt(rset, 3);
		cr->next         = NULL;
	}

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * uri が示すリソースに対し開封通知を仕掛けているユーザのユーザプロパティ一覧を
 * 取得する.
 * (note)
 *   非アクティブユーザのプロパティは取得しません.
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_confirmreading_users(request_rec *r,
					const char *uri, divy_rdbo_usr **usr_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	divy_rdbo_usr *usr, *tmp_usr;
	int support_extstatus   = divy_support_extenduserstatus(r);
	divy_sbuf *sql_buf      = NULL;
	int idx;

	*usr_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"uri is EMPTY");
		return 1;
	}

	if (!divy_support_confirmreading(r)) {
		return 0;	/* 未サポートならば開封通知は有り得ない */
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の作成 */
	divy_sbuf_create(p, &sql_buf, 512);
	divy_sbuf_append(sql_buf,
				"SELECT"
				" u.usr_usr_id_vc"
				",u.usr_fullname_vc"
				",usr_mailaddr_vc");
	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
				",u.usr_expiration_bi"
				",u.usr_extended_status_c");
	}
	divy_sbuf_append(sql_buf,
				" FROM divy_usr u"
				" INNER JOIN divy_confirmreading cr"
				" ON u.usr_usr_id_vc = cr.cr_usr_id_vc"
				" WHERE cr.cr_uri_txt = ?");

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt for divy_confirmreading. "
				"(uri = %s) Reason: %s", uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, uri);

	/* SQLの実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get divy_confirmreading table. "
			"(uri = %s) Reason: %s", uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 
	 * 結果の取得
	 */
	usr = NULL;
	while (rset->next(rset) == DB_TRUE) {
		tmp_usr = apr_pcalloc(p, sizeof(divy_rdbo_usr));
		tmp_usr->next = NULL;

		idx = 1;
		tmp_usr->usrid    = rset->getString(rset, idx++);
		tmp_usr->fullname = rset->getString(rset, idx++);
		tmp_usr->mailaddr = rset->getString(rset, idx++);
		if (support_extstatus) {
			tmp_usr->expiration  = (time_t) rset->getBigInt(rset, idx++);
			tmp_usr->extstatus	 = divy_rdbo_parse_extstatus(p, rset->getString(rset, idx++), EXTSTATUS_TYPE_USR);

			/* 非アクティブユーザはリストに追加しない */
			if (!divy_rdbo_is_active_user(tmp_usr->extstatus)) {
				continue;
			}

			/* 期限切れユーザもリストに追加しない */
			if (tmp_usr->expiration != 0 && tmp_usr->expiration < apr_time_sec(r->request_time)) {
				continue;
			}
		}

		if (*usr_pr == NULL) {
			*usr_pr = usr = tmp_usr;
		}
		else {
			usr->next = tmp_usr;
			usr       = usr->next;
		}
	}

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * src_userid が持っていた開封通知をest_userid のユーザに付け替える.
 *
 */
DIVY_DECLARE(int) divy_rdbo_move_confirmreading_userid(request_rec *r,
					const char *src_userid, const char *dst_userid,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;

	if (IS_EMPTY(src_userid) || IS_EMPTY(dst_userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"src_userid or dst_userid is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn,
					"UPDATE divy_confirmreading "
					"SET cr_usr_id_vc = ? "
					"WHERE cr_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_userid, dst_userid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, dst_userid);
	stmt->setString(stmt, 2, src_userid);

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_confirmreading."
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_userid, dst_userid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * src_uri が持っていた開封通知をdst_uri のリソースに付け替える.
 *
 */
DIVY_DECLARE(int) divy_rdbo_move_confirmreading_uri(request_rec *r,
					const char *src_uri, const char *dst_uri,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;

	if (IS_EMPTY(src_uri) || IS_EMPTY(dst_uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"src_uri or dst_uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn,
					"UPDATE divy_confirmreading "
					"SET cr_uri_txt = "
					DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
					DIVY_DBFUNC_SUBSTRING("cr_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")" "
					" WHERE cr_uri_txt = ?"
					" OR cr_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(src_uri = %s, dst_uri = %s) Reason: %s",
				src_uri, dst_uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, dst_uri);
	stmt->setString(stmt, 2, src_uri);
	stmt->setString(stmt, 3, src_uri);
	stmt->setString(stmt, 4, apr_pstrcat(p,
				stmt->escWildCard(stmt, src_uri), "/%", NULL));

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_confirmreading."
				"(src_uri = %s, dst_uri = %s) Reason: %s",
				src_uri, dst_uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * private 関数 _insert_grpmem の公開版
 *
 * @param r request_rec *
 * @param grpid const char * グループID
 * @param usrid const char * ユーザID
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_grpmem(request_rec *r,
					const char *grpid, const char *usrid,
					divy_db_transaction_ctx *ts_ctx)
{
	return _insert_grpmem(r, grpid, usrid, ts_ctx);
}

/*------------------------------------------------------------------------------
  Define private functions 
  ----------------------------------------------------------------------------*/
/**
 * 指定されたrdb_r のuriとrsid を使って、dav_dead_property から
 * 一致するデータをDELETE する。
 *
 * delete 対象テーブル: dav_dead_property
 * (note) rdb_r 必須項目
 * 	uri, rsid, resourcetype
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 * (note)
 * 	この関数は、Dead プロパティが定義されているリソースorコレクション
 * 	そのものが削除される時に呼び出されると仮定しています。
 * 	そのため、削除対象に定義されていたプロパティを全て削除します。
 *
 * @param r request_rec *
 * @param rdb_r const divy_rdbo_resource *
 * @param ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理結果 (0: 成功 / 1: 失敗)
 */
static int _remove_hierarchy_dead_property(request_rec *r,
                                     const divy_rdbo_resource *rdb_r,
			             divy_db_transaction_ctx *ts_ctx)
{
	DbConn *dbconn       = NULL;
	DbPreparedStmt *stmt = NULL;
	apr_pool_t *p        = r->pool;
	int iscommit         = 0;	/* トランザクションを確定するかどうか */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* 削除対象がリソースの場合 */
	if (rdb_r->resourcetype == DIVY_TYPE_RESOURCE) {
		stmt = dbconn->prepareStatement(dbconn, 
			"DELETE FROM dav_dead_property "
			"WHERE dp_rs_id_c = ?", p);
	}
	/* 削除対象がコレクションの場合 */
	else {
		stmt = dbconn->prepareStatement(dbconn, 
			"DELETE FROM dav_dead_property "
			"WHERE dp_rs_id_c IN ("
			"SELECT rs_rs_id_c FROM dav_resource "
			"WHERE rs_rs_id_c = ? OR rs_uri_txt LIKE ? "
			DIVY_DBFUNC_ESCAPE_CLAUSE")", p);
	}

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for dav_dead_property."
			"(uri = %s, rsid = %s) Reason: %s",
			rdb_r->uri, rdb_r->rsid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setString(stmt, 1, rdb_r->rsid);

	if (rdb_r->resourcetype != DIVY_TYPE_RESOURCE) {
		/* uri に含まれるワイルドカード文字をエスケープする */
		stmt->setString(stmt, 2, apr_pstrcat(p,
				stmt->escWildCard(stmt, rdb_r->uri), "/%", NULL));
	}

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to DELETE dav_dead_property."
			"(uri = %s, rsid = %s) Reason: %s",
			rdb_r->uri, rdb_r->rsid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* 使用した資源の片付け */
	if (stmt != NULL) stmt->close(stmt);
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたrdb_r のrsid を使って、dav_link_collection から
 * 一致するデータをDELETE する。
 * 削除するデータは、ソースコレクション側と共有コレクション側の両方です。
 *
 * delete 対象テーブル: dav_link_collection
 * (note) rdb_r 必須項目: rsid, uri
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 *
 * @param r request_rec *
 * @param rdb_r const divy_rdbo_resource *
 * @param ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理結果 (0: 成功 / 1: 失敗)
 */
static int _remove_hierarchy_sharedcollection_property(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn *dbconn       = NULL;
	DbPreparedStmt *stmt = NULL;
	apr_pool_t *p        = r->pool;
	int iscommit         = 0;	/* トランザクションを確定するかどうか */
	char *rsid           = rdb_r->rsid;
	char *uri	     = rdb_r->uri;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* dav_link_collection からエントリの削除 */
	stmt = dbconn->prepareStatement(dbconn, 
			"DELETE FROM dav_link_collection "
			"WHERE lc_uri_txt = ? OR lc_share_rs_id_c = ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for dav_link_collection."
			"(rsid = %s, uri = %s) \nReason: %s",
			rsid, uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	stmt->setString(stmt, 1, uri);
	stmt->setString(stmt, 2, rsid);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to DELETE dav_link_collection."
			"(rsid = %s, uri = %s)\n Reason: %s",
			rsid, uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* 使用した資源の片付け */
	if (stmt != NULL) stmt->close(stmt);
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたuri, userid が持つメール監視状態を削除する。
 * (note)
 * 	・is_logicaldel = 1 の時には、mw_l_del_flag を1 にUPDATEするだけ。
 *
 * delete, update 対象テーブル: divy_mailwatch
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 *
 * @param r request_rec *
 * @param uri const char * 削除対象コレクション
 * @param userid const char * 削除を行うユーザ
 * @param delmode int 削除モード
 * 			DIVY_REMWATCH_NORMAL      : 物理削除 & usrid とuriを使用して削除
 * 			DIVY_REMWATCH_LOGICAL     : 論理削除 & usrid とuriを使用して更新
 * 			DIVY_REMWATCH_IGNORE_USER : usrid を削除/更新条件から外す
 * 			DIVY_REMWATCH_IGNORE_URI  : uri を削除/更新条件から外す
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _remove_mailwatch_property(request_rec *r,
					const char *uri,
					const char *usrid,
					int delmode,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn *dbconn       = NULL;
	DbPreparedStmt *stmt = NULL;
	apr_pool_t *p        = r->pool;
	int iscommit         = 0;
	int update_cnt	     = 0;
	char *sql;

	TRACE(p);

	/* 入力値のチェック */

	/* ユーザIDとURIを処理に利用する場合 */
	if (!(delmode & DIVY_REMWATCH_IGNORE_USER) && !(delmode & DIVY_REMWATCH_IGNORE_URI)) {
		if (IS_EMPTY(uri) || IS_EMPTY(usrid)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"uri or userid is EMPTY.");
			return 1;
		}
	}
	/* ユーザIDを利用しない場合 ( = uri を使う) */
	else if (delmode & DIVY_REMWATCH_IGNORE_USER) {
		if (IS_EMPTY(uri)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"uri is EMPTY.");
			return 1;
		}
	}
	/* URIを利用しない場合( = usrid を使う) */
	else if (delmode & DIVY_REMWATCH_IGNORE_URI) {
		if (IS_EMPTY(usrid)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"userid is EMPTY.");
			return 1;
		}
	}
	/* どっちも使わない */
	else {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"You could not specify ignore-uri and ignore-user.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Could not operation. Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_mailwatch テーブルの更新または削除
	 */
	if (delmode & DIVY_REMWATCH_LOGICAL) {
		sql =	"UPDATE divy_mailwatch "
			"SET mw_l_del_flag = 1 ";
	}
	else {
		sql =	"DELETE FROM divy_mailwatch ";
	}

	if (!(delmode & DIVY_REMWATCH_IGNORE_USER) && !(delmode & DIVY_REMWATCH_IGNORE_URI)) {
		sql = apr_pstrcat(p, sql, "WHERE mw_rs_uri_txt = ? AND mw_usr_id_vc = ?", NULL);
	}
	else if (delmode & DIVY_REMWATCH_IGNORE_USER) {
		sql = apr_pstrcat(p, sql, "WHERE mw_rs_uri_txt = ?", NULL);
	}
	else if (delmode & DIVY_REMWATCH_IGNORE_URI) {
		sql = apr_pstrcat(p, sql, "WHERE mw_usr_id_vc = ?", NULL);
	}

	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for divy_mailwatch. (uri = %s, usrid = %s) "
			"Reason: %s", REPLACE_NULL(uri), REPLACE_NULL(usrid), stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */

	if (!(delmode & DIVY_REMWATCH_IGNORE_USER) && !(delmode & DIVY_REMWATCH_IGNORE_URI)) {
		stmt->setString(stmt, 1, uri);
		stmt->setString(stmt, 2, usrid);
	}
	else if (delmode & DIVY_REMWATCH_IGNORE_USER) {
		stmt->setString(stmt, 1, uri);
	}
	else if (delmode & DIVY_REMWATCH_IGNORE_URI) {
		stmt->setString(stmt, 1, usrid);
	}

	/* SQL実行 */
	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to UPDATE or DELETE divy_mailwatch."
			"(uri = %s, userid = %s) Reason: %s",
			REPLACE_NULL(uri), REPLACE_NULL(usrid), stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* UPDATE 文の実行で更新件数が０件なのは問題 */
	if ((delmode & DIVY_REMWATCH_LOGICAL) && update_cnt == 0) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"Status is OK, but the count of updating is 0. "
			"Please check divy_mailwatch. (uri = %s, usrid = %s)",
			REPLACE_NULL(uri), REPLACE_NULL(usrid));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * uri が示す論理削除されたメール監視フラグ情報を保持するレコードを削除する。
 *
 * ユーザIDは削除条件には入れません。
 * また、論理削除されていないレコードは削除しません。uri はメール監視フラグを
 * マーク可能なコレクションのURIでなければなりません。
 * 削除件数が0件であっても正常終了します。
 *
 * @param r request_rec *
 * @param uri const char * 削除対象コレクションのURI
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 * @param ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
static int _clean_gargage_mailwatch_property(request_rec *r,
					const char *uri,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn *dbconn       = NULL;
	DbPreparedStmt *stmt = NULL;
	apr_pool_t *p        = r->pool;
	int iscommit         = 0;

	TRACE(p);

	/* 引数チェック */
	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_mailwatch テーブルエントリの削除
	 */

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, 
				"DELETE FROM divy_mailwatch "
				"WHERE mw_rs_uri_txt = ? "
				"AND mw_l_del_flag = 1", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt "
				"for divy_mailwatch. (uri = %s) Reason: %s",
				uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, uri);

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to DELETE divy_mailwatch.(uri = %s) Reason: %s",
			uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたdepth で、rdb_r->uri 以下に存在する全てのリソースまたは
 * コレクションのプロパティを取得する。(内部専用)
 *
 * rdb_r->uri 自身の情報も取得します。このrdb_r->uri が示す
 * divy_db_transaction_ctx 構造体は、必ず、*rdb_r が作る
 * リスト構造の開始点となっています。
 *
 * (note) rdb_r 必須項目
 * 	rdb_r->uri, rdb_r->depth (depth = 1のときだけ必須)
 * 	その他はあっても無くてもDBの値で上書きされてしまいます。
 *
 * トランザクションが継続中であれば引き継ぎます。
 *
 * @param r request_rec * request_rec 構造体へのポインタ
 * @param wp apr_pool_t * 作業用のプール
 * @param rdb_r divy_rdbo_resource *
 * 		検索対象となるuri を表す構造体. 検索結果もここに格納されます。
 * 		なお、1件もヒットしなければ、rsid は null になります。
 * @param depth int Depth ヘッダの値 (0 / 1 / DAV_INFINITY)
 * @param propflag apr_uint32_t 特別に取得するプロパティの種類 (ORで指定する)
 * 	DIVY_GET_PROP_fullname      : 作成者,更新者名称プロパティを取得する
 * 	DIVY_GET_PROP_dead          : Dead プロパティを取得する
 * 	DIVY_GET_PROP_mailwatch     : mailwatch プロパティを取得する
 * 	DIVY_GET_PROP_shared        : 共有コレクションプロパティを取得する
 * 	DIVY_GET_PROP_trash         : ごみ箱のプロパティを取得する
 * 	DIVY_GET_PROP_notrashfolder : ごみ箱だけを除外する
 * 	DIVY_GET_PROP_resourcestate : 状態, 属性プロパティを取得する
 * 	DIVY_DEL_PROP_noviewattr    : View属性が"付いていない"リソースを削り落とす(基点は除外)
 * 	DIVY_GET_PROP_confirmreading: 開封通知プロパティを取得する
 * @param sort divy_rdbo_sortorder * ソートキー(NULL でデフォルト)
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理ステータス 0: 成功 / 1: 失敗
 */
static int _get_hierarchy_property(request_rec *r,
					   apr_pool_t *wp, 
					   divy_rdbo_resource *rdb_r, 
					   int depth,
					   apr_uint32_t propflag,
					   divy_rdbo_sortorder *sort,
					   divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	dav_divy_dir_conf *dconf= dav_divy_get_dir_config(r);
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int ret;

	const char *uri         = rdb_r->uri;
	int hierarchy_depth     = rdb_r->depth;
	divy_uri_spec *u_spec   = rdb_r->u_spec;	/* NULLなら後で取る */
	int found_root, ignore_prop = 0, is_root;
	char *rsid, *geturi;
	divy_rdbo_resource *n_rdb_r = NULL, *first = NULL, *swap_rdb_r = NULL;
	apr_hash_t *dp_list_hash= NULL, *userid_hash = NULL, *cr_pr_hash = NULL;
	apr_hash_t *trash_pr_hash = NULL;	/* rsid - (divy_rdbo_trash *) */
	divy_rdbo_mailwatch *mw = NULL; 
	divy_rdbo_usr ldapusr   = { 0 };
	divy_sbuf *sql_buf      = NULL;
	char *fullname, *displayname;
	divy_rdbo_resourcetype resourcetype;
	apr_hash_t *rstate_pr_hash = NULL;	/* rsid - (divy_rdbo_resourcestate *) */
	divy_rdbo_resourcestate *rstate_pr;
	int trash_depth = 0, getdepth;
	int do_skip_trash = 0, found_trash = 0;
	char *orderby_clause = "";
	divy_rdbo_grp *grp_pr = NULL;

	/* 条件句を含まないSQL文 */
	char *sql_base_stmt     =
			"SELECT"
			" rs.rs_rs_id_c,"
			" rs.rs_uri_txt,"
			" rs.rs_dispname_vc,"
			" rs.rs_resourcetype_i,"
			" rs.rs_depth_i,"
			" rs.rs_create_bi,"
			" rs.rs_get_cont_lang_vc,"
			" rs.rs_get_cont_len_bi,"
			" rs.rs_get_cont_type_vc,"
			" rs.rs_get_etag_txt,"
			" rs.rs_get_lastmodified_bi,"
			" rs.rs_isversioned_i,"
			" rs.rs_checkin_i,"
			" rs.rs_checkout_i,"
			" rs.rs_physical_path_txt,"
			" rs.rs_creator_usr_id_vc,"
			" rs.rs_lastmodifier_usr_id_vc ";

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* u_spec が無ければ生成する */
	if (u_spec == NULL) {
		/* (note) r->pool を使うこと */
		divy_parse_uri(p, dav_divy_get_root_uri(r), uri, &u_spec);
		rdb_r->u_spec = u_spec;	/* ついでに入れておく */
	}

	/* ORDER BY 句の構成 */
	if (sort == NULL) {
		orderby_clause = "ORDER BY rs.rs_uri_txt ASC";
	}
	else {
		switch (sort->key) {
			case DIVY_SORTKEY_URI:
				if (sort->order == DIVY_SORTORDER_ASC) {
					orderby_clause = "ORDER BY rs.rs_uri_txt ASC";
				}
				else {
					orderby_clause = "ORDER BY rs.rs_uri_txt DESC";
				}
				break;
			case DIVY_SORTKEY_NAME:
				if (sort->order == DIVY_SORTORDER_ASC) {
					orderby_clause = "ORDER BY rs.rs_dispname_vc ASC";
				}
				else {
					orderby_clause = "ORDER BY rs.rs_dispname_vc DESC";
				}
				break;
			case DIVY_SORTKEY_SIZE:
				if (sort->order == DIVY_SORTORDER_ASC) {
					orderby_clause = "ORDER BY rs.rs_get_cont_len_bi ASC";
				}
				else {
					orderby_clause = "ORDER BY rs.rs_get_cont_len_bi DESC";
				}
				break;
			case DIVY_SORTKEY_TYPE:
				if (sort->order == DIVY_SORTORDER_ASC) {
					orderby_clause = "ORDER BY rs.rs_get_cont_type_vc ASC";
				}
				else {
					orderby_clause = "ORDER BY rs.rs_get_cont_type_vc DESC";
				}
				break;
			case DIVY_SORTKEY_DATE:
				if (sort->order == DIVY_SORTORDER_ASC) {
					orderby_clause = "ORDER BY  ASC";
				}
				else {
					orderby_clause = "ORDER BY  DESC";
				}
				break;
			case DIVY_SORTKEY_MODIFIER:
				if (sort->order == DIVY_SORTORDER_ASC) {
					orderby_clause = "ORDER BY rs.rs_get_lastmodified_bi ASC";
				}
				else {
					orderby_clause = "ORDER BY rs.rs_get_lastmodified_bi DESC";
				}
				break;
			case DIVY_SORTKEY_RESTYPE:
				if (sort->order == DIVY_SORTORDER_ASC) {
					orderby_clause = "ORDER BY rs.rs_resourcetype_i DESC, rs.rs_uri_txt ASC";
				}
				else {
					orderby_clause = "ORDER BY rs.rs_resourcetype_i ASC, rs.rs_uri_txt DESC";
				}
				break;
			default:
				orderby_clause = "ORDER BY rs.rs_uri_txt ASC";
		}
	}

	/*
	 * 必要であれば、Dead プロパティ, mailwatchプロパティ, 共有コレクション
	 * ごみ箱プロパティ, 状態/属性の値を取得する
	 * (note)
	 * 	dav_resource よりも先に取ってしまうのは後でループを回してrdb_r
	 * 	にプロパティ構造体を入れるので先に取っておきたかったため。
	 */

	/*
	 * 状態/属性 プロパティの取得
	 */
	if (propflag & DIVY_GET_PROP_resourcestate) {
		ret = _get_hierarchy_resourcestate_property(r, uri, depth, &rstate_pr_hash, ts_ctx);
		if (ret) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get resourcestate property."
				"(uri = %s, depth = %d)", uri, depth);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/* View属性の付いていないリソースは不要であると指示されて、結果
		 * View属性が全く取得出来なかった場合(グループフォルダは除く) */
		if (rstate_pr_hash == NULL && (propflag & DIVY_DEL_PROP_noviewattr) &&
				(u_spec->infotype == DIVY_INFOTYPE_group_e_regular)) {
			/* 取得リソースの特殊プロパティは最終的に無視されるので
			 * 要らないプロパティは取らないようにする(パフォーマンス向上) */
			ignore_prop = 1;
		}
	}

	/*
	 * Dead プロパティの取得
	 */
	if ((propflag & DIVY_GET_PROP_dead) && !ignore_prop) {
		ret = _get_dav_dead_property(r, uri, depth, &dp_list_hash, ts_ctx);
		if (ret) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get dead property.(uri = %s, "
				"depth = %d)", uri, depth);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/* 
	 * mailwatch プロパティの取得(watchownerなし)
	 */
	if ((propflag & DIVY_GET_PROP_mailwatch) && !ignore_prop) {
		ret = _get_merge_mailwatch_property(r, uri,
						divy_get_userid(r), &mw, ts_ctx);
		if (ret) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get mailwatch property."
				"(uri = %s, depth = %d)", uri, depth);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	rdb_r->shlink_pr = NULL;
	/* 
	 * 共有コレクションプロパティを取得する
	 */
	if ((propflag & DIVY_GET_PROP_shared) && !ignore_prop) {
		ret = _get_sharedcollection_property(r, uri, &rdb_r->shlink_pr, ts_ctx);
		if (ret) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get sharedcollection property."
				"(uri = %s, depth = %d)", uri, depth);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * trashinformation プロパティの取得
	 */
	if ((propflag & DIVY_GET_PROP_trash) && !ignore_prop) {
		ret = _get_trash_property(r, uri, depth, &trash_pr_hash, ts_ctx);
		if (ret) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get trashinformation property."
				"(uri = %s, depth = %d)", uri, depth);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * groupstate プロパティの取得
	 */
	if ((propflag & DIVY_GET_PROP_groupstate) && !ignore_prop) {
		ret = divy_rdbo_get_hierarchy_group_property(r, u_spec->other_part, 0, &grp_pr, ts_ctx);
		if (ret) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get group property."
					"(uri = %s, depth = %d)", uri, depth);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * confirmreading プロパティの取得
	 */
	if ((propflag & DIVY_GET_PROP_confirmreading) && !ignore_prop) {
		ret = _get_confirmreading(r, uri, NULL/* ユーザIDなし*/, depth, &cr_pr_hash, ts_ctx);
		if (ret) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get confirmreading property."
					"(uri = %s, depth = %d)", uri, depth);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/* 
	 * dav_resource テーブルに対する結合条件:
	 */
	divy_sbuf_create(wp, &sql_buf, 800);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf, sql_base_stmt);

	if (propflag & DIVY_GET_PROP_fullname) {
		divy_sbuf_append(sql_buf,
				",u1.usr_fullname_vc AS rs_creator_vc,"
				" u2.usr_fullname_vc AS rs_lastmodifier_vc "
				"FROM dav_resource rs "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"LEFT JOIN divy_usr u1 "
				"ON rs.rs_creator_usr_id_vc = u1.usr_usr_id_vc "
				"LEFT JOIN divy_usr u2 "
				"ON rs.rs_lastmodifier_usr_id_vc = u2.usr_usr_id_vc "
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				", divy_usr u1, divy_usr u2 "
#endif
				);
		if (depth == 0) {
			divy_sbuf_append(sql_buf,
#if defined(DIVY_DBMS_ORACLE) /* oracle */
				"WHERE rs.rs_creator_usr_id_vc = u1.usr_usr_id_vc(+) "
				"AND rs.rs_lastmodifier_usr_id_vc = u2.usr_usr_id_vc(+) "
				"AND rs_uri_txt = ?"
#else
				"WHERE rs_uri_txt = ?"
#endif
				);
		}
		else if (depth == 1) {
			divy_sbuf_append(sql_buf,
#if defined(DIVY_DBMS_ORACLE) /* oracle */
				"WHERE rs.rs_creator_usr_id_vc = u1.usr_usr_id_vc(+) "
				"AND rs.rs_lastmodifier_usr_id_vc = u2.usr_usr_id_vc(+) "
				"AND (rs_uri_txt = ?"
				" OR (rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND rs_depth_i = ?)) "
#else
				"WHERE rs_uri_txt = ? "
				"OR (rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND rs_depth_i = ?) "
#endif
				);
			divy_sbuf_append(sql_buf, orderby_clause);
		}
		else {
			divy_sbuf_append(sql_buf,
#if defined(DIVY_DBMS_ORACLE) /* oracle */
				"WHERE rs.rs_creator_usr_id_vc = u1.usr_usr_id_vc(+) "
				"AND rs.rs_lastmodifier_usr_id_vc = u2.usr_usr_id_vc(+) "
				"AND (rs_uri_txt = ?"
				" OR rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE") "
#else
				"WHERE rs_uri_txt = ? "
				"OR rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE" "
#endif
				);
			divy_sbuf_append(sql_buf, orderby_clause);
		}
	}
	/* フルネーム系プロパティを含めない場合 */
	else {
		divy_sbuf_append(sql_buf, "FROM dav_resource rs ");
		if (depth == 0) {
			divy_sbuf_append(sql_buf,
				"WHERE rs_uri_txt = ?");
		}
		else if (depth == 1) {
			divy_sbuf_append(sql_buf,
				"WHERE rs_uri_txt = ? "
				"OR (rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND rs_depth_i = ?) ");
			divy_sbuf_append(sql_buf, orderby_clause);
		}
		else {
			divy_sbuf_append(sql_buf,
				"WHERE rs_uri_txt = ? "
				"OR rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE" ");
			divy_sbuf_append(sql_buf, orderby_clause);
		}
	}

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), wp); 
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s) Reason: %s",
			uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	stmt->setString(stmt, 1, uri);
	if (depth != 0) {
		/* uri に含まれるワイルドカード文字をエスケープする */
		stmt->setString(stmt, 2,
			apr_pstrcat(wp, stmt->escWildCard(stmt, uri), "/%", NULL));
	}
	if (depth == 1) {
		stmt->setInt(stmt, 3, hierarchy_depth + 1);
	}

	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet. (uri = %s) Reason: %s",
			uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 
	 * 結果の取得 
	 */
	found_root  = 0;	/* 検索の基点となったURIのプロパティが見つかった */
	rsid        = NULL;
	userid_hash = apr_hash_make(wp);	/* userid - 正しいフルネーム */
	rstate_pr   = NULL;
	is_root     = 0;	/* 検索の基点となるURIのプロパティではない */

	/* ゴミ箱をスキップする操作を行うか? */
	if ((propflag & DIVY_GET_PROP_notrashfolder) &&
			(u_spec->infotype == DIVY_INFOTYPE_user_e ||
			 u_spec->infotype == DIVY_INFOTYPE_group_e)) {
		do_skip_trash = 1;	/* スキップ操作をやります */
		trash_depth = hierarchy_depth + 1;	/* ゴミ箱は1つした */
	}

	while (rset->next(rset) == DB_TRUE) {

		/* rsid, uri の取得 */
		rsid   = rset->getString(rset, 1);
		geturi = rset->getString(rset, 2);
		displayname  = rset->getString(rset, 3);
		resourcetype = rset->getInt(rset, 4);
		getdepth     = rset->getInt(rset, 5);

		/* 状態/属性プロパティの取り出し */
		rstate_pr = NULL;
		if (rstate_pr_hash != NULL) {
			rstate_pr = apr_hash_get(rstate_pr_hash, rsid, APR_HASH_KEY_STRING);
		}

		is_root = 0;
		/* 基点となったリソースのURIかどうか */
		if (found_root == 0 && strcmp(uri, geturi) == 0) {
			swap_rdb_r = n_rdb_r;	/* 現在のプロパティを一時退避 */
			n_rdb_r = rdb_r;
			found_root = 1;	/* 見つかった */
			is_root = 1;	/* 基点リソースだった */

			/* (note) "View属性の付いていないリソースは要らない"と指示されていても
			 * 基点となるリソースは返却しないとNULL リソースとの区別が出来なく
			 * るので削ってはならない。またゴミ箱の場合も同様です */
		}
		/* 基点リソースがリストの先頭ではなかった場合 */
		else if (n_rdb_r == NULL) {
			/* View属性が付いていないリソースは要らない場合 */
			if (IS_INEFFECTIVE_STATE(rstate_pr) && (propflag & DIVY_DEL_PROP_noviewattr)) {
				continue;	/* 以下は無視 */
			}

			/* ゴミ箱外し */
			if (do_skip_trash && !found_trash && resourcetype == DIVY_TYPE_COLLECTION &&
					trash_depth == getdepth &&
					strcmp(DIVY_TRASHFOLDER_NAME, displayname) == 0) {
				found_trash = 1;	/* もう見つかったので今後は無視 */
				continue;
			}

			first = n_rdb_r = apr_pcalloc(wp, sizeof(*n_rdb_r));
		}
		else {
			/* View属性が付いていないリソースは要らない場合 */
			if (IS_INEFFECTIVE_STATE(rstate_pr) && (propflag & DIVY_DEL_PROP_noviewattr)) {
				continue;	/* 以下は無視 */
			}

			/* ゴミ箱外し */
			if (do_skip_trash && !found_trash && resourcetype == DIVY_TYPE_COLLECTION &&
					trash_depth == getdepth &&
					strcmp(DIVY_TRASHFOLDER_NAME, displayname) == 0) {
				found_trash = 1;	/* もう見つかったので今後は無視 */
				continue;
			}

			n_rdb_r->next = apr_pcalloc(wp, sizeof(*n_rdb_r));
			n_rdb_r = n_rdb_r->next;
		}
		n_rdb_r->p         = wp;
		n_rdb_r->rstate_pr = rstate_pr;	/* 状態プロパティの記録 */
		n_rdb_r->next = NULL;	/* sentinel */


		/* dead property の取り出し */
		if (dp_list_hash != NULL) {
			n_rdb_r->d_pr  = apr_hash_get(dp_list_hash, rsid, APR_HASH_KEY_STRING);
		}

		/* mailwatch property の取り出し */
		n_rdb_r->mwatch_pr = mw;

		/* trashinformation の取り出し */
		if (trash_pr_hash != NULL) {
			n_rdb_r->trash_pr = apr_hash_get(trash_pr_hash, rsid, APR_HASH_KEY_STRING);
			if (n_rdb_r->trash_pr != NULL) {
				fullname = apr_hash_get(userid_hash,
							n_rdb_r->trash_pr->deleter_id,
							APR_HASH_KEY_STRING);
				if (IS_FILLED(fullname)) {
					n_rdb_r->trash_pr->deleter_name = fullname;
				}
				else {
					/* ユーザ名称の置き換えが必要 */
					ldapusr.fullname = n_rdb_r->trash_pr->deleter_name;
					if (IS_FILLED(dconf->ldapmapfullnameattr) && 
							divy_util_ldap_get_user_property(r, wp,
							n_rdb_r->trash_pr->deleter_id,
							&ldapusr) == TF_LDAP_TRUE) {
						n_rdb_r->trash_pr->deleter_name = ldapusr.fullname;
					}

					/* ユーザID - 正式名称の組合せを記録しておく */
					apr_hash_set(userid_hash,
							n_rdb_r->trash_pr->deleter_id,
							APR_HASH_KEY_STRING,
							n_rdb_r->trash_pr->deleter_name);
				}
			}
		}

		/* groupstate の取り出し
		 * 基点リソースのみが対象 */
		if (grp_pr != NULL && is_root) {
			n_rdb_r->grp_pr = grp_pr;
		}

		/* confirmreading プロパティの取り出し */
		if (cr_pr_hash != NULL) {
			n_rdb_r->confirmreading_pr  = apr_hash_get(cr_pr_hash, rsid, APR_HASH_KEY_STRING);
		}

		/* 値の取り出し */
		n_rdb_r->rsid               = rsid;
		n_rdb_r->uri                = geturi;
		n_rdb_r->displayname        = displayname;
		n_rdb_r->resourcetype       = resourcetype;
		n_rdb_r->depth              = getdepth;
		n_rdb_r->creationdate       = (time_t) rset->getBigInt(rset, 6);
		n_rdb_r->getcontentlanguage = rset->getString(rset, 7);
		if (IS_EMPTY(n_rdb_r->getcontentlanguage)) {
			n_rdb_r->getcontentlanguage = NULL;
		}
		n_rdb_r->getcontentlength   = rset->getBigInt(rset, 8);
		n_rdb_r->getcontenttype     = rset->getString(rset, 9);
		if (IS_EMPTY(n_rdb_r->getcontenttype)) {
			n_rdb_r->getcontenttype = NULL;
		}
		n_rdb_r->getetag            = rset->getString(rset, 10);
		n_rdb_r->getlastmodified    = (time_t) rset->getBigInt(rset, 11);
		n_rdb_r->isversioned        = rset->getInt(rset,   12);
		n_rdb_r->checkin            = rset->getInt(rset,   13);
		n_rdb_r->checkout           = rset->getInt(rset,   14);
		n_rdb_r->physicalpath       = rset->getString(rset,15);
		n_rdb_r->creator_userid     = rset->getString(rset,16);
		n_rdb_r->lastmodifier_userid= rset->getString(rset,17);

		if (propflag & DIVY_GET_PROP_fullname) {
			n_rdb_r->creator      = rset->getString(rset,18);
			n_rdb_r->lastmodifier = rset->getString(rset,19);

			fullname = apr_hash_get(userid_hash,
						n_rdb_r->creator_userid,
						APR_HASH_KEY_STRING);
			if (IS_FILLED(fullname)) {
				n_rdb_r->creator = fullname;
			}
			else {
				/*  LDAPから名前を取得して置換する */
				ldapusr.fullname = n_rdb_r->creator;
				if (IS_FILLED(dconf->ldapmapfullnameattr) && 
						divy_util_ldap_get_user_property(r, wp,
							n_rdb_r->creator_userid,
							&ldapusr) == TF_LDAP_TRUE) {
					n_rdb_r->creator = ldapusr.fullname;
				}
				/* ユーザID - 正式名称の組合せを記録しておく */
				apr_hash_set(userid_hash, n_rdb_r->creator_userid,
						APR_HASH_KEY_STRING, n_rdb_r->creator);
			}

			fullname = apr_hash_get(userid_hash,
					n_rdb_r->lastmodifier_userid, APR_HASH_KEY_STRING);
			if (IS_FILLED(fullname)) {
				n_rdb_r->lastmodifier = fullname;
			}
			else {
				/*  LDAPから名前を取得して置換する */
				ldapusr.fullname = n_rdb_r->lastmodifier;
				if (IS_FILLED(dconf->ldapmapfullnameattr) && 
						divy_util_ldap_get_user_property(r, wp,
							n_rdb_r->lastmodifier_userid,
							&ldapusr) == TF_LDAP_TRUE) {
					n_rdb_r->lastmodifier = ldapusr.fullname;
				}
				/* ユーザID - 正式名称の組合せを記録しておく */
				apr_hash_set(userid_hash,n_rdb_r->lastmodifier_userid,
						APR_HASH_KEY_STRING, n_rdb_r->lastmodifier);
			}
		}

		/* 退避プロパティを元に戻す */
		if (swap_rdb_r != NULL) {
			n_rdb_r = swap_rdb_r;
			rdb_r->next = first;	/* リストを繋げる */
			swap_rdb_r = NULL;	/* 退避プロパティを忘れる   */
		}
	}

	/* 検索の基点となったリソースが見つからなかった場合 */
	if (rsid && found_root == 0) {
		/* 更新動作との競合で発生する可能性あり */
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get root-resource property."
			"It is probably occurred by concurrent operation, "
			"Please refresh data and retry operation.(uri = %s)", uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * src_rdb_r のリソースが持つDead property を dist_rdb_r のリソースへと
 * コピーする。
 *
 * src_rdb_r  必須項目: uri, rsid
 * dist_rdb_r 必須項目: uri
 *
 * (note)
 * トランザクションが継続中であれば引き継ぎます。
 *
 * @param r request_rec *
 * @param src_rdb_r const divy_rdbo_resource *  ソースリソース
 * @param dist_rdb_r const divy_rdbo_resource * Desitinationリソース
 * @param depth int Depth ヘッダの値 (0, DAV_INFINITY)
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理ステータス 0: 成功 / 1: 失敗
 */
static int _copy_dav_dead_property(request_rec *r,
                                   const divy_rdbo_resource *src_rdb_r,
                                   const divy_rdbo_resource *dist_rdb_r,
				   int depth,
			           divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL, *stmt2 = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	char *dist_uri, *src_rsid;
	char *sql_stmt =
			"INSERT INTO dav_dead_property "
			"(dp_rs_id_c,"
			" dp_ns_id_i,"
			" dp_name_vc,"
			" dp_value_txt,"
			" dp_lang_tag_vc) " 
			"(SELECT"
			" (SELECT rs_rs_id_c FROM dav_resource"
			"  WHERE rs_uri_txt = ?),"
			" dp.dp_ns_id_i,"
			" dp.dp_name_vc,"
			" dp.dp_value_txt,"
			" dp.dp_lang_tag_vc"
			" FROM dav_dead_property dp"
			" WHERE dp.dp_rs_id_c = ?)";

	TRACE(p);

	/* depth = 1 は許可されていません。*/
	if (depth == 1) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The copy of dead property is not allowd by depth = 1");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * dav_dead_property をコピーする
	 * (depth = 0)
	 */
	if (depth == 0) {
		stmt = dbconn->prepareStatement(dbconn, sql_stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt."
				"(src = %s, dist = %s) Reason: %s",
				src_rdb_r->uri, dist_rdb_r->uri,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);

			return 1;
		}

		stmt->setString(stmt, 1, dist_rdb_r->uri);
		stmt->setString(stmt, 2, src_rdb_r->rsid);

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to copy dav_dead_property(depth=0). "
				"(src = %s, dist = %s) Reason: %s", 
				src_rdb_r->uri, dist_rdb_r->uri,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);

			return 1;
		}

		if (stmt != NULL) stmt->close(stmt);
		if (iscommit) divy_db_commit_transaction(ts_ctx);

		return 0;
	}

	/*
	 * dav_dead_property をコピーする
	 * (depth = DAV_INFINITY)
	 */
	/* 既存のDead プロパティを検索し、dist のuriとsrc のrsidを取得する */
	stmt = dbconn->prepareStatement(dbconn,
			"SELECT"
			" DISTINCT dp.dp_rs_id_c,"
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
			DIVY_DBFUNC_SUBSTRING("r.rs_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")" "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
			"FROM dav_resource r"
			" INNER JOIN dav_dead_property dp"
			" ON r.rs_rs_id_c = dp.dp_rs_id_c "
			"WHERE r.rs_uri_txt = ?"
			" OR r.rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE, 
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
			"FROM dav_resource r, dav_dead_property dp"
			" WHERE r.rs_rs_id_c = dp.dp_rs_id_c"
			" AND (r.rs_uri_txt = ?"
			" OR r.rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE")", 
#endif
			p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(src = %s, dist = %s) Reason: %s",
			src_rdb_r->uri, dist_rdb_r->uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	stmt->setString(stmt, 1, dist_rdb_r->uri);
	stmt->setString(stmt, 2, src_rdb_r->uri);
	stmt->setString(stmt, 3, src_rdb_r->uri);
	/* uri に含まれるワイルドカード文字をエスケープする */
	stmt->setString(stmt, 4, apr_pstrcat(p,
			stmt->escWildCard(stmt, src_rdb_r->uri), "/%", NULL));

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select dav_dead_property. (src = %s) "
			"Reason: %s", src_rdb_r->uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 取得した全てのDead プロパティをコピーする */
	while (rset->next(rset) == DB_TRUE) {
		src_rsid = rset->getString(rset, 1);
		dist_uri = rset->getString(rset, 2);

		stmt2 = dbconn->prepareStatement(dbconn, sql_stmt, p);
		if (stmt2->getCode(stmt2) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt."
				"(dist_uri = %s, src_rsid = %s) Reason: %s",
				dist_uri, src_rsid, stmt2->getMsg(stmt2));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
			if (rset != NULL)  rset->close(rset); rset = NULL;
			if (stmt != NULL)  stmt->close(stmt); stmt = NULL;

			return 1;
		}

		stmt2->setString(stmt2, 1, dist_uri);
		stmt2->setString(stmt2, 2, src_rsid);

		(void) stmt2->executeUpdate(stmt2, p);
		if (stmt2->getCode(stmt2) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to copy dav_dead_property"
				"(depth=infinity). "
				"(dist_uri = %s, src_rsid = %s) Reason: %s", 
				dist_uri, src_rsid, stmt2->getMsg(stmt2));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
			if (rset != NULL)  rset->close(rset); rset = NULL;
			if (stmt != NULL)  stmt->close(stmt); stmt = NULL;

			return 1;
		}

		if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;

	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/*
 * 指定されたrdb_r->uri とそれ以下に存在する全てのリソースまたはコレクションの
 * Dead プロパティを取得して、リソースIDとキーとしてdivy_rdbo_dproperty を持つ
 * ハッシュを返却する。
 *
 * (note)
 * この関数は、１リソース単位で実施されるDB 検索を避ける目的で導入されました。
 * トランザクションが継続中であれば、それを引き継ぎます。
 * 検索結果は、リソースIDとネームスペースIDでorder by されています。よって、
 * あるリソースIDとリンクしているdivy_rdbo_dproperty のListは、必ず
 * ネームスペースIDの小さい順にリンクされています。(重要)
 *
 * @param r request_rec *
 * @param uri const char * URI を表す文字列。2重スラッシュ、末尾スラッシュは禁止.
 * @param depth int Depth ヘッダの値 (0, 1, DAV_INFINITY)
 * @param dp_list_hash apr_hash_t ** 取得したdivy_rdbo_property を持つハッシュ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _get_dav_dead_property(request_rec *r,
				  const char *uri,
				  int depth,
				  apr_hash_t **dp_list_hash,
			          divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;

	char *rsid              = NULL;
	divy_rdbo_dproperty *dp = NULL;
	int cnt                 = 0;
	apr_hash_t *work_hash   = NULL;
	int hierarchy_depth     = divy_count_dirs(uri);
	char *sql_stmt;
	char *sql_stmt_body     =
			"SELECT "
			"dp_rs_id_c, "
			"dp_ns_id_i, "
			"dp_name_vc, "
			"dp_value_txt, "
			"dp_lang_tag_vc "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2*/
			"FROM dav_dead_property dp"
			" INNER JOIN dav_resource r"
			" ON dp.dp_rs_id_c = r.rs_rs_id_c "
			"WHERE r.rs_uri_txt = ? ";
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
			"FROM dav_dead_property dp, dav_resource r"
			" WHERE dp.dp_rs_id_c = r.rs_rs_id_c"
			" AND r.rs_uri_txt = ? ";
#endif
	*dp_list_hash = NULL;	/* 初期化 */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* 
	 * dav_dead_property テーブルを検索し、uri 以下のリソースが
	 * 持つDead プロパティを取得する。
	 */
	if (depth == 0) {
		sql_stmt = apr_pstrcat(p, sql_stmt_body, 
				"ORDER BY dp_rs_id_c, dp_ns_id_i", NULL);
	}
	else if (depth == 1) {
		sql_stmt = apr_pstrcat(p, sql_stmt_body, 
			"OR (r.rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
			" AND r.rs_depth_i = ?) "
			"ORDER BY dp_rs_id_c, dp_ns_id_i", NULL);
	}
	else {
		sql_stmt = apr_pstrcat(p, sql_stmt_body, 
			"OR r.rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
			" ORDER BY dp_rs_id_c, dp_ns_id_i", NULL);
	}

	stmt = dbconn->prepareStatement(dbconn, sql_stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for dav_dead_property. "
			"(uri = %s) Reason: %s", uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	stmt->setString(stmt, 1, uri);
	if (depth != 0) {
		/* uri に含まれるワイルドカード文字をエスケープする */
		stmt->setString(stmt, 2, apr_pstrcat(p,
				stmt->escWildCard(stmt, uri), "/%", NULL));
	}

	if (depth == 1) {
		stmt->setInt(stmt, 3, hierarchy_depth + 1);
	}

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet for dav_dead_property."
			"(uri = %s) Reason: %s", uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 
	 * 結果の取得
	 * 取得結果を、divy_rdbo_dproperty に入れて、その構造体を
	 * rsid でハッシュに格納する。同一のrsid が現れた場合には
	 * divy_rdbo_dproperty のnext につないでList構造を作る。
	 */
	*dp_list_hash = apr_hash_make(p);
	work_hash     = apr_hash_make(p);
	while (rset->next(rset) == DB_TRUE) {
		rsid = rset->getString(rset, 1);

		dp = apr_hash_get(work_hash, rsid, APR_HASH_KEY_STRING);
		if (dp == NULL) {
			dp = apr_pcalloc(p, sizeof(*dp));
			dp->next = NULL;	/* sentinel */

			/* divy_dead_property からなるListの先頭を保存しておく*/
			apr_hash_set(*dp_list_hash, rsid, APR_HASH_KEY_STRING, dp);
		}
		else {
			dp->next = apr_pcalloc(p, sizeof(*dp));
			dp       = dp->next;
			dp->next = NULL;	/* sentinel */
		}

		dp->rsid     = rsid;
		dp->ns_id    = rset->getInt(rset, 2);
		dp->name     = rset->getString(rset, 3);
		dp->value    = rset->getString(rset, 4);
		dp->lang_tag = rset->getString(rset, 5);

		/* 現在使用した divy_rdbo_property を憶えておく */
		apr_hash_set(work_hash, rsid, APR_HASH_KEY_STRING, dp);
		cnt++;
	}

	if (cnt == 0) *dp_list_hash = NULL; 

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたDead プロパティを新規登録する。
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます
 *
 * @param r request_rec *
 * @param d_pr const divy_rdbo_dproperty * 登録するDeadプロパティ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _insert_dead_property(request_rec *r,
					const divy_rdbo_dproperty *d_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t      *p      = r->pool;
	int iscommit            = 0;
	int insert_cnt          = 0;

	if (d_pr == NULL || IS_EMPTY(d_pr->rsid) || IS_EMPTY(d_pr->name)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"d_pr or rsid or name is NULL !!");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * dav_dead_property テーブルへの新規追加
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO dav_dead_property ("
				"dp_rs_id_c,"
				"dp_ns_id_i,"
				"dp_name_vc,"
				"dp_value_txt,"
				"dp_lang_tag_vc) "
				"VALUES (?,?,?,?,?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (rsid = %s) Reason: %s",
			d_pr->rsid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	stmt->setString(stmt, 1, d_pr->rsid);
	stmt->setInt(stmt,    2, d_pr->ns_id);
	stmt->setString(stmt, 3, d_pr->name);
	stmt->setString(stmt, 4, REPLACE_EMPTYSTR(d_pr->value));
	stmt->setString(stmt, 5, REPLACE_EMPTYSTR(d_pr->lang_tag));

	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert dav_dead_property. (rsid = %s)"
			"Reason: %s", d_pr->rsid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 登録出来なかった場合 */
	if (insert_cnt == 0) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Status is OK, but the count of inserting is 0."
			"Please check dav_dead_property.(rsid = %s)", d_pr->rsid);

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 終了処理 */
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたDead プロパティを削除する。
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます
 *
 * @param r request_rec *
 * @param d_pr const divy_rdbo_dproperty * 削除するDeadプロパティ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _remove_dead_property(request_rec *r,
					const divy_rdbo_dproperty *d_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t      *p      = r->pool;
	int iscommit            = 0;

	if (d_pr == NULL || IS_EMPTY(d_pr->rsid) || IS_EMPTY(d_pr->name)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"d_pr or rsid or name is EMPTY");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * dav_dead_property テーブルから削除する
	 */
	stmt = dbconn->prepareStatement(dbconn, 
			"DELETE FROM dav_dead_property "
			"WHERE dp_rs_id_c = ? "
			"AND dp_ns_id_i = ? "
			"AND dp_name_vc = ?",p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (rsid = %s) Reason: %s",
			d_pr->rsid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	stmt->setString(stmt, 1, d_pr->rsid);
	stmt->setInt(stmt,    2, d_pr->ns_id);
	stmt->setString(stmt, 3, d_pr->name);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete dav_dead_property. (rsid = %s) "
			"Reason: %s", d_pr->rsid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 終了処理 */
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたdivy_rdbo_resource のプロパティを DB に書き込む。(内部専用)
 *
 * inset 対象テーブル： dav_resource
 * rdb_r 必須項目:
 * 	挿入したい値, rsid, getetag
 * (note)
 * トランザクションが継続中であれば、引き継ぎます。
 * (note) DB キー(rs_key1_vc, rs_key2_vc ...) の登録
 * 	DB キーを登録するには、rdb_r->keylist_pr にdivy_rdbo_keylist の
 * 	リストを入れます。
 * 	但し、keylist_pr のメンバindex の歯抜けは許されません。
 * 	呼び出し元にて保証して下さい。
 *
 * @param r request_rec *
 * @param rdb_r const divy_rdbo_resource * DBに入れるデータ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
static int _insert_property(request_rec *r, const divy_rdbo_resource *rdb_r,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int insert_cnt          = 0;
	int iscommit            = 0;
	divy_rdbo_keylist *pr   = NULL;
	int i, db_idx;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * dav_resource テーブルへの新規追加
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"INSERT INTO dav_resource ("
			"rs_rs_id_c,"
			"rs_uri_txt,"
			"rs_dispname_vc,"
			"rs_create_bi,"
			"rs_get_cont_lang_vc,"
			"rs_get_cont_len_bi,"
			"rs_get_cont_type_vc,"
			"rs_get_etag_txt,"
			"rs_get_lastmodified_bi,"
			"rs_resourcetype_i,"
			"rs_depth_i,"
			"rs_isversioned_i,"
			"rs_checkin_i,"
			"rs_checkout_i,"
			"rs_physical_path_txt,"
			"rs_creator_usr_id_vc,"
			"rs_lastmodifier_usr_id_vc) "
			"VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for dav_resource insert. "
			"(rsid = %s, uri = %s) Reason: %s", 
			rdb_r->rsid, rdb_r->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	stmt->setString(stmt, 1, rdb_r->rsid);
	stmt->setString(stmt, 2, rdb_r->uri);
	stmt->setString(stmt, 3, rdb_r->displayname);
	stmt->setBigInt(stmt, 4, rdb_r->creationdate);
	stmt->setString(stmt, 5, REPLACE_EMPTYSTR(rdb_r->getcontentlanguage));
	stmt->setBigInt(stmt, 6, rdb_r->getcontentlength);
	stmt->setString(stmt, 7, REPLACE_EMPTYSTR(rdb_r->getcontenttype));
	stmt->setString(stmt, 8, rdb_r->getetag);
	stmt->setBigInt(stmt, 9, rdb_r->getlastmodified);
	stmt->setInt(stmt,   10, rdb_r->resourcetype);
	stmt->setInt(stmt,   11, rdb_r->depth);
	stmt->setInt(stmt,   12, rdb_r->isversioned);
	stmt->setInt(stmt,   13, rdb_r->checkin);
	stmt->setInt(stmt,   14, rdb_r->checkout);
	stmt->setString(stmt,15, REPLACE_EMPTYSTR(rdb_r->physicalpath));
	stmt->setString(stmt,16, rdb_r->creator_userid);
	stmt->setString(stmt,17, rdb_r->lastmodifier_userid);

	/* SQL 実行 */
	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert dav_resource data. "
			"(rsid = %s, uri = %s) Reason: %s", 
			rdb_r->rsid, rdb_r->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}
	if (insert_cnt == 0) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Status is OK, but the count of inserting is 0."
			"Please check dav_resource.(rsid = %s, uri = %s)", 
			rdb_r->rsid, rdb_r->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 使い終わった資源の解放 */
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_dbkey テーブルへの挿入
	 */
	if (rdb_r->keylist_pr != NULL) {
		stmt = dbconn->prepareStatement(dbconn,
				"INSERT INTO divy_dbkey ("
				"dk_rs_id_c,"
				"dk_key1_vc,"
				"dk_key2_vc,"
				"dk_key3_vc,"
				"dk_key4_vc,"
				"dk_key5_vc,"
				"dk_key6_vc,"
				"dk_key7_vc,"
				"dk_key8_vc,"
				"dk_key9_vc,"
				"dk_key10_vc) "
				"VALUES (?,?,?,?,?,?,?,?,?,?,?)", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt for divy_dbkey insert. "
				"(rsid = %s, uri = %s) Reason: %s", 
				rdb_r->rsid, rdb_r->uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);

			return 1;
		}
		stmt->setString(stmt, 1, rdb_r->rsid);
		db_idx = 1;
		for (pr = rdb_r->keylist_pr; pr; pr = pr->next) {
			db_idx = 1 + pr->index;
			stmt->setString(stmt, db_idx, REPLACE_EMPTYSTR(pr->key));
		}

		/* 残りの値はNULL値にする */
		for (i = db_idx + 1; i <= 11; i++) {
			stmt->setString(stmt, i, NULL);
		}

		/* SQL 実行 */
		insert_cnt = stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_dbkey. "
				"(rsid = %s, uri = %s) Reason: %s", 
				rdb_r->rsid, rdb_r->uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		if (insert_cnt == 0) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Status is OK, but the count of inserting is 0."
				"Please check dav_resource.(rsid = %s, uri = %s)", 
				rdb_r->rsid, rdb_r->uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/* 使い終わった資源の解放 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/*
 * 指定されたuri のトップコレクション[グループ｜ldbsearch結果｜共有コレクション]
 * 以下に存在する全てのmailwatchプロパティを取得して、トリガをマージ後
 * 返却する。
 * userid が存在した場合、このユーザ自身がメール監視を行っていたら、
 * owner_trigger_methos にトリガメソッドを設定します。
 * なお、usrid 自身がメール監視していない場合、owner_trigger_methos は
 * 常にNULLとなります。
 *
 * (note)
 * 返却件数は常に1件です。トリガのみを返却します。 
 * 論理削除フラグは意識しません。
 * トランザクションが継続中であれば、それを引き継ぎます。
 *
 * @param r request_rec *
 * @param uri const char * URI を表す文字列。2重スラッシュ、末尾スラッシュは禁止.
 * @param usrid const char * watchownerかどうか調べるusrid
 * @param mw divy_rdbo_mailwatch ** 取得したmailwatch のトリガマージ結果
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _get_merge_mailwatch_property(request_rec *r, 
					const char *uri, const char *usrid, 
					divy_rdbo_mailwatch **mw,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int m_search            = ap_method_number_of("SEARCH");
	char *sql_stmt		= "SELECT mw_trigger_get"
					", mw_trigger_put"
					", mw_trigger_post"
					", mw_trigger_delete"
					", mw_trigger_options"
					", mw_trigger_propfind"
					", mw_trigger_proppatch"
					", mw_trigger_mkcol"
					", mw_trigger_copy"
					", mw_trigger_move"
					", mw_trigger_lock"
					", mw_trigger_unlock"
					", mw_trigger_search";
	char *mwtopuri		= NULL;
	apr_int32_t trigger_methods[13] = { 0 };
	int i;

	*mw = NULL;	/* 初期化 */

	/* 
	 * uriから先頭を切り出しトップ階層のURIを作成する 
	 */
	if (divy_extract_mailwatch_uri(p, dav_divy_get_root_uri(r),
						uri, &mwtopuri)) {
		return 1;
	}
	else if (mwtopuri == NULL) {
		return 0;	/* mailwatch が存在しない場合 */
	}

	/* usridがあればselect句追加 */
	if (IS_FILLED(usrid)) {
		sql_stmt = apr_pstrcat(p, sql_stmt,
				", mw_usr_id_vc",
#ifdef DIVY_SUPPORT_MESSENGER
				", mw_notification_i"
#endif	/* DIVY_SUPPORT_MESSENGER */
				" FROM divy_mailwatch"
				" WHERE mw_rs_uri_txt = ?", NULL);
	}
	else {
		/* from,where */
		sql_stmt = apr_pstrcat(p, sql_stmt,
				" FROM divy_mailwatch"
				" WHERE mw_rs_uri_txt = ?", NULL);
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* 
	 * divy_mailwatch テーブルを検索し、uri のトップコレクション
	 * 以下のmailwatch プロパティを取得する。
	 */
	stmt = dbconn->prepareStatement(dbconn, sql_stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt "
				"for divy_mailwatch. (uri = %s) Reason: %s", 
				uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	stmt->setString	(stmt, 1, mwtopuri);

	rset = stmt->executeQuery(stmt, p);

	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet for divy_mailwatch."
				"(uri = %s) Reason: %s", 
				uri, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 
	 * 結果の取得
	 */
	while (rset->next(rset) == DB_TRUE) {
		/* 領域確保,owner_trigger_methos の初期化 */
		if (!(*mw)){
			*mw = apr_pcalloc(p, sizeof(divy_rdbo_mailwatch));
			(*mw)->owner_trigger_methos = NULL;
	        }

		/* トリガメソッドの取得 */
		for (i = 0; i < 13; i++) {
			trigger_methods[i] = rset->getInt(rset, i+1);
		}

		/* トリガを取得しマージしていく */
		(*mw)->trigger_methods[M_GET]       |= trigger_methods[0];
		(*mw)->trigger_methods[M_PUT]       |= trigger_methods[1];
		(*mw)->trigger_methods[M_POST]      |= trigger_methods[2];
		(*mw)->trigger_methods[M_DELETE]    |= trigger_methods[3];
		(*mw)->trigger_methods[M_OPTIONS]   |= trigger_methods[4];
		(*mw)->trigger_methods[M_PROPFIND]  |= trigger_methods[5];
		(*mw)->trigger_methods[M_PROPPATCH] |= trigger_methods[6];
		(*mw)->trigger_methods[M_MKCOL]     |= trigger_methods[7];
		(*mw)->trigger_methods[M_COPY]      |= trigger_methods[8];
		(*mw)->trigger_methods[M_MOVE]      |= trigger_methods[9];
		(*mw)->trigger_methods[M_LOCK]      |= trigger_methods[10];
		(*mw)->trigger_methods[M_UNLOCK]    |= trigger_methods[11];
		(*mw)->trigger_methods[m_search]    |= trigger_methods[12];

		/* usridがあればownerかどうかチェック */
		if (IS_FILLED(usrid) && (*mw)->owner_trigger_methos == NULL) {
			if (strcmp(usrid, rset->getString(rset, 14)) == 0) {
				(*mw)->owner_trigger_methos = apr_pcalloc(p, sizeof(apr_int32_t) * METHODS);
				(*mw)->owner_trigger_methos[M_GET]       |= trigger_methods[0];
				(*mw)->owner_trigger_methos[M_PUT]       |= trigger_methods[1];
				(*mw)->owner_trigger_methos[M_POST]      |= trigger_methods[2];
				(*mw)->owner_trigger_methos[M_DELETE]    |= trigger_methods[3];
				(*mw)->owner_trigger_methos[M_OPTIONS]   |= trigger_methods[4];
				(*mw)->owner_trigger_methos[M_PROPFIND]  |= trigger_methods[5];
				(*mw)->owner_trigger_methos[M_PROPPATCH] |= trigger_methods[6];
				(*mw)->owner_trigger_methos[M_MKCOL]     |= trigger_methods[7];
				(*mw)->owner_trigger_methos[M_COPY]      |= trigger_methods[8];
				(*mw)->owner_trigger_methos[M_MOVE]      |= trigger_methods[9];
				(*mw)->owner_trigger_methos[M_LOCK]      |= trigger_methods[10];
				(*mw)->owner_trigger_methos[M_UNLOCK]    |= trigger_methods[11];
				(*mw)->owner_trigger_methos[m_search]    |= trigger_methods[12];
#ifdef DIVY_SUPPORT_MESSENGER
				(*mw)->notification = rset->getInt(rset, 15);
#else	/* !DIVY_SUPPORT_MESSENGER */
				(*mw)->notification = DIVY_NOTICE_MAIL;
#endif	/* DIVY_SUPPORT_MESSENGER */
			}
		}

	}
	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたuriのクライアントモジュール情報を divy_clmodule から削除する。
 * (内部専用)
 * 
 * (note)
 * ・トランザクションが継続中であれば引き継ぎます。
 * ・トランザクションが開始されていなければ開始します。
 *   トランザクションの確定・破棄はしません。
 * ・ts_ctx がNULLであれば、トランザクションは確定or破棄します。
 * 
 * @param r request_rec *
 * @param uri const char * 処理対象のuri
 * @param ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理結果 (0: 成功 / 1: 失敗)
 */
int _remove_clmodule_property(request_rec *r,
				const char *uri,
				divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int delete_cnt		= 0;

	TRACE(p);
	/* 必須項目のチェック */
	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is empty.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_clmodule "
			"WHERE cm_rs_id_c = ("
			"SELECT rs_rs_id_c "
			"FROM dav_resource "
			"WHERE rs_uri_txt = ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s) Reason: %s",
				uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, uri);

	/* SQL実行　失敗時はエラー */
	delete_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_clmodule.(uri = %s) Reason: %s",
				uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* データが既に削除されていた場合はwarningで継続 */
	if (delete_cnt == 0){
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"divy_clmodule\" table did not have a recorde of "
			"this clmodule.(uri = %s) "
			"Probably data is already deleted.", uri);
	}

	/* 使用した資源の片付け */
	if (stmt != NULL) stmt->close(stmt);
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}


/**
 * 指定されたソースコレクション( = linkdbsearch の結果フォルダ) のuri が
 * 持っている共有コレクションの情報を取得して返却する。
 *
 * @param r request_rec *
 * @param uri const char * ソースコレクションのuri
 * @param shlink_pr divy_rdbo_shlink ** 検索結果
 * @param ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理結果 (0: 成功 / 1: 失敗)
 */
static int _get_sharedcollection_property(request_rec *r,
					const char *uri,
					divy_rdbo_shlink **shlink_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	DbResultSet     *rset    = NULL;
	apr_pool_t *p            = r->pool;
	int iscommit            = 0;
	divy_rdbo_shlink *_shlink_pr = NULL;

	/* 初期化 */
	*shlink_pr = NULL;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* 
	 * dav_link_collection テーブルを検索し、
	 * sharedcollection プロパティを取得する。
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				"r.rs_rs_id_c,"
				"r.rs_uri_txt,"
				"r.rs_dispname_vc "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2)/* postgres */
				"FROM dav_resource r"
				" INNER JOIN dav_link_collection lc"
				" ON r.rs_rs_id_c = lc.lc_share_rs_id_c "
				"WHERE lc.lc_uri_txt = ?", 
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
				"FROM dav_resource r, dav_link_collection lc"
				" WHERE r.rs_rs_id_c = lc.lc_share_rs_id_c"
				" AND lc.lc_uri_txt = ?",
#endif
				p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for dav_link_collection. "
			"(uri = %s) Reason: %s", uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	stmt->setString(stmt, 1, uri);

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select dav_link_collection."
			"(uri = %s) Reason: %s", uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 
	 * 結果の取得
	 */
	while (rset->next(rset) == DB_TRUE) {
		if (*shlink_pr == NULL) {
			_shlink_pr       = apr_pcalloc(p, sizeof(divy_rdbo_shlink));
			_shlink_pr->next = NULL;
			*shlink_pr       = _shlink_pr;
		}
		else {
			_shlink_pr->next = apr_pcalloc(p, sizeof(divy_rdbo_shlink));
			_shlink_pr       = _shlink_pr->next;
			_shlink_pr->next = NULL; 
		}

		_shlink_pr->rsid        = rset->getString(rset, 1);
		_shlink_pr->uri         = rset->getString(rset, 2);
		_shlink_pr->displayname = rset->getString(rset, 3);
	}

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたmwatch_pr->uri が示すコレクションにmailwatch_pr->trigger_methods 
 * が示すメール監視状態を追加する。
 *
 * insert 対象テーブル: divy_mailwatch
 * mwatch_pr 必須項目 : uri, usrid, trigger_methods
 * (note) delflag は設定されていても無視します。常に0を設定します。
 *        owner_trigger_methos は無視します。この変数は歴史的な経緯から
 *        書き込み時には監視ユーザの監視フラグを表さないためです。
 *        mwatch_pr->next は無視します。
 * (note)
 *   uri, usrid の組が既にdivy_mailwatch に登録されていないことを呼び出し元で
 *   保証して下さい。
 * (note)
 * トランザクションが継続中であれば、それを引き継ぎます。
 *
 * @param r request_rec *
 * @param mwatch_pr divy_rdbo_mailwatch * 追加するメール監視状況
 * @param ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _insert_mailwatch_property(request_rec *r,
					const divy_rdbo_mailwatch *mwatch_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int insert_cnt          = 0;
	int iscommit            = 0;
	int m_search            = ap_method_number_of("SEARCH");

	TRACE(p);

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Could not operation. "
				"Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_mailwatch テーブルへの新規追加
	 */
	stmt = dbconn->prepareStatement(dbconn, 
			"INSERT INTO divy_mailwatch ("
			"  mw_rs_uri_txt"
			", mw_usr_id_vc"
			", mw_trigger_get"
			", mw_trigger_put"
			", mw_trigger_post"
			", mw_trigger_delete"
			", mw_trigger_options"
			", mw_trigger_propfind"
			", mw_trigger_proppatch"
			", mw_trigger_mkcol"
			", mw_trigger_copy"
			", mw_trigger_move"
			", mw_trigger_lock"
			", mw_trigger_unlock"
			", mw_trigger_search"
#ifdef DIVY_SUPPORT_MESSENGER
			", mw_notification_i"
#endif	/* DIVY_SUPPORT_MESSENGER */
			", mw_l_del_flag)"
			" VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?"
#ifdef DIVY_SUPPORT_MESSENGER
			",?"
#endif	/* DIVY_SUPPORT_MESSENGER */
			",0)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt for "
				"divy_mailwatch insert. "
				"(uri = %s, usrid = %s) Reason: %s", 
				mwatch_pr->uri, mwatch_pr->usrid, 
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	stmt->setString	(stmt, 1,  mwatch_pr->uri);
	stmt->setString	(stmt, 2,  mwatch_pr->usrid);
	stmt->setInt	(stmt, 3,  mwatch_pr->trigger_methods[M_GET]);
	stmt->setInt	(stmt, 4,  mwatch_pr->trigger_methods[M_PUT]);
	stmt->setInt	(stmt, 5,  mwatch_pr->trigger_methods[M_POST]);
	stmt->setInt	(stmt, 6,  mwatch_pr->trigger_methods[M_DELETE]);
	stmt->setInt	(stmt, 7,  mwatch_pr->trigger_methods[M_OPTIONS]);
	stmt->setInt	(stmt, 8,  mwatch_pr->trigger_methods[M_PROPFIND]);
	stmt->setInt	(stmt, 9,  mwatch_pr->trigger_methods[M_PROPPATCH]);
	stmt->setInt	(stmt, 10, mwatch_pr->trigger_methods[M_MKCOL]);
	stmt->setInt	(stmt, 11, mwatch_pr->trigger_methods[M_COPY]);
	stmt->setInt	(stmt, 12, mwatch_pr->trigger_methods[M_MOVE]);
	stmt->setInt	(stmt, 13, mwatch_pr->trigger_methods[M_LOCK]);
	stmt->setInt	(stmt, 14, mwatch_pr->trigger_methods[M_UNLOCK]);
	stmt->setInt	(stmt, 15, mwatch_pr->trigger_methods[m_search]);
#ifdef DIVY_SUPPORT_MESSENGER
	stmt->setInt	(stmt, 16, (int) mwatch_pr->notification);
#endif	/* DIVY_SUPPORT_MESSENGER */

	/* SQLの実行 */
	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_mailwatch data. "
				"(uri = %s, usrid = %s) Reason: %s",
				mwatch_pr->uri, mwatch_pr->usrid,
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	if (insert_cnt == 0) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"Status is OK, but the count of inserting "
				"is 0. Please check divy_mailwatch. "
				"(uri = %s, usrid = %s)",
				mwatch_pr->uri, mwatch_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたmwatch_pr->uri が示すコレクションをmailwatch_pr->trigger_methods 
 * が示すメール監視状態で更新する。
 *
 * update 対象テーブル: divy_mailwatch
 * mwatch_pr 必須項目 : uri, usrid, trigger_methods
 * (note) delflag は設定されていても無視します。常に0を設定します。
 *        owner_trigger_methos は無視します。この変数は歴史的な経緯から
 *        書き込み時には監視ユーザの監視フラグを表さないためです。
 *        mwatch_pr->next は無視します。
 * (note)
 *   uri, usrid の組が既にdivy_mailwatch に登録されていないことを呼び出し元で
 *   保証して下さい。
 * (note)
 * トランザクションが継続中であれば、それを引き継ぎます。
 *
 * @param r request_rec *
 * @param mwatch_pr divy_rdbo_mailwatch * 更新するメール監視状況
 * @param ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _update_mailwatch_property(request_rec *r,
					const divy_rdbo_mailwatch *mwatch_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int update_cnt          = 0;
	int iscommit            = 0;
	int m_search            = ap_method_number_of("SEARCH");

	TRACE(p);

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Could not operation. "
				"Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_mailwatch テーブルの更新
	 */
	stmt = dbconn->prepareStatement(dbconn, 
			"UPDATE divy_mailwatch SET "
			"mw_trigger_get = ?, "
			"mw_trigger_put = ?, "
			"mw_trigger_post = ?, "
			"mw_trigger_delete = ?, "
			"mw_trigger_options = ?, "
			"mw_trigger_propfind = ?, "
			"mw_trigger_proppatch = ?, "
			"mw_trigger_mkcol = ?, "
			"mw_trigger_copy = ?, "
			"mw_trigger_move = ?, "
			"mw_trigger_lock = ?, "
			"mw_trigger_unlock = ?, "
			"mw_trigger_search = ? "
#ifdef DIVY_SUPPORT_MESSENGER
			",mw_notification_i = ? "
#endif	/* DIVY_SUPPORT_MESSENGER */
			"WHERE mw_rs_uri_txt = ? "
			"AND mw_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(uri = %s, usrid = %s) Reason: %s",
				mwatch_pr->uri, mwatch_pr->usrid,
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setInt	(stmt, 1,  mwatch_pr->trigger_methods[M_GET]);
	stmt->setInt	(stmt, 2,  mwatch_pr->trigger_methods[M_PUT]);
	stmt->setInt	(stmt, 3,  mwatch_pr->trigger_methods[M_POST]);
	stmt->setInt	(stmt, 4,  mwatch_pr->trigger_methods[M_DELETE]);
	stmt->setInt	(stmt, 5,  mwatch_pr->trigger_methods[M_OPTIONS]);
	stmt->setInt	(stmt, 6,  mwatch_pr->trigger_methods[M_PROPFIND]);
	stmt->setInt	(stmt, 7,  mwatch_pr->trigger_methods[M_PROPPATCH]);
	stmt->setInt	(stmt, 8,  mwatch_pr->trigger_methods[M_MKCOL]);
	stmt->setInt	(stmt, 9,  mwatch_pr->trigger_methods[M_COPY]);
	stmt->setInt	(stmt, 10, mwatch_pr->trigger_methods[M_MOVE]);
	stmt->setInt	(stmt, 11, mwatch_pr->trigger_methods[M_LOCK]);
	stmt->setInt	(stmt, 12, mwatch_pr->trigger_methods[M_UNLOCK]);
	stmt->setInt	(stmt, 13, mwatch_pr->trigger_methods[m_search]);
#ifdef DIVY_SUPPORT_MESSENGER
	stmt->setInt	(stmt, 14, mwatch_pr->notification);
	stmt->setString	(stmt, 15, mwatch_pr->uri);
	stmt->setString	(stmt, 16, mwatch_pr->usrid);
#else	/* !DIVY_SUPPORT_MESSENGER */
	stmt->setString	(stmt, 14, mwatch_pr->uri);
	stmt->setString	(stmt, 15, mwatch_pr->usrid);
#endif	/* DIVY_SUPPORT_MESSENGER */

	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_mailwatch."
				"(uri = %s, usrid = %s) Reason: %s",
				mwatch_pr->uri, mwatch_pr->usrid,
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* 更新件数が1件以外はWARNING */
	if (update_cnt != 1){
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"Status is OK, but the count of updating "
				"is not 1. Please check divy_mailwatch. "
				"(uri = %s, usrid = %s)", 
				mwatch_pr->uri, mwatch_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたusrid, grpidが示すグループリレーションをdivy_grpmem テーブルに
 * 追加し、usrid が示すユーザにグループが持っていたデフォルトメール監視フラグ
 * (groupmailwatch) を設定する。
 * (note)
 *   ユーザが制限ユーザであればメール監視状態は設定されません。
 *
 * @param r request_rec *
 * @param grpid const char * グループID
 * @param usrid const char * ユーザID
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _insert_grpmem(request_rec *r,
					const char *grpid,
					const char *usrid,
					divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;

	/*
	 * divy_grpmem への追加
	 */
	if (_insert_divy_grpmem(r, grpid, usrid, ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert divy_grpmem.");
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/*
	 * グループが持っていたメール監視フラグのデフォルト値を
	 * ユーザにつける
	 * (note)
	 * 	mailwatch uri  = $root/Group Folder/[$groupid | $grpname]
	 * 	mailwatch user = rusr->usrid
	 */
	if (divy_rdbo_copy_groupmailwatch_to_user(r, grpid, usrid, DIVY_CPMWATCH_NORMAL, ts_ctx)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy mailwatch property."
			"(grpid = %s, usrid = %s)",grpid, usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたrusr が示すユーザリレーションをdivy_grpmem テーブルに追加し、
 * rusr にrusr->grp_uri のグループが持っていたメール監視状態を設定する。
 * (note)
 *   ユーザが制限ユーザであればメール監視状態は設定されません。
 *
 * (note) rusr 必須項目
 * 	grp_uri, usrid
 *
 * (note)
 * 	トランザクションがあれば継続します。
 *
 * @param r request_rec *
 * @param rusr divy_rdbo_rusr * 追加対象ユーザ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
static int _insert_rusr(request_rec *r, const divy_rdbo_rusr *rusr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	char *grpid = NULL;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* グループID を取得する */
	stmt = dbconn->prepareStatement(dbconn,
			"SELECT grp_grp_id_c "
			"FROM divy_grp "
			"WHERE grp_relative_uri_txt = ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (grpid = %s) "
			"Reason: %s", grpid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, rusr->grp_uri);
	/* SQL 実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_grp.(grpuri = %s) "
			"Reason: %s", rusr->grp_uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		grpid = rset->getString(rset, 1);
	}
	else {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get grpid from divy_grp."
			"Maybe this group was deleted before this operation."
			"Please check client cache.(grpuri = %s, usrid = %s)",
			rusr->grp_uri, rusr->usrid);

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_grpmem への追加
	 */
	if (_insert_divy_grpmem(r, grpid, rusr->usrid, ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to insert divy_grpmem.");
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/*
	 * グループが持っていたメール監視フラグのデフォルト値をユーザにつける
	 * (note)
	 * 	mailwatch uri  = $root/Group Folder/[$groupid | $grpname]
	 * 	mailwatch user = rusr->usrid
	 */
	if (divy_rdbo_copy_groupmailwatch_to_user(r, grpid, rusr->usrid, DIVY_CPMWATCH_NORMAL, ts_ctx)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to copy mailwatch property."
			"(grpid = %s, usrid = %s)",grpid, rusr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたgrpid, usrid が示すリレーションエントリを削除する。
 *
 * (note) delete 対象テーブル: divy_grpmem, divy_mailwatch
 *
 * @param r request_rec *
 * @param grpid const char * グループID
 * @param usrid const char * ユーザID
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _remove_grpmem(request_rec *r, const char *grpid, const char *usrid,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	char *grp_col_uri;

	if (IS_EMPTY(grpid) || IS_EMPTY(usrid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"grpid or usrid is EMPTY");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_grpmem の削除
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_grpmem "
			"WHERE grpm_grp_id_c = ?"
			" AND grpm_usr_id_vc = ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(groupid = %s, userid = %s) Reason: %s",
			grpid, usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	stmt->setString(stmt, 1, grpid);
	stmt->setString(stmt, 2, usrid);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_grpmem. "
			"(groupid = %s, userid = %s) Reason: %s",
			grpid, usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * グループコレクションURIの取得
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"SELECT rs_uri_txt "
			"FROM dav_resource "
			"WHERE rs_rs_id_c ="
			" (SELECT grp_rs_id_c FROM divy_grp"
			" WHERE grp_grp_id_c = ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (grpid = %s) "
			"Reason: %s", grpid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, grpid);
	/* SQL 実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select dav_resource and divy_grp.(grpid = %s) "
			"Reason: %s", grpid, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		grp_col_uri = rset->getString(rset, 1);
	}
	else {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get a rsid of group from dav_resource."
			"Maybe this group was deleted before this operation."
			"Please check client cache.(grpid = %s)", grpid);

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	/*
	 * divy_mailwatch の物理削除
	 * (note)
	 * ここは無条件に物理削除してよい。
	 * (メール監視フラグの影響はdav_resourceのリソースにのみ有効であるため。)
	 */
	if (_remove_mailwatch_property(r, grp_col_uri, usrid,
					DIVY_REMWATCH_NORMAL, ts_ctx)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to delete divy_mailwatch. "
			"(groupid = %s, userid = %s)", grpid, usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/*
	 * divy_confirmreading (開封通知)の削除
	 */
	if (divy_support_confirmreading(r)) {
		if (divy_rdbo_remove_confirmreading(r, grp_col_uri, usrid, DAV_INFINITY, ts_ctx)) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to delete divy_confirmreading. "
					"(groupid = %s, userid = %s)", grpid, usrid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return 1;
		}
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたgrpid, sqlname を使って、divy_sqlmem テーブルにエントリを追加する。
 *
 * @param r request_rec *
 * @param grpid const char * グループID
 * @param sqlname const char * SQL名称 
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _insert_sqlmem_using_name(request_rec *r,
					const char *grpid,
					const char *sqlname,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int insert_cnt          = 0;
	int iscommit            = 0;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"INSERT INTO divy_sqlmem "
			"(sqlm_grp_id_c, sqlm_sql_id_c) "
			"(SELECT "
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_GRPID_LEN))", sql_id_c"
			" FROM divy_sql"
			" WHERE sql_label_name_vc = ?)", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(groupid = %s, sqlname = %s) Reason: %s",
			grpid, sqlname, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	stmt->setString(stmt, 1, grpid);
	stmt->setString(stmt, 2, sqlname);

	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_grpmem. "
			"(groupid = %s, sqlname = %s) Reason: %s",
			grpid, sqlname, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (insert_cnt == 0) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Maybe specified sql was deleted before this operation."
			"Please check client cache and divy_grpmem table."
			"(groupid = %s, sqlname = %s)", grpid, sqlname);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 使い終わった資源の解放 */
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたrsql を使って、divy_sqlmem テーブルにエントリを追加する
 * 
 * (note) rsql 必須項目
 * 	labelname, grp_uri
 *
 * @param r request_rec *
 * @param rsql const divy_rdbo_rsql * 追加対象のSQLリレーション
 * @param ts_ctx divy_db_transaction_ctx *
 */
static int _insert_rsql(request_rec *r, const divy_rdbo_rsql *rsql,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int insert_cnt          = 0;
	int iscommit            = 0;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"INSERT INTO divy_sqlmem "
			"(sqlm_grp_id_c, sqlm_sql_id_c) "
			"VALUES ("
			"(SELECT grp_grp_id_c FROM divy_grp"
			" WHERE grp_relative_uri_txt = ?),"
			" (SELECT sql_id_c FROM divy_sql"
			" WHERE sql_label_name_vc = ?))", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(grpuri = %s, sqlname = %s) Reason: %s",
			rsql->grp_uri, rsql->labelname, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	stmt->setString(stmt, 1, rsql->grp_uri);
	stmt->setString(stmt, 2, rsql->labelname);

	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_grpmem. "
			"(grpuri = %s, sqlname = %s) Reason: %s",
			rsql->grp_uri, rsql->labelname, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (insert_cnt == 0) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Status is OK, but the count of inserting is 0."
			"Please check divy_sqlmem.(grpuri = %s, sqlname = %s)",
			rsql->grp_uri, rsql->labelname);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 使い終わった資源の解放 */
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;

}

/**
 * 指定されたgrpid, sqlid のdivy_sqlmem テーブルのエントリを削除する。
 *
 * @param r request_rec *
 * @param grpid const char * グループID
 * @param sqlid const char * SQL シーケンス番号
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _remove_sqlmem(request_rec *r, const char *grpid, const char *sqlid,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_sqlmem "
			"WHERE sqlm_grp_id_c = ?"
			" AND sqlm_sql_id_c = ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(groupid = %s, sqlid = %s) Reason: %s",
			grpid, sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	stmt->setString(stmt, 1, grpid);
	stmt->setString(stmt, 2, sqlid);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_sqlmem. "
			"(groupid = %s, sqlid = %s) Reason: %s",
			grpid, sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}


	/* 使い終わった資源の解放 */
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたgrpid, usrid をdivy_grpmem テーブルに追加する。
 *
 * (note) トランザクションがあれば、継続します。
 *
 * @param r request_rec *
 * @param grpid const char * グループID
 * @param usrid const char * ユーザID
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _insert_divy_grpmem(request_rec *r,
				const char *grpid,
				const char *usrid,
				divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int insert_cnt          = 0;
	int iscommit            = 0;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_grpmem への追加
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"INSERT INTO divy_grpmem "
			"(grpm_grp_id_c, "
			"grpm_usr_id_vc) "
			"VALUES (?, ?)", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(grpid = %s, userid = %s) Reason: %s",
			grpid, usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	stmt->setString(stmt, 1, grpid);
	stmt->setString(stmt, 2, usrid);

	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_grpmem. "
			"(groupid = %s, userid = %s) Reason: %s",
			grpid, usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (insert_cnt == 0) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Status is OK, but the count of inserting is 0."
			"Please check divy_grpmem.(grpid = %s, userid = %s)",
			grpid, usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * rdb_r が示すリソース構造体にコレクションの生成に必要な値を詰める。
 *
 * (note) rdb_r 必須項目
 * 	uri, displayname, rsid
 * (note) 自動的に設定される値
 *	getcontentlength, getcontenttype, resourcetype, physicalpath,
 *	creationdate, getlastmodified, getetag, isversioned,
 *	checkin, checkout, creator, lastmodifier, creator_userid,
 *	lastmodifier_userid
 *
 * @param r request_rec *
 * @param uri const char * URI文字列
 * @param displayname const char * 表示名
 * @param rsid const char * リソースID
 * @param rdb_r divy_rdbo_resource * リソース構造体
 * 			領域を割り当ててから渡してください。
 */
static void _fill_default_collection_property(request_rec *r,
					divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p = r->pool;

	if (rdb_r == NULL || IS_EMPTY(rdb_r->uri) ||
		IS_EMPTY(rdb_r->rsid) || IS_EMPTY(rdb_r->displayname)) {
		return;
	}

	/*
	 * デフォルト値を詰める
	 */
	rdb_r->creationdate        = dav_divy_get_now_epoch();
	rdb_r->getlastmodified     = rdb_r->creationdate;
	rdb_r->getcontentlength    = APR_INT64_C(0);
	rdb_r->getcontenttype      = DIR_MAGIC_TYPE;
	rdb_r->getetag             = (char *) dav_divy_get_etag_string(p,
						rdb_r->rsid,
						rdb_r->getcontentlength,
						rdb_r->creationdate);
	rdb_r->resourcetype        = DIVY_TYPE_COLLECTION;
	rdb_r->depth               = divy_count_dirs(rdb_r->uri);
	rdb_r->isversioned         = 0;
	rdb_r->checkin             = -1;
	rdb_r->checkout            = -1;
	rdb_r->creator             = NULL;
	rdb_r->lastmodifier        = NULL;

	rdb_r->creator_userid      = apr_pstrdup(p, divy_get_userid(r));
	rdb_r->lastmodifier_userid = rdb_r->creator_userid;
	rdb_r->p                   = p;
	rdb_r->next                = NULL;
}

/**
 * 指定されたusr_pr の示すユーザ(usr_pr->next があればそれらのユーザ群)が
 * ユーザQuotaの制限に抵触しなければ、ユーザの消費Quota(使用ディスク容量
 * およびファイル数)を変更する。
 * (note)
 * 	追加サイズ・ファイル数がマイナスの場合、Quotaを負の方向に消費する
 * 	つまり、Quotaを返却します。
 * 	なお、追加サイズ、ファイル数の符号は一致している必要があります。
 * 	一致していない場合、正しい変更が実施出来ません。
 *
 * update 対象テーブル: divy_usrdiskquota
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 *
 * @param r request_rec *
 * @param usr_pr divy_rdbo_usr * ユーザQuotaを保持する構造体
 * @param ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理結果
 * 	(0: 成功 / 1: 失敗 / DIVY_STCODE_QUOTAOVER: ユーザQuota Over)
 */
static int _change_used_userquota(request_rec *r,
					divy_rdbo_usr *usr_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	DbPreparedStmt  *stmt2   = NULL;
	DbResultSet     *rset    = NULL;
	apr_pool_t      *p       = r->pool;
	divy_rdbo_usr uquota     = { 0 };
	divy_rdbo_usr *c_uquota  = NULL;
	int iscommit             = 0;
	int is_reduction;	/* Quotaを消費させる変更かどうか */
	apr_int64_t sum_usedst, sum_usedres;
	char *msg;
	apr_hash_t *uquota_h     = apr_hash_make(p);
	char *sql;
	divy_linkedlist_t *in_clause;
	divy_db_bind_ctx *bindctx = NULL;
	divy_db_bind_ctx_idx *idx;
	const char *in_clause_str = NULL;
	divy_cset_t *user_set     = NULL;
	int i;

	TRACE(p);

	/* 入力不正のチェック */
	if (usr_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"usr_pr is NULL");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_usrdiskquota から対象ユーザ全てのエントリを取得する
	 */

	/* 更新対象者のユーザID集合を算出 */
	
	for (c_uquota = usr_pr; c_uquota; c_uquota = c_uquota->next) {

		/* 入力されたユーザ情報の検証 */
		if ((c_uquota->usedst  > APR_INT64_C(0) &&
		     c_uquota->usedres < APR_INT64_C(0)) ||
		    (c_uquota->usedst  < APR_INT64_C(0) &&
		     c_uquota->usedres > APR_INT64_C(0))) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The amount of Quota is invalid."
				"(userid = %s, usedst = %"APR_INT64_T_FMT
				", usedres = %"APR_INT64_T_FMT")",
				c_uquota->usrid, usr_pr->usedst, usr_pr->usedres);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/* 全て0ならばユーザQuotaの変更をする必要なし */
		if (c_uquota->usedst  == APR_INT64_C(0) &&
		    c_uquota->usedres == APR_INT64_C(0)) {
			continue;	/* 次のユーザへ */
		}

		if (user_set == NULL) {
			user_set = divy_cset_make(p);
		}
		divy_cset_set(user_set, c_uquota->usrid);

		/* ついでにユーザ情報をハッシュに入れておく */
		apr_hash_set(uquota_h, c_uquota->usrid, APR_HASH_KEY_STRING, c_uquota);
	}

	/* 更新すべきユーザが一人もいなかったら？ */
	if (apr_hash_count(uquota_h) == 0) {
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		return 0;	/* 正常終了です */
	}

	/* バインドコンテキスト/インデックスを生成 */
	bindctx = divy_db_bindcontext_make(p, user_set, -1);
	idx     = divy_db_bindcontext_first(bindctx);

	/* バインド変数文字列の取得 */
	divy_db_bindcontext_get_bindstr(idx, &in_clause_str);

	/* SQL文の組み立て */
	sql = apr_psprintf(p,	"SELECT "
				"usqt_usr_id_vc,"
				"usqt_used_st_bi,"
				"usqt_max_st_bi,"
				"usqt_used_res_bi,"
				"usqt_max_res_bi "
				"FROM divy_usrdiskquota "
				"WHERE usqt_usr_id_vc IN (%s) "
				"FOR UPDATE", in_clause_str);

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for update "
			"divy_usrdiskquota). Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	i = 0;
	divy_db_bindcontext_get_list(idx, &in_clause);
	for (; in_clause; in_clause = in_clause->next) {
		stmt->setString(stmt, ++i, in_clause->val);
	}

	/* SQLの実行 */
	rset = stmt->executeQueryForUpdate(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to SELECT divy_usrdiskquota. Reason: %s",
			rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset);	rset = NULL;
		if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

		return 1;
	}

	/* ユーザQuotaテーブルの更新用SQLの準備 */
	stmt2 = dbconn->prepareStatement(dbconn, 
				"UPDATE divy_usrdiskquota "
				"SET "
				"usqt_used_st_bi = ?,"
				"usqt_used_res_bi = ? "
				"WHERE usqt_usr_id_vc = ? ", p);
	if (stmt2->getCode(stmt2) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for update "
			"divy_usrdiskquota). Reason: %s", stmt2->getMsg(stmt2));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)  rset->close(rset);	rset = NULL;
		if (stmt != NULL)  stmt->close(stmt);	stmt = NULL;
		if (stmt2 != NULL) stmt2->close(stmt2);	stmt2 = NULL;

		return 1;
	}

	/*
	 * ユーザQuotaの検証と更新処理
	 */
	while (rset->next(rset) == DB_TRUE) {
		/* 値の取得 */
		uquota.usrid   = rset->getString(rset, 1);
		uquota.usedst  = rset->getBigInt(rset, 2);
		uquota.maxst   = rset->getBigInt(rset, 3);
		uquota.usedres = rset->getBigInt(rset, 4);
		uquota.maxres  = rset->getBigInt(rset, 5);

		/* 変更したいユーザQuotaの取得 */
		c_uquota = apr_hash_get(uquota_h, uquota.usrid, APR_HASH_KEY_STRING);

		/* この変更はQuotaを消費する or 返すのか？ */
		if (c_uquota->usedst >= APR_INT64_C(0) && c_uquota->usedres >= APR_INT64_C(0)) {
			is_reduction = 1;	/* 消費する */
		}
		else {
			is_reduction = 0;	/* 返す */
		}

		/*
		 * ユーザQuotaの検証
		 * [ 処理開始条件 ]
		 * 	* 無制限Quotaを持たないこと
		 *
		 * [ Quota エラーの条件 ]
		 * 	変更サイズが"正"の場合:
		 * 	------------------------
		 * 	　・サイズQuota制限に抵触した (error)
		 * 	　・ファイス数Quota制限に抵触した (error)
		 * 	  ・変更後の消費Quotaが負になった (データ不正,warning) 
		 *
		 * 	変更サイズが"負"の場合:
		 * 	------------------------
		 * 	  ・変更後の消費Quotaが負になった (データ不正,warning)
		 *  (note)
		 *  	変更サイズがマイナスの場合、ユーザは消費Quotaを
		 *  	減らそうと努力しているということを意味しているので、
		 *　	QuotaOverを出してはならない。
		 */
		sum_usedst  = c_uquota->usedst  + uquota.usedst;
		sum_usedres = c_uquota->usedres + uquota.usedres;

		/* サイズQuota に抵触した */
		if (uquota.maxst != DIVY_UNLIMITED_USRQUOTA_BYTES &&
		    is_reduction && sum_usedst > uquota.maxst) {
			/* メッセージの作成 */
			msg = apr_psprintf(p,   "userid = %s, "
					"change size = %"APR_INT64_T_FMT
					", usedst = %"APR_INT64_T_FMT
					", maxst = %"APR_INT64_T_FMT,
					c_uquota->usrid, c_uquota->usedst,
					uquota.usedst, uquota.maxst);

			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"There is not enough storage to write to "
				"this resource for user-quota limit.(%s)", msg);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL)  rset->close(rset);	rset  = NULL;
			if (stmt != NULL)  stmt->close(stmt);	stmt  = NULL;
			if (stmt2 != NULL) stmt2->close(stmt2);	stmt2 = NULL;

			return DIVY_STCODE_QUOTAOVER;
		}
		/* ファイス数Quota制限に抵触した */
		else if (uquota.maxres != DIVY_UNLIMITED_USRQUOTA_FILES &&
			 is_reduction && sum_usedres > uquota.maxres) {
			/* メッセージの作成 */
			msg = apr_psprintf(p,   "userid = %s, "
					"change size = %"APR_INT64_T_FMT
					", usedres = %"APR_INT64_T_FMT
					", maxres = %"APR_INT64_T_FMT,
					c_uquota->usrid, c_uquota->usedres,
					uquota.usedres, uquota.maxres);
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"There is not enough storage to write to "
				"this resource for user-quota limit.(%s)", msg);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL)  rset->close(rset);	rset  = NULL;
			if (stmt != NULL)  stmt->close(stmt);	stmt  = NULL;
			if (stmt2 != NULL) stmt2->close(stmt2);	stmt2 = NULL;

			return DIVY_STCODE_QUOTAOVER;
		}

		/* 変更後の消費Quotaサイズが負になった */
		if (sum_usedst < APR_INT64_C(0)) {
			ERRLOG3(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"The capacity of used storage is negative value. "
				"Maybe it show trouble, but we set 0 value."
				"(userid = %s, change size = %"APR_INT64_T_FMT
				", usedst = %"APR_INT64_T_FMT")",
				c_uquota->usrid, c_uquota->usedst, uquota.usedst);
			sum_usedst = APR_INT64_C(0);
		}

		/* 変更後の消費Quotaファイル数が負になった */
		if (sum_usedres < APR_INT64_C(0)) {
			ERRLOG3(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"The capacity of used resource is negative value. "
				"Maybe it show trouble, but we set 0 value."
				"(userid = %s, change count = %"APR_INT64_T_FMT
				", usedres = %"APR_INT64_T_FMT")",
				c_uquota->usrid, c_uquota->usedres, uquota.usedres);
			sum_usedres = APR_INT64_C(0);
		}

		/*
		 * divy_usrdiskquota テーブルの更新
		 */
		/* バインド */
		stmt2->setBigInt(stmt2,  1, sum_usedst);
		stmt2->setBigInt(stmt2,  2, sum_usedres);
		stmt2->setString(stmt2,  3, c_uquota->usrid);

		/* SQLの実行 */
		(void) stmt2->executeUpdate(stmt2, p);
		if (stmt2->getCode(stmt2) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to UPDATE divy_usrdiskquota. (userid = %s) "
				"Reason: %s", c_uquota->usrid, stmt2->getMsg(stmt2));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL)  rset->close(rset);	rset  = NULL;
			if (stmt != NULL)  stmt->close(stmt);	stmt  = NULL;
			if (stmt2 != NULL) stmt2->close(stmt2);	stmt2 = NULL;

			return 1;
		}
	}

	/* 使用した資源を解放する */
	if (rset != NULL)  rset->close(rset);   rset = NULL;
	if (stmt != NULL)  stmt->close(stmt);   stmt = NULL;
	if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * 指定された使用容量を加えてもシステムQuotaの制限に抵触しなければ、
 * システムQuota(使用ディスク容量およびファイル数)を変更する。
 * (note)
 * 	追加サイズ・ファイル数がマイナスの場合、Quotaを負の方向に消費する
 * 	つまり、Quotaを返却します。
 * 	なお、追加サイズ、ファイル数の符号は一致している必要があります。
 * 	一致していない場合、正しい変更が実施出来ません。
 *
 * update 対象テーブル: divy_diskquota
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 *
 * @param r request_rec *
 * @param sysquota_pr divy_rdbo_diskquota * システムQuotaの変更量
 * @param ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理結果
 * 	(0: 成功 / 1: 失敗 / DIVY_STCODE_QUOTAOVER: システムQuota Over)
 */
static int _change_used_sysquota(request_rec *r,
					divy_rdbo_diskquota *sysquota_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	apr_pool_t      *p       = r->pool;
	divy_rdbo_diskquota squota = { 0 };
	int iscommit             = 0;
	int is_reduction;	/* Quotaを消費させる変更かどうか */
	int ret;
	int is_unlimited_bytes   = 0;	/* 無制限システムQuotaサイズかどうか */
	int is_unlimited_files   = 0;	/* 無制限システムQuotaファイル数かどうか */
	apr_int64_t sum_usedst, sum_usedres;
	char *msg;

	TRACE(p);

	/* 入力不正のチェック */
	if (sysquota_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"sysquota_pr is NULL");
		return 1;
	}

	if ((sysquota_pr->usedst > 0 && sysquota_pr->usedres < 0) ||
	    (sysquota_pr->usedst < 0 && sysquota_pr->usedres > 0)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The amount of Quota is invalid."
			"(usedst = %"APR_INT64_T_FMT
			", usedres = %"APR_INT64_T_FMT")",
			sysquota_pr->usedst, sysquota_pr->usedres);
		return 1;
	}

	/* 全て0バイトならば以下は必要なし */
	if (sysquota_pr->usedst == APR_INT64_C(0) &&
	    sysquota_pr->usedres == APR_INT64_C(0)) {
		return 0;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* この変更はQuotaを消費する or 返すのか？ */
	if (sysquota_pr->usedst >= 0 && sysquota_pr->usedres >= 0) {
		is_reduction = 1;	/* 消費する */
	}
	else {
		is_reduction = 0;	/* 返す */
	}

	/*
	 * 無制限システムQuotaを持つかどうか検証する
	 * (エントリロックを行う)
	 */
	squota.uri  = apr_pstrdup(p, dav_divy_get_root_uri(r));
	squota.type = DIVY_QTTYPE_SYSTEM;

	ret = divy_rdbo_get_systemquota(r, &squota, 1/* lock */, ts_ctx);
	if (ret == DIVY_STCODE_SYSQUOTA_NOT_EXIST) {
		/* システムQuotaテーブルエントリが存在しなかった */
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		return 0;	/* 正常 (互換性に関する考慮) */
	}
	else if (ret) {
		/* 予期せぬエラー発生 */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get system-quota information. "
			"(location = %s)", squota.uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	if (squota.maxst == DIVY_UNLIMITED_SYSQUOTA_BYTES) {
		is_unlimited_bytes = 1;	/* 無制限システムQuota(サイズ) を持つ */
	}

	if (squota.maxres == DIVY_UNLIMITED_SYSQUOTA_FILES) {
		is_unlimited_files = 1;	/* 無制限システムQuota(ファイル数) を持つ */
	}

	/*
	 * システムQuotaの検証
	 *
	 * [ 処理開始条件 ]
	 * 	* 無制限Quotaを持たないこと
	 *
	 * [ Quota エラーの条件 ]
	 * 	変更サイズが"正"の場合:
	 * 	------------------------
	 * 	　・サイズQuota制限に抵触した (error)
	 * 	　・ファイス数Quota制限に抵触した (error)
	 * 	  ・変更後の消費Quotaが負になった (データ不正,warning) 
	 *
	 * 	変更サイズが"負"の場合:
	 * 	------------------------
	 * 	  ・変更後の消費Quotaが負になった (データ不正,warning)
	 *  (note)
	 *  	変更サイズがマイナスの場合、ユーザは消費Quotaを
	 *  	減らそうと努力しているということを意味しているので、
	 *　	QuotaOverを出してはならない。
	 */
	sum_usedst  = sysquota_pr->usedst  + squota.usedst;
	sum_usedres = sysquota_pr->usedres + squota.usedres;

	/* サイズQuota に抵触した */
	if (!is_unlimited_bytes && is_reduction && sum_usedst > squota.maxst) {
		/* メッセージの作成 */
		msg = apr_psprintf(p,	"location = %s"
					", change size = %"APR_INT64_T_FMT
					", usedst = %"APR_INT64_T_FMT
					", maxst = %"APR_INT64_T_FMT,
					squota.uri, sysquota_pr->usedst,
					squota.usedst, squota.maxst);

		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"There is not enough storage to write to "
			"this resource for system-quota limit.(%s)", msg);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return DIVY_STCODE_QUOTAOVER;
	}
	/* ファイス数Quota制限に抵触した */
	else if (!is_unlimited_files && is_reduction && sum_usedres > squota.maxres) {
		/* メッセージの作成 */
		msg = apr_psprintf(p,	"location = %s"
					", change size = %"APR_INT64_T_FMT
					", usedres = %"APR_INT64_T_FMT
					", maxres = %"APR_INT64_T_FMT,
					squota.uri, sysquota_pr->usedres,
					squota.usedres, squota.maxres);
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"There is not enough storage to write to "
			"this resource for system-quota limit.(%s)", msg);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return DIVY_STCODE_QUOTAOVER;
	}

	/* 変更後の消費Quotaサイズが負になった */
	if (sum_usedst < 0) {
		ERRLOG3(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"The capacity of used storage is negative value. "
			"Maybe it show trouble, but we set 0 value."
			"(location = %s, change size = %"APR_INT64_T_FMT
			", usedst = %"APR_INT64_T_FMT")",
			squota.uri, sysquota_pr->usedst, squota.usedst);
		sum_usedst = APR_INT64_C(0);
	}

	/* 変更後の消費Quotaファイル数が負になった */
	if (sum_usedres < 0) {
		ERRLOG3(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"The capacity of used resource is negative value. "
			"Maybe it show trouble, but we set 0 value."
			"(location = %s, change count = %"APR_INT64_T_FMT
			", usedres = %"APR_INT64_T_FMT")",
			squota.uri, sysquota_pr->usedres, squota.usedres);
		sum_usedres = APR_INT64_C(0);
	}

	/*
	 * ユーザQuotaテーブルの更新
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"UPDATE divy_diskquota "
				"SET "
				"dkqt_used_st_bi = ?,"
				"dkqt_used_res_bi = ? "
				"WHERE dkqt_uri_txt = ? "
				"AND dkqt_type_i = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for update "
			"divy_diskquota). (location = %s) "
			"Reason: %s", squota.uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setBigInt(stmt, 1, sum_usedst);
	stmt->setBigInt(stmt, 2, sum_usedres);
	stmt->setString(stmt, 3, squota.uri);
	stmt->setInt(stmt,    4, squota.type);

	/* SQLの実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to UPDATE divy_diskquota. (location = %s) "
			"Reason: %s", squota.uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

#ifdef DIVY_SUPPORT_GROUPQUOTA
/**
 * 指定された仕様容量を加えてもグループQuotaの制限に接触しなければ
 * グループQuota（ディスク容量及びファイル数）を変更する。
 *
 * update 対象テーブル: divy_grpdiskquota
 * (note)
 *   トランザクションが継続中であれば引き継ぐ
 *
 * @param r request_rec *
 * @param grpquota_pr divy_rdbo_grpquota * グループQuotaの変更量
 * @param ctx divy_db_transaction_ctx * トランザクションコンテキスト
 *  (0: 成功 / 1:失敗 / DIVY_STCODE_QUOTAOVER: Quota Over)
 */
static int _change_used_grpquota(request_rec *r,
				divy_rdbo_grpquota *grpquota_pr,
				divy_db_transaction_ctx *tx_ctx)
{
	apr_pool_t *p = r->pool;

	TRACE(p);
	/* FIXME コードを追加しなさい*/

	return 0;
}
#endif /* DIVY_SUPPORT_GROUPQUOTA */

/**
 * 指定されたuri のdav_resource エントリを取得して、該当レコードをDBロックする。
 * (note)
 * 	rdb_r は呼び出し側で必ずalloc しておくこと。
 *
 * 必須パラメータ: rdb_r->uri
 *
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource * 取得したレコード(呼び出し側でallocすること)
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 * 	(0: 成功 / 1: 予期しないエラー /
 * 	DIVY_STCODE_NOT_EXIST : 対象リソースが存在しなかった)
 */
static int _lock_resource_entry(request_rec *r,
					divy_rdbo_resource *rdb_r,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	int iscommit            = 0;

	TRACE(p);

	/* 入力不正のチェック */
	if (rdb_r == NULL || IS_EMPTY(rdb_r->uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rdb_r is NULL or uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT"
				" rs_rs_id_c,"
				" rs_uri_txt,"
				" rs_dispname_vc,"
				" rs_create_bi,"
				" rs_get_cont_lang_vc,"
				" rs_get_cont_len_bi,"
				" rs_get_cont_type_vc,"
				" rs_get_etag_txt,"
				" rs_get_lastmodified_bi,"
				" rs_resourcetype_i,"
				" rs_depth_i,"
				" rs_isversioned_i,"
				" rs_checkin_i,"
				" rs_checkout_i,"
				" rs_physical_path_txt,"
				" rs_creator_usr_id_vc,"
				" rs_lastmodifier_usr_id_vc "
				"FROM dav_resource "
				"WHERE rs_uri_txt = ? "
				"FOR UPDATE", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s) "
			"Reason: %s", rdb_r->uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, rdb_r->uri);

	/* SQL実行 */
	rset = stmt->executeQueryForUpdate(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select dav_resource. (uri = %s) "
			"Reason: %s", rdb_r->uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/*
	 * 結果の取得 
	 */
	if (rset->next(rset) == DB_TRUE) {
		/* rsid, uri の取得 */
		rdb_r->rsid               = rset->getString(rset, 1);
		rdb_r->uri                = rset->getString(rset, 2);
		rdb_r->displayname        = rset->getString(rset, 3);
		rdb_r->creationdate       = (time_t) rset->getBigInt(rset, 4);
		rdb_r->getcontentlanguage = rset->getString(rset, 5);
		rdb_r->getcontentlength   = rset->getBigInt(rset, 6);
		rdb_r->getcontenttype     = rset->getString(rset, 7);
		rdb_r->getetag            = rset->getString(rset, 8);
		rdb_r->getlastmodified    = (time_t) rset->getBigInt(rset, 9);
		rdb_r->resourcetype       = rset->getInt(rset,   10);
		rdb_r->depth              = rset->getInt(rset,   11);
		rdb_r->isversioned        = rset->getInt(rset,   12);
		rdb_r->checkin            = rset->getInt(rset,   13);
		rdb_r->checkout           = rset->getInt(rset,   14);
		rdb_r->physicalpath       = rset->getString(rset,15);
		rdb_r->creator_userid     = rset->getString(rset,16);
		rdb_r->lastmodifier_userid= rset->getString(rset,17);

		rdb_r->p    = p;
		rdb_r->next = NULL;	/* sentinel */
	}
	else {
		/* エントリが存在しなかった
		 * (note)
		 * 	この関数はエントリが存在しなくてもエラーにはしません。
		 * 	それが何を意味しているのかは呼び出し側で判断して下さい。
		 */
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset);	rset = NULL;
		if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

		return DIVY_STCODE_NOT_EXIST;
	}

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * uri が持っているごみ箱プロパティ(trashinformation) の情報を取得する。
 * (note)
 * 	* トランザクションが継続中であれば引き継ぎます。
 * 	* uri がごみ箱またはその直下のURIではなければ情報は取得出来ません。
 *
 * @param r request_rec *
 * @param uri const char * ごみ箱またはその直下のURI
 * @param depth int
 * @param trash_pr_hash apr_hash_t ** rsid - divy_rdbo_trash * のハッシュ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 正常 / 1: 異常)
 */
static int _get_trash_property(request_rec *r,
					const char *uri,
					int depth,
					apr_hash_t **trash_pr_hash,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	int iscommit            = 0;
	divy_sbuf *sql_buf      = NULL;
	divy_rdbo_trash *trash  = NULL;
	char *sql_base          =
				"SELECT"
				" tr.tr_rs_id_c,"
				" tr.tr_deletion_bi,"
				" tr.tr_deleter_id_vc,"
				" u.usr_fullname_vc AS tr_deleter_name_vc "
				"FROM divy_trash_info tr ";

	/* 入力不正のチェック */
	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}
	/* depth がinfinity ならば1 に置き換える
	 * <-- ごみ箱プロパティはdepth = 0, 1 にしかないので
	 */
	if (depth == DAV_INFINITY) {
		depth = 1;
	}

	*trash_pr_hash = NULL;	/* 初期化 */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL 文の組み立て */
	divy_sbuf_create(p, &sql_buf, 800);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf, sql_base);
	if (depth == 0) {
		divy_sbuf_append(sql_buf,
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"INNER JOIN dav_resource r"
				" ON tr.tr_rs_id_c = r.rs_rs_id_c "
				"LEFT JOIN divy_usr u "
				"ON tr.tr_deleter_id_vc = u.usr_usr_id_vc "
				"WHERE r.rs_uri_txt = ? "
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				", dav_resource r, divy_usr u "
				"WHERE tr.tr_rs_id_c = r.rs_rs_id_c "
				"AND tr.tr_deleter_id_vc = u.usr_usr_id_vc "
				"AND r.rs_uri_txt = ? "
#endif

				);
	}
	else {
		divy_sbuf_append(sql_buf,
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"INNER JOIN dav_resource r"
				" ON tr.tr_rs_id_c = r.rs_rs_id_c "
				"LEFT JOIN divy_usr u "
				"ON tr.tr_deleter_id_vc = u.usr_usr_id_vc "
				"WHERE r.rs_uri_txt = ? "
				"OR (r.rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND r.rs_depth_i = ?) "
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				", dav_resource r, divy_usr u "
				"WHERE tr.tr_rs_id_c = r.rs_rs_id_c "
				"AND tr.tr_deleter_id_vc = u.usr_usr_id_vc "
				"AND (r.rs_uri_txt = ? "
				" OR (r.rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND r.rs_depth_i = ?)) "
#endif

				);
	}

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s) "
			"Reason: %s", uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, uri);
	if (depth == 1) {
		stmt->setString(stmt, 2, apr_pstrcat(p,
				stmt->escWildCard(stmt, uri), "/%", NULL));
		stmt->setInt(stmt, 3, divy_count_dirs(uri) + 1);
	}

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_trash_info. (uri = %s) "
			"Reason: %s", uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/*
	 * 結果の取得 
	 */
	while (rset->next(rset) == DB_TRUE) {

		trash = apr_pcalloc(p, sizeof(divy_rdbo_trash));
		trash->rsid         = rset->getString(rset, 1);
		divy_format_time_t(p, rset->getBigInt(rset, 2),
				DIVY_TIME_STYLE_ISO8601, &trash->daletion_date);
		trash->deleter_id   = rset->getString(rset, 3);
		trash->deleter_name = rset->getString(rset, 4);

		/* rsid - (divy_rdbo_trash *) のハッシュを作る */
		if (*trash_pr_hash == NULL) {
			*trash_pr_hash = apr_hash_make(p);
		}
		apr_hash_set(*trash_pr_hash, trash->rsid,
						APR_HASH_KEY_STRING, trash);
	}

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * リソースID rsid、削除日付delete_id, 削除ユーザID deleter_id の情報を
 * ごみ箱プロパティとして登録する。
 * (note)
 * 	* トランザクションが継続中であれば引き継ぎます。
 * 	* 古いエントリが既にDBに格納されていれば、それらを削除して
 * 	  新たにINSERT します。
 *
 * @param r request_rec *
 * @param rsid const char * リソースID
 * @param deletion_bi time_t 削除日付
 * @param deleter_id const char * 削除ユーザID
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス(0: 正常 / 1: 失敗)
 */
static int _insert_trash_property(request_rec *r,
					const char *rsid,
					time_t deletion_bi,
					const char *deleter_id,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	apr_pool_t      *p       = r->pool;
	int iscommit             = 0;

	/* 入力不正のチェック */
	if (IS_EMPTY(rsid) || deletion_bi == 0 || IS_EMPTY(deleter_id)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There were some EMPTY value.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_trash_info から同rsid のエントリを削除する
	 * (note) この対応はディレクティブを運用中に不正に操作された結果
	 * 	生じ得るデータ不整合を防ぐためのコードです。大抵の場合
	 * 	無駄になりますが、仕方ありません。
	 */
	if (_remove_trash_property(r, rsid, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to remove previous entry from divy_trash_info "
			"(rsid = %s).", rsid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/*
	 * divy_trash_info へのデータ追加
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO divy_trash_info "
				"(tr_rs_id_c,"
				" tr_deletion_bi,"
				" tr_deleter_id_vc) "
				"VALUES (?,?,?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (rsid = %s) "
			"Reason: %s", rsid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, rsid);
	stmt->setBigInt(stmt, 2, deletion_bi);
	stmt->setString(stmt, 3, deleter_id);

	/* SQLの実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to INSERT divy_trash_info. (rsid = %s) "
			"Reason: %s", rsid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * リソースID rsid のごみ箱プロパティを削除する。
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 *
 * @param r request_rec *
 * @param rsid const char * リソースID
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス(0: 正常 / 1: 失敗)
 */
static int _remove_trash_property(request_rec *r,
					const char *rsid,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	apr_pool_t      *p       = r->pool;
	int iscommit             = 0;

	/* 入力不正のチェック */
	if (IS_EMPTY(rsid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rsid is EMPTY");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_trash_info からデータを削除
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"DELETE FROM divy_trash_info "
				"WHERE tr_rs_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (rsid = %s) "
			"Reason: %s", rsid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, rsid);

	/* SQLの実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to DELETE divy_trash_info. (rsid = %s) "
			"Reason: %s", rsid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたごみ箱フォルダのURIであるtrash_uri 以下リソースが持っている全ての
 * ごみ箱プロパティ(divy_trash_info エントリ)を削除する。
 *
 * @param r request_rec *
 * @param trash_uri const char *
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
static int _remove_hierarchy_trash_property(request_rec *r,
					const char *trash_uri,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	apr_pool_t      *p       = r->pool;
	int iscommit             = 0;

	/* 入力不正のチェック */
	if (IS_EMPTY(trash_uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"trash_uri is EMPTY");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_trash_info からデータを削除
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"DELETE FROM divy_trash_info "
				"WHERE tr_rs_id_c IN "
				"(SELECT rs_rs_id_c FROM dav_resource "
				" WHERE rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE")", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (trash_uri = %s) "
			"Reason: %s", trash_uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, apr_pstrcat(p,
			stmt->escWildCard(stmt, trash_uri), "/%", NULL));

	/* SQLの実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to DELETE divy_trash_info. (uri = %s) "
			"Reason: %s", trash_uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたuri, depth で取得されるリソースの状態/属性プロパティを取得する。
 * (note)
 *	* depth がinifinity の場合にはステータスを返却しません。(パフォーマンスの観点)
 *	* この関数はパフォーマンス上の考慮のためPROPFINDに特化した形式で
 *	  状態/属性プロパティを返却します。他からは使わない方がいいでしょう。
 *
 * [ 状態/属性プロパティの付き方 ]
 *   * 対象リソースが存在しない   -> なし
 *   * グループ自身               -> 有効/無効な状態/属性プロパティはありません
 *   * グループダ直下のリソース   -> 有効/無効な状態/属性プロパティが記録されます
 *   * グループ直下の下のリソース -> 間接的に有効化された状態/属性プロパティは全て記録
 *                                   間接的に無効化された状態/属性プロパティは無視
 *   * 上記以外                   -> なし (外部仕様)
 *
 * @param r request_rec *
 * @param uri const char *
 * @param depth int
 * @param rstate_pr_hash apr_hash_t ** rsid がキー、divy_rdbo_resourcestate * がval
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
static int _get_hierarchy_resourcestate_property(request_rec *r,
					const char *uri,
					int depth,
					apr_hash_t **rstate_pr_hash,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	int iscommit            = 0;
	divy_rdbo_resourcestate *rt, *first = NULL;
	char *escaped_uri, *rsid, *geturi;
	const char *root_uri;
	int rs_depth;
	divy_uri_spec *u_spec   = NULL;
	divy_sbuf *sql_buf      = NULL;
	divy_rdbo_box *box_pr   = NULL;
	int idx                 = 0;

	*rstate_pr_hash = NULL;	/* 初期化 */

	/* 入力不正のチェック */
	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}

	/* infinity には答えられません */
	if (depth == DAV_INFINITY) {
		return 0;	/* 正常に検索したようにする */
	}

	/* uri のパース */
	root_uri = dav_divy_get_root_uri(r);
	divy_parse_uri(p, root_uri, uri, &u_spec);

	/* グループ自身(depth = 1) またはそれ以下のリソースでなければ存在し得ません */
	if (!((u_spec->infotype == DIVY_INFOTYPE_group_e && depth == 1) ||
		u_spec->infotype == DIVY_INFOTYPE_group_e_regular)) {
		return 0;
	}

	TRACE(p);

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の組み立て */
	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */

	rs_depth = divy_count_dirs(uri);	/* URI の階層数を取る */

	/*
	 * uri がグループコレクション自身の場合 (depth = 1)
	 * (depth = 0 は事前に排除しています)
	 */
	if (u_spec->infotype == DIVY_INFOTYPE_group_e) {
		/* グループコレクション自身は状態/属性プロパティを持たないが, 直下の
		 * リソース/コレクションは保持している可能性があるので検索する */

		/* 状態/属性プロパティが有効or無効であるリソースの集合の取得 */
		divy_sbuf_append(sql_buf,
					"SELECT"
					" rs.rs_rs_id_c,"
					" rt.rt_uri_txt,"
					" rs.rs_uri_txt,"
					" rt.rt_type_i,"
					" rt.rt_effective_i ");

		if (divy_support_tfbox(r)) {
			divy_sbuf_append(sql_buf,
					",box.box_allowed_origin_c,"
					" box.box_shorten_c,"
					" box.box_flag_i,"
					" box.box_passwd_vc,"
					" box.box_create_bi,"
					" box.box_expiration_bi,"
					" box.box_creator_usr_id_vc,"
					" usr.usr_fullname_vc,"
					" box.box_greeting_vc,"
					" box.box_message_vc,"
					" box.box_viewcount_bi ");

			if (divy_support_2FA(r)) {
				divy_sbuf_append(sql_buf, ",box_to_mailaddr_vc");
			}

		}


		divy_sbuf_append(sql_buf,
					" FROM dav_resource rs "
					" LEFT JOIN divy_resourcestate rt"
					" ON rs.rs_uri_txt = rt.rt_uri_txt ");

		if (divy_support_tfbox(r)) {
			divy_sbuf_append(sql_buf,
					" LEFT OUTER JOIN divy_box box"
					" ON rs.rs_uri_txt = box.box_uri_txt "
					" LEFT OUTER JOIN divy_usr usr"
					" ON box.box_creator_usr_id_vc = usr.usr_usr_id_vc "
					);
		}

		divy_sbuf_append(sql_buf,
					"WHERE rs.rs_uri_txt LIKE ? "
					DIVY_DBFUNC_ESCAPE_CLAUSE" "
					"AND rs.rs_depth_i = ? ");

		/* SQL文の準備 */
		stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (uri = %s, depth = %d) "
				"Reason: %s", uri, depth, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		escaped_uri = apr_pstrcat(p, stmt->escWildCard(stmt, uri), "/%", NULL);

		/* バインド */
		stmt->setString(stmt, 1, escaped_uri);
		stmt->setInt(stmt,    2, rs_depth + 1);

		/* SQL実行 */
		rset = stmt->executeQuery(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get resource state. (uri = %s, depth = %d) "
				"Reason: %s", uri, depth, rset->getMsg(rset));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* 結果の取得 */
		while (rset->next(rset) == DB_TRUE) {
			idx = 1;
			rsid   = rset->getString(rset, idx++);
			geturi = rset->getString(rset, idx++);

			first = NULL;
			/* このrsid は別のtypeで既に取得済みか? */
			if (*rstate_pr_hash != NULL) {
				first = apr_hash_get(*rstate_pr_hash, rsid, APR_HASH_KEY_STRING);
			}

			if (first == NULL) {
				first = rt = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
				/* ハッシュに詰める */
				if (*rstate_pr_hash == NULL) {
					*rstate_pr_hash = apr_hash_make(p);
				}
				apr_hash_set(*rstate_pr_hash, rsid, APR_HASH_KEY_STRING, first);
			}
			else {
				/* 末尾になるまでスキップ */
				for (rt = first; rt->next; rt = rt->next);
				rt->next = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
				rt = rt->next;
			}

			rt->rsid = rsid;
			rt->uri  = rset->getString(rset, idx++);
			rt->depth= 0;	/* 記録しない */

			/* 状態/属性プロパティが存在した場合 */
			if (IS_FILLED(geturi)) {
				rt->type      = rset->getInt(rset, idx++);
				rt->effective = rset->getInt(rset, idx++);
				if (divy_support_tfbox(r)) {
					box_pr = apr_pcalloc(p, sizeof(divy_rdbo_box));

					box_pr->allowed_origin = rset->getString(rset, idx++);
					box_pr->shorten        = rset->getString(rset, idx++);
					box_pr->flag           = rset->getInt(rset, idx++);
					box_pr->password       = rset->getString(rset, idx++);
					box_pr->creationdate   = (time_t)rset->getBigInt(rset, idx++);
					box_pr->expirationdate = (time_t)rset->getBigInt(rset, idx++);
					box_pr->creator_usr_id = rset->getString(rset, idx++);
					box_pr->creator        = rset->getString(rset, idx++);
					box_pr->greeting       = rset->getString(rset, idx++);
					box_pr->message        = rset->getString(rset, idx++);
					box_pr->viewcount      = rset->getBigInt(rset, idx++);
					if (divy_support_2FA(r)) {
						box_pr->tomailaddr = rset->getString(rset, idx++);
					}

					rt->optional.box_pr = box_pr;
				}

			}
			else {
				rt->type      = DIVY_RSTATE_TYPE_VIEW;
				rt->effective = 0;	/* 状態/属性は無効 */
			}
			rt->inherit = 0;	/* 自分自身の属性値(グループ直下は特別) */
			rt->next    = NULL;
		}
	}
	/*
	 * uri がグループ直下 or 直下よりも下の場合
	 */
	else {
		divy_rdbo_resourcestate *rt0 = NULL, *rtx = NULL;
		int len = RESOURCESTATE_TYPES_LEN, type, i;

		/*
		 * uri自身の状態/属性プロパティを取得
		 */
		if (_get_resourcestate_property_with_rsid(r, uri, &rt0, ts_ctx)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get own resourcestate.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/* uri のリソースが存在しなかった場合 */
		if (rt0 == NULL) {
			/* 状態/属性プロパティは存在し得ません */
			if (iscommit) divy_db_commit_transaction(ts_ctx);
			return 0;	/* もう終わり */
		}

		/* 少なくともuri のリソースは存在していた場合 */

		/* 自身が状態/属性プロパティを持っていない */
		first = NULL;
		if (rt0->effective == -1) {
			/* 親がグループフォルダの場合 */
			if (u_spec->p_infotype == DIVY_INFOTYPE_group_e) {
				for (rt = rt0; rt; rt = rt->next) {
					rt->effective = 0;	/* 無効なプロパティとする */
				}
			}
			/* 親がグループフォルダの下のフォルダの場合 */
			else {
				/* type 毎に処理 */
				for (i = 0; i < len; i++) {
					type = _resourcestate_types[i];
					for (rt = rt0; rt; rt = rt->next) {
						if (rt->type == type) {
							break;
						}
					}

					/* 自身より上位階層にある全ての親の状態/属性を取得する */
					if (_get_parent_resourcestate_property(r, uri, type,
										&rtx, ts_ctx)) {
						ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
							"Failed to get resource state of parents. "
							"(uri = %s, type = %d)", uri, type);
						ts_ctx->status |= DIVY_TRANS_ABORT;
						if (iscommit) divy_db_rollback_transaction(ts_ctx);
						return 1;
					}
					/* 状態/属性プロパティが上位階層にも存在しない */
					if (rtx == NULL) {
						/* first には繋がない */
					}
					/* 有効 or 無効な状態/属性プロパティがあった */
					else {
						if (first == NULL) {
							first = rt = apr_pcalloc(p,
									sizeof(divy_rdbo_resourcestate));
						}
						else {
							rt->next = apr_pcalloc(p,
									sizeof(divy_rdbo_resourcestate));
							rt = rt->next;
						}

						/* 自身及び親のプロパティを引き継ぐ */
						rt->rsid      = rt0->rsid;
						rt->uri       = rt0->uri;
						rt->depth     = 0;
						rt->inherit   = 1;	/* 親から継承した */
						rt->type      = type;
						rt->hierarchy = rtx->hierarchy;
						rt->effective = rtx->effective;	/* 親から引き継ぐ */
						if (divy_support_tfbox(r)) {
							rt->optional.box_pr = rtx->optional.box_pr;
						}
						rt->next = NULL;
					}
				}
				rt0 = first;
			}
		}

		if (*rstate_pr_hash == NULL) {
			*rstate_pr_hash = apr_hash_make(p);
		}
		if (rt0 != NULL) {
			apr_hash_set(*rstate_pr_hash, rt0->rsid, APR_HASH_KEY_STRING, rt0);
		}

		/*
		 * uri の下位リソースの状態/属性プロパティを取得(depth = 1)
		 */
		if (depth == 1) {
			/* type 毎に処理 */
			for (i = 0; i < len; i++) {
				type = _resourcestate_types[i];
				for (rt = rt0; rt; rt = rt->next) {
					if (rt->type == type) {
						break;
					}
				}

				/* 有効な状態/属性プロパティを持っていた場合 */
				if (IS_EFFECTIVE_STATE(rt)) {

					/* 下位リソースの継承されていたプロパティを取得する */
					if (_get_inherited_resourcestate_property_with_rsid_by_hash(r,
								uri, type, rstate_pr_hash, ts_ctx)) {
						ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
							"Failed to get inherited resourcestate.");
						ts_ctx->status |= DIVY_TRANS_ABORT;
						if (iscommit) divy_db_rollback_transaction(ts_ctx);
						return 1;
					}
				}
			}
		}
	}

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	return 0;
}

/**
 * 指定されたuri の親コレクションに付いているtype の状態/属性を取得する。
 * type が -1 の場合、type を無視して検索します。
 *
 * (note) トランザクション開始されていれば継続します。
 *
 * @param r request_rec *
 * @param uri const char *
 * @param type int 状態/属性の種類 or -1
 * @param rstate_pr divy_rdbo_resourcestate ** 状態/属性プロパティ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _get_parent_resourcestate_property(request_rec *r,
					const char *uri,
					int type,
					divy_rdbo_resourcestate **rstate_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;
	divy_cset_t *uri_set    = NULL;
	divy_db_bind_ctx *bindctx = NULL;
	divy_db_bind_ctx_idx *idx;
	const char *in_clause_str = NULL;
	divy_linkedlist_t *in_clause;
	divy_rdbo_resourcestate *rt = NULL, *first = NULL;
	int rs_depth, i;
	const char *root_uri;
	divy_sbuf *sql_buf      = NULL;
	divy_rdbo_box	*box_pr = NULL;

	*rstate_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* 現在のURI階層数を数える */
	rs_depth = divy_count_dirs(uri);

	/* uri を親URIの集合に分解する */
	uri_set = divy_get_parenturi_set(p, uri);
	if (uri_set == NULL) {
		/* 分解できないuri は対象外である */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Found invalid uri. (%s)", uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/* 少なくともグループフォルダよりも上位階層のURIは不要である */
	root_uri = dav_divy_get_root_uri(r);
	divy_cset_remove(uri_set, root_uri);
	divy_cset_remove(uri_set, divy_build_group_uri(p, root_uri, NULL));

	/* バインドコンテキスト / インデックスを生成 */
	bindctx = divy_db_bindcontext_make(p, uri_set, -1);
	idx     = divy_db_bindcontext_first(bindctx);

	/* バインド変数文字列の取得 */
	divy_db_bindcontext_get_bindstr(idx, &in_clause_str);

	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */

	/* SQL文の組み立て */
	divy_sbuf_append(sql_buf,
			"SELECT"
			" rt_uri_txt,"
			" rt_type_i,"
			" rt_depth_i,"
			" rt_effective_i ");

	if (divy_support_tfbox(r)) {
		divy_sbuf_append(sql_buf,
				",box_uri_txt,"
				" box_allowed_origin_c,"
				" box_shorten_c,"
				" box_flag_i,"
				" box_passwd_vc,"
				" box_create_bi,"
				" box_expiration_bi,"
				" box_creator_usr_id_vc,"
				" usr_fullname_vc,"
				" box_greeting_vc,"
				" box_message_vc,"
				" box_viewcount_bi ");
		if (divy_support_2FA(r)) {
			divy_sbuf_append(sql_buf, ",box_to_mailaddr_vc");
		}
	}

	divy_sbuf_append(sql_buf,
			" FROM divy_resourcestate ");

	if (divy_support_tfbox(r)) {
		divy_sbuf_append(sql_buf,
				" LEFT OUTER JOIN "
				" divy_box ON "
				" rt_uri_txt = box_uri_txt "
				" LEFT OUTER JOIN divy_usr "
				" ON box_creator_usr_id_vc = usr_usr_id_vc "
				);
	}

	divy_sbuf_append(sql_buf, apr_psprintf(p,
				"WHERE rt_uri_txt IN (%s)", in_clause_str));

	if (type != -1) {
		divy_sbuf_append(sql_buf,
				" AND rt_type_i = ? ");
	}

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s) "
			"Reason: %s", uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	i = 0;
	divy_db_bindcontext_get_list(idx, &in_clause);
	for (; in_clause; in_clause = in_clause->next) {
		stmt->setString(stmt, ++i, in_clause->val);
	}

	if (type != -1) {
		/* バインド */
		stmt->setInt(stmt, ++i, type);
	}

	/* SQL文の実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_resourcestate."
			"(uri = %s) Reason: %s", uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 結果の取得 */
	while (rset->next(rset) == DB_TRUE) {
		if (first == NULL) {
			first = rt = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
		}
		else {
			rt->next = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
			rt = rt->next;
		}
		rt->uri       = rset->getString(rset, 1);
		rt->type      = rset->getInt(rset, 2);
		rt->depth     = rset->getInt(rset, 3);
		rt->effective = rset->getInt(rset, 4);
		rt->inherit   = 1;	/* 親から継承された */
		if (divy_support_tfbox(r)) {

			box_pr = apr_pcalloc(p, sizeof(divy_rdbo_box));

			box_pr->uri            = rset->getString(rset, 5);
			box_pr->allowed_origin = rset->getString(rset, 6);
			box_pr->shorten        = rset->getString(rset, 7);
			box_pr->flag           = rset->getInt(rset, 8);
			box_pr->password       = rset->getString(rset, 9);
			box_pr->creationdate   = (time_t)rset->getBigInt(rset, 10);
			box_pr->expirationdate = (time_t)rset->getBigInt(rset, 11);
			box_pr->creator_usr_id = rset->getString(rset, 12);
			box_pr->creator        = rset->getString(rset, 13);
			box_pr->greeting       = rset->getString(rset, 14);
			box_pr->message        = rset->getString(rset, 15);
			box_pr->viewcount      = rset->getBigInt(rset, 16);
			if (divy_support_2FA(r)) {
				box_pr->tomailaddr = rset->getString(rset, 17);
			}

			rt->optional.box_pr = box_pr;
		}
		rt->next      = NULL;
	}

	/* 不要な資源を解放 */
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	/* 状態/属性が存在しなかった場合 */
	if (first == NULL) {
		return 0;
	}

	/*
	 * 状態/属性の評価
	 * (note) 評価方法
	 * 	rs_depth を1個ずつ減らしていき(= 親を遡っていく)
	 * 	順番に属性を調べていく
	 */
	for (i = rs_depth; i > 0; i--) {
		for (rt = first; rt; rt = rt->next) {
			/* depth が一致した -> 調べたい親だった場合 */
			if (i == rt->depth) {
				/* この属性が親の属性となります */
				*rstate_pr = rt;
				rt->next = NULL;	/* 後ろは要らない */
				break;
			}
		}
		if (*rstate_pr != NULL) break;
	}

	return 0;
}

/**
 * 指定されたuri,type の状態/属性を取得する。
 *
 * (note) トランザクション開始されていれば継続します。
 *
 * @param r request_rec *
 * @param uri const char * 対象リソースのURI
 * @param type int 状態・属性の種類
 * @param rstate_pr divy_rdbo_resourcestate ** 状態/属性プロパティ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _get_resourcestate_property(request_rec *r,
					const char *uri,
					int type,
					divy_rdbo_resourcestate **rstate_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;
	divy_sbuf *sql_buf      = NULL;
	divy_rdbo_box	*box_pr = NULL;

	*rstate_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の組み立て */
	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT"
				" rt_uri_txt,"
				" rt_depth_i,"
				" rt_effective_i ");

	if (divy_support_tfbox(r)) {
		divy_sbuf_append(sql_buf,
				",box_allowed_origin_c,"
				" box_shorten_c,"
				" box_flag_i,"
				" box_passwd_vc,"
				" box_create_bi,"
				" box_expiration_bi,"
				" box_creator_usr_id_vc,"
				" usr_fullname_vc,"
				" box_greeting_vc,"
				" box_message_vc,"
				" box_viewcount_bi "
				);
		if (divy_support_2FA(r)) {
			divy_sbuf_append(sql_buf, ",box_to_mailaddr_vc");
		}
	}

	divy_sbuf_append(sql_buf,
				" FROM divy_resourcestate ");

	if (divy_support_tfbox(r)) {
		divy_sbuf_append(sql_buf,
				" LEFT OUTER JOIN "
				" divy_box ON "
				" rt_uri_txt = box_uri_txt "
				" LEFT OUTER JOIN "
				" divy_usr ON "
				" usr_usr_id_vc = box_creator_usr_id_vc "
				);
	}

	divy_sbuf_append(sql_buf,
				"WHERE rt_uri_txt = ? "
				"AND rt_type_i = ?");

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p); 
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s, type = %d) "
			"Reason: %s", uri, type, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, uri);
	stmt->setInt(stmt, 2, type);

	/* SQL文の実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_resourcestate. (uri = %s, type = %d) "
			"Reason: %s", uri, type, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 結果の取得 */
	if (rset->next(rset) == DB_TRUE) {
		*rstate_pr = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
		(*rstate_pr)->uri       = rset->getString(rset, 1);
		(*rstate_pr)->type      = type;
		(*rstate_pr)->depth     = rset->getInt(rset, 2);
		(*rstate_pr)->effective = rset->getInt(rset, 3);
		(*rstate_pr)->inherit   = 0;	/* 自分自身の属性値である */

		if (divy_support_tfbox(r)) {

			box_pr = apr_pcalloc(p, sizeof(divy_rdbo_box));

			box_pr->uri            = rset->getString(rset, 1);
			box_pr->allowed_origin = rset->getString(rset, 4);
			box_pr->shorten        = rset->getString(rset, 5);
			box_pr->flag           = rset->getInt(rset, 6);
			box_pr->password       = rset->getString(rset, 7);
			box_pr->creationdate   = (time_t)rset->getBigInt(rset, 8);
			box_pr->expirationdate = (time_t)rset->getBigInt(rset, 9);
			box_pr->creator_usr_id = rset->getString(rset, 10);
			box_pr->creator        = rset->getString(rset, 11);
			box_pr->greeting   = rset->getString(rset, 12);
			box_pr->message    = rset->getString(rset, 13);
			box_pr->viewcount  = rset->getBigInt(rset, 14);
			if (divy_support_2FA(r)) {
				box_pr->tomailaddr = rset->getString(rset, 15);
			}

			(*rstate_pr)->optional.box_pr = box_pr;

		}

	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (rset != NULL) rset->close(rset); rset = NULL;

	return 0;
}

/**
 * 指定されたuri の状態/属性プロパティを取得する。(内部処理専用)
 * rsid も一緒に取得します。
 * (note)
 *	* uri の示すリソースが存在しなければ*rstate_pr はNULL
 *	* リソースは存在したが状態/属性プロパティがなければ effective = -1
 *	* 状態/属性プロパティがあればその値を取得
 * 
 *
 * (note) トランザクション開始されていれば継続します。
 *
 * @param r request_rec *
 * @param uri const char * 対象リソースのURI
 * @param rstate_pr divy_rdbo_resourcestate ** 状態/属性プロパティ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _get_resourcestate_property_with_rsid(request_rec *r,
					const char *uri,
					divy_rdbo_resourcestate **rstate_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;
	divy_rdbo_resourcestate *rstate = NULL;
	char *geturi = NULL, *rsid = NULL;
	int notfound_property = 0;
	divy_sbuf *sql_buf      = NULL;
	int idx                 = 0;
	divy_rdbo_box	*box_pr = NULL;

	*rstate_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の組み立て */
	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT"
				" rs.rs_rs_id_c,"
				" rt.rt_uri_txt,"
				" rt.rt_type_i,"
				" rt.rt_depth_i,"
				" rt.rt_hierarchy_i,"
				" rt.rt_effective_i ");

	if (divy_support_tfbox(r)) {
		divy_sbuf_append(sql_buf,
				",box.box_uri_txt,"
				" box.box_allowed_origin_c,"
				" box.box_shorten_c,"
				" box.box_flag_i,"
				" box.box_passwd_vc,"
				" box.box_create_bi,"
				" box.box_expiration_bi,"
				" box.box_creator_usr_id_vc,"
				" usr.usr_fullname_vc,"
				" box.box_greeting_vc,"
				" box.box_message_vc,"
				" box.box_viewcount_bi ");
		if (divy_support_2FA(r)) {
			divy_sbuf_append(sql_buf, ",box_to_mailaddr_vc");
		}
	}

	divy_sbuf_append(sql_buf,
				" FROM dav_resource rs"
				" LEFT JOIN divy_resourcestate rt"
				" ON rs.rs_uri_txt = rt.rt_uri_txt ");
	
	if (divy_support_tfbox(r)) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_box box"
				" ON rs.rs_uri_txt = box.box_uri_txt "
				" LEFT OUTER JOIN divy_usr usr"
				" ON usr.usr_usr_id_vc = box.box_creator_usr_id_vc "
				);

	}

	divy_sbuf_append(sql_buf,
				"WHERE rs.rs_uri_txt = ?");

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf) ,p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s) "
			"Reason: %s", uri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, uri);

	/* SQL文の実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_resourcestate. (uri = %s) "
			"Reason: %s", uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 結果の取得 */
	while (rset->next(rset) == DB_TRUE) {
		idx = 1;
		rsid   = rset->getString(rset, idx++);
		geturi = rset->getString(rset, idx++);

		/* リソースは存在したが状態/属性プロパティは存在しなかった場合 */
		if (IS_FILLED(rsid) && IS_EMPTY(geturi)) {
			notfound_property = 1;
			break;
		}

		if (*rstate_pr == NULL) {
			*rstate_pr = rstate = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
		}
		else {
			rstate->next = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
			rstate = rstate->next;
		}
		rstate->rsid      = rsid;
		rstate->uri       = geturi;
		rstate->type      = rset->getInt(rset, idx++);
		rstate->depth     = rset->getInt(rset, idx++);
		rstate->hierarchy = rset->getInt(rset, idx++);
		rstate->effective = rset->getInt(rset, idx++);
		rstate->inherit = 0;	/* 自分自身の属性値である */

		if (divy_support_tfbox(r)) {
			box_pr = apr_pcalloc(p, sizeof(divy_rdbo_box));
			box_pr->uri            = rset->getString(rset, idx++);
			box_pr->allowed_origin = rset->getString(rset, idx++);
			box_pr->shorten        = rset->getString(rset, idx++);
			box_pr->flag           = rset->getInt(rset, idx++);
			box_pr->password       = rset->getString(rset, idx++);
			box_pr->creationdate   = (time_t)rset->getBigInt(rset, idx++);
			box_pr->expirationdate = (time_t)rset->getBigInt(rset, idx++);
			box_pr->creator_usr_id = rset->getString(rset, idx++);
			box_pr->creator        = rset->getString(rset, idx++);
			box_pr->greeting       = rset->getString(rset, idx++);
			box_pr->message        = rset->getString(rset, idx++);
			box_pr->viewcount      = rset->getBigInt(rset, idx++);
			if (divy_support_2FA(r)) {
				box_pr->tomailaddr = rset->getString(rset, idx++);
			}

			rstate->optional.box_pr = box_pr;
		}

		rstate->next    = NULL;
	}

	/* 状態/属性プロパティは存在していたのか? */
	if (notfound_property) {
		/* リソースは存在したのだがプロパティは無かった場合には全type の
		 * プロパティを作って返す */
		int len = RESOURCESTATE_TYPES_LEN, i;

		for (i = 0; i < len; i++) {
			if (*rstate_pr == NULL) {
				*rstate_pr = rstate = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
			}
			else {
				rstate->next = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
				rstate = rstate->next;
			}

			rstate->rsid      = rsid;
			rstate->uri       = (char *) uri;
			rstate->type      = _resourcestate_types[i];
			rstate->depth     = 0;
			rstate->hierarchy = 1;
			rstate->effective = -1;	/* 目印 */
			rstate->inherit   = 0;
			rstate->next      = NULL;
		}
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (rset != NULL) rset->close(rset); rset = NULL;

	return 0;
}

/**
 * 有効な状態/属性プロパティをuri が保持するとき、uriの直下にあるリソースの
 * 継承された有効な状態/属性プロパティ及び
 * 直接無効化された状態/属性プロパティを検索して返却する。(内部処理専用)
 * rsid も一緒に取得します。
 * (note)
 * 	この関数は、uri自身またはその親に既に"有効な"状態・属性が付いていた
 * 	ものと仮定して処理を進めます。これは呼び出し側で保証して下さい。
 * 	さもないと正しい結果を得ることはできません。
 *
 * (note)
 * 	* トランザクション開始されていれば継続します。
 * 	* rstate_pr のinherit, hierarchy メンバは正しく設定されません
 *
 * @param r request_rec *
 * @param uri const char * 対象リソースのURI
 * @param type int 状態・属性プロパティの種類
 * @param rstate_pr divy_rdbo_resourcestate ** 状態/属性プロパティ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _get_inherited_resourcestate_property_with_rsid(request_rec *r,
					const char *uri,
					int type,
					divy_rdbo_resourcestate **rstate_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;
	divy_rdbo_resourcestate *rstate = NULL;
	int rs_depth, effective, type0, inherit;
	char *escaped_uri, *geturi, *rs_uri, *rs_rsid;
	divy_sbuf *sql_buf      = NULL;
	int idx                 = 0;
	divy_rdbo_box *box_pr   = NULL;

	*rstate_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の組み立て */
	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */

	divy_sbuf_append(sql_buf,
				"SELECT"
				" rs.rs_rs_id_c,"
				" rs.rs_uri_txt,"
				" rt.rt_uri_txt,"
				" rt.rt_type_i,"
				" rt.rt_effective_i ");

	if (divy_support_tfbox(r)) {
		divy_sbuf_append(sql_buf,
				",box_allowed_origin_c,"
				" box_shorten_c,"
				" box_flag_i,"
				" box_passwd_vc,"
				" box_create_bi,"
				" box_expiration_bi,"
				" box_creator_usr_id_vc,"
				" usr.usr_fullname_vc,"
				" box_greeting_vc,"
				" box_message_vc,"
				" box_viewcount_bi ");
		if (divy_support_2FA(r)) {
			divy_sbuf_append(sql_buf, ",box_to_mailaddr_vc");
		}
	}

	divy_sbuf_append(sql_buf,
				" FROM dav_resource rs"
				" LEFT JOIN divy_resourcestate rt"
				" ON rs.rs_uri_txt = rt.rt_uri_txt ");

	if (divy_support_tfbox(r)) {
		divy_sbuf_append(sql_buf,
				" LEFT OUTER JOIN "
				" divy_box box ON "
				" rs.rs_uri_txt = box.box_uri_txt "
				" LEFT OUTER JOIN "
				" divy_usr usr ON "
				" usr.usr_usr_id_vc = box.box_creator_usr_id_vc "
				);
	}

	divy_sbuf_append(sql_buf,
				"WHERE "
				"rs.rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE" "
				"AND rs.rs_depth_i = ? ");

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s, type = %d) "
			"Reason: %s", uri, type, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	rs_depth    = divy_count_dirs(uri);
	escaped_uri = apr_pstrcat(p, stmt->escWildCard(stmt, uri), "/%", NULL);

	/* バインド */
	stmt->setString(stmt, 1, escaped_uri);
	stmt->setInt(stmt,    2, rs_depth + 1);

	/* SQL文の実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_resourcestate. (uri = %s) "
			"Reason: %s", uri, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 結果の取得 */
	while (rset->next(rset) == DB_TRUE) {
		idx = 1;
		rs_rsid = rset->getString(rset, idx++);
		rs_uri  = rset->getString(rset, idx++);
		geturi  = rset->getString(rset, idx++);

		/* 無効な状態/属性プロパティを持っていた場合 */
		if (IS_FILLED(geturi)) {
			type0     = rset->getInt(rset, idx++);
			if (type0 != type) {
				continue;	/* 興味のあるタイプではない */
			}
			effective = rset->getInt(rset, idx++);
			inherit   = 0;	/* 自分自身が持っている */
		}
		/* 親から継承されたことで有効な状態/属性プロパティを持つことになります */
		else {
			effective = 1;	/* 有効である */
			inherit   = 1;	/* 継承されました */
		}

		if (*rstate_pr == NULL) {
			*rstate_pr = rstate = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
		}
		else {
			rstate->next = apr_pcalloc(p, sizeof(divy_rdbo_resourcestate));
			rstate = rstate->next;
		}
		rstate->rsid      = rs_rsid;
		rstate->uri       = rs_uri;
		rstate->type      = type;
		rstate->effective = effective;
		rstate->depth     = 0;
		rstate->hierarchy = 1;
		rstate->inherit   = inherit;

		if (divy_support_tfbox(r)) {
			box_pr = apr_pcalloc(p, sizeof(divy_rdbo_box));

			box_pr->allowed_origin = rset->getString(rset, idx++);
			box_pr->shorten        = rset->getString(rset, idx++);
			box_pr->flag           = rset->getInt(rset, idx++);
			box_pr->password       = rset->getString(rset, idx++);
			box_pr->creationdate   = (time_t)rset->getBigInt(rset, idx++);
			box_pr->expirationdate = (time_t)rset->getBigInt(rset, idx++);
			box_pr->creator_usr_id = rset->getString(rset, idx++);
			box_pr->creator        = rset->getString(rset, idx++);
			box_pr->greeting       = rset->getString(rset, idx++);
			box_pr->message        = rset->getString(rset, idx++);
			box_pr->viewcount      = rset->getBigInt(rset, idx++);
			if (divy_support_2FA(r)) {
				box_pr->tomailaddr = rset->getString(rset, idx++);
			}

			rstate->optional.box_pr = box_pr;
		}

		rstate->next      = NULL;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (rset != NULL) rset->close(rset); rset = NULL;

	return 0;
}

/**
 * 関数 _get_inherited_resourcestate_property_with_rsid で取得した結果を
 * rsid をキーとして divy_rdbo_resourcestate * を保持するハッシュ*rstate_pr_hash
 * として返却する。 (殆どサブルーチン)
 *
 * *rstate_pr_hash がNULLでなければそのまま利用します。なければ作ります。
 *
 */
static int _get_inherited_resourcestate_property_with_rsid_by_hash(request_rec *r,
					const char *uri,
					int type,
					apr_hash_t **rstate_pr_hash,
					divy_db_transaction_ctx *ts_ctx)
{
	int iscommit    = 0;
	apr_pool_t *p   = r->pool;
	divy_rdbo_resourcestate *rstate_pr = NULL, *next, *first;

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;

	/*
	 * 継承されていたプロパティを取得する
	 */
	if (_get_inherited_resourcestate_property_with_rsid(r, uri, type, &rstate_pr, ts_ctx)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/* ハッシュに結果を詰める */
	while (rstate_pr != NULL) {
		next = rstate_pr->next;
		rstate_pr->next = NULL;	/* rstate_pr からのリンクを切る */

		if (*rstate_pr_hash == NULL) {
			*rstate_pr_hash = apr_hash_make(p);
		}

		/* このrsid は別のtypeで既に取得済みか? */
		first = apr_hash_get(*rstate_pr_hash, rstate_pr->rsid, APR_HASH_KEY_STRING);
		if (first == NULL) {
			apr_hash_set(*rstate_pr_hash, rstate_pr->rsid, APR_HASH_KEY_STRING, rstate_pr);
		}
		else {
			/* 末尾になるまでスキップ */
			for (; first->next; first = first->next);
			first->next = rstate_pr;
		}
		rstate_pr = next;	/* 次を指す */
	}

	return 0;
}

/**
 * uri 以下にある状態/属性プロパティを削除する。
 * depth = 0 ならばuri自身のみを、depth = infinity ならばuri以下全てを削除します。
 * typeに応じて削除する対象を変更します。NULLですべて削除します
 * (note)
 * 	uri のリソースが単に状態/属性を"継承"しているだけであれば
 * 	テーブル divy_resourcestate には何ら変更はありません。
 * 	uri のリソースが直接に状態/属性を持つ場合のみに意味を持ちます。
 *
 * [ 削除対象テーブル ] divy_resourcestate
 *
 * @param r request_rec *
 * @param depth int 深さ (0, DAV_INFINITY)
 * @param uri const char * 削除する状態/属性プロパティを持つURI
 * @param type int リソースタイプ see tf_rdbo_h divy_rdbo_resourcestate.type
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 */
static int _remove_resourcestate_property(request_rec *r,
					int depth,
					const char *uri,
					int type,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;
	divy_sbuf *sql_buf      = NULL;
	int idx                 = 0;

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"uri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */

	/* SQL文の準備 */
	if (depth == 0) {
		divy_sbuf_append(sql_buf,
				"DELETE FROM divy_resourcestate "
				"WHERE rt_uri_txt = ?");
	}
	else {
		divy_sbuf_append(sql_buf,
				"DELETE FROM divy_resourcestate "
				"WHERE rt_uri_txt = ? "
				"OR rt_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE);
	}

	if (type != DIVY_RSTATE_TYPE_UNKNOWN) {
		divy_sbuf_append(sql_buf, " AND rt_type_i = ? ");
	}

	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s, depth = %d) "
			"Reason: %s", uri, depth, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	idx = 1;
	/* バインド */
	stmt->setString(stmt, idx++, uri);
	if (depth != 0) {
		stmt->setString(stmt, idx++, apr_pstrcat(p, stmt->escWildCard(stmt, uri), "/%", NULL));
	}
	if (type != DIVY_RSTATE_TYPE_UNKNOWN) {
		stmt->setInt(stmt, idx++, type);
	}

	/* SQL文の実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_resourcestate.(uri = %s, depth = %d) "
			"Reason: %s", uri, depth, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* BOXのサポートに関係なく削除は試す */
	if (type != DIVY_RSTATE_TYPE_VIEW) {
		if (_remove_box_property(r, uri, ts_ctx)) {
			/* エラーは _remove_box_propertyで出力済 */

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
	}

	/* 不要な資源を解放 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * BOX情報を削除する
 * 
 * 対象テーブル: divy_box
 *
 * @param r request_rec *
 * @param uri const char* 
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
static int _remove_box_property(request_rec *r,
					const char *uri,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	char            *sqlstr = "DELETE "
							  " FROM divy_box "
							  " WHERE box_uri_txt LIKE ?";

	TRACE(p);

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, sqlstr, p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s) "
			"Reason: %s", uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/* バインド */
	stmt->setString(stmt, 1, apr_pstrcat(p,
				stmt->escWildCard(stmt, uri), "%", NULL));

	/* sql実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_box. (uri = %s) "
			"Reason: %s", uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/* 解放 */
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * rstate_pr の状態・属性プロパティを登録する。
 * 単にINSERTするだけです。外部仕様から要請される整合性は気にしません。
 *
 * 対象テーブル: divy_resourcestate
 * (note)
 * 	継続中のトランザクションは引き継ぎます。
 *
 * @param r request_rec *
 * @param rstate_pr divy_rdbo_resourcestate * 追加する状態・属性プロパティ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
static int _insert_resourcestate_property(request_rec *r,
					divy_rdbo_resourcestate *rstate_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	int iscommit            = 0;
	apr_pool_t *p           = r->pool;
	divy_rdbo_box	*rbox= NULL;
	divy_sbuf *sql_buf      = NULL;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);

	if (rstate_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rstate_pr is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
			"INSERT INTO divy_resourcestate "
			"(rt_uri_txt,"
			" rt_type_i,"
			" rt_depth_i,"
			" rt_hierarchy_i,"
			" rt_effective_i) "
			"VALUES (?, ?, ?, ?, ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, rstate_pr->uri);
	stmt->setInt(stmt,    2, rstate_pr->type);
	stmt->setInt(stmt,    3, rstate_pr->depth);
	stmt->setInt(stmt,    4, rstate_pr->hierarchy);
	stmt->setInt(stmt,    5, rstate_pr->effective);

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_resourcestate. "
			"(uri = %s, type = %d) Reason: %s",
			rstate_pr->uri, rstate_pr->type, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* box機能をサポートしているなら設定する */
	if (rstate_pr->type == DIVY_RSTATE_TYPE_BOX && divy_support_tfbox(r)) {

		divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */

		rbox = rstate_pr->optional.box_pr;
		divy_sbuf_append(sql_buf, "INSERT INTO divy_box "
								  "(box_uri_txt,"
								  " box_allowed_origin_c,"
								  " box_shorten_c,"
								  " box_flag_i,"
								  " box_passwd_vc,"
								  " box_create_bi,"
								  " box_expiration_bi,"
								  " box_creator_usr_id_vc,"
								  " box_greeting_vc,"
								  " box_message_vc,"
								  " box_viewcount_bi");
		if (divy_support_2FA(r)) {
			divy_sbuf_append(sql_buf, ",box_to_mailaddr_vc");
		}

		divy_sbuf_append(sql_buf, ") VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?");

		if (divy_support_2FA(r)) {
			divy_sbuf_append(sql_buf, ",?");
		}

		divy_sbuf_append(sql_buf, ")");

		stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		/* バインド */
		stmt->setString(stmt, 1, rstate_pr->uri);
		stmt->setString(stmt, 2, rbox->allowed_origin);
		stmt->setString(stmt, 3, rbox->shorten);
		stmt->setInt(stmt,    4, rbox->flag);
		stmt->setString(stmt, 5, rbox->password);
		stmt->setBigInt(stmt, 6, rbox->creationdate);
		stmt->setBigInt(stmt, 7, rbox->expirationdate);
		stmt->setString(stmt, 8, rbox->creator_usr_id);
		stmt->setString(stmt, 9, rbox->greeting);
		stmt->setString(stmt,10, rbox->message);
		stmt->setBigInt(stmt,11, rbox->viewcount);
		if (divy_support_2FA(r)) {
			stmt->setString(stmt, 12, rbox->tomailaddr);
		}
		
		/* SQL実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to insert divy_box. "
					"(uri = %s, type = %d) Reason: %s",
					rstate_pr->uri, rstate_pr->type, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* メールを送信 */
		if (rbox->flag & BOX_FLAG_OPEN) {

			/* 公開かつ新規公開の場合はBOX公開した本人にメールを送る */
			if (rbox->flag & BOX_FLAG_NEW) {
				(void)divy_ml_box(r, DIVY_ML_BOX_TYPE_OPEN_REPORT, rbox);
			}

			/* サーバポリシーでメール送信全面禁止の場合は送らない */
			if (conf->boxwithoutmail != BOX_WITHOUTMAIL_ALL) {
				if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN) {
					/* 相手先メールがあれば対象者には毎回送る */
					(void)divy_ml_box_invitation(r, rbox);
				}
				else 
				if (conf->boxwithoutmail != BOX_WITHOUTMAIL_USER) {
					/* 相手先メールがあれば対象者には毎回送る */
					(void)divy_ml_box_invitation(r, rbox);
				}
			}
		}
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたgrp_pr が示すグループフォルダ以下からbordertime_sec よりも
 * 古い更新日付を持つフォルダを削除する. (自動削除用)
 * (note)
 *   この関数は、対象グループフォルダ以下のリソース及びフォルダが
 *   DBロック済みであることを前提に記述されております。利用には注意を。
 */
static int _remove_old_groupfolders(request_rec *r,
					divy_rdbo_grp *grp_pr,
					apr_time_t bordertime_sec,
					divy_autodel_dresource **del_res,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn         *dbconn     = NULL;
	DbPreparedStmt *stmt       = NULL, *stmt2 = NULL;
	DbResultSet    *rset       = NULL;
	apr_pool_t     *p          = r->pool;
	dav_divy_dir_conf *dconf   = dav_divy_get_dir_config(r);
	int iscommit               = 0;
	int support_trash          = divy_support_trashfolder(r);
	int support_extstatus      = divy_support_extenduserstatus(r);
	int uri_count, i;
	char *str, *slapos, *cururi;
	divy_cset_t *uri_set        = NULL;
	divy_rdbo_delfolder *first  = NULL, *delfolder = NULL;
	divy_autodel_dresource *res = NULL;
	char *esc_grpcol_uri        = NULL, *esc_folder_uri;
	apr_time_t sec;
	divy_rdbo_usr ldapusr       = { 0 };

	*del_res = NULL;

	if (grp_pr == NULL || bordertime_sec < APR_INT64_C(0)) {
		return 0;	/* 何もやることはない */
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * (1) grp_pr->grpcol_uri 以下のbordertime_sec よりも更新日付が
	 *     古いフォルダ一覧を階層の深い順に取得.
	 *     階層の深い順に取得するのは次の削除ロジックで利用するため.
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"SELECT "
				"rs.rs_rs_id_c,"
				"rs.rs_uri_txt,"
				"rs.rs_get_cont_type_vc,"
				"rs.rs_dispname_vc,"
				"rs.rs_lastmodifier_usr_id_vc,"
				"rs.rs_get_lastmodified_bi, "
				"u.usr_fullname_vc AS rs_lastmodifier_vc "
				"FROM dav_resource rs "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				"LEFT JOIN divy_usr u "
				"ON rs.rs_lastmodifier_usr_id_vc = u.usr_usr_id_vc "
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				", divy_usr u "
#endif
#if defined(DIVY_DBMS_ORACLE) /* oracle */
				"WHERE rs.rs_lastmodifier_usr_id_vc = u.usr_usr_id_vc(+) "
				"AND rs.rs_uri_txt LIKE ? "
#else
				"WHERE rs.rs_uri_txt LIKE ? "
#endif
				DIVY_DBFUNC_ESCAPE_CLAUSE" "
				" AND rs.rs_resourcetype_i = 1 "
				" AND rs.rs_get_lastmodified_bi < ? "
				"ORDER BY rs.rs_depth_i DESC ", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* uri に含まれるワイルドカード文字をエスケープする */
	esc_grpcol_uri = apr_pstrcat(p, stmt->escWildCard(stmt, grp_pr->grpcol_uri), "/%", NULL);

	stmt->setString(stmt, 1, esc_grpcol_uri);
	stmt->setBigInt(stmt, 2, bordertime_sec);

	/* SQLの実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to SELECT dav_resource. Reason: %s", rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {

		str = rset->getString(rset, 3);
		if (IS_FILLED(str) && strcmp(str, DIVY_TRASHFOLDER_CONTENTTYPE) == 0) {
			continue;	/* ごみ箱は除外 */
		}

		if (first == NULL) {
			first = delfolder = apr_pcalloc(p, sizeof(divy_rdbo_delfolder));
		}
		else {
			delfolder->next = apr_pcalloc(p, sizeof(divy_rdbo_delfolder));
			delfolder = delfolder->next;
		}
		delfolder->rsid                = rset->getString(rset, 1);
		delfolder->uri                 = rset->getString(rset, 2);
		delfolder->getcontenttype      = str;
		delfolder->displayname         = rset->getString(rset, 4);
		delfolder->lastmodifier_userid = rset->getString(rset, 5);
		sec                            = rset->getBigInt(rset, 6);	/* DB には秒で入っている */
		delfolder->getlastmodified     = apr_time_from_sec(sec);	/* sec -> micro sec */
		delfolder->lastmodifier        = rset->getString(rset, 7);
		delfolder->next                = NULL;

		/* LDAP から探してみる */
		if (IS_FILLED(dconf->ldapmapfullnameattr) &&
				divy_util_ldap_get_user_property(r, p, delfolder->lastmodifier_userid, &ldapusr) == TF_LDAP_TRUE) {
			delfolder->lastmodifier = ldapusr.fullname;
		}
	}
	if (rset != NULL) rset->close(rset);	rset = NULL;
	if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

	/* 1件も削除対象がなければ正常終了する */
	if (first == NULL) {
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		return 0;	/* 正常とする */
	}

	/*
	 * (2) delfolder->uri 以下にリソースやフォルダがあるかどうかを確認する
	 *     あれば、消してはならないという仕様になっている.
	 * (note)
	 *   delfolder の並び順は、より階層の深いものほど先頭に来るように上記SQLで
	 *   調整されています.
	 */
	/* 事前にprepareする(再利用前提) */
	stmt = dbconn->prepareStatement(dbconn, 
				"SELECT "
				"1 "
				"FROM dav_resource "
				"WHERE rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE" "
				"LIMIT 1", p);	/* 1件取得できれば十分である */

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	for (delfolder = first; delfolder != NULL; delfolder = delfolder->next) {

		/* 無駄URIリストに登録されている？ */
		if (uri_set != NULL && divy_cset_contains(uri_set, delfolder->uri)) {
			continue;	/* DB 引く必要もない */
		}
		esc_folder_uri = apr_pstrcat(p, stmt->escWildCard(stmt, delfolder->uri), "/%", NULL);
		stmt->setString(stmt, 1, esc_folder_uri);

		/* SQL の実行 */
		rset = stmt->executeQuery(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbResultSet for select dav_resource."
					"Reason: %s", rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		/* 下にファイル or フォルダなりが存在した場合 */
		if (rset->next(rset) == DB_TRUE) {

			/* より上位のフォルダは同様の理由で必ず失敗するだろうから、
			 * 無駄なSQLを投げないようにSET を作っておく */
			if (uri_set == NULL) {
				uri_set = divy_cset_make(p);
			}

			slapos = cururi = apr_pstrdup(p, delfolder->uri);
			uri_count = divy_count_dirs(cururi);

			/* スラッシュ数-1(最初の / はスキップ) だけループ */
			for (i = 0; i < uri_count - 1; i++) {
				/* 末尾から探していく */
				slapos = divy_strrchr(cururi, '/');
				if (slapos) *slapos = '\0';

				/* URI集合に追加 */
				divy_cset_set(uri_set, apr_pstrdup(p, cururi));
			}
			/* (note) stmt は再利用するのでclose してはなりません */
			if (rset != NULL) rset->close(rset); rset = NULL;
			continue;	/* 削除できないので諦める */
		}
		if (rset != NULL) rset->close(rset); rset = NULL;

		/* ここまで来てようやく削除される準備が整ったので、*del_res に追加 */
		if (*del_res == NULL) {
			*del_res = res = apr_pcalloc(p, sizeof(divy_autodel_dresource));
		}
		else {
			res->next = apr_pcalloc(p, sizeof(divy_autodel_dresource));
			res = res->next;
		}
		res->displayname         = delfolder->displayname;
		res->getcontenttype      = delfolder->getcontenttype; 
		res->lastmodifier_userid = delfolder->lastmodifier_userid;
		res->getlastmodified     = delfolder->getlastmodified;
		res->lastmodifier        = delfolder->lastmodifier;
		res->deletion            = APR_INT64_C(0);	/* 後で */
		res->next                = NULL;

		/*
		 * (3) Dead プロパティを削除する
		 *     (dav_resource を消すよりも前にやること)
		 */
		stmt2 = dbconn->prepareStatement(dbconn, 
					"DELETE FROM dav_dead_property "
					"WHERE dp_rs_id_c = ?", p);
		if (stmt2->getCode(stmt2) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s", stmt2->getMsg(stmt2));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt2 != NULL) stmt2->close(stmt2);
			if (stmt  != NULL) stmt->close(stmt);
			return 1;
		}
		stmt2->setString(stmt2, 1, delfolder->rsid);

		(void) stmt2->executeUpdate(stmt2, p);
		if (stmt2->getCode(stmt2) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to execute DELETE dead property. Reason: %s", stmt2->getMsg(stmt2));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt2 != NULL) stmt2->close(stmt2);
			if (stmt  != NULL) stmt->close(stmt);
			return 1;
		}
		if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;

		/*
		 * (4) ごみ箱プロパティの削除
		 */
		if (support_trash) {
			stmt2 = dbconn->prepareStatement(dbconn, 
						"DELETE FROM divy_trash_info "
						"WHERE tr_rs_id_c = ?", p);
			if (stmt2->getCode(stmt2) != DB_SUCCESS) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. Reason: %s", stmt2->getMsg(stmt2));

				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (stmt2 != NULL) stmt2->close(stmt2);
				if (stmt  != NULL) stmt->close(stmt);
				return 1;
			}
			stmt2->setString(stmt2, 1, delfolder->rsid);

			(void) stmt2->executeUpdate(stmt2, p);
			if (stmt2->getCode(stmt2) != DB_SUCCESS) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to DELETE trash property. Reason: %s", stmt2->getMsg(stmt2));

				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (stmt2 != NULL) stmt2->close(stmt2);
				if (stmt  != NULL) stmt->close(stmt);
				return 1;
			}
			if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
		}

		/*
		 * (5) 状態/属性プロパティの削除
		 */
		if (support_extstatus) {
			stmt2 = dbconn->prepareStatement(dbconn, 
						"DELETE FROM divy_resourcestate "
						"WHERE rt_uri_txt = ?", p);
			if (stmt2->getCode(stmt2) != DB_SUCCESS) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. Reason: %s", stmt2->getMsg(stmt2));

				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (stmt2 != NULL) stmt2->close(stmt2);
				if (stmt  != NULL) stmt->close(stmt);
				return 1;
			}
			stmt2->setString(stmt2, 1, delfolder->uri);

			(void) stmt2->executeUpdate(stmt2, p);
			if (stmt2->getCode(stmt2) != DB_SUCCESS) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to DELETE resourcestate property. Reason: %s", stmt2->getMsg(stmt2));

				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (stmt2 != NULL) stmt2->close(stmt2);
				if (stmt  != NULL) stmt->close(stmt);
				return 1;
			}
			if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
		}

		/*
		 * (6) ロック情報を削除する
		 * (note) NULL ロックがあるかもしれないので、LIKEで消す
		 */
		stmt2 = dbconn->prepareStatement(dbconn, 
					"DELETE FROM dav_lock "
					"WHERE lk_uri_txt = ? "
					" OR lk_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE, p);

		if (stmt2->getCode(stmt2) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s", stmt2->getMsg(stmt2));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt2 != NULL) stmt2->close(stmt2);

			return 1;
		}
		stmt2->setString(stmt2, 1, delfolder->uri);
		stmt2->setString(stmt2, 2, esc_folder_uri);

		(void) stmt2->executeUpdate(stmt2, p);
		if (stmt2->getCode(stmt2) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to DELETE lock information. Reason: %s", stmt2->getMsg(stmt2));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt2 != NULL) stmt2->close(stmt2);

			return 1;
		}
		if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;

		/*
		 * (7) リソース情報を削除する
		 */
		stmt2 = dbconn->prepareStatement(dbconn, 
					"DELETE FROM dav_resource "
					"WHERE rs_rs_id_c = ?", p);

		if (stmt2->getCode(stmt2) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. Reason: %s", stmt2->getMsg(stmt2));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt2 != NULL) stmt2->close(stmt2);

			return 1;
		}
		stmt2->setString(stmt2, 1, delfolder->rsid);

		(void) stmt2->executeUpdate(stmt2, p);
		if (stmt2->getCode(stmt2) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to DELETE resource information. Reason: %s", stmt2->getMsg(stmt2));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt2 != NULL) stmt2->close(stmt2);

			return 1;
		}
		if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
		/* (note) stmt は再利用するのでclose してはなりません */
	}
	if (stmt != NULL) stmt->close(stmt);

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * confirmreading プロパティを取得する.
 *
 * uri == NULL    : userid に合致するプロパティを取得
 * userid == NULL :
 * 		* depth == 0             : uri のプロパティ
 * 		* depth == 1             : 直下
 * 		* depth == DAV_INFINITY  : uri 以下全て
 */
static int _get_confirmreading(request_rec *r, const char *uri,
					const char *userid, int depth, apr_hash_t **cr_pr_hash,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;

	char *rsid              = NULL;
	int cnt                 = 0;
	apr_hash_t *work_hash   = NULL;
	int hierarchy_depth     = divy_count_dirs(uri);
	divy_sbuf *sql_buf      = NULL;
	int idx;
	divy_rdbo_confirmreading *cr_pr;

	*cr_pr_hash = NULL;	/* 初期化 */

	if (IS_EMPTY(uri) && IS_EMPTY(userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"uri and userid is EMPTY");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の組み立て */
	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT"
				" rs.rs_rs_id_c,"
				" cr.cr_uri_txt"
				",cr.cr_usr_id_vc"
				",cr.cr_creation_bi"
				" FROM divy_confirmreading cr"
				" INNER JOIN dav_resource rs"
				" ON cr.cr_uri_txt = rs.rs_uri_txt");

	if (IS_EMPTY(uri)) {
		if (IS_FILLED(userid)) {
			divy_sbuf_append(sql_buf,
						" WHERE cr.cr_usr_id_vc = ?");
		}
		else {	/* このケースは既に除外してあります */
		}
	}
	else {
		if (depth == 0) {
			divy_sbuf_append(sql_buf,
					" WHERE cr.cr_uri_txt = ?");
		}
		else if (depth == 1) {
			divy_sbuf_append(sql_buf,
					" WHERE (cr.cr_uri_txt = ?"
					" OR (rs.rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
					" AND rs.rs_depth_i = ?))");
		}
		else {
			divy_sbuf_append(sql_buf,
					" WHERE (cr.cr_uri_txt = ?"
					" OR rs.rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE")");

		}
		if (IS_FILLED(userid)) {
			divy_sbuf_append(sql_buf,
					" AND cr.cr_usr_id_vc = ?");
		}
	}

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt for divy_confirmreading. "
				"(uri = %s, userid = %s) Reason: %s",
				REPLACE_NULL(uri), REPLACE_NULL(userid), stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* バインド */
	idx = 1;
	if (IS_EMPTY(uri)) {
		if (IS_FILLED(userid)) {
			stmt->setString(stmt, idx++, userid);
		}
	}
	else {
		if (depth == 0) {
			stmt->setString(stmt, idx++, uri);
		}
		else if (depth == 1) {
			stmt->setString(stmt, idx++, uri);
			stmt->setString(stmt, idx++, apr_pstrcat(p,
						stmt->escWildCard(stmt, uri), "/%", NULL));
			stmt->setInt(stmt, idx++, hierarchy_depth + 1);
		}
		else {
			stmt->setString(stmt, idx++, uri);
			stmt->setString(stmt, idx++, apr_pstrcat(p,
						stmt->escWildCard(stmt, uri), "/%", NULL));
		}

		if (IS_FILLED(userid)) {
			stmt->setString(stmt, idx++, userid);
		}
	}

	/* SQLの実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get divy_confirmreading table. "
			"(uri = %s, userid = %s) Reason: %s",
			REPLACE_NULL(uri), REPLACE_NULL(userid), rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 
	 * 結果の取得
	 * 取得結果を、divy_rdbo_confirmreading に入れて、その構造体を
	 * rsid でハッシュに格納する。同一のrsid が現れた場合には
	 * divy_rdbo_confirmreading のnext につないでList構造を作る。
	 */
	*cr_pr_hash = apr_hash_make(p);
	work_hash   = apr_hash_make(p);
	while (rset->next(rset) == DB_TRUE) {
		rsid = rset->getString(rset, 1);

		cr_pr = apr_hash_get(work_hash, rsid, APR_HASH_KEY_STRING);
		if (cr_pr == NULL) {
			cr_pr = apr_pcalloc(p, sizeof(*cr_pr));
			cr_pr->next = NULL;	/* sentinel */

			/* divy_rdbo_confirmreading からなるListの先頭を保存しておく*/
			apr_hash_set(*cr_pr_hash, rsid, APR_HASH_KEY_STRING, cr_pr);
		}
		else {
			cr_pr->next = apr_pcalloc(p, sizeof(*cr_pr));
			cr_pr       = cr_pr->next;
			cr_pr->next = NULL;	/* sentinel */
		}

		cr_pr->uri          = rset->getString(rset, 2);
		cr_pr->userid       = rset->getString(rset, 3);
		cr_pr->creationdate = (time_t) rset->getBigInt(rset, 4);

		/* 現在使用した divy_rdbo_confirmreading を憶えておく */
		apr_hash_set(work_hash, rsid, APR_HASH_KEY_STRING, cr_pr);
		cnt++;
	}

	if (cnt == 0) *cr_pr_hash = NULL; 

	/* 終了処理 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * セッションIDを作成する
 * @param r request_rec
 * @return セッションID char *
 */
static char* _make_session_sid(apr_pool_t *pool)
{
	apr_uuid_t uuid;
	char buff[APR_UUID_FORMATTED_LENGTH + 1];
	unsigned char md5[APR_MD5_DIGESTSIZE+1] = {0};
	const char hex[16] = "0123456789abcdef";
	unsigned char ch;
	char *digest, *first;
	int i = 0;
	apr_status_t st;

	apr_uuid_get(&uuid);
	apr_uuid_format(buff, &uuid);

	st = apr_md5(md5, buff, strlen(buff));
	if (st != APR_SUCCESS) return NULL; 

	digest = first = apr_pcalloc(pool, APR_MD5_DIGESTSIZE * 2 + 1);

    for(i=0; i < APR_MD5_DIGESTSIZE; i++) {
        ch = (unsigned char)md5[i];
        *digest++ = hex[ch >> 4];
        *digest++ = hex[ch & 0xF];
    }
    *digest++ = '\0';

	return first;
}


