/**
 * $Id$
 *
 * BOX機能関連をまとめたファイル
 *
 */
#include "httpd.h"
#include "http_protocol.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"
#include "util_md5.h"

#include "mod_dav_tf.h"
#include "tf_db.h"
#include "tf_rdbo.h"
#include "tf_rdbo_box.h"
#include "tf_box.h"
#include "util.h"

/*------------------------------------------------------------------------------
  Declare private prototype functions 
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define public functions 
  ----------------------------------------------------------------------------*/

/*
 * 短縮URLのBOX参照カウントを+1する
 */
DIVY_DECLARE(int) divy_rdbo_box_update_viewcount(request_rec *r, const char* shorten)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	char            *sqlstr = NULL;

	TRACE(p);

	/* BOX機能をサポートしていなければNULL */
	if (divy_support_tfbox(r) == 0) return 1;

	if (IS_EMPTY(shorten)) return 1;

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbConn.");
		return 1; 
	}

	/* 接続 */
	dbconn->startTrans(dbconn, 0);

	sqlstr = "UPDATE divy_box"
			 " SET "
			 " box_viewcount_bi= box_viewcount_bi + 1"
			 " WHERE "
			 " box_shorten_c = ?";

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, sqlstr, p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. update box count(shorten = %s) "
			"Reason: %s", shorten, stmt->getMsg(stmt));
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, shorten);

	/* sql実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update divy_box. (shorten = %s) "
			"Reason: %s", shorten, rset->getMsg(rset));
		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	dbconn->commit(dbconn);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/*
 * 短縮URLの文字列を元にBOXテーブルにある情報を取得する
 * 未使用
 *
 */
DIVY_DECLARE(divy_rdbo_box *) divy_rdbo_get_box_property_by_shorten(request_rec *r, const char* shorten)
{

	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	int             idx     = 0;
	divy_rdbo_box   *box    = NULL;
	divy_sbuf *sql_buf      = NULL;

	TRACE(p);

	/* BOX機能をサポートしていなければNULL */
	if (divy_support_tfbox(r) == 0) return NULL;

	if (IS_EMPTY(shorten)) return NULL;

	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf, "SELECT "
							  " box_uri_txt"
							  ",box_allowed_origin_c"
							  ",box_shorten_c"
							  ",box_flag_i"
							  ",box_passwd_vc"
							  ",box_create_bi"
							  ",box_expiration_bi"
							  ",box_creator_usr_id_vc"
							  ",box_greeting_vc"
							  ",box_message_vc"
							  ",box_viewcount_bi");

	if (divy_support_2FA(r)) {
		divy_sbuf_append(sql_buf, ",box_to_mailaddr_vc");
	}

	divy_sbuf_append(sql_buf, " FROM divy_box "
							  " WHERE box_shorten_c = ?");

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbConn.");
		return NULL; 
	}

	/* 接続 */
	dbconn->startTrans(dbconn, 0);

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", 
				REPLACE_NULL(stmt->getMsg(stmt)));
		dbconn->rollback(dbconn);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return NULL;
	}

	stmt->setString(stmt, 1, shorten);

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(shorten = %s) "
			"Reason: %s", shorten, rset->getMsg(rset));

		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return NULL;
	}

	/* 結果の取得 */
	idx = 1;
	if (rset->next(rset) == DB_TRUE) {
		/* 成功 一件だけしかしかない */
		box  = apr_pcalloc(p, sizeof(divy_rdbo_box));

		box->uri            = rset->getString(rset, idx++);
		box->allowed_origin = rset->getString(rset, idx++);
		box->shorten 		= rset->getString(rset, idx++);
		box->flag           = rset->getInt(rset,    idx++);
		box->password       = rset->getString(rset, idx++);
		box->creationdate   = rset->getBigInt(rset, idx++);
		box->expirationdate = rset->getBigInt(rset, idx++);
		box->creator_usr_id = rset->getString(rset, idx++);
		box->greeting       = rset->getString(rset, idx++);
		box->message        = rset->getString(rset, idx++);
		box->viewcount      = rset->getBigInt(rset, idx++);

		dbconn->commit(dbconn);
	}
	else {
		/* 取得出来なかった */
		dbconn->rollback(dbconn);
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* TODO もしくはなかったことになるためログを出したほうがいいのでは？ */

	return box;
}

/*
 * 指定したグループに対してシステム権限のユーザ(名前は._guest_)を投入する
 * トランザクションは必ずなければなりません。この関数では作りません
 * 既に存在していた場合はdivy_rdbo_insert_rusr()でエラーとなりますが、
 * 通常のオペレーションでは追加エラーはでません
 */
DIVY_DECLARE(int) divy_rdbo_box_append_guest_user(request_rec *r, const divy_rdbo_grp *grp_pr, divy_db_transaction_ctx* ts_ctx)
{
	apr_pool_t *p = r->pool;
	divy_rdbo_rusr *rusr_pr = NULL;
	divy_rdbo_rusr *rusr_tmp = NULL;
	char *rusr_id = NULL;
	char *rusr_uri = NULL;

	if (grp_pr == NULL) return 1;

	/* トランザクションがなければエラー */
	if (ts_ctx == NULL) return 1;

	rusr_id = apr_psprintf(p, DIVY_PREFIX_RUSER"%s", DIVY_GUEST_ID);
	rusr_uri = divy_build_m_group_uri(p, dav_divy_get_root_uri(r),
			apr_psprintf(p, "%s/%s/", grp_pr->relativeuri, rusr_id));

	/* 所属ユーザの一覧を取得しシステムユーザが存在するか確認する */
	if (divy_rdbo_get_rusr(r, rusr_uri, &rusr_tmp)) {
		return 1;
	}

	/* システムユーザを追加する但しシステムユーザが既にいたらしない */
	(void)divy_rdbo_parse_rusr_uri(r, rusr_uri, &rusr_pr);
	if (rusr_tmp == NULL && divy_rdbo_insert_rusr(r, rusr_pr, ts_ctx)) {
		return 1;
	}

	return 0;
}

/*
 * 指定したグループのBOX管理システムユーザを削除する
 */
DIVY_DECLARE(int) divy_rdbo_box_remove_guest_user(request_rec *r,const divy_rdbo_grp *grp_pr)
{
	divy_rdbo_rusr rusr_pr;

	rusr_pr.grpid = grp_pr->grpid;
	rusr_pr.usrid = DIVY_GUEST_ID;

	if (divy_rdbo_remove_rusr(r, &rusr_pr)) {
		return 1;
	}

	return 0;
}

