/**
 * $Id$
 *
 * 特殊フォルダ「クライアント更新」関連の関数をまとめたファイル
 *
 * (note)
 *   ここで定義された関数群はtf_rdbo.c にあったものです.
 *   tf_rdbo.c が巨大化し過ぎたため、関連のあるものをまとめました.
 */
#include "httpd.h"
#include "http_protocol.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"

#include "mod_dav_tf.h"
#include "tf_db.h"
#include "util_db.h"
#include "tf_rdbo.h"
#include "util.h"
#include "tf_folder.h"
#include "tf_rdbo_clupdate.h"
#include "tf_rdbo_util.h"
#include "tf_rdbo.h"

/*------------------------------------------------------------------------------
  Define public functions 
  ----------------------------------------------------------------------------*/
/**
 * 指定されたuri が示す更新クライアントの情報を取得して返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_clmodule_property(request_rec *r, const char *uri,
					divy_rdbo_clmodule **clmodule_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;

	*clmodule_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"uri is empty !!");
		return 1;
	}

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbConn.(uri = %s)", uri);
		return 1;
	}

	/* 接続 */
	dbconn->startTrans(dbconn, 0);

	/* sql文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"SELECT "
			"rs.rs_rs_id_c, "
			"rs.rs_dispname_vc, "
			"cm.cm_ver_vc, "
			"cm.cm_lineup_vc, "
			"cm_digest_vc, "
			"rs.rs_create_bi, "
			"rs.rs_get_lastmodified_bi "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
			"FROM dav_resource rs "
			"INNER JOIN divy_clmodule cm "
			"ON rs.rs_rs_id_c = cm.cm_rs_id_c "
			"WHERE rs.rs_uri_txt = ?", 
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
			" FROM dav_resource rs, divy_clmodule cm"
			" WHERE rs.rs_rs_id_c = cm.cm_rs_id_c"
			" AND rs.rs_uri_txt = ?",
#endif
			p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (uri = %s)\n "
			"Reason: %s", uri, stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, uri);

	/* sql実行　失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(uri = %s) "
			"Reason: %s", uri, rset->getMsg(rset));
		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* データがあれば取得(ない場合は初期化時のNULL) */
	if (rset->next(rset) == DB_TRUE) {
		/* 領域確保 */
		*clmodule_pr = apr_pcalloc(p, sizeof(divy_rdbo_clmodule));

		/* データ格納 */
		(*clmodule_pr)->rsid     = rset->getString(rset, 1);
		(*clmodule_pr)->name     = rset->getString(rset, 2);
		(*clmodule_pr)->version  = rset->getString(rset, 3);
		(*clmodule_pr)->lineup   = rset->getString(rset, 4);
		(*clmodule_pr)->digest   = rset->getString(rset, 5);
		divy_format_time_t(p, rset->getBigInt(rset, 6),
				DIVY_TIME_STYLE_ISO8601, &(*clmodule_pr)->registdt);
		divy_format_time_t(p, rset->getBigInt(rset, 7),
				DIVY_TIME_STYLE_ISO8601, &(*clmodule_pr)->updatedt);
		/* 次のclmodule_prはNULL */
		(*clmodule_pr)->next       = NULL;
	}

	/* クローズ */
	dbconn->commit(dbconn);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたclmodule_pr が表す更新クライアントの情報を新規登録する。
 * (note)
 * 同じモジュール名を持つエントリは登録できません。
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_clmodule_property(request_rec *r,
					const divy_rdbo_clmodule *clmodule_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit = 0;       /* トランザクションを確定するかどうか */
	int insert_cnt		= 0;

	TRACE(p);

	/* 必須項目のチェック */
	if (clmodule_pr == NULL || IS_EMPTY(clmodule_pr->uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Could not operation. There are empty value.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Could not operation. Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, 
			"INSERT INTO divy_clmodule ("
			"cm_rs_id_c, "
			"cm_ver_vc, "
			"cm_lineup_vc, "
			"cm_digest_vc) "
			"VALUES ((SELECT rs_rs_id_c FROM dav_resource WHERE rs_uri_txt = ?),"
			"?, ?, ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.(uri = %s) "
			"Reason: %s", 
			clmodule_pr->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, clmodule_pr->uri);
	stmt->setString(stmt, 2, clmodule_pr->version);
	stmt->setString(stmt, 3, clmodule_pr->lineup);
	stmt->setString(stmt, 4, clmodule_pr->digest);

	/* SQL実行　失敗時はエラー */
	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_clmodule.(uri = %s) "
			"Reason: %s", clmodule_pr->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	/* 挿入件数が0 件はエラー */
	if (insert_cnt == 0) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Status is OK, but the count of inserting is 0. "
			"Maybe the entry of dav_resource is missing."
			"(uri = %s)", clmodule_pr->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 変更を確定 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたclmodule_pr が表す内容でクライアント更新情報を更新する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_clmodule_property(request_rec *r,
					const divy_rdbo_clmodule *clmodule_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit = 0;       /* トランザクションを確定するかどうか */
	int update_cnt;

	TRACE(p);

	/* 必須項目のチェック */
	if (clmodule_pr == NULL || IS_EMPTY(clmodule_pr->uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"URI is NULL. Could not do process !!");
		return 1;
	}
	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Could not operation. Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"UPDATE divy_clmodule SET "
			"cm_ver_vc = ?, "
			"cm_lineup_vc = ?, "
			"cm_digest_vc = ? "
			"WHERE cm_rs_id_c = ("
			"SELECT rs_rs_id_c "
			"FROM dav_resource "
			"WHERE rs_uri_txt = ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.(uri = %s) "
			"Reason: %s", 
			clmodule_pr->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, clmodule_pr->version);
	stmt->setString(stmt, 2, clmodule_pr->lineup);
	stmt->setString(stmt, 3, clmodule_pr->digest);
	stmt->setString(stmt, 4, clmodule_pr->uri);

	/* SQL実行　失敗時はエラー */
	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update divy_clmodule.(uri = %s) "
			"Reason: %s", clmodule_pr->uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	/* 更新件数が1件以外はエラー */
	if (update_cnt != 1) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Status is OK, but the count of updating is 0. "
			"Maybe the entry of dav_resource is missing."
			"(uri = %s)", clmodule_pr->uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 変更を確定 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

