/**
 * $Id$
 *
 * 特殊フォルダ「DBMS」関連の関数をまとめたファイル
 *
 * (note)
 *   ここで定義された関数群はtf_rdbo.c にあったものです.
 *   tf_rdbo.c が巨大化し過ぎたため、関連のあるものをまとめました.
 */
#include "httpd.h"
#include "http_protocol.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"

#include "mod_dav_tf.h"
#include "tf_db.h"
#include "util_db.h"
#include "tf_rdbo.h"
#include "util.h"
#include "search.h"
#include "tf_folder.h"
#include "tf_valuecache.h"
#include "tf_linkedlist.h"
#include "tf_rdbo_util.h"
#include "tf_rdbo.h"
#include "tf_rdbo_dbms.h"


/*------------------------------------------------------------------------------
  Define public functions 
  ----------------------------------------------------------------------------*/
/**
 * リポジトリDBを検索する(dbmsinformationsearch)
 *
 */
DIVY_DECLARE(int) divy_rdbo_dbmsinformationsearch(divy_rdbo_search_params *params,
						divy_search_dbmsis_iscreen *screen,
						divy_rdbo_search_output *output)
{
	request_rec *r          = params->r;
	apr_pool_t *p		= params->r->pool;
	apr_pool_t *wp          = params->wp;
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;

	divy_rdbo_dbms *dbms    = NULL;
	char *type ,*dummy, *dbmsid;
	apr_hash_t *licensed_dbmstype_h = NULL;
	RET_STAT retstat;
	divy_db_transaction_ctx *ts_ctx   = NULL;
	char *sqlstr            = NULL;
 
	output->list.dbms_pr = NULL;  /* 初期化 */
 
	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SELECT・FROM句作成 */
	sqlstr =	"SELECT"
			" dbms.ds_id_name_vc"
			", dbms.ds_ds_id_c"
			", dbms.ds_type_vc"
			", dbms.ds_hostname_vc"
			", dbms.ds_port_i"
			", dbms.ds_dbname_vc"
			", dbms.ds_username_vc"
			", dbms.ds_password_vc"
			", dbms.ds_regist_c"
			", dbms.ds_update_c"
			", dbms.ds_comment_vc"
			", dbms.ds_active_i"
			" FROM divy_dbms dbms";

	/* content */
	if (params->optflg == DIVY_SEARCH_OPT_CONTENT) {
		sqlstr = apr_pstrcat(wp, sqlstr,
				" WHERE dbms.ds_ds_id_c = ?", NULL);
	}
	/* detaillist */
	else if (params->optflg == DIVY_SEARCH_OPT_DETAILLIST) {
		/* デフォルト値でよい */
	}
	/* activedbms */
	else if (params->optflg == DIVY_SEARCH_OPT_ACTIVEDBMS) {
		sqlstr =	"SELECT"
				" dbms.ds_id_name_vc"
				", dbms.ds_ds_id_c"
				", dbms.ds_type_vc"
				" FROM divy_dbms dbms"
				" WHERE dbms.ds_active_i = 1";
	}
	else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
			"Unknown optflg found. (code = %d)", params->optflg);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * SQL文の作成
	 */
	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, sqlstr, wp);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", 
			stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	if (params->optflg == DIVY_SEARCH_OPT_CONTENT) {
		stmt->setString(stmt, 1, screen->dbmsid);
	}
    
	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.Reason: %s", 
			rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* DBMS のライセンスを取得 */
	if (params->optflg == DIVY_SEARCH_OPT_ACTIVEDBMS) {

		retstat = divy_db_get_licensed_dbmstype(r, &licensed_dbmstype_h);
		if (retstat !=  DB_SUCCESS) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get licensed dbmstype list.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
	}

	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE) {
		dbmsid = rset->getString(rset, 2);

		/* リポジトリDB用エントリの場合にはtype を補う */
		if (strcmp(dbmsid, DIVY_REPOS_DBMSID) == 0) {
			dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
			type = apr_pstrdup(p, dconf->dbmstype);
		}
		else {
			type = rset->getString(rset, 3);
		}

		/* activedbms でライセンスなしの場合はスキップ */
		if (params->optflg == DIVY_SEARCH_OPT_ACTIVEDBMS) {
			if (licensed_dbmstype_h && (dummy =
				apr_hash_get(licensed_dbmstype_h, type,
						APR_HASH_KEY_STRING)) == NULL) {
				continue;	/* ライセンスがなかった */
			}
		}

		/* 領域確保 */
		if (output->list.dbms_pr == NULL) {
			output->list.dbms_pr = dbms =
					apr_pcalloc(wp, sizeof(divy_rdbo_dbms));
		}
		else {
			dbms->next = apr_pcalloc(wp, sizeof(divy_rdbo_dbms));
			dbms = dbms->next;
		}

		/* データを取得 */
		dbms->name   = rset->getString(rset, 1);
		dbms->dbmsid = dbmsid;
		dbms->type   = type;

		/* detaillist,content はさらに取得 */
		if (params->optflg == DIVY_SEARCH_OPT_DETAILLIST ||
			params->optflg == DIVY_SEARCH_OPT_CONTENT) {

			/* リポジトリDB用エントリの場合には特別扱い */
			if (strcmp(dbms->dbmsid, DIVY_REPOS_DBMSID) == 0) {
				dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

				dbms->hostname  = apr_pstrdup(p, dconf->hostname);
				dbms->port      = atoi(dconf->hostport);
				dbms->dbname    = apr_pstrdup(p, dconf->dbname);
				dbms->username  = apr_pstrdup(p, dconf->username);
				dbms->password  = apr_pstrdup(p, dconf->password);
				dbms->active    = DIVY_DBMS_ACTIVE;	/* 常にアクティブ */
			}
			else {
				dbms->hostname  = rset->getString(rset, 4);
				dbms->port      = rset->getInt	 (rset, 5);
				dbms->dbname    = rset->getString(rset, 6);
				dbms->username  = rset->getString(rset, 7);
				dbms->password  = rset->getString(rset, 8);
				dbms->active    = (divy_rdbo_dbmsactive) rset->getInt(rset, 12);
			}
			dbms->registdt 	= rset->getString(rset, 9);
			dbms->updatedt 	= rset->getString(rset, 10);
			dbms->comment 	= rset->getString(rset, 11);
		}
	}

	divy_db_commit_transaction(ts_ctx);

	/* 使用した領域を解放 (dbconn は何もしてはならない) */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
    
	return 0;
}

/**
 * リポジトリDB に登録されているアクテイブなDBMS接続情報を取得して返却する。
 * リポジトリDB用の擬似エントリも取得しますが、接続情報は正しくありません.
 * ディレクティブの値を使ってきちんと置き換えてから利用して下さい.
 * (note)
 * 	この関数はDBプロバイダの管理モジュールが、プロバイダを初期化する際に
 * 	必要な情報を取得する目的で使用されます。
 * 	その他の関数からは使用してはなりません。
 */
DIVY_DECLARE(int) divy_rdbo_get_dbmsinfo(apr_pool_t *p,
					DbConn *dbconn,
					divy_rdbo_dbms **dbms_pr)
{
	DbPreparedStmt  *stmt        = NULL;
	DbResultSet     *rset        = NULL;
	divy_rdbo_dbms  *dbms        = NULL;

	*dbms_pr = NULL;	/* 初期化 */

	/* (前堤) トランザクションは開始されていない */
	dbconn->startTrans(dbconn, 0);
	stmt =	dbconn->prepareStatement(dbconn, 
			"SELECT "
			"ds_id_name_vc,"
			"ds_hostname_vc,"
			"ds_port_i,"
			"ds_dbname_vc,"
			"ds_username_vc,"
			"ds_password_vc, "
			"ds_type_vc,"
			"ds_db_pool_i,"
			"ds_min_spare_conn_i,"
			"ds_max_spare_conn_i,"
			"ds_active_i,"
			"ds_ds_id_c "
			"FROM divy_dbms "
			"WHERE ds_active_i = 1", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt."
			"Reason: %s", stmt->getMsg(stmt));

		dbconn->rollback(dbconn);
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_dbms table."
			"Reason: %s", rset->getMsg(rset));

		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* データの取得 */
	while (rset->next(rset) == DB_TRUE) {
		/* 領域確保 */
		if (*dbms_pr == NULL) {
			*dbms_pr = apr_pcalloc(p, sizeof(divy_rdbo_dbms));
			dbms = *dbms_pr;
		}
		else {
			dbms->next = apr_pcalloc(p, sizeof(divy_rdbo_dbms));
			dbms       = dbms->next;
		}

		dbms->name           = rset->getString(rset, 1);
		dbms->hostname       = rset->getString(rset, 2);
		dbms->port           = rset->getInt(rset,    3);
		dbms->dbname         = rset->getString(rset, 4);
		dbms->username       = rset->getString(rset, 5);
		dbms->password       = rset->getString(rset, 6);
		dbms->type           = rset->getString(rset, 7);
		dbms->dbpool         = rset->getInt(rset,    8);
		dbms->dbminspareconn = rset->getInt(rset,    9);
		dbms->dbmaxspareconn = rset->getInt(rset,   10);
		dbms->active         = (divy_rdbo_dbmsactive) rset->getInt(rset, 11);
		dbms->dbmsid         = rset->getString(rset,12);

		dbms->next           = NULL;
	}

	dbconn->commit(dbconn);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたdbmsidname が示すDBMS接続情報を取得して返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_dbms_property(request_rec *r,
						const char *dbmsidname,
						divy_rdbo_dbms **dbms_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	dav_divy_dir_conf *dconf= dav_divy_get_dir_config(r);

	*dbms_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(dbmsidname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"dbmsidname is empty.");
		return 1;
	}

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.(dbmsidname = %s)", dbmsidname);
		return 1;
	}

	dbconn->startTrans(dbconn, 0);
	stmt = dbconn->prepareStatement(dbconn,
			"SELECT "
			"ds_ds_id_c,"
			"ds_type_vc,"
			"ds_id_name_vc,"
			"ds_hostname_vc,"
			"ds_port_i,"
			"ds_dbname_vc,"
			"ds_username_vc,"
			"ds_password_vc,"
			"ds_comment_vc,"
			"ds_active_i,"
			"ds_regist_c,"
			"ds_update_c "
			"FROM divy_dbms "
			"WHERE ds_id_name_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (dbmsidname = %s) "
			"Reason: %s", dbmsidname, stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, dbmsidname);

	/* sql実行　失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(dbmsidname = %s) "
			"Reason: %s", dbmsidname, rset->getMsg(rset));
		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* データがあれば取得(ない場合は初期化時のNULL) */
	if (rset->next(rset) == DB_TRUE) {
		*dbms_pr = apr_pcalloc(p, sizeof(divy_rdbo_dbms));

		/* データ格納 */
		(*dbms_pr)->dbmsid   = rset->getString(rset, 1);
		(*dbms_pr)->name     = rset->getString(rset, 3);
		(*dbms_pr)->comment  = rset->getString(rset, 9);
		(*dbms_pr)->registdt = rset->getString(rset,11);
		(*dbms_pr)->updatedt = rset->getString(rset,12);

		/* リポジトリDB用エントリの場合には特別扱い */
		if (strcmp((*dbms_pr)->dbmsid, DIVY_REPOS_DBMSID) == 0) {
			(*dbms_pr)->type     = apr_pstrdup(p, dconf->dbmstype);
			(*dbms_pr)->hostname = apr_pstrdup(p, dconf->hostname);
			(*dbms_pr)->port     = atoi(dconf->hostport);
			(*dbms_pr)->dbname   = apr_pstrdup(p, dconf->dbname);
			(*dbms_pr)->username = apr_pstrdup(p, dconf->username);
			(*dbms_pr)->password = apr_pstrdup(p, dconf->password);
			(*dbms_pr)->active   = DIVY_DBMS_ACTIVE; /* 常にアクティブ */
		}
		else {
			(*dbms_pr)->type     = rset->getString(rset, 2);
			(*dbms_pr)->hostname = rset->getString(rset, 4);
			(*dbms_pr)->port     = rset->getInt(rset,    5);
			(*dbms_pr)->dbname   = rset->getString(rset, 6);
			(*dbms_pr)->username = rset->getString(rset, 7);
			(*dbms_pr)->password = rset->getString(rset, 8);
			(*dbms_pr)->active   = (divy_rdbo_dbmsactive) rset->getInt(rset, 10);
		}

		/* 次のdbms_prはNULL */
		(*dbms_pr)->next       = NULL;
	}

	/* クローズ */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	dbconn->commit(dbconn);

	return 0;
}

/**
 * 指定されたdbms_pr が表すDBMS情報を新規登録する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_dbms_property(request_rec *r,
						const divy_rdbo_dbms *dbms_pr,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int insert_cnt          = 0;
	time_t short_time       = dav_divy_get_now_epoch();
	char *datebuf           = NULL;

	TRACE(p);

	if (dbms_pr == NULL || IS_EMPTY(dbms_pr->name)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Could not operation. There are empty value.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_dbms テーブルへの新規追加
	 */
	stmt = dbconn->prepareStatement(dbconn, 
			"INSERT INTO divy_dbms "
			"(ds_ds_id_c,"
			" ds_type_vc,"
			" ds_id_name_vc,"
			" ds_hostname_vc,"
			" ds_port_i,"
			" ds_dbname_vc,"
			" ds_username_vc,"
			" ds_password_vc,"
			" ds_db_pool_i,"
			" ds_min_spare_conn_i,"
			" ds_max_spare_conn_i,"
			" ds_comment_vc,"
			" ds_active_i,"
			" ds_regist_c,"
			" ds_update_c) "
			"VALUES ("DIVY_DBFUNC_CREATE_DS_SEQ","
			"?,?,?,?,?,?,?,?,?,?,?,?,?,?)", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (name = %s) "
			"Reason: %s", dbms_pr->name, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	stmt->setString(stmt,  1, dbms_pr->type);
	stmt->setString(stmt,  2, dbms_pr->name);
	stmt->setString(stmt,  3, dbms_pr->hostname);
	stmt->setInt(stmt,     4, dbms_pr->port);
	stmt->setString(stmt,  5, dbms_pr->dbname);
	stmt->setString(stmt,  6, dbms_pr->username);
	stmt->setString(stmt,  7, dbms_pr->password);
	stmt->setInt(stmt,     8, dbms_pr->dbpool);
	stmt->setInt(stmt,     9, dbms_pr->dbminspareconn);
	stmt->setInt(stmt,    10, dbms_pr->dbmaxspareconn);
	stmt->setString(stmt, 11, REPLACE_EMPTYSTR(dbms_pr->comment));
	stmt->setInt(stmt,    12, (apr_int32_t) dbms_pr->active);
	stmt->setString(stmt, 13, datebuf);
	stmt->setString(stmt, 14, datebuf);

	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_dbms.(name = %s) "
			"Reason: %s", dbms_pr->name, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	else if (insert_cnt == 0) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"Status is OK, but the count of inserting is 0. "
			"Please check divy_dbms. (name = %s)", dbms_pr->name);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたdbms_pr が表すDBMS情報の内容で更新する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_dbms_property(request_rec *r,
						const divy_rdbo_dbms *dbms_pr,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	time_t short_time       = dav_divy_get_now_epoch();
	char *datebuf           = NULL;
	int update_cnt          = 0;

	TRACE(p);

	if (dbms_pr == NULL || IS_EMPTY(dbms_pr->dbmsid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Could not operation. usrid is NULL.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない -> ここでは作成せずエラーにする */
	if (ts_ctx == NULL) {
	 	/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Could not operation. Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_dbms テーブルの更新
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"UPDATE divy_dbms "
				"SET "
				"ds_type_vc = ?,"
				"ds_hostname_vc = ?,"
				"ds_port_i = ?,"
				"ds_dbname_vc = ?,"
				"ds_username_vc = ?,"
				"ds_password_vc = ?,"
				"ds_db_pool_i = ?,"
				"ds_min_spare_conn_i = ?,"
				"ds_max_spare_conn_i = ?,"
				"ds_comment_vc = ?,"
				"ds_active_i = ?,"
				"ds_update_c = ? "
				"WHERE ds_ds_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (dbmsid = %s) "
			"Reason: %s", dbms_pr->dbmsid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	stmt->setString(stmt,  1, dbms_pr->type);
	stmt->setString(stmt,  2, dbms_pr->hostname);
	stmt->setInt(stmt,     3, dbms_pr->port);
	stmt->setString(stmt,  4, dbms_pr->dbname);
	stmt->setString(stmt,  5, dbms_pr->username);
	stmt->setString(stmt,  6, dbms_pr->password);
	stmt->setInt(stmt,     7, dbms_pr->dbpool);
	stmt->setInt(stmt,     8, dbms_pr->dbminspareconn);
	stmt->setInt(stmt,     9, dbms_pr->dbmaxspareconn);
	stmt->setString(stmt, 10, REPLACE_EMPTYSTR(dbms_pr->comment));
	stmt->setInt(stmt,    11, (apr_int32_t) dbms_pr->active);
	stmt->setString(stmt, 12, datebuf);
	stmt->setString(stmt, 13, dbms_pr->dbmsid);

	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update divy_dbms. (dbmsid = %s) "
			"Reason: %s", dbms_pr->dbmsid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	else if (update_cnt == 0) {
		/* 恐らく、select してからupdate されるまでの間に
		 * エントリが消されてしまったということ */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Status is OK, but the count of updating is 0."
			"Probably because of competing with remove "
			"operation, this entry of user already have "
			"been lost. "
			"Please check your data.(dbmsid = %s)",
			dbms_pr->dbmsid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* 変更を確定 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたdbms_pr が表すDBMS情報を削除する
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_dbms_property(request_rec *r,
						const divy_rdbo_dbms *dbms_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int delete_cnt          = 0;
	divy_db_transaction_ctx *ts_ctx = NULL;

	TRACE(p);

	if (dbms_pr == NULL || IS_EMPTY(dbms_pr->dbmsid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Could not operation. There are empty value.");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_dbms テーブルの削除
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"DELETE FROM divy_dbms "
				"WHERE ds_ds_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (dbmsid = %s) "
			"Reason: %s", dbms_pr->dbmsid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, dbms_pr->dbmsid);

	/* sql実行 失敗時はエラー */
	delete_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_dbms.(dbmsid = %s) "
			"Reason: %s", dbms_pr->dbmsid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	/* データが既に削除されていた場合はwarning */
	if (delete_cnt == 0) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The \"divy_dbms\" table did not have a recorde of "
			"this dbms.(dbmsid = %s) "
			"Probably data is already deleted. ", dbms_pr->dbmsid);
	}

	/* 反映を確定する */
	divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたsrc_dbms_pr が表すDBMS情報の内容をdst_dbms_pr が示す
 * DBMS情報としてコピーする。
 *
 */
DIVY_DECLARE(int) divy_rdbo_copy_dbms_property(request_rec *r,
						const divy_rdbo_dbms *src_dbms_pr,
						const divy_rdbo_dbms *dst_dbms_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	divy_db_transaction_ctx *ts_ctx;
	char *datebuf           = NULL;
	time_t short_time       = dav_divy_get_now_epoch();
	int insert_cnt          = 0;
	dav_divy_dir_conf *dconf= dav_divy_get_dir_config(r);

	TRACE(p);

	/* 必須項目のチェック */
	if (src_dbms_pr == NULL || dst_dbms_pr == NULL ||
		IS_EMPTY(src_dbms_pr->dbmsid) || IS_EMPTY(dst_dbms_pr->name)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There are empty values.");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_dbms の内容をコピーする
	 */
	/* リポジトリDB用エントリの場合には特別扱い */
	if (strcmp(src_dbms_pr->dbmsid, DIVY_REPOS_DBMSID) == 0) {
		stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO divy_dbms "
				"(ds_ds_id_c,"
				" ds_type_vc,"
				" ds_id_name_vc,"
				" ds_hostname_vc,"
				" ds_port_i,"
				" ds_dbname_vc,"
				" ds_username_vc,"
				" ds_password_vc,"
				" ds_db_pool_i,"
				" ds_min_spare_conn_i,"
				" ds_max_spare_conn_i,"
				" ds_comment_vc,"
				" ds_active_i,"
				" ds_regist_c,"
				" ds_update_c) "
				"(SELECT "DIVY_DBFUNC_CREATE_DS_SEQ","
				" "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_DSTYPE_LEN))","
				" "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_DBIDNAME_LEN))","
				" "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_DSHOSTNAME_LEN))","
				" ?,"
				" ds_dbname_vc,"
				" "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_DSUSERNAME_LEN))","
				" "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_DSPASSWORD_LEN))","
				" ds_db_pool_i,"
				" ds_min_spare_conn_i,"
				" ds_max_spare_conn_i,"
				" ds_comment_vc,"
				" ?,"
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN))","
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN))" "
				"FROM divy_dbms "
				"WHERE ds_ds_id_c = ?)", p);
	}
	else {
		stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO divy_dbms "
				"(ds_ds_id_c,"
				" ds_type_vc,"
				" ds_id_name_vc,"
				" ds_hostname_vc,"
				" ds_port_i,"
				" ds_dbname_vc,"
				" ds_username_vc,"
				" ds_password_vc,"
				" ds_db_pool_i,"
				" ds_min_spare_conn_i,"
				" ds_max_spare_conn_i,"
				" ds_comment_vc,"
				" ds_active_i,"
				" ds_regist_c,"
				" ds_update_c) "
				"(SELECT "DIVY_DBFUNC_CREATE_DS_SEQ","
				" ds_type_vc,"
				" "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_DBIDNAME_LEN))","
				" ds_hostname_vc,"
				" ds_port_i,"
				" ds_dbname_vc,"
				" ds_username_vc,"
				" ds_password_vc,"
				" ds_db_pool_i,"
				" ds_min_spare_conn_i,"
				" ds_max_spare_conn_i,"
				" ds_comment_vc,"
				" ds_active_i,"
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN))","
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN))" "
				"FROM divy_dbms "
				"WHERE ds_ds_id_c = ?)", p);
	}
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (dbmsid = %s) "
			"Reason: %s", src_dbms_pr->dbmsid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* バインド */
	if (strcmp(src_dbms_pr->dbmsid, DIVY_REPOS_DBMSID) == 0) {
		stmt->setString(stmt,  1, dconf->dbmstype);
		stmt->setString(stmt,  2, dst_dbms_pr->name);
		stmt->setString(stmt,  3, dconf->hostname);
		stmt->setInt(stmt,     4, atoi(dconf->hostport));
		stmt->setString(stmt,  5, dconf->username);
		stmt->setString(stmt,  6, dconf->password);
		stmt->setInt(stmt,     7, DIVY_DBMS_ACTIVE);
		stmt->setString(stmt,  8, datebuf);
		stmt->setString(stmt,  9, datebuf);
		stmt->setString(stmt, 10, src_dbms_pr->dbmsid);
	}
	else {
		stmt->setString(stmt,  1, dst_dbms_pr->name);
		stmt->setString(stmt,  2, datebuf);
		stmt->setString(stmt,  3, datebuf);
		stmt->setString(stmt,  4, src_dbms_pr->dbmsid);
	}

	/* sql実行 失敗時はエラー */
	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_dbms."
			"(src_dbmsid = %s, dst_name = %s) Reason: %s", 
			src_dbms_pr->dbmsid, dst_dbms_pr->name,
			stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	else if (insert_cnt == 0) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"Status is OK, but the count of inserting is 0. "
			"Please check divy_dbms. (dbmsid = %s)",
			src_dbms_pr->dbmsid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* 反映を確定する */
	divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	return 0;
}

/**
 * 指定されたsrc_dbms_pr が表すDBMS情報の内容をdst_dbms_pr が示す
 * DBMS情報として移動(名称変更)する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_move_dbms_property(request_rec *r,
						const divy_rdbo_dbms *src_dbms_pr,
						const divy_rdbo_dbms *dst_dbms_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	divy_db_transaction_ctx *ts_ctx;
	char *datebuf           = NULL;
	time_t short_time       = dav_divy_get_now_epoch();
	int update_cnt          = 0;

	TRACE(p);

	/* 必須項目のチェック */
	if (src_dbms_pr == NULL || dst_dbms_pr == NULL ||
		IS_EMPTY(src_dbms_pr->dbmsid) || IS_EMPTY(dst_dbms_pr->name)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There are empty values.");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

       /*
	* divy_dbms の内容を更新する
	*/
	stmt = dbconn->prepareStatement(dbconn, 
				"UPDATE divy_dbms "
				"SET "
				"ds_id_name_vc = ?,"
				"ds_update_c = ? "
				"WHERE ds_ds_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (dbmsid = %s) "
			"Reason: %s", src_dbms_pr->dbmsid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* バインド */
	stmt->setString(stmt,  1, dst_dbms_pr->name);
	stmt->setString(stmt,  2, datebuf);
	stmt->setString(stmt,  3, src_dbms_pr->dbmsid);

	/* sql実行 失敗時はエラー */
	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update divy_dbms."
			"(src_dbmsid = %s, dst_name = %s) Reason: %s", 
			src_dbms_pr->dbmsid, dst_dbms_pr->name,
			stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	else if (update_cnt == 0) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Status is OK, but the count of updating is 0."
			"Probably because of competing with remove "
			"operation, this entry of user already have "
			"been lost. "
			"Please check your data.(dbmsid = %s)",
			src_dbms_pr->dbmsid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* 反映を確定する */
	divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	return 0;
}

