/**
 * $Id$
 *
 * 特殊フォルダ「グループ」関連の関数をまとめたファイル
 *
 * (note)
 *   ここで定義された関数群はtf_rdbo.c にあったものです.
 *   tf_rdbo.c が巨大化し過ぎたため、関連のあるものをまとめました.
 */
#include "httpd.h"
#include "http_protocol.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"

#include "mod_dav_tf.h"
#include "tf_db.h"
#include "util_db.h"
#include "util_ldap.h"
#include "util_ml.h"
#include "tf_rdbo.h"
#include "util.h"
#include "search.h"
#include "repos.h"
#include "tf_folder.h"
#include "tf_linkedlist.h"
#include "tf_rdbo_util.h"
#include "tf_rdbo_group.h"
#include "tf_rdbo_box.h"
#include "tf_plugin.h"
#include "tf_confirmreading.h"
#include "tf_box.h"

APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Define enum and structures
  ----------------------------------------------------------------------------*/
/**
 * _get_group_property_by_condition で使用する条件値
 */
typedef enum divy_whereclause_type	divy_whereclause_type;
enum divy_whereclause_type {
	DIVY_WHERECLAUSE_GRPNAME = 0,	/* グループ名で検索する */
	DIVY_WHERECLAUSE_GRPID,			/* グループIDで検索する */
	DIVY_WHERECLAUSE_RESURI,		/* リソースURIで検索する */
};

/*------------------------------------------------------------------------------
  Declare private functions 
  ----------------------------------------------------------------------------*/
static int _get_group_property_by_condition(request_rec *r,
					divy_whereclause_type wheretype, const char *condition,
					divy_rdbo_grp **grp_pr, divy_db_transaction_ctx *ts_ctx);

static int _divy_rdbo_insert_upload_policy(request_rec *r,
					const char* grpid, const divy_rdbo_grp *grp_pr,
					divy_db_transaction_ctx *ts_ctx);

static int _divy_rdbo_exists_upload_policy(request_rec *r,
					const divy_rdbo_grp *grp_pr,
					divy_db_transaction_ctx *ts_ctx);

static int _divy_rdbo_remove_upload_policy(request_rec *r,
					const divy_rdbo_grp *grp_pr,
					divy_db_transaction_ctx *ts_ctx);

/*------------------------------------------------------------------------------
  Define public functions 
  ----------------------------------------------------------------------------*/
/**
 * リポジトリDBを検索する(groupinformationsearch のcontent, detail, treelist)
 * 
 */
DIVY_DECLARE(int) divy_rdbo_groupinformationsearch(divy_rdbo_search_params *params,
						divy_search_grpis_iscreen *screen,
						divy_rdbo_search_output *output)
{
	request_rec *r      = params->r;
	apr_pool_t *p       = params->r->pool;
	apr_pool_t *wp      = params->wp;
	apr_pool_t *scratchpool = params->scratchpool;
	DbConn *dbconn 		= NULL;
	DbPreparedStmt *stmt= NULL;
	DbResultSet *rset 	= NULL;
	divy_db_transaction_ctx *ts_ctx  = NULL;
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
	int support_uploadpolicy   = divy_support_upload_policy(r);
	int support_groupquota     = divy_support_groupquota(r);
	int extra_bind = 0, idx, use_ownerinfo = 0;
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	int ret = 0;
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	apr_hash_t *gmw_h = NULL;
	divy_sbuf *sql_buf = NULL;
	char *grpuri = NULL;
	divy_rdbo_grp *grp = NULL;
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	const char *own_userid = divy_get_userid(r);
	apr_hash_t *grp_count_h = NULL;

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen is EMPTY.");
		return 1;
	}

	output->list.grp_pr  = NULL;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* detaillist,content の場合、グループメール監視状態を先に取得しておく */
	if (params->optflg == DIVY_SEARCH_OPT_DETAILLIST ||
		params->optflg == DIVY_SEARCH_OPT_DETAILLIST_TOP ||
		params->optflg == DIVY_SEARCH_OPT_CONTENT) {

		if (divy_rdbo_get_groupmailwatch_properties(r, &gmw_h, ts_ctx)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get groupmailwatch information.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * グループIDが検索に指定された場合 (detaillist, treelist)、
	 * グループの相対URLが判らないので予め取得しておく.
	 */
	if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPID &&
		(params->optflg == DIVY_SEARCH_OPT_TREELIST ||
		 params->optflg == DIVY_SEARCH_OPT_DETAILLIST)) {

		/* SQL文の準備 */
		stmt = dbconn->prepareStatement(dbconn,
					"SELECT"
					" grp_relative_uri_txt"
					" FROM divy_grp"
					" WHERE grp_grp_id_c = ?", wp);
		if (stmt->getCode(stmt) != DB_SUCCESS){
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		/* バインド */
		stmt->setString(stmt, 1, screen->grpid);

		/* SQL実行 */
		rset = stmt->executeQuery(stmt, wp);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to select divy_grp. (Reason: %s)", rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* 結果の取得 */
		if (rset->next(rset) == DB_TRUE) {
			grpuri = rset->getString(rset, 1);
		}
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* グループURIが取得出来なかったら終了 */
		if (IS_EMPTY(grpuri)) {
			divy_db_commit_transaction(ts_ctx);
			return 0;
		}
	}

	divy_sbuf_create(wp, &sql_buf, 512);	/* SQLバッファの作成 */

	/*
	 * 下位グループに何件のグループがあるかどうかを取得する.
	 * treelist のときのみ
	 */
	if (support_groupleader &&
		(params->optflg == DIVY_SEARCH_OPT_TREELIST ||
		 params->optflg == DIVY_SEARCH_OPT_TREELIST_TOP)) {
		char *relative_uri;
		int count;

		divy_sbuf_append(sql_buf,
					"SELECT"
					" grp.parent_uri,"
					" count(grp.parent_uri) AS cnt"
					" FROM"
					" (SELECT"
					"  grp_relative_uri_txt,"
					"  "DIVY_DBFUNC_SUBSTRING2("grp_relative_uri_txt", "1",
						DIVY_DBFUNC_CHARLEN("grp_relative_uri_txt")" - " 
						DIVY_DBFUNC_CHARLEN("substring(grp_relative_uri_txt, '/[^/]*$')"))
					"  AS parent_uri"
					"  FROM divy_grp"
					"  WHERE grp_relative_uri_txt");

		/* 1階層目移行 */
		if (IS_FILLED(screen->grpid)) {
			divy_sbuf_append(sql_buf,
					"  LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE
					"  AND grp_depth_i IN (?, ?)");
		}
		/* トップ階層 -> ルート自身は要らない(仕様) */
		else {
			divy_sbuf_append(sql_buf,
					"  LIKE '/%'"
					"  AND grp_depth_i = 2");
		}

		if (divy_rdbo_is_groupleader(own_extstatus)) {
			divy_sbuf_append(sql_buf,
					" AND grp_owner_usr_id_vc = ?");
		}

		divy_sbuf_append(sql_buf,
					") AS grp"
					" GROUP BY grp.parent_uri");

		stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), wp);
		if (stmt->getCode(stmt) != DB_SUCCESS){
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* バインド */
		idx = 1;
		if (IS_FILLED(screen->grpid)) {
			stmt->setString(stmt, idx++, apr_pstrcat(wp,
						stmt->escWildCard(stmt, grpuri), "/%", NULL));
			stmt->setInt(stmt, idx++, divy_count_dirs(grpuri) + 1);
			stmt->setInt(stmt, idx++, divy_count_dirs(grpuri) + 2);
		}
		if (divy_rdbo_is_groupleader(own_extstatus)) {
			stmt->setString(stmt, idx++, own_userid);
		}

		/* SQL実行 */
		rset = stmt->executeQuery(stmt, wp);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to select divy_grp. (Reason: %s)", rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* 結果の取得 */
		while (rset->next(rset) == DB_TRUE) {
			relative_uri = rset->getString(rset, 1);
			count        = rset->getInt(rset, 2);

			if (grp_count_h == NULL) {
				grp_count_h = apr_hash_make(wp);
			}
			apr_hash_set(grp_count_h, relative_uri, APR_HASH_KEY_STRING, (const void *)count);
		}

		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		divy_sbuf_clear(sql_buf);	/* バッファのクリア */
	}

	/*
	 * SQL文の作成
	 */
	divy_sbuf_append(sql_buf,
					"SELECT"
					" grp.grp_relative_uri_txt"
					", grp.grp_grp_id_c"
					", grp.grp_name_vc"
					", grp.grp_depth_i"
					", grp.grp_regist_c"
					", grp.grp_update_c"
					", grp.grp_comment_vc"
					", rs.rs_uri_txt");

	/* グループ制約 / グループ管理者機能をサポートしている場合 */
	if (support_grpconstraints || support_groupleader) {
		divy_sbuf_append(sql_buf,
					", grp.grp_extended_status_c");
	}

	/* グループ管理者機能をサポートしている場合 */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
					", grp.grp_owner_usr_id_vc"
					", usr2.usr_fullname_vc AS ownername");
	}

	/* アップロードポリシー機能をサポートしている場合 */
	if (support_uploadpolicy) {
		divy_sbuf_append(sql_buf,
					", up.upload_flag_i"
					", up.upload_rules_filename_vc"
					", up.upload_rules_suffix_vc"
					", up.upload_rules_chartype_i"
					", up.upload_rules_length_vc"
					", up.upload_rules_matchtype_i"
					", up.upload_rules_hidden_i"
					", up.upload_update_bi");
	}

	/* グループQuota機能をサポートしている場合 */
	if (support_groupquota) {
		divy_sbuf_append(sql_buf,
					",gq.gsqt_used_st_bi"
					",gq.gsqt_max_st_bi"
					",gq.gsqt_used_res_bi"
					",gq.gsqt_max_res_bi");
	}

	divy_sbuf_append(sql_buf,
					" FROM divy_grp grp "
					" INNER JOIN dav_resource rs "
					" ON grp.grp_rs_id_c = rs.rs_rs_id_c ");

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usr usr2"
				" ON (grp.grp_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}

	if (support_uploadpolicy) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_uploadpolicy up"
				" ON (grp.grp_grp_id_c= up.upload_grp_id_c)");
	}

	if (support_groupquota) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_grpdiskquota gq"
				" ON (grp.grp_grp_id_c = gq.gsqt_grp_id_c)");
	}

	if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPID) {
		divy_sbuf_append(sql_buf,
					" WHERE (grp.grp_grp_id_c = ?");
	}
	else if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPURI) {
		divy_sbuf_append(sql_buf,
					" WHERE (grp.grp_relative_uri_txt = ?");
	}
	else if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPCOLURI) {
		divy_sbuf_append(sql_buf,
					" WHERE (grp.grp_rs_id_c = (SELECT rs_rs_id_c FROM dav_resource WHERE rs_uri_txt = ?)");
	}
	else {
		divy_sbuf_append(sql_buf,
					" WHERE (grp.grp_depth_i = 1");
	}
	
	/*
	 * URIが検索条件ならばwhere句を工夫してSQLを1回減らす
	 * (note)
	 *   * content はリストが1つしか無いので通してはならない
	 *   * treelist(top), detailist(top) はgrp_depth_i で取得範囲を絞り済みなので、除外
	 */
	if ((screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPURI||
		 screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPID) &&
		(params->optflg == DIVY_SEARCH_OPT_TREELIST ||
		 params->optflg == DIVY_SEARCH_OPT_DETAILLIST)) {

		divy_sbuf_append(sql_buf,
					" OR (grp.grp_depth_i = ?"
					" AND grp.grp_relative_uri_txt LIKE ? "
					DIVY_DBFUNC_ESCAPE_CLAUSE")) ");
		extra_bind = 1;	/* バインド変数を増やす */
	}
	else {
		divy_sbuf_append(sql_buf,	")");
	}

	/*
	 * グループリーダ自身がこのSEARCHを実行した場合,
	 * 自身が管理するグループだけに限定する.
	 *
	 * ただし、DIVY_SEARCH_GRPIS_USE_GRPCOLURI && content は例外です.
	 * 他人のグループの制約だけを見るパスがあるためです.
	 *
	 */
	if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus) &&
		!(screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPCOLURI &&
		  params->optflg == DIVY_SEARCH_OPT_CONTENT)) {
		divy_sbuf_append(sql_buf,
					" AND grp.grp_owner_usr_id_vc = ?");
		use_ownerinfo = 1;
	}

	divy_sbuf_append(sql_buf,
					" ORDER BY grp.grp_relative_uri_txt");

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), wp);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	idx = 1;
	/* 必要ならばバインド */
	if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPID) {
		stmt->setString(stmt, idx++, screen->grpid);
	}
	else if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPURI) {
		stmt->setString(stmt, idx++, screen->grpuri);
	}
	else if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPCOLURI) {
		stmt->setString(stmt, idx++, screen->grpcol_uri);
	}

	if (extra_bind) {
		char *bind_grpuri = "";
		if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPURI) {
			bind_grpuri = (char *)screen->grpuri;
		}
		/* DIVY_SEARCH_GRPIS_USE_GRPID */
		else {
			bind_grpuri = grpuri;
		}
		stmt->setInt(stmt,    idx++, divy_count_dirs(bind_grpuri) + 1); 
		stmt->setString(stmt, idx++, apr_pstrcat(wp,
				stmt->escWildCard(stmt, bind_grpuri), "/%", NULL));
	}

	if (use_ownerinfo) {
		stmt->setString(stmt, idx++, own_userid);
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* グループの情報を取得 */
	while (rset->next(rset) == DB_TRUE) {

		if (output->list.grp_pr == NULL) {
			output->list.grp_pr = grp = apr_pcalloc(wp, sizeof(divy_rdbo_grp));
		}
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		else if (!params->is_streaming_mode)
#else	/* !DAV_SUPPORT_EXTENDED_SEARCH */
		else
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
		{
			grp->next = apr_pcalloc(wp, sizeof(divy_rdbo_grp));
			grp = grp->next;
		}

		idx = 1;
		/* データ格納(グループ情報のみの各オプション共通) */
		grp->relativeuri = rset->getString(rset, idx++);
		grp->grpid       = rset->getString(rset, idx++);
		grp->name        = rset->getString(rset, idx++);
		grp->depth       = rset->getInt(rset, idx++);
		grp->registdt    = rset->getString(rset, idx++);
		grp->updatedt    = rset->getString(rset, idx++);
		grp->comment     = rset->getString(rset, idx++);
		grp->grpcol_uri  = rset->getString(rset, idx++);

		grp->mailwatch_pr = NULL;
		/* グループメール監視状態を取得する */
		if (IS_FILLED(grp->relativeuri) && gmw_h != NULL) {
			grp->mailwatch_pr = apr_hash_get(gmw_h, grp->relativeuri, APR_HASH_KEY_STRING);
		}

		if (support_grpconstraints || support_groupleader) {
			grp->grp_extstatus = divy_rdbo_parse_extstatus(scratchpool, rset->getString(rset, idx++), EXTSTATUS_TYPE_GRP);
			/* 値が取れなければデフォルト値を使用する(互換性) */
			if (grp->grp_extstatus == NULL) {
				grp->grp_extstatus = divy_rdbo_create_default_extstatus(scratchpool, EXTSTATUS_TYPE_GRP);
			}
		}
		else {
			/* サポートしていなければデフォルト値を使用(互換性) */
			grp->grp_extstatus = divy_rdbo_create_default_extstatus(scratchpool, EXTSTATUS_TYPE_GRP);
		}

		if (support_groupleader) {
			grp->ownerid   = rset->getString(rset, idx++);
			grp->ownername = rset->getString(rset, idx++);
		}

		if (grp_count_h != NULL) {
			grp->childcount = (int)apr_hash_get(grp_count_h, grp->relativeuri, APR_HASH_KEY_STRING);
		}
		else {
			grp->childcount = 0;	/* 初期値 */
		}

		if (support_uploadpolicy) {
			grp->policy = apr_pcalloc(p, sizeof(divy_rdbo_upload_policy));
			grp->policy->active = rset->getInt(rset, idx++);
			grp->policy->rules_fnames = rset->getString(rset, idx++);
			grp->policy->rules_suffix = rset->getString(rset, idx++);
			grp->policy->rules_chartype = rset->getInt(rset, idx++);
			grp->policy->rules_length = rset->getString(rset, idx++);
			grp->policy->rules_matchtype = rset->getInt(rset, idx++);
			grp->policy->hidden = rset->getInt(rset, idx++);
			grp->policy->updatedate = rset->getBigInt(rset, idx++);
		}

		if (support_groupquota) {
#ifdef DIVY_SUPPORT_GROUPQUOTA
			grp->grp_q = apr_pcalloc(p, sizeof(divy_rdbo_grpquota));
			grp->grp_q->usedst = rset->getBigInt(rset, idx++);
			grp->grp_q->maxst  = rset->getBigInt(rset, idx++);
			grp->grp_q->usedres= rset->getBigInt(rset, idx++);
			grp->grp_q->maxres = rset->getBigInt(rset, idx++);
#endif /* DIVY_SUPPORT_GROUPQUOTA */
		}

		grp->next = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		/* ストリーミングモードであればレスポンス出力を実施する */
		if (params->is_streaming_mode) {
			ret = params->output_response(params, output);
			if (ret) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to output response.");
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (rset != NULL) rset->close(rset); rset = NULL;
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
		}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	}
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	/* 1個以上のレスポンス出力が正常終了していたら最後のタグを出力させる */
	if (ret == 0 && output->list.grp_pr != NULL && params->is_streaming_mode) {
		divy_rdbo_search_output last = { 0 };
		last.list.grp_pr = NULL;
		last.response_cnt = output->response_cnt;

		ret = params->output_response(params, &last);
		if (ret) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to output last response.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	divy_db_commit_transaction(ts_ctx);

	return 0;    
}

/**
 * groupinformationsearch のavailableuser の実装.
 *
 */
DIVY_DECLARE(int) divy_rdbo_groupinformation_availableuser(divy_rdbo_search_params *params,
						divy_search_grpis_iscreen *screen,
						divy_rdbo_search_output *output)
{
	request_rec *r          = params->r;
	apr_pool_t *p		= params->r->pool;
	apr_pool_t *wp          = params->wp;
	apr_pool_t *scratchpool = params->scratchpool;
	DbConn *dbconn 		= NULL;
	DbPreparedStmt *stmt 	= NULL;
	DbResultSet *rset 	= NULL;
	divy_db_transaction_ctx *ts_ctx  = NULL;
	int support_extstatus   = divy_support_extenduserstatus(r);
	int support_groupleader = divy_support_groupleader(r);
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	int ret = 0, idx;
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	divy_rdbo_usr *usr = NULL;
	divy_rdbo_grp *grp = NULL, *prev = NULL;
	divy_rdbo_grp *grp_tmp = NULL;
	divy_sbuf *sql_buf = NULL;
	char *group_ownerid = NULL;
	divy_rdbo_mailwatch *mwatch_pr = NULL;
	apr_hash_t *gmw_t = NULL;

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen is EMPTY.");
		return 1;
	}

	output->list.grp_pr  = NULL;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* あらかじめメール監視は取得しておく */
	/* グループIDで来るのは.management/GROUP のケース
	 * です。つまり管理者でなければこの検索は行われないことになります
	 */
	if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPID) {
		/* グループの情報を取得 */
		if (divy_rdbo_get_group_property_by_groupid(r, screen->grpid,
														&grp_tmp, ts_ctx)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get groupmailwatch information.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/* 対象のメール監視一覧を取得する */
		if (grp_tmp && divy_rdbo_get_mailwatch_property(r, grp_tmp->grpcol_uri, NULL, &mwatch_pr, ts_ctx)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get groupmailwatch information.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/* メールウォッチが存在した場合ユーザをキーとしたハッシュテーブルを
		 * 作成する。
		 */
		if (mwatch_pr) {
			gmw_t = apr_hash_make(p);
			while(mwatch_pr) {
				apr_hash_set(gmw_t, mwatch_pr->usrid, APR_HASH_KEY_STRING, mwatch_pr);
				mwatch_pr = mwatch_pr->next;
			}
		}
	}

	/*
	 * SQL文の作成
	 */
	/* SQL文を組み立てる */
	divy_sbuf_create(wp, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
					"SELECT"
					" grp.grp_relative_uri_txt"
					", grp.grp_grp_id_c"
					", grp.grp_depth_i"
					", usr.usr_usr_id_vc"
					", usr.usr_fullname_vc"
					", usr.usr_mailaddr_vc"
					", usr.usr_regist_c"
					", usr.usr_update_c"
					", usr.usr_admin_mod_i");

	/* ユーザ拡張ステータスをサポートしていたら取得 */
	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
					", usr.usr_expiration_bi"
					", usr.usr_extended_status_c");
	}

	/* グループ管理者機能をサポートしているか? */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
					",usr.usr_owner_usr_id_vc"
					",usr.usr_maxcreation_i"
					",usr2.usr_fullname_vc AS ownername"
					",grp.grp_owner_usr_id_vc");
	}

	divy_sbuf_append(sql_buf,
					" FROM divy_grp grp "
					" INNER JOIN divy_grpmem grpm"
					" ON (grp.grp_grp_id_c = grpm.grpm_grp_id_c)"
					" INNER JOIN divy_usr usr"
					" ON (grpm.grpm_usr_id_vc = usr.usr_usr_id_vc)");

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usr usr2"
				" ON (usr.usr_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}

	if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPID) {
		divy_sbuf_append(sql_buf,
					"WHERE grp.grp_grp_id_c = ?");
	}
	else if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPURI) {
		divy_sbuf_append(sql_buf,
					"WHERE grp.grp_relative_uri_txt = ?");
	}
	else if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPCOLURI) {
		divy_sbuf_append(sql_buf,
					"WHERE grp.grp_rs_id_c = (SELECT rs_rs_id_c FROM dav_resource WHERE rs_uri_txt = ?)");
	}

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), wp);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 必要ならばバインド */
	if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPID) {
		stmt->setString(stmt, 1, screen->grpid);
	}
	else if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPURI) {
		stmt->setString(stmt, 1, screen->grpuri);
	}
	else if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPCOLURI) {
		stmt->setString(stmt, 1, screen->grpcol_uri);
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	if (params->is_streaming_mode) {
		/* scratchpool をプールとして設定する */
		divy_db_set_rset_outputpool(rset, scratchpool);
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	prev = NULL;
	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE) {
		/* 領域確保 */
		if (output->list.grp_pr == NULL) {
			/* (note) scratchpool からアロケートしてはならない */
			output->list.grp_pr = grp = apr_pcalloc(wp, sizeof(divy_rdbo_grp));
			output->list.grp_pr->usr_pr = usr = apr_pcalloc(wp, sizeof(divy_rdbo_usr));
		}
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		else if (!params->is_streaming_mode)
#else	/* !DAV_SUPPORT_EXTENDED_SEARCH */
		else
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
		{
			usr->next = apr_pcalloc(wp, sizeof(divy_rdbo_usr));
			usr = usr->next;
		}
				        
		/* データ格納 */
		idx = 1;
		grp->relativeuri= rset->getString(rset, idx++);
		grp->grpid      = rset->getString(rset, idx++);
		grp->depth      = rset->getInt(rset, idx++);
		usr->usrid 	    = rset->getString(rset, idx++);
		usr->fullname 	= rset->getString(rset, idx++);
		usr->mailaddr 	= rset->getString(rset, idx++);
		usr->registdt 	= rset->getString(rset, idx++);
		usr->updatedt 	= rset->getString(rset, idx++);
		usr->adminmode 	= rset->getInt(rset, idx++);
		if (gmw_t) {
			usr->mwatch_pr = apr_hash_get(gmw_t, usr->usrid, APR_HASH_KEY_STRING);
		}
		if (support_extstatus) {
			usr->expiration = (time_t) rset->getBigInt(rset, idx++);
			usr->extstatus  = divy_rdbo_parse_extstatus(scratchpool, rset->getString(rset, idx++), EXTSTATUS_TYPE_USR);

			/* システム実行権限を持つユーザはリスティングしてはならない */
			if (divy_rdbo_has_sysexec_privilege(usr->extstatus)) {
				if (prev == NULL) {
					output->list.grp_pr = grp = NULL;
				}
				else {
					prev->next = NULL;
				}
				continue;
			}
		}

		if (support_groupleader) {
			usr->ownerid         = rset->getString(rset, idx++);
			usr->maxusercreation = rset->getInt(rset, idx++);
			usr->ownername       = rset->getString(rset, idx++);
			group_ownerid        = rset->getString(rset, idx++);

			/* 自身が管理しているユーザかどうか */
			usr->is_otheruser = divy_is_otheruser(r, usr->ownerid);

			/* このグループのグループリーダに任命されているかどうか.
			 * トップグループでのみ任命されていると言える */
			if (IS_FILLED(group_ownerid) && strcmp(usr->usrid, group_ownerid) == 0 && grp->depth == 1) {
				usr->is_appointed_groupleader = 1;	/* 任命されている */
			}
			else {
				usr->is_appointed_groupleader = 0;
			}
		}

		/* LDAPにユーザ名を問い合わせる */
		(void)divy_util_ldap_get_user_property(r, scratchpool, usr->usrid, usr);

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		/* ストリーミングモードであればレスポンス出力を実施する */
		if (params->is_streaming_mode) {
			ret = params->output_response(params, output);
			if (ret) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to output response.");
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (rset != NULL) rset->close(rset); rset = NULL;
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
		}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
		prev = grp;
	}

	/* 使用した領域を解放 (dbconn は何もしてはならない) */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	/* 1個以上のレスポンス出力が正常終了していたら最後のタグを出力させる */
	if (ret == 0 && output->list.grp_pr != NULL && params->is_streaming_mode) {
		divy_rdbo_search_output last = { 0 };
		last.list.grp_pr = NULL;
		last.response_cnt = output->response_cnt;

		ret = params->output_response(params, &last);
		if (ret) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to output last response.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	divy_db_commit_transaction(ts_ctx);

	return 0;    
}

/**
 * groupinformationsearch のavailablesql の実装.
 *
 */
DIVY_DECLARE(int) divy_rdbo_groupinformation_availablesql(divy_rdbo_search_params *params,
						divy_search_grpis_iscreen *screen,
						divy_rdbo_search_output *output)
{
	request_rec *r      = params->r;
	apr_pool_t *p        = params->r->pool;
	apr_pool_t *wp       = params->wp;
	DbConn *dbconn 		 = NULL;
	DbPreparedStmt *stmt = NULL;
	DbResultSet *rset    = NULL;
	divy_db_transaction_ctx *ts_ctx  = NULL;
	char *sqlstr         = NULL;
	char  *sql_where     = NULL;
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	int ret = 0;
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	divy_rdbo_sql *sql = NULL, *prev = NULL;
	char *sqlstmt, *grpid, *sqlid;
	divy_linkedlist_t *in_clause;
	divy_cset_t *uri_set = NULL;
	divy_db_bind_ctx *bindctx = NULL;
	divy_db_bind_ctx_idx *idx;
	const char *in_clause_str = NULL;
	int i;
	apr_hash_t *sql_h       = NULL;
	apr_hash_index_t *hi;

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"screen is EMPTY.");
		return 1;
	}

	output->list.grp_pr  = NULL;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * SQL文の作成
	 */
	if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPID) {
		sql_where = " grp.grp_grp_id_c = ?";
	}
	else if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPURI) {
		sql_where = " grp.grp_relative_uri_txt = ?";
	}

	sqlstr = apr_pstrcat(wp,
					"SELECT"
					" grp.grp_grp_id_c "
					", grp.grp_relative_uri_txt"
					" FROM divy_grp grp"
					" WHERE", sql_where, NULL);

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, sqlstr, wp);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 必要ならばバインド */
	if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPID) {
		stmt->setString(stmt, 1, screen->grpid);
	}
	else if (screen->param_type == DIVY_SEARCH_GRPIS_USE_GRPURI) {
		stmt->setString(stmt, 1, screen->grpuri);
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 1度目の検索はgrpid,uri取得のみ データなしは終了 */
	if (rset->next(rset) == DB_TRUE) {
		/* 領域確保・データ取得 */
		output->list.grp_pr = apr_pcalloc(wp, sizeof(divy_rdbo_grp));
		output->list.grp_pr->grpid       = rset->getString(rset, 1);
		output->list.grp_pr->relativeuri = rset->getString(rset, 2);
	}
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (output->list.grp_pr == NULL) {
		divy_db_commit_transaction(ts_ctx);
		return 0;
	}

	/* 
	 * 親grpの持つsql情報も含めて検索のためsqlを作成しなおし 
	 */
	uri_set = divy_get_uri_set(wp, output->list.grp_pr->relativeuri);

	/* バインドコンテキスト/インデックスを生成 */
	bindctx = divy_db_bindcontext_make(wp, uri_set, -1);
	idx     = divy_db_bindcontext_first(bindctx);

	/* バインド変数文字列の取得 */
	divy_db_bindcontext_get_bindstr(idx, &in_clause_str);

	/* sqlのほかの部分を付加する */
	sqlstmt = apr_pstrcat(wp,
				"SELECT DISTINCT"
				" divy_sql.SQL_ID_C"
				", divy_sql.SQL_LABEL_NAME_VC"
				", divy_sql.SQL_SUB_NAME_VC"
				", divy_sql.SQL_TYPE_I"
				", divy_sql.SQL_ACTIVE_I"
				", divy_dbms.DS_ID_NAME_VC"
				", divy_sql.SQL_REGIST_C"
				", divy_sql.SQL_UPDATE_C"
				", divy_sql.SQL_COMMENT_VC"
				", divy_sqlmem.sqlm_grp_id_c"
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				" FROM ((divy_grp INNER JOIN divy_sqlmem"
				" ON (divy_grp.GRP_GRP_ID_C = divy_sqlmem.SQLM_GRP_ID_C))"
				" INNER JOIN divy_sql"
				" ON (divy_sqlmem.SQLM_SQL_ID_C = divy_sql.SQL_ID_C))"
				" INNER JOIN divy_dbms"
				" ON (divy_sql.SQL_DS_ID_C = divy_dbms.DS_DS_ID_C)"
				" WHERE divy_grp.GRP_RELATIVE_URI_TXT IN (", in_clause_str, ")", 
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
				" FROM divy_grp, divy_sqlmem, divy_sql, divy_dbms"
				" WHERE (divy_grp.GRP_GRP_ID_C = divy_sqlmem.SQLM_GRP_ID_C)"
				" AND (divy_sqlmem.SQLM_SQL_ID_C = divy_sql.SQL_ID_C)"
				" AND (divy_sql.SQL_DS_ID_C = divy_dbms.DS_DS_ID_C)"
				" AND divy_grp.GRP_RELATIVE_URI_TXT IN (", in_clause_str, ")",
#endif
				" ORDER BY divy_sql.SQL_LABEL_NAME_VC",
				NULL);

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, sqlstmt, wp);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	i = 0;
	divy_db_bindcontext_get_list(idx, &in_clause);
	for (; in_clause; in_clause = in_clause->next) {
		stmt->setString(stmt, ++i, in_clause->val);
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	output->list.grp_pr->sql_pr = NULL;	/* 初期化 */

	/* (note) ストリームモードは機能しません。全リストを作り、それから
	 * 	出力動作を行うことにします。これは、直接割り当てられたSQLと
	 * 	継承SQLのバッティングチェックを全てのリスト取得なしに実施することが
	 * 	原理的に出来ないためです。仕方ありません。
	 * 	divy_db_set_rset_outputpool もコールしてはなりません。*/

	sql_h = apr_hash_make(wp);
	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE) {
		sqlid = rset->getString(rset, 1);

		sql = apr_hash_get(sql_h, sqlid, APR_HASH_KEY_STRING);
		/* 既にsql_h に登録されており、かつそれが直接割り当てられた
		 * SQLであればsqlid のSQLをsql_h に再登録する必要はない
		 * (継承SQLの場合だけsql_h への上書き登録を行う) */
		if (sql != NULL && sql->inheritsql == DIVY_SQL_INHERIT_FALSE) {
			continue;
		}

		if (sql == NULL) {
			sql = apr_pcalloc(wp, sizeof(divy_rdbo_sql));
		}
		/* データ格納 */
		sql->sqlid 	= sqlid;
		sql->labelname 	= rset->getString(rset, 2);
		sql->subname 	= rset->getString(rset, 3);
		sql->type 	= rset->getInt(rset, 4);
		sql->active 	= rset->getInt(rset, 5);
		sql->usedbmsname= rset->getString(rset, 6);
		sql->registdt	= rset->getString(rset, 7);
		sql->updatedt 	= rset->getString(rset, 8);
		sql->comment 	= rset->getString(rset, 9);
		grpid           = rset->getString(rset, 10);
		if (strcmp(output->list.grp_pr->grpid, grpid) == 0) {
			/* 継承されていない */
			sql->inheritsql = DIVY_SQL_INHERIT_FALSE;
		}
		else {
			/* 継承されている */
			sql->inheritsql = DIVY_SQL_INHERIT_TRUE;
		}

		apr_hash_set(sql_h, sqlid, APR_HASH_KEY_STRING, sql);
	}

	/* 取得データなしの場合は結果を初期化して終了 */
	if (apr_hash_count(sql_h) == 0) {
		output->list.grp_pr = NULL;
		divy_db_commit_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 0;
	}

	/* SQLハッシュをリストに直す */
	for (hi = apr_hash_first(wp, sql_h); hi; hi = apr_hash_next(hi)) {
		apr_hash_this(hi, (const void **)&sqlid, NULL, (void **)&sql);
		if (output->list.grp_pr->sql_pr == NULL) {
			output->list.grp_pr->sql_pr = prev = sql;
		}
		else {
			prev->next = sql;
			prev = prev->next;
		}
		prev->next = NULL;
	}

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	/* ストリーミングモードであればここでレスポンス出力を実施する */
	if (params->is_streaming_mode && output->list.grp_pr->sql_pr != NULL) {
		divy_rdbo_search_output sql_output = { 0 };
		divy_rdbo_grp dummy = { 0 };

		sql_output.optflg       = output->optflg;
		sql_output.list.grp_pr  = &dummy;
		sql_output.list.grp_pr->grpid       = output->list.grp_pr->grpid;
		sql_output.list.grp_pr->relativeuri = output->list.grp_pr->relativeuri;
		sql_output.list.grp_pr->sql_pr      = output->list.grp_pr->sql_pr;

		for (; sql_output.list.grp_pr->sql_pr;
			sql_output.list.grp_pr->sql_pr = sql_output.list.grp_pr->sql_pr->next) {
			ret = params->output_response(params, &sql_output);
			if (ret) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to output response.");
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (rset != NULL) rset->close(rset); rset = NULL;
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
		}
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	/* 使用した領域を解放 (dbconn は何もしてはならない) */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	/* 1個以上のレスポンス出力が正常終了していたら最後のタグを出力させる */
	if (ret == 0 && output->list.grp_pr != NULL && params->is_streaming_mode) {
		divy_rdbo_search_output last = { 0 };
		last.list.grp_pr = NULL;
		last.response_cnt = output->response_cnt;

		ret = params->output_response(params, &last);
		if (ret) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to output last response.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	divy_db_commit_transaction(ts_ctx);

	return 0;    
}


/**
 * 指定されたrelativeuri(相対URI),depth が示すグループの情報を取得して返却する。
 * (note)
 *	grpuri = /$groupname_p1/$groupname_p2/$groupname
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_hierarchy_group_property(request_rec *r,
						const char *relativeuri,
						int depth,
						divy_rdbo_grp **grp_pr,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	divy_rdbo_grp *n_grp_pr = NULL;
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
	int support_groupquota     = divy_support_groupquota(r);
	int support_uploadpolicy   = divy_support_upload_policy(r);
	divy_sbuf *sql_buf      = NULL;
	/* 条件句を含まないSQL文 */
	int idx;

	*grp_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(relativeuri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"relativeuri is empty.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の生成 */
	divy_sbuf_create(p, &sql_buf, 256);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT "
				"g.grp_grp_id_c,"
				"g.grp_name_vc,"
				"g.grp_relative_uri_txt,"
				"g.grp_depth_i,"
				"g.grp_rs_id_c,"
				"g.grp_comment_vc,"
				"g.grp_regist_c,"
				"g.grp_update_c,"
				"r.rs_uri_txt ");

	/* グループ制約 / グループ管理者機能をサポートしていればカラムを追加 */
	if (support_grpconstraints || support_groupleader) {
		divy_sbuf_append(sql_buf,
				", g.grp_extended_status_c ");
	}
	
	/* グループ管理者機能をサポートしている場合 */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				", g.grp_owner_usr_id_vc "
				", usr2.usr_fullname_vc AS ownername");
	}

	/* アップロードポリシー機能をサポートしている場合 */
	if (support_uploadpolicy) {
		divy_sbuf_append(sql_buf,
				", up.upload_flag_i"
				", up.upload_rules_filename_vc"
				", up.upload_rules_suffix_vc"
				", up.upload_rules_chartype_i"
				", up.upload_rules_length_vc"
				", up.upload_rules_matchtype_i"
				", up.upload_rules_hidden_i"
				", up.upload_update_bi");
	}

	/* グループQuotaをサポート */
	if (support_groupquota) {
		divy_sbuf_append(sql_buf,
				", q.gsqt_used_st_bi "
				", q.gsqt_max_st_bi "
				", q.gsqt_used_res_bi "
				", q.gsqt_max_res_bi ");
	}

	divy_sbuf_append(sql_buf,
				" FROM divy_grp g "
				" INNER JOIN dav_resource r "
				" ON g.grp_rs_id_c = r.rs_rs_id_c ");

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usr usr2"
				" ON (g.grp_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}

	if (support_uploadpolicy) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_uploadpolicy up"
				" ON (g.grp_grp_id_c= up.upload_grp_id_c)");
	}


	if (support_groupquota) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_grpdiskquota q"
				" ON (g.grp_grp_id_c = q.gsqt_grp_id_c)");
	}

	if (depth == 0) {
		divy_sbuf_append(sql_buf,
				" WHERE g.grp_relative_uri_txt = ?");
	}
	else if (depth == 1) {
		divy_sbuf_append(sql_buf,
				" WHERE (g.grp_relative_uri_txt = ?"
				" OR (g.grp_relative_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE
				" AND g.grp_depth_i = ?))");
	}
	else {
		divy_sbuf_append(sql_buf,
				" WHERE (g.grp_relative_uri_txt = ?"
				" OR g.grp_relative_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE")");
	}

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (relativeuri = %s) "
				"Reason: %s", relativeuri, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	stmt->setString(stmt, 1, relativeuri);
	if (depth != 0) {
		stmt->setString(stmt, 2, apr_pstrcat(p,
			stmt->escWildCard(stmt, relativeuri), "/%", NULL));
	}

	if (depth == 1) {
		stmt->setInt(stmt, 3, depth);
	}

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_grp.(relativeuri = %s) "
			"Reason: %s", relativeuri, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {
		if (*grp_pr == NULL) {
			*grp_pr = n_grp_pr =
				apr_pcalloc(p, sizeof(divy_rdbo_grp));
		}
		else {
			n_grp_pr->next =
				apr_pcalloc(p, sizeof(divy_rdbo_grp));
			n_grp_pr = n_grp_pr->next;
		}

		idx = 1;
		n_grp_pr->grpid      = rset->getString(rset, idx++);
		n_grp_pr->name       = rset->getString(rset, idx++);
		n_grp_pr->relativeuri= rset->getString(rset, idx++);
		n_grp_pr->depth      = rset->getInt(rset,    idx++);
		n_grp_pr->rsid       = rset->getString(rset, idx++);
		n_grp_pr->comment    = rset->getString(rset, idx++);
		n_grp_pr->registdt   = rset->getString(rset, idx++);
		n_grp_pr->updatedt   = rset->getString(rset, idx++);
		n_grp_pr->grpcol_uri = rset->getString(rset, idx++);
		if (support_grpconstraints || support_groupleader) {
			n_grp_pr->grp_extstatus = divy_rdbo_parse_extstatus(p, rset->getString(rset, idx++), EXTSTATUS_TYPE_GRP);
			/* 値が取れなければデフォルト値を使用する(互換性) */
			if (n_grp_pr->grp_extstatus == NULL) {
				n_grp_pr->grp_extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_GRP);
			}
		}
		else {
			/* サポートしていなければデフォルト値を使用(互換性) */
			n_grp_pr->grp_extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_GRP);
		}

		if (support_groupleader) {
			n_grp_pr->ownerid   = rset->getString(rset, idx++);
			n_grp_pr->ownername = rset->getString(rset, idx++);
		}

		if (support_uploadpolicy) {
			n_grp_pr->policy = apr_pcalloc(p, sizeof(divy_rdbo_upload_policy));
			n_grp_pr->policy->active = rset->getInt(rset, idx++);
			n_grp_pr->policy->rules_fnames = rset->getString(rset, idx++);
			n_grp_pr->policy->rules_suffix = rset->getString(rset, idx++);
			n_grp_pr->policy->rules_chartype = rset->getInt(rset, idx++);
			n_grp_pr->policy->rules_length = rset->getString(rset, idx++);
			n_grp_pr->policy->rules_matchtype = rset->getInt(rset, idx++);
			n_grp_pr->policy->hidden = rset->getInt(rset, idx++);
			n_grp_pr->policy->updatedate = rset->getBigInt(rset, idx++);
		}

		if (support_groupquota) {
#ifdef DIVY_SUPPORT_GROUPQUOTA
			n_grp_pr->grp_q = apr_pcalloc(p, sizeof(divy_rdbo_grpquota));
			n_grp_pr->grp_q->usedst = rset->getBigInt(rset, idx++);
			n_grp_pr->grp_q->maxst  = rset->getBigInt(rset, idx++);
			n_grp_pr->grp_q->usedres= rset->getBigInt(rset, idx++);
			n_grp_pr->grp_q->maxres = rset->getBigInt(rset, idx++);
#endif /* DIVY_SUPPORT_GROUPQUOTA */
		}

		n_grp_pr->next       = NULL;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * grpname が示すグループのプロパティを検索する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_group_property_by_name(request_rec *r,
					const char *grpname,
					divy_rdbo_grp **grp_pr)
{
	return _get_group_property_by_condition(r, DIVY_WHERECLAUSE_GRPNAME, grpname, grp_pr, NULL);
}

/**
 * グループフォルダに格納されたリソースのURI resource_uri の所属グループの
 * グループプロパティを取得する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_group_property_by_resuri(request_rec *r,
					const char *resource_uri,
					divy_rdbo_grp **grp_pr)
{
	apr_pool_t *p = r->pool;
	char *uri = NULL;

	*grp_pr = NULL;	/* 初期化 */

	/* resource_uri からグループフォルダのURIを算出する */
	if (divy_extract_groupcol_uri(p, dav_divy_get_root_uri(r), resource_uri, &uri)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to parse uri (uri = %s)", resource_uri);
		return 1;
	}
	if (IS_EMPTY(uri)) {
		return 0;	/* グループコレクション以下のURIではなかった */
	}

	return _get_group_property_by_condition(r, DIVY_WHERECLAUSE_RESURI, uri, grp_pr, NULL);
}

/**
 * groupid が示すグループのプロパティを取得する.
 * (note)
 *   トランザクションが継続していれば引き継ぎます
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_group_property_by_groupid(request_rec *r,
					const char *groupid,
					divy_rdbo_grp **grp_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	return _get_group_property_by_condition(r, DIVY_WHERECLAUSE_GRPID, groupid, grp_pr, ts_ctx);
}

/**
 * 指定されたuserid のユーザが所属するグループ一覧のリソースをoffset から limit個まで
 * 取得して返却する。
 * グループURIとグループ名を同期化する場合には、2階層目以降のグループは存在
 * しても表示しません。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_group_resource_by_userid(request_rec *r,
						const char * userid,
						apr_int32_t offset,
						apr_int32_t limit,
						divy_rdbo_resource **rdb_r)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	DbResultSet     *rset    = NULL;
	apr_pool_t *p            = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	int support_groupleader  = divy_support_groupleader(r);
	divy_db_transaction_ctx *ts_ctx = NULL;
	divy_rdbo_resource *n_rdb_r = NULL;
	divy_sbuf *sql_buf       = NULL;
	int cnt                  = 0;
	int found_extra_data     = 0;
	apr_int32_t totallimit   = 0;
	char *grpid, *grpcomment;
	int idx;

	if (IS_EMPTY(userid) || (limit > 0 && offset < 0)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"There are some wrong values.");
		return 1;
	}

	*rdb_r = NULL;	/* 初期化 */

	/* 残りデータがあることを確認する目的で1行余計に取得する */
#if defined(DIVY_DBMS_POSTGRES)	/* postgres */
	totallimit = (limit > 0) ? limit + 1 : -1;
#elif defined(DIVY_DBMS_ORACLE) || defined(DIVY_DBMS_DB2) /* oracle, DB2 */
	if (offset == 0) {
		totallimit = (limit > 0) ? limit + 1 : -1;
	}
	else {
		totallimit = (limit > 0) ? offset*limit + 1 : -1;
	}
#endif

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * 指定されたuserid のユーザが所属しているグループコレクションを
	 * 取得する。
	 * (note)
	 * 	グループコレクション以下のリソースは取得しない
	 */
	/* SQL文の生成 */
	divy_sbuf_create(p, &sql_buf, 900);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT"
				" rs.rs_rs_id_c,"
				" rs.rs_uri_txt,"
				" rs.rs_dispname_vc,"
				" rs.rs_create_bi,"
				" rs.rs_get_cont_lang_vc,"
				" rs.rs_get_cont_len_bi,"
				" rs.rs_get_cont_type_vc,"
				" rs.rs_get_etag_txt,"
				" rs.rs_get_lastmodified_bi,"
				" rs.rs_resourcetype_i,"
				" rs.rs_depth_i,"
				" rs.rs_isversioned_i,"
				" rs.rs_checkin_i,"
				" rs.rs_checkout_i,"
				" rs.rs_physical_path_txt,"
				" rs.rs_creator_usr_id_vc,"
				" rs.rs_lastmodifier_usr_id_vc, "
				" grp.grp_grp_id_c, "
				" grp.grp_comment_vc ");

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				", usr2.usr_fullname_vc AS ownername, "
				"  grp.grp_extended_status_c ");
	}

	divy_sbuf_append(sql_buf,
				" FROM dav_resource rs "
				" INNER JOIN divy_grp grp "
				" ON rs.rs_rs_id_c = grp.grp_rs_id_c ");

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usr usr2"
				" ON (grp.grp_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}

	divy_sbuf_append(sql_buf,
				" WHERE "
				" grp.grp_grp_id_c IN "
				"(SELECT grpm_grp_id_c"
				" FROM divy_grpmem"
				" WHERE grpm_usr_id_vc = ?)");

	if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON) {
		divy_sbuf_append(sql_buf,
				" AND grp.grp_depth_i = 1 "); 
	}

	/* 非アクティブグループはリスティングしない */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" AND (grp.grp_extended_status_c NOT LIKE '___-%' "DIVY_DBFUNC_ESCAPE_CLAUSE
				" OR grp.grp_extended_status_c IS NULL)");
	}

	/* limit 指定が有効な時 */
	if (totallimit > 0) {
		divy_sbuf_append(sql_buf,
#if defined(DIVY_DBMS_POSTGRES)	/* postgres */
				"ORDER BY rs_dispname_vc ASC "
				"OFFSET ? LIMIT ?"
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				"AND ROWNUM <= ? "
				"ORDER BY rs_dispname_vc ASC"
#elif defined(DIVY_DBMS_DB2)	/* DB2 */
				"ORDER BY rs_dispname_vc ASC "
				"FETCH FIRST ? ROWS ONLY"
#endif
				);
	}
	else {
		divy_sbuf_append(sql_buf, "ORDER BY rs_dispname_vc ASC"); 
	}

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) "
			"Reason: %s", userid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	stmt->setString(stmt, 1, userid);
#if defined(DIVY_DBMS_POSTGRES)	/* postgres */
	stmt->setInt(stmt, 2, offset);
	stmt->setInt(stmt, 3, totallimit);
#elif defined(DIVY_DBMS_ORACLE) || defined(DIVY_DBMS_DB2) /* oracle, DB2 */
	stmt->setInt(stmt, 2, totallimit);
#endif

	/* SQLの実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select dav_resource and divy_grpmem."
			"(userid = %s) Reason: %s", userid, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 
	 * 結果の取得 
	 */
	while (rset->next(rset) == DB_TRUE) {
		++cnt;	/* 取得件数をカウント */
#if defined(DIVY_DBMS_ORACLE) || defined(DIVY_DBMS_DB2)	/* oracle, DB2 */
		if (totallimit > 0 && cnt < offset) {
			/* offset 位置にくるまで飛ばす */
			continue;
		}
#endif
		if (totallimit == cnt) {
			found_extra_data = 1;	/* 指定されたlimit件以上のデータが存在した */
			break;	/* 最終行のデータは要らない */
		}

		if (*rdb_r == NULL) {
			*rdb_r = n_rdb_r =
				apr_pcalloc(p, sizeof(divy_rdbo_resource));
		}
		else {
			/* 新規作成 */
			n_rdb_r->next = apr_pcalloc(p, sizeof(divy_rdbo_resource));
			n_rdb_r = n_rdb_r->next;
		}

		/* 値の取り出し */
		idx = 1;
		n_rdb_r->rsid               = rset->getString(rset, idx++);
		n_rdb_r->uri                = rset->getString(rset, idx++);
		n_rdb_r->displayname        = rset->getString(rset, idx++);
		n_rdb_r->creationdate       = (time_t) rset->getBigInt(rset, idx++);
		n_rdb_r->getcontentlanguage = rset->getString(rset, idx++);

		if (IS_EMPTY(n_rdb_r->getcontentlanguage)) {
			n_rdb_r->getcontentlanguage = NULL;
		}
		n_rdb_r->getcontentlength   = rset->getBigInt(rset, idx++);
		n_rdb_r->getcontenttype     = rset->getString(rset, idx++);

		if (IS_EMPTY(n_rdb_r->getcontenttype)) {
			n_rdb_r->getcontenttype = NULL;
		}
		n_rdb_r->getetag             = rset->getString(rset, idx++);
		n_rdb_r->getlastmodified     = (time_t) rset->getBigInt(rset, idx++);
		n_rdb_r->resourcetype        = rset->getInt(rset,   idx++);
		n_rdb_r->depth               = rset->getInt(rset,   idx++);
		n_rdb_r->isversioned         = rset->getInt(rset,   idx++);
		n_rdb_r->checkin             = rset->getInt(rset,   idx++);
		n_rdb_r->checkout            = rset->getInt(rset,   idx++);
		n_rdb_r->physicalpath        = rset->getString(rset,idx++);
		n_rdb_r->creator_userid      = rset->getString(rset,idx++);
		n_rdb_r->lastmodifier_userid = rset->getString(rset,idx++);
		grpid                        = rset->getString(rset,idx++);
		grpcomment                   = rset->getString(rset,idx++);
		if (support_groupleader) {
			n_rdb_r->lastmodifier    = rset->getString(rset,idx++);
		}
		else {
			n_rdb_r->lastmodifier    = NULL;
		}
		n_rdb_r->grp_pr = apr_pcalloc(p, sizeof(divy_rdbo_grp));
		n_rdb_r->grp_pr->grpid = grpid;
		n_rdb_r->grp_pr->comment = grpcomment;
		if (support_groupleader) {
			n_rdb_r->grp_pr->grp_extstatus = divy_rdbo_parse_extstatus(p, rset->getString(rset, idx++), EXTSTATUS_TYPE_GRP);
		}
		else {
			n_rdb_r->grp_pr->grp_extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_GRP);
		}
		n_rdb_r->next                = NULL;	/* sentinel */
	}

	/* 終了処理 */
	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	if (found_extra_data) {
		return DIVY_STCODE_REMAIN_DATA;	/* まだ取得可能なデータが残っていた */
	}
	else {
		return 0;
	}
}

/**
 * 指定されたuserid のユーザがアクセス可能なグループのグループコレクションURI
 * 一覧を取得してset にして返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_grpcoluris_by_userid(request_rec *r,
						const char *userid,
						divy_cset_t **grpcol_uri_set,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	dav_divy_dir_conf *dconf= dav_divy_get_dir_config(r);
	int support_groupleader = divy_support_groupleader(r);
	char *sql, *cond_depth, *cond_active;
	char *sql_fmt           =
				"SELECT rs_uri_txt "
				"FROM dav_resource "
				"WHERE rs_rs_id_c IN"
				" (SELECT g.grp_rs_id_c FROM divy_grp g"
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
				" INNER JOIN divy_grpmem gm"
				" ON g.grp_grp_id_c = gm.grpm_grp_id_c"
				" WHERE "
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
				", divy_grpmem gm"
				" WHERE g.grp_grp_id_c = gm.grpm_grp_id_c"
				" AND "
#endif
				" gm.grpm_usr_id_vc = ?%s%s)";

	*grpcol_uri_set = NULL;	/* 初期化 */

	if (IS_EMPTY(userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"userid is empty.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQLの作成 */
	if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON) {
		/* 1階層のグループだけをアクセス可能 */
		cond_depth = " AND g.grp_depth_i = 1";
	}
	else {
		cond_depth = "";
	}

	/* 非アクティブグループはリスティングしない */
	if (support_groupleader) {
		cond_active =	" AND (g.grp_extended_status_c NOT LIKE '___-%' "DIVY_DBFUNC_ESCAPE_CLAUSE
						" OR g.grp_extended_status_c IS NULL)";
	}
	else {
		cond_active = "";
	}

	sql = apr_psprintf(p, sql_fmt, cond_depth, cond_active);

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (userid = %s) (Reason: %s)",
				userid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}
	stmt->setString(stmt, 1, userid);

	/* SQL文の実行 */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select divy_grpmem. (userid = %s) (Reason: %s)",
				userid, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {
		if (*grpcol_uri_set == NULL) {
			*grpcol_uri_set = divy_cset_make(p);
		}

		/* グループコレクションURIを入れる */
		divy_cset_set(*grpcol_uri_set, rset->getString(rset, 1));
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);

	/* 後片付け */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * grpuri (グループ相対URI)  が示すグループの"トップグループ"を取得する.
 * (note)
 *   grpuri 自身がトップグループであれば、その情報を返します.
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_topgroup_property_by_grpuri(request_rec *r,
					const char *grpuri,
					divy_rdbo_grp **grp_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p = r->pool;
	char *uri     = NULL;
	char *slapos;

	if (IS_EMPTY(grpuri)) {
		return 1;
	}

	*grp_pr = NULL;	/* 初期化 */
 
	/* 末尾のスラッシュを消去 */
	grpuri = dav_divy_remove_endslash(p, grpuri);

	slapos = divy_strchr((char*)&grpuri[1], '/');	/* 次のスラッシュを調べる */
	if (slapos != NULL) {
		int len = slapos - grpuri;
		uri = apr_pcalloc(p, sizeof(char) * (len + 1));
		apr_cpystrn(uri, grpuri, len + 1);
	}
	else {
		uri = (char *) grpuri;
	}

	return divy_rdbo_get_hierarchy_group_property(r, uri, 0, grp_pr, ts_ctx);
}


/**
 * 指定されたgrp_pr が表すグループを新規登録する。
 * グループが持っているグループコレクションも作成する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_group_property(request_rec *r,
						const divy_rdbo_grp *grp_pr,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	apr_pool_t *p            = r->pool;
	int iscommit             = 0;
	divy_rdbo_resource rdb_r = { 0 };
	char *datebuf            = NULL;
	char *grpid		 = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char *root_uri     = dav_divy_get_root_uri(r);
	const char *trash_uri;
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
	int support_box            = divy_support_tfbox(r);
	int support_upload_policy  = divy_support_upload_policy(r);
	divy_sbuf *sql_buf = NULL;
	int idx;

	TRACE(p);

	if (grp_pr == NULL || IS_EMPTY(grp_pr->name) || IS_EMPTY(grp_pr->relativeuri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There are empty value.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/**
	 * グループコレクションの作成
	 */
	if (divy_rdbo_create_group_collection(r, grp_pr->name, &rdb_r, &grpid, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to create group collection. "
				"(groupname = %s)", grp_pr->name);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/**
	 * グループごみ箱の作成
	 */
	if (divy_support_trashfolder(r)) {
		if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON ||
		    dconf->syncgrpuri == DIVY_SYNCGRPURI_MIDDLE) {
			trash_uri = divy_build_group_trash_uri(p, root_uri, grp_pr->name);
		}
		else {
			trash_uri = divy_build_group_trash_uri(p, root_uri, grpid);
		}

		if (divy_rdbo_create_trash_property(r, trash_uri, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to create group trash folder. "
				"(trash_uri = %s)", trash_uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return 1;
		}
	}

	/* SQLの組み立て */
	divy_sbuf_create(p, &sql_buf, 256);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"INSERT INTO divy_grp ("
				"grp_grp_id_c,"
				"grp_name_vc,"
				"grp_relative_uri_txt,"
				"grp_depth_i,"
				"grp_rs_id_c,"
				"grp_comment_vc,"
				"grp_regist_c,"
				"grp_update_c");

	if (support_grpconstraints || support_groupleader) {
		divy_sbuf_append(sql_buf,
				",grp_extended_status_c");
	}

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				",grp_owner_usr_id_vc");
	}

	divy_sbuf_append(sql_buf,
				")"
				" VALUES (?, ?, ?, ?, ?, ?, ?, ?");
	if (support_grpconstraints || support_groupleader) {
		divy_sbuf_append(sql_buf, ",?");
	}
	if (support_groupleader) {
		divy_sbuf_append(sql_buf, ",?");
	}
	divy_sbuf_append(sql_buf, ")");

	/* SQL実行 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (groupname = %s) "
				"Reason: %s", grp_pr->name, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, rdb_r.creationdate, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	idx = 1;
	stmt->setString(stmt, idx++, grpid);
	stmt->setString(stmt, idx++, grp_pr->name);
	stmt->setString(stmt, idx++, grp_pr->relativeuri);
	stmt->setInt(stmt,    idx++, divy_count_dirs(grp_pr->relativeuri));
	stmt->setString(stmt, idx++, rdb_r.rsid);
	stmt->setString(stmt, idx++, REPLACE_EMPTYSTR(grp_pr->comment));
	stmt->setString(stmt, idx++, datebuf);
	stmt->setString(stmt, idx++, datebuf);
	if (support_grpconstraints || support_groupleader) {
		stmt->setString(stmt, idx++, divy_rdbo_extstatus2str(grp_pr->grp_extstatus, EXTSTATUS_TYPE_GRP));
	}
	if (support_groupleader) {
		stmt->setString(stmt, idx++, grp_pr->ownerid);
	}

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_grp.(groupname = %s) "
				"Reason: %s", grp_pr->name, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * groupmailwatch が存在すれば、divy_mailwatch テーブルへの追加
	 */
	if (grp_pr->mailwatch_pr &&
	    divy_rdbo_insert_mailwatch_property(r, grp_pr->mailwatch_pr, ts_ctx)) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to add groupmailwatch element.(groupname = %s)",
			grp_pr->name);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/**
	 * BOXの管理ユーザを自動で登録します。グループが存在する必要があります。
	 */
	if (support_box && divy_rdbo_is_box_group(grp_pr->grp_extstatus)) {
		if (divy_rdbo_insert_grpmem(r, grpid, DIVY_GUEST_ID, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to append systemuser member.(groupname = %s)",
					grp_pr->name);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/**
	 * アップロードポリシーを設定する
	 */
	if (support_upload_policy
					&& divy_rdbo_is_uploadpolicy_group(grp_pr->grp_extstatus)) {

		if (_divy_rdbo_insert_upload_policy(r, grpid, grp_pr, ts_ctx)) {
			ts_ctx->status |= DIVY_TRANS_ABORT;	 
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたgrp_pr が表す内容でグループを更新する。
 * グループ階層間の整合性を取る上で必要なメンバ(uri) が変更された場合には、
 * 対象グループ以下の全てのグループを変更します。
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_group_property(request_rec *r,  
					const divy_rdbo_grp *grp_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	time_t short_time   	= dav_divy_get_now_epoch();
	dav_divy_dir_conf *dconf= dav_divy_get_dir_config(r);
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
	int support_box            = divy_support_tfbox(r);
	int support_uploadpolicy   = divy_support_upload_policy(r);
	char *datebuf           = NULL;
	int update_cnt;
	divy_rdbo_mailwatch mw = { 0 };
	divy_rdbo_usr *usr_pr = NULL;
	int idx;

	TRACE(p);

	/* 必須項目(UPDATEするデータのuri)のチェック */
	if (grp_pr == NULL || IS_EMPTY(grp_pr->grpid) || IS_EMPTY(grp_pr->rsid) ||
	    IS_EMPTY(grp_pr->relativeuri) || IS_EMPTY(grp_pr->grpcol_uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There are empty values.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Could not operation. Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_grp テーブルの更新
	 */
	if (support_grpconstraints || support_groupleader) {
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_grp SET "
				"grp_comment_vc = ?, "
				"grp_update_c = ?, "
				"grp_extended_status_c = ? "
				"WHERE grp_relative_uri_txt = ?", p);
	}
	else {
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_grp SET "
				"grp_comment_vc = ?, "
				"grp_update_c = ? "
				"WHERE grp_relative_uri_txt = ?", p);
	}
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (relativeuri = %s) "
			"Reason: %s", grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	idx = 1;
	/* バインド */
	stmt->setString(stmt, idx++, REPLACE_EMPTYSTR(grp_pr->comment));
	stmt->setString(stmt, idx++, datebuf);
	if (support_grpconstraints || support_groupleader) {
		stmt->setString(stmt, idx++, divy_rdbo_extstatus2str(grp_pr->grp_extstatus, EXTSTATUS_TYPE_GRP));
	}
	stmt->setString(stmt, idx++, grp_pr->relativeuri);

	/* SQL実行　失敗時はエラー */
	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_grp.(relativeuri = %s) "
				"Reason: %s", grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* 更新件数が1件以外はエラー */
	if (update_cnt != 1) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to update divy_grp. data is not found."
				"(relativeuri = %s)", grp_pr->relativeuri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_mailwatch(groupmailwatchプロパティ)の更新
	 * (note)
	 * 	この更新処理はdivy_grpの更新ロジックよりも"後"でなければなりません。
	 * 	divy_mailwatchテーブルに対するphantom readを現在の分離レベルでは
	 * 	回避できないため、divy_grpテーブルの更新ロックを利用することで
	 * 	間接的にこの問題を排除しています。(分離レベルを変えられないのは
	 * 	DBプロバイダによる仮想化がうまくいかなかったことのしわ寄せです。)
	 *
	 * 	また手でデータを更新してしまう際にも発生するphantom readを防止する
	 * 	目的で、mailwatchがあるか無いかに関わらず、まずはgroupmailwatch
	 * 	を削除することにしています。好ましい方法ではありませんが、
	 * 	ここで原理原則に基づいた処理をしても別のコードがダウンすることに
	 * 	なるので、データ不正を正す意味もこめてこのように処理しています。
	 */
	/* グループの絶対URIを取得 */
	mw.uri     = divy_build_m_group_uri(p, dav_divy_get_root_uri(r), grp_pr->relativeuri);
	mw.usrid   = NULL;
	mw.delflag = 0;

	/* groupmailwatch の削除 */
	if (divy_rdbo_remove_mailwatch_property(r, &mw, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to remove groupmailwatch element."
			"(mw_uri = %s)", mw.uri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/* サーバ主体のメール送信の場合 */
	if (dconf->mlserversend == DIVY_MLSERVERSEND_ON) {

		/* 所属ユーザのメール監視フラグ(mailwatch)も削除 */
		mw.uri = grp_pr->grpcol_uri;	/* グループコレクションのURIで削除を指示する */
		if (divy_rdbo_remove_mailwatch_property(r, &mw, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove user-mailwatch group."
					"(mw_uri = %s)", grp_pr->grpcol_uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * メールを追加する条件
	 * * mailwatchがある場合
	 * * BOX属性がついていない
	 */
	if (grp_pr->mailwatch_pr != NULL && !divy_rdbo_is_box_group(grp_pr->grp_extstatus)) {

		/* groupmailwatch の追加 */
		if (divy_rdbo_insert_mailwatch_property(r, grp_pr->mailwatch_pr, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to insert groupmailwatch for updating group."
				"(relativeuri = %s)", grp_pr->relativeuri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		/* サーバ主体のメール送信の場合 */
		if (dconf->mlserversend == DIVY_MLSERVERSEND_ON) {
			/* groupmailwatch を所属ユーザ全員にコピーする */
			if (divy_rdbo_copy_groupmailwatch_to_user(r, grp_pr->grpid, NULL,
						DIVY_CPMWATCH_IGNORE_USER, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to copy mailwatch property "
					"to all users.(grpid = %s)",grp_pr->grpid);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}
	}

	/*
	 * BOXの情報を更新する
	 * 設定された場合
	 * 1. システムユーザ（GUESTユーザ）が自動的に登録されます
	 *
	 * 解除された場合
	 * 1. システムユーザ（GUESTユーザ）が自動的に解除されます
	 * 2. resourcestateで設定されているBOX属性がすべてクリアする
	 * 3. divy_boxに設定されているBOX情報をすべてクリアする
	 *
	 */
	if (support_box) {
		if (divy_rdbo_is_box_group(grp_pr->grp_extstatus)) {
			/* BOX属性が設定されている */
			if (divy_rdbo_box_append_guest_user(r, grp_pr, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to append box system user."
					"(grpid = %s)",grp_pr->grpid);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}
		else {
			/* BOX属性が設定されていない */
			if (divy_rdbo_box_remove_guest_user(r, grp_pr)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove box system user."
					"(grpid = %s)",grp_pr->grpid);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}

			/* グループ内の属性(resourcestate)をすべてクリアする */
			if (divy_rdbo_remove_resourcestate_property(r,
												grp_pr->grpcol_uri,
												DIVY_RSTATE_TYPE_BOX, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove resourcestate property."
					"(grpuri= %s)",grp_pr->grpcol_uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}

			/* BOX情報をすべてクリアする */
			if (divy_rdbo_remove_box_property(r, grp_pr->grpcol_uri, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove divy_box property."
					"(grpuri = %s)",grp_pr->grpcol_uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}
	}

	/*
	 * アップロードポリシーの設定を更新する
	 */
	if (support_uploadpolicy) {
		/* 同じポリシーが設定されていない */
		if (_divy_rdbo_exists_upload_policy(r, grp_pr, ts_ctx) == 0) {

			/* ここに来るのは新規もしくは更新 */
			if (_divy_rdbo_remove_upload_policy(r, grp_pr, ts_ctx)) {
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}

			/* 
			 * ここの実装は確実に失敗です。
			 * ポリシーの更新を行うべきかのチェックは内容が同じかどうかを
			 * DBを検索して調べますが、レコードがあって変更ないとレコードがない
			 * という二つの結果を同じに考えてしまっています。
			 * 本来、このロジックに入ってくるのは、ポリシーが変更されたという
			 * ものが望ましいですが、新規も来てしまいます。
			 * その為、新規と更新の区別をつけるためにルール内容にNULLが
			 * 含まれているかという判断をしてしまっています。
			 *
			 * NULLになるのはデータがない時だけです。設定されると空でありNULL
			 * にはなりません。
			 */
			if (grp_pr->policy->rules_fnames != NULL) {
				if (_divy_rdbo_insert_upload_policy(r, grp_pr->grpid,
							grp_pr, ts_ctx)) {
					if (iscommit) divy_db_rollback_transaction(ts_ctx);
					return 1;
				}
				/* ポリシーが適用されたメールをメンバに送ります */
				if (divy_rdbo_get_group_member(r, &usr_pr, grp_pr, ts_ctx)) {
					if (iscommit) divy_db_rollback_transaction(ts_ctx);
					return 1;
				}
				if (grp_pr->policy->isChanged && usr_pr != NULL) {
					/* メンバーがいたらメールを送る */
					divy_ml_group_policy_apply_notify(r, grp_pr, usr_pr);
				}
			}
		}
	}

	/*
	 * dav_resource テーブルの更新者ユーザID、更新日付を変更する
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"UPDATE dav_resource "
				"SET"
				" rs_lastmodifier_usr_id_vc = ?,"
				" rs_get_lastmodified_bi = ? "
				"WHERE rs_rs_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (relativeuri = %s) "
			"Reason: %s", grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setString	(stmt, 1, divy_get_userid(r));
	stmt->setBigInt(stmt,  2, short_time);
	stmt->setString	(stmt, 3, grp_pr->rsid);

	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update dav_resource. (relativeuri = %s) "
			"Reason: %s", grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 変更を確定 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたgrp_pr が示すグループを削除する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_group_property(request_rec *r,
					const divy_rdbo_grp *grp_pr,
					divy_linkedlist_t **del_filelist,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	char *del_full_grpuri;
	divy_rdbo_grp *grp_pr_list = NULL, *n_grp_pr;
	char *where_like_uri;	/* LIKE 句で使用するワイルドカード付きURI */
	divy_linkedlist_t *filelist, *flist;
	divy_rdbo_resource rdb_r = { 0 };
	char *groupid;
	divy_cset_t *groupid_set = NULL;
	divy_rdbo_mailwatch mw = { 0 };
	int support_confirmreading = divy_support_confirmreading(r);

	TRACE(p);

	*del_filelist = NULL;	/* 初期化 */

	if (grp_pr == NULL ||
	    IS_EMPTY(grp_pr->grpid) || IS_EMPTY(grp_pr->relativeuri)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There are empty value.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* 
	 * 指定されたグループ以下のグループコレクションが持つ
	 * グループ情報を取得する
	 */
	if (divy_rdbo_get_hierarchy_group_property(r, grp_pr->relativeuri,
					DAV_INFINITY, &grp_pr_list, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get hierarchy group information."
			"(relativeuri = %s)", grp_pr->relativeuri);
		return 1;
	}

	/* 削除すべきグループが1つもなかったら以下は実施しても意味がない */
	if (grp_pr_list == NULL) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The specified group not found."
			"Maybe this group was deleted before this operation."
			"(grpid = %s, grpuri = %s)",
			grp_pr->grpid, grp_pr->relativeuri);

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 0;	/* どうせ消えてしまったのだから正常終了にしておく */
	}

	/* 残ってしまったロックの削除用SQLを準備 */
	stmt = dbconn->prepareStatement(dbconn,
				"DELETE FROM dav_lock "
				"WHERE lk_uri_txt = ? "
				"OR lk_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt "
			"(delete group collection). (top_grpid = %s) Reason: %s",
			grp_pr->grpid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 
	 * dav_resourceのデータを削除
	 * 削除件数が1件以外はグループがリソースを持っているので終了
	 */
	for (n_grp_pr = grp_pr_list; n_grp_pr; n_grp_pr = n_grp_pr->next) {

		/*
		 * グループコレクションの削除 以下の削除(dav_resource)
		 */
		rdb_r.uri          = n_grp_pr->grpcol_uri;
		rdb_r.rsid         = n_grp_pr->rsid;
		rdb_r.resourcetype = DIVY_TYPE_COLLECTION;

		filelist = NULL;
		if (divy_rdbo_remove_resource(r, &rdb_r, &filelist, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete group collection."
				"(uri = %s)", rdb_r.uri);

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;	/* 使い回しstmt 用 */
			return 1;
		}

		if (*del_filelist == NULL) {
			*del_filelist = filelist;
		}
		else {
			if (filelist != NULL) {
				for (flist = *del_filelist; flist->next; flist = flist->next);
				flist->next = filelist;
			}
		}

		/*
	 	 * グループコレクションのmailwatch を削除 (divy_mailwatch)
		 */
		mw.uri = n_grp_pr->grpcol_uri;
		mw.usrid = NULL;
		mw.delflag = 0;
		if (divy_rdbo_remove_mailwatch_property(r, &mw, ts_ctx)) {

			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to delete divy_mailwatch for grp collection."
					"(rsuri = %s)", n_grp_pr->grpcol_uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		/* グループコレクションの下に残ってしまっていたロックがあれば
		 * それを削除する */
		stmt->setString(stmt, 1, n_grp_pr->grpcol_uri);
		stmt->setString(stmt, 2, apr_pstrcat(p, stmt->escWildCard(stmt,
					n_grp_pr->grpcol_uri), "/%", NULL));

		/* SQLを実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to delete dav_lock for deleting grp."
					"(rsuri = %s) Reason: %s",
				n_grp_pr->grpcol_uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		/*
		 * グループコレクションに付いていた開封通知を削除
		 */
		if (support_confirmreading) {
			if (divy_rdbo_remove_confirmreading(r, n_grp_pr->grpcol_uri,
								NULL/*ユーザID無視*/, DAV_INFINITY, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to remove confirmreading of group collection. "
						"(uri = %s)", n_grp_pr->grpcol_uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;	/* 使い回しstmt 用 */

				return 1;
			}
		}
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_grpmemの削除
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"DELETE FROM divy_grpmem "
				"WHERE grpm_grp_id_c IN ("
				"SELECT grp_grp_id_c FROM divy_grp "
				"WHERE grp_grp_id_c = ? "
				"OR grp_relative_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE")", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt."
			"(grpid = %s, grpuri = %s) Reason: %s",
			grp_pr->grpid, grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* uri に含まれるワイルドカード文字をエスケープする */
	where_like_uri  = apr_pstrcat(p,
			stmt->escWildCard(stmt, grp_pr->relativeuri), "/%", NULL);
	/* バインド */
	stmt->setString(stmt, 1, grp_pr->grpid);
	stmt->setString(stmt, 2, where_like_uri);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_grpmem."
			"(grpid = %s, grpuri = %s) Reason: %s",
			grp_pr->grpid, grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_sqlmemの削除
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"DELETE FROM divy_sqlmem "
				"WHERE sqlm_grp_id_c IN ("
				"SELECT grp_grp_id_c FROM divy_grp "
				"WHERE grp_grp_id_c = ? "
				"OR grp_relative_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE")", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(grpid = %s, grpuri = %s) Reason: %s",
			grp_pr->grpid, grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, grp_pr->grpid);
	stmt->setString(stmt, 2, where_like_uri);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_sqlmem. "
			"(grpid = %s, grpuri = %s) Reason: %s",
			grp_pr->grpid, grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * groupmailwatch の削除 (divy_mailwatch)
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"DELETE FROM divy_mailwatch "
				"WHERE mw_rs_uri_txt = ? "
				"OR mw_rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(grpid = %s, grpuri = %s) Reason: %s",
			grp_pr->grpid, grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* アップロードポリシー の削除 (divy_uploadpolicy) */
	if (_divy_rdbo_remove_upload_policy(r, grp_pr, ts_ctx)) {
		/* エラーは_divy_rdbo_remove_upload_policyで通知済み */
		return 1;
	}

	/* グループの絶対URIを取得 */
	del_full_grpuri = divy_build_m_group_uri(p, dav_divy_get_root_uri(r),
						grp_pr->relativeuri);

	stmt->setString(stmt, 1, del_full_grpuri);
	/* uri に含まれるワイルドカード文字をエスケープする */
	stmt->setString(stmt, 2, apr_pstrcat(p,
			stmt->escWildCard(stmt, del_full_grpuri), "/%", NULL));

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_mailwatch."
			"(grpid = %s, full_grpuri = %s) Reason: %s",
			grp_pr->grpid, del_full_grpuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DIVY_SUPPORT_GROUPQUOTA
	/* 
	 * groupdiskquotaの削除 (divy_grpdiskquota)
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"DELETE FROM divy_grpdiskquota "
				"WHERE gsqt_grp_id_c = ?"
				DIVY_DBFUNC_ESCAPE_CLAUSE, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt."
			"(grpid = %s, grpuri = %s) Reason: %s",
			grp_pr->grpid, grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, grp_pr->grpid);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_grpdiskquota. "
			"(grpid = %s, grpuri = %s) Reason: %s",
			grp_pr->grpid, grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
#endif /* DIVY_SUPPORT_GROUPQUOTA */

#ifdef DIVY_SUPPORT_PLUGIN
	/* 削除する前に全てのグループIDを取得する */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				"grp_grp_id_c "
				"FROM divy_grp "
				"WHERE grp_grp_id_c = ? "
				"OR grp_relative_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(grpid = %s, grpuri = %s) Reason: %s",
			grp_pr->grpid, grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, grp_pr->grpid);
	stmt->setString(stmt, 2, where_like_uri);

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_grp.(groupid = %s) "
			"Reason: %s", grp_pr->grpid, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {
		groupid = rset->getString(rset, 1);

		if (groupid_set == NULL) {
			groupid_set = divy_cset_make(p);
		}
		/* グループIDを記録する */
		divy_cset_set(groupid_set, groupid);

		/* グループIDの除去を依頼する */
		(void) divy_pi_excludeTarget(r, groupid);
	}

	/* 自動削除対象からgroupid_set のグループ群を除外する.
	 * (note) 失敗しても何もできないのでエラーチェックしない */
	(void) divy_autodel_excludeTargets(r, groupid_set);

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;


#endif	/* DIVY_SUPPORT_PLUGIN */
	/*
	 * divy_grpの削除
	 */
 	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
				"DELETE FROM divy_grp "
				"WHERE grp_grp_id_c = ? "
				"OR grp_relative_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt."
			"(grpid = %s, grpuri = %s) Reason: %s",
			grp_pr->grpid, grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, grp_pr->grpid);
	stmt->setString(stmt, 2, where_like_uri);

	/* sql実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_grp."
			"(grpid = %s, grpuri = %s) Reason: %s",
			grp_pr->grpid, grp_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 反映を確定する */
	divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたsrc_grp_pr が示すグループをdst_grp_pr が示すグループとしてコピーする。
 * depth = DAV_INFINITYの場合、src_grp_pr の下にあったグループも一緒にコピー
 * します。
 *
 */
DIVY_DECLARE(int) divy_rdbo_copy_group_property(request_rec *r, int depth,
					const divy_rdbo_grp *src_grp_pr,
					const divy_rdbo_grp *dst_grp_pr)
{
	DbConn          *dbconn   = NULL;
	DbPreparedStmt  *stmt0    = NULL, *stmt = NULL;
	DbResultSet    	*rset0    = NULL;
	divy_db_transaction_ctx *ts_ctx;
	apr_pool_t *p             = r->pool;
	time_t short_time         = dav_divy_get_now_epoch();
	dav_divy_dir_conf *dconf  = dav_divy_get_dir_config(r);
	char *datebuf             = NULL;
	const char *root_uri      = dav_divy_get_root_uri(r);
	const char *trash_uri;
	int support_trash         = divy_support_trashfolder(r);
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
	char *grpconstraints_str = "";

	int srcdepth, dstdepth;
	char *dstrsuri, *src_mw_uri, *dst_mw_uri, *src_grpuri;
	char *srcgrpid = NULL, *srcrsid, *dstgrpid, *dstrsid;
	char *sql;
	char *ins_grp_sql1 = NULL, *ins_grp_sql2 = NULL;
	char *ins_res_sql1 = NULL, *ins_res_sql2 = NULL;
	int insert_cnt, one_loop;
	char *fmt_ins_grp =
			"INSERT INTO divy_grp ("
			"grp_grp_id_c, "
			"grp_relative_uri_txt, "
			"grp_depth_i,"
			"grp_rs_id_c,"
			"grp_comment_vc,"
			"grp_regist_c, grp_update_c, grp_name_vc %s) "
			"(SELECT "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_GRPID_LEN))", "
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
			DIVY_DBFUNC_SUBSTRING("grp_relative_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")","
			"grp_depth_i + "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_INTEGER)","
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_CHAR(DIVY_DB_RSID_LEN))","
			"grp_comment_vc,"
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN))","
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN))","
			" %s %s "
			"FROM divy_grp "
			"WHERE grp_grp_id_c = ?)";

	char *fmt_ins_res =
			"INSERT INTO dav_resource ("
			"rs_rs_id_c,"
			"rs_uri_txt,"
			"rs_create_bi,"
			"rs_get_cont_lang_vc,"
			"rs_get_cont_len_bi,"
			"rs_get_cont_type_vc,"
			"rs_get_etag_txt,"
			"rs_get_lastmodified_bi,"
			"rs_resourcetype_i,"
			"rs_depth_i,"
			"rs_isversioned_i, rs_checkin_i, rs_checkout_i,"
			"rs_physical_path_txt,"
			"rs_creator_usr_id_vc, rs_lastmodifier_usr_id_vc,"
			"rs_dispname_vc) "
			"(SELECT "
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_CHAR(DIVY_DB_RSID_LEN))","
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))","
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_BIGINT)","
			"rs_get_cont_lang_vc,"
			"rs_get_cont_len_bi,"
			"rs_get_cont_type_vc,"
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_ETAG_LEN))","
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_BIGINT)","
			"rs_resourcetype_i,"
			"rs_depth_i + "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_INTEGER)","
			"0, -1, -1,"
			"rs_physical_path_txt,"
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_USRID_LEN))","
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_USRID_LEN))","
			"%s "
			"FROM dav_resource "
			"WHERE rs_rs_id_c = ?)";

	char *sql_mw_body =
			"INSERT INTO divy_mailwatch ("
			"mw_rs_uri_txt, mw_usr_id_vc,"
			"mw_trigger_get,"
			"mw_trigger_put,"
			"mw_trigger_post,"
			"mw_trigger_delete,"
			"mw_trigger_options,"
			"mw_trigger_propfind,"
			"mw_trigger_proppatch,"
			"mw_trigger_mkcol,"
			"mw_trigger_copy,"
			"mw_trigger_move,"
			"mw_trigger_lock,"
			"mw_trigger_unlock,"
			"mw_trigger_search,"
			"mw_l_del_flag) "
			"(SELECT "
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
			DIVY_DBFUNC_SUBSTRING("mw_rs_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")","
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_USRID_LEN))","
			"mw_trigger_get,"
			"mw_trigger_put,"
			"mw_trigger_post,"
			"mw_trigger_delete,"
			"mw_trigger_options,"
			"mw_trigger_propfind,"
			"mw_trigger_proppatch,"
			"mw_trigger_mkcol,"
			"mw_trigger_copy,"
			"mw_trigger_move,"
			"mw_trigger_lock,"
			"mw_trigger_unlock,"
			"mw_trigger_search,"
			"mw_l_del_flag "
			"FROM divy_mailwatch "
			"WHERE mw_rs_uri_txt = ? ";

	TRACE(p);

        /* 必須項目のチェック */
	if (src_grp_pr == NULL || dst_grp_pr == NULL ||
	    IS_EMPTY(src_grp_pr->grpid) || IS_EMPTY(src_grp_pr->name) ||
	    IS_EMPTY(src_grp_pr->relativeuri) || IS_EMPTY(src_grp_pr->rsid) ||
	    IS_EMPTY(dst_grp_pr->name) || IS_EMPTY(dst_grp_pr->relativeuri)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"There are empty values.");
		return 1;
	}
	/* depthを調べる */
	srcdepth = divy_count_dirs(src_grp_pr->relativeuri);
	dstdepth = divy_count_dirs(dst_grp_pr->relativeuri);

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	if (depth != 0) {
		/* 
		 * divy_grpからコピー元データをselectします
		 */
		stmt0 = dbconn->prepareStatement(dbconn,
			"SELECT grp_grp_id_c, "
			"grp_rs_id_c, "
			"grp_relative_uri_txt "
			"FROM divy_grp "
			"WHERE grp_grp_id_c = ? "
			"OR grp_relative_uri_txt LIKE ? "
			DIVY_DBFUNC_ESCAPE_CLAUSE, p);
		if (stmt0->getCode(stmt0) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. "
					"(grpid = %s) Reason: %s", 
					src_grp_pr->grpid, stmt0->getMsg(stmt0));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt0 != NULL) stmt0->close(stmt0);
			return 1;
		}
		/* バインド */
		stmt0->setString(stmt0, 1, src_grp_pr->grpid);
		/* uri に含まれるワイルドカード文字をエスケープする */
		stmt0->setString(stmt0, 2, apr_pstrcat(p,
			stmt0->escWildCard(stmt0, src_grp_pr->relativeuri), "/%", NULL));

		/* sql実行 失敗時はエラー */
		rset0 = stmt0->executeQuery(stmt0, p);
		if (rset0->getCode(rset0) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbResultSet."
					"(grpid = %s) Reason: %s", 
					src_grp_pr->grpid, rset0->getMsg(rset0));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}
	}
	/* depth = 0 ならば、不要なSQLを発行しない */
	else {
		rset0 = NULL;
		stmt0 = NULL;
	}

	/* グループ制約 / グループ管理者機能をサポートしていればその値もコピーする */
	if (support_grpconstraints || support_groupleader) {
		grpconstraints_str = ", grp_extended_status_c";
	}

	/*
	 * 取得データをループしながら各テーブルにデータ挿入
	 */
	one_loop = (depth == 0) ? 1: 0; /* depth = 0 は１ループ回ればOK */

	while (one_loop || (rset0 && rset0->next(rset0) == DB_TRUE)) {

		if (depth == 0) {
			srcgrpid   = src_grp_pr->grpid;
			srcrsid    = src_grp_pr->rsid;
			src_grpuri = src_grp_pr->relativeuri;
			one_loop = 0;	/* １度回ればOK */
		}
		else {
			srcgrpid   = rset0->getString(rset0, 1);
			srcrsid    = rset0->getString(rset0, 2);
			src_grpuri = rset0->getString(rset0, 3);
		}
		dstgrpid = NULL;
		dstrsid  = NULL;
		sql      = NULL;

		/*
		 * 新規登録データのIDを取得しておく
		 */
		if (divy_rdbo_create_rsgrpid(r, &dstrsid, &dstgrpid, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get rsid or grpid. (src_grpid = %s)",
				src_grp_pr->grpid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;

			return 1;
		}

		/* 
		 * divy_grp のコピー
	 	 */
		/* sql文の作成
		 * グループ名称を変更するかどうかで分岐
		 * ( = グループのトップであるかどうか) */
		if (strcmp(src_grp_pr->grpid, srcgrpid) == 0) {
			if (ins_grp_sql1 == NULL) {
				ins_grp_sql1 = apr_psprintf(p, fmt_ins_grp,
						grpconstraints_str,
						DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_DBIDNAME_LEN)),
						grpconstraints_str);
			}
			sql = ins_grp_sql1;
		}
		else {
			if (ins_grp_sql2 == NULL) {
				ins_grp_sql2 = apr_psprintf(p, fmt_ins_grp,
						grpconstraints_str, "grp_name_vc", grpconstraints_str);
			}
			sql = ins_grp_sql2;
		}

		stmt = dbconn->prepareStatement(dbconn, sql, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (srcgrpid = %s) "
				"Reason: %s", srcgrpid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}

		/* バインド (名前を変更するトップとそれ以外で分岐) */
		stmt->setString(stmt, 1, dstgrpid);
		stmt->setString(stmt, 2, dst_grp_pr->relativeuri);
		stmt->setString(stmt, 3, src_grp_pr->relativeuri);
		stmt->setInt(stmt,    4, dstdepth - srcdepth);
		stmt->setString(stmt, 5, dstrsid);
		stmt->setString(stmt, 6, datebuf);
		stmt->setString(stmt, 7, datebuf);

		if (strcmp(src_grp_pr->grpid, srcgrpid) == 0) {
			stmt->setString(stmt, 8, dst_grp_pr->name);
			stmt->setString(stmt, 9, srcgrpid);
		}
		else {
			stmt->setString(stmt, 8, srcgrpid);
		}

		/* sql実行 失敗時はエラー */
		insert_cnt = stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_grp. "
				"(srcgrpid = %s) Reason: %s",
				srcgrpid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}

		/* 追加できなかった場合(レアケース) */
		if (insert_cnt == 0){
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to insert divy_grp. "
				"data is not found. (src_grpid = %s)", srcgrpid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}

		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* 
	 	 * dav_resource のコピー
	 	 */

		/* dstのグループコレクションの絶対URIを取得 */
		if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON ||
		    dconf->syncgrpuri == DIVY_SYNCGRPURI_MIDDLE) {
			dstrsuri = divy_build_group_uri(p, root_uri, dst_grp_pr->name);
		}
		else {
			dstrsuri = divy_build_group_uri(p, root_uri, dstgrpid);
		}

		/* sql文の作成
		 * 名前を変更するトップとそれ以外で分岐 */

		if (strcmp(src_grp_pr->grpid, srcgrpid) == 0) {
			if (ins_res_sql1 == NULL) {
				ins_res_sql1 = apr_psprintf(p, fmt_ins_res,
					DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_DISPNAME_LEN))); 
			}
			sql = ins_res_sql1;
		}
		else {
			if (ins_res_sql2 == NULL) {
				ins_res_sql2 = apr_psprintf(p, fmt_ins_res, "rs_dispname_vc");
			}
			sql = ins_res_sql2;
		}

		/* sql文の準備 失敗時はエラー */
		stmt = dbconn->prepareStatement(dbconn, sql, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (src_grpid = %s) "
				"Reason: %s", srcgrpid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}

		/* バインド (名前を変更するトップとそれ以外で分岐) */
		stmt->setString(stmt,  1, dstrsid);
		stmt->setString(stmt,  2, dstrsuri);
		stmt->setBigInt(stmt,  3, short_time);
		stmt->setString(stmt,  4, (char *) dav_divy_get_etag_string(p,
						dstrsid, 0, short_time));
		stmt->setBigInt(stmt,  5, short_time);
		stmt->setInt(stmt,     6, dstdepth - srcdepth);
		stmt->setString(stmt,  7, divy_get_userid(r));
		stmt->setString(stmt,  8, divy_get_userid(r));

		if (strcmp(src_grp_pr->grpid, srcgrpid) == 0){
			stmt->setString(stmt,  9, dst_grp_pr->name);
			stmt->setString(stmt, 10, srcrsid);
		}
		else {
			stmt->setString(stmt,  9, srcrsid);
		}

		/* sql実行 失敗時はエラー */
		insert_cnt = stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert dav_resource. "
				"(src_grpid = %s, src_rsid = %s) Reason: %s",
				srcgrpid, srcrsid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}
		/* 追加できなかった場合 */
		if (insert_cnt == 0) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to insert dav_resource. "
				"data is not found. (src_grpid = %s, "
				"src_rsid = %s)", srcgrpid, srcrsid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}

		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* 
		 * divy_grpmem のコピー
	 	 */
		stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO divy_grpmem ("
				"grpm_grp_id_c,"
				"grpm_usr_id_vc) "
				"(SELECT "
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_GRPID_LEN))","
				" grpm_usr_id_vc "
				"FROM divy_grpmem "
				"WHERE grpm_grp_id_c = ?)", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (src_grpid = %s) "
				"Reason: %s", srcgrpid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}

		/* バインド */
		stmt->setString(stmt, 1, dstgrpid);
		stmt->setString(stmt, 2, srcgrpid);

		/* sql実行 失敗時はエラー */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_grpmem. "
				"(src_grpid = %s, dst_grpid = %s) Reason: %s",
				srcgrpid, dstgrpid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}

		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* 
	 	 * divy_sqlmemのコピー
		 */
		stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO divy_sqlmem ("
				"sqlm_sql_id_c,"
				"sqlm_grp_id_c) "
				"(SELECT sqlm_sql_id_c, "
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_GRPID_LEN))" "
				"FROM divy_sqlmem "
				"WHERE sqlm_grp_id_c = ?)", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (src_grpid = %s) "
				"Reason: %s", srcgrpid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}

		/* バインド */
		stmt->setString(stmt, 1, dstgrpid);
		stmt->setString(stmt, 2, srcgrpid);

		/* sql実行 失敗時はエラー */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_sqlmem. "
				"(src_grpid = %s, dst_grpid = %s) Reason: %s",
				srcgrpid, dstgrpid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}

		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/*
		 * divy_mailwatch (mailwatch) のコピー
		 */
		if (divy_rdbo_copy_usermailwatch_to_group(r, src_grp_pr->grpcol_uri, dstrsuri, ts_ctx)) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to copy usermailwatch to group."
					"(srcgrpuri = %s, dstgrpuri = %s)", src_grp_pr->grpcol_uri, dstrsuri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
			if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
			return 1;
		}

		/**
		 * グループごみ箱の作成
		 */
		if (support_trash) {
			if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON ||
			    dconf->syncgrpuri == DIVY_SYNCGRPURI_MIDDLE) {
				trash_uri = divy_build_group_trash_uri(p, root_uri,
								dst_grp_pr->name);
			}
			else {
				trash_uri = divy_build_group_trash_uri(p, root_uri,
								dstgrpid);
			}

			if (divy_rdbo_create_trash_property(r, trash_uri, ts_ctx)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to create group trash folder. "
					"(trash_uri = %s)", trash_uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
				if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
				return 1;
			}
		}
	}

	if (rset0 != NULL) rset0->close(rset0); rset0 = NULL;
	if (stmt0 != NULL) stmt0->close(stmt0); stmt0 = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_mailwatch (groupmailwatch) のコピー
	 */

	/* src グループが持つgroupmailwatch のURIを生成する */

	/* グループの絶対URIを取得 */
	src_mw_uri = divy_build_m_group_uri(p, root_uri, src_grp_pr->relativeuri);
	dst_mw_uri = divy_build_m_group_uri(p, root_uri, dst_grp_pr->relativeuri);

	if (depth != 0) {
		sql = apr_pstrcat(p, sql_mw_body, "OR mw_rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE")", NULL);
	}
	else {
		sql = apr_pstrcat(p, sql_mw_body, ")", NULL);
	}

	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (src_grpid = %s) "
			"Reason: %s", srcgrpid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	stmt->setString(stmt, 1, dst_mw_uri);
	stmt->setString(stmt, 2, src_mw_uri);
	stmt->setString(stmt, 3, divy_get_userid(r));
	stmt->setString(stmt, 4, src_mw_uri);
	if (depth != 0) {
		/* uri に含まれるワイルドカード文字をエスケープする */
		stmt->setString(stmt, 5, apr_pstrcat(p,
				stmt->escWildCard(stmt, src_mw_uri), "/%", NULL));
	}

	/* sql実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_mailwatch. "
			"(src_mw_uri = %s, dst_mw_uri = %s) Reason: %s",
			src_mw_uri, dst_mw_uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 変更を確定 */
	divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたsrc_grp_pr が示すグループをdst_grp_pr が示すグループとして移動する。
 * src_grp_pr の下にあったグループも一緒に移動します。
 *
 */
DIVY_DECLARE(int) divy_rdbo_move_group_property(request_rec *r,
					const divy_rdbo_grp *src_grp_pr,
					const divy_rdbo_grp *dst_grp_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int srcdepth		= 0;
	int dstdepth		= 0;
	int update_cnt		= 0;
	time_t short_time       = dav_divy_get_now_epoch();
	char *datebuf           = NULL;
	const char *root_uri    = dav_divy_get_root_uri(r);
	divy_db_transaction_ctx *ts_ctx = NULL;
	char *src_full_uri, *dst_full_uri;
	dav_divy_dir_conf *dconf= dav_divy_get_dir_config(r);
	int isRename        = 0;	/* リネーム操作を含むかどうか */
	int support_groupleader = divy_support_groupleader(r);
	int update_ownerid  = 0;
	int idx;
	char *ownerid       = NULL;
	divy_sbuf *sql_buf  = NULL;

	TRACE(p);

	if (src_grp_pr == NULL || dst_grp_pr == NULL ||
	    IS_EMPTY(src_grp_pr->grpid) || IS_EMPTY(src_grp_pr->name) ||
	    IS_EMPTY(src_grp_pr->relativeuri) || IS_EMPTY(src_grp_pr->grpcol_uri) ||
	    IS_EMPTY(dst_grp_pr->name) || IS_EMPTY(dst_grp_pr->relativeuri)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There are empty values.");
		return 1;
	}

	/* depthを調べる */
	srcdepth = divy_count_dirs(src_grp_pr->relativeuri);
	dstdepth = divy_count_dirs(dst_grp_pr->relativeuri);

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* リネーム操作かどうか */
	if (strcmp(src_grp_pr->name, dst_grp_pr->name) != 0) {
		isRename = 1;	/* 名称が変わるのでリネームが必要である(階層移動もあるかもしれない) */
	}

	/* DBに書き込む日付の生成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/*
	 * リネームが必要な場合
	 * src のトップ階層の名称変更 (divy_grp, dav_resource の更新)
	 */
	if (isRename) {

		/* (2007/07/04 Wed)
		 *  操作ログにおいて過去に同名のグループが存在していた場合、
		 *  そのグループのログが表示されるという問題があった.
		 *  これを改善する目的で、グループの作成日付以前のログは表示しないことになりました.
		 *  しかし、MOVEでも過去名称の再利用が起き得るため、ここでも更新することとします.
		 *  あまり相応しくありませんが、更新日付は利用できないので、仕方ありません.
		 */

		/* SQL文の準備 失敗時はエラー */
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_grp "
				"SET grp_name_vc = ?,"
				"grp_regist_c = ? "
				"WHERE grp_grp_id_c = ?", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (grpid = %s) "
				"Reason: %s",
				src_grp_pr->grpid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		/* バインド */
		stmt->setString(stmt, 1, dst_grp_pr->name);
		stmt->setString(stmt, 2, datebuf);
		stmt->setString(stmt, 3, src_grp_pr->grpid);

		/* sql実行 失敗時はエラー */
		update_cnt = stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_grp.(grpid = %s) "
				"Reason: %s", 
				src_grp_pr->grpid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		/* 更新対象データが見つからなかった場合 */
		if (update_cnt == 0) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Status is OK, but count of updating divy_grp is 0."
				"(grpid = %s)", src_grp_pr->grpid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/*
		 * dav_resource の表示名を変更
		 */

		/* SQL文の準備 */
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE dav_resource "
				"SET rs_dispname_vc = ? "
				"WHERE rs_uri_txt = ?", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (grpid = %s) "
				"Reason: %s",
				src_grp_pr->grpid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		/* バインド */
		stmt->setString(stmt, 1, dst_grp_pr->name);
		stmt->setString(stmt, 2, src_grp_pr->grpcol_uri);

		/* sql実行 失敗時はエラー */
		update_cnt = stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_grp.(grpid = %s, "
				"srcgrp_uri = %s) Reason: %s", src_grp_pr->grpid,
				src_grp_pr->grpcol_uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		/* 更新対象データが見つからなかった場合 */
		if (update_cnt == 0) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Status is OK, but count of updating dav_resource is 0."
				"(grpid = %s, srcgrp_uri = %s)", src_grp_pr->grpid,
				src_grp_pr->grpcol_uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/*
	 * グループを移動するとき、グループリーダの任命状態(= グループのオーナ情報)が
	 * どのように変わるかどうか判断する
	 */
	if (support_groupleader) {
		
		/* src のトップグループのプロパティを取得する */
		int is_src_top = 0 ,is_dst_top = 0;

		
		/* src はtop？それともサブ？ */
		if (srcdepth <= 1) {
			is_src_top = 1;
		}

		/* dst はtop？それともサブ？ */
		if (dstdepth <= 1) {
			is_dst_top = 1;
		}
		/*
		 * [ ルール ]
		 *   * src がtopか?  -> top への移動 = これはリネームのこと. 何もしない
		 *                   -> sub への移動 = ownerid とtop に合わせる (旧リーダは降格扱い)
		 *
		 *   * src がサブか? -> top への移動 = ownerid をNULL にする.
		 *                   -> sub への移動 = ownerid とtop に合わせる
		 */
		if (!is_dst_top) {
			divy_rdbo_grp *top_grp = NULL;

			if (divy_rdbo_get_topgroup_property_by_grpuri(r, dst_grp_pr->relativeuri, &top_grp, ts_ctx)) {

				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get top-group information. (uri = %s)", dst_grp_pr->relativeuri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
			ownerid = top_grp->ownerid;
			update_ownerid = 1;	/* 更新の必要あり */
		}
		else if (!is_src_top && is_dst_top) {
			ownerid = NULL;	/* NULL 化する */
			update_ownerid = 1;	/* 更新の必要あり */
		}
		else {
			update_ownerid = 0;	/* 更新の必要なし */
		}
	}
	else {
		update_ownerid = 0;	/* 更新の必要なし */
	}

	/*
	 * グループのURI, Depth の変更 (divy_grp の更新)
	 */
	divy_sbuf_create(p, &sql_buf, 128);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
					"UPDATE divy_grp SET "
					"grp_relative_uri_txt = "
					DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
					DIVY_DBFUNC_SUBSTRING("grp_relative_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")", "
					"grp_depth_i = grp_depth_i + "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_INTEGER)", "
					"grp_update_c = "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN))" ");

	if (update_ownerid) {
		divy_sbuf_append(sql_buf,
					",grp_owner_usr_id_vc = ? ");
	}

	divy_sbuf_append(sql_buf,
					" WHERE grp_grp_id_c = ? "
					" OR grp_relative_uri_txt LIKE ? "
					DIVY_DBFUNC_ESCAPE_CLAUSE);
		
	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (grpid = %s) "
			"Reason: %s",
			src_grp_pr->grpid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	idx = 1;
	/* バインド */
	stmt->setString	(stmt, idx++, dst_grp_pr->relativeuri);
	stmt->setString	(stmt, idx++, src_grp_pr->relativeuri);
	stmt->setInt	(stmt, idx++, dstdepth - srcdepth);
	stmt->setString	(stmt, idx++, datebuf);
	if (update_ownerid) {
		stmt->setString	(stmt, idx++, ownerid);
	}
	stmt->setString	(stmt, idx++, src_grp_pr->grpid);
	/* uri に含まれるワイルドカード文字をエスケープする */
	stmt->setString	(stmt, idx++, apr_pstrcat(p,
			stmt->escWildCard(stmt, src_grp_pr->relativeuri), "/%", NULL));

	/* sql実行 失敗時はエラー */
	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_grp.(grpid = %s) "
				"Reason: %s", src_grp_pr->grpid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 更新対象データが見つからなかった場合 */
	if (update_cnt == 0) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to update divy_grp. data is not found."
				"(grpid = %s)", src_grp_pr->grpid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * groupmailwatch の移動 (divy_mailwath テーブルの更新)
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"UPDATE divy_mailwatch "
			"SET mw_rs_uri_txt = "
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
			DIVY_DBFUNC_SUBSTRING("mw_rs_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")" "
			"WHERE mw_rs_uri_txt = ? "
			"OR mw_rs_uri_txt LIKE ? "DIVY_DBFUNC_ESCAPE_CLAUSE, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get stmt for select divy_mailwatch. "
				"(grpid = %s) Reason: %s",
				src_grp_pr->grpid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	/* グループの絶対URIを取得 */
	src_full_uri = divy_build_m_group_uri(p, root_uri, src_grp_pr->relativeuri);
	dst_full_uri = divy_build_m_group_uri(p, root_uri, dst_grp_pr->relativeuri);

	stmt->setString	(stmt, 1, dst_full_uri);
	stmt->setString	(stmt, 2, src_full_uri);
	stmt->setString	(stmt, 3, src_full_uri);
	/* uri に含まれるワイルドカード文字をエスケープする */
	stmt->setString	(stmt, 4, apr_pstrcat(p,
			stmt->escWildCard(stmt, src_full_uri), "/%", NULL));

	/* groupmailwatchのエントリは無いこともあるので、更新行数は気にしない */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_grp.(grpid = %s) "
				"Reason: %s", 
				src_grp_pr->grpid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * グループコレクションのURIにグループ名が使用されている場合で
	 * 名称の変更を検出したら、
	 * dav_resourceのリソース全てとdivy_mailwatchを更新する
	 *
	 * [ URI変更条件 ]
	 * 	* グループURIとグループ名を同期化し (syncgrpuri == ON)
	 * 	* リネームの場合 (isRename == 1)
	 *
	 * (note)
	 * 	syncgrpuri == OFF でsrc_grp_pr->name == dst_grp_pr->name の
	 * 	ケースは救済しません。これは運用中に変えてはならない
	 * 	ディレクティブを変更したことに相当するためです。
	 * 	なおこのケースは事前にチェックされているはず。
	 */
	if ((dconf->syncgrpuri == DIVY_SYNCGRPURI_ON ||
	     dconf->syncgrpuri == DIVY_SYNCGRPURI_MIDDLE) && isRename) {
		char *dst_grpcol_uri = NULL;

		/*
		 * dav_resource のuriを変更
		 */
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE dav_resource "
				"SET rs_uri_txt = "
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
				DIVY_DBFUNC_SUBSTRING("rs_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")" "
				"WHERE rs_uri_txt = ? OR rs_uri_txt LIKE ? "
				DIVY_DBFUNC_ESCAPE_CLAUSE, p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (grpid = %s) "
				"Reason: %s", src_grp_pr->grpid, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		/* グループコレクションのURIを生成 */
		dst_grpcol_uri = divy_build_group_uri(p, root_uri, dst_grp_pr->name);

		stmt->setString(stmt, 1, dst_grpcol_uri);
		stmt->setString(stmt, 2, src_grp_pr->grpcol_uri);
		stmt->setString(stmt, 3, src_grp_pr->grpcol_uri);
		/* uri に含まれるワイルドカード文字をエスケープする */
		stmt->setString(stmt, 4, apr_pstrcat(p,
				stmt->escWildCard(stmt, src_grp_pr->grpcol_uri), "/%", NULL));

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to executeUpdate for dav_resource update. "
				"(src_grpcol_uri = %s, dst_grpcol_uri = %s) "
				"Reason: %s", src_grp_pr->grpcol_uri,
				dst_grpcol_uri, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/*
		 * divy_mailwatch のuriを変更
		 */
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_mailwatch "
				"SET mw_rs_uri_txt = "
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN))" "DIVY_DBOP_CONCAT" "
				DIVY_DBFUNC_SUBSTRING("mw_rs_uri_txt", DIVY_DBFUNC_CHARLEN(DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_URI_LEN)))" + 1")" "
				"WHERE mw_rs_uri_txt = ?", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (src_grpcol_uri = %s) "
				"Reason: %s", src_grp_pr->grpcol_uri, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		stmt->setString(stmt, 1, dst_grpcol_uri);
		stmt->setString(stmt, 2, src_grp_pr->grpcol_uri);
		stmt->setString(stmt, 3, src_grp_pr->grpcol_uri);

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to executeUpdate for divy_mailwatch update. "
				"(src_grpcol_uri = %s, dst_grpcol_uri = %s) Reason: %s", 
				src_grp_pr->grpcol_uri, dst_grpcol_uri,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/*
		 * 状態/属性プロパティのURIを変更する(divy_resourcestate)
		 */
		if (divy_support_extenduserstatus(r)) {

			if (divy_rdbo_move_resourcestate_property(r, src_grp_pr->grpcol_uri,
							dst_grpcol_uri, ts_ctx)) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to update resource state property."
						"(src_grpcol_uri = %s, dst_grpcol_uri = %s)",
						src_grp_pr->grpcol_uri, dst_grpcol_uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}

		/*
		 * 開封通知のURIを変更する (divy_confirmreading)
		 */
		if (divy_support_confirmreading(r)) {

			if (divy_rdbo_move_confirmreading_uri(r, src_grp_pr->grpcol_uri,
							dst_grpcol_uri, ts_ctx)) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to update confirmreading property."
						"(src_grpcol_uri = %s, dst_grpcol_uri = %s)",
						src_grp_pr->grpcol_uri, dst_grpcol_uri);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}
	}

	/* 反映を確定する */
	divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定された相対URI(top_grpuri) が示すトップグループとそのサブグループの
 * オーナをowerid に変更する.
 * (note)
 *   * サブグループのオーナIDも全て更新されます (外部仕様)
 *   * ownerid がNULLの場合、オーナは存在しないことになります (外部仕様)
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_group_ownerinfo(request_rec *r, const char *top_grpuri,
						const char *ownerid, divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;

	TRACE(p);

	if (!divy_support_groupleader(r)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The groupleader function is not supported.");
		return 1;
	}

	/* 必須項目(UPDATEするデータのuri)のチェック */
	if (IS_EMPTY(top_grpuri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The top_grpuri is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Could not operation. Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_grp テーブルの更新
	 */
	stmt = dbconn->prepareStatement(dbconn,
					"UPDATE divy_grp SET "
					"grp_owner_usr_id_vc = ?"
					" WHERE grp_relative_uri_txt = ? "
					" OR grp_relative_uri_txt LIKE ? "
					DIVY_DBFUNC_ESCAPE_CLAUSE, p);
	
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (relativeuri = %s) "
				"Reason: %s", top_grpuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
    stmt->setString(stmt, 1, ownerid);
	stmt->setString(stmt, 2, top_grpuri);
	stmt->setString(stmt, 3, apr_pstrcat(p, stmt->escWildCard(stmt, top_grpuri), "/%", NULL));

	/* SQL実行　失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_grp.(relativeuri = %s) "
				"Reason: %s", top_grpuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたownerid が管理する全てグループでのグループリーダの役割を「解任」する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_dismiss_groupleader(request_rec *r, const char *ownerid,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;

	TRACE(p);

	if (!divy_support_groupleader(r)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The groupleader function is not supported.");
		return 1;
	}

	/* 必須項目(UPDATEするデータのuri)のチェック */
	if (IS_EMPTY(ownerid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The ownerid is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_grp テーブルの更新
	 */
	stmt = dbconn->prepareStatement(dbconn,
					"UPDATE divy_grp SET "
					"grp_owner_usr_id_vc = NULL"
					" WHERE grp_owner_usr_id_vc = ? ", p);
	
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (ownerid = %s) "
				"Reason: %s", ownerid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
    stmt->setString(stmt, 1, ownerid);

	/* SQL実行　失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_grp.(ownerid = %s) "
				"Reason: %s", ownerid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

DIVY_DECLARE(int) divy_rdbo_get_group_member(request_rec *r,
						divy_rdbo_usr **usr_pr, const divy_rdbo_grp *grp_pr,
						divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p 		= r->pool;
	DbConn *dbconn 		= NULL;
	DbPreparedStmt *stmt= NULL;
	DbResultSet *rset	= NULL;
	divy_sbuf *sql_buf  = NULL;
	divy_rdbo_usr *tmp_usr_pr = NULL;
	int iscommit            = 0;
	int idx                 = 0;
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);

	if (!support_grpconstraints && !support_groupleader) {
		return 0;	/* 一覧の取得はできないが正常とする */
	}

	if (grp_pr == NULL || IS_EMPTY(grp_pr->grpid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
									"There are group proprty empty values.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT "
				" usr_usr_id_vc"
				",usr_fullname_vc"
				",usr_mailaddr_vc"
				",usr_extended_status_c"
				",usr_expiration_bi"
				" FROM divy_usr "
				" LEFT JOIN divy_grpmem ON grpm_usr_id_vc = usr_usr_id_vc"
				" WHERE "
				" grpm_grp_id_c = ?");

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (grpid = %s) "
				"Reason: %s", grp_pr->grpid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, grp_pr->grpid);

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_usr(divy_grpmem).(grpid = %s) "
			"Reason: %s", grp_pr->grpid, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {
		if (tmp_usr_pr == NULL) {
			*usr_pr = tmp_usr_pr = apr_pcalloc(p, sizeof(divy_rdbo_usr));
		}
		else {
			tmp_usr_pr->next = apr_pcalloc(p, sizeof(divy_rdbo_usr));
			tmp_usr_pr = tmp_usr_pr->next;
		}

		idx = 1;
		tmp_usr_pr->usrid     = rset->getString(rset, idx++);
		tmp_usr_pr->fullname  = rset->getString(rset, idx++);
		tmp_usr_pr->mailaddr  = rset->getString(rset, idx++);
		tmp_usr_pr->extstatus = divy_rdbo_parse_extstatus(p,
							rset->getString(rset, idx++), EXTSTATUS_TYPE_USR);
		tmp_usr_pr->expiration = (time_t)rset->getBigInt(rset, idx++);
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);

	/* 結果セットとSQL文開放 */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/*------------------------------------------------------------------------------
  Define private functions 
  ----------------------------------------------------------------------------*/
/**
 * type が示すWHERE句、値condition を使ってグループプロパティを検索する.
 */
static int _get_group_property_by_condition(request_rec *r,
					divy_whereclause_type wheretype, const char *condition,
					divy_rdbo_grp **grp_pr, divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p 		= r->pool;
	DbConn *dbconn 		= NULL;
	DbPreparedStmt *stmt= NULL;
	DbResultSet *rset	= NULL;
	int iscommit        = 0;
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
	int support_groupquota     = divy_support_groupquota(r);
	int support_uploadpolicy   = divy_support_upload_policy(r);
	divy_sbuf *sql_buf  = NULL;
	int idx;

	*grp_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(condition)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"condition is EMPTY.");
		return 1;
	}
 
	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT "
				"g.grp_grp_id_c,"
				"g.grp_name_vc,"
				"g.grp_relative_uri_txt,"
				"g.grp_depth_i,"
				"g.grp_rs_id_c,"
				"g.grp_comment_vc,"
				"g.grp_regist_c,"
				"g.grp_update_c,"
				"r.rs_uri_txt ");

	/* グループ制約 / グループリーダをサポートしていればその値も取得する */
	if (support_grpconstraints || support_groupleader) {
		divy_sbuf_append(sql_buf,
				", g.grp_extended_status_c ");
	}
	/* グループQuotaをサポートしていればその値も取得する */
	if (support_groupquota) {
		divy_sbuf_append(sql_buf,
				", q.gsqt_used_st_bi"
				", q.gsqt_max_st_bi "
				", q.gsqt_used_res_bi "
				", q.gsqt_max_res_bi ");
	}

	/* グループ管理者機能をサポートしている場合 */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				", g.grp_owner_usr_id_vc "
				", usr2.usr_fullname_vc AS ownername");
	}
	/* アップロードポリシーをサポートしていればその値も取得する */
	if (support_uploadpolicy) {
		divy_sbuf_append(sql_buf,
				", up.upload_flag_i"
				", up.upload_update_bi"
				", up.upload_rules_filename_vc"
				", up.upload_rules_suffix_vc"
				", up.upload_rules_chartype_i"
				", up.upload_rules_length_vc"
				", up.upload_rules_matchtype_i"
				", up.upload_rules_hidden_i");
	}

	divy_sbuf_append(sql_buf,
				" FROM divy_grp g "
				" INNER JOIN dav_resource r "
				" ON g.grp_rs_id_c = r.rs_rs_id_c ");

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usr usr2"
				" ON (g.grp_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}

	/* アップロードポリシー をサポートしている場合 */
	if (support_uploadpolicy) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_uploadpolicy up"
				" ON (g.grp_grp_id_c = up.upload_grp_id_c) ");
	}

	/* グループQuota をサポートしている場合 */
	if (support_groupquota) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_grpdiskquota q"
				" ON (g.grp_grp_id_c = q.gsqt_grp_id_c) ");
	}

	/* グループ名 */
	if (wheretype == DIVY_WHERECLAUSE_GRPNAME) {
		divy_sbuf_append(sql_buf,
				" WHERE g.grp_name_vc = ?");
	}
	/* グループID */
	else if (wheretype == DIVY_WHERECLAUSE_GRPID) {
		divy_sbuf_append(sql_buf,
				"WHERE g.grp_grp_id_c = ?");
	}
	/* リソースURI */
	else if (wheretype == DIVY_WHERECLAUSE_RESURI) {
		divy_sbuf_append(sql_buf,
				" WHERE r.rs_uri_txt = ?");
	}
	else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
				"Invalid wheretype found (type = %d)", wheretype);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p); 
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, condition);

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet. Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		*grp_pr = apr_pcalloc(p, sizeof(divy_rdbo_grp));

		idx = 1;
		(*grp_pr)->grpid      = rset->getString(rset, idx++);
		(*grp_pr)->name       = rset->getString(rset, idx++);
		(*grp_pr)->relativeuri= rset->getString(rset, idx++);
		(*grp_pr)->depth      = rset->getInt(rset,    idx++);
		(*grp_pr)->rsid       = rset->getString(rset, idx++);
		(*grp_pr)->comment    = rset->getString(rset, idx++);
		(*grp_pr)->registdt   = rset->getString(rset, idx++);
		(*grp_pr)->updatedt   = rset->getString(rset, idx++);
		(*grp_pr)->grpcol_uri = rset->getString(rset, idx++);
		if (support_grpconstraints || support_groupleader) {
			(*grp_pr)->grp_extstatus = divy_rdbo_parse_extstatus(p, rset->getString(rset, idx++), EXTSTATUS_TYPE_GRP);
			/* 値が取れなければデフォルト値を使用する(互換性) */
			if ((*grp_pr)->grp_extstatus == NULL) {
				(*grp_pr)->grp_extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_GRP);
			}
		}
		else {
			/* サポートしていなければデフォルト値を使用(互換性) */
			(*grp_pr)->grp_extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_GRP);
		}

		if (support_groupleader) {
			(*grp_pr)->ownerid   = rset->getString(rset, idx++);
			(*grp_pr)->ownername = rset->getString(rset, idx++);
		}

		if (support_uploadpolicy) {
			(*grp_pr)->policy = apr_pcalloc(p, sizeof(divy_rdbo_upload_policy));

			(*grp_pr)->policy->active = rset->getInt(rset, idx++);
			(*grp_pr)->policy->updatedate = rset->getBigInt(rset, idx++);
			(*grp_pr)->policy->rules_fnames = rset->getString(rset, idx++);
			(*grp_pr)->policy->rules_suffix = rset->getString(rset, idx++);
			(*grp_pr)->policy->rules_chartype = rset->getInt(rset, idx++);
			(*grp_pr)->policy->rules_length = rset->getString(rset, idx++);
			(*grp_pr)->policy->rules_matchtype= rset->getInt(rset, idx++);
			(*grp_pr)->policy->hidden = rset->getInt(rset, idx++);
		}

		if (support_groupquota) {
#ifdef DIVY_SUPPORT_GROUPQUOTA
			divy_rdbo_grpquota *grp_q = apr_pcalloc(p, sizeof(divy_rdbo_grpquota));
			grp_q->usedst = rset->getBigInt(rset, idx++);
			grp_q->maxst  = rset->getBigInt(rset, idx++);
			grp_q->usedres= rset->getBigInt(rset, idx++);
			grp_q->maxres = rset->getBigInt(rset, idx++);
			(*grp_pr)->grp_q = grp_q;
#endif /* DIVY_SUPPORT_GROUPQUOTA */
		}
		(*grp_pr)->next       = NULL;
	}
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	/* 結果セットとSQL文開放 */
	if (rset != NULL)   rset->close(rset); rset = NULL;
	if (stmt != NULL)   stmt->close(stmt); stmt = NULL;

	return 0;
}

/*
 * アップロードポリシーを追加する
 */
static int _divy_rdbo_insert_upload_policy(request_rec *r,
					const char* grpid, const divy_rdbo_grp *grp_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	apr_pool_t		*p		 = r->pool;
	divy_sbuf *sql_buf = NULL;
	int idx = 0; 

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	dbconn = ts_ctx->dbconn;

	/* 更新処理 */
	divy_sbuf_create(p, &sql_buf, 256);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
					"INSERT INTO divy_uploadpolicy ("
					"upload_grp_id_c,"
					"upload_flag_i,"
					"upload_rules_filename_vc,"
					"upload_rules_suffix_vc,"
					"upload_rules_chartype_i,"
					"upload_rules_length_vc,"
					"upload_rules_matchtype_i,"
					"upload_rules_hidden_i,"
					"upload_update_bi"
					") VALUES (?,?,?,?,?,?,?,?,?)"
					);

	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (groupname = %s) "
				"Reason: %s", grp_pr->name, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	idx = 1;
	stmt->setString(stmt, idx++, grpid);
	stmt->setInt(stmt, idx++, grp_pr->policy->active);
	stmt->setString(stmt, idx++, grp_pr->policy->rules_fnames);
	stmt->setString(stmt, idx++, grp_pr->policy->rules_suffix);
	stmt->setInt(stmt, idx++, grp_pr->policy->rules_chartype);
	stmt->setString(stmt, idx++, grp_pr->policy->rules_length);
	stmt->setInt(stmt, idx++, grp_pr->policy->rules_matchtype);
	stmt->setInt(stmt, idx++, grp_pr->policy->hidden);
	stmt->setBigInt(stmt, idx++, grp_pr->policy->updatedate);

	/* SQL実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt,p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/* 
 * 同じアップロードポリシーが存在しているか確認する
 * @ result int		(1: 同じポリシーがあった / 0:同じポリシーがなかったか新規)
 */
static int _divy_rdbo_exists_upload_policy(request_rec *r,
					const divy_rdbo_grp *grp_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	DbResultSet 	*rset    = NULL;
	apr_pool_t		*p		 = r->pool;
	int idx = 0; 

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 0;

	dbconn = ts_ctx->dbconn;

	stmt = dbconn->prepareStatement(dbconn,
					"SELECT"
					" count(*) FROM divy_uploadpolicy"	
					" WHERE upload_grp_id_c = ?"
					" AND upload_flag_i = ?"
					" AND upload_rules_filename_vc = ?"
					" AND upload_rules_suffix_vc = ?"
					" AND upload_rules_chartype_i = ?"
					" AND upload_rules_length_vc = ?"
					" AND upload_rules_matchtype_i = ?"
					" AND upload_rules_hidden_i = ?" , p);

	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)"
				, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 0;
	}
	/* バインド */
	idx = 1;
	stmt->setString(stmt, idx++, grp_pr->grpid);
	stmt->setInt(stmt, idx++, grp_pr->policy->active);
	stmt->setString(stmt, idx++, grp_pr->policy->rules_fnames);
	stmt->setString(stmt, idx++, grp_pr->policy->rules_suffix);
	stmt->setInt(stmt, idx++, grp_pr->policy->rules_chartype);
	stmt->setString(stmt, idx++, grp_pr->policy->rules_length);
	stmt->setInt(stmt, idx++, grp_pr->policy->rules_matchtype);
	stmt->setInt(stmt, idx++, grp_pr->policy->hidden);

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select divy_grp. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 0;
	}

	/* 結果の取得 */
	if (rset->next(rset) == DB_TRUE) {
		if (rset->getInt(rset , 1) == 0) {
			/* 一致しなかったもしくは新規 */
			grp_pr->policy->isChanged = 1;
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 0;
		}
	}

	/* 同じルールが存在した */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	stmt = dbconn->prepareStatement(dbconn,
					"SELECT"
					" upload_flag_i"
					" FROM divy_uploadpolicy"	
					" WHERE upload_grp_id_c = ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)"
				, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 0;
	}
	/* バインド */
	idx = 1;
	stmt->setString(stmt, idx++, grp_pr->grpid);

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select divy_grp. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 0;
	}

	/* 結果の取得 */
	if (rset->next(rset) == DB_TRUE) {
		grp_pr->policy->isChanged =
						(rset->getInt(rset , 1) != grp_pr->policy->active);
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 1;
}

/*
 * 指定されたアップロードポリシーを削除します
 */
static int _divy_rdbo_remove_upload_policy(request_rec *r,
					const divy_rdbo_grp *grp_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p   = r->pool;
	DbConn *dbconn  = NULL;
	DbPreparedStmt *stmt = NULL;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	dbconn = ts_ctx->dbconn;

	stmt = dbconn->prepareStatement(dbconn, 
					"DELETE FROM divy_uploadpolicy"
					" WHERE upload_grp_id_c = ?" ,p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (grp_pr->grpid = %s) "
			"Reason: %s", grp_pr->grpid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, grp_pr->grpid);

	/* SQL実行　失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to delete divy_uploadpolicy.(grp_pr->grpid = %s) "
				"Reason: %s", grp_pr->grpid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

