/**
 * $Id$
 *
 * linkdbsearch、リポジトリ検索関連の関数をまとめたファイル
 *
 * (note)
 *   ここで定義された関数群はtf_rdbo.c にあったものです.
 *   tf_rdbo.c が巨大化し過ぎたため、関連のあるものをまとめました.
 */
#include "httpd.h"
#include "http_protocol.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"

#include "mod_dav_tf.h"
#include "tf_db.h"
#include "util_db.h"
#include "tf_rdbo.h"
#include "util.h"
#include "search.h"
#include "tf_sqlparser.h"
#include "tf_folder.h"
#include "tf_valuecache.h"
#include "tf_linkedlist.h"
#include "tf_rdbo_util.h"
#include "tf_rdbo.h"
#include "tf_rdbo_linkdb.h"

/*------------------------------------------------------------------------------
  Fixed value 
  ----------------------------------------------------------------------------*/
/**
 * リポジトリ検索におけるカラム名の数値表現 (関数 divy_rdbo_linkdbsearch 専用)
 */
enum {
	DIVY_LDBS_COLUMN_URI = 0,		/* rs_uri_txt          */
	DIVY_LDBS_COLUMN_DISPNAME,		/* rs_dispname_vc      */
	DIVY_LDBS_COLUMN_CONTENTLEN,		/* rs_get_cont_len_bi  */
	DIVY_LDBS_COLUMN_CONTENTTYPE,		/* rs_get_cont_type_vc */
	DIVY_LDBS_COLUMN_LASTMIDIFIED,		/* rs_get_lastmodified_bi */
	DIVY_LDBS_COLUMN_RESOURCETYPE,		/* rs_resourcetype_i   */
	DIVY_LDBS_COLUMN_CREATE,		/* rs_create_bi        */
	DIVY_LDBS_COLUMN_CREATOR,		/* rs_creator_vc       */
	DIVY_LDBS_COLUMN_LASTMIDIFIER,		/* rs_lastmodifier_vc  */

	DIVY_LDBS_COLUMN_END			/* sentinel */
};

/*------------------------------------------------------------------------------
  Declare private prototype functions 
  ----------------------------------------------------------------------------*/
static divy_rdbo_resource * _set_linkdbsearch_error(apr_pool_t *p, const char *uri,
					const char *errcode, const char *errmsg);

/*------------------------------------------------------------------------------
  Define public functions 
  ----------------------------------------------------------------------------*/
/**
 * リポジトリDB,顧客DBを検索する(linkdbsearch)
 * 
 */
DIVY_DECLARE(int) divy_rdbo_linkdbsearch(divy_rdbo_search_params *params,
						divy_search_ldbs_iscreen *screen,
						divy_rdbo_search_output *output)
{
	request_rec *r			= params->r;
	apr_pool_t *p			= params->r->pool;
	apr_pool_t *wp			= params->wp;
	DbConn *dbconn_rep	 	= NULL, *dbconn_link = NULL;
	DbPreparedStmt *stmt_rep 	= NULL, *stmt_link   = NULL;
	DbResultSet *rset_rep 	 	= NULL, *rset_link   = NULL;
	char *sqluri, *ldbsql, *ldbsql_replaced = NULL, *dsidname;
	divy_rdbo_sqltype sqltype;
	int i, colcnt, result, ldbs_value_cnt = 0, is_first_row = 0;
	divy_rdbo_resource *resource    = NULL;
	char *data, *colname;
	divy_sql_parser *parser         = NULL;
	apr_int32_t sqlccnt = 0;
	apr_hash_t *repmustcol_h	= NULL;
	divy_sbuf *sbuf                 = NULL;
	char *sqlstr			= NULL;
	divy_search_ldbs_bind *ldbs_value = NULL;
	divy_array_t *bindvals          = NULL;
	apr_int32_t bindvals_len;
	int repos_columns[DIVY_LDBS_COLUMN_END] = { 0 };
	DbDataSource *dbds_link = NULL;	/* 顧客DBのデータソース */

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen is EMPTY.");
		return 1;
	}

	output->list.rdb_r = NULL;	/* 初期化 */

	/* SQL パーサー作成 */
	if (divy_sql_parser_create(p, &parser) != DIVY_SQLP_ST_OK){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to create sql parser.");
		return DIVY_LDB_ST_ERR_IERR;
	}

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn_rep);
	if (dbconn_rep == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbConn.(reposdb)");
		return DIVY_LDB_ST_ERR_IERR;
	}
	/* トランザクション開始 */
	dbconn_rep->startTrans(dbconn_rep, 0);

	/*
	 * (1) divy_sql から URI, SQL文、sqltype、データソースを取得
	 * (note) バインド変数の数のチェックより先に行うのは 
	 * 		DBMSERR 返却時に必要な URI を取得するためです。
	 */

	/* SQL文の作成 */
	sqlstr =	"SELECT "
			" sql.sql_relative_uri_txt,"
			" sql.sql_stmt_txt,"
			" sql.sql_type_i,"
			" dbms.ds_id_name_vc"
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_DB2) /* postgres / db2 */
			" FROM divy_sql sql"
			" INNER JOIN divy_dbms dbms"
			" ON (sql.sql_ds_id_c = dbms.ds_ds_id_c)"
			" WHERE sql.sql_id_c = ?" 
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
			" FROM divy_sql sql, divy_dbms dbms"
			" WHERE sql.sql_ds_id_c = dbms.ds_ds_id_c"
			" AND sql.sql_id_c = ?"
#endif
			;

	switch (params->optflg) {
		/* 通常のlinkdbsearch */
		case DIVY_SEARCH_LDB_NORMAL:
			/* タイプが通常でアクティブのみ */
			sqlstr = apr_pstrcat(p, sqlstr,
					" AND sql.sql_type_i = 0"
					" AND sql.sql_active_i = 1", NULL);
			break;
		/* リポジトリ検索 */
		case DIVY_SEARCH_LDB_REPOSITORY:
			/* タイプがrepositoryでアクティブのみ */
			sqlstr = apr_pstrcat(p, sqlstr,
					" AND sql.sql_type_i = 1"
					" AND sql.sql_active_i = 1", NULL);
			break;
		/* SQLの試し実行 */
		case DIVY_SEARCH_LDB_MLKDB:
			/* タイプがバインド変数以外全て */
			sqlstr = apr_pstrcat(p, sqlstr,
					" AND sql.sql_type_i <> 3", NULL);
			break;
	}

	/* SQL文の準備 失敗時はエラー */
	stmt_rep = dbconn_rep->prepareStatement(dbconn_rep, sqlstr, wp);
	if (stmt_rep->getCode(stmt_rep) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.(reposdb) Reason: %s", 
				stmt_rep->getMsg(stmt_rep));
		dbconn_rep->rollback(dbconn_rep);
		if (rset_rep != NULL) rset_rep->close(rset_rep); rset_rep = NULL;
		if (stmt_rep != NULL) stmt_rep->close(stmt_rep); stmt_rep = NULL;
		return DIVY_LDB_ST_ERR_IERR;
	}

	/* バインド */
	stmt_rep->setString(stmt_rep, 1, screen->sqlid);

	/* SQL実行 失敗時はエラー */
	rset_rep = stmt_rep->executeQuery(stmt_rep, wp);
	if (rset_rep->getCode(rset_rep) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(reposdb) Reason: %s", 
			rset_rep->getMsg(rset_rep));
		dbconn_rep->rollback(dbconn_rep);
		if (rset_rep != NULL) rset_rep->close(rset_rep); rset_rep = NULL;
		if (stmt_rep != NULL) stmt_rep->close(stmt_rep); stmt_rep = NULL;
		return DIVY_LDB_ST_ERR_IERR;
	}

	/* フェッチ(データなしはエラー(badrequest)) */
	if (rset_rep->next(rset_rep) != DB_TRUE) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"SQL statement for linkdb is not found.");
		dbconn_rep->rollback(dbconn_rep);
		if (rset_rep != NULL) rset_rep->close(rset_rep); rset_rep = NULL;
		if (stmt_rep != NULL) stmt_rep->close(stmt_rep); stmt_rep = NULL;
		return DIVY_LDB_ST_ERR_BADREQ;
	}

	/* URI sql文 データソース名 SQLタイプ取得 */
	sqluri 		= rset_rep->getString(rset_rep, 1);
	ldbsql 		= rset_rep->getString(rset_rep, 2);
	sqltype 	= (divy_rdbo_sqltype) rset_rep->getInt(rset_rep, 3);
	dsidname 	= rset_rep->getString(rset_rep, 4);

	/* クローズ */
	if (rset_rep != NULL) rset_rep->close(rset_rep); rset_rep = NULL;
	if (stmt_rep != NULL) stmt_rep->close(stmt_rep); stmt_rep = NULL;

	/* この関数以前のステータスが正常でなかった場合
	 * エラー情報返却のためのURI を取得したので終了 */
	if (screen->retcd != DIVY_LDB_ST_OK) {
		output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);

		dbconn_rep->rollback(dbconn_rep);
		/* 以前のステータスをそのまま返却 */
		return screen->retcd;
	}

	if (IS_EMPTY(dsidname)) {
		output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);
		dbconn_rep->rollback(dbconn_rep);
		return DIVY_LDB_ST_ERR_IERR;
	}

	/*
	 * (2) 文法チェック
	 *	更新系SQL の場合はエラー
	 */
	if (divy_sql_parser_validate_reqsql(parser, sqltype, ldbsql) == DIVY_SQLP_ST_FOUND_UPDATESTMT) {
		output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);

		dbconn_rep->rollback(dbconn_rep);
		return DIVY_LDB_ST_SQL_NOT_SELECT;
	}

	/*
	 * (3) requiredSQL, 名前付きバインド変数をバインド変数パラメータに置き換え、
	 * 	バインドに必要なバインド変数配列を取得する。
	 * (note)
	 * 	バインド変数だけで構成されていても、この関数を通さなければならない。
	 */

	result = divy_sql_parser_replace_reqsql(parser, ldbsql, screen->ldbs_value,
						screen->ldbs_rsvalue, &ldbsql_replaced, &bindvals);
	if (result != DIVY_SQLP_ST_OK) {
		/* パーサ側エラー出力済み・簡単なメッセージです */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"An error occured while replacing required.");

		dbconn_rep->rollback(dbconn_rep);
		if (result == DIVY_SQLP_ST_ERR){
			/* Internal Server Error */
			return DIVY_LDB_ST_ERR_IERR;
		} 
		/* required 文法エラー */
		else {
			output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);
			return DIVY_LDB_ST_MISMATCH_REQVAL;
		}
	}

	/*
	 * (4) divy_sqlcontent をカウントしバインド変数と数があっているかチェック
	 */
	/* SQL文の準備(sqlcontent) 失敗時はエラー */
	stmt_rep = dbconn_rep->prepareStatement(dbconn_rep,
					"SELECT"
					" COUNT(sqlc.sqlc_where_id_i)"
					" FROM divy_sqlcontent sqlc"
					" WHERE sqlc.sqlc_id_c = ?"
					" AND sqlc_where_contype_i = 0", wp);
	if (stmt_rep->getCode(stmt_rep) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.(reposdb) Reason: %s", 
			stmt_rep->getMsg(stmt_rep));
		dbconn_rep->rollback(dbconn_rep);
		if (rset_rep != NULL) rset_rep->close(rset_rep); rset_rep = NULL;
		if (stmt_rep != NULL) stmt_rep->close(stmt_rep); stmt_rep = NULL;
		return DIVY_LDB_ST_ERR_IERR;
	}

	/* バインド */
	stmt_rep->setString(stmt_rep, 1, screen->sqlid);

	/* SQL実行 失敗時はエラー */
	rset_rep = stmt_rep->executeQuery(stmt_rep, wp);
	if (rset_rep->getCode(rset_rep) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(reposdb) Reason: %s", 
			rset_rep->getMsg(rset_rep));
		dbconn_rep->rollback(dbconn_rep);
		if (rset_rep != NULL) rset_rep->close(rset_rep); rset_rep = NULL;
		if (stmt_rep != NULL) stmt_rep->close(stmt_rep); stmt_rep = NULL;
		return DIVY_LDB_ST_ERR_IERR;
	}

	/* フェッチ(データなしはエラー(badrequest)) */
	if (rset_rep->next(rset_rep) != DB_TRUE) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get count of sqlcontent.");
		dbconn_rep->rollback(dbconn_rep);
		if (rset_rep != NULL) rset_rep->close(rset_rep); rset_rep = NULL;
		if (stmt_rep != NULL) stmt_rep->close(stmt_rep); stmt_rep = NULL;
		return DIVY_LDB_ST_ERR_BADREQ;
	}
	sqlccnt = rset_rep->getInt(rset_rep, 1);

	/* バインド変数とsqlcontentのデータ数があわない場合 */
	for (ldbs_value = screen->ldbs_value; ldbs_value; ldbs_value = ldbs_value->next) {
		ldbs_value_cnt++;
	}
	if ((!screen->ldbs_value && sqlccnt != 0) || (ldbs_value_cnt != sqlccnt)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Count of bind value is wrong.");
		dbconn_rep->rollback(dbconn_rep);
		if (rset_rep != NULL) rset_rep->close(rset_rep); rset_rep = NULL;
		if (stmt_rep != NULL) stmt_rep->close(stmt_rep); stmt_rep = NULL;

		output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);
		return DIVY_LDB_ST_MISMATCH_BINDVAL;
	}

	/* もう使用しないdbconn_rep, rset_rep, stmt_rep をクローズ */
	dbconn_rep->commit(dbconn_rep);
	if (rset_rep != NULL) rset_rep->close(rset_rep); rset_rep = NULL;
	if (stmt_rep != NULL) stmt_rep->close(stmt_rep); stmt_rep = NULL;

	/*
	 * (5) dsidname のデータベースに SQL文 ldbsql_replaced をプリペアさせる
	 */

	/* アクティブコネクションの dbconn 取得 see tf_db.h */

	/*
	 * (note) 2006/10/12 Thu
	 *   キャッシュされたコネクションを取得する以下のコードはコメントアウト.
	 *   何故か顧客DBがリポジトリの時にコネクションがclose されないという
	 *   問題が発覚したため(顧客からの指摘).
	 *   ただ、本来であれば顧客DBのコネクションはキャッシュしてはならないので
	 *   そもそも間違い. だからキャッシュしないように直にコネクションを取得
	 *   するようにしました.
	 *   もし、コネクションキャッシュ周りが修正されたらもう一度考えてください.
	 */
	//lookup_cached_DbConn(r, dsidname, &dbconn_link);

	/* 顧客DBのデータソースを取得 */
	dbds_link = lookup_db_provider(r, dsidname);
	if (dbds_link == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get datasource.(dbmsid = %s) ", dsidname);
		return DIVY_LDB_ST_ERR_IERR;
	}
	dbconn_link = dbds_link->getDbConn(dbds_link, r->pool);
	if (dbconn_link == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.(dbmsid = %s) ", dsidname);
		return DIVY_LDB_ST_ERR_IERR;
	}

	/* !! 以降 dbconn_link のclose が必須です(エラーリターンする時も) !! */

	/* linkdb に対するトランザクションを開始 */
	dbconn_link->startTrans(dbconn_link, 0);

	/* SQL文の準備 失敗時はエラー */
	stmt_link = dbconn_link->prepareStatement(dbconn_link, ldbsql_replaced, wp);
	if (stmt_link->getCode(stmt_link) != DB_SUCCESS) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.(DB: %s) ",
			stmt_link->getMsg(stmt_link));

		/* management の場合は DBMS エラー情報を取得 */
		if (params->optflg == DIVY_SEARCH_LDB_MLKDB) {
			output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri,
						stmt_link->getNativeCode(stmt_link),
						stmt_link->getMsg(stmt_link));
		}
		else {
			output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);
		}
		/* エラーコードを取得後クローズ */
		dbconn_link->rollback(dbconn_link);
		if (rset_link != NULL) rset_link->close(rset_link); rset_link = NULL;
		if (stmt_link != NULL) stmt_link->close(stmt_link); stmt_link = NULL;
		if (dbconn_link != NULL) dbconn_link->close(dbconn_link); dbconn_link = NULL;

		/* エラーステータスを返却 */
		return DIVY_LDB_ST_ERR_DBPROVIDER;
	}

	/*
	 * (6) dsidname のデータベースに バインド変数を設定する
	 */
	bindvals_len = divy_array_getlength(bindvals);
	for (i = 0; i < bindvals_len; i++) {
		/* バインド */
		/* ##### FIXME ワイルドカード問題に対応するには? */
		stmt_link->setString(stmt_link, i+1, divy_array_get(bindvals, i));
		//stmt_link->setString(stmt_link, i+1, stmt_link->escWildCard(stmt_link, divy_array_get(bindvals, i)));
		if (stmt_link->getCode(stmt_link) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to bind(setString).(dbmsid = %s) Reason: %s",
				dsidname, stmt_link->getMsg(stmt_link));

			/* management の場合は DBMS エラー情報を取得 */
			if (params->optflg == DIVY_SEARCH_LDB_MLKDB) {
				output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri,
							stmt_link->getNativeCode(stmt_link),
							stmt_link->getMsg(stmt_link));
			}
			else {
				output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);
			}
			/* エラーコードを取得後クローズ */
			dbconn_link->rollback(dbconn_link);
			if (rset_link != NULL) rset_link->close(rset_link); rset_link = NULL;
			if (stmt_link != NULL) stmt_link->close(stmt_link); stmt_link = NULL;
			if (dbconn_link != NULL) dbconn_link->close(dbconn_link); dbconn_link = NULL;

			/* エラーステータスを返却 */
			return DIVY_LDB_ST_ERR_DBPROVIDER;
		}
	}
 
	/*
	 * (7) dsidname のデータベースでSQLを実行する
	 * ここではsql文がselect以外の場合も失敗します(DBプロバイダ側の処理)
	 */
	rset_link = stmt_link->executeQuery(stmt_link, wp);

	/* rset クローズ後も使用するためとっておく */
	result = rset_link->getCode(rset_link);
	if (result != DB_SUCCESS){
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(dbmsid = %s) Reason: %s",
			dsidname, rset_link->getMsg(rset_link));

		/* management の場合 DBMS エラー情報を取得 */
		if (params->optflg == DIVY_SEARCH_LDB_MLKDB){
			output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri,
						rset_link->getNativeCode(rset_link),
						rset_link->getMsg(rset_link));
		}
		else {
			output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);
		}

		/* エラー情報取得後クローズ */
		dbconn_link->rollback(dbconn_link);
		if (rset_link != NULL) rset_link->close(rset_link); rset_link = NULL;
		if (stmt_link != NULL) stmt_link->close(stmt_link); stmt_link = NULL;
		if (dbconn_link != NULL) dbconn_link->close(dbconn_link); dbconn_link = NULL;

		return DIVY_LDB_ST_ERR_DBPROVIDER; 
	}

	/*
	 * (8) 結果の取得
	 */

	/* 列数取得 */
	colcnt = rset_link->getColumnCount(rset_link);

	/* ループに入る前に sqltype 毎のエラーチェック */
	switch (sqltype){
		case DIVY_SQL_TYPE_NORMAL:	/* 通常SQL */
			/* select カラムが少なすぎる */
			if (colcnt < DIVY_NORMALSQL_MIN_SELECT_COLUMN){
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Two or more select elements are required "
					"for the normal linkdbsearch.");

				output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);

				/* クローズ */
				dbconn_link->rollback(dbconn_link);
				if (rset_link != NULL) rset_link->close(rset_link); rset_link = NULL;
				if (stmt_link != NULL) stmt_link->close(stmt_link); stmt_link = NULL;
				if (dbconn_link != NULL) dbconn_link->close(dbconn_link); dbconn_link = NULL;

				return DIVY_LDB_ST_NSQL_TOOFEW_COLUMN;
			}
			/* select カラムが多すぎる */
			else if (colcnt > DIVY_NORMALSQL_MAX_SELECT_COLUMN){
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Too many columns exist in normalsql statement.");

				output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);

				/* クローズ */
				dbconn_link->rollback(dbconn_link);
				if (rset_link != NULL) rset_link->close(rset_link); rset_link = NULL;
				if (stmt_link != NULL) stmt_link->close(stmt_link); stmt_link = NULL;
				if (dbconn_link != NULL) dbconn_link->close(dbconn_link); dbconn_link = NULL;

				return DIVY_LDB_ST_NSQL_TOOMANY_COLUMN;
			}
			break;
		case DIVY_SQL_TYPE_REPOSITORY:
			/* リポジトリSQLチェック */
			if ((result = divy_sql_parser_validate_selectcol(parser, ldbsql_replaced,
							&repmustcol_h)) != DIVY_SQLP_ST_OK) {
				/* クローズ */
				dbconn_link->rollback(dbconn_link);
				if (rset_link != NULL) rset_link->close(rset_link); rset_link = NULL;
				if (stmt_link != NULL) stmt_link->close(stmt_link); stmt_link = NULL;
				if (dbconn_link != NULL) dbconn_link->close(dbconn_link); dbconn_link = NULL;

				/* 通常エラー */
				if (result == DIVY_SQLP_ST_ERR) {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
						"An error occured while validate repository sql.");
					return DIVY_LDB_ST_ERR_BADREQ;
				}
				/* 文法エラー */
				else {
					output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);
					return DIVY_LDB_ST_REPSQL_SYNTAX_ERR;
				}
			}
			break;
		case DIVY_SQL_TYPE_REQUIRED:
			/* select カラムの数が正しくない */
			if (colcnt != DIVY_REQSQL_MAX_SELECT_COLUMN){
				/* クローズ */
				dbconn_link->rollback(dbconn_link);
				if (rset_link != NULL) rset_link->close(rset_link); rset_link = NULL;
				if (stmt_link != NULL) stmt_link->close(stmt_link); stmt_link = NULL;
				if (dbconn_link != NULL) dbconn_link->close(dbconn_link); dbconn_link = NULL;

				output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);
				return DIVY_LDB_ST_REQSQL_WRONG_COLUMN;
			}
			break;
		default:
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Unexpected sqltype (%d)", sqltype);
			break;
	}

	/* バッファの作成 */
	divy_sbuf_create(p, &sbuf, 1000);

	/* フェッチ(データありの間) */
	is_first_row = 1;
	while (rset_link->next(rset_link) == DB_TRUE) {
		/* 領域確保 */
		if (output->list.rdb_r == NULL) {
			output->list.rdb_r = resource = apr_pcalloc(wp, sizeof(divy_rdbo_resource));
			output->list.rdb_r->search_pr = apr_pcalloc(wp, sizeof(divy_rdbo_search));
		} else {
			resource->next 	= apr_pcalloc(wp, sizeof(divy_rdbo_resource));
			resource = resource->next;
			resource->search_pr = apr_pcalloc(wp, sizeof(divy_rdbo_search));
		}
		/* SQLタイプを記録 */
		resource->search_pr->sqltype = sqltype;

		/* 通常SQL / 通常SQLの試し実行の場合 */
		if (params->optflg == DIVY_SEARCH_LDB_NORMAL ||
			(params->optflg == DIVY_SEARCH_LDB_MLKDB && sqltype == DIVY_SQL_TYPE_NORMAL)) {

			/* key エレメントの値を保持する構造体の作成 */
			resource->search_pr->linkdb_key_values =
					apr_pcalloc(wp, sizeof(char *) * (colcnt + 1));
			divy_sbuf_clear(sbuf);	/* バッファのクリア */

			/* 列数分ループし結果格納 */
			for (i = 1; i <= colcnt; i++) {
				/* データの取得 */
				data = rset_link->getString(rset_link, i);

				/* uri・displayname作成 */ 
				if (i == colcnt) {
					/* 最終行の場合はdisplayname設定 */
					resource->uri = apr_pstrdup(wp, divy_sbuf_tostring(sbuf));
					resource->displayname = data;
					resource->search_pr->linkdb_key_values[i-1] = NULL;
				}
				else {
					/* DB キーが1つでも空なら終了 */
					if (IS_EMPTY(data)) {
						output->list.rdb_r = _set_linkdbsearch_error(wp,
								sqluri, NULL, NULL);

						dbconn_link->rollback(dbconn_link);
						if (rset_link != NULL) rset_link->close(rset_link);
						rset_link = NULL;
						if (stmt_link != NULL) stmt_link->close(stmt_link);
						stmt_link = NULL;
						if (dbconn_link != NULL) dbconn_link->close(dbconn_link); dbconn_link = NULL;

						return DIVY_LDB_ST_EMPTY_DBKEY;
					}

					/* key エレメントの値を格納 */
					resource->search_pr->linkdb_key_values[i-1] = data;

					/* uri成分をバッファに追加 */
					divy_sbuf_append(sbuf, data);
				}
			}
		}
		/* リポジトリSQL / リポジトリSQLの試し実行の場合 */
		else if (params->optflg == DIVY_SEARCH_LDB_REPOSITORY ||
			(params->optflg == DIVY_SEARCH_LDB_MLKDB && sqltype == DIVY_SQL_TYPE_REPOSITORY)) {

			/* オプション指定のフラグ構造体初期化 */
			memset(resource->search_pr->reposdb_selected, 0, 3);

			/* 初回のみカラム名とカラム名を表す数値とのマッピング配列を作る
			 * (パフォーマンス向上のため) */
			if (is_first_row) {
				for (i = 1; i <= colcnt; i++) {
					colname = rset_link->getColumnName(rset_link, i);

					/* 比較のため小文字に変換する */
					ap_str_tolower(colname);

					/* 列名により分岐 */
					if(strcmp(colname, "rs_uri_txt")== 0){
						repos_columns[DIVY_LDBS_COLUMN_URI] = i;
					}
					else if(strcmp(colname, "rs_dispname_vc")== 0){
						repos_columns[DIVY_LDBS_COLUMN_DISPNAME] = i;
					}
					else if(strcmp(colname, "rs_get_cont_len_bi") == 0){
						repos_columns[DIVY_LDBS_COLUMN_CONTENTLEN] = i;
					}
					else if(strcmp(colname, "rs_get_cont_type_vc") == 0){
						repos_columns[DIVY_LDBS_COLUMN_CONTENTTYPE] = i;
					}
					else if(strcmp(colname, "rs_get_lastmodified_bi") == 0){
						repos_columns[DIVY_LDBS_COLUMN_LASTMIDIFIED] = i;
					}
					else if(strcmp(colname, "rs_resourcetype_i") == 0){
						repos_columns[DIVY_LDBS_COLUMN_RESOURCETYPE] = i;
					} 
					else if(strcmp(colname, "rs_create_bi") == 0){
						repos_columns[DIVY_LDBS_COLUMN_CREATE] = i;
					}
					else if(strcmp(colname, "rs_creator_vc") == 0){
						repos_columns[DIVY_LDBS_COLUMN_CREATOR] = i;
					}
					else if(strcmp(colname, "rs_lastmodifier_vc") == 0){
						repos_columns[DIVY_LDBS_COLUMN_LASTMIDIFIER] = i;
					} 
					else {
						/* 上記以外が select にあった場合
						 * 現在は無視します。*/
					}
				}
				is_first_row = 0;
			}
   
			/* 列数分ループし結果格納 */
			for (i = 1; i <= colcnt; i++) {

				if (i == repos_columns[DIVY_LDBS_COLUMN_URI]) {
					resource->uri =
						rset_link->getString(rset_link, i);
				}
				else if (i == repos_columns[DIVY_LDBS_COLUMN_DISPNAME]) {
					resource->displayname =
						rset_link->getString(rset_link, i);
				}
				else if (i == repos_columns[DIVY_LDBS_COLUMN_CONTENTLEN]) {
					resource->getcontentlength =
						rset_link->getBigInt(rset_link, i);
				}
				else if (i == repos_columns[DIVY_LDBS_COLUMN_CONTENTTYPE]) {
					resource->getcontenttype =
						rset_link->getString(rset_link, i);
				}
				else if (i == repos_columns[DIVY_LDBS_COLUMN_LASTMIDIFIED]) {
					resource->getlastmodified = (time_t)
						rset_link->getBigInt(rset_link, i);
				}
				else if (i == repos_columns[DIVY_LDBS_COLUMN_RESOURCETYPE]) {
					resource->resourcetype = 
						rset_link->getInt(rset_link, i);
				} 
				/* 以下はオプション */
				else if (i == repos_columns[DIVY_LDBS_COLUMN_CREATE]) {
					resource->creationdate = (time_t)
						rset_link->getBigInt(rset_link, i);
					resource->search_pr->reposdb_selected[DIVY_REPOSDBOPT_CREATIONDATE] = 1;
				}
				else if (i == repos_columns[DIVY_LDBS_COLUMN_CREATOR]) {
					resource->creator =
						rset_link->getString(rset_link, i);
					resource->search_pr->reposdb_selected[DIVY_REPOSDBOPT_CREATOR] = 1;
				}
				else if (i == repos_columns[DIVY_LDBS_COLUMN_LASTMIDIFIER]) {
					resource->lastmodifier =
						rset_link->getString(rset_link, i);
					resource->search_pr->reposdb_selected[DIVY_REPOSDBOPT_LASTMODIFIER] = 1;
				} 
				else {
					/* 上記以外が select にあった場合 
				 	* 現在は無視します。*/
				}
			}
		}
		/* RequiredSQLの試し実行の場合 */
		else if (params->optflg == DIVY_SEARCH_LDB_MLKDB && sqltype == DIVY_SQL_TYPE_REQUIRED) {

			/* key エレメントの値を保持する構造体の作成 */
			resource->search_pr->linkdb_key_values = apr_pcalloc(wp, sizeof(char *) * 2);

			resource->uri = rset_link->getString(rset_link, 1);
			/* DB キーが空なら終了 */
			if (IS_EMPTY(resource->uri)) {
				output->list.rdb_r = _set_linkdbsearch_error(wp, sqluri, NULL, NULL);

				dbconn_link->rollback(dbconn_link);
				if (rset_link != NULL) rset_link->close(rset_link); rset_link = NULL;
				if (stmt_link != NULL) stmt_link->close(stmt_link); stmt_link = NULL;
				if (dbconn_link != NULL) dbconn_link->close(dbconn_link); dbconn_link = NULL;

				return DIVY_LDB_ST_EMPTY_DBKEY;
			}

			resource->displayname = resource->uri;
			resource->search_pr->linkdb_key_values[0] = resource->uri;
			resource->search_pr->linkdb_key_values[1] = NULL; /* sentinel */
		}
	}

	/* クローズ */
	dbconn_link->commit(dbconn_link);
    
	if (rset_link != NULL) rset_link->close(rset_link); rset_link = NULL;
	if (stmt_link != NULL) stmt_link->close(stmt_link); stmt_link = NULL;
	if (dbconn_link != NULL) dbconn_link->close(dbconn_link); dbconn_link = NULL;

	return DIVY_LDB_ST_OK;
}

/*------------------------------------------------------------------------------
  Define private functions 
  ----------------------------------------------------------------------------*/
/**
 * errcode, errmsg のエラーコード及びエラーメッセージをdivy_rdbo_resource に詰めて
 * 返却する. (linkdbsearch 専用)
 *
 * @param p apr_pool_t *
 * @param uri const char *
 * @param errcode const char * エラーコード
 * @param errmsg  const char * エラーメッセージ
 * @return divy_rdbo_resource * エラー情報を格納した構造体へのポインタ
 */
static divy_rdbo_resource * _set_linkdbsearch_error(apr_pool_t *p, const char *uri,
					const char *errcode, const char *errmsg)
{
	divy_rdbo_resource *rdb_r = NULL;

	rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));

	if (IS_FILLED(errcode) || IS_FILLED(errmsg)) {
		/* エラーメッセージとコードを格納 */
		rdb_r->search_pr = apr_pcalloc(p, sizeof(divy_rdbo_search));
		rdb_r->search_pr->dbmserrcode = apr_pstrdup(p, errcode);
		rdb_r->search_pr->dbmserrmsg  = apr_pstrdup(p, errmsg);
	}
	rdb_r->next = NULL;

	return rdb_r;
}

