/**
 * $Id$
 *
 * 特殊フォルダ「トップノード」関連の関数をまとめたファイル
 *
 * (note)
 *   ここで定義された関数群はtf_rdbo.c にあったものです.
 *   tf_rdbo.c が巨大化し過ぎたため、関連のあるものをまとめました.
 */
#include "httpd.h"
#include "http_protocol.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"

#include "mod_dav_tf.h"
#include "tf_db.h"
#include "util_db.h"
#include "tf_rdbo.h"
#include "tf_rdbo_root.h"
#include "util.h"
#include "tf_folder.h"
#include "tf_linkedlist.h"
#include "tf_rdbo_util.h"

/*------------------------------------------------------------------------------
  Declare private functions 
  ----------------------------------------------------------------------------*/
static int _roottree_sql(request_rec *r, divy_rdbo_resource **rdb_r, int isSqlstmt);

/*------------------------------------------------------------------------------
  Define public functions 
  ----------------------------------------------------------------------------*/
/**
 * rootreesearch (alllist) のDB検索.
 * この関数では、ユーザフォルダの情報だけを取得しています.
 * その他の特殊フォルダはtf_folder.c が保持する内部データだけで表現します.
 *
 */
DIVY_DECLARE(int) divy_rdbo_roottree_alllist(request_rec *r, divy_rdbo_resource **rdb_r)
{
	apr_pool_t *p        = r->pool;
	DbConn *dbconn       = NULL;
	DbPreparedStmt *stmt = NULL;
	DbResultSet *rset    = NULL;
	divy_db_transaction_ctx *ts_ctx  = NULL;

	*rdb_r = NULL;	/* 初期化 */
    
	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				" rs.rs_uri_txt"
				" from dav_resource rs"
				" WHERE rs.rs_rs_id_c ="
				" (SELECT usr.usr_rs_id_c"
				" FROM divy_usr usr"
				" WHERE usr.usr_usr_id_vc = ?)", p);

	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, divy_get_userid(r));

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* データありの場合は取得 */
	if (rset->next(rset) == DB_TRUE){
		if (*rdb_r == NULL) {
			*rdb_r 	= apr_pcalloc(p, sizeof(divy_rdbo_resource));
		}
		(*rdb_r)->uri = rset->getString(rset, 1);
	} 

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * rootreesearch (sqlstmt) のDB検索.
 *
 */
DIVY_DECLARE(int) divy_rdbo_roottree_sql(request_rec *r, divy_rdbo_resource **rdb_r)
{
	return _roottree_sql(r, rdb_r, 1/* sqlstmt */);
}


/**
 * rootreesearch (reposdbsqlstmt) のDB検索.
 *
 */
DIVY_DECLARE(int) divy_rdbo_roottree_reposdbsql(request_rec *r, divy_rdbo_resource **rdb_r)
{
	return _roottree_sql(r, rdb_r, 0/* reposdbsqlstmt */);
}

/**
 * rootreesearch (execsqlstmt) のDB検索.
 *
 */
DIVY_DECLARE(int) divy_rdbo_roottree_execsql(request_rec *r, divy_rdbo_resource **rdb_r)
{
	apr_pool_t *p        = r->pool;
	DbConn *dbconn       = NULL;
	DbPreparedStmt *stmt = NULL;
	DbResultSet *rset    = NULL;
	divy_rdbo_sql *sql   = NULL;
	divy_db_transaction_ctx *ts_ctx  = NULL;

	*rdb_r = NULL;	/* 初期化 */
    
	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT"
				" sql.sql_id_c,"
				" sql.sql_relative_uri_txt,"
				" sql.sql_label_name_vc "
				" FROM divy_sql sql"
				" WHERE sql.sql_type_i <> ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setInt(stmt, 1, DIVY_SQL_TYPE_BIND);

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select \"divy_sql\". (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {
		if (*rdb_r == NULL) {
			*rdb_r 	= apr_pcalloc (p, sizeof(divy_rdbo_resource));
			(*rdb_r)->sql_pr = sql = apr_pcalloc(p, sizeof(divy_rdbo_sql));
		}
		else {
			sql->next = apr_pcalloc(p, sizeof(divy_rdbo_sql));
			sql = sql->next;
		}

		/* データ格納 */ 
		sql->sqlid       = rset->getString(rset, 1);
		sql->relativeuri = rset->getString(rset, 2);
		sql->labelname   = rset->getString(rset, 3);

		sql->next = NULL;
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * rootreesearch (grouplist) のDB検索.
 *
 */
DIVY_DECLARE(int) divy_rdbo_roottree_grouplist(request_rec *r, divy_rdbo_resource **rdb_r)
{
	apr_pool_t *p        = r->pool;
	DbConn *dbconn       = NULL;
	DbPreparedStmt *stmt = NULL;
	DbResultSet *rset    = NULL;
	divy_db_transaction_ctx *ts_ctx  = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	int support_groupleader = divy_support_groupleader(r);
#ifdef DIVY_SUPPORT_GROUPQUOTA
	int support_groupquota = divy_support_groupquota(r);
#endif /* DIVY_SUPPORT_GROUPQUOTA */
	divy_rdbo_grp *grp = NULL;
	divy_sbuf *sql_buf = NULL;
	int idx = 0;

	*rdb_r = NULL;	/* 初期化 */
    
	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * SQL文の作成
	 */
	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
					"SELECT"
					" g.grp_grp_id_c"
					",g.grp_name_vc"
					",g.grp_comment_vc"
					",r.rs_uri_txt");
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
					",g.grp_extended_status_c"
					",usr2.usr_fullname_vc AS ownername");
	}

#ifdef DIVY_SUPPORT_GROUPQUOTA
	if (support_groupquota) {
		divy_sbuf_append(sql_buf,
					",q.gsqt_used_st_bi"
					",q.gsqt_max_st_bi"
					",q.gsqt_used_res_bi"
					",q.gsqt_max_res_bi"
					);
	}
#endif /* DIVY_SUPPORT_GROUPQUOTA */

	divy_sbuf_append(sql_buf,
					" FROM divy_grp g"
					" INNER JOIN dav_resource r"
					" ON g.grp_rs_id_c = r.rs_rs_id_c");
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
					" LEFT JOIN divy_usr usr2"
					" ON (g.grp_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}

#ifdef DIVY_SUPPORT_GROUPQUOTA
	if (support_groupquota) {
		divy_sbuf_append(sql_buf,
					" LEFT JOIN divy_grpdiskquota q"
					" ON (g.grp_rs_id_c = q.gsqt_grp_id_c)");
	}
#endif /* DIVY_SUPPORT_GROUPQUOTA */

	divy_sbuf_append(sql_buf,
					" WHERE"
					" g.grp_grp_id_c IN "
					"(SELECT grpm.grpm_grp_id_c"
					" FROM divy_grpmem grpm"
					" WHERE grpm.grpm_usr_id_vc = ?)");

	/* グループURIと名称を同一にする場合、
	 * 深さ1 のグループに割り当てられているSQLだけを取得する */
	if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON) {
		divy_sbuf_append(sql_buf,
					" AND g.grp_depth_i = 1");
	}

	/* 非アクティブグループはリスティングしない */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
					" AND (g.grp_extended_status_c NOT LIKE '___-%' "DIVY_DBFUNC_ESCAPE_CLAUSE
					" OR g.grp_extended_status_c IS NULL)");
	}

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, divy_get_userid(r));

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select \"divy_grp\". (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}


	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE) {
		if (*rdb_r == NULL) {
			*rdb_r 	= apr_pcalloc(p, sizeof(divy_rdbo_resource));
			(*rdb_r)->grp_pr = grp = apr_pcalloc(p, sizeof(divy_rdbo_grp));
		}
		else {
			grp->next = apr_pcalloc(p, sizeof(divy_rdbo_grp));
			grp 	  = grp->next;
		}

		/* データ格納 */ 
		idx = 1; /* index */
		grp->grpid 	= rset->getString(rset, idx++);
		grp->name 	= rset->getString(rset, idx++);
		grp->comment= rset->getString(rset, idx++);
		grp->grpcol_uri = rset->getString(rset, idx++);
		if (support_groupleader) {
			grp->grp_extstatus = divy_rdbo_parse_extstatus(p, rset->getString(rset, idx++), EXTSTATUS_TYPE_GRP);
			grp->ownername = rset->getString(rset, idx++);
		}
#ifdef DIVY_SUPPORT_GROUPQUOTA
		if (support_groupquota) {
			grp->grp_q = apr_pcalloc(p, sizeof(divy_rdbo_grpquota));
			grp->grp_q->usedst = rset->getBigInt(rset, idx++);
			grp->grp_q->maxst  = rset->getBigInt(rset, idx++);
			grp->grp_q->usedres= rset->getBigInt(rset, idx++);
			grp->grp_q->maxres = rset->getBigInt(rset, idx++);
		}
#endif /* DIVY_SUPPORT_GROUPQUOTA */

		grp->next = NULL;
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}


/*------------------------------------------------------------------------------
  Define private functions 
  ----------------------------------------------------------------------------*/
static int _roottree_sql(request_rec *r, divy_rdbo_resource **rdb_r, int isSqlstmt)
{
	apr_pool_t *p 		= r->pool;
	DbConn *dbconn 		= NULL;
	DbPreparedStmt *stmt 	= NULL;
	DbResultSet *rset	= NULL;
	divy_rdbo_sql *sql	= NULL;
	char *sqlstr = NULL, *cond_depth;
	int i;
	divy_db_transaction_ctx *ts_ctx  = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	int support_groupleader = divy_support_groupleader(r);

	divy_cset_t *uri_set = NULL, *sqlid_set = NULL;
	int uri_count;
	divy_linkedlist_t *in_clause;
	divy_db_bind_ctx *bindctx = NULL;
	divy_db_bind_ctx_idx *idx;
	const char *in_clause_str = NULL;
	char *uri, *slapos, *cururi, *cur_sql, *sqlid;

	*rdb_r = NULL;	/* 初期化 */
    
	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * SQL文の作成
	 */

	/* グループURIと名称を同一にする場合、
	 * 深さ1 のグループに割り当てられているSQLだけを取得する */
	if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON) {
		cond_depth = " AND g.grp_depth_i = 1";
	}
	else {
		cond_depth = "";
	}

	/* 全所属グループの相対URIを取得する */
	if (support_groupleader) {
		/* (note) 非アクティブなグループはリスティングしないこと */
		sqlstr = apr_pstrcat(p,
				"SELECT "
				" g.grp_relative_uri_txt"
				" FROM divy_grp g"
				" WHERE g.grp_grp_id_c IN"
				" (SELECT grpm_grp_id_c"
				" FROM divy_grpmem"
				" WHERE grpm_usr_id_vc = ?)"
				, cond_depth,
				" AND (g.grp_extended_status_c NOT LIKE '___-%' "DIVY_DBFUNC_ESCAPE_CLAUSE
				" OR g.grp_extended_status_c IS NULL)", NULL);
	}
	else {
		sqlstr = apr_pstrcat(p,
				"SELECT "
				" g.grp_relative_uri_txt"
				" FROM divy_grp g"
				" WHERE g.grp_grp_id_c IN"
				" (SELECT grpm_grp_id_c"
				" FROM divy_grpmem"
				" WHERE grpm_usr_id_vc = ?)"
				, cond_depth, NULL);
	}

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, sqlstr, p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, divy_get_userid(r));

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}


	while (rset->next(rset) == DB_TRUE) {
		/* 所属grp + 親grp のURI集合を作成します */
		if (uri_set == NULL) {
			uri_set = divy_cset_make(p);
		}

		uri = rset->getString(rset, 1);

		/* 所属grp の uri 格納 */
		divy_cset_set(uri_set, uri);

		/* URIの切り出し */
		slapos = cururi = apr_pstrdup(p, uri);
		uri_count = divy_count_dirs(cururi);

		/* スラッシュ数-1(最初の / はスキップ) だけループ */
		for (i = 0; i < uri_count - 1; i++) {
			/* 末尾から探していく */
			slapos = divy_strrchr(cururi, '/');
			if (slapos) *slapos = '\0';

			/* URI集合に追加 */
			divy_cset_set(uri_set, apr_pstrdup(p, cururi));
		}
	}
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* バインドコンテキストを生成 */
	bindctx = divy_db_bindcontext_make(p, uri_set, 50);
	for (idx = divy_db_bindcontext_first(bindctx); idx; idx = divy_db_bindcontext_next(idx)) {
		/* 
		 * divy_sql から所属grp,親grp のSQL 情報を取得 
		 */

		/* バインド変数文字列の取得 */
		divy_db_bindcontext_get_bindstr(idx, &in_clause_str);
		cur_sql = apr_psprintf(p,
					"SELECT sql_id_c"
					", sql_type_i"
					", sql_relative_uri_txt"
					", sql_label_name_vc"
					" FROM divy_sql"
					" WHERE sql_type_i = ?"
					" AND sql_active_i = 1"
					" AND sql_id_c IN"
					" (SELECT sqlm_sql_id_c"
					" FROM divy_sqlmem"
					" WHERE sqlm_grp_id_c IN"
					" (SELECT grp_grp_id_c"
					" FROM divy_grp"
					" WHERE grp_relative_uri_txt IN"
					" (%s)))", in_clause_str);

		/* SQL文の準備 失敗時はエラー */
		stmt = dbconn->prepareStatement(dbconn, cur_sql, p);
		if (stmt->getCode(stmt) != DB_SUCCESS){
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		/* バインド 通常とrepos で切り替えます */
		i = 0;
		if (isSqlstmt) {
			stmt->setInt(stmt, ++i, 0);
		}
		else {
			stmt->setInt(stmt, ++i, 1);
		}
		divy_db_bindcontext_get_list(idx, &in_clause);
		for (; in_clause; in_clause = in_clause->next) {
			stmt->setString(stmt, ++i, in_clause->val);
		}

		/* SQL実行 失敗時はエラー */
		rset = stmt->executeQuery(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS){
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to select \"divy_sql\". (Reason: %s)", rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* フェッチ(データありの間) */
		while (rset->next(rset) == DB_TRUE) {
			if (sqlid_set == NULL) {
				sqlid_set = divy_cset_make(p);
			}
			sqlid = rset->getString(rset, 1);

			/* このSQLIDのデータは既に取得済みか? */
			if (divy_cset_contains(sqlid_set, sqlid)) {
				continue;	/* だったら次 */
			}
			divy_cset_set(sqlid_set, sqlid);

			if (*rdb_r == NULL) {
				*rdb_r 	= apr_pcalloc(p, sizeof(divy_rdbo_resource));
				sql = apr_pcalloc(p, sizeof(divy_rdbo_sql));
				(*rdb_r)->sql_pr = sql;
			}
			else {
				sql->next = apr_pcalloc (p, sizeof(divy_rdbo_sql));
				sql = sql->next;
			}

			/* データ格納 */ 
			sql->sqlid 	 = sqlid;
			sql->type 	 = (divy_rdbo_sqltype) rset->getInt(rset, 2);
			sql->relativeuri = rset->getString(rset, 3);
			sql->labelname 	 = rset->getString(rset, 4);

			sql->next = NULL;
		}
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

