/**
 * $Id$
 *
 * 特殊フォルダ「SQL」関連の関数をまとめたファイル
 *
 * (note)
 *   ここで定義された関数群はtf_rdbo.c にあったものです.
 *   tf_rdbo.c が巨大化し過ぎたため、関連のあるものをまとめました.
 */
#include "httpd.h"
#include "http_protocol.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"

#include "mod_dav_tf.h"
#include "tf_db.h"
#include "util_db.h"
#include "tf_rdbo.h"
#include "util.h"
#include "search.h"
#include "tf_sqlparser.h"
#include "tf_folder.h"
#include "tf_valuecache.h"
#include "tf_linkedlist.h"
#include "tf_rdbo_util.h"
#include "tf_rdbo_sql.h"
#include "tf_rdbo.h"

/*------------------------------------------------------------------------------
  Declare private prototype functions 
  ----------------------------------------------------------------------------*/
static int _remove_sqldepend(request_rec *r,
					const char *top_sqlid,
					divy_db_transaction_ctx *ts_ctx);
static int _insert_sqldepend(request_rec *r,
					divy_rdbo_sql *sql_pr,
					divy_rdbo_sqldepend *all_dependlist,
					divy_db_transaction_ctx *ts_ctx);
static int _lock_all_divy_sql(request_rec *r, divy_db_transaction_ctx *ts_ctx);

/*------------------------------------------------------------------------------
  Define public functions 
  ----------------------------------------------------------------------------*/
/**
 *  リポジトリDBを検索する(sqlinformationsearch)
 *
 *  
 */
DIVY_DECLARE(int) divy_rdbo_sqlinformationsearch(request_rec *r,
					divy_rdbo_resource **rdb_r, 
					const char *sqlstr, 
					apr_hash_t *sqldtopid_h,
					apr_hash_t *cachedreqsql,
					divy_rdbo_rsvalue *rsvalue,
					int searchopt) 
{
	apr_pool_t *p 		= r->pool;
	DbConn *dbconn 		= NULL;
	DbPreparedStmt *stmt 	= NULL;
	DbResultSet *rset 	= NULL;
	divy_rdbo_sqlcontent *sqlc = NULL;
	divy_sql_parser *parser;
	apr_hash_index_t *hashind;
	char *_sqlstr           = NULL;
	const char *hashkey;
	divy_rdbo_sqldepend *sqld = NULL, *tmp_sqld;
	divy_rdbo_clist *subname_cl = NULL, *first_subname_cl = NULL;
	divy_rdbo_sql *sql_pr = NULL;
	int reqflg, rsvalflg;
	divy_rdbo_reqsql  *reqsql;
	divy_rdbo_rsvalue *tmp_rsvalue;
	apr_hash_t *sql_pr_h 	= NULL, *top_rsvalue_hash = NULL;
	divy_cset_t *cset_t;
	int result, idx;
	divy_db_transaction_ctx *ts_ctx;
	int iscommit 		= 1;	/* この関数が反映するかどうかを決定する */
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
	divy_rdbo_sqltype type;

	*rdb_r = NULL;  /* 初期化 */

	/* sqlparser 作成 */
	if (divy_sql_parser_create(p, &parser) != DIVY_SQLP_ST_OK){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to create sql parser.");
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		/* Internal Server Error */
		return 1;
	}

	/* トランザクションコンテキストの生成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to start transaction.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
    	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, sqlstr, p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", 
				REPLACE_NULL(stmt->getMsg(stmt)));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet.Reason: %s", 
				REPLACE_NULL(rset->getMsg(rset)));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (searchopt == DIVY_SEARCH_OPT_TREELIST ||
		searchopt == DIVY_SEARCH_OPT_DETAILLIST) {
		divy_rdbo_sql *sql = NULL;

		/* フェッチ(データありの間) */
		while (rset->next(rset) == DB_TRUE) {

			/* まずはsqltype を取得 */
			type = rset->getInt(rset, 4);

			/* グループ管理者機能がサポートされていて、
			 * グループリーダの場合、RequiredSQLと名前付きバインド変数は
			 * リスティングしないこと */
			if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus) &&
				(type == DIVY_SQL_TYPE_REQUIRED || type == DIVY_SQL_TYPE_BIND)) {
				continue;
			}

			/* 領域確保 */
			if (!(*rdb_r)) {
				*rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
				sql = apr_pcalloc(p, sizeof(divy_rdbo_sql));
				(*rdb_r)->sql_pr = sql;
			}
			else {
				sql->next = apr_pcalloc(p, sizeof(divy_rdbo_sql));
				sql = sql->next;
			}
	    
			/* データを取得 */
			sql->sqlid 	= rset->getString(rset, 1);
			sql->labelname 	= rset->getString(rset, 2);

			/* detaillistの場合はさらに取得 */
			if (searchopt == DIVY_SEARCH_OPT_DETAILLIST){
				sql->subname     = rset->getString(rset, 3);
				sql->type        = type;
				sql->active      = rset->getInt(rset, 5);
				sql->comment     = rset->getString(rset, 6);
				sql->registdt    = rset->getString(rset, 7);
				sql->updatedt    = rset->getString(rset, 8);
				sql->usedbmsname = rset->getString(rset, 9);
			}
		}
	}
	else if (searchopt == DIVY_SEARCH_OPT_CONTENT) {

		/* フェッチ(データありの間) */
		while (rset->next(rset) == DB_TRUE) {

			/* sqltype を取得 */
			type = rset->getInt(rset, 4);

			/* グループ管理者機能がサポートされていて、
			 * グループリーダの場合、RequiredSQLと名前付きバインド変数は
			 * プロパティを見せない */
			if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus) &&
				(type == DIVY_SQL_TYPE_REQUIRED || type == DIVY_SQL_TYPE_BIND)) {
				continue;
			}

			/* 領域確保 1件目の場合はSQLの本体に関する情報を格納 */
			if (!(*rdb_r)) {
				*rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
				(*rdb_r)->sql_pr = apr_pcalloc(p, sizeof(divy_rdbo_sql));
				sqlc = apr_pcalloc(p, sizeof(divy_rdbo_sqlcontent));
				(*rdb_r)->sql_pr->sqlcnt_pr = sqlc;
	    
				(*rdb_r)->sql_pr->sqlid       = rset->getString(rset, 1);
				(*rdb_r)->sql_pr->labelname   = rset->getString(rset, 2);
				(*rdb_r)->sql_pr->subname     = rset->getString(rset, 3);
				(*rdb_r)->sql_pr->type        = type;
				(*rdb_r)->sql_pr->cachemode   = rset->getInt(rset, 5);
				(*rdb_r)->sql_pr->sql         = rset->getString(rset, 6);
				(*rdb_r)->sql_pr->active      = rset->getInt(rset, 7);
				(*rdb_r)->sql_pr->comment     = rset->getString(rset, 8);
				(*rdb_r)->sql_pr->usedbmsname = rset->getString(rset, 9);
				(*rdb_r)->sql_pr->registdt    = rset->getString(rset, 10);
				(*rdb_r)->sql_pr->updatedt    = rset->getString(rset, 11);
			}
			else {
				sqlc->next = apr_pcalloc(p, sizeof(divy_rdbo_sqlcontent));
				sqlc= sqlc->next;
			}

			/* content情報格納 */
			sqlc->id	= rset->getInt(rset, 12);
			sqlc->contype	= rset->getInt(rset, 13);
			sqlc->name 	= rset->getString(rset, 14);
			sqlc->colname 	= rset->getString(rset, 15);
			sqlc->fmt	= rset->getString(rset, 16);

			/* RequiredSQL, 名前付きバインド変数 */
			if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED){
				sqlc->reqsql = apr_pcalloc(p, sizeof(divy_rdbo_reqsql));
				sqlc->reqsql->rsvalue       = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
				sqlc->reqsql->rsvalue->name = rset->getString(rset, 17);
				sqlc->reqsql->sqlmode       = rset->getInt(rset, 18);
				/* 
				 * sqlmodelist 作成
				 */
				/* WHERE句プロパティとして
				 * $$Bxxx が指定されていた場合 */
				if (IS_NBIND_NODE
					(sqlc->reqsql->rsvalue->name)) {
					/* (note)
					 *  これだけを特別扱いしたのはループの
					 *  中でSQLを引きたくなかったため。
					 *  まずは集めておいて、後でデフォルト値
					 *  があるかどうか一括して調べる
					 */
					if (subname_cl == NULL) {
						subname_cl = first_subname_cl = apr_pcalloc(p, sizeof(divy_rdbo_clist));
					}
					else {
						subname_cl->next = apr_pcalloc(p, sizeof(divy_rdbo_clist));
						subname_cl = subname_cl->next;
					}
					subname_cl->val = apr_pstrdup(p, sqlc->reqsql->rsvalue->name);
					subname_cl->next = NULL;
				}
				else {
					/* sqlmodelist の設定 */
					(void) divy_sql_parser_get_sqlmodelist(parser,
							(*rdb_r)->sql_pr->type,
							sqlc->reqsql->rsvalue->name,
							DIVY_SQL_NO_DEFAULT_VAL,
							&sqlc->reqsql->sqlmodelist);
				}
			}
			sqlc->sqlposition = rset->getInt(rset, 19);
		}

		/* もう使わないので閉じておく */
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/*
		 * このSQL(RequiredSQL, 名前付きバインド変数)を利用している
		 * SQL名称の一覧(usingsqllist)を取得
		 */
		if (*rdb_r &&
		    ((*rdb_r)->sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
		     (*rdb_r)->sql_pr->type == DIVY_SQL_TYPE_BIND)) {

			/* sqldepend からデータ取得(topid指定なし) */
			if (divy_rdbo_get_sqldepend_list(r, NULL, &sqld, ts_ctx) != 0) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"An error occurred while getting "
					"sqldepend of RequiredSQL.");
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}

			/* このSQLを利用しているSQLを取得する */
			result = divy_sql_parser_find_usingsql(parser,
						(*rdb_r)->sql_pr->subname,
						sqld, &cset_t);

			/* 使用しているSQL あり */
			if (result == DIVY_SQLP_ST_USINGSQL_FOUND && cset_t) {
				/* 使用SQL ハッシュ(sqlid と subname)から
				   ラベル名称を取得 */
				if (divy_rdbo_get_sql_labelname_by_idsubname(r,
						divy_cset_tohash(cset_t),
						 &((*rdb_r)->sql_pr->usingsql_hash),
						 ts_ctx) != 0) {
					ERRLOG0(p, APLOG_ERR,
						DIVY_FST_IERR + DIVY_SST_PROC,
						"An error occurred while getting "
						"sqllabelname with dependency.");
					ts_ctx->status |= DIVY_TRANS_ABORT;
					if (iscommit)
						divy_db_rollback_transaction(ts_ctx);
					return 1;
				}
			} 
			/* 実行エラー */
			else if (result == DIVY_SQLP_ST_ERR){
				ERRLOG0(p, APLOG_ERR,
					DIVY_FST_IERR + DIVY_SST_PROC,
					"An error occurred while getting "
					"using sql of RequiredSQL.");
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit)
					divy_db_rollback_transaction(ts_ctx);
				return 1;
			}
		}

		/*
		 * WHERE句に指定された名前付きバインド変数がデフォルト値を
		 * 持っていたかどうかを調べる
		 */
		/* divy_sql テーブルを検索して名前付きバインド変数の
		 * SQL文を取得する */
		sql_pr_h = NULL;
		if (first_subname_cl &&
		    divy_rdbo_get_sql_by_subname(r, first_subname_cl, &sql_pr_h, ts_ctx) != 0) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"An error occured while getting value "
				"of namedbind.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		if (*rdb_r != NULL) {
			/* ループしながら値を取得する */
			for (sqlc = (*rdb_r)->sql_pr->sqlcnt_pr; first_subname_cl && sqlc; sqlc = sqlc->next) {

				/* 名前付きバインド変数以外には興味なし */
				if (sqlc->contype != DIVY_SQL_CONTYPE_REQUIRED ||
					!IS_NBIND_NODE(sqlc->reqsql->rsvalue->name)) {
					continue;
				}

				/* DB には存在しない名前付きバインド変数が
				 * WHERE句プロパティとして指定されていた(不整合) */
				if (sql_pr_h == NULL ||
						((sql_pr = apr_hash_get(sql_pr_h,
												sqlc->reqsql->rsvalue->name,
												APR_HASH_KEY_STRING)) && sql_pr == NULL)) {

					/* ちゃんとエラーも報告できないので
					 * 		大目に見ておきましょう */
					(void) divy_sql_parser_get_sqlmodelist(parser,
														   (*rdb_r)->sql_pr->type,
														   sqlc->reqsql->rsvalue->name,
														   DIVY_SQL_NO_DEFAULT_VAL,
														   &sqlc->reqsql->sqlmodelist);
					continue;
				}

				/* SQL文をパースして値があるかどうかを調べる */
				result = divy_sql_parser_is_nbind_setting(parser, sql_pr->sql);

				/* デフォルト値があった */
				if (result == DIVY_SQLP_ST_FOUND_NBIND_VAL) {

					/* sqlmodelist の設定 */
					(void) divy_sql_parser_get_sqlmodelist(parser,
														   (*rdb_r)->sql_pr->type,
														   sqlc->reqsql->rsvalue->name,
														   DIVY_SQL_HAS_DEFAULT_VAL,
														   &sqlc->reqsql->sqlmodelist);
				}
				/* なければshow */
				else if (result == DIVY_SQLP_ST_NF_NBIND_VAL) {

					/* sqlmodelist の設定 */
					(void) divy_sql_parser_get_sqlmodelist(parser,
														   (*rdb_r)->sql_pr->type,
														   sqlc->reqsql->rsvalue->name,
														   DIVY_SQL_NO_DEFAULT_VAL,
														   &sqlc->reqsql->sqlmodelist);
				}
				else {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
							"An error occured while checking value of namedbind.");
					ts_ctx->status |= DIVY_TRANS_ABORT;
					if (iscommit) divy_db_rollback_transaction(ts_ctx);

					/* Internal Server Error */
					return 1;
				}
			}
		}
	}
	else if (searchopt == DIVY_SEARCH_OPT_QUERYCONTENT) {

		reqflg = 0;
		/* フェッチ(データありの間) */
		while (rset->next(rset) == DB_TRUE){

			/* 領域確保 1件目の場合はSQLの本体に関する情報を格納 */
			if (!(*rdb_r)){
				*rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
				(*rdb_r)->sql_pr = apr_pcalloc(p, sizeof(divy_rdbo_sql));
				sqlc = apr_pcalloc(p, sizeof(divy_rdbo_sqlcontent));

				(*rdb_r)->sql_pr->sqlcnt_pr   = sqlc;
				(*rdb_r)->sql_pr->sqlid       = rset->getString(rset, 1);
				(*rdb_r)->sql_pr->labelname   = rset->getString(rset, 2);
				(*rdb_r)->sql_pr->subname     = rset->getString(rset, 3);
				(*rdb_r)->sql_pr->type        = rset->getInt(rset, 4);
				(*rdb_r)->sql_pr->comment     = rset->getString(rset, 5);
				(*rdb_r)->sql_pr->usedbmsname = rset->getString(rset, 6);
			} else {
				sqlc->next = apr_pcalloc(p, sizeof(divy_rdbo_sqlcontent));
				sqlc= sqlc->next;
			}

			/* content情報格納 */
			sqlc->id	= rset->getInt(rset, 7);
			sqlc->contype	= rset->getInt(rset, 8);
			sqlc->name 	= rset->getString(rset, 9);
			sqlc->fmt	= rset->getString(rset, 10);

			/* required、名前付きバインド変数だった場合 */
			if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED){
				reqflg = 1;	/* フラグ設定 */

				/* 領域確保 */
				reqsql = apr_pcalloc(p, sizeof(divy_rdbo_reqsql));
				tmp_rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
				sqlc->reqsql = reqsql;
				sqlc->reqsql->rsvalue = tmp_rsvalue;

				/* 値を格納 */
				tmp_rsvalue->name = rset->getString(rset, 11);
				reqsql->sqlmode   = rset->getInt(rset, 12);
			}
		}

		/* クローズする */
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* required、名前付きバインド変数があった場合 */
		if (reqflg == 1) {

			sqld = NULL;	/* 初期化 */
			/* sqldepend からデータ取得 */
			if (divy_rdbo_get_sqldepend_list(r, sqldtopid_h, &sqld, ts_ctx) != 0) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get sqldependlist.");
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				return 1;
			}

			/* 指定されたsqlno = topid のデータが divy_sqldepend
		  	 * にない場合は結果を空にして返す(NOT FOUND) 
			 * これはsqlpreference と同じ仕様です */
			if (sqld == NULL) {
				*rdb_r = NULL;
				if (iscommit) divy_db_commit_transaction(ts_ctx);
				return 0;
			}

			/* RequiredSQL の値を解決する */

			/* 
			 * 依存関係のあるSQLが値を取得出来なかったときには、
			 * Multistatus & Not Found を返却する
			 * 戻り値DIVY_SQLP_ST_EXEC_FAILEDが返却された時には、
			 * top_rsvalue には実行失敗の原因となった
			 * SQLの名前(subname)を格納します。
			 */
			result = divy_sql_parser_resolve_reqsql(parser, r, 
							ts_ctx,
							sqld, 
							rsvalue, 
							&top_rsvalue_hash);
			/* 依存SQLの実行に失敗した場合 */
			if (result == DIVY_SQLP_ST_EXEC_FAILED ||
			    result == DIVY_SQLP_ST_TOOMANY_COLUMN ||
			    result == DIVY_SQLP_ST_TOOFEW_COLUMN) {
				divy_rdbo_sql *sql = NULL;
				divy_sbuf *sbuf    = NULL;
				int first = 1;

				divy_sbuf_create(p, &sbuf, 256);

				/* SQL 文作成 */
				divy_sbuf_append(sbuf,
						"SELECT"
						" sql_relative_uri_txt"
						" FROM divy_sql"
						" WHERE sql_sub_name_vc IN (");
				/* top_rsvalue をループしてwhere句作成 */
				for (hashind = apr_hash_first(p, 
							top_rsvalue_hash); 
					hashind;
					hashind = apr_hash_next(hashind)){

					apr_hash_this(hashind, (const void **)&hashkey,
							NULL, NULL);
					if (first) {
						divy_sbuf_appendbyte(sbuf, 1, "'");
						divy_sbuf_append(sbuf, hashkey);
						divy_sbuf_appendbyte(sbuf, 1, "'");
						first = 0;
					}
					else {
						divy_sbuf_appendbyte(sbuf, 2, ",'");
						divy_sbuf_append(sbuf, hashkey);
						divy_sbuf_appendbyte(sbuf, 1, "'");
					}
				}
				divy_sbuf_appendbyte(sbuf, 1, ")");

				ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
					"execfailedsql = %s", divy_sbuf_tostring(sbuf));
				/* 
				 * divy_sql からrequiredsql の情報取得
				 */
				/* sql文の準備 失敗時はエラー */
				stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sbuf), p);
				if (stmt->getCode(stmt) != DB_SUCCESS) {
					ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. "
						"(divy_sql) Reason: %s", stmt->getMsg(stmt));

					ts_ctx->status |= DIVY_TRANS_ABORT;
					if (iscommit) divy_db_rollback_transaction(ts_ctx);
					if (stmt != NULL) stmt->close(stmt); stmt = NULL;
					return 1;
				}
				/* sql実行 失敗時はエラー */
				rset = stmt->executeQuery(stmt, p);
				if (rset->getCode(rset) != DB_SUCCESS) {
					ERRLOG1(p, APLOG_ERR, 
						DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbResultSet."
						"(divy_sql) Reason: %s",
						rset->getMsg(rset));
					ts_ctx->status |= DIVY_TRANS_ABORT;
					if (iscommit) divy_db_rollback_transaction(ts_ctx);
					if (rset != NULL) rset->close(rset); rset = NULL;
					if (stmt != NULL) stmt->close(stmt); stmt = NULL;
					return 1;
				}

				/* sql_pr をつぶしてしまうので先にログを出力 */
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to resolve requiredsql."
					"(sqlno of top SQL : %s)", (*rdb_r)->sql_pr->sqlid);

				/* フェッチ(データありの間) */
				while (rset->next(rset) == DB_TRUE){

					/* 領域確保 */
					if (!sql){
						sql = apr_pcalloc(p, sizeof(divy_rdbo_sql));
						(*rdb_r)->sql_pr = sql;
					} else {
						sql->next = apr_pcalloc(p, sizeof(divy_rdbo_sql));
						sql = sql->next;
					}
					/* uri 格納 */
					sql->relativeuri = rset->getString(rset, 1);
				}
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				if (rset != NULL) rset->close(rset); rset = NULL;
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;

				return DIVY_STCODE_SQLISFAILEDDEPENDENCY;
			}
			else if (result == DIVY_SQLP_ST_ERR){

				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to resolve requiredsql."
					"(sqlno of top SQL : %s)", (*rdb_r)->sql_pr->sqlid);
				ts_ctx->status |= DIVY_TRANS_ABORT;
				if (iscommit) divy_db_rollback_transaction(ts_ctx);
				/* Internal Server Error */
				return 1;
			}

			/* 結果構造体のsqlcontent 構造体をループし、
			 * parser で取得したrsvalue を割り当てていく */
			for (sqlc = (*rdb_r)->sql_pr->sqlcnt_pr; sqlc;
							sqlc = sqlc->next){
				/* required、名前付きバインド変数だった場合 */
				if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED){
					rsvalflg = 0;

					/* parser で取得したrsvalue を検索 */
					if (!(tmp_rsvalue = apr_hash_get
						(top_rsvalue_hash, 
						 sqlc->reqsql->rsvalue->name, 
						 APR_HASH_KEY_STRING))){

						/* 見つからない場合はエラー */
						ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"There is data mismatching between divy_sqlconten and "
							"divy_sqldepend. (subname = %s)", sqlc->reqsql->rsvalue->name);
						ts_ctx->status |= DIVY_TRANS_ABORT;
						if (iscommit)
							divy_db_rollback_transaction(ts_ctx);
						return 1;
					}

					/* 割り当て */
					sqlc->reqsql->rsvalue = tmp_rsvalue;
				}
			}
		}
	}
	else if (searchopt == DIVY_SEARCH_OPT_AVAILABLEGROUP) {
		divy_rdbo_grp *grp = NULL;

		/* フェッチ(データありの間) */
		while (rset->next(rset) == DB_TRUE) {
			/* 領域確保 1件目の場合はSQLの本体に関する情報を格納 */
			if (!(*rdb_r)) {
				*rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
				(*rdb_r)->sql_pr = apr_pcalloc(p, sizeof(divy_rdbo_sql));
				grp = apr_pcalloc(p, sizeof(divy_rdbo_grp));
				(*rdb_r)->sql_pr->grp_pr = grp;

				(*rdb_r)->sql_pr->sqlid     = rset->getString(rset, 1);
				(*rdb_r)->sql_pr->labelname = rset->getString(rset, 7);
			}
			else {
				grp->next = apr_pcalloc(p, sizeof(divy_rdbo_grp));
				grp = grp->next;
			}

			/* grp情報格納 */
			grp->grpid	= rset->getString(rset, 2);
			grp->name 	= rset->getString(rset, 3);
			grp->comment 	= rset->getString(rset, 4);
			grp->registdt 	= rset->getString(rset, 5);
			grp->updatedt 	= rset->getString(rset, 6);

			idx = 8;
			if (support_grpconstraints || support_groupleader) {
				grp->grp_extstatus = divy_rdbo_parse_extstatus(p, rset->getString(rset, idx++), EXTSTATUS_TYPE_GRP);
				/* 値が取れなければデフォルト値を使用する(互換性) */
				if (grp->grp_extstatus == NULL) {
					grp->grp_extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_GRP);
				}
			}
			else {
				/* サポートしていなければデフォルト値を使用(互換性) */
				grp->grp_extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_GRP);
			}

			if (support_groupleader) {
				grp->ownerid   = rset->getString(rset, idx++);
				grp->ownername = rset->getString(rset, idx++);
			}
		}
	}
	else if (searchopt == DIVY_SEARCH_OPT_DEPENDSQLLIST){
		char *topuri;
		apr_hash_t *reqsqlname_h;

		/* フェッチ(データなしはエラー(badrequest)) */
		if (rset->next(rset) != DB_TRUE){
			/*
			 * dependsqllist でトップ階層のURLが取得出来ない場合は
			 * エラー(badrequest)
			 * この仕様はlinkdbsearch でSQL文が取得できない
			 * 場合と同じです。
			 */
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Failed to get URL of topsql.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return DIVY_STCODE_SQLISDEPENDNOTOPURI;
		}
		/* データ取得 */
		topuri = apr_pstrdup(p, rset->getString(rset, 1));

		/* いったんクローズ */
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		sqld = NULL;	/* 初期化 */
		/* sqldepend からデータ取得 */
		if (divy_rdbo_get_sqldepend_list(r, sqldtopid_h,
						&sqld, ts_ctx) != 0){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get sqldependlist.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		if (sqld){
			/* ハッシュに格納 */
			reqsqlname_h = apr_hash_make(p);
			for (tmp_sqld = sqld; tmp_sqld;
					tmp_sqld = tmp_sqld->next){
				/* 親はシーケンス番号の場合があるので覗く */
				if (IS_REQSQL_NODE(tmp_sqld->ptsubname)) {
					apr_hash_set(reqsqlname_h,
						tmp_sqld->ptsubname,
						APR_HASH_KEY_STRING, "");
				}
				apr_hash_set(reqsqlname_h, tmp_sqld->clsubname, APR_HASH_KEY_STRING, "");
			}
			/* 領域確保 */
			if (!(*rdb_r)){
				*rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
				(*rdb_r)->search_pr = apr_pcalloc(p, sizeof(divy_rdbo_search));
			}
			(*rdb_r)->search_pr->reqsqlname_hash = reqsqlname_h;
			/* URI を設定 */
			(*rdb_r)->uri = topuri;
		}
	}
	else if (searchopt == DIVY_SEARCH_OPT_SQLPREFERENCE){
		char *topuri;
		apr_hash_t *must_reqsql_set;
		divy_rdbo_sql *sql = NULL;

		/* 
		 * top のSQL 文取得
		 */
		/* フェッチ(データなしはエラー(badrequest)) */
		if (rset->next(rset) != DB_TRUE){
			/* 
			 * sqlpreference でトップ階層のSQLが取得出来ない場合は
			 * エラー(badrequest)
			 * この仕様はlinkdbsearch でSQL文が取得できない
			 * 場合と同じです。
			 */
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Failed to get SQL statement of topsql.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return DIVY_STCODE_SQLISDEPENDNOTOPURI;
		}
		/* データ取得 */
		topuri = apr_pstrdup(p, rset->getString(rset, 1));

		/* いったんクローズ */
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* 
		 * divy_sqldepend から依存関係取得 
		 */
		sqld = NULL;    /* 初期化 */
		/* sqldepend からデータ取得 */
		if (divy_rdbo_get_sqldepend_list(r, sqldtopid_h, &sqld, ts_ctx) != 0){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get sqldependlist.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}

		 /* 指定されたsqlno = topid のデータが divy_sqldepend
		  * にない場合はなにもしない(NOT FOUND) */
		if (!sqld){
			if (iscommit) divy_db_commit_transaction(ts_ctx);
			return 0;
		}

		/* sqlparser 作成 */
		if (divy_sql_parser_create(p, &parser) != DIVY_SQLP_ST_OK){
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to create sql parser.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			/* Internal Server Error */
			return 1;
		}

		/* sql 依存関係で解決されていない最下部を取得する */
		if (divy_sql_parser_find_sqldependency(parser, r, ts_ctx,
							sqld, 
							cachedreqsql, 
							&must_reqsql_set) != DIVY_SQLP_ST_OK){

			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to find sql dependency."
					"(rerativeuri of top SQL : %s)",
					topuri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			/* Internal Server Error */
			return 1;
		}

		 /*  解決されていない部分がない場合は終了(NOT FOUND) */
		if (!must_reqsql_set){
			if (iscommit) divy_db_commit_transaction(ts_ctx);
			return 0;
		}

		_sqlstr = NULL;	/* 初期化 */
		/* ハッシュをループしてwhere句作成 */
		for (hashind = apr_hash_first(p, must_reqsql_set); hashind;
					hashind = apr_hash_next(hashind)){
			apr_hash_this(hashind, (const void **)&hashkey, NULL, NULL);

			if (!_sqlstr){
				_sqlstr = apr_psprintf(p, "'%s'", hashkey);
			} else {
				_sqlstr = apr_pstrcat(p, _sqlstr, ",'", hashkey, "'", NULL);
			}
		}
		/* SQL 文作成 */
		_sqlstr = apr_pstrcat(p, "SELECT sql_label_name_vc"
					", sql_sub_name_vc"
					", sql_cachemode_i"
					", sql_comment_vc"
					" FROM divy_sql"
					" WHERE sql_sub_name_vc IN"
					" (", _sqlstr, ")", NULL);
		ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"reqsqlnamesql = %s", REPLACE_NULL(_sqlstr));

		/* 
		 * divy_sql から詳細情報取得
		 */
		/* sql文の準備 失敗時はエラー */
		stmt = dbconn->prepareStatement(dbconn, _sqlstr, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. "
					"(divy_sql) Reason: %s", 
					stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* sql実行 失敗時はエラー */
		rset = stmt->executeQuery(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbResultSet."
					"(divy_sql) Reason: %s", rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* フェッチ(データありの間) */
		while (rset->next(rset) == DB_TRUE){

			/* 領域確保 */
			if (!(*rdb_r)){
				*rdb_r = apr_pcalloc(p, sizeof(divy_rdbo_resource));
				sql = apr_pcalloc(p, sizeof(divy_rdbo_sql));
				(*rdb_r)->sql_pr = sql;
				/* URI 設定 */
				(*rdb_r)->uri = topuri;
			} else {
				sql->next = apr_pcalloc(p, sizeof(divy_rdbo_sql));
				sql = sql->next;
			}

			/* データ取得 */
			sql->labelname 	= rset->getString(rset, 1); 
			sql->subname 	= rset->getString(rset, 2);
			sql->cachemode 	= rset->getInt(rset, 3);
			sql->comment 	= rset->getString(rset, 4);
		}
	}

	/* content はdbms リスト作成 */
	if (*rdb_r && (searchopt == DIVY_SEARCH_OPT_ANALYSIS || 
			       	searchopt == DIVY_SEARCH_OPT_CONTENT)){
		divy_rdbo_dbms *dbms = NULL;
		char *dbmstype, *dummy, *dbmsid;
		RET_STAT retstat;
		apr_hash_t *licensed_dbmstype_h = NULL;

		/* いったんクローズ */
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		/* sql文の準備 失敗時はエラー */
		stmt = dbconn->prepareStatement(dbconn,
						"SELECT "
						"ds_ds_id_c,"
						"ds_type_vc,"
						"ds_id_name_vc "
						"FROM divy_dbms "
						"WHERE ds_active_i = 1 "
						"ORDER BY ds_id_name_vc",p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. "
					"(divy_dbms) Reason: %s", 
					stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* sql実行 失敗時はエラー */
		rset = stmt->executeQuery(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbResultSet.(divy_dbms) "
					"Reason: %s", rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		(*rdb_r)->dbms_pr = NULL;	/* 初期化 */

		/* DBMS のライセンスを取得 */
		retstat = divy_db_get_licensed_dbmstype(r, &licensed_dbmstype_h);
		if (retstat != DB_SUCCESS) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get licensed dbmstype list.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/* データ取得 */
		while (rset->next(rset) == DB_TRUE) {
			dbmsid   = rset->getString(rset, 1);
			/* リポジトリDB用エントリの場合には特別扱い */
			if (strcmp(dbmsid, DIVY_REPOS_DBMSID) == 0) {
				dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
				dbmstype = apr_pstrdup(p, dconf->dbmstype);
			}
			else {
				dbmstype = rset->getString(rset, 2);
			}

			/* このプロバイダにはライセンスがあるのか? */
			if ((dummy = apr_hash_get(licensed_dbmstype_h,
					dbmstype, APR_HASH_KEY_STRING)) == NULL) {
				continue;	/* なければ駄目 */
			}

			if (!((*rdb_r)->dbms_pr)){
				dbms = apr_pcalloc(p, sizeof(divy_rdbo_dbms));
				(*rdb_r)->dbms_pr = dbms;
			}
			else {
				dbms->next = apr_pcalloc(p, sizeof(divy_rdbo_dbms));
				dbms = dbms->next;
			}
			dbms->name  = rset->getString(rset, 3);
		}
	}
	if (iscommit) divy_db_commit_transaction(ts_ctx);
    
	/* 使用した領域を解放 (dbconn は何もしてはならない) */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたuri が示すSQLの情報を取得して返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_sql_property(request_rec *r, const char *sqluri,
							divy_rdbo_sql **sql_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;

	*sql_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(sqluri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"sqluri is empty.");
		return 1;
	}

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.(sqluri = %s)", sqluri);
		return 1;
	}

	dbconn->startTrans(dbconn, 0);
	stmt = dbconn->prepareStatement(dbconn,
			"SELECT "
			"sql_id_c,"
			"sql_label_name_vc,"
			"sql_sub_name_vc,"
			"sql_type_i,"
			"sql_relative_uri_txt,"
			"sql_stmt_txt,"
			"sql_active_i,"
			"sql_ds_id_c,"
			"sql_comment_vc,"
			"sql_regist_c,"
			"sql_update_c,"
			"sql_cachemode_i "
			"FROM divy_sql "
			"WHERE sql_relative_uri_txt = ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (sqluri = %s) "
			"Reason: %s", sqluri, stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, sqluri);

	/* sql実行　失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet.(sqluri = %s) "
				"Reason: %s", sqluri, rset->getMsg(rset));
		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* データがあれば取得(ない場合は初期化時のNULL) */
	if (rset->next(rset) == DB_TRUE) {

		/* 領域確保 */
		*sql_pr = apr_pcalloc(p, sizeof(divy_rdbo_sql));

		/* データ格納 */
		(*sql_pr)->sqlid       = rset->getString(rset, 1);
		(*sql_pr)->labelname   = rset->getString(rset, 2);
		(*sql_pr)->subname     = rset->getString(rset, 3);
		(*sql_pr)->type	       = rset->getInt(rset,    4);
		(*sql_pr)->relativeuri = rset->getString(rset, 5);
		(*sql_pr)->sql	       = rset->getString(rset, 6);
		(*sql_pr)->active      = (divy_rdbo_sqlactive) rset->getInt(rset, 7);
		(*sql_pr)->dbid        = rset->getString(rset, 8);
		(*sql_pr)->comment     = rset->getString(rset, 9);
		(*sql_pr)->registdt    = rset->getString(rset, 10);
		(*sql_pr)->updatedt    = rset->getString(rset, 11);
		(*sql_pr)->cachemode   = (divy_rdbo_sqlcachemode) rset->getInt(rset, 12);

		/* 次のsql_prはNULL */
		(*sql_pr)->next       = NULL;
	}

	/* クローズ */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	/* トランザクションの反映 */
	dbconn->commit(dbconn);

	return 0;
}

/**
 * 指定されたsql_pr が表すSQLを新規登録する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_sql_property(request_rec *r,
						const divy_rdbo_sql *sql_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL, *stmt1 = NULL, *stmt2 = NULL;
	apr_pool_t *p           = r->pool;
	time_t short_time       = dav_divy_get_now_epoch();
	char *datebuf           = NULL;
	int insert_cnt;
	divy_rdbo_sqldepend *all_dependlist = NULL;
	divy_rdbo_sqlcontent *sqlcnt_cur    = NULL;
	divy_db_transaction_ctx *ts_ctx   = NULL;

	TRACE(p);

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * 新規作成に先立ち、依存関係の破壊を防ぐ目的で
	 * divy_sqlテーブルをロックします
	 */
	if (_lock_all_divy_sql(r, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to lock divy_sql.(labelname = %s)",
			sql_pr->labelname);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		return 1;	/* ロックに失敗したら何もしない */
	}

	/* 
	 * divy_sqlへの新規追加
	 */

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, 
			"INSERT INTO divy_sql ("
			"sql_id_c,"
			"sql_label_name_vc,"
			"sql_relative_uri_txt,"
			"sql_stmt_txt,"
			"sql_active_i,"
			"sql_ds_id_c,"
			"sql_comment_vc,"
			"sql_regist_c,"
			"sql_update_c,"
			"sql_sub_name_vc,"
			"sql_type_i,"
			"sql_cachemode_i) "
			"VALUES ("DIVY_DBFUNC_CREATE_SQL_SEQ",?,?,?,?,"
			"(SELECT ds_ds_id_c FROM divy_dbms WHERE ds_id_name_vc = ?),"
			"?,?,?,?,?,?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (sqlname = %s) "
				"Reason: %s", 
				sql_pr->labelname, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, sql_pr->labelname);
	stmt->setString(stmt, 2, sql_pr->relativeuri);
	stmt->setString(stmt, 3, REPLACE_EMPTYSTR(sql_pr->sql));
	stmt->setInt(stmt,    4, (apr_int32_t) sql_pr->active);
	stmt->setString(stmt, 5, sql_pr->usedbmsname);
	stmt->setString(stmt, 6, REPLACE_EMPTYSTR(sql_pr->comment));
	stmt->setString(stmt, 7, datebuf);
	stmt->setString(stmt, 8, datebuf);
	stmt->setString(stmt, 9, REPLACE_EMPTYSTR(sql_pr->subname));
	stmt->setInt(stmt,   10, (apr_int32_t) sql_pr->type);
	stmt->setInt(stmt,   11, (apr_int32_t) sql_pr->cachemode);

	/* SQL実行　失敗時はエラー */
	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_sql.(sqlname = %s) "
			"Reason: %s", 
			sql_pr->labelname, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* insert 件数が0 件の場合 */
	if (insert_cnt == 0) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"Status is OK, but count of inserting is 0. "
			"(sqlname = %s, dbsname = %s) ",
			sql_pr->labelname, sql_pr->usedbmsname);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_sqlcontentへの新規追加
	 */
	for (sqlcnt_cur = sql_pr->sqlcnt_pr; sqlcnt_cur;
					sqlcnt_cur = sqlcnt_cur->next) {

		/* SQL文の準備　失敗時はエラー */
		if (sqlcnt_cur->contype == DIVY_SQL_CONTYPE_REQUIRED) {
			/* RequiredSQL が存在した場合 */
			if (stmt1 == NULL) {
				stmt1 = dbconn->prepareStatement(dbconn,
					"INSERT INTO divy_sqlcontent ("
					"sqlc_id_c,"
					"sqlc_where_id_i,"
					"sqlc_where_contype_i,"
					"sqlc_where_name_vc,"
					"sqlc_where_colname_vc,"
					"sqlc_where_fmt_vc,"
					"sqlc_where_sql_position_i,"
					"sqlc_depsql_sub_name_vc,"
					"sqlc_depsql_mode_i) "
					"VALUES ("
					"(SELECT sql_id_c FROM divy_sql "
					"WHERE sql_relative_uri_txt = ?), "
					"?, ?, ?, ?, ?, ?, ?, ?)", p);

				if (stmt1->getCode(stmt1) != DB_SUCCESS) {
					ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. "
						"(sqlname = %s) Reason: %s", 
						sql_pr->labelname,
						stmt1->getMsg(stmt1));
					ts_ctx->status |= DIVY_TRANS_ABORT;
					divy_db_rollback_transaction(ts_ctx);
					if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
					if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
					return 1;
				}
			}
			stmt = stmt1;
		}
		else {
			/* RequiredSQL がない場合 */
			if (stmt2 == NULL) {
				stmt2 = dbconn->prepareStatement(dbconn,
					"INSERT INTO divy_sqlcontent ("
					"sqlc_id_c,"
					"sqlc_where_id_i,"
					"sqlc_where_contype_i,"
					"sqlc_where_name_vc,"
					"sqlc_where_colname_vc,"
					"sqlc_where_fmt_vc,"
					"sqlc_where_sql_position_i,"
					"sqlc_depsql_sub_name_vc,"
					"sqlc_depsql_mode_i) "
					"VALUES ("
					"(SELECT sql_id_c FROM divy_sql "
					"WHERE sql_relative_uri_txt = ?), "
					"?, ?, ?, ?, ?, ?, NULL, NULL)", p);

				if (stmt2->getCode(stmt2) != DB_SUCCESS) {
					ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. "
						"(sqlname = %s) Reason: %s", 
						sql_pr->labelname,
						stmt2->getMsg(stmt2));
					ts_ctx->status |= DIVY_TRANS_ABORT;
					divy_db_rollback_transaction(ts_ctx);
					if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
					if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
					return 1;
				}
			}
			stmt = stmt2;
		}

		stmt->setString(stmt, 1, sql_pr->relativeuri);
		stmt->setInt(stmt,    2, sqlcnt_cur->id);
		stmt->setInt(stmt,    3, (apr_int32_t) sqlcnt_cur->contype);
		stmt->setString(stmt, 4, sqlcnt_cur->name);
		stmt->setString(stmt, 5, sqlcnt_cur->colname);
		stmt->setString(stmt, 6, REPLACE_EMPTYSTR(sqlcnt_cur->fmt));
		stmt->setInt(stmt,    7, sqlcnt_cur->sqlposition);
		if (sqlcnt_cur->contype == DIVY_SQL_CONTYPE_REQUIRED) {
			stmt->setString(stmt, 8, sqlcnt_cur->reqsql->rsvalue->name);
			stmt->setInt(stmt,    9, (apr_int32_t) sqlcnt_cur->reqsql->sqlmode);
		}

		/* SQL実行　失敗時はエラー */
	        (void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to insert divy_sql."
					"(sqlname = %s) Reason: %s",
					sql_pr->labelname, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			/* (note) stmt はリファレンスなのでcloseしません */
			if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
			if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;

			return 1;
		}
	}
	/* (note) stmt はリファレンスなのでcloseしません */
	if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
	if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;

	/*
	 * divy_sqldepend への新規追加
	 */
	/* DB から現存する全依存リスト(親子関係)を取得する */
	if (divy_rdbo_get_sqldepend_list(r, NULL, &all_dependlist, ts_ctx)) {

		/* 予期しないエラー発生 */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get all sql-depend list."
			"(sqluri = %s)", sql_pr->relativeuri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/* 新規追加 */
	if (_insert_sqldepend(r, (divy_rdbo_sql *) sql_pr,
						all_dependlist, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_sqldepend."
			"(sqlname = %s) ", sql_pr->labelname);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/* 変更を確定 */
	divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたsql_pr が表す内容でSQLを更新する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_sql_property(request_rec *r, 
					const divy_rdbo_sql *sql_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL, *stmt1 = NULL, *stmt2 = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int update_cnt          = 0;
	time_t short_time       = dav_divy_get_now_epoch();
	char *datebuf           = NULL;
	divy_rdbo_sql *old_sql_pr, *sqlpr;
	divy_rdbo_sqlcontent *sqlcnt;
	divy_cset_t *old_subname_set = NULL, *new_subname_set = NULL;
	divy_rdbo_sqldepend *all_dependlist = NULL, *sql_d = NULL, *new_dependlist = NULL;
	apr_hash_t *sqlpr_h     = NULL;
	apr_hash_index_t *hi;
	divy_cset_index_t *ci;
	char *subname, *sqlid, *dummy, *sql, *pt_name;
	dav_buffer sbuf = { 0 };
	int is_subname_changed = 0, is_sqltype_changed = 0;

	TRACE(p);

	/*
	 * 処理をはじめる前のチェック
	 */
	if (sql_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"sql_pr is NULL.");
		return 1;
	}

	/* 更新前のsql_pr を取得する */
	old_sql_pr = sql_pr->next;
	if (old_sql_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"old_sql_pr is NULL.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/*
		 * ここでは作成せずエラーにする
		 * (note) liveproperty プロバイダの絶対的な制約
		 */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Could not operation. Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * 更新処理に先立ち、依存関係の破壊を防ぐ目的で
	 * divy_sqlテーブルをロックします
	 */
	if (_lock_all_divy_sql(r, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to lock divy_sql.(labelname = %s)",
			sql_pr->labelname);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;	/* ロックに失敗したら何もしない */
	}

	/*
	 * ロック待ちをしている間、誰かがdivy_sqlテーブルを更新していたか
	 * どうかを判定する。
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT"
				" sql_update_c "
				"FROM divy_sql "
				"WHERE sql_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(sqlid = %s) Reason: %s", 
			sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
       	stmt->setString(stmt,1, sql_pr->sqlid);

	/* sql実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_sql.(sqlid = %s) "
			"Reason: %s", sql_pr->sqlid, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 更新日付の取得 */
	if (rset->next(rset) == DB_TRUE) {
		char *update_dt = rset->getString(rset, 1);
		/* 更新日付が途中で変わってしまっていないかどうか判定 */
		if (update_dt && old_sql_pr->updatedt &&
			strcmp(update_dt, old_sql_pr->updatedt) != 0) {

			/* あきらめる */
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"This sql was update by another."
				"We gave up to update this sql, because "
				"the dependency of sql was changed."
				"(sqlid = %s)", sql_pr->sqlid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
	}
	else {
		/* 誰かに消されてしまったということ */
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"Status is OK, but the count of selecting is 0. "
			"Maybe the entry of divy_sql is missing."
			"(sqlid = %s)", sql_pr->sqlid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 片付け */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* サブ名称を変更する必要があるかどうかを判定 */
	if (IS_FILLED(sql_pr->subname) && IS_FILLED(old_sql_pr->subname) &&
	    strcmp(sql_pr->subname, old_sql_pr->subname) != 0) {

		is_subname_changed = 1;	/* サブ名称が変更されていた */
	}

	/* SQLの種類(sqltype)がRequiredSQL・名前付きバインド変数から
	 * 通常・リポジトリSQL,またはその逆に変更されていないかどうかの判定 */
	if (((old_sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
	      old_sql_pr->type == DIVY_SQL_TYPE_BIND) &&
	     (sql_pr->type == DIVY_SQL_TYPE_NORMAL ||
	      sql_pr->type == DIVY_SQL_TYPE_REPOSITORY))  ||
	    ((old_sql_pr->type == DIVY_SQL_TYPE_NORMAL ||
	      old_sql_pr->type == DIVY_SQL_TYPE_REPOSITORY) &&
	     (sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
	      sql_pr->type == DIVY_SQL_TYPE_BIND))) {

		is_sqltype_changed = 1;	/* 変更されていた */
	}

	/*
	 * 更新項目を調べるため
	 * divy_sqlcontent から情報を取得
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"SELECT "
			"sqlc_where_contype_i,"
			"sqlc_depsql_sub_name_vc "
			"FROM divy_sqlcontent "
			"WHERE sqlc_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (sqlid = %s) "
			"Reason: %s", sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, sql_pr->sqlid);

	/* sql実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(sqlid = %s) "
			"Reason: %s", sql_pr->sqlid, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* データ取得 */
	while (rset->next(rset) == DB_TRUE) {

		/* RequiredSQL の場合 */
		if ((divy_rdbo_contype) rset->getInt(rset, 1) ==
						DIVY_SQL_CONTYPE_REQUIRED) {
			if (old_subname_set == NULL) {
				old_subname_set = divy_cset_make(p);
			}
			/* old_subname_set にサブ名称を入れる */
			divy_cset_set(old_subname_set, rset->getString(rset, 2));
		}
	}
	/* クローズ */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_sql テーブルの更新
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_sql "
				"SET "
                        	"sql_stmt_txt = ?,"
	                        "sql_active_i = ?,"
        	                "sql_ds_id_c ="
				" (SELECT ds_ds_id_c"
				"  FROM divy_dbms"
				"  WHERE ds_id_name_vc = ?), "
                	        "sql_comment_vc = ?,"
                        	"sql_update_c = ?,"
				"sql_sub_name_vc = ?,"
				"sql_type_i = ?,"
				"sql_cachemode_i = ? "
				"WHERE sql_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(sqluri = %s) Reason: %s", 
			sql_pr->relativeuri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

        /* 日付の作成 (ISO8601形式) */
       	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

        /* バインド */
       	stmt->setString(stmt, 1, REPLACE_EMPTYSTR(sql_pr->sql));
	stmt->setInt(stmt,    2, (apr_int32_t) sql_pr->active);
       	stmt->setString(stmt, 3, sql_pr->usedbmsname);
        stmt->setString(stmt, 4, REPLACE_EMPTYSTR(sql_pr->comment));
       	stmt->setString(stmt, 5, datebuf);
        stmt->setString(stmt, 6, REPLACE_EMPTYSTR(sql_pr->subname));
        stmt->setInt(stmt,    7, (apr_int32_t) sql_pr->type);
        stmt->setInt(stmt,    8, (apr_int32_t) sql_pr->cachemode);
       	stmt->setString(stmt, 9, sql_pr->sqlid);

        /* SQL実行　失敗時はエラー */
       	update_cnt = stmt->executeUpdate(stmt, p);
        if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update divy_sql."
			"(sqluri = %s, sqlid = %s) Reason: %s",
			sql_pr->relativeuri, sql_pr->sqlid,
			stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	/* 更新件数が1件以外はエラー */
	if (update_cnt != 1) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to update divy_sql. data is not found."
			"(sqluri = %s, sqlid = %s)",
			sql_pr->relativeuri, sql_pr->sqlid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL)   stmt->close(stmt); stmt = NULL;

	/*
	 * divy_sqlcontentへの更新
	 * (note)
	 * 	sqlid が示すdivy_sqlcontentテーブルのエントリを
	 * 	一旦削除してから新規挿入するという操作を行います。
 	 */

	/* 対象SQL文のdivy_sqlcontentを全て削除 */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_sqlcontent "
			"WHERE sqlc_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(sqluri = %s, sqlid = %s) Reason: %s",
			sql_pr->relativeuri, sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, sql_pr->sqlid);

	/* SQL実行　失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_sqlcontent. "
			"(sqluri = %s, sqlid = %s) Reason: %s",
			sql_pr->relativeuri, sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 指定されたdivy_sqlcontent を新規追加 */
	for (sqlcnt = sql_pr->sqlcnt_pr; sqlcnt; sqlcnt = sqlcnt->next) {

		/* SQL文の準備　失敗時はエラー */
		if (sqlcnt->contype == DIVY_SQL_CONTYPE_REQUIRED) {
			/* RequiredSQL が存在した場合 */
			if (stmt1 == NULL) {
				stmt1 = dbconn->prepareStatement(dbconn,
					"INSERT INTO divy_sqlcontent ("
					"sqlc_id_c,"
					"sqlc_where_id_i,"
					"sqlc_where_contype_i,"
					"sqlc_where_name_vc,"
					"sqlc_where_colname_vc,"
					"sqlc_where_fmt_vc,"
					"sqlc_where_sql_position_i,"
					"sqlc_depsql_sub_name_vc,"
					"sqlc_depsql_mode_i) "
					"VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)", p);

				if (stmt1->getCode(stmt1) != DB_SUCCESS) {
					ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. "
						"(sqluri = %s, sqlid = %s) Reason: %s", 
						sql_pr->relativeuri, sql_pr->sqlid,
						stmt1->getMsg(stmt1));
					ts_ctx->status |= DIVY_TRANS_ABORT;
					if (iscommit) divy_db_rollback_transaction(ts_ctx);
					if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
					if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
					return 1;
				}
			}
			stmt = stmt1;
			if (new_subname_set == NULL) {
				new_subname_set = divy_cset_make(p);
			}
			/* RequiredSQLのサブ名称の集合を記録しておく */
			divy_cset_set(new_subname_set, sqlcnt->reqsql->rsvalue->name);
		}
		else {
			/* RequiredSQL がない場合 */
			if (stmt2 == NULL) {
				stmt2 = dbconn->prepareStatement(dbconn,
					"INSERT INTO divy_sqlcontent ("
					"sqlc_id_c,"
					"sqlc_where_id_i,"
					"sqlc_where_contype_i,"
					"sqlc_where_name_vc,"
					"sqlc_where_colname_vc,"
					"sqlc_where_fmt_vc,"
					"sqlc_where_sql_position_i,"
					"sqlc_depsql_sub_name_vc,"
					"sqlc_depsql_mode_i) "
					"VALUES (?, ?, ?, ?, ?, ?, ?, NULL, NULL)", p);
				if (stmt2->getCode(stmt2) != DB_SUCCESS) {
					ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. "
						"(sqluri = %s, sqlid = %s) Reason: %s", 
						sql_pr->relativeuri, sql_pr->sqlid,
						stmt2->getMsg(stmt2));
					ts_ctx->status |= DIVY_TRANS_ABORT;
					if (iscommit) divy_db_rollback_transaction(ts_ctx);
					if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
					if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
					return 1;
				}
			}
			stmt = stmt2;
		}

		/* バインド */
		stmt->setString(stmt, 1, sql_pr->sqlid);
		stmt->setInt(stmt,    2, sqlcnt->id);
		stmt->setInt(stmt,    3, (apr_int32_t) sqlcnt->contype);
		stmt->setString(stmt, 4, sqlcnt->name);
		stmt->setString(stmt, 5, sqlcnt->colname);
		stmt->setString(stmt, 6, REPLACE_EMPTYSTR(sqlcnt->fmt));
		stmt->setInt(stmt,    7, sqlcnt->sqlposition);

		if (sqlcnt->contype == DIVY_SQL_CONTYPE_REQUIRED) {
			stmt->setString(stmt, 8, sqlcnt->reqsql->rsvalue->name);
			stmt->setInt(stmt,    9, (apr_int32_t) sqlcnt->reqsql->sqlmode);
		}

		/* SQL実行　失敗時はエラー */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_sqlcontent. "
				"(sqluri = %s, sqlid = %s) Reason: %s", 
				sql_pr->relativeuri, sql_pr->sqlid,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			/* (note) stmt はリファレンスなのでcloseしません */
			if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
			if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
			return 1;
		}
	}
	/* (note) stmt はリファレンスなのでcloseしません */
	if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
	if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;

	/*
	 * divy_sqlconent テーブルのsqlc_depsql_sub_name_vc を更新
	 * [ 開始条件 ]
	 * 	・sql_pr はRequiredSQL または名前付きバインド変数である
	 * 	・サブ名称が変更されていること
	 */
	if ((sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
	     sql_pr->type == DIVY_SQL_TYPE_BIND) && is_subname_changed) {

		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_sqlcontent "
				"SET sqlc_depsql_sub_name_vc = ? "
				"WHERE sqlc_depsql_sub_name_vc = ?", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(sqluri = %s, sqlid = %s) Reason: %s", 
				sql_pr->relativeuri, sql_pr->sqlid,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		/* バインド */
		stmt->setString(stmt, 1, sql_pr->subname);
		stmt->setString(stmt, 2, old_sql_pr->subname);

		/* １件も無いかもしれないので更新カウントはとらない */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_sqlcontent for updating "
				"subname.(subname = %s, sqlid = %s) "
				"Reason: %s", sql_pr->subname, sql_pr->sqlid,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/* 親ノードの名前を取得 */
	if (sql_pr->type == DIVY_SQL_TYPE_NORMAL ||
	    sql_pr->type == DIVY_SQL_TYPE_REPOSITORY) {
		/* 通常SQL, リポジトリSQL */
		pt_name = sql_pr->sqlid;
	}
	else { /* RequiredSQL, 名前付きバインド変数 */
		pt_name = sql_pr->subname;
	}

	/*
	 * divy_sqldepend テーブルのsqld_pt_subname_vcを更新
	 * [ 開始条件 ]
	 * 	・sql_pr はRequiredSQL または名前付きバインド変数である
	 * 	・サブ名称が変更されていること
	 * 	or
	 * 	・sqltype がRequiredSQL・名前付きバインド変数から
	 * 	  通常・リポジトリSQLに,またはその逆に変更されている
	 */
	if (((sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
	      sql_pr->type == DIVY_SQL_TYPE_BIND) &&
				is_subname_changed) || is_sqltype_changed) {

		/* sqld_pt_subname_vc を更新する */
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_sqldepend "
				"SET sqld_pt_subname_vc = ? "
				"WHERE sqld_pt_subname_vc = ?", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(sqluri = %s, sqlid = %s) Reason: %s", 
				sql_pr->relativeuri, sql_pr->sqlid,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		/* バインド */
		stmt->setString(stmt, 1, pt_name);
		if (is_sqltype_changed &&
		    (old_sql_pr->type == DIVY_SQL_TYPE_NORMAL ||
		     old_sql_pr->type == DIVY_SQL_TYPE_REPOSITORY)) {
			/* 昔は依存SQLではなかった */
			stmt->setString(stmt, 2, sql_pr->sqlid);
		}
		else {
			/* 今はともかく、昔は依存SQL */
			stmt->setString(stmt, 2, old_sql_pr->subname);
		}

		/* 実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_sqldepend for updating "
				"sqld_pt_subname_vc.(sqluri = %s, sqlid = %s) "
				"Reason: %s", sql_pr->relativeuri,
				sql_pr->sqlid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/*
	 * divy_sqldepend テーブルの sqld_cl_subname_vcを更新
	 * [ 開始条件 ]
	 * 	・sql_pr はRequiredSQL または名前付きバインド変数である
	 * 	・サブ名称が変更されていること
	 */
	if ((sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
	      sql_pr->type == DIVY_SQL_TYPE_BIND) && is_subname_changed) {

		/* sqld_pt_subname_vc を更新する */
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_sqldepend "
				"SET sqld_cl_subname_vc = ? "
				"WHERE sqld_cl_subname_vc = ?", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(sqluri = %s, sqlid = %s) Reason: %s", 
				sql_pr->relativeuri, sql_pr->sqlid,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		/* バインド */
		stmt->setString(stmt, 1, sql_pr->subname);
		stmt->setString(stmt, 2, old_sql_pr->subname);

		/* 実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_sqldepend for updating "
				"sqld_cl_subname_vc.(sqluri = %s, sqlid = %s) "
				"Reason: %s", sql_pr->relativeuri,
				sql_pr->sqlid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/*
	 * 他のSQLとの依存関係(divy_sqldepend) の更新
	 *
	 * 他のSQLとの依存関係リストは、topSQLシーケンス番号がdivy_sqldepend
	 * に存在することでSQL間の関係が至る所に複製されてしまっているため
	 * 変更のある場所だけを直すには、非常に手数がかかる。
	 * そこで、少々乱暴だが、WHERE句プロパティに含まれていたRequiredSQL
	 * と関係している全ての関係を再更新することにした。
	 *
	 * [ 依存関係のパターン ]
	 * 	a) 新しい関係     : S -> n, n -> n1, n1 -> n2
	 * 	b) 失われた関係   : S -> l, l -> l1
	 * 	c) 変わらない関係 : S -> c, c -> c1
	 * 	d) a)の変形       : * -> S, S -> n,  n -> n1, n1 -> n2
	 * 	e) b)の変形       : * -> S, S -> l,  l -> l1
	 * 	f) c)の変形       : * -> S, S -> c,  c -> c1
	 * 		S: 自分 / n,l,c: WHERE句に指定された依存SQL /
	 * 			n1,c1,l1... : 間接的に依存している依存SQL
	 * [ 更新手順 ]
	 * 	(1) 新しい関係(a,d) を求める
	 * 	(2) 失われた関係(b,e) を求める
	 * 	(3) 全依存関係 + (1) - (2) を算出する -> 新しい依存関係
	 * 	(4) 更新対象SQLのdivy_sql, divy_sqlcontentを取得する
	 * 	(5) (4)で対象となったSQLのdivy_sqldependエントリを削除
	 * 	(6) (4)のSQLの依存関係を再作成する
	 */
	/* (1) 新しい関係は存在するか？ */
	for (sqlcnt = sql_pr->sqlcnt_pr; sqlcnt; sqlcnt = sqlcnt->next) {

		/* 通常バインド変数だったら何もしない */
		if (sqlcnt->contype == DIVY_SQL_CONTYPE_BIND) continue;

		subname = sqlcnt->reqsql->rsvalue->name;
		/* このsubnameは古いWHERE句プロパティセットにはなかった */
		if (divy_cset_contains(old_subname_set, subname) == 0) {
			if (new_dependlist == NULL) {
				new_dependlist = sql_d = apr_pcalloc(p,
						sizeof(divy_rdbo_sqldepend));
			}
			else {
				sql_d->next = apr_pcalloc(p,
						sizeof(divy_rdbo_sqldepend));
				sql_d       = sql_d->next;
			}
			sql_d->topid     = NULL;
			sql_d->ptsubname = pt_name;
			sql_d->clsubname = subname;
			sql_d->next      = NULL;
		}
	}

	/* (2) 失われた関係を取得 */
	/* 昔からWHERE句プロパティはなかった */
	if (old_subname_set == NULL) {
		/* 失われた関係はなかった */
		sql = 	apr_pstrdup(p,	"SELECT"
					" sqld_pt_subname_vc,"
					" sqld_cl_subname_vc"
					" FROM divy_sqldepend"
					" GROUP BY"
					" sqld_pt_subname_vc,"
					" sqld_cl_subname_vc");
	}
	/* 昔はWHERE句プロパティがあった */
	else {
		dav_set_bufsize(p, &sbuf, 512);
		sbuf.cur_len = 0;
		for (ci = divy_cset_first(old_subname_set);
						ci; ci = divy_cset_next(ci)) {
			divy_cset_this(ci, (const char **)&subname);

			/* 今は全くWHERE句プロパティがない or 
			 * 幾つか持っているが、無くなってしまったプロパティあり */
			if (sql_pr->sqlcnt_pr == NULL ||
			    (sql_pr->sqlcnt_pr &&
			     divy_cset_contains(new_subname_set, subname) == 0)) {

				/* 依存関係リストには含めないリストを作る */
				if (sbuf.cur_len) {
					dav_buffer_append(p, &sbuf, " OR ");
				}
				dav_buffer_append(p, &sbuf, "(sqld_pt_subname_vc ='");
				dav_buffer_append(p, &sbuf, pt_name);
				dav_buffer_append(p, &sbuf, "' AND sqld_cl_subname_vc ='");
				dav_buffer_append(p, &sbuf, subname);
				dav_buffer_append(p, &sbuf, "')");
			}
		}
		/* リストがあった時 */
		if (sbuf.cur_len) {
			sql = apr_psprintf(p,
					"SELECT"
					" sqld_pt_subname_vc,"
					" sqld_cl_subname_vc"
					" FROM divy_sqldepend"
					" WHERE NOT (%s)"
					" GROUP BY"
					" sqld_pt_subname_vc,"
					" sqld_cl_subname_vc", sbuf.buf);
		}
		else {
			sql = apr_pstrdup(p,
					"SELECT"
					" sqld_pt_subname_vc,"
					" sqld_cl_subname_vc"
					" FROM divy_sqldepend"
					" GROUP BY"
					" sqld_pt_subname_vc,"
					" sqld_cl_subname_vc");
		}
	}

	/* 新しい関係も失われた関係も存在しなかった場合(c,fの場合) */
	if (new_dependlist == NULL && sbuf.cur_len == 0) {
		/* もう何もすることはありません */
		if (iscommit) divy_db_commit_transaction(ts_ctx);
		return 0;
	}

	/* (3) DB から現存する全依存リスト(親子関係)を取得する */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.(divy_sqldepend) "
			"(sqlid = %s, sql = %s) Reason: %s",
			sql_pr->sqlid, sql, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_sqldepend."
			"(sqlid = %s, sql = %s) Reason: %s",
			sql_pr->sqlid, sql, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* 依存関係リストの作成 */
	while (rset->next(rset) == DB_TRUE) {

		if (all_dependlist == NULL) {
			all_dependlist = sql_d = apr_pcalloc(p, 
					sizeof(divy_rdbo_sqldepend));
		}
		else {
			sql_d->next = apr_pcalloc(p, 
					sizeof(divy_rdbo_sqldepend));
			sql_d = sql_d->next;
		}
		sql_d->ptsubname = rset->getString(rset, 1);
		sql_d->clsubname = rset->getString(rset, 2);
		sql_d->next      = NULL;
	}
	/* クローズ */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* (1) のリストをall_dependlist に追加する */
	if (all_dependlist) {
		/* 先のループで使ったsql_d を利用させてもらいます */
		sql_d->next = new_dependlist;
	}
	else {
		all_dependlist = new_dependlist;
	}

	/* (4) sql_pr 以外のSQLプロパティ(divy_sql, divy_sqlcontent) を取得 */
	stmt = dbconn->prepareStatement(dbconn,
			"SELECT "
			" sql.sql_id_c,"
			" sql.sql_sub_name_vc,"
			" sql.sql_type_i,"
			" sql.sql_relative_uri_txt,"
			" sqlc.sqlc_depsql_sub_name_vc,"
			" sqlc.sqlc_where_contype_i "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2 ) /* postgres */
			"FROM divy_sql sql "
			"LEFT JOIN divy_sqlcontent sqlc "
			"ON sql.sql_id_c = sqlc.sqlc_id_c "
			"WHERE sql_id_c IN"
			" (SELECT sqld_top_sql_id_c"
			"  FROM divy_sqldepend"
			"  WHERE sqld_pt_subname_vc = ?"
			"  OR sqld_cl_subname_vc = ?"
			"  GROUP BY sqld_top_sql_id_c) "
			"AND sql.sql_id_c != ? "
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
			"FROM divy_sql sql, divy_sqlcontent sqlc"
			" WHERE sql.sql_id_c = sqlc.sqlc_id_c(+)"
			" AND (sql.sql_id_c IN"
			" (SELECT sqld_top_sql_id_c"
			"  FROM divy_sqldepend"
			"  WHERE sqld_pt_subname_vc = ?"
			"  OR sqld_cl_subname_vc = ?"
			"  GROUP BY sqld_top_sql_id_c) "
			"AND sql.sql_id_c != ?) "
#endif
			"ORDER BY sql.sql_id_c", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (sqlid = %s) "
			"Reason: %s", sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, pt_name);
	stmt->setString(stmt, 2, pt_name);
	stmt->setString(stmt, 3, sql_pr->sqlid);

	/* sql実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_sql and divy_sqlcontent."
			"(sqlid = %s, subname = %s) Reason: %s", sql_pr->sqlid,
			REPLACE_NULL(sql_pr->subname), rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	sqlpr_h = apr_hash_make(p);	/* sqlid をキー,sql_pr を値に持つハッシュ */

	/* 自分自身を追加しておく */
	apr_hash_set(sqlpr_h, sql_pr->sqlid, APR_HASH_KEY_STRING, sql_pr);

	/* データ取得 */
	while (rset->next(rset) == DB_TRUE) {
		sqlid = rset->getString(rset, 1);

		sqlpr = apr_hash_get(sqlpr_h, sqlid, APR_HASH_KEY_STRING);
		/* divy_sql のエントリが取得できていなかった時 */
		if (sqlpr == NULL) {
			sqlpr = apr_pcalloc(p, sizeof(divy_rdbo_sql));
			sqlpr->sqlid       = sqlid;
			sqlpr->subname     = rset->getString(rset, 2);
			sqlpr->type        = (divy_rdbo_sqltype) rset->getInt(rset, 3);
			sqlpr->relativeuri = rset->getString(rset, 4);
			sqlpr->sqlcnt_pr   = NULL;

			/* ハッシュに記憶 */
			apr_hash_set(sqlpr_h, sqlid, APR_HASH_KEY_STRING, sqlpr);
		}

		subname = rset->getString(rset, 5);
		if (IS_FILLED(subname)) {
			if (sqlpr->sqlcnt_pr == NULL) {
				sqlpr->sqlcnt_pr = sqlcnt =
					apr_pcalloc(p, sizeof(divy_rdbo_sqlcontent));
			}
			else {
				for (sqlcnt = sqlpr->sqlcnt_pr;
						sqlcnt->next; sqlcnt = sqlcnt->next);
				sqlcnt->next = apr_pcalloc(p,
						sizeof(divy_rdbo_sqlcontent));
				sqlcnt       = sqlcnt->next;
			}

			sqlcnt->contype = (divy_rdbo_contype) rset->getInt(rset, 6);
			sqlcnt->reqsql  = apr_pcalloc(p, sizeof(divy_rdbo_reqsql));
			sqlcnt->reqsql->rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_reqsql));
			sqlcnt->reqsql->rsvalue->name = subname;
			sqlcnt->reqsql->rsvalue->next = NULL;
			sqlcnt->next    = NULL;
		}
	}
	/* クローズ */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* (5) */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_sqldepend "
			"WHERE sqld_top_sql_id_c IN"
			" (SELECT sqld_top_sql_id_c"
			" FROM divy_sqldepend"
			" WHERE sqld_pt_subname_vc = ?"
			"  OR sqld_cl_subname_vc = ?"
			" GROUP BY sqld_top_sql_id_c)", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(relativeuri = %s, sqlid = %s) Reason: %s",
			sql_pr->relativeuri, sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, pt_name);
	stmt->setString(stmt, 2, pt_name);

	/* SQL実行　失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_sqldepend. "
			"(relativeuri = %s, sqlid = %s) Reason: %s",
			sql_pr->relativeuri, sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* (6) 順番に追加処理を行う */
	for (hi = apr_hash_first(p, sqlpr_h); hi; hi = apr_hash_next(hi)) {

		apr_hash_this(hi, (const void **)&dummy, NULL, (void **)&sqlpr); 

		/* sqlpr のデータをDBに挿入する */
		if (_insert_sqldepend(r, (divy_rdbo_sql *) sqlpr,
							all_dependlist, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_sqldepend for updating. "
				"(sqlid = %s)", sqlpr->sqlid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return 1;
		}
	}

	/* 変更を確定 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたsql_pr が示すSQLを削除する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_sql_property(request_rec *r,
						const divy_rdbo_sql *sql_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int delete_cnt		= 0;
	divy_db_transaction_ctx *ts_ctx = NULL;

	TRACE(p);

	/* 必須項目(UPDATEするデータのID)のチェック */
	if (sql_pr == NULL || IS_EMPTY(sql_pr->sqlid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"sqlid is NULL.");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

     	/*
	 * divy_sqlの削除
	 */

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_sql "
			"WHERE sql_id_c = ?", p);
        if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (sqlid = %s) Reason: %s", 
			sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}

        /* バインド */
        stmt->setString(stmt, 1, sql_pr->sqlid);

        /* SQL実行　失敗時はエラー */
        delete_cnt = stmt->executeUpdate(stmt, p);
        if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_sql.(sqlid = %s) Reason: %s", 
				sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* データが既に削除されていた場合はwarningで継続 */
	if (delete_cnt == 0){
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The \"divy_sql\" table did not have a recorde of "
			"this sql.(sqlid = %s) "
			"Probably data is already deleted.", sql_pr->sqlid);
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 
	 * divy_sqlcontentの削除
	 */
	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_sqlcontent "
			"WHERE sqlc_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(content-sqlid = %s) Reason: %s",
				sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, sql_pr->sqlid);

	/* SQL実行　失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to delete divy_sqlcontent. "
				"(content-sqlid = %s) Reason: %s",
				sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;							       
	/*
	 * divy_sqldepend の削除
	 * (note) このSQL自身が持っていた他のRequiredSQLの依存関係だけを
	 *        削除します。
	 */
	if (_remove_sqldepend(r, sql_pr->sqlid, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_sqldepend. "
			"(content-sqlid = %s)", sql_pr->sqlid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/*
	 * divy_sqlmemの削除
	 */
 	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_sqlmem "
			"WHERE sqlm_sql_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(sqlmem-sqlid = %s) Reason: %s",
				sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);

		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, sql_pr->sqlid);

	/* SQL実行　失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to delete divy_sqlmem. "
				"(sqlmem-sqlid = %s) Reason: %s",
				sql_pr->sqlid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 変更を確定 */
	divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたsrc_sql_pr が示すSQLをdst_sql_pr が示すSQLとして移動する。
 * 
 * src_sql_pr の必須項目: relativeuri, sqlid
 * dst_sql_pr の必須項目: relativeuri, labelname
 */
DIVY_DECLARE(int) divy_rdbo_move_sql_property(request_rec *r,
					const divy_rdbo_sql *src_sql_pr,
					const divy_rdbo_sql *dst_sql_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	time_t short_time       = dav_divy_get_now_epoch();
	char *datebuf           = NULL;
	int update_cnt;

	TRACE(p);

	/* 必須項目(SQL相対URI)のチェック */
	if (src_sql_pr == NULL || dst_sql_pr == NULL ||
	    IS_EMPTY(src_sql_pr->relativeuri) ||
	    IS_EMPTY(dst_sql_pr->relativeuri)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"URI is NULL.");
		return 1;
	}

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbConn.(srcuri = %s)", 
				src_sql_pr->relativeuri);
		return 1;
	}
	/* トランザクション開始 */
	dbconn->startTrans(dbconn, 0);

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"UPDATE divy_sql "
			"SET sql_label_name_vc = ?,"
			"sql_relative_uri_txt = ?,"
			"sql_update_c = ? "
			"WHERE sql_relative_uri_txt = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(src_uri = %s, dst_uri = %s) Reason: %s",
			src_sql_pr->relativeuri, dst_sql_pr->relativeuri,
			stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		dbconn->rollback(dbconn);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, dst_sql_pr->labelname);
	stmt->setString(stmt, 2, dst_sql_pr->relativeuri);
	stmt->setString(stmt, 3, datebuf);
	stmt->setString(stmt, 4, src_sql_pr->relativeuri);

	/* SQL実行　失敗時はエラー */
	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update divy_sql."
			"(src_uri = %s, dst_uri = %s) Reason: %s",
			src_sql_pr->relativeuri, dst_sql_pr->relativeuri,
			stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	/* 更新データが見つからなかった場合 */
	if (update_cnt == 0){
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to update divy_sql. data is not found."
			"(src_uri = %s, dst_uri = %s)",
			src_sql_pr->relativeuri, dst_sql_pr->relativeuri);
		dbconn->rollback(dbconn);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 変更を確定 */
	dbconn->commit(dbconn);
	return 0;
}

/**
 * 指定されたサブ名称のリストsubname_listを使ってSQL文を取得する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_sql_by_subname(request_rec *r,
						divy_rdbo_clist *subname_list,
						apr_hash_t **sql_pr_hash,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	dav_buffer *in_buf      = NULL;
	divy_rdbo_clist *clist  = NULL;
	divy_rdbo_sql *sql_pr   = NULL;
	char *sql;
	int iscommit            = 0;
	char *sql_fmt           = "SELECT"
				  " sql_sub_name_vc,"
				  " sql_stmt_txt "
				  "FROM divy_sql "
				  "WHERE sql_sub_name_vc IN (%s)";

	if (subname_list == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"subname_list is NULL.");
		return 1;
	}

	*sql_pr_hash = NULL;	/* 初期化 */

	/*
	 * IN 句の作成
	 */
	in_buf = apr_pcalloc(p, sizeof(dav_buffer));
	dav_set_bufsize(p, in_buf, 512);
	in_buf->cur_len = 0;

	for (clist = subname_list; clist; clist = clist->next) {
		dav_buffer_append(p, in_buf, "\'");
		dav_buffer_append(p, in_buf, clist->val);
		dav_buffer_append(p, in_buf, "\'");
		if (clist->next) {
			dav_buffer_append(p, in_buf, ",");
		}
	}

	/* SQL文の作成 */
	sql = apr_psprintf(p, sql_fmt, in_buf->buf);

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (sql = %s) "
			"Reason: %s", sql, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_sql. (sql = %s) Reason: %s", 
			sql, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {
		if (*sql_pr_hash == NULL) {
			*sql_pr_hash = apr_hash_make(p);
		}

		sql_pr = apr_pcalloc(p, sizeof(divy_rdbo_sql));

		sql_pr->subname = rset->getString(rset, 1);
		sql_pr->sql     = rset->getString(rset, 2);
		sql_pr->next    = NULL;

		apr_hash_set(*sql_pr_hash, sql_pr->subname, APR_HASH_KEY_STRING, sql_pr);
	}

	/* 後片付け */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * dbmsname が示すDBMS に対しSQL文sql を実行し、result_value に値をいれて返却する.
 *
 * (note) 2006/10/12 Thu
 *  検索対象データベースがリポジトリDBの場合、DBコネクションキャッシュが
 *  開放されないという問題が発覚しました.
 *  そのため、RequiredSQLのコネクションを維持して回るという行為は止めます.
 *  後で原因がわかったら修正して下さい. コネクションキャッシュ周りです.
 *  そういう訳で、引数のts_ctx は利用しません.
 *  驚かないで下さい.
 *
 *  本来、ここで扱うDBは顧客DBかもしれないので、キャッシュしない方が良かったのでしょうが、
 *  SQL階層が深い時のパフォーマンス劣化を避ける目的でキャッシュしていました.
 *
 */
DIVY_DECLARE(int) divy_rdbo_exec_sql(request_rec *r,
						char *dbmsname,
						char *sql,
						divy_array_t *bindvals,
						int isTopOnly,
						divy_rdbo_rsvalue **result_value,
						divy_db_transaction_ctx *ts_ctx)	/* ts_ctx は使いません */
{
	DbDataSource    *dbds   = NULL;
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int columncount         = 0;
	divy_rdbo_rsvalueset *valueset    = NULL;
	divy_db_transaction_ctx *wk_ctx = NULL;
	apr_int32_t len, i;

	if (IS_EMPTY(dbmsname) || IS_EMPTY(sql)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"dbmsname or sql is EMPTY.");
		return 1;
	}
	*result_value = NULL;	/* 初期化 */

	/* dbmsname が示すDBプロバイダのデータソースを取得し、
	 * DbConn を生成. トランザクションコンテキストも自前で作る */
	dbds = lookup_db_provider(r, dbmsname);
	if (dbds == NULL) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get datasource.(dbmsname = %s, sql = %s)", dbmsname, sql);
		return 1;
	}
	dbconn = dbds->getDbConn(dbds, p);
	if (dbconn == NULL) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.(dbmsname = %s, sql = %s)", dbmsname, sql);
		return 1;
	}
	/* 以降、dbconn を必ずcloseしなさい !! */

	/* トランザクションコンテキストの生成 */
	if (divy_db_create_transaction_ctx_by_dbconn(r, dbconn, &wk_ctx)) {
		if (dbconn != NULL) dbconn->close(dbconn); dbconn = NULL;
		return 1;
	}

	/* トランザクションを開始 */
	dbconn->startTrans(dbconn, 0);
	wk_ctx->status |= DIVY_TRANS_START;

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (sql = %s) "
			"Reason: %s", sql, stmt->getMsg(stmt));
		wk_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(wk_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		if (dbconn != NULL) dbconn->close(dbconn); dbconn = NULL;

		return 1;
	}

	/* バインド */
	len = divy_array_getlength(bindvals);
	for (i = 0; i < len; i++) {
		stmt->setString(stmt, i+1, divy_array_get(bindvals, i));
	}

	/* SQL実行 */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select sql. (sql = %s) Reason: %s", 
			sql, rset->getMsg(rset));
		wk_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(wk_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		if (dbconn != NULL) dbconn->close(dbconn); dbconn = NULL;

		return 1;
	}

	/* カラム数の取得 */
	columncount = rset->getColumnCount(rset);

	/* カラム数のチェック */
	if (columncount != DIVY_REQSQL_MAX_SELECT_COLUMN) {

		divy_db_commit_transaction(wk_ctx);
		/* 後片付け */
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		if (dbconn != NULL) dbconn->close(dbconn); dbconn = NULL;

		if (columncount < DIVY_REQSQL_MAX_SELECT_COLUMN) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
				"The number of SELECT column is small."
				"(column count = %d, sql = %s)", columncount, sql);
			return DIVY_STCODE_TOOFEW_COLUMN;
		}
		else if (columncount > DIVY_REQSQL_MAX_SELECT_COLUMN) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
				"The number of SELECT column is large."
				"(column count = %d, sql = %s)", columncount, sql);
			return DIVY_STCODE_TOOMANY_COLUMN;
		}
	}

	/* 値を取得して result_value につめる */
	while (rset->next(rset) == DB_TRUE) {
		if (*result_value == NULL) {
			*result_value = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
			(*result_value)->valueset = valueset =
					apr_pcalloc(p, sizeof(divy_rdbo_rsvalueset));
		}
		else {
			valueset->next = apr_pcalloc(p, sizeof(divy_rdbo_rsvalueset));
			valueset = valueset->next;
		}

		valueset->value     = rset->getString(rset, 1);
		valueset->dispvalue = rset->getString(rset, 2);
		valueset->next      = NULL;

		if (isTopOnly) break;	/* 先頭の１件だけ取得する場合 */
	}

	/* 後片付け */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	divy_db_commit_transaction(wk_ctx);
	if (dbconn != NULL) dbconn->close(dbconn); dbconn = NULL;

	return 0;
}

/**
 *  requiredsql,名前付きバインド変数の存在チェックをする
 *  
 */
DIVY_DECLARE(int) divy_rdbo_check_required(request_rec *r,
					apr_hash_t *subname_h,
					apr_hash_t **subname_nf_h,
					int retflg,
					divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t 	*p	 = r->pool;
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	DbResultSet     *rset    = NULL;
	int 		iscommit = 0;
	apr_hash_t *subname_db_h = NULL;
	apr_hash_index_t *hashind;
	const char	*hashkey = NULL;
	divy_sbuf	*sbuf    = NULL;
	int		first;

	*subname_nf_h = NULL;	/* 初期化 */

	if (subname_h == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"subname_h is NULL.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	divy_sbuf_create(p, &sbuf, 256);	/* 初期化 */
	first = 1;
	/* SQL 文作成 */
	divy_sbuf_append(sbuf,	"SELECT sql_sub_name_vc"
				" FROM divy_sql"
				" WHERE sql_active_i = 1"
				" AND sql_sub_name_vc IN (");
	/* ハッシュをループしてwhere句作成 */
	for (hashind = apr_hash_first(p, subname_h); hashind;
				hashind = apr_hash_next(hashind)){

		apr_hash_this(hashind, (const void **)&hashkey, NULL, NULL);

		if (first) {
			divy_sbuf_appendbyte(sbuf, 1, "'");
			divy_sbuf_append(sbuf, hashkey);
			divy_sbuf_appendbyte(sbuf, 1, "'");
			first = 0;
		}
		else {
			divy_sbuf_appendbyte(sbuf, 2, ",'");
			divy_sbuf_append(sbuf, hashkey);
			divy_sbuf_appendbyte(sbuf, 1, "'");
		}
	}
	divy_sbuf_appendbyte(sbuf, 1, ")");

	ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"reqsqlnamesql = %s", divy_sbuf_tostring(sbuf));

	/*
	 * DB から subname を検索しハッシュに格納する
	 */
	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sbuf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt(divy_sql) Reason: %s",
			REPLACE_NULL(stmt->getMsg(stmt)));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.Reason: %s",
			REPLACE_NULL(rset->getMsg(rset)));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE){

		/* ハッシュテーブルを作成 */
		if (!subname_db_h){
			subname_db_h = apr_hash_make(p);
		}

		/* データをハッシュにセット */
		hashkey = rset->getString(rset, 1);
		apr_hash_set(subname_db_h, hashkey, APR_HASH_KEY_STRING, "");
	}

	/* DB 操作は終了 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 指定されたsubname のハッシュをループして存在チェック */
	for (hashind = apr_hash_first(p, subname_h); hashind;
				hashind = apr_hash_next(hashind)){
		apr_hash_this(hashind, (const void **)&hashkey, NULL, NULL);

		/* DB データのハッシュになし */
		if (!subname_db_h  || apr_hash_get(subname_db_h, hashkey, 
						APR_HASH_KEY_STRING) == NULL) {
			/* 存在しないsubname を返却する場合 */
			if (retflg){
				if (!(*subname_nf_h)){
					*subname_nf_h = apr_hash_make(p);
				}
				/* データをハッシュにセット */
				apr_hash_set(*subname_nf_h, hashkey,
						APR_HASH_KEY_STRING, "");
			} else {
				/* 返却しない場合は終了
				 * せめてログに出しておきます */
				ERRLOG1(p, APLOG_ERR, 
					DIVY_FST_IERR + DIVY_SST_DATA,
					"Requiredsql or namedbind is "
					"not found. (subname = %s)",
					REPLACE_NULL(hashkey));
				return DIVY_STCODE_REQUIRED_NF;
			}
		}
	}

	/* 存在しないsubname あり */
	if (*subname_nf_h){
		return DIVY_STCODE_REQUIRED_NF;
	} 

	return 0;
}

/**
 * 指定されたサブ名称の集合subname_setを使ってSQLプロパティを取得する.
 *
 * select 対象テーブル: divy_sql
 * (note) 取得プロパティ
 * 	sqlid, subname, sql, active, type, usedbmsname
 */
DIVY_DECLARE(int) divy_rdbo_get_sql_properties_by_subnames(request_rec *r,
						divy_cset_t *subname_set,
						apr_hash_t **sql_pr_h,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	divy_rdbo_sql *sql_pr   = NULL;
	divy_linkedlist_t *in_clause = NULL;
	divy_db_bind_ctx *bindctx = NULL;
	divy_db_bind_ctx_idx *idx;
	const char *in_clause_str = NULL;
	int i;
	char *sql;
	int iscommit            = 0;

	if (subname_set == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"subname_set is NULL.");
		return 1;
	}

	*sql_pr_h = NULL;	/* 初期化 */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* バインドコンテキストを生成 */
	bindctx = divy_db_bindcontext_make(p, subname_set, 50);

	/* 50 個のIN句毎にSQLを発行 */
	for (idx = divy_db_bindcontext_first(bindctx); idx; idx = divy_db_bindcontext_next(idx)) {

		/* バインド変数文字列の取得 */
		divy_db_bindcontext_get_bindstr(idx, &in_clause_str);

		/* SQL 文の作成 */
		sql = apr_psprintf(p,	"SELECT "
					" s.sql_id_c,"
					" s.sql_sub_name_vc,"
					" s.sql_stmt_txt,"
					" d.ds_id_name_vc,"
					" s.sql_active_i,"
					" s.sql_type_i "
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
					"FROM divy_sql s INNER JOIN divy_dbms d"
					" ON (s.sql_ds_id_c = d.ds_ds_id_c) "
					"WHERE s.sql_sub_name_vc IN (%s)"
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
					"FROM divy_sql s, divy_dbms d"
					" WHERE s.sql_ds_id_c = d.ds_ds_id_c"
					" AND s.sql_sub_name_vc IN (%s)"
#endif
					, in_clause_str);

		/* SQLの準備 */
		stmt = dbconn->prepareStatement(dbconn, sql, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (in_clause = %s) "
				"Reason: %s", in_clause_str, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		/* バインド */
		i = 0;
		divy_db_bindcontext_get_list(idx, &in_clause);
		for (; in_clause; in_clause = in_clause->next) {
			stmt->setString(stmt, ++i, in_clause->val);
		}

		/* SQL文の実行 */
		rset = stmt->executeQuery(stmt,p);
		if (rset->getCode(rset) != DB_SUCCESS){
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select divy_sql. (in_clause = %s) Reason: %s",
				in_clause_str, rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (rset != NULL) rset->close(rset); rset = NULL;
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		while (rset->next(rset) == DB_TRUE) {
			if (*sql_pr_h == NULL) {
				*sql_pr_h = apr_hash_make(p);
			}

			sql_pr = apr_pcalloc(p, sizeof(divy_rdbo_sql));

			sql_pr->sqlid       = rset->getString(rset, 1);
			sql_pr->subname     = rset->getString(rset, 2);
			sql_pr->sql         = rset->getString(rset, 3);
			sql_pr->usedbmsname = rset->getString(rset, 4);
			sql_pr->active      = rset->getInt(rset,    5);
			sql_pr->type        = rset->getInt(rset,    6);
			sql_pr->next        = NULL;

			apr_hash_set(*sql_pr_h, sql_pr->subname,
						APR_HASH_KEY_STRING, sql_pr);
		}

		/* 後片付け */
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたid または subname の集合idsubname_hを使ってSQLラベル名称を取得する
 *  
 */
DIVY_DECLARE(int) divy_rdbo_get_sql_labelname_by_idsubname(request_rec *r,
						apr_hash_t *idsubname_h,
						apr_hash_t **labelname_h,
						divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p	 	= r->pool;
	DbConn *dbconn  	= NULL;
	DbPreparedStmt *stmt    = NULL;
	DbResultSet *rset    	= NULL;
	int iscommit            = 0;
	char *sqlstr, *sql_id, *sql_subname;
	apr_hash_index_t *hashind;
	const char *hkey	= NULL;

	*labelname_h = NULL;	/* 初期化 */

	if (idsubname_h == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"idsubname_h is NULL.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	sqlstr		= NULL;
	sql_id		= NULL;
	sql_subname	= NULL;	/* 初期化 */

	/* ハッシュをループしてwhere句作成 */
	for (hashind = apr_hash_first(p, idsubname_h); hashind;
				hashind = apr_hash_next(hashind)){
		apr_hash_this(hashind, (const void **)&hkey, NULL, NULL);

		/* id だった */
		if (!IS_REQSQL_NODE(hkey)){
			if (!sql_id){
				sql_id = apr_psprintf(p, "'%s'", hkey);
			} else {
				sql_id = apr_pstrcat(p, sql_id, 
						",'", hkey, "'", NULL);
			}
		}
		/* subname だった */
		else {
			if (!sql_subname){
				sql_subname = apr_psprintf(p, "'%s'", hkey);
			} else {
				sql_subname = apr_pstrcat(p, sql_subname, 
						",'", hkey, "'", NULL);
			}
		}
	}

	/* SQL 文作成 */
	sqlstr = apr_pstrdup(p, "SELECT sql_label_name_vc"
					" FROM divy_sql"
					" WHERE");
	if (sql_id){
		sqlstr = apr_pstrcat(p, sqlstr,
				" sql_id_c IN (", sql_id, ")", NULL);
		if (sql_subname){
			sqlstr = apr_pstrcat(p, sqlstr,
					" OR sql_sub_name_vc"
					" IN (", sql_subname, ")", NULL);
		}
	} else if (sql_subname){
		sqlstr = apr_pstrcat(p, sqlstr,
				 " sql_sub_name_vc IN (", sql_subname, ")", 
				 NULL);
	}

	ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"labelnamesql = %s", REPLACE_NULL(sqlstr));

	/*
	 * DB から labalname を検索し格納する
	 */
	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, sqlstr, p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s",
				REPLACE_NULL(stmt->getMsg(stmt)));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. Reason: %s",
				REPLACE_NULL(rset->getMsg(rset)));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE){
		/* ハッシュ作成 */
		if (!(*labelname_h)){
			*labelname_h = apr_hash_make(p);
		}
    
		/* データ格納 */
			apr_hash_set(*labelname_h, rset->getString(rset, 1),
					APR_HASH_KEY_STRING, "");
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);

	/* 使用した領域を解放 (dbconn は何もしてはならない) */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたid または subname の集合idsubname_h
 * かつ 指定されたsubname を sqlcontent にもつ
 * かつ 指定された値取得モード を sqlcontent にもつ
 * SQL 文の SQLラベル名称を取得する
 *  
 */
DIVY_DECLARE(int) divy_rdbo_get_sql_labelname_by_idsubname_sqlcsubnamemode
					(request_rec *r,
					apr_hash_t *idsubname_h,
					char *sqlcsubname,
					divy_rdbo_sqlmode sqlcdepmode,
					apr_hash_t **labelname_h,
					divy_db_transaction_ctx *ts_ctx)
{
	apr_pool_t *p	 	= r->pool;
	DbConn *dbconn  	= NULL;
	DbPreparedStmt *stmt    = NULL;
	DbResultSet *rset    	= NULL;
	int iscommit            = 0;
	char *sqlstr, *sql_id, *sql_subname;
	apr_hash_index_t *hashind;
	const char *hkey	= NULL;

	*labelname_h = NULL;	/* 初期化 */

	if (idsubname_h == NULL || IS_EMPTY(sqlcsubname) ||
			sqlcdepmode == DIVY_SQL_MODE_UNKNOWN) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Where conditions are not specified.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	sqlstr		= NULL;
	sql_id		= NULL;
	sql_subname	= NULL;	/* 初期化 */
	/* ハッシュをループしてwhere句作成 */
	for (hashind = apr_hash_first(p, idsubname_h); hashind;
				hashind = apr_hash_next(hashind)){
		apr_hash_this(hashind, (const void **)&hkey, NULL, NULL);

		/* id だった */
		if (!IS_REQSQL_NODE(hkey)){
			if (!sql_id){
				sql_id = apr_psprintf(p, "'%s'", hkey);
			} else {
				sql_id = apr_pstrcat(p, sql_id, 
						",'", hkey, "'", NULL);
			}
		}
		/* subname だった */
		else {
			if (!sql_subname){
				sql_subname = apr_psprintf(p, "'%s'", hkey);
			} else {
				sql_subname = apr_pstrcat(p, sql_subname, 
						",'", hkey, "'", NULL);
			}
		}
	}

	/* SQL 文作成 */
	sqlstr = apr_pstrdup(p, "SELECT sql.sql_label_name_vc"
#if defined(DIVY_DBMS_POSTGRES)	|| defined(DIVY_DBMS_DB2) /* postgres / db2 */
					" FROM divy_sql sql"
					" INNER JOIN divy_sqlcontent sqlc"
					" ON sql.sql_id_c = sqlc.sqlc_id_c"
					" WHERE"
					" sqlc.sqlc_depsql_sub_name_vc = ?"
					" AND sqlc.sqlc_depsql_mode_i = ?"
#elif defined(DIVY_DBMS_ORACLE)	/* oracle */
					" FROM divy_sql sql"
					", divy_sqlcontent sqlc"
					" WHERE sql.sql_id_c = sqlc.sqlc_id_c"
					" AND"
					" (sqlc.sqlc_depsql_sub_name_vc = ?"
					" AND sqlc.sqlc_depsql_mode_i = ?)"
#endif
					);
	if (sql_id && sql_subname){
		sqlstr = apr_pstrcat(p, sqlstr,
			" AND (sql.sql_id_c IN (", sql_id, ")",
			" OR sql.sql_sub_name_vc IN (", sql_subname, "))",
			NULL);
	} else if (sql_id){
		sqlstr = apr_pstrcat(p, sqlstr,
				" AND sql.sql_id_c IN (", sql_id, ")", NULL);
	} else if (sql_subname){
		sqlstr = apr_pstrcat(p, sqlstr,
				 " AND sql_sub_name_vc IN (", sql_subname, ")", 
				 NULL);
	}

	ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"labelnamesql = %s", REPLACE_NULL(sqlstr));

	/*
	 * DB から labalname を検索し格納する
	 */
	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, sqlstr, p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s",
				REPLACE_NULL(stmt->getMsg(stmt)));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString	(stmt, 1, sqlcsubname);
	stmt->setInt	(stmt, 2, (apr_int32_t) sqlcdepmode);

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. Reason: %s",
				REPLACE_NULL(rset->getMsg(rset)));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE){
		/* ハッシュ作成 */
		if (!(*labelname_h)){
			*labelname_h = apr_hash_make(p);
		}
    
		/* データ格納 */
			apr_hash_set(*labelname_h, rset->getString(rset, 1),
					APR_HASH_KEY_STRING, "");
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);

	/* 使用した領域を解放 (dbconn は何もしてはならない) */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * dbmsname のDBMS情報を使っているSQLの絶対URI一覧を取得して返却する。
 * 使用されていなければ*usingdbms_set がNULLになります。
 *
 * なお、DBMSを直接利用していたSQLに関してのみ、情報を返却することにします。
 * 間接的にこのDBMSを利用していたものについては特に報告しません。
 *
 */
DIVY_DECLARE(int) divy_rdbo_find_sql_usingdbms(request_rec *r,
						const char *dbmsname,
						apr_hash_t **usingdbms_set)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;
	char *part, *sql_uri;

	if (IS_EMPTY(dbmsname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There are empty values.");
		return 1;
	}

	*usingdbms_set = NULL;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL名称の上部に付くURIのpath segmentを取得する */
	part = divy_build_m_sql_uri(p, dav_divy_get_root_uri(r), NULL);

	/*
	 * 指定されたDBMS識別名称を直接利用しているSQL一覧の取得
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				"sql_relative_uri_txt "
				"FROM divy_sql "
				"WHERE sql_ds_id_c ="
				" (SELECT ds_ds_id_c"
				" FROM divy_dbms"
				" WHERE ds_id_name_vc = ?)", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (dbmsname = %s) "
			"Reason: %s",
			dbmsname, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setString(stmt, 1, dbmsname);

	/* SQLの実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_dbms and divy_sql."
			"(dbmsname = %s) Reason: %s", dbmsname, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 
	 * 結果の取得 
	 */
	while (rset->next(rset) == DB_TRUE) {
		if (*usingdbms_set == NULL) {
			*usingdbms_set = apr_hash_make(p);
		}

		/* SQL絶対URIを求める */
		sql_uri = dav_divy_make_uri(p, part,
					rset->getString(rset, 1), "/", NULL);
		/* 絶対URIを設定する */
		apr_hash_set(*usingdbms_set, sql_uri, APR_HASH_KEY_STRING, "");
	}

	/* 終了処理 */
	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたtop_sqlid が持っているSQLの依存関係リストを取得して返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_sqldepend_list(request_rec *r,
					apr_hash_t *topid_h,
					divy_rdbo_sqldepend **sqldepend,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	divy_rdbo_sqldepend *sql_d = NULL;
	char *sqlstr  = NULL;
	apr_hash_index_t *hashind;
	divy_array_t *bindvals  = NULL, *sqlidvals = NULL;
	apr_int32_t len, i, offset;

	*sqldepend = NULL;	/* 初期化 */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* topid_h が指定されていた場合 */
	if (topid_h != NULL) {
		dav_buffer sbuf = { 0 }, idbuf = { 0 };
		char *str;

		/* ハッシュをループしてwhere句作成 */
		for (hashind = apr_hash_first(p, topid_h); hashind;
					hashind = apr_hash_next(hashind)) {

			apr_hash_this(hashind, (const void **)&str, NULL, NULL);

			/* RequiredSQL, 名前付きバインド変数だった */
			if (IS_REQSQL_NODE(str)) {
				if (sbuf.cur_len == 0) {
					dav_set_bufsize(p, &sbuf, 512);
					sbuf.cur_len = 0;
					dav_buffer_append(p, &sbuf, "?");

					bindvals = divy_array_make(p, 10);
				}
				else {
					dav_buffer_append(p, &sbuf, ",?");
				}
				/* バインドする値を配列に入れる */
				divy_array_add(bindvals, str);
			}
			/* SQLシーケンス番号だった */
			else {
				if (idbuf.cur_len == 0) {
					dav_set_bufsize(p, &idbuf, 512);
					idbuf.cur_len = 0;
					dav_buffer_append(p, &idbuf, "?");

					sqlidvals = divy_array_make(p, 10);
				}
				else {
					dav_buffer_append(p, &idbuf, ",?");
				}
				/* バインドする値を配列に入れる */
				divy_array_add(sqlidvals, str);
			}
		}

		/* sql 文作成 */
		sqlstr = 	"SELECT"
				" sqld_pt_subname_vc,"
				" sqld_cl_subname_vc"
				" FROM divy_sqldepend";

		if (idbuf.cur_len && sbuf.cur_len) {
			sqlstr = apr_psprintf(p, "%s"
				" WHERE sqld_top_sql_id_c IN (%s)"
				" AND sqld_top_sql_id_c IN "
				"(SELECT sql_id_c FROM divy_sql"
				" WHERE sql_sub_name_vc IN (%s))"
				" GROUP BY"
				" sqld_pt_subname_vc,"
				" sqld_cl_subname_vc",
				sqlstr, idbuf.buf, sbuf.buf);
		}
		else if (idbuf.cur_len && sbuf.cur_len == 0) {
			sqlstr = apr_psprintf(p, "%s"
				" WHERE sqld_top_sql_id_c IN (%s)"
				" GROUP BY"
				" sqld_pt_subname_vc,"
				" sqld_cl_subname_vc",
				sqlstr, idbuf.buf);
		}
		else if (idbuf.cur_len == 0 && sbuf.cur_len) {
			sqlstr = apr_psprintf(p, "%s "
				" WHERE sqld_top_sql_id_c IN "
				"(SELECT sql_id_c FROM divy_sql"
				" WHERE sql_sub_name_vc IN (%s))"
				" GROUP BY"
				" sqld_pt_subname_vc,"
				" sqld_cl_subname_vc",
				sqlstr, sbuf.buf);
		}
		else {
			/* バインド変数なし。何もしない */
		}

		stmt = dbconn->prepareStatement(dbconn, sqlstr, p);
	}
	/* topid_h がなかった場合 */
	else {
		stmt = dbconn->prepareStatement(dbconn,
				"SELECT"
				" sqld_pt_subname_vc,"
				" sqld_cl_subname_vc "
				"FROM divy_sqldepend "
				"GROUP BY"
				" sqld_pt_subname_vc,"
				" sqld_cl_subname_vc", p);

	}

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.(divy_sqldepend) "
			"Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	if (sqlidvals != NULL && bindvals != NULL) {
		offset = len = divy_array_getlength(sqlidvals);
		for (i = 0; i < len; i++) {
			stmt->setString(stmt, i+1, divy_array_get(sqlidvals, i));
		}
		len = divy_array_getlength(bindvals);
		for (i = 0; i < len; i++) {
			stmt->setString(stmt, offset+i+1, divy_array_get(bindvals, i));
		}
	}
	else if (sqlidvals != NULL && bindvals == NULL) {
		len = divy_array_getlength(sqlidvals);
		for (i = 0; i < len; i++) {
			stmt->setString(stmt, i+1, divy_array_get(sqlidvals, i));
		}
	}
	else if (sqlidvals == NULL && bindvals != NULL) {
		len = divy_array_getlength(bindvals);
		for (i = 0; i < len; i++) {
			stmt->setString(stmt, i+1, divy_array_get(bindvals, i));
		}
	}
	else {
		/* バインドしなくていい */
	}

	/* sql実行　失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_sqldepend."
			"Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {

		if (*sqldepend == NULL) {
			*sqldepend = sql_d = apr_pcalloc(p, 
					sizeof(divy_rdbo_sqldepend));
		}
		else {
			sql_d->next = apr_pcalloc(p, 
					sizeof(divy_rdbo_sqldepend));
			sql_d = sql_d->next;
		}
		sql_d->ptsubname = rset->getString(rset, 1);
		sql_d->clsubname = rset->getString(rset, 2);
		sql_d->next      = NULL;
	}

	/* クローズ */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたnames(サブ名称またはSQLシーケンス番号)の配列を使って
 * divy_sqlテーブルを検索し、uriを取得する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_sqluri_by_subname(request_rec *r,
							const char **names,
							divy_rdbo_sql **sql_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;

	divy_rdbo_sql *new_sql  = NULL;
	dav_buffer *subname_buf = NULL, *sqlid_buf = NULL, *tmp_buf = NULL;
	char *sql_base          = "SELECT"
				  " sql_relative_uri_txt,"
				  " sql_id_c,"
				  " sql_sub_name_vc "
				  "FROM divy_sql WHERE ";
	char *sql;
	int subname_cnt = 0, sqlid_cnt = 0;

	*sql_pr = NULL;	/* 初期化 */

	if (*names == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"names is EMPTY");
		return 1;
	}

	/*
	 * IN句の作成
	 */
	subname_buf = apr_pcalloc(p, sizeof(dav_buffer));
	dav_set_bufsize(p, subname_buf, 512);
	subname_buf->cur_len = 0;

	sqlid_buf = apr_pcalloc(p, sizeof(dav_buffer));
	dav_set_bufsize(p, sqlid_buf, 512);
	sqlid_buf->cur_len = 0;

	for (; *names; names++) {
		if (IS_REQSQL_NODE((*names))) {
			tmp_buf = subname_buf;	/* サブ名称の場合 */

			if (subname_cnt) {
				dav_buffer_append(p, tmp_buf, ",");
			}
			subname_cnt++;
		}
		else {
			tmp_buf = sqlid_buf;	/* SQLシーケンス番号の場合 */
			if (sqlid_cnt) {
				dav_buffer_append(p, tmp_buf, ",");
			}
			sqlid_cnt++;
		}

		dav_buffer_append(p, tmp_buf, "\'");
		dav_buffer_append(p, tmp_buf, *names);
		dav_buffer_append(p, tmp_buf, "\'");
	}

	/* SQL文の作成 */
	sql = sql_base;
	if (subname_buf->cur_len == 0 && sqlid_buf->cur_len == 0) {
		/* これは呼び出し規約違反 */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"names have no data.");
		return 1;
	}

	if (subname_buf->cur_len > 0) {
		sql = apr_pstrcat(p, sql,
				"sql_sub_name_vc IN (", subname_buf->buf, ")",
				NULL);
		if (sqlid_buf->cur_len > 0) {
			sql = apr_pstrcat(p, sql, " OR ", NULL);
		}
	}

	if (sqlid_buf->cur_len > 0) {
		sql = apr_pstrcat(p, sql,
				"sql_id_c IN (", sqlid_buf->buf, ")",
				NULL);
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* sql文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt.(sql = %s) "
			"Reason: %s", sql, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* sql実行　失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_sqldepend.(sql = %s) "
			"Reason: %s", sql, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	while (rset->next(rset) == DB_TRUE) {

		if (*sql_pr == NULL) {
			*sql_pr = new_sql = apr_pcalloc(p, sizeof(divy_rdbo_sql));
		}
		else {
			new_sql->next = apr_pcalloc(p, sizeof(divy_rdbo_sql));
			new_sql = new_sql->next;
		}

		new_sql->relativeuri = rset->getString(rset, 1);
		new_sql->sqlid       = rset->getString(rset, 2);
		new_sql->subname     = rset->getString(rset, 3);
		new_sql->next        = NULL;
	}

	/* クローズ */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	divy_db_commit_transaction(ts_ctx);
	return 0;
}


/*------------------------------------------------------------------------------
  Define private functions 
  ----------------------------------------------------------------------------*/
/**
 * 指定されたtop_sqlid をdivy_sqldependテーブルのsqld_top_sql_id_cカラムに持つ
 * レコードをdivy_sqldepend から削除する。
 * このSQLの依存関係は操作しません。※
 * ※
 *   この関数を呼び出す前に、top_sqlid が示すSQLが他のSQLから
 *   必要とされていないことを保証して下さい。
 *   この保証がなければ、依存関係グラフが壊れてしまいます。
 *
 * (note)
 * 	トランザクションが継続中であれば、そのまま引き継ぎます。
 *
 * @param r request_rec *
 * @param top_sqlid const char * SQLシーケンス番号
 * @param subname const char * top_sqlid のサブ名称
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @param int 処理ステータス
 */
static int _remove_sqldepend(request_rec *r,
					const char *top_sqlid,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_sqldepend の削除
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_sqldepend "
			"WHERE sqld_top_sql_id_c = ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"(top_sqlid = %s) Reason: %s", top_sqlid,
			stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	stmt->setString(stmt, 1, top_sqlid);

	/* エントリが無いかもしれないので削除件数は数えない */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_sqldepend. "
			"(top_sqlid = %s) Reason: %s", top_sqlid,
			stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	return 0;
}

/**
 * sql_pr が表すSQLの依存関係をdivy_sqldepend テーブルに追加する。
 * RequiredSQLとの関係が全くなければ、何もしません。
 * (note) 前提条件
 * 	sql_pr->sqlid がdivy_sqldepend テーブルのsqld_top_sql_id_cに
 * 	存在しないこと。
 *
 * (note) sql_pr 必須項目
 * 	relativeuri, subname, type, sqlcnt_pr
 *
 * (note) 追加のアルゴリズム
 * ---------------------------
 *  (1) WHERE句プロパティ(sql_pr->sqlcnt_pr) のRequiredSQL一覧を取得
 *  (2) (1)の一覧を使って、追加するSQL(sql_pr)とWHERE句のRequiredSQLとの
 *  	親子関係(divy_rdbo_sqldepend) を作る
 *  (3) all_depnedlist から(1) より下位階層の依存リストを取得する
 *  (4) (2) +(3) が今回追加するsql_pr の依存関係
 *
 * @param r request_rec *
 * @param sql_pr divy_rdbo_sql * SQLプロパティ
 * @param all_dependlist divy_rdbo_sqldepend * 全依存関係リスト
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 */
static int _insert_sqldepend(request_rec *r,
					divy_rdbo_sql *sql_pr,
					divy_rdbo_sqldepend *all_dependlist,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL, *stmt1 = NULL, *stmt2 = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	int insert_cnt          = 0, flag = 0, ret = 0;
	divy_rdbo_sqlcontent *sqlcnt;
	divy_cset_t *subname_set = NULL;
	divy_sql_parser *parser  = NULL;
	divy_rdbo_sqldepend *edgelist = NULL, *new_dependlist = NULL, *list = NULL;

	/* 他のSQLを必要としているかどうか調べる */
	for (sqlcnt = sql_pr->sqlcnt_pr; sqlcnt; sqlcnt = sqlcnt->next) {
		if (sqlcnt->contype == DIVY_SQL_CONTYPE_REQUIRED) {
			flag = 1;
			break;
		}
	}

	if (!flag) return 0;	/* 何もする必要ない */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * (1)(2)
	 * WHERE句プロパティに指定されたRequiredSQLのサブ名称をsubname_set に記録
	 * また、sql_pr とWHERE句プロパティに指定されたRequiredSQLとの
	 * 親子関係をnew_dependlistに記録。
	 */
	subname_set = divy_cset_make(p);
	for (sqlcnt = sql_pr->sqlcnt_pr; sqlcnt; sqlcnt = sqlcnt->next) {

		/* WHERE句プロパティがRequiredSQLと関係なければスキップ */
		if (sqlcnt->contype != DIVY_SQL_CONTYPE_REQUIRED) continue;

		/* 同名のサブ名称が登録されていないかどうか調べる */
		if (divy_cset_contains(subname_set,
				sqlcnt->reqsql->rsvalue->name))	continue;

		/* subname_set に追加 */
		divy_cset_set(subname_set, sqlcnt->reqsql->rsvalue->name);

		/* 依存リストに追加 */
		if (new_dependlist == NULL) {
			new_dependlist = list =
				apr_pcalloc(p, sizeof(divy_rdbo_sqldepend));
		}
		else {
			list->next = apr_pcalloc(p, sizeof(divy_rdbo_sqldepend));
			list = list->next;
		}

		if (sql_pr->type == DIVY_SQL_TYPE_NORMAL ||
		    sql_pr->type == DIVY_SQL_TYPE_REPOSITORY) {
			list->ptsubname = NULL;	/* SQLシーケンス番号はDBからとる */
		}
		else {
			list->ptsubname = sql_pr->subname;
		}
		list->clsubname = sqlcnt->reqsql->rsvalue->name;
		list->next      = NULL;
	}

	/*
	 * WHERE句プロパティに指定されたRequiredSQLが持っていた
	 * 他のRequiredSQLとの親子関係リスト(edgelist) を取得する
	 */
	(void) divy_sql_parser_create(p, &parser);
	ret = divy_sql_parser_find_edgelist(parser, subname_set,
						all_dependlist, &edgelist);
	if (ret != DIVY_SQLP_ST_OK) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get edeg list.(sqluri = %s)",
			sql_pr->relativeuri);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/* new_dependlist の後ろにedgelist をつなげる(挿入する親子関係) */
	list->next = edgelist;

	/*
	 * divy_sqldepend への追加
	 * (note)
	 * 	sql_pr が通常SQLの場合、最初の親はsql_prのSQLシーケンス番号で
	 * 	子はWHERE句プロパティのサブ名称にしなければならない。
	 * 	sql_pr がRequiredSQLの場合には、最初の親もサブ名称となります。
	 */

	/* ループしながら親子関係をdivy_sqldepend にINSERTする */
	for (list = new_dependlist; list; list = list->next) {

		if (IS_EMPTY(list->ptsubname)) {
			if (stmt1 == NULL) {
				/* SQL文の準備　失敗時はエラー */
				stmt1 = dbconn->prepareStatement(dbconn,
					"INSERT INTO divy_sqldepend "
					"(sqld_top_sql_id_c,"
					" sqld_pt_subname_vc,"
					" sqld_cl_subname_vc) "
					"VALUES "
					"((SELECT sql_id_c"
					" FROM divy_sql WHERE sql_relative_uri_txt = ?),"
					"(SELECT sql_id_c FROM divy_sql"
					" WHERE sql_relative_uri_txt = ?), ?)", p);
				if (stmt1->getCode(stmt1) != DB_SUCCESS) {
					ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. "
						"(sqluri = %s) Reason: %s",
						sql_pr->relativeuri, stmt1->getMsg(stmt1));
					ts_ctx->status |= DIVY_TRANS_ABORT;
					if (iscommit) divy_db_rollback_transaction(ts_ctx);
					if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
					if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;

					return 1;
				}
			}
			stmt = stmt1;
		}
		else {
			if (stmt2 == NULL) {
				/* SQL文の準備　失敗時はエラー */
				stmt2 = dbconn->prepareStatement(dbconn,
					"INSERT INTO divy_sqldepend "
					"(sqld_top_sql_id_c,"
					" sqld_pt_subname_vc,"
					" sqld_cl_subname_vc) "
					"VALUES "
					"((SELECT sql_id_c"
					" FROM divy_sql WHERE sql_relative_uri_txt = ?),"
					"?, ?)", p);
				if (stmt2->getCode(stmt2) != DB_SUCCESS) {
					ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
						"Failed to get DbPreparedStmt. "
						"(sqluri = %s) Reason: %s",
						sql_pr->relativeuri, stmt2->getMsg(stmt2));
					ts_ctx->status |= DIVY_TRANS_ABORT;
					if (iscommit) divy_db_rollback_transaction(ts_ctx);
					if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
					if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;

					return 1;
				}
			}
			stmt = stmt2;
		}

		stmt->setString(stmt, 1, sql_pr->relativeuri);

		/* 通常SQL、リポジトリ検索SQLのトップ階層の場合 */
		if (IS_EMPTY(list->ptsubname)) {
			stmt->setString(stmt, 2, sql_pr->relativeuri);
		}
		else {
			stmt->setString(stmt, 2, list->ptsubname);
		}

		stmt->setString(stmt, 3, list->clsubname);

		/* 実行 */
		insert_cnt = stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_sqldepend. "
				"(sqluri = %s, clsubname = %s) Reason: %s",
				sql_pr->relativeuri, list->clsubname,
				stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			/* (note) stmt はリファレンスなのでcloseしません */
			if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
			if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;

			return 1;
		}

		if (insert_cnt == 0) {
			  ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Status is OK, but the count of inserting is 0."
				"Please check divy_sqldepend." 
				"(sqluri = %s, clsubname = %s)",
				sql_pr->relativeuri, list->clsubname);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			/* (note) stmt はリファレンスなのでcloseしません */
			if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
			if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;

			return 1;
		}
	}
	/* (note) stmt はリファレンスなのでcloseしません */
	if (stmt1 != NULL) stmt1->close(stmt1); stmt1 = NULL;
	if (stmt2 != NULL) stmt2->close(stmt2); stmt2 = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * divy_sql テーブルの全エントリをロックします。
 * この関数はts_ctxとして与えられたトランザクションを引き継ぎ、
 * 最後にはトランザクションを"終了せず"に戻ります。
 * (note)
 * 	この関数は、SQLの新規追加・更新処理に先立ち、依存関係が途中で
 * 	壊れて壊れる可能性を防ぐ目的で使用します。
 * 	その他の用途には使用しないで下さい。
 *
 * @param r request_rec *
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (1: 異常終了 / 0: 成功)
 */
static int _lock_all_divy_sql(request_rec *r, divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない -> ここでは作成せずエラーにする */
	if (ts_ctx == NULL) {
	 	/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Could not operation. Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_sql テーブルの全ロック
	 * (note)
	 * 	テーブルレベルのロック構文はDBMSによってはサポート
	 * 	していない場合もあるため、ここではdivy_sqlテーブルの
	 * 	sql_active_iを同じ値でUPDATEすることによって代用します。
	 * 	sql_active_iを選択したのは、not nullであることと、integer
	 *	なので更新によってテーブルの断片化が起き難いからです。
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_sql "
				"SET sql_active_i = sql_active_i", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for update"
			"Reason: %s", stmt->getMsg(stmt));
		if (stmt != NULL) stmt->close(stmt);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		/* トランザクションを終了させてはならない !! */

		return 1;
	}

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to dummy-UPDATE divy_sql. "
			"Reason: %s", stmt->getMsg(stmt));
		if (stmt != NULL) stmt->close(stmt);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		/* トランザクションを終了させてはならない !! */

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt);

	/* トランザクションを終了させてはならない !! */
	return 0;
}

