/**
 * $Id$
 *
 * 特殊フォルダ「システムメッセージ」関連の関数をまとめたファイル
 *
 * (note)
 *   ここで定義された関数群はtf_rdbo.c にあったものです.
 *   tf_rdbo.c が巨大化し過ぎたため、関連のあるものをまとめました.
 */
#include "httpd.h"
#include "http_protocol.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"

#include "mod_dav_tf.h"
#include "tf_db.h"
#include "util_db.h"
#include "tf_rdbo.h"
#include "util.h"
#include "search.h"
#include "tf_sqlparser.h"
#include "tf_folder.h"
#include "tf_valuecache.h"
#include "tf_linkedlist.h"
#include "tf_rdbo_util.h"
#include "tf_rdbo.h"
#include "tf_rdbo_sysmsg.h"

/*------------------------------------------------------------------------------
  Define public functions 
  ----------------------------------------------------------------------------*/
/**
 * リポジトリDBを検索する(sysmsginformationsearch)
 * 
 */
DIVY_DECLARE(int) divy_rdbo_sysmsginformationsearch(divy_rdbo_search_params *params,
						divy_search_sysmsgis_iscreen *screen,
						divy_rdbo_search_output *output)
{
	request_rec *r          = params->r;
	apr_pool_t *p		= params->r->pool;
	apr_pool_t *wp          = params->wp;
	DbConn          *dbconn	= NULL;
	DbPreparedStmt  *stmt	= NULL;
	DbResultSet     *rset	= NULL;
	divy_db_transaction_ctx *ts_ctx   = NULL;
	char *sqlstr            = NULL;
	divy_rdbo_sysmsg *sysmsg = NULL;
	int support_extendsysmsg = divy_support_extendsysmsg(r);
	const char* lang         = divy_get_language_param(r);
	int idx=0;

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen is EMPTY.");
		return 1;
	}

	output->list.sysmsg_pr = NULL;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * SQL文の作成
	 */
	sqlstr =	"SELECT "
			" sysmsg.smsg_msg_id_i"
			", sysmsg.smsg_msg_txt"
			", sysmsg.smsg_update_c";

	/* extendsysmsg */
	if (support_extendsysmsg)
	{
		sqlstr = apr_pstrcat(wp, sqlstr,
					", sysmsg.smsg_lang_c"
					", sysmsg.smsg_scope_c"
					", sysmsg.smsg_open_bi"
					", sysmsg.smsg_close_bi"
					", sysmsg.smsg_subject_vc"
					", sysmsg.smsg_regist_c"
					, NULL);
	}

	/* currentmsg */
	if (params->optflg == DIVY_SEARCH_OPT_CURRENTMSG) {

		sqlstr = apr_pstrcat(wp, sqlstr,
					" FROM divy_sysmsg sysmsg"
					" WHERE sysmsg.SMSG_ACTIVE_I = 1",
					NULL);

		if (support_extendsysmsg) {
			sqlstr = apr_pstrcat(wp, sqlstr,
					" AND "
					"(sysmsg.smsg_lang_c = ?"
					" OR "
					"sysmsg.smsg_lang_c is null)", 
					NULL);
		}
	}
	/* detaillist */
	else if (params->optflg == DIVY_SEARCH_OPT_DETAILLIST) {
		sqlstr = apr_pstrcat(wp, sqlstr,
					", sysmsg.smsg_active_i"
					" FROM divy_sysmsg sysmsg",
					NULL);
	}
	/* content */
	else if (params->optflg == DIVY_SEARCH_OPT_CONTENT) {

		sqlstr = apr_pstrcat(wp, sqlstr,
					", sysmsg.smsg_active_i"
					" FROM divy_sysmsg sysmsg"
					" WHERE sysmsg.SMSG_MSG_ID_I = ?",
					NULL);
	}
	else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
			"Unknown optflg found. (code = %d)", params->optflg);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, sqlstr, wp);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. Reason: %s", 
			stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	idx = 1;
	if (params->optflg == DIVY_SEARCH_OPT_CURRENTMSG && support_extendsysmsg) {
		stmt->setString(stmt, idx++, lang);
	}

	if (params->optflg == DIVY_SEARCH_OPT_CONTENT) {
		stmt->setInt(stmt, idx++, screen->msgid);
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet. Reason: %s", 
			rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE) {
		idx = 1;
		/* 領域確保 */
		if (output->list.sysmsg_pr == NULL) {
			output->list.sysmsg_pr = sysmsg =
				apr_pcalloc(wp, sizeof(divy_rdbo_sysmsg));
		}
		else {
			sysmsg->next = apr_pcalloc(wp, sizeof(divy_rdbo_sysmsg));
			sysmsg = sysmsg->next;
		}
 
		/* データを取得 IDはintを変換 */
		sysmsg->msgid	 = apr_itoa(wp,	rset->getInt(rset, idx++));
		sysmsg->msg	 = rset->getString(rset, idx++);
		sysmsg->updatedt = rset->getString(rset, idx++);
		if (support_extendsysmsg) {
			sysmsg->lang = rset->getString(rset, idx++);
			sysmsg->scope = rset->getString(rset, idx++);
			sysmsg->opendt = rset->getBigInt(rset, idx++);
			sysmsg->closedt = rset->getBigInt(rset, idx++);
			sysmsg->subject = rset->getString(rset, idx++);
			sysmsg->registdt = rset->getString(rset, idx++);
		}

		/* currentmsg以外(detaillist,content)の場合はさらに取得 */
		if (params->optflg != DIVY_SEARCH_OPT_CURRENTMSG) {
			if (support_extendsysmsg) {
				sysmsg->active 	 = rset->getInt(rset, idx++);
			}
			else {
				sysmsg->active 	 = rset->getInt(rset, idx++);
			}
		}
	}

	divy_db_commit_transaction(ts_ctx);

	/* 使用した領域を解放 (dbconn は何もしてはならない) */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたmsgid が示すシステムメッセージを取得して返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_sysmsg_property(request_rec *r, const char *msgid,
						divy_rdbo_sysmsg **sysmsg_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int support_extendsysmsg = divy_support_extendsysmsg(r);
	divy_sbuf *sql_buf = NULL;

	*sysmsg_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(msgid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"msgid is empty.");
		return 1;
	}

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn.(msgid = %s)", msgid);
		return 1;
	}

	divy_sbuf_create(p, &sql_buf, 256); /* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
			"SELECT "
			"smsg_msg_id_i,"
			"smsg_msg_txt,"
			"smsg_active_i,"
			"smsg_regist_c,"
			"smsg_update_c ");

	if (support_extendsysmsg) {
		divy_sbuf_append(sql_buf,
			",smsg_lang_c "
			",smsg_scope_c "
			",smsg_open_bi "
			",smsg_close_bi "
			",smsg_subject_vc ");
	}

	divy_sbuf_append(sql_buf,
			"FROM divy_sysmsg "
			"WHERE smsg_msg_id_i = ?");

	dbconn->startTrans(dbconn, 0);
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (msgid = %s) "
			"Reason: %s", msgid, stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setInt(stmt, 1, atoi(msgid));

	/* sql実行　失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(msgid = %s) "
			"Reason: %s", msgid, rset->getMsg(rset));
		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* データがあれば取得(ない場合は初期化時のNULL) */
	if (rset->next(rset) == DB_TRUE) {
		*sysmsg_pr = apr_pcalloc(p, sizeof(divy_rdbo_sysmsg));
		/* データ格納 */
		(*sysmsg_pr)->msgid    = rset->getString(rset, 1);
		(*sysmsg_pr)->msg      = rset->getString(rset, 2);
		(*sysmsg_pr)->active   = rset->getInt(rset,    3);
		(*sysmsg_pr)->registdt = rset->getString(rset, 4);
		(*sysmsg_pr)->updatedt = rset->getString(rset, 5);

		if (support_extendsysmsg) {
			(*sysmsg_pr)->lang   = rset->getString(rset, 6);
			(*sysmsg_pr)->scope  = rset->getString(rset, 7);
			(*sysmsg_pr)->opendt = rset->getBigInt(rset, 8);
			(*sysmsg_pr)->closedt= rset->getBigInt(rset, 9);
			(*sysmsg_pr)->subject= rset->getString(rset,10);
		}

		/* 次のsysmsg_prはNULL */
		(*sysmsg_pr)->next     = NULL;
	}

	/* クローズ */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	dbconn->commit(dbconn);

	return 0;
}

/**
 * アクティブなシステムメッセージを取得する。
 * FIXME この関数はブラウザからメッセージを選択した時に利用されます。
 * 2.1.3から利用しなくなる可能性があります
 * TfExetendSysmsg onにすることによりここは利用されない
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_active_sysmsg(request_rec *r,
						divy_rdbo_sysmsg **sysmsg_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	divy_rdbo_sysmsg *sysmsg = NULL;
	divy_db_transaction_ctx *ts_ctx = NULL;
	int support_extendsysmsg = divy_support_extendsysmsg(r);
	// FIXME const char* lang         = divy_get_language_param(r);
	divy_sbuf *sql_buf;

	*sysmsg_pr = NULL;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	divy_sbuf_create(p, &sql_buf, 512); /* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
			"SELECT "
			"smsg_msg_id_i,"
			"smsg_msg_txt,"
			"smsg_active_i,"
			"smsg_regist_c,"
			"smsg_update_c ");

	if (support_extendsysmsg) {
		divy_sbuf_append(sql_buf,
			",smsg_lang_c"
			",smsg_scope_c "
			",smsg_open_bi "
			",smsg_close_bi "
			",smsg_subject_vc "
			);
	}

	divy_sbuf_append(sql_buf,
			"FROM divy_sysmsg "
			"WHERE smsg_active_i = 1 ");

	if (support_extendsysmsg) {
		apr_time_t now = apr_time_now();
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"time = %"APR_INT64_T_FMT"", now);
	}

	divy_sbuf_append(sql_buf,
			"ORDER BY smsg_update_c DESC");

	/*
	 * アクティブなメッセージを取得する
	 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* SQLの実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to execute select divy_sysmsg. "
			"Reason: %s", rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 結果を取得する */
	while (rset->next(rset) == DB_TRUE) {
		if (*sysmsg_pr == NULL) {
			sysmsg = *sysmsg_pr =
				apr_pcalloc(p, sizeof(divy_rdbo_sysmsg));
		}
		else {
			sysmsg->next = apr_pcalloc(p, sizeof(divy_rdbo_sysmsg));
			sysmsg = sysmsg->next;
		}

		sysmsg->msgid    = rset->getString(rset, 1);
		sysmsg->msg      = rset->getString(rset, 2);
		sysmsg->active   = rset->getInt(rset,    3);
		sysmsg->registdt = rset->getString(rset, 4);
		sysmsg->updatedt = rset->getString(rset, 5);

		if (support_extendsysmsg) {
			sysmsg->lang = rset->getString(rset, 6);
			sysmsg->scope= rset->getString(rset, 7);
			sysmsg->opendt=rset->getBigInt(rset, 8);
			sysmsg->closedt=rset->getBigInt(rset, 9);
			sysmsg->subject=rset->getString(rset,10);
		}
		sysmsg->next     = NULL;

	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset);	rset = NULL;
	if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

	return 0;
}

/**
 * アクティブなシステムメッセージ数を返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_count_active_sysmsg(request_rec *r, int *count)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;

	*count = 1;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * アクティブなメッセージを取得する
	 */
	stmt = dbconn->prepareStatement(dbconn, 
			"SELECT "
			"COUNT(smsg_msg_id_i) AS cnt "
			"FROM divy_sysmsg "
			"WHERE smsg_active_i = 1", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. "
			"Reason: %s", stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}


	/* SQLの実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to execute select divy_sysmsg. "
			"Reason: %s", rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 結果を取得する */
	if (rset->next(rset) == DB_TRUE) {
		*count = rset->getInt(rset, 1);
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset);	rset = NULL;
	if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

	return 0;
}

/**
 * 指定されたsysmsg_pr が表すシステムメッセージを新規登録する。
 * (note)
 * 同じタイムスタンプを持つシステムメッセージは登録できません。
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_sysmsg_property(request_rec *r,
					const divy_rdbo_sysmsg *sysmsg_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	time_t short_time       = dav_divy_get_now_epoch();
	char *datebuf           = NULL;
	divy_db_transaction_ctx *ts_ctx = NULL;
	int support_extendsysmsg= divy_support_extendsysmsg(r);
	divy_sbuf *sql_buf;
	int idx=0;

	TRACE(p);

	/* 必須項目のチェック */
	if (sysmsg_pr == NULL || sysmsg_pr->msgid == NULL || 
			(sysmsg_pr->active != 0 && sysmsg_pr->active != 1)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Could not operation. There are empty value.");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	if (dbconn == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbConn.(msgid = %s)", 
				sysmsg_pr->msgid);
		return 1;
	}

	divy_sbuf_create(p, &sql_buf, 1024); /* SQLバッファの作成 */
	/* SQL文の準備　失敗時はエラー */
	divy_sbuf_append(sql_buf,
			"INSERT INTO divy_sysmsg ("
			"smsg_msg_id_i,"
			"smsg_msg_txt,"
			"smsg_active_i,"
			"smsg_regist_c,"
			"smsg_update_c ");
	if (support_extendsysmsg) {
		divy_sbuf_append(sql_buf,
			",smsg_lang_c"
			",smsg_scope_c"
			",smsg_open_bi"
			",smsg_close_bi"
			",smsg_subject_vc");
	}

	divy_sbuf_append(sql_buf,
			")VALUES (?, ?, ?, ?, ?");

	if (support_extendsysmsg) {
		divy_sbuf_append(sql_buf, ",?,?,?,?,?");
	}

	divy_sbuf_append(sql_buf, ")");
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (msgid = %s) "
				"Reason: %s", 
				sysmsg_pr->msgid, stmt->getMsg(stmt));
		
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	idx=1;
	/* バインド */
	stmt->setInt(stmt,    idx++, atoi(sysmsg_pr->msgid));
	stmt->setString(stmt, idx++, REPLACE_EMPTYSTR(sysmsg_pr->msg));
	stmt->setInt(stmt,    idx++, sysmsg_pr->active);
	stmt->setString(stmt, idx++, datebuf);
	stmt->setString(stmt, idx++, datebuf);

	if (support_extendsysmsg) {
		stmt->setString(stmt, idx++, sysmsg_pr->lang);
		stmt->setString(stmt, idx++, sysmsg_pr->scope);
		stmt->setString(stmt, idx++, NULL); /* FIXME */
		stmt->setString(stmt, idx++, NULL); /* FIXME */
		stmt->setString(stmt, idx++, sysmsg_pr->subject);

	}

	/* SQL実行　失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_sysmsg.(msgid = %s) "
				"Reason: %s",
				sysmsg_pr->msgid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 変更を確定 */
	divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたsysmsg_pr が表す内容でシステムメッセージを更新する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_sysmsg_property(request_rec *r,
					const divy_rdbo_sysmsg *sysmsg_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit = 0;       /* トランザクションを確定するかどうか */
	time_t short_time       = dav_divy_get_now_epoch();
	char *datebuf           = NULL;
	int update_cnt;
	int support_extendsysmsg = divy_support_extendsysmsg(r);
	divy_sbuf *sql_buf;
	int idx=0;

	TRACE(p);
	/* 必須項目(UPDATEするデータのID,active)のチェック */
	if (sysmsg_pr == NULL || IS_EMPTY(sysmsg_pr->msgid) || 
			(sysmsg_pr->active != 0 && sysmsg_pr->active != 1)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Msgid or active is NULL. Could not do "
				"process !!");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Could not operation. Transaction_ctx is "
				"NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	divy_sbuf_create(p, &sql_buf, 1024);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
			"UPDATE divy_sysmsg SET "
			"smsg_msg_txt = ?, "
			"smsg_active_i = ?, "
			"smsg_update_c = ? ");

	if (support_extendsysmsg) {
		divy_sbuf_append(sql_buf,
			",smsg_lang_c = ? "
			",smsg_scope_c = ? "
			",smsg_open_bi = ? "
			",smsg_close_bi = ? "
			",smsg_subject_vc = ? ");
	}

	divy_sbuf_append(sql_buf, "WHERE smsg_msg_id_i = ?");
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (msgid = %s) "
				"Reason: %s",
				sysmsg_pr->msgid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}

	idx=1;
	/* バインド */
	stmt->setString(stmt, idx++, REPLACE_EMPTYSTR(sysmsg_pr->msg));
	stmt->setInt(stmt,    idx++, sysmsg_pr->active);
	stmt->setString(stmt, idx++, datebuf);

	if (support_extendsysmsg) {
		stmt->setString(stmt, idx++, sysmsg_pr->lang);
		stmt->setString(stmt, idx++, sysmsg_pr->scope);
		stmt->setBigInt(stmt, idx++, 0);
		stmt->setBigInt(stmt, idx++, 0);
		stmt->setString(stmt, idx++, REPLACE_EMPTYSTR(sysmsg_pr->subject));
		stmt->setInt(stmt, idx++, atoi(sysmsg_pr->msgid));

	}
	else {
		stmt->setInt(stmt,    idx++, atoi(sysmsg_pr->msgid));
	}

	/* SQL実行　失敗時はエラー */
	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_sysmsg.(msgid = %s) "
				"Reason: %s",
				sysmsg_pr->msgid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	/* 更新件数が1件以外はエラー */
	if (update_cnt != 1){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to update divy_sysmsg. data is "
				"not found.(msgid = %s)", sysmsg_pr->msgid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 変更を確定 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたsysmsg_pr が示すシステムメッセージを削除する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_sysmsg_property(request_rec *r,
					const divy_rdbo_sysmsg *sysmsg_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int delete_cnt		= 0;

	TRACE(p);

	/* 必須項目(DELETEするデータのID)のチェック */
	if (sysmsg_pr == NULL || IS_EMPTY(sysmsg_pr->msgid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"msgid is NULL.");
		return 1;
	}

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbConn.(msgid = %s)", 
				sysmsg_pr->msgid);
		return 1;
	}
	/* トランザクション開始 */
	dbconn->startTrans(dbconn, 0);

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_sysmsg "
			"WHERE smsg_msg_id_i = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (msgid = %s) "
				"Reason: %s", 
				sysmsg_pr->msgid, stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	/* バインド */
	stmt->setInt(stmt, 1, atoi(sysmsg_pr->msgid));
	/* SQL実行　失敗時はエラー */
	delete_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to delete divy_sysmsg.(msgid = %s) "
				"Reason: %s",
				sysmsg_pr->msgid, stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* データが既に削除されていた場合はwarningで継続 */
	if (delete_cnt == 0){
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The \"divy_sysmsg\" table did not have a recorde of "
			"this sysmsg.(msgid = %s) "
			"Probably data is already deleted. ", sysmsg_pr->msgid);
	}

	if (stmt != NULL) stmt->close(stmt);

	/* 変更を確定 */
	dbconn->commit(dbconn);
	return 0;
}

/**
 * 指定されたsrc_sysmsg_pr が示すシステムメッセージをdst_sysmsg_pr が示す
 * システムメッセージとしてコピーする。
 * 
 */
DIVY_DECLARE(int) divy_rdbo_copy_sysmsg_property(request_rec *r,
					const divy_rdbo_sysmsg *src_sysmsg_pr,
					const divy_rdbo_sysmsg *dst_sysmsg_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	time_t short_time       = dav_divy_get_now_epoch();
	char *datebuf           = NULL;
	int update_cnt;
	divy_sbuf *sql_buf      = NULL;
	int support_extendsysmsg = divy_support_extendsysmsg(r);

	TRACE(p);

	/* 必須項目のチェック */
	if (src_sysmsg_pr == NULL || dst_sysmsg_pr == NULL ||
	    IS_EMPTY(src_sysmsg_pr->msgid) || IS_EMPTY(dst_sysmsg_pr->msgid)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"There are empty values.");
		return 1;
	}

	/* リポジトリDB の DbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbConn.(srcmsgid = %s)",
				src_sysmsg_pr->msgid);
		return 1;
	}

	divy_sbuf_create(p, &sql_buf, 2048);
	divy_sbuf_append(sql_buf, 
			"INSERT INTO divy_sysmsg ("
			"smsg_msg_id_i,"
			"smsg_msg_txt,"
			"smsg_active_i,"
			"smsg_regist_c,"
			"smsg_update_c) "
			"(SELECT "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_INTEGER)","
			"smsg_msg_txt,"
			"smsg_active_i,"
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN))","
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN)) );

	if (support_extendsysmsg) {
		divy_sbuf_append(sql_buf, 
							"smsg_lang_c,"
							"smsg_scope_c,"
							"smsg_open_bi,"
							"smsg_close_bi,"
							"smsg_subject_vc");
	}

	divy_sbuf_append(sql_buf, 
			"FROM divy_sysmsg "
			"WHERE smsg_msg_id_i = ?)");

	/* トランザクション開始 */
	dbconn->startTrans(dbconn, 0);

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (srcmsgid = %s) "
				"Reason: %s",
				src_sysmsg_pr->msgid, stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		dbconn->rollback(dbconn);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setInt(stmt,    1, atoi(dst_sysmsg_pr->msgid));
	stmt->setString(stmt, 2, datebuf);
	stmt->setString(stmt, 3, datebuf);
	stmt->setInt(stmt,    4, atoi(src_sysmsg_pr->msgid));

	/* SQL実行　失敗時はエラー */
	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_sysmsg."
			"(src_msgid = %s, dst_msgid = %s) Reason: %s",
			src_sysmsg_pr->msgid, dst_sysmsg_pr->msgid,
			stmt->getMsg(stmt));
		dbconn->rollback(dbconn);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}
	/* 更新データが見つからなかった場合 */
	if (update_cnt == 0){
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to insert divy_sysmsg. data is not found."
			"(src_msgid = %s, dst_msgid = %s)",
			src_sysmsg_pr->msgid, dst_sysmsg_pr->msgid);
		dbconn->rollback(dbconn);
		if (stmt != NULL)   stmt->close(stmt);
		return 1;
	}

	/* 変更を確定 */
	dbconn->commit(dbconn);

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	return 0;
}

