/**
 * $Id$
 *
 * 特殊フォルダ「ユーザ」関連の関数をまとめたファイル
 *
 * (note)
 *   ここで定義された関数群はtf_rdbo.c にあったものです.
 *   tf_rdbo.c が巨大化し過ぎたため、関連のあるものをまとめました.
 */
#include "httpd.h"
#include "http_protocol.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"
#include "util_md5.h"

#include "mod_dav_tf.h"
#include "tf_db.h"
#include "util_db.h"
#include "util_ldap.h"
#include "tf_rdbo.h"
#include "util.h"
#include "search.h"
#include "tf_folder.h"
#include "tf_valuecache.h"
#include "tf_linkedlist.h"
#include "tf_rdbo_util.h"
#include "tf_rdbo.h"
#include "tf_rdbo_user.h"
#include "tf_rdbo_group.h"
#include "tf_confirmreading.h"
#include "tf_validator.h"

APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Declare private prototype functions 
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define public functions 
  ----------------------------------------------------------------------------*/
/**
 * 指定されたparams, screen によりユーザ情報をリポジトリDBから検索して返却する
 * (userinformationsearch のdetaillist)
 *
 */
DIVY_DECLARE(int) divy_rdbo_userinformation_detaillist(divy_rdbo_search_params *params,
						divy_search_useris_iscreen *screen,
						divy_rdbo_search_output *output)
{
	request_rec *r          = params->r;
	apr_pool_t *p		= params->r->pool;
	apr_pool_t *wp          = params->wp;
	apr_pool_t *scratchpool = params->scratchpool;
	DbConn *dbconn		= NULL;
	DbPreparedStmt *stmt	= NULL;
	DbResultSet *rset 	= NULL;
	char *swap_pw		= NULL;
	int support_extstatus   = divy_support_extenduserstatus(r);
	int support_groupleader = divy_support_groupleader(r);
#ifdef DIVY_SUPPORT_PASSPOLICY
	int support_passpolicy  = divy_support_passpolicy(r);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	int support_access_control = divy_support_access_control(r);
	divy_db_transaction_ctx *ts_ctx  = NULL;
	int ret;
	divy_rdbo_usr *usr = NULL, *prev = NULL;
	char *acuserid = NULL;
	int idx;
	divy_sbuf *sql_buf = NULL;
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	const char *own_userid = divy_get_userid(r);

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen is EMPTY.");
		return 1;
	}
    
	output->list.usr_pr = NULL;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文を組み立てる */
	divy_sbuf_create(wp, &sql_buf, 1024);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT"
				" usr.usr_usr_id_vc"
				", usr.usr_fullname_vc"
				", usr.usr_mailaddr_vc"
				", usr.usr_regist_c"
				", usr.usr_update_c"
				", usr.usr_admin_mod_i"
				", usr.usr_comment_vc"
				", usr.usr_last_access_bi");

	/* ユーザ拡張ステータスをサポートしていればカラムを追加 */
	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
				", usr.usr_expiration_bi"
				", usr.usr_extended_status_c");
	}

	/* グループ管理者機能をサポートしているか? */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				",usr.usr_owner_usr_id_vc"
				",usr.usr_maxcreation_i"
				",usr2.usr_fullname_vc AS ownername");
	}

	/* アクセス制御をサポートしているか? */
	if (support_access_control) {
		divy_sbuf_append(sql_buf,
				",usr.usr_allowhosts_vc");
	}

	divy_sbuf_append(sql_buf,
				", usr.usr_passwd_vc"
				", usrdq.usqt_used_st_bi"
				", usrdq.usqt_max_st_bi"
				", usrdq.usqt_used_res_bi"
				", usrdq.usqt_max_res_bi"
				", ua.usad_usr_id_vc"
				", ua.usad_deny_user"
				", ua.usad_deny_group"
				", ua.usad_deny_dblink"
				", ua.usad_deny_reposdblink"
				", ua.usad_deny_management"
				", ua.usad_deny_m_update"
				", ua.usad_deny_m_user"
				", ua.usad_deny_m_group"
				", ua.usad_deny_m_sql"
				", ua.usad_deny_m_status"
				", ua.usad_deny_m_msg"
				", ua.usad_deny_m_dbms"
				", ua.usad_deny_m_execsql");

#ifdef DIVY_SUPPORT_PASSPOLICY
	if (support_passpolicy) {
		divy_sbuf_append(sql_buf,
				", ps.ps_policy_id_i"
				", ps.ps_send_expiredmail_bi"
				", ps.ps_last_change_bi"
				", ps.ps_firstlogin_bi"
				", ps.ps_special_start_bi");
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	divy_sbuf_append(sql_buf,
				" FROM divy_usr usr"
				" INNER JOIN divy_usrdiskquota usrdq"
				" ON (usr.usr_usr_id_vc = usrdq.usqt_usr_id_vc)"
				" LEFT JOIN divy_usraccessdeny ua"
				" ON (usr.usr_usr_id_vc = ua.usad_usr_id_vc)");

#ifdef DIVY_SUPPORT_PASSPOLICY
	if (support_passpolicy) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_passpolicystatus ps"
				" ON usr.usr_usr_id_vc = ps.ps_usr_usr_id_vc ");
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usr usr2"
				" ON (usr.usr_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}

	/* グループリーダの場合の取得条件
	 *  * 自分自身
	 *  * 自身がオーナとなっているユーザ群
	 *  * グループフォルダに所属しているユーザ群
	 */
	if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus)) {
		divy_sbuf_append(sql_buf,
				" WHERE usr.usr_usr_id_vc = ?"
				" OR usr.usr_owner_usr_id_vc = ?"
				" OR usr.usr_usr_id_vc "
				" IN (SELECT gm.grpm_usr_id_vc FROM divy_grpmem gm "
				"     INNER JOIN divy_grp g"
				"     ON gm.grpm_grp_id_c = g.grp_grp_id_c"
				"     WHERE g.grp_owner_usr_id_vc = ?)");
	}

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), wp);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus)) {
		stmt->setString(stmt, 1, own_userid);
		stmt->setString(stmt, 2, own_userid);
		stmt->setString(stmt, 3, own_userid);
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	ret = 0;
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	if (params->is_streaming_mode) {
		/* scratchpool をプールとして設定する */
		divy_db_set_rset_outputpool(rset, scratchpool);
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE) {
		idx = 1;
		/* 領域確保 */
		if (output->list.usr_pr == NULL) {
			output->list.usr_pr = usr = apr_pcalloc(wp, sizeof(divy_rdbo_usr));
		}
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		else if (!params->is_streaming_mode)
#else	/* !DAV_SUPPORT_EXTENDED_SEARCH */
		else
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
		{
			usr->next = apr_pcalloc(wp, sizeof(divy_rdbo_usr));
			usr = usr->next;
		}

			/* データ格納(各オプション共通) */
		usr->usrid    = rset->getString(rset, idx++);
		usr->fullname = rset->getString(rset, idx++);
		usr->mailaddr	= rset->getString(rset, idx++);
		usr->registdt	= rset->getString(rset, idx++);
		usr->updatedt 	= rset->getString(rset, idx++);
		usr->adminmode 	= rset->getInt(rset, idx++);
		usr->comment	= rset->getString(rset, idx++);
		usr->lastaccess = rset->getBigInt(rset, idx++);
		if (support_extstatus) {
			usr->expiration	= (time_t) rset->getBigInt(rset, idx++);
			usr->extstatus	= divy_rdbo_parse_extstatus(scratchpool, rset->getString(rset, idx++), EXTSTATUS_TYPE_USR);

			/* システム実行権限を持つユーザはリスティングしてはならない */
			if (divy_rdbo_has_sysexec_privilege(usr->extstatus)) {
				if (prev == NULL) {
					output->list.usr_pr = usr = NULL;
				}
				else {
					prev->next = NULL;
				}
				continue;
			}
		}

		if (support_groupleader) {
			usr->ownerid         = rset->getString(rset, idx++);
			usr->maxusercreation = rset->getInt(rset, idx++);
			usr->ownername       = rset->getString(rset, idx++);

			/* 自身が管理しているユーザかどうか */
			usr->is_otheruser = divy_is_otheruser(r, usr->ownerid);
			usr->is_appointed_groupleader = 0;	/* 任命状態は不明 */
		}

		if (support_access_control) {
			usr->allowhosts      = rset->getString(rset, idx++);
		}

		usr->password = rset->getString(rset, idx++);
		usr->usedst   = rset->getBigInt(rset, idx++);
		usr->maxst    = rset->getBigInt(rset, idx++);
		usr->usedres  = rset->getBigInt(rset, idx++);
		usr->maxres   = rset->getBigInt(rset, idx++);

		/* アクセス拒否リスト */
		acuserid = rset->getString(rset, idx++);
		if (IS_FILLED(acuserid)) {
			usr->accessdeny = apr_pcalloc(scratchpool, sizeof(int) * DIVY_FOLDER_ID_END);
			usr->accessdeny[DIVY_FOLDER_ID_user]        = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_group]       = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_dblink]      = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_reposdblink] = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_management]  = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_update]    = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_user]      = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_group]     = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_sql]       = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_status]    = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_msg]       = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_dbms]      = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_execsql]   = rset->getInt(rset, idx++);
		}
		else {
			idx += 13;
		}
#ifdef DIVY_SUPPORT_PASSPOLICY
		if (support_passpolicy) {
 			usr->passpolicy_status = apr_pcalloc(scratchpool, sizeof(divy_rdbo_passpolicystatus));
			usr->passpolicy_status->policyid        = rset->getInt(rset, idx++);
			usr->passpolicy_status->usrid           = usr->usrid;
			usr->passpolicy_status->sendexpiredmail = (time_t) rset->getBigInt(rset, idx++);
			usr->passpolicy_status->lastchange      = (time_t) rset->getBigInt(rset, idx++);
			usr->passpolicy_status->firstlogin      = (time_t) rset->getBigInt(rset, idx++);
			usr->passpolicy_status->specialstart    = (time_t) rset->getBigInt(rset, idx++);
		}
		else {
			usr->passpolicy_status = NULL;
		}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

		/* パスワードは置き換えられては困るので、一時退避する */
		swap_pw = usr->password;

		/* LDAPの内容にリプレース処理をする */
		(void) divy_util_ldap_get_user_property(r, scratchpool, usr->usrid, usr);
		/* 一時退避したパスワードを書き戻す */
		usr->password = swap_pw;

		usr->next = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		/* ストリーミングモードであればレスポンス出力を実施する */
		if (params->is_streaming_mode) {
			ret = params->output_response(params, output);
			if (ret) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to output response.");
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (rset != NULL) rset->close(rset); rset = NULL;
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
		}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
		prev = usr;
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	/* 1個以上のレスポンス出力が正常終了していたら最後のタグを出力させる */
	if (ret == 0 && output->list.usr_pr != NULL && params->is_streaming_mode) {
		divy_rdbo_search_output last = { 0 };
		last.list.usr_pr = NULL;
		last.response_cnt = output->response_cnt;

		ret = params->output_response(params, &last);
		if (ret) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to output last response.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたparams, screen によりユーザ情報をリポジトリDBから検索して返却する.
 * (userinformationsearch のcontent)
 *
 */
DIVY_DECLARE(int) divy_rdbo_userinformation_content(divy_rdbo_search_params *params,
						divy_search_useris_iscreen *screen,
						divy_rdbo_search_output *output)
{
	request_rec *r          = params->r;
	apr_pool_t *p		= params->r->pool;
	apr_pool_t *wp          = params->wp;
	apr_pool_t *scratchpool = params->scratchpool;
	DbConn *dbconn		= NULL;
	DbPreparedStmt *stmt	= NULL;
	DbResultSet *rset 	= NULL;
	char *swap_pw		= NULL;
	int support_extstatus   = divy_support_extenduserstatus(r);
	int support_groupleader = divy_support_groupleader(r);
#ifdef DIVY_SUPPORT_PASSPOLICY
	int support_passpolicy  = divy_support_passpolicy(r);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	int support_access_control = divy_support_access_control(r);
	divy_db_transaction_ctx *ts_ctx  = NULL;
	int ret;
	divy_rdbo_usr *usr = NULL;
	char *acuserid = NULL;
	int idx;
	divy_sbuf *sql_buf = NULL;

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"screen is EMPTY.");
		return 1;
	}
    
	output->list.usr_pr = NULL;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文を組み立てる */
	divy_sbuf_create(wp, &sql_buf, 1024);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT"
				" usr.usr_usr_id_vc"
				", usr.usr_fullname_vc"
				", usr.usr_mailaddr_vc"
				", usr.usr_regist_c"
				", usr.usr_update_c"
				", usr.usr_admin_mod_i"
				", usr.usr_comment_vc"
				", usr.usr_last_access_bi");

	/* ユーザ拡張ステータスをサポートしていればカラムを追加 */
	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
				", usr.usr_expiration_bi"
				", usr.usr_extended_status_c");
	}

	/* グループ管理者機能をサポートしているか? */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				",usr.usr_owner_usr_id_vc"
				",usr.usr_maxcreation_i"
				",usr2.usr_fullname_vc AS ownername");
	}

	/* アクセス制御をサポートしているか? */
	if (support_access_control) {
		divy_sbuf_append(sql_buf,
				",usr.usr_allowhosts_vc");
	}

	divy_sbuf_append(sql_buf,
				", usr.usr_passwd_vc"
				", usrdq.usqt_used_st_bi"
				", usrdq.usqt_max_st_bi"
				", usrdq.usqt_used_res_bi"
				", usrdq.usqt_max_res_bi"
				", ua.usad_usr_id_vc"
				", ua.usad_deny_user"
				", ua.usad_deny_group"
				", ua.usad_deny_dblink"
				", ua.usad_deny_reposdblink"
				", ua.usad_deny_management"
				", ua.usad_deny_m_update"
				", ua.usad_deny_m_user"
				", ua.usad_deny_m_group"
				", ua.usad_deny_m_sql"
				", ua.usad_deny_m_status"
				", ua.usad_deny_m_msg"
				", ua.usad_deny_m_dbms"
				", ua.usad_deny_m_execsql");

#ifdef DIVY_SUPPORT_PASSPOLICY
	if (support_passpolicy) {
		divy_sbuf_append(sql_buf,
				", ps.ps_policy_id_i"
				", ps.ps_send_expiredmail_bi"
				", ps.ps_last_change_bi"
				", ps.ps_firstlogin_bi"
				", ps.ps_special_start_bi");
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	divy_sbuf_append(sql_buf,
				" FROM divy_usr usr"
				" INNER JOIN divy_usrdiskquota usrdq"
				" ON (usr.usr_usr_id_vc = usrdq.usqt_usr_id_vc)"
				" LEFT JOIN divy_usraccessdeny ua"
				" ON (usr.usr_usr_id_vc = ua.usad_usr_id_vc)");

#ifdef DIVY_SUPPORT_PASSPOLICY
	if (support_passpolicy) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_passpolicystatus ps"
				" ON usr.usr_usr_id_vc = ps.ps_usr_usr_id_vc ");
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usr usr2"
				" ON (usr.usr_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}
	divy_sbuf_append(sql_buf,
				" WHERE usr.usr_usr_id_vc = ?");

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), wp);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, screen->userid);

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. (userid = %s, Reason: %s)",
				screen->userid, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	ret = 0;
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	if (params->is_streaming_mode) {
		/* scratchpool をプールとして設定する */
		divy_db_set_rset_outputpool(rset, scratchpool);
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	if (rset->next(rset) == DB_TRUE) {
		idx = 1;
		/* 領域確保 */
		if (output->list.usr_pr == NULL) {
			output->list.usr_pr = usr = apr_pcalloc(wp, sizeof(divy_rdbo_usr));
			usr->next = NULL;
		}

		/* データ格納(各オプション共通) */
		usr->usrid    = rset->getString(rset, idx++);
		usr->fullname = rset->getString(rset, idx++);
		usr->mailaddr	= rset->getString(rset, idx++);
		usr->registdt	= rset->getString(rset, idx++);
		usr->updatedt 	= rset->getString(rset, idx++);
		usr->adminmode 	= rset->getInt(rset, idx++);
		usr->comment	= rset->getString(rset, idx++);
		usr->lastaccess = rset->getBigInt(rset, idx++);
		if (support_extstatus) {
			usr->expiration	= (time_t) rset->getBigInt(rset, idx++);
			usr->extstatus	= divy_rdbo_parse_extstatus(scratchpool, rset->getString(rset, idx++), EXTSTATUS_TYPE_USR);
		}

		if (support_groupleader) {
			usr->ownerid         = rset->getString(rset, idx++);
			usr->maxusercreation = rset->getInt(rset, idx++);
			usr->ownername       = rset->getString(rset, idx++);

			/* 自身が管理しているユーザかどうか */
			usr->is_otheruser = divy_is_otheruser(r, usr->ownerid);
			usr->is_appointed_groupleader = 0;	/* 任命状態は不明 */
		}

		if (support_access_control) {
			usr->allowhosts      = rset->getString(rset, idx++);
		}

		usr->password = rset->getString(rset, idx++);
		usr->usedst   = rset->getBigInt(rset, idx++);
		usr->maxst    = rset->getBigInt(rset, idx++);
		usr->usedres  = rset->getBigInt(rset, idx++);
		usr->maxres   = rset->getBigInt(rset, idx++);

		/* アクセス拒否リスト */
		acuserid = rset->getString(rset, idx++);
		if (IS_FILLED(acuserid)) {
			usr->accessdeny = apr_pcalloc(scratchpool, sizeof(int) * DIVY_FOLDER_ID_END);
			usr->accessdeny[DIVY_FOLDER_ID_user]        = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_group]       = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_dblink]      = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_reposdblink] = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_management]  = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_update]    = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_user]      = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_group]     = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_sql]       = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_status]    = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_msg]       = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_dbms]      = rset->getInt(rset, idx++);
			usr->accessdeny[DIVY_FOLDER_ID_m_execsql]   = rset->getInt(rset, idx++);
		}
		else {
			idx += 13;
		}
#ifdef DIVY_SUPPORT_PASSPOLICY
		if (support_passpolicy) {
 			usr->passpolicy_status = apr_pcalloc(scratchpool, sizeof(divy_rdbo_passpolicystatus));
			usr->passpolicy_status->policyid        = rset->getInt(rset, idx++);
			usr->passpolicy_status->usrid           = usr->usrid;
			usr->passpolicy_status->sendexpiredmail = (time_t) rset->getBigInt(rset, idx++);
			usr->passpolicy_status->lastchange      = (time_t) rset->getBigInt(rset, idx++);
			usr->passpolicy_status->firstlogin      = (time_t) rset->getBigInt(rset, idx++);
			usr->passpolicy_status->specialstart    = (time_t) rset->getBigInt(rset, idx++);
		}
		else {
			usr->passpolicy_status = NULL;
		}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

		/* パスワードは置き換えられては困るので、一時退避する */
		swap_pw = usr->password;

		/* LDAPの内容にリプレース処理をする */
		(void) divy_util_ldap_get_user_property(r, scratchpool, usr->usrid, usr);
		/* 一時退避したパスワードを書き戻す */
		usr->password = swap_pw;

		usr->next = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		/* ストリーミングモードであればレスポンス出力を実施する */
		if (params->is_streaming_mode) {
			ret = params->output_response(params, output);
			if (ret) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to output response.");
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (rset != NULL) rset->close(rset); rset = NULL;
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
		}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	/* 1個以上のレスポンス出力が正常終了していたら最後のタグを出力させる */
	if (ret == 0 && output->list.usr_pr != NULL && params->is_streaming_mode) {
		divy_rdbo_search_output last = { 0 };
		last.list.usr_pr = NULL;
		last.response_cnt = output->response_cnt;

		ret = params->output_response(params, &last);
		if (ret) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to output last response.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたparams, screen によりユーザのグループ所属情報
 * をリポジトリDBから検索して返却する(userinformationsearch の availablegroup)
 *
 */
DIVY_DECLARE(int) divy_rdbo_userinformation_availablegroup(divy_rdbo_search_params *params,
						divy_search_useris_iscreen *screen,
						divy_rdbo_search_output *output)
{
	request_rec *r          = params->r;
	apr_pool_t *p		= params->r->pool;
	apr_pool_t *wp          = params->wp;
	apr_pool_t *scratchpool = params->scratchpool;
	DbConn *dbconn		= NULL;
	DbPreparedStmt *stmt	= NULL;
	DbResultSet *rset 	= NULL;
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
	divy_db_transaction_ctx *ts_ctx  = NULL;
	int ret, idx;
	divy_rdbo_grp *grp = NULL;
	divy_rdbo_usr *usr = NULL;
	divy_sbuf *sql_buf = NULL;

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"screen is EMPTY.");
		return 1;
	}
    
	output->list.usr_pr = NULL;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * SQL文の作成
	 */
	/* (note) グループコレクションURIは操作ログの表示アクセス制御に利用します */
	divy_sbuf_create(wp, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT"
				" grpm.grpm_usr_id_vc"
				", grp.grp_grp_id_c"
				", grp.grp_name_vc"
				", grp.grp_regist_c"
				", grp.grp_update_c"
				", grp.grp_comment_vc"
				",rs.rs_uri_txt");

	/* グループ制約 / グループ管理者機能をサポートしていればカラムを追加 */
	if (support_grpconstraints || support_groupleader) {
		divy_sbuf_append(sql_buf,
				", grp.grp_extended_status_c");
	}

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				", grp.grp_owner_usr_id_vc"
				", usr2.usr_fullname_vc AS ownername");
	}

	divy_sbuf_append(sql_buf,
				" FROM divy_grpmem grpm"
				" INNER JOIN divy_grp grp"
				" ON (grpm.grpm_grp_id_c = grp.grp_grp_id_c)"
				" INNER JOIN dav_resource rs"
				" ON (grp.grp_rs_id_c = rs.rs_rs_id_c)");

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usr usr2"
				" ON (grp.grp_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}

	divy_sbuf_append(sql_buf,
				" WHERE grpm.grpm_usr_id_vc = ?");

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), wp);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, screen->userid);

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	ret = 0;
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	if (params->is_streaming_mode) {
		/* scratchpool をプールとして設定する */
		divy_db_set_rset_outputpool(rset, scratchpool);
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE) {
    
		/* 領域確保 */
		if (output->list.usr_pr == NULL) {
			/* (note) scratchpool からアロケートしてはならない */
			output->list.usr_pr = usr = apr_pcalloc(wp, sizeof(divy_rdbo_usr));
			output->list.usr_pr->grp_pr = grp = apr_pcalloc(wp, sizeof(divy_rdbo_grp));
		}
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		else if (!params->is_streaming_mode)
#else	/* !DAV_SUPPORT_EXTENDED_SEARCH */
		else
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
		{
			grp->next = apr_pcalloc(wp, sizeof(divy_rdbo_grp));
			grp = grp->next;
		}
    
		idx = 1;
		/* データ格納 */
		usr->usrid      = rset->getString(rset, idx++);
		grp->grpid      = rset->getString(rset, idx++);
		grp->name       = rset->getString(rset, idx++);
		grp->registdt 	= rset->getString(rset, idx++);
		grp->updatedt 	= rset->getString(rset, idx++);
		grp->comment 	= rset->getString(rset, idx++);
		grp->grpcol_uri = rset->getString(rset, idx++);
		if (support_grpconstraints || support_groupleader) {
			grp->grp_extstatus = divy_rdbo_parse_extstatus(scratchpool, rset->getString(rset, idx++), EXTSTATUS_TYPE_GRP);
			/* 値が取れなければデフォルト値を使用する(互換性) */
			if (grp->grp_extstatus == NULL) {
				grp->grp_extstatus = divy_rdbo_create_default_extstatus(scratchpool, EXTSTATUS_TYPE_GRP);
			}
		}
		else {
			/* サポートしていなければデフォルト値を使用(互換性) */
			grp->grp_extstatus = divy_rdbo_create_default_extstatus(scratchpool, EXTSTATUS_TYPE_GRP);
		}

		if (support_groupleader) {
			grp->ownerid   = rset->getString(rset, idx++);
			grp->ownername = rset->getString(rset, idx++);
		}

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		/* ストリーミングモードであればレスポンス出力を実施する */
		if (params->is_streaming_mode) {
			ret = params->output_response(params, output);
			if (ret) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to output response.");
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (rset != NULL) rset->close(rset); rset = NULL;
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
		}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	/* 1個以上のレスポンス出力が正常終了していたら最後のタグを出力させる */
	if (ret == 0 && output->list.usr_pr != NULL && params->is_streaming_mode) {
		divy_rdbo_search_output last = { 0 };
		last.list.usr_pr = NULL;
		last.response_cnt = output->response_cnt;

		ret = params->output_response(params, &last);
		if (ret) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to output last response.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたparams, screen によりユーザ情報をリポジトリDBから検索して返却する
 * (userinformationsearch のtreelist)
 *
 */
DIVY_DECLARE(int) divy_rdbo_userinformation_treelist(divy_rdbo_search_params *params,
						divy_search_useris_iscreen *screen,
						divy_rdbo_search_output *output)
{
	request_rec *r          = params->r;
	apr_pool_t *p			= params->r->pool;
	apr_pool_t *wp          = params->wp;
	apr_pool_t *scratchpool = params->scratchpool;
	DbConn *dbconn			= NULL;
	DbPreparedStmt *stmt	= NULL;
	DbResultSet *rset 		= NULL;
	divy_db_transaction_ctx *ts_ctx  = NULL;
	int ret;
	divy_rdbo_usr *usr = NULL;
	int idx;

	if (screen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"screen is EMPTY.");
		return 1;
	}
    
	output->list.usr_pr = NULL;	/* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
					"SELECT"
					" usr.usr_usr_id_vc"
					", usr.usr_fullname_vc"
					" FROM divy_usr usr", wp);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	ret = 0;
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	if (params->is_streaming_mode) {
		/* scratchpool をプールとして設定する */
		divy_db_set_rset_outputpool(rset, scratchpool);
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE) {
		idx = 1;
		/* 領域確保 */
		if (output->list.usr_pr == NULL) {
			output->list.usr_pr = usr = apr_pcalloc(wp, sizeof(divy_rdbo_usr));
		}
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		else if (!params->is_streaming_mode)
#else	/* !DAV_SUPPORT_EXTENDED_SEARCH */
		else
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
		{
			usr->next = apr_pcalloc(wp, sizeof(divy_rdbo_usr));
			usr = usr->next;
		}

			/* データ格納(各オプション共通) */
		usr->usrid    = rset->getString(rset, idx++);
		usr->fullname = rset->getString(rset, idx++);
		usr->next = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		/* ストリーミングモードであればレスポンス出力を実施する */
		if (params->is_streaming_mode) {
			ret = params->output_response(params, output);
			if (ret) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to output response.");
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (rset != NULL) rset->close(rset); rset = NULL;
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
		}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	/* 1個以上のレスポンス出力が正常終了していたら最後のタグを出力させる */
	if (ret == 0 && output->list.usr_pr != NULL && params->is_streaming_mode) {
		divy_rdbo_search_output last = { 0 };
		last.list.usr_pr = NULL;
		last.response_cnt = output->response_cnt;

		ret = params->output_response(params, &last);
		if (ret) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to output last response.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	divy_db_commit_transaction(ts_ctx);

	return 0;
}
 
/**
 * リポジトリDBを検索する(statusinformationsearch)
 * 
 */
DIVY_DECLARE(int) divy_rdbo_statusinformationsearch(divy_rdbo_search_params *params,
							divy_rdbo_search_output *output)
{
	request_rec *r          = params->r;
	apr_pool_t *p           = params->r->pool;
	apr_pool_t *wp          = params->wp;
	apr_pool_t *scratchpool = params->scratchpool;
	DbConn *dbconn	    	= NULL;
	DbPreparedStmt *stmt	= NULL;
	DbResultSet *rset	    = NULL;
	divy_rdbo_usr *usr      = NULL, *prev = NULL;
	int support_extstatus   = divy_support_extenduserstatus(r);
	int support_groupleader = divy_support_groupleader(r);
	divy_db_transaction_ctx *ts_ctx  = NULL;
	int ret, idx;
#ifdef DIVY_SUPPORT_PASSPOLICY
	int support_passpolicy  = divy_support_passpolicy(r);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	divy_sbuf *sql_buf = NULL;
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	const char *own_userid = divy_get_userid(r);

	output->list.usr_pr = NULL;  /* 初期化 */

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文を組み立てる */
	divy_sbuf_create(wp, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT "
				" usr.usr_usr_id_vc"
				", usr.usr_fullname_vc"
				", usrdq.usqt_used_st_bi"
				", usrdq.usqt_max_st_bi"
				", usrdq.usqt_used_res_bi"
				", usrdq.usqt_max_res_bi"
				", usr.usr_last_access_bi"
				", usr.usr_last_accesscl_vc"
				", usr.usr_mailaddr_vc"
				", usr.usr_comment_vc"
				", usr.usr_regist_c"
				", usr.usr_update_c"
				", usr.usr_admin_mod_i");

	/* ユーザ拡張ステータスをサポートしていればカラムを追加 */
	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
				", usr.usr_expiration_bi"
				", usr.usr_extended_status_c");
	}

	/* グループ管理者機能をサポートしているか? */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				", usr.usr_owner_usr_id_vc"
				", usr.usr_maxcreation_i"
				",usr2.usr_fullname_vc AS ownername");
	}

#ifdef DIVY_SUPPORT_PASSPOLICY
	if (support_passpolicy) {
		divy_sbuf_append(sql_buf,
				", ps.ps_policy_id_i"
				", ps.ps_send_expiredmail_bi"
				", ps.ps_last_change_bi"
				", ps.ps_firstlogin_bi"
				", ps.ps_special_start_bi");
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	divy_sbuf_append(sql_buf,
				" FROM divy_usr usr"
				" INNER JOIN divy_usrdiskquota usrdq"
				" ON (usr.usr_usr_id_vc = usrdq.usqt_usr_id_vc)");

#ifdef DIVY_SUPPORT_PASSPOLICY
	if (support_passpolicy) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_passpolicystatus ps"
				" ON usr.usr_usr_id_vc = ps.ps_usr_usr_id_vc "
				);
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usr usr2"
				" ON (usr.usr_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}

	/* グループリーダの場合の取得条件
	 *  * 自分自身
	 *  * 自身がオーナとなっているユーザ群
	 *  * グループフォルダに所属しているユーザ群
	 */
	if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus)) {
		divy_sbuf_append(sql_buf,
				" WHERE usr.usr_usr_id_vc = ?"
				" OR usr.usr_owner_usr_id_vc = ?"
				" OR usr.usr_usr_id_vc "
				" IN (SELECT gm.grpm_usr_id_vc FROM divy_grpmem gm "
				"     INNER JOIN divy_grp g"
				"     ON gm.grpm_grp_id_c = g.grp_grp_id_c"
				"     WHERE g.grp_owner_usr_id_vc = ?)");
	}

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), wp);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus)) {
		stmt->setString(stmt, 1, own_userid);
		stmt->setString(stmt, 2, own_userid);
		stmt->setString(stmt, 3, own_userid);
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, wp);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	if (params->is_streaming_mode) {
		/* scratchpool をプールとして設定する */
		divy_db_set_rset_outputpool(rset, scratchpool);
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */

	ret = 0; prev = NULL;
	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE) {
		/* 領域確保 */
		if (output->list.usr_pr == NULL) {
			/* (note) scratchpool からアロケートしてはならない */
			output->list.usr_pr = usr = apr_pcalloc(wp, sizeof(divy_rdbo_usr));
		}
#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		else if (!params->is_streaming_mode)
#else	/* !DAV_SUPPORT_EXTENDED_SEARCH */
		else
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
		{
			usr->next = apr_pcalloc(wp, sizeof(divy_rdbo_usr));
			usr 	  = usr->next;
		}
		idx = 1;

		/* データを取得 */
		usr->usrid        = rset->getString(rset, idx++);
		usr->fullname     = rset->getString(rset, idx++);
		usr->usedst       = rset->getBigInt(rset, idx++);
		usr->maxst        = rset->getBigInt(rset, idx++);
		usr->usedres      = rset->getBigInt(rset, idx++);
		usr->maxres       = rset->getBigInt(rset, idx++);
		usr->lastaccess	  = (time_t) rset->getBigInt(rset, idx++);
		usr->lastaccesscl = rset->getString(rset, idx++);
		usr->mailaddr     = rset->getString(rset, idx++);
		usr->comment      = rset->getString(rset,idx++);
		usr->registdt     = rset->getString(rset,idx++);
		usr->updatedt     = rset->getString(rset,idx++);
		usr->adminmode 	  = rset->getInt(rset, idx++);

		if (support_extstatus) {
			usr->expiration = (time_t) rset->getBigInt(rset, idx++);
			usr->extstatus  = divy_rdbo_parse_extstatus(scratchpool, rset->getString(rset,idx++), EXTSTATUS_TYPE_USR);
			/* システム実行権限を持つユーザはリスティングしてはならない */
			if (divy_rdbo_has_sysexec_privilege(usr->extstatus)) {
				if (prev == NULL) {
					output->list.usr_pr = usr = NULL;
				}
				else {
					prev->next = NULL;
				}
				continue;
			}
		}

		if (support_groupleader) {
			usr->ownerid         = rset->getString(rset, idx++);
			usr->maxusercreation = rset->getInt(rset, idx++);
			usr->ownername       = rset->getString(rset, idx++);

			/* 自身が管理しているユーザかどうか */
			usr->is_otheruser = divy_is_otheruser(r, usr->ownerid);
			usr->is_appointed_groupleader = 0;	/* 任命状態は不明 */
		}

#ifdef DIVY_SUPPORT_PASSPOLICY
		if (support_passpolicy) {
	 		usr->passpolicy_status = apr_pcalloc(scratchpool,
											sizeof(divy_rdbo_passpolicystatus));
			usr->passpolicy_status->policyid        = rset->getInt(rset, idx++);
			usr->passpolicy_status->usrid           = usr->usrid;
			usr->passpolicy_status->sendexpiredmail = (time_t) rset->getBigInt(rset, idx++);
			usr->passpolicy_status->lastchange      = (time_t) rset->getBigInt(rset, idx++);
			usr->passpolicy_status->firstlogin      = (time_t) rset->getBigInt(rset, idx++);
			usr->passpolicy_status->specialstart    = (time_t) rset->getBigInt(rset, idx++);
		}
		else {
			usr->passpolicy_status = NULL;
		}
#endif	/* DIVY_SUPPORT_PASSPOLICY */
		/* LDAPにユーザ名を問い合わせる */
		(void) divy_util_ldap_get_user_property(r, scratchpool, usr->usrid, usr);

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
		/* ストリーミングモードであればレスポンス出力を実施する */
		if (params->is_streaming_mode) {
			ret = params->output_response(params, output);
			if (ret) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to output response.");
				ts_ctx->status |= DIVY_TRANS_ABORT;
				divy_db_rollback_transaction(ts_ctx);
				if (rset != NULL) rset->close(rset); rset = NULL;
				if (stmt != NULL) stmt->close(stmt); stmt = NULL;
				return 1;
			}
		}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
		prev = usr;
	}
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DAV_SUPPORT_EXTENDED_SEARCH
	/* 1個以上のレスポンス出力が正常終了していたら最後のタグを出力させる */
	if (ret == 0 && output->list.usr_pr != NULL && params->is_streaming_mode) {
		divy_rdbo_search_output last = { 0 };
		last.list.usr_pr  = NULL;
		last.response_cnt = output->response_cnt;

		ret = params->output_response(params, &last);
		if (ret) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to output last response.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
#endif	/* DAV_SUPPORT_EXTENDED_SEARCH */
	divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 指定されたuserid が示すユーザ情報を取得して返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_user_property(request_rec *r,
						const char *userid,
						divy_rdbo_usr **usr_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int support_extstatus   = divy_support_extenduserstatus(r);
	int support_groupleader = divy_support_groupleader(r);
#ifdef DIVY_SUPPORT_PASSPOLICY
	int support_passpolicy  = divy_support_passpolicy(r);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	int support_access_control = divy_support_access_control(r);
	char *sql_base_stmt     =
			"SELECT "
			"u.usr_usr_id_vc,"
			"u.usr_passwd_vc,"
			"u.usr_fullname_vc,"
			"u.usr_mailaddr_vc,"
			"u.usr_admin_mod_i,"
			"u.usr_rs_id_c,"
			"u.usr_usr_seq_i,"
			"u.usr_last_access_bi,"
			"u.usr_last_accesscl_vc,"
			"u.usr_regist_c,"
			"u.usr_update_c,"
			"u.usr_comment_vc,"
			"q.usqt_used_st_bi,"
			"q.usqt_max_st_bi,"
			"q.usqt_used_res_bi,"
			"q.usqt_max_res_bi,"
			"r.rs_uri_txt,"
			"ua.usad_usr_id_vc,"
			"ua.usad_deny_user,"
			"ua.usad_deny_group,"
			"ua.usad_deny_dblink,"
			"ua.usad_deny_reposdblink,"
			"ua.usad_deny_management,"
			"ua.usad_deny_m_update,"
			"ua.usad_deny_m_user,"
			"ua.usad_deny_m_group,"
			"ua.usad_deny_m_sql,"
			"ua.usad_deny_m_status,"
			"ua.usad_deny_m_msg,"
			"ua.usad_deny_m_dbms,"
			"ua.usad_deny_m_execsql ";
	divy_sbuf *sql_buf = NULL;
	divy_db_transaction_ctx *ts_ctx = NULL;

	*usr_pr = NULL;	/* 初期化 */

	if (IS_EMPTY(userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"userid is empty.");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文を組み立てる */
	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf, sql_base_stmt);

	/* 拡張ユーザステータスをサポートしているか? */
	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
						",u.usr_expiration_bi"
						",u.usr_extended_status_c ");
	}

	/* グループ管理者機能をサポートしているか? */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
						",u.usr_owner_usr_id_vc"
						",u.usr_maxcreation_i "
						",usr2.usr_fullname_vc AS ownername");
	}

	/* アクセス制御をサポートしているか? */
	if (support_access_control) {
		divy_sbuf_append(sql_buf,
						",u.usr_allowhosts_vc");
	}

#ifdef DIVY_SUPPORT_PASSPOLICY
	if (support_passpolicy) {
		divy_sbuf_append(sql_buf,
						",ps.ps_policy_id_i"
						",ps.ps_send_expiredmail_bi"
						",ps.ps_last_change_bi"
						",ps.ps_firstlogin_bi"
						",ps.ps_special_start_bi");
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	divy_sbuf_append(sql_buf,
					" FROM divy_usr u"
					" INNER JOIN divy_usrdiskquota q"
					" ON u.usr_usr_id_vc = q.usqt_usr_id_vc"
					" INNER JOIN dav_resource r"
					" ON u.usr_rs_id_c = r.rs_rs_id_c"
					" LEFT JOIN divy_usraccessdeny ua"
					" ON u.usr_usr_id_vc = ua.usad_usr_id_vc");

#ifdef DIVY_SUPPORT_PASSPOLICY
	if (support_passpolicy) {
		divy_sbuf_append(sql_buf,
						" LEFT JOIN divy_passpolicystatus ps"
						" ON u.usr_usr_id_vc = ps.ps_usr_usr_id_vc ");
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
						" LEFT JOIN divy_usr usr2"
						" ON (u.usr_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}

	divy_sbuf_append(sql_buf,
						" WHERE u.usr_usr_id_vc = ?");

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (userid = %s) "
				"Reason: %s", userid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	stmt->setString(stmt, 1, userid);
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet.(userid = %s) "
			"Reason: %s", userid, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
#ifdef DIVY_SUPPORT_PASSPOLICY
		char *policyid;
#endif	/* DIVY_SUPPORT_PASSPOLICY */
		char *acuserid  = NULL;
		int idx = 0;
		*usr_pr = apr_pcalloc(p, sizeof(divy_rdbo_usr));

		(*usr_pr)->usrid      = rset->getString(rset, ++idx);
		(*usr_pr)->password   = rset->getString(rset, ++idx);
		(*usr_pr)->fullname   = rset->getString(rset, ++idx);
		(*usr_pr)->mailaddr   = rset->getString(rset, ++idx);
		(*usr_pr)->adminmode  = rset->getInt(rset,    ++idx);
		(*usr_pr)->rsid       = rset->getString(rset, ++idx);
		(*usr_pr)->usrseq     = rset->getInt(rset,    ++idx);
		(*usr_pr)->lastaccess = (time_t) rset->getBigInt(rset, ++idx);
		(*usr_pr)->lastaccesscl=rset->getString(rset, ++idx);
		(*usr_pr)->registdt   = rset->getString(rset, ++idx);
		(*usr_pr)->updatedt   = rset->getString(rset, ++idx);
		(*usr_pr)->comment    = rset->getString(rset, ++idx);
		(*usr_pr)->usedst     = rset->getBigInt(rset, ++idx);
		(*usr_pr)->maxst      = rset->getBigInt(rset, ++idx);
		(*usr_pr)->usedres    = rset->getBigInt(rset, ++idx);
		(*usr_pr)->maxres     = rset->getBigInt(rset, ++idx);
		(*usr_pr)->prvcol_uri = rset->getString(rset, ++idx);

		acuserid = rset->getString(rset, ++idx); 
		if (IS_FILLED(acuserid)) {
			(*usr_pr)->accessdeny = apr_pcalloc(p, sizeof(int) * DIVY_FOLDER_ID_END);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_user]        = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_group]       = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_dblink]      = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_reposdblink] = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_management]  = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_m_update]    = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_m_user]      = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_m_group]     = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_m_sql]       = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_m_status]    = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_m_msg]       = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_m_dbms]      = rset->getInt(rset, ++idx);
			(*usr_pr)->accessdeny[DIVY_FOLDER_ID_m_execsql]   = rset->getInt(rset, ++idx);
		}
		else {
			idx += 13;
		}

		if (support_extstatus) {
			(*usr_pr)->expiration = (time_t) rset->getBigInt(rset, ++idx);
			(*usr_pr)->extstatus = divy_rdbo_parse_extstatus(p, rset->getString(rset, ++idx), EXTSTATUS_TYPE_USR);
			/* 値が不正だったらデフォルト値を使用(互換性) */
			if ((*usr_pr)->extstatus == NULL) {
				(*usr_pr)->extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_USR);
			}
		}
		else {
			/* サポートしていなければデフォルト値を使用(互換性) */
			(*usr_pr)->extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_USR);
		}

		if (support_groupleader) {
			(*usr_pr)->ownerid         = rset->getString(rset, ++idx);
			(*usr_pr)->maxusercreation = rset->getInt(rset, ++idx);
			(*usr_pr)->ownername       = rset->getString(rset, idx++);

			/* 自身が管理しているユーザかどうか */
			(*usr_pr)->is_otheruser = divy_is_otheruser(r, (*usr_pr)->ownerid);
			(*usr_pr)->is_appointed_groupleader = 0;	/* 任命状態は不明 */
		}

		if (support_access_control) {
			(*usr_pr)->allowhosts      = rset->getString(rset, ++idx);
		}

#ifdef DIVY_SUPPORT_PASSPOLICY
		if (support_passpolicy) {
			/* パスワードポリシー状態があれば取得する */
			policyid = rset->getString(rset, ++idx);
			if (IS_FILLED(policyid)) {
				(*usr_pr)->passpolicy_status = apr_pcalloc(p, sizeof(divy_rdbo_passpolicystatus));
				(*usr_pr)->passpolicy_status->policyid        = atoi(policyid);
				(*usr_pr)->passpolicy_status->usrid           = (*usr_pr)->usrid;
				(*usr_pr)->passpolicy_status->sendexpiredmail = (time_t) rset->getBigInt(rset, ++idx);
				(*usr_pr)->passpolicy_status->lastchange      = (time_t) rset->getBigInt(rset, ++idx);
				(*usr_pr)->passpolicy_status->firstlogin      = (time_t) rset->getBigInt(rset, ++idx);
				(*usr_pr)->passpolicy_status->specialstart    = (time_t) rset->getBigInt(rset, ++idx);
			}
		}
		else {
			(*usr_pr)->passpolicy_status = NULL;
		}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

		(*usr_pr)->next = NULL;
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 自分自身のユーザ情報を取得する (auth.c 用)
 *
 * @param r request_rec *
 * @param force int	0: r->notesにユーザ情報がセットされていれば終了
 * 			1: 強制的にユーザ情報を取得する
 * @return int 0: 成功 / 1: 失敗 / DIVY_STCODE_USER_NOT_EXIST: ユーザが存在しない
 *
 */
DIVY_DECLARE(int) divy_rdbo_cache_userinfo(request_rec *r, int force)
{

	DbConn *dbconn       = NULL;
	DbPreparedStmt *stmt = NULL;
	DbResultSet *rset    = NULL;
	apr_pool_t *p        = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;
	dav_divy_dir_conf *conf;
	int support_extstatus;
	char *sql_base_stmt     =
				"SELECT "
				" u.usr_usr_id_vc"
				",u.usr_passwd_vc"
				",u.usr_fullname_vc"
				",u.usr_mailaddr_vc"
				",u.usr_comment_vc"
				",u.usr_admin_mod_i"
				",u.usr_rs_id_c"
				",u.usr_usr_seq_i"
				",u.usr_last_access_bi"
				",u.usr_last_accesscl_vc"
				",u.usr_regist_c"
				",u.usr_update_c"
				",ua.usad_usr_id_vc"
				",ua.usad_deny_user"
				",ua.usad_deny_group"
				",ua.usad_deny_dblink"
				",ua.usad_deny_reposdblink"
				",ua.usad_deny_management"
				",ua.usad_deny_m_update"
				",ua.usad_deny_m_user"
				",ua.usad_deny_m_group"
				",ua.usad_deny_m_sql"
				",ua.usad_deny_m_status"
				",ua.usad_deny_m_msg"
				",ua.usad_deny_m_dbms"
				",ua.usad_deny_m_execsql"
				",usrdq.usqt_used_st_bi"
				",usrdq.usqt_max_st_bi"
				",usrdq.usqt_used_res_bi"
				",usrdq.usqt_max_res_bi";
	divy_sbuf *sql_buf = NULL;
	divy_rdbo_usr *usr_pr = NULL;
#ifdef DIVY_SUPPORT_PASSPOLICY
	int support_passpolicy  = divy_support_passpolicy(r);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
#ifdef DIVY_SUPPORT_GROUPQUOTA
	int support_groupquota     = divy_support_groupquota(r);
#endif /* DIVY_SUPPORT_GROUPQUOTA */
	int support_access_control = divy_support_access_control(r);
	int support_loginlockout = divy_support_failedlogin_lockout(r);
	divy_rdbo_grp *grp_pr = NULL;
	apr_hash_t *grp_pr_h = NULL;
	int idx;

	/* 一度通ったことがあった場合(所属グループ情報は特に見ません) */
	if (force == 0 &&
			(usr_pr = divy_pcache_get_data(r->pool, DIVY_PCACHE_DAT_REQ_USERINFO)) != NULL) {
		return 0;
	}
	conf = dav_divy_get_dir_config(r);
	support_extstatus = divy_support_extenduserstatus(r);

	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の組み立て */
	divy_sbuf_create(p, &sql_buf, 1024);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf, sql_base_stmt);

	/* 拡張ユーザステータスをサポートしているか? */
	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
						",u.usr_expiration_bi"
						",u.usr_extended_status_c");
	}

	/* グループ管理者機能をサポートしているか? */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
						",u.usr_owner_usr_id_vc"
						",u.usr_maxcreation_i "
						",usr2.usr_fullname_vc AS ownername");
	}

	/* アクセス制御機能をサポートしているか? */
	if (support_access_control) {
		divy_sbuf_append(sql_buf,
						",u.usr_allowhosts_vc");
	}

	/* ロックアウト機能をサポートしているか? */
	if (support_loginlockout) {
		divy_sbuf_append(sql_buf,
						",u.usr_failed_access_count_i");
	}

#ifdef DIVY_SUPPORT_PASSPOLICY
	if (support_passpolicy) {
		divy_sbuf_append(sql_buf,
						",ps.ps_policy_id_i"
						",ps.ps_send_expiredmail_bi"
						",ps.ps_last_change_bi"
						",ps.ps_firstlogin_bi"
						",ps.ps_special_start_bi");
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	divy_sbuf_append(sql_buf,
					" FROM divy_usr u"
					" INNER JOIN divy_usrdiskquota usrdq"
					" ON u.usr_usr_id_vc = usrdq.usqt_usr_id_vc"
					" LEFT JOIN divy_usraccessdeny ua"
					" ON u.usr_usr_id_vc = ua.usad_usr_id_vc");

#ifdef DIVY_SUPPORT_PASSPOLICY
	if (support_passpolicy) {
		divy_sbuf_append(sql_buf,
						" LEFT JOIN divy_passpolicystatus ps"
						" ON u.usr_usr_id_vc = ps.ps_usr_usr_id_vc");
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
						" LEFT JOIN divy_usr usr2"
						" ON (u.usr_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}
	divy_sbuf_append(sql_buf, " WHERE u.usr_usr_id_vc = ?");

	/* ユーザ情報の取得 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL )  stmt->close(stmt);
		return 1;
	}

	stmt->setString(stmt, 1, r->user);

	rset = stmt->executeQuery(stmt, p);
	/*
	 * ユーザが存在しなかった場合でLDAPあったら作成フラグが
	 * 設定されていた場合にはエラーコードを変化させる
	 * この時点ではLDAPが利用可なのかのチェックは行わない
	 */

	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet.Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/*
	 * ユーザIDで複数人いるはずが無い為、一度だけFETCH
	 */
	if (rset->next(rset) == DB_TRUE) {
		char *acuserid  = NULL;
#ifdef DIVY_SUPPORT_PASSPOLICY
		char *policyid;
#endif	/* DIVY_SUPPORT_PASSPOLICY */

		usr_pr = apr_pcalloc(p, sizeof(divy_rdbo_usr));

		idx = 0;
		usr_pr->usrid        = rset->getString(rset, ++idx);
		usr_pr->password     = rset->getString(rset, ++idx);
		usr_pr->fullname     = rset->getString(rset, ++idx);
		usr_pr->mailaddr     = rset->getString(rset, ++idx);
		usr_pr->comment      = rset->getString(rset, ++idx);
		usr_pr->adminmode    = rset->getInt(rset,    ++idx);
		usr_pr->rsid         = rset->getString(rset, ++idx);
		usr_pr->usrseq       = rset->getInt(rset,    ++idx);
		usr_pr->lastaccess   = (time_t) rset->getBigInt(rset, ++idx);
		usr_pr->lastaccesscl = rset->getString(rset, ++idx);
		usr_pr->registdt     = rset->getString(rset, ++idx);
		usr_pr->updatedt     = rset->getString(rset, ++idx);

		/* アクセス拒否リストは存在するか？ */
		acuserid = rset->getString(rset, ++idx); 
		if (IS_FILLED(acuserid)) {
			usr_pr->accessdeny = apr_pcalloc(p, sizeof(int) * DIVY_FOLDER_ID_END);
			usr_pr->accessdeny[DIVY_FOLDER_ID_user]        = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_group]       = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_dblink]      = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_reposdblink] = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_management]  = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_m_update]    = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_m_user]      = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_m_group]     = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_m_sql]       = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_m_status]    = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_m_msg]       = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_m_dbms]      = rset->getInt(rset, ++idx);
			usr_pr->accessdeny[DIVY_FOLDER_ID_m_execsql]   = rset->getInt(rset, ++idx);
		}
		else {
			idx += 13;
		}
		/* Quotaの取得 */
		usr_pr->usedst  = rset->getBigInt(rset, ++idx);
		usr_pr->maxst   = rset->getBigInt(rset, ++idx);
		usr_pr->usedres = rset->getBigInt(rset, ++idx);
		usr_pr->maxres  = rset->getBigInt(rset, ++idx);

		if (support_extstatus) {
			usr_pr->expiration = (time_t) rset->getBigInt(rset, ++idx);
			usr_pr->extstatus  = divy_rdbo_parse_extstatus(p, rset->getString(rset, ++idx), EXTSTATUS_TYPE_USR);
			/* 値が不正だったらデフォルト値を使用(互換性) */
			if (usr_pr->extstatus == NULL) {
				usr_pr->extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_USR);
			}
		}
		else {
			/* サポートしていなければデフォルト値を使用(互換性) */
			usr_pr->extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_USR);
		}

		if (support_groupleader) {
			usr_pr->ownerid         = rset->getString(rset, ++idx);
			usr_pr->maxusercreation = rset->getInt(rset, ++idx);
			usr_pr->ownername       = rset->getString(rset, ++idx);

			/* 自身が管理しているユーザかどうか */
			usr_pr->is_otheruser = divy_is_otheruser(r, usr_pr->ownerid);
			usr_pr->is_appointed_groupleader = 0;	/* 任命状態は不明 */
		}

		if (support_access_control) {
			usr_pr->allowhosts      = rset->getString(rset, ++idx);
		}
		if (support_loginlockout) {
			usr_pr->loginfailedcount = rset->getInt(rset, ++idx);
		}

#ifdef DIVY_SUPPORT_PASSPOLICY
		if (support_passpolicy) {
			/* パスワードポリシー状態があれば取得する */
			policyid = rset->getString(rset, ++idx);
			if (IS_FILLED(policyid)) {
				usr_pr->passpolicy_status = apr_pcalloc(p, sizeof(divy_rdbo_passpolicystatus));
				usr_pr->passpolicy_status->policyid        = atoi(policyid);
				usr_pr->passpolicy_status->usrid           = usr_pr->usrid;
				usr_pr->passpolicy_status->sendexpiredmail = (time_t) rset->getBigInt(rset, ++idx);
				usr_pr->passpolicy_status->lastchange      = (time_t) rset->getBigInt(rset, ++idx);
				usr_pr->passpolicy_status->firstlogin      = (time_t) rset->getBigInt(rset, ++idx);
				usr_pr->passpolicy_status->specialstart    = (time_t) rset->getBigInt(rset, ++idx);
			}
		}
		else {
			usr_pr->passpolicy_status = NULL;
		}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

		/* r->pool にusr_pr をキャッシュする */
		divy_pcache_set_data(p, usr_pr, DIVY_PCACHE_DAT_REQ_USERINFO);
	}
	else if (conf->ldap == TF_LDAP_ON && conf->ldapfoundcreate) {
		/*
		 * データが一件もなかった場合でLDAPが有効になっており
		 * 存在しない場合に作成するオプションが設定されていた
		 */
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;

		return DIVY_STCODE_USER_NOT_EXIST;
	}
	else {
		/* 一件も存在しなくてLDAPでもない */
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;

		return 1; /* 失敗 */
	}

	/* SQLが異なるので初期化 */
	if (rset != NULL)   rset->close(rset); rset = NULL;
	if (stmt != NULL)   stmt->close(stmt); stmt = NULL;

	/*
	 * 所属グループの情報を取得する.
	 */
	divy_sbuf_clear(sql_buf);
	divy_sbuf_append(sql_buf,
				"SELECT"
				" grpm.grpm_usr_id_vc"
				", grp.grp_grp_id_c"
				", grp.grp_name_vc"
				", grp.grp_regist_c"
				", grp.grp_update_c"
				", grp.grp_comment_vc"
				", rs.rs_uri_txt");

	/* グループ制約属性も取得する */
	if (support_grpconstraints || support_groupleader) {
		divy_sbuf_append(sql_buf,
				", grp.grp_extended_status_c");
	}

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
					", grp.grp_owner_usr_id_vc"
					", usr2.usr_fullname_vc AS ownername");
	}
	
	/* グループQuota情報を取得する */
#ifdef DIVY_SUPPORT_GROUPQUOTA
	if (support_groupquota) {
		divy_sbuf_append(sql_buf,
					", gq.gsqt_used_st_bi"
					", gq.gsqt_max_st_bi"
					", gq.gsqt_used_res_bi"
					", gq.gsqt_max_res_bi");
	}
#endif /* DIVY_SUPPORT_GROUPQUOTA */

	divy_sbuf_append(sql_buf,
				" FROM divy_grpmem grpm"
				" INNER JOIN divy_grp grp"
				" ON grpm.grpm_grp_id_c = grp.grp_grp_id_c"
				" INNER JOIN dav_resource rs "
				" ON grp.grp_rs_id_c = rs.rs_rs_id_c ");

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usr usr2"
				" ON (grp.grp_owner_usr_id_vc = usr2.usr_usr_id_vc)");
	}

#ifdef DIVY_SUPPORT_GROUPQUOTA
	if (support_groupquota) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_grpdiskquota gq"
				" ON (grp.grp_grp_id_c = gq.gsqt_grp_id_c)");
	}
#endif /* DIVY_SUPPORT_GROUPQUOTA */

	divy_sbuf_append(sql_buf,
				" WHERE grpm.grpm_usr_id_vc = ?");

	/* 非アクティブグループはリスティングしない */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" AND (grp.grp_extended_status_c NOT LIKE '___-%' "DIVY_DBFUNC_ESCAPE_CLAUSE
				" OR grp.grp_extended_status_c IS NULL)");
	}

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL )  stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, r->user);

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet.Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* データ取得 */
	while (rset->next(rset) == DB_TRUE) {
		if (grp_pr_h == NULL) {
			grp_pr_h = apr_hash_make(p);
		}
		grp_pr = apr_pcalloc(p, sizeof(divy_rdbo_grp));
		idx = 1;
		grp_pr->relativeuri = rset->getString(rset, idx++);
		grp_pr->grpid       = rset->getString(rset, idx++);
		grp_pr->name        = rset->getString(rset, idx++);
		grp_pr->registdt    = rset->getString(rset, idx++);
		grp_pr->updatedt    = rset->getString(rset, idx++);
		grp_pr->comment     = rset->getString(rset, idx++);
		grp_pr->grpcol_uri  = rset->getString(rset, idx++);
		if (support_grpconstraints || support_groupleader) {
			grp_pr->grp_extstatus = divy_rdbo_parse_extstatus(p, rset->getString(rset, idx++), EXTSTATUS_TYPE_GRP);
			/* 値が取れなければデフォルト値を使用する(互換性) */
			if (grp_pr->grp_extstatus == NULL) {
				grp_pr->grp_extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_GRP);
			}
		}
		else {
			/* サポートしていなければデフォルト値を使用(互換性) */
			grp_pr->grp_extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_GRP);
		}

		if (support_groupleader) {
			grp_pr->ownerid   = rset->getString(rset, idx++);
			grp_pr->ownername = rset->getString(rset, idx++);
		}
#ifdef DIVY_SUPPORT_GROUPQUOTA
		if (support_groupquota) {
			grp_pr->grp_q = apr_pcalloc(p, sizeof(divy_rdbo_grpquota));
			grp_pr->grp_q->usedst  = rset->getBigInt(rset, idx++);
			grp_pr->grp_q->maxst   = rset->getBigInt(rset, idx++);
			grp_pr->grp_q->usedres = rset->getBigInt(rset, idx++);
			grp_pr->grp_q->maxres  = rset->getBigInt(rset, idx++);
		}
#endif /* DIVY_SUPPORT_GROUPQUOTA */
		grp_pr->next = NULL;

		/* グループコレクションのURI でgrp_pr をキャッシュする */
		apr_hash_set(grp_pr_h, grp_pr->grpcol_uri, APR_HASH_KEY_STRING, grp_pr);
	}

	if (grp_pr_h != NULL) {
		apr_pool_t *cache_p = (r->main != NULL) ? r->main->pool : r->pool;

		/* メインの r->pool にgrp_pr_h をキャッシュする */
		divy_pcache_vset_data(cache_p, grp_pr_h, DIVY_PCACHE_DAT_REQ_AVAILABLE_GRPINFO, r->user, NULL);
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたユーザのユーザQuotaを取得する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_userquota(request_rec *r,
					divy_rdbo_usr *usr_pr,
					int do_entrylock,
			   		divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t      *p      = r->pool;
	int iscommit            = 0;

	if (usr_pr == NULL || IS_EMPTY(usr_pr->usrid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The value of input is invalid.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (do_entrylock) {
			/* do_entrylock が立っていて、かつ iscommit = 1 は
			 * この関数の仕様に反する
			 */
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The transaction context is nothing, but "
				"do_entrylock is set. It is impossible "
				"to execute it");
			return 1;
		}
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_usrdiskquota から値を取り出す。
	 */
	if (do_entrylock) {
		/* レコードをロックする */
		stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				"usqt_used_st_bi, "
				"usqt_max_st_bi, "
				"usqt_used_res_bi, "
				"usqt_max_res_bi "
				"FROM divy_usrdiskquota "
				"WHERE usqt_usr_id_vc = ? "
				"FOR UPDATE", p);
	}
	else {
		stmt = dbconn->prepareStatement(dbconn,
				"SELECT "
				"usqt_used_st_bi, "
				"usqt_max_st_bi, "
				"usqt_used_res_bi, "
				"usqt_max_res_bi "
				"FROM divy_usrdiskquota "
				"WHERE usqt_usr_id_vc = ?", p);
	}
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt for divy_usrdiskquota. "
			"(userid = %s, do_lock = %d) Reason: %s",
			usr_pr->usrid, do_entrylock, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, usr_pr->usrid);

	/* SQLの実行 */
	if (do_entrylock) {
		rset = stmt->executeQueryForUpdate(stmt, p);
	}
	else {
		rset = stmt->executeQuery(stmt, p);
	}
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to SELECT divy_usrdiskquota. (userid = %s) "
			"Reason: %s", usr_pr->usrid, rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset);	rset = NULL;
		if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		usr_pr->usedst  = rset->getBigInt(rset, 1);
		usr_pr->maxst   = rset->getBigInt(rset, 2);
		usr_pr->usedres = rset->getBigInt(rset, 3);
		usr_pr->maxres  = rset->getBigInt(rset, 4);
	}
	else {
		/*
		 * usr_pr が示すユーザが存在しなかった場合
		 * (note)
		 * 	これは正常ケースでも発生するので、エラーログを
		 * 	出してはならない
		 */
		if (do_entrylock) {
			/* ロックしようとして該当ユーザが存在しないのなら
			 * これは失敗させるしかない */
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
		}
		else {
			if (iscommit) divy_db_commit_transaction(ts_ctx);
		}

		if (rset != NULL) rset->close(rset);	rset = NULL;
		if (stmt != NULL) stmt->close(stmt);	stmt = NULL;

		return DIVY_STCODE_USER_NOT_EXIST;
	}

	/* 使用した資源を解放する */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * 現時点で全ユーザに割り当てられたQuota数の合計を取得する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_allocated_userquota(request_rec *r,
						divy_rdbo_usr **sum_uquota)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	divy_db_transaction_ctx *ts_ctx = NULL;

	TRACE(p);

	*sum_uquota = NULL;	/* 初期化 */

	/* トランザクションコンテキストの作成 */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, 
				"SELECT"
				" SUM(usqt_used_st_bi) AS usedst,"
				" SUM(usqt_max_st_bi) AS maxst,"
				" SUM(usqt_used_res_bi) AS usedres,"
				" SUM(usqt_max_res_bi) AS maxres "
				"FROM divy_usrdiskquota ", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt."
			"Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	/* SQL文の実行 */
	rset = stmt->executeQuery(stmt,p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to select divy_usrdiskquota. "
			"Reason: %s", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		*sum_uquota = apr_pcalloc(p, sizeof(divy_rdbo_usr));
		(*sum_uquota)->usedst  = rset->getBigInt(rset, 1);
		(*sum_uquota)->maxst   = rset->getBigInt(rset, 2);
		(*sum_uquota)->usedres = rset->getBigInt(rset, 3);
		(*sum_uquota)->maxres  = rset->getBigInt(rset, 4);
	}

	divy_db_commit_transaction(ts_ctx);

	/* 後片付け */
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたusr_pr が表すユーザを新規登録する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_user_property(request_rec *r,
						const divy_rdbo_usr *usr_pr,
						divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	apr_pool_t *p            = r->pool;
	int iscommit             = 0;
	divy_rdbo_resource rdb_r = { 0 };
	char *datebuf            = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char *trash_uri;
	int support_extstatus   = divy_support_extenduserstatus(r);
	int support_groupleader = divy_support_groupleader(r);
#ifdef DIVY_SUPPORT_PASSPOLICY
	int support_passpolicy   = divy_support_passpolicy(r);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	int support_access_control = divy_support_access_control(r);
	divy_sbuf *sql_buf = NULL;
	int idx;

	TRACE(p);

	if (usr_pr == NULL || IS_EMPTY(usr_pr->usrid) ||
	    IS_EMPTY(usr_pr->password) || IS_EMPTY(usr_pr->fullname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Could not operation. There are empty value.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/**
	 * プライベートコレクションの作成
	 */
	if (divy_rdbo_create_private_collection(r, usr_pr->usrid, &rdb_r, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to create private collection. "
			"(userid = %s)", usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/**
	 * ユーザごみ箱フォルダの作成
	 */
	if (divy_support_trashfolder(r)) {

		trash_uri = divy_build_user_trash_uri(p, dav_divy_get_root_uri(r), usr_pr->usrid);
		if (divy_rdbo_create_trash_property(r, trash_uri, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to create user trash folder. "
				"(trash_uri = %s)", trash_uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);

			return 1;
		}
	}

	/*
	 * divy_usr テーブルへの新規追加
	 */
	divy_sbuf_create(p, &sql_buf, 128);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"INSERT INTO divy_usr ("
				"usr_usr_id_vc,"
				"usr_passwd_vc,"
				"usr_fullname_vc,"
				"usr_mailaddr_vc,"
				"usr_admin_mod_i,"
				"usr_rs_id_c,"
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_ORACLE) || defined(DIVY_DBMS_DB2)
				"usr_usr_seq_i,"
#elif defined(DIVY_DBMS_SYBASE) /* sybase (何も入れない) */
#endif
				"usr_last_access_bi,"
				"usr_regist_c,"
				"usr_update_c,"
				"usr_comment_vc"
			);

	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
				",usr_expiration_bi, usr_extended_status_c");
	}

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				",usr_owner_usr_id_vc, usr_maxcreation_i");
	}

	if (support_access_control) {
		divy_sbuf_append(sql_buf,
				",usr_allowhosts_vc");
	}

	divy_sbuf_append(sql_buf,
				") "
				"VALUES (?, ?, ?, ?, ?, ?,"
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_ORACLE) || defined(DIVY_DBMS_DB2)
				DIVY_DBFUNC_CREATE_USR_SEQ","
#elif defined(DIVY_DBMS_SYBASE) /* sybase (何も入れない) */
#endif
				"?, ?, ?, ?");

	if (support_extstatus) {
		divy_sbuf_append(sql_buf, ",?, ?");
	}

	if (support_groupleader) {
		divy_sbuf_append(sql_buf, ",?, ?");
	}

	if (support_access_control) {
		divy_sbuf_append(sql_buf, ",?");
	}

	divy_sbuf_append(sql_buf, ")");

	/* SQL準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf) , p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (userid = %s) Reason: %s", 
				usr_pr->usrid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, rdb_r.creationdate, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	idx = 1;
	stmt->setString(stmt, idx++, usr_pr->usrid);
	/* ユーザパスワードを暗号化して利用している場合はMD5する */
	stmt->setString(stmt, idx++,
		       	(dconf->encryptpassword == DIVY_ENCRYPTPASSWORD_ON) ?
		       	ap_md5(p, (const unsigned char*)usr_pr->password) : 
			usr_pr->password);
	stmt->setString(stmt, idx++, usr_pr->fullname);
	stmt->setString(stmt, idx++, REPLACE_EMPTYSTR(usr_pr->mailaddr));
	stmt->setInt(stmt,    idx++, (int) usr_pr->adminmode);
	stmt->setString(stmt, idx++, rdb_r.rsid);
	stmt->setBigInt(stmt, idx++, APR_INT64_C(0));	/* 作成直後は0 とする */
	stmt->setString(stmt, idx++, datebuf);
	stmt->setString(stmt, idx++, datebuf);
	stmt->setString(stmt, idx++, REPLACE_EMPTYSTR(usr_pr->comment));
	if (support_extstatus) {
		stmt->setBigInt(stmt, idx++, usr_pr->expiration);
		stmt->setString(stmt, idx++, divy_rdbo_extstatus2str(usr_pr->extstatus, EXTSTATUS_TYPE_USR));
	}
	if (support_groupleader) {
		stmt->setString(stmt, idx++, usr_pr->ownerid);
		stmt->setInt(stmt,    idx++, usr_pr->maxusercreation);
	}
	if (support_access_control) {
		stmt->setString(stmt, idx++, usr_pr->allowhosts);
	}

	/* SQL実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_usr.(userid = %s) Reason: %s",
				usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_usrdiskquota テーブルへの新規追加
	 */
	stmt = dbconn->prepareStatement(dbconn, 
			"INSERT INTO divy_usrdiskquota ("
			"usqt_usr_id_vc,"
			"usqt_used_st_bi,"
			"usqt_max_st_bi,"
			"usqt_used_res_bi,"
			"usqt_max_res_bi) "
			"VALUES (?, 0, ?, 0, ?)", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) "
			"Reason: %s", 
			usr_pr->usrid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	stmt->setString(stmt,  1, usr_pr->usrid);
	stmt->setBigInt(stmt,  2, usr_pr->maxst);
	stmt->setBigInt(stmt,  3, usr_pr->maxres);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_usrdiskquota."
			"(userid = %s) Reason: %s", 
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_usraccessdeny テーブルに追加する必要があればINSERTする
	 */
	if (usr_pr->accessdeny != NULL) {
		stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO divy_usraccessdeny "
				"(usad_usr_id_vc,"
				"usad_deny_user,"
				"usad_deny_group,"
				"usad_deny_dblink,"
				"usad_deny_reposdblink,"
				"usad_deny_management,"
				"usad_deny_m_update,"
				"usad_deny_m_sql,"
				"usad_deny_m_group,"
				"usad_deny_m_user,"
				"usad_deny_m_status,"
				"usad_deny_m_msg,"
				"usad_deny_m_dbms,"
				"usad_deny_m_execsql) "
				"VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?)", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (userid = %s) "
				"Reason: %s", 
				usr_pr->usrid, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);

			return 1;
		}

		stmt->setString(stmt,  1, usr_pr->usrid);
		stmt->setInt(stmt,     2, usr_pr->accessdeny[DIVY_FOLDER_ID_user]);
		stmt->setInt(stmt,     3, usr_pr->accessdeny[DIVY_FOLDER_ID_group]);
		stmt->setInt(stmt,     4, usr_pr->accessdeny[DIVY_FOLDER_ID_dblink]);
		stmt->setInt(stmt,     5, usr_pr->accessdeny[DIVY_FOLDER_ID_reposdblink]);
		stmt->setInt(stmt,     6, usr_pr->accessdeny[DIVY_FOLDER_ID_management]);
		stmt->setInt(stmt,     7, usr_pr->accessdeny[DIVY_FOLDER_ID_m_update]);
		stmt->setInt(stmt,     8, usr_pr->accessdeny[DIVY_FOLDER_ID_m_sql]);
		stmt->setInt(stmt,     9, usr_pr->accessdeny[DIVY_FOLDER_ID_m_group]);
		stmt->setInt(stmt,    10, usr_pr->accessdeny[DIVY_FOLDER_ID_m_user]);
		stmt->setInt(stmt,    11, usr_pr->accessdeny[DIVY_FOLDER_ID_m_status]);
		stmt->setInt(stmt,    12, usr_pr->accessdeny[DIVY_FOLDER_ID_m_msg]);
		stmt->setInt(stmt,    13, usr_pr->accessdeny[DIVY_FOLDER_ID_m_dbms]);
		stmt->setInt(stmt,    14, usr_pr->accessdeny[DIVY_FOLDER_ID_m_execsql]);

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to insert divy_usraccessdeny."
					"(userid = %s) Reason: %s", usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);

			return 1;
		}
	}
#ifdef DIVY_SUPPORT_PASSPOLICY
	/*
	 * ユーザのパスワードポリシーを登録する
	 */
	if (support_passpolicy && usr_pr->passpolicy_status != NULL) {

		usr_pr->passpolicy_status->lastchange = rdb_r.creationdate;
		if (divy_rdbo_insert_user_passwordpolicy(r, usr_pr->passpolicy_status, ts_ctx)) {

			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to insert divy_passpolicystatus.(userid = %s) ", usr_pr->usrid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたusr_pr が表す内容でユーザを更新する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_user_property(request_rec *r,
					const divy_rdbo_usr *usr_pr,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit = 0;	/* トランザクションを確定するかどうか */
	time_t short_time       = dav_divy_get_now_epoch();
	char *datebuf           = NULL;
	int update_cnt          = 0;
	int support_extstatus   = divy_support_extenduserstatus(r);
	int support_groupleader = divy_support_groupleader(r);
	int support_access_control = divy_support_access_control(r);
	int support_userlockout = divy_support_failedlogin_lockout(r);
	divy_sbuf *sql_buf = NULL;
	int idx;
	divy_rdbo_usr *old_usr_pr = NULL;	/* 変更前のユーザ情報 */
	int clearfailedlogin = 0;	/* 失敗ログイン回数をクリアするなら1*/

	TRACE(p);

	if (usr_pr == NULL || IS_EMPTY(usr_pr->usrid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Could not operation. usrid is NULL.");
		return 1;
	}

	/* 必須項目のチェック */
	if (IS_EMPTY(usr_pr->password) || IS_EMPTY(usr_pr->fullname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Could not operation. There is empty value.");
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/* アップデート対象ユーザがアクティブになる更新の場合のみ
	   ライセンス数のチェックを行う
	*/
	/* 更新対象ユーザがアクティブの場合、変更前のユーザを取得 */
	if (support_extstatus && divy_rdbo_is_active_user(usr_pr->extstatus)) {
		if (divy_rdbo_get_user_property(r, usr_pr->usrid, &old_usr_pr) == 0) {
			if (!divy_rdbo_is_active_user(old_usr_pr->extstatus)) {
				if (divy_validate_update_user_limit(r) != NULL) 
					return 1;
			}
		}
		/* 更新対象がアクティブユーザならばログイン失敗はクリアする */
		if (support_userlockout) {
			clearfailedlogin = 1;	/* ログイン失敗回数はクリアする */
		}
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない -> ここでは作成せずエラーにする */
	if (ts_ctx == NULL) {
	 	/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Could not operation. Transaction_ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_usr テーブルの更新(対象者のエントリ更新)
	 */
	divy_sbuf_create(p, &sql_buf, 256);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"UPDATE divy_usr "
					"SET "
					"usr_passwd_vc = ?,"
					"usr_fullname_vc = ?,"
					"usr_mailaddr_vc = ?,"
					"usr_admin_mod_i = ?,"
					"usr_update_c = ?,"
					"usr_comment_vc = ?");
	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
					",usr_expiration_bi = ?"
					",usr_extended_status_c = ?");
	}

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
					",usr_owner_usr_id_vc = ?"
					",usr_maxcreation_i = ?");
	}

	if (support_access_control) {
		divy_sbuf_append(sql_buf,
					",usr_allowhosts_vc = ?");
	}

	if (support_userlockout && clearfailedlogin) {
		divy_sbuf_append(sql_buf,
					",usr_failed_access_count_i = 0");
	}

	divy_sbuf_append(sql_buf,
					" WHERE usr_usr_id_vc = ?");

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (userid = %s) Reason: %s",
				usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	idx = 1;
	stmt->setString	(stmt, idx++, usr_pr->password);
	stmt->setString	(stmt, idx++, usr_pr->fullname);
	stmt->setString	(stmt, idx++, REPLACE_EMPTYSTR(usr_pr->mailaddr));
	stmt->setInt	(stmt, idx++, (int) usr_pr->adminmode);
	stmt->setString	(stmt, idx++, datebuf);
	stmt->setString	(stmt, idx++, REPLACE_EMPTYSTR(usr_pr->comment));
	if (support_extstatus) {
		stmt->setBigInt(stmt,  idx++, usr_pr->expiration);
		stmt->setString(stmt,  idx++, divy_rdbo_extstatus2str(usr_pr->extstatus, EXTSTATUS_TYPE_USR));
	}
	if (support_groupleader) {
		stmt->setString(stmt, idx++, usr_pr->ownerid);
		stmt->setInt   (stmt, idx++, usr_pr->maxusercreation);
	}
	if (support_access_control) {
		stmt->setString(stmt, idx++, usr_pr->allowhosts);
	}

	stmt->setString	(stmt, idx++, usr_pr->usrid);

	/* SQL実行 */
	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_usr. (userid = %s) Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (update_cnt == 0) {
		/* 恐らく、select してからupdate されるまでの間に
		 * エントリが消されてしまったということ */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Status is OK, but the count of updating is 0."
				"Probably because of competing with remove "
				"operation, this entry of user already have been lost. "
				"Please check your data.(userid = %s)",
				usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	update_cnt = 0;

	/*
	 * divy_session テーブルの更新
	 * memcacheが無効の場合のみ更新する
	 */
	if (divy_support_session(r) && divy_enable_memcache(r) == 0) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"session update password = %s", usr_pr->password);
		stmt = dbconn->prepareStatement(dbconn,
					"UPDATE divy_session "
					"SET "
					"ses_usr_password_vc = ? "
					"WHERE ses_usr_usr_id_vc = ?", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. (update divy_session) "
					"(userid = %s) Reason: %s",
					usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		stmt->setString(stmt,  1, usr_pr->password);
		stmt->setString(stmt,  2, usr_pr->usrid);

		/* SQL実行 */
		(void)stmt->executeUpdate(stmt, p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to update divy_session. (userid = %s) Reason: %s",
				usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/*
	 * divy_usrdiskquota テーブルの更新
	 */
	update_cnt = 0;
	stmt = dbconn->prepareStatement(dbconn, 
				"UPDATE divy_usrdiskquota "
				"SET "
				"usqt_max_st_bi = ?,"
				"usqt_max_res_bi = ? "
				"WHERE usqt_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setBigInt(stmt,  1, usr_pr->maxst);
	stmt->setBigInt(stmt,  2, usr_pr->maxres);
	stmt->setString(stmt,  3, usr_pr->usrid);

	update_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update divy_usrdiskquota. (userid = %s) Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (update_cnt == 0) {
		/* 恐らく、select してからupdate されるまでの間に
		 * エントリが消されてしまったということ */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Status is OK, but the count of updating is 0."
			"Probably because of competing with remove "
			"operation, this entry of user already have "
			"been lost.Please check your data.(userid = %s)",
			usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_usraccessdeny テーブルを更新する必要があれば更新する
	 */
	/* まずは削除する */
	stmt = dbconn->prepareStatement(dbconn, 
			"DELETE FROM divy_usraccessdeny "
			"WHERE usad_usr_id_vc = ?", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) Reason: %s", 
			usr_pr->usrid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	stmt->setString(stmt,  1, usr_pr->usrid);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_usraccessdeny.(userid = %s) Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/* 更新する必要があった場合 */
	if (usr_pr->accessdeny != NULL) {
		stmt = dbconn->prepareStatement(dbconn, 
				"INSERT INTO divy_usraccessdeny "
				"(usad_usr_id_vc,"
				"usad_deny_user,"
				"usad_deny_group,"
				"usad_deny_dblink,"
				"usad_deny_reposdblink,"
				"usad_deny_management,"
				"usad_deny_m_update,"
				"usad_deny_m_sql,"
				"usad_deny_m_group,"
				"usad_deny_m_user,"
				"usad_deny_m_status,"
				"usad_deny_m_msg,"
				"usad_deny_m_dbms,"
				"usad_deny_m_execsql) "
				"VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?)", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (userid = %s) "
				"Reason: %s", 
				usr_pr->usrid, stmt->getMsg(stmt));

			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);

			return 1;
		}

		stmt->setString(stmt,  1, usr_pr->usrid);
		stmt->setInt(stmt,     2, usr_pr->accessdeny[DIVY_FOLDER_ID_user]);
		stmt->setInt(stmt,     3, usr_pr->accessdeny[DIVY_FOLDER_ID_group]);
		stmt->setInt(stmt,     4, usr_pr->accessdeny[DIVY_FOLDER_ID_dblink]);
		stmt->setInt(stmt,     5, usr_pr->accessdeny[DIVY_FOLDER_ID_reposdblink]);
		stmt->setInt(stmt,     6, usr_pr->accessdeny[DIVY_FOLDER_ID_management]);
		stmt->setInt(stmt,     7, usr_pr->accessdeny[DIVY_FOLDER_ID_m_update]);
		stmt->setInt(stmt,     8, usr_pr->accessdeny[DIVY_FOLDER_ID_m_sql]);
		stmt->setInt(stmt,     9, usr_pr->accessdeny[DIVY_FOLDER_ID_m_group]);
		stmt->setInt(stmt,    10, usr_pr->accessdeny[DIVY_FOLDER_ID_m_user]);
		stmt->setInt(stmt,    11, usr_pr->accessdeny[DIVY_FOLDER_ID_m_status]);
		stmt->setInt(stmt,    12, usr_pr->accessdeny[DIVY_FOLDER_ID_m_msg]);
		stmt->setInt(stmt,    13, usr_pr->accessdeny[DIVY_FOLDER_ID_m_dbms]);
		stmt->setInt(stmt,    14, usr_pr->accessdeny[DIVY_FOLDER_ID_m_execsql]);

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_usraccessdeny.(userid = %s) Reason: %s",
				usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);

			return 1;
		}
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * dav_resource テーブルの更新者ユーザIDおよび更新者名、
	 * 更新日付を変更する
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"UPDATE dav_resource "
				"SET"
				" rs_lastmodifier_usr_id_vc = ?,"
				" rs_get_lastmodified_bi = ? "
				"WHERE rs_rs_id_c = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) "
			"Reason: %s", usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	stmt->setString	(stmt, 1, divy_get_userid(r));
	stmt->setBigInt(stmt,  2, short_time);
	stmt->setString	(stmt, 3, usr_pr->rsid);

	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update dav_resource. (userid = %s) "
			"Reason: %s", usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * 制限ユーザであれば自身のメール監視フラグを全て消去する
	 */
	if (support_extstatus && !divy_rdbo_is_trusty_user(usr_pr->extstatus)) {
		divy_rdbo_mailwatch mw = { 0 };
		mw.uri     = NULL;
		mw.usrid   = usr_pr->usrid;
		mw.delflag = 0;

		if (divy_rdbo_remove_mailwatch_property(r, &mw, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete divy_mailwatch.(userid = %s) ", usr_pr->usrid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

#ifdef DIVY_SUPPORT_PASSPOLICY
	/*
	 * ユーザの持つパスワードポリシー状態があれば反映する
	 */
	if (usr_pr->passpolicy_status != NULL &&
		divy_rdbo_update_user_passwordpolicy(r, usr_pr->passpolicy_status, ts_ctx)) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to update divy_passpolicystatus.(userid = %s) ", usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * ユーザのパスワード履歴に追加する
	 * [ 条件 ]
	 *   * 自分自身のパスワード変更であること(ユーザの種別に依らない)
	 */
	if (strcmp(divy_get_userid(r), usr_pr->usrid) == 0) {
		if (divy_rdbo_update_user_passhistory(r, usr_pr->usrid, usr_pr->password, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update user-passwordpolicy history.(userid = %s)",
					usr_pr->usrid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/*
	 * 対象ユーザが管理者ユーザの場合、そのユーザがオーナーとなっている
	 * ユーザがいたらそのユーザのオーナーIDを全てNULLにする、
	 * つまり、対象ユーザがオーナーの子ユーザは存在しないようにする.
	 * (note)
	 * 本当は、グループリーダ -> 管理者になったときだけやればいいのだが
	 * 判らないので、更新してしまっています. データ不整合を訂正する意味でもいいでしょう. */
	if (support_groupleader && usr_pr->adminmode == DIVY_ADMINMODE_ADMIN) {

		/* SQLの準備 */
		stmt = dbconn->prepareStatement(dbconn,
					"UPDATE divy_usr "
					" SET"
					" usr_owner_usr_id_vc = NULL"
					" WHERE usr_owner_usr_id_vc = ?", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. (userid = %s) Reason: %s",
					usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}

		/* バインド */
		stmt->setString(stmt, 1, usr_pr->usrid);

		/* SQL実行 */
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to update divy_usr for cleaning ownerid. "
					"(userid = %s) Reason: %s", usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt);
			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/*
	 * グループリーダ以外のユーザは、グループのオーナであってはならない
	 * (note)
	 *  グループリーダが降格されたり、昇格した場合にのみ実施すればいいのだが、
	 *  データ不正が生じているかもしれないので、トライしてみる.
	 */
	if (support_groupleader && !divy_rdbo_is_groupleader(usr_pr->extstatus)) {

		if (divy_rdbo_dismiss_groupleader(r, usr_pr->usrid, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to dismiss groupleader.(userid = %s)", usr_pr->usrid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * 制限ユーザであれば、そのユーザが持っていた開封通知を全て削除する -> やめました.
	 * (note)
	 *    この関数の呼び出し元に移動しました. なぜなら、闇雲に更新してしまうと
	 *    コメントを変更したり、パスワードを変えただけでも開封通知が取れてしまうためです.
	 *    非制限ユーザ -> 制限ユーザに限り、削除動作をすべきです.
	 */

	/* 変更を確定 */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたusr_pr が示すユーザを削除する。
 * ユーザが保持していたプライベートコレクション以下のリソースを全て削除します。
 * 
 */
DIVY_DECLARE(int) divy_rdbo_remove_user_property(request_rec *r,
					const divy_rdbo_usr *usr_pr,
					divy_linkedlist_t **del_filelist,
					divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	DbResultSet     *rset    = NULL;
	apr_pool_t *p            = r->pool;
	int iscommit             = 0;
	divy_rdbo_resource rdb_r = { 0 };
	divy_rdbo_mailwatch mw   = { 0 };
	const char *parentid     = NULL;

	TRACE(p);

	*del_filelist = NULL;	/* 初期化 */

	if (usr_pr == NULL || IS_EMPTY(usr_pr->usrid) ||
	    IS_EMPTY(usr_pr->rsid) || IS_EMPTY(usr_pr->prvcol_uri)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There are empty value.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * dav_resource (プライベートコレクション) 以下の削除
	 */
	rdb_r.uri          = usr_pr->prvcol_uri;
	rdb_r.rsid         = usr_pr->rsid;
	rdb_r.resourcetype = DIVY_TYPE_COLLECTION;

	if (divy_rdbo_remove_resource(r, &rdb_r, del_filelist, ts_ctx)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete private collection. (uri = %s, "
			"userid = %s) ", usr_pr->prvcol_uri, usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * ユーザを削除すると2.1.6-5までは削除されるユーザIDのディスク
	 * クォータは全てなくなり、ユーザIDも不明なユーザに付け替えられ
	 * ていました。
	 * システムクォータとしては不整合はなくなるが、グループリーダー
	 * がユーザを作って運用している場合、グループリーダーの設定枠を
	 * 超えたところにファイルが存在することになってしまいます。
	 * その為グループリーダーが作ったユーザだけは削除された場合、
	 * そのユーザのオーナーに付け替える処理を追加します。
	 * オーナーは突然ファイルが増えることになりますのでオーナー自身が
	 * クォータ制御されている場合は、突然容量がいっぱいになる可能性も
	 * あります。注意してください。プログラムでは無条件に増やします。
	 * 
	 * オーナーがいないユーザ（つまり管理者が作ったユーザ）は次の条件
	 * で処理を行います。
	 *
	 * 条件：
	 *   管理者が一人の場合はその管理者に全て押し付けます。
	 *   管理者が複数いる場合は、不明ユーザ（従来通り）へ割当てます。
	 */
	parentid = usr_pr->ownerid;
	if (parentid == NULL) {
		/* parentid(オーナー)がいないなら管理者を調べる */
		stmt = dbconn->prepareStatement(dbconn,
				"SELECT usr_usr_id_vc FROM divy_usr "
				"WHERE usr_admin_mod_i = 1", p);

		/* SQL実行 */
		rset = stmt->executeQuery(stmt, p);
		if (rset->getCode(rset) != DB_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Reason: %s", rset->getMsg(rset));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		/*
		 * 検索結果の初回だけ(parent=NULL)管理者のユーザIDを取得します。
		 * 管理者が複数いる場合はparentid はNULLに設定されて
		 * ループを抜けます
		 */
		while (rset->next(rset) == DB_TRUE) {
			if (parentid != NULL) { 
				parentid = NULL;
				break;
			}
			parentid = rset->getString(rset, 1);
		}

		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	}

	/* 
	 * parentidにユーザIDがある場合はそのユーザに対してディスククォータを
	 * 付け替えます。
	 */
	if (parentid != NULL) {
		/*
		 * 無条件にファイル容量とファイル数をparentid(オーナー）に加算します
		 * Quotaのチェックは行いません。つまりこれで溢れるかも知れません。
		 */
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_usrdiskquota "
				" SET "
				" usqt_used_st_bi  = sub.st, "
				" usqt_used_res_bi = sub.rs  "
					" FROM (SELECT "
							" SUM(usqt_used_st_bi)  as st,"
							" SUM(usqt_used_res_bi) as rs "
					       	" FROM divy_usrdiskquota "
							" WHERE usqt_usr_id_vc IN (?, ?)) sub "
				" WHERE usqt_usr_id_vc = ?", p);

		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt."
				"(UPDATE FROM divy_usrdiskquota.) (userid = %s) "
				"Reason: %s", parentid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		stmt->setString(stmt, 1, parentid);
		stmt->setString(stmt, 2, usr_pr->usrid);
		stmt->setString(stmt, 3, parentid);

		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to UPDATE divy_usrdiskquota. (userid = %s) "
					"Reason: %s", parentid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			return 1;
		}

		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
			
	}

	/*
	 * divy_usrdiskquota の削除
	 * (note) divy_usr よりも前に削除すること。
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_usrdiskquota "
			"WHERE usqt_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) "
			"Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, usr_pr->usrid);

	/* sql実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_usrdiskquota.(userid = %s) "
			"Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_usraccessdeny の削除
	 * (note) divy_usr よりも前に削除すること。
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_usraccessdeny "
			"WHERE usad_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) "
			"Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, usr_pr->usrid);

	/* sql実行 失敗時はエラー(無い場合が殆ど) */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_usraccessdeny.(userid = %s) "
			"Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_usrの削除
	 */
	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_usr "
			"WHERE usr_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) "
			"Reason: %s", usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, usr_pr->usrid);

	/* sql実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_usr.(userid = %s) "
			"Reason: %s", usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_grpmemの削除
	 */
 	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM divy_grpmem "
			"WHERE grpm_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (userid = %s) "
				"Reason: %s",
				usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, usr_pr->usrid);

	/* sql実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete divy_grpmem.(userid = %s) "
			"Reason: %s", usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_mailwatchの削除
	 */
	mw.uri     = NULL;
	mw.usrid   = usr_pr->usrid;
	mw.delflag = 0;
	if (divy_rdbo_remove_mailwatch_property(r, &mw, ts_ctx)) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete divy_mailwatch.(userid = %s) ", usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * dav_lockの削除
	 * (note)
	 *   強制削除機能が有効な場合には、ここで削除されるロックは正常なものです.
	 *   そうでなければ、ここで削除されるロックは、リソースは存在しないのに残っていた
	 *   不正なロックです. (事前にリソースが存在しないことをチェックをしているので)
	 */
	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn,
			"DELETE FROM dav_lock "
			"WHERE lk_uri_txt = ? "
			"OR lk_uri_txt LIKE ? "
			DIVY_DBFUNC_ESCAPE_CLAUSE, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) "
			"Reason: %s", usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, usr_pr->prvcol_uri);
	stmt->setString(stmt, 2, apr_pstrcat(p,
			stmt->escWildCard(stmt, usr_pr->prvcol_uri), "/%", NULL));

	/* sql実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to delete dav_lock."
			"(uri = %s) Reason: %s",
			usr_pr->prvcol_uri, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DIVY_SUPPORT_PASSPOLICY
	/*
	 * パスワードポリシー状態テーブルの削除
	 */
	if (divy_rdbo_remove_user_passwordpolicy(r, usr_pr->usrid, ts_ctx)) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete divy_passpolicystatus.(userid = %s) ", usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}

	/*
	 * ユーザのパスワード履歴を削除
	 */
	if (divy_rdbo_remove_user_passhistory(r, usr_pr->usrid, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to delete user's password-policy history.(userid = %s)",
				usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		return 1;
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/*
	 * 対象ユーザが作成していたリソースの作成者IDを上書き (dav_resource)
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"UPDATE dav_resource "
				"SET rs_creator_usr_id_vc = ? "
				"WHERE rs_creator_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	stmt->setString(stmt, 1, (parentid != NULL) ? parentid : DIVY_DELETED_USER_ID);
	stmt->setString(stmt, 2, usr_pr->usrid);

	/* sql実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update rs_creator_usr_id_vc on dav_resource. "
			"(userid = %s) Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * 対象ユーザが更新していたリソースの更新者IDを上書き (dav_resource)
	 * (note) ユーザID再利用によるQuotaの不正現在を防止
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"UPDATE dav_resource "
				"SET rs_lastmodifier_usr_id_vc = ? "
				"WHERE rs_lastmodifier_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	stmt->setString(stmt, 1, (parentid != NULL) ? parentid : DIVY_DELETED_USER_ID);
	stmt->setString(stmt, 2, usr_pr->usrid);

	/* sql実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to update rs_lastmodifier_usr_id_vc on dav_resource. "
			"(userid = %s) Reason: %s",
			usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * 対象ユーザが持っていた開封通知を削除
	 */
	if (divy_support_confirmreading(r)) {

		if (divy_rdbo_remove_confirmreading(r, NULL/*URI無視*/, usr_pr->usrid, 0, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove confirmreading property of deleted user."
					"(userid = %s)", usr_pr->usrid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/*
	 * 対象ユーザがグループリーダに任命されていた場合、その任を解く
	 * (note)
	 *   グループリーダユーザに対してのみ実施すればいいのだけれど、
	 *   データ不正が生じているかもしれないので、全ユーザでトライしてみる
	 */
	if (divy_support_groupleader(r)) {
		if (divy_rdbo_dismiss_groupleader(r, usr_pr->usrid, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to dismiss groupleader."
					"(userid = %s)", usr_pr->usrid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			if (iscommit) divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}

	/* 反映を確定する */
	if (iscommit) divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたsrc_usr_pr が示すユーザをdst_usr_pr が示すユーザとしてコピーする。
 * 
 * divy_usr, divy_usrdiskquota, divy_usraccessdeny, divy_grpmem は
 * 元のデータを元に新たに作成しなおします。
 * dav_resourceには新規のプライベートコレクションを作ります。
 *
 */
DIVY_DECLARE(int) divy_rdbo_copy_user_property(request_rec *r,
					const divy_rdbo_usr *src_usr_pr,
					const divy_rdbo_usr *dst_usr_pr)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	apr_pool_t *p            = r->pool;
	divy_rdbo_resource rdb_r = { 0 };
	divy_db_transaction_ctx *ts_ctx;
	char *datebuf            = NULL;
	const char *trash_uri;
	int insert_cnt           = 0;
	int support_extstatus    = divy_support_extenduserstatus(r);
	int support_groupleader  = divy_support_groupleader(r);
#ifdef DIVY_SUPPORT_PASSPOLICY
	int support_passpolicy   = divy_support_passpolicy(r);
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	int support_access_control = divy_support_access_control(r);
	divy_sbuf *sql_buf;
	int idx;

	TRACE(p);

	/* 必須項目のチェック */
	if (src_usr_pr == NULL || IS_EMPTY(src_usr_pr->usrid) ||
	    dst_usr_pr == NULL || IS_EMPTY(dst_usr_pr->usrid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There are empty values.");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/**
	 * プライベートコレクションの作成
	 */
	if (divy_rdbo_create_private_collection(r, dst_usr_pr->usrid, &rdb_r, ts_ctx)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to create private collection. "
			"(userid = %s)", src_usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/**
	 * ユーザごみ箱フォルダの作成
	 */
	if (divy_support_trashfolder(r)) {
		trash_uri = divy_build_user_trash_uri(p, dav_divy_get_root_uri(r),
								dst_usr_pr->usrid);
		if (divy_rdbo_create_trash_property(r, trash_uri, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to create user trash folder. "
				"(trash_uri = %s)", trash_uri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);

			return 1;
		}
	}

	/*
	 * divy_usrへの新規追加
	 */
	divy_sbuf_create(p, &sql_buf, 512);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"INSERT INTO divy_usr ("
				"usr_usr_id_vc, "
				"usr_passwd_vc, "
				"usr_fullname_vc, "
				"usr_mailaddr_vc, "
				"usr_admin_mod_i, "
				"usr_rs_id_c, "
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_ORACLE) || defined(DIVY_DBMS_DB2)
				"usr_usr_seq_i, "
#elif defined(DIVY_DBMS_SYBASE) /* sybase (何も入れない) */
#endif
				"usr_last_access_bi, "
				"usr_regist_c, "
				"usr_update_c, "
				"usr_comment_vc");

	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
				",usr_expiration_bi"
				",usr_extended_status_c");
	}

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				",usr_owner_usr_id_vc"
				",usr_maxcreation_i");
	}

	if (support_access_control) {
		divy_sbuf_append(sql_buf,
				",usr_allowhosts_vc");
	}

	divy_sbuf_append(sql_buf,
				") "
				"(SELECT "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_USRID_LEN))", "
				"usr_passwd_vc, "
				"usr_fullname_vc, "
				"usr_mailaddr_vc, "
				"usr_admin_mod_i, "
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_CHAR(DIVY_DB_RSID_LEN))", "
#if defined(DIVY_DBMS_POSTGRES) || defined(DIVY_DBMS_ORACLE) || defined(DIVY_DBMS_DB2)
				DIVY_DBFUNC_CREATE_USR_SEQ", "
#elif defined(DIVY_DBMS_SYBASE) /* sybase (何も入れない) */
#endif
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_BIGINT)","
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN))","
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN))","
				"usr_comment_vc ");
	
	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
				",usr_expiration_bi"
				","DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_CHAR(DIVY_DB_USREXTSTATUS_LEN)));
	}
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				","DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_CHAR(DIVY_DB_USRID_LEN))
				",?");
	}
	if (support_access_control) {
		divy_sbuf_append(sql_buf,
				",usr_allowhosts_vc");
	}
	divy_sbuf_append(sql_buf,
				" FROM divy_usr "
				" WHERE usr_usr_id_vc = ?)");

	/* SQL文の準備　失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (userid = %s) "
				"Reason: %s", src_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, rdb_r.creationdate, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */
	idx = 1;
	stmt->setString(stmt, idx++, dst_usr_pr->usrid);
	stmt->setString(stmt, idx++, rdb_r.rsid);
	stmt->setBigInt(stmt, idx++, rdb_r.creationdate);
	stmt->setString(stmt, idx++, datebuf);
	stmt->setString(stmt, idx++, datebuf);
	if (support_extstatus) {
		stmt->setString(stmt, idx++, divy_rdbo_extstatus2str(dst_usr_pr->extstatus, EXTSTATUS_TYPE_USR));
	}
	if (support_groupleader) {
		stmt->setString(stmt, idx++, dst_usr_pr->ownerid);
		stmt->setInt(stmt, idx++, dst_usr_pr->maxusercreation);
	}

	stmt->setString(stmt, idx++, src_usr_pr->usrid);

	/* sql実行　失敗時はエラー */
	insert_cnt = stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_usr.(userid = %s) "
			"Reason: %s", src_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	/* 更新件数が1件以外はエラー (データが見つからないレアケース)*/
	if (insert_cnt != 1){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to insert divy_usr. data is not found."
			"(userid = %s)", src_usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_usrdiskquota の内容をコピーする
	 * [ コピー項目 ]
	 *   * 最大ストレージ容量
	 *   * 最大ストレージファイル数
	 */
	/* sql文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, 
			"INSERT INTO divy_usrdiskquota ("
			"usqt_usr_id_vc,"
			"usqt_used_st_bi,"
			"usqt_max_st_bi,"
			"usqt_used_res_bi,"
			"usqt_max_res_bi) "
			"(SELECT "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_USRID_LEN))", 0, "
			"usqt_max_st_bi, "
			"0, "
			"usqt_max_res_bi "
			"FROM divy_usrdiskquota "
			"WHERE usqt_usr_id_vc = ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) "
			"Reason: %s", src_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt,  1, dst_usr_pr->usrid);
	stmt->setString(stmt,  2, src_usr_pr->usrid);

	/* sql実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_usrdiskquota."
			"(userid = %s) Reason: %s", 
			src_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_usraccessdeny の内容をコピーする
	 * (note)
	 * 	ユーザID以外の全てをコピーします
	 */
	stmt = dbconn->prepareStatement(dbconn, 
			"INSERT INTO divy_usraccessdeny "
			"(usad_usr_id_vc,"
			"usad_deny_user,"
			"usad_deny_group,"
			"usad_deny_dblink,"
			"usad_deny_reposdblink,"
			"usad_deny_management,"
			"usad_deny_m_update,"
			"usad_deny_m_sql,"
			"usad_deny_m_group,"
			"usad_deny_m_user,"
			"usad_deny_m_status,"
			"usad_deny_m_msg,"
			"usad_deny_m_dbms,"
			"usad_deny_m_execsql) "
			"(SELECT "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_USRID_LEN))", "
			"usad_deny_user,"
			"usad_deny_group,"
			"usad_deny_dblink,"
			"usad_deny_reposdblink,"
			"usad_deny_management,"
			"usad_deny_m_update,"
			"usad_deny_m_sql,"
			"usad_deny_m_group,"
			"usad_deny_m_user,"
			"usad_deny_m_status,"
			"usad_deny_m_msg,"
			"usad_deny_m_dbms,"
			"usad_deny_m_execsql "
			"FROM divy_usraccessdeny "
			"WHERE usad_usr_id_vc = ?)", p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) "
			"Reason: %s", 
			src_usr_pr->usrid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, dst_usr_pr->usrid);
	stmt->setString(stmt, 2, src_usr_pr->usrid);

	/* SQLの実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_usraccessdeny."
			"(userid = %s) Reason: %s",
			src_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

        /* 
	 * divy_grpmemテーブルへの新規追加
	 * (note) srcが持っていた所属関係を全てコピーする
	 */
	stmt = dbconn->prepareStatement(dbconn,
			"INSERT INTO divy_grpmem ("
			"grpm_grp_id_c, "
			"grpm_usr_id_vc) "
			"(SELECT grpm_grp_id_c,"
			DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_USRID_LEN))" "
			"FROM divy_grpmem "
			"WHERE grpm_usr_id_vc = ?)", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (userid = %s) "
			"Reason: %s", src_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt,  1, dst_usr_pr->usrid);
	stmt->setString(stmt,  2, src_usr_pr->usrid);

	/* sql実行 失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to insert divy_grpmem.(userid = %s) "
			"Reason: %s", src_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DIVY_SUPPORT_PASSPOLICY
	/*
	 * ユーザと関連したパスワードポリシーを新規作成する.
	 * (note) src_usr_pr のプロパティはコピーしないこと. コピーすることには何の意味もないから.
	 */
	if (support_passpolicy && dst_usr_pr->passpolicy_status != NULL) {

		dst_usr_pr->passpolicy_status->lastchange = rdb_r.creationdate;
		if (divy_rdbo_insert_user_passwordpolicy(r, dst_usr_pr->passpolicy_status, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to create user based passwordpolicy. "
					"(dst_userid = %s)", dst_usr_pr->usrid);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			return 1;
		}
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* 反映を確定する */
	divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * 指定されたsrc_usr_pr が示すユーザをdst_usr_pr が示すユーザとして移動する。
 * uri変更のためユーザIDも変更します
 * リソース(配下含む)のuriも変更します(最終更新日は変更しません)
 */
DIVY_DECLARE(int) divy_rdbo_move_user_property(request_rec *r,
					const divy_rdbo_usr *src_usr_pr,
					const divy_rdbo_usr *dst_usr_pr)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	time_t short_time       = dav_divy_get_now_epoch();
	char *datebuf           = NULL;
	divy_db_transaction_ctx *ts_ctx;
	divy_rdbo_resource src_rdb_r = { 0 };
	divy_rdbo_resource dst_rdb_r = { 0 };
	int support_extstatus   = divy_support_extenduserstatus(r);
	int support_groupleader = divy_support_groupleader(r);
	int support_access_control = divy_support_access_control(r);
	divy_sbuf *sql_buf = NULL;

	TRACE(p);

	if (src_usr_pr == NULL || dst_usr_pr == NULL ||
	    IS_EMPTY(src_usr_pr->usrid) || IS_EMPTY(src_usr_pr->rsid) ||
	    IS_EMPTY(src_usr_pr->prvcol_uri) || IS_EMPTY(dst_usr_pr->usrid)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"There are empty values.");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* 
	 * 移動に必要なリソースを作る
	 */
	src_rdb_r.uri          = src_usr_pr->prvcol_uri;
	src_rdb_r.displayname  = src_usr_pr->usrid;
	src_rdb_r.resourcetype = DIVY_TYPE_COLLECTION;
	src_rdb_r.depth        = divy_count_dirs(src_rdb_r.uri);
	src_rdb_r.rsid         = src_usr_pr->rsid;
	divy_parse_uri(p, dav_divy_get_root_uri(r),
				src_rdb_r.uri, &src_rdb_r.u_spec);
	dst_rdb_r.uri          = divy_build_user_uri(p, dav_divy_get_root_uri(r),
							dst_usr_pr->usrid);
	dst_rdb_r.displayname  = dst_usr_pr->usrid;
	dst_rdb_r.resourcetype = DIVY_TYPE_COLLECTION;
	dst_rdb_r.depth        = divy_count_dirs(dst_rdb_r.uri);

	/*
	 * ##### FIXME ユーザIDの変更を実施するとデッドロックする可能性が極めて高い。
	 * 	現テーブルの構成上、これを防ぐことは非常に難しい。
	 */

	/*
	 * ユーザ情報の移動 (divy_usr の更新)
	 * (note)
	 * 	divy_usrdiskquota から参照されているため、単純にUPDATEすることが
	 * 	出来ない。そのため、エントリーをコピー＆削除することで、
	 * 	対応している。
	 */
	/* SQL文を組み立てる */
	divy_sbuf_create(p, &sql_buf, 256);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"INSERT INTO divy_usr "
				"(usr_usr_id_vc,"
				" usr_passwd_vc,"
				" usr_fullname_vc,"
				" usr_mailaddr_vc,"
				" usr_admin_mod_i,"
				" usr_rs_id_c,"
				" usr_usr_seq_i,"
				" usr_last_access_bi,"
				" usr_last_accesscl_vc,"
				" usr_regist_c,"
				" usr_update_c,"
				" usr_comment_vc");

	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
				", usr_expiration_bi, usr_extended_status_c");
	}

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				", usr_owner_usr_id_vc, usr_maxcreation_i");
	}

	if (support_access_control) {
		divy_sbuf_append(sql_buf,
				", usr_allowhosts_vc");
	}

	divy_sbuf_append(sql_buf,
				") "
				"(SELECT "DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_USRID_LEN))", "
				" usr_passwd_vc,"
				" usr_fullname_vc,"
				" usr_mailaddr_vc,"
				" usr_admin_mod_i,"
				" usr_rs_id_c,"
				" usr_usr_seq_i,"
				" usr_last_access_bi,"
				" usr_last_accesscl_vc,"
				" usr_regist_c,"
				DIVY_DBEXPR_CAST("?", DIVY_COLTYPE_VARCHAR(DIVY_DB_UPDATEDT_LEN))","
				" usr_comment_vc");

	if (support_extstatus) {
		divy_sbuf_append(sql_buf,
				", usr_expiration_bi, usr_extended_status_c");
	}

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				", usr_owner_usr_id_vc, usr_maxcreation_i");
	}

	if (support_access_control) {
		divy_sbuf_append(sql_buf,
				", usr_allowhosts_vc");
	}

	divy_sbuf_append(sql_buf,
				" FROM divy_usr"
				" WHERE usr_usr_id_vc = ?)");

	/* SQLの準備 */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* 日付の作成 (ISO8601形式) */
	if (divy_format_time_t(p, short_time, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	stmt->setString(stmt, 1, dst_usr_pr->usrid);
	stmt->setString(stmt, 2, datebuf);
	stmt->setString(stmt, 3, src_usr_pr->usrid);
        /* SQL実行　失敗時はエラー */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to insert divy_usr for dst_usr_pr."
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * ユーザQuota のユーザIDを変更 (divy_usrdiskquota)
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_usrdiskquota "
				"SET usqt_usr_id_vc = ? "
				"WHERE usqt_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	stmt->setString(stmt, 1, dst_usr_pr->usrid);
	stmt->setString(stmt, 2, src_usr_pr->usrid);
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_usrdiskquota."
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * divy_usraccessdeny の変更
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_usraccessdeny "
				"SET usad_usr_id_vc = ? "
				"WHERE usad_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	stmt->setString(stmt, 1, dst_usr_pr->usrid);
	stmt->setString(stmt, 2, src_usr_pr->usrid);
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_usraccessdeny for MOVE."
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * src_usr_pr のエントリを削除する
	 * (note) dst_usr_pr のエントリはCOPYして作ったので、古いエントリは削除する
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"DELETE FROM divy_usr "
				"WHERE usr_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	stmt->setString(stmt, 1, src_usr_pr->usrid);
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to delete divy_usr of src_usr_pr."
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * プライベートコレクションのURI変更
	 */
	if (divy_rdbo_move_resource(r, &src_rdb_r, &dst_rdb_r, ts_ctx)) {

		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to move private collection."
				"(src_userid = %s, dst_userid = %s)",
				src_usr_pr->usrid, dst_usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/*
	 * ロックテーブル の認証ユーザを変更 (dav_lock)
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"UPDATE dav_lock "
				"SET lk_auth_usr_id_vc = ? "
				"WHERE lk_auth_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	stmt->setString(stmt, 1, dst_usr_pr->usrid);
	stmt->setString(stmt, 2, src_usr_pr->usrid);
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update dav_lock."
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * mailwatch の監視ユーザIDを変更 (divy_mailwatch)
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_mailwatch "
				"SET mw_usr_id_vc = ? "
				"WHERE mw_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	stmt->setString(stmt, 1, dst_usr_pr->usrid);
	stmt->setString(stmt, 2, src_usr_pr->usrid);
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_mailwatch."
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * グループ所属ユーザのユーザIDを変更 (divy_grpmem)
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_grpmem "
				"SET grpm_usr_id_vc = ? "
				"WHERE grpm_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	stmt->setString(stmt, 1, dst_usr_pr->usrid);
	stmt->setString(stmt, 2, src_usr_pr->usrid);
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_grpmem."
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * リソース の作成者ユーザIDを変更 (dav_resource)
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"UPDATE dav_resource "
				"SET rs_creator_usr_id_vc = ? "
				"WHERE rs_creator_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	stmt->setString(stmt, 1, dst_usr_pr->usrid);
	stmt->setString(stmt, 2, src_usr_pr->usrid);
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update dav_resource."
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	/*
	 * リソース更新者ユーザIDを変更 (dav_resource)
	 */
	stmt = dbconn->prepareStatement(dbconn,
				"UPDATE dav_resource "
				"SET rs_lastmodifier_usr_id_vc = ? "
				"WHERE rs_lastmodifier_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. "
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}

	stmt->setString(stmt, 1, dst_usr_pr->usrid);
	stmt->setString(stmt, 2, src_usr_pr->usrid);
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update dav_resource."
				"(src_userid = %s, dst_userid = %s) Reason: %s",
				src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

#ifdef DIVY_SUPPORT_PASSPOLICY
	/*
	 * パスワードポリシー状態のユーザIDを変更する
	 */
	if (divy_rdbo_move_user_passwordpolicy(r, src_usr_pr->usrid, dst_usr_pr->usrid, ts_ctx)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to move user based passwordpolicy state."
				"(src_userid = %s, dst_userid = %s)",
				src_usr_pr->usrid, dst_usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}

	/*
	 * パスワード履歴ののユーザIDを変更する
	 */
	if (divy_rdbo_move_user_passhistory(r, src_usr_pr->usrid, dst_usr_pr->usrid, ts_ctx)) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to move user based password history data."
				"(src_userid = %s, dst_userid = %s)",
				src_usr_pr->usrid, dst_usr_pr->usrid);
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);

		return 1;
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/*
	 * ごみ箱の削除者IDを変更する
	 */
	if (divy_support_trashfolder(r)) {

		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_trash_info "
				"SET tr_deleter_id_vc = ? "
				"WHERE tr_deleter_id_vc = ?", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. "
					"(src_userid = %s, dst_userid = %s) Reason: %s",
					src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		stmt->setString(stmt, 1, dst_usr_pr->usrid);
		stmt->setString(stmt, 2, src_usr_pr->usrid);
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to update divy_trash_info."
					"(src_userid = %s, dst_userid = %s) Reason: %s",
					src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/*
	 * src がオーナとなっていたユーザのオーナIDを変更する
	 */
	if (support_groupleader) {

		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_usr "
				"SET usr_owner_usr_id_vc = ? "
				"WHERE usr_owner_usr_id_vc = ?", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. "
					"(src_userid = %s, dst_userid = %s) Reason: %s",
					src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		stmt->setString(stmt, 1, dst_usr_pr->usrid);
		stmt->setString(stmt, 2, src_usr_pr->usrid);
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to update divy_usr."
					"(src_userid = %s, dst_userid = %s) Reason: %s",
					src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/*
	 * src が任命されていたグループのリーダIDを変更する
	 */
	if (support_groupleader) {
		stmt = dbconn->prepareStatement(dbconn,
				"UPDATE divy_grp "
				"SET grp_owner_usr_id_vc = ? "
				"WHERE grp_owner_usr_id_vc = ?", p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to get DbPreparedStmt. "
					"(src_userid = %s, dst_userid = %s) Reason: %s",
					src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}

		stmt->setString(stmt, 1, dst_usr_pr->usrid);
		stmt->setString(stmt, 2, src_usr_pr->usrid);
		(void) stmt->executeUpdate(stmt, p);
		if (stmt->getCode(stmt) != DB_SUCCESS) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
					"Failed to update divy_grp."
					"(src_userid = %s, dst_userid = %s) Reason: %s",
					src_usr_pr->usrid, dst_usr_pr->usrid, stmt->getMsg(stmt));
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			if (stmt != NULL) stmt->close(stmt); stmt = NULL;

			return 1;
		}
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	}

	/*
	 * 開封通知のユーザIDを変更する
	 */
	if (divy_support_confirmreading(r)) {
		if (divy_rdbo_move_confirmreading_userid(r, src_usr_pr->usrid,
													dst_usr_pr->usrid, ts_ctx)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to update divy_confirmreading.");
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);

			return 1;
		}
	}

	/* 反映を確定する */
	divy_db_commit_transaction(ts_ctx);
	return 0;
}

/**
 * ownerid が示すグループリーダが、自分自身の管理下に置かれた
 * ユーザを保持しているかどうか.
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_owner_users(request_rec *r, const char *ownerid, int *has_user)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int support_groupleader = divy_support_groupleader(r);
	divy_db_transaction_ctx *ts_ctx = NULL;

	*has_user = 0;	/* 持っていない */

	if (IS_EMPTY(ownerid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"ownerid is empty.");
		return 1;
	}

	/* グループリーダ機能がサポートされていなければ、
	 * オーナユーザは一人も居ないとするしかない */
	if (!support_groupleader) return 0;

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, 
					"SELECT "
					"usr_usr_id_vc "
					"FROM divy_usr "
					"WHERE usr_owner_usr_id_vc = ? "
#if defined(DIVY_DBMS_POSTGRES)	/* postgres */
					"OFFSET 0 LIMIT ?"
#elif defined(DIVY_DBMS_ORACLE) /* oracle */
					"AND ROWNUM <= ?"
#elif defined(DIVY_DBMS_DB2)	/* DB2 */
					"FETCH FIRST ? ROWS ONLY"
#endif
				, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (userid = %s) "
				"(Reason: %s)", ownerid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, ownerid);
	stmt->setInt(stmt, 2, 1);

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select divy_usr.(userid = %s) "
				"Reason: %s", ownerid, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		*has_user = 1;	/* ユーザがいた */
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * ownerid が示すグループリーダが管理しているユーザの人数を返却する
 *
 */
DIVY_DECLARE(int) divy_rdbo_count_owner_users(request_rec *r, const char *ownerid, int *nuser)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	int support_groupleader = divy_support_groupleader(r);
	divy_db_transaction_ctx *ts_ctx = NULL;

	*nuser = 0;	/* 0人 */

	if (IS_EMPTY(ownerid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"ownerid is empty.");
		return 1;
	}

	/* グループリーダ機能がサポートされていなければ、
	 * オーナユーザは一人も居ないとするしかない */
	if (!support_groupleader) return 0;

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文の準備 */
	stmt = dbconn->prepareStatement(dbconn, 
					"SELECT "
					"COUNT(usr_usr_id_vc) "
					" FROM divy_usr "
					" WHERE usr_owner_usr_id_vc = ? ", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (userid = %s) "
				"(Reason: %s)", ownerid, stmt->getMsg(stmt));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	/* バインド */
	stmt->setString(stmt, 1, ownerid);
	stmt->setInt(stmt, 2, 1);

	/* SQL実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to select divy_usr.(userid = %s) "
				"Reason: %s", ownerid, rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	if (rset->next(rset) == DB_TRUE) {
		/* (note) 作成ユーザ数は、INT_MAX の範囲に収まるはず */
		*nuser = (int) rset->getBigInt(rset, 1);
	}

	divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * Other ユーザuserid をownerid が示すユーザの管理下に置くようDBを変更する.
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_user_ownerinfo(request_rec *r, const char *userid,
							const char *ownerid, divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	apr_pool_t *p           = r->pool;
	int iscommit            = 0;
	char *datebuf           = NULL;
	time_t updatedt;

	if (IS_EMPTY(userid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"userid is EMPTY.");
		return 1;
	}

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		/* ここでは作成せずエラーにする */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Could not operation. Transaction ctx is NULL.");
		return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/*
	 * divy_usr テーブルのオーナIDと更新日付を更新する
	 */
	stmt = dbconn->prepareStatement(dbconn, 
				"UPDATE divy_usr "
				"SET "
				" usr_owner_usr_id_vc = ?,"
				" usr_update_c = ? "
				"WHERE usr_usr_id_vc = ?", p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. Reason: %s",
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	updatedt = dav_divy_get_now_epoch();
	if (divy_format_time_t(p, updatedt, DIVY_TIME_STYLE_ISO8601, &datebuf)) {
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* バインド */
	stmt->setString(stmt, 1, ownerid);
	stmt->setString(stmt, 2, datebuf);
	stmt->setString(stmt, 3, userid);

	/* SQLを実行 */
	(void) stmt->executeUpdate(stmt, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to update divy_usr. (target user = %s, Reason: %s)",
				userid, stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;
	if (iscommit) divy_db_commit_transaction(ts_ctx);

	return 0;
}

/**
 * mail addressが該当するユーザ一覧をすべて取得する。主にSAML向け
 *
 * @param r request_rec *
 * @param mailaddr const char* メールアドレス
 * @param usr_pr divy_rdbo_usr **	ユーザ構造体
 * @param count	int * 該当ユーザ数
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_get_userlist_from_mail(request_rec *r, const char* mailaddr, divy_rdbo_usr **usr_pr, int *count)
{
	DbConn          *dbconn = NULL;
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	apr_pool_t *p           = r->pool;
	divy_sbuf *sql_buf		= NULL;
	divy_db_transaction_ctx *ts_ctx = NULL;
	divy_rdbo_usr *usr = NULL;
	int support_groupleader = divy_support_groupleader(r);
	int idx = 0;

	*usr_pr = NULL;
	*count = 0; /* 0人 */

	if (IS_EMPTY(mailaddr)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"mail address is empty");
		return 1;
	}

	/* トランザクションコンテキストを生成する */
	if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	/* SQL文を組み立てる */
	divy_sbuf_create(p, &sql_buf, 1024);	/* SQLバッファの作成 */
	divy_sbuf_append(sql_buf,
				"SELECT"
				" usr.usr_usr_id_vc"
				", usr.usr_fullname_vc"
				", usr.usr_mailaddr_vc"
				", usr.usr_regist_c"
				", usr.usr_update_c"
				", usr.usr_last_access_bi");

	/* グループ管理者機能をサポートしているか? */
	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				",usr.usr_owner_usr_id_vc"
				",usr2.usr_fullname_vc AS ownername");
	}

	divy_sbuf_append(sql_buf,
				" FROM divy_usr usr");

	if (support_groupleader) {
		divy_sbuf_append(sql_buf,
				" LEFT JOIN divy_usr usr2"
				" ON (usr.usr_owner_usr_id_vc = usr2.usr_usr_id_vc)"
				" WHERE usr.usr_mailaddr_vc LIKE ?");
	}

	/* SQL文の準備 失敗時はエラー */
	stmt = dbconn->prepareStatement(dbconn, divy_sbuf_tostring(sql_buf), p);
	if (stmt->getCode(stmt) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt. (Reason: %s)", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* バインド */
	if (support_groupleader) {
		stmt->setString(stmt, 1, apr_pstrcat(p, mailaddr, "%", NULL));
	}

	/* SQL実行 失敗時はエラー */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS){
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet. (Reason: %s)", rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL)   rset->close(rset); rset = NULL;
		if (stmt != NULL)   stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* フェッチ(データありの間) */
	while (rset->next(rset) == DB_TRUE) {
		idx = 1;
		if (*usr_pr == NULL) {
			*usr_pr = usr = apr_pcalloc(p, sizeof(divy_rdbo_usr));
		}
		else {
			usr->next = apr_pcalloc(p, sizeof(divy_rdbo_usr));
			usr = usr->next;
		}

		/* データ格納(各オプション共通) */
		usr->usrid    = rset->getString(rset, idx++);
		usr->fullname = rset->getString(rset, idx++);
		usr->mailaddr	= rset->getString(rset, idx++);
		usr->registdt	= rset->getString(rset, idx++);
		usr->updatedt 	= rset->getString(rset, idx++);
		usr->lastaccess = rset->getBigInt(rset, idx++);
		if (support_groupleader) {
			usr->ownerid         = rset->getString(rset, idx++);
			usr->ownername       = rset->getString(rset, idx++);
		}

		(*count)++;
	}

	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}



/*------------------------------------------------------------------------------
  Define private functions 
  ----------------------------------------------------------------------------*/

