/**
 * $Id$
 *
 * tf_rdbo には分類できなかった関数を集めたファイル.
 *
 */
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_hash.h"
#include "apr_time.h"
#include "apr_thread_mutex.h"

#include "httpd.h"

#include "mod_dav_tf.h"
#include "tf_rdbo.h"
#include "tf_rdbo_util.h"
#include "util_common.h"
#include "tf_valuecache.h"
#include "util.h"
#include "tf_folder.h"
#include "tf_valuecache.h"
#include "liveprop.h"

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * divy_rdbo_usr.extstatus の定義 & 利用する定義値および配列の定義
 * (note)
 * 	値およびフォーマット位置などはテーブルdivy_usr の仕様に従います。
 * 	ですので勝手に変更することは厳禁です。変えるのなら仕様と一緒に。
 */
/* ユーザ/グループ拡張ステータスフィールドの"OFF"を表す文字 */
#define EXTSTATUS_C_OFF '-'

/* ユーザ/グループ拡張ステータスフィールドの"将来拡張フィールド"を表す文字 */
#define EXTSTATUS_C_RESERVED '*'

/* ユーザ/グループ拡張ステータスフィールドの長さ(byte)
 * (note) テーブルのフォーマットとあわせること */
#define EXTSTATUS_LEN	32

/* divy_extstatus_elt.state のON/OFF 値 */
enum {
	EXTSTATUS_UNDEFINE = 0,
	EXTSTATUS_OFF,
	EXTSTATUS_ON
};

/*
 * ある1つのユーザ/グループ拡張ステータス値を保持する構造体
 */
typedef struct __divy_extstatus_elt	divy_extstatus_elt;
struct __divy_extstatus_elt {
	/* val を分類付ける値 (USR_EXTSTATUS_TYPE_xxx) */
	int type;

	/* データとして受け取った値 */
	char raw;

	/* このステータスがON/OFFどちらの状態なのか? */
	int state;	/* UNDEFINE / ON / OFF */
};

/*
 * 拡張ステータスを保持するコンテナ構造体の定義
 * (note)
 * 	不完全型の定義です。宣言は tf_rdbo_util.h にあります */
struct __divy_rdbo_extstatus {
	apr_pool_t *p;	/* 作業用のプール */

	/* パースされていない生の拡張ステータスを表す文字列 */
	const char *raw;

	/* 各ステータスを保持する構造体の配列(raw のパース結果) */
	divy_extstatus_elt *elts;

	/* elts の長さ */
	int elen;

	/* 拡張ステータスの種類 */
	divy_rdbo_extstatus_type type;
};

/* type に応じてelen の長さを算出するマクロ */
#define GET_ELEN(type)	(((type) == EXTSTATUS_TYPE_USR) ? USR_EXTSTATUS_TYPE_END : GRP_EXTSTATUS_TYPE_END)

/* type に応じてextstatus がon の時の値を取得するマクロ */
#define GET_EXTSTATUS_ON(type,i)	(((type) == EXTSTATUS_TYPE_USR) ? _usr_extstatus_on[i] : _grp_extstatus_on[i])

/**
 * ユーザ拡張ステータス専用の定義
 */

/* divy_rdbo_extstatus.elt, _usr_extstatus_default, _usr_extstatus_on の
 * インデックス番号。
 * この番号に対応したステータスがelt のデータとして格納されてます */
enum {
	USR_EXTSTATUS_TYPE_ACTIVE    = 0,  /* アクティブかどうか             */
	USR_EXTSTATUS_TYPE_READ      = 1,  /* read 権限があるかどうか        */
	USR_EXTSTATUS_TYPE_UPLOAD    = 2,  /* upload 権限があるかどうか      */
	USR_EXTSTATUS_TYPE_READWRITE = 3,  /* readwrite 権限があるかどうか   */
	USR_EXTSTATUS_TYPE_SETVIEW   = 4,  /* view属性付加権限があるかどうか */
	USR_EXTSTATUS_TYPE_TRUST     = 5,  /* 信頼されたユーザかどうか       */
	USR_EXTSTATUS_TYPE_SYSEXEC   = 6,  /* システム実行権限を持つユーザか */
	USR_EXTSTATUS_TYPE_GRPCONSTRAINTS_IGN = 7,	/* グループ制約属性「無視」の有無 */
	USR_EXTSTATUS_TYPE_GROUPLEADER = 8,	/* グループリーダかどうか        */
	USR_EXTSTATUS_TYPE_CONTROLOTHER = 9,/* Other ユーザを管理下に置くことができるかどうか */
	USR_EXTSTATUS_TYPE_LOCKOUT   = 10, /* アカウントがロックされた */

	USR_EXTSTATUS_TYPE_END	/* sentinel */
};

/**
 * グループ拡張ステータス(=グループ制約属性) 専用の定義
 */

/* グループ拡張ステータスを表す文字列データのインデックス番号 */
enum {
	GRP_EXTSTATUS_TYPE_WRITE    = 0,	/* 書き込み制約       */
	GRP_EXTSTATUS_TYPE_OPLOG    = 1,	/* 操作ログ表示制約   */
	GRP_EXTSTATUS_TYPE_SHOWPROP = 2,	/* プロパティ表示制約 */
	GRP_EXTSTATUS_ACTIVE_GROUP  = 3,	/* アクティブかどうか */
	GRP_EXTSTATUS_PROP_BOX      = 4,    /* BOXグループかどうか*/
	GRP_EXTSTATUS_UPLOADPOLICY  = 5, /* アップロードポリシーがあるか*/

	GRP_EXTSTATUS_TYPE_END	/* sentinel */
};

/*------------------------------------------------------------------------------
  Declare and Define structure
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define static values
  ----------------------------------------------------------------------------*/
/* ユーザ拡張ステータスフィールドのデフォルト値 */
static const char  _usr_extstatus_default[] = "a--wvt-----*********************";

/* ユーザ拡張ステータスフィールドが全てONになった場合の文字列 */
static const char _usr_extstatus_on[] = "aruwvtsgLox*********************";

/* グループ拡張ステータス(グループ制約属性) のデフォルト値 */
static const char  _grp_extstatus_default[] = "---a--**************************";

/* グループ拡張ステータス(グループ制約属性) のフィールドが全てONになった場合の文字列 */
static const char _grp_extstatus_on[] = "wopabu**************************";
/*                                       ||||||
 * 書き込み制約         -----------------+|||||
 * 操作ログ制約         -------------------+|||
 * プロパティ表示制約   --------------------+||
 * TeamFileBOX利用可能  ---------------------+|
 * アップロードポリシー ----------------------+
 */

/**
 * divy_rdbo_tableinfo の配列定義(basicsearch 用)
 */
static const divy_rdbo_tableinfo prop_tableinfo_list[] = {

	{	DAV_DIVY_PROPID_creationdate,
		"rs_create_bi",
		DIVY_RESOURCE_TABLE,
		DIVY_RESOURCE_TABLE_REF,
		DIVY_JOIN_NONE, NULL, NULL
	},
	{ 	DAV_DIVY_PROPID_displayname,
		"rs_dispname_vc",
		DIVY_RESOURCE_TABLE,
		DIVY_RESOURCE_TABLE_REF,
		DIVY_JOIN_NONE, NULL, NULL
	},
	{ 	DAV_DIVY_PROPID_getcontentlanguage,
		"rs_get_cont_lang_vc",
		DIVY_RESOURCE_TABLE,
		DIVY_RESOURCE_TABLE_REF,
		DIVY_JOIN_NONE, NULL, NULL
	},
	{ 	DAV_DIVY_PROPID_getcontentlength,
		"rs_get_cont_len_bi",
		DIVY_RESOURCE_TABLE,
		DIVY_RESOURCE_TABLE_REF,
		DIVY_JOIN_NONE, NULL, NULL
	},
	{ 	DAV_DIVY_PROPID_getcontenttype,
		"rs_get_cont_type_vc",
		DIVY_RESOURCE_TABLE,
		DIVY_RESOURCE_TABLE_REF,
		DIVY_JOIN_NONE, NULL, NULL
	},
	{ 	DAV_DIVY_PROPID_getetag,
		"rs_get_etag_txt",
		DIVY_RESOURCE_TABLE,
		DIVY_RESOURCE_TABLE_REF,
		DIVY_JOIN_NONE, NULL, NULL
	},
	{ 	DAV_DIVY_PROPID_getlastmodified,
		"rs_get_lastmodified_bi",
		DIVY_RESOURCE_TABLE,
		DIVY_RESOURCE_TABLE_REF,
		DIVY_JOIN_NONE, NULL, NULL
	},
	{ 	DAV_DIVY_PROPID_resourcetype,
		"rs_resourcetype_i",
		DIVY_RESOURCE_TABLE,
		DIVY_RESOURCE_TABLE_REF,
		DIVY_JOIN_NONE, NULL, NULL
	},
	{ 	DAV_DIVY_PROPID_creator,
		"usr_fullname_vc AS rs_creator_vc",
		"divy_usr",
		"u1",
		DIVY_JOIN_LEFT,
		DIVY_RESOURCE_TABLE_REF".rs_creator_usr_id_vc",
		"u1.usr_usr_id_vc"
	},
	{ 	DAV_DIVY_PROPID_lastmodifier,
		"usr_fullname_vc AS rs_lastmodifier_vc",
		"divy_usr",
		"u2",
		DIVY_JOIN_LEFT,
		DIVY_RESOURCE_TABLE_REF".rs_lastmodifier_usr_id_vc",
		"u2.usr_usr_id_vc"
	},
	{0, 	NULL , NULL, NULL, DIVY_JOIN_NONE, NULL }	/* sentinel */
};

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static void _set_extstatus(divy_rdbo_extstatus *extstatus, int idx, int on);
static int _has_extstatus(const divy_rdbo_extstatus *extstatus, int idx);

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 *  リポジトリDB からリソースIDを採番して取得する。
 *  (note)
 *    ts_ctx のトランザクションを継続します.
 * 
 */
DIVY_DECLARE(int) divy_rdbo_create_rsid(request_rec *r, char **rsid, divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	DbResultSet     *rset    = NULL;
	apr_pool_t      *p       = r->pool;
	int iscommit             = 0;

	*rsid = NULL;	/* 初期化 */

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	stmt = dbconn->prepareStatement(dbconn, 
			"SELECT "DIVY_DBFUNC_CREATE_RES_SEQ 
			/* oracle の場合取得元テーブルが必要 */
#if defined(DIVY_DBMS_ORACLE)	/* oracle */
			" FROM dual"
#elif defined(DIVY_DBMS_DB2)	/* db2	*/
			/* db2 はシステムテーブルsysdummyが必要 */
			" FROM SYSIBM.SYSDUMMY1"
#endif
			, p);

	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt for select res_seq. "
				"Reason: %s", stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		*rsid = NULL;
		return 1;
	}

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbResultSet for select res_seq."
			"Reason: %s", rset->getMsg(rset));

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* 結果の取得 */
	if (rset->next(rset) == DB_TRUE) {
		*rsid = rset->getString(rset, 1);
	}
	else {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA, "Failed to get res_seq.");

		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	if (iscommit) divy_db_commit_transaction(ts_ctx);

	ERRLOG3(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"create resource id. (rsid = %s | shorten = %s | uri = %s)", *rsid, divy_get_rid2shorten(p, *rsid, NULL), r->uri);

	return 0;
}

/**
 * リソースID・グループIDを採番号して返却する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_create_rsgrpid(request_rec *r, char **rsid, char **grpid,
											divy_db_transaction_ctx *ts_ctx)
{
	DbConn          *dbconn  = NULL;
	DbPreparedStmt  *stmt    = NULL;
	DbResultSet     *rset    = NULL;
	apr_pool_t      *p       = r->pool;
	int iscommit             = 0;

	/* 現在のトランザクションの状態を調べる */
	if (!divy_db_is_transaction_valid_state(ts_ctx)) return 1;

	/* トランザクションがない */
	if (ts_ctx == NULL) {
		iscommit = 1;
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) return 1;
	}

	/* トランザクションを開始する */
	if (divy_db_start_transaction(ts_ctx)) return 1;
	dbconn = ts_ctx->dbconn;

	stmt = dbconn->prepareStatement(dbconn, 
			"SELECT "DIVY_DBFUNC_CREATE_RES_SEQ", "
			DIVY_DBFUNC_CREATE_GRP_SEQ
			/* oracle の場合取得元テーブルが必要 */
#if defined(DIVY_DBMS_ORACLE)	/* oracle */
			" FROM dual"
#elif defined(DIVY_DBMS_DB2)	/* db2	*/
			/* db2の場合はシステムテーブルが必要 */
			" FROM SYSIBM.SYSDUMMY1"
#endif
			, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbPreparedStmt for select "
				"res_seq or grp_seq. Reason: %s", 
				stmt->getMsg(stmt));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (stmt != NULL) stmt->close(stmt);

		*rsid = NULL;
		*grpid = NULL;
		return 1;
	}

	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get DbResultSet for select "
				"res_seq or grp_seq. Reason: %s", 
				rset->getMsg(rset));
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;

		*rsid = NULL;
		*grpid = NULL;
		return 1;
	}

	/* 結果の取得 */
	if (rset->next(rset) == DB_TRUE) {
		*rsid  = rset->getString(rset, 1);
		*grpid = rset->getString(rset, 2);
	}
	else {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to get res_seq or grp_seq.");
		ts_ctx->status |= DIVY_TRANS_ABORT;
		if (iscommit) divy_db_rollback_transaction(ts_ctx);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		*rsid = NULL;
		*grpid = NULL;

		return 1;
	}

	if (iscommit) divy_db_commit_transaction(ts_ctx);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定された文字列 str からエスケープされるワイルドカードを探し、
 * dbに対応した形に変換して返却する。
 * 
 */
DIVY_DECLARE(char *) divy_rdbo_escape_wildcard(apr_pool_t *p, const char *str)
{
	char convstr[(strlen(str) * 2) + 1];
	char *start, *stop, *find;
	char esc[] 	= "\\";
	int esclen	= strlen(esc);

	/* 文字列なしはNULL */
	if (IS_EMPTY(str)) return NULL;

	/* 
	 * 文字列コピー・開始/終了設定
	 */
	strcpy(convstr, str);
	start 	= (char *) &convstr;
	stop 	= (char *) &convstr + strlen(convstr);

	/* 
	 * エスケープ＋変換文字を探して変換していく
	 */
	/* '\'を探す */
	while ((find = divy_strstr(start, esc)) != NULL){

		start = (char *) (find + 1);	/* 隣へ移動 */

		if (strncmp(start, "%", 1) == 0 || strncmp(start, "_", 1) 
									== 0){

			/* 隣が%か_の場合は後ろへ移動後\を追加 */
			memmove(start + esclen, start, stop - start + 1);
			memcpy(start, esc, esclen);

			/* 開始・終了位置移動 */
			start 	= start + esclen;
			stop 	= stop + esclen;
		}

	}
	return apr_pstrdup(p, convstr);
}

/**
 * 指定されたdbconn がアクティブなコネクションであることを検証する。
 * (note)
 * 	この関数は、DBコネクション管理マネージャが使用します。
 * 	特殊な考慮が入っていますので、他からは使用しないで下さい。
 */
DIVY_DECLARE(int) divy_rdbo_validate_activedbconn(apr_pool_t *p, DbConn *dbconn)
{
	DbPreparedStmt  *stmt   = NULL;
	DbResultSet     *rset   = NULL;
	const char *sql;

	if (dbconn == NULL) {
		return 1;	/* 実行できません */
	}

	/* 実行対象のSQLを取得する */
	sql = dbconn->getExecutableSQL(dbconn);
	if (IS_EMPTY(sql)) return 0;	/* 正常であるとします */

	/* トランザクション開始 */
	dbconn->startTrans(dbconn, 0);

	/* SQL文の解析 */
	stmt = dbconn->prepareStatement(dbconn, sql, p);
	if (stmt->getCode(stmt) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbPreparedStmt. (provider = %s)\n "
			"Reason: %s", dbconn->dbds->dbmsname, stmt->getMsg(stmt));

		dbconn->rollback(dbconn);
		if (stmt != NULL) stmt->close(stmt);
		return 1;
	}

	/* SQLの実行 */
	rset = stmt->executeQuery(stmt, p);
	if (rset->getCode(rset) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to execute \"executable sql\"(provider = %s)."
			"Reason: %s", dbconn->dbds->dbmsname, rset->getMsg(rset));

		dbconn->rollback(dbconn);
		if (rset != NULL) rset->close(rset); rset = NULL;
		if (stmt != NULL) stmt->close(stmt); stmt = NULL;
		return 1;
	}

	/* ここまで来たということは正常に終了したということ */
	dbconn->commit(dbconn);
	if (rset != NULL) rset->close(rset); rset = NULL;
	if (stmt != NULL) stmt->close(stmt); stmt = NULL;

	return 0;
}

/**
 * 指定されたプロパティ名の取得元となっているテーブル名とカラム名を
 * 構造体divy_rdbo_tableinfoとして返却する
 */
DIVY_DECLARE(divy_rdbo_tableinfo *)divy_rdbo_get_tableinfo(request_rec *r, 
                                             		const char *propname)
{
	apr_hash_t *table_hash = NULL;
	apr_pool_t *p          = r->server->process->pool;

	table_hash = divy_pcache_get_data(p, DIVY_PCACHE_DAT_GL_TABLEINFO);
	if (table_hash != NULL) {
		return apr_hash_get(table_hash, propname, APR_HASH_KEY_STRING);
	}
	else {
		int i;
		const dav_liveprop_spec *info;
		apr_status_t st;

		/* ロックをかける */
		st = apr_thread_mutex_lock(init_mutex);
		if (st != APR_SUCCESS) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to lock for creating table_hash. "
			        "Code = %d", st);
			/* 継続してみる */
		}

		/*
		 * (note) r->server->process->poolに格納しても良い根拠
		 * 	・このハッシュはコードが書き直されるまで変更されない。
		 * 	・全てのユーザ、ディレクティブが同じ値をReadできればよい
		 * 以上の条件を１つでも満たさない状況になったら、考え直して下さい。
		 */
		if (table_hash == NULL) {	/* ダブルチェックパターン */
			table_hash = apr_hash_make(p);
			/* 
			 * prop_tableinfo_list からプロパティIDを取得して
			 * その値を使ってLiveプロパティのプロパティ名を取り出す
			 */
			for (i = 0; prop_tableinfo_list[i].propid; i++) {
				dav_get_liveprop_info(
						prop_tableinfo_list[i].propid,
						&dav_divy_liveprop_group, &info);
				/* プロパティ名 - (divy_rdbo_tableinfo *) をセット */
				apr_hash_set(table_hash, info->name,
						APR_HASH_KEY_STRING,
						&prop_tableinfo_list[i]);
			}
			/* table_hash を サーバプールに入れる */
			divy_pcache_set_data(p, table_hash, DIVY_PCACHE_DAT_GL_TABLEINFO);
		}

		/* ロックを解除 */
		if (st == APR_SUCCESS) {	/* ロックを掛けるのに成功していた場合 */
			st = apr_thread_mutex_unlock(init_mutex);
		}
		if (st != APR_SUCCESS) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to unlock for creating table_hash. "
			        "Code = %d", st);
		}

		return apr_hash_get(table_hash, propname, APR_HASH_KEY_STRING);
	}

	return NULL;
}

/**
 * ログインしているユーザは有効期限切れか?
 *
 */
DIVY_DECLARE(int) divy_rdbo_is_expired_user(request_rec *r)
{
	time_t expiration = divy_get_expiration(r);

	/* 無期限 / 期限内以外ののアカウントか? */
	return (expiration != 0 && expiration < apr_time_sec(r->request_time));
}

/**
 * type が示す拡張ステータスを表す文字列 extstatus_str 値が正しい状態に
 * なっているかどうか検証する。
 *
 */
DIVY_DECLARE(int) divy_rdbo_validate_extstatus(apr_pool_t *p,
						const char *extstatus_str, divy_rdbo_extstatus_type type)
{
	int i;
	apr_size_t len;

	if (IS_EMPTY(extstatus_str)) return 1;	/* 空やNULLは不正ですよ */

	len = strlen(extstatus_str);
	if (len != EXTSTATUS_LEN) {
		return 1;	/* 長さが足りません */
	}

	for (i = 0; i < EXTSTATUS_LEN; i++) {
		if (!(extstatus_str[i] == EXTSTATUS_C_OFF ||
		     extstatus_str[i] == EXTSTATUS_C_RESERVED ||
		     extstatus_str[i] == GET_EXTSTATUS_ON(type,i))) {
			/* 上記の何れにも一致しない文字があれば不正である */
			return 1;
		}
	}

	return 0;	/* ここまで残ったら正常 */
}

/**
 * type が示す拡張ステータスを表す文字列 extstatus_str をパースする。
 *
 */
DIVY_DECLARE(divy_rdbo_extstatus *) divy_rdbo_parse_extstatus(
					apr_pool_t *p, const char *extstatus_str, divy_rdbo_extstatus_type type)
{
	divy_rdbo_extstatus *extstatus = NULL;
	divy_extstatus_elt *elt;
	int i;
	apr_size_t len;

	if (IS_EMPTY(extstatus_str) || (type != EXTSTATUS_TYPE_USR && type != EXTSTATUS_TYPE_GRP)) {
		return NULL;	/* 何もしない */
	}

	extstatus = apr_pcalloc(p, sizeof(divy_rdbo_extstatus));
	extstatus->p    = p;
	extstatus->raw  = extstatus_str;
	extstatus->elen = GET_ELEN(type);
	extstatus->elts = apr_pcalloc(p, sizeof(divy_extstatus_elt) * extstatus->elen);
	extstatus->type = type;

	len = strlen(extstatus_str);

	/* extstatus_str のパース */
	for (i = 0; i < extstatus->elen; i++) {
		elt = &extstatus->elts[i];
		elt->type = i;
		if (i < len) {
			elt->raw  = extstatus_str[i];
			/* 将来拡張のための予約の時 */
			if (extstatus_str[i] == EXTSTATUS_C_RESERVED) {
				elt->state = EXTSTATUS_UNDEFINE;
			}
			/* ON 値の時 */
			else if (extstatus_str[i] == GET_EXTSTATUS_ON(type,i)) {
				elt->state = EXTSTATUS_ON;
			}
			/* OFF の時 */
			else {
				elt->state = EXTSTATUS_OFF;
			}
		}
		/* len よりも短い時 */
		else {
			/* 外部仕様を厳密に適用すると len 
			 * よりも長くないとエラーであるはず。
			 * だが、ちょっと大目に見ましょう。
			 * 設定されていないところはOFFとします*/
			elt->raw   = EXTSTATUS_C_OFF;
			elt->state = EXTSTATUS_OFF;
		}
	}

	return extstatus;
}

/**
 * デフォルトのtype に応じた拡張ステータスを取得する。
 *
 */
DIVY_DECLARE(divy_rdbo_extstatus *) divy_rdbo_create_default_extstatus(apr_pool_t *p, divy_rdbo_extstatus_type type)
{
	if (type == EXTSTATUS_TYPE_USR) {
		return divy_rdbo_parse_extstatus(p, _usr_extstatus_default, type);
	}
	else if (type == EXTSTATUS_TYPE_GRP) {
		return divy_rdbo_parse_extstatus(p, _grp_extstatus_default, type);
	}

	return NULL;
}

/**
 * type が示す拡張ステータスの文字列表現を取得する。
 *
 */
DIVY_DECLARE(char *) divy_rdbo_extstatus2str(const divy_rdbo_extstatus *extstatus, divy_rdbo_extstatus_type type)
{
	char *extstatus_str;
	int i, state;

	if (extstatus == NULL || (type != EXTSTATUS_TYPE_USR && type != EXTSTATUS_TYPE_GRP)) return NULL;

	extstatus_str = apr_pcalloc(extstatus->p, sizeof(char) * (EXTSTATUS_LEN + 1));

	for (i = 0; i < extstatus->elen; i++) {
		state = extstatus->elts[i].state;
		if (state == EXTSTATUS_UNDEFINE) {
			extstatus_str[i] = EXTSTATUS_C_RESERVED;
		}
		else if (state == EXTSTATUS_ON) {
			extstatus_str[i] = GET_EXTSTATUS_ON(type,i);
		}
		else {
			extstatus_str[i] = EXTSTATUS_C_OFF;
		}
	}

	/* 残りには将来拡張用文字を詰める */
	memset(&extstatus_str[i], EXTSTATUS_C_RESERVED, (EXTSTATUS_LEN - extstatus->elen));
	extstatus_str[EXTSTATUS_LEN] = '\0'; /* 文字列にする */

	return extstatus_str;
}

/**
 * 指定されたtype の拡張ステータスextstatus に含まれるステータスのフラグ値を
 * 全てリセット(非アクティブ)にする.
 */
DIVY_DECLARE(void) divy_rdbo_reset_extstatus(divy_rdbo_extstatus *extstatus, divy_rdbo_extstatus_type type)
{
	int i;
	if (extstatus == NULL) return;

	for (i = 0; i < extstatus->elen; i++) {
		_set_extstatus(extstatus, i, 0/* OFF */);
	}
}

/**
 * type の示す拡張ステータスのワイルドカード文字列表現を取得する。
 * ワイルドカード文字列になるのはアクティブでないステータスだけです。
 *
 */
DIVY_DECLARE(char *) divy_rdbo_extstatus2wildcardstr(const divy_rdbo_extstatus *extstatus, divy_rdbo_extstatus_type type)
{
	char *extstatus_str;
	int i, state;

	if (extstatus == NULL || (type != EXTSTATUS_TYPE_USR && type != EXTSTATUS_TYPE_GRP)) return NULL;

	extstatus_str = apr_pcalloc(extstatus->p, sizeof(char) * (EXTSTATUS_LEN + 1));

	for (i = 0; i < extstatus->elen; i++) {
		state = extstatus->elts[i].state;
		if (state == EXTSTATUS_UNDEFINE) {
			extstatus_str[i] = EXTSTATUS_C_RESERVED;
		}
		else if (state == EXTSTATUS_ON) {
			extstatus_str[i] = GET_EXTSTATUS_ON(type,i);
		}
		else {
			extstatus_str[i] = '_';	/* ワイルドカード */
		}
	}

	/* 残りには将来拡張用文字を詰める */
	memset(&extstatus_str[i], EXTSTATUS_C_RESERVED, (EXTSTATUS_LEN - extstatus->elen));
	extstatus_str[EXTSTATUS_LEN] = '\0'; /* 文字列にする */

	return extstatus_str;
}

/**
 * 指定されたユーザ拡張ステータスextstatus に含まれるステータスの内、ユーザ権限フラグだけを
 * リセットする. 全てをリセットするには、divy_rdbo_reset_extstatus を使用して下さい.
 *
 */
DIVY_DECLARE(void) divy_rdbo_reset_userprivilege(divy_rdbo_extstatus *extstatus)
{
	if (extstatus == NULL) return;

	/* 権限フラグだけをリセットする */
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_READ,      0);
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_UPLOAD,    0);
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_READWRITE, 0);
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_SETVIEW,   0);
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_SYSEXEC,   0);
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_GRPCONSTRAINTS_IGN, 0);
}

/**
 * ユーザ拡張ステータスにアクティブフラグが立っているかどうか
 *
 */
DIVY_DECLARE(int) divy_rdbo_is_active_user(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, USR_EXTSTATUS_TYPE_ACTIVE);
}

/**
 * ユーザ拡張ステータスにアクティブフラグを立てる
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_active_user(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_ACTIVE, on);
}

/**
 * ユーザ拡張ステータスに信頼ユーザフラグが立っているかどうか
 *
 */
DIVY_DECLARE(int) divy_rdbo_is_trusty_user(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, USR_EXTSTATUS_TYPE_TRUST);
}

/**
 * ユーザ拡張ステータスに信頼ユーザフラグを立てる
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_trusty_user(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_TRUST, on);
}

/**
 * ユーザ拡張ステータスにグループリーダフラグが立っているかどうか.
 *
 */
DIVY_DECLARE(int) divy_rdbo_is_groupleader(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, USR_EXTSTATUS_TYPE_GROUPLEADER);
}

/**
 * ユーザ拡張ステータスにグループリーダフラグを立てる
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_groupleader(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_GROUPLEADER, on);
}

/**
 * ユーザ拡張ステータスにread 権限が付いているかどうか
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_read_privilege(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, USR_EXTSTATUS_TYPE_READ);
}

/**
 * ユーザ拡張ステータスにread 権限をつける
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_read_privilege(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_READ, on);
}

/**
 * ユーザ拡張ステータスにupload 権限が付いているかどうか
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_upload_privilege(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, USR_EXTSTATUS_TYPE_UPLOAD);
}

/**
 * ユーザ拡張ステータスにupload 権限をつける
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_upload_privilege(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_UPLOAD, on);
}

/**
 * ユーザ拡張ステータスにreadwrite 権限が付いているかどうか
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_readwrite_privilege(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, USR_EXTSTATUS_TYPE_READWRITE);
}

/**
 * ユーザ拡張ステータスにreadwrite 権限をつける
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_readwrite_privilege(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_READWRITE, on);
}

/**
 * ユーザ拡張ステータスにview属性設定 権限が付いているかどうか
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_setview_privilege(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, USR_EXTSTATUS_TYPE_SETVIEW);
}

/**
 * ユーザ拡張ステータスにview属性設定 権限をつける
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_setview_privilege(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_SETVIEW, on);
}

/**
 * ユーザ拡張ステータスにシステム実行権限が付いているかどうか
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_sysexec_privilege(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, USR_EXTSTATUS_TYPE_SYSEXEC);
}

/**
 * ユーザ拡張ステータスにシステム実行権限を付ける
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_sysexec_privilege(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_SYSEXEC, on);
}

/**
 * ユーザ拡張ステータスにグループ制約属性「無視」するが付いているかどうか.
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_user_groupconstraints_ignore(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, USR_EXTSTATUS_TYPE_GRPCONSTRAINTS_IGN);
}

/**
 * ユーザ拡張ステータスにグループ制約属性の「無視」状態を設定する.
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_user_groupconstraints_ignore(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_GRPCONSTRAINTS_IGN, on);
}

/**
 * ユーザ拡張ステータスに「Otherユーザを管理下に置くことができる」が付いているかどうか.
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_control_otheruser(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, USR_EXTSTATUS_TYPE_CONTROLOTHER);
}

/**
 * ユーザ拡張ステータスに「Otherユーザを管理下に置くことができる」状態を設定する.
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_control_otheruser(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_CONTROLOTHER, on);
}

/**
 * ユーザ拡張ステータスに「アカウントがロックされた」がついているかどうか
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_lockout(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, USR_EXTSTATUS_TYPE_LOCKOUT);
}

/**
 * ユーザ拡張ステータスに「アカウントがロックされた」を設定する
 */
DIVY_DECLARE(void) divy_rdbo_set_lockout(divy_rdbo_extstatus *extstatus, int on){
	_set_extstatus(extstatus, USR_EXTSTATUS_TYPE_LOCKOUT, on);
}

/**
 * グループ制約に書き込み制約がついているかどうか.
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_write_groupconstraints(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, GRP_EXTSTATUS_TYPE_WRITE);
}

/**
 * r を持つアクセスユーザが、指定されたuri, infotype のリソース・コレクションに対し
 * ユーザのグループ制約適用の有無やグループの書き込み制約を総合的に評価した上で
 * 書き込み制限が掛かっているかどうか判定する. (ユーティリティ関数)
 *
 * (note)
 *   * ユーザ拡張ステータスとグループ制約を総合的に判断します.
 *   * ただし、ユーザのread-only や upload-only は評価しません.
 *   * グループ単体の書き込み制約の有無は関数divy_rdbo_has_write_groupconstraints を使います.
 *   * アクセスユーザという観点で評価する点に注意. 他の関数群とは趣が違います.
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_write_constraints_on_uri(request_rec *r, const char *uri, divy_infotype infotype)
{
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	int support_groupleader = divy_support_groupleader(r);

	/*
	 * [ 条件 ]
	 *   * グループ制約属性機能が有効であること
	 *   * グループコレクション以下
	 *   * 一般ユーザ、制限ユーザグループリーダ(将来)&単なる所属グループ = 管理者以外
	 *   * グループ制約が"無視"されていないこと(制限ユーザは無視の有無に関わらず制約あり)
	 *   * グループ制約属性プロパティの書き込み制約を持っていること
	 *
	 * (note)
	 *   管理者 or グループ管理者&自身の管理グループであれば無視
	 */
	if (divy_support_grpconstraints(r) &&
		(infotype == DIVY_INFOTYPE_group_e        ||
		 infotype == DIVY_INFOTYPE_group_trash    ||
		 infotype == DIVY_INFOTYPE_group_trash_e0 ||
		 infotype == DIVY_INFOTYPE_group_trash_ex ||
		 infotype == DIVY_INFOTYPE_group_e_regular) &&
		divy_get_adminmode(r) != DIVY_ADMINMODE_ADMIN) {

		divy_rdbo_grp *grp_pr = NULL;
		divy_rdbo_extstatus *grp_extstatus = NULL;
		int use_user_groupconstraints = 0;	/* グループ制約による影響を受けるかどうか(0: 受けない) */

		/* グループリーダで自分自身のグループならば例外とする */
		if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus)) {
			/* グループプロパティを取得 */
			(void) divy_get_cached_availablegroup_by_uri(r, uri, &grp_pr);
			if (grp_pr != NULL &&
				IS_FILLED(grp_pr->ownerid) && strcmp(grp_pr->ownerid, divy_get_userid(r)) == 0) {
				return 0;	/* 制約が掛かっていないものとする */
			}
		}

		/* 制限ユーザ or グループ制約を"無視できない" ユーザか? */
		if (!divy_rdbo_is_trusty_user(own_extstatus) ||
			!divy_rdbo_has_user_groupconstraints_ignore(own_extstatus)) {
			/* (note)
			 * own_extstatus が存在しなければ、グループ制約の適用は「あり」が外部仕様.
			 * つまり、書き込み制約を受ける方向になるので問題は無い.
			 * 制限ユーザかどうかには影響するが、ここでは議論しません. */
			use_user_groupconstraints = 1;
		}

		/* グループ拡張ステータスを取得 */
		(void) divy_get_cached_availablegroupextstatus(r, uri, &grp_extstatus);

		/* グループ制約が適用されるべきユーザであり、書き込み制約を持っているかどうか */
		if (use_user_groupconstraints && divy_rdbo_has_write_groupconstraints(grp_extstatus)) {
			return 1;
		}
	}
	return 0;
}

/**
 * グループ制約に書き込み制約を設定する
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_write_groupconstraints(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, GRP_EXTSTATUS_TYPE_WRITE, on);
}

/**
 * グループ制約に操作ログ制約がついているかどうか.
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_oplog_groupconstraints(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, GRP_EXTSTATUS_TYPE_OPLOG);
}

/**
 * グループ制約に操作ログ制約を設定する
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_oplog_groupconstraints(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, GRP_EXTSTATUS_TYPE_OPLOG, on);
}

/**
 * グループ制約にプロパティ表示制約がついているかどうか.
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_showprop_groupconstraints(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, GRP_EXTSTATUS_TYPE_SHOWPROP);
}

/**
 * グループ制約にプロパティ表示制約を設定する
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_showprop_groupconstraints(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, GRP_EXTSTATUS_TYPE_SHOWPROP, on);
}

/**
 * グループ拡張ステータスにアクティブフラグが立っているかどうか.
 *
 */
DIVY_DECLARE(int) divy_rdbo_is_active_group(const divy_rdbo_extstatus *extstatus)
{
	int state;

	if (extstatus == NULL) {
		return 1;	/* デフォルトはアクティブ */
	}

	/* (note) 非アクティブに明示的に指定していなければアクティブとみなす.
	 * これは他のロジックとは異なる動きです. データ移行を忘れても
	 * 動作させるようにするためです. */
	state = extstatus->elts[GRP_EXTSTATUS_ACTIVE_GROUP].state;
	if (state != EXTSTATUS_OFF) {
		return 1;
	}
	return 0;
}

/**
 * グループ拡張ステータスにアクティブフラグをセットする.
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_active_group(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, GRP_EXTSTATUS_ACTIVE_GROUP, on);
}

/**
 * グループ拡張ステータスにBOX機能フラグが立っているかどうか.
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @return int	BOX機能が有効かどうか(1: 有効 / 0: 無効)
 */
DIVY_DECLARE(int) divy_rdbo_is_box_group(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, GRP_EXTSTATUS_PROP_BOX);
}

/**
 * グループ拡張ステータスにBOX有効フラグをセットする.
 *
 */
DIVY_DECLARE(void) divy_rdbo_set_box_group(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, GRP_EXTSTATUS_PROP_BOX, on);
}

/**
 * グループ拡張ステータスにアップロードポリシーフラグが立っているかどうか
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @return int	アップロードポリシー機能が有効かどうか(1: 有効 / 0: 無効)
 */
DIVY_DECLARE(int) divy_rdbo_is_uploadpolicy_group(const divy_rdbo_extstatus *extstatus)
{
	return _has_extstatus(extstatus, GRP_EXTSTATUS_UPLOADPOLICY);
}

/**
 * グループ拡張ステータスにアップロードポリシーフラグをセットする
 *
 * @param extstatus const divy_rdbo_extstatus * グループ拡張ステータスへののポインタ
 * @param on int フラグを立てることを表す変数(1: ON / 0: OFF)
 */
DIVY_DECLARE(void) divy_rdbo_set_uploadpolicy_group(divy_rdbo_extstatus *extstatus, int on)
{
	_set_extstatus(extstatus, GRP_EXTSTATUS_UPLOADPOLICY, on);
}

/**
 * r が示すアクセスユーザが管理リソースにアクセスできる権限を持っているかどうか.
 *
 * [ アクセスできる条件 ]
 *   * 管理者であること or
 *   * グループリーダであること
 *
 */
DIVY_DECLARE(int) divy_rdbo_has_administrative_privilege(request_rec *r)
{
	const divy_rdbo_extstatus *extstatus = NULL;

	if (divy_support_extenduserstatus(r)) {
		extstatus = divy_get_extstatus(r);
	}

	if (extstatus != NULL) {
		if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN ||
			(divy_support_groupleader(r) && divy_rdbo_is_groupleader(extstatus))) {
			return 1;
		}
	}
	else {
		if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN) {
			return 1;
		}
	}
	return 0;
}


/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * 拡張ステータスextstatus のidx 要素を設定する.
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータス
 * @param idx int ステータスタイプのインデックス
 * @param on int ON値 (1: ON / 0: OFF)
 */
static void _set_extstatus(divy_rdbo_extstatus *extstatus, int idx, int on)
{
	if (extstatus == NULL) return;

	/* idx 値が不正の場合 */
	if (idx < 0 || idx > extstatus->elen) {
		return;	/* 持っていないものとする */
	}

	if (on) {
		extstatus->elts[idx].state = EXTSTATUS_ON;
	}
	else {
		extstatus->elts[idx].state = EXTSTATUS_OFF;
	}
}

/**
 * 拡張ステータスextstatus がidxの示す種類の値を持っているかどうか.
 *
 * @param extstatus const divy_rdbo_extstatus * ユーザ拡張ステータス
 * @param idx int ステータスタイプのインデックス
 * @return int 持っているかどうか (1: 持っている / 0: 持っていない)
 */
static int _has_extstatus(const divy_rdbo_extstatus *extstatus, int idx)
{
	if (extstatus == NULL) return 0;	/* 持っていないものとする */

	/* idx 値が不正の場合 */
	if (idx < 0 || idx > extstatus->elen) {
		return 0;	/* 持っていないものとする */
	}

	/* ステータスがONの場合 */
	if (extstatus->elts[idx].state == EXTSTATUS_ON) {
		return 1;
	}
	else {
		return 0;
	}
}

