/**
 * $Id$
 *
 * tf_sqlparser.c
 *
 * SQLのパースを行う関数を定義するファイル
 *
 */
#include "httpd.h"
#include "http_config.h"
#include "apr_strings.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_thread_mutex.h"

/* グローバル変数の定義を行うためにdefineする */
#define SQLP_GLOBAL_VALUE_DEFINE

#include "tf_sqlparser.h"
#include "sqlparser.h"
#include "tf_rdbo.h"
#include "tf_rdbo_sql.h"
#include "util_common.h"

/*--------------------------------------------------------------
  Define fixed values and macro
  --------------------------------------------------------------*/

/*--------------------------------------------------------------
  Declare structures
  --------------------------------------------------------------*/
typedef struct __divy_sql_dependgraph	divy_sql_dependgraph;

/*--------------------------------------------------------------
  Define structures
  --------------------------------------------------------------*/
/**
 * SQL依存グラフを表す構造体
 */
struct __divy_sql_dependgraph {

	/*
	 * このグラフを構成する辺(Edge)の数
	 */
	unsigned int edge_cnt;

	/*
	 * グラフを構成するノードを保持するハッシュ
	 *   key = RequiredSQL/名前付きバインド変数名
	 *   val = SQL依存グラフノードへのポインタ
	 */
	apr_hash_t * top_h;
};
/* divy_sql_dependgraph を操作するためのマクロ */
#define GET_DGRAPH_NODE(g,name)      (apr_hash_get(g->top_h, name, APR_HASH_KEY_STRING))
#define SET_DGRAPH_NODE(g,name,node) (apr_hash_set(g->top_h, name, APR_HASH_KEY_STRING, node))
#define COUNT_DGRAPH_NODE(g)         (apr_hash_count(g->top_h))

/*--------------------------------------------------------------
  Define array 
  --------------------------------------------------------------*/
/**
 * divy_sql_parser_validate_selectcol 関数のlistに指定するカラムリスト.
 * teamfile の処理に特化したデータ構造です。
 * (note)
 * 	リポジトリDB検索SQLのSELECT句に指定可能なカラムの一覧
 * 	全て小文字で記述して下さい。
 */
static const char * reposdb_allowed_cols[] = {
	"rs_uri_txt",
	"rs_create_bi",
	"rs_dispname_vc",
	"rs_get_lastmodified_bi",
	"rs_resourcetype_i",
	"rs_get_cont_len_bi",
	"rs_get_cont_type_vc",
	"rs_creator_vc",
	"rs_lastmodifier_vc",
	NULL			/* sentinel */
};

/**
 * システムで特別に予約されたSQLのサブ名称一覧を保持する配列
 * (note)
 *	このテーブルの検索にはバイナリサーチを利用するため、
 *	必ずASCIIの昇順に並んでいなければなりません。
 */
static const char * reserved_sql_subname[] = {
	"$$Buserid"
};

/* 配列reserved_sql_subname の長さ(バイナリサーチに必要) */
#define RESERVED_SQL_SUBNAME_TBL_LEN 	sizeof(reserved_sql_subname)/sizeof(char *)

/*--------------------------------------------------------------
  Declare lex scanner and yacc parser functions 
  --------------------------------------------------------------*/
DIVY_DECLARE(extern int) tf_scanner_init(divy_sql_parser_ctx *p_ctx);
DIVY_DECLARE(extern int) tf_scanner_finish(divy_sql_parser_ctx *p_ctx);
extern int yyparse(divy_sql_parser_ctx *p_ctx);
extern int tf_yylex(YYSTYPE *p_yylval, divy_sql_parser_ctx *p_ctx);

/*--------------------------------------------------------------
  Declare private functions
  --------------------------------------------------------------*/
/* SQL文の解析に使用する関数群 */ 
static int _is_nbind_setting(apr_pool_t *p, char *sql);
static char * _get_nbind_value(apr_pool_t *p, char *sql);
static int _make_sqlelemnode(divy_sql_parser *parser,
					const char *sql,
					divy_sql_elemnode **sqlenode);
static divy_sql_parser_err * _get_yylex_err(apr_pool_t *p, int yylex_ret);

/* chk_mode の値 */
#define VALIDATE_WITHOUT_BODY 0	/* Body タグとの整合性は検証しない */
#define VALIDATE_WITH_BODY 1	/* Body タグとの整合性も検証する   */
static int _validate_reqsql(divy_sql_parser *parser,
					int chk_mode,
					divy_rdbo_sqltype sqltype,
					const char *sql,
					divy_rdbo_sqlcontent *where_pr);
static int _convert_sqlenode_to_sqlcnt(apr_pool_t *p,
					int *idx, int *ridx,
					divy_sql_elemnode *sqlenode,
					divy_rdbo_sqlcontent **sqlcnt_pr);
static divy_rdbo_sqlcontent * _create_sqlcnt(apr_pool_t *p,
						int id,
						divy_rdbo_contype contype,
						int sqlposition,
						char *colname,
						char *rsname);
static int _is_special_reservedsql(const char *subname);

/* 以下はRequireSQLの依存グラフを解決するための関数群 */
static divy_sql_dependnode * _create_dependnode(
					apr_pool_t *p,
					unsigned int nid,
					char *name);
static divy_sql_dependnodeList * _create_dependnodeList(
					apr_pool_t *p,
					divy_sql_dependnode *d_node);
static int _append_dependnodeList(divy_sql_dependnodeList *l1,
					divy_sql_dependnodeList *l2);
static int _make_dependgraph(apr_pool_t *p,
					divy_rdbo_sqldepend *sqldepend,
					divy_sql_dependgraph **d_graph);
static void _clear_dependnode_state(divy_sql_dependnode *d_node);
static int _get_top_dependnode(apr_pool_t *p,
					divy_sql_dependgraph *d_graph,
					divy_sql_dependnode **d_node);
static void _find_indispensable_node(apr_pool_t *p,
					divy_sql_dependnode *d_node,
					divy_cset_t **indispensable_set);  
static int _is_closedpath(apr_pool_t *p,
					divy_sql_dependgraph *d_graph);
static divy_sql_dependnode * _find_closedpath_node(
					divy_sql_dependnode *d_node,
					unsigned int nid);
static int _find_maxchain(divy_sql_dependnode *d_node, int chain_cnt);

static void _make_edgelist(apr_pool_t *p,
					divy_sql_dependnode *d_node,
					apr_hash_t **edge_h);
static int _resolve_reqsql(divy_sql_parser *parser,
					divy_sql_dependnode *d_node,
					apr_hash_t *sql_pr_hash,
					divy_rdbo_rsvalue **rsvalue);
static char * _print_dnode_child(apr_pool_t *p,
					divy_sql_dependnode *d_node);
static char * _print_sqlenode(apr_pool_t *p, divy_sql_elemnode *sqlenode);

/* 以下は汎用的に使用するユーティリティ関数群 */
static int _validate_parser(divy_sql_parser *parser);
static int _is_keyword_existance(const char *keyword, int tbllen,
					const char *keywordtbl[]);

/*--------------------------------------------------------------
  Define public function
  --------------------------------------------------------------*/
/**
 * SQL parser を生成する。
 *
 */
DIVY_DECLARE(int) divy_sql_parser_create(apr_pool_t *p, divy_sql_parser **parser)
{
	*parser = NULL;		/* 初期化 */

	TRACE(p);

	/* parser 構造体の作成 */
	*parser = apr_pcalloc(p, sizeof(divy_sql_parser));
	(*parser)->p      = p;
	(*parser)->r      = NULL;
	(*parser)->ts_ctx = NULL;

	/* パーサコンテキストの生成と初期化 */
	(void) divy_sql_parser_init_pctx(p, &(*parser)->p_ctx);

	return DIVY_SQLP_ST_OK;
}

/**
 * パーサのコンテキスト構造体を初期化する. (内部専用)
 * (note)
 * 	この関数は、字句解析・構文解析パーサとのやり取りでのみ使用するように
 * 	して下さい。それ以外の用途では、divy_sql_parser_create でパーサ
 * 	コンテキストも一緒に初期化しますので、そちらを使用して下さい。
 *
 */
DIVY_DECLARE(int) divy_sql_parser_init_pctx(apr_pool_t *p,
					divy_sql_parser_ctx **p_ctx)
{
	*p_ctx = apr_pcalloc(p, sizeof(divy_sql_parser_ctx));
	(*p_ctx)->p            = p;
	(*p_ctx)->str          = NULL;
	(*p_ctx)->errcd        = DIVY_SQLP_ST_OK;	/* "正常"で初期化 */
	(*p_ctx)->select_colcnt= 0;
	(*p_ctx)->max_select_colcnt = 0;
	(*p_ctx)->select_id    = 0;
	(*p_ctx)->sqlenode     = NULL;
	(*p_ctx)->sqlposition  = 0;
	(*p_ctx)->scanbuf      = NULL;
	(*p_ctx)->xstart       = SCANNER_START_MODE_NORMAL;
	(*p_ctx)->new_sql_buf  = NULL;
	(*p_ctx)->bindvals     = NULL;
	(*p_ctx)->bvalue       = NULL;
	(*p_ctx)->ipos         = 0;
	(*p_ctx)->ibpos        = 0;
	(*p_ctx)->reqsql_value = NULL;
	(*p_ctx)->keyword      = NULL;

	return DIVY_SQLP_ST_OK;
}

/**
 * SQLパーサの戻り値retcd から、詳細なエラー情報(divy_sql_parser_err)を
 * 取得して返却する.
 *
 */
DIVY_DECLARE(divy_sql_parser_err *) divy_sql_parser_get_parser_err(
						apr_pool_t *p, int retcd)
{
	divy_sql_parser_err *err = NULL;
	int i, len = SQL_PARSER_ERR_LEN;

	for (i = 0; i < len; i++) {
		err = (divy_sql_parser_err *) &sql_parser_errs[i];
		if (err && err->retcd == retcd) {
			return err;
		}
	}

	return NULL;
}


/**
 * SELECT に指定されたカラムが決められたカラム集合に含まれている
 * かどうかを検証する。
 *
 */
DIVY_DECLARE(int) divy_sql_parser_validate_selectcol(divy_sql_parser *parser,
					const char *sql,
					apr_hash_t **must_selcol_hash)
{
	apr_pool_t *p;
	apr_hash_t *repcol_h         = NULL;
	divy_sql_elemnode *sqlenode  = NULL, *node;
	int i, ret;
	char *colname;
	
	p = parser->p;
	TRACE(p);
	
	*must_selcol_hash = NULL;	/* 初期化 */

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;

	if (IS_EMPTY(sql)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"sql is NULL.");
		return DIVY_SQLP_ST_ERR;
	}

	/*
	 * SQL解析ノードの生成
	 */
	ret = _make_sqlelemnode(parser, sql, &sqlenode);
	if (ret != DIVY_SQLP_ST_OK) {
		return ret;
	}

	/* 先頭はselect か？ */
	if (sqlenode == NULL || CHILD_NODE(sqlenode) == NULL ||
	    sqlenode->nodetype != DIVY_NT_SELECT) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"SQL must start from \'select\'");
		return DIVY_SQLP_ST_NONSELECT;
	}

	/* SELECT に含まれるべき全カラムのハッシュを作成します */
	repcol_h = apr_hash_make(p);
	for (i = 0; reposdb_allowed_cols[i]; i++){
		apr_hash_set(repcol_h, reposdb_allowed_cols[i],
				APR_HASH_KEY_STRING, "");
	}

	/* SELECT 句を探す */
	for (node = CHILD_NODE(sqlenode); node; node = node->next) {
		if (node->belong_nodetype == DIVY_NT_SELECT_CLAUSE) break;
	}
	
	/* SELECT 句カラムを調べる */
	for (; node; node = node->next) {
		/* SELECT句に含まれていなければスキップ */
		if (node->belong_nodetype != DIVY_NT_SELECT_CLAUSE) {
			continue;
		}

		/* '.' があればその後だけ比較する */
		if ((colname = divy_strchr(node->name, '.')) != NULL) {
			colname = apr_pstrdup(p, colname++);
		} else {
			colname = apr_pstrdup(p, node->name);
		}

		/* 大文字・小文字を無視して比較するため小文字に変換する
		 * (note) オリジナル文字列を変えないようにするためコピーする */
		ap_str_tolower(colname);

		/* 作成済みのハッシュから一致データを削除 */
		apr_hash_set(repcol_h, colname, APR_HASH_KEY_STRING, NULL);
	}

	/* 必須カラムが存在していたか？ */
	if (apr_hash_count(repcol_h) > 0) {
		*must_selcol_hash = repcol_h;
		return DIVY_SQLP_ST_MUSTCOL_NF;
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * [ RequiredSQLの依存関係の最下層を調べる（MUSTなRequiredを調べる）]
 *
 * dependlist のような依存関係がある状態で、SQL文sqlと値が解決済みの
 * RequiredSQL集合reqsql_set が与えられた時、
 * このSQLの実行に絶対に必要なRequiredSQLの一覧を返却する。
 *
 */
DIVY_DECLARE(int) divy_sql_parser_find_sqldependency(divy_sql_parser *parser,
						request_rec *r,
						divy_db_transaction_ctx *ts_ctx,
						divy_rdbo_sqldepend *dependlist,
						apr_hash_t *reqsql_set,
						apr_hash_t **must_reqsql_set)
{
	apr_pool_t *p;
	divy_sql_dependgraph *d_graph  = NULL;
	divy_sql_dependnode  *d_node   = NULL;
	divy_cset_t *indispensable_set = NULL;
	divy_rdbo_sql *sql_pr          = NULL;
	divy_cset_index_t *ci;
	apr_hash_t *sql_pr_hash        = NULL;
	divy_rdbo_clist *subname_list  = NULL, *first = NULL;
	char *name;
	int ret;
	int i, tbl_len = RESERVED_SQL_SUBNAME_TBL_LEN;

	*must_reqsql_set = NULL;	/* 初期化 */

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;
	p = parser->p;

	TRACE(p);

	/*
	 * RequiredSQLと名前付きバインド変数からなる
	 * 依存有向グラフを生成する
	 */
	ret = _make_dependgraph(p, dependlist, &d_graph);
	if (ret != DIVY_SQLP_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to make depend graph.");

		return DIVY_SQLP_ST_ERR;
	}
	else if (d_graph == NULL) {
		/* 依存関係が存在しなかったので何もしない */
		return DIVY_SQLP_ST_OK;
	}

	/*
	 * SQL依存グラフが解決済みであるとマークする
	 * (マーク対象)
	 * 	・reqsql_set に含まれるRequiredSQL
	 * 	・システムで予約された特別なバインド変数
	 */
	if (reqsql_set) {
		apr_hash_index_t *iterator;

		for (iterator = apr_hash_first(p, reqsql_set);
			iterator; iterator = apr_hash_next(iterator)) {

			/* RequiredSQLの名称を取得 */
			apr_hash_this(iterator, (const void **)&name, NULL, NULL);

			/* 解決済みであるとマークする */
			d_node = GET_DGRAPH_NODE(d_graph, name);
			if (d_node) d_node->state = RESOLVE;
		}
	}

	/* システム予約変数の検索 */
	for (i = 0; i < tbl_len; i++) {
		name = (char *) reserved_sql_subname[i];

		/* d_node が存在すれば、解決済みであるとマークする */
		d_node = GET_DGRAPH_NODE(d_graph, name);
		if (d_node) d_node->state = RESOLVE;
	}

	/*
	 * データ不正に備えて、SQL依存グラフが閉回路を形成していないか調べる
	 */
	ret = _is_closedpath(p, d_graph);
	if (ret == DIVY_SQLP_ST_LOOP_DETECTED) {
		/* 閉回路が存在した場合 */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to search sqldependency, because the "
			"dependence-relations of RequiredSQL have some closed-loop."
			"Please check sql and correct relations.");
		return ret;
	}
	else if (ret != DIVY_SQLP_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to find closed-loop for sql.");
		return ret;
	}

	/*
	 * 親を持たないノード(つまり頂点ノード)を開始点として
	 * Childノードを調べ、解決しなければならないRequiredSQL一覧を取得
	 */
	/* 頂点ノードの取得 */
	ret = _get_top_dependnode(p, d_graph, &d_node);
	if (ret == DIVY_SQLP_ST_MULTIPLE_TOP) {
		return DIVY_SQLP_ST_ERR;
	}
	else if (ret == DIVY_SQLP_ST_NOTFOUND) {
		/* 頂点が見つからなかったのはデータ不正 */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get top-sql node. "
			"Maybe relationship for RequiredSQL was broken. "
			"Please correct this.");
		return DIVY_SQLP_ST_ERR;
	}

	/* d_node から見た必須SQL依存グラフノードの取得 */
	_find_indispensable_node(p, d_node, &indispensable_set);
	if (indispensable_set == NULL) {
		return DIVY_SQLP_ST_OK;
	}

	/*
	 * 名前付きバインド変数: 管理者によって値補完が行われていたら
	 *			 "必須である"とは報告しない
	 * RequiredSQL         : indispensable_set に含まれているRequiredSQLは無視。
	 * 			 WHERE句を持たないはずなので。
	 */
	for (ci = divy_cset_first(indispensable_set); ci; ci = divy_cset_next(ci)) {

		divy_cset_this(ci, (const char **)&name);
		if (IS_NBIND_NODE(name)) {
			/* 後でSQLを引くので一時的に記憶しておく */
			if (first == NULL) {
				first = subname_list =
					apr_pcalloc(p, sizeof(divy_rdbo_clist));
			}
			else {
				subname_list->next = apr_pcalloc(p, sizeof(divy_rdbo_clist));
				subname_list = subname_list->next;
			}
			subname_list->val  = name;
			subname_list->next = NULL;
		}
		else {
			/* 無視する
			 * (note)
			 * indispensable_set に含まれるRequiredSQLは、"必須"SQLには
			 * なり得ない。RequiredSQLを含むWHERE句を持たないので。
			 * データ不整合でWHERE句があったとしてもSELECT文の結果セットを
			 * エンドユーザに入力させる訳には行かないので無視します */
		}
	}

	subname_list = first;
	if (subname_list) {
		if (divy_rdbo_get_sql_by_subname(r, subname_list, &sql_pr_hash, ts_ctx)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get sql statement.");
			return DIVY_SQLP_ST_EXEC_FAILED;
		}
	}

	/* 名前付きバインド変数をパースして値を持っているかどうか判定しながら
	 * 値が解決されていない名前付きバインド変数を追加する */
	for (; subname_list; subname_list = subname_list->next) {
		sql_pr = NULL;
		if (sql_pr_hash) {
			sql_pr = apr_hash_get(sql_pr_hash,
					subname_list->val, APR_HASH_KEY_STRING);
		}

		if (sql_pr == NULL || !_is_nbind_setting(p, sql_pr->sql)) {

			/* 値が設定されていなければmust_reqsql_set に追加 */
			if (*must_reqsql_set == NULL) {
				*must_reqsql_set = apr_hash_make(p);
			}
			apr_hash_set(*must_reqsql_set, subname_list->val,
							APR_HASH_KEY_STRING, "");
		}
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * 指定されたSQL文sqlに含まれるRequiredSQLまたは名前付きバインド変数を
 * rsvalueによって指定される値で置換して返却する。
 * また、IN句指定された場合にはこれらの句の性質に合致するよう値を変形する。
 *
 * (note) !!注意!!
 * 	この関数はmutexを使ったロックを使用する関数を使っています。
 * 	mutex の中からこの関数をCallすることは厳禁です。
 */
DIVY_DECLARE(int) divy_sql_parser_replace_reqsql(divy_sql_parser *parser,
						const char *sql,
						divy_search_ldbs_bind *bvalue,
						divy_rdbo_rsvalue *rsvalue,
						char **new_sql,
						divy_array_t **bindvals)
{
	apr_pool_t *p = NULL;
	int ret = 0, count = 0;
	apr_status_t st;
	divy_rdbo_rsvalue *_rsvalue = NULL;
	divy_sql_parser_err *err    = NULL;
	YYSTYPE _yylval;

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;

	p = parser->p;
	TRACE(p);

	if (IS_EMPTY(sql)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"sql is NULL.");
		return DIVY_SQLP_ST_ERR;
	}

	*new_sql  = NULL;	/* 初期化 */
	*bindvals = NULL;

	/*
	 * mutex ロックを掛ける
	 */
	st = apr_thread_mutex_lock(scanner_mutex);
	if (st != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to lock for sqlparser. Code = %d", st);
		return DIVY_SQLP_ST_ERR;
	}

	/*
	 * パーサコンテキストの初期化と値の設定
	 * (note) 値置換モードにする
	 */
	divy_sql_parser_init_pctx(p, &parser->p_ctx);

	parser->p_ctx->str          = sql;
	parser->p_ctx->xstart       = SCANNER_START_MODE_REPLACE;
	parser->p_ctx->ipos         = 0;
	parser->p_ctx->reqsql_value = rsvalue;
	parser->p_ctx->bvalue       = bvalue;
	divy_sbuf_create(p, &parser->p_ctx->new_sql_buf, 1024);

	/* 字句解析パーサの初期化 */
	tf_scanner_init(parser->p_ctx);

	/*
	 * 字句解析パーサを使用して名前付きバインド変数及び、
	 * RequiredSQLを値で置換する。
	 * (note) 構文解析パーサは使用しません。
	 * 	また、緩衝ルーチン入りの字句解析パーサも使用しません。
	 */
	ret = tf_yylex(&_yylval, parser->p_ctx);
	if ((err = _get_yylex_err(p, ret)) != NULL) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to extract reqsql from sql, because %s"
			"(sql = %s)", err->msg, sql);
		ret = err->retcd;
		goto proc_finish;
	}

	/* rsvalue の中のリスト数とSQL文中のサブ名称の個数が同じかどうか */
	for (_rsvalue = rsvalue; _rsvalue; _rsvalue = _rsvalue->next) {
		count++;
	}

	if (count != parser->p_ctx->ipos) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"Failed to replace the RequiredSQL in the specified value, "
			"because the count of resolve-value list is diffecrent "
			"from the count of RequiredSQL in the sql."
			"(resolve-value count = %d, reqsql count = %d, sql = %s)",
			count, parser->p_ctx->ipos, sql);
		ret = DIVY_SQLP_ST_WRONG_REQSQLNUM;
		goto proc_finish;
	}

	*new_sql  = divy_sbuf_tostring(parser->p_ctx->new_sql_buf);
	*bindvals = parser->p_ctx->bindvals;
	ret = DIVY_SQLP_ST_OK; 

proc_finish:
	/* 字句解析パーサの終了処理 */
	tf_scanner_finish(parser->p_ctx);

	/*
	 * mutex ロックの解除
	 */
	st = apr_thread_mutex_unlock(scanner_mutex);
	if (st != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to unlock for sql parser. Code = %d", st);
		return DIVY_SQLP_ST_ERR;
	}

	return ret;
}

/**
 * [ WHERE句プロパティの作成 ]
 *
 * 指定されたSQL文sqlを解析し、WHERE句プロパティwhere_prop を生成して返却
 * する。
 */
DIVY_DECLARE(int) divy_sql_parser_make_whereprop(divy_sql_parser *parser,
						const char *sql,
						divy_rdbo_sqlcontent **where_pr)
{
	apr_pool_t *p;
	divy_sql_elemnode *sqlenode  = NULL;
	int ret, idx = 0, ridx = 0;

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;

	p = parser->p;
	TRACE(p);

	if (IS_EMPTY(sql)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"sql is NULL.");
		return DIVY_SQLP_ST_ERR;
	}

	*where_pr = NULL;	/* 初期化 */

	/*
	 * SQL解析ノードの生成
	 */
	ret = _make_sqlelemnode(parser, sql, &sqlenode);
	if (ret != DIVY_SQLP_ST_OK) {
		return ret;
	}

	/*
	 * WHERE句プロパティが存在しなかった or
	 * BINDVAL 構文だった場合
	 */
	if (sqlenode == NULL || CHILD_NODE(sqlenode) == NULL) {
		/* WHERE句プロパティは存在しない */
		return DIVY_SQLP_ST_OK;
	}

	/*
	 * SQL解析ノードをWHERE句プロパティに焼き直す
	 */
	ret = _convert_sqlenode_to_sqlcnt(p, &idx, &ridx, sqlenode, where_pr);
	if (ret != DIVY_SQLP_ST_OK) {
		return ret;
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * 指定されたSQL文sqlに含まれるRequiredSQL及び名前付きバインド変数が指定規則に
 * 基づいて指定されているかどうか検証する。
 *
 */
DIVY_DECLARE(int) divy_sql_parser_validate_reqsql(divy_sql_parser *parser,
						divy_rdbo_sqltype sqltype,
						const char *sql)
{
	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;

	TRACE(parser->p);

	return _validate_reqsql(parser, VALIDATE_WITHOUT_BODY, sqltype, sql, NULL);
}

/**
 * 指定されたSQL文sqlに含まれるRequiredSQL及び名前付きバインド変数が指定規則に
 * 基づいて指定されているかどうか検証し、さらに指定されたWHERE句プロパティとの
 * 整合性を確認する.
 *
 */
DIVY_DECLARE(int) divy_sql_parser_validate_reqsql_withbody(
						divy_sql_parser *parser,
						divy_rdbo_sqltype sqltype,
						const char *sql,
						divy_rdbo_sqlcontent *where_pr)
{
	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;

	TRACE(parser->p);

	return _validate_reqsql(parser, VALIDATE_WITH_BODY, sqltype, sql, where_pr);
}

/**
 * [ RequiredSQL の値を解決する ]  
 *
 * 値解決が既に行われているRequiredSQLの集合rsvalueを使って、RequiredSQL間の
 * 依存関係dependlistの中で値が解決されていないRequiredSQLの値を求め、
 * トップ階層(RequiredSQLを使用する最上位)の値のみを返却する。
 *
 */
DIVY_DECLARE(int) divy_sql_parser_resolve_reqsql(divy_sql_parser *parser,
						request_rec *r,
						divy_db_transaction_ctx *ts_ctx,
						divy_rdbo_sqldepend *dependlist,
						divy_rdbo_rsvalue *rsvalue,
						apr_hash_t **top_rsvalue_hash)
{
	apr_pool_t *p;
	apr_hash_t *sql_pr_hash         = NULL;
	divy_sql_dependgraph *d_graph   = NULL;
	divy_sql_dependnode  *d_node    = NULL;
	divy_sql_dependnodeList *d_list = NULL;
	divy_rdbo_rsvalue *tmp_rsvalue  = NULL;
	divy_cset_t *resolve_node       = NULL, *subname_set = NULL;
	divy_rdbo_sqldepend *sqldepend  = NULL;
	int ret;

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;

	p = parser->p;
	parser->r      = r;
	parser->ts_ctx = ts_ctx;

	*top_rsvalue_hash = NULL;	/* 初期化 */

	TRACE(p);

	if (dependlist == NULL) return DIVY_SQLP_ST_OK;

	/*
	 * RequiredSQLと名前付きバインド変数からなる
	 * 依存有向グラフを生成する
	 */
	ret = _make_dependgraph(p, dependlist, &d_graph);
	if (ret != DIVY_SQLP_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to make depend graph.");

		return DIVY_SQLP_ST_ERR;
	}
	else if (d_graph == NULL) {
		return DIVY_SQLP_ST_OK;
	}

	/* 値が解決済みのRequiredSQLの値をSQL依存グラフノードに設定する */
	if (rsvalue) {
		resolve_node = divy_cset_make(p);
		for (tmp_rsvalue = rsvalue;
			tmp_rsvalue; tmp_rsvalue = tmp_rsvalue->next) {

			/* ノードの取得 */
			d_node = GET_DGRAPH_NODE(d_graph, tmp_rsvalue->name);
			if (d_node) {
				d_node->state    = RESOLVE;
				d_node->valueset = tmp_rsvalue->valueset;

				/* 解決済みのノード名を記録しておく */
				divy_cset_set(resolve_node, tmp_rsvalue->name);
			}
		}
	}

	/*
	 * 解決されていないSQLノードのSQLプロパティを取得する
	 */
	subname_set = divy_cset_make(p);
	for (sqldepend = dependlist; sqldepend; sqldepend = sqldepend->next) {

		/* 値が解決済みでない親と子を共にsubname_setに入れる */
		if (!divy_cset_contains(resolve_node, sqldepend->ptsubname)) {
			divy_cset_set(subname_set, sqldepend->ptsubname);
		}

		if (!divy_cset_contains(resolve_node, sqldepend->clsubname)) {
			divy_cset_set(subname_set, sqldepend->clsubname);
		}
	}

	/* SQLプロパティの取得 */
	ret = divy_rdbo_get_sql_properties_by_subnames(r, subname_set, &sql_pr_hash, ts_ctx);
	if (ret) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get sql properties by subnames.");
		return DIVY_SQLP_ST_ERR;
	}
	
	/*
	 * 頂点ノードの取得
	 * (note)
	 * 	ここでいう頂点ノードとは、本関数においてWHERE句に値を
	 * 	補完する対象のSQLノードを指す。これは当然１つのはずである。
	 */
	ret = _get_top_dependnode(p, d_graph, &d_node);
	if (ret == DIVY_SQLP_ST_MULTIPLE_TOP) {
		return DIVY_SQLP_ST_ERR;
	}
	else if (ret == DIVY_SQLP_ST_NOTFOUND) {
		/* 頂点が見つからなかったのはデータ不正 */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get top-sql node. "
			"Maybe relationship for RequiredSQL was broken. "
			"Please correct this.");
		return DIVY_SQLP_ST_ERR;
	}

	/*
	 * 頂点ノードの子ノードのSQLから値を取得する
	 * (note)
	 * 	この関数で欲しいのは、頂点ノードのSQL実行結果ではなく、
	 * 	頂点ノードの子ノードが補完してくれるRequiredSQLの値。
	 * 	ですので、子ノードだけが対象となります。
	 */
	for (d_list = d_node->child_list; d_list; d_list = d_list->next) {
		tmp_rsvalue = NULL;

		/* d_list->d_node が示すSQLノードから値を取得する */
		ret = _resolve_reqsql(parser, d_list->d_node,
						sql_pr_hash, &tmp_rsvalue);
		if (ret == DIVY_SQLP_ST_EXEC_FAILED ||
		    ret == DIVY_SQLP_ST_TOOMANY_COLUMN ||
		    ret == DIVY_SQLP_ST_TOOFEW_COLUMN) {
			/* 実行に失敗した場合 */
			/* tmp_rsvalue には失敗したSQLの名前が入っているので
			 * そのまま返却する */
			*top_rsvalue_hash = apr_hash_make(p);
			apr_hash_set(*top_rsvalue_hash, tmp_rsvalue->name,
					APR_HASH_KEY_STRING, tmp_rsvalue);
			return ret;
		}
		
		/* *top_rsvalue に取得値をリストで連結する */
		if (*top_rsvalue_hash == NULL) {
			*top_rsvalue_hash = apr_hash_make(p);
		}

		/* サブ名称をキーとしてRequiredSQLの名称と値を詰める */
		apr_hash_set(*top_rsvalue_hash, tmp_rsvalue->name,
					APR_HASH_KEY_STRING, tmp_rsvalue);
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * dependlist のような依存関係がある時、この依存関係から作られるSQL依存グラフが
 * 閉回路(closed path)を作らないかどうか判定する。
 *
 */
DIVY_DECLARE(int) divy_sql_parser_validate_closedpath(divy_sql_parser *parser,
						divy_rdbo_sqldepend *dependlist)
{
	apr_pool_t *p;
	divy_sql_dependgraph *d_graph = NULL;
	int ret;

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;
	p = parser->p;

	TRACE(p);

	/*
	 * RequiredSQLと名前付きバインド変数からなる
	 * 依存有向グラフを生成する
	 */
	ret = _make_dependgraph(p, dependlist, &d_graph);
	if (ret != DIVY_SQLP_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to make depend graph.");

		return DIVY_SQLP_ST_ERR;
	}
	else if (d_graph == NULL) {
		return DIVY_SQLP_ST_OK;
	}

	/*
	 * SQL依存グラフが閉回路を形成していないか調べる
	 */
	ret = _is_closedpath(p, d_graph);
	if (ret == DIVY_SQLP_ST_LOOP_DETECTED) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The dependence-relations of RequiredSQL "
			"have some closed-loop.");
		return ret;
	}
	else if (ret != DIVY_SQLP_ST_OK) {
		return ret;
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * dependlist のような依存関係がある時、この依存関係から作られるSQL依存グラフの
 * 依存連鎖数が最大値(DIVY_REQSQL_MAX_CHAING) を超えないかどうかを判定する。
 *
 * (note)
 * 	dependlist は、入次数が０である唯一SQLを頂点とするグラフに
 * 	なっていなければなりません。
 * 	さもなければ、全く依存関係の無いSQLグループで発生していた
 * 	DIVY_SQLP_ST_OVER_MAX_CHAINを見てしまいます。
 */
DIVY_DECLARE(int) divy_sql_parser_validate_maxchain(divy_sql_parser *parser,
						divy_rdbo_sqldepend *dependlist)
{
	apr_pool_t *p;
	divy_sql_dependgraph    *d_graph = NULL;
	divy_sql_dependnode     *d_node  = NULL;
	int chain_cnt = 0, ret;

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;
	p = parser->p;

	TRACE(p);

	/*
	 * RequiredSQLと名前付きバインド変数からなる
	 * 依存有向グラフを生成する
	 */
	ret = _make_dependgraph(p, dependlist, &d_graph);
	if (ret != DIVY_SQLP_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to make depend graph.");

		return DIVY_SQLP_ST_ERR;
	}
	else if (d_graph == NULL) {
		return DIVY_SQLP_ST_OK;
	}

	/*
	 * SQL依存グラフの頂点ノード(入次数０) が１つだけであることの検証
	 */
	ret = _get_top_dependnode(p, d_graph, &d_node);
	if (ret == DIVY_SQLP_ST_MULTIPLE_TOP) {
		return ret;
	}
	else if (ret == DIVY_SQLP_ST_NOTFOUND) {
		/* 頂点が見つからなかったのはデータ不正 */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get top-sql node. "
			"Maybe relationship for RequiredSQL was broken. "
			"Please correct this.");
		return DIVY_SQLP_ST_ERR;
	}

	/*
	 * 依存SQLの連鎖数が最大値をオーバーしていないかどうか調べる
	 */
	chain_cnt = _find_maxchain(d_node, chain_cnt);
	if (chain_cnt > DIVY_REQSQL_MAX_CHAIN) {
		/* 最大連鎖数を超えた場合
		 * (note) node の名前は内部固定値の場合があるので
		 * 	ログには出しません */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The count of sql-dependence exceeded "
			"the max value. (current chain count = %d, "
			"max = %d)", chain_cnt,
			DIVY_REQSQL_MAX_CHAIN);
		return DIVY_SQLP_ST_OVER_MAX_CHAIN;
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * dependlist のような依存関係がある時、subname が示すSQLが依存グラフに
 * 含まれていないかどうか検証し、含まれていれば、subnameのSQLを直接利用するSQLの
 * 名称をリストして返却する。
 *
 */
DIVY_DECLARE(int) divy_sql_parser_find_usingsql(divy_sql_parser *parser,
						const char *subname,
						divy_rdbo_sqldepend *dependlist,
						divy_cset_t **usingsql_set)
{
	apr_pool_t *p;
	divy_sql_dependgraph *d_graph = NULL;
	divy_sql_dependnode  *d_node  = NULL;
	divy_sql_dependnodeList *list = NULL;
	int ret;

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;
	p = parser->p;

	TRACE(p);

	if (IS_EMPTY(subname)) {
		/* サブ名称がなければ使われていないはず
		 * (note) データ不整合の可能性は十分あるが、寛容に扱う */
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"subname is EMPTY, so we suppose that "
			"this sql is not used.");
		return DIVY_SQLP_ST_OK;
	}

	/* 初期化 */
	*usingsql_set = NULL;

	/*
	 * RequiredSQLと名前付きバインド変数からなる
	 * 依存有向グラフを生成する
	 */
	ret = _make_dependgraph(p, dependlist, &d_graph);
	if (ret != DIVY_SQLP_ST_OK) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to make depend graph.(sqlsubname = %s)", subname);

		return DIVY_SQLP_ST_ERR;
	}
	else if (d_graph == NULL) {
		return DIVY_SQLP_ST_OK;
	}

	/*
	 * subname のSQLが他のSQLから使用されていないかどうかを検証する
	 * (読み替え)
	 *   --> subname のSQLの親依存グラフノードを取得する
	 */
	d_node = GET_DGRAPH_NODE(d_graph, subname);
	if (d_node != NULL) {
		/* 親ノードを記録する */
		for (list = d_node->parent_list; list; list = list->next) {
			if (*usingsql_set == NULL) {
				*usingsql_set = divy_cset_make(p); 
			}
			divy_cset_set(*usingsql_set, list->d_node->name);
		}

		/* 他のSQLから参照されていたことを報告する */
		if (*usingsql_set) return DIVY_SQLP_ST_USINGSQL_FOUND; 
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * dependlist のような依存関係がある時、subname_setの集合に含まれるRequiredSQL
 * が利用するRequiredSQLと名前付きバインド変数との親子関係、つまり依存グラフの
 * 辺(edge)を全て取得してdivy_rdbo_sqldepend のリストとして返却する。
 *
 */
DIVY_DECLARE(int) divy_sql_parser_find_edgelist(divy_sql_parser *parser,
						divy_cset_t *subname_set,
						divy_rdbo_sqldepend *dependlist,
						divy_rdbo_sqldepend **edgelist)
{
	apr_pool_t *p;
	divy_sql_dependgraph *d_graph = NULL;
	divy_sql_dependnode  *d_node  = NULL;
	divy_cset_index_t *ci         = NULL;
	apr_hash_index_t *hi          = NULL;
	apr_hash_t *edge_h            = NULL;
	int ret;
	char *key;
	const char *subname;

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;
	p = parser->p;

	TRACE(p);

	if (subname_set == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"subname_set is EMPTY.");
		return DIVY_SQLP_ST_ERR;
	}

	/* 初期化 */
	*edgelist = NULL;

	/*
	 * RequiredSQLと名前付きバインド変数からなる
	 * 依存有向グラフを生成する
	 */
	ret = _make_dependgraph(p, dependlist, &d_graph);
	if (ret != DIVY_SQLP_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to make depend graph.");

		return DIVY_SQLP_ST_ERR;
	}
	else if (d_graph == NULL) {
		return DIVY_SQLP_ST_OK;
	}

	/*
	 * subname_set の要素がChildを持っていないかどうか調べ、持っていれば
	 * そのChildとの辺を取り出す
	 */
	for (ci = divy_cset_first(subname_set); ci; ci = divy_cset_next(ci)) {
		/* 要素を１つ取り出す */
		divy_cset_this(ci, &subname);

		/* SQL依存グラフノードの取り出し */
		d_node = GET_DGRAPH_NODE(d_graph, subname);
		if (d_node != NULL) {
			/* d_node 以下の辺を持つハッシュを作成する */
			_make_edgelist(p, d_node, &edge_h);
		}
	}

	/* edge_h からedgelist を生成する */
	if (edge_h) {
		divy_rdbo_sqldepend *wk = NULL, *sdepend = NULL;
		for (hi = apr_hash_first(p, edge_h); hi; hi = apr_hash_next(hi)) {
			/* 要素の取得 */
			apr_hash_this(hi, (const void **)&key,
							NULL, (void **)&sdepend);
			if (*edgelist == NULL) {
				*edgelist = wk = sdepend;
			}
			else {
				wk->next = sdepend;
				wk = wk->next;
			}
		}
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * 指定されたSQLが持っているRequiredSQL及び名前付きバインド変数を取得して、
 * rsql_set につめて返却する。(通常バインド変数は返却されません)
 * なお、同名のRequiredSQLや名前付きバインド変数が複数使用されていたとしても
 * それは１つであるとカウントします。
 *
 * (note) !!注意!!
 * 	この関数は、scanner_mutex を使用しています。
 * 	scanner_mutex を使用する他の関数を絶対に内部から呼び出さないで下さい。
 */
DIVY_DECLARE(int) divy_sql_parser_extract_reqsql(divy_sql_parser *parser,
						const char *sql,
						apr_hash_t **reqsql_hash)
{
	apr_status_t st;
	apr_pool_t *p;
	int ret = 0, count = 0;
	int *idx = NULL;
	YYSTYPE _yylval;
	char *name;
	divy_sql_parser_err *err = NULL;

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;
	p = parser->p;

	TRACE(p);

	*reqsql_hash = NULL;	/* 初期化 */

	/*
	 * mutex ロックを掛ける
	 */
	st = apr_thread_mutex_lock(scanner_mutex);
	if (st != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to lock for sqlparser. Code = %d", st);
		return DIVY_SQLP_ST_ERR;
	}

	/*
	 * パーサコンテキストの初期化と値の設定
	 */
	divy_sql_parser_init_pctx(p, &parser->p_ctx);
	parser->p_ctx->str = sql;

	/* 字句解析パーサの初期化 */
	tf_scanner_init(parser->p_ctx);

	/*
	 * 字句解析パーサを使用して名前付きバインド変数
	 * 及び、RequiredSQLを取得する
	 * (note) 構文解析パーサは使用しません。
	 */
	while ((ret = tf_yylex(&_yylval, parser->p_ctx)) != 0) {
		if (ret == NAMEDBIND ||
		    ret == S_REQUIREDSQL || ret == M_REQUIREDSQL) { 
			/* 文字列集合に名称を追加 */
			if (*reqsql_hash == NULL) {
				*reqsql_hash =  apr_hash_make(p);
			}

			idx = apr_palloc(p, sizeof(int));
			*idx = ++count;
			name = apr_pstrdup(p, _yylval.str);

			/* name - idx で登録する */
			apr_hash_set(*reqsql_hash, name, APR_HASH_KEY_STRING, idx);
		}
		/* エラー判定 */
		else if ((err = _get_yylex_err(p, ret)) != NULL) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to extract reqsql from sql, because %s"
				"(sql = %s)", err->msg, sql);
			ret = err->retcd;
			goto proc_finish;
		}
	}

	ret = DIVY_SQLP_ST_OK;	/* 正常に終わった */

proc_finish:
	/* 字句解析パーサの終了処理 */
	tf_scanner_finish(parser->p_ctx);

	/*
	 * mutex ロックの解除
	 */
	st = apr_thread_mutex_unlock(scanner_mutex);
	if (st != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to unlock for sql parser. Code = %d", st);
		return DIVY_SQLP_ST_ERR;
	}

	return ret;
}

/**
 * 名前付きバインド変数にデフォルト値が設定されているかどうかを判別する。
 *
 */
DIVY_DECLARE(int) divy_sql_parser_is_nbind_setting(divy_sql_parser *parser,
						const char *sql)
{
	int ret;

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;

	/* 値があるかどうかを取得 */
	ret = _is_nbind_setting(parser->p, (char *) sql);
	if (ret) {
		/* 値が見つかった */
		return DIVY_SQLP_ST_FOUND_NBIND_VAL;
	}
	else {
		/* 値は未設定 */
		return DIVY_SQLP_ST_NF_NBIND_VAL;
	}
}

/**
 * SQLタイプtypeのWHERE句プロパティにサブ名称subnameのSQLが指定された時、
 * 持つことが出来るSQLモード一覧を返却する。
 * なお、subname が名前付きバインド変数の場合には、デフォルト値があるか
 * どうかをhas_default_valに指定する。
 *
 */
DIVY_DECLARE(int) divy_sql_parser_get_sqlmodelist(divy_sql_parser *parser,
						 divy_rdbo_sqltype type,
						 const char *subname,
						 int has_default_val,
						 divy_rdbo_sqlmode **sqlmodelist)
{
	apr_pool_t *p;
	divy_rdbo_sqlmode *list = NULL;

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;

	p = parser->p;
	if (IS_EMPTY(subname)) {
		return DIVY_SQLP_ST_ERR;
	}

	*sqlmodelist = NULL;	/* 初期化 */
	
	/*
	 * RequiredSQL の場合
	 */
	if (type == DIVY_SQL_TYPE_REQUIRED) {

		/* 名前付きバインド変数かつ
		 * 	デフォルト値を持っている or 特別なSQL */
		if (IS_NBIND_NODE(subname) &&
		    (has_default_val == DIVY_SQL_HAS_DEFAULT_VAL ||
		     _is_special_reservedsql(subname))) {
			list = apr_pcalloc(p, sizeof(int) * 3);

			list[0] = DIVY_SQL_MODE_SHOW;
			list[1] = DIVY_SQL_MODE_HIDDEN;
			list[2] = DIVY_SQL_MODE_UNKNOWN; /* sentinel */
		}
		else {
			list = apr_pcalloc(p, sizeof(int) * 2);

			list[0] = DIVY_SQL_MODE_SHOW;
			list[1] = DIVY_SQL_MODE_UNKNOWN; /* sentinel */
		}
	} 
	/*
	 * 通常SQLまたはリポジトリSQLの場合
	 */
	else if (type == DIVY_SQL_TYPE_NORMAL || type == DIVY_SQL_TYPE_REPOSITORY) {

		/* 名前付きバインド変数($$Bxxx) の場合 */
		if (IS_NBIND_NODE(subname)) {
			/* デフォルト値を持っている or 特別なSQL */
			if (has_default_val == DIVY_SQL_HAS_DEFAULT_VAL ||
			    _is_special_reservedsql(subname)) {

				list = apr_pcalloc(p, sizeof(int) * 3);

				list[0] = DIVY_SQL_MODE_SHOW;
				list[1] = DIVY_SQL_MODE_HIDDEN;
				list[2] = DIVY_SQL_MODE_UNKNOWN; /* sentinel */
			}
			else {
				list = apr_pcalloc(p, sizeof(int) * 2);

				list[0] = DIVY_SQL_MODE_SHOW;
				list[1] = DIVY_SQL_MODE_UNKNOWN; /* sentinel */
			}
		}
		/* RequiredSQL($$SSxxx) の場合 */
		else if (IS_S_REQSQL_NODE(subname)) {
			list = apr_pcalloc(p, sizeof(int) * 3);

			list[0] = DIVY_SQL_MODE_SHOW;
			list[1] = DIVY_SQL_MODE_HIDDEN;
			list[2] = DIVY_SQL_MODE_UNKNOWN; /* sentinel */
		}
		/* RequiredSQL($$SMxxx) の場合 */
		else if (IS_M_REQSQL_NODE(subname)){
			list = apr_pcalloc(p, sizeof(int) * 5);

			list[0] = DIVY_SQL_MODE_SHOW;
			list[1] = DIVY_SQL_MODE_HIDDEN;
			list[2] = DIVY_SQL_MODE_SELECTED;
			list[3] = DIVY_SQL_MODE_MULTISELECTED;
			list[4] = DIVY_SQL_MODE_UNKNOWN; /* sentinel */
		}
	}
	/* 名前付きバインド変数構文の場合 */
	else { /* 無視 */
	}

	/* list を格納する */
	*sqlmodelist = list;

	return DIVY_SQLP_ST_OK;
}

/**
 * 指定されたサブ名称のSQLがシステムで特別に予約されたものであるかどうかを判定する。
 */
DIVY_DECLARE(int) divy_sql_parser_is_special_reservedsql(divy_sql_parser *parser,
							const char * subname)
{
	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;

	if (_is_special_reservedsql(subname)) {
		return DIVY_SQLP_ST_RESERVED_SQL;	/* 予約されていた */
	}
	else {
		return DIVY_SQLP_ST_UNRESERVED_SQL;	/* 予約されていなかった */
	}
}

/**
 * サブ名称subname とsqluri, sqltype との組み合わせ検証
 *
 */
DIVY_DECLARE(int) divy_sql_parser_validate_subname(divy_sql_parser *parser,
						request_rec *r,
						divy_rdbo_sqltype sqltype,
						const char *sqluri,
						int isnew,
						const char *subname)
{
	int ret;
	apr_pool_t *p;
	divy_rdbo_sql *subname_sql_pr = NULL;
	const char *names[] = { subname, NULL };

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;

	p = parser->p;

	/* RequiredSQL、名前付きバインド変数の場合 */
	if (sqltype == DIVY_SQL_TYPE_REQUIRED || sqltype == DIVY_SQL_TYPE_BIND) {

		/* サブ名称は存在するか？ */
		if (IS_EMPTY(subname)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"subname\"(\"sqldiscovery\" child) "
				"element is missing or empty.");
			return DIVY_SQLP_ST_MISSING_REQSUB;
		}

		/* サブ名称の形式は正しいか？ */
		if (!IS_REQSQL_NODE(subname)) {

			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The format of \"subname\"(\"sqldiscovery\" child) "
				"element is invalid. (subname = %s)", subname);
			return DIVY_SQLP_ST_WRONG_SUBNAME;
		}

		/* サブ名称とSQL種類との組合せは正しいか */
		if (sqltype == DIVY_SQL_TYPE_BIND && !IS_NBIND_NODE(subname)) {
			/* 名前付きバインド変数構文のサブ名称と
			 * sqltypeの組み合わせがおかしい */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The kind of \"sqltype\" and type of \"subname\" "
				"are mismatch.(sqltype = %d, subname = %s)",
				sqltype, subname);

			return DIVY_SQLP_ST_WRONG_COMB_TYPE;
		}
		else if (sqltype == DIVY_SQL_TYPE_REQUIRED &&
			(!IS_S_REQSQL_NODE(subname) && !IS_M_REQSQL_NODE(subname))) {
			/* RequiredSQL構文のサブ名称と
			 * sqltypeの組み合わせがおかしい */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The kind of \"sqltype\" and type of \"subname\" "
				"are mismatch.(sqltype = %d, subname = %s)",
				sqltype, subname);

			return DIVY_SQLP_ST_WRONG_COMB_TYPE;
		}

		/* サブ名称がDB に登録されているか？ */
		ret = divy_rdbo_get_sqluri_by_subname(r, names, &subname_sql_pr);
		if (ret == 1) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"Failed to get sql uri by subname.(subname = %s)",
				subname);

			return DIVY_SQLP_ST_IERR; 
		}

		/* 
		 * (新規登録) : 同名のサブ名称が既にあってはならない
		 * (更新)     : 同名のサブ名称があってもよいが、持っているのは自分 */
		if ((isnew && subname_sql_pr) || (!isnew && subname_sql_pr &&
			  strcmp(sqluri, subname_sql_pr->relativeuri) != 0)) {

			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The name of \"subname\" (\"sqldiscovery\" child)"
				"element is already used."
				"So, the server could not continue process."
				"(subname = %s)", subname);

			return DIVY_SQLP_ST_MISMATCH_REQURI;
		}
	}
	/* 通常SQL、リポジトリSQL */
	else {
		/* サブ名称は指定出来ません */
		if (IS_FILLED(subname)) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Could not set \"subname\" element for normalsql "
				"or repositorysql.(sqltype = %d, subname = %s)",
				sqltype, subname);
			return DIVY_SQLP_ST_USELESS_SUBNAME;
		}
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * キャッシュモードcachemode とsqluri, sqltype との組み合わせ検証
 *
 */
DIVY_DECLARE(int) divy_sql_parser_validate_cachemode(divy_sql_parser *parser,
						divy_rdbo_sqltype sqltype,
						const char *subname,
						const char *sql,
						divy_rdbo_sqlcachemode cachemode)
{
	int ret;
	apr_pool_t *p;

	/* SQLパーサの検証 */
	if (_validate_parser(parser)) return DIVY_SQLP_ST_ERR;

	p = parser->p;

	/* 名前付きバインド変数構文ではなかった */
	if (sqltype != DIVY_SQL_TYPE_BIND || !IS_NBIND_NODE(subname)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"cache\" element must be used for namedbind."
			"(subname = %s)", subname);
		return DIVY_SQLP_ST_CACHE_NONBIND;
	}
	else {
		/* デフォルト値が設定されているか？ */	
		ret = divy_sql_parser_is_nbind_setting(parser, sql);
		if (ret == DIVY_SQLP_ST_FOUND_NBIND_VAL) {
			/* 設定されていた場合はエラー */
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The Default value of namedbind "
				"should not be set with \"cache\" element.");
			return DIVY_SQLP_ST_CACHE_NBIND_VAL;
		} 
		/* 通常のエラー */
		else if (ret == DIVY_SQLP_ST_ERR){
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_PROC,
				"An error occured while checking "
				"default value of namedbind.(subname = %s)",
				subname);
				return ret;
		}

		/* システムで予約された特別なSQLか？ */
		if (IS_FILLED(subname)) {
			ret = divy_sql_parser_is_special_reservedsql(parser,
								subname);
			if (ret == DIVY_SQLP_ST_RESERVED_SQL) {
				ERRLOG0(p, APLOG_ERR,
					DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The reserved sql should not "
					"be set with \"cache\" element.");
				return ret;
			}
		}
	}

	return DIVY_SQLP_ST_OK;
}

/*--------------------------------------------------------------
  Define private function
  --------------------------------------------------------------*/
/*
 * SQL文の解析に使用する関数群
 */

/**
 * 名前付きバインド変数が値をもっているかどうか
 *
 * @param p apr_pool_t *
 * @param sql char * 名前付きバインド変数の定義文
 * @return int 値を持っているかどうか
 * 	1 : 持っていた
 * 	0 : 持っていない
 */
static int _is_nbind_setting(apr_pool_t *p, char *sql)
{
	if (IS_FILLED(sql)) {
		return 1;
	}

	return 0;
}


/**
 * 名前付きバインド変数のSQL文からデフォルト値を切り出す.
 * (note) !!注意!!
 * 	sql はバインド変数であることを呼び出し元にて保証して下さい。
 *
 * @param p apr_pool_t *
 * @param sql char * SQL文
 * @return char * 切り出した値。値がなければNULL
 */
static char * _get_nbind_value(apr_pool_t *p, char *sql)
{
	if (IS_EMPTY(sql)) return NULL;

	return apr_pstrdup(p, sql);
}

/**
 * 指定されたSQL文sql からSQLのSyntax Treeを生成して返却する。
 *
 * (note) !!注意!!
 * 	この関数は、scanner_mutex を使用しています。
 * 	scanner_mutex を使用する他の関数を絶対に内部から呼び出さないで下さい。
 *
 * @param parser divy_sql_parser *
 * @param sql const char * 解析対象のSQL
 * @param sqlenode divy_sql_elemnode ** Syntax Treeを持つノード
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK               : 成功
 * 	DIVY_SQLP_ST_ERR              : 失敗
 * 	DIVY_SQLP_ST_INVALID          : 誤りのある可能性が高いSQL構文エラーを発見した
 *	DIVY_SQLP_ST_WRONG_LVAL       : RequiredSQLが左辺値として指定された
 * 	DIVY_SQLP_ST_WRONG_M_PREF     : $$SMxxx がIN句の右辺値以外に指定された
 * 	DIVY_SQLP_ST_NONSELECT        : 有効な開始識別子(SELECT句)が無かった
 * 	DIVY_SQLP_ST_UNQUOTED_STR     : SQL文中に閉じられていないシングルクオートがある 
 * 	DIVY_SQLP_ST_FOUND_UPDATESTMT : 更新系DML、DDL、DCLが指定された
 * 	DIVY_SQLP_ST_UNKNOWN_STMT     : 理解出来ないSQL構文・識別子が指定された
 * 	DIVY_SQLP_ST_UNQUOTED_DQSTR   : ダブルクオート文字が正しく閉じられていない
 * 	DIVY_SQLP_ST_FOUND_ASTERISK   : SELECT句に'*'が指定された
 * 	DIVY_SQLP_ST_UNQUOTED_CTSTR   : SQL文中に閉じられていないSQLコメントがある
 */
static int _make_sqlelemnode(divy_sql_parser *parser,
				const char *sql,
				divy_sql_elemnode **sqlenode)
{
	apr_status_t st;
	apr_pool_t *p = parser->p;
	int ret;

	/*
	 * 前回解析されたSQLと同じSQLを解析するか？
	 */
	if (parser->p_ctx && parser->p_ctx->sqlenode &&
	    IS_FILLED(parser->p_ctx->str) &&
	    strcmp(parser->p_ctx->str, sql) == 0) {

		/* 前と同じでよいはず */
		*sqlenode = parser->p_ctx->sqlenode;
		return DIVY_SQLP_ST_OK;
	}

	/*
	 * mutex ロックを掛ける
	 */
	st = apr_thread_mutex_lock(scanner_mutex);
	if (st != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to lock for sqlparser. Code = %d", st);
		return DIVY_SQLP_ST_ERR;
	}

	/*
	 * パーサコンテキストの初期化と値の設定
	 */
	divy_sql_parser_init_pctx(p, &parser->p_ctx);
	parser->p_ctx->str = sql;

	/* 字句解析パーサの初期化 */
	tf_scanner_init(parser->p_ctx);

	/*
	 * 字句解析・構文解析パーサの呼び出し
	 */
	ret = yyparse(parser->p_ctx);
	if (ret) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parser sql.(yyparse ret = %d, "
			"detail ret = %d, sql = \"%s\")",
			ret, parser->p_ctx->errcd, sql);

		ret = parser->p_ctx->errcd;	/* パーサが出力した詳細エラー */
		goto proc_finish;
	}
	else {
		/* 取得したノードをセット */
		*sqlenode = parser->p_ctx->sqlenode;
		ret = DIVY_SQLP_ST_OK;

		ERRLOG2(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"Parse Node (cnt = %d): %s",
			parser->p_ctx->max_select_colcnt,
			_print_sqlenode(p, *sqlenode));
	}
	
proc_finish:
	/* 字句解析パーサの終了処理 */
	tf_scanner_finish(parser->p_ctx);

	/*
	 * mutex ロックの解除
	 */
	st = apr_thread_mutex_unlock(scanner_mutex);
	if (st != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to unlock for sql parser. Code = %d", st);
		return DIVY_SQLP_ST_ERR;
	}

	return ret;
}

/**
 * 字句解析パーサでエラーが戻されていないかどうかを検証し、
 * エラーがあればdivy_sql_parser_err *を返却する
 *
 * @param p apr_pool_t *
 * @param yylex_ret int 字句解析パーサからの戻り値
 * @return divy_sql_parser_err * エラー情報を表す構造体へのポインタ
 * 				NULL の時には正常
 */
static divy_sql_parser_err * _get_yylex_err(apr_pool_t *p, int yylex_ret)
{
	divy_sql_parser_err *err = NULL;

	switch (yylex_ret) {
		default:
			err = NULL;
			break;
		/* シングルクオートが正しく閉じられていなかった */
		case EOF_SCONST:
			err = divy_sql_parser_get_parser_err(p,
						DIVY_SQLP_ST_UNQUOTED_STR); 
			break;
		/* ダブルクオートが正しく閉じられていなかった */
		case EOF_QT_IDENT:
			err = divy_sql_parser_get_parser_err(p,
						DIVY_SQLP_ST_UNQUOTED_DQSTR);
			break;
		/* SQLコメントが正しく閉じられていなかった */
		case EOF_COMMENT:
			err = divy_sql_parser_get_parser_err(p,
						DIVY_SQLP_ST_UNQUOTED_CTSTR);
			break;
		/* rsvalue->id が歯抜けだったため、置換できなかった */ 
		case ST_YYLEX_MISSING_ID:
			err = divy_sql_parser_get_parser_err(p,
						DIVY_SQLP_ST_MISSING_ID);
			break;

		/* sql中のRequiredSQL出現位置とrsvalue->idの値が異なる */
		case ST_YYLEX_MISMATCH_ID:
			err = divy_sql_parser_get_parser_err(p,
						DIVY_SQLP_ST_MISMATCH_ID);
			break;
	}

	return err;
}

/*
 * 指定されたSQL文sqlに含まれるRequiredSQL及び名前付きバインド変数が指定規則に
 * 基づいて指定されているかどうか検証し、必要に応じて指定されたWHERE句プロパティ
 * との整合性を確認する.
 *
 * (note)
 * 	名前付きバインド変数構文(sqltype = DIVY_SQL_TYPE_BIND) のsql の内容は
 * 	単なる値とみなすようになりました。常に正常であると返却されます。
 *
 * @param parser divy_sql_parser * divy_sql_parser_create で作成したパーサ
 * @param sqltype divy_rdbo_sqltype 解析対象SQLのsqltype
 * @param sql const char * 解析対象のSQL
 * @param chk_mode int Bodyタグとの整合性をチェックするかどうか
 * 	VALIDATE_WITHOUT_BODY        : チェックしない
 * 	VALIDATE_WITH_BODY           : チェックする
 * @param where_pr divy_rdbo_sqlcontent * WHERE句プロパティ.存在しない場合にはNULL
 * @return int 処理ステータスと検証結果
 * 	DIVY_SQLP_ST_OK               : 問題なし。成功。 
 * 	DIVY_SQLP_ST_ERR              : 予期しないエラーのため中止
 * 	DIVY_SQLP_ST_INVALID          : 誤りのある可能性が高いSQL構文エラーを発見した
 * 	DIVY_SQLP_ST_WRONG_LVAL       : RequiredSQLが左辺値として指定された
 * 	DIVY_SQLP_ST_WRONG_OPER       : RequiredSQLに比較演算子以外が適用された
 * 	DIVY_SQLP_ST_WRONG_M_PREF     : $$SMxxx がIN句の右辺値以外に指定された
 * 	DIVY_SQLP_ST_CONFLICT_SQLCNT  : SQL文のWHEREプロパティとkeyvalue エレメントの内容に矛盾
 * 	DIVY_SQLP_ST_WRONG_BIND_POS   : 通常バインド変数がRequiredSQLの中に指定された
 * 	DIVY_SQLP_ST_UNQUOTED_STR     : SQL文中に閉じられていないシングルクオートがある 
 * 	DIVY_SQLP_ST_MISMATCH_SQLTYPE : sqltype とSQL文の種類が一致しない
 * 	DIVY_SQLP_ST_TOOMANY_COLUMN   : SELECT句に指定されたカラム数が多い
 * 	DIVY_SQLP_ST_TOOFEW_COLUMN    : SELECT句に指定されたカラム数が少ない
 * 	DIVY_SQLP_ST_FOUND_UPDATESTMT : 更新系DML、DDL、DCLが指定された
 * 	DIVY_SQLP_ST_UNKNOWN_STMT     : 理解出来ないSQL構文・識別子が指定された
 * 	DIVY_SQLP_ST_UNQUOTED_DQSTR   : ダブルクオート文字が正しく閉じられていない
 */
static int _validate_reqsql(divy_sql_parser *parser,
					int chk_mode,
					divy_rdbo_sqltype sqltype,
					const char *sql,
					divy_rdbo_sqlcontent *where_pr)
{
	apr_pool_t *p = parser->p;	/* 既に検証は済んでいるはず */
	int ret = 0;
	divy_sql_elemnode *sqlenode  = NULL;
	divy_rdbo_sqlcontent *sqlcnt_pr = NULL, *pr;
	const char *str1, *str2;
	int select_colcnt, min_cnt, max_cnt;
	int idx = 0, ridx = 0;

	/*
	 * 名前付きバインド変数構文ならばすべて正常とする。
	 */
	if (sqltype == DIVY_SQL_TYPE_BIND) {
		return DIVY_SQLP_ST_OK;
	}

	/* 以下は、RequiredSQLと通常SQL、リポジトリSQLの場合の検証 */

	if (IS_EMPTY(sql)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"sql is NULL.");
		return DIVY_SQLP_ST_ERR;
	}

	/*
	 * SQL解析ノードの生成
	 */
	ret = _make_sqlelemnode(parser, sql, &sqlenode);
	if (ret != DIVY_SQLP_ST_OK) {
		return ret;
	}

	/*
	 * sqltype と SQL文の種類を比較する
	 */
	if (sqlenode && sqlenode->nodetype != DIVY_NT_SELECT &&
		(sqltype == DIVY_SQL_TYPE_NORMAL ||
		 sqltype == DIVY_SQL_TYPE_REPOSITORY ||
		 sqltype == DIVY_SQL_TYPE_REQUIRED)) {

		/* sqltype がSELECT文だったのに、SQL文はSELECT構文ではない */
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The value of \"sqltype\" element indicates to be "
			"\"SELECT\" sentence, but the kind of sql statement "
			"is different from that.(kind of sql = %d)",
			sqlenode->nodetype);
		return DIVY_SQLP_ST_MISMATCH_SQLTYPE;
	}

	/*
	 * SELECT句に指定されたカラム数の判定
	 */
	select_colcnt = parser->p_ctx->max_select_colcnt;
	if (sqltype == DIVY_SQL_TYPE_REQUIRED) {
		min_cnt = DIVY_REQSQL_MAX_SELECT_COLUMN;
		max_cnt = DIVY_REQSQL_MAX_SELECT_COLUMN;
	}
	else {
		min_cnt = DIVY_NORMALSQL_MIN_SELECT_COLUMN;
		max_cnt = DIVY_NORMALSQL_MAX_SELECT_COLUMN;
	}

	/* 最小値よりも少ない */
	if (select_colcnt < min_cnt) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The number of SELECT column is small."
			"(count = %d, min = %d)", select_colcnt, min_cnt);
		return DIVY_SQLP_ST_TOOFEW_COLUMN;
	}
	/* 最大値よりも多い */
	else if (select_colcnt > max_cnt) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The number of SELECT column is large."
			"(count = %d, max = %d)", select_colcnt, max_cnt);
		return DIVY_SQLP_ST_TOOMANY_COLUMN;
	}

	/*
	 * SQL解析ノードをWHERE句プロパティに焼き直す
	 */
	ret = _convert_sqlenode_to_sqlcnt(p, &idx, &ridx, sqlenode, &sqlcnt_pr);
	if (ret != DIVY_SQLP_ST_OK) {
		return ret;
	}

	/*
	 * WHERE句プロパティとSQL文との整合性チェック
	 */
	/* 簡易解析 */
	if (chk_mode == VALIDATE_WITH_BODY) {
		int nwhere_pr  = 0;	/* where_pr の要素数  */
		int nsqlcnt_pr = 0;	/* sqlcnt_pr の要素数 */

		/* where_pr が無いのにSQL文にはWHERE句プロパティがあった */
		if (where_pr == NULL && sqlcnt_pr != NULL) {
			ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The condition of sql statement exist, but "
				"the content of where_pr is missing."
				"The both of these set must correspond.");	

			return DIVY_SQLP_ST_CONFLICT_SQLCNT;
		}
		/* where_pr があったのに、SQL文には条件句がなかった */
		else if (where_pr != NULL && sqlcnt_pr == NULL) {
			ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The content of where_pr exist, but the "
				"condition of sql statement is missing."
				"The both of these set must correspond.");

			return DIVY_SQLP_ST_CONFLICT_SQLCNT;
		}

		/* where_pr, sqlcnt_pr の要素数を算出 */
		for (pr = where_pr; pr; pr = pr->next)
			nwhere_pr++;

		for (pr = sqlcnt_pr; pr; pr = pr->next)
			nsqlcnt_pr++;

		/* SQL文中のWHERE句プロパティの集合の方が
		 * 	where_prの集合よりも大きかった */
		if (nsqlcnt_pr > nwhere_pr) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The count of sql-statement's WHERE property set "
				"is larger than the count of where_pr set. "
				"The both of these set must correspond."
				"(nsqlcnt = %d, nwhere_pr = %d)", nsqlcnt_pr, nwhere_pr);
			return DIVY_SQLP_ST_CONFLICT_SQLCNT;
		}
		/* SQL文中のWHERE句プロパティの集合の方が
		 * 	where_pr の集合よりも小さかった */
		else if (nsqlcnt_pr < nwhere_pr) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The count of where_pr set is larger than "
				"the count of sql-statement's WHERE property set. "
				"The both of these set must correspond."
				"(nsqlcnt = %d, nwhere_pr = %d)", nsqlcnt_pr, nwhere_pr);
			return DIVY_SQLP_ST_CONFLICT_SQLCNT;
		}
	}

	/*
	 * 詳細解析
	 * (note) divy_validate_sql_property のWHEREプロパティ検証との住み分け
	 * 	SQLパーサ側で行うWHEREプロパティの検証では、SQL解析ノードと
	 * 	BODYで与えられたWHERE句プロパティとの差異とSQL解析ノードが
	 * 	なければ分からないもの、を実施します。
	 * 	それ以外は、ここでやるべきではありません。
	 */
	/* where_pr の中身がSQL文にもちゃんとあるかどうかを検証する */
	for ( ; sqlcnt_pr; sqlcnt_pr = sqlcnt_pr->next) {

		/* RequiredSQLの場合、"?" が指定されていないこと */
		if (sqltype == DIVY_SQL_TYPE_REQUIRED &&
		    sqlcnt_pr->contype == DIVY_SQL_CONTYPE_BIND) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"Could not specify the BIND variable in the RequiredSQL."
				"(sqlstmt_id = %d, sqlstmt_colname = \"%s\")",
				sqlcnt_pr->id, sqlcnt_pr->colname);

			return DIVY_SQLP_ST_WRONG_BIND_POS;

		}

		if (chk_mode == VALIDATE_WITH_BODY) {

			/* id とsqltype の組合せが一致するwhere_pr をprに取得する */
			for (pr = where_pr; pr; pr = pr->next) {
				/* 見つかった */
				if (sqlcnt_pr->id == pr->id &&
				    sqlcnt_pr->contype == pr->contype) {
					break;
				}
			}

			/* id と sqltype の組合せがwhere_pr になかった時 */
			if (pr == NULL) {
				ERRLOG3(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"There are no conbinations of \"id\" "
					"element value and sqltype in WHERE property."
					"(sqlstmt_colname = \"%s\", sqlstmt_id = %d, "
					"sqlstmt_sqltype = %d)",
					REPLACE_NULL(sqlcnt_pr->colname),
					sqlcnt_pr->id, sqlcnt_pr->contype);
				return DIVY_SQLP_ST_CONFLICT_SQLCNT;
			}

			/* 右辺値に指定されたRequiredSQLの名前が違っていた */
			if (pr->contype == DIVY_SQL_CONTYPE_REQUIRED) {
				str1 = pr->reqsql->rsvalue->name;
				str2 = sqlcnt_pr->reqsql->rsvalue->name;

				if ((str1 == NULL && str2 != NULL) ||
				    (str1 != NULL && str2 == NULL) ||
				    (str1 != NULL && str2 != NULL &&
				     			strcmp(str1, str2) != 0)) {
					ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
						"The value of \"reqsqlname\" element "
						"is different from the value of sql "
						"statement. (bodywhere_reqsqlname = \"%s\", "
						"sqlstmt_reqsqlname = \"%s\")",
						REPLACE_NULL(str1), REPLACE_NULL(str2));
					return DIVY_SQLP_ST_CONFLICT_SQLCNT;
				}
			}

			/* 左辺値が異なっていた */
			if ((pr->colname == NULL && sqlcnt_pr->colname != NULL) ||
			    (pr->colname != NULL && sqlcnt_pr->colname == NULL) ||
			    (pr->colname != NULL && sqlcnt_pr->colname != NULL &&
			     strcmp(pr->colname, sqlcnt_pr->colname) != 0)) {
				ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"The content of \"columnname\" element "
					"is different from the columnname of "
					"sql statement.(bodywhere_colname = \"%s\","
					" sqlstmt_colname = \"%s\")",
					REPLACE_NULL(pr->colname),
					REPLACE_NULL(sqlcnt_pr->colname));
				return DIVY_SQLP_ST_CONFLICT_SQLCNT;
			}
		}
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * SQL解析ノードsqlenodeをsqlcnt_pr に焼きなおして返却する。(再帰呼び出し用)
 * (note)
 * 	SQL解析ノードがBINDVAL構文を表していた時にはNULLを返却
 *
 * (note) ノードの構成
 * 	SELECT $$Bxxx, id, val FROM abc WHERE id2 = $$SSxxx and id3 =
 * 		(SELECT id4 FROM zzz WHERE id5 IN $$SMxxx)
 * 	UNION SELECT ...
 * -->
 * 	SELECT
 * 	  |
 * 	  $$Bxxx -> '='     ->      '='    -> SELECT
 * 	             |               |     
 * 	             id2 -> $$SSxxx  id3 -> SELECT
 * 	                                       |-- IN
 * 	                                            |
 * 	                                            id5 -> $$SMxxx
 *  "->"  : next
 *  "|"   : CHILD_NODE
 *
 * @param p apr_pool_t *
 * @param idx int * 通常バインド変数のid 属性の値
 * @param ridx int * 名前付きバインド変数、RequiredSQLのid 属性の値
 * @param sqlenode divy_sql_elemnode * SQL解析ノード
 * @param sqlcnt_pr divy_rdbo_sqlcontent ** 焼きなおしたプロパティ
 * 					プロパティがなければNULL
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK  : 成功
 */
static int _convert_sqlenode_to_sqlcnt(apr_pool_t *p,
					int *idx, int *ridx,
					divy_sql_elemnode *sqlenode,
					divy_rdbo_sqlcontent **sqlcnt_pr)
{
	divy_sql_elemnode *lval, *rval, *node, *rvalx;
	divy_rdbo_sqlcontent *sqlcnt = NULL, *last = NULL;

	if (sqlenode == NULL) return DIVY_SQLP_ST_OK;

	/*
	 * sqlenode をsqlcnt_pr に焼きなおす
	 */
	for (node = sqlenode; node; node = node->next) {

		/*
		 * このノードがSELECTノードを示していた
		 * (SELECT ....)
		 */
		if (node->nodetype == DIVY_NT_SELECT) {
			/* SELECT ノードの下を処理させる */
			_convert_sqlenode_to_sqlcnt(p, idx, ridx,
						CHILD_NODE(node), sqlcnt_pr);
			continue;
		}

		/* Lvalue とRvalue の取得 */
		lval = LVALUE(node);
		rval = RVALUE(node);

		/*
		 * 単体の式だった場合
		 * id, $$Bxxx, $$SSxxx, ...
		 */
		if (lval == NULL && rval == NULL) {

			/* 通常バインド変数単体 */
			if (node->nodetype == DIVY_NT_BIND) {
				/* sqlcnt_pr の作成 */
				sqlcnt = _create_sqlcnt(p, ++(*idx),
							DIVY_SQL_CONTYPE_BIND,
							node->pos, NULL, NULL);
			}
			/* RequiredSQL, 名前付きバインド変数単体 */
			else if (node->nodetype == DIVY_NT_NAMEDBIND ||
				 node->nodetype == DIVY_NT_S_REQUIREDSQL ||
				 node->nodetype == DIVY_NT_M_REQUIREDSQL) {

				/* sqlcnt_pr の作成 */
				sqlcnt = _create_sqlcnt(p, ++(*ridx),
							DIVY_SQL_CONTYPE_REQUIRED,
							node->pos, NULL, node->name);
			}
			/* DB カラム単体など --> sqlcnt_pr に焼き直す必要なし */
			else {
				continue;
			}

			/* *sqlcnt_pr にsqlcnt を連結する */
			if (*sqlcnt_pr == NULL) {
				*sqlcnt_pr = sqlcnt;
			}
			else {
				/* 一番後ろを探す */
				for (last = *sqlcnt_pr; last->next; last = last->next);
				last->next = sqlcnt;
			}
		}
		/*
		 * 左辺値と右辺値が共に存在する場合
		 * xxx = expr 
		 */
		else if (lval && rval) {

			/* 左辺値がSELECT句 (SELECT ....) */
			if (lval->nodetype == DIVY_NT_SELECT) {
				/* SELECT ノードの下を処理させる */
				_convert_sqlenode_to_sqlcnt(p, idx, ridx,
							CHILD_NODE(lval), sqlcnt_pr);
				/* 左辺値がSELECTノードならば右辺値はその値と無関係とする */
				lval->name = NULL;
			}

			/* 右辺値がSELECT句 ( xxx != (SELECT ...) ) */
			if (rval->nodetype == DIVY_NT_SELECT) {
				/* SELECT ノードの下を処理させる */
				_convert_sqlenode_to_sqlcnt(p, idx, ridx,
							CHILD_NODE(rval), sqlcnt_pr);
				/* 右辺値がSELECTノードならば左辺値はその値と無関係とする */
				if (lval->nodetype == DIVY_NT_BIND) {
					/* ? = (SELECT ...) */
					sqlcnt = _create_sqlcnt(p, ++(*idx),
								DIVY_SQL_CONTYPE_BIND,
								lval->pos,
								lval->name, NULL /* 右辺消去 */);
				}
				else if (lval->nodetype == DIVY_NT_NAMEDBIND ||
					 lval->nodetype == DIVY_NT_S_REQUIREDSQL ||
					 lval->nodetype == DIVY_NT_M_REQUIREDSQL) {
					/* $$Bxxx = (SELECT ...) */
					sqlcnt = _create_sqlcnt(p, ++(*ridx),
								DIVY_SQL_CONTYPE_REQUIRED,
								lval->pos,
								lval->name, NULL /* 右辺消去 */);
				}
				else {
					continue;
				}

				/* *sqlcnt_pr にsqlcnt を連結する */
				if (*sqlcnt_pr == NULL) {
					*sqlcnt_pr = sqlcnt;
				}
				else {
					for (last = *sqlcnt_pr; last->next; last = last->next);
					last->next = sqlcnt;
				}
				continue;	/* 下を実施しなくていい */
			}

			/*
			 * 右辺値がリストになっていれば、それを抽出する
			 * xxx = $$Bxxx || $$SSxxx || ? || id
			 */
			for (rvalx = rval; rvalx; rvalx = rvalx->next) {
				sqlcnt = NULL;	/* 初期化 */

				/* 通常バインド変数 */
				if (rvalx->nodetype == DIVY_NT_BIND) {
					sqlcnt = _create_sqlcnt(p, ++(*idx),
								DIVY_SQL_CONTYPE_BIND,
								rvalx->pos,
								lval->name, NULL);
				}
				/* RequiredSQL, 名前付きバインド変数 */
				else if (rvalx->nodetype == DIVY_NT_NAMEDBIND     ||
					 rvalx->nodetype == DIVY_NT_S_REQUIREDSQL ||
					 rvalx->nodetype == DIVY_NT_M_REQUIREDSQL) {

					sqlcnt = _create_sqlcnt(p, ++(*ridx),
								DIVY_SQL_CONTYPE_REQUIRED,
								rvalx->pos,
								lval->name, rvalx->name);
				}
				/* 演算子だった場合 */
				else if (rvalx->nodetype == DIVY_NT_OPERATOR) {
					/* この演算子ノードの下もsqlcnt_pr に焼く */
					_convert_sqlenode_to_sqlcnt(p, idx,
								ridx, rvalx, sqlcnt_pr);
					continue;	/* 次へ */
				}
				/* ##### FIXME もっと統一的に出来ないものかな */
				/* DB カラムの場合 (? = id )*/
				else if (rvalx->nodetype == DIVY_NT_COLUMN) {
					/* 左辺が通常バインド変数だったら
					 * 		式として認識する */
					if (lval->nodetype == DIVY_NT_BIND) {
						/* 左と右を入れ替える */
						sqlcnt = _create_sqlcnt(p, ++(*idx),
								DIVY_SQL_CONTYPE_BIND,
								rvalx->pos,
								rvalx->name, NULL);
					}
					else {
						continue;	/* 何もしない */
					}
				}
				else { /* 何もしない */
					continue;
				}

				/* *sqlcnt_pr にsqlcnt を連結する */
				if (*sqlcnt_pr == NULL) {
					*sqlcnt_pr = sqlcnt;
				}
				else {
					for (last = *sqlcnt_pr; last->next; last = last->next);
					last->next = sqlcnt;
				}
			}
		}
		else { /* ここでは使わないので無視 */
			continue;
		}
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * 指定された値を使ってdivy_rdbo_sqlcontent を生成して返却する。
 * (note)
 * 	rsname が長さ１以上の文字列を持っているとき、divy_rdbo_sqlcontent
 * 	構造体のメンバreqsql を生成します。
 * 	なおこの時、sqlmode は常にDIVY_SQL_MODE_SHOWを設定します。
 *
 * @param p apr_pool_t *
 * @param id int id 属性の値
 * @param contype divy_rdbo_contype sqlcntの種類
 * @param sqlposition int SQL解析位置
 * @param colname char * DBカラム名
 * @param rsname char * RequiredSQLまたは名前付きバインド変数の名前
 * @return divy_rdbo_sqlcontent * 作成したsqlcnt
 */
static divy_rdbo_sqlcontent * _create_sqlcnt(apr_pool_t *p,
						int id,
						divy_rdbo_contype contype,
						int sqlposition,
						char *colname,
						char *rsname)
{
	divy_rdbo_sqlcontent *sqlcnt = apr_pcalloc(p, sizeof(divy_rdbo_sqlcontent));

	sqlcnt->colname     = colname;
	sqlcnt->id          = id;
	sqlcnt->contype     = contype;
	sqlcnt->sqlposition = sqlposition;

	if (IS_FILLED(rsname)) {
		sqlcnt->reqsql  = apr_pcalloc(p, sizeof(divy_rdbo_reqsql));
		sqlcnt->reqsql->sqlmode = DIVY_SQL_MODE_SHOW;	/* デフォルト */

		sqlcnt->reqsql->rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		sqlcnt->reqsql->rsvalue->id   = sqlcnt->id;
		sqlcnt->reqsql->rsvalue->name = rsname;
		sqlcnt->reqsql->rsvalue->next = NULL;
	}
	else {
		sqlcnt->reqsql = NULL;
	}
	sqlcnt->next = NULL;

	return sqlcnt;
}

/**
 * 指定されたサブ名称のSQLがシステムで特別に予約されたものであるかどうかを判定する。
 * (note)
 * 	システムで特別に予約されたSQLとは、クライアント・サーバ間で意識している
 * 	SQLのことを指します。
 *
 * @param subname const char * 調べるSQLのサブ名称
 * @return int 予約されたSQLかどうか
 *	1 : 予約されたSQLであった / 0 : 予約されていないSQLであった
 */
static int _is_special_reservedsql(const char *subname)
{
	if (IS_EMPTY(subname)) {
		return 0;
	}

	/*
	 * subname がreserved_sql_subname配列の中に無いかどうか探す
	 */
	if (_is_keyword_existance(subname, RESERVED_SQL_SUBNAME_TBL_LEN,
					reserved_sql_subname)) {
		/* 配列の中に含まれていた場合 */
		return 1;
	}
	else {
		return 0;
	}
}

/*
 * 以下はRequireSQLのSQL依存グラフを解決するための関数群
 */

/**
 * divy_sql_dependnode を初期化して生成する
 */
static divy_sql_dependnode * _create_dependnode(apr_pool_t *p,
				unsigned int nid, char *name)
{
	divy_sql_dependnode *d_node =
		apr_pcalloc(p, sizeof(divy_sql_dependnode));

	d_node->nid          = nid;
	d_node->name         = apr_pstrdup(p, name);
	d_node->isRequired   = IS_REQSQL_NODE(d_node->name);
	d_node->valueset     = NULL;
	d_node->parent_list  = NULL;
	d_node->child_list   = NULL;
	d_node->state        = INIT;
	d_node->path_nid     = 0;

	return d_node;
}

/**
 * d_node を要素として持つリストを作成して返却する
 */
static divy_sql_dependnodeList * _create_dependnodeList(
				apr_pool_t *p, divy_sql_dependnode *d_node)
{
	divy_sql_dependnodeList *list
		= apr_pcalloc(p, sizeof(divy_sql_dependnodeList));
	list->d_node = d_node;
	list->next   = NULL;

	return list;
}

/**
 * リストl1 の末尾にリストl2を連結する.
 * (note)
 * 	l1にl2の要素が既に含まれていれば何もしない
 *
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK           : 連結できた
 * 	DIVY_SQLP_ST_NOTHING_TODO : 連結する必要は無かった
 */
static int _append_dependnodeList(divy_sql_dependnodeList *l1,
				divy_sql_dependnodeList *l2)
{
	divy_sql_dependnodeList *list;

	if (l1 == NULL || l2 == NULL) return DIVY_SQLP_ST_NOTHING_TODO;

	for (list = l1; list->next; list = list->next) {
		/* 同じノードが含まれていないか？*/
		if (list->d_node == l2->d_node) {
			return DIVY_SQLP_ST_NOTHING_TODO;
		}
	}
	list->next = l2;	/* 連結する */

	return DIVY_SQLP_ST_OK; 
}

/**
 * 与えられたsqldepend から依存グラフd_graphを生成して返却する。
 * (note)
 * 	sqldepend が同じ親子関係を持っていてもそれらは１つの
 * 	関係に集約します。
 *
 * @param p apr_pool_t *
 * @param sqldepend divy_rdbo_sqldepend * SQL依存関係リスト
 * @param d_graph divy_sql_dependgraph ** SQL依存グラフ
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK             : 成功
 * 	DIVY_SQLP_ST_WRONG_RELATION : 子ノードとして通常SQLが指定されていた
 */
static int _make_dependgraph(apr_pool_t *p,
				divy_rdbo_sqldepend *sqldepend,
				divy_sql_dependgraph **d_graph)
{
	divy_sql_dependnode  *p_node, *c_node;
	divy_rdbo_sqldepend  *d_sql;
	char *p_name, *c_name;
	int ret, duplicate_edge;
	unsigned int nid = 0;	/* ノードID (1から始まる) */

	*d_graph = NULL;	/* 初期化 */

	if (sqldepend == NULL) return DIVY_SQLP_ST_OK;

	/* SQL依存グラフ構造体の初期化 */
	*d_graph = apr_pcalloc(p, sizeof(divy_sql_dependgraph));
	(*d_graph)->edge_cnt = 0;
	(*d_graph)->top_h    = apr_hash_make(p);

	/*
	 * sqldepend を使ってグラフを作る
	 * (note)
	 * 	d_sql は重複しているかもしれないので
	 * 	それらをdistinctしなければならない
	 */
	for (d_sql = sqldepend; d_sql; d_sql = d_sql->next) {

		p_name = d_sql->ptsubname;	/* 親 */
		c_name = d_sql->clsubname;	/* 子 */
		duplicate_edge = 0;	/* 初期化 */

		if (p_name == NULL || c_name == NULL) continue;	/* データ不正 */

		/* 親ノードの取り出し */
		p_node = GET_DGRAPH_NODE((*d_graph), p_name);
		if (p_node == NULL) {
			/* SQL依存グラフノードの生成と格納 */
			p_node = _create_dependnode(p, ++nid, p_name);
			SET_DGRAPH_NODE((*d_graph), p_node->name, p_node);
		}

		/* 子ノードの取り出し */
		c_node = GET_DGRAPH_NODE((*d_graph), c_name);
		if (c_node == NULL) {
			c_node = _create_dependnode(p, ++nid, c_name);
			SET_DGRAPH_NODE((*d_graph), c_node->name, c_node);
		}

		/* 子ノードは必ずRequiredSQLでなければならない */
		if (!c_node->isRequired) {
			/* 多分、手で書き換えて失敗したので直してもらう */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The child sql-node was not RequiredSQL or "
				"NamedBindSQL. Maybe the relation was broken."
				"Please correct it.(parent = \"%s\", "
				"child = \"%s\")", p_node->name, c_node->name);
			return DIVY_SQLP_ST_WRONG_RELATION; 
		}

		/* 子ノードを親の子ノードのリスト(child_list) に追加する */
		if (p_node->child_list == NULL) {
			p_node->child_list = _create_dependnodeList(p, c_node);
		}
		else {
			/* リストを繋げる */
			ret = _append_dependnodeList(p_node->child_list,
					_create_dependnodeList(p, c_node));
			if (ret == DIVY_SQLP_ST_NOTHING_TODO) {
				/* 辺が重複してしまった */
				duplicate_edge = 1;
			}
		}

		/* 親ノードを子の親ノードリスト(parent_list) に追加する */
		if (c_node->parent_list == NULL) {
			c_node->parent_list = _create_dependnodeList(p, p_node);
		}
		else {
			(void) _append_dependnodeList(c_node->parent_list,
					_create_dependnodeList(p, p_node));
		}

		/* 親から子へ辺(edge)が１本引かれるのでカウントアップ */
		if (!duplicate_edge) (*d_graph)->edge_cnt++;
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * 指定されたd_node が示すSQL依存グラフノードをクリア(INIT)する.(再起呼び出し用)
 *
 * @param d_node divy_sql_dependnode * SQL依存グラフノード
 */
static void _clear_dependnode_state(divy_sql_dependnode *d_node)
{
	divy_sql_dependnodeList *list;

	/* 終端に達していたか ? */
	if (d_node == NULL) return;

	/* クリア */
	d_node->state = INIT;

	/* 子に対しても同様の操作を行う(再帰呼び出し) */
	for (list = d_node->child_list; list; list = list->next) {
		_clear_dependnode_state(list->d_node);
	}
}

/**
 * SQL依存グラフd_graph からトップノードを取得してd_node に詰めて返却する.
 * なお、トップノードが複数件あった場合には、エラーとします。
 * (note) トップノード
 * 	parent_list を持たないノードのこと。自分の子ノードしか持たない
 * 	ノードともいえます。
 *
 * @param p apr_pool_t *
 * @param d_graph divy_sql_dependgraph * SQL依存グラフ
 * @param d_node divy_sql_dependnode **
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK           : 成功
 * 	DIVY_SQLP_ST_MULTIPLE_TOP : トップノードが複数個あった
 *	DIVY_SQLP_ST_NOTFOUND     : 見つからなかった
 */
static int _get_top_dependnode(apr_pool_t *p,
					divy_sql_dependgraph *d_graph,
					divy_sql_dependnode **d_node)
{
	apr_hash_index_t *iterator;
	divy_sql_dependnode *tmp_node;
	int found_count = 0;
	char *name;

	*d_node = NULL;

	if (d_graph == NULL) return DIVY_SQLP_ST_NOTFOUND;

	for (iterator = apr_hash_first(p, d_graph->top_h);
			iterator; iterator = apr_hash_next(iterator)) {

		/* SQL依存ノードの取得 */
		apr_hash_this(iterator, (const void **)&name, NULL,
					(void **)&tmp_node);

		if (tmp_node->parent_list == NULL) {
			if (found_count == 0) {
				*d_node = tmp_node;
				found_count++;
			}
			else {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"The server found the top-sql-node was "
					"registered with the multiples. "
					"Maybe relationship for RequiredSQL "
					"was broken. Please correct this."
					"(1st top node = \"%s\", 2nd top node = \"%s\")",
					(*d_node)->name, tmp_node->name);

				return DIVY_SQLP_ST_MULTIPLE_TOP;
			}
		}
	}

	if (found_count) {
		return DIVY_SQLP_ST_OK;
	}

	return DIVY_SQLP_ST_NOTFOUND;	/* 見つからなかった */
}

/**
 * 指定されたSQL依存グラフノードd_node の値算出に必要不可欠な依存
 * SQLグラフノードを求める(再帰呼び出し)
 *
 * (note) 必須SQL依存グラフノードになるための条件
 * ----------------------------------------------
 *   (1) 値解決が行われていないこと
 *   (2) Leaf(child が存在しない)ノードであること
 *   (3) 自身を必要とするSQLが１つ以上存在すること 
 *       --> (読み替え)
 *           どのノードからも探索されなければ、必須ノードではない
 * 	         (中間ノードが解決されていれば、それ以下のノードは探索
 * 	          しないという仕様に依存)
 * 	 --> (対偶)
 * 	     必須ノードならば、１つ以上の他のノードから探索される
 *   (4) RequiredSQLまたは名前付きバインド変数である
 *   さらに、同一ノードをmust_reqsql_set に追加しないように以下を追加。
 *   (5) 初めて探索が行われた (state == INIT)
 *
 * @param p apr_pool_t *
 * @param d_node divy_sql_dependnode * SQL依存グラフノード
 * @param indispensable_set divy_cset_t ** 必須SQL依存グラフノードの集合 
 */
static void _find_indispensable_node(apr_pool_t *p,
					divy_sql_dependnode *d_node,
					divy_cset_t **indispensable_set)
{
	divy_sql_dependnodeList *list;

	if (d_node == NULL) return;

	/* 既に探索済み or 解決済みならばスキップ */
	if (d_node->state == VISIT || d_node->state == RESOLVE) {
		return;
	}

	/*
	 * 必須SQL依存グラフノードかどうかの判定
	 * (note)
	 *   (3) の条件は、このノードが検索対象になったという事実
	 *   だけで無条件に成立します。ですので、if 文には現れません。
	 */
	if (d_node->state == INIT && d_node->isRequired &&
				     d_node->child_list == NULL) {
		/* ノード名を記録 */
		if (*indispensable_set == NULL) {
			*indispensable_set = divy_cset_make(p);
		}

		divy_cset_set(*indispensable_set, d_node->name);
	}

	d_node->state = VISIT;	/* 訪問済み */

	if (d_node->child_list == NULL) return;

	/* 自分の子に対して調べる(再帰呼び出し) */ 
	for (list = d_node->child_list; list; list = list->next) {

		_find_indispensable_node(p, list->d_node, indispensable_set);
	}
}

/**
 * SQL依存グラフが閉回路(closed path) を作っていないか判定する。
 *
 * @param p apr_pool_t *
 * @param d_graph divy_sql_dependgraph * SQL依存グラフ
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK            : 閉回路は存在しない
 * 	DIVY_SQLP_ST_LOOP_DETECTED : 閉回路が存在した
 * 	DIVY_SQLP_ST_ERR           : 予期しないエラーが発生した
 */
static int _is_closedpath(apr_pool_t *p,
				divy_sql_dependgraph *d_graph)
{
	divy_sql_dependnode  *d_node, *closed_d_node;
	char *name;
	apr_hash_index_t *iterator;

	if (d_graph == NULL) return DIVY_SQLP_ST_OK;	/* 判定できません */

	/*
	 * SQL依存グラフが"木"であった場合の簡易判定
	 *
	 * (note) この判定は、木の性質を利用して依存グラフのノードを
	 *        総当りで調べなくても済むようにする目的で導入しました。
	 * (note) 木の性質
	 *   (SQL依存ノードの数) = (グラフの辺の数 + 1) ならばこのグラフは"木"
	 *
	 *   木とはサイクル(循環経路)を持たない特別なグラフであるので、
	 *   閉回路を形成するようなパスは存在し得ない。
	 */
	if (COUNT_DGRAPH_NODE(d_graph) == d_graph->edge_cnt + 1) {
		return DIVY_SQLP_ST_OK;
	}

	/*
	 * SQL依存グラフに含まれるSQL依存グラフノードを総当りで調べる
	 */
	for (iterator = apr_hash_first(p, d_graph->top_h);
			iterator; iterator = apr_hash_next(iterator)) {
		
		/* SQL依存ノードの取得 */
		apr_hash_this(iterator, (const void **)&name, NULL,
					(void **)&d_node);
		
		if (d_node == NULL) continue;	/* データ不正対応 */

		/* トップ階層 or 最下層に存在するノードは関係ない */
		if (d_node->parent_list == NULL || d_node->child_list == NULL) {
			continue;
		}

		/* 閉回路を形成するSQL依存グラフノードを取得する */
		closed_d_node = _find_closedpath_node(d_node, d_node->nid);
		if (closed_d_node) {
			/* 閉回路が存在した場合 */
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The closed-loop was detected !! "
				"This was started at \"%s\" node.", closed_d_node->name);
			return DIVY_SQLP_ST_LOOP_DETECTED; 
		}
	}

	return DIVY_SQLP_ST_OK;
}

/**
 * 閉回路を形成するSQL依存グラフノードからなるリストを取得する(再帰呼び出し用)
 *
 * (note) 閉回路が存在することとは、自身に戻ってくる道(path)があるということ。
 */
static divy_sql_dependnode * _find_closedpath_node(
				divy_sql_dependnode *d_node, unsigned int nid)
{
	divy_sql_dependnode *closed_d_node;
	divy_sql_dependnodeList *list;

	if (d_node == NULL) return NULL;

	/* 初めて通る道(path)ならば通ったことを記録しておく */
	if (d_node->path_nid != nid) {
		d_node->path_nid = nid;
	}
	/* １度通ったことがあった場合 */
	else {
		/* 自分に戻ってきてしまった */
		if (d_node->path_nid == d_node->nid) {
			/* ループを検出しました */
			return d_node;
		}
		else {
			/* 以降現れる子は探索済み */
			return NULL;
		}
	}

	/*
	 * (note)
	 * もし値解決されたRequiredSQLのChildを検索しないことになったら、
	 * ここにRESOLVEマークが付いているかどうか判定するコードを
	 * 入れてください。
	 */

	if (d_node->child_list == NULL) return NULL;

	/* 自身の子に対して調べる(再帰呼び出し) */
	for (list = d_node->child_list; list; list = list->next) {
		closed_d_node = _find_closedpath_node(list->d_node, nid);
		if (closed_d_node) return closed_d_node;
	}

	return NULL;
}

/**
 * d_node が示すSQL依存グラフノードの連鎖数を調べて
 * 現在の連鎖数chain_cnt に連鎖数を加えて返却する。(再帰呼び出し用)
 *
 * @param d_node divy_sql_dependnode * SQL依存グラフノード
 * @param chain_cnt int 現在の連鎖数(正の整数)
 * @return int 連鎖数
 */
static int _find_maxchain(divy_sql_dependnode *d_node, int chain_cnt)
{
	divy_sql_dependnodeList *d_list = NULL;
	int cnt, maxcnt;

	if (d_node == NULL) return chain_cnt;

	/* d_node を通ったので連鎖数に１を加える */
	++chain_cnt;
	maxcnt = chain_cnt;

	/* 自分の子供を調べる */
	for (d_list = d_node->child_list; d_list; d_list = d_list->next) {
		cnt = _find_maxchain(d_list->d_node, chain_cnt);
		if (cnt > maxcnt) {
			maxcnt = cnt;	/* 最大値を記録する */
		}
	}

	return maxcnt;
}


/**
 * d_node が示すSQL依存グラフノードからChild方向への有向グラフ辺(edge)を
 * 取り出す。
 * d_node のChildがあれば、それに対して実施する。(再帰呼び出し用)
 *
 * @param p apr_pool_t *p
 * @param d_node divy_sql_dependnode * SQL依存グラフノード
 * @param edge_h apr_hash_t ** 親と子のノード識別子(nid)の組合せからなる
 * 			文字列をキーとし、辺リスト(divy_rdbo_sqldepend *)を
 * 			持つハッシュテーブル
 */
static void _make_edgelist(apr_pool_t *p,
					divy_sql_dependnode *d_node,
					apr_hash_t **edge_h)
{
	divy_sql_dependnodeList *wk;
	divy_rdbo_sqldepend *edge = NULL;
	char *key;

	if (d_node == NULL) return;

	/* edge ハッシュの作成 */
	if (*edge_h == NULL) {
		*edge_h = apr_hash_make(p);
	}

	/* 辺の生成 */
	for (wk = d_node->child_list; wk; wk = wk->next) {
		/* edge_h に登録するためのキー */
		key = apr_psprintf(p, "%d\a\b%d",d_node->nid, wk->d_node->nid);

		/* edge は登録済みか？ */
		edge = apr_hash_get(*edge_h, key, APR_HASH_KEY_STRING); 
		if (edge == NULL) {
			/* (d_node -> wk->d_node) という向きの辺を作る */
			edge = apr_pcalloc(p, sizeof(divy_rdbo_sqldepend));
			edge->topid     = NULL;
			edge->ptsubname = d_node->name;
			edge->clsubname = wk->d_node->name;
			edge->next      = NULL;

			apr_hash_set(*edge_h, key, APR_HASH_KEY_STRING, edge);
		}
		/* 子ノードに対しても実施 */
		_make_edgelist(p, wk->d_node, edge_h);
	}
	return;
}


/**
 * d_node の依存SQLの値を解決し、rsvalue に詰めて返却する.(再帰呼び出し用)
 *
 * (note) !!注意!!
 * 	この関数はmutexを使ったロックを使用する関数を使っています。
 * 	mutex の中からこの関数をCallすることは厳禁です。
 *
 * @param parser divy_sql_parser * パーサを表す構造体
 * @param d_node divy_sql_dependnode *
 * @param sql_pr_hash apr_hash_t * サブ名称をキーとしてdivy_rdbo_sqlを持つハッシュ
 * @param rsvalue divy_rdbo_rsvalue ** 解決した値
 * @return int 処理ステータス
 * 	DIVY_SQLP_ST_OK             : 成功
 * 	DIVY_SQLP_ST_EXEC_FAILED    : RequiredSQLからの値取得に失敗した
 *	DIVY_SQLP_ST_TOOMANY_COLUMN : SELECT句のカラムが多すぎた
 *	DIVY_SQLP_ST_TOOFEW_COLUMN  : SELECT句のカラムが少なすぎた
 */
static int _resolve_reqsql(divy_sql_parser *parser,
					divy_sql_dependnode *d_node,
					apr_hash_t *sql_pr_hash,
					divy_rdbo_rsvalue **rsvalue)
{
	divy_sql_dependnodeList *d_list = NULL;
	divy_rdbo_rsvalue *cond_rsvalue = NULL, *first = NULL;
	apr_pool_t *p = parser->p;
	char *sql = NULL;
	divy_rdbo_sql *sql_pr;
	apr_hash_t *subname_idx_hash = NULL;
	int ret = 0;
	divy_array_t *bindvals = NULL;

	if (d_node == NULL) return DIVY_SQLP_ST_OK;	/* 何もしない */

	*rsvalue = NULL;	/* 初期化 */

	/*
	 * クライアントからキャッシュ値が提出されていた場合
	 * 値解決の手続きは実施しなくてよい
	 */
	if (d_node->state == RESOLVE) {

		/* 値は解決されたのでもうやることはない */
		*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		(*rsvalue)->name      = d_node->name;
		(*rsvalue)->valueset  = d_node->valueset;

		return DIVY_SQLP_ST_OK;
	}

	/* SQL文が取得出来なかった場合 */
	if (sql_pr_hash == NULL) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"sql_pr_hash is NULL.(subname = %s). "
			"Could not resolve RequiredSQL value", d_node->name);
		*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		(*rsvalue)->name = d_node->name;

		return DIVY_SQLP_ST_EXEC_FAILED;
	}

	/* SQL 文の取得 */
	sql_pr = apr_hash_get(sql_pr_hash, d_node->name, APR_HASH_KEY_STRING);
	if (sql_pr == NULL) {
		/* SQLプロパティを取得出来なかった */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"sql_pr is NULL.(subname = %s). "
			"Could not resolve RequiredSQL value", d_node->name);
		*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		(*rsvalue)->name = d_node->name;

		return DIVY_SQLP_ST_EXEC_FAILED;
	}
	/* SQLが非アクティブな場合は実行してはならない */
	else if (sql_pr->active != DIVY_SQL_ACTIVE) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The specified sql is not executable, because "
			"the \"active\" flag is INACTIVE.(subname = %s, "
			"sql = %s) ", d_node->name, REPLACE_NULL(sql_pr->sql));
		*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		(*rsvalue)->name = d_node->name;

		return DIVY_SQLP_ST_EXEC_FAILED;
	}

	/*
	 * 名前付きバインド変数構文の解析
	 * (note)
	 * 	名前付きバインド変数用SQLだった場合、DBMSに投げると
	 * 	エラーになるので、前段階で解決する必要がある
	 */
	if (sql_pr->type == DIVY_SQL_TYPE_BIND) {
		char *val = NULL;

		*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		(*rsvalue)->name = d_node->name;

		/* デフォルト値の取得 */
		val = _get_nbind_value(p, sql_pr->sql);
		if (val == NULL) {
			/* エンドユーザからも値を与えられず、デフォルト値も
			 * なかったら何もできません */
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to resolve namedbind sentence, "
				"because the namedbind sentence had no value "
				"and the value was not given by end user."
				"(subname = %s)", d_node->name);

			return DIVY_SQLP_ST_EXEC_FAILED;
		}

		(*rsvalue)->valueset = apr_pcalloc(p, sizeof(divy_rdbo_rsvalueset));
		(*rsvalue)->valueset->value      = val;
		(*rsvalue)->valueset->dispvalue  = NULL; /* 値を入れないこと */
		(*rsvalue)->valueset->next       = NULL;
		d_node->state    = RESOLVE;	/* 解決した */
		d_node->valueset = (*rsvalue)->valueset;

		/* これ以上やることはありません */
		return DIVY_SQLP_ST_OK;
	}

	/*
	 * 通常SQL、リポジトリSQL、RequiredSQLのSQL文を解析して
	 * WHERE句プロパティに含まれる変数($$B, $$SS, $$SM) の出現位置を算出する
	 */
	ret = divy_sql_parser_extract_reqsql(parser, sql_pr->sql, &subname_idx_hash);
	if (ret != DIVY_SQLP_ST_OK) {
		/* 出現位置の解析に失敗した */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to parse sql for postion-parse."
			"(subname = %s, sql = %s)", d_node->name, sql_pr->sql);
		*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		(*rsvalue)->name = d_node->name;

		return DIVY_SQLP_ST_EXEC_FAILED;
	}

	/* データ不整合のチェック */
	if (subname_idx_hash == NULL && d_node->child_list) {
		/*
		 * SQL文には、RequiredSQLが無いことになっているが、
		 * 依存グラフには依存関係が存在していた。
		 * (divy_sqldependテーブルとdivy_sqlテーブルのSQLの矛盾)
		 */ 
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The data mismatching was detected. "
			"Some effective RequiredSQL in an SQL sentence "
			"was not described, but the sql-dependency existed. "
			"(subname = %s, sql = %s)", d_node->name, sql_pr->sql);
		*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		(*rsvalue)->name = d_node->name;

		return DIVY_SQLP_ST_EXEC_FAILED; 
	}
	else if (subname_idx_hash && d_node->child_list == NULL) {
		/*
		 * SQL文には、RequiredSQLがWHERE句プロパティとして指定されているが、
		 * 依存グラフには依存関係が存在しないことになっている。
		 * (divy_sqldependテーブルとdivy_sqlテーブルのSQLの矛盾)
		 */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The data mismatching was detected. "
			"The sql-dependency did not exist, "
			"but some effective RequiredSQL was necessary "
			"for an SQL sentence. "
			"(subname = %s, sql = %s)", d_node->name, sql_pr->sql);
		*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		(*rsvalue)->name = d_node->name;

		return DIVY_SQLP_ST_EXEC_FAILED; 
	}

	/*
	 * 直下の子ノードから値をかき集める
	 */
	for (d_list = d_node->child_list; d_list; d_list = d_list->next) {
		/* 値の取得 */
		if (d_list->d_node->valueset) {
			*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
			(*rsvalue)->valueset  = d_list->d_node->valueset;
		}
		else {
			/* 無ければ値をとってくる(再帰呼び出し) */
			ret = _resolve_reqsql(parser, d_list->d_node,
							sql_pr_hash, rsvalue);
			if (ret == DIVY_SQLP_ST_EXEC_FAILED) {
				return ret;	/* SQL実行に失敗した */
			}
		}

		/* 条件句リストに設定 */
		if (first == NULL) {
			first = cond_rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		}
		else {
			cond_rsvalue->next = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
			cond_rsvalue = cond_rsvalue->next;
		}

		cond_rsvalue->id       = *((int *)apr_hash_get(subname_idx_hash,
								d_list->d_node->name,
					     			APR_HASH_KEY_STRING));
		cond_rsvalue->name     = d_list->d_node->name;
		cond_rsvalue->valueset = (*rsvalue)->valueset;
		cond_rsvalue->next     = NULL;
	}

	*rsvalue = NULL;
	cond_rsvalue = first;

	/*
	 * 子ノードから集めた値を使ってSQL文を組み立てる
	 */
	if (cond_rsvalue) {

		/* SQL文のRequiredSQLを置き換える */
		ret = divy_sql_parser_replace_reqsql(parser, sql_pr->sql, NULL, cond_rsvalue,
							&sql, &bindvals);
		if (ret != DIVY_SQLP_ST_OK) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to replace RequiredSQL in sql statement."
				"(subname = %s, sql = %s)", d_node->name, sql_pr->sql);
			*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
			(*rsvalue)->name = d_node->name;

			return DIVY_SQLP_ST_EXEC_FAILED;
		}
	}
	else {
		sql = sql_pr->sql;	/* SQL文はそのまま */
	}

	/*
	 * SQLを実行する
	 * (note) $$SS のRequiredSQLならば結果セットは１件だけになります
 	 * (note) 2006/10/12 Thu
	 *   divy_rdbo_exec_sql で発生していた問題によりts_ctx は利用しないことに
	 *   なりました。なので、明示的にNULLにしておきます.
	 */
	ret = divy_rdbo_exec_sql(parser->r, sql_pr->usedbmsname, sql, bindvals,
			//IS_S_REQSQL_NODE(d_node->name), rsvalue, parser->ts_ctx);
			IS_S_REQSQL_NODE(d_node->name), rsvalue, NULL);
	if (ret == DIVY_STCODE_TOOMANY_COLUMN) {
		/* カラム数が多い */
		if (*rsvalue == NULL) {
			*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		}
		(*rsvalue)->name = d_node->name;

		return DIVY_SQLP_ST_TOOMANY_COLUMN; 
	}
	else if (ret == DIVY_STCODE_TOOFEW_COLUMN) {
		/* カラム数が少ない */
		if (*rsvalue == NULL) {
			*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		}
		(*rsvalue)->name = d_node->name;

		return DIVY_SQLP_ST_TOOFEW_COLUMN; 
	}
	else if (*rsvalue == NULL) {
		/* 値が無かった場合は警告(通常起こりうるので) */
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"Status is OK, but the result set of RequiredSQL is EMPTY. "
			"Maybe the data of database is changed."
			"(sql = %s, dbmsname = %s)", sql, sql_pr->usedbmsname);
		*rsvalue = apr_pcalloc(p, sizeof(divy_rdbo_rsvalue));
		(*rsvalue)->name = d_node->name;

		return DIVY_SQLP_ST_EXEC_FAILED;
	}

	/* 値をd_node にも記録しておく */
	(*rsvalue)->name = d_node->name;
	d_node->state    = RESOLVE;	/* 値は解決された */
	d_node->valueset = (*rsvalue)->valueset;

	return DIVY_SQLP_ST_OK;
}

/**
 * (デバッグ用)
 *   d_node が持つ直下のChild SQL依存ノードを見易いように整形して文字列として返却する
 *   (再帰呼び出しを行います)
 */
static char * _print_dnode_child(apr_pool_t *p, divy_sql_dependnode *d_node)
{
	char *ret = NULL, *c_ret = NULL;
	divy_sql_dependnodeList *wk;

	if (d_node == NULL) return "NIL";
	if (d_node->child_list == NULL) {
		return apr_psprintf(p, "%s -> NIL", d_node->name);
	}

	for (wk = d_node->child_list; wk; wk = wk->next) {
		if (ret == NULL) {
			/* 子のリストを取得する */
			c_ret = _print_dnode_child(p, wk->d_node);
			ret = apr_psprintf(p, "%s -> %s,%s", d_node->name,
					wk->d_node->name, c_ret);
		}
		else {
			/* 子のリストを取得する */
			c_ret = _print_dnode_child(p, wk->d_node);
			ret = apr_psprintf(p, "%s / %s -> %s,%s",
						ret, d_node->name,
						wk->d_node->name, c_ret);
		}
	}
	return ret;
}

/**
 * (デバッグ用)
 *  sqlenode が持つ関係を見やすいよう整形して文字列として返却する。
 * [ 書式 ]
 *	(オペレータ, lval, rval) -> (オペレータ, lval, rval1,rval2,...) ...
 */
static char * _print_sqlenode(apr_pool_t *p, divy_sql_elemnode *sqlenode)
{
	divy_sql_elemnode *node = NULL, *lval, *rval, *node2;
	char *str = "", *tmp;

	if (sqlenode == NULL || CHILD_NODE(sqlenode) == NULL) {
		return "NIL";
	}

	for (node = CHILD_NODE(sqlenode); node; node = node->next) {
		/* SELECT ノード */
		if (node->nodetype == DIVY_NT_SELECT) {
			tmp = apr_psprintf(p, "(,%s=>(%s),)",
					node->name, _print_sqlenode(p, node));
		}
		/* 演算子 */
		else if (node->nodetype == DIVY_NT_OPERATOR) {
			lval = LVALUE(node);
			rval = RVALUE(node);
			tmp = apr_psprintf(p, "(\"%s\",%s,", node->name,
						(lval ? lval->name : "(NIL)"));
			if (lval && lval->nodetype == DIVY_NT_SELECT) {
				tmp = apr_pstrcat(p, tmp, "[",
					_print_sqlenode(p, lval), "]", NULL);	
			}

			tmp = apr_pstrcat(p, tmp, (rval ? rval->name : "(NIL)"), NULL);
			if (rval && rval->nodetype == DIVY_NT_SELECT) {
				tmp = apr_pstrcat(p, tmp, "[",
					_print_sqlenode(p, rval), "]", NULL);
			}

			/* 右辺値が連なっていく時 */
			if (rval && rval->next) {
				for (node2 = rval->next; node2; node2 = node2->next) {
					tmp = apr_pstrcat(p, tmp, ",", node2->name, NULL);
					/* SELECT ノードが現れた時 */
					if (node2->nodetype == DIVY_NT_SELECT) {
						tmp = apr_pstrcat(p, tmp, "[",
							_print_sqlenode(p, node2), "]", NULL);	
					}

					if (node2->next == NULL) {
						tmp = apr_pstrcat(p, tmp, ")", NULL);
					}
				}
			}
			else {
				tmp = apr_pstrcat(p, tmp, ")", NULL);
			}
		}
		else {
			tmp = apr_psprintf(p, "(,%s,)", node->name);
		}

		str = apr_pstrcat(p, str, tmp, NULL);
		if (node->next) {
			str = apr_pstrcat(p, str, " -> ", NULL);
		}
	}
	return str;
}

/*-----------------------------------------------------------------------------
 * 以下は内部で使用するユーティリティ関数群
 ----------------------------------------------------------------------------*/
/**
 * SQLパーサが正しい状態になっているかどうか検証する。
 *
 * @param parser divy_sql_parser * パーサを表す構造体
 * @return SQLパーサの状態
 * 	0: 正常 / 1:初期化されていないまたは不正な状態
 */
static int _validate_parser(divy_sql_parser *parser)
{
	if (parser == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The \"parser\" structure not initialized.");
		return 1;
	}

	return 0;
}

/**
 * 指定されたkeyword が keywordtblの中に含まれているかどうか判定する。
 * (note)
 * 	大文字・小文字は無視しません。
 *
 * @param keyword const char * キーワード
 * @param tbllen int キーワードテーブルの長さ
 * @param keywordtbl const char *[] キーワードテーブル(char *を格納する配列)
 * @return int	(1: 含まれていた / 0: 含まれていなかった)
 */
static int _is_keyword_existance(const char *keyword, int tbllen,
					const char *keywordtbl[])
{
	int low, high, mid, ret;

	if (tbllen == 0 || IS_EMPTY(keyword)) {
		return 0;
	}

	/*
	 * keywordtbl をバイナリサーチする
	 */
	for (low = 0, high = tbllen; low < high; ) {
		mid = (low + high) / 2;
		ret = strcmp(keywordtbl[mid], keyword);
		if (ret == 0) {
			return 1;	/* 見つかった */
		}
		else if (ret < 0) {
			low = mid + 1;
		}
		else {
			high = mid;
		}
	}

	return 0;
}


