/**
 * $Id$
 *
 * ファイルストレージ管理用関数の定義
 *
 * (note) このファイル導入の背景
 * 	ここに定義されているものは、元々リポジトリプロバイダ(repos.c) に
 * 	実装されていたもので、これを多少汎用化しています。
 * 	ストレージを触る処理が多くなったので、単純に外に出したという経緯が
 * 	あります。なのでまとまっていません。時間に余裕が出来たら直して下さい。
 */

#include "apr.h"
#include "apr_strings.h"
#include "apr_pools.h"
#include "apr_time.h"
#include "apr_file_info.h"
#include "apr_file_io.h"
#include "apr_buckets.h"

#if APR_HAVE_UNISTD_H
#include <unistd.h>     /* for getpid */
#endif	/* APR_HAVE_UNISTD_H */

#include "httpd.h"
#include "http_protocol.h"
#include "util_filter.h"

#include "util_common.h"
#include "tf_cset.h"
#include "tf_storage.h"
#include "mod_dav_tf.h"
#include "util.h"
#include "tf_rdbo.h"
#include "tf_thumbnail.h"

APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * pfile からERRLOG用のプールを取得するマクロ
 *
 * @param pfile divy_pfile_t *
 */
#define GET_ERRLOGP(pf)	(pfile && pfile->pdir && pfile->pdir->fstorage ? \
			 pfile->pdir->fstorage->log_p : NULL)

/*------------------------------------------------------------------------------
  Declare and Define structure
  ----------------------------------------------------------------------------*/
/**
 * ファイルストレージのマッピングタイプをあらわす列挙型
 */
typedef enum __divy_fstorage_mptype	divy_fstorage_mptype;
enum __divy_fstorage_mptype {
	DIVY_FSMPTYPE_HASH = 0,		/* 計算でディレクトリを算出するマッピング */
	DIVY_FSMPTYPE_URI,		/* URL階層と同じマッピング  */
};

/**
 * ハッシュマッピング方式のディレクトリのみが持つ情報を運ぶコンテナ
 */
typedef struct __divy_hashdir_t		divy_hashdir_t;
struct __divy_hashdir_t {
	/* ユーザディレクトリSEQ   */
	int usr_dirseq;

	/* リソースディレクトリSEQ */
	int rs_dirseq;

	/* ユーザSEQディレクトリ */
	char *usr_dirpath;

	/* このディレクトリの下に存在していたファイル数 */
	int filecount;
};

/**
 * 1つのストレージ(仮想的なDisk)を表す構造体 [ 不完全型の定義 ]
 */
struct __divy_fstorage_t {
	/* マッピングタイプ */
	divy_fstorage_mptype mptype;

	/* マッピングされる物理Diskの基点となるパス */
	char *fsrootpath;

	/* 1つの物理ディレクトリに格納可能な最大ファイル数 */
	apr_int32_t maxresource_per_dir;

	/* 特殊なプロパティ */
	char *usrid;		/* 操作ユーザのユーザID */
	int usrseq;		/* ユーザID SEQ         */
	char *location;		/* Location のURI       */
	char *hostname;		/* 動作サーバのホスト名 */
	pid_t pid;		/* 動作プロセスのpid    */

	/* このストレージが開いているかどうか(1: 開いている) */
	int isActive;

	/* 作業用のプール */
	apr_pool_t *p;

	/* エラーログ用のプール */
	apr_pool_t *log_p;
};

/**
 * 物理ファイルの抽象パスを表す構造体 [ 不完全型の定義 ]
 */
struct __divy_pfile_t {

	/* この抽象パスの親ディレクトリ(内部利用のみ) */
	divy_pdir_t *pdir;

	/* この抽象パスの名前(final path segment) */
	const char *name;

	/* この抽象パスの相対パス */
	const char *relativepath;

	/* この抽象パスのフルパス */
	const char *fullpath;

	/* 作業用のプール */
	apr_pool_t *p;
};

/**
 * 物理ディレクトリの抽象パスを表す構造体 [ 不完全型の定義 ]
 *
 */
struct __divy_pdir_t {

	/* この抽象パスの仮想ストレージへのポインタ */
	divy_fstorage_t *fstorage;

	/* このディレクトリの名前 */
	const char * name;

	/* このディレクトリの相対パス */
	const char *relativepath;

	/* このディレクトリのフルパス */
	const char *fullpath;

	/* 作業用のプール */
	apr_pool_t *p;

	/* 以下はハッシュマッピング方式のディレクトリのみが持つ
	 * 特殊なプロパティ。存在しない時にはNULL。
	 * 但し、ハッシュマッピング方式であっても不正確になる場合には
	 * NULLになります。(例えばdivy_pfile_move した場合など) */
	divy_hashdir_t *hdir;

	/* hdir の情報が正確かどうか
	 * (1: 正確 / 0: 正確ではないかもしれない) */
	int isEffective;
};

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static int _validate_fstorage(divy_fstorage_t *fstorage, int use_mptype);
static int _make_usr_dirseq(int usrseq);
static char * _make_usr_dirpath(apr_pool_t *p, const char *fsrootpath,
				int usr_dirseq, int usrseq);
static char * _make_res_dirpath(apr_pool_t *p, const char *usr_dirpath,
					const char *name);
static int _setup_pdir_env(divy_fstorage_t *fstorage, apr_pool_t *p,
				divy_pdir_t **pdir);
static apr_status_t _count_entry_in_directory(apr_pool_t *p,
					const char *dirpath,
					int *filecount);
static int _move_relativepath(divy_pfile_t *pfile, const char *relativepath);
static int _parse_hdir_relativepath(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *relativepath,
					divy_hashdir_t **hdir);

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * ファイルストレージを表す構造体のインスタンスを生成し、ファイルストレージが
 * "開いた"とみなせる状態にする。(ファイルストレージが開いていれば、いつでも
 * ファイル操作が可能である)
 *
 */
DIVY_DECLARE(int) divy_fstorage_open(request_rec *r, apr_pool_t *p,
						divy_fstorage_t **fstorage)
{
	dav_divy_dir_conf *dconf;
	const char *usrid, *usrseq_str;
	apr_pool_t *log_p = r->pool;

	if (r == NULL || p == NULL) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"request_rec or apr_pool_t is NULL.");
		return DIVY_FS_ST_INVALID_PARAM;
	}

	*fstorage = NULL;	/* 初期化 */

	/*
	 * 各種設定値の内容を検証する
	 */
	dconf = dav_divy_get_dir_config(r);
	usrid = divy_get_userid(r);
	usrseq_str = divy_get_userseq(r);

	if (IS_EMPTY(dconf->fsrootpath)) {
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The root of storage path is EMPTY."
			"You MUST set the path.");
		return DIVY_FS_ST_INVALID_CONFIG;
	}
	else if (IS_EMPTY(usrid) || IS_EMPTY(usrseq_str)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The userid or userseq is EMPTY.");
		return DIVY_FS_ST_ERR;
	}

	/* ディレクトリコンフィグの内容を転写する */
	*fstorage = apr_pcalloc(p, sizeof(divy_fstorage_t));
	(*fstorage)->mptype     = DIVY_FSMPTYPE_HASH;	/* 現時点ではhashに固定 */
	(*fstorage)->fsrootpath = apr_pstrdup(p, dconf->fsrootpath);
	ap_no2slash((char *) (*fstorage)->fsrootpath);
	if (dconf->maxresource_per_dir == DIVY_INT32_C_UNSET ||
	    dconf->maxresource_per_dir <= 0) {
		/* コンフィグに設定されていなかったということなので
		 * デフォルト値を使用する */
		(*fstorage)->maxresource_per_dir = DIVY_MAX_RESOURCE_PER_DIR;
	}
	else {
		(*fstorage)->maxresource_per_dir = dconf->maxresource_per_dir;
	}
	(*fstorage)->usrid      = apr_pstrdup(p, usrid);
	(*fstorage)->usrseq     = atoi(usrseq_str);
	(*fstorage)->location   = apr_pstrdup(p, dav_divy_get_root_uri(r));
	(*fstorage)->hostname   = apr_pstrdup(p, r->server->server_hostname);
	(*fstorage)->pid        = getpid();
	(*fstorage)->isActive   = 1;	/* 開いた */
	(*fstorage)->p          = p;
	(*fstorage)->log_p      = log_p;

	return DIVY_FS_ST_OK;
}

/**
 * ファイルストレージを"閉じた"とみなせる状態にする。
 *
 */
DIVY_DECLARE(int) divy_fstorage_close(divy_fstorage_t *fstorage)
{
	if (fstorage == NULL) return DIVY_FS_ST_OK;	/* 何も出来ない */

	fstorage->isActive = 0;	/* 閉じた */

	return DIVY_FS_ST_OK;
}

/**
 * 一時ファイルの"抽象パス表現"を生成する。
 *
 */
DIVY_DECLARE(int) divy_pfile_mktemp(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *suffix, divy_pfile_t **pfile)
{
	int ret;
	const char *name;
	divy_pdir_t *pdir = NULL;

	/* ストレージの検証 */
	ret = _validate_fstorage(fstorage, DIVY_FSMPTYPE_HASH);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"fstorage is invalid.");
		return ret;
	}
	/* suffixe が空だった */
	else if (IS_EMPTY(suffix)) {
		ERRLOG0(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"suffix is EMPTY.");
		return DIVY_FS_ST_INVALID_PARAM;
	}

	*pfile = NULL;	/* 初期化 */

	/* 一時ファイルのfinal path segment を組み立てる */
	name = divy_make_temporaryname(fstorage, suffix);

	/* 親パスの算出 */
	ret = _setup_pdir_env(fstorage, p, &pdir);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG0(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to setup parent directory environment.");
		return DIVY_FS_ST_ERR;
	}

	/* pfile の生成 */
	*pfile = apr_pcalloc(p, sizeof(divy_pfile_t));
	(*pfile)->pdir         = pdir;
	(*pfile)->name         = name;
	(*pfile)->relativepath = ap_make_full_path(p, pdir->relativepath, name);
	(*pfile)->fullpath     = ap_make_full_path(p, pdir->fullpath, name);
	(*pfile)->p            = p;

	return DIVY_FS_ST_OK;
}

/**
 * 一時ファイルの"抽象パス表現"を生成する。(URI版)
 *
 */
DIVY_DECLARE(int) divy_pfile_mktempByUri(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *uri, const char *suffix,
					divy_pfile_t **pfile)
{
	/* ##### TODO
	 * URI階層と物理階層を一致させる実装をすることになったら実装して下さい。 */
	ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
		"##### Sorry, this function is not ready. #####");
	return DIVY_FS_ST_ERR;
}

/**
 * 一時ファイルの"抽象パス表現"を生成する。(autoindex, サーバ主体メール専用)
 *
 */
DIVY_DECLARE(int) divy_pfile_mkfixedtemp(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *suffix, divy_pfile_t **pfile)
{
	int ret;
	char *tmp;
	const char *name, *relativepath;

	/* ストレージの検証 */
	ret = _validate_fstorage(fstorage, DIVY_FSMPTYPE_HASH);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"fstorage is invalid.");
		return ret;
	}
	/* suffixe が空だった */
	else if (IS_EMPTY(suffix)) {
		ERRLOG0(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"suffix is EMPTY.");
		return DIVY_FS_ST_INVALID_PARAM;
	}

	/* 一時ファイル名を取得 */
	name = divy_make_temporaryname(fstorage, suffix);

	/*
	 * 相対パスを組み立てる
	 */
	/* ユーザディレクトリパス */
	tmp = _make_usr_dirpath(p, "", _make_usr_dirseq(fstorage->usrseq),
							fstorage->usrseq);

	/* リソースディレクトリパス */
	tmp = _make_res_dirpath(p, tmp, DIVY_FIXED_RESDIR_NAME);

	/* 相対パス */
	relativepath = ap_make_full_path(p, tmp, name);

	/* 相対パスからpfile を生成する */
	return divy_pfile_create(fstorage, p, relativepath, pfile);
}

/**
 * 指定されたpath の"抽象パス表現"を生成する。
 *
 */
DIVY_DECLARE(int) divy_pfile_create(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *relativepath,
					divy_pfile_t **pfile)
{
	apr_status_t rv;
	int ret;
	divy_pdir_t *pdir = NULL;
	apr_dir_t *res_dir = NULL;

	/* ストレージの検証 */
	ret = _validate_fstorage(fstorage, DIVY_FSMPTYPE_HASH);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"fstorage is invalid.");
		return ret;
	}
	/* relativepath が空だった */
	else if (IS_EMPTY(relativepath)) {
		ERRLOG0(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"relativepath is EMPTY.");
		return DIVY_FS_ST_INVALID_PARAM;
	}

	*pfile = NULL;	/* 初期化 */

	/* 親パスの算出 */
	pdir = apr_pcalloc(p, sizeof(divy_pdir_t));
	pdir->fstorage     = fstorage;
	pdir->relativepath = divy_extract_parentath(p, relativepath);
	if (pdir->relativepath != NULL) {
		/* "/" 以外の場合 */
		ap_no2slash((char *) pdir->relativepath);
		pdir->name     = divy_extract_finalpath(p, pdir->relativepath);
		pdir->fullpath = ap_make_full_path(p, fstorage->fsrootpath,
							pdir->relativepath);
		ap_no2slash((char *) pdir->fullpath);
	}
	else {
		/* "/"の場合 */
		pdir->name     = NULL;
		pdir->fullpath = apr_pstrdup(p, fstorage->fsrootpath);
	}
	pdir->p = p;

	/* pdir->fullpath が存在しなければ作成する */
	rv = apr_dir_open(&res_dir, pdir->fullpath, p);
	if (rv == APR_SUCCESS) {
		rv = apr_dir_close(res_dir);
		if (rv != APR_SUCCESS) {
			ERRLOG2(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to close directory. (path = %s, "
				"code = %d)", pdir->fullpath, rv);
			return DIVY_FS_ST_ERR;
		}
	}
	else if (APR_STATUS_IS_ENOENT(rv)) {
		/* pdir->fullpath を作成する */
		rv = apr_dir_make_recursive(pdir->fullpath,
					APR_UREAD | APR_UWRITE | APR_UEXECUTE, p);
		if (rv != APR_SUCCESS) {
			ERRLOG2(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to make resource dir. "
				"(path = %s, code = %d)", pdir->fullpath, rv);
			return DIVY_FS_ST_ERR;
		}
	}
	/* 予期しないエラー */
	else if (rv != APR_SUCCESS) {
		ERRLOG2(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
			"Failed to open resource dir. "
			"(path = %s, code = %d)", pdir->fullpath, rv);
		return DIVY_FS_ST_ERR;
	}

	/* hdir を算出 */
	ret = _parse_hdir_relativepath(fstorage, p,
					pdir->relativepath, &pdir->hdir);
	if (ret == DIVY_FS_ST_OK) {
		pdir->isEffective  = 1;	/* 情報を正しく取得出来た */
	}

	/* pfile の生成 */
	*pfile = apr_pcalloc(p, sizeof(divy_pfile_t));
	(*pfile)->pdir         = pdir;
	(*pfile)->name         = divy_extract_finalpath(p, relativepath);
	(*pfile)->relativepath = apr_pstrdup(p, relativepath);
	(*pfile)->fullpath     = ap_make_full_path(p, fstorage->fsrootpath,
						   relativepath);
	ap_no2slash((char *)(*pfile)->fullpath);
	(*pfile)->p            = p;

	return DIVY_FS_ST_OK;
}

/**
 * 指定されたURI文字列から"抽象パス表現"を生成する。
 *
 */
DIVY_DECLARE(int) divy_pfile_createByUri(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *uri, divy_pfile_t **pfile)
{
	/* ##### TODO
	 * URI階層と物理階層を一致させる実装をすることになったら実装して下さい。 */
	ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
		"##### Sorry, this function is not ready. #####");
	return DIVY_FS_ST_ERR;
}

/**
 * ソースとディスティネーションの情報を持つcopyinfo から"抽象パス表現"を生成する。
 * 生成したdivy_pfile_t は、copyinfo に詰めて返却します。
 *
 */
DIVY_DECLARE(int) divy_pfile_createByCopyinfo(divy_fstorage_t *fstorage, apr_pool_t *p,
					divy_copyfile_info *copyinfo)
{
	int ret, i_maxdir_name;
	apr_status_t rv;
	divy_pdir_t *pdir = NULL;
	divy_copyfile_info *cp;
	char *res_dirpath, *name;
	divy_pfile_t *src_pfile, *dst_pfile;
	divy_hashdir_t *swap_hdir;

	/* ストレージの検証 */
	ret = _validate_fstorage(fstorage, DIVY_FSMPTYPE_HASH);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"fstorage is invalid.");
		return ret;
	}
	/* copyinfo が空だった */
	else if (copyinfo == NULL) {
		ERRLOG0(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"copyinfo is NULL.");
		return DIVY_FS_ST_INVALID_PARAM;
	}

	/* 親パスの算出 */
	ret = _setup_pdir_env(fstorage, p, &pdir);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG0(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to setup parent directory environment.");
		return DIVY_FS_ST_ERR;
	}

	for (cp = copyinfo; cp; cp = cp->next) {
		if (!cp->isfile) continue;

		/* dst のpfile を生成 */
		dst_pfile = apr_pcalloc(p, sizeof(divy_pfile_t));
		dst_pfile->name = cp->rsid;

		if (++pdir->hdir->filecount > fstorage->maxresource_per_dir) {
			swap_hdir     = pdir->hdir;	/* 一時退避 */
			i_maxdir_name = swap_hdir->rs_dirseq + 1;
			name          = apr_itoa(p, i_maxdir_name);

			/* 最大リソースディレクトリSEQ 数を超えたとき
			   (i_maxdir_name + 1) のディレクトリを作成する */
			res_dirpath = _make_res_dirpath(p, swap_hdir->usr_dirpath, name);
			rv = apr_dir_make(res_dirpath, 
					APR_UREAD | APR_UWRITE | APR_UEXECUTE, p);
			if (rv != APR_SUCCESS) {
				ERRLOG2(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
					"Failed to make resource dir. (path = %s,"
					"code = %d)", res_dirpath, rv);
				return DIVY_FS_ST_ERR;
			}

			pdir = apr_pcalloc(p, sizeof(divy_pdir_t));
			pdir->fstorage     = fstorage;
			pdir->name         = name;
			pdir->relativepath = apr_pstrdup(p,
					&res_dirpath[strlen(fstorage->fsrootpath)]);
			pdir->fullpath     = res_dirpath;
			pdir->isEffective  = 1;	/* 有効である */
			pdir->p            = p;

			pdir->hdir = apr_pcalloc(p, sizeof(divy_hashdir_t));
			pdir->hdir->usr_dirseq   = swap_hdir->usr_dirseq;
			pdir->hdir->rs_dirseq    = i_maxdir_name;
			pdir->hdir->usr_dirpath  = apr_pstrdup(p, swap_hdir->usr_dirpath);
			pdir->hdir->filecount    = 1;
		}

		dst_pfile->pdir         = pdir;
		dst_pfile->relativepath = ap_make_full_path(p,
							pdir->relativepath, cp->rsid);
		dst_pfile->fullpath     = ap_make_full_path(p,
							pdir->fullpath, cp->rsid);
		dst_pfile->p            = p;

		/* src のpfile を生成 */
		src_pfile = apr_pcalloc(p, sizeof(divy_pfile_t));
		src_pfile->pdir         = NULL;	/* 不要なのでつけないことにしました */
		src_pfile->name         = divy_extract_finalpath(p, cp->src_relativepath);
		src_pfile->relativepath = cp->src_relativepath;
		src_pfile->fullpath     = ap_make_full_path(p, fstorage->fsrootpath,
							cp->src_relativepath); 
		ap_no2slash((char *)src_pfile->fullpath);
		src_pfile->p            = p;

		/* cp への代入 */
		cp->dst_relativepath = (char *) dst_pfile->relativepath;
		cp->dst_pfile        = dst_pfile;
		cp->src_pfile        = src_pfile;
	}

	return DIVY_FS_ST_OK;
}


/**
 * 指定されたpfile1 とpfile2 の抽象パス表現が等しいかどうか判定する。
 *
 */
DIVY_DECLARE(int) divy_pfile_compare(divy_pfile_t *pfile1, divy_pfile_t *pfile2)
{
	if (pfile1 == NULL || pfile2 == NULL) {
		return DIVY_FS_ST_ERR;
	}

	if (IS_EMPTY(pfile1->fullpath) && IS_FILLED(pfile2->fullpath)) {
		return DIVY_FS_ST_LESSTHAN;
	}
	else if (IS_FILLED(pfile1->fullpath) && IS_EMPTY(pfile2->fullpath)) {
		return DIVY_FS_ST_MORETHAN;
	}
	else if (IS_EMPTY(pfile1->fullpath) && IS_EMPTY(pfile2->fullpath)) {
		return DIVY_FS_ST_EQUAL;
	}
	else {
		int ret = strcmp(pfile1->fullpath, pfile2->fullpath);
		if (ret < 0) {
			return DIVY_FS_ST_LESSTHAN;
		}
		else if (ret == 0) {
			return DIVY_FS_ST_EQUAL;
		}
		else {
			return DIVY_FS_ST_MORETHAN;
		}
	}
}

/**
 * 指定されたpfile の名前(Final Path Segment) をname に変更する。
 *
 */
DIVY_DECLARE(int) divy_pfile_rename(divy_pfile_t *pfile, const char *name)
{
	apr_pool_t *p;

	if (pfile == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"pfile is NULL.");
		return DIVY_FS_ST_INVALID_PARAM;
	}
	else if (IS_EMPTY(name)) {
		apr_pool_t *log_p = GET_ERRLOGP(pfile);
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"name is EMPTY.");
		return DIVY_FS_ST_INVALID_PARAM;
	}
	p = pfile->p;

	/* final path segment の変更 */
	pfile->name = apr_pstrdup(p, name);
	if (pfile->pdir != NULL) {
		pfile->relativepath = ap_make_full_path(p,
						pfile->pdir->relativepath, name);
		pfile->fullpath     = ap_make_full_path(p,
						pfile->pdir->fullpath, name);
	}
	else {
		const char *p_relativepath, *p_fullpath;

		/* 自分のパスをパースして算出する */

		p_relativepath = divy_extract_parentath(p, pfile->relativepath);
		if (IS_EMPTY(p_relativepath)) {
			p_relativepath = apr_psprintf(p, "%c", PATH_DELIMITER_CH);
		}
		p_fullpath = divy_extract_parentath(p, pfile->fullpath);
		if (IS_EMPTY(p_fullpath)) {
			p_fullpath = apr_psprintf(p, "%c", PATH_DELIMITER_CH);
		}

		pfile->relativepath = ap_make_full_path(p, p_relativepath, name);
		pfile->fullpath     = ap_make_full_path(p, p_fullpath, name);
	}

	return DIVY_FS_ST_OK;
}

/**
 * 指定されたpfile の相対パスをrelativepath に変更する。
 * また、pfile を開いたfstorageがNULLであればこれも失敗します。
 *
 */
DIVY_DECLARE(int) divy_pfile_relativemove(divy_pfile_t *pfile, const char *relativepath)
{
	apr_pool_t *log_p = NULL;

	if (pfile == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"pfile is NULL.");
		return DIVY_FS_ST_INVALID_PARAM;
	}
	else if (IS_EMPTY(relativepath)) {
		log_p = GET_ERRLOGP(pfile);
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"relativepath is EMPTY.You MUST set it.");
		return DIVY_FS_ST_INVALID_PARAM;
	}
	else if (pfile->pdir == NULL || pfile->pdir->fstorage == NULL ||
		 IS_EMPTY(pfile->pdir->fstorage->fsrootpath)) {
		log_p = GET_ERRLOGP(pfile);
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"fstorage is invalid.");
		return DIVY_FS_ST_INVALID_PARAM;
	}

	/* パスの変更処理を実施 */
	return _move_relativepath(pfile, relativepath);
}


/**
 * 指定されたpfile のフルパスをfullpath に変更する。
 *
 */
DIVY_DECLARE(int) divy_pfile_fullmove(divy_pfile_t *pfile, const char *fullpath)
{
	char *fsrootpath;
	const char *relativepath;
	int ret;
	apr_size_t len;
	apr_pool_t *log_p = NULL;

	if (pfile == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"pfile is NULL.");
		return DIVY_FS_ST_INVALID_PARAM;
	}
	else if (IS_EMPTY(fullpath)) {
		log_p = GET_ERRLOGP(pfile);
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"name is EMPTY.You MUST set it.");
		return DIVY_FS_ST_INVALID_PARAM;
	}
	else if (pfile->pdir == NULL || pfile->pdir->fstorage == NULL ||
		 IS_EMPTY(pfile->pdir->fstorage->fsrootpath)) {
		log_p = GET_ERRLOGP(pfile);
		ERRLOG0(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"fstorage is invalid.");
		return DIVY_FS_ST_INVALID_PARAM;
	}

	log_p = GET_ERRLOGP(pfile);

	/* fullpath とpfile のルートパスが同じかどうか検証する */
	fsrootpath = pfile->pdir->fstorage->fsrootpath;
	len = strlen(fsrootpath);
	ret = strncmp(fullpath, fsrootpath, len);
	if (ret != 0) {
		ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The root path of fullpath is different from fsrootpath."
			"(fullpath = %s, fsrootpath = %s)",
			fullpath, fsrootpath);
		return DIVY_FS_ST_DIFFERENT_ROOT;
	}

	/* 相対パスの算出 */
	relativepath = &fullpath[strlen(fsrootpath)];

	/* パスの変更処理を実施 */
	return _move_relativepath(pfile, relativepath);
}


/*
 * 以下はアクセサの宣言
 */

/**
 * 指定されたpfile のフルパスを取得する。
 *
 */
DIVY_DECLARE(const char *) divy_pfile_get_fullpath(divy_pfile_t *pfile)
{
	if (pfile == NULL) return NULL;	/* 何も出来ない */

	/* 書き換えられても影響がないようにする目的でコピーして渡す */
	return apr_pstrdup(pfile->p, pfile->fullpath);
}

/**
 * 指定されたpfile の相対パス(ストレージルートを除いたパス)を取得する。
 *
 */
DIVY_DECLARE(const char *) divy_pfile_get_relativepath(divy_pfile_t *pfile)
{
	if (pfile == NULL) return NULL;	/* 何も出来ない */

	/* 書き換えられても影響がないようにする目的でコピーして渡す */
	return apr_pstrdup(pfile->p, pfile->relativepath);
}

/**
 * 指定されたpfile の名前(final path segment)を取得する。
 *
 */
DIVY_DECLARE(const char *) divy_pfile_get_name(divy_pfile_t *pfile)
{
	if (pfile == NULL) return NULL;	/* 何も出来ない */

	/* 書き換えられても影響がないようにする目的でコピーして渡す */
	return apr_pstrdup(pfile->p, pfile->name);
}

/**
 * 指定されたpfile の仮想ストレージを取得する。
 *
 */
DIVY_DECLARE(divy_fstorage_t *) divy_pfile_get_fstorage(divy_pfile_t *pfile)
{
	if (pfile == NULL || pfile->pdir == NULL) return NULL;	/* 何も出来ない */

	return pfile->pdir->fstorage;
}

/**
 * 指定されたpfile の親ディレクトリのフルパスを取得する。
 *
 */
DIVY_DECLARE(const char *) divy_pfile_get_pfullpath(divy_pfile_t *pfile)
{
	if (pfile == NULL) return NULL;	/* 何も出来ない */

	if (pfile->pdir == NULL) {
		/* pfileのフルパスをパースして返す */
		return divy_extract_parentath(pfile->p, pfile->fullpath);
	}
	return apr_pstrdup(pfile->p, pfile->pdir->fullpath);
}

/**
 * 指定されたpfile の親ディレクトリの相対パス(ストレージルートを除いたパス)を
 * 取得する。
 *
 */
DIVY_DECLARE(const char *) divy_pfile_get_prelativepath(divy_pfile_t *pfile)
{
	if (pfile == NULL) return NULL;	/* 何も出来ない */

	if (pfile->pdir == NULL) {
		/* pfileの相対パスをパースして返す */
		return divy_extract_parentath(pfile->p, pfile->relativepath);
	}
	return apr_pstrdup(pfile->p, pfile->pdir->relativepath);
}


/*
 * 以下はユーティリティ
 */

/**
 * fullpath が示す物理ファイルパスから名称パート(final path segment) を取り出して返却する。
 *
 */
DIVY_DECLARE(const char *) divy_extract_finalpath(apr_pool_t *p, const char *fullpath)
{
	char *first, *ps;

	if (IS_EMPTY(fullpath)) return NULL;

	/* 先頭がPATH_DELIMITER_CH でなければ付ける */
	if (strlen(fullpath) > 0 && fullpath[0] != PATH_DELIMITER_CH) {
		first = ps = apr_psprintf(p, "%c%s", PATH_DELIMITER_CH, fullpath);
	}
	else {
		first = ps = apr_pstrdup(p, fullpath);	/* コピー */
	}

	while ((ps = divy_strrchr(first, PATH_DELIMITER_CH)) != NULL) {
		if (*(ps + 1) != '\0') {
			return ++ps;
		}
		*ps = '\0';	/* 見つかった位置を終端にしてしまう */
		if (first == ps) {
			/* 先頭まで移動しても見つからなかった場合(/の場合のみ) */
			return NULL;
		}
		ps--;	/* 1つ前に戻す */
	}
	if (first && strlen(first) > 0) {
		return first;
	}
	return NULL;
}

/**
 * fullpath が示す物理ファイルパスから親ディレクトリのパスを取り出して返却する。
 *
 */
DIVY_DECLARE(const char *) divy_extract_parentath(apr_pool_t *p, const char *fullpath)
{
	char *first, *ps;

	if (IS_EMPTY(fullpath)) return NULL;

	/* 先頭がPATH_DELIMITER_CH でなければ付ける */
	if (strlen(fullpath) > 0 && fullpath[0] != PATH_DELIMITER_CH) {
		first = ps = apr_psprintf(p, "%c%s", PATH_DELIMITER_CH, fullpath);
	}
	else {
		first = ps = apr_pstrdup(p, fullpath);	/* コピー */
	}

	while ((ps = divy_strrchr(first, PATH_DELIMITER_CH)) != NULL) {
		if (*(ps + 1) != '\0') {
			*ps = '\0';	/* "/" を終端にしてしまう */
			/* 先頭まで移動していないか? */
			if (ps == first) {
				/* つまり「/xxx」のような１階層だったということ
				 * この場合の親は「/」である */
				return apr_psprintf(p, "%c", PATH_DELIMITER_CH);
			}
			return first;
		}
		*ps = '\0';	/* 見つかった位置を終端にする */
		if (first == ps) {
			/* 先頭まで移動しても見つからなかった場合(/の場合のみ) */
			return NULL;
		}
		ps--;	/* 1つ前に戻す */
	}

	return NULL;
}


/**
 * relativepath の絶対パスをfstorage のストレージ上に作成する。
 *
 */
DIVY_DECLARE(const char *) divy_make_fullpath(divy_fstorage_t *fstorage,
						const char *relativepath)
{
	int ret;
	char *fullpath;

	if (fstorage == NULL) return NULL;

	/* ストレージの検証 */
	ret = _validate_fstorage(fstorage, DIVY_FSMPTYPE_HASH);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"fstorage is invalid.");
		return NULL;
	}
	else if (IS_EMPTY(relativepath)) {
		return apr_pstrdup(fstorage->p, fstorage->fsrootpath);
	}

	/* URIの合成 */
	fullpath = (char *) ap_make_full_path(fstorage->p,
					fstorage->fsrootpath, relativepath);
	ap_no2slash(fullpath);

	return (const char *) fullpath;
}

/**
 * 一時ファイル名称を組み立てて返却する
 *
 */
DIVY_DECLARE(const char *) divy_make_temporaryname(divy_fstorage_t *fstorage,
							const char *suffix)
{
	if (fstorage == NULL) return NULL;

	 /*
	  * [ 書式 ]
	  * 	(ユーザSeq).(プロセスID)@(ホスト名).(エポックタイム).(suffix)
	  */
	return apr_psprintf(fstorage->p,
				"%d.%"APR_PID_T_FMT"@%s.%"APR_TIME_T_FMT".%s",
				fstorage->usrseq, fstorage->pid,
				fstorage->hostname, apr_time_now(), suffix);
}

/**
 * サイズがgetcontentlength バイトのfilepath が示すファイルを
 * output ストリームに流す。
 *
 */
DIVY_DECLARE(int) divy_sendfile2stream(request_rec *r,
					ap_filter_t *output,
					apr_int64_t getcontentlength,
					const char *filepath)
{
	apr_status_t rv;
	int ret;
	apr_pool_t *p          = r->pool;
	apr_file_t *fd         = NULL;
	apr_finfo_t finfo      = { 0 };
	apr_bucket_brigade *bb = NULL;
	apr_bucket *bkt        = NULL;
	apr_off_t fsize;
	const char *ua         = NULL; /* MacのFinder調査の為のUserAgent */

	/* 条件付GETの検査 */
	ret = ap_meets_conditions(r);
	if (ret != OK) {
		/* エラーとは限らないのでエラーログは出さない */
		return ret;
	}

	/* ファイルを開く */
	rv = apr_file_open(&fd, filepath, APR_READ | APR_BINARY, 0, p);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to open file. (filepath = %s, code = %d)",
			filepath, rv);
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	/* ファイルサイズの取得 */
	rv = apr_file_info_get(&finfo, APR_FINFO_SIZE, fd);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to get file size. (code = %d)", rv);
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	/* DBのサイズとファイルサイズがずれていないか? */
	if (getcontentlength != APR_INT64_C(-1) && finfo.size != getcontentlength) {
		ERRLOG3(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
			"The size of physical file are different from "
			"the size of storing in repository db. We use "
			"physical file size, but please check the file."
			"(filepath = %s, physicalsize = %s,"
			"db filesize = %"APR_INT64_T_FMT")", filepath,
			apr_off_t_toa(p, finfo.size), getcontentlength);
	}

	fsize = finfo.size;

	/* bucket brigade を作成する */
	bb = apr_brigade_create(p, output->c->bucket_alloc);

#ifdef DIVY_SUPPORT_LFS
	/* 対象ファイルが2GBよりも大きいか? */
	if (fsize > APR_INT64_C(0x7FFFFFFF)) {
		apr_bucket *bkt2;
		bkt = apr_bucket_file_create(fd, 0, AP_MAX_SENDFILE, p,
						output->c->bucket_alloc);
		/* ファイル型のapr_bucketをAP_MAX_SENDFILE(byte)刻みで作成する */
		while (fsize > AP_MAX_SENDFILE) {
			/* コピーして作成する (byte列はコピーされません) */
			apr_bucket_copy(bkt, &bkt2);
			APR_BRIGADE_INSERT_TAIL(bb, bkt2);

			/* 次に作成するbacketのoffset を計算する */
			bkt->start += AP_MAX_SENDFILE;
			fsize -= AP_MAX_SENDFILE;
		}
		bkt->length = (apr_size_t) fsize; /* Resize just the last bucket */
	}
	else {
#endif	/* DIVY_SUPPORT_LFS */
		/* この場合はサイズをapr_size_tにキャストしてもよい */
		bkt = apr_bucket_file_create(fd, 0,
					(apr_size_t) fsize,
				     	p, output->c->bucket_alloc);
#ifdef DIVY_SUPPORT_LFS
	}
#endif	/* DIVY_SUPPORT_LFS */

	APR_BRIGADE_INSERT_TAIL(bb, bkt);

	/* Create an End of Stream bucket */
	bkt = apr_bucket_eos_create(output->c->bucket_alloc);
	APR_BRIGADE_INSERT_TAIL(bb, bkt);

	/* 出力フィルタに渡す */
	rv = ap_pass_brigade(output, bb);
	if (APR_STATUS_IS_ECONNABORTED(rv)) {
		/* accept の時にクライアントがコネクションを強制切断した場合 */
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_OS,
			"Failed to write content to filter."
			"Maybe the client closed a connection forcibly, so "
			"ignore this message.(filepath = %s, code = %d)",
			filepath, rv);
		return HTTP_INTERNAL_SERVER_ERROR;
	}
	else if (rv != APR_SUCCESS) {
		/* accept の時にクライアントがコネクションを強制切断した場合 */

		/* WebDAVFS(Mac)は自分勝手にコネクションを切断することを行います
		 * 画像ファイルをアップロードしたときなどそのファイルのサムネイル
		 * を取得しようとしてファイルの先頭を少し読むRangeヘッダが入っています
		 *
		 * 従来のエラーコードはHTTP_INTERNAL_ERRORだったが、
		 * このエラーを返すと、Macはスネてしまい処理をやめてしまいます
		 * またこの動作はMacでは意味正常な動きであるためログが大量に
		 * 出力されるのを防ぐため、MacのFinderの時だけはログを出力しない
		 * ようにします。
		 */
		ua = dav_divy_get_user_agent(r);
		if (!ua || strncmp(ua, "WebDAVFS", 8) == 0) {
			/* デバッグ時の時だけログ出力とする */
			ERRLOG2(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
					"client closed a connection. User Agent = %s normal operation. code = %d", ua, rv);
			return HTTP_FORBIDDEN;
		}

		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to write content to filter."
			"(filepath = %s, code = %d)", filepath, rv);
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	return OK;
}

/**
 * pathlist が示す物理ファイルをストレージから削除する。
 *
 */
DIVY_DECLARE(int) divy_remove_physical_files(request_rec *r,
					divy_linkedlist_t *pathlist)
{
	apr_status_t rv;
	int ret;
	char *tmp;
	const char *filepath;
	apr_pool_t *p            = r->pool;
	divy_cset_t *path_set    = divy_cset_make(p);
	divy_cset_index_t *ci    = NULL;
	divy_linkedlist_t *del_file;
	divy_fstorage_t *fstorage= NULL;

	if (pathlist == NULL) return 0;	/* 何もしない */

	/* ストレージのopen */
	ret = divy_fstorage_open(r, p, &fstorage);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC, "Failed to open storage.");
		return 1;
	}

	/*
	 * 物理ファイルの削除
	 */
	for (del_file = pathlist; del_file; del_file = del_file->next) {

		/* サムネイルファイルの削除 */
		(void) divy_thumbnail_remove(p, r, del_file->val);

		/* 削除ファイルパスの組み立て */
		filepath = divy_make_fullpath(fstorage, del_file->val);

		/* 物理ファイルの削除 */
		rv = apr_file_remove(filepath, p);
		if (APR_STATUS_IS_ENOENT(rv)) {
			/* 既に無いのなら消さなくていい。ログにも書かない */
			continue;
		}
		else if (rv != APR_SUCCESS) {
			/* 
			 * 既にDB のエントリは消えてしまったので、失敗しても
			 * どうしようもない。ここは成功したものとみなす。
			 * (ゴミが残るだけなので)
			 */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_FS,
					"Failed to remove physical file for DELETE. "
					"Maybe this file became garbage,"
					"Please check the file and delete it."
					"(path = %s, code = %d)", filepath, rv);
		}
		else {
			/* 削除したファイルパス(ファイル名パートは除く) を憶えておく */
			tmp = divy_strrchr((char *)filepath, '/');
			*tmp = '\0';	/* filepath から名称パートを切り落とす */
			divy_cset_set(path_set, filepath);
		}
	}

	/*
	 * ファイルを削除した結果、物理ファイル格納ディレクトリが空になった時には
	 * そのディレクトリを消しておく
	 */
	for (ci = divy_cset_first(path_set); ci; ci = divy_cset_next(ci)) {

		/* ファイル格納パスの取り出し */
		divy_cset_this(ci, (const char **)&filepath);

		/* サムネイルディレクトリの削除をトライしてみる */
		(void) divy_thumbnail_removepdir(p, r, filepath);

		/* ディレクトリの削除をトライしてみる(ちょっと乱暴だけど) */
		rv = apr_dir_remove(filepath, p);
		if (APR_STATUS_IS_ENOTEMPTY(rv)) {
			/* ディレクトリの中にエントリがあったので失敗 */
			continue;	/* 構わず続ける。問題ないので */
		}
		else if (APR_STATUS_IS_ENOENT(rv)) {
			/* そのディレクトリエントリが既に存在しなかった */
			continue;	/* だったら消さなくていい */
		}
		else if (rv != APR_SUCCESS) {
			/* 何らかの問題が起きた。だけど気にしない */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_FS,
					"Failed to remove directory. but it is "
					"probably occurred by concurrent access,"
					"so please ignore this.(dir path = %s, "
					"code = %d)", filepath, rv);
		}
	}

	/* ストレージを閉じる */
	(void) divy_fstorage_close(fstorage);

	return 0;
}

/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * fstorage が示す仮想ストレージが、その下にリソースを作成するのに十分な情報を
 * 持っているかどうかを検証する。
 *
 * @param fstorage divy_fstorage_t * ファイルストレージを表す構造体へのポインタ
 * @param use_mptype int このストレージで使用するマッピングタイプの値
 * @return int 処理ステータス
 *	DIVY_FS_ST_OK               : 正常
 *	DIVY_FS_ST_INVALID_PARAM    : fstorage がNULLだった(引数不正)
 *	DIVY_FS_ST_CLOSED           : fstorage は既にclose されていた
 *	DIVY_FS_ST_MPTYPE_MISSMATCH : use_mptype と fstorage のマッピングタイプが不一致
 */
static int _validate_fstorage(divy_fstorage_t *fstorage, int use_mptype)
{
	if (fstorage == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"fstorage is NULL.");
		return DIVY_FS_ST_INVALID_PARAM;
	}
	/* ストレージが閉じられていた場合 */
	else if (!fstorage->isActive) {
		ERRLOG0(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The fstorage already had been closed.");
		return DIVY_FS_ST_CLOSED;
	}
	else if (fstorage->mptype != use_mptype) {
		ERRLOG1(fstorage->log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"This function is only used for \"DIVY_FSMPTYPE_HASH\" "
			"mapping.(mptype = %d)", fstorage->mptype);
		return DIVY_FS_ST_MPTYPE_MISSMATCH;
	}

	return DIVY_FS_ST_OK;
}

/**
 * ユーザディレクトリSEQを計算する
 * (note) 2004.09.03 Fir takehara
 * 	本当は剰余算のはずだったのだが、潜在不良によって必ず0番目の
 * 	ディレクトリにマップされていたことに気づいた。これはまずいので
 * 	急遽、割り算に変更することにした。
 *
 * @param usrseq int ユーザID SEQ
 * @return int ユーザディレクトリSEQ
 */
static int _make_usr_dirseq(int usrseq)
{
	return usrseq / DIVY_MAX_USER_PER_DIR;
}

/**
 * ユーザディレクトリパスを生成して返却する。
 * [ フォーマット ]
 * 	$root/(ユーザディレクトリSEQ)/(ユーザID SEQ)
 *
 * @param p apr_pool_t * 戻り値が割り当てられるプール
 * @param fsrootpath const char * ストレージルートパス
 * @param usr_dirseq int ユーザディレクトリSEQ
 * @param usrseq int ユーザID SEQ
 * @return char * ユーザディレクトリパス
 * 		fsrootpath をNULLまたは空文字にしたら相対パスが返却されます。
 */
static char * _make_usr_dirpath(apr_pool_t *p, const char *fsrootpath,
				int usr_dirseq, int usrseq)
{
	char *usr_dirpath;

	if (IS_EMPTY(fsrootpath)) {
		fsrootpath = apr_psprintf(p, "%c", PATH_DELIMITER_CH);
	}

	usr_dirpath = ap_make_full_path(p, fsrootpath,
				ap_make_full_path(p, apr_itoa(p, usr_dirseq),
						     apr_itoa(p, usrseq)));
	ap_no2slash(usr_dirpath);

	return usr_dirpath;
}

/**
 * リソースディレクトリパスを生成して返却する。
 * [ フォーマット ]
 * 	(ユーザディレクトリパス)/(名前)
 *
 * @param p apr_pool_t * 戻り値が割り当てられるプール
 * @param usr_dirpath const char * ユーザディレクトリパス
 * @param name const char * 名称(final path segment)
 * @return char * リソースディレクトリパス
 */
static char * _make_res_dirpath(apr_pool_t *p, const char *usr_dirpath,
					const char *name)
{
	char *res_dirpath = ap_make_full_path(p, usr_dirpath, name);
	ap_no2slash(res_dirpath);

	return res_dirpath;
}

/**
 * fstorage が示すストレージ上に物理ディレクトリ(divy_pdir_t) の環境を算出し、
 * 必要であれば物理ディレクトリを作成する。
 * (note)
 * 	この関数は、仮想ではなく、実際に物理ディレクトリを作成してしまいます。
 * 	呼び出し環境に注意が必要です。
 *
 * @param fstorage divy_fstorage_t * ファイルストレージを表す構造体へのポインタ
 * @param p apr_pool_t * 作業用のプール (*pdir とライフサイクルを共にする)
 * @param pdir divy_pdir_t ** 親パス情報へのポインタ
 * @return int 処理ステータス (DIVY_FS_ST_OK: 成功 / 上記以外: 失敗)
 */
static int _setup_pdir_env(divy_fstorage_t *fstorage, apr_pool_t *p,
				divy_pdir_t **pdir)
{
	apr_status_t rv;
	char *usr_dirpath, *res_dirpath = NULL;
	apr_finfo_t finfo         = { 0 };
	apr_dir_t *usr_dir        = NULL;
	int filecount             = 0;
	int i_name, i_maxdir_name = 0;
	int usr_dirseq;
	apr_pool_t *log_p         = NULL;

	if (fstorage == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"fstorage is NULL.");
		return DIVY_FS_ST_INVALID_PARAM;
	}

	*pdir = NULL;	/* 初期化 */
	log_p = fstorage->log_p;

	/* ユーザディレクトリSEQ の算出 */
	usr_dirseq = _make_usr_dirseq(fstorage->usrseq);

	/* ユーザディレクトリパスの算出 */
	usr_dirpath = _make_usr_dirpath(p, fstorage->fsrootpath,
					usr_dirseq, fstorage->usrseq);
	/* 
	 * リソースディレクトリSEQ の取得 
	 * [ $root/(ユーザディレクトリSEQ)/(ユーザID SEQ)/$i_maxdir_name/ ]
	 *
	 * リソースディレクトリSEQ ディレクトリの下にあるディレクトリ名を取得し、
	 * その名前を数値に直したときに最も大きくなる値を求める。
	 * この値が１ディレクトリあたりに格納可能な最大ディレクトリ数を
	 * 超えていたら、 (最大値 + 1)をリソースディレクトリSEQとする
	 */

	/* ユーザディレクトリパスのディレクトリを開く */
	rv = apr_dir_open(&usr_dir, usr_dirpath, p);
	if (rv == APR_SUCCESS) {

		/* i_maxdir_name を求める */
		i_maxdir_name = 0;
		while(apr_dir_read(&finfo, 0, usr_dir) != APR_ENOENT) {
			i_name = atoi(finfo.name);	/* int に変換 */
			if (i_maxdir_name < i_name) {
				i_maxdir_name = i_name;
			}
		}
		/* usr_dir を閉じる */	
		rv = apr_dir_close(usr_dir);
		if (rv != APR_SUCCESS) {
			ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
					"Failed to close user-resource dir. "
					"(path = %s, code = %d)", usr_dirpath, rv);
			return DIVY_FS_ST_ERR;
		}

		usr_dir = NULL;

		/* 
		 * リソースディレクトリSEQ ($usr_dir/$i_maxdir_name) ディレクトリ直下の
		 * エントリ(ファイル)を数える
		 */
		res_dirpath = _make_res_dirpath(p, usr_dirpath, apr_itoa(p, i_maxdir_name));
		rv = _count_entry_in_directory(p, res_dirpath, &filecount);
		if (APR_STATUS_IS_ENOENT(rv)) {
			/* ディレクトリが無かったら作成する */
			rv = apr_dir_make(res_dirpath, 
					APR_UREAD | APR_UWRITE | APR_UEXECUTE, p);
			if (rv != APR_SUCCESS) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
						"Failed to make resource dir. "
						"(path = %s, code = %d)", res_dirpath, rv);
				return DIVY_FS_ST_ERR;
			}
		}
		else if (rv != APR_SUCCESS) {
			return DIVY_FS_ST_ERR;
		}
	}
	/* ディレクトリが無かった場合 (初めてアップロードした時だけ) */
	else if (APR_STATUS_IS_ENOENT(rv)) {

		/* $usr_dirpath/0 のディレクトリを作成する */
		res_dirpath = _make_res_dirpath(p, usr_dirpath, "0");
		rv = apr_dir_make_recursive(res_dirpath,
					APR_UREAD | APR_UWRITE | APR_UEXECUTE, p);
		if (rv != APR_SUCCESS) {
			ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to make resource dir. "
				"(path = %s, code = %d)", res_dirpath, rv);
			return DIVY_FS_ST_ERR;
		}

		i_maxdir_name = 0;
	}
	/* Permission Danied */
	else if (APR_STATUS_IS_EACCES(rv)) {
		ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
			"Failed to open user-resource dir. (path = %s, code = %d) "
			"Maybe, Two or more Apache with different uid has "
			"started, or the permission of a file system "
			"is not right. Please check file system.", usr_dirpath, rv);
		return DIVY_FS_ST_ERR;
	}
	/* 予期しないエラー */
	else if (rv != APR_SUCCESS) {
		ERRLOG2(log_p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
			"Failed to open user-resource dir. "
			"(path = %s, code = %d)", usr_dirpath, rv);
		return DIVY_FS_ST_ERR;
	}

	/* 最大リソースディレクトリSEQ 数を超えたとき (i_maxdir_name + 1) の
	 * ディレクトリを作成する */
	if ((filecount + 1) > fstorage->maxresource_per_dir) {
		res_dirpath = _make_res_dirpath(p, usr_dirpath,
						apr_itoa(p, ++i_maxdir_name));
		rv = apr_dir_make(res_dirpath,
					APR_UREAD | APR_UWRITE | APR_UEXECUTE, p);
		if (rv != APR_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to make resource dir. "
				"(path = %s,code = %d)", res_dirpath, rv);
			return DIVY_FS_ST_ERR;
		}
		filecount = 0;
	}

	/* pdir の生成 */
	*pdir = apr_pcalloc(p, sizeof(divy_pdir_t));
	(*pdir)->fstorage     = fstorage;
	(*pdir)->name         = apr_itoa(p, i_maxdir_name);
	(*pdir)->relativepath = apr_pstrdup(p, &res_dirpath[strlen(fstorage->fsrootpath)]);
	(*pdir)->fullpath     = res_dirpath;
	(*pdir)->isEffective  = 1;	/* 有効である */
	(*pdir)->p            = p;

	(*pdir)->hdir = apr_pcalloc(p, sizeof(divy_hashdir_t));
	(*pdir)->hdir->usr_dirseq  = usr_dirseq;
	(*pdir)->hdir->rs_dirseq   = i_maxdir_name;
	(*pdir)->hdir->usr_dirpath = usr_dirpath;
	(*pdir)->hdir->filecount   = filecount;

	return DIVY_FS_ST_OK;
}

/**
 * 指定されたdirpath のディレクトリを開き、この中に格納されているエントリ
 * (ファイルまたはディレクトリ) 数を数えて、filecount に入れて返却する。
 * なお、"."と".."は除外します。
 *
 * @param p apr_pool_t *
 * @param dirpath const char * 調査したいディレクトリのフルパス
 * @param filecount int * エントリの数(返却値)
 * 			エラー発生時には0となります。
 * @return apr_status_t
 *	APR_SUCCESS: 成功
 *	APR_ENOENT : dirpath のディレクトリが存在しなかった
 *	上記以外   : 何からのエラーが発生してエントリを数えられなかった
 */
static apr_status_t _count_entry_in_directory(apr_pool_t *p,
						const char *dirpath,
						int *filecount)
{
	apr_status_t rv;
	apr_dir_t *dirpath_dir = NULL;
	apr_finfo_t finfo      = { 0 };

	*filecount = 0;	/* 初期化 */

	/* dirpath のディレクトリを開く */
	rv = apr_dir_open(&dirpath_dir, dirpath, p); 
	if (APR_STATUS_IS_ENOENT(rv)) {
		/* ディレクトリが存在しなかった */
		return rv;
	}
	else if (rv != APR_SUCCESS) {
		/* その他のエラー */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
			"Failed to open dir for counting entry."
			"(path = %s, code = %d)", dirpath, rv);
		return rv;
	}

	/* エントリを数える */
	while(apr_dir_read(&finfo, 0, dirpath_dir) != APR_ENOENT) {
		(*filecount)++;
	}
	if (*filecount >= 2) {
		/* "."と".."は除外する */
		*filecount -= 2;
	}

	/* ディレクトリを閉じる */
	rv = apr_dir_close(dirpath_dir);
	if (rv != APR_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
			"Failed to close directory. (path = %s, dir count = %d, "
			"code = %d)", dirpath, *filecount, rv);
		return rv;
	}

	return APR_SUCCESS;
}

/**
 * 指定されたpfile の相対パスをrelativepath に変更する。
 *
 * @param pfile divy_pfile_t * 抽象パス表現へのポインタ
 * @param relativepath const char * 変更後の相対パス
 * @return int 処理ステータス
 *	DIVY_FS_ST_OK               : 正常
 */
static int _move_relativepath(divy_pfile_t *pfile, const char *relativepath)
{
	char *fsrootpath;
	apr_pool_t *p;

	if (pfile == NULL || IS_EMPTY(relativepath) ||
	    pfile->pdir == NULL || pfile->pdir->fstorage == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Some parameter is NULL.");
		return DIVY_FS_ST_INVALID_PARAM;
	}

	fsrootpath = pfile->pdir->fstorage->fsrootpath;
	p          = pfile->p;

	/* fullpath の変更 */
	pfile->name         = divy_extract_finalpath(p, relativepath);
	pfile->relativepath = apr_pstrdup(p, relativepath);
	pfile->fullpath     = ap_make_full_path(p, fsrootpath, relativepath);
	ap_no2slash((char *)pfile->fullpath);

	/* pdir の情報を一部更新
	 * (note)
	 * 	全ての情報を更新するのは難しい。何故なら、usr_dirpath などの
	 * 	パラメータがそもそもfullpath に含まれているかどうか判断出来ないからだ。
	 * 	仕方がないので、更新できるところだけを何とかして、残りは無視します。*/
	pfile->pdir->relativepath = divy_extract_parentath(p, relativepath);
	ap_no2slash((char *)pfile->pdir->relativepath);
	pfile->pdir->name     = divy_extract_finalpath(p, pfile->pdir->relativepath);
	pfile->pdir->fullpath = ap_make_full_path(p, fsrootpath, pfile->pdir->relativepath);
	ap_no2slash((char *)pfile->pdir->fullpath);
	pfile->pdir->isEffective = 0;	 /* 情報が正確ではなくなった */
	pfile->pdir->hdir        = NULL; /* 正確ではないのでNULLクリア */

	return DIVY_FS_ST_OK;
}

/**
 * relativepath をパースしてhdir を生成して返却する.
 *
 * @param fstorage divy_fstorage_t * ファイルストレージを表す構造体へのポインタ
 * @param p apr_pool_t * 作業用プール
 * @param relativepath const char * ディレクトリ相対パス
 * @param hdir divy_hashdir_t ** 生成したhdir。
 * 	relativepath が所定の形式でなければNULL。
 * @return int 処理ステータス
 *	DIVY_FS_ST_OK               : 正常
 *	DIVY_FS_ST_FORMAT_MISSMATH  : relativepath が所定の形式ではなかった
 *	DIVY_FS_ST_FORMAT_SPECIAL   : 固定一時ディレクトリ用のフォーマットだった
 */
static int _parse_hdir_relativepath(divy_fstorage_t *fstorage, apr_pool_t *p,
					const char *relativepath,
					divy_hashdir_t **hdir)
{
	const char *path, *str_rs_dirseq, *str_usrseq, *str_usr_dirseq;
	char *usr_dirpath;
	int found_specialname = 0;

	*hdir = NULL;	/* 初期化 */

	/* 空orNULLならば何も出来ない */
	if (IS_EMPTY(relativepath)) {
		return DIVY_FS_ST_FORMAT_MISSMATH;
	}

	/*
	 * pdir->relativepath が以下のフォーマットを持っていたときパースする
	 * 	(ユーザディレクトリSEQ)/(ユーザIDSEQ)/(リソースディレクトリSEQ)
	 */
	/* リソースディレクトリSEQ の算出 */
	str_rs_dirseq = divy_extract_finalpath(p, relativepath);
	if (IS_FILLED(str_rs_dirseq) &&
	    strcmp(str_rs_dirseq, DIVY_FIXED_RESDIR_NAME) == 0) {
		/* DIVY_FIXED_RESDIR_NAME は特別扱いする */
		found_specialname = 1;
	}
	else if (!dav_divy_isdigit_str(str_rs_dirseq)) {
		/* 数値でなければフォーマット不一致 */
		return DIVY_FS_ST_FORMAT_MISSMATH;
	}

	/* ユーザIDSEQの整合性チェック */
	path =  divy_extract_parentath(p, relativepath);
	str_usrseq = divy_extract_finalpath(p, path);
	if (!dav_divy_isdigit_str(str_usrseq)) {
		/* 数値でなければフォーマット不一致 */
		return DIVY_FS_ST_FORMAT_MISSMATH;
	}
	usr_dirpath = ap_make_full_path(p, fstorage->fsrootpath, path);
	ap_no2slash(usr_dirpath);

	/* ユーザディレクトリSEQ の算出 */
	path =  divy_extract_parentath(p, path);
	str_usr_dirseq = divy_extract_finalpath(p, path);
	if (!dav_divy_isdigit_str(str_usr_dirseq)) {
		/* 数値でなければエラー */
		return DIVY_FS_ST_FORMAT_MISSMATH;
	}

	*hdir = apr_pcalloc(p, sizeof(divy_hashdir_t));
	(*hdir)->usr_dirseq  = atoi(str_usr_dirseq);
	(*hdir)->usr_dirpath = usr_dirpath;
	(*hdir)->filecount   = 0;	/* カウントしていない */
	if (found_specialname) {
		/* (note)
		 * rs_dirseq を0固定にするので、ディレクトリのフルパスと
		 * 差異が出てしまう。だが数値に変換できないので仕方がない。
		 * 将来、これらの整合性を確認する必要が出てきたら
		 * 特別扱いしてください。*/
		(*hdir)->rs_dirseq   = 0;	/* 0 にしておく */
		return DIVY_FS_ST_FORMAT_SPECIAL;
	}
	else {
		(*hdir)->rs_dirseq   = atoi(str_rs_dirseq);
	}

	return DIVY_FS_ST_OK;
}


