/**
 * $Id$
 *
 * サムネイルファイル周りの操作を行う関数、構造体の定義
 *
 */
#include "apr.h"
#include "apr_strings.h"
#include "apr_pools.h"
#include "apr_time.h"
#include "apr_file_info.h"
#include "apr_file_io.h"

#if APR_HAVE_UNISTD_H
#include <unistd.h>     /* for getpid */
#endif	/* APR_HAVE_UNISTD_H */

#include "httpd.h"
#include "http_protocol.h"
#include "util_filter.h"

#include "tf_thumbnail.h"
#include "tf_storage.h"
#include "util_common.h"
#include "tf_sbuf.h"
#include "mod_dav_tf.h"
#include "util.h"

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * サムネイルファイルが格納されるディレクトリの名称
 */
#define DIVY_THUMBNAIL_DIRNAME	".thumbnail"

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static int _support_thumbnail(request_rec *r);
static char * _make_thumbnail_path(apr_pool_t *p, request_rec *r, const char *relativepath);
static char * _make_thumbnail_ppath(apr_pool_t *p, request_rec *r, const char *relativepath);

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * 抽出元イメージの相対パスrelativepath からサムネイルファイルを算出し、
 * 存在すれば、サムネイルデータを読み込んで文字列として返却する.
 *
 */
DIVY_DECLARE(int) divy_thumbnail_read(apr_pool_t *p, request_rec *r,
                                       const char *relativepath,
                                       char **data,
                                       apr_size_t *nreadbyte)
{
	apr_status_t rv;
	char *path, *buf;
	divy_sbuf *sbuf = NULL;
	apr_file_t *fd = NULL;
	apr_size_t nbytes;
	int ret = DIVY_THMNL_ST_OK;

	*data = NULL;	/* 初期化 */
	*nreadbyte = 0;

	if (!_support_thumbnail(r)) {
		return DIVY_THMNL_ST_NOTSUPPORTED;
	}

	/* サムネイルファイルのパスを取得 */
	path = _make_thumbnail_path(p, r, relativepath);
	if (IS_EMPTY(path)) {
		return DIVY_THMNL_ST_INVALID_PARAM;
	}

	/* サムネイルファイルを開く */
	rv = apr_file_open(&fd, path, APR_READ | APR_BINARY, 0, p);
	if (rv == APR_ENOENT) {
		ret = DIVY_THMNL_ST_NOTFOUND;
		/* サムネイルファイルが存在しないのは正常ケース */
		goto cleanup_thumbnail_file;
	}
	else if (rv != APR_SUCCESS) {
		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to open thumbnail file.(path = %s, code = %d)", path, rv);
		ret = DIVY_THMNL_ST_READERR;
		goto cleanup_thumbnail_file;
	}

	/* 可変長バッファの初期化 */
	divy_sbuf_create(p, &sbuf, 43710);

	/* ファイルからデータを取得する */
	buf = apr_pcalloc(p, sizeof(char) * 4096);
	while (1) {
		nbytes = 4096;
		rv = apr_file_read(fd, buf, &nbytes);
		if (rv == APR_EOF) {
			if (nbytes) {
				divy_sbuf_appendbyte(sbuf, nbytes, buf);
			}
			break;
		}
		else if (rv != APR_SUCCESS) {
			ret = DIVY_THMNL_ST_READERR;
			goto cleanup_thumbnail_file;
		}
		if (nbytes == 0) break;

		divy_sbuf_appendbyte(sbuf, nbytes, buf);
	}
	*data = divy_sbuf_tostring(sbuf);
	*nreadbyte = divy_sbuf_getlength(sbuf);

cleanup_thumbnail_file:
	/* サムネイルファイルを閉じる */
	if (fd != NULL) {
		(void) apr_file_close(fd);
	}

	return ret;
}

/**
 * 抽出元イメージの相対パスrelativepath からサムネイルファイルパスを算出し、
 * 指定されたサムネイルデータdata を書き込む.
 *
 */
DIVY_DECLARE(int) divy_thumbnail_write(apr_pool_t *p, request_rec *r,
                                        const char *relativepath,
                                        char *data,
                                        apr_size_t nwritebyte)
{
	apr_status_t rv;
	char *path;
	apr_file_t *fd = NULL;
	apr_size_t nbytes = nwritebyte;
	int ret = DIVY_THMNL_ST_OK;

	if (!_support_thumbnail(r)) {
		return DIVY_THMNL_ST_NOTSUPPORTED;
	}

	/* サムネイルファイルのパスを取得 */
	path = _make_thumbnail_path(p, r, relativepath);
	if (IS_EMPTY(path)) {
		return DIVY_THMNL_ST_INVALID_PARAM;
	}

	/* サムネイルファイルを開く */
	rv = apr_file_open(&fd, path,
			APR_WRITE | APR_CREATE | APR_TRUNCATE | APR_BINARY, APR_OS_DEFAULT, p);
	if (rv == APR_ENOENT) {
		/* 親ディレクトリのチェック */
		apr_dir_t *pdir = NULL;
		const char *ppath = divy_extract_parentath(p, path);

		rv = apr_dir_open(&pdir, ppath, p);
		if (rv == APR_ENOENT) {
			rv = apr_dir_make_recursive(ppath, APR_UREAD | APR_UWRITE | APR_UEXECUTE, p);
			if (rv != APR_SUCCESS) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
						"Failed to make thumbnail dir (path = %s, code = %d", ppath, rv);
				ret = DIVY_THMNL_ST_WRITEERR;
			}
		}
		else if (rv != APR_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
					"Failed to open thumbnail dir (path = %s, code = %d", ppath, rv);
			ret = DIVY_THMNL_ST_WRITEERR;
		}

		if (pdir != NULL) {
			apr_dir_close(pdir);
		}
		if (ret != DIVY_THMNL_ST_OK) goto cleanup_thumbnail_file;

		/* もう一度開いてみる */
		rv = apr_file_open(&fd, path,
				APR_WRITE | APR_CREATE | APR_TRUNCATE | APR_BINARY, APR_OS_DEFAULT, p);
	}

	if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to open thumbnail file.(path = %s, code = %d)", path, rv);
		ret = DIVY_THMNL_ST_WRITEERR;
		goto cleanup_thumbnail_file;
	}

	/* サムネイルデータを書き込む */
	rv = apr_file_write(fd, data, &nbytes);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to write thumbnail data. (path = %s, code = %d)", path, rv);
		ret = DIVY_THMNL_ST_WRITEERR;

		/* ファイルを削除 */
		(void) apr_file_remove(path, p);
		goto cleanup_thumbnail_file;
	}

cleanup_thumbnail_file:
	/* サムネイルファイルを閉じる */
	if (fd != NULL) {
		(void) apr_file_close(fd);
	}

	return ret;
}

/**
 * 抽出元イメージの相対パスrelativepath からサムネイルファイルパスを算出し、
 * このファイルの持つサムネイルファイルを物理的に削除する.
 *
 */
DIVY_DECLARE(int) divy_thumbnail_remove(apr_pool_t *p, request_rec *r,
                                        const char *relativepath)
{
	apr_status_t rv;
	int ret = DIVY_THMNL_ST_OK;
	const char *path;

	/* サムネイルファイルのパスを取得 */
	path = _make_thumbnail_path(p, r, relativepath);
	if (IS_EMPTY(path)) {
		return DIVY_THMNL_ST_INVALID_PARAM;
	}

	/* サムネイルファイルを削除 */
	rv = apr_file_remove(path, p);
	if (rv != APR_ENOENT && rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to remove thumbnail file.(path = %s, code = %d)", path, rv);
		ret = DIVY_THMNL_ST_REMOVEERR;
	}

	return ret;
}

/**
 * 抽出元イメージが格納された親フォルダのパスpfullpath から
 * サムネイルファイルパスを算出し、このサムネイルパスの親ディレクトリが
 * 削除できれば削除する.
 * (サムネイル機能をサポートしていなくてもこの関数はコールできます)
 *
 */
DIVY_DECLARE(int) divy_thumbnail_removepdir(apr_pool_t *p, request_rec *r,
                                            const char *pfullpath)
{
	apr_status_t rv;
	const char *path;
	int ret = DIVY_THMNL_ST_OK;

	if (IS_EMPTY(pfullpath)) {
		return DIVY_THMNL_ST_INVALID_PARAM;
	}

	/* サムネイルパスを取得 */
	path = ap_make_full_path(p, pfullpath, DIVY_THUMBNAIL_DIRNAME);
	if (IS_EMPTY(path)) {
		return DIVY_THMNL_ST_INVALID_PARAM;
	}

	/* 対象ディレクトリを削除してみる */
	rv = apr_dir_remove(path, p);
	if (APR_STATUS_IS_ENOTEMPTY(rv)) {
		/* ディレクトリの中にエントリがあったので失敗 */
		ret = DIVY_THMNL_ST_NOREMOVEDIR;
	}
	else if (APR_STATUS_IS_ENOENT(rv)) {
		/* そのディレクトリエントリが既に存在しなかった */
		ret = DIVY_THMNL_ST_OK;	/* 無いのだから成功 */
	}
	else if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to remove thumbnail directory.(path = %s, code = %d) "
				"but is is probably occurred by concurrent access, so "
				"please ignore this.", path, rv);
		ret = DIVY_THMNL_ST_REMOVEERR;
	}

	return ret;
}



/**
 * ソースの抽出元イメージを相対パスsrc_relativepath から dst_relativepath に
 * コピーするとき、サムネイルファイルもコピーする.
 *
 */
DIVY_DECLARE(int) divy_thumbnail_copy(apr_pool_t *p, request_rec *r,
                                       const char *src_relativepath,
                                       const char *dst_relativepath)
{
	apr_status_t rv;
	char *src, *dst;
	const char *ppath;
	int ret = DIVY_THMNL_ST_OK;
	apr_dir_t *pdir = NULL;
	apr_file_t *fd  = NULL;

	if (!_support_thumbnail(r)) {
		return DIVY_THMNL_ST_NOTSUPPORTED;
	}

	if (IS_EMPTY(src_relativepath) || IS_EMPTY(dst_relativepath) ||
		p == NULL || r == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Empty values were found.");
		return DIVY_THMNL_ST_INVALID_PARAM;
	}

	/* サムネイルファイルパスのsrc と dst パスを取得 */
	src = _make_thumbnail_path(p, r, src_relativepath);
	if (IS_EMPTY(src)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"src filepath is EMPTY.");
		return DIVY_THMNL_ST_INVALID_PARAM;
	}

	/* そもそもサムネイルファイルは存在するのか? */
	rv = apr_file_open(&fd, src, APR_READ | APR_BINARY, 0, p);
	if (rv == APR_ENOENT) {
		if (fd != NULL) {
			apr_file_close(fd);
		}
		return DIVY_THMNL_ST_OK;	/* 成功扱いとする */
	}
	if (fd != NULL) {
		apr_file_close(fd);
	}

	dst = _make_thumbnail_path(p, r, dst_relativepath);
	if (IS_EMPTY(dst)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"dst filepath is EMPTY.");
		return DIVY_THMNL_ST_INVALID_PARAM;
	}


	/* dst の親ディレクトリは存在するのか? */
	ppath = divy_extract_parentath(p, dst);
	rv = apr_dir_open(&pdir, ppath, p);
	if (rv == APR_ENOENT) {
		/* ディレクトリを作る */
		rv = apr_dir_make_recursive(ppath, APR_UREAD | APR_UWRITE | APR_UEXECUTE, p);
		if (rv != APR_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
					"Failed to make thumbnail dir (path = %s, code = %d", ppath, rv);
			ret = DIVY_THMNL_ST_WRITEERR;
		}
	}
	else if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to open thumbnail dir (path = %s, code = %d", ppath, rv);
		ret = DIVY_THMNL_ST_WRITEERR;
	}
	if (pdir != NULL) {
		apr_dir_close(pdir);
	}
	if (ret != DIVY_THMNL_ST_OK) {
		return ret;
	}

	/* サムネイルファイルのコピー */
	rv = apr_file_copy(src, dst, APR_FILE_SOURCE_PERMS, p);
	if (rv == APR_ENOENT) {
		ret = DIVY_THMNL_ST_OK;	/* 成功したこととする */
	}
	else if (rv != APR_SUCCESS) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to copy thumbnail file (src = %s, dst = %s, code = %d).",
				src, dst, rv);
		ret = DIVY_THMNL_ST_WRITEERR;
	}
	return ret;
}


/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * サムネイル機能をサポートしているかどうか(0/1)
 */
static int _support_thumbnail(request_rec *r)
{
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

	return (dconf->thumbnail == DIVY_THUMBNAIL_ON) ? 1 : 0;
}

/**
 * 抽出元イメージファイルのパスからサムネイルファイルのパスを算出して
 * 返却する.
 *
 */
static char * _make_thumbnail_path(apr_pool_t *p, request_rec *r, const char *relativepath)
{
	char *ppath = NULL, *fullpath;

	if (IS_EMPTY(relativepath) || p == NULL || r == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Empty values were found.");
		return NULL;
	}
	
	/* 親パスを取得 */
	ppath = _make_thumbnail_ppath(p, r, relativepath);
	if (IS_EMPTY(ppath)) {
		return NULL;
	}
	fullpath = ap_make_full_path(p, ppath, divy_extract_finalpath(p, relativepath));
	ap_no2slash(fullpath);

	return fullpath;
}

/**
 * 抽出元イメージファイルのパスからサムネイルファイルのパスを算出して
 * その親ディレクトリパスを返却する.
 *
 */
static char * _make_thumbnail_ppath(apr_pool_t *p, request_rec *r, const char *relativepath)
{
	char *path, *fullpath;
	dav_divy_dir_conf *dconf;

	if (IS_EMPTY(relativepath) || p == NULL || r == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Empty values were found.");
		return NULL;
	}
	dconf = dav_divy_get_dir_config(r);
	if (IS_EMPTY(dconf->fsrootpath)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"fsrootpath is EMPTY.");
		return NULL;
	}

	path = ap_make_full_path(p, divy_extract_parentath(p, relativepath), DIVY_THUMBNAIL_DIRNAME);
	if (IS_EMPTY(path)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to make fullpath for thumbnail directory.");
		return NULL;
	}

	fullpath = ap_make_full_path(p, dconf->fsrootpath, path);
	ap_no2slash(fullpath);

	return fullpath;
}



